/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection;

import io.netty.util.Timeout;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.redisson.api.NodeType;
import org.redisson.client.RedisClient;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.config.BaseMasterSlaveServersConfig;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.config.ReadMode;
import org.redisson.config.ReplicatedServersConfig;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.MasterSlaveConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.redisson.connection.ServiceManager;
import org.redisson.misc.RedisURI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicatedConnectionManager
extends MasterSlaveConnectionManager {
    private static final String ROLE_KEY = "role";
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final AtomicReference<InetSocketAddress> currentMaster = new AtomicReference();
    private volatile Timeout monitorFuture;
    private ReplicatedServersConfig cfg;

    public ReplicatedConnectionManager(ReplicatedServersConfig cfg, ServiceManager serviceManager) {
        super(cfg, serviceManager);
    }

    @Override
    public void doConnect(Set<RedisURI> disconnectedSlaves, Function<RedisURI, String> hostnameMapper) {
        if (this.cfg.getNodeAddresses().isEmpty()) {
            throw new IllegalArgumentException("At least one Redis node should be defined!");
        }
        for (String address : this.cfg.getNodeAddresses()) {
            RedisURI addr = new RedisURI(address);
            CompletionStage<RedisConnection> connectionFuture = this.connectToNode(this.cfg, addr, addr.getHost());
            RedisConnection connection = null;
            try {
                connection = connectionFuture.toCompletableFuture().join();
            }
            catch (Exception exception) {
                // empty catch block
            }
            if (connection == null) continue;
            Role role = Role.valueOf(connection.sync(RedisCommands.INFO_REPLICATION, new Object[0]).get(ROLE_KEY));
            if (Role.master.equals((Object)role)) {
                this.currentMaster.set(connection.getRedisClient().getAddr());
                this.log.info("{} is the master", (Object)addr);
                this.config.setMasterAddress(addr.toString());
                continue;
            }
            this.log.info("{} is a slave", (Object)addr);
            this.config.addSlaveAddress(addr.toString());
        }
        if (this.currentMaster.get() == null) {
            this.internalShutdown();
            throw new RedisConnectionException("Can't connect to servers!");
        }
        if (this.config.getReadMode() != ReadMode.MASTER && this.config.getSlaveAddresses().isEmpty()) {
            this.log.warn("ReadMode = {}, but slave nodes are not found! Please specify all nodes in replicated mode.", (Object)this.config.getReadMode());
        }
        super.doConnect(disconnectedSlaves, hostnameMapper);
        this.scheduleMasterChangeCheck(this.cfg);
    }

    @Override
    protected void startDNSMonitoring(RedisClient masterHost) {
    }

    @Override
    protected MasterSlaveServersConfig create(BaseMasterSlaveServersConfig<?> cfg) {
        this.cfg = (ReplicatedServersConfig)cfg;
        MasterSlaveServersConfig res = super.create(cfg);
        res.setDatabase(((ReplicatedServersConfig)cfg).getDatabase());
        return res;
    }

    private void scheduleMasterChangeCheck(ReplicatedServersConfig cfg) {
        if (this.serviceManager.isShuttingDown()) {
            return;
        }
        this.monitorFuture = this.serviceManager.newTimeout(t -> {
            if (this.serviceManager.isShuttingDown()) {
                return;
            }
            Set slaveIPs = Collections.newSetFromMap(new ConcurrentHashMap());
            List<CompletableFuture> roles = cfg.getNodeAddresses().stream().map(address -> {
                RedisURI uri = new RedisURI((String)address);
                return this.checkNode(uri, cfg, slaveIPs);
            }).collect(Collectors.toList());
            CompletableFuture<Void> f = CompletableFuture.allOf(roles.toArray(new CompletableFuture[0]));
            f.whenComplete((r, e) -> {
                if (e == null) {
                    if (roles.stream().noneMatch(role -> Role.master.equals((Object)role.getNow(Role.slave)))) {
                        this.log.error("No master available among the configured addresses, please check your configuration.");
                    }
                    this.checkFailedSlaves(slaveIPs);
                }
                this.scheduleMasterChangeCheck(cfg);
            });
        }, cfg.getScanInterval(), TimeUnit.MILLISECONDS);
    }

    private void checkFailedSlaves(Set<InetSocketAddress> slaveIPs) {
        MasterSlaveEntry entry = this.getEntry(this.singleSlotRange.getStartSlot());
        Set failedSlaves = entry.getAllEntries().stream().filter(e -> e.getNodeType() == NodeType.SLAVE && !slaveIPs.contains(e.getClient().getAddr())).map(e -> e.getClient()).collect(Collectors.toSet());
        for (RedisClient slave : failedSlaves) {
            if (!entry.slaveDown(slave.getAddr(), ClientConnectionsEntry.FreezeReason.MANAGER)) continue;
            this.log.info("slave: {} is down", (Object)slave);
            this.disconnectNode(new RedisURI(slave.getConfig().getAddress().getScheme(), slave.getAddr().getAddress().getHostAddress(), slave.getAddr().getPort()));
        }
    }

    private CompletableFuture<Role> checkNode(RedisURI uri, ReplicatedServersConfig cfg, Set<InetSocketAddress> slaveIPs) {
        CompletionStage<RedisConnection> connectionFuture = this.connectToNode(cfg, uri, uri.getHost());
        return connectionFuture.thenCompose(c -> {
            if (cfg.isMonitorIPChanges()) {
                return this.serviceManager.resolveIP(uri);
            }
            return CompletableFuture.completedFuture(uri);
        }).thenCompose(ip -> {
            if (this.serviceManager.isShuttingDown()) {
                return CompletableFuture.completedFuture(null);
            }
            RedisConnection connection = (RedisConnection)connectionFuture.toCompletableFuture().join();
            if (cfg.isMonitorIPChanges() && !ip.equals(connection.getRedisClient().getAddr())) {
                this.disconnectNode(uri);
                this.log.info("Hostname: {} has changed IP from: {} to {}", uri, connection.getRedisClient().getAddr(), ip);
                return CompletableFuture.completedFuture(null);
            }
            return connection.async(RedisCommands.INFO_REPLICATION, new Object[0]);
        }).thenCompose(r -> {
            if (r == null) {
                return CompletableFuture.completedFuture(null);
            }
            RedisConnection connection = (RedisConnection)connectionFuture.toCompletableFuture().join();
            InetSocketAddress addr = connection.getRedisClient().getAddr();
            Role role = Role.valueOf((String)r.get(ROLE_KEY));
            if (Role.master.equals((Object)role)) {
                InetSocketAddress master = this.currentMaster.get();
                if (master.equals(addr)) {
                    this.log.debug("Current master {} unchanged", (Object)master);
                } else if (this.currentMaster.compareAndSet(master, addr)) {
                    CompletableFuture<RedisClient> changeFuture = this.changeMaster(this.singleSlotRange.getStartSlot(), uri);
                    return changeFuture.handle((ignored, e) -> {
                        if (e != null) {
                            this.log.error("Unable to change master to {}", (Object)addr, e);
                            this.currentMaster.compareAndSet(addr, master);
                        }
                        return role;
                    });
                }
            } else if (!this.config.isSlaveNotUsed()) {
                CompletableFuture<Void> f = this.slaveUp(addr, uri);
                slaveIPs.add(addr);
                return f.thenApply(re -> role);
            }
            return CompletableFuture.completedFuture(role);
        }).whenComplete((r, ex) -> {
            if (ex != null) {
                this.log.error(ex.getMessage(), (Throwable)ex);
            }
        }).toCompletableFuture();
    }

    private CompletableFuture<Void> slaveUp(InetSocketAddress address, RedisURI uri) {
        MasterSlaveEntry entry = this.getEntry(this.singleSlotRange.getStartSlot());
        if (!entry.hasSlave(address)) {
            CompletableFuture<Void> f = entry.addSlave(address, uri, uri.getHost());
            return f.whenComplete((r, e) -> {
                if (e != null) {
                    this.log.error("Unable to add slave", (Throwable)e);
                    return;
                }
                this.log.info("slave: {} added", (Object)address);
            });
        }
        return entry.slaveUpAsync(address, ClientConnectionsEntry.FreezeReason.MANAGER).thenAccept(r -> {
            if (r.booleanValue()) {
                this.log.info("slave: {} is up", (Object)address);
            }
        });
    }

    @Override
    public void shutdown(long quietPeriod, long timeout, TimeUnit unit) {
        if (this.monitorFuture != null) {
            this.monitorFuture.cancel();
        }
        this.closeNodeConnections();
        super.shutdown(quietPeriod, timeout, unit);
    }

    private static enum Role {
        master,
        slave;

    }
}

