/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection.pool;

import java.net.InetSocketAddress;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.redisson.api.NodeType;
import org.redisson.api.RFuture;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.protocol.RedisCommand;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class ConnectionPool<T extends RedisConnection> {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    protected final Queue<ClientConnectionsEntry> entries = new ConcurrentLinkedQueue<ClientConnectionsEntry>();
    final ConnectionManager connectionManager;
    final MasterSlaveServersConfig config;
    final MasterSlaveEntry masterSlaveEntry;

    ConnectionPool(MasterSlaveServersConfig config, ConnectionManager connectionManager, MasterSlaveEntry masterSlaveEntry) {
        this.config = config;
        this.masterSlaveEntry = masterSlaveEntry;
        this.connectionManager = connectionManager;
    }

    public CompletableFuture<Void> add(ClientConnectionsEntry entry) {
        return this.initConnections(entry, true);
    }

    public void addEntry(ClientConnectionsEntry entry) {
        this.entries.add(entry);
    }

    public CompletableFuture<Void> initConnections(ClientConnectionsEntry entry) {
        return this.initConnections(entry, false);
    }

    private CompletableFuture<Void> initConnections(ClientConnectionsEntry entry, boolean checkFreezed) {
        int minimumIdleSize = this.getMinimumIdleSize(entry);
        if (minimumIdleSize == 0 || checkFreezed && entry.isFreezed()) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<Void> initPromise = new CompletableFuture<Void>();
        AtomicInteger initializedConnections = new AtomicInteger(minimumIdleSize);
        int startAmount = Math.min(1, minimumIdleSize);
        AtomicInteger requests = new AtomicInteger(startAmount);
        for (int i = 0; i < startAmount; ++i) {
            this.createConnection(checkFreezed, requests, entry, initPromise, minimumIdleSize, initializedConnections);
        }
        return initPromise;
    }

    private void createConnection(boolean checkFreezed, AtomicInteger requests, ClientConnectionsEntry entry, CompletableFuture<Void> initPromise, int minimumIdleSize, AtomicInteger initializedConnections) {
        if (checkFreezed && (entry.isFreezed() || !this.isHealthy(entry))) {
            int totalInitializedConnections = minimumIdleSize - initializedConnections.get();
            RedisConnectionException cause = new RedisConnectionException("Unable to init enough connections amount! Only " + totalInitializedConnections + " of " + minimumIdleSize + " were initialized. Server: " + entry.getClient().getAddr());
            initPromise.completeExceptionally(cause);
            return;
        }
        CompletableFuture<Void> f = this.acquireConnection(entry, null);
        f.thenAccept(r -> {
            CompletableFuture promise = new CompletableFuture();
            this.createConnection(entry, promise);
            promise.whenComplete((conn, e) -> {
                if (e == null) {
                    if (this.changeUsage()) {
                        conn.decUsage();
                    }
                    if (!initPromise.isDone()) {
                        entry.addConnection((RedisConnection)conn);
                    } else {
                        conn.closeAsync();
                    }
                }
                this.releaseConnection(entry);
                if (e != null) {
                    void var10_17;
                    if (initPromise.isDone()) {
                        return;
                    }
                    for (RedisConnection redisConnection : entry.getAllConnections()) {
                        if (redisConnection.isClosed()) continue;
                        redisConnection.closeAsync();
                    }
                    entry.getAllConnections().clear();
                    for (RedisConnection redisConnection : entry.getAllSubscribeConnections()) {
                        if (redisConnection.isClosed()) continue;
                        redisConnection.closeAsync();
                    }
                    entry.getAllSubscribeConnections().clear();
                    int totalInitializedConnections = minimumIdleSize - initializedConnections.get();
                    if (totalInitializedConnections == 0) {
                        String string = "Unable to connect to Redis server: " + entry.getClient().getAddr();
                    } else {
                        String string = "Unable to init enough connections amount! Only " + totalInitializedConnections + " of " + minimumIdleSize + " were initialized. Redis server: " + entry.getClient().getAddr();
                    }
                    RedisConnectionException cause = new RedisConnectionException((String)var10_17, (Throwable)e);
                    initPromise.completeExceptionally(cause);
                    return;
                }
                int value = initializedConnections.decrementAndGet();
                if (value == 0) {
                    if (initPromise.complete(null)) {
                        this.log.info("{} connections initialized for {}", (Object)minimumIdleSize, (Object)entry.getClient().getAddr());
                    }
                } else if (value > 0 && !initPromise.isDone() && requests.incrementAndGet() <= minimumIdleSize) {
                    this.createConnection(checkFreezed, requests, entry, initPromise, minimumIdleSize, initializedConnections);
                }
            });
        });
    }

    protected CompletableFuture<Void> acquireConnection(ClientConnectionsEntry entry, RedisCommand<?> command) {
        return entry.acquireConnection(command);
    }

    protected abstract int getMinimumIdleSize(ClientConnectionsEntry var1);

    public CompletableFuture<T> get(RedisCommand<?> command) {
        LinkedList<ClientConnectionsEntry> entriesCopy = new LinkedList<ClientConnectionsEntry>(this.entries);
        Iterator iterator = entriesCopy.iterator();
        while (iterator.hasNext()) {
            ClientConnectionsEntry entry = (ClientConnectionsEntry)iterator.next();
            if ((!entry.isFreezed() || entry.isMasterForRead()) && this.isHealthy(entry)) continue;
            iterator.remove();
        }
        if (!entriesCopy.isEmpty()) {
            ClientConnectionsEntry entry = this.config.getLoadBalancer().getEntry(entriesCopy, command);
            return this.acquireConnection(command, entry);
        }
        LinkedList<InetSocketAddress> failed = new LinkedList<InetSocketAddress>();
        LinkedList<InetSocketAddress> freezed = new LinkedList<InetSocketAddress>();
        for (ClientConnectionsEntry entry : this.entries) {
            if (entry.isFailed()) {
                failed.add(entry.getClient().getAddr());
                continue;
            }
            if (!entry.isFreezed()) continue;
            freezed.add(entry.getClient().getAddr());
        }
        StringBuilder errorMsg = new StringBuilder(this.getClass().getSimpleName() + " no available Redis entries. Master entry host: " + this.masterSlaveEntry.getClient().getAddr());
        if (!freezed.isEmpty()) {
            errorMsg.append(" Disconnected hosts: ").append(freezed);
        }
        if (!failed.isEmpty()) {
            errorMsg.append(" Hosts disconnected due to errors during `failedSlaveCheckInterval`: ").append(failed);
        }
        RedisConnectionException exception = new RedisConnectionException(errorMsg.toString());
        CompletableFuture result = new CompletableFuture();
        result.completeExceptionally(exception);
        return result;
    }

    public CompletableFuture<T> get(RedisCommand<?> command, ClientConnectionsEntry entry) {
        return this.acquireConnection(command, entry);
    }

    protected final CompletableFuture<T> acquireConnection(RedisCommand<?> command, ClientConnectionsEntry entry) {
        CompletableFuture result = new CompletableFuture();
        CompletableFuture<Void> f = this.acquireConnection(entry, command);
        f.thenAccept(r -> this.connectTo(entry, result, command));
        result.whenComplete((r, e) -> {
            if (e != null) {
                f.completeExceptionally((Throwable)e);
            }
        });
        return result;
    }

    private boolean isHealthy(ClientConnectionsEntry entry) {
        if (entry.getNodeType() == NodeType.SLAVE && entry.isFailed()) {
            this.checkForReconnect(entry, null);
            return false;
        }
        return true;
    }

    protected T poll(ClientConnectionsEntry entry, RedisCommand<?> command) {
        return (T)entry.pollConnection(command);
    }

    protected CompletionStage<T> connect(ClientConnectionsEntry entry) {
        return entry.connect();
    }

    private void connectTo(ClientConnectionsEntry entry, CompletableFuture<T> promise, RedisCommand<?> command) {
        if (promise.isDone()) {
            this.connectionManager.getServiceManager().getGroup().submit(() -> this.releaseConnection(entry));
            return;
        }
        T conn = this.poll(entry, command);
        if (conn != null) {
            if (!((RedisConnection)conn).isActive() && entry.getNodeType() == NodeType.SLAVE) {
                entry.trySetupFistFail();
            }
            this.connectedSuccessful(entry, promise, conn);
            return;
        }
        this.createConnection(entry, promise);
    }

    private void createConnection(ClientConnectionsEntry entry, CompletableFuture<T> promise) {
        CompletionStage<RedisConnection> connFuture = this.connect(entry);
        connFuture.whenComplete((conn, e) -> {
            if (e != null) {
                this.promiseFailure(entry, promise, (T)e);
                return;
            }
            if (!conn.isActive()) {
                this.promiseFailure(entry, promise, conn);
                return;
            }
            if (this.changeUsage()) {
                promise.thenApply(c -> c.incUsage());
            }
            this.connectedSuccessful(entry, promise, conn);
        });
    }

    protected boolean changeUsage() {
        return true;
    }

    private void connectedSuccessful(ClientConnectionsEntry entry, CompletableFuture<T> promise, T conn) {
        if (((RedisConnection)conn).isActive() && entry.getNodeType() == NodeType.SLAVE) {
            entry.resetFirstFail();
        }
        if (!promise.complete(conn)) {
            this.releaseConnection(entry, conn);
            this.releaseConnection(entry);
        }
    }

    private void promiseFailure(ClientConnectionsEntry entry, CompletableFuture<T> promise, Throwable cause) {
        if (entry.getNodeType() == NodeType.SLAVE) {
            entry.trySetupFistFail();
            if (entry.isFailed()) {
                this.checkForReconnect(entry, cause);
            }
        }
        this.releaseConnection(entry);
        promise.completeExceptionally(cause);
    }

    private void promiseFailure(ClientConnectionsEntry entry, CompletableFuture<T> promise, T conn) {
        if (entry.getNodeType() == NodeType.SLAVE) {
            entry.trySetupFistFail();
            if (entry.isFailed()) {
                ((RedisConnection)conn).closeAsync();
                entry.getAllConnections().remove(conn);
                this.checkForReconnect(entry, null);
            } else {
                this.releaseConnection(entry, conn);
            }
        } else {
            this.releaseConnection(entry, conn);
        }
        this.releaseConnection(entry);
        RedisConnectionException cause = new RedisConnectionException(conn + " is not active!");
        promise.completeExceptionally(cause);
    }

    private void checkForReconnect(ClientConnectionsEntry entry, Throwable cause) {
        this.masterSlaveEntry.slaveDownAsync(entry, ClientConnectionsEntry.FreezeReason.RECONNECT).thenAccept(r -> {
            if (r.booleanValue()) {
                this.log.error("slave {} has been disconnected after {} ms interval since moment of the first failed connection", entry.getClient().getAddr(), this.config.getFailedSlaveCheckInterval(), cause);
                this.scheduleCheck(entry);
            }
        });
    }

    private void scheduleCheck(ClientConnectionsEntry entry) {
        this.connectionManager.getServiceManager().newTimeout(timeout -> {
            ClientConnectionsEntry clientConnectionsEntry = entry;
            synchronized (clientConnectionsEntry) {
                if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT || this.connectionManager.getServiceManager().isShuttingDown()) {
                    return;
                }
            }
            RFuture<RedisConnection> connectionFuture = entry.getClient().connectAsync();
            connectionFuture.whenComplete((c, e) -> {
                ClientConnectionsEntry clientConnectionsEntry = entry;
                synchronized (clientConnectionsEntry) {
                    if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT) {
                        return;
                    }
                }
                if (e != null) {
                    this.scheduleCheck(entry);
                    return;
                }
                if (!c.isActive()) {
                    c.closeAsync();
                    this.scheduleCheck(entry);
                    return;
                }
                RFuture f = c.async(RedisCommands.PING, new Object[0]);
                f.whenComplete((t, ex) -> {
                    try {
                        ClientConnectionsEntry clientConnectionsEntry = entry;
                        // MONITORENTER : clientConnectionsEntry
                        if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT) {
                            // MONITOREXIT : clientConnectionsEntry
                            return;
                        }
                        // MONITOREXIT : clientConnectionsEntry
                        if ("PONG".equals(t)) {
                            if (!this.masterSlaveEntry.slaveUp(entry, ClientConnectionsEntry.FreezeReason.RECONNECT)) return;
                            this.log.info("slave {} has been successfully reconnected", (Object)entry.getClient().getAddr());
                            return;
                        }
                        this.scheduleCheck(entry);
                        return;
                    }
                    finally {
                        c.closeAsync();
                    }
                });
            });
        }, this.config.getFailedSlaveReconnectionInterval(), TimeUnit.MILLISECONDS);
    }

    public void returnConnection(ClientConnectionsEntry entry, T connection) {
        if (entry == null) {
            ((RedisConnection)connection).closeAsync();
            return;
        }
        if (entry.isFreezed() && entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.SYSTEM) {
            ((RedisConnection)connection).closeAsync();
            entry.getAllConnections().remove(connection);
        } else {
            this.releaseConnection(entry, connection);
        }
        this.releaseConnection(entry);
    }

    protected void releaseConnection(ClientConnectionsEntry entry) {
        entry.releaseConnection();
    }

    protected void releaseConnection(ClientConnectionsEntry entry, T conn) {
        entry.releaseConnection((RedisConnection)conn);
    }
}

