/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.cluster;

import io.netty.resolver.AddressResolver;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.ScheduledFuture;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.stream.Collectors;
import org.redisson.api.NatMapper;
import org.redisson.api.NodeType;
import org.redisson.api.RFuture;
import org.redisson.client.RedisClient;
import org.redisson.client.RedisClientConfig;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.RedisException;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.client.protocol.RedisStrictCommand;
import org.redisson.cluster.ClusterNodeInfo;
import org.redisson.cluster.ClusterPartition;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.Config;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.config.ReadMode;
import org.redisson.connection.CRC16;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.MasterSlaveConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.redisson.connection.SingleEntry;
import org.redisson.misc.RedisURI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterConnectionManager
extends MasterSlaveConnectionManager {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ConcurrentMap<Integer, ClusterPartition> lastPartitions = new ConcurrentHashMap<Integer, ClusterPartition>();
    private ScheduledFuture<?> monitorFuture;
    private volatile RedisURI lastClusterNode;
    private RedisStrictCommand<List<ClusterNodeInfo>> clusterNodesCommand;
    private String configEndpointHostName;
    private final NatMapper natMapper;
    private final AtomicReferenceArray<MasterSlaveEntry> slot2entry = new AtomicReferenceArray(16384);
    private final Map<RedisClient, MasterSlaveEntry> client2entry = new ConcurrentHashMap<RedisClient, MasterSlaveEntry>();

    public ClusterConnectionManager(ClusterServersConfig cfg, Config config, UUID id) {
        super(config, id);
        if (cfg.getNodeAddresses().isEmpty()) {
            throw new IllegalArgumentException("At least one cluster node should be defined!");
        }
        this.natMapper = cfg.getNatMapper();
        this.config = this.create(cfg);
        this.initTimer(this.config);
        Throwable lastException = null;
        ArrayList<String> failedMasters = new ArrayList<String>();
        for (String address : cfg.getNodeAddresses()) {
            RedisURI addr = new RedisURI(address);
            CompletionStage<RedisConnection> connectionFuture = this.connectToNode(cfg, addr, addr.getHost());
            try {
                RedisConnection connection = connectionFuture.toCompletableFuture().join();
                if (cfg.getNodeAddresses().size() == 1 && !addr.isIP()) {
                    this.configEndpointHostName = addr.getHost();
                }
                this.clusterNodesCommand = RedisCommands.CLUSTER_NODES;
                if (addr.isSsl()) {
                    this.clusterNodesCommand = RedisCommands.CLUSTER_NODES_SSL;
                }
                List<ClusterNodeInfo> nodes = connection.sync(this.clusterNodesCommand, new Object[0]);
                StringBuilder nodesValue = new StringBuilder();
                for (ClusterNodeInfo clusterNodeInfo : nodes) {
                    nodesValue.append(clusterNodeInfo.getNodeInfo()).append("\n");
                }
                this.log.info("Redis cluster nodes configuration got from {}:\n{}", (Object)connection.getRedisClient().getAddr(), (Object)nodesValue);
                this.lastClusterNode = addr;
                CompletableFuture<Collection<ClusterPartition>> partitionsFuture = this.parsePartitions(nodes);
                Collection<ClusterPartition> partitions = partitionsFuture.join();
                ArrayList<CompletableFuture<Void>> masterFutures = new ArrayList<CompletableFuture<Void>>();
                for (ClusterPartition partition : partitions) {
                    if (partition.isMasterFail()) {
                        failedMasters.add(partition.getMasterAddress().toString());
                        continue;
                    }
                    if (partition.getMasterAddress() == null) {
                        throw new IllegalStateException("Master node: " + partition.getNodeId() + " doesn't have address.");
                    }
                    CompletableFuture<Void> masterFuture = this.addMasterEntry(partition, cfg);
                    masterFutures.add(masterFuture);
                }
                CompletableFuture<Void> masterFuture = CompletableFuture.allOf(masterFutures.toArray(new CompletableFuture[0]));
                try {
                    masterFuture.join();
                }
                catch (CompletionException e) {
                    lastException = e.getCause();
                }
                break;
            }
            catch (Exception e) {
                if (e instanceof CompletionException) {
                    e = (Exception)e.getCause();
                }
                lastException = e;
                this.log.warn(e.getMessage());
            }
        }
        if (this.lastPartitions.isEmpty()) {
            this.stopThreads();
            if (failedMasters.isEmpty()) {
                throw new RedisConnectionException("Can't connect to servers!", lastException);
            }
            throw new RedisConnectionException("Can't connect to servers! Failed masters according to cluster status: " + failedMasters, lastException);
        }
        if (cfg.isCheckSlotsCoverage() && this.lastPartitions.size() != 16384) {
            this.stopThreads();
            if (failedMasters.isEmpty()) {
                throw new RedisConnectionException("Not all slots covered! Only " + this.lastPartitions.size() + " slots are available. Set checkSlotsCoverage = false to avoid this check.", lastException);
            }
            throw new RedisConnectionException("Not all slots covered! Only " + this.lastPartitions.size() + " slots are available. Set checkSlotsCoverage = false to avoid this check. Failed masters according to cluster status: " + failedMasters, lastException);
        }
        this.scheduleClusterChangeCheck(cfg);
    }

    @Override
    public Collection<MasterSlaveEntry> getEntrySet() {
        return this.client2entry.values();
    }

    @Override
    protected MasterSlaveEntry getEntry(RedisURI addr) {
        for (MasterSlaveEntry entry : this.client2entry.values()) {
            if (RedisURI.compare(entry.getClient().getAddr(), addr)) {
                return entry;
            }
            if (!entry.hasSlave(addr)) continue;
            return entry;
        }
        return null;
    }

    @Override
    public MasterSlaveEntry getEntry(RedisClient redisClient) {
        MasterSlaveEntry entry = this.client2entry.get(redisClient);
        if (entry != null) {
            return entry;
        }
        for (MasterSlaveEntry mentry : this.client2entry.values()) {
            if (!mentry.hasSlave(redisClient)) continue;
            return mentry;
        }
        return null;
    }

    @Override
    public MasterSlaveEntry getEntry(InetSocketAddress address) {
        for (MasterSlaveEntry entry : this.client2entry.values()) {
            InetSocketAddress addr = entry.getClient().getAddr();
            if (!addr.getAddress().equals(address.getAddress()) || addr.getPort() != address.getPort()) continue;
            return entry;
        }
        return null;
    }

    @Override
    protected CompletableFuture<RedisClient> changeMaster(int slot, RedisURI address) {
        MasterSlaveEntry entry = this.getEntry(slot);
        RedisClient oldClient = entry.getClient();
        CompletableFuture<RedisClient> future = super.changeMaster(slot, address);
        return future.whenComplete((res, e) -> {
            if (e == null) {
                this.client2entry.remove(oldClient);
                this.client2entry.put(entry.getClient(), entry);
            }
        });
    }

    @Override
    public MasterSlaveEntry getEntry(int slot) {
        return this.slot2entry.get(slot);
    }

    private void addEntry(Integer slot, MasterSlaveEntry entry) {
        MasterSlaveEntry oldEntry = this.slot2entry.getAndSet(slot, entry);
        if (oldEntry != entry) {
            entry.incReference();
            this.shutdownEntry(oldEntry);
        }
        this.client2entry.put(entry.getClient(), entry);
    }

    private void removeEntry(Integer slot) {
        MasterSlaveEntry entry = this.slot2entry.getAndSet(slot, null);
        this.shutdownEntry(entry);
    }

    private void shutdownEntry(MasterSlaveEntry entry) {
        if (entry != null && entry.decReference() == 0) {
            this.client2entry.remove(entry.getClient());
            entry.getAllEntries().forEach(e -> entry.nodeDown((ClientConnectionsEntry)e));
            entry.masterDown();
            entry.shutdownAsync();
            this.subscribeService.remove(entry);
            String slaves = entry.getAllEntries().stream().filter(e -> !e.getClient().getAddr().equals(entry.getClient().getAddr())).map(e -> e.getClient().toString()).collect(Collectors.joining(","));
            this.log.info("{} master and related slaves: {} removed", (Object)entry.getClient().getAddr(), (Object)slaves);
        }
    }

    @Override
    protected RedisClientConfig createRedisConfig(NodeType type, RedisURI address, int timeout, int commandTimeout, String sslHostname) {
        RedisClientConfig result = super.createRedisConfig(type, address, timeout, commandTimeout, sslHostname);
        result.setReadOnly(type == NodeType.SLAVE && this.config.getReadMode() != ReadMode.MASTER);
        return result;
    }

    private CompletableFuture<Void> addMasterEntry(ClusterPartition partition, ClusterServersConfig cfg) {
        CompletableFuture<Void> result = new CompletableFuture<Void>();
        if (partition.isMasterFail()) {
            RedisException e = new RedisException("Failed to add master: " + partition.getMasterAddress() + " for slot ranges: " + partition.getSlotRanges() + ". Reason - server has FAIL flag");
            if (partition.getSlotsAmount() == 0) {
                e = new RedisException("Failed to add master: " + partition.getMasterAddress() + ". Reason - server has FAIL flag");
            }
            result.completeExceptionally(e);
            return result;
        }
        CompletionStage<RedisConnection> connectionFuture = this.connectToNode(cfg, partition.getMasterAddress(), this.configEndpointHostName);
        connectionFuture.whenComplete((connection, ex1) -> {
            MasterSlaveEntry entry;
            if (ex1 != null) {
                this.log.error("Can't connect to master: {} with slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                result.completeExceptionally((Throwable)ex1);
                return;
            }
            MasterSlaveServersConfig config = this.create(cfg);
            config.setMasterAddress(partition.getMasterAddress().toString());
            if (config.checkSkipSlavesInit()) {
                entry = new SingleEntry(this, config);
            } else {
                Set<String> slaveAddresses = partition.getSlaveAddresses().stream().map(r -> r.toString()).collect(Collectors.toSet());
                config.setSlaveAddresses(slaveAddresses);
                entry = new MasterSlaveEntry(this, config);
            }
            CompletableFuture<RedisClient> f = entry.setupMasterEntry(new RedisURI(config.getMasterAddress()), this.configEndpointHostName);
            f.whenComplete((masterClient, ex3) -> {
                if (ex3 != null) {
                    this.log.error("Can't add master: " + partition.getMasterAddress() + " for slot ranges: " + partition.getSlotRanges(), (Throwable)ex3);
                    result.completeExceptionally((Throwable)ex3);
                    return;
                }
                for (Integer slot : partition.getSlots()) {
                    this.addEntry(slot, entry);
                    this.lastPartitions.put(slot, partition);
                }
                if (!config.checkSkipSlavesInit()) {
                    CompletableFuture<Void> fs = entry.initSlaveBalancer(partition.getFailedSlaveAddresses(), this.configEndpointHostName);
                    fs.whenComplete((r, ex) -> {
                        if (ex != null) {
                            this.log.error("unable to add slave for: " + partition.getMasterAddress() + " slot ranges: " + partition.getSlotRanges(), (Throwable)ex);
                            result.completeExceptionally((Throwable)ex);
                            return;
                        }
                        if (!partition.getSlaveAddresses().isEmpty()) {
                            this.log.info("slaves: {} added for slot ranges: {}", (Object)partition.getSlaveAddresses(), (Object)partition.getSlotRanges());
                            if (!partition.getFailedSlaveAddresses().isEmpty()) {
                                this.log.warn("slaves: {} are down for slot ranges: {}", (Object)partition.getFailedSlaveAddresses(), (Object)partition.getSlotRanges());
                            }
                        }
                        if (result.complete(null)) {
                            this.log.info("master: {} added for slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                        } else {
                            this.log.error("unable to add master: {} for slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                        }
                    });
                } else if (result.complete(null)) {
                    this.log.info("master: {} added for slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                } else {
                    this.log.error("unable to add master: {} for slot ranges: {}", (Object)partition.getMasterAddress(), (Object)partition.getSlotRanges());
                }
            });
        });
        return result;
    }

    private void scheduleClusterChangeCheck(final ClusterServersConfig cfg) {
        this.monitorFuture = this.group.schedule(new Runnable(){

            @Override
            public void run() {
                if (ClusterConnectionManager.this.configEndpointHostName != null) {
                    String address = cfg.getNodeAddresses().iterator().next();
                    final RedisURI uri = new RedisURI(address);
                    AddressResolver resolver = ClusterConnectionManager.this.resolverGroup.getResolver(ClusterConnectionManager.this.getGroup().next());
                    Future allNodes = resolver.resolveAll(InetSocketAddress.createUnresolved(uri.getHost(), uri.getPort()));
                    allNodes.addListener(new FutureListener<List<InetSocketAddress>>(){

                        @Override
                        public void operationComplete(Future<List<InetSocketAddress>> future) throws Exception {
                            AtomicReference<Throwable> lastException = new AtomicReference<Throwable>(future.cause());
                            if (!future.isSuccess()) {
                                ClusterConnectionManager.this.checkClusterState(cfg, Collections.emptyIterator(), lastException);
                                return;
                            }
                            ArrayList<RedisURI> nodes = new ArrayList<RedisURI>();
                            for (InetSocketAddress addr : future.getNow()) {
                                RedisURI address = ClusterConnectionManager.this.toURI(uri.getScheme(), addr.getAddress().getHostAddress(), "" + addr.getPort());
                                nodes.add(address);
                            }
                            Iterator nodesIterator = nodes.iterator();
                            ClusterConnectionManager.this.checkClusterState(cfg, nodesIterator, lastException);
                        }
                    });
                } else {
                    AtomicReference lastException = new AtomicReference();
                    ArrayList<RedisURI> nodes = new ArrayList<RedisURI>();
                    ArrayList<RedisURI> slaves = new ArrayList<RedisURI>();
                    for (ClusterPartition partition : ClusterConnectionManager.this.getLastPartitions()) {
                        if (!partition.isMasterFail()) {
                            nodes.add(partition.getMasterAddress());
                        }
                        HashSet<RedisURI> partitionSlaves = new HashSet<RedisURI>(partition.getSlaveAddresses());
                        partitionSlaves.removeAll(partition.getFailedSlaveAddresses());
                        slaves.addAll(partitionSlaves);
                    }
                    Collections.shuffle(nodes);
                    Collections.shuffle(slaves);
                    nodes.addAll(slaves);
                    Iterator nodesIterator = nodes.iterator();
                    ClusterConnectionManager.this.checkClusterState(cfg, nodesIterator, lastException);
                }
            }
        }, (long)cfg.getScanInterval(), TimeUnit.MILLISECONDS);
    }

    private void checkClusterState(ClusterServersConfig cfg, Iterator<RedisURI> iterator, AtomicReference<Throwable> lastException) {
        if (!iterator.hasNext()) {
            if (lastException.get() != null) {
                this.log.error("Can't update cluster state", lastException.get());
            }
            this.scheduleClusterChangeCheck(cfg);
            return;
        }
        if (!this.getShutdownLatch().acquire()) {
            return;
        }
        RedisURI uri = iterator.next();
        CompletionStage<RedisConnection> connectionFuture = this.connectToNode(cfg, uri, this.configEndpointHostName);
        connectionFuture.whenComplete((connection, e) -> {
            if (e != null) {
                lastException.set((Throwable)e);
                this.getShutdownLatch().release();
                this.checkClusterState(cfg, iterator, lastException);
                return;
            }
            this.updateClusterState(cfg, (RedisConnection)connection, iterator, uri, lastException);
        });
    }

    private void updateClusterState(ClusterServersConfig cfg, RedisConnection connection, Iterator<RedisURI> iterator, RedisURI uri, AtomicReference<Throwable> lastException) {
        RFuture future = connection.async(this.clusterNodesCommand, new Object[0]);
        future.whenComplete((nodes, e) -> {
            if (e != null) {
                this.log.error("Unable to execute " + this.clusterNodesCommand, (Throwable)e);
                lastException.set((Throwable)e);
                this.getShutdownLatch().release();
                this.checkClusterState(cfg, iterator, lastException);
                return;
            }
            if (nodes.isEmpty()) {
                this.log.debug("cluster nodes state got from {}: doesn't contain any nodes", (Object)connection.getRedisClient().getAddr());
                this.getShutdownLatch().release();
                this.checkClusterState(cfg, iterator, lastException);
                return;
            }
            this.lastClusterNode = uri;
            StringBuilder nodesValue = new StringBuilder();
            if (this.log.isDebugEnabled()) {
                for (ClusterNodeInfo clusterNodeInfo : nodes) {
                    nodesValue.append(clusterNodeInfo.getNodeInfo()).append("\n");
                }
                this.log.debug("cluster nodes state got from {}:\n{}", (Object)connection.getRedisClient().getAddr(), (Object)nodesValue);
            }
            CompletableFuture<Collection<ClusterPartition>> newPartitionsFuture = this.parsePartitions((List<ClusterNodeInfo>)nodes);
            newPartitionsFuture.whenComplete((newPartitions, ex) -> {
                CompletableFuture<Void> masterFuture = this.checkMasterNodesChange(cfg, (Collection<ClusterPartition>)newPartitions);
                this.checkSlaveNodesChange((Collection<ClusterPartition>)newPartitions);
                masterFuture.whenComplete((res, exc) -> {
                    this.checkSlotsMigration((Collection<ClusterPartition>)newPartitions);
                    this.checkSlotsChange((Collection<ClusterPartition>)newPartitions);
                    this.getShutdownLatch().release();
                    this.scheduleClusterChangeCheck(cfg);
                });
            });
        });
    }

    private void checkSlaveNodesChange(Collection<ClusterPartition> newPartitions) {
        Map<RedisURI, ClusterPartition> lastPartitions = this.getLastPartitonsByURI();
        for (ClusterPartition newPart : newPartitions) {
            ClusterPartition currentPart = lastPartitions.get(newPart.getMasterAddress());
            if (currentPart == null) continue;
            MasterSlaveEntry entry = this.getEntry(currentPart.slots().nextSetBit(0));
            Set<RedisURI> addedSlaves = this.addRemoveSlaves(entry, currentPart, newPart);
            this.upDownSlaves(entry, currentPart, newPart, addedSlaves);
        }
    }

    private void upDownSlaves(MasterSlaveEntry entry, ClusterPartition currentPart, ClusterPartition newPart, Set<RedisURI> addedSlaves) {
        List<RedisURI> c = currentPart.getFailedSlaveAddresses().stream().filter(uri -> !addedSlaves.contains(uri) && !newPart.getFailedSlaveAddresses().contains(uri)).collect(Collectors.toList());
        c.forEach(uri -> {
            currentPart.removeFailedSlaveAddress((RedisURI)uri);
            if (entry.hasSlave((RedisURI)uri) && entry.slaveUp((RedisURI)uri, ClientConnectionsEntry.FreezeReason.MANAGER)) {
                this.log.info("slave: {} is up for slot ranges: {}", uri, (Object)currentPart.getSlotRanges());
            }
        });
        newPart.getFailedSlaveAddresses().stream().filter(uri -> !currentPart.getFailedSlaveAddresses().contains(uri)).forEach(uri -> {
            currentPart.addFailedSlaveAddress((RedisURI)uri);
            if (entry.slaveDown((RedisURI)uri, ClientConnectionsEntry.FreezeReason.MANAGER)) {
                this.disconnectNode((RedisURI)uri);
                this.log.warn("slave: {} has down for slot ranges: {}", uri, (Object)currentPart.getSlotRanges());
            }
        });
    }

    private Set<RedisURI> addRemoveSlaves(MasterSlaveEntry entry, ClusterPartition currentPart, ClusterPartition newPart) {
        HashSet<RedisURI> removedSlaves = new HashSet<RedisURI>(currentPart.getSlaveAddresses());
        removedSlaves.removeAll(newPart.getSlaveAddresses());
        for (RedisURI uri : removedSlaves) {
            currentPart.removeSlaveAddress(uri);
            if (!entry.slaveDown(uri, ClientConnectionsEntry.FreezeReason.MANAGER)) continue;
            this.log.info("slave {} removed for slot ranges: {}", (Object)uri, (Object)currentPart.getSlotRanges());
        }
        HashSet<RedisURI> addedSlaves = new HashSet<RedisURI>(newPart.getSlaveAddresses());
        addedSlaves.removeAll(currentPart.getSlaveAddresses());
        for (RedisURI uri : addedSlaves) {
            ClientConnectionsEntry slaveEntry = entry.getEntry(uri);
            if (slaveEntry != null) {
                currentPart.addSlaveAddress(uri);
                entry.slaveUp(uri, ClientConnectionsEntry.FreezeReason.MANAGER);
                this.log.info("slave: {} added for slot ranges: {}", (Object)uri, (Object)currentPart.getSlotRanges());
                continue;
            }
            CompletableFuture<Void> future = entry.addSlave(uri, false, NodeType.SLAVE, this.configEndpointHostName);
            future.whenComplete((res, ex) -> {
                if (ex != null) {
                    this.log.error("Can't add slave: " + uri, (Throwable)ex);
                    return;
                }
                currentPart.addSlaveAddress(uri);
                entry.slaveUp(uri, ClientConnectionsEntry.FreezeReason.MANAGER);
                this.log.info("slave: {} added for slot ranges: {}", (Object)uri, (Object)currentPart.getSlotRanges());
            });
        }
        return addedSlaves;
    }

    private ClusterPartition find(Collection<ClusterPartition> partitions, Integer slot) {
        return partitions.stream().filter(p -> p.hasSlot(slot)).findFirst().orElseThrow(() -> new IllegalStateException("Unable to find partition with slot " + slot));
    }

    private CompletableFuture<Void> checkMasterNodesChange(ClusterServersConfig cfg, Collection<ClusterPartition> newPartitions) {
        Map<RedisURI, ClusterPartition> lastPartitions = this.getLastPartitonsByURI();
        HashMap<RedisURI, ClusterPartition> addedPartitions = new HashMap<RedisURI, ClusterPartition>();
        HashSet<RedisURI> mastersElected = new HashSet<RedisURI>();
        for (ClusterPartition newPart : newPartitions) {
            boolean masterFound;
            if (newPart.getSlotsAmount() == 0) continue;
            ClusterPartition currentPart = lastPartitions.get(newPart.getMasterAddress());
            boolean bl = masterFound = currentPart != null;
            if (masterFound && newPart.isMasterFail()) {
                for (Integer slot : currentPart.getSlots()) {
                    ClusterPartition newMasterPart = this.find(newPartitions, slot);
                    if (Objects.equals(newMasterPart.getMasterAddress(), currentPart.getMasterAddress())) continue;
                    RedisURI newUri = newMasterPart.getMasterAddress();
                    RedisURI oldUri = currentPart.getMasterAddress();
                    mastersElected.add(newUri);
                    CompletableFuture<RedisClient> future = this.changeMaster(slot, newUri);
                    currentPart.setMasterAddress(newUri);
                    future.whenComplete((res, e) -> {
                        if (e != null) {
                            currentPart.setMasterAddress(oldUri);
                        } else {
                            this.disconnectNode(oldUri);
                        }
                    });
                }
            }
            if (masterFound || newPart.isMasterFail()) continue;
            addedPartitions.put(newPart.getMasterAddress(), newPart);
        }
        addedPartitions.keySet().removeAll(mastersElected);
        if (addedPartitions.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        for (ClusterPartition newPart : addedPartitions.values()) {
            CompletableFuture<Void> future = this.addMasterEntry(newPart, cfg);
            futures.add(future);
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).exceptionally(e -> null);
    }

    private void checkSlotsChange(Collection<ClusterPartition> newPartitions) {
        int newSlotsAmount = newPartitions.stream().mapToInt(ClusterPartition::getSlotsAmount).sum();
        if (newSlotsAmount == this.lastPartitions.size() && this.lastPartitions.size() == 16384) {
            return;
        }
        Set removedSlots = this.lastPartitions.keySet().stream().filter(s -> newPartitions.stream().noneMatch(p -> p.hasSlot((int)s))).collect(Collectors.toSet());
        this.lastPartitions.keySet().removeAll(removedSlots);
        if (!removedSlots.isEmpty()) {
            this.log.info("{} slots found to remove", (Object)removedSlots.size());
        }
        for (Integer slot : removedSlots) {
            this.removeEntry(slot);
        }
        Integer addedSlots = 0;
        for (ClusterPartition clusterPartition : newPartitions) {
            MasterSlaveEntry entry = this.getEntry(clusterPartition.getMasterAddress());
            for (Integer slot : clusterPartition.getSlots()) {
                if (this.lastPartitions.containsKey(slot) || entry == null) continue;
                this.addEntry(slot, entry);
                this.lastPartitions.put(slot, clusterPartition);
                Integer n = addedSlots;
                Integer n2 = addedSlots = Integer.valueOf(addedSlots + 1);
            }
        }
        if (addedSlots > 0) {
            this.log.info("{} slots found to add", (Object)addedSlots);
        }
    }

    private void checkSlotsMigration(Collection<ClusterPartition> newPartitions) {
        Set<ClusterPartition> clusterLastPartitions = this.getLastPartitions();
        Map<String, MasterSlaveEntry> nodeEntries = clusterLastPartitions.stream().collect(Collectors.toMap(p -> p.getNodeId(), p -> this.getEntry(p.slots().nextSetBit(0))));
        HashSet changedSlots = new HashSet();
        block0: for (ClusterPartition currentPartition : clusterLastPartitions) {
            String nodeId = currentPartition.getNodeId();
            for (ClusterPartition newPartition : newPartitions) {
                if (!Objects.equals(nodeId, newPartition.getNodeId())) continue;
                MasterSlaveEntry entry = nodeEntries.get(nodeId);
                BitSet addedSlots = newPartition.copySlots();
                addedSlots.andNot(currentPartition.slots());
                addedSlots.stream().forEach(slot -> {
                    this.addEntry(slot, entry);
                    this.lastPartitions.put(slot, currentPartition);
                    changedSlots.add(slot);
                });
                if (!addedSlots.isEmpty()) {
                    this.log.info("{} slots added to {}", (Object)addedSlots.cardinality(), (Object)currentPartition.getMasterAddress());
                }
                BitSet removedSlots = currentPartition.copySlots();
                removedSlots.andNot(newPartition.slots());
                removedSlots.stream().forEach(slot -> {
                    if (this.lastPartitions.remove(slot, currentPartition)) {
                        this.removeEntry(slot);
                        changedSlots.add(slot);
                    }
                });
                if (!removedSlots.isEmpty()) {
                    this.log.info("{} slots removed from {}", (Object)removedSlots.cardinality(), (Object)currentPartition.getMasterAddress());
                }
                if (addedSlots.isEmpty() && removedSlots.isEmpty()) continue block0;
                currentPartition.clear();
                currentPartition.addSlotRanges(newPartition.getSlotRanges());
                continue block0;
            }
        }
        changedSlots.forEach(this.subscribeService::reattachPubSub);
    }

    private int indexOf(byte[] array, byte element) {
        for (int i = 0; i < array.length; ++i) {
            if (array[i] != element) continue;
            return i;
        }
        return -1;
    }

    @Override
    public int calcSlot(byte[] key) {
        int end;
        if (key == null) {
            return 0;
        }
        int start = this.indexOf(key, (byte)123);
        if (start != -1 && (end = this.indexOf(key, (byte)125)) != -1 && start + 1 < end) {
            key = Arrays.copyOfRange(key, start + 1, end);
        }
        int result = CRC16.crc16(key) % 16384;
        return result;
    }

    @Override
    public int calcSlot(String key) {
        int end;
        if (key == null) {
            return 0;
        }
        int start = key.indexOf(123);
        if (start != -1 && (end = key.indexOf(125)) != -1 && start + 1 < end) {
            key = key.substring(start + 1, end);
        }
        int result = CRC16.crc16(key.getBytes()) % 16384;
        this.log.debug("slot {} for {}", (Object)result, (Object)key);
        return result;
    }

    @Override
    public RedisURI applyNatMap(RedisURI address) {
        return this.natMapper.map(address);
    }

    private CompletableFuture<Collection<ClusterPartition>> parsePartitions(List<ClusterNodeInfo> nodes) {
        ConcurrentHashMap partitions = new ConcurrentHashMap();
        ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
        for (ClusterNodeInfo clusterNodeInfo : nodes) {
            String masterId;
            if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.NOADDR) || clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.HANDSHAKE) || clusterNodeInfo.getAddress() == null || clusterNodeInfo.getSlotRanges().isEmpty() && clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.MASTER) || (masterId = clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.SLAVE) ? clusterNodeInfo.getSlaveOf() : clusterNodeInfo.getNodeId()) == null) continue;
            CompletableFuture<RedisURI> ipFuture = this.resolveIP(clusterNodeInfo.getAddress());
            CompletionStage f = ipFuture.thenAccept(address -> {
                if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.SLAVE)) {
                    ClusterPartition masterPartition = partitions.computeIfAbsent(masterId, k -> new ClusterPartition(masterId));
                    ClusterPartition slavePartition = partitions.computeIfAbsent(clusterNodeInfo.getNodeId(), k -> new ClusterPartition(clusterNodeInfo.getNodeId()));
                    slavePartition.setType(ClusterPartition.Type.SLAVE);
                    slavePartition.setParent(masterPartition);
                    masterPartition.addSlaveAddress((RedisURI)address);
                    if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.FAIL)) {
                        masterPartition.addFailedSlaveAddress((RedisURI)address);
                    }
                } else if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.MASTER)) {
                    ClusterPartition masterPartition = partitions.computeIfAbsent(masterId, k -> new ClusterPartition(masterId));
                    masterPartition.addSlotRanges(clusterNodeInfo.getSlotRanges());
                    masterPartition.setMasterAddress((RedisURI)address);
                    masterPartition.setType(ClusterPartition.Type.MASTER);
                    if (clusterNodeInfo.containsFlag(ClusterNodeInfo.Flag.FAIL)) {
                        masterPartition.setMasterFail(true);
                    }
                }
            });
            futures.add(f);
        }
        CompletableFuture<Void> future = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
        return future.handle((r, e) -> {
            this.addCascadeSlaves(partitions.values());
            List ps = partitions.values().stream().filter(cp -> cp.getType() == ClusterPartition.Type.MASTER && cp.getMasterAddress() != null).collect(Collectors.toList());
            return ps;
        });
    }

    private void addCascadeSlaves(Collection<ClusterPartition> partitions) {
        Iterator<ClusterPartition> iter = partitions.iterator();
        while (iter.hasNext()) {
            ClusterPartition cp = iter.next();
            if (cp.getType() != ClusterPartition.Type.SLAVE) continue;
            if (cp.getParent() != null && cp.getParent().getType() == ClusterPartition.Type.MASTER) {
                ClusterPartition parent = cp.getParent();
                for (RedisURI addr : cp.getSlaveAddresses()) {
                    parent.addSlaveAddress(addr);
                }
                for (RedisURI addr : cp.getFailedSlaveAddresses()) {
                    parent.addFailedSlaveAddress(addr);
                }
            }
            iter.remove();
        }
    }

    @Override
    public void shutdown() {
        if (this.monitorFuture != null) {
            this.monitorFuture.cancel(true);
        }
        this.closeNodeConnections();
        super.shutdown();
    }

    private Map<RedisURI, ClusterPartition> getLastPartitonsByURI() {
        return this.lastPartitions.values().stream().collect(Collectors.toMap(p -> p.getMasterAddress(), p -> p, (e1, e2) -> e1));
    }

    private Set<ClusterPartition> getLastPartitions() {
        return new HashSet<ClusterPartition>(this.lastPartitions.values());
    }

    @Override
    public RedisURI getLastClusterNode() {
        return this.lastClusterNode;
    }

    @Override
    public boolean isClusterMode() {
        return true;
    }
}

