/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.client.handler;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.group.ChannelGroup;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.ImmediateEventExecutor;
import io.netty.util.concurrent.Promise;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisException;
import org.redisson.client.RedisPubSubConnection;
import org.redisson.client.codec.Codec;
import org.redisson.client.protocol.CommandData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionWatchdog
extends ChannelInboundHandlerAdapter {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final Bootstrap bootstrap;
    private final ChannelGroup channels;
    private static final int BACKOFF_CAP = 12;

    public ConnectionWatchdog(Bootstrap bootstrap, ChannelGroup channels) {
        this.bootstrap = bootstrap;
        this.channels = channels;
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        this.channels.add(ctx.channel());
        ctx.fireChannelActive();
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        Object connection = RedisConnection.getFrom(ctx.channel());
        ((RedisConnection)connection).onDisconnect();
        if (!((RedisConnection)connection).isClosed()) {
            EventLoopGroup group = ctx.channel().eventLoop().parent();
            this.reconnect(group, (RedisConnection)connection);
        }
        ctx.fireChannelInactive();
    }

    private void reconnect(final EventLoopGroup group, final RedisConnection connection) {
        group.schedule(new Runnable(){

            @Override
            public void run() {
                ConnectionWatchdog.this.tryReconnect(group, connection, 1);
            }
        }, 100L, TimeUnit.MILLISECONDS);
    }

    private void tryReconnect(final EventLoopGroup group, final RedisConnection connection, final int attempts) {
        if (connection.isClosed() || group.isShuttingDown()) {
            return;
        }
        this.log.debug("reconnecting {} to {} ", connection, connection.getRedisClient().getAddr(), connection);
        this.bootstrap.connect().addListener(new ChannelFutureListener(){

            @Override
            public void operationComplete(ChannelFuture future) throws Exception {
                if (connection.isClosed() || group.isShuttingDown()) {
                    return;
                }
                try {
                    if (future.isSuccess()) {
                        ConnectionWatchdog.this.log.debug("{} connected to {}", (Object)connection, (Object)connection.getRedisClient().getAddr());
                        ConnectionWatchdog.this.reconnect(connection, future.channel());
                        return;
                    }
                }
                catch (RedisException e) {
                    ConnectionWatchdog.this.log.warn("Can't connect " + connection + " to " + connection.getRedisClient().getAddr(), e);
                }
                int timeout = 2 << attempts;
                group.schedule(new Runnable(){

                    @Override
                    public void run() {
                        ConnectionWatchdog.this.tryReconnect(group, connection, Math.min(12, attempts + 1));
                    }
                }, (long)timeout, TimeUnit.MILLISECONDS);
            }
        });
    }

    private void reconnect(final RedisConnection connection, final Channel channel) {
        if (connection.getReconnectListener() != null) {
            RedisConnection rc = new RedisConnection(connection.getRedisClient(), channel);
            Promise<RedisConnection> connectionFuture = ImmediateEventExecutor.INSTANCE.newPromise();
            connection.getReconnectListener().onReconnect(rc, connectionFuture);
            connectionFuture.addListener((GenericFutureListener<Future<RedisConnection>>)new FutureListener<RedisConnection>(){

                @Override
                public void operationComplete(Future<RedisConnection> future) throws Exception {
                    if (future.isSuccess()) {
                        ConnectionWatchdog.this.refresh(connection, channel);
                    }
                }
            });
        } else {
            this.refresh(connection, channel);
        }
    }

    private void reattachPubSub(RedisConnection connection) {
        if (connection instanceof RedisPubSubConnection) {
            RedisPubSubConnection conn = (RedisPubSubConnection)connection;
            for (Map.Entry<String, Codec> entry : conn.getChannels().entrySet()) {
                conn.subscribe(entry.getValue(), entry.getKey());
            }
            for (Map.Entry<String, Codec> entry : conn.getPatternChannels().entrySet()) {
                conn.psubscribe(entry.getValue(), entry.getKey());
            }
        }
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        ctx.channel().close();
    }

    private void refresh(RedisConnection connection, Channel channel) {
        CommandData commandData = connection.getCurrentCommand();
        connection.updateChannel(channel);
        this.reattachBlockingQueue(connection, commandData);
        this.reattachPubSub(connection);
    }

    private void reattachBlockingQueue(RedisConnection connection, final CommandData<?, ?> commandData) {
        if (commandData == null || !commandData.isBlockingCommand()) {
            return;
        }
        ChannelFuture future = connection.send(commandData);
        future.addListener(new ChannelFutureListener(){

            @Override
            public void operationComplete(ChannelFuture future) throws Exception {
                if (!future.isSuccess()) {
                    ConnectionWatchdog.this.log.error("Can't reconnect blocking queue to new connection. {}", (Object)commandData);
                }
            }
        });
    }
}

