/*
 * This file is part of dependency-check-core.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright (c) 2022 Jeremy Long. All Rights Reserved.
 */
package org.owasp.dependencycheck.analyzer;

import java.util.Map;
import java.util.Set;
import javax.annotation.concurrent.ThreadSafe;
import org.owasp.dependencycheck.Engine;
import org.owasp.dependencycheck.analyzer.exception.AnalysisException;
import org.owasp.dependencycheck.data.nvdcve.DatabaseException;
import org.owasp.dependencycheck.dependency.Dependency;
import org.owasp.dependencycheck.dependency.Vulnerability;
import org.owasp.dependencycheck.exception.InitializationException;
import org.owasp.dependencycheck.utils.Settings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This analyzer adds information about known exploited vulnerabilities.
 *
 * @author Jeremy Long
 */
@ThreadSafe
public class KnownExploitedVulnerabilityAnalyzer extends AbstractAnalyzer {

    /**
     * The Logger for use throughout the class
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(KnownExploitedVulnerabilityAnalyzer.class);
    /**
     * The map of known exploited vulnerabilities.
     */
    private Map<String, org.owasp.dependencycheck.data.knownexploited.json.Vulnerability> knownExploited = null;
    /**
     * The name of the analyzer.
     */
    private static final String ANALYZER_NAME = "Known Exploited Vulnerability Analyzer";
    /**
     * The phase that this analyzer is intended to run in.
     */
    private static final AnalysisPhase ANALYSIS_PHASE = AnalysisPhase.POST_FINDING_ANALYSIS;

    /**
     * Returns the name of the analyzer.
     *
     * @return the name of the analyzer.
     */
    @Override
    public String getName() {
        return ANALYZER_NAME;
    }

    /**
     * Returns the phase that the analyzer is intended to run in.
     *
     * @return the phase that the analyzer is intended to run in.
     */
    @Override
    public AnalysisPhase getAnalysisPhase() {
        return ANALYSIS_PHASE;
    }

    /**
     * <p>
     * Returns the setting key to determine if the analyzer is enabled.</p>
     *
     * @return the key for the analyzer's enabled property
     */
    @Override
    protected String getAnalyzerEnabledSettingKey() {
        return Settings.KEYS.ANALYZER_KNOWN_EXPLOITED_ENABLED;
    }

    /**
     * The prepare method does nothing for this Analyzer.
     *
     * @param engine a reference the dependency-check engine
     * @throws InitializationException thrown if there is an exception
     */
    @Override
    public void prepareAnalyzer(Engine engine) throws InitializationException {
        try {
            this.knownExploited = engine.getDatabase().getknownExploitedVulnerabilities();
        } catch (DatabaseException ex) {
            LOGGER.debug("Unable to load the known exploited vulnerabilities", ex);
            throw new InitializationException("Unable to load the known exploited vulnerabilities", ex);
        }
    }

    /**
     * Adds information about the known exploited vulnerabilities to the
     * analysis.
     *
     * @param dependency The dependency being analyzed
     * @param engine The scanning engine
     * @throws AnalysisException is thrown if there is an exception analyzing
     * the dependency.
     */
    @Override
    protected void analyzeDependency(Dependency dependency, Engine engine) throws AnalysisException {
        final Set<Vulnerability> vulns = dependency.getVulnerabilities();
        for (Vulnerability v : vulns) {
            final org.owasp.dependencycheck.data.knownexploited.json.Vulnerability kev = knownExploited.get(v.getName());
            if (kev != null) {
                v.setKnownExploitedVulnerability(kev);
            }
        }
    }

}
