/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.dependencycheck.analyzer;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import org.apache.commons.lang3.builder.CompareToBuilder;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TopDocs;
import org.owasp.dependencycheck.Engine;
import org.owasp.dependencycheck.analyzer.AnalysisPhase;
import org.owasp.dependencycheck.analyzer.Analyzer;
import org.owasp.dependencycheck.analyzer.exception.AnalysisException;
import org.owasp.dependencycheck.data.cpe.CpeMemoryIndex;
import org.owasp.dependencycheck.data.cpe.IndexEntry;
import org.owasp.dependencycheck.data.cpe.IndexException;
import org.owasp.dependencycheck.data.lucene.LuceneUtils;
import org.owasp.dependencycheck.data.nvdcve.CveDB;
import org.owasp.dependencycheck.data.nvdcve.DatabaseException;
import org.owasp.dependencycheck.dependency.Confidence;
import org.owasp.dependencycheck.dependency.Dependency;
import org.owasp.dependencycheck.dependency.Evidence;
import org.owasp.dependencycheck.dependency.EvidenceCollection;
import org.owasp.dependencycheck.dependency.Identifier;
import org.owasp.dependencycheck.dependency.VulnerableSoftware;
import org.owasp.dependencycheck.exception.InitializationException;
import org.owasp.dependencycheck.utils.DependencyVersion;
import org.owasp.dependencycheck.utils.DependencyVersionUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CPEAnalyzer
implements Analyzer {
    private static final Logger LOGGER = LoggerFactory.getLogger(CPEAnalyzer.class);
    static final int MAX_QUERY_RESULTS = 25;
    static final String WEIGHTING_BOOST = "^5";
    static final String CLEANSE_CHARACTER_RX = "[^A-Za-z0-9 ._-]";
    static final String CLEANSE_NONALPHA_RX = "[^A-Za-z]*";
    static final int STRING_BUILDER_BUFFER = 20;
    private CpeMemoryIndex cpe;
    private CveDB cve;
    public static final String NVD_SEARCH_URL = "https://web.nvd.nist.gov/view/vuln/search-results?adv_search=true&cves=on&cpe_version=%s";

    @Override
    public String getName() {
        return "CPE Analyzer";
    }

    @Override
    public AnalysisPhase getAnalysisPhase() {
        return AnalysisPhase.IDENTIFIER_ANALYSIS;
    }

    @Override
    public void initialize() throws InitializationException {
        try {
            this.open();
        }
        catch (IOException ex) {
            LOGGER.debug("Exception initializing the Lucene Index", (Throwable)ex);
            throw new InitializationException("An exception occurred initializing the Lucene Index", ex);
        }
        catch (DatabaseException ex) {
            LOGGER.debug("Exception accessing the database", (Throwable)ex);
            throw new InitializationException("An exception occurred accessing the database", ex);
        }
    }

    public void open() throws IOException, DatabaseException {
        if (!this.isOpen()) {
            this.cve = new CveDB();
            this.cve.open();
            this.cpe = CpeMemoryIndex.getInstance();
            try {
                LOGGER.info("Creating the CPE Index");
                long creationStart = System.currentTimeMillis();
                this.cpe.open(this.cve);
                LOGGER.info("CPE Index Created ({} ms)", (Object)(System.currentTimeMillis() - creationStart));
            }
            catch (IndexException ex) {
                LOGGER.debug("IndexException", (Throwable)ex);
                throw new DatabaseException(ex);
            }
        }
    }

    @Override
    public void close() {
        if (this.cpe != null) {
            this.cpe.close();
            this.cpe = null;
        }
        if (this.cve != null) {
            this.cve.close();
            this.cve = null;
        }
    }

    public boolean isOpen() {
        return this.cpe != null && this.cpe.isOpen();
    }

    protected void determineCPE(Dependency dependency) throws CorruptIndexException, IOException, ParseException {
        String vendors = "";
        String products = "";
        for (Confidence confidence : Confidence.values()) {
            List<IndexEntry> entries;
            if (dependency.getVendorEvidence().contains(confidence)) {
                vendors = this.addEvidenceWithoutDuplicateTerms(vendors, dependency.getVendorEvidence(), confidence);
                LOGGER.debug("vendor search: {}", (Object)vendors);
            }
            if (dependency.getProductEvidence().contains(confidence)) {
                products = this.addEvidenceWithoutDuplicateTerms(products, dependency.getProductEvidence(), confidence);
                LOGGER.debug("product search: {}", (Object)products);
            }
            if (vendors.isEmpty() || products.isEmpty() || (entries = this.searchCPE(vendors, products, dependency.getVendorEvidence().getWeighting(), dependency.getProductEvidence().getWeighting())) == null) continue;
            boolean identifierAdded = false;
            for (IndexEntry e : entries) {
                LOGGER.debug("Verifying entry: {}", (Object)e);
                if (!this.verifyEntry(e, dependency)) continue;
                String vendor = e.getVendor();
                String product = e.getProduct();
                LOGGER.debug("identified vendor/product: {}/{}", (Object)vendor, (Object)product);
                identifierAdded |= this.determineIdentifiers(dependency, vendor, product, confidence);
            }
            if (identifierAdded) break;
        }
    }

    private String addEvidenceWithoutDuplicateTerms(String text, EvidenceCollection ec, Confidence confidenceFilter) {
        String txt = text == null ? "" : text;
        StringBuilder sb = new StringBuilder(txt.length() + 20 * ec.size());
        sb.append(' ').append(txt).append(' ');
        for (Evidence e : ec.iterator(confidenceFilter)) {
            String value = e.getValue();
            if (value.startsWith("http://")) {
                value = value.substring(7).replaceAll("\\.", " ");
            }
            if (value.startsWith("https://")) {
                value = value.substring(8).replaceAll("\\.", " ");
            }
            if (sb.indexOf(" " + value + " ") >= 0) continue;
            sb.append(value).append(' ');
        }
        return sb.toString().trim();
    }

    protected List<IndexEntry> searchCPE(String vendor, String product, Set<String> vendorWeightings, Set<String> productWeightings) {
        ArrayList<IndexEntry> ret = new ArrayList<IndexEntry>(25);
        String searchString = this.buildSearch(vendor, product, vendorWeightings, productWeightings);
        if (searchString == null) {
            return ret;
        }
        try {
            TopDocs docs = this.cpe.search(searchString, 25);
            for (ScoreDoc d : docs.scoreDocs) {
                if (!((double)d.score >= 0.08)) continue;
                Document doc = this.cpe.getDocument(d.doc);
                IndexEntry entry = new IndexEntry();
                entry.setVendor(doc.get("vendor"));
                entry.setProduct(doc.get("product"));
                entry.setSearchScore(d.score);
                if (ret.contains(entry)) continue;
                ret.add(entry);
            }
            return ret;
        }
        catch (ParseException ex) {
            LOGGER.warn("An error occurred querying the CPE data. See the log for more details.");
            LOGGER.info("Unable to parse: {}", (Object)searchString, (Object)ex);
        }
        catch (IOException ex) {
            LOGGER.warn("An error occurred reading CPE data. See the log for more details.");
            LOGGER.info("IO Error with search string: {}", (Object)searchString, (Object)ex);
        }
        return null;
    }

    protected String buildSearch(String vendor, String product, Set<String> vendorWeighting, Set<String> productWeightings) {
        String v = vendor;
        String p = product;
        StringBuilder sb = new StringBuilder(v.length() + p.length() + "product".length() + "vendor".length() + 20);
        if (!this.appendWeightedSearch(sb, "product", p, productWeightings)) {
            return null;
        }
        sb.append(" AND ");
        if (!this.appendWeightedSearch(sb, "vendor", v, vendorWeighting)) {
            return null;
        }
        return sb.toString();
    }

    private boolean appendWeightedSearch(StringBuilder sb, String field, String searchText, Set<String> weightedText) {
        sb.append(' ').append(field).append(":( ");
        String cleanText = this.cleanseText(searchText);
        if (cleanText.isEmpty()) {
            return false;
        }
        if (weightedText == null || weightedText.isEmpty()) {
            LuceneUtils.appendEscapedLuceneQuery(sb, cleanText);
        } else {
            StringTokenizer tokens = new StringTokenizer(cleanText);
            while (tokens.hasMoreElements()) {
                String word = tokens.nextToken();
                StringBuilder temp = null;
                for (String weighted : weightedText) {
                    String weightedStr = this.cleanseText(weighted);
                    if (!this.equalsIgnoreCaseAndNonAlpha(word, weightedStr)) continue;
                    temp = new StringBuilder(word.length() + 2);
                    LuceneUtils.appendEscapedLuceneQuery(temp, word);
                    temp.append(WEIGHTING_BOOST);
                    if (word.equalsIgnoreCase(weightedStr)) break;
                    temp.append(' ');
                    LuceneUtils.appendEscapedLuceneQuery(temp, weightedStr);
                    temp.append(WEIGHTING_BOOST);
                    break;
                }
                sb.append(' ');
                if (temp == null) {
                    LuceneUtils.appendEscapedLuceneQuery(sb, word);
                    continue;
                }
                sb.append((CharSequence)temp);
            }
        }
        sb.append(" ) ");
        return true;
    }

    private String cleanseText(String text) {
        return text.replaceAll(CLEANSE_CHARACTER_RX, " ");
    }

    private boolean equalsIgnoreCaseAndNonAlpha(String l, String r) {
        if (l == null || r == null) {
            return false;
        }
        String left = l.replaceAll(CLEANSE_NONALPHA_RX, "");
        String right = r.replaceAll(CLEANSE_NONALPHA_RX, "");
        return left.equalsIgnoreCase(right);
    }

    private boolean verifyEntry(IndexEntry entry, Dependency dependency) {
        boolean isValid = false;
        if (this.collectionContainsString(dependency.getProductEvidence(), entry.getProduct()) && this.collectionContainsString(dependency.getVendorEvidence(), entry.getVendor())) {
            isValid = true;
        }
        return isValid;
    }

    private boolean collectionContainsString(EvidenceCollection ec, String text) {
        if (text == null) {
            return false;
        }
        String[] words = text.split("[\\s_-]");
        ArrayList<String> list = new ArrayList<String>();
        String tempWord = null;
        for (String word : words) {
            if (tempWord != null) {
                list.add(tempWord + word);
                tempWord = null;
                continue;
            }
            if (word.length() <= 2) {
                tempWord = word;
                continue;
            }
            list.add(word);
        }
        if (tempWord != null) {
            if (!list.isEmpty()) {
                String tmp = (String)list.get(list.size() - 1) + tempWord;
                list.add(tmp);
            } else {
                list.add(tempWord);
            }
        }
        if (list.isEmpty()) {
            return false;
        }
        boolean contains = true;
        for (String word : list) {
            contains &= ec.containsUsedString(word);
        }
        return contains;
    }

    @Override
    public synchronized void analyze(Dependency dependency, Engine engine) throws AnalysisException {
        try {
            this.determineCPE(dependency);
        }
        catch (CorruptIndexException ex) {
            throw new AnalysisException("CPE Index is corrupt.", ex);
        }
        catch (IOException ex) {
            throw new AnalysisException("Failure opening the CPE Index.", ex);
        }
        catch (ParseException ex) {
            throw new AnalysisException("Unable to parse the generated Lucene query for this dependency.", ex);
        }
    }

    protected boolean determineIdentifiers(Dependency dependency, String vendor, String product, Confidence currentConfidence) throws UnsupportedEncodingException {
        Set<VulnerableSoftware> cpes = this.cve.getCPEs(vendor, product);
        DependencyVersion bestGuess = new DependencyVersion("-");
        Confidence bestGuessConf = null;
        boolean hasBroadMatch = false;
        ArrayList<IdentifierMatch> collected = new ArrayList<IdentifierMatch>();
        for (Confidence conf : Confidence.values()) {
            for (Evidence evidence : dependency.getVersionEvidence().iterator(conf)) {
                DependencyVersion evVer = DependencyVersionUtil.parseVersion(evidence.getValue());
                if (evVer == null) continue;
                for (VulnerableSoftware vs : cpes) {
                    IdentifierMatch match;
                    String url;
                    DependencyVersion dbVer = vs.getUpdate() != null && !vs.getUpdate().isEmpty() ? DependencyVersionUtil.parseVersion(vs.getVersion() + '.' + vs.getUpdate()) : DependencyVersionUtil.parseVersion(vs.getVersion());
                    if (dbVer == null) {
                        hasBroadMatch = true;
                        url = String.format(NVD_SEARCH_URL, URLEncoder.encode(vs.getName(), "UTF-8"));
                        match = new IdentifierMatch("cpe", vs.getName(), url, IdentifierConfidence.BROAD_MATCH, conf);
                        collected.add(match);
                        continue;
                    }
                    if (evVer.equals(dbVer)) {
                        url = String.format(NVD_SEARCH_URL, URLEncoder.encode(vs.getName(), "UTF-8"));
                        match = new IdentifierMatch("cpe", vs.getName(), url, IdentifierConfidence.EXACT_MATCH, conf);
                        collected.add(match);
                        continue;
                    }
                    if (evVer.getVersionParts().size() > dbVer.getVersionParts().size() || !evVer.matchesAtLeastThreeLevels(dbVer) || bestGuessConf != null && bestGuessConf.compareTo(conf) <= 0 || bestGuess.getVersionParts().size() >= dbVer.getVersionParts().size()) continue;
                    bestGuess = dbVer;
                    bestGuessConf = conf;
                }
                if (bestGuessConf != null && bestGuessConf.compareTo(conf) <= 0 || bestGuess.getVersionParts().size() >= evVer.getVersionParts().size()) continue;
                bestGuess = evVer;
                bestGuessConf = conf;
            }
        }
        String cpeName = String.format("cpe:/a:%s:%s:%s", vendor, product, bestGuess.toString());
        String url = null;
        if (hasBroadMatch) {
            String cpeUrlName = String.format("cpe:/a:%s:%s", vendor, product);
            url = String.format(NVD_SEARCH_URL, URLEncoder.encode(cpeUrlName, "UTF-8"));
        }
        if (bestGuessConf == null) {
            bestGuessConf = Confidence.LOW;
        }
        IdentifierMatch match = new IdentifierMatch("cpe", cpeName, url, IdentifierConfidence.BEST_GUESS, bestGuessConf);
        collected.add(match);
        Collections.sort(collected);
        IdentifierConfidence bestIdentifierQuality = ((IdentifierMatch)collected.get(0)).getConfidence();
        Confidence bestEvidenceQuality = ((IdentifierMatch)collected.get(0)).getEvidenceConfidence();
        boolean identifierAdded = false;
        for (IdentifierMatch m : collected) {
            if (!bestIdentifierQuality.equals((Object)m.getConfidence()) || !bestEvidenceQuality.equals((Object)m.getEvidenceConfidence())) continue;
            Identifier i = m.getIdentifier();
            if (bestIdentifierQuality == IdentifierConfidence.BEST_GUESS) {
                i.setConfidence(Confidence.LOW);
            } else {
                i.setConfidence(bestEvidenceQuality);
            }
            dependency.addIdentifier(i);
            identifierAdded = true;
        }
        return identifierAdded;
    }

    private static class IdentifierMatch
    implements Comparable<IdentifierMatch> {
        private Confidence evidenceConfidence;
        private IdentifierConfidence confidence;
        private Identifier identifier;

        IdentifierMatch(String type, String value, String url, IdentifierConfidence identifierConfidence, Confidence evidenceConfidence) {
            this.identifier = new Identifier(type, value, url);
            this.confidence = identifierConfidence;
            this.evidenceConfidence = evidenceConfidence;
        }

        public Confidence getEvidenceConfidence() {
            return this.evidenceConfidence;
        }

        public void setEvidenceConfidence(Confidence evidenceConfidence) {
            this.evidenceConfidence = evidenceConfidence;
        }

        public IdentifierConfidence getConfidence() {
            return this.confidence;
        }

        public void setConfidence(IdentifierConfidence confidence) {
            this.confidence = confidence;
        }

        public Identifier getIdentifier() {
            return this.identifier;
        }

        public void setIdentifier(Identifier identifier) {
            this.identifier = identifier;
        }

        public String toString() {
            return "IdentifierMatch{evidenceConfidence=" + (Object)((Object)this.evidenceConfidence) + ", confidence=" + (Object)((Object)this.confidence) + ", identifier=" + this.identifier + '}';
        }

        public int hashCode() {
            int hash = 5;
            hash = 97 * hash + (this.evidenceConfidence != null ? this.evidenceConfidence.hashCode() : 0);
            hash = 97 * hash + (this.confidence != null ? this.confidence.hashCode() : 0);
            hash = 97 * hash + (this.identifier != null ? this.identifier.hashCode() : 0);
            return hash;
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            IdentifierMatch other = (IdentifierMatch)obj;
            if (this.evidenceConfidence != other.evidenceConfidence) {
                return false;
            }
            if (this.confidence != other.confidence) {
                return false;
            }
            return this.identifier == other.identifier || this.identifier != null && this.identifier.equals(other.identifier);
        }

        @Override
        public int compareTo(IdentifierMatch o) {
            return new CompareToBuilder().append((Object)this.confidence, (Object)o.confidence).append((Object)this.evidenceConfidence, (Object)o.evidenceConfidence).append((Object)this.identifier, (Object)o.identifier).toComparison();
        }
    }

    private static enum IdentifierConfidence {
        EXACT_MATCH,
        BEST_GUESS,
        BROAD_MATCH;

    }
}

