/**
 * EasyBeans
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EZBApplicationJNDIResolver.java 2981 2008-04-27 15:58:00Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.easybeans.console.jmxbrowser.service;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.MBeanServer;
import javax.management.MBeanServerConnection;
import javax.management.MBeanServerFactory;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Manages the POJO service used by flex clients.
 * @author Florent Benoit
 */
public class JMXBrowserService implements Serializable {


    /**
     * Serial Version UID.
     */
    private static final long serialVersionUID = -876338222735762564L;

    public JMXBrowserService() {
    }

    private int counter = 0;

    public int getCounter() {
        return ++counter;
    }

    public Document getXML() {

        // Create builder with factory
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = null;
        try {
            builder = factory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        // Create Document
        Document document = builder.newDocument();

        // Append root element
        Element root = document.createElement("root");
        document.appendChild(root);

        // Map between domain
        Map<String, Map<String, List<ObjectName>>> mBeansMap = new TreeMap<String, Map<String, List<ObjectName>>>();

        // Get all MBeans of this MBean server
        Set<ObjectName> names = null;
        try {
            names = getMBeanServerConnection().queryNames(null, null);
        } catch (IOException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        } catch (Exception e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }

        // Iterator
        Iterator<ObjectName> it = names.iterator();

        // Loop on all ObjectName
        while (it.hasNext()) {
            ObjectName objectName = it.next();

            // J2EE, Standard or other MBean ?
            String j2eeType = objectName.getKeyProperty("j2eeType");
            String standardtype = objectName.getKeyProperty("type");

            // Key to use
            String key = null;
            String keyValue = null;
            if (j2eeType != null) {
                key = "Java EE";
                keyValue = j2eeType;
            } else if (standardtype != null) {
                key = "Standard";
                keyValue = standardtype;
            } else {
                key = "Others";
                keyValue = "other";
            }

            // Get element of this domain
            Map<String, List<ObjectName>> typeList = mBeansMap.get(key);

            // Create a new Map if null
            if (typeList == null) {
                typeList = new TreeMap<String, List<ObjectName>>();
                mBeansMap.put(key, typeList);
            }

            // Get list
            List<ObjectName> lst = typeList.get(keyValue);
            if (lst == null) {
                lst = new ArrayList<ObjectName>();
                typeList.put(keyValue, lst);
            }

            // Add value to the list
            lst.add(objectName);
        }

        // Loop on sorted MBeans
        Iterator<String> itMBean = mBeansMap.keySet().iterator();
        while (itMBean.hasNext()) {
            String keyMBean = itMBean.next();
            Element domainElement = document.createElement("node");
            domainElement.setAttribute("label", keyMBean);
            domainElement.setAttribute("isBranch", "true");
            root.appendChild(domainElement);

            // Subtype
            Map<String, List<ObjectName>> subTypeMap = mBeansMap.get(keyMBean);
            Iterator<String> subTypeIt = subTypeMap.keySet().iterator();
            while (subTypeIt.hasNext()) {
                String type = subTypeIt.next();
                Element subElement = document.createElement("node");
                subElement.setAttribute("label", type);
                subElement.setAttribute("isBranch", "true");
                domainElement.appendChild(subElement);

                List<ObjectName> oNames = subTypeMap.get(type);
                for (ObjectName oName : oNames) {
                    Element txtElement = document.createElement("mbean");
                    txtElement.setAttribute("isBranch", "false");
                    txtElement.appendChild(document.createTextNode(oName.toString()));
                    subElement.appendChild(txtElement);
                }

            }

        }

        // Prepare the DOM document for writing
        Source source = new DOMSource(document);

        // Prepare the output file
        StringWriter sw = new StringWriter();
        Result result = new StreamResult(sw);

        // Write the DOM document to the file
        Transformer xformer;
        try {
            xformer = TransformerFactory.newInstance().newTransformer();
            xformer.transform(source, result);
        } catch (TransformerConfigurationException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (TransformerFactoryConfigurationError e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (TransformerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        // System.out.println(sw.toString());

        return document;
    }

    // For combobox, return an array of String
    public String[] getAttributes(final String oName) {

        List<String> lst = new ArrayList<String>();

        // Use first MBean server
        MBeanServerConnection mBeanServerConnection = null;
        try {
            mBeanServerConnection = getMBeanServerConnection();
        } catch (Exception e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }

        MBeanInfo info = null;
        try {
            info = mBeanServerConnection.getMBeanInfo(new ObjectName(oName));
        } catch (InstanceNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IntrospectionException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (ReflectionException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (MalformedObjectNameException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (NullPointerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        MBeanAttributeInfo[] attributeArray = info.getAttributes();
        for (int x = 0; x < attributeArray.length; x++) {
            String attributeName = attributeArray[x].getName();

            lst.add(attributeName);
            /*
             * try { Object objectValue = server.getAttribute( on, attributeName );
             * if( objectValue != null ) { attributeElement.addContent(
             * objectValue.toString() );
             */
        }

        String[] array = lst.toArray(new String[lst.size()]);
        Arrays.sort(array);
        return array;
    }

    class MBeanAttributeData {
        public MBeanAttributeInfo mBeanAttributeInfo = null;

        public String value = null;
    }

    public void updateAttributeValue(final String oNameStr, final String name, final Object value)
            throws InstanceNotFoundException, AttributeNotFoundException, MBeanException, ReflectionException, Exception {

        // Get Object Name
        ObjectName oName = getObjectName(oNameStr);

        javax.management.Attribute attribute = new javax.management.Attribute(name, value);

        getMBeanServerConnection().setAttribute(oName, attribute);

    }

    public Object callOperation(final String oNameStr, final String operationName, final Object[] signatureObj,
            final Object[] params) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException,
            Exception {

        // Get Object Name
        ObjectName oName = getObjectName(oNameStr);

        String[] signature = null;
        if (signatureObj != null) {
            signature = new String[signatureObj.length];
            int i = 0;
            for (Object o : signatureObj) {
                signature[i++] = o.toString();
            }
        }

        // Invoke
        return getMBeanServerConnection().invoke(oName, operationName, params, signature);

    }

    protected ObjectName getObjectName(final String oNameStr) {
        ObjectName oName = null;
        try {
            oName = new ObjectName(oNameStr);
        } catch (MalformedObjectNameException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        } catch (NullPointerException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }
        return oName;
    }

    public List<JMXAttributes> getJMXAttributes(final String oNameStr) {

        // Create a Map with key = MBean name and value the value of the MBean
        // data info.
        Map<String, MBeanAttributeData> attributesMap = new TreeMap<String, MBeanAttributeData>();

        MBeanInfo info = null;

        ObjectName oName = getObjectName(oNameStr);

        try {
            info = getMBeanServerConnection().getMBeanInfo(oName);
        } catch (InstanceNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IntrospectionException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (ReflectionException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (NullPointerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        MBeanAttributeInfo[] attributeArray = info.getAttributes();
        for (int x = 0; x < attributeArray.length; x++) {
            MBeanAttributeInfo mBeanAttributeInfo = attributeArray[x];

            String attributeName = mBeanAttributeInfo.getName();

            MBeanAttributeData mBeanAttributeData = new MBeanAttributeData();
            mBeanAttributeData.mBeanAttributeInfo = mBeanAttributeInfo;

            Object objectValue = null;
            try {
                objectValue = getMBeanServerConnection().getAttribute(oName, attributeName);
            } catch (AttributeNotFoundException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (InstanceNotFoundException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (MBeanException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (ReflectionException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (Exception e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }

            mBeanAttributeData.value = "null";
            if (objectValue != null) {
                // is Array
                boolean b = objectValue.getClass().isArray();
                if (b) {
                    mBeanAttributeData.value = Arrays.asList((Object[]) objectValue).toString();
                } else {
                    mBeanAttributeData.value = objectValue.toString();
                }

            }

            attributesMap.put(attributeName, mBeanAttributeData);

        }

        List<JMXAttributes> list = new ArrayList<JMXAttributes>();

        // Loop on sorted MBeans attributes data
        Iterator<String> itMBeanData = attributesMap.keySet().iterator();
        while (itMBeanData.hasNext()) {
            String attributeName = itMBeanData.next();

            MBeanAttributeData mBeanAttributeData = attributesMap.get(attributeName);

            JMXAttributes jmxAttributes = new JMXAttributes();
            list.add(jmxAttributes);

            jmxAttributes.setName(attributeName);
            jmxAttributes.setDesc(mBeanAttributeData.mBeanAttributeInfo.getDescription());
            jmxAttributes.setType(mBeanAttributeData.mBeanAttributeInfo.getType());
            jmxAttributes.setValue(mBeanAttributeData.value);
            jmxAttributes.setReadable(mBeanAttributeData.mBeanAttributeInfo.isReadable());
            jmxAttributes.setWritable(mBeanAttributeData.mBeanAttributeInfo.isWritable());
        }
        // System.out.println("list = " + list);
        return list;
    }

    public List<JMXOperation> getJMXOperations(final String oNameStr) {

        List<JMXOperation> jmxOperations = new ArrayList<JMXOperation>();

        ObjectName oName = getObjectName(oNameStr);

        MBeanInfo mBeanInfo = null;
        try {
            mBeanInfo = getMBeanServerConnection().getMBeanInfo(oName);
        } catch (InstanceNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IntrospectionException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (ReflectionException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (NullPointerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        MBeanOperationInfo[] mBeanOperationInfos = mBeanInfo.getOperations();

        for (MBeanOperationInfo mBeanOperationInfo : mBeanOperationInfos) {
            // Build a new JMX operation and add it to the list
            JMXOperation jmxOperation = new JMXOperation();
            jmxOperations.add(jmxOperation);

            // Get data
            jmxOperation.setName(mBeanOperationInfo.getName());
            jmxOperation.setDesc(mBeanOperationInfo.getDescription());
            jmxOperation.setReturnType(mBeanOperationInfo.getReturnType());

            MBeanParameterInfo[] mBeanParameterInfos = mBeanOperationInfo.getSignature();
            for (MBeanParameterInfo mBeanParameterInfo : mBeanParameterInfos) {
                JMXParameterOperation jmxParameterOperation = new JMXParameterOperation();
                jmxOperation.addParameter(jmxParameterOperation);
                jmxParameterOperation.setName(mBeanParameterInfo.getName());
                jmxParameterOperation.setDesc(mBeanParameterInfo.getDescription());
                jmxParameterOperation.setType(mBeanParameterInfo.getType());
            }

        }

        // Sort the operations !
        Collections.sort(jmxOperations);
        return jmxOperations;
    }

    public List<String> getProtocols() {
        // Get all MBeans of this MBean server
        Set<ObjectName> oNames = null;

        StringBuffer sb = new StringBuffer();
        sb.append(":j2eeType=JNDIResource");
        sb.append(",*");
        ObjectName oName = null;
        try {
            oName = new ObjectName(sb.toString());
        } catch (MalformedObjectNameException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (NullPointerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        try {
            oNames = getMBeanServerConnection().queryNames(oName, null);
        } catch (IOException e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        } catch (Exception e1) {
            // TODO Auto-generated catch block
            e1.printStackTrace();
        }

        // Get Name for each objectNames
        List<String> protocolNames = new ArrayList<String>();
        for (ObjectName objectName : oNames) {
            protocolNames.add(objectName.getKeyProperty("name"));
        }

        return protocolNames;

    }

    private MBeanServerConnection getMBeanServerConnection() throws Exception {
        MBeanServer mBeanServer = MBeanServer.class.cast(MBeanServerFactory.findMBeanServer(null).get(0));
        return mBeanServer;
    }

}
