/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.updater.vehicle_position;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.util.JsonFormat;
import com.google.transit.realtime.GtfsRealtime;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.opentripplanner.framework.geometry.WgsCoordinate;
import org.opentripplanner.framework.lang.StringUtils;
import org.opentripplanner.framework.time.ServiceDateUtils;
import org.opentripplanner.model.UpdateError;
import org.opentripplanner.model.UpdateSuccess;
import org.opentripplanner.service.vehiclepositions.VehiclePositionRepository;
import org.opentripplanner.service.vehiclepositions.model.RealtimeVehiclePosition;
import org.opentripplanner.service.vehiclepositions.model.RealtimeVehiclePositionBuilder;
import org.opentripplanner.transit.model.framework.FeedScopedId;
import org.opentripplanner.transit.model.framework.Result;
import org.opentripplanner.transit.model.network.TripPattern;
import org.opentripplanner.transit.model.site.StopLocation;
import org.opentripplanner.transit.model.timetable.Trip;
import org.opentripplanner.transit.model.timetable.TripTimes;
import org.opentripplanner.updater.spi.ResultLogger;
import org.opentripplanner.updater.spi.UpdateResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VehiclePositionPatternMatcher {
    private static final Logger LOG = LoggerFactory.getLogger(VehiclePositionPatternMatcher.class);
    private final String feedId;
    private final VehiclePositionRepository repository;
    private final ZoneId timeZoneId;
    private final Function<FeedScopedId, Trip> getTripForId;
    private final Function<Trip, TripPattern> getStaticPattern;
    private final BiFunction<Trip, LocalDate, TripPattern> getRealtimePattern;
    private Set<TripPattern> patternsInPreviousUpdate = Set.of();

    public VehiclePositionPatternMatcher(String feedId, Function<FeedScopedId, Trip> getTripForId, Function<Trip, TripPattern> getStaticPattern, BiFunction<Trip, LocalDate, TripPattern> getRealtimePattern, VehiclePositionRepository repository, ZoneId timeZoneId) {
        this.feedId = feedId;
        this.getTripForId = getTripForId;
        this.getStaticPattern = getStaticPattern;
        this.getRealtimePattern = getRealtimePattern;
        this.repository = repository;
        this.timeZoneId = timeZoneId;
    }

    public UpdateResult applyVehiclePositionUpdates(List<GtfsRealtime.VehiclePosition> vehiclePositions) {
        List<Result> matchResults = vehiclePositions.stream().map(vehiclePosition -> this.toRealtimeVehiclePosition(this.feedId, (GtfsRealtime.VehiclePosition)vehiclePosition)).toList();
        Map<TripPattern, List> positions = matchResults.stream().filter(Result::isSuccess).map(Result::successValue).collect(Collectors.groupingBy(PatternAndVehiclePosition::pattern)).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((List)e.getValue()).stream().map(PatternAndVehiclePosition::position).collect(Collectors.toList())));
        positions.forEach(this.repository::setVehiclePositions);
        Set<TripPattern> patternsInCurrentUpdate = positions.keySet();
        Sets.SetView toDelete = Sets.difference(this.patternsInPreviousUpdate, patternsInCurrentUpdate);
        toDelete.forEach(this.repository::clearVehiclePositions);
        this.patternsInPreviousUpdate = patternsInCurrentUpdate;
        if (!vehiclePositions.isEmpty() && patternsInCurrentUpdate.isEmpty()) {
            LOG.error("Could not match any vehicle positions for feedId '{}'. Are you sure that the updater is using the correct feedId?", (Object)this.feedId);
        }
        List<Result> results = matchResults.stream().map(e -> e.mapSuccess(ignored -> UpdateSuccess.noWarnings())).toList();
        UpdateResult updateResult = UpdateResult.ofResults(new ArrayList<Result<UpdateSuccess, UpdateError>>(results));
        ResultLogger.logUpdateResult(this.feedId, "gtfs-rt-vehicle-positions", updateResult);
        return updateResult;
    }

    private LocalDate inferServiceDate(Trip trip) {
        TripTimes staticTripTimes = this.getStaticPattern.apply(trip).getScheduledTimetable().getTripTimes(trip);
        return VehiclePositionPatternMatcher.inferServiceDate(staticTripTimes, this.timeZoneId, Instant.now());
    }

    protected static LocalDate inferServiceDate(TripTimes staticTripTimes, ZoneId zoneId, Instant now) {
        int start = staticTripTimes.getScheduledDepartureTime(0);
        int end = staticTripTimes.getScheduledDepartureTime(staticTripTimes.getNumStops() - 1);
        LocalDate today = now.atZone(zoneId).toLocalDate();
        LocalDate yesterday = today.minusDays(1L);
        LocalDate tomorrow = today.plusDays(1L);
        return Stream.of(yesterday, today, tomorrow).flatMap(day -> {
            Instant startTime = ServiceDateUtils.toZonedDateTime(day, zoneId, start).toInstant();
            Instant endTime = ServiceDateUtils.toZonedDateTime(day, zoneId, end).toInstant();
            return Stream.of(Duration.between(startTime, now), Duration.between(endTime, now)).map(Duration::abs).map(duration -> new TemporalDistance((LocalDate)day, duration.toSeconds()));
        }).min(Comparator.comparingLong(TemporalDistance::distance)).map(TemporalDistance::date).orElse(today);
    }

    private static RealtimeVehiclePosition mapVehiclePosition(GtfsRealtime.VehiclePosition vehiclePosition, List<StopLocation> stopsOnVehicleTrip, Trip trip) {
        RealtimeVehiclePositionBuilder newPosition = RealtimeVehiclePosition.builder();
        if (vehiclePosition.hasPosition()) {
            GtfsRealtime.Position position = vehiclePosition.getPosition();
            newPosition.setCoordinates(new WgsCoordinate(position.getLatitude(), position.getLongitude()));
            if (position.hasSpeed()) {
                newPosition.setSpeed(position.getSpeed());
            }
            if (position.hasBearing()) {
                newPosition.setHeading(position.getBearing());
            }
        }
        if (vehiclePosition.hasVehicle()) {
            GtfsRealtime.VehicleDescriptor vehicle = vehiclePosition.getVehicle();
            FeedScopedId id = new FeedScopedId(trip.getId().getFeedId(), vehicle.getId());
            newPosition.setVehicleId(id).setLabel(Optional.ofNullable(vehicle.getLabel()).orElse(vehicle.getLicensePlate()));
        }
        if (vehiclePosition.hasTimestamp()) {
            newPosition.setTime(Instant.ofEpochSecond(vehiclePosition.getTimestamp()));
        }
        if (vehiclePosition.hasCurrentStatus()) {
            newPosition.setStopStatus(VehiclePositionPatternMatcher.toModel(vehiclePosition.getCurrentStatus()));
        }
        if (vehiclePosition.hasStopId()) {
            List<StopLocation> matchedStops = stopsOnVehicleTrip.stream().filter(stop -> stop.getId().getId().equals(vehiclePosition.getStopId())).toList();
            if (matchedStops.size() == 1) {
                newPosition.setStop(matchedStops.get(0));
            } else {
                LOG.warn("Stop ID {} is not in trip {}. Not setting stopRelationship.", (Object)vehiclePosition.getStopId(), (Object)trip.getId());
            }
        } else if (vehiclePosition.hasCurrentStopSequence()) {
            StopLocation stop2 = stopsOnVehicleTrip.get(vehiclePosition.getCurrentStopSequence());
            newPosition.setStop(stop2);
        }
        newPosition.setTrip(trip);
        return newPosition.build();
    }

    private static RealtimeVehiclePosition.StopStatus toModel(GtfsRealtime.VehiclePosition.VehicleStopStatus currentStatus) {
        return switch (currentStatus) {
            default -> throw new IncompatibleClassChangeError();
            case GtfsRealtime.VehiclePosition.VehicleStopStatus.IN_TRANSIT_TO -> RealtimeVehiclePosition.StopStatus.IN_TRANSIT_TO;
            case GtfsRealtime.VehiclePosition.VehicleStopStatus.INCOMING_AT -> RealtimeVehiclePosition.StopStatus.INCOMING_AT;
            case GtfsRealtime.VehiclePosition.VehicleStopStatus.STOPPED_AT -> RealtimeVehiclePosition.StopStatus.STOPPED_AT;
        };
    }

    private static String toString(GtfsRealtime.VehiclePosition vehiclePosition) {
        try {
            return JsonFormat.printer().omittingInsignificantWhitespace().print((MessageOrBuilder)vehiclePosition);
        }
        catch (InvalidProtocolBufferException ignored) {
            return vehiclePosition.toString();
        }
    }

    private Result<PatternAndVehiclePosition, UpdateError> toRealtimeVehiclePosition(String feedId, GtfsRealtime.VehiclePosition vehiclePosition) {
        if (!vehiclePosition.hasTrip()) {
            LOG.debug("Realtime vehicle positions {} has no trip ID. Ignoring.", (Object)VehiclePositionPatternMatcher.toString(vehiclePosition));
            return Result.failure(UpdateError.noTripId(UpdateError.UpdateErrorType.INVALID_INPUT_STRUCTURE));
        }
        String tripId = vehiclePosition.getTrip().getTripId();
        if (!StringUtils.hasValue(tripId)) {
            return Result.failure(UpdateError.noTripId(UpdateError.UpdateErrorType.NO_TRIP_ID));
        }
        FeedScopedId scopedTripId = new FeedScopedId(feedId, tripId);
        Trip trip = this.getTripForId.apply(scopedTripId);
        if (trip == null) {
            LOG.debug("Unable to find trip ID in feed '{}' for vehicle position with trip ID {}", (Object)feedId, (Object)tripId);
            return UpdateError.result(scopedTripId, UpdateError.UpdateErrorType.TRIP_NOT_FOUND);
        }
        LocalDate serviceDate = Optional.of(vehiclePosition.getTrip().getStartDate()).map(Strings::emptyToNull).flatMap(ServiceDateUtils::parseStringToOptional).orElseGet(() -> this.inferServiceDate(trip));
        TripPattern pattern = this.getRealtimePattern.apply(trip, serviceDate);
        if (pattern == null) {
            LOG.debug("Unable to match OTP pattern ID for vehicle position with trip ID {}", (Object)tripId);
            return UpdateError.result(scopedTripId, UpdateError.UpdateErrorType.NO_SERVICE_ON_DATE);
        }
        RealtimeVehiclePosition newPosition = VehiclePositionPatternMatcher.mapVehiclePosition(vehiclePosition, pattern.getStops(), trip);
        return Result.success(new PatternAndVehiclePosition(pattern, newPosition));
    }

    record PatternAndVehiclePosition(TripPattern pattern, RealtimeVehiclePosition position) {
    }

    private record TemporalDistance(LocalDate date, long distance) {
    }
}

