/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.graph_builder.module.ned;

import java.util.HashMap;
import java.util.Map;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.impl.PackedCoordinateSequence;
import org.opentripplanner.astar.model.BinHeap;
import org.opentripplanner.graph_builder.issue.api.DataImportIssueStore;
import org.opentripplanner.graph_builder.issues.ElevationFlattened;
import org.opentripplanner.graph_builder.issues.ElevationProfileFailure;
import org.opentripplanner.street.model.edge.Edge;
import org.opentripplanner.street.model.edge.StreetEdge;
import org.opentripplanner.street.model.edge.StreetElevationExtension;
import org.opentripplanner.street.model.vertex.Vertex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MissingElevationHandler {
    private static final Logger LOG = LoggerFactory.getLogger(MissingElevationHandler.class);
    private final DataImportIssueStore issueStore;
    private final Map<Vertex, Double> existingElevationForVertices;
    private final double maxElevationPropagationMeters;

    public MissingElevationHandler(DataImportIssueStore issueStore, Map<Vertex, Double> existingElevationsForVertices, double maxElevationPropagationMeters) {
        this.issueStore = issueStore;
        this.existingElevationForVertices = existingElevationsForVertices;
        this.maxElevationPropagationMeters = maxElevationPropagationMeters;
    }

    void run() {
        LOG.debug("Assigning missing elevations");
        BinHeap<ElevationRepairState> pq = this.createPriorityQueue(this.existingElevationForVertices);
        HashMap<Vertex, Double> elevations = new HashMap<Vertex, Double>(this.existingElevationForVertices);
        this.propagateElevationToNearbyVertices(pq, elevations);
        elevations.keySet().forEach(vertex -> {
            vertex.getIncomingStreetEdges().forEach(edge -> this.assignElevationToEdgeIfPossible(elevations, (StreetEdge)edge));
            vertex.getOutgoingStreetEdges().forEach(edge -> this.assignElevationToEdgeIfPossible(elevations, (StreetEdge)edge));
        });
    }

    private BinHeap<ElevationRepairState> createPriorityQueue(Map<Vertex, Double> elevations) {
        BinHeap<ElevationRepairState> pq = new BinHeap<ElevationRepairState>();
        elevations.forEach((vertex, elevation) -> {
            vertex.getIncoming().forEach(edge -> {
                if (edge.getDistanceMeters() < this.maxElevationPropagationMeters) {
                    pq.insert(new ElevationRepairState((Vertex)vertex, (Double)elevation, edge.getFromVertex(), edge.getDistanceMeters()), edge.getDistanceMeters());
                }
            });
            vertex.getOutgoing().forEach(edge -> {
                if (edge.getDistanceMeters() < this.maxElevationPropagationMeters) {
                    pq.insert(new ElevationRepairState((Vertex)vertex, (Double)elevation, edge.getToVertex(), edge.getDistanceMeters()), edge.getDistanceMeters());
                }
            });
        });
        return pq;
    }

    private void propagateElevationToNearbyVertices(BinHeap<ElevationRepairState> pq, Map<Vertex, Double> elevations) {
        HashMap<Vertex, ElevationRepairState> pending = new HashMap<Vertex, ElevationRepairState>();
        while (!pq.empty()) {
            double nsDistance;
            Vertex nsVertex;
            ElevationRepairState currentState = pq.extract_min();
            if (elevations.containsKey(currentState.currentVertex)) continue;
            if (pending.containsKey(currentState.currentVertex)) {
                ElevationRepairState otherState = (ElevationRepairState)pending.get(currentState.currentVertex);
                if (otherState.initialVertex == currentState.currentVertex) continue;
                this.interpolateElevationsAlongBackPath(otherState, currentState, elevations, pending);
                this.interpolateElevationsAlongBackPath(currentState, otherState, elevations, pending);
            } else {
                pending.put(currentState.currentVertex, currentState);
            }
            for (Edge e : currentState.currentVertex.getIncoming()) {
                nsVertex = e.getFromVertex();
                nsDistance = currentState.distance + e.getDistanceMeters();
                if (elevations.containsKey(nsVertex) || nsDistance > this.maxElevationPropagationMeters) continue;
                pq.insert(new ElevationRepairState(currentState, nsVertex, nsDistance), nsDistance);
            }
            for (Edge e : currentState.currentVertex.getOutgoing()) {
                nsVertex = e.getToVertex();
                nsDistance = currentState.distance + e.getDistanceMeters();
                if (elevations.containsKey(nsVertex) || nsDistance > this.maxElevationPropagationMeters) continue;
                pq.insert(new ElevationRepairState(currentState, nsVertex, nsDistance), nsDistance);
            }
        }
        pending.forEach((vertex, elevationRepairState) -> {
            if (!elevations.containsKey(vertex)) {
                elevations.put((Vertex)vertex, this.lastKnowElevationForState((ElevationRepairState)elevationRepairState, elevations));
            }
        });
    }

    private Double lastKnowElevationForState(ElevationRepairState elevationRepairState, Map<Vertex, Double> elevations) {
        ElevationRepairState backState = elevationRepairState.previousState;
        while (backState != null) {
            if (elevations.containsKey(backState.currentVertex)) {
                return elevations.get(backState.currentVertex);
            }
            backState = backState.previousState;
        }
        return elevationRepairState.initialElevation;
    }

    private void interpolateElevationsAlongBackPath(ElevationRepairState stateToBackTrack, ElevationRepairState alternateState, Map<Vertex, Double> elevations, Map<Vertex, ElevationRepairState> pending) {
        double elevationDiff = alternateState.initialElevation - stateToBackTrack.initialElevation;
        double totalDistance = stateToBackTrack.distance + alternateState.distance;
        ElevationRepairState currentState = stateToBackTrack;
        while (currentState != null) {
            if (!elevations.containsKey(currentState.currentVertex)) {
                double elevation = currentState.initialElevation + elevationDiff * (currentState.distance / totalDistance);
                elevation = (double)Math.round(elevation * 10.0) / 10.0;
                elevations.put(currentState.currentVertex, elevation);
                pending.remove(currentState.currentVertex);
            }
            currentState = currentState.previousState;
        }
    }

    private void assignElevationToEdgeIfPossible(Map<Vertex, Double> elevations, StreetEdge edge) {
        if (edge.getElevationProfile() != null) {
            return;
        }
        Double fromElevation = elevations.get(edge.getFromVertex());
        Double toElevation = elevations.get(edge.getToVertex());
        if (fromElevation == null || toElevation == null) {
            if (!edge.isElevationFlattened() && !edge.isSlopeOverride()) {
                this.issueStore.add(new ElevationProfileFailure(edge, "Failed to propagate elevation data"));
            }
            return;
        }
        Coordinate[] coords = new Coordinate[]{new Coordinate(0.0, fromElevation.doubleValue()), new Coordinate(edge.getDistanceMeters(), toElevation.doubleValue())};
        PackedCoordinateSequence.Double profile = new PackedCoordinateSequence.Double(coords);
        try {
            StreetElevationExtension.addToEdge(edge, (PackedCoordinateSequence)profile, true);
            if (edge.isElevationFlattened()) {
                this.issueStore.add(new ElevationFlattened(edge));
            }
        }
        catch (Exception ex) {
            this.issueStore.add(new ElevationProfileFailure(edge, ex.getMessage()));
        }
    }

    private static class ElevationRepairState {
        final ElevationRepairState previousState;
        final Vertex initialVertex;
        final Double initialElevation;
        final Vertex currentVertex;
        final Double distance;

        ElevationRepairState(ElevationRepairState previousState, Vertex currentVertex, Double distance) {
            this.previousState = previousState;
            this.initialVertex = previousState.initialVertex;
            this.initialElevation = previousState.initialElevation;
            this.currentVertex = currentVertex;
            this.distance = distance;
        }

        ElevationRepairState(Vertex initialVertex, Double initialElevation, Vertex currentVertex, Double distance) {
            this.previousState = null;
            this.initialVertex = initialVertex;
            this.initialElevation = initialElevation;
            this.currentVertex = currentVertex;
            this.distance = distance;
        }

        public String toString() {
            return "ElevationRepairState{initialVertex=" + this.initialVertex + ", initialElevation=" + this.initialElevation + ", currentVertex=" + this.currentVertex + ", distance=" + this.distance + "}";
        }
    }
}

