/*
 * Decompiled with CFR 0.152.
 */
package org.opentripplanner.util.geometry;

import java.util.Arrays;
import java.util.List;
import org.geojson.GeoJsonObject;
import org.geojson.LngLatAlt;
import org.geojson.MultiLineString;
import org.geojson.MultiPolygon;
import org.geojson.Point;
import org.geotools.referencing.CRS;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.CoordinateSequenceFactory;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.impl.PackedCoordinateSequenceFactory;
import org.locationtech.jts.linearref.LengthLocationMap;
import org.locationtech.jts.linearref.LinearLocation;
import org.locationtech.jts.linearref.LocationIndexedLine;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opentripplanner.common.geometry.UnsupportedGeometryException;
import org.opentripplanner.common.model.P2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GeometryUtils {
    private static final Logger LOG = LoggerFactory.getLogger(GeometryUtils.class);
    private static final CoordinateSequenceFactory csf = new PackedCoordinateSequenceFactory();
    private static final GeometryFactory gf = new GeometryFactory(csf);
    public static final CoordinateReferenceSystem WGS84_XY;

    public static LineString makeLineString(double ... coords) {
        GeometryFactory factory = GeometryUtils.getGeometryFactory();
        Coordinate[] coordinates = new Coordinate[coords.length / 2];
        for (int i = 0; i < coords.length; i += 2) {
            coordinates[i / 2] = new Coordinate(coords[i], coords[i + 1]);
        }
        return factory.createLineString(coordinates);
    }

    public static LineString makeLineString(List<Coordinate> coordinates) {
        GeometryFactory factory = GeometryUtils.getGeometryFactory();
        return factory.createLineString(coordinates.toArray(new Coordinate[0]));
    }

    public static LineString makeLineString(Coordinate[] coordinates) {
        GeometryFactory factory = GeometryUtils.getGeometryFactory();
        return factory.createLineString(coordinates);
    }

    public static LineString concatenateLineStrings(List<LineString> lineStrings) {
        GeometryFactory factory = GeometryUtils.getGeometryFactory();
        return factory.createLineString((Coordinate[])lineStrings.stream().flatMap(t -> Arrays.stream(t.getCoordinates())).toArray(Coordinate[]::new));
    }

    public static LineString addStartEndCoordinatesToLineString(Coordinate startCoord, LineString lineString, Coordinate endCoord) {
        Coordinate[] coordinates = new Coordinate[lineString.getCoordinates().length + 2];
        coordinates[0] = startCoord;
        for (int j = 0; j < lineString.getCoordinates().length; ++j) {
            coordinates[j + 1] = lineString.getCoordinates()[j];
        }
        coordinates[lineString.getCoordinates().length + 1] = endCoord;
        return GeometryUtils.makeLineString(coordinates);
    }

    public static LineString removeStartEndCoordinatesFromLineString(LineString lineString) {
        Coordinate[] coordinates = new Coordinate[lineString.getCoordinates().length - 2];
        for (int j = 1; j < lineString.getCoordinates().length - 1; ++j) {
            coordinates[j - 1] = lineString.getCoordinates()[j];
        }
        return GeometryUtils.makeLineString(coordinates);
    }

    public static GeometryFactory getGeometryFactory() {
        return gf;
    }

    public static P2<LineString> splitGeometryAtPoint(Geometry geometry, Coordinate nearestPoint) {
        LocationIndexedLine line = new LocationIndexedLine(geometry);
        LinearLocation l = line.indexOf(nearestPoint);
        LineString beginning = (LineString)line.extractLine(line.getStartIndex(), l);
        LineString ending = (LineString)line.extractLine(l, line.getEndIndex());
        return new P2<LineString>(beginning, ending);
    }

    public static P2<LineString> splitGeometryAtFraction(Geometry geometry, double fraction) {
        LineString empty = new LineString(null, gf);
        Coordinate[] coordinates = geometry.getCoordinates();
        CoordinateSequence sequence = gf.getCoordinateSequenceFactory().create(coordinates);
        LineString total = new LineString(sequence, gf);
        if (coordinates.length < 2) {
            return new P2<LineString>(empty, empty);
        }
        if (fraction <= 0.0) {
            return new P2<LineString>(empty, total);
        }
        if (fraction >= 1.0) {
            return new P2<LineString>(total, empty);
        }
        double totalDistance = total.getLength();
        double requestedDistance = totalDistance * fraction;
        LocationIndexedLine line = new LocationIndexedLine(geometry);
        LinearLocation l = LengthLocationMap.getLocation((Geometry)geometry, (double)requestedDistance);
        LineString beginning = (LineString)line.extractLine(line.getStartIndex(), l);
        LineString ending = (LineString)line.extractLine(l, line.getEndIndex());
        return new P2<LineString>(beginning, ending);
    }

    public static LineString getInteriorSegment(Geometry geomerty, Coordinate first, Coordinate second) {
        P2<LineString> splitGeom = GeometryUtils.splitGeometryAtPoint(geomerty, first);
        splitGeom = GeometryUtils.splitGeometryAtPoint((Geometry)splitGeom.second, second);
        return (LineString)splitGeom.first;
    }

    public static double segmentFraction(double x0, double y0, double x1, double y1, double xp, double yp, double xscale) {
        double dx = (x1 - x0) * xscale;
        double dy = y1 - y0;
        double len2 = dx * dx + dy * dy;
        if (len2 == 0.0) {
            return 0.0;
        }
        double r = ((xp - x0) * xscale * dx + (yp - y0) * dy) / len2;
        if (r < 0.0) {
            return 0.0;
        }
        if (r > 1.0) {
            return 1.0;
        }
        return r;
    }

    public static Geometry convertGeoJsonToJtsGeometry(GeoJsonObject geoJsonGeom) throws UnsupportedGeometryException {
        if (geoJsonGeom instanceof Point) {
            Point geoJsonPoint = (Point)geoJsonGeom;
            return gf.createPoint(new Coordinate(geoJsonPoint.getCoordinates().getLongitude(), geoJsonPoint.getCoordinates().getLatitude(), geoJsonPoint.getCoordinates().getAltitude()));
        }
        if (geoJsonGeom instanceof org.geojson.Polygon) {
            org.geojson.Polygon geoJsonPolygon = (org.geojson.Polygon)geoJsonGeom;
            LinearRing shell = gf.createLinearRing(GeometryUtils.convertPath(geoJsonPolygon.getExteriorRing()));
            LinearRing[] holes = new LinearRing[geoJsonPolygon.getInteriorRings().size()];
            int i = 0;
            for (List hole : geoJsonPolygon.getInteriorRings()) {
                holes[i++] = gf.createLinearRing(GeometryUtils.convertPath(hole));
            }
            return gf.createPolygon(shell, holes);
        }
        if (geoJsonGeom instanceof MultiPolygon) {
            MultiPolygon geoJsonMultiPolygon = (MultiPolygon)geoJsonGeom;
            Polygon[] jtsPolygons = new Polygon[geoJsonMultiPolygon.getCoordinates().size()];
            int i = 0;
            for (List geoJsonRings : geoJsonMultiPolygon.getCoordinates()) {
                org.geojson.Polygon geoJsonPoly = new org.geojson.Polygon();
                for (List geoJsonRing : geoJsonRings) {
                    geoJsonPoly.add((Object)geoJsonRing);
                }
                jtsPolygons[i++] = (Polygon)GeometryUtils.convertGeoJsonToJtsGeometry((GeoJsonObject)geoJsonPoly);
            }
            return gf.createMultiPolygon(jtsPolygons);
        }
        if (geoJsonGeom instanceof org.geojson.LineString) {
            org.geojson.LineString geoJsonLineString = (org.geojson.LineString)geoJsonGeom;
            return gf.createLineString(GeometryUtils.convertPath(geoJsonLineString.getCoordinates()));
        }
        if (geoJsonGeom instanceof MultiLineString) {
            MultiLineString geoJsonMultiLineString = (MultiLineString)geoJsonGeom;
            LineString[] jtsLineStrings = new LineString[geoJsonMultiLineString.getCoordinates().size()];
            int i = 0;
            for (List geoJsonPath : geoJsonMultiLineString.getCoordinates()) {
                org.geojson.LineString geoJsonLineString = new org.geojson.LineString(geoJsonPath.toArray(new LngLatAlt[geoJsonPath.size()]));
                jtsLineStrings[i++] = (LineString)GeometryUtils.convertGeoJsonToJtsGeometry((GeoJsonObject)geoJsonLineString);
            }
            return gf.createMultiLineString(jtsLineStrings);
        }
        throw new UnsupportedGeometryException(geoJsonGeom.getClass().toString());
    }

    private static Coordinate[] convertPath(List<LngLatAlt> path) {
        Coordinate[] coords = new Coordinate[path.size()];
        int i = 0;
        for (LngLatAlt p : path) {
            coords[i++] = new Coordinate(p.getLongitude(), p.getLatitude(), p.getAltitude());
        }
        return coords;
    }

    static {
        try {
            WGS84_XY = CRS.getAuthorityFactory((boolean)true).createCoordinateReferenceSystem("EPSG:4326");
        }
        catch (Exception ex) {
            LOG.error("Unable to create longitude-first WGS84 CRS", (Throwable)ex);
            throw new RuntimeException("Could not create longitude-first WGS84 coordinate reference system.");
        }
    }
}

