/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.processor.date;

import io.micrometer.core.instrument.Counter;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.opensearch.dataprepper.expression.ExpressionEvaluator;
import org.opensearch.dataprepper.logging.DataPrepperMarkers;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.annotations.DataPrepperPlugin;
import org.opensearch.dataprepper.model.annotations.DataPrepperPluginConstructor;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.plugin.InvalidPluginConfigurationException;
import org.opensearch.dataprepper.model.processor.AbstractProcessor;
import org.opensearch.dataprepper.model.processor.Processor;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.plugins.processor.date.DateProcessorConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@DataPrepperPlugin(name="date", pluginType=Processor.class, pluginConfigurationType=DateProcessorConfig.class)
public class DateProcessor
extends AbstractProcessor<Record<Event>, Record<Event>> {
    private static final Logger LOG = LoggerFactory.getLogger(DateProcessor.class);
    private static final String OUTPUT_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSSXXX";
    private static final int LENGTH_OF_EPOCH_IN_MILLIS = 13;
    private static final int LENGTH_OF_EPOCH_SECONDS = 10;
    private static final int LENGTH_OF_EPOCH_MICROSECONDS = 16;
    static final String DATE_PROCESSING_MATCH_SUCCESS = "dateProcessingMatchSuccess";
    static final String DATE_PROCESSING_MATCH_FAILURE = "dateProcessingMatchFailure";
    private String keyToParse;
    private List<DateTimeFormatter> dateTimeFormatters;
    private Set<String> epochFormatters;
    private String outputFormat;
    private final DateProcessorConfig dateProcessorConfig;
    private final ExpressionEvaluator expressionEvaluator;
    private final Counter dateProcessingMatchSuccessCounter;
    private final Counter dateProcessingMatchFailureCounter;

    @DataPrepperPluginConstructor
    public DateProcessor(PluginMetrics pluginMetrics, DateProcessorConfig dateProcessorConfig, ExpressionEvaluator expressionEvaluator) {
        super(pluginMetrics);
        this.dateProcessorConfig = dateProcessorConfig;
        this.expressionEvaluator = expressionEvaluator;
        this.outputFormat = dateProcessorConfig.getOutputFormat();
        this.dateProcessingMatchSuccessCounter = pluginMetrics.counter(DATE_PROCESSING_MATCH_SUCCESS);
        this.dateProcessingMatchFailureCounter = pluginMetrics.counter(DATE_PROCESSING_MATCH_FAILURE);
        if (dateProcessorConfig.getMatch() != null) {
            this.extractKeyAndFormatters();
        }
        if (dateProcessorConfig.getDateWhen() != null && !expressionEvaluator.isValidExpressionStatement(dateProcessorConfig.getDateWhen()).booleanValue()) {
            throw new InvalidPluginConfigurationException(String.format("date_when \"%s\" is not a valid expression statement. See https://opensearch.org/docs/latest/data-prepper/pipelines/expression-syntax/ for valid expression syntax", dateProcessorConfig.getDateWhen()));
        }
    }

    public Collection<Record<Event>> doExecute(Collection<Record<Event>> records) {
        for (Record<Event> record : records) {
            try {
                if (Objects.nonNull(this.dateProcessorConfig.getDateWhen()) && !this.expressionEvaluator.evaluateConditional(this.dateProcessorConfig.getDateWhen(), (Event)record.getData()).booleanValue()) continue;
                String zonedDateTime = null;
                if (Boolean.TRUE.equals(this.dateProcessorConfig.getFromTimeReceived())) {
                    zonedDateTime = this.getDateTimeFromTimeReceived(record);
                } else if (this.keyToParse != null && !this.keyToParse.isEmpty()) {
                    Pair<String, Instant> result = this.getDateTimeFromMatch(record);
                    if (result != null) {
                        zonedDateTime = (String)result.getLeft();
                        Instant timeStamp = (Instant)result.getRight();
                        if (this.dateProcessorConfig.getToOriginationMetadata().booleanValue()) {
                            Event event = (Event)record.getData();
                            event.getMetadata().setExternalOriginationTime(timeStamp);
                            event.getEventHandle().setExternalOriginationTime(timeStamp);
                        }
                    }
                    this.populateDateProcessorMetrics(zonedDateTime);
                }
                if (zonedDateTime == null) continue;
                ((Event)record.getData()).put(this.dateProcessorConfig.getDestination(), (Object)zonedDateTime);
            }
            catch (Exception e) {
                LOG.error(DataPrepperMarkers.NOISY, "An exception occurred while attempting to process Event: ", (Throwable)e);
            }
        }
        return records;
    }

    private void populateDateProcessorMetrics(String zonedDateTime) {
        if (zonedDateTime != null) {
            this.dateProcessingMatchSuccessCounter.increment();
        } else {
            this.dateProcessingMatchFailureCounter.increment();
        }
    }

    private void extractKeyAndFormatters() {
        for (DateProcessorConfig.DateMatch entry : this.dateProcessorConfig.getMatch()) {
            this.keyToParse = entry.getKey();
            this.epochFormatters = entry.getPatterns().stream().filter(pattern -> pattern.contains("epoch")).collect(Collectors.toSet());
            this.dateTimeFormatters = entry.getPatterns().stream().filter(pattern -> !pattern.contains("epoch")).map(this::getSourceFormatter).collect(Collectors.toList());
        }
    }

    private DateTimeFormatter getSourceFormatter(String pattern) {
        LocalDate localDateForDefaultValues = LocalDate.now(this.dateProcessorConfig.getSourceZoneId());
        DateTimeFormatterBuilder dateTimeFormatterBuilder = new DateTimeFormatterBuilder().appendPattern(pattern).parseDefaulting(ChronoField.MONTH_OF_YEAR, localDateForDefaultValues.getMonthValue()).parseDefaulting(ChronoField.DAY_OF_MONTH, localDateForDefaultValues.getDayOfMonth()).parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0L).parseDefaulting(ChronoField.SECOND_OF_MINUTE, 0L);
        if (!pattern.contains("a") && !pattern.contains("k")) {
            dateTimeFormatterBuilder.parseDefaulting(ChronoField.HOUR_OF_DAY, 0L);
        }
        if (!pattern.contains("y") && !pattern.contains("u")) {
            dateTimeFormatterBuilder.parseDefaulting(ChronoField.YEAR_OF_ERA, localDateForDefaultValues.getYear());
        }
        return dateTimeFormatterBuilder.toFormatter(this.dateProcessorConfig.getSourceLocale()).withZone(this.dateProcessorConfig.getSourceZoneId());
    }

    private String getDateTimeFromTimeReceived(Record<Event> record) {
        Instant timeReceived = ((Event)record.getData()).getMetadata().getTimeReceived();
        return timeReceived.atZone(this.dateProcessorConfig.getDestinationZoneId()).format(this.getOutputFormatter());
    }

    private Pair<String, Instant> getDateTimeFromMatch(Record<Event> record) {
        String sourceTimestamp = this.getSourceTimestamp(record);
        if (sourceTimestamp == null) {
            return null;
        }
        return this.getFormattedDateTimeString(sourceTimestamp);
    }

    private String getSourceTimestamp(Record<Event> record) {
        try {
            return (String)((Event)record.getData()).get(this.keyToParse, String.class);
        }
        catch (Exception e) {
            LOG.debug("Unable to find {} in event data.", (Object)this.keyToParse);
            return null;
        }
    }

    private Pair<String, Instant> getEpochFormatOutput(Instant time) {
        if (this.outputFormat.equals("epoch_second")) {
            return Pair.of((Object)Long.toString(time.getEpochSecond()), (Object)time);
        }
        if (this.outputFormat.equals("epoch_milli")) {
            return Pair.of((Object)Long.toString(time.toEpochMilli()), (Object)time);
        }
        if (this.outputFormat.equals("epoch_micro")) {
            long micro = time.getEpochSecond() * 1000000L + (long)time.getNano() / 1000L;
            return Pair.of((Object)Long.toString(micro), (Object)time);
        }
        long nano = time.getEpochSecond() * 1000000000L + (long)time.getNano();
        return Pair.of((Object)Long.toString(nano), (Object)time);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Pair<String, Instant> getFormattedDateTimeString(String sourceTimestamp) {
        ZoneId srcZoneId = this.dateProcessorConfig.getSourceZoneId();
        ZoneId dstZoneId = this.dateProcessorConfig.getDestinationZoneId();
        Long numberValue = null;
        if (this.epochFormatters.size() > 0) {
            try {
                numberValue = Long.parseLong(sourceTimestamp);
            }
            catch (NumberFormatException e) {
                numberValue = null;
            }
        }
        if (numberValue != null) {
            Instant epochTime;
            int timestampLength = sourceTimestamp.length();
            if (timestampLength > 16) {
                if (!this.epochFormatters.contains("epoch_nano")) {
                    LOG.warn("Source time value is larger than epoch pattern configured. epoch_nano is expected but not present in the patterns list");
                    return null;
                }
                epochTime = Instant.ofEpochSecond(numberValue / 1000000000L, numberValue % 1000000000L);
            } else if (timestampLength > 13) {
                if (!this.epochFormatters.contains("epoch_micro")) {
                    LOG.warn("Source time value is larger than epoch pattern configured. epoch_micro is expected but not present in the patterns list");
                    return null;
                }
                epochTime = Instant.ofEpochSecond(numberValue / 1000000L, numberValue % 1000000L * 1000L);
            } else if (timestampLength > 10) {
                if (!this.epochFormatters.contains("epoch_milli")) {
                    LOG.warn("Source time value is larger than epoch pattern configured. epoch_milli is expected but not present in the patterns list");
                    return null;
                }
                epochTime = Instant.ofEpochMilli(numberValue);
            } else {
                epochTime = Instant.ofEpochSecond(numberValue);
            }
            srcZoneId = ZoneId.of("UTC");
            try {
                if (this.outputFormat.startsWith("epoch_")) {
                    return this.getEpochFormatOutput(epochTime);
                }
                DateTimeFormatter outputFormatter = this.getOutputFormatter().withZone(dstZoneId);
                ZonedDateTime tmp = ZonedDateTime.ofInstant(epochTime, srcZoneId);
                return Pair.of((Object)tmp.format(outputFormatter.withZone(dstZoneId)), (Object)tmp.toInstant());
            }
            catch (Exception outputFormatter) {
                // empty catch block
            }
        }
        Iterator<DateTimeFormatter> iterator = this.dateTimeFormatters.iterator();
        while (true) {
            if (!iterator.hasNext()) {
                LOG.debug("Unable to parse {} with any of the provided patterns", (Object)sourceTimestamp);
                return null;
            }
            DateTimeFormatter formatter = iterator.next();
            try {
                ZonedDateTime tmp = ZonedDateTime.parse(sourceTimestamp, formatter);
                if (this.outputFormat.startsWith("epoch_")) {
                    return this.getEpochFormatOutput(tmp.toInstant());
                }
                return Pair.of((Object)tmp.format(this.getOutputFormatter().withZone(dstZoneId)), (Object)tmp.toInstant());
            }
            catch (Exception exception) {
                continue;
            }
            break;
        }
    }

    private DateTimeFormatter getOutputFormatter() {
        return DateTimeFormatter.ofPattern(this.outputFormat);
    }

    public void prepareForShutdown() {
    }

    public boolean isReadyForShutdown() {
        return true;
    }

    public void shutdown() {
    }
}

