/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.confluence;

import io.micrometer.core.instrument.Counter;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Queue;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import javax.inject.Named;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.plugin.InvalidPluginConfigurationException;
import org.opensearch.dataprepper.plugins.source.confluence.ConfluenceItemInfo;
import org.opensearch.dataprepper.plugins.source.confluence.ConfluenceSourceConfig;
import org.opensearch.dataprepper.plugins.source.confluence.configuration.NameConfig;
import org.opensearch.dataprepper.plugins.source.confluence.models.ConfluenceItem;
import org.opensearch.dataprepper.plugins.source.confluence.models.ConfluencePaginationLinks;
import org.opensearch.dataprepper.plugins.source.confluence.models.ConfluenceSearchResults;
import org.opensearch.dataprepper.plugins.source.confluence.models.ConfluenceServerMetadata;
import org.opensearch.dataprepper.plugins.source.confluence.rest.ConfluenceRestClient;
import org.opensearch.dataprepper.plugins.source.confluence.utils.ConfluenceConfigHelper;
import org.opensearch.dataprepper.plugins.source.confluence.utils.ConfluenceContentType;
import org.opensearch.dataprepper.plugins.source.source_crawler.model.ItemInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

@Named
public class ConfluenceService {
    private static final Logger log = LoggerFactory.getLogger(ConfluenceService.class);
    public static final String CONTENT_TYPE = "ContentType";
    public static final String CQL_LAST_MODIFIED_DATE_FORMAT = "yyyy-MM-dd HH:mm";
    private static final String SEARCH_RESULTS_FOUND = "searchResultsFound";
    private ZoneId confluenceServerZoneId = null;
    private final ConfluenceSourceConfig confluenceSourceConfig;
    private final ConfluenceRestClient confluenceRestClient;
    private final Counter searchResultsFoundCounter;

    public ConfluenceService(ConfluenceSourceConfig confluenceSourceConfig, ConfluenceRestClient confluenceRestClient, PluginMetrics pluginMetrics) {
        this.confluenceSourceConfig = confluenceSourceConfig;
        this.confluenceRestClient = confluenceRestClient;
        this.searchResultsFoundCounter = pluginMetrics.counter(SEARCH_RESULTS_FOUND);
    }

    public void getPages(ConfluenceSourceConfig configuration, Instant timestamp, Queue<ItemInfo> itemInfoQueue) {
        log.trace("Started to fetch entities");
        this.searchForNewContentAndAddToQueue(configuration, timestamp, itemInfoQueue);
        log.trace("Creating item information and adding in queue");
    }

    public String getContent(String contentId) {
        return this.confluenceRestClient.getContent(contentId);
    }

    private void initializeConfluenceServerMetadata() {
        ConfluenceServerMetadata confluenceServerMetadata = this.confluenceRestClient.getConfluenceServerMetadata();
        this.confluenceServerZoneId = confluenceServerMetadata.getDefaultTimeZone();
    }

    private void searchForNewContentAndAddToQueue(ConfluenceSourceConfig configuration, Instant timestamp, Queue<ItemInfo> itemInfoQueue) {
        log.trace("Looking for Add/Modified tickets with a Search API call");
        StringBuilder cql = this.createContentFilterCriteria(configuration, timestamp);
        int total = 0;
        int startAt = 0;
        ConfluencePaginationLinks paginationLinks = null;
        do {
            ConfluenceSearchResults searchContentItems = this.confluenceRestClient.getAllContent(cql, startAt, paginationLinks);
            ArrayList<ConfluenceItem> contentList = new ArrayList<ConfluenceItem>(searchContentItems.getResults());
            startAt += searchContentItems.getResults().size();
            this.addItemsToQueue(contentList, itemInfoQueue);
            log.debug("Content items fetched so far: {}", (Object)(total += searchContentItems.getSize().intValue()));
            paginationLinks = searchContentItems.getLinks();
            this.searchResultsFoundCounter.increment((double)searchContentItems.getSize().intValue());
        } while (paginationLinks != null && paginationLinks.getNext() != null);
        log.info("Number of content items found in search api call: {}", (Object)total);
    }

    private void addItemsToQueue(List<ConfluenceItem> contentList, Queue<ItemInfo> itemInfoQueue) {
        contentList.forEach(contentItem -> itemInfoQueue.add(ConfluenceItemInfo.builder().withEventTime(Instant.now()).withContentBean((ConfluenceItem)contentItem).build()));
    }

    StringBuilder createContentFilterCriteria(ConfluenceSourceConfig configuration, Instant ts) {
        log.info("Creating content filter criteria");
        if (!CollectionUtils.isEmpty(ConfluenceConfigHelper.getSpacesNameIncludeFilter(configuration)) || !CollectionUtils.isEmpty(ConfluenceConfigHelper.getSpacesNameExcludeFilter(configuration))) {
            this.validateSpaceFilters(configuration);
        }
        if (!CollectionUtils.isEmpty(ConfluenceConfigHelper.getContentTypeIncludeFilter(configuration)) || !CollectionUtils.isEmpty(ConfluenceConfigHelper.getContentTypeExcludeFilter(configuration))) {
            this.validatePageTypeFilters(configuration);
        }
        if (this.confluenceServerZoneId == null) {
            this.initializeConfluenceServerMetadata();
        }
        String formattedTimeStamp = ts.atZone(this.confluenceServerZoneId).format(DateTimeFormatter.ofPattern(CQL_LAST_MODIFIED_DATE_FORMAT));
        StringBuilder cQl = new StringBuilder("lastModified>=\"" + formattedTimeStamp + "\"");
        if (!CollectionUtils.isEmpty(ConfluenceConfigHelper.getSpacesNameIncludeFilter(configuration))) {
            cQl.append(" AND space in (").append(ConfluenceConfigHelper.getSpacesNameIncludeFilter(configuration).stream().collect(Collectors.joining("\",\"", "\"", "\""))).append(")");
        }
        if (!CollectionUtils.isEmpty(ConfluenceConfigHelper.getSpacesNameExcludeFilter(configuration))) {
            cQl.append(" AND space not in (").append(ConfluenceConfigHelper.getSpacesNameExcludeFilter(configuration).stream().collect(Collectors.joining("\",\"", "\"", "\""))).append(")");
        }
        if (!CollectionUtils.isEmpty(ConfluenceConfigHelper.getContentTypeIncludeFilter(configuration))) {
            cQl.append(" AND type in (").append(ConfluenceConfigHelper.getContentTypeIncludeFilter(configuration).stream().collect(Collectors.joining("\",\"", "\"", "\""))).append(")");
        }
        if (!CollectionUtils.isEmpty(ConfluenceConfigHelper.getContentTypeExcludeFilter(configuration))) {
            cQl.append(" AND type not in (").append(ConfluenceConfigHelper.getContentTypeExcludeFilter(configuration).stream().collect(Collectors.joining("\",\"", "\"", "\""))).append(")");
        }
        cQl.append(" order by lastModified asc ");
        log.info("Created content filter criteria ConfluenceQl query: {}", (Object)cQl);
        return cQl;
    }

    private void validatePageTypeFilters(ConfluenceSourceConfig configuration) {
        log.trace("Validating Page Type filters");
        ArrayList badFilters = new ArrayList();
        HashSet includedPageType = new HashSet();
        ArrayList includedAndExcludedPageType = new ArrayList();
        ConfluenceConfigHelper.getContentTypeIncludeFilter(configuration).forEach(pageTypeFilter -> {
            if (ConfluenceContentType.fromString(pageTypeFilter) == null) {
                badFilters.add(pageTypeFilter);
            } else {
                includedPageType.add(pageTypeFilter);
            }
        });
        ConfluenceConfigHelper.getContentTypeExcludeFilter(configuration).forEach(pageTypeFilter -> {
            if (includedPageType.contains(pageTypeFilter)) {
                includedAndExcludedPageType.add(pageTypeFilter);
            }
            if (ConfluenceContentType.fromString(pageTypeFilter) == null) {
                badFilters.add(pageTypeFilter);
            }
        });
        if (!badFilters.isEmpty()) {
            String filters = String.join((CharSequence)("\"" + String.valueOf(badFilters) + "\""), ", ");
            log.error("One or more invalid Page Types found in filter configuration: {}", badFilters);
            throw new InvalidPluginConfigurationException("Bad request exception occurred Invalid Page Type key found in filter configuration " + filters);
        }
        if (!includedAndExcludedPageType.isEmpty()) {
            String filters = String.join((CharSequence)("\"" + String.valueOf(includedAndExcludedPageType) + "\""), ", ");
            log.error("One or more Page types found in both include and exclude: {}", includedAndExcludedPageType);
            throw new InvalidPluginConfigurationException("Bad request exception occurred Page Type filters is invalid because the following Page types are listed in both include and exclude" + filters);
        }
    }

    private void validateSpaceFilters(ConfluenceSourceConfig configuration) {
        log.trace("Validating space filters");
        ArrayList badFilters = new ArrayList();
        HashSet includedSpaces = new HashSet();
        ArrayList includedAndExcludedSpaces = new ArrayList();
        ConfluenceConfigHelper.getSpacesNameIncludeFilter(configuration).forEach(spaceFilter -> {
            Matcher matcher = NameConfig.VALID_SPACE_KEY_REGEX.matcher((CharSequence)spaceFilter);
            includedSpaces.add(spaceFilter);
            if (!matcher.find() || spaceFilter.length() <= 1 || spaceFilter.length() > 100) {
                badFilters.add(spaceFilter);
            }
        });
        ConfluenceConfigHelper.getSpacesNameExcludeFilter(configuration).forEach(spaceFilter -> {
            Matcher matcher = NameConfig.VALID_SPACE_KEY_REGEX.matcher((CharSequence)spaceFilter);
            if (includedSpaces.contains(spaceFilter)) {
                includedAndExcludedSpaces.add(spaceFilter);
            }
            if (!matcher.find() || spaceFilter.length() <= 1 || spaceFilter.length() > 100) {
                badFilters.add(spaceFilter);
            }
        });
        if (!badFilters.isEmpty()) {
            String filters = String.join((CharSequence)("\"" + String.valueOf(badFilters) + "\""), ", ");
            log.error("One or more invalid Space keys found in filter configuration: {}", badFilters);
            throw new InvalidPluginConfigurationException("Bad request exception occurred Invalid Space key found in filter configuration for " + filters);
        }
        if (!includedAndExcludedSpaces.isEmpty()) {
            String filters = String.join((CharSequence)("\"" + String.valueOf(includedAndExcludedSpaces) + "\""), ", ");
            log.error("One or more Space keys found in both include and exclude: {}", includedAndExcludedSpaces);
            throw new InvalidPluginConfigurationException("Bad request exception occurred Space filters is invalid because the following space are listed in both include and exclude" + filters);
        }
    }
}

