/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.indices;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: indices.get_template.Request

/**
 * Returns an index template.
 *
 */

public class GetTemplateRequest extends RequestBase {
    @Nullable
    private final Boolean flatSettings;

    @Nullable
    private final Boolean local;

    @Deprecated
    @Nullable
    private final Time masterTimeout;

    @Nullable
    private final Time clusterManagerTimeout;

    private final List<String> name;

    // ---------------------------------------------------------------------------------------------

    private GetTemplateRequest(Builder builder) {

        this.flatSettings = builder.flatSettings;
        this.local = builder.local;
        this.masterTimeout = builder.masterTimeout;
        this.clusterManagerTimeout = builder.clusterManagerTimeout;
        this.name = ApiTypeHelper.unmodifiable(builder.name);

    }

    public static GetTemplateRequest of(Function<Builder, ObjectBuilder<GetTemplateRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Return settings in flat format (default: false)
     * <p>
     * API name: {@code flat_settings}
     */
    @Nullable
    public final Boolean flatSettings() {
        return this.flatSettings;
    }

    /**
     * Return local information, do not retrieve the state from cluster-manager node
     * (default: false)
     * <p>
     * API name: {@code local}
     */
    @Nullable
    public final Boolean local() {
        return this.local;
    }

    /**
     * Explicit operation timeout for connection to master node
     * <p>
     * API name: {@code master_timeout}
     */
    @Deprecated
    @Nullable
    public final Time masterTimeout() {
        return this.masterTimeout;
    }

    /**
     * Explicit operation timeout for connection to cluster-manager node
     * <p>
     * API name: {@code cluster_manager_timeout}
     */
    @Nullable
    public final Time clusterManagerTimeout() {
        return this.clusterManagerTimeout;
    }

    /**
     * The comma separated names of the index templates
     * <p>
     * API name: {@code name}
     */
    public final List<String> name() {
        return this.name;
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link GetTemplateRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<GetTemplateRequest> {
        @Nullable
        private Boolean flatSettings;

        @Nullable
        private Boolean local;

        @Deprecated
        @Nullable
        private Time masterTimeout;

        @Nullable
        private Time clusterManagerTimeout;

        @Nullable
        private List<String> name;

        /**
         * Return settings in flat format (default: false)
         * <p>
         * API name: {@code flat_settings}
         */
        public final Builder flatSettings(@Nullable Boolean value) {
            this.flatSettings = value;
            return this;
        }

        /**
         * Return local information, do not retrieve the state from cluster-manager node
         * (default: false)
         * <p>
         * API name: {@code local}
         */
        public final Builder local(@Nullable Boolean value) {
            this.local = value;
            return this;
        }

        /**
         * Explicit operation timeout for connection to master node
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(@Nullable Time value) {
            this.masterTimeout = value;
            return this;
        }

        /**
         * Explicit operation timeout for connection to master node
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.masterTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Explicit operation timeout for connection to cluster-manager node
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(@Nullable Time value) {
            this.clusterManagerTimeout = value;
            return this;
        }

        /**
         * Explicit operation timeout for connection to cluster-manager node
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.clusterManagerTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * The comma separated names of the index templates
         * <p>
         * API name: {@code name}
         * <p>
         * Adds all elements of <code>list</code> to <code>name</code>.
         */
        public final Builder name(List<String> list) {
            this.name = _listAddAll(this.name, list);
            return this;
        }

        /**
         * The comma separated names of the index templates
         * <p>
         * API name: {@code name}
         * <p>
         * Adds one or more values to <code>name</code>.
         */
        public final Builder name(String value, String... values) {
            this.name = _listAdd(this.name, value, values);
            return this;
        }

        /**
         * Builds a {@link GetTemplateRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public GetTemplateRequest build() {
            _checkSingleUse();

            return new GetTemplateRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code indices.get_template}".
     */
    public static final Endpoint<GetTemplateRequest, GetTemplateResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "GET";

        },

        // Request path
        request -> {
            final int _name = 1 << 0;

            int propsSet = 0;

            if (ApiTypeHelper.isDefined(request.name())) propsSet |= _name;

            if (propsSet == 0) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_template");
                return buf.toString();
            }
            if (propsSet == (_name)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_template");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.name.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.masterTimeout != null) {
                params.put("master_timeout", request.masterTimeout._toJsonString());
            }
            if (request.clusterManagerTimeout != null) {
                params.put("cluster_manager_timeout", request.clusterManagerTimeout._toJsonString());
            }
            if (request.flatSettings != null) {
                params.put("flat_settings", String.valueOf(request.flatSettings));
            }
            if (request.local != null) {
                params.put("local", String.valueOf(request.local));
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        false,
        GetTemplateResponse._DESERIALIZER
    );
}
