/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.indices;

import jakarta.json.stream.JsonGenerator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.PlainJsonSerializable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.ExpandWildcard;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: indices.put_settings.Request

/**
 * Updates the index settings.
 *
 */
@JsonpDeserializable
public class PutIndicesSettingsRequest extends RequestBase implements PlainJsonSerializable {
    @Nullable
    private final Boolean allowNoIndices;

    private final List<ExpandWildcard> expandWildcards;

    @Nullable
    private final Boolean flatSettings;

    @Nullable
    private final Boolean ignoreUnavailable;

    private final List<String> index;

    @Deprecated
    @Nullable
    private final Time masterTimeout;

    @Nullable
    private final Time clusterManagerTimeout;

    @Nullable
    private final Boolean preserveExisting;

    @Nullable
    private final Time timeout;

    private final IndexSettings settings;

    // ---------------------------------------------------------------------------------------------

    private PutIndicesSettingsRequest(Builder builder) {

        this.allowNoIndices = builder.allowNoIndices;
        this.expandWildcards = ApiTypeHelper.unmodifiable(builder.expandWildcards);
        this.flatSettings = builder.flatSettings;
        this.ignoreUnavailable = builder.ignoreUnavailable;
        this.index = ApiTypeHelper.unmodifiable(builder.index);
        this.masterTimeout = builder.masterTimeout;
        this.clusterManagerTimeout = builder.clusterManagerTimeout;
        this.preserveExisting = builder.preserveExisting;
        this.timeout = builder.timeout;
        this.settings = ApiTypeHelper.requireNonNull(builder.settings, this, "settings");

    }

    public static PutIndicesSettingsRequest of(Function<Builder, ObjectBuilder<PutIndicesSettingsRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Whether to ignore if a wildcard indices expression resolves into no concrete
     * indices. (This includes <code>_all</code> string or when no indices have been
     * specified)
     * <p>
     * API name: {@code allow_no_indices}
     */
    @Nullable
    public final Boolean allowNoIndices() {
        return this.allowNoIndices;
    }

    /**
     * Whether to expand wildcard expression to concrete indices that are open,
     * closed or both.
     * <p>
     * API name: {@code expand_wildcards}
     */
    public final List<ExpandWildcard> expandWildcards() {
        return this.expandWildcards;
    }

    /**
     * Return settings in flat format (default: false)
     * <p>
     * API name: {@code flat_settings}
     */
    @Nullable
    public final Boolean flatSettings() {
        return this.flatSettings;
    }

    /**
     * Whether specified concrete indices should be ignored when unavailable
     * (missing or closed)
     * <p>
     * API name: {@code ignore_unavailable}
     */
    @Nullable
    public final Boolean ignoreUnavailable() {
        return this.ignoreUnavailable;
    }

    /**
     * A comma-separated list of index names; use <code>_all</code> or empty string
     * to perform the operation on all indices
     * <p>
     * API name: {@code index}
     */
    public final List<String> index() {
        return this.index;
    }

    /**
     * Specify timeout for connection to master
     * <p>
     * API name: {@code master_timeout}
     */
    @Deprecated
    @Nullable
    public final Time masterTimeout() {
        return this.masterTimeout;
    }

    /**
     * Specify timeout for connection to cluster-manager
     * <p>
     * API name: {@code cluster_manager_timeout}
     */
    @Nullable
    public final Time clusterManagerTimeout() {
        return this.clusterManagerTimeout;
    }

    /**
     * Whether to update existing settings. If set to <code>true</code> existing
     * settings on an index remain unchanged, the default is <code>false</code>
     * <p>
     * API name: {@code preserve_existing}
     */
    @Nullable
    public final Boolean preserveExisting() {
        return this.preserveExisting;
    }

    /**
     * Explicit operation timeout
     * <p>
     * API name: {@code timeout}
     */
    @Nullable
    public final Time timeout() {
        return this.timeout;
    }

    /**
     * Required - Request body.
     * <p>
     * API name: {@code _value_body}
     */
    public final IndexSettings settings() {
        return this.settings;
    }

    /**
     * Serialize this value to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        this.settings.serialize(generator, mapper);

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link PutIndicesSettingsRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<PutIndicesSettingsRequest> {
        @Nullable
        private Boolean allowNoIndices;

        @Nullable
        private List<ExpandWildcard> expandWildcards;

        @Nullable
        private Boolean flatSettings;

        @Nullable
        private Boolean ignoreUnavailable;

        @Nullable
        private List<String> index;

        @Deprecated
        @Nullable
        private Time masterTimeout;

        @Nullable
        private Time clusterManagerTimeout;

        @Nullable
        private Boolean preserveExisting;

        @Nullable
        private Time timeout;

        private IndexSettings settings;

        /**
         * Whether to ignore if a wildcard indices expression resolves into no concrete
         * indices. (This includes <code>_all</code> string or when no indices have been
         * specified)
         * <p>
         * API name: {@code allow_no_indices}
         */
        public final Builder allowNoIndices(@Nullable Boolean value) {
            this.allowNoIndices = value;
            return this;
        }

        /**
         * Whether to expand wildcard expression to concrete indices that are open,
         * closed or both.
         * <p>
         * API name: {@code expand_wildcards}
         * <p>
         * Adds all elements of <code>list</code> to <code>expandWildcards</code>.
         */
        public final Builder expandWildcards(List<ExpandWildcard> list) {
            this.expandWildcards = _listAddAll(this.expandWildcards, list);
            return this;
        }

        /**
         * Whether to expand wildcard expression to concrete indices that are open,
         * closed or both.
         * <p>
         * API name: {@code expand_wildcards}
         * <p>
         * Adds one or more values to <code>expandWildcards</code>.
         */
        public final Builder expandWildcards(ExpandWildcard value, ExpandWildcard... values) {
            this.expandWildcards = _listAdd(this.expandWildcards, value, values);
            return this;
        }

        /**
         * Return settings in flat format (default: false)
         * <p>
         * API name: {@code flat_settings}
         */
        public final Builder flatSettings(@Nullable Boolean value) {
            this.flatSettings = value;
            return this;
        }

        /**
         * Whether specified concrete indices should be ignored when unavailable
         * (missing or closed)
         * <p>
         * API name: {@code ignore_unavailable}
         */
        public final Builder ignoreUnavailable(@Nullable Boolean value) {
            this.ignoreUnavailable = value;
            return this;
        }

        /**
         * A comma-separated list of index names; use <code>_all</code> or empty string
         * to perform the operation on all indices
         * <p>
         * API name: {@code index}
         * <p>
         * Adds all elements of <code>list</code> to <code>index</code>.
         */
        public final Builder index(List<String> list) {
            this.index = _listAddAll(this.index, list);
            return this;
        }

        /**
         * A comma-separated list of index names; use <code>_all</code> or empty string
         * to perform the operation on all indices
         * <p>
         * API name: {@code index}
         * <p>
         * Adds one or more values to <code>index</code>.
         */
        public final Builder index(String value, String... values) {
            this.index = _listAdd(this.index, value, values);
            return this;
        }

        /**
         * Specify timeout for connection to master
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(@Nullable Time value) {
            this.masterTimeout = value;
            return this;
        }

        /**
         * Specify timeout for connection to master
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.masterTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Specify timeout for connection to cluster-manager
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(@Nullable Time value) {
            this.clusterManagerTimeout = value;
            return this;
        }

        /**
         * Specify timeout for connection to cluster-manager
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.clusterManagerTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Whether to update existing settings. If set to <code>true</code> existing
         * settings on an index remain unchanged, the default is <code>false</code>
         * <p>
         * API name: {@code preserve_existing}
         */
        public final Builder preserveExisting(@Nullable Boolean value) {
            this.preserveExisting = value;
            return this;
        }

        /**
         * Explicit operation timeout
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(@Nullable Time value) {
            this.timeout = value;
            return this;
        }

        /**
         * Explicit operation timeout
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.timeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Required - Request body.
         * <p>
         * API name: {@code _value_body}
         */
        public final Builder settings(IndexSettings value) {
            this.settings = value;
            return this;
        }

        /**
         * Required - Request body.
         * <p>
         * API name: {@code _value_body}
         */
        public final Builder settings(Function<IndexSettings.Builder, ObjectBuilder<IndexSettings>> fn) {
            return this.settings(fn.apply(new IndexSettings.Builder()).build());
        }

        /**
         * Builds a {@link PutIndicesSettingsRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public PutIndicesSettingsRequest build() {
            _checkSingleUse();

            return new PutIndicesSettingsRequest(this);
        }
    }

    public static final JsonpDeserializer<PutIndicesSettingsRequest> _DESERIALIZER = createPutIndicesSettingsRequestDeserializer();

    protected static JsonpDeserializer<PutIndicesSettingsRequest> createPutIndicesSettingsRequestDeserializer() {

        JsonpDeserializer<IndexSettings> valueDeserializer = IndexSettings._DESERIALIZER;

        return JsonpDeserializer.of(
            valueDeserializer.acceptedEvents(),
            (parser, mapper) -> new Builder().settings(valueDeserializer.deserialize(parser, mapper)).build()
        );
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code indices.put_settings}".
     */
    public static final Endpoint<PutIndicesSettingsRequest, PutIndicesSettingsResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "PUT";

        },

        // Request path
        request -> {
            final int _index = 1 << 0;

            int propsSet = 0;

            if (ApiTypeHelper.isDefined(request.index())) propsSet |= _index;

            if (propsSet == 0) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_settings");
                return buf.toString();
            }
            if (propsSet == (_index)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/");
                SimpleEndpoint.pathEncode(request.index.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                buf.append("/_settings");
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.masterTimeout != null) {
                params.put("master_timeout", request.masterTimeout._toJsonString());
            }
            if (request.clusterManagerTimeout != null) {
                params.put("cluster_manager_timeout", request.clusterManagerTimeout._toJsonString());
            }
            if (request.flatSettings != null) {
                params.put("flat_settings", String.valueOf(request.flatSettings));
            }
            if (ApiTypeHelper.isDefined(request.expandWildcards)) {
                params.put("expand_wildcards", request.expandWildcards.stream().map(v -> v.jsonValue()).collect(Collectors.joining(",")));
            }
            if (request.ignoreUnavailable != null) {
                params.put("ignore_unavailable", String.valueOf(request.ignoreUnavailable));
            }
            if (request.allowNoIndices != null) {
                params.put("allow_no_indices", String.valueOf(request.allowNoIndices));
            }
            if (request.preserveExisting != null) {
                params.put("preserve_existing", String.valueOf(request.preserveExisting));
            }
            if (request.timeout != null) {
                params.put("timeout", request.timeout._toJsonString());
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        true,
        PutIndicesSettingsResponse._DESERIALIZER
    );
}
