/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.cluster;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: cluster.stats.Request

/**
 * Returns high-level overview of cluster statistics.
 *
 */

public class ClusterStatsRequest extends RequestBase {
    @Nullable
    private final Boolean flatSettings;

    private final List<String> nodeId;

    @Nullable
    private final Time timeout;

    // ---------------------------------------------------------------------------------------------

    private ClusterStatsRequest(Builder builder) {

        this.flatSettings = builder.flatSettings;
        this.nodeId = ApiTypeHelper.unmodifiable(builder.nodeId);
        this.timeout = builder.timeout;

    }

    public static ClusterStatsRequest of(Function<Builder, ObjectBuilder<ClusterStatsRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Return settings in flat format (default: false)
     * <p>
     * API name: {@code flat_settings}
     */
    @Nullable
    public final Boolean flatSettings() {
        return this.flatSettings;
    }

    /**
     * Comma-separated list of node filters used to limit returned information.
     * Defaults to all nodes in the cluster.
     * <p>
     * API name: {@code node_id}
     */
    public final List<String> nodeId() {
        return this.nodeId;
    }

    /**
     * Period to wait for each node to respond. If a node does not respond before
     * its timeout expires, the response does not include its stats. However, timed
     * out nodes are included in the response's _nodes.failed property. Defaults to
     * no timeout.
     * <p>
     * API name: {@code timeout}
     */
    @Nullable
    public final Time timeout() {
        return this.timeout;
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link ClusterStatsRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<ClusterStatsRequest> {
        @Nullable
        private Boolean flatSettings;

        @Nullable
        private List<String> nodeId;

        @Nullable
        private Time timeout;

        /**
         * Return settings in flat format (default: false)
         * <p>
         * API name: {@code flat_settings}
         */
        public final Builder flatSettings(@Nullable Boolean value) {
            this.flatSettings = value;
            return this;
        }

        /**
         * Comma-separated list of node filters used to limit returned information.
         * Defaults to all nodes in the cluster.
         * <p>
         * API name: {@code node_id}
         * <p>
         * Adds all elements of <code>list</code> to <code>nodeId</code>.
         */
        public final Builder nodeId(List<String> list) {
            this.nodeId = _listAddAll(this.nodeId, list);
            return this;
        }

        /**
         * Comma-separated list of node filters used to limit returned information.
         * Defaults to all nodes in the cluster.
         * <p>
         * API name: {@code node_id}
         * <p>
         * Adds one or more values to <code>nodeId</code>.
         */
        public final Builder nodeId(String value, String... values) {
            this.nodeId = _listAdd(this.nodeId, value, values);
            return this;
        }

        /**
         * Period to wait for each node to respond. If a node does not respond before
         * its timeout expires, the response does not include its stats. However, timed
         * out nodes are included in the response's _nodes.failed property. Defaults to
         * no timeout.
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(@Nullable Time value) {
            this.timeout = value;
            return this;
        }

        /**
         * Period to wait for each node to respond. If a node does not respond before
         * its timeout expires, the response does not include its stats. However, timed
         * out nodes are included in the response's _nodes.failed property. Defaults to
         * no timeout.
         * <p>
         * API name: {@code timeout}
         */
        public final Builder timeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.timeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Builds a {@link ClusterStatsRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public ClusterStatsRequest build() {
            _checkSingleUse();

            return new ClusterStatsRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code cluster.stats}".
     */
    public static final Endpoint<ClusterStatsRequest, ClusterStatsResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "GET";

        },

        // Request path
        request -> {
            final int _nodeId = 1 << 0;

            int propsSet = 0;

            if (ApiTypeHelper.isDefined(request.nodeId())) propsSet |= _nodeId;

            if (propsSet == 0) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_cluster");
                buf.append("/stats");
                return buf.toString();
            }
            if (propsSet == (_nodeId)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_cluster");
                buf.append("/stats");
                buf.append("/nodes");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.nodeId.stream().map(v -> v).collect(Collectors.joining(",")), buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.flatSettings != null) {
                params.put("flat_settings", String.valueOf(request.flatSettings));
            }
            if (request.timeout != null) {
                params.put("timeout", request.timeout._toJsonString());
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        false,
        ClusterStatsResponse._DESERIALIZER
    );
}
