/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.xmlsec.encryption.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.xml.namespace.QName;

import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.IndexedXMLObjectChildrenList;
import org.opensaml.xmlsec.encryption.AgreementMethod;
import org.opensaml.xmlsec.encryption.KANonce;
import org.opensaml.xmlsec.encryption.OriginatorKeyInfo;
import org.opensaml.xmlsec.encryption.RecipientKeyInfo;

/**
 * Concrete implementation of {@link org.opensaml.xmlsec.encryption.AgreementMethod}.
 */
public class AgreementMethodImpl extends AbstractXMLObject implements AgreementMethod {
    
    /** Algorithm attribute value. */
    private String algorithm;
    
    /** KA-Nonce child element value. */
    private KANonce kaNonce;
    
    /** OriginatorKeyInfo child element value. */
    private OriginatorKeyInfo originatorKeyInfo;
    
    /** RecipientKeyInfo child element value. */
    private RecipientKeyInfo recipientKeyInfo;
    
    /** List of wildcard &lt;any&gt; XMLObject children. */
    private IndexedXMLObjectChildrenList<XMLObject> xmlChildren;

    /**
     * Constructor.
     *
     * @param namespaceURI namespace URI
     * @param elementLocalName element local name
     * @param namespacePrefix namespace prefix
     */
    protected AgreementMethodImpl(final String namespaceURI, final String elementLocalName,
            final String namespacePrefix) {
        super(namespaceURI, elementLocalName, namespacePrefix);
        xmlChildren = new IndexedXMLObjectChildrenList<>(this);
    }

    /** {@inheritDoc} */
    public String getAlgorithm() {
        return algorithm;
    }

    /** {@inheritDoc} */
    public void setAlgorithm(final String newAlgorithm) {
        algorithm = prepareForAssignment(algorithm, newAlgorithm);
    }

    /** {@inheritDoc} */
    public KANonce getKANonce() {
        return this.kaNonce;
    }

    /** {@inheritDoc} */
    public void setKANonce(final KANonce newKANonce) {
        kaNonce = prepareForAssignment(kaNonce, newKANonce);
    }

    /** {@inheritDoc} */
    public OriginatorKeyInfo getOriginatorKeyInfo() {
        return originatorKeyInfo;
    }

    /** {@inheritDoc} */
    public void setOriginatorKeyInfo(final OriginatorKeyInfo newOriginatorKeyInfo) {
        originatorKeyInfo = prepareForAssignment(originatorKeyInfo, newOriginatorKeyInfo);
    }

    /** {@inheritDoc} */
    public RecipientKeyInfo getRecipientKeyInfo() {
        return recipientKeyInfo;
    }

    /** {@inheritDoc} */
    public void setRecipientKeyInfo(final RecipientKeyInfo newRecipientKeyInfo) {
        recipientKeyInfo = prepareForAssignment(recipientKeyInfo, newRecipientKeyInfo);
    }

    /** {@inheritDoc} */
    public List<XMLObject> getUnknownXMLObjects() {
        return xmlChildren;
    }
    /** {@inheritDoc} */
    public List<XMLObject> getUnknownXMLObjects(final QName typeOrName) {
        return (List<XMLObject>) xmlChildren.subList(typeOrName);
    }

    /** {@inheritDoc} */
    public List<XMLObject> getOrderedChildren() {
        final ArrayList<XMLObject> children = new ArrayList<>();
        
        if (kaNonce != null) {
            children.add(kaNonce);
        }
        
        children.addAll(xmlChildren);
        
        if (originatorKeyInfo != null) {
            children.add(originatorKeyInfo);
        }
        if (recipientKeyInfo != null) {
            children.add(recipientKeyInfo);
        }
        
        if (children.size() == 0) {
            return null;
        }
        
        return Collections.unmodifiableList(children);
    }

}
