/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.security.x509;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.crypto.SecretKey;

import org.opensaml.security.credential.BasicCredential;
import org.opensaml.security.credential.Credential;

import net.shibboleth.utilities.java.support.annotation.ParameterName;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A basic implementation of {@link X509Credential}.
 */
public class BasicX509Credential extends BasicCredential implements X509Credential {

    /** Entity certificate. */
    private X509Certificate entityCert;

    /** Entity certificate chain, must include entity certificate. */
    private Collection<X509Certificate> entityCertChain;

    /** CRLs for this credential. */
    private Collection<X509CRL> crls;
    
    /**
     * Constructor.
     *
     * @param entityCertificate the credential entity certificate
     */
    public BasicX509Credential(
            @Nonnull @ParameterName(name="entityCertificate") final X509Certificate entityCertificate) {
        super();
        setEntityCertificate(entityCertificate);
    }
    
    /**
     * Constructor.
     *
     * @param entityCertificate the credential entity certificate
     * @param privateKey the credential private key
     */
    public BasicX509Credential(
            @Nonnull @ParameterName(name="entityCertificate") final X509Certificate entityCertificate,
            @ParameterName(name="privateKey") @Nonnull final PrivateKey privateKey) {
        super();
        setEntityCertificate(entityCertificate);
        setPrivateKey(privateKey);
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Class<? extends Credential> getCredentialType() {
        return X509Credential.class;
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Collection<X509CRL> getCRLs() {
        return crls;
    }

    /**
     * Sets the CRLs for this credential.
     * 
     * @param newCRLs CRLs for this credential
     */
    public void setCRLs(@Nullable final Collection<X509CRL> newCRLs) {
        crls = newCRLs;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public X509Certificate getEntityCertificate() {
        return entityCert;
    }

    /**
     * Sets the entity certificate for this credential.
     * 
     * @param newEntityCertificate entity certificate for this credential
     */
    public void setEntityCertificate(@Nonnull final X509Certificate newEntityCertificate) {
        Constraint.isNotNull(newEntityCertificate, "Credential certificate cannot be null");
        entityCert = newEntityCertificate;
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull public PublicKey getPublicKey() {
        return getEntityCertificate().getPublicKey();
    }
    
    /**
     * This operation is unsupported for X.509 credentials. The public key will be retrieved
     * automatically from the entity certificate.
     * 
     * @param newPublicKey not supported
     */
    @Override
    public void setPublicKey(final PublicKey newPublicKey) {
        throw new UnsupportedOperationException("Public key may not be set explicitly on an X509 credential");
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Collection<X509Certificate> getEntityCertificateChain() {
        if (entityCertChain == null) {
            final LazySet<X509Certificate> constructedChain = new LazySet<>();
            constructedChain.add(entityCert);
            return constructedChain;
        }
        return entityCertChain;
    }

    /**
     * Sets the entity certificate chain for this credential. This <strong>MUST</strong> include the entity
     * certificate.
     * 
     * @param newCertificateChain entity certificate chain for this credential
     */
    public void setEntityCertificateChain(@Nonnull final Collection<X509Certificate> newCertificateChain) {
        Constraint.isNotNull(newCertificateChain, "Certificate chain collection cannot be null");
        Constraint.isNotEmpty(newCertificateChain, "Certificate chain collection cannot be empty");
        entityCertChain = new ArrayList<>(newCertificateChain);
    }
    
    /**
     *  This operation is unsupported for X.509 credentials.
     *  
     *  @return null
     */
    @Override
    @Nullable public SecretKey getSecretKey() {
        return null;
    }
    
    /**
     *  This operation is unsupported for X.509 credentials.
     *  
     *  @param newSecretKey unsupported
     */
    @Override
    public void setSecretKey(final SecretKey newSecretKey) {
        throw new UnsupportedOperationException("An X509Credential may not contain a secret key");
    }

}