"use strict";
/**
 * @fileoverview Named exports should not use the name annotation if it is redundant to the name that would be generated by the export name
 * @author Yann Braga
 */
const csf_1 = require("@storybook/csf");
const ast_1 = require("../utils/ast");
const constants_1 = require("../utils/constants");
const create_storybook_rule_1 = require("../utils/create-storybook-rule");
module.exports = (0, create_storybook_rule_1.createStorybookRule)({
    name: 'no-redundant-story-name',
    defaultOptions: [],
    meta: {
        type: 'suggestion',
        fixable: 'code',
        hasSuggestions: true,
        docs: {
            description: 'A story should not have a redundant name property',
            categories: [constants_1.CategoryId.CSF, constants_1.CategoryId.RECOMMENDED],
            recommended: 'warn',
        },
        messages: {
            removeRedundantName: 'Remove redundant name',
            storyNameIsRedundant: 'Named exports should not use the name annotation if it is redundant to the name that would be generated by the export name',
        },
        schema: [],
    },
    create(context) {
        //----------------------------------------------------------------------
        // Public
        //----------------------------------------------------------------------
        return {
            // CSF3
            ExportNamedDeclaration: function (node) {
                // if there are specifiers, node.declaration should be null
                if (!node.declaration)
                    return;
                const decl = node.declaration;
                if ((0, ast_1.isVariableDeclaration)(decl)) {
                    const declaration = decl.declarations[0];
                    if (declaration == null)
                        return;
                    const { id, init } = declaration;
                    if ((0, ast_1.isIdentifier)(id) && (0, ast_1.isObjectExpression)(init)) {
                        const storyNameNode = init.properties.find((prop) => {
                            var _a, _b;
                            return (0, ast_1.isProperty)(prop) &&
                                (0, ast_1.isIdentifier)(prop.key) &&
                                (((_a = prop.key) === null || _a === void 0 ? void 0 : _a.name) === 'name' || ((_b = prop.key) === null || _b === void 0 ? void 0 : _b.name) === 'storyName');
                        });
                        if (!storyNameNode) {
                            return;
                        }
                        const { name } = id;
                        const resolvedStoryName = (0, csf_1.storyNameFromExport)(name);
                        if (!(0, ast_1.isSpreadElement)(storyNameNode) &&
                            (0, ast_1.isLiteral)(storyNameNode.value) &&
                            storyNameNode.value.value === resolvedStoryName) {
                            context.report({
                                node: storyNameNode,
                                messageId: 'storyNameIsRedundant',
                                suggest: [
                                    {
                                        messageId: 'removeRedundantName',
                                        fix: function (fixer) {
                                            return fixer.remove(storyNameNode);
                                        },
                                    },
                                ],
                            });
                        }
                    }
                }
            },
            // CSF2
            AssignmentExpression: function (node) {
                if (!(0, ast_1.isExpressionStatement)(node.parent))
                    return;
                const { left, right } = node;
                if ('property' in left &&
                    (0, ast_1.isIdentifier)(left.property) &&
                    !(0, ast_1.isMetaProperty)(left) &&
                    left.property.name === 'storyName') {
                    if (!('name' in left.object && 'value' in right))
                        return;
                    const propertyName = left.object.name;
                    const propertyValue = right.value;
                    const resolvedStoryName = (0, csf_1.storyNameFromExport)(propertyName);
                    if (propertyValue === resolvedStoryName) {
                        context.report({
                            node: node,
                            messageId: 'storyNameIsRedundant',
                            suggest: [
                                {
                                    messageId: 'removeRedundantName',
                                    fix: function (fixer) {
                                        return fixer.remove(node);
                                    },
                                },
                            ],
                        });
                    }
                }
            },
        };
    },
});
