/**
 * An integer number
 */
declare type int = number;

/**
 * A number, potentially with decimals
 */
declare type float = number;

/**
 * An alternative name for the global type Element.
 *
 * UI5 defines its own type sap.ui.core.Element, which, in the scope of the sap.ui.core
 * namespace, hides the global type. As there are other entities in the same namespace
 * that refer to the global Element in their signatures, this naming conflict can cause
 * type mismatches between sap.ui.core code and other code subclassing from it.
 *
 * To avoid these issues, the alternative name can be used in sap.ui.core signatures.
 *
 * @todo check if HTMLElement could be used instead in UI5 method signatures
 */
declare type global_Element = Element;

declare type jQuery<T = HTMLElement> = JQuery<T>;
declare namespace jQuery {
  export type Event = JQuery.Event;
  export type Deferred<T = any> = JQuery.Deferred<T>;
  export type Promise<T = any> = JQuery.Promise<T>;
}

declare namespace QUnit {
  export type Assert = globalThis.Assert;
}

interface JQuery<TElement = HTMLElement> extends Iterable<TElement> {
  /**
   * Adds the given ID reference to the aria-describedby attribute.
   */
  addAriaDescribedBy(
    /**
     * The ID reference of an element
     */
    sId: string,
    /**
     * whether prepend or not
     */
    bPrepend?: boolean
  ): jQuery;

  /**
   * Adds the given ID reference to the aria-labelledby attribute.
   */
  addAriaLabelledBy(
    /**
     * The ID reference of an element
     */
    sId: string,
    /**
     * Whether prepend or not
     */
    bPrepend?: boolean
  ): jQuery;

  /**
   * Extension function to the jQuery.fn which identifies SAPUI5 controls in the given jQuery context.
   * @deprecated As of version 1.106, use {@link sap.ui.core.Element.closestTo} instead.
   */
  control(
    /**
     * Whether or not to respect the associated DOM elements to a control via <code>data-sap-ui-related</code> attribute.
     */
    includeRelated?: boolean
  ): Array<import("sap/ui/core/Control").default>;

  /**
   * Extension function to the jQuery.fn which identifies SAPUI5 controls in the given jQuery context.
   * @deprecated As of version 1.106, use {@link sap.ui.core.Element.closestTo} instead.
   */
  control(
    /**
     * Parameter to return the control instance at the given index in the array.
     */
    index: int,
    /**
     * Whether or not to respect the associated DOM elements to a control via <code>data-sap-ui-related</code> attribute.
     */
    includeRelated?: boolean
  ): import("sap/ui/core/Control").default | null;

  /**
   * Gets the position of the cursor in an element that supports cursor positioning.
   */
  cursorPos(): int;

  /**
   * Sets the position of the cursor in an element that supports cursor positioning.
   */
  cursorPos(
    /**
     * The cursor position to set
     */
    iPos: int
  ): jQuery;

  /**
   * Disable HTML elements selection.
   */
  disableSelection(): jQuery;

  /**
   * Enable HTML elements to get selected.
   */
  enableSelection(): jQuery;

  /**
   * Returns the first focusable domRef in a given container (the first element of the collection)
   */
  firstFocusableDomRef(): Element;

  /**
   * Retrieve the selected text in the first element of the collection.
   *
   * <b>Note</b>: This feature is only supported for input element’s type of text, search, url, tel and password.
   */
  getSelectedText(): string;

  /**
   * Returns <code>true</code> if the first element has a set tabindex.
   */
  hasTabIndex(): boolean;

  /**
   * Returns the last focusable domRef in a given container
   */
  lastFocusableDomRef(): Element;

  /**
   * Gets the next parent DOM element with a given attribute and attribute value starting above the first given element
   */
  parentByAttribute(
    /**
     * Name of the attribute
     */
    sAttribute: string,
    /**
     * Value of the attribute (optional)
     */
    sValue: string
  ): Element;

  /**
   * Returns a rectangle describing the current visual positioning of the first DOM object in the collection
   * (or <code>null</code> if no element was given).
   */
  rect(): object;

  /**
   * Returns whether a point described by X and Y is inside this Rectangle's boundaries.
   */
  rectContains(
    /**
     * The X coordinate
     */
    posX: int,
    /**
     * The Y coordinate
     */
    posY: int
  ): boolean;

  /**
   * Removes the given ID reference from the aria-describedby attribute.
   */
  removeAriaDescribedBy(
    /**
     * The ID reference of an element
     */
    sId: string
  ): jQuery;

  /**
   * Removes the given ID reference from the aria-labelledby attribute.
   */
  removeAriaLabelledBy(
    /**
     * The ID reference of an element
     */
    sId: string
  ): jQuery;

  /**
   * Returns the scrollLeft value of the first element in the given jQuery collection in right-to-left mode.
   *
   * Precondition: The element is rendered in RTL mode.
   *
   * Reason for this method is that the major browsers use three different values for the same scroll position
   * when in RTL mode. This method hides those differences and returns/applies the same value that would be
   * returned in LTR mode: The distance in px how far the given container is scrolled away from the leftmost
   * scroll position.
   *
   * Returns "undefined" if no element is given.
   */
  scrollLeftRTL(): int | undefined;

  /**
   * Sets the scrollLeft value of the first element in the given jQuery collection in right-to-left mode.
   *
   * Precondition: The element is rendered in RTL mode.
   *
   * Reason for this method is that the major browsers use three different values for the same scroll position
   * when in RTL mode. This method hides those differences and returns/applies the same value that would be
   * returned in LTR mode: The distance in px how far the given container is scrolled away from the leftmost
   * scroll position.
   */
  scrollLeftRTL(
    /**
     * The desired scroll position
     */
    iPos: int
  ): jQuery;

  /**
   * Returns the MIRRORED scrollLeft value of the first element in the given jQuery collection in right-to-left mode.
   *
   * Precondition: The element is rendered in RTL mode.
   *
   * Reason for this method is that the major browsers return three different values for the same scroll position
   * when in RTL mode. This method hides those differences and returns the value that would be returned in LTR mode
   * if the UI would be mirrored horizontally: The distance in px how far the given container is scrolled away
   * from the rightmost scroll position.
   *
   * Returns "undefined" if no element is in the given jQuery collection.
   */
  scrollRightRTL(): int | undefined;

  /**
   * Sets the text selection in the first element of the collection.
   *
   * <b>Note</b>: This feature is only supported for input element's type of text, search, url, tel and password.
   */
  selectText(
    /**
     * Start position of the selection (inclusive)
     */
    iStart: int,
    /**
     * End position of the selection (exclusive)
     */
    iEnd: int
  ): jQuery;

  /**
   * Get the z-index for an element.
   */
  zIndex(): number;

  /**
   * Set the z-index for an element.
   */
  zIndex(
    /**
     * The z-index to set
     */
    zIndex: int
  ): jQuery;
}

declare module "sap/ui/thirdparty/jquery" {
  export default jQuery;
}
declare module "sap/ui/thirdparty/qunit-2" {
  export default QUnit;
}

declare namespace sap {
  interface IUI5DefineDependencyNames {
    "sap/ui/thirdparty/jquery": undefined;
    "sap/ui/thirdparty/qunit-2": undefined;
  }
}
// For Library Version: 1.136.5

declare module "sap/base/assert" {
  /**
   * A simple assertion mechanism that logs a message when a given condition is not met.
   *
   * **Note:** Calls to this method might be removed when the JavaScript code is optimized during build. Therefore,
   * callers should not rely on any side effects of this method.
   *
   * @since 1.58
   */
  export default function assert(
    /**
     * Result of the checked assertion
     */
    bResult: boolean,
    /**
     * Message that will be logged when the result is `false`. In case this is a function, the return value
     * of the function will be displayed. This can be used to execute complex code only if the assertion fails.
     */
    vMessage: string | (() => any)
  ): void;
}

declare module "sap/base/i18n/date/CalendarType" {
  /**
   * The types of `Calendar`.
   *
   * @since 1.120
   */
  enum CalendarType {
    /**
     * The Thai buddhist calendar
     */
    Buddhist = "Buddhist",
    /**
     * The Gregorian calendar
     */
    Gregorian = "Gregorian",
    /**
     * The Islamic calendar
     */
    Islamic = "Islamic",
    /**
     * The Japanese emperor calendar
     */
    Japanese = "Japanese",
    /**
     * The Persian Jalali calendar
     */
    Persian = "Persian",
  }
  export default CalendarType;
}

declare module "sap/base/i18n/date/CalendarWeekNumbering" {
  /**
   * The `CalendarWeekNumbering` enum defines how to calculate calendar weeks. Each value defines:
   * 	 - The first day of the week,
   * 	 - the first week of the year.
   *
   * @since 1.120
   */
  enum CalendarWeekNumbering {
    /**
     * The default calendar week numbering:
     *
     * The framework determines the week numbering scheme; currently it is derived from the active format locale.
     * Future versions of UI5 might select a different week numbering scheme.
     */
    Default = "Default",
    /**
     * Official calendar week numbering in most of Europe (ISO 8601 standard):
     * 	Monday is first day of the week, the week containing January 4th is first week of the year.
     */
    ISO_8601 = "ISO_8601",
    /**
     * Official calendar week numbering in much of the Middle East (Middle Eastern calendar):
     * 	Saturday is first day of the week, the week containing January 1st is first week of the year.
     */
    MiddleEastern = "MiddleEastern",
    /**
     * Official calendar week numbering in the United States, Canada, Brazil, Israel, Japan, and other countries
     * (Western traditional calendar):
     * 	Sunday is first day of the week, the week containing January 1st is first week of the year.
     */
    WesternTraditional = "WesternTraditional",
  }
  export default CalendarWeekNumbering;
}

declare module "sap/base/i18n/Formatting" {
  import CalendarType from "sap/base/i18n/date/CalendarType";

  import CalendarWeekNumbering from "sap/base/i18n/date/CalendarWeekNumbering";

  import LanguageTag from "sap/base/i18n/LanguageTag";

  /**
   * Configuration for formatting specific parameters
   *
   * @since 1.120
   */
  interface Formatting {
    /**
     * Adds custom currencies. There is a special currency code named "DEFAULT" that is optional. In case it
     * is set it is used for all currencies not contained in the list, otherwise currency digits as defined
     * by the CLDR are used as a fallback.
     *
     * **Note:** Adding custom currencies affects all applications running with the current UI5 core instance.
     *
     * @since 1.120
     */
    addCustomCurrencies(
      /**
       * A map with the currency code as key and a custom currency definition as value; already existing custom
       * currencies are replaced, new ones are added; the custom currency code must contain at least one non-digit
       * character, so that the currency part can be distinguished from the amount part; see {@link #.getCustomCurrencies Formatting.getCustomCurrencies }
       * for an example
       */
      mCurrencies?: Record<string, CustomCurrency>
    ): void;
    /**
     * Adds custom units.
     *
     * **Note:** Adding custom units affects all applications running with the current UI5 core instance.
     *
     * @since 1.123
     */
    addCustomUnits(
      /**
       * A map with the unit code as key and a custom unit definition as value; already existing custom units
       * are replaced, new ones are added; see {@link #.getCustomUnits Formatting.getCustomUnits} for an example
       */
      mUnits: Record<string, CustomUnit>
    ): void;
    /**
     * Attaches the `fnFunction` event handler to the {@link #event:change change} event of `module:sap/base/i18n/Formatting`.
     *
     * @since 1.120
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Formatting$ChangeEvent) => void
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `module:sap/base/i18n/Formatting`.
     *
     * @since 1.120
     */
    detachChange(
      /**
       * Function to be called when the event occurs
       */
      fnFunction: (p1: Formatting$ChangeEvent) => void
    ): void;
    /**
     * Returns the currently set ABAP date format (its id) or undefined if none has been set.
     *
     * @since 1.120
     *
     * @returns ID of the ABAP date format, if not set or set to `""`, `undefined` will be returned
     */
    getABAPDateFormat():
      | "1"
      | "2"
      | "3"
      | "4"
      | "5"
      | "6"
      | "7"
      | "8"
      | "9"
      | "A"
      | "B"
      | "C"
      | undefined;
    /**
     * Returns the currently set ABAP number format (its id) or undefined if none has been set.
     *
     * @since 1.120
     *
     * @returns ID of the ABAP number format, if not set or set to `""`, `undefined` will be returned
     */
    getABAPNumberFormat(): " " | "X" | "Y" | undefined;
    /**
     * Returns the currently set ABAP time format (its id) or undefined if none has been set.
     *
     * @since 1.120
     *
     * @returns ID of the ABAP date format, if not set or set to `""`, `undefined` will be returned
     */
    getABAPTimeFormat(): "0" | "1" | "2" | "3" | "4" | undefined;
    /**
     * Returns the calendar type which is being used in locale dependent functionality.
     *
     * When it's explicitly set by calling `setCalendarType`, the set calendar type is returned. Otherwise,
     * the calendar type is determined by checking the format settings and current locale.
     *
     * @since 1.120
     *
     * @returns the current calendar type, e.g. `Gregorian`
     */
    getCalendarType(): CalendarType;
    /**
     * Returns the calendar week numbering algorithm used to determine the first day of the week and the first
     * calendar week of the year, see {@link module:sap/base/i18n/date/CalendarWeekNumbering CalendarWeekNumbering}.
     *
     * @since 1.120
     *
     * @returns The calendar week numbering algorithm
     */
    getCalendarWeekNumbering(): CalendarWeekNumbering;
    /**
     * Gets the custom currencies that have been set via {@link #.addCustomCurrencies Formatting.addCustomCurrencies }
     * or {@link #.setCustomCurrencies Formatting.setCustomCurrencies}. There is a special currency code named
     * "DEFAULT" that is optional. If it is set it is used for all currencies not contained in the list, otherwise
     * currency digits as defined by the CLDR are used as a fallback.
     *
     * @since 1.120
     *
     * @returns A map with the currency code as key and a custom currency definition containing the number of
     * decimals as value; or `undefined` if there are no custom currencies
     */
    getCustomCurrencies(): Record<string, CustomCurrency> | undefined;
    /**
     * Returns the currently set customizing data for Islamic calendar support.
     *
     * See: {@link module:sap/base/i18n/Formatting.CustomIslamicCalendarData}
     *
     * @since 1.120
     *
     * @returns Returns an array that contains the customizing data. Each element in the array has properties:
     * dateFormat, islamicMonthStart, gregDate. For details, please see {@link #.setCustomIslamicCalendarData}
     */
    getCustomIslamicCalendarData(): CustomIslamicCalendarData[] | undefined;
    /**
     * Gets the custom units that have been set via {@link #.addCustomUnits Formatting.addCustomUnits} or {@link #.setCustomUnits Formatting.setCustomUnits}.
     *
     * @since 1.123
     *
     * @returns A map with the unit code as key and a custom unit definition containing a display name and different
     * unit patterns as value; or `undefined` if there are no custom units
     */
    getCustomUnits(): Record<string, CustomUnit> | undefined;
    /**
     * Returns the currently set date pattern or undefined if no pattern has been defined.
     *
     * @since 1.120
     *
     * @returns The resulting date pattern
     */
    getDatePattern(
      /**
       * The date style (short, medium, long or full)
       */
      sStyle: "short" | "medium" | "long" | "full"
    ): string;
    /**
     * Returns the LanguageTag to be used for formatting.
     *
     * If no such LanguageTag has been defined, this method falls back to the language, see {@link module:sap/base/i18n/Localization.getLanguage Localization.getLanguage()}.
     *
     * If any user preferences for date, time or number formatting have been set, and if no format LanguageTag
     * has been specified, then a special private use subtag is added to the LanguageTag, indicating to the
     * framework that these user preferences should be applied.
     *
     * @since 1.120
     *
     * @returns the format LanguageTag
     */
    getLanguageTag(): LanguageTag;
    /**
     * Returns the currently set number symbol of the given type or undefined if no symbol has been defined.
     *
     * @since 1.120
     *
     * @returns A non-numerical symbol used as part of a number for the given type, e.g. for locale de_DE:
     *
     * 	 - "group": "." (grouping separator)
     * 	 - "decimal": "," (decimal separator)
     * 	 - "plusSign": "+" (plus sign)
     * 	 - "minusSign": "-" (minus sign)
     */
    getNumberSymbol(
      /**
       * the type of symbol
       */
      sType: "group" | "decimal" | "plusSign" | "minusSign"
    ): string;
    /**
     * Returns the currently set time pattern or undefined if no pattern has been defined.
     *
     * @since 1.120
     *
     * @returns The resulting time pattern
     */
    getTimePattern(
      /**
       * The time style (short, medium, long or full)
       */
      sStyle: "short" | "medium" | "long" | "full"
    ): string;
    /**
     * Returns current trailingCurrencyCode configuration for new NumberFormatter instances
     *
     * @since 1.120
     *
     * @returns Whether currency codes shall always be placed after the numeric value
     */
    getTrailingCurrencyCode(): boolean;
    /**
     * Allows to specify one of the ABAP date formats.
     *
     * This method modifies the date patterns for 'short' and 'medium' style with the corresponding ABAP format.
     * When called with a null or undefined format id, any previously applied format will be removed.
     *
     * After changing the date format, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * @since 1.120
     */
    setABAPDateFormat(
      /**
       * ID of the ABAP date format, `""` will reset the date patterns for 'short' and 'medium' style to the locale-specific
       * ones.
       */
      sFormatId?:
        | ""
        | "1"
        | "2"
        | "3"
        | "4"
        | "5"
        | "6"
        | "7"
        | "8"
        | "9"
        | "A"
        | "B"
        | "C"
    ): void;
    /**
     * Allows to specify one of the ABAP number format.
     *
     * This method will modify the 'group' and 'decimal' symbols. When called with a null or undefined format
     * id, any previously applied format will be removed.
     *
     * After changing the number format, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * @since 1.120
     */
    setABAPNumberFormat(
      /**
       * ID of the ABAP number format set, `""` will reset the 'group' and 'decimal' symbols to the locale-specific
       * ones.
       */
      sFormatId?: "" | " " | "X" | "Y"
    ): void;
    /**
     * Allows to specify one of the ABAP time formats.
     *
     * This method sets the time patterns for 'short' and 'medium' style to the corresponding ABAP formats and
     * sets the day period texts to "AM"/"PM" or "am"/"pm" respectively. When called with a null or undefined
     * format id, any previously applied format will be removed.
     *
     * After changing the time format, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * @since 1.120
     */
    setABAPTimeFormat(
      /**
       * ID of the ABAP time format, `""` will reset the time patterns for 'short' and 'medium' style and the
       * day period texts to the locale-specific ones.
       */
      sFormatId?: "" | "0" | "1" | "2" | "3" | "4"
    ): void;
    /**
     * Sets the new calendar type to be used from now on in locale dependent functionality (for example, formatting,
     * translation texts, etc.).
     *
     * @since 1.120
     */
    setCalendarType(
      /**
       * the new calendar type. Set it with null to clear the calendar type and the calendar type is calculated
       * based on the format settings and current locale.
       */
      sCalendarType: (CalendarType | keyof typeof CalendarType) | null
    ): void;
    /**
     * Sets the calendar week numbering algorithm which is used to determine the first day of the week and the
     * first calendar week of the year, see {@link module:sap/base/i18n/date/CalendarWeekNumbering CalendarWeekNumbering}.
     *
     * @since 1.120
     */
    setCalendarWeekNumbering(
      /**
       * The calendar week numbering algorithm
       */
      sCalendarWeekNumbering:
        | CalendarWeekNumbering
        | keyof typeof CalendarWeekNumbering
    ): void;
    /**
     * Replaces existing custom currencies by the given custom currencies. There is a special currency code
     * named "DEFAULT" that is optional. In case it is set, it is used for all currencies not contained in the
     * list, otherwise currency digits as defined by the CLDR are used as a fallback.
     *
     * **Note:** Setting custom units affects all applications running with the current UI5 core instance.
     * See:
     * 	{@link module:sap/base/i18n/Formatting.addCustomCurrencies Formatting.addCustomCurrencies}
     *
     * @since 1.120
     */
    setCustomCurrencies(
      /**
       * A map with the currency code as key and a custom currency definition as value; the custom currency code
       * must contain at least one non-digit character, so that the currency part can be distinguished from the
       * amount part; `mCurrencies` replaces the current custom currencies; if not given, all custom currencies
       * are deleted; see {@link #.getCustomCurrencies Formatting.getCustomCurrencies} for an example
       */
      mCurrencies?: Record<string, CustomCurrency>
    ): void;
    /**
     * Allows to specify the customizing data for Islamic calendar support
     *
     * See: {@link module:sap/base/i18n/Formatting.CustomIslamicCalendarData}
     *
     * @since 1.120
     */
    setCustomIslamicCalendarData(
      /**
       * Contains the customizing data for the support of Islamic calendar. One JSON object in the array represents
       * one row of data from Table TISLCAL
       */
      aCustomCalendarData: CustomIslamicCalendarData[]
    ): void;
    /**
     * Replaces existing custom units by the given custom units.
     *
     * **Note:** Setting custom units affects all applications running with the current UI5 core instance.
     * See:
     * 	{@link module:sap/base/i18n/Formatting.addCustomUnits Formatting.addCustomUnits}
     *
     * @since 1.123
     */
    setCustomUnits(
      /**
       * A map with the unit code as key and a custom unit definition as value; `mUnits` replaces the current
       * custom units; if not given, all custom units are deleted; see {@link #.getCustomUnits Formatting.getCustomUnits }
       * for an example
       */
      mUnits?: Record<string, CustomUnit>
    ): void;
    /**
     * Defines the preferred format pattern for the given date format style.
     *
     * Calling this method with a null or undefined pattern removes a previously set pattern.
     *
     * If a pattern is defined, it will be preferred over patterns derived from the current locale.
     *
     * See class {@link sap.ui.core.format.DateFormat DateFormat} for details about the pattern syntax.
     *
     * After changing the date pattern, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * @since 1.120
     */
    setDatePattern(
      /**
       * must be one of short, medium, long or full.
       */
      sStyle: "short" | "medium" | "long" | "full",
      /**
       * the format pattern to be used in LDML syntax.
       */
      sPattern: string
    ): void;
    /**
     * Sets a new language tag to be used from now on for retrieving language specific formatters. Modifying
     * this setting does not have an impact on the retrieval of translated texts!
     *
     * Can either be set to a concrete value (a BCP47 or Java locale compliant language tag) or to `null`. When
     * set to `null` (default value) then locale specific formatters are retrieved for the current language.
     *
     * After changing the format locale, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * **Note**: When a language tag is set, it has higher priority than a number, date or time format defined
     * with a call to `setABAPNumberFormat`, `setABAPDateFormat` or `setABAPTimeFormat`.
     *
     * **Note**: See documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for restrictions.
     *
     * @since 1.120
     */
    setLanguageTag(
      /**
       * the new BCP47 compliant language tag; case doesn't matter and underscores can be used instead of dashes
       * to separate components (compatibility with Java Locale IDs)
       */
      vLanguageTag: string | LanguageTag | null
    ): void;
    /**
     * Defines the string to be used for the given number symbol.
     *
     * Calling this method with a null or undefined symbol removes a previously set symbol string. Note that
     * an empty string is explicitly allowed.
     *
     * If a symbol is defined, it will be preferred over symbols derived from the current locale.
     *
     * See class {@link sap.ui.core.format.NumberFormat NumberFormat} for details about the symbols.
     *
     * After changing the number symbol, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * @since 1.120
     */
    setNumberSymbol(
      /**
       * the type of symbol
       */
      sType: "group" | "decimal" | "plusSign" | "minusSign",
      /**
       * will be used to represent the given symbol type
       */
      sSymbol: string
    ): void;
    /**
     * Defines the preferred format pattern for the given time format style.
     *
     * Calling this method with a null or undefined pattern removes a previously set pattern.
     *
     * If a pattern is defined, it will be preferred over patterns derived from the current locale.
     *
     * See class {@link sap.ui.core.format.DateFormat DateFormat} for details about the pattern syntax.
     *
     * After changing the time pattern, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage() }
     * for details and restrictions.
     *
     * @since 1.120
     */
    setTimePattern(
      /**
       * must be one of short, medium, long or full.
       */
      sStyle: "short" | "medium" | "long" | "full",
      /**
       * the format pattern to be used in LDML syntax.
       */
      sPattern: string
    ): void;
    /**
     * Define whether the NumberFormatter shall always place the currency code after the numeric value, with
     * the only exception of right-to-left locales, where the currency code shall be placed before the numeric
     * value. Default configuration setting is `true`.
     *
     * When set to `false` the placement of the currency code is done dynamically, depending on the configured
     * locale using data provided by the Unicode Common Locale Data Repository (CLDR).
     *
     * Each currency instance ({@link sap.ui.core.format.NumberFormat.getCurrencyInstance NumberFormat.getCurrencyInstance})
     * will be created with this setting unless overwritten on instance level.
     *
     * @since 1.120
     */
    setTrailingCurrencyCode(
      /**
       * Whether currency codes shall always be placed after the numeric value
       */
      bTrailingCurrencyCode: boolean
    ): void;
  }
  const Formatting: Formatting;
  export default Formatting;

  /**
   * Definition of a custom currency.
   */
  export type CustomCurrency = {
    /**
     * The number of decimal digits to be used for the currency
     */
    digits: int;
  };

  /**
   * Customizing data for the support of Islamic calendar. Represents one row of data from Table TISLCAL.
   */
  export type CustomIslamicCalendarData = {
    /**
     * The date format. Column DATFM in TISLCAL.
     */
    dateFormat: "A" | "B";
    /**
     * The Islamic date in format: 'yyyyMMdd'. Column ISLMONTHSTART in TISLCAL.
     */
    islamicMonthStart: string;
    /**
     * The corresponding Gregorian date format: 'yyyyMMdd'. Column GREGDATE in TISLCAL.
     */
    gregDate: string;
  };

  /**
   * Definition of a custom unit.
   * See:
   * 	{@link sap.ui.core.LocaleData#getPluralCategories}
   */
  export type CustomUnit = {
    /**
     * The unit's display name
     */
    displayName: string;
    /**
     * The unit pattern for the plural form "zero"; `{0}` in the pattern is replaced by the number
     */
    "unitPattern-count-zero"?: string;
    /**
     * The unit pattern for the plural form "one"; `{0}` in the pattern is replaced by the number
     */
    "unitPattern-count-one"?: string;
    /**
     * The unit pattern for the plural form "two"; `{0}` in the pattern is replaced by the number
     */
    "unitPattern-count-two"?: string;
    /**
     * The unit pattern for the plural form "few"; `{0}` in the pattern is replaced by the number
     */
    "unitPattern-count-few"?: string;
    /**
     * The unit pattern for the plural form "many"; `{0}` in the pattern is replaced by the number
     */
    "unitPattern-count-many"?: string;
    /**
     * The unit pattern for all other numbers which do not match the plural forms of the other given patterns;
     * `{0}` in the pattern is replaced by the number
     */
    "unitPattern-count-other": string;
  };

  /**
   * The formatting change event. Contains only the parameters which were changed.
   *
   * The list below shows the possible combinations of parameters available as part of the change event.
   *
   *
   * 	 - {@link module:sap/base/i18n/Formatting.setLanguageTag Formatting.setLanguageTag}:
   * 	`languageTag`
   * 	 - {@link module:sap/base/i18n/Formatting.setCustomIslamicCalendarData Formatting.setCustomIslamicCalendarData}:
   *
   * 	`customIslamicCalendarData`
   * 	 - {@link module:sap/base/i18n/Formatting.setCalendarWeekNumbering Formatting.setCalendarWeekNumbering}:
   *
   * 	`calendarWeekNumbering`
   * 	 - {@link module:sap/base/i18n/Formatting.setCalendarType Formatting.setCalendarType}:
   * 	`calendarType`
   * 	 - {@link module:sap/base/i18n/Formatting.addCustomCurrencies Formatting.addCustomCurrencies} / {@link module:sap/base/i18n/Formatting.setCustomCurrencies Formatting.setCustomCurrencies}:
   *
   * 	`currency`
   * 	 - {@link module:sap/base/i18n/Formatting.setABAPDateFormat Formatting.setABAPDateFormat} (all parameters
   *     listed below):
   * 	`ABAPDateFormat`
   * 	 - `"dateFormats-short"`
   * 	 - `"dateFormats-medium"`
   * 	 - {@link module:sap/base/i18n/Formatting.setABAPTimeFormat Formatting.setABAPTimeFormat} (all parameters
   *     listed below):
   * 	`ABAPTimeFormat`
   * 	 - `"timeFormats-short"`
   * 	 - `"timeFormats-medium"`
   * 	 - `"dayPeriods-format-abbreviated"`
   * 	 - {@link module:sap/base/i18n/Formatting.setABAPNumberFormat Formatting.setABAPNumberFormat} (all parameters
   *     listed below):
   * 	`ABAPNumberFormat`
   * 	 - `"symbols-latn-group"`
   * 	 - `"symbols-latn-decimal"`
   * 	 - {@link module:sap/base/i18n/Formatting.setDatePattern Formatting.setDatePattern} (one of the parameters
   *     listed below):
   * 	`"dateFormats-short"`
   * 	 - `"dateFormats-medium"`
   * 	 - `"dateFormats-long"`
   * 	 - `"dateFormats-full"`
   * 	 - {@link module:sap/base/i18n/Formatting.setTimePattern Formatting.setTimePattern} (one of the parameters
   *     listed below):
   * 	`"timeFormats-short"`
   * 	 - `"timeFormats-medium"`
   * 	 - `"timeFormats-long"`
   * 	 - `"timeFormats-full"`
   * 	 - {@link module:sap/base/i18n/Formatting.setNumberSymbol Formatting.setNumberSymbol} (one of the parameters
   *     listed below):
   * 	`"symbols-latn-group"`
   * 	 - `"symbols-latn-decimal"`
   * 	 - `"symbols-latn-plusSign"`
   * 	 - `"symbols-latn-minusSign"`
   *
   * @since 1.120
   */
  export type Formatting$ChangeEvent = {
    /**
     * The formatting language tag.
     */
    languageTag?: string;
    /**
     * The ABAP date format.
     */
    ABAPDateFormat?: string;
    /**
     * The ABAP time format.
     */
    ABAPTimeFormat?: string;
    /**
     * The ABAP number format.
     */
    ABAPNumberFormat?: string;
    /**
     * The legacy date calendar customizing.
     */
    legacyDateCalendarCustomizing?: object[];
    /**
     * The calendar week numbering.
     */
    calendarWeekNumbering?: object;
    /**
     * The calendar type.
     */
    calendarType?: object;
    /**
     * The short date format.
     */
    "dateFormats-short"?: string;
    /**
     * The medium date format.
     */
    "dateFormats-medium"?: string;
    /**
     * The long date format.
     */
    "dateFormats-long"?: string;
    /**
     * The full date format.
     */
    "dateFormats-full"?: string;
    /**
     * The short time format.
     */
    "timeFormats-short"?: string;
    /**
     * The medium time format.
     */
    "timeFormats-medium"?: string;
    /**
     * The long time format.
     */
    "timeFormats-long"?: string;
    /**
     * The full time format.
     */
    "timeFormats-full"?: string;
    /**
     * The latin symbols group.
     */
    "symbols-latn-group"?: string;
    /**
     * The latin symbols decimal.
     */
    "symbols-latn-decimal"?: string;
    /**
     * The latin symbols plusSign.
     */
    "symbols-latn-plusSign"?: string;
    /**
     * The latin symbols minusSign.
     */
    "symbols-latn-minusSign"?: string;
    /**
     * The currency.
     */
    currency?: Record<string, string>;
    /**
     * The abbreviated day periods format.
     */
    "dayPeriods-format-abbreviated"?: string[];
  };
}

declare module "sap/base/i18n/LanguageTag" {
  /**
   * Creates an LanguageTag instance. LanguageTag represents a BCP-47 language tag, consisting of a language,
   * script, region, variants, extensions and private use section.
   */
  export default class LanguageTag {
    constructor(
      /**
       * the language tag identifier, in format en-US or en_US.
       */
      sLanguageTag: string
    );
    /**
     * Get the extension as a single string or `null`.
     *
     * The extension always consists of a singleton character (not 'x'), a dash '-' and one or more extension
     * token, each separated again with a dash.
     */
    extension: string | null;

    /**
     * Get the extensions as an array of tokens.
     *
     * The leading singleton and the separating dashes are not part of the result. If there is no extensions
     * section in the language tag, an empty array is returned.
     */
    extensionSubtags: string[];

    /**
     * Get the language.
     *
     * Note that the case might differ from the original script tag (Lower case is enforced as recommended by
     * BCP47/ISO639).
     */
    language: string;

    /**
     * Get the region or `null` if none was specified.
     *
     * Note that the case might differ from the original script tag (Upper case is enforced as recommended by
     * BCP47/ISO3166-1).
     */
    region: string;

    /**
     * Get the script or `null` if none was specified.
     *
     * Note that the case might differ from the original language tag (Upper case first letter and lower case
     * reminder enforced as recommended by BCP47/ISO15924)
     */
    script: string | null;

    /**
     * Get the variants as a single string or `null`.
     *
     * Multiple variants are separated by a dash '-'.
     */
    variant: string | null;

    /**
     * Get the variants as an array of individual variants.
     *
     * The separating dashes are not part of the result. If there is no variant section in the language tag,
     * an empty array is returned.
     */
    variantSubtags: string[];
  }
}

declare module "sap/base/i18n/Localization" {
  import LanguageTag from "sap/base/i18n/LanguageTag";

  /**
   * Configuration for localization specific parameters
   *
   * @since 1.118
   */
  interface Localization {
    /**
     * Attaches the `fnFunction` event handler to the {@link #event:change change} event of `module:sap/base/i18n/Localization`.
     *
     * @since 1.120.0
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Localization$ChangeEvent) => void
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `module:sap/base/i18n/Localization`.
     *
     * @since 1.120.0
     */
    detachChange(
      /**
       * Function to be called when the event occurs
       */
      fnFunction: (p1: Localization$ChangeEvent) => void
    ): void;
    /**
     * Returns the list of active terminologies defined via the Configuration.
     *
     * @since 1.119.0
     *
     * @returns if no active terminologies are set, the default value `undefined` is returned.
     */
    getActiveTerminologies(): string[] | undefined;
    /**
     * Returns a string that identifies the current language.
     *
     * The value returned by config method in most cases corresponds to the exact value that has been configured
     * by the user or application or that has been determined from the user agent settings. It has not been
     * normalized, but has been validated against a relaxed version of {@link http://www.ietf.org/rfc/bcp/bcp47.txt BCP47},
     * allowing underscores ('_') instead of the suggested dashes ('-') and not taking the case of letters into
     * account.
     *
     * The exceptions mentioned above affect languages that have been specified via the URL parameter `sap-language`.
     * That parameter by definition represents an SAP logon language code ('ABAP language'). Most but not all
     * of these language codes are valid ISO639 two-letter languages and as such are valid BCP47 language tags.
     * For better BCP47 compliance, the framework maps the following non-BCP47 SAP logon codes to a BCP47 substitute:
     *
     * ```javascript
     *
     *    "ZH"  -->  "zh-Hans"         // script 'Hans' added to distinguish it from zh-Hant
     *    "ZF"  -->  "zh-Hant"         // ZF is not a valid ISO639 code, use the compliant language + script 'Hant'
     *    "1Q"  -->  "en-US-x-saptrc"  // special language code for supportability (tracing),
     *                                    represented as en-US with a private extension
     *    "2Q"  -->  "en-US-x-sappsd"  // special language code for supportability (pseudo translation),
     *                                    represented as en-US with a private extension
     *    "3Q"  -->  "en-US-x-saprigi" // special language code for the Rigi pseudo language,
     *                                    represented as en-US with a private extension
     * ```
     *
     *
     * Call {@link module:sap/base/i18n/Localization.getLanguageTag getLanguageTag} to get a {@link module:sap/base/i18n/LanguageTag LanguageTag }
     * object matching the language. For a normalized BCP47 tag, call {@link module:sap/base/i18n/LanguageTag.toString toString() }
     * on the returned `LanguageTag`
     *
     * @since 1.120.0
     *
     * @returns Language string as configured
     */
    getLanguage(): string;
    /**
     * Returns a LanguageTag object for the current language.
     *
     * The LanguageTag is derived from {@link module:sap/base/i18n/Localization.getLanguage Localization.getLanguage}.
     *
     * @since 1.120.0
     *
     * @returns The LanguageTag
     */
    getLanguageTag(): LanguageTag;
    /**
     * Returns whether the page uses the RTL text direction.
     *
     * If no mode has been explicitly set (neither `true` nor `false`), the mode is derived from the current
     * language setting.
     *
     * @since 1.120.0
     *
     * @returns whether the page uses the RTL text direction
     */
    getRTL(): boolean;
    /**
     * Returns an SAP logon language for the current language.
     *
     * It will be returned in uppercase. e.g. "EN", "DE"
     *
     * @since 1.120.0
     *
     * @returns The SAP logon language code for the current language
     */
    getSAPLogonLanguage(): string;
    /**
     * Retrieves the configured IANA timezone ID.
     *
     * @since 1.120.0
     *
     * @returns The configured IANA timezone ID, e.g. "America/New_York"
     */
    getTimezone(): string;
    /**
     * Sets a new language to be used from now on for language/region dependent functionality (e.g. formatting,
     * data types, translated texts, ...).
     *
     * When the language can't be interpreted as a BCP47 language (using the relaxed syntax described in {@link #getLanguage},
     * an error will be thrown.
     *
     * When the language has changed, the Localization will fire its {@link module:sap/base/i18n/Localization.change change }
     * event.
     *
     * Restrictions:
     *
     * The framework **does not** guarantee that already created, language dependent objects will be updated
     * by config call. It therefore remains best practice for applications to switch the language early, e.g.
     * before any language dependent objects are created. Applications that need to support more dynamic changes
     * of the language should listen to the `localizationChanged` event and adapt all language dependent objects
     * that they use (e.g. by rebuilding their UI).
     *
     * Currently, the framework notifies the following objects about a change of the localization settings before
     * it fires the `localizationChanged` event:
     *
     *
     * 	 - date and number data types that are used in property bindings or composite bindings in existing Elements,
     *     Controls, UIAreas or Components
     * 	 - ResourceModels currently assigned to the Core, a UIArea, Component, Element or Control
     * 	 - Elements or Controls that implement the `onLocalizationChanged` hook
     *
     * It furthermore derives the RTL mode from the new language, if no explicit RTL mode has been set. If the
     * RTL mode changes, the following additional actions will be taken:
     *
     *
     * 	 - the URLs of already loaded library theme files will be changed
     * 	 - the `dir` attribute of the page will be changed to reflect the new mode.
     * 	 - all UIAreas will be invalidated (which results in a rendering of the whole UI5 UI)
     *
     * config method does not accept SAP language codes for `sLanguage`. Instead, a second parameter `sSAPLogonLanguage`
     * can be provided with an SAP language code corresponding to the given language. A given value will be
     * returned by the {@link module:sap/base/i18n/Localization.getSAPLogonLanguage getSAPLogonLanguage} method.
     * It is up to the caller to provide a consistent pair of BCP47 language and SAP language code. The SAP
     * language code is only checked to be of length 2 and must consist of letters or digits only.
     *
     * **Note**: When using config method please take note of and respect the above mentioned restrictions.
     * See:
     * 	http://scn.sap.com/docs/DOC-14377
     *
     * @since 1.120.0
     */
    setLanguage(
      /**
       * the new language as a BCP47 compliant language tag; case doesn't matter and underscores can be used instead
       * of dashes to separate components (compatibility with Java Locale IDs)
       */
      sLanguage: string,
      /**
       * SAP language code that corresponds to the `sLanguage`; if a value is specified, future calls to `getSAPLogonLanguage`
       * will return that value; if no value is specified, the framework will use the ISO639 language part of
       * `sLanguage` as SAP Logon language.
       */
      sSAPLogonLanguage?: string
    ): void;
    /**
     * Sets the character orientation mode to be used from now on.
     *
     * Can either be set to a concrete value (true meaning right-to-left, false meaning left-to-right) or to
     * `null` which means that the character orientation mode should be derived from the current language (incl.
     * region) setting.
     *
     * After changing the character orientation mode, the framework tries to update localization specific parts
     * of the UI. See the documentation of {@link module:sap/base/i18n/Localization.setLanguage setLanguage }
     * for details and restrictions.
     *
     * **Note**: See documentation of {@link module:sap/base/i18n/Localization.setLanguage setLanguage} for
     * restrictions.
     *
     * @since 1.120.0
     */
    setRTL(
      /**
       * new character orientation mode or `null`
       */
      bRTL: boolean | null
    ): void;
    /**
     * Sets the timezone such that all date and time based calculations use config timezone.
     *
     * **Important:** It is strongly recommended to only use config API at the earliest point of time while
     * initializing a UI5 app. A later adjustment of the time zone should be avoided. It can lead to unexpected
     * data inconsistencies in a running application, because date objects could still be related to a previously
     * configured time zone. Instead, the app should be completely restarted with the new time zone. For more
     * information, see {@link https://ui5.sap.com/#/topic/6c9e61dc157a40c19460660ece8368bc Dates, Times, Timestamps, and Time Zones}.
     *
     * When the timezone has changed, the Localization will fire its {@link #event:change change} event.
     *
     * @since 1.120.0
     */
    setTimezone(
      /**
       * IANA timezone ID, e.g. "America/New_York". Use `null` to reset the timezone to the browser's local timezone.
       * An invalid IANA timezone ID will fall back to the browser's timezone.
       */
      sTimezone?: string | null
    ): void;
  }
  const Localization: Localization;
  export default Localization;

  /**
   * The localization change event. Contains only the parameters which were changed.
   *
   * The list below shows the possible combinations of parameters available as part of the change event.
   *
   *
   * 	 - {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage}:
   * 	`language`
   * 	 - `rtl?` (only if language change also changed RTL)
   * 	 - {@link module:sap/base/i18n/Localization.setRTL Localization.setRTL}:
   * 	`rtl`
   * 	 - {@link module:sap/base/i18n/Localization.setTimezone Localization.setTimezone}:
   * 	`timezone`
   *
   * @since 1.120.0
   */
  export type Localization$ChangeEvent = {
    /**
     * The newly set language.
     */
    language?: string;
    /**
     * Whether the page uses the RTL text direction.
     */
    rtl?: boolean;
    /**
     * The newly set timezone.
     */
    timezone?: string;
  };
}

declare module "sap/base/i18n/ResourceBundle" {
  /**
   * Contains locale-specific texts.
   *
   * If you need a locale-specific text within your application, you can use the resource bundle to load the
   * locale-specific file from the server and access the texts of it.
   *
   * Use {@link module:sap/base/i18n/ResourceBundle.create} to create an instance of sap/base/i18n/ResourceBundle
   * (.properties without any locale information, e.g. "mybundle.properties"), and optionally a locale. The
   * locale is defined as a string of the language and an optional country code separated by underscore (e.g.
   * "en_GB" or "fr"). If no locale is passed, the default locale is "en" if the SAPUI5 framework is not available.
   * Otherwise the default locale is taken from the SAPUI5 configuration.
   *
   * With the getText() method of the resource bundle, a locale-specific string value for a given key will
   * be returned.
   *
   * With the given locale, the resource bundle requests the locale-specific properties file (e.g. "mybundle_fr_FR.properties").
   * If no file is found for the requested locale or if the file does not contain a text for the given key,
   * a sequence of fallback locales is tried one by one. First, if the locale contains a region information
   * (fr_FR), then the locale without the region is tried (fr). If that also can't be found or doesn't contain
   * the requested text, a fallback language will be used, if given (defaults to en (English), assuming that
   * most development projects contain at least English texts). If that also fails, the file without locale
   * (base URL of the bundle, often called the 'raw' bundle) is tried.
   *
   * If none of the requested files can be found or none of them contains a text for the given key, then the
   * key itself is returned as text.
   *
   * Exception: Fallback for "zh_HK" is "zh_TW" before "zh".
   *
   * @since 1.58
   */
  export default class ResourceBundle {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Creates and returns a new instance of {@link module:sap/base/i18n/ResourceBundle} using the given URL
     * and locale to determine what to load.
     *
     * Before loading the ResourceBundle, the locale is evaluated with a fallback chain. Sample fallback chain
     * for locale="de-DE" and fallbackLocale="fr_FR" `"de-DE" -> "de" -> "fr_FR" -> "fr" -> raw`
     *
     * Only those locales are considered for loading, which are in the supportedLocales array (if the array
     * is supplied and not empty).
     *
     * Note: The fallbackLocale should be included in the supportedLocales array.
     *
     *
     * @returns A new resource bundle or a Promise on that bundle (in asynchronous case)
     */
    static create(
      /**
       * Parameters used to initialize the resource bundle
       */
      mParams?: {
        /**
         * URL pointing to the base .properties file of a bundle (.properties file without any locale information,
         * e.g. "mybundle.properties") if not provided, `bundleUrl` or `bundleName` can be used; if both are set,
         * `bundleName` wins
         */
        url?: string;
        /**
         * URL pointing to the base .properties file of a bundle (.properties file without any locale information,
         * e.g. "i18n/mybundle.properties")
         */
        bundleUrl?: string;
        /**
         * UI5 module name in dot notation pointing to the base .properties file of a bundle (.properties file without
         * any locale information, e.g. "i18n.mybundle")
         */
        bundleName?: string;
        /**
         * Optional locale (aka 'language tag') to load the texts for. Can either be a BCP47 language tag or a JDK
         * compatible locale string (e.g. "en-GB", "en_GB" or "en"). Defaults to the current session locale if `sap.ui.getCore`
         * is available, otherwise to the provided `fallbackLocale`
         */
        locale?: string;
        /**
         * Whether to include origin information into the returned property values
         */
        includeInfo?: boolean;
        /**
         * List of supported locales (aka 'language tags') to restrict the fallback chain. Each entry in the array
         * can either be a BCP47 language tag or a JDK compatible locale string (e.g. "en-GB", "en_GB" or "en").
         * An empty string (`""`) represents the 'raw' bundle. **Note:** The given language tags can use modern
         * or legacy ISO639 language codes. Whatever language code is used in the list of supported locales will
         * also be used when requesting a file from the server. If the `locale` contains a legacy language code
         * like "iw" and the `supportedLocales` contains [...,"he",...], "he" will be used in the URL. This mapping
         * works in both directions.
         */
        supportedLocales?: string[];
        /**
         * A fallback locale to be used after all locales derived from `locale` have been tried, but before the
         * 'raw' bundle is used. Can either be a BCP47 language tag or a JDK compatible locale string (e.g. "en-GB",
         * "en_GB" or "en"). To prevent a generic fallback, use the empty string (`""`). E.g. by providing `fallbackLocale:
         * ""` and `supportedLocales: ["en"]`, only the bundle "en" is requested without any fallback.
         */
        fallbackLocale?: string;
        /**
         * map of terminologies. The key is the terminology identifier and the value is a ResourceBundle terminology
         * configuration. A terminology is a resource bundle configuration for a specific use case (e.g. "oil").
         * It does neither have a `fallbackLocale` nor can it be enhanced with `enhanceWith`.
         */
        terminologies?: Record<string, TerminologyConfiguration>;
        /**
         * The list of active terminologies, e.g. `["oil", "retail"]`. The order in this array represents the lookup
         * order.
         */
        activeTerminologies?: string[];
        /**
         * List of ResourceBundle configurations which enhance the current one. The order of the enhancements is
         * significant, because the lookup checks the last enhancement first. Each enhancement represents a ResourceBundle
         * with limited options ('bundleUrl', 'bundleName', 'terminologies', 'fallbackLocale', 'supportedLocales').
         * Note: supportedLocales and fallbackLocale are inherited from the parent ResourceBundle if not present.
         */
        enhanceWith?: Configuration[];
        /**
         * Whether the first bundle should be loaded asynchronously Note: Fallback bundles loaded by {@link #getText }
         * are always loaded synchronously. **As of version 1.135, synchronous loading is deprecated.** The `async`
         * parameter must have the value `true`.
         */
        async?: boolean;
      }
    ): ResourceBundle | Promise<ResourceBundle>;
    /**
     * Returns a locale-specific string value for the given key sKey.
     *
     * The text is searched in this resource bundle according to the fallback chain described in {@link module:sap/base/i18n/ResourceBundle}.
     * If no text could be found, the key itself is used as text.
     *
     * Placeholders:
     *
     * A text can contain placeholders that will be replaced with concrete values when `getText` is called.
     * The replacement is triggered by the `aArgs` parameter.
     *
     * Whenever this parameter is given, then the text and the arguments are additionally run through the {@link module:sap/base/strings/formatMessage }
     * API to replace placeholders in the text with the corresponding values from the arguments array. The resulting
     * string is returned by `getText`.
     *
     * As the `formatMessage` API imposes some requirements on the input text (regarding curly braces and single
     * apostrophes), text authors need to be aware of the specifics of the `formatMessage` API. Callers of `getText`,
     * on the other side, should only supply `aArgs` when the text has been created with the `formatMessage`
     * API in mind. Otherwise, single apostrophes in the text might be removed unintentionally.
     *
     * When `getText` is called without `aArgs`, the `formatMessage` API is not applied and the transformation
     * reg. placeholders and apostrophes does not happen.
     *
     * For more details on the replacement mechanism refer to {@link module:sap/base/strings/formatMessage}.
     *
     *
     * @returns The value belonging to the key, if found; otherwise the key itself or `undefined` depending
     * on `bIgnoreKeyFallback`.
     */
    getText(
      /**
       * Key to retrieve the text for
       */
      sKey: string,
      /**
       * List of parameter values which should replace the placeholders "{n}" (n is the index) in
       * the found locale-specific string value. Note that the replacement is done whenever `aArgs` is given (not
       * `undefined`), no matter whether the text contains placeholders or not and no matter whether `aArgs` contains
       * a value for n or not.
       */
      aArgs?: any[],
      /**
       * If set, `undefined` is returned instead of the key string, when the key is not found in any bundle or
       * fallback bundle.
       */
      bIgnoreKeyFallback?: boolean
    ): string | undefined;
    /**
     * Checks whether a text for the given key can be found in the first loaded resource bundle or not. Neither
     * the custom resource bundles nor the fallback chain will be processed.
     *
     * This method allows to check for the existence of a text without triggering requests for the fallback
     * locales.
     *
     * When requesting the resource bundle asynchronously this check must only be used after the resource bundle
     * has been loaded.
     *
     *
     * @returns Whether the text has been found in the concrete bundle
     */
    hasText(
      /**
       * Key to check
       */
      sKey: string
    ): boolean;
  }
  /**
   * ResourceBundle Configuration
   *
   * A ResourceBundle Configuration holds information on where to load the ResourceBundle from using the fallback
   * chain and terminologies. The location is retrieved from the `bundleUrl` and `bundleName` parameters The
   * locale used is influenced by the `supportedLocales` and `fallbackLocale` parameters Terminologies of
   * this ResourceBundle are loaded via the `terminologies` parameter
   *
   * Note: If omitted, the supportedLocales and the fallbackLocale are inherited from the parent ResourceBundle
   * Configuration
   */
  export type Configuration = {
    /**
     * URL pointing to the base .properties file of a bundle (.properties file without any locale information,
     * e.g. "i18n/mybundle.properties")
     */
    bundleUrl?: string;
    /**
     * UI5 module name in dot notation pointing to the base .properties file of a bundle (.properties file without
     * any locale information, e.g. "i18n.mybundle")
     */
    bundleName?: string;
    /**
     * List of supported locales (aka 'language tags') to restrict the fallback chain. Each entry in the array
     * can either be a BCP47 language tag or a JDK compatible locale string (e.g. "en-GB", "en_GB" or "en").
     * An empty string (`""`) represents the 'raw' bundle. **Note:** The given language tags can use modern
     * or legacy ISO639 language codes. Whatever language code is used in the list of supported locales will
     * also be used when requesting a file from the server. If the `locale` contains a legacy language code
     * like "iw" and the `supportedLocales` contains [...,"he",...], "he" will be used in the URL. This mapping
     * works in both directions.
     */
    supportedLocales?: string[];
    /**
     * A fallback locale to be used after all locales derived from `locale` have been tried, but before the
     * 'raw' bundle is used. Can either be a BCP47 language tag or a JDK compatible locale string (e.g. "en-GB",
     * "en_GB" or "en"), defaults to "en" (English). To prevent a generic fallback, use the empty string (`""`).
     * E.g. by providing `fallbackLocale: ""` and `supportedLocales: ["en"]`, only the bundle "en" is requested
     * without any fallback.
     */
    fallbackLocale?: string;
    /**
     * An object, mapping a terminology identifier (e.g. "oil") to a `ResourceBundle.TerminologyConfiguration`.
     * A terminology is a resource bundle configuration for a specific use case (e.g. "oil"). It does neither
     * have a `fallbackLocale` nor can it be enhanced with `enhanceWith`.
     */
    terminologies?: Record<string, TerminologyConfiguration>;
  };

  /**
   * ResourceBundle Terminology Configuration
   *
   * Terminologies represent a variant of a ResourceBundle. They can be used to provide domain specific texts,
   * e.g. for industries, e.g. "oil", "retail" or "health". While "oil" could refer to a user as "driller",
   * in "retail" a user could be a "customer" and in "health" a "patient". While "oil" could refer to a duration
   * as "hitch", in "retail" a duration could be a "season" and in "health" an "incubation period".
   *
   * Note: Terminologies do neither support a fallbackLocale nor nested terminologies in their configuration.
   */
  export type TerminologyConfiguration = {
    /**
     * URL pointing to the base .properties file of a bundle (.properties file without any locale information,
     * e.g. "i18n/mybundle.properties")
     */
    bundleUrl?: string;
    /**
     * UI5 module name in dot notation pointing to the base .properties file of a bundle (.properties file without
     * any locale information, e.g. "i18n.mybundle")
     */
    bundleName?: string;
    /**
     * List of supported locales (aka 'language tags') to restrict the fallback chain. Each entry in the array
     * can either be a BCP47 language tag or a JDK compatible locale string (e.g. "en-GB", "en_GB" or "en").
     * An empty string (`""`) represents the 'raw' bundle. **Note:** The given language tags can use modern
     * or legacy ISO639 language codes. Whatever language code is used in the list of supported locales will
     * also be used when requesting a file from the server. If the `locale` contains a legacy language code
     * like "iw" and the `supportedLocales` contains [...,"he",...], "he" will be used in the URL. This mapping
     * works in both directions.
     */
    supportedLocales?: string[];
  };
}

declare module "sap/base/Log" {
  /**
   * A Logging API for JavaScript.
   *
   * Provides methods to manage a client-side log and to create entries in it. Each of the logging methods
   * {@link module:sap/base/Log.debug}, {@link module:sap/base/Log.info}, {@link module:sap/base/Log.warning},
   * {@link module:sap/base/Log.error} and {@link module:sap/base/Log.fatal} creates and records a log entry,
   * containing a timestamp, a log level, a message with details and a component info. The log level will
   * be one of {@link module:sap/base/Log.Level} and equals the name of the concrete logging method.
   *
   * By using the {@link module:sap/base/Log.setLevel} method, consumers can determine the least important
   * log level which should be recorded. Less important entries will be filtered out. (Note that higher numeric
   * values represent less important levels). The initially set level depends on the mode that UI5 is running
   * in. When the optimized sources are executed, the default level will be {@link module:sap/base/Log.Level.ERROR}.
   * For normal (debug sources), the default level is {@link module:sap/base/Log.Level.DEBUG}.
   *
   * All logging methods allow to specify a **component**. These components are simple strings and don't have
   * a special meaning to the UI5 framework. However they can be used to semantically group log entries that
   * belong to the same software component (or feature). There are two APIs that help to manage logging for
   * such a component. With {@link module:sap/base/Log.getLogger}, one can retrieve a logger that automatically
   * adds the given `sComponent` as component parameter to each log entry, if no other component is specified.
   * Typically, JavaScript code will retrieve such a logger once during startup and reuse it for the rest
   * of its lifecycle. Second, the {@link module:sap/base/Log.setLevel}(iLevel, sComponent) method allows
   * to set the log level for a specific component only. This allows a more fine grained control about the
   * created logging entries. {@link module:sap/base/Log.getLevel} allows to retrieve the currently effective
   * log level for a given component.
   *
   * {@link module:sap/base/Log.getLogEntries} returns an array of the currently collected log entries.
   *
   * Furthermore, a listener can be registered to the log. It will be notified whenever a new entry is added
   * to the log. The listener can be used for displaying log entries in a separate page area, or for sending
   * it to some external target (server).
   *
   * @since 1.58
   */
  interface Log {
    /**
     * Enumeration of the configurable log levels that a Logger should persist to the log.
     *
     * Only if the current LogLevel is higher than the level {@link module:sap/base/Log.Level} of the currently
     * added log entry, then this very entry is permanently added to the log. Otherwise it is ignored.
     *
     * This enum is part of the 'sap/base/Log' module export and must be accessed by the property 'Level'.
     */
    Level: typeof Level;

    /**
     * Allows to add a new listener that will be notified for new log entries.
     *
     * The given object must provide method `onLogEntry` and can also be informed about `onDetachFromLog`, `onAttachToLog`
     * and `onDiscardLogEntries`.
     */
    addLogListener(
      /**
       * The new listener object that should be informed
       */
      oListener: Listener
    ): void;
    /**
     * Creates a new debug-level entry in the log with the given message, details and calling component.
     */
    debug(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged with
       * the stack.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Creates a new error-level entry in the log with the given message, details and calling component.
     */
    error(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged together
       * with its stacktrace.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Creates a new fatal-level entry in the log with the given message, details and calling component.
     */
    fatal(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged together
       * with its stacktrace.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Returns the log level currently effective for the given component. If no component is given or when no
     * level has been configured for a given component, the log level for the default component of this logger
     * is returned.
     *
     *
     * @returns The log level for the given component or the default log level
     */
    getLevel(
      /**
       * Name of the component to retrieve the log level for
       */
      sComponent?: string
    ): Level;
    /**
     * Returns the logged entries recorded so far as an array.
     *
     * Log entries are plain JavaScript objects with the following properties
     * 	timestamp {number} point in time when the entry was created level {module:sap/base/Log.Level} LogLevel
     * level of the entry message {string} message text of the entry  The default amount of stored
     * log entries is limited to 3000 entries.
     *
     *
     * @returns an array containing the recorded log entries
     */
    getLogEntries(): Entry[];
    /**
     * Returns a dedicated logger for a component.
     *
     * The logger comes with the same API as the `sap/base/Log` module:
     * 	`#fatal` - see: {@link module:sap/base/Log.fatal} `#error` - see: {@link module:sap/base/Log.error }
     * `#warning` - see: {@link module:sap/base/Log.warning} `#info` - see: {@link module:sap/base/Log.info }
     * `#debug` - see: {@link module:sap/base/Log.debug} `#trace` - see: {@link module:sap/base/Log.trace }
     * `#setLevel` - see: {@link module:sap/base/Log.setLevel} `#getLevel` - see: {@link module:sap/base/Log.getLevel }
     * `#isLoggable` - see: {@link module:sap/base/Log.isLoggable}
     *
     *
     * @returns A logger with a specified component
     */
    getLogger(
      /**
       * Name of the component which should be logged
       */
      sComponent: string,
      /**
       * The default log level
       */
      iDefaultLogLevel?: Level
    ): Logger;
    /**
     * Creates a new info-level entry in the log with the given message, details and calling component.
     */
    info(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged with
       * the stack.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Checks whether logging is enabled for the given log level, depending on the currently effective log level
     * for the given component.
     *
     * If no component is given, the default component of this logger will be taken into account.
     *
     *
     * @returns Whether logging is enabled or not
     */
    isLoggable(
      /**
       * The log level in question
       */
      iLevel?: Level,
      /**
       * Name of the component to check the log level for
       */
      sComponent?: string
    ): boolean;
    /**
     * Allows to remove a registered LogListener.
     */
    removeLogListener(
      /**
       * The listener object that should be removed
       */
      oListener: Listener
    ): void;
    /**
     * Defines the maximum `sap/base/Log.Level` of log entries that will be recorded. Log entries with a higher
     * (less important) log level will be omitted from the log. When a component name is given, the log level
     * will be configured for that component only, otherwise the log level for the default component of this
     * logger is set. For the global logger, the global default level is set.
     *
     * **Note**: Setting a global default log level has no impact on already defined component log levels. They
     * always override the global default log level.
     */
    setLevel(
      /**
       * The new log level
       */
      iLogLevel: Level,
      /**
       * The log component to set the log level for
       */
      sComponent?: string
    ): void;
    /**
     * Creates a new trace-level entry in the log with the given message, details and calling component.
     */
    trace(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged with
       * the stack.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Creates a new warning-level entry in the log with the given message, details and calling component.
     */
    warning(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged together
       * with its stacktrace.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
  }
  const Log: Log;
  export default Log;

  export type Entry = {
    /**
     * The number of milliseconds since the epoch
     */
    timestamp: float;
    /**
     * Time string in format HH:mm:ss:mmmnnn
     */
    time: string;
    /**
     * Date string in format yyyy-MM-dd
     */
    date: string;
    /**
     * The level of the log entry, see {@link module:sap/base/Log.Level}
     */
    level: Level;
    /**
     * The message of the log entry
     */
    message: string;
    /**
     * The detailed information of the log entry
     */
    details: string;
    /**
     * The component that creates the log entry
     */
    component: string;
    /**
     * Callback that returns an additional support object to be logged in support mode.
     */
    supportInfo?: () => any;
  };

  /**
   * Enumeration of the configurable log levels that a Logger should persist to the log.
   *
   * Only if the current LogLevel is higher than the level {@link module:sap/base/Log.Level} of the currently
   * added log entry, then this very entry is permanently added to the log. Otherwise it is ignored.
   *
   * This enum is part of the 'sap/base/Log' module export and must be accessed by the property 'Level'.
   */
  enum Level {
    /**
     * Trace level to log everything.
     */
    ALL = "undefined",
    /**
     * Debug level. Use this for logging information necessary for debugging
     */
    DEBUG = "undefined",
    /**
     * Error level. Use this for logging of erroneous but still recoverable situations
     */
    ERROR = "undefined",
    /**
     * Fatal level. Use this for logging unrecoverable situations
     */
    FATAL = "undefined",
    /**
     * Info level. Use this for logging information of purely informative nature
     */
    INFO = "undefined",
    /**
     * Do not log anything
     */
    NONE = "undefined",
    /**
     * Trace level. Use this for tracing the program flow.
     */
    TRACE = "undefined",
    /**
     * Warning level. Use this for logging unwanted but foreseen situations
     */
    WARNING = "undefined",
  }
  /**
   * Interface to be implemented by a log listener.
   *
   * Typically, a listener will at least implement the {@link #.onLogEntry} method, but in general, all methods
   * are optional.
   */
  export interface Listener {
    __implements__sap_base_Log_Listener: boolean;

    /**
     * The function that is called once the Listener is attached
     */
    onAttachToLog?(
      /**
       * The Log instance where the listener is attached
       */
      oLog: Log
    ): void;
    /**
     * The function that is called once the Listener is detached
     */
    onDetachFromLog?(
      /**
       * The Log instance where the listener is detached
       */
      oLog: Log
    ): void;
    /**
     * The function that is called once log entries are discarded due to the exceed of total log entry amount
     */
    onDiscardLogEntries?(
      /**
       * The discarded log entries
       */
      aDiscardedEntries: Entry[]
    ): void;
    /**
     * The function that is called when a new log entry is created
     */
    onLogEntry?(
      /**
       * The newly created log entry
       */
      oLogEntry: Entry
    ): void;
  }

  /**
   * The logger comes with a subset of the API of the `sap/base/Log` module:
   * 	`#fatal` - see: {@link module:sap/base/Log.fatal} `#error` - see: {@link module:sap/base/Log.error }
   * `#warning` - see: {@link module:sap/base/Log.warning} `#info` - see: {@link module:sap/base/Log.info }
   * `#debug` - see: {@link module:sap/base/Log.debug} `#trace` - see: {@link module:sap/base/Log.trace }
   * `#setLevel` - see: {@link module:sap/base/Log.setLevel} `#getLevel` - see: {@link module:sap/base/Log.getLevel }
   * `#isLoggable` - see: {@link module:sap/base/Log.isLoggable}
   */
  export interface Logger {
    __implements__sap_base_Log_Logger: boolean;

    /**
     * Creates a new debug-level entry in the log with the given message, details and calling component.
     */
    debug(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged with
       * the stack.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Creates a new error-level entry in the log with the given message, details and calling component.
     */
    error(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged together
       * with its stacktrace.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Creates a new fatal-level entry in the log with the given message, details and calling component.
     */
    fatal(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged together
       * with its stacktrace.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Returns the log level currently effective for the given component. If no component is given or when no
     * level has been configured for a given component, the log level for the default component of this logger
     * is returned.
     *
     *
     * @returns The log level for the given component or the default log level
     */
    getLevel(
      /**
       * Name of the component to retrieve the log level for
       */
      sComponent?: string
    ): Level;
    /**
     * Creates a new info-level entry in the log with the given message, details and calling component.
     */
    info(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged with
       * the stack.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Checks whether logging is enabled for the given log level, depending on the currently effective log level
     * for the given component.
     *
     * If no component is given, the default component of this logger will be taken into account.
     *
     *
     * @returns Whether logging is enabled or not
     */
    isLoggable(
      /**
       * The log level in question
       */
      iLevel?: Level,
      /**
       * Name of the component to check the log level for
       */
      sComponent?: string
    ): boolean;
    /**
     * Defines the maximum `sap/base/Log.Level` of log entries that will be recorded. Log entries with a higher
     * (less important) log level will be omitted from the log. When a component name is given, the log level
     * will be configured for that component only, otherwise the log level for the default component of this
     * logger is set. For the global logger, the global default level is set.
     *
     * **Note**: Setting a global default log level has no impact on already defined component log levels. They
     * always override the global default log level.
     */
    setLevel(
      /**
       * The new log level
       */
      iLogLevel: Level,
      /**
       * The log component to set the log level for
       */
      sComponent?: string
    ): void;
    /**
     * Creates a new trace-level entry in the log with the given message, details and calling component.
     */
    trace(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged with
       * the stack.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
    /**
     * Creates a new warning-level entry in the log with the given message, details and calling component.
     */
    warning(
      /**
       * Message text to display
       */
      sMessage: string,
      /**
       * Optional details about the message, might be omitted. Can be an Error object which will be logged together
       * with its stacktrace.
       */
      vDetails?: string | Error,
      /**
       * Name of the component that produced the log entry
       */
      sComponent?: string,
      /**
       * Callback that returns an additional support object to be logged in support mode. This function is only
       * called if support info mode is turned on via the Support Assistant. To avoid negative effects regarding
       * execution times and memory consumption, the returned object should be a simple immutable JSON object
       * with mostly static and stable content.
       */
      fnSupportInfo?: Function
    ): void;
  }
}

declare module "sap/base/security/encodeCSS" {
  /**
   * Encode the string for inclusion into CSS string literals or identifiers.
   *
   * @since 1.58
   *
   * @returns The encoded string
   */
  export default function encodeCSS(
    /**
     * The string to be escaped
     */
    sString: string
  ): string;
}

declare module "sap/base/security/encodeJS" {
  /**
   * Encode the string for inclusion into a JS string literal.
   *
   * @since 1.58
   *
   * @returns The encoded string
   */
  export default function encodeJS(
    /**
     * The string to be escaped
     */
    sString: string
  ): string;
}

declare module "sap/base/security/encodeURL" {
  /**
   * Encode the string for inclusion into a URL parameter.
   *
   * Unescaped characters: alphabetic, decimal digits, -_. (dash, underscore, point)
   *
   * @since 1.58
   *
   * @returns The encoded string
   */
  export default function encodeURL(
    /**
     * The string to be escaped
     */
    sString: string
  ): string;
}

declare module "sap/base/security/encodeURLParameters" {
  /**
   * Encode a map of parameters into a combined URL parameter string.
   *
   * @since 1.58
   *
   * @returns The URL encoded parameter string
   */
  export default function encodeURLParameters(
    /**
     * The map of parameters to encode
     */
    mParams: Record<string, string | number | boolean>
  ): string;
}

declare module "sap/base/security/encodeXML" {
  /**
   * Encode the string for inclusion into XML content/attribute.
   *
   * @since 1.58
   *
   * @returns The encoded string
   */
  export default function encodeXML(
    /**
     * The string to be escaped
     */
    sString: string
  ): string;
}

declare module "sap/base/security/URLListValidator" {
  /**
   * Registry to manage allowed URLs and validate against them.
   *
   * @since 1.85
   */
  interface URLListValidator {
    /**
     * Adds an allowed entry.
     *
     * Note: Adding the first entry to the list of allowed entries will disallow all URLs but the ones matching
     * the newly added entry.
     *
     * **Note**: It is strongly recommended to set a path only in combination with an origin (never set a path
     * alone). There's almost no case where checking only the path of a URL would allow to ensure its validity.
     */
    add(
      /**
       * The protocol of the URL, can be falsy to allow all protocols for an entry e.g. "", "http", "mailto"
       */
      protocol?: string,
      /**
       * The host of the URL, can be falsy to allow all hosts. A wildcard asterisk can be set at the beginning,
       * e.g. "examples.com", "*.example.com"
       */
      host?: string,
      /**
       * The port of the URL, can be falsy to allow all ports, e.g. "", "8080"
       */
      port?: string,
      /**
       * the path of the URL, path of the url, can be falsy to allow all paths. A wildcard asterisk can be set
       * at the end, e.g. "/my-example*", "/my-news"
       */
      path?: string
    ): void;
    /**
     * Clears the allowed entries for URL validation. This makes all URLs allowed.
     */
    clear(): void;
    /**
     * Gets the list of allowed entries.
     *
     *
     * @returns The allowed entries
     */
    entries(): Entry[];
    /**
     * Validates a URL. Check if it's not a script or other security issue.
     *
     * **Note**: It is strongly recommended to validate only absolute URLs. There's almost no case where checking
     * only the path of a URL would allow to ensure its validity. For compatibility reasons, this API cannot
     * automatically resolve URLs relative to `document.baseURI`, but callers should do so. In that case, and
     * when the allow list is not empty, an entry for the origin of `document.baseURI` must be added to the
     * allow list.
     *
     * Details: Splits the given URL into components and checks for allowed characters according to RFC 3986:
     *
     *
     * ```javascript
     *
     * authority     = [ userinfo "@" ] host [ ":" port ]
     * userinfo      = *( unreserved / pct-encoded / sub-delims / ":" )
     * host          = IP-literal / IPv4address / reg-name
     *
     * IP-literal    = "[" ( IPv6address / IPvFuture  ) "]"
     * IPvFuture     = "v" 1*HEXDIG "." 1*( unreserved / sub-delims / ":" )
     * IPv6address   =                            6( h16 ":" ) ls32
     *               /                       "::" 5( h16 ":" ) ls32
     *               / [               h16 ] "::" 4( h16 ":" ) ls32
     *               / [ *1( h16 ":" ) h16 ] "::" 3( h16 ":" ) ls32
     *               / [ *2( h16 ":" ) h16 ] "::" 2( h16 ":" ) ls32
     *               / [ *3( h16 ":" ) h16 ] "::"    h16 ":"   ls32
     *               / [ *4( h16 ":" ) h16 ] "::"              ls32
     *               / [ *5( h16 ":" ) h16 ] "::"              h16
     *               / [ *6( h16 ":" ) h16 ] "::"
     * ls32          = ( h16 ":" h16 ) / IPv4address
     *               ; least-significant 32 bits of address
     * h16           = 1*4HEXDIG
     *               ; 16 bits of address represented in hexadecimal
     *
     * IPv4address   = dec-octet "." dec-octet "." dec-octet "." dec-octet
     * dec-octet     = DIGIT                 ; 0-9
     *               / %x31-39 DIGIT         ; 10-99
     *               / "1" 2DIGIT            ; 100-199
     *               / "2" %x30-34 DIGIT     ; 200-249
     *               / "25" %x30-35          ; 250-255
     *
     * reg-name      = *( unreserved / pct-encoded / sub-delims )
     *
     * pct-encoded   = "%" HEXDIG HEXDIG
     * reserved      = gen-delims / sub-delims
     * gen-delims    = ":" / "/" / "?" / "#" / "[" / "]" / "@"
     * sub-delims    = "!" / "$" / "&" / "'" / "(" / ")"
     *               / "*" / "+" / "," / ";" / "="
     * unreserved    = ALPHA / DIGIT / "-" / "." / "_" / "~"
     * pchar         = unreserved / pct-encoded / sub-delims / ":" / "@"
     *
     * path          = path-abempty    ; begins with "/" or is empty
     *               / path-absolute   ; begins with "/" but not "//"
     *               / path-noscheme   ; begins with a non-colon segment
     *               / path-rootless   ; begins with a segment
     *               / path-empty      ; zero characters
     *
     * path-abempty  = *( "/" segment )
     * path-absolute = "/" [ segment-nz *( "/" segment ) ]
     * path-noscheme = segment-nz-nc *( "/" segment )
     * path-rootless = segment-nz *( "/" segment )
     * path-empty    = 0
     * segment       = *pchar
     * segment-nz    = 1*pchar
     * segment-nz-nc = 1*( unreserved / pct-encoded / sub-delims / "@" )
     *               ; non-zero-length segment without any colon ":"
     *
     * query         = *( pchar / "/" / "?" )
     *
     * fragment      = *( pchar / "/" / "?" )
     * ```
     *
     *
     * For the hostname component, we are checking for valid DNS hostnames according to RFC 952 / RFC 1123:
     *
     *
     * ```javascript
     *
     * hname         = name *("." name)
     * name          = let-or-digit ( *( let-or-digit-or-hyphen ) let-or-digit )
     * ```
     *
     *
     * When the URI uses the protocol 'mailto:', the address part is additionally checked against the most commonly
     * used parts of RFC 6068:
     *
     *
     * ```javascript
     *
     * mailtoURI     = "mailto:" [ to ] [ hfields ]
     * to            = addr-spec *("," addr-spec )
     * hfields       = "?" hfield *( "&" hfield )
     * hfield        = hfname "=" hfvalue
     * hfname        = *qchar
     * hfvalue       = *qchar
     * addr-spec     = local-part "@" domain
     * local-part    = dot-atom-text              // not accepted: quoted-string
     * domain        = dot-atom-text              // not accepted: "[" *dtext-no-obs "]"
     * dtext-no-obs  = %d33-90 / ; Printable US-ASCII
     *                 %d94-126  ; characters not including
     *                           ; "[", "]", or "\"
     * qchar         = unreserved / pct-encoded / some-delims
     * some-delims   = "!" / "$" / "'" / "(" / ")" / "*"
     *               / "+" / "," / ";" / ":" / "@"
     *
     * Note:
     * A number of characters that can appear in <addr-spec> MUST be
     * percent-encoded.  These are the characters that cannot appear in
     * a URI according to [STD66] as well as "%" (because it is used for
     * percent-encoding) and all the characters in gen-delims except "@"
     * and ":" (i.e., "/", "?", "#", "[", and "]").  Of the characters
     * in sub-delims, at least the following also have to be percent-
     * encoded: "&", ";", and "=".  Care has to be taken both when
     * encoding as well as when decoding to make sure these operations
     * are applied only once.
     *
     * ```
     *
     *
     * When a list of allowed entries has been configured using {@link #add}, any URL that passes the syntactic
     * checks above, additionally will be tested against the content of this list.
     *
     *
     * @returns true if valid, false if not valid
     */
    validate(
      /**
       * URL to be validated
       */
      sUrl: string
    ): boolean;
  }
  const URLListValidator: URLListValidator;
  export default URLListValidator;

  /**
   * Entry object of the URLListValidator.
   */
  export type Entry = {
    /**
     * The protocol of the URL, can be falsy to allow all protocols for an entry e.g. "", "http", "mailto"
     */
    protocol?: string;
    /**
     * The host of the URL, can be falsy to allow all hosts. A wildcard asterisk can be set at the beginning,
     * e.g. "examples.com", "*.example.com"
     */
    host?: string;
    /**
     * The port of the URL, can be falsy to allow all ports, e.g. "", "8080"
     */
    port?: string;
    /**
     * the path of the URL, path of the url, can be falsy to allow all paths. A wildcard asterisk can be set
     * at the end, e.g. "/my-example*", "/my-news"
     */
    path?: string;
  };
}

declare module "sap/base/security/URLWhitelist" {
  /**
   * Entry object of the URLWhitelist.
   *
   * @deprecated As of version 1.85. use {@link module:sap/base/security/URLListValidator.Entry} instead.
   */
  export type Entry = {
    /**
     * The protocol of the URL
     */
    protocol: string;
    /**
     * The host of the URL
     */
    host: string;
    /**
     * The port of the URL
     */
    port: string;
    /**
     * the path of the URL
     */
    path: string;
  };
}

declare module "sap/base/strings/camelize" {
  /**
   * Transforms a hyphen separated string to a camel case string.
   *
   * @since 1.58
   *
   * @returns The transformed string
   */
  export default function camelize(
    /**
     * Hyphen separated string
     */
    sString: string
  ): string;
}

declare module "sap/base/strings/capitalize" {
  /**
   * Converts first character of the string to upper case.
   *
   * @since 1.58
   *
   * @returns String input with first character uppercase
   */
  export default function capitalize(
    /**
     * String for which first character should be converted
     */
    sString: string
  ): string;
}

declare module "sap/base/strings/escapeRegExp" {
  /**
   * Escapes all characters that would have a special meaning in a regular expression.
   *
   * This method can be used when a string with arbitrary content has to be integrated into a regular expression
   * and when the whole string should match literally.
   *
   * @since 1.58
   *
   * @returns The escaped string
   */
  export default function escapeRegExp(
    /**
     * String to escape
     */
    sString: string
  ): string;
}

declare module "sap/base/strings/formatMessage" {
  /**
   * Creates a string from a pattern by replacing placeholders with concrete values.
   *
   * The syntax of the pattern is inspired by (but not fully equivalent to) the java.util.MessageFormat.
   *
   * Placeholders have the form `{ integer }`, where any occurrence of `{0}` is replaced by the value with
   * index 0 in `aValues`, `{1}` by the value with index 1 in `aValues` etc.
   *
   * To avoid interpretation of curly braces as placeholders, any non-placeholder fragment of the pattern
   * can be enclosed in single quotes. The surrounding single quotes will be omitted from the result. Single
   * quotes that are not meant to escape a fragment and that should appear in the result, need to be doubled.
   * In the result, only a single single quote will occur.
   *
   * Example: Pattern Strings
   * ```javascript
   *
   *  formatMessage("Say {0}",     ["Hello"]) -> "Say Hello"    // normal use case
   *  formatMessage("Say '{0}'",   ["Hello"]) -> "Say {0}"      // escaped placeholder
   *  formatMessage("Say ''{0}''", ["Hello"]) -> "Say 'Hello'"  // doubled single quote
   *  formatMessage("Say '{0}'''", ["Hello"]) -> "Say {0}'"     // doubled single quote in quoted fragment
   * ```
   *  In contrast to java.util.MessageFormat, format types or format styles are not supported. Everything
   * after the argument index and up to the first closing curly brace is ignored. Nested placeholders (as
   * supported by java.lang.MessageFormat for the format type choice) are not ignored but reported as a parse
   * error.
   *
   * This method throws an Error when the pattern syntax is not fulfilled (e.g. unbalanced curly braces, nested
   * placeholders or a non-numerical argument index).
   *
   * This method can also be used as a formatter within a binding. The first part of a composite binding will
   * be used as pattern, the following parts as aValues. If there is only one value and this value is an array
   * it will be handled like the default described above.
   *
   * @since 1.58
   *
   * @returns The formatted result string
   */
  export default function formatMessage(
    /**
     * A pattern string in the described syntax
     */
    sPattern: string,
    /**
     * The values to be used instead of the placeholders.
     */
    aValues?: any[]
  ): string;
}

declare module "sap/base/strings/hyphenate" {
  /**
   * Transforms a camel case string (camelCase) into a hyphen separated string (kebab-case).
   *
   * @since 1.58
   *
   * @returns The transformed string
   */
  export default function hyphenate(
    /**
     * camel case string
     */
    sString: string
  ): string;
}

declare module "sap/base/util/array/diff" {
  /**
   * Calculates delta of old list and new list.
   *
   * This function implements the algorithm described in "A Technique for Isolating Differences Between Files"
   * (Commun. ACM, April 1978, Volume 21, Number 4, Pages 264-268).
   *
   * Items in the arrays are not compared directly. Instead, a substitute symbol is determined for each item
   * by applying the provided function `fnSymbol` to it. Items with strictly equal symbols are assumed to
   * represent the same logical item:
   * ```javascript
   *
   *   fnSymbol(a) === fnSymbol(b)   <=>   a 'is logically the same as' b
   * ```
   *  As an additional constraint, casting the symbols to string should not modify the comparison result.
   * If this second constraint is not met, this method might report more diffs than necessary.
   *
   * If no symbol function is provided, a default implementation is used which applies `JSON.stringify` to
   * non-string items and reduces the strings to a hash code. It is not guaranteed that this default implementation
   * fulfills the above constraint in all cases, but it is a compromise between implementation effort, generality
   * and performance. If items are known to be non-stringifiable (e.g. because they may contain cyclic references)
   * or when hash collisions are likely, an own `symbol` function must be provided via the configuration object.
   *
   * The result of the diff is a sequence of update operations, each consisting of a `type` (either `"insert"`,
   * `"delete"` or `"replace"` when enabled via configuration object) and an `index`. By applying the operations
   * one after the other to the old array, it can be transformed to an array whose items are equal to the
   * new array.
   *
   * @since 1.58
   *
   * @returns List of update operations
   */
  export default function diff(
    /**
     * Old Array
     */
    aOld: any[],
    /**
     * New Array
     */
    aNew: any[],
    /**
     * Configuration object or a function to calculate substitute symbols for array items
     */
    vConfigOrSymbol?:
      | {
          /**
           * Function to calculate substitute symbols for array items
           */
          symbol?: Function;
          /**
           * Switch for the `replace` type which specifies that an item within the array has been changed
           */
          replace?: boolean;
        }
      | Function
  ): Array<{
    type: string;

    index: int;
  }>;
}

declare module "sap/base/util/array/uniqueSort" {
  /**
   * Sorts the given array in-place and removes any duplicates (identified by "===").
   *
   * Uses Array#sort()
   * See:
   * 	https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/sort
   *
   * Use `jQuery.uniqueSort()` for arrays of DOMElements.
   *
   * @since 1.58
   *
   * @returns Same array as given (for chaining)
   */
  export default function uniqueSort(
    /**
     * An Array of any type
     */
    aArray: any[]
  ): any[];
}

declare module "sap/base/util/deepClone" {
  /**
   * Creates a deep clone of the source value.
   *
   * Only arrays, JavaScript Date objects and objects that pass the {@link module:sap/base/util/isPlainObject isPlainObject }
   * check will be cloned. For other object types, a `TypeError` will be thrown as there's no standard way
   * to clone them. Primitive values (boolean, number, string) as well as `null` and `undefined` will be copied,
   * they have value semantics anyhow.
   *
   * `deepClone` is designed to match the semantics of {@link module:sap/base/util/deepEqual deepEqual}. Any
   * deeply cloned object should be deep-equal to the source. However, not every object that can be handled
   * by `deepEqual` can also be deeply cloned (e.g. `deepClone` fails on non-plain objects).
   *
   * To limit the time needed for a deep clone and to avoid endless recursion in case of cyclic structures,
   * the recursion depth is limited by the parameter `maxDepth`, which defaults to 10. When the recursion
   * depth exceeds the given limit, a `TypeError` is thrown.
   *
   * Note that object identities are not honored by the clone operation. If the original source contained
   * multiple references to the same plain object instance, the clone will contain a different clone for each
   * reference.
   *
   * @since 1.63
   *
   * @returns A clone of the source value
   */
  export default function deepClone(
    /**
     * Source value that shall be cloned
     */
    src: any,
    /**
     * Maximum recursion depth for the clone operation, deeper structures will throw an error
     */
    maxDepth?: int
  ): any;
}

declare module "sap/base/util/deepEqual" {
  /**
   * Compares the two given values for equality, especially by comparing the content.
   *
   * **Note:** Function does not work with comparing XML objects.
   *
   * @since 1.58
   *
   * @returns Whether a and b are equal
   */
  export default function deepEqual(
    /**
     * A value of any type
     */
    a: any,
    /**
     * A value of any type
     */
    b: any,
    /**
     * Maximum recursion depth
     */
    maxDepth?: int,
    /**
     * Whether all existing properties in a are equal as in b
     */
    contains?: boolean
  ): boolean;
}

declare module "sap/base/util/deepExtend" {
  /**
   * Performs object extension by merging source objects into a target object. Copies are always deep.
   *
   * If during merging a key in the target object exists it is overwritten with the source object's value.
   * Usage is the same as `jQuery.extend(true, ...)`. Values that are `undefined` are ignored.
   *
   * For shallow copies, you may use {@link module:sap/base/util/extend sap/base/util/extend} or `Object.assign`,
   * but note that `Object.assign` only copies enumerable and own properties and doesn't copy properties on
   * the prototype and non-enumerable properties. Also, values that are `undefined` are NOT ignored.
   *
   * @since 1.71
   *
   * @returns the target object which is the result of the merge
   */
  export default function deepExtend(
    /**
     * The object that will receive new properties
     */
    target: object,
    /**
     * One or more objects which get merged into the target object
     */
    ...source: object[]
  ): object;
}

declare module "sap/base/util/Deferred" {
  /**
   * Creates a `Deferred` instance which represents a future value.
   *
   * While a `Promise` can only be resolved or rejected by calling the respective methods in its constructor,
   * a `Deferred` can be resolved or rejected via `resolve` or `reject` methods at any point. A `Deferred`
   * object creates a `Promise` instance which functions as a proxy for the future result. This `Promise`
   * object can be accessed via the `promise` property of the `Deferred` object.
   *
   * @since 1.90
   */
  export default class Deferred<T extends any = any> {
    constructor();
    /**
     * Promise instance of the Deferred
     */
    promise: Promise<T>;

    /**
     * Proxy call to the `reject` method of the wrapped Promise
     */
    reject(
      /**
       * Failure reason
       */
      reason?: any
    ): void;
    /**
     * Proxy call to the `resolve` method of the wrapped Promise
     */
    resolve(
      /**
       * Fulfillment value
       */
      value?: T
    ): void;
  }
}

declare module "sap/base/util/each" {
  /**
   * Iterates over elements of the given object or array.
   *
   * Numeric indexes are only used for instances of `Array`. For all other objects, including those with a
   * numeric `length` property, the properties are iterated by name.
   *
   * When `fnCallback` returns `false`, then the iteration stops (break).
   *
   * @since 1.58
   *
   * @returns the given `oObject`
   */
  export default function each(
    /**
     * object or array to enumerate the properties of
     */
    oObject: object | any[],
    /**
     * function to call for each property name
     */
    fnCallback: (this: any, p1: Key, p2: any) => boolean
  ): object | any[];
  /**
   * The key that is passed to the callback as the first parameter
   */
  export type Key = int | string;
}

declare module "sap/base/util/extend" {
  /**
   * Performs object extension by merging source objects into a target object. Generates a shallow copy.
   *
   * If during merging a key in the target object exists it is overwritten with the source object's value.
   * Usage is the same as `jQuery.extend(...)`. Values that are `undefined` are ignored.
   *
   * As alternative you may also use `Object.assign`, but note that `Object.assign` only copies enumerable
   * and own properties and doesn't copy properties on the prototype and non-enumerable properties. Also,
   * values that are `undefined` are NOT ignored.
   *
   * For deep copies, you may use {@link module:sap/base/util/deepExtend sap/base/util/deepExtend}.
   *
   * @since 1.71
   *
   * @returns the target object which is the result of the merge
   */
  export default function extend(
    /**
     * The object that will receive new properties
     */
    target: object,
    /**
     * One or more objects which get merged into the target object
     */
    ...source: object[]
  ): object;
}

declare module "sap/base/util/includes" {
  /**
   * Checks if value is included in collection.
   *
   * @since 1.58
   * @deprecated As of version 1.90. Use the `Array.prototype.includes` or `String.prototype.includes` instead,
   * but note that `Array.prototype.includes` or `String.prototype.includes` fail when called on null values.
   *
   * @returns - true if value is in the collection, false otherwise
   */
  export default function includes(
    /**
     * Collection to be checked
     */
    vCollection: any[] | object | string,
    /**
     * The value to be checked
     */
    vValue: any,
    /**
     * optional start index, negative start index will start from the end
     */
    iFromIndex?: int
  ): boolean;
}

declare module "sap/base/util/isEmptyObject" {
  /**
   * Validates if the given object is empty, that is that it has no enumerable properties.
   *
   * Note that `null` and `undefined` comply with this definition of 'empty'. The behavior for non-object
   * values is undefined and might change in future.
   *
   * @since 1.65
   *
   * @returns whether or not the given object is empty
   */
  export default function isEmptyObject(
    /**
     * the object which is checked
     */
    obj: Object
  ): boolean;
}

declare module "sap/base/util/isPlainObject" {
  /**
   * Checks whether the object is a plain object (created using "{}" or "new Object").
   *
   * @since 1.58
   *
   * @returns whether or not the object is a plain object (created using "{}" or "new Object").
   */
  export default function isPlainObject(
    /**
     * the object which is checked
     */
    obj: Object
  ): boolean;
}

declare module "sap/base/util/merge" {
  /**
   * Performs object extension by merging source objects into a target object. Copies are always deep.
   *
   * If during merging a key in the target object exists it is overwritten with the source object's value.
   * Usage is the same as `jQuery.extend(true, ...)`, but values that are `undefined` are NOT ignored.
   *
   * For shallow copies, you may use `Object.assign`, but note that `Object.assign` only copies enumerable
   * and own properties and doesn't copy properties on the prototype and non-enumerable properties.
   *
   * @since 1.58
   *
   * @returns the target object which is the result of the merge
   */
  export default function merge(
    /**
     * The object that will receive new properties
     */
    target: object,
    /**
     * One or more objects which get merged into the target object
     */
    ...source: object[]
  ): object;
}

declare module "sap/base/util/now" {
  /**
   * Returns a high resolution timestamp in microseconds. The timestamp is based on 01/01/1970 00:00:00 (UNIX
   * epoch) as float with microsecond precision. The fractional part of the timestamp represents fractions
   * of a millisecond. Converting to a `Date` is possible by using `require(["sap/base/util/now"], function(now){new
   * Date(now());}`
   *
   * @since 1.58
   *
   * @returns timestamp in microseconds
   */
  export default function now(): float;
}

declare module "sap/base/util/ObjectPath" {
  /**
   * Manages an object path.
   *
   * The object path can be just created with {@link #.create}, then an empty nested object path will be created
   * from the provided string. If a value is set for an object path {@link #.set} it is also created if it
   * not already exists. Values can be retrieved from the objectpath with {@link #get}.
   *
   * @since 1.58
   */
  interface ObjectPath {
    /**
     * Creates a object path from the provided path in the provided root context.
     *
     * The provided path is used to navigate through the nested objects, starting with the root context.
     *
     *
     * @returns The newly created context object, e.g. base.my.test.module
     */
    create(
      /**
       * Path as string where each name is separated by '.'. Can also be an array of names.
       */
      vObjectPath: string | string[],
      /**
       * Root context where the path starts
       */
      oRootContext?: Object
    ): Object;
    /**
     * Returns a value located in the provided path. If the provided path cannot be resolved completely, `undefined`
     * is returned.
     *
     * The provided object path is used to navigate through the nested objects, starting with the root context.
     * If no root context is provided, the object path begins with `globalThis`.
     *
     *
     * @returns Returns the value located in the provided path, or `undefined` if the path does not exist completely.
     */
    get(
      /**
       * Path as string where each name is separated by '.'. Can also be an array of names.
       */
      vObjectPath: string | string[],
      /**
       * Root context where the path starts
       */
      oRootContext?: Object
    ): any | undefined;
    /**
     * Sets a value located in the provided path.
     *
     * The provided path is used to navigate through the nested objects, starting with the root context.
     *
     * **Note:** Ensures that the object path exists.
     */
    set(
      /**
       * vObjectPath Path as string where each name is separated by '.'. Can also be an array of names.
       */
      vObjectPath: string | string[],
      /**
       * The value to be set in the root context's object path
       */
      vValue: any,
      /**
       * Root context where the path starts
       */
      oRootContext?: Object
    ): void;
  }
  const ObjectPath: ObjectPath;
  export default ObjectPath;
}

declare module "sap/base/util/Properties" {
  /**
   * Represents a collection of string properties (key/value pairs).
   *
   * Each key and its corresponding value in the collection is a string, keys are case-sensitive.
   *
   * Use {@link module:sap/base/util/Properties.create} to create an instance of {@link module:sap/base/util/Properties}.
   *
   * The {@link #getProperty} method can be used to retrieve a value from the collection, {@link #setProperty }
   * to store or change a value for a key and {@link #getKeys} can be used to retrieve an array of all keys
   * that are currently stored in the collection.
   *
   * @since 1.58
   */
  export default class Properties {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Creates and returns a new instance of {@link module:sap/base/util/Properties}.
     *
     * If option 'url' is passed, immediately a load request for the given target is triggered. A property file
     * that is loaded can contain comments with a leading ! or #. The loaded property list does not contain
     * any comments.
     *
     *
     * @returns A new property collection (synchronous case) or `null` if the file could not be loaded and `returnNullIfMissing`
     * was set; in case of asynchronous loading, always a Promise is returned, which resolves with the property
     * collection or with `null` if the file could not be loaded and `returnNullIfMissing` was set to true
     */
    static create(
      /**
       * Parameters used to initialize the property list
       */
      mParams?: {
        /**
         * The URL to the .properties file which should be loaded
         */
        url?: string;
        /**
         * Whether the .properties file should be loaded asynchronously or not
         */
        async?: boolean;
        /**
         * A map of additional header key/value pairs to send along with the request (see `headers` option of `jQuery.ajax`)
         */
        headers?: Record<string, any>;
        /**
         * Whether `null` should be returned for a missing properties file; by default an empty collection is returned
         */
        returnNullIfMissing?: boolean;
      }
    ): Properties | null | Promise<Properties | null>;
    /**
     * Creates and returns a clone of the property collection.
     *
     *
     * @returns A clone of the property collection
     */
    clone(): Properties;
    /**
     * Returns an array of all keys in the property collection.
     *
     *
     * @returns All keys in the property collection
     */
    getKeys(): string[];
    /**
     * Returns the value for the given key or `null` if the collection has no value for the key.
     *
     * Optionally, a default value can be given which will be returned if the collection does not contain a
     * value for the key; only non-empty default values are supported.
     *
     *
     * @returns Value for the given key or the default value or `null` if no default value or a falsy default
     * value was given
     */
    getProperty(
      /**
       * Key to return the value for
       */
      sKey: string,
      /**
       * Optional, a default value that will be returned if the requested key is not in the collection
       */
      sDefaultValue?: string
    ): string | null;
    /**
     * Stores or changes the value for the given key in the collection.
     *
     * If the given value is not a string, the collection won't be modified. The key is always cast to a string.
     */
    setProperty(
      /**
       * Key of the property
       */
      sKey: string,
      /**
       * String value for the key
       */
      sValue: string
    ): void;
  }
}

declare module "sap/base/util/uid" {
  /**
   * Creates and returns a pseudo-unique ID.
   *
   * No means for detection of overlap with already present or future UIDs.
   *
   * @since 1.58
   *
   * @returns A pseudo-unique id.
   */
  export default function uid(): string;
}

declare module "sap/base/util/UriParameters" {
  /**
   * Provides access to the individual parameters of a URL query string.
   *
   * This class parses the query string from a URL and provides access to the values of individual parameters.
   * There are methods to check whether the query string {@link #has contains a parameter (`has()`)}, to {@link #get get a single value (`get()`) }
   * for a parameter and to {@link #getAll get a list of all values (`getAll()`)} for a parameter. Another
   * method allows to {@link #keys iterate over all parameter names (`keys()`)}.
   *
   * Decoding:
   *
   * The constructor and the factory methods expect percentage encoded input whereas all other APIs expect
   * and return decoded strings. After parsing the query string, any plus sign (0x2b) in names or values is
   * replaced by a blank (0x20) and the resulting strings are percentage decoded (`decodeURIComponent`).
   *
   * Deprecation:
   *
   * This class is deprecated in favor of the URL web standard classes `URLSearchParams` / `URL`.
   *
   * `UriParameters.fromQuery(input)` can be migrated to `new URLSearchParams(input)`
   *
   * `UriParameters.fromURL(input)` can be migrated to `new URL(input).searchParams`
   *
   * Caveats:
   *
   * The API has already been designed to be a drop-in replacement but there are some important caveats to
   * consider when switching to the standard APIs `URLSearchParams` / `URL`:
   * 	 - `new URL(input).searchParams` validates the given URL according to the WHATWG URL Standard ({@link https://url.spec.whatwg.org}).
   *     `UriParameters.fromURL(input)` only extracts the query string from the given input but does not perform
   *     any validation.
   * 	 - In some edge cases, especially for incomplete/invalid encoding, decoding behaves differently. Decoding
   *     in `UriParameters` is described in the section above. For details about the encoding/decoding of `URLSearchParams`,
   *     see the WHATWG URL Standard ({@link https://url.spec.whatwg.org}).
   * 	 - The `get` method's second parameter, `bAll`, is not available; use the `getAll` method instead.
   * 	 - The `keys` method's return value contains an entry for every occurrence of the key within the query,
   *     in the defined order and including duplicates. `UriParameters#keys()` yields unique key values, even
   *     when there are multiple values for the same key.
   * 	 - The internal `mParams` property is not available anymore (you should never access internal properties
   *     of UI5 classes or objects). With the predecessor of this API, access to `mParams` was often used to check
   *     whether a parameter is defined at all. Using the `has` method or checking the result of `get` against
   *     `null` serves the same purpose.
   *
   * @since 1.68
   * @deprecated As of version 1.119. See class description for details.
   */
  export default class UriParameters {
    /**
     * See:
     * 	https://url.spec.whatwg.org/#interface-urlsearchparams
     */
    constructor(
      /**
       * URL with parameters
       */
      sURL?: string
    );

    /**
     * Parses the given query string and returns an interface to access the individual parameters.
     *
     * Callers using `UriParameters.fromQuery(input)` can be migrated to `new URLSearchParams(input)` once that
     * API is available (or polyfilled) in all supported browsers.
     *
     *
     * @returns Object providing read access to the query parameters
     */
    static fromQuery(
      /**
       * Query string to parse, a leading question mark (?) will be ignored
       */
      sQuery?: string
    ): UriParameters;
    /**
     * Parses the query portion of the given URL and returns an object to access the individual parameters.
     *
     * Callers using `UriParameters.fromURL(input)` can be migrated to `new URL(input).searchParams` once that
     * API is available (or polyfilled) in all supported browsers.
     *
     *
     * @returns Object providing read access to the query parameters
     */
    static fromURL(
      /**
       * to parse the query portion of.
       */
      sURL: string
    ): UriParameters;
    /**
     * Returns the first value of the named query parameter.
     *
     * The value of the first occurrence of the parameter with name `sName` in the query string is returned.
     * If that first occurrence does not contain a value (it does not contain an equal sign), then an empty
     * string is returned.
     *
     * If (and only if) the parameter does not occur in the query string, `null` is returned.
     *
     *
     * @returns First value of the query parameter with the given name or `null`
     */
    get(
      /**
       * Name of the query parameter to get the value for
       */
      sName: string,
      /**
       * Whether all values for the parameter should be returned; the use of this parameter is deprecated and
       * highly discouraged; use the {@link #getAll} method instead
       */
      bAll?: boolean
    ): string | null;
    /**
     * Returns all values of the query parameter with the given name.
     *
     * An array of string values of all occurrences of the parameter with the given name is returned. This array
     * is empty if (and only if) the parameter does not occur in the query string.
     *
     *
     * @returns Array with all values of the query parameter with the given name
     */
    getAll(
      /**
       * Name of the query parameter
       */
      sName: string
    ): string[];
    /**
     * Checks whether a parameter occurs at least once in the query string.
     *
     *
     * @returns Whether the parameter has been defined
     */
    has(
      /**
       * Name of the query parameter to check
       */
      sName: string
    ): boolean;
    /**
     * Returns an iterator for all contained parameter names.
     *
     *
     * @returns Iterator for all parameter names.
     */
    keys(): Iterator<string>;
  }
}

declare module "sap/base/util/values" {
  /**
   * Returns values from an object.
   *
   * **Note:**Whenever possible, please try to use the native function `Object.values` instead. Especially,
   * if you don't need to rely on handling null values as argument.
   *
   * @since 1.58
   *
   * @returns - array of object values, if object does not contain values, an empty array will be returned
   */
  export default function values(
    /**
     * Object to be extracted
     */
    mObject: Object | undefined
  ): any[];
}

declare module "sap/base/util/Version" {
  /**
   * Represents a version consisting of major, minor, patch version, and suffix, for example '1.2.7-SNAPSHOT'.
   *
   * @since 1.58
   */
  export default class Version {
    /**
     * Returns a Version instance created from the given parameters.
     *
     * This function can either be called as a constructor (using `new`) or as a normal function. It always
     * returns an immutable Version instance.
     *
     * The parts of the version number (major, minor, patch, suffix) can be provided in several ways:
     * 	 - Version("1.2.3-SNAPSHOT") - as a dot-separated string. Any non-numerical char or a dot followed by
     *     a non-numerical char starts the suffix portion. Any missing major, minor or patch versions will be set
     *     to 0.
     * 	 - Version(1,2,3,"-SNAPSHOT") - as individual parameters. Major, minor and patch must be integer numbers
     *     or empty, suffix must be a string not starting with digits.
     * 	 - Version([1,2,3,"-SNAPSHOT"]) - as an array with the individual parts. The same type restrictions
     *     apply as before.
     * 	 - Version(otherVersion) - as a Version instance (cast operation). Returns the given instance instead
     *     of creating a new one.
     *
     * To keep the code size small, this implementation mainly validates the single string variant. All other
     * variants are only validated to some degree. It is the responsibility of the caller to provide proper
     * parts.
     */
    constructor(
      /**
       * the major part of the version (int) or any of the single parameter variants explained above.
       */
      vMajor: int | string | any[] | Version,
      /**
       * the minor part of the version number
       */
      iMinor?: int,
      /**
       * the patch part of the version number
       */
      iPatch?: int,
      /**
       * the suffix part of the version number
       */
      sSuffix?: string
    );

    /**
     * Compares this version with a given one.
     *
     * The version with which this version should be compared can be given as a `sap/base/util/Version` instance,
     * as a string (e.g. `v.compareTo("1.4.5")`). Or major, minor, patch and suffix values can be given as separate
     * parameters (e.g. `v.compareTo(1, 4, 5)`) or in an array (e.g. `v.compareTo([1, 4, 5])`).
     *
     *
     * @returns 0, if the given version is equal to this version, a negative value if the given other version
     * is greater and a positive value otherwise
     */
    compareTo(
      /**
       * The major part (an integer) of the version to compare to or the full version in any of the single parameter
       * variants, as documented for the {@link module:sap/base/util/Version constructor}.
       */
      vOtherMajor: int | string | any[] | Version,
      /**
       * A minor version to compare to (only valid when `vOther` is a single integer)
       */
      iOtherMinor?: int,
      /**
       * A patch version to compare to (only valid when `vOther` is a single integer)
       */
      iOtherPatch?: int,
      /**
       * A version suffix like "-SNAPSHOT" to compare to (only valid when `vOther` is an integer)
       */
      sOtherSuffix?: string
    ): int;
    /**
     * Returns the major version part of this version.
     *
     *
     * @returns the major version part of this version
     */
    getMajor(): int;
    /**
     * Returns the minor version part of this version.
     *
     *
     * @returns the minor version part of this version
     */
    getMinor(): int;
    /**
     * Returns the patch (or micro) version part of this version.
     *
     *
     * @returns the patch version part of this version
     */
    getPatch(): int;
    /**
     * Returns the version suffix of this version.
     *
     *
     * @returns the version suffix of this version
     */
    getSuffix(): string;
    /**
     * Checks whether this version is in the range of the given interval (start inclusive, end exclusive).
     *
     * The boundaries against which this version should be checked can be given as `sap/base/util/Version` instances
     * (e.g. `v.inRange(v1, v2)`), as strings (e.g. `v.inRange("1.4", "2.7")`) or as arrays (e.g. `v.inRange([1,4],
     * [2,7])`).
     *
     *
     * @returns `true` if this version is greater or equal to `vMin` and smaller than `vMax`, `false` otherwise.
     */
    inRange(
      /**
       * the start of the range (inclusive)
       */
      vMin: string | any[] | Version,
      /**
       * the end of the range (exclusive)
       */
      vMax: string | any[] | Version
    ): boolean;
    /**
     * Returns a string representation of this version.
     *
     *
     * @returns a string representation of this version.
     */
    toString(): string;
  }
}

declare module "sap/ui/util/XMLHelper" {
  import { URI } from "sap/ui/core/library";

  /**
   * Error information as provided by the `DOMParser`.
   *
   * Note that the set of properties with meaningful content differs between browsers.
   */
  export type XMLParseErrorInfo = {
    errorCode?: int;

    url?: URI;

    reason?: string;

    srcText?: string;

    line?: int;

    linepos?: int;

    filepos?: int;

    type?: "error" | "warning";
  };

  /**
   * Provides functionality for parsing XML formatted strings and serializing XML documents.
   *
   * @since 1.58
   */
  interface XMLHelper {
    /**
     * Extracts parse error information from the specified document (if any).
     *
     * If an error was found, the returned object contains a browser-specific subset of the properties described
     * in {@link module:sap/base/util/XMLHelper.XMLParseErrorInfo XMLParseErrorInfo}. Otherwise, it just contains
     * an `errorCode` property with value 0.
     *
     *
     * @returns A browser-specific error info object if errors were found, or an object with an errorCode
     * of 0 only
     */
    getParseError(
      /**
       * The parsed XML document
       */
      oDocument: XMLDocument
    ): XMLParseErrorInfo;
    /**
     * Parses the specified XML string into an XML document, using the native parsing functionality of the browser.
     * If an error occurs during parsing, a {@link module:sap/base/util/XMLHelper.XMLParseErrorInfo parse error info object }
     * is attached as the `parseError` property of the returned document.
     *
     *
     * @returns the parsed XML document with a `parseError` property as described in {@link #getParseError}.
     * An error occurred if the `errorCode` property of the `parseError` is not 0.
     */
    parse(
      /**
       * An XML string
       */
      sXMLText: string
    ): XMLDocument;
    /**
     * Serializes the specified DOM tree into a string representation.
     * See:
     * 	{@link https://developer.mozilla.org/en-US/docs/Web/API/XMLSerializer/serializeToString}
     *
     *
     * @returns the serialized XML string
     */
    serialize(
      /**
       * the XML document object to be serialized as string
       */
      oXMLDocument: Node | Attr
    ): string;
  }
  const XMLHelper: XMLHelper;
  export default XMLHelper;
}

declare module "sap/ui/core/AnimationMode" {
  /**
   * Enumerable list with available animation modes.
   *
   * This enumerable is used to validate the animation mode. Animation modes allow to specify different animation
   * scenarios or levels. The implementation of the Control (JavaScript or CSS) has to be done differently
   * for each animation mode.
   *
   * @since 1.120
   */
  enum AnimationMode {
    /**
     * `basic` can be used for a reduced, more light-weight set of animations.
     */
    basic = "basic",
    /**
     * `full` represents a mode with unrestricted animation capabilities.
     */
    full = "full",
    /**
     * `minimal` includes animations of fundamental functionality.
     */
    minimal = "minimal",
    /**
     * `none` deactivates the animation completely.
     */
    none = "none",
  }
  export default AnimationMode;
}

declare module "sap/ui/core/ComponentRegistry" {
  import { ID } from "sap/ui/core/library";

  import Component from "sap/ui/core/Component";

  /**
   * Registry of all `Component`s that currently exist.
   *
   * @since 1.120
   */
  interface ComponentRegistry {
    /**
     * Number of existing components.
     */
    size: int;

    /**
     * Return an object with all instances of `sap.ui.core.Component`, keyed by their ID.
     *
     * Each call creates a new snapshot object. Depending on the size of the UI, this operation therefore might
     * be expensive. Consider to use the `forEach` or `filter` method instead of executing similar operations
     * on the returned object.
     *
     * **Note**: The returned object is created by a call to `Object.create(null)`, and therefore lacks all
     * methods of `Object.prototype`, e.g. `toString` etc.
     *
     *
     * @returns Object with all components, keyed by their ID
     */
    all(): Record<ID, Component>;
    /**
     * Returns an array with components for which the given `callback` returns a value that coerces to `true`.
     *
     * The expected signature of the callback is
     * ```javascript
     *
     *    function callback(oComponent, sID)
     * ```
     *  where `oComponent` is the currently visited component instance and `sID` is the ID of that instance.
     *
     * If components are created or destroyed within the `callback`, then the behavior is not specified. Newly
     * added objects might or might not be visited. When a component is destroyed during the filtering and was
     * not visited yet, it might or might not be visited. As the behavior for such concurrent modifications
     * is not specified, it may change in newer releases.
     *
     * If a `thisArg` is given, it will be provided as `this` context when calling `callback`. The `this` value
     * that the implementation of `callback` sees, depends on the usual resolution mechanism. E.g. when `callback`
     * was bound to some context object, that object wins over the given `thisArg`.
     *
     * This function returns an array with all components matching the given predicate. The order of the components
     * in the array is not specified and might change between calls (over time and across different versions
     * of UI5).
     *
     *
     * @returns Array of components matching the predicate; order is undefined and might change in newer versions
     * of UI5
     */
    filter(
      /**
       * predicate against which each Component is tested
       */
      callback: (p1: Component, p2: ID) => boolean,
      /**
       * context object to provide as `this` in each call of `callback`
       */
      thisArg?: Object
    ): Component[];
    /**
     * Calls the given `callback` for each existing component.
     *
     * The expected signature of the callback is
     * ```javascript
     *
     *    function callback(oComponent, sID)
     * ```
     *  where `oComponent` is the currently visited component instance and `sID` is the ID of that instance.
     *
     * The order in which the callback is called for components is not specified and might change between calls
     * (over time and across different versions of UI5).
     *
     * If components are created or destroyed within the `callback`, then the behavior is not specified. Newly
     * added objects might or might not be visited. When a component is destroyed during the filtering and was
     * not visited yet, it might or might not be visited. As the behavior for such concurrent modifications
     * is not specified, it may change in newer releases.
     *
     * If a `thisArg` is given, it will be provided as `this` context when calling `callback`. The `this` value
     * that the implementation of `callback` sees, depends on the usual resolution mechanism. E.g. when `callback`
     * was bound to some context object, that object wins over the given `thisArg`.
     */
    forEach(
      /**
       * Function to call for each Component
       */
      callback: (p1: Component, p2: ID) => void,
      /**
       * Context object to provide as `this` in each call of `callback`
       */
      thisArg?: Object
    ): void;
    /**
     * Retrieves a Component by its ID.
     *
     * When the ID is `null` or `undefined` or when there's no Component with the given ID, then `undefined`
     * is returned.
     *
     *
     * @returns Component with the given ID or `undefined`
     */
    get(
      /**
       * ID of the Component to retrieve
       */
      id: ID
    ): Component | undefined;
  }
  const ComponentRegistry: ComponentRegistry;
  export default ComponentRegistry;
}

declare module "sap/ui/core/ComponentSupport" {
  /**
   * The module `sap/ui/core/ComponentSupport` provides functionality which is used to find declared Components
   * in the HTML page and to create the Component instances which will be put into a {@link sap.ui.core.ComponentContainer}.
   *
   * The {@link module:sap/ui/core/ComponentSupport.run} function is called automatically once the module
   * has been required. This allows declarative support for components.
   *
   * Usage: To enable the `sap/ui/core/ComponentSupport` include it as the `oninit` module in the bootstrap:
   *
   * ```javascript
   *
   * <script id="sap-ui-bootstrap"
   *     src="/resources/sap-ui-core.js"
   *     ...
   *     data-sap-ui-oninit="module:sap/ui/core/ComponentSupport"
   *     ...>
   * </script>
   * ```
   *
   *
   * To load and render components inside the HTML page, a special data attribute has to be specified on the
   * respective DOM elements: `data-sap-ui-component`. All DOM elements marked with this data attribute will
   * be regarded as container elements for the created `ComponentContainer` instances.
   *
   *
   * ```javascript
   *
   * <body id="content" class="sapUiBody sapUiSizeCompact">
   *     ...
   *     <div data-sap-ui-component
   *         data-id="container"
   *         data-name="sap.ui.core.samples.formatting"
   *         ...
   *         data-handle-validation="true"
   *         ...>
   *     </div>
   *     ...
   * </body>
   * ```
   *
   *
   * Configuration: All configuration settings for the `ComponentContainer` have to be defined as `data` attributes
   * on the respective HTML tags. Each data attribute will be interpreted as a setting and parsed considering
   * the data type of the matching property in the `ComponentContainer`.
   *
   * **NOTE:** The following `data` attributes for registering event handlers have been deprecated since UI5
   * version 1.120 and won't work in the next major version because of the removal of accessing the global
   * namespace:
   * 	 - `data-component-created`
   * 	 - `data-component-failed`
   *
   * Alternatively, you can provide your own module in the bootstrap via `oninit`, in which you create an
   * instance of the {@link sap.ui.core.ComponentContainer ComponentContainer} in the JavaScript code.
   *
   * As HTML is case-insensitive, in order to define a property with upper-case characters, you have to "escape"
   * them with a dash character, similar to CSS attributes. The following code gives an example:
   *
   *
   * ```javascript
   *
   * <div data-sap-ui-component ... data-handle-validation="true" ...></div>
   * ```
   *
   *
   * **Beware:**
   *
   * The `ComponentSupport` module enforces asynchronous loading of the respective component and its library
   * dependencies. This is done by applying default settings for the following properties of the `ComponentContainer`:
   *
   *
   * 	 - `async` {boolean} (**forced to `true`**)
   * 	 - `manifest` {boolean|string} (**forced to `true` if no string is provided to ensure manifest first**)
   *
   * 	 - `lifecycle` {sap.ui.core.ComponentLifecycle} (defaults to `Container`)
   * 	 - `autoPrefixId` {boolean} (defaults to `true`)
   *
   * See {@link https://ui5.sap.com/#/topic/82a0fcecc3cb427c91469bc537ebdddf Declarative API for Initial Components}.
   *
   * @since 1.58.0
   */
  interface ComponentSupport {
    /**
     * Find all DOM elements with the attribute `data-sap-ui-component` and parse the attributes from these
     * DOM elements for the settings of the `ComponentContainer` which will be placed into these DOM elements.
     *
     * This function is called automatically once the module has been required.
     */
    run(): void;
  }
  const ComponentSupport: ComponentSupport;
  export default ComponentSupport;
}

declare module "sap/ui/core/ControlBehavior" {
  import AnimationMode from "sap/ui/core/AnimationMode";

  /**
   * Provides control behavior relevant configuration options
   *
   * @since 1.120
   */
  interface ControlBehavior {
    /**
     * Returns the current animation mode.
     *
     * @since 1.120
     *
     * @returns The current animationMode
     */
    getAnimationMode(): AnimationMode;
    /**
     * Returns whether the accessibility mode is enabled or not.
     *
     * @since 1.120
     *
     * @returns whether the accessibility mode is enabled or not
     */
    isAccessibilityEnabled(): boolean;
    /**
     * Sets the current animation mode.
     *
     * Expects an animation mode as string and validates it. If a wrong animation mode was set, an error is
     * thrown. If the mode is valid it is set, then the attributes `data-sap-ui-animation` and `data-sap-ui-animation-mode`
     * of the HTML document root element are also updated. If the `animationMode` is `AnimationMode.none` the
     * old `animation` property is set to `false`, otherwise it is set to `true`.
     *
     * @since 1.120
     */
    setAnimationMode(
      /**
       * A valid animation mode
       */
      sAnimationMode: AnimationMode | keyof typeof AnimationMode
    ): void;
  }
  const ControlBehavior: ControlBehavior;
  export default ControlBehavior;
}

declare module "sap/ui/core/date/CalendarUtils" {
  import CalendarWeekNumbering from "sap/base/i18n/date/CalendarWeekNumbering";

  import Locale from "sap/ui/core/Locale";

  /**
   * Provides calendar-related utilities.
   *
   * @since 1.108.0
   */
  interface CalendarUtils {
    /**
     * Resolves calendar week configuration.
     *
     * Returns an object with the following fields:
     * 	 - `firstDayOfWeek`: specifies the first day of the week starting with `0` (which is Sunday)
     * 	 - `minimalDaysInFirstWeek`: minimal days at the beginning of the year which define the first calendar
     *     week
     *
     * @since 1.108.0
     *
     * @returns The calendar week configuration, or `undefined for an invalid value of module:sap/base/i18n/date/CalendarWeekNumbering`.
     */
    getWeekConfigurationValues(
      /**
       * The calendar week numbering; if omitted, the calendar week numbering of the configuration is used; see
       * {@link module:sap/base/i18n/Formatting.getCalendarWeekNumbering Formatting.getCalendarWeekNumbering}.
       * If this value is `Default` the returned calendar week configuration is derived from the given `oLocale`.
       */
      sCalendarWeekNumbering?:
        | CalendarWeekNumbering
        | keyof typeof CalendarWeekNumbering,
      /**
       * The locale to use; if no locale is given, a locale for the currently configured language is used; see
       * {@link module:sap/base/i18n/Formatting.getLanguageTag Formatting.getLanguageTag}. Is only used when `sCalendarWeekNumbering`
       * is set to `Default`.
       */
      oLocale?: Locale
    ):
      | {
          firstDayOfWeek: int;

          minimalDaysInFirstWeek: int;
        }
      | undefined;
  }
  const CalendarUtils: CalendarUtils;
  export default CalendarUtils;
}

declare module "sap/ui/core/date/UI5Date" {
  /**
   * A date implementation considering the configured time zone
   *
   * A subclass of JavaScript `Date` that considers the configured time zone, see {@link module:sap/base/i18n/Localization.getTimezone Localization.getTimezone}.
   * All JavaScript `Date` functions that use the local browser time zone, like `getDate`, `setDate`, and
   * `toString`, are overwritten and use the configured time zone to compute the values.
   *
   * Use {@link module:sap/ui/core/date/UI5Date.getInstance} to create new date instances.
   *
   * **Note:** Adjusting the time zone in a running application can lead to unexpected data inconsistencies.
   * For more information, see {@link module:sap/base/i18n/Localization.setTimezone Localization.setTimezone}.
   *
   * @since 1.111.0
   */
  export default class UI5Date extends Date {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Creates a date instance (either JavaScript Date or `UI5Date`) which considers the configured time zone
     * wherever JavaScript Date uses the local browser time zone, for example in `getDate`, `toString`, or `setHours`.
     * The supported parameters are the same as the ones supported by the JavaScript Date constructor.
     *
     * **Note:** Adjusting the time zone in a running application can lead to unexpected data inconsistencies.
     * For more information, see {@link module:sap/base/i18n/Localization.setTimezone Localization.setTimezone}.
     * See:
     * 	module:sap/base/i18n/Localization.getTimezone
     *
     *
     * @returns The date instance that considers the configured time zone in all local getters and setters.
     */
    static getInstance(
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vYearOrValue?: int | string | Date | UI5Date | null,
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vMonthIndex?: int | string,
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vDay?: int | string,
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vHours?: int | string,
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vMinutes?: int | string,
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vSeconds?: int | string,
      /**
       * Same meaning as in the JavaScript Date constructor
       */
      vMilliseconds?: int | string
    ): Date | UI5Date;
    /**
     * Returns the day of the month of this date instance according to the configured time zone, see `Date.prototype.getDate`.
     *
     *
     * @returns A number between 1 and 31 representing the day of the month of this date instance according
     * to the configured time zone
     */
    getDate(): int;
    /**
     * Returns the day of the week of this date instance according to the configured time zone, see `Date.prototype.getDay`.
     *
     *
     * @returns A number between 0 (Sunday) and 6 (Saturday) representing the day of the week of this date instance
     * according to the configured time zone
     */
    getDay(): int;
    /**
     * Returns the year of this date instance according to the configured time zone, see `Date.prototype.getFullYear`.
     *
     *
     * @returns The year of this date instance according to the configured time zone
     */
    getFullYear(): int;
    /**
     * Returns the hours of this date instance according to the configured time zone, see `Date.prototype.getHours`.
     *
     *
     * @returns A number between 0 and 23 representing the hours of this date instance according to the configured
     * time zone
     */
    getHours(): int;
    /**
     * Returns the milliseconds of this date instance according to the configured time zone, see `Date.prototype.getMilliseconds`.
     *
     *
     * @returns A number between 0 and 999 representing the milliseconds of this date instance according to
     * the configured time zone
     */
    getMilliseconds(): int;
    /**
     * Returns the minutes of this date instance according to the configured time zone, see `Date.prototype.getMinutes`.
     *
     *
     * @returns A number between 0 and 59 representing the minutes of this date instance according to the configured
     * time zone
     */
    getMinutes(): int;
    /**
     * Returns the month index of this date instance according to the configured time zone, see `Date.prototype.getMonth`.
     *
     *
     * @returns The month index between 0 (January) and 11 (December) of this date instance according to the
     * configured time zone
     */
    getMonth(): int;
    /**
     * Returns the seconds of this date instance according to the configured time zone, see `Date.prototype.getSeconds`.
     *
     *
     * @returns A number between 0 and 59 representing the seconds of this date instance according to the configured
     * time zone
     */
    getSeconds(): int;
    /**
     * Returns this date object to the given time represented by a number of milliseconds based on the UNIX
     * epoch, see `Date.prototype.getTime`.
     *
     *
     * @returns The timestamp in milliseconds of this date based on the UNIX epoch, or `NaN` if the date is
     * an invalid date
     */
    getTime(): int;
    /**
     * Returns the difference in minutes between the UTC and the configured time zone for this date, see `Date.prototype.getTimezoneOffset`.
     *
     *
     * @returns The difference in minutes between the UTC and the configured time zone for this date
     */
    getTimezoneOffset(): int;
    /**
     * Returns the day of the month of this date instance according to universal time, see `Date.prototype.getUTCDate`.
     *
     *
     * @returns A number between 1 and 31 representing the day of the month of this date instance according
     * to universal time
     */
    getUTCDate(): int;
    /**
     * Returns the day of the week of this date instance according to universal time, see `Date.prototype.getUTCDay`.
     *
     *
     * @returns A number between 0 (Sunday) and 6 (Saturday) representing the day of the week of this date instance
     * according to universal time
     */
    getUTCDay(): int;
    /**
     * Returns the year of this date instance according to universal time, see `Date.prototype.getUTCFullYear`.
     *
     *
     * @returns The year of this date instance according to universal time
     */
    getUTCFullYear(): int;
    /**
     * Returns the hours of this date instance according to universal time, see `Date.prototype.getUTCHours`.
     *
     *
     * @returns A number between 0 and 23 representing the hours of this date instance according to universal
     * time
     */
    getUTCHours(): int;
    /**
     * Returns the milliseconds of this date instance according to universal time, see `Date.prototype.getUTCMilliseconds`.
     *
     *
     * @returns A number between 0 and 999 representing the milliseconds of this date instance according to
     * universal time
     */
    getUTCMilliseconds(): int;
    /**
     * Returns the minutes of this date instance according to universal time, see `Date.prototype.getUTCMinutes`.
     *
     *
     * @returns A number between 0 and 59 representing the minutes of this date instance according to universal
     * time
     */
    getUTCMinutes(): int;
    /**
     * Returns the month index of this date instance according to universal time, see `Date.prototype.getUTCMonth`.
     *
     *
     * @returns The month index between 0 (January) and 11 (December) of this date instance according to universal
     * time
     */
    getUTCMonth(): int;
    /**
     * Returns the seconds of this date instance according to universal time, see `Date.prototype.getUTCSeconds`.
     *
     *
     * @returns A number between 0 and 59 representing the seconds of this date instance according to universal
     * time
     */
    getUTCSeconds(): int;
    /**
     * Returns the year of this date instance minus 1900 according to the configured time zone, see `Date.prototype.getYear`.
     *
     * @deprecated As of version 1.111. as it is deprecated in the base class JavaScript Date; use {@link #getFullYear }
     * instead
     *
     * @returns The year of this date instance minus 1900 according to the configured time zone
     */
    getYear(): int;
    /**
     * Sets the day of the month for this date instance considering the configured time zone, see `Date.prototype.setDate`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setDate(
      /**
       * An integer representing the new day value, see `Date.prototype.setDate`
       */
      iDay: int
    ): int;
    /**
     * Sets the year, month and day for this date instance considering the configured time zone, see `Date.prototype.setFullYear`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setFullYear(
      /**
       * An integer representing the new year value
       */
      iYear: int,
      /**
       * An integer representing the new month index
       */
      iMonth?: int,
      /**
       * An integer representing the new day value
       */
      iDay?: int
    ): int;
    /**
     * Sets the hours, minutes, seconds and milliseconds for this date instance considering the configured time
     * zone, see `Date.prototype.setHours`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setHours(
      /**
       * An integer representing the new hour value
       */
      iHours: int,
      /**
       * An integer representing the new minutes value
       */
      iMinutes?: int,
      /**
       * An integer representing the new seconds value
       */
      iSeconds?: int,
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds?: int
    ): int;
    /**
     * Sets the milliseconds for this date instance considering the configured time zone, see `Date.prototype.setMilliseconds`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setMilliseconds(
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds: int
    ): int;
    /**
     * Sets the minutes, seconds and milliseconds for this date instance considering the configured time zone,
     * see `Date.prototype.setMinutes`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setMinutes(
      /**
       * An integer representing the new minutes value
       */
      iMinutes: int,
      /**
       * An integer representing the new seconds value
       */
      iSeconds?: int,
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds?: int
    ): int;
    /**
     * Sets the month and day for this date instance considering the configured time zone, see `Date.prototype.setMonth`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setMonth(
      /**
       * An integer representing the new month index
       */
      iMonth: int,
      /**
       * An integer representing the new day value
       */
      iDay?: int
    ): int;
    /**
     * Sets the seconds and milliseconds for this date instance considering the configured time zone, see `Date.prototype.setSeconds`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setSeconds(
      /**
       * An integer representing the new seconds value
       */
      iSeconds: int,
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds?: int
    ): int;
    /**
     * Sets this date object to the given time represented by a number of milliseconds based on the UNIX epoch
     * and resets the previously set date parts, see `Date.prototype.setTime`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setTime(
      /**
       * The date time in milliseconds based in the UNIX epoch
       */
      iTime: int
    ): int;
    /**
     * Sets the day of the month for this date instance according to universal time, see `Date.prototype.setUTCDate`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCDate(
      /**
       * An integer representing the new day value, see `Date.prototype.setUTCDate`
       */
      iDay: int
    ): int;
    /**
     * Sets the year, month and day for this date instance according to universal time, see `Date.prototype.setUTCFullYear`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCFullYear(
      /**
       * An integer representing the new year value
       */
      iYear: int,
      /**
       * An integer representing the new month index
       */
      iMonth?: int,
      /**
       * An integer representing the new day value
       */
      iDay?: int
    ): int;
    /**
     * Sets the hours, minutes, seconds and milliseconds for this date instance according to universal time,
     * see `Date.prototype.setUTCHours`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCHours(
      /**
       * An integer representing the new hour value
       */
      iHours: int,
      /**
       * An integer representing the new minutes value
       */
      iMinutes?: int,
      /**
       * An integer representing the new seconds value
       */
      iSeconds?: int,
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds?: int
    ): int;
    /**
     * Sets the milliseconds for this date instance according to universal time, see `Date.prototype.setUTCMilliseconds`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCMilliseconds(
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds: int
    ): int;
    /**
     * Sets the minutes, seconds and milliseconds for this date instance according to universal time, see `Date.prototype.setUTCMinutes`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCMinutes(
      /**
       * An integer representing the new minutes value
       */
      iMinutes: int,
      /**
       * An integer representing the new seconds value
       */
      iSeconds?: int,
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds?: int
    ): int;
    /**
     * Sets the month and day for this date instance according to universal time, see `Date.prototype.setUTCMonth`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCMonth(
      /**
       * An integer representing the new month index
       */
      iMonth: int,
      /**
       * An integer representing the new day value
       */
      iDay?: int
    ): int;
    /**
     * Sets the seconds and milliseconds for this date instance according to universal time, see `Date.prototype.setUTCSeconds`.
     *
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setUTCSeconds(
      /**
       * An integer representing the new seconds value
       */
      iSeconds: int,
      /**
       * An integer representing the new milliseconds value
       */
      iMilliseconds?: int
    ): int;
    /**
     * Sets the year for this date instance plus 1900 considering the configured time zone, see `Date.prototype.setYear`.
     *
     * @deprecated As of version 1.111. as it is deprecated in the base class JavaScript Date; use {@link #setFullYear }
     * instead
     *
     * @returns The milliseconds of the new timestamp based on the UNIX epoch, or `NaN` if the timestamp could
     * not be updated
     */
    setYear(
      /**
       * The year which is to be set for this date. If iYear is a number between 0 and 99 (inclusive), then the
       * year for this date is set to 1900 + iYear. Otherwise, the year for this date is set to iYear.
       */
      iYear: int
    ): int;
    /**
     * Returns the date portion of this date object interpreted in the configured time zone in English, see
     * `Date.prototype.toDateString`.
     *
     *
     * @returns The date portion of this date object interpreted in the configured time zone in English
     */
    toDateString(): string;
    /**
     * Converts this date to a string, interpreting it in the UTC time zone, see `Date.prototype.toGMTString`.
     *
     *
     * @returns The converted date as string in the UTC time zone
     */
    toGMTString(): string;
    /**
     * Converts this date to a string in ISO format in the UTC offset zero time zone, as denoted by the suffix
     * `Z`, see `Date.prototype.toISOString`.
     *
     *
     * @returns The converted date as a string in ISO format, in the UTC offset zero time zone
     */
    toISOString(): string;
    /**
     * Returns a string representation of this date object, see `Date.prototype.toJSON`.
     *
     *
     * @returns The date object representation as a string
     */
    toJSON(): string;
    /**
     * Returns a string with a language-dependent representation of the date part of this date object interpreted
     * by default in the configured time zone, see `Date.prototype.toLocaleDateString`.
     *
     *
     * @returns The language-dependent representation of the date part of this date object
     */
    toLocaleDateString(
      /**
       * The locale used for formatting; by default, the string representation of {@link module:sap/base/i18n/Localization.getLanguageTag Localization.getLanguageTag}
       */
      sLocale?: string,
      /**
       * The options object used for formatting, corresponding to the options parameter of the `Intl.DateTimeFormat`
       * constructor
       */
      oOptions?: {
        /**
         * The IANA time zone ID; by default {@link module:sap/base/i18n/Localization.getTimezone Localization.getTimezone}
         */
        timeZone?: string;
      }
    ): string;
    /**
     * Returns a string with a language-dependent representation of this date object interpreted by default
     * in the configured time zone, see `Date.prototype.toLocaleString`.
     *
     *
     * @returns The language-dependent representation of this date object
     */
    toLocaleString(
      /**
       * The locale used for formatting; by default, the string representation of {@link module:sap/base/i18n/Localization.getLanguageTag Localization.getLanguageTag}
       */
      sLocale?: string,
      /**
       * The options object used for formatting, corresponding to the options parameter of the `Intl.DateTimeFormat`
       * constructor
       */
      oOptions?: {
        /**
         * The IANA time zone ID; by default {@link module:sap/base/i18n/Localization.getTimezone Localization.getTimezone}
         */
        timeZone?: string;
      }
    ): string;
    /**
     * Returns a string with a language-dependent representation of the time part of this date object interpreted
     * by default in the configured time zone, see `Date.prototype.toLocaleTimeString`.
     *
     *
     * @returns The language-dependent representation of the time part of this date object
     */
    toLocaleTimeString(
      /**
       * The locale used for formatting; by default, the string representation of {@link module:sap/base/i18n/Localization.getLanguageTag Localization.getLanguageTag}
       */
      sLocale?: string,
      /**
       * The options object used for formatting, corresponding to the options parameter of the `Intl.DateTimeFormat`
       * constructor
       */
      oOptions?: {
        /**
         * The IANA time zone ID; by default {@link module:sap/base/i18n/Localization.getTimezone Localization.getTimezone}
         */
        timeZone?: string;
      }
    ): string;
    /**
     * Returns a string representing this date object interpreted in the configured time zone.
     *
     *
     * @returns A string representing this date object interpreted in the configured time zone
     */
    toString(): string;
    /**
     * Returns the time portion of this date object interpreted in the configured time zone in English.
     *
     *
     * @returns The time portion of this date object interpreted in the configured time zone in English
     */
    toTimeString(): string;
    /**
     * Converts this date to a string, interpreting it in the UTC time zone, see `Date.prototype.toUTCString`.
     *
     *
     * @returns The converted date as a string in the UTC time zone
     */
    toUTCString(): string;
    /**
     * Returns the value of this date object in milliseconds based on the UNIX epoch, see `Date.prototype.valueOf`.
     *
     *
     * @returns The primitive value of this date object in milliseconds based on the UNIX epoch
     */
    valueOf(): int;
  }
}

declare module "sap/ui/core/ElementRegistry" {
  import { ID } from "sap/ui/core/library";

  import UI5Element from "sap/ui/core/Element";

  /**
   * Registry of all `sap.ui.core.Element`s that currently exist.
   *
   * @since 1.120
   */
  interface ElementRegistry {
    /**
     * Number of existing elements.
     */
    size: int;

    /**
     * Return an object with all instances of `sap.ui.core.Element`, keyed by their ID.
     *
     * Each call creates a new snapshot object. Depending on the size of the UI, this operation therefore might
     * be expensive. Consider to use the `forEach` or `filter` method instead of executing similar operations
     * on the returned object.
     *
     * **Note**: The returned object is created by a call to `Object.create(null)`, and therefore lacks all
     * methods of `Object.prototype`, e.g. `toString` etc.
     *
     *
     * @returns Object with all elements, keyed by their ID
     */
    all(): Record<ID, UI5Element>;
    /**
     * Returns an array with elements for which the given `callback` returns a value that coerces to `true`.
     *
     * The expected signature of the callback is
     * ```javascript
     *
     *    function callback(oElement, sID)
     * ```
     *  where `oElement` is the currently visited element instance and `sID` is the ID of that instance.
     *
     * If elements are created or destroyed within the `callback`, then the behavior is not specified. Newly
     * added objects might or might not be visited. When an element is destroyed during the filtering and was
     * not visited yet, it might or might not be visited. As the behavior for such concurrent modifications
     * is not specified, it may change in newer releases.
     *
     * If a `thisArg` is given, it will be provided as `this` context when calling `callback`. The `this` value
     * that the implementation of `callback` sees, depends on the usual resolution mechanism. E.g. when `callback`
     * was bound to some context object, that object wins over the given `thisArg`.
     *
     * This function returns an array with all elements matching the given predicate. The order of the elements
     * in the array is not specified and might change between calls (over time and across different versions
     * of UI5).
     *
     *
     * @returns Array of elements matching the predicate; order is undefined and might change in newer versions
     * of UI5
     */
    filter(
      /**
       * predicate against which each element is tested
       */
      callback: (p1: UI5Element, p2: ID) => boolean,
      /**
       * context object to provide as `this` in each call of `callback`
       */
      thisArg?: Object
    ): UI5Element[];
    /**
     * Calls the given `callback` for each element.
     *
     * The expected signature of the callback is
     * ```javascript
     *
     *    function callback(oElement, sID)
     * ```
     *  where `oElement` is the currently visited element instance and `sID` is the ID of that instance.
     *
     * The order in which the callback is called for elements is not specified and might change between calls
     * (over time and across different versions of UI5).
     *
     * If elements are created or destroyed within the `callback`, then the behavior is not specified. Newly
     * added objects might or might not be visited. When an element is destroyed during the filtering and was
     * not visited yet, it might or might not be visited. As the behavior for such concurrent modifications
     * is not specified, it may change in newer releases.
     *
     * If a `thisArg` is given, it will be provided as `this` context when calling `callback`. The `this` value
     * that the implementation of `callback` sees, depends on the usual resolution mechanism. E.g. when `callback`
     * was bound to some context object, that object wins over the given `thisArg`.
     */
    forEach(
      /**
       * Function to call for each element
       */
      callback: (p1: UI5Element, p2: ID) => void,
      /**
       * Context object to provide as `this` in each call of `callback`
       */
      thisArg?: Object
    ): void;
    /**
     * Retrieves an Element by its ID.
     *
     * When the ID is `null` or `undefined` or when there's no element with the given ID, then `undefined` is
     * returned.
     *
     *
     * @returns Element with the given ID or `undefined`
     */
    get(
      /**
       * ID of the element to retrieve
       */
      id: ID
    ): UI5Element | undefined;
  }
  const ElementRegistry: ElementRegistry;
  export default ElementRegistry;
}

declare module "sap/ui/core/fieldhelp/FieldHelpUtil" {
  import UI5Element from "sap/ui/core/Element";

  /**
   * Utility class to set field help information for controls for which field help information cannot be deduced
   * automatically from OData metadata or for which the automatically deduced field help needs to be overwritten.
   * These can be controls like filter fields that don't have OData property bindings.
   *
   * @since 1.133.0
   */
  export default class FieldHelpUtil {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Sets the field help information for the given element as `sap-ui-DocumentationRef` custom data. Note
     * that field help inferred from data bindings of control properties is overwritten by this method unless
     * an empty array is given in parameter `vDocumentationRefs`.
     */
    static setDocumentationRef(
      /**
       * The element on which to set the field help
       */
      oElement: UI5Element,
      /**
       * The string value or an array of string values of `com.sap.vocabularies.Common.v1.DocumentationRef` OData
       * annotations, for example `"urn:sap-com:documentation:key?=type=DE&id=MY_ID&origin=MY_ORIGIN"`"
       */
      vDocumentationRefs: string | string[]
    ): void;
  }
}

declare module "sap/ui/core/getCompatibilityVersion" {
  import Version from "sap/base/util/Version";

  /**
   * Returns the used compatibility version for the given feature.
   *
   * @deprecated As of version 1.119. without a replacement. All features that have been controlled by a compatibility
   * version in UI5 1.x will abandon their legacy behavior, starting with the next major version. In other
   * words, they will behave as if compatibility version "edge" was configured. Due to this, no more access
   * to the compatibility version will be required starting with the next major version.
   *
   * @returns the used compatibility version
   */
  export default function getCompatibilityVersion(
    /**
     * the key of desired feature
     */
    sFeature: string
  ): Version;
}

declare module "sap/ui/core/InvisibleRenderer" {
  import Control from "sap/ui/core/Control";

  import RenderManager from "sap/ui/core/RenderManager";

  import UI5Element from "sap/ui/core/Element";

  /**
   * Provides the default renderer for the controls that have set their `visible` property to `false`.
   *
   * @since 1.66.0
   * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
   */
  interface InvisibleRenderer {
    /**
     * Creates the ID to be used for the invisible placeholder DOM element.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The ID used for the invisible placeholder of this element
     */
    createInvisiblePlaceholderId(
      /**
       * The `control` instance for which to create the placeholder ID
       */
      oControl: Control
    ): string;
    /**
     * Returns the placeholder DOM element of the provided control.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The placeholder DOM element
     */
    getDomRef(
      /**
       * The `control` instance for which to get the placeholder DOM element
       */
      oControl: Control
    ): HTMLElement | null;
    /**
     * Renders an invisible placeholder to identify the location of the invisible control within the DOM tree.
     *
     * The standard implementation renders an invisible <span> element for controls with `visible:false`
     * to improve re-rendering performance. Due to the fault tolerance of the HTML5 standard, such <span>
     * elements are accepted in many scenarios and won't appear in the render tree of the browser. However,
     * in some cases, controls might need to write a different element if <span> is not an allowed element
     * (for example, within the <tr> or <li> group). In this case, the caller can require this module
     * and use the third parameter to define the HTML tag.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    render(
      /**
       * The `RenderManager` instance
       */
      oRm?: RenderManager,
      /**
       * The instance of the invisible element
       */
      oElement?: UI5Element,
      /**
       * HTML tag of the invisible placeholder; void tags are not allowed.
       */
      sTagName?: string
    ): void;
  }
  const InvisibleRenderer: InvisibleRenderer;
  export default InvisibleRenderer;
}

declare module "sap/ui/core/message/MessageType" {
  /**
   * Specifies possible message types.
   *
   * @since 1.118
   */
  enum MessageType {
    /**
     * Message is an error
     */
    Error = "Error",
    /**
     * Message should be just an information
     */
    Information = "Information",
    /**
     * Message has no specific level
     */
    None = "None",
    /**
     * Message is a success message
     */
    Success = "Success",
    /**
     * Message is a warning
     */
    Warning = "Warning",
  }
  export default MessageType;
}

declare module "sap/ui/core/Messaging" {
  import Message from "sap/ui/core/message/Message";

  import MessageModel from "sap/ui/model/message/MessageModel";

  import MessageProcessor from "sap/ui/core/message/MessageProcessor";

  import ManagedObject from "sap/ui/base/ManagedObject";

  /**
   * Messaging provides a central place for managing `sap.ui.core.message.Messages`.
   *
   * @since 1.118.0
   */
  interface Messaging {
    /**
     * Add messages to Messaging
     */
    addMessages(
      /**
       * Array of sap.ui.core.message.Message or single sap.ui.core.message.Message
       */
      vMessages: Message | Message[]
    ): void;
    /**
     * Get the MessageModel
     *
     *
     * @returns oMessageModel The Message Model
     */
    getMessageModel(): MessageModel;
    /**
     * Register MessageProcessor
     */
    registerMessageProcessor(
      /**
       * The MessageProcessor
       */
      oProcessor: MessageProcessor
    ): void;
    /**
     * When using the databinding type system, the validation/parsing of a new property value could fail. In
     * this case, a validationError/parseError event is fired. These events bubble up to the core. For registered
     * ManagedObjects, the Messaging attaches to these events and creates a `sap.ui.core.message.Message` (bHandleValidation=true)
     * for each of these errors and cancels the event bubbling.
     */
    registerObject(
      /**
       * The sap.ui.base.ManagedObject
       */
      oObject: ManagedObject,
      /**
       * Handle validationError/parseError events for this object. If set to true, the Messaging creates a Message
       * for each validation/parse error. The event bubbling is canceled in every case.
       */
      bHandleValidation: boolean
    ): void;
    /**
     * Remove all messages
     */
    removeAllMessages(): void;
    /**
     * Remove given Messages
     */
    removeMessages(
      /**
       * The message(s) to be removed.
       */
      vMessages: Message | Message[]
    ): void;
    /**
     * Deregister MessageProcessor
     */
    unregisterMessageProcessor(
      /**
       * The MessageProcessor
       */
      oProcessor: MessageProcessor
    ): void;
    /**
     * Unregister ManagedObject
     */
    unregisterObject(
      /**
       * The sap.ui.base.ManagedObject
       */
      oObject: ManagedObject
    ): void;
    /**
     * Update Messages by providing two arrays of old and new messages.
     *
     * The old ones will be removed, the new ones will be added.
     */
    updateMessages(
      /**
       * Array of old messages to be removed
       */
      aOldMessages: Message[],
      /**
       * Array of new messages to be added
       */
      aNewMessages: Message[]
    ): void;
  }
  const Messaging: Messaging;
  export default Messaging;
}

declare module "sap/ui/core/StaticArea" {
  import UIArea from "sap/ui/core/UIArea";

  /**
   * Helper module to access the static area.
   *
   * The static area is a hidden DOM element with a unique ID and managed by the framework. It is typically
   * used for hidden or temporarily hidden elements like InvisibleText, Popup, Shadow, Blocklayer etc.
   *
   * All methods throw an `Error` when they are called before the document is ready.
   */
  interface StaticArea {
    /**
     * Checks whether the given DOM element is part of the static area.
     *
     *
     * @returns Whether the given DOM reference is part of the static UIArea
     */
    contains(
      /**
       * The DOM element to check
       */
      oDomRef: Element
    ): boolean;
    /**
     * Returns the root element of the static, hidden area.
     *
     * It can be used e.g. for hiding elements like Popup, Shadow, Blocklayer etc. If it is not yet available,
     * a DIV element is created and appended to the body.
     *
     *
     * @returns the root DOM element of the static, hidden area
     */
    getDomRef(): Element;
    /**
     * Returns the `UIArea` instance for the static area. If none exists yet, one gets created.
     *
     *
     * @returns The `UIArea` instance for the static area
     */
    getUIArea(): UIArea;
  }
  const StaticArea: StaticArea;
  export default StaticArea;
}

declare module "sap/ui/core/syncStyleClass" {
  import Control from "sap/ui/core/Control";

  /**
   * Search ancestors of the given source DOM element for the specified CSS class name.
   *
   * If the class name is found, set it to the root DOM element of the target control. If the class name is
   * not found, it is also removed from the target DOM element.
   *
   * @since 1.58
   *
   * @returns Target element
   */
  export default function syncStyleClass(
    /**
     * CSS class name
     */
    sStyleClass: string,
    /**
     * jQuery object, control or an id of the source element.
     */
    vSource: jQuery | Control | string,
    /**
     * target jQuery object or a control.
     */
    vDestination: jQuery | Control
  ): jQuery | Element;
}

declare module "sap/ui/core/Theming" {
  /**
   * Provides theming related API
   *
   * @since 1.118
   */
  interface Theming {
    /**
     * Attaches event handler `fnFunction` to the {@link #event:applied applied} event.
     *
     * The given handler is called when the the applied event is fired. If the theme is already applied the
     * handler will be called immediately. The handler stays attached to the applied event for future theme
     * changes.
     *
     * @since 1.118.0
     */
    attachApplied(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Theming$AppliedEvent) => void
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:applied applied} event
     *
     * The passed function must match the one used for event registration.
     *
     * @since 1.118.0
     */
    detachApplied(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Theming$AppliedEvent) => void
    ): void;
    /**
     * Returns the theme name
     *
     * @since 1.118
     *
     * @returns the theme name
     */
    getTheme(): string;
    /**
     * Notify content density changes
     *
     * @since 1.118.0
     */
    notifyContentDensityChanged(): void;
    /**
     * Sets the favicon. The path must be relative to the current origin. Absolute URLs are not allowed.
     *
     * @since 1.135
     *
     * @returns A promise that resolves when the favicon has been set with undefined
     */
    setFavicon(
      /**
       * A string containing a specific relative path to the favicon, 'true' to use a favicon from custom theme
       * or the default favicon in case no custom favicon is maintained, 'false' or undefined to disable the favicon
       */
      vFavicon: string | boolean | undefined
    ): Promise<undefined>;
    /**
     * Allows setting the theme name
     *
     * @since 1.118
     */
    setTheme(
      /**
       * the theme name
       */
      sTheme: string
    ): void;
  }
  const Theming: Theming;
  export default Theming;

  /**
   * The theme applied Event.
   *
   * @since 1.118.0
   */
  export type Theming$AppliedEvent = {
    /**
     * The newly set theme.
     */
    theme: string;
  };
}

declare module "sap/ui/dom/containsOrEquals" {
  /**
   * Returns whether `oDomRefChild` is contained in or equal to `oDomRefContainer`.
   *
   * For compatibility reasons it returns `true` if `oDomRefContainer` and `oDomRefChild` are equal.
   *
   * This method intentionally does not operate on the jQuery object, as the original `jQuery.contains()`
   * method also does not do so.
   *
   * @since 1.58
   *
   * @returns Whether `oDomRefChild` is contained in or equal to `oDomRefContainer`
   */
  export default function containsOrEquals(
    /**
     * The container element
     */
    oDomRefContainer: Element,
    /**
     * The child element (must not be a text node, must be an element)
     */
    oDomRefChild: Element
  ): boolean;
}

declare module "sap/ui/dom/denormalizeScrollBeginRTL" {
  /**
   * For the given scroll position measured from the "beginning" of a container (the right edge in RTL mode)
   * this method returns the scrollLeft value as understood by the current browser in RTL mode. This value
   * is specific to the given DOM element, as the computation may involve its dimensions.
   *
   * So when oDomRef should be scrolled 2px from the beginning, the number "2" must be given as `iNormalizedScrollBegin`
   * and the result of this method (which may be a large or even negative number, depending on the browser)
   * can then be set as `oDomRef.scrollLeft` to achieve the desired (cross-browser-consistent) scrolling position.
   * Low values make the right part of the content visible, high values the left part.
   *
   * This method does no scrolling on its own, it only calculates the value to set (so it can also be used
   * for animations).
   *
   * Only use this method in RTL mode, as the behavior in LTR mode is undefined and may change!
   *
   * @since 1.58
   *
   * @returns The scroll position that must be set for the DOM element
   */
  export default function denormalizeScrollBeginRTL(
    /**
     * The distance from the rightmost position to which the element should be scrolled
     */
    iNormalizedScrollBegin: int,
    /**
     * The DOM Element to which scrollLeft will be applied
     */
    oDomRef: Element
  ): int;
}

declare module "sap/ui/dom/denormalizeScrollLeftRTL" {
  /**
   * For the given scrollLeft value this method returns the scrollLeft value as understood by the current
   * browser in RTL mode. This value is specific to the given DOM element, as the computation may involve
   * its dimensions.
   *
   * So when oDomRef should be scrolled 2px from the leftmost position, the number "2" must be given as `iNormalizedScrollLeft`
   * and the result of this method (which may be a large or even negative number, depending on the browser)
   * can then be set as `oDomRef.scrollLeft` to achieve the desired (cross-browser-consistent) scrolling position.
   *
   * This method does no scrolling on its own, it only calculates the value to set (so it can also be used
   * for animations).
   *
   * @since 1.58
   *
   * @returns The scroll position that must be set for the DOM element
   */
  export default function denormalizeScrollLeftRTL(
    /**
     * The distance from the leftmost position to which the element should be scrolled
     */
    iNormalizedScrollLeft: int,
    /**
     * The DOM Element to which scrollLeft will be applied
     */
    oDomRef: Element
  ): int;
}

declare module "sap/ui/dom/getOwnerWindow" {
  /**
   * Returns the window reference for a DomRef.
   *
   * @since 1.58
   *
   * @returns Window reference
   */
  export default function getOwnerWindow(
    /**
     * The DOM reference
     */
    oDomRef: Element
  ): Window;
}

declare module "sap/ui/dom/getScrollbarSize" {
  /**
   * Returns the size (width of the vertical / height of the horizontal) native browser scrollbars.
   *
   * This function must only be used when the DOM is ready.
   *
   * @since 1.58
   *
   * @returns Object with properties `width` and `height` (the values are of type number and are pixels).
   */
  export default function getScrollbarSize(
    /**
     * The CSS class that should be added to the test element.
     */
    sClasses?: string,
    /**
     * Force recalculation of size (e.g. when CSS was changed). When no classes are passed all calculated sizes
     * are reset.
     */
    bForce?: boolean
  ): {
    width: number;

    height: number;
  };
}

declare module "sap/ui/dom/includeScript" {
  /**
   * Includes the script (via <script>-tag) into the head for the specified `sUrl` and optional `sId`.
   *
   * @since 1.58
   *
   * @returns When using the configuration object a `Promise` will be returned. The documentation for the
   * `fnLoadCallback` applies to the `resolve` handler of the `Promise` and the one for the `fnErrorCallback`
   * applies to the `reject` handler of the `Promise`.
   */
  export default function includeScript(
    /**
     * the URL of the script to load or a configuration object
     */
    vUrl:
      | string
      | {
          /**
           * the URL of the script to load
           */
          url: string;
          /**
           * id that should be used for the script tag
           */
          id?: string;
          /**
           * map of attributes that should be used for the script tag
           */
          attributes?: object;
        },
    /**
     * id that should be used for the script tag or map of attributes
     */
    vId?: string | object,
    /**
     * callback function to get notified once the script has been loaded
     */
    fnLoadCallback?: (p1: Event) => void,
    /**
     * callback function to get notified once the script loading failed
     */
    fnErrorCallback?: (p1: Event) => void
  ): void | Promise<Event>;
}

declare module "sap/ui/dom/includeStylesheet" {
  /**
   * Includes the specified stylesheet via a <link>-tag in the head of the current document.
   *
   * If `includeStylesheet` is called with an `sId` of an already included stylesheet and:
   * 	 - either `fnLoadCallback` or `fnErrorCallback` is given: the old stylesheet is deleted and a new one
   *     is inserted
   * 	 - `vUrl` is different from the existing one's: the old stylesheet is deleted and a new one is inserted
   *
   * 	 - otherwise: no action
   *
   * @since 1.58
   *
   * @returns When using the configuration object a `Promise` will be returned. The documentation for the
   * `fnLoadCallback` applies to the `resolve` handler of the `Promise` and the one for the `fnErrorCallback`
   * applies to the `reject` handler of the `Promise`.
   */
  export default function includeStylesheet(
    /**
     * the URL of the stylesheet to load or a configuration object
     */
    vUrl:
      | string
      | {
          /**
           * the URL of the stylesheet to load
           */
          url: string;
          /**
           * id that should be used for the link tag
           */
          id?: string;
          /**
           * map of attributes that should be used for the script tag
           */
          attributes?: object;
        },
    /**
     * id that should be used for the link tag or map of attributes
     */
    vId?: string | object,
    /**
     * callback function to get notified once the stylesheet has been loaded
     */
    fnLoadCallback?: (p1: Event) => void,
    /**
     * callback function to get notified once the stylesheet loading failed.
     */
    fnErrorCallback?: (p1: Event) => void
  ): void | Promise<Event>;
}

declare module "sap/ui/events/checkMouseEnterOrLeave" {
  /**
   * Checks a given mouseover or mouseout event whether it is equivalent to a mouseenter or mouseleave event
   * regarding the given DOM reference.
   *
   * @since 1.58
   *
   * @returns True if the provided event is equivalent
   */
  export default function checkMouseEnterOrLeave(
    /**
     * The Mouse Event
     */
    oEvent: jQuery.Event,
    /**
     * The domref of the element to check
     */
    oDomRef: Element
  ): boolean;
}

declare module "sap/ui/events/ControlEvents" {
  /**
   * @since 1.58
   */
  interface ControlEvents {
    /**
     * List of DOM events that a UIArea automatically takes care of.
     *
     * A control/element doesn't have to bind listeners for these events. It instead can implement an `onevent(oEvent)`
     * method for any of the following events that it wants to be notified about:
     *
     * click, dblclick, contextmenu, focusin, focusout, keydown, keypress, keyup, mousedown, mouseout, mouseover,
     * mouseup, select, selectstart, dragstart, dragenter, dragover, dragleave, dragend, drop, paste, cut, input,
     * touchstart, touchend, touchmove, touchcancel, tap, swipe, swipeleft, swiperight, scrollstart, scrollstop
     *
     * The mouse events and touch events are supported simultaneously on both desktop and mobile browsers. Do
     * NOT create both onmouse* and ontouch* functions to avoid one event being handled twice on the same control.
     */
    events: undefined;

    /**
     * Binds all events for listening with the given callback function.
     */
    bindAnyEvent(
      /**
       * Callback function
       */
      fnCallback: (p1: Event) => void
    ): void;
    /**
     * Unbinds all events for listening with the given callback function.
     */
    unbindAnyEvent(
      /**
       * Callback function
       */
      fnCallback: (p1: Event) => void
    ): void;
  }
  const ControlEvents: ControlEvents;
  export default ControlEvents;
}

declare module "sap/ui/events/KeyCodes" {
  /**
   * @since 1.58
   */
  enum KeyCodes {
    A = "65",

    ALT = "18",

    ARROW_DOWN = "40",

    ARROW_LEFT = "37",

    ARROW_RIGHT = "39",

    ARROW_UP = "38",

    B = "66",

    BACKSLASH = "226",

    BACKSPACE = "8",

    BREAK = "19",

    C = "67",

    CAPS_LOCK = "20",

    COMMA = "188",

    CONTEXT_MENU = "93",

    CONTROL = "17",

    D = "68",

    DELETE = "46",

    DIGIT_0 = "48",

    DIGIT_1 = "49",

    DIGIT_2 = "50",

    DIGIT_3 = "51",

    DIGIT_4 = "52",

    DIGIT_5 = "53",

    DIGIT_6 = "54",

    DIGIT_7 = "55",

    DIGIT_8 = "56",

    DIGIT_9 = "57",

    DOT = "190",

    E = "69",

    END = "35",

    ENTER = "13",

    EQUALS = "221",

    ESCAPE = "27",

    F = "70",

    F1 = "112",

    F10 = "121",

    F11 = "122",

    F12 = "123",

    F2 = "113",

    F3 = "114",

    F4 = "115",

    F5 = "116",

    F6 = "117",

    F7 = "118",

    F8 = "119",

    F9 = "120",

    G = "71",

    GREAT_ACCENT = "220",

    H = "72",

    HOME = "36",

    I = "73",

    INSERT = "45",

    J = "74",

    K = "75",

    L = "76",

    M = "77",

    MINUS = "219",

    N = "78",

    NUM_LOCK = "144",

    NUMPAD_0 = "96",

    NUMPAD_1 = "97",

    NUMPAD_2 = "98",

    NUMPAD_3 = "99",

    NUMPAD_4 = "100",

    NUMPAD_5 = "101",

    NUMPAD_6 = "102",

    NUMPAD_7 = "103",

    NUMPAD_8 = "104",

    NUMPAD_9 = "105",

    NUMPAD_ASTERISK = "106",

    NUMPAD_COMMA = "110",

    NUMPAD_MINUS = "109",

    NUMPAD_PLUS = "107",

    NUMPAD_SLASH = "111",

    O = "79",

    OPEN_BRACKET = "186",

    P = "80",

    PAGE_DOWN = "34",

    PAGE_UP = "33",

    PIPE = "191",

    PLUS = "187",

    PRINT = "44",

    Q = "81",

    R = "82",

    S = "83",

    SCROLL_LOCK = "145",

    SEMICOLON = "192",

    SHIFT = "16",

    SINGLE_QUOTE = "222",

    SLASH = "189",

    SLEEP = "95",

    SPACE = "32",

    T = "84",

    TAB = "9",

    TURN_OFF = "94",

    U = "85",

    V = "86",

    W = "87",

    WINDOWS = "91",

    X = "88",

    Y = "89",

    Z = "90",
  }
  export default KeyCodes;
}

declare module "sap/ui/events/PseudoEvents" {
  /**
   * @since 1.58
   */
  interface PseudoEvents {
    /**
     * Map of all so called "pseudo events", a useful classification of standard browser events as implied by
     * SAP product standards.
     *
     * This map is intended to be used internally in UI5 framework and UI5 Controls.
     *
     * Whenever a browser event is recognized as one or more pseudo events, then this classification is attached
     * to the original {@link jQuery.Event} object and thereby delivered to any jQuery-style listeners registered
     * for that browser event.
     *
     * Pure JavaScript listeners can evaluate the classification information using the {@link jQuery.Event.prototype.isPseudoType }
     * method.
     *
     * Instead of using the procedure as described above, the SAPUI5 controls and elements should simply implement
     * an `onpseudo-event(oEvent)` method. It will be invoked only when that specific pseudo event has
     * been recognized. This simplifies event dispatching even further.
     */
    events: Record<
      string,
      {
        sName: string;

        aTypes: string[];

        fnCheck: (p1: Event) => boolean;
      }
    >;

    /**
     * Ordered array of the {@link module:sap/ui/events/PseudoEvents.events}.
     *
     * Order is significant as some check methods rely on the fact that they are tested before other methods.
     * The array is processed during event analysis (when classifying browser events as pseudo events).
     */
    order: undefined;
  }
  const PseudoEvents: PseudoEvents;
  export default PseudoEvents;
}

declare module "sap/ui/model/FilterProcessor" {
  import Filter from "sap/ui/model/Filter";

  /**
   * Helper class for processing of filter objects
   *
   * @since 1.71
   */
  interface FilterProcessor {
    /**
     * Groups filters according to their path and combines filters on the same path using "OR" and filters on
     * different paths using "AND", all multi-filters contained are ANDed.
     *
     * @since 1.71
     *
     * @returns A single filter containing all filters of the array combined or `undefined` if no filters are
     * given
     */
    groupFilters(
      /**
       * The filters to be grouped
       */
      aFilters?: Filter[]
    ): Filter | undefined;
  }
  const FilterProcessor: FilterProcessor;
  export default FilterProcessor;
}

declare module "sap/ui/model/odata/ODataExpressionAddons" {
  /**
   * Module to enable the expression binding syntax elements `odata.*`
   */
  interface ODataExpressionAddons {}
  const ODataExpressionAddons: ODataExpressionAddons;
  export default ODataExpressionAddons;
}

declare module "sap/ui/model/odata/v2/ODataModel" {
  import {
    default as Model,
    Model$RequestCompletedEventParameters,
    Model$RequestFailedEventParameters,
    Model$RequestSentEventParameters,
  } from "sap/ui/model/Model";

  import BindingMode from "sap/ui/model/BindingMode";

  import CountMode from "sap/ui/model/odata/CountMode";

  import OperationMode from "sap/ui/model/odata/OperationMode";

  import UpdateMethod from "sap/ui/model/odata/UpdateMethod";

  import Context from "sap/ui/model/Context";

  import ODataContextBinding from "sap/ui/model/odata/v2/ODataContextBinding";

  import Sorter from "sap/ui/model/Sorter";

  import Filter from "sap/ui/model/Filter";

  import ODataListBinding from "sap/ui/model/odata/v2/ODataListBinding";

  import PropertyBinding from "sap/ui/model/PropertyBinding";

  import ODataTreeBinding from "sap/ui/model/odata/v2/ODataTreeBinding";

  import Context1 from "sap/ui/model/odata/v2/Context";

  import MessageScope from "sap/ui/model/odata/MessageScope";

  import Metadata from "sap/ui/base/Metadata";

  import ODataMetaModel from "sap/ui/model/odata/ODataMetaModel";

  import Event from "sap/ui/base/Event";

  import { Source } from "sap/ui/model/odata/v2/ODataAnnotations";

  /**
   * The error object passed to the retry after callback.
   *
   * @since 1.134.0
   */
  export type RetryAfterError = Error & {
    /**
     * Error message returned by the 503 HTTP status response
     */
    message: string;
    /**
     * The earliest point in time the request may be repeated
     */
    retryAfter: Date;
  };

  /**
   * Model implementation based on the OData protocol.
   *
   * See chapter {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2 OData V2 Model} for a
   * general introduction.
   *
   * This model is not prepared to be inherited from.
   *
   * @since 1.24.0
   */
  export default class ODataModel extends Model {
    /**
     * Constructor for a new ODataModel.
     */
    constructor(
      /**
       * Base URI of the service to request data from; additional URL parameters appended here will be appended
       * to every request. If you pass an object, it will be interpreted as the parameter object (second parameter).
       * Then `mParameters.serviceUrl` becomes a mandatory parameter.
       */
      vServiceUrl: string | object,
      /**
       * Map which contains the following parameter properties:
       */
      mParameters?: {
        /**
         * The URL (or an array of URLs) from which the annotation metadata should be loaded
         */
        annotationURI?: string | string[];
        /**
         * Set this array to make custom response headers bindable via the entity's "__metadata/headers" property
         */
        bindableResponseHeaders?: string[];
        /**
         * Whether the model tries to calculate canonical URLs to request the data.
         *
         * **For example:** An application displays the details of a sales order in a form with an absolute binding
         * path `/SalesOrderSet("1")`. The form embeds a table for the sales order line items with a relative binding
         * path `ToLineItems`. If the user selects a sales order line item (e.g. Item "10"), the details of this
         * sales order line item are displayed in another form, which also contains a table for the sales order
         * line item's schedules with a relative binding path `ToSchedules`.
         *
         * If the `canonicalRequests` parameter has the default value `false`, then the OData model would request
         * the data for the sales order line item's details form with the following requests:
         * ```javascript
         *
         *   GET /<serviceUrl>/SalesOrderSet("1")/ToLineItems(SalesOrderID="1",ItemPosition="10")
         *   GET /<serviceUrl>/SalesOrderSet("1")/ToLineItems(SalesOrderID="1",ItemPosition="10")/ToSchedules```
         *
         *
         * Some back-end implementations do not support more than one navigation property in the resource URL. In
         * this case, set the `canonicalRequests` parameter to `true`. The OData model then converts the long resource
         * URLs to canonical URLs and requests the data for the sales order line item's details form with the following
         * requests:
         * ```javascript
         *
         *   GET /<serviceUrl>/SalesOrderLineItemsSet(SalesOrderID="1",ItemPosition="10")
         *   GET /<serviceUrl>/SalesOrderLineItemsSet(SalesOrderID="1",ItemPosition="10")/ToSchedules```
         */
        canonicalRequests?: boolean;
        /**
         * Sets the default binding mode for the model
         */
        defaultBindingMode?: BindingMode | keyof typeof BindingMode;
        /**
         * Sets the default count mode for the model
         */
        defaultCountMode?: CountMode | keyof typeof CountMode;
        /**
         * Sets the default operation mode for the model
         */
        defaultOperationMode?: OperationMode | keyof typeof OperationMode;
        /**
         * Default update method which is used for all update requests
         */
        defaultUpdateMethod?: UpdateMethod;
        /**
         * Set this flag to `true` if your service does not support `HEAD` requests for fetching the service document
         * (and thus the security token) to avoid sending a `HEAD`-request before falling back to `GET`
         */
        disableHeadRequestForToken?: boolean;
        /**
         * Set this flag to `true` if you don´t want to start a new soft state session with context ID (`SID`) through
         * header mechanism. This is useful if you want to share a `SID` between different browser windows
         */
        disableSoftStateHeader?: boolean;
        /**
         * Whether the security token is requested at the earliest convenience, if parameter `tokenHandling` is
         * `true`; supported since 1.79.0.
         */
        earlyTokenRequest?: boolean;
        /**
         * Map of custom headers (name/value pairs) like {"myHeader":"myHeaderValue",...}
         */
        headers?: Record<string, string>;
        /**
         * Whether to ignore all annotations from service metadata, so that they are not available as V4 annotations
         * in this model's metamodel; see {@link #getMetaModel}. Only annotations from annotation files are loaded;
         * see the `annotationURI` parameter. Supported since 1.121.0
         */
        ignoreAnnotationsFromMetadata?: boolean;
        /**
         * If set to `true`, request payloads will be JSON, XML for `false`
         */
        json?: boolean;
        /**
         * Whether the `metadataLoaded` event will be fired only after all annotations have been loaded as well
         */
        loadAnnotationsJoined?: boolean;
        /**
         * Please use the following string format e.g. '2.0' or '3.0'. OData version supported by the ODataModel:
         * '2.0'
         */
        maxDataServiceVersion?: string;
        /**
         * Map of namespace aliases (alias => URI) that can be used in metadata binding paths; each alias is mapped
         * to a corresponding namespace URI; when an alias is used in a metadata binding path, it addresses a metadata
         * extension that belongs to the corresponding namespace URI; if `metadataNamespaces` is not given, the
         * following default mappings will be used:
         * 	 - `"sap": "sap:"http://www.sap.com/Protocols/SAPData"`
         * 	 - `"m": "http://schemas.microsoft.com/ado/2007/08/dataservices/metadata"`
         * 	 - `"": "http://schemas.microsoft.com/ado/2007/06/edmx`
         */
        metadataNamespaces?: Record<string, string>;
        /**
         * Map of URL parameters for metadata requests - only attached to a `$metadata` request
         */
        metadataUrlParams?: Record<string, string>;
        /**
         * Whether technical messages should always be treated as persistent, since 1.83.0
         */
        persistTechnicalMessages?: boolean;
        /**
         * Whether a preliminary context will be created/used by a binding. When set to `true`, the model can bundle
         * the OData calls for dependent bindings into fewer $batch requests. For more information, see {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio62149734b5c24507868e722fe87a75db Optimizing Dependent Bindings}
         */
        preliminaryContext?: boolean;
        /**
         * Enable/disable automatic refresh after change operations
         */
        refreshAfterChange?: boolean;
        /**
         * Whether to sequentialize all requests, needed in case the service cannot handle parallel requests. **Deprecated**
         * as of version 1.128.0, the concept has been discarded.
         */
        sequentializeRequests?: boolean;
        /**
         * Base URI of the service to request data from; this property is mandatory when the first method parameter
         * `serviceUrl` is omitted, but ignored otherwise
         */
        serviceUrl?: string;
        /**
         * Map of URL parameters (name/value pairs) - these parameters will be attached to all requests, except
         * for the `$metadata` request
         */
        serviceUrlParams?: Record<string, string>;
        /**
         * Enable/disable security token handling. If the "skipServerCache" string value is provided, the security
         * token is not cached with the server as key in order to avoid failing $batch requests when accessing services
         * running on different back-end systems behind a reverse proxy (since 1.119).
         *  Use this option only if the system landscape is known.
         */
        tokenHandling?: boolean | "skipServerCache";
        /**
         * Send security token for GET requests in case read access logging is activated for the OData Service in
         * the backend.
         */
        tokenHandlingForGet?: boolean;
        /**
         * Whether all requests should be sent in batch requests
         */
        useBatch?: boolean;
        /**
         * If set to `true`, the user credentials are included in a cross-origin request. **Note:** This only works
         * if all requests are asynchronous.
         */
        withCredentials?: boolean;
        /**
         * **Deprecated** for security reasons. Use strong server side authentication instead. Password for the
         * service.
         */
        password?: string;
        /**
         * **Deprecated** This parameter does not prevent creation of annotations from the metadata document in
         * this model's metamodel. Whether to skip the automated loading of annotations from the metadata document.
         * Loading annotations from metadata does not have any effects (except the lost performance by invoking
         * the parser) if there are no annotations inside the metadata document
         */
        skipMetadataAnnotationParsing?: boolean;
        /**
         * **Deprecated** for security reasons. Use strong server side authentication instead. UserID for the service.
         */
        user?: string;
      }
    );

    /**
     * Returns a metadata object for class sap.ui.model.odata.v2.ODataModel.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Returns a promise that resolves with an array containing information about the initially loaded annotations.
     *
     * **Important**: This covers the annotations that were given to the model constructor, not the ones that
     * might have been added later on using the API method {@link sap.ui.model.odata.ODataMetaModel#getODataValueLists}.
     * In order to get information about those, the event `annotationsLoaded` can be used.
     *
     * @since 1.42
     *
     * @returns A promise that resolves with an array containing information about the initially loaded annotations
     */
    annotationsLoaded(): Promise<any>;
    /**
     * Attaches event handler `fnFunction` to the `annotationsFailed` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAnnotationsFailed(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$AnnotationsFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `annotationsFailed` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAnnotationsFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$AnnotationsFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `annotationsLoaded` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAnnotationsLoaded(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$AnnotationsLoadedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `annotationsLoaded` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAnnotationsLoaded(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$AnnotationsLoadedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:batchRequestCompleted batchRequestCompleted }
     * event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBatchRequestCompleted(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestCompletedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:batchRequestCompleted batchRequestCompleted }
     * event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBatchRequestCompleted(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestCompletedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:batchRequestFailed batchRequestFailed} event
     * of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBatchRequestFailed(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:batchRequestFailed batchRequestFailed} event
     * of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBatchRequestFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:batchRequestSent batchRequestSent} event of
     * this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBatchRequestSent(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestSentEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:batchRequestSent batchRequestSent} event of
     * this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBatchRequestSent(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestSentEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `metadataFailed` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachMetadataFailed(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$MetadataFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `metadataFailed` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachMetadataFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$MetadataFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `metadataLoaded` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachMetadataLoaded(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$MetadataLoadedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the `metadataLoaded` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachMetadataLoaded(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$MetadataLoadedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.model.odata.v2.ODataModel` itself
       */
      oListener?: object
    ): this;
    /**
     * Creates a context binding for this model.
     * See:
     * 	sap.ui.model.Model.prototype.bindContext
     *
     *
     * @returns The new context binding
     */
    bindContext(
      /**
       * The binding path in the model
       */
      sPath: string,
      /**
       * The context which is required as base for a relative path.
       */
      oContext?: Context,
      /**
       * A map which contains additional parameters for the binding.
       */
      mParameters?: {
        /**
         * Whether a preliminary context is created
         */
        createPreliminaryContext?: boolean;
        /**
         * An optional map of custom query parameters. Custom parameters must not start with `$`.
         */
        custom?: Record<string, string>;
        /**
         * Value for the OData `$expand` query option parameter which is included in the request after URL encoding
         * of the given value.
         */
        expand?: string;
        /**
         * The group id to be used for requests originating from the binding
         */
        groupId?: string;
        /**
         * Value for the OData `$select` query option parameter which is included in the request after URL encoding
         * of the given value.
         */
        select?: string;
        /**
         * Whether a preliminary context is used. When set to `true`, the model can bundle the OData calls for dependent
         * bindings into fewer $batch requests. For more information, see {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio62149734b5c24507868e722fe87a75db Optimizing Dependent Bindings}.
         */
        usePreliminaryContext?: boolean;
        /**
         * **Deprecated**, use `groupId` instead. Sets the batch group id to be used for requests originating from
         * the binding.
         */
        batchGroupId?: string;
      }
    ): ODataContextBinding;
    /**
     * Creates a new list binding for this model.
     * See:
     * 	sap.ui.model.Model.prototype.bindList
     *
     *
     * @returns The new list binding
     */
    bindList(
      /**
       * The binding path in the model
       */
      sPath: string,
      /**
       * The context which is required as base for a relative path.
       */
      oContext?: Context,
      /**
       * The sorters used initially; call {@link sap.ui.model.odata.v2.ODataListBinding#sort} to replace them
       */
      aSorters?: Sorter[] | Sorter,
      /**
       * The filters to be used initially with type {@link sap.ui.model.FilterType.Application}; call {@link sap.ui.model.odata.v2.ODataListBinding#filter }
       * to replace them
       */
      aFilters?: Filter[] | Filter,
      /**
       * A map which contains additional parameters for the binding.
       */
      mParameters?: {
        /**
         * Defines the count mode of the binding; if not specified, the default count mode of the `oModel` is applied.
         */
        countMode?: CountMode | keyof typeof CountMode;
        /**
         * A key used in combination with the resolved path of the binding to identify the entities created by the
         * binding's {@link #create} method.
         *
         * **Note:** Different controls or control aggregation bindings to the same collection must have different
         * `createdEntitiesKey` values.
         */
        createdEntitiesKey?: string;
        /**
         * An optional map of custom query parameters. Custom parameters must not start with `$`.
         */
        custom?: Record<string, string>;
        /**
         * Value for the OData `$expand` query option parameter which is included in the data request after URL
         * encoding of the given value.
         */
        expand?: string;
        /**
         * Turns on the fault tolerance mode, data is not reset if a back-end request returns an error.
         */
        faultTolerant?: boolean;
        /**
         * The group id to be used for requests originating from the binding
         */
        groupId?: string;
        /**
         * The operation mode of the binding
         */
        operationMode?: OperationMode | keyof typeof OperationMode;
        /**
         * Value for the OData `$select` query option parameter which is included in the data request after URL
         * encoding of the given value.
         */
        select?: string;
        /**
         * Whether the list binding only requests transition messages from the back end. If messages for entities
         * of this collection need to be updated, use {@link sap.ui.model.odata.v2.ODataModel#read} on the parent
         * entity corresponding to the list binding's context, with the parameter `updateAggregatedMessages` set
         * to `true`.
         */
        transitionMessagesOnly?: boolean;
        /**
         * Whether a preliminary context is used. When set to `true`, the model can bundle the OData calls for dependent
         * bindings into fewer $batch requests. For more information, see {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio62149734b5c24507868e722fe87a75db Optimizing Dependent Bindings}.
         */
        usePreliminaryContext?: boolean;
        /**
         * **Deprecated**, use `groupId` instead. Sets the batch group id to be used for requests originating from
         * the binding.
         */
        batchGroupId?: string;
        /**
         * Deprecated since 1.102.0, as {@link sap.ui.model.odata.OperationMode.Auto} is deprecated; the threshold
         * that defines how many entries should be fetched at least by the binding if `operationMode` is set to
         * `Auto`.
         */
        threshold?: int;
      }
    ): ODataListBinding;
    /**
     * Creates a new property binding for this model.
     * See:
     * 	sap.ui.model.Model#bindProperty
     * 	#getProperty
     *
     *
     * @returns The new property binding
     */
    bindProperty(
      /**
       * Path pointing to the property that should be bound; either an absolute path or a path relative to a given
       * `oContext`
       */
      sPath: string,
      /**
       * A context object for the new binding
       */
      oContext?: object,
      /**
       * Map of optional parameters for the binding
       */
      mParameters?: {
        /**
         * Whether this binding does not propagate model messages to the control; supported since 1.82.0. Some composite
         * types like {@link sap.ui.model.type.Currency} automatically ignore model messages for some of their parts
         * depending on their format options; setting this parameter to `true` or `false` overrules the automatism
         * of the type
         *
         * For example, a binding for a currency code is used in a composite binding for rendering the proper number
         * of decimals, but the currency code is not displayed in the attached control. In that case, messages for
         * the currency code shall not be displayed at that control, only messages for the amount
         */
        ignoreMessages?: boolean;
        /**
         * Whether the value of the created property binding is `undefined` if it is unresolved; if not set, its
         * value is `null`. Supported since 1.100.0
         */
        useUndefinedIfUnresolved?: boolean;
      }
    ): PropertyBinding;
    /**
     * Creates a new tree binding for this model.
     *
     * Hierarchy Annotations: To use the v2.ODataTreeBinding with an OData service which exposes hierarchy annotations,
     * see the **"SAP Annotations for OData Version 2.0"** specification. The required property annotations
     * as well as accepted / default values are documented in this specification.
     *
     * Services which include the `hierarchy-node-descendant-count-for` annotation and expose the data points
     * sorted in a depth-first, pre-order manner, can use an optimized auto-expand feature by specifying the
     * `numberOfExpandedLevels` in the binding parameters. This will pre-expand the hierarchy to the given number
     * of levels, with only a single initial OData request.
     *
     * For services without the `hierarchy-node-descendant-count-for` annotation, the `numberOfExpandedLevels`
     * property is not supported and deprecated.
     *
     * Operation Modes: For a full definition and explanation of all OData binding operation modes, see {@link sap.ui.model.odata.OperationMode}.
     *
     * OperationMode.Server: Filtering on the `ODataTreeBinding` is only supported with filters of type {@link sap.ui.model.FilterType.Application}.
     * Be aware that this applies only to filters which do not prevent the creation of a hierarchy. So filtering
     * on a property (e.g. a "Customer") is fine, as long as the application ensures that the responses from
     * the back end are sufficient to create a valid hierarchy on the client. Subsequent paging requests for
     * sibling and child nodes must also return responses, since the filters are sent with every request. Using
     * control-defined filters (see {@link sap.ui.model.FilterType.Control}) via the {@link #filter} function
     * is not supported for the operation mode `Server`.
     *
     * OperationMode.Client and OperationMode.Auto: The ODataTreeBinding supports control-defined filters only
     * in operation modes `Client` and `Auto`. With these operation modes, the filters and sorters are applied
     * on the client, like for the {@link sap.ui.model.odata.v2.ODataListBinding}.
     *
     * The operation modes `Client` and `Auto` are only supported for services which expose the hierarchy annotations
     * mentioned above, but do **not** expose the `hierarchy-node-descendant-count-for` annotation. Services
     * with hierarchy annotations including the `hierarchy-node-descendant-count-for` annotation, do **not**
     * support the operation modes `Client` and `Auto`. **Note:** {@link sap.ui.model.odata.OperationMode.Auto }
     * is deprecated since 1.102.0.
     *
     * **Note:** OData tree bindings do neither support {@link sap.ui.model.Binding#suspend suspend} nor {@link sap.ui.model.Binding#resume resume}.
     * See:
     * 	{@link http://www.sap.com/protocols/SAPData "SAP Annotations for OData Version 2.0" Specification}
     *
     *
     * @returns The new tree binding
     */
    bindTree(
      /**
       * The binding path, either absolute or relative to a given `oContext`
       */
      sPath: string,
      /**
       * The parent context which is required as base for a relative path
       */
      oContext?: Context,
      /**
       * The filters to be used initially with type {@link sap.ui.model.FilterType.Application}; call {@link sap.ui.model.odata.v2.ODataTreeBinding#filter }
       * to replace them; depending on the operation mode, there are restrictions for using filters; see above
       */
      vFilters?: Filter | Filter[],
      /**
       * Map of binding parameters
       */
      mParameters?: {
        /**
         * Whether the tree binding only requests transition messages from the back end. If messages for entities
         * of this collection need to be updated, use {@link sap.ui.model.odata.v2.ODataModel#read} on the parent
         * entity corresponding to the tree binding's context, with the parameter `updateAggregatedMessages` set
         * to `true`.
         */
        transitionMessagesOnly?: boolean;
        /**
         * The mapping between data properties and the hierarchy used to visualize the tree, if not provided by
         * the service's metadata. For the correct metadata annotations, check the "SAP Annotations for OData Version
         * 2.0" specification
         */
        treeAnnotationProperties?: {
          /**
           * The property name in the same type holding the hierarchy level information; the type of the referenced
           * property has to be an integer type
           */
          hierarchyLevelFor?: string;
          /**
           * The property name in the same type holding the hierarchy node id
           */
          hierarchyNodeFor?: string;
          /**
           * The property name in the same type holding the parent node id
           */
          hierarchyParentNodeFor?: string;
          /**
           * The property name in the same type holding the drill state for the node; the referenced property may
           * have the values "collapsed", "expanded" or "leaf"
           */
          hierarchyDrillStateFor?: string;
          /**
           * The property name in the same type holding the descendant count for the node; the type of the referenced
           * property has to be an integer type
           */
          hierarchyNodeDescendantCountFor?: string;
        };
        /**
         * The number of levels that are auto-expanded initially. Setting this property might lead to multiple back-end
         * requests. The auto-expand feature is **deprecated for services without the `hierarchy-node-descendant-count-for`
         * annotation**
         */
        numberOfExpandedLevels?: number;
        /**
         * The level of the topmost tree nodes
         */
        rootLevel?: number;
        /**
         * The group id to be used for requests originating from this binding
         */
        groupId?: string;
        /**
         * The operation mode for this binding; defaults to the model's default operation mode if not specified.
         * {@link sap.ui.model.odata.OperationMode.Auto OperationMode.Auto} is only supported for services which
         * expose the hierarchy annotations, yet do **NOT** expose the `hierarchy-node-descendant-count-for` annotation.
         * **Note:** {@link sap.ui.model.odata.OperationMode.Auto} is deprecated since 1.102.0.
         */
        operationMode?: OperationMode | keyof typeof OperationMode;
        /**
         * Deprecated since 1.102.0, as {@link sap.ui.model.odata.OperationMode.Auto} is deprecated; the threshold
         * that defines how many entries should be fetched at least by the binding if `operationMode` is set to
         * `Auto`
         */
        threshold?: number;
        /**
         * Deprecated since 1.102.0, as {@link sap.ui.model.odata.OperationMode.Auto} is deprecated; whether `$filter`
         * statements should be used for the `$count` / `$inlinecount` requests and for the data request if the
         * operation mode is {@link sap.ui.model.odata.OperationMode.Auto OperationMode.Auto}. Use this feature
         * only if your back end supports pre-filtering the tree and is capable of responding with a complete tree
         * hierarchy, including all inner nodes. To construct the hierarchy on the client, it is mandatory that
         * all filter matches include their complete parent chain up to the root level. If {@link sap.ui.model.odata.OperationMode.Client OperationMode.Client }
         * is used, the complete collection without filters is requested; filters are applied on the client side.
         */
        useServersideApplicationFilters?: boolean;
        /**
         * A tree state handle can be given to the `ODataTreeBinding` when two conditions are met:
         * 	 - The binding is running in {@link sap.ui.model.odata.OperationMode.Client OperationMode.Client}, and
         *
         * 	 - the {@link sap.ui.table.TreeTable} is used.  The feature is only available when using the {@link sap.ui.table.TreeTable}.
         *     The tree state handle will contain all necessary information to expand the tree to the given state.
         *
         * This feature is not supported if {@link sap.ui.model.odata.OperationMode.Server OperationMode.Server }
         * or {@link sap.ui.model.odata.OperationMode.Auto OperationMode.Auto} is used.
         */
        treeState?: any;
        /**
         * Defines the count mode of this binding; if not specified, the default count mode of the binding's model
         * is applied. The resulting count mode must not be {@link sap.ui.model.odata.CountMode.None}.
         */
        countMode?: CountMode | keyof typeof CountMode;
        /**
         * Whether a preliminary context is used; defaults to the value of the parameter `preliminaryContext` given
         * on construction of the binding's model, see {@link sap.ui.model.odata.v2.ODataModel}
         */
        usePreliminaryContext?: boolean;
        /**
         * **Deprecated**, use `groupId` instead. Sets the batch group id to be used for requests originating from
         * this binding
         */
        batchGroupId?: string;
        /**
         * A map describing the navigation properties between entity sets, which is used for constructing and paging
         * the tree. Keys in this object are entity names, whereas the values name the navigation properties.
         *
         * **Deprecated: since 1.44** The use of navigation properties to build up the hierarchy structure is deprecated.
         * It is recommended to use the hierarchy annotations mentioned above instead.
         */
        navigation?: object;
      },
      /**
       * The sorters used initially; call {@link sap.ui.model.odata.v2.ODataTreeBinding#sort} to replace them
       */
      vSorters?: Sorter[] | Sorter
    ): ODataTreeBinding;
    /**
     * Triggers a request for the given function import.
     *
     * If the return type of the function import is either an entity type or a collection of an entity type,
     * then this OData model's cache is updated with the values of the returned entities. Otherwise they are
     * ignored, and the `response` can be processed in the `success` callback.
     *
     * The `contextCreated` property of the returned object is a function that returns a Promise which resolves
     * with an `sap.ui.model.odata.v2.Context`. This context can be used to modify the function import parameter
     * values and to bind the function call's result. Changes of a parameter value via that context after the
     * function import has been processed lead to another function call with the modified parameters. Changed
     * function import parameters are considered as pending changes, see {@link #hasPendingChanges} or {@link #getPendingChanges},
     * and can be reset via {@link #resetChanges}. If the function import returns an entity or a collection
     * of entities, the `$result` property relative to that context can be used to bind the result to a control,
     * see {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio6cb8d585ed594ee4b447b5b560f292a4 Binding of Function Import Parameters}.
     *
     *
     * @returns An object which has a `contextCreated` function that returns a `Promise`. This resolves with
     * the created {@link sap.ui.model.Context}. In addition it has an `abort` function to abort the current
     * request. The Promise returned by `contextCreated` is rejected if the function name cannot be found in
     * the metadata or if the parameter `expand` is used and the function does not return a single entity.
     */
    callFunction(
      /**
       * The name of the function import starting with a slash, for example `/Activate`.
       */
      sFunctionName: string,
      /**
       * The parameter map containing any of the following properties:
       */
      mParameters?: {
        /**
         * Defines a callback function to adjust the deep path for the resulting entity of the function import call;
         * since 1.82. The deep path of an entity is the resolved path relative to the parent contexts of the binding
         * in the UI hierarchy. For example, for a `ToBusinessPartner` relative context binding with a `/SalesOrder('42')`
         * parent context, the resulting deep path for the `BusinessPartner` is `/SalesOrder('42')/ToBusinessPartner`.
         * This deep path is used to properly assign messages and show them correctly on the UI.
         *
         * The callback function returns a `string` with the deep path for the entity returned by the function import
         * and gets the parameter map `mParameters` containing the following properties:
         * 	 - `{string} mParameters.deepPath`: The deep path of the resulting entity, as far as the framework is
         *     able to determine from the metadata and the OData response
         * 	 - `{object} mParameters.response`: A copy of the OData response object
         */
        adjustDeepPath?: Function;
        /**
         * ID of the `ChangeSet` that this request belongs to
         */
        changeSetId?: string;
        /**
         * A callback function which is called when the request failed. The handler can have the parameter: `oError`
         * which contains additional error information. If the request has been aborted, the error has an `aborted`
         * flag set to `true`.
         */
        error?: Function;
        /**
         * If the function import changes an entity, the ETag for this entity can be passed with this parameter
         */
        eTag?: string;
        /**
         * A comma-separated list of navigation properties to be expanded for the entity returned by the function
         * import; since 1.83.0. The navigation properties are requested with an additional GET request in
         * the same `$batch` request as the POST request for the function import. The given `mParameters.headers`
         * are not considered in the GET request. **Note:** The following prerequisites must be fulfilled:
         *
         * 	 - batch mode must be enabled; see constructor parameter `useBatch`,
         * 	 - the HTTP method used for the function import is "POST",
         * 	 - the function import returns a single entity,
         * 	 - the back-end service must support the "Content-ID" header,
         * 	 - the back end must allow GET requests relative to this content ID outside the changeset within the
         *     `$batch` request.  The success and error callback functions are called only once, even if there
         *     are two requests in the `$batch` related to a single call of {@link #callFunction}.
         * 	 - If both requests succeed, the success callback is called with the merged data of the POST and the
         *     GET request and with the response of the POST request.
         * 	 - If the POST request fails, the GET request also fails. In that case the error callback is called
         *     with the error response of the POST request.
         * 	 - If the POST request succeeds but the GET request for the navigation properties fails, the success
         *     callback is called with the data and the response of the POST request. The response object of the success
         *     callback call and the response parameter of the corresponding `requestFailed` and `requestCompleted`
         *     events have an additional property `expandAfterFunctionCallFailed` set to `true`.
         */
        expand?: string;
        /**
         * ID of a request group; requests belonging to the same group are bundled in one batch request
         */
        groupId?: string;
        /**
         * A map of headers for this request
         */
        headers?: Record<string, string>;
        /**
         * The HTTP method used for the function import call as specified in the metadata definition of the function
         * import
         */
        method?: string;
        /**
         * Defines whether to update all bindings after submitting this change operation; since 1.46. See {@link #setRefreshAfterChange}.
         * If given, this overrules the model-wide `refreshAfterChange` flag for this operation only.
         */
        refreshAfterChange?: boolean;
        /**
         * A callback function which is called when the data has been successfully retrieved; the handler can have
         * the following parameters: `oData` and `response`.
         */
        success?: Function;
        /**
         * Maps the function import parameter name as specified in the function import's metadata to its value;
         * the value is formatted based on the parameter's type as specified in the metadata
         */
        urlParameters?: Record<string, any>;
        /**
         * **Deprecated - use `groupId` instead**
         */
        batchGroupId?: string;
      }
    ): object;
    /**
     * Whether the canonical requests calculation is switched on, see the `canonicalRequests` parameter of the
     * model constructor.
     *
     *
     * @returns Whether the canonical requests calculation is switched on
     */
    canonicalRequestsEnabled(): boolean;
    /**
     * Trigger a `POST` request to the OData service that was specified in the model constructor; see {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio4c4cd99af9b14e08bb72470cc7cabff4 Creating Entities documentation }
     * for comprehensive information on the topic.
     *
     * **Note:** This function does not support a "deep create" scenario. Use {@link #createEntry} or {@link sap.ui.model.odata.v2.ODataListBinding#create }
     * instead.
     *
     *
     * @returns An object which has an `abort` function to abort the current request.
     */
    create(
      /**
       * A string containing the path to the collection where an entry should be created. The path is concatenated
       * to the service URL which was specified in the model constructor.
       */
      sPath: string,
      /**
       * Data of the entry that should be created.
       */
      oData: object,
      /**
       * Optional parameter map containing any of the following properties:
       */
      mParameters?: {
        /**
         * If specified , `sPath` has to be relative to the path given with the context.
         */
        context?: object;
        /**
         * A callback function which is called when the data has been successfully retrieved. The handler can have
         * the following parameters: `oData` and `response`. The `oData` parameter contains the data of the newly
         * created entry if it is provided by the backend. The `response` parameter contains information about the
         * response of the request.
         */
        success?: Function;
        /**
         * A callback function which is called when the request failed. The handler can have the parameter `oError`
         * which contains additional error information. If the `POST` request has been aborted, the error has an
         * `aborted` flag set to `true`.
         */
        error?: Function;
        /**
         * A map containing the parameters that will be passed as query strings
         */
        urlParameters?: Record<string, string>;
        /**
         * A map of headers for this request
         */
        headers?: Record<string, string>;
        /**
         * Deprecated - use `groupId` instead
         */
        batchGroupId?: string;
        /**
         * ID of a request group; requests belonging to the same group will be bundled in one batch request
         */
        groupId?: string;
        /**
         * ID of the `ChangeSet` that this request should belong to
         */
        changeSetId?: string;
        /**
         * Since 1.46; defines whether to update all bindings after submitting this change operation. See {@link #setRefreshAfterChange}.
         * If given, this overrules the model-wide `refreshAfterChange` flag for this operation only.
         */
        refreshAfterChange?: boolean;
      }
    ): object;
    /**
     * Creates a binding context for the given path.
     *
     * If the data of the context is not yet available, it can not be created, but first the entity needs to
     * be fetched from the server asynchronously. In case no callback function is provided, the request will
     * not be triggered.
     *
     * If a callback function is given, the created binding context for a fetched entity is passed as argument
     * to the given callback function.
     * See:
     * 	sap.ui.model.Model.prototype.createBindingContext
     *
     *
     * @returns The created binding context, only if the data is already available and the binding context could
     * be created synchronously; `undefined` otherwise
     */
    createBindingContext(
      /**
       * Binding path
       */
      sPath: string,
      /**
       * Binding context
       */
      oContext?: object,
      /**
       * Map which contains additional parameters for the binding
       */
      mParameters?: {
        /**
         * Value for the OData `$expand` query parameter which should be included in the request
         */
        expand?: string;
        /**
         * Value for the OData `$select` query parameter which should be included in the request
         */
        select?: string;
        /**
         * Whether a preliminary context will be created. When set to `true`, the model can bundle the OData calls
         * for dependent bindings into fewer $batch requests. For more information, see {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio62149734b5c24507868e722fe87a75db Optimizing Dependent Bindings}
         */
        createPreliminaryContext?: boolean;
        /**
         * Optional map of custom query parameters, names of custom parameters must not start with `$`.
         */
        custom?: Record<string, string>;
      },
      /**
       * The function to be called when the context has been created. The parameter of the callback function is
       * the newly created binding context, an instance of {@link sap.ui.model.odata.v2.Context}.
       */
      fnCallBack?: Function,
      /**
       * Whether to reload data
       */
      bReload?: boolean
    ): Context1 | undefined;
    /**
     * Creates a new entry object which is described by the metadata of the entity type of the specified `sPath`
     * Name. A context object is returned which can be used to bind against the newly created object. See {@link https://ui5.sap.com/#/topic/6c47b2b39db9404582994070ec3d57a2#loio4c4cd99af9b14e08bb72470cc7cabff4 Creating Entities documentation }
     * for comprehensive information on the topic.
     *
     * For each created entry a request is created and stored in a request queue. The request queue can be submitted
     * by calling {@link #submitChanges}. As long as the context is transient (see {@link sap.ui.model.odata.v2.Context#isTransient}),
     * {@link sap.ui.model.odata.v2.ODataModel#resetChanges} with the `bDeleteCreatedEntities` parameter set
     * to `true` can be used to delete the created entity again.
     *
     * If the creation of the entity on the server failed, it is repeated automatically.
     *
     * The optional parameter `mParameters.properties` can be used as follows:
     * 	 - `properties` could be an array containing the property names which should be included in the new
     *     entry. Other properties defined in the entity type won't be included.
     * 	 - `properties` could be an object which includes the desired properties and the corresponding values
     *     which should be used for the created entry.
     *
     * The `properties` can be modified via property bindings relative to the returned context instance.
     *
     * The parameter `expand` is supported since 1.78.0. If this parameter is set, the given navigation properties
     * are expanded automatically with the same $batch request in which the POST request for the creation is
     * contained. Ensure that the batch mode is used and the back-end service supports GET requests relative
     * to a content ID outside the changeset. The success and error callback functions are called only once,
     * even if there are two requests in the `$batch` related to a single call of {@link #createEntry}:
     *
     * 	 - a POST request for creating an entity,
     * 	 - a GET request for requesting the navigation properties for the just created entity.   The following
     *     outcomes are possible:
     * 	 - If both requests succeed, the success handler is called with the merged data of the POST and the
     *     GET request and with the response of the POST request.
     * 	 - If the POST request fails, the GET request also fails. In that case the error callback handler is
     *     called with the error response of the POST request.
     * 	 - If the POST request succeeds but the GET request for the navigation properties fails, the success
     *     handler is called with the data and the response of the POST request. The response object of the success
     *     handler call and the response parameter of the corresponding `requestFailed` and `requestCompleted` events
     *     have an additional property `expandAfterCreateFailed` set to `true`.
     *
     * Note: If a server requires a property in the request, you must supply this property in the initial data,
     * for example if the server requires a unit for an amount. This also applies if this property has a default
     * value.
     *
     * Note: Deep create is only supported since 1.108.0, where "deep create" means creation of a sub-entity
     * for a navigation property of a transient, not yet persisted root entity. Before 1.108.0, the sub-entity
     * had to be created after the transient entity had been saved successfully in the back-end system. Since
     * 1.108.0, a deep create is triggered when the `sPath` parameter is a navigation property for the entity
     * type associated with the transient context given in `mParameters.context`. The payload of the OData request
     * to create the root entity then contains its sub-entities. On creation of a sub-entity, only the `sPath`,
     * `mParameters.context` and `mParameters.properties` method parameters are allowed; the context given in
     * `mParameters.context` must not be inactive.
     *
     *
     * @returns An OData V2 context object that points to the newly created entry; or `undefined` if the service
     * metadata are not yet loaded or if a `created` callback parameter is given
     */
    createEntry(
      /**
       * The path to the EntitySet
       */
      sPath: string,
      /**
       * A map of the following parameters:
       */
      mParameters: {
        /**
         * Deprecated - use `groupId` instead
         */
        batchGroupId?: string;
        /**
         * The ID of the `ChangeSet` that this request should belong to
         */
        changeSetId?: string;
        /**
         * The binding context
         */
        context?: Context;
        /**
         * The callback function that is called after the metadata of the service is loaded and the {@link sap.ui.model.odata.v2.Context }
         * instance for the newly created entry is available; The {@link sap.ui.model.odata.v2.Context} instance
         * for the newly created entry is passed as the first and only parameter.
         */
        created?: Function;
        /**
         * The error callback function
         */
        error?: Function;
        /**
         * A comma-separated list of navigation properties to be expanded for the newly created entity; since 1.78.0. The navigation properties are requested with an additional GET request in the same `$batch` request
         * as the POST request for the entity creation; the given `mParameters.headers` are not considered in the
         * GET request. **Note:** The following prerequisites must be fulfilled:
         * 	 - batch mode must be enabled; see constructor parameter `useBatch`,
         * 	 - the back-end service must support the "Content-ID" header,
         * 	 - the back end must allow GET requests relative to this content ID outside the changeset within the
         *     `$batch` request.
         */
        expand?: string;
        /**
         * The ID of a request group; requests belonging to the same group will be bundled in one batch request
         */
        groupId?: string;
        /**
         * A map of headers
         */
        headers?: Record<string, string>;
        /**
         * Whether the created context is inactive. An inactive context will only be sent to the server after the
         * first property update. From then on it behaves like any other created context. Supported since 1.98.0
         */
        inactive?: boolean;
        /**
         * The initial values of the entry, or an array that specifies a list of property names to be initialized
         * with `undefined`; **Note:** Passing a list of property names is deprecated since 1.120; pass the initial
         * values as an object instead
         */
        properties?: object | string[];
        /**
         * Whether to update all bindings after submitting this change operation, see {@link #setRefreshAfterChange};
         * if given, this overrules the model-wide `refreshAfterChange` flag for this operation only; since 1.46
         */
        refreshAfterChange?: boolean;
        /**
         * The success callback function
         */
        success?: Function;
        /**
         * A map of URL parameters
         */
        urlParameters?: Record<string, string>;
      }
    ): Context1 | undefined;
    /**
     * Creates the key from the given collection name and property map.
     *
     * Please make sure that the metadata document is loaded before using this function.
     *
     *
     * @returns Key of the entry
     */
    createKey(
      /**
       * Name of the collection
       */
      sCollection: string,
      /**
       * Object containing at least all the key properties of the entity type
       */
      oKeyProperties: object
    ): string;
    /**
     * Deletes a created entry from the request queue and from the model.
     *
     * **Note:** Controls are not updated. Use {@link #resetChanges} instead to update also the controls, for
     * example: `oModel.resetChanges([oContext.getPath()], undefined, true);`
     *
     * @deprecated As of version 1.95.0. use {@link #resetChanges} instead
     */
    deleteCreatedEntry(
      /**
       * The context object pointing to the created entry
       */
      oContext: Context
    ): void;
    /**
     * See:
     * 	sap.ui.model.Model.prototype.destroy
     */
    destroy(): void;
    /**
     * Detaches event handler `fnFunction` from the `annotationsFailed` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAnnotationsFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$AnnotationsFailedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the `annotationsLoaded` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAnnotationsLoaded(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$AnnotationsLoadedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:batchRequestCompleted batchRequestCompleted }
     * event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBatchRequestCompleted(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestCompletedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:batchRequestFailed batchRequestFailed} event
     * of this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBatchRequestFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestFailedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:batchRequestSent batchRequestSent} event of
     * this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBatchRequestSent(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$BatchRequestSentEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the `metadataFailed` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachMetadataFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$MetadataFailedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the `metadataLoaded` event of this `sap.ui.model.odata.v2.ODataModel`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachMetadataLoaded(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (evt: ODataModel$MetadataLoadedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:annotationsFailed annotationsFailed} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireAnnotationsFailed(
      /**
       * Parameters to pass along with the event
       */
      oParameters?: ODataModel$AnnotationsFailedEventParameters
    ): this;
    /**
     * Fires event {@link #event:annotationsLoaded annotationsLoaded} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireAnnotationsLoaded(
      /**
       * Parameters to pass along with the event
       */
      oParameters?: ODataModel$AnnotationsLoadedEventParameters
    ): this;
    /**
     * Fires event {@link #event:batchRequestCompleted batchRequestCompleted} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireBatchRequestCompleted(
      /**
       * parameters to add to the fired event
       */
      oParameters: ODataModel$BatchRequestCompletedEventParameters
    ): this;
    /**
     * Fires event {@link #event:batchRequestFailed batchRequestFailed} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireBatchRequestFailed(
      /**
       * Parameters to pass along with the event
       */
      oParameters: ODataModel$BatchRequestFailedEventParameters
    ): this;
    /**
     * Fires event {@link #event:batchRequestSent batchRequestSent} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireBatchRequestSent(
      /**
       * Parameters to pass along with the event
       */
      oParameters?: ODataModel$BatchRequestSentEventParameters
    ): this;
    /**
     * Fires event {@link #event:metadataFailed metadataFailed} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireMetadataFailed(
      /**
       * Parameters to pass along with the event
       */
      oParameters?: ODataModel$MetadataFailedEventParameters
    ): this;
    /**
     * Fires event {@link #event:metadataLoaded metadataLoaded} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` to allow method chaining
     */
    fireMetadataLoaded(
      /**
       * Parameters to pass along with the event
       */
      oParameters?: ODataModel$MetadataLoadedEventParameters
    ): this;
    /**
     * Force the update on the server of an entity by setting its ETag to '*'.
     *
     * ETag handling must be active so the force update will work.
     */
    forceEntityUpdate(
      /**
       * The key to an Entity e.g.: Customer(4711)
       */
      sKey: string
    ): void;
    /**
     * Returns the definition of batch groups per entity type for two-way binding changes.
     *
     * @deprecated As of version 1.36. use {@link #getChangeGroups} instead
     *
     * @returns Definition of batch groups for two-way binding changes, keyed by entity names.
     */
    getChangeBatchGroups(): Record<string, ChangeGroupDefinition>;
    /**
     * Returns the definition of groups per entity type for two-way binding changes
     *
     *
     * @returns Definition of groups for two-way binding changes, keyed by entity names.
     */
    getChangeGroups(): Record<string, ChangeGroupDefinition>;
    /**
     * Return requested data as object if the data has already been loaded and stored in the model.
     *
     * @deprecated As of version 1.24. please use {@link #getProperty} instead
     *
     * @returns Object containing the requested data if the path is valid.
     */
    getData(
      /**
       * A string containing the path to the data object that should be returned.
       */
      sPath: string,
      /**
       * The optional context which is used with the `sPath` to retrieve the requested data.
       */
      oContext?: object,
      /**
       * This parameter should be set when a URI or custom parameter with a `$expand` system query option was
       * used to retrieve associated entries embedded. If set to `true` then the `getProperty` function returns
       * a desired property value or entry and includes the associated expand entries (if any). If set to `false`
       * the associated/expanded entry properties are removed and not included in the desired entry as properties
       * at all. This is useful for performing updates on the base entry only. Note: A copy, not a reference of
       * the entry will be returned.
       */
      bIncludeExpandEntries?: boolean
    ): object;
    /**
     * Returns the default count mode for retrieving the count of collections
     *
     * @since 1.20
     *
     * @returns Returns the default count mode for this model
     */
    getDefaultCountMode(): CountMode;
    /**
     * Returns the array of batch group IDs that are set as deferred
     *
     * @deprecated As of version 1.32. use {@link #getDeferredGroups} instead
     *
     * @returns aGroupIds The array of deferred batch group IDs
     */
    getDeferredBatchGroups(): any[];
    /**
     * Returns the array of group IDs that are set as deferred.
     *
     *
     * @returns aGroupIds The array of deferred group IDs
     */
    getDeferredGroups(): any[];
    /**
     * Returns the ETag for a given binding path/context or data object.
     *
     *
     * @returns The found ETag (or `null` if none could be found)
     */
    getETag(
      /**
       * The binding path
       */
      sPath?: string,
      /**
       * The binding context
       */
      oContext?: Context,
      /**
       * The entity data
       */
      oEntity?: object
    ): string | null;
    /**
     * Returns all headers and custom headers which are stored in this OData model.
     *
     *
     * @returns The header map
     */
    getHeaders(): object;
    /**
     * Returns the key part for the given the canonical entry URI, model context or data object or `undefined`
     * when the `vValue` can't be interpreted.
     *
     *
     * @returns Key of the entry or `undefined`
     */
    getKey(
      /**
       * The canonical entry URI, the context or entry object
       */
      vValue: string | object | Context
    ): string | undefined;
    /**
     * Returns this model's message scope.
     * See:
     * 	sap.ui.model.odata.MessageScope
     *
     * @since 1.76.0
     *
     * @returns The message scope
     */
    getMessageScope(): MessageScope;
    /**
     * Returns an instance of an OData meta model which offers a unified access to both OData V2 metadata and
     * V4 annotations. It uses the existing {@link sap.ui.model.odata.ODataMetadata} as a foundation and merges
     * V4 annotations from the existing {@link sap.ui.model.odata.v2.ODataAnnotations} directly into the corresponding
     * model element.
     *
     * **BEWARE:** Access to this OData meta model will fail before the promise returned by {@link sap.ui.model.odata.ODataMetaModel#loaded loaded }
     * has been resolved!
     *
     *
     * @returns The meta model for this `ODataModel`
     */
    getMetaModel(): ODataMetaModel;
    /**
     * Returns a JSON object that is a copy of the entity data referenced by the given `sPath` and `oContext`.
     * It does not load any data and may not return all requested data if it is not available.
     *
     * With the `mParameters.select` parameter it is possible to specify comma-separated property or navigation
     * property names which should be included in the result object. This works like the OData `$select` URL
     * parameter. With the `mParameters.expand` parameter it is possible to specify comma-separated navigation
     * property names which should be included inline in the result object. This works like the OData `$expand`
     * parameter.
     *
     * **Note:** `mParameters.expand` can only be used if the corresponding navigation properties have been
     * read via {@link sap.ui.model.odata.v2.ODataModel#read} using the OData `$expand` URL parameter. If a
     * navigation property has not been read via the OData `$expand` URL parameter, it is left out in the result.
     * Keep in mind that navigation properties referencing a collection are usually not loaded via the OData
     * `$expand` URL parameter but directly via its navigation property.
     *
     * **Note:** If `mParameters.select` is not specified, the returned object may contain model-internal attributes.
     * This may lead to problems when submitting this data to the service for an update or create operation.
     * To get a copy of the entity without internal attributes, use `{select: "*"}` instead.
     *
     * **Note:** If `mParameters.select` is given and not all selected properties are available, this method
     * returns `undefined` instead of incomplete data.
     *
     * **Note:** If `mParameters.select` is not given, all properties and navigation properties available on
     * the client are returned.
     *
     * Example:
     *  With `mParameters` given as `{select: "Products/ProductName, Products", expand:"Products"}` no properties
     * of the entity itself are returned, but only the `ProductName` property of the `Products` navigation property.
     * If `Products/ProductName` has not been loaded before, `undefined` is returned.
     *
     *
     * @returns The value for the given path and context or `undefined` if data or entity type cannot be found
     * or if not all selected properties are available
     */
    getObject(
      /**
       * The path referencing the object
       */
      sPath: string,
      /**
       * The optional context which is used with the `sPath` to reference the object.
       */
      oContext?: Context,
      /**
       * Map of parameters
       */
      mParameters?: {
        /**
         * Comma-separated list of properties or paths to properties to select
         */
        select?: string;
        /**
         * Comma-separated list of navigation properties or paths to navigation properties to expand
         */
        expand?: string;
      }
    ): any | undefined;
    /**
     * Returns a JSON object that is a copy of the entity data referenced by the given `sPath` and `oContext`.
     * It does not load any data and may not return all requested data if it is not available.
     *
     * With the `mParameters.select` parameter it is possible to specify comma-separated property or navigation
     * property names which should be included in the result object. This works like the OData `$select` URL
     * parameter. With the `mParameters.expand` parameter it is possible to specify comma-separated navigation
     * property names which should be included inline in the result object. This works like the OData `$expand`
     * parameter.
     *
     * **Note:** `mParameters.expand` can only be used if the corresponding navigation properties have been
     * read via {@link sap.ui.model.odata.v2.ODataModel#read} using the OData `$expand` URL parameter. If a
     * navigation property has not been read via the OData `$expand` URL parameter, it is left out in the result.
     * Keep in mind that navigation properties referencing a collection are usually not loaded via the OData
     * `$expand` URL parameter but directly via its navigation property.
     *
     * **Note:** If `mParameters.select` is not specified, the returned object may contain model-internal attributes.
     * This may lead to problems when submitting this data to the service for an update or create operation.
     * To get a copy of the entity without internal attributes, use `{select: "*"}` instead.
     *
     * **Note:** If `mParameters.select` is given and not all selected properties are available, this method
     * returns `undefined` instead of incomplete data.
     *
     * **Note:** If `mParameters.select` is not given, all properties and navigation properties available on
     * the client are returned.
     *
     * Example:
     *  With `mParameters` given as `{select: "Products/ProductName, Products", expand:"Products"}` no properties
     * of the entity itself are returned, but only the `ProductName` property of the `Products` navigation property.
     * If `Products/ProductName` has not been loaded before, `undefined` is returned.
     *
     *
     * @returns The value for the given path and context or `undefined` if data or entity type cannot be found
     * or if not all selected properties are available
     */
    getObject(
      /**
       * The path referencing the object
       */
      sPath: string,
      /**
       * Map of parameters
       */
      mParameters?: {
        /**
         * Comma-separated list of properties or paths to properties to select
         */
        select?: string;
        /**
         * Comma-separated list of navigation properties or paths to navigation properties to expand
         */
        expand?: string;
      }
    ): any | undefined;
    /**
     * Returns the original value for the property with the given path and context. The original value is the
     * value that was last responded by the server.
     *
     *
     * @returns the value of the property
     */
    getOriginalProperty(
      /**
       * The path/name of the property
       */
      sPath: string,
      /**
       * The context if available to access the property value
       */
      oContext?: object
    ): any;
    /**
     * Returns the pending changes in this model.
     *
     * Only changes triggered through {@link #createEntry} or {@link #setProperty}, and tree hierarchy changes
     * are taken into account. Changes are returned as a map from the changed entity's key to an object containing
     * the changed properties. A node removed from a tree hierarchy has the empty object as value in this map;
     * all other pending entity deletions are not contained in the map.
     * See:
     * 	#hasPendingChanges
     * 	#resetChanges
     *
     *
     * @returns The map of pending changes
     */
    getPendingChanges(): Record<string, object>;
    /**
     * Returns the value for the property with the given `sPath`. Since 1.100, a path starting with "@$ui5."
     * which represents an instance annotation is supported. The following instance annotations are allowed;
     * they return information on the given oContext, which must be set and be an {@link sap.ui.model.odata.v2.Context}:
     *
     * 	 - `@$ui5.context.isInactive`: The return value of {@link sap.ui.model.odata.v2.Context#isInactive }
     *
     * 	 - `@$ui5.context.isTransient`: The return value of {@link sap.ui.model.odata.v2.Context#isTransient }
     *
     *
     *
     * @returns Value of the property
     */
    getProperty(
      /**
       * Path/name of the property
       */
      sPath: string,
      /**
       * Context if available to access the property value
       */
      oContext?: object,
      /**
       * Deprecated, use {@link #getObject} function with 'select' and 'expand' parameters instead. Whether entities
       * for navigation properties of this property which have been read via `$expand` are part of the return
       * value.
       */
      bIncludeExpandEntries?: boolean
    ): any;
    /**
     * Whether all affected bindings are refreshed after a change operation.
     *
     * This flag can be overruled on request level by providing the `refreshAfterChange` parameter to the corresponding
     * function (for example {@link #update}).
     *
     * @since 1.46.0
     *
     * @returns Whether to automatically refresh after changes
     */
    getRefreshAfterChange(): boolean;
    /**
     * Returns the current security token if available; triggers a request to fetch the security token if it
     * is not available.
     *
     * @deprecated As of version 1.119.0. use {@link #securityTokenAvailable} instead
     *
     * @returns The security token; `undefined` if it is not available
     */
    getSecurityToken(): string | undefined;
    /**
     * Return the annotation object. Please note that the metadata is loaded asynchronously and this function
     * might return undefined because the metadata has not been loaded yet. In this case attach to the `annotationsLoaded`
     * event to get notified when the annotations are available and then call this function.
     *
     *
     * @returns Metadata object
     */
    getServiceAnnotations(): object;
    /**
     * Return the parsed XML metadata as a Javascript object.
     *
     * Please note that the metadata is loaded asynchronously and this function might return `undefined` because
     * the metadata has not been loaded yet. In this case attach to the `metadataLoaded` event to get notified
     * when the metadata is available and then call this function.
     *
     *
     * @returns Metadata object
     */
    getServiceMetadata(): Object | undefined;
    /**
     * Returns this model's base URI of the data service (as defined by the "serviceUrl" model parameter; see
     * {@link #constructor}), without query options.
     *
     * @since 1.130.0
     *
     * @returns The service's base URI without query options
     */
    getServiceUrl(): string;
    /**
     * Checks if there exist pending changes in the model.
     *
     * By default, only client data changes triggered through {@link #createEntry} or {@link #setProperty},
     * and tree hierarchy changes are taken into account.
     *
     * If `bAll` is set to `true`, also deferred requests triggered through {@link #create}, {@link #update},
     * and {@link #remove} are taken into account.
     * See:
     * 	#getPendingChanges
     * 	#resetChanges
     *
     *
     * @returns `true` if there are pending changes, `false` otherwise.
     */
    hasPendingChanges(
      /**
       * If set to true, deferred requests are also taken into account.
       */
      bAll?: boolean
    ): boolean;
    /**
     * Checks if there are pending requests, either ongoing or sequential.
     *
     *
     * @returns Whether there are pending requests
     */
    hasPendingRequests(): boolean;
    /**
     * Invalidate the model data.
     *
     * Mark all entries in the model cache as invalid. Next time a context or list is bound (binding), the respective
     * entries will be detected as invalid and will be refreshed from the server.
     *
     * To refresh all model data use {@link sap.ui.model.odata.v2.ODataModel#refresh}
     *
     * @since 1.52.1
     */
    invalidate(
      /**
       * A function which can be used to restrict invalidation to specific entries, gets the entity key and object
       * as parameters and should return true for entities to invalidate.
       */
      fnCheckEntry?: Function
    ): void;
    /**
     * Invalidate all entries of the given entity type in the model data.
     *
     * Mark entries of the provided entity type in the model cache as invalid. Next time a context binding or
     * list binding is done, the entry will be detected as invalid and will be refreshed from the server.
     *
     * @since 1.52.1
     */
    invalidateEntityType(
      /**
       * The qualified name of the entity type. A qualified name consists of two parts separated by a dot. The
       * first part is the namespace of the schema in which the entity type is defined, such as "NorthwindModel".
       * The second part is the entity type name such as "Customer". This results in a qualified name such as
       * "NorthwindModel.Customer". The qualified name can be found in the data sent from the server in JSON format
       * under `__metadata.type` or in XML format in the `term` attribute of the entity's `category` tag.
       */
      sEntityType: string
    ): void;
    /**
     * Invalidate a single entry in the model data.
     *
     * Mark the selected entry in the model cache as invalid. Next time a context binding or list binding is
     * done, the entry will be detected as invalid and will be refreshed from the server.
     *
     * @since 1.52.1
     */
    invalidateEntry(
      /**
       * the reference to the entry, either by key, absolute path or context object
       */
      vEntry: string | Context
    ): void;
    /**
     * Checks whether metadata loading has failed in the past.
     *
     * @since 1.38
     *
     * @returns Whether metadata request has failed
     */
    isMetadataLoadingFailed(): boolean;
    /**
     * Checks whether the service has set the OData V2 annotation "message-scope-supported" on the `EntityContainer`
     * with the value `true`. This is a a precondition for the setting of {@link sap.ui.model.odata.MessageScope.BusinessObject }
     * via {@link #setMessageScope}.
     * See:
     * 	sap.ui.model.odata.MessageScope
     *
     * @since 1.76.0
     *
     * @returns A promise resolving with `true` if the OData V2 annotation "message-scope-supported" on the
     * `EntityContainer` is set to `true`
     */
    messageScopeSupported(): Promise<any>;
    /**
     * Returns a promise for the loaded state of the metadata.
     *
     * The metadata needs to be loaded prior to performing OData calls. Chaining to the returned promise ensures
     * that all required parameters have been loaded, e.g. the security token, see {@link #getSecurityToken}.
     *
     * The returned promise depends on the optional parameter `bRejectOnFailure`.
     *
     * `bRejectOnFailure=false`: The promise won't get rejected in case the metadata or annotation loading failed
     * but is only resolved if
     * 	 - the metadata are loaded successfully,
     * 	 - the annotations are processed, provided the model parameter `loadAnnotationsJoined` has been set.
     *      Use this promise for delaying OData calls until all required information is available, i.e. this
     *     promise is resolved.
     *
     * `bRejectOnFailure=true`: Since 1.79, the parameter `bRejectOnFailure` allows to request a promise that
     * is rejected when one of the following fails:
     * 	 - the loading of the metadata,
     * 	 - the loading of the annotations, provided the model parameter `loadAnnotationsJoined` has been set.
     *      The promise is fulfilled upon successful loading of both. This promise can be used to start processing
     *     OData calls when it is fulfilled and to display an error message when it is rejected. See also the example
     *     below.
     *
     * If the method `refreshMetadata` is called after the returned promise is already resolved or rejected,
     * you should use the promise returned by `refreshMetadata` to get information about the refreshed state.
     *
     * @since 1.30
     *
     * @returns A promise on metadata loaded state
     */
    metadataLoaded(
      /**
       * Determines since 1.79 whether the returned promise is rejected when the initial loading of the metadata
       * fails. In case the model parameter `loadAnnotationsJoined` is set, the returned promise fails also if
       * loading the annotations fails.
       */
      bRejectOnFailure?: boolean
    ): Promise<any>;
    /**
     * Trigger a `GET` request to the OData service that was specified in the model constructor.
     *
     * The data will be stored in the model. The requested data is returned with the response.
     *
     *
     * @returns An object which has an `abort` function to abort the current request.
     */
    read(
      /**
       * An absolute path or a path relative to the context given in `mParameters.context`; if the path contains
       * a query string, the query string is ignored, use `mParameters.urlParameters` instead
       */
      sPath: string,
      /**
       * Optional parameter map containing any of the following properties:
       */
      mParameters?: {
        /**
         * If specified, `sPath` has to be relative to the path given with the context.
         */
        context?: object;
        /**
         * A map containing the parameters that will be passed as query strings
         */
        urlParameters?: Record<string, string>;
        /**
         * An array of filters to be included in the request URL
         */
        filters?: Filter[];
        /**
         * An array of sorters to be included in the request URL
         */
        sorters?: Sorter[];
        /**
         * A callback function which is called when the data has been successfully retrieved. The handler can have
         * the following parameters: `oData` and `response`. The `oData` parameter contains the data of the retrieved
         * data. The `response` parameter contains further information about the response of the request.
         */
        success?: Function;
        /**
         * A callback function which is called when the request failed. The handler can have the parameter: `oError`
         * which contains additional error information. If the `GET` request has been aborted, the error has an
         * `aborted` flag set to `true`.
         */
        error?: Function;
        /**
         * Deprecated - use `groupId` instead
         */
        batchGroupId?: string;
        /**
         * ID of a request group; requests belonging to the same group will be bundled in one batch request
         */
        groupId?: string;
        /**
         * Whether messages for child entities belonging to the same business object as the requested or changed
         * resources are updated. It is considered only if {@link sap.ui.model.odata.MessageScope.BusinessObject }
         * is set using {@link #setMessageScope} and if the OData service supports message scope.
         */
        updateAggregatedMessages?: boolean;
      }
    ): object;
    /**
     * Refresh the model.
     *
     * This will reload all data stored in the model. This will check all bindings for updated data and update
     * the controls if data has been changed.
     *
     * Note: In contrast to an individual Binding refresh, the model refresh ignores Binding-specific parameters/queries.
     */
    refresh(
      /**
       * Force update of controls
       */
      bForceUpdate?: boolean,
      /**
       * If set to `true` then the model data will be removed/cleared. Please note that the data might not be
       * there when calling e.g. `getProperty` too early before the refresh call returned.
       */
      bRemoveData?: boolean,
      /**
       * ID of a request group; requests belonging to the same group will be bundled in one batch request
       */
      sGroupId?: string
    ): void;
    /**
     * Refreshes the metadata for this model, for example when the request for metadata has failed. Returns
     * a new promise which can be resolved or rejected depending on the metadata loading state.
     *
     * @deprecated As of version 1.42. this API may cause data inconsistencies and should not be used.
     *
     * @returns A promise on metadata loaded state or `undefined` if metadata is not initialized or currently
     * refreshed
     */
    refreshMetadata(): Promise<any> | undefined;
    /**
     * Refresh XSRF token by performing a GET request against the service root URL.
     *
     *
     * @returns An object which has an `abort` function to abort the current request.
     */
    refreshSecurityToken(
      /**
       * Callback function which is called when the data has been successfully retrieved.
       */
      fnSuccess?: Function,
      /**
       * Callback function which is called when the request failed. The handler can have the parameter: oError
       * which contains additional error information.
       */
      fnError?: Function,
      /**
       * Whether the request should be sent asynchronously
       */
      bAsync?: boolean
    ): object;
    /**
     * Trigger a `DELETE` request to the OData service that was specified in the model constructor.
     *
     *
     * @returns An object which has an `abort` function to abort the current request.
     */
    remove(
      /**
       * A string containing the path to the data that should be removed. The path is concatenated to the service
       * URL which was specified in the model constructor.
       */
      sPath: string,
      /**
       * Optional, can contain the following attributes:
       */
      mParameters?: {
        /**
         * If specified, `sPath` has to be relative to the path given with the context.
         */
        context?: object;
        /**
         * A callback function which is called when the data has been successfully retrieved. The handler can have
         * the following parameters: `oData` and `response`.
         */
        success?: Function;
        /**
         * A callback function which is called when the request failed. The handler can have the parameter: `oError`
         * which contains additional error information. If the `DELETE` request has been aborted, the error has
         * an `aborted` flag set to `true`.
         */
        error?: Function;
        /**
         * If specified, the `If-Match` header will be set to this ETag.
         */
        eTag?: string;
        /**
         * A map containing the parameters that will be passed as query strings
         */
        urlParameters?: Record<string, string>;
        /**
         * A map of headers for this request
         */
        headers?: Record<string, string>;
        /**
         * Deprecated - use `groupId` instead
         */
        batchGroupId?: string;
        /**
         * ID of a request group; requests belonging to the same group will be bundled in one batch request
         */
        groupId?: string;
        /**
         * ID of the `ChangeSet` that this request should belong to
         */
        changeSetId?: string;
        /**
         * Since 1.46; defines whether to update all bindings after submitting this change operation, see {@link #setRefreshAfterChange}.
         * If given, this overrules the model-wide `refreshAfterChange` flag for this operation only.
         */
        refreshAfterChange?: boolean;
      }
    ): object;
    /**
     * Resets pending changes and aborts corresponding requests.
     *
     * By default, only changes triggered through {@link #createEntry} or {@link #setProperty}, and tree hierarchy
     * changes are taken into account. If `bAll` is set, also deferred requests triggered through {@link #create},
     * {@link #update} or {@link #remove} are taken into account.
     *
     * With a given `aPath` only specified entities are reset. Note that tree hierarchy changes are only affected
     * if a given path is equal to the tree binding's resolved binding path.
     *
     * If `bDeleteCreatedEntities` is set, the entity is completely removed, provided it has been created by
     * one of the following methods:
     * 	 - {@link #createEntry}, provided it is not yet persisted in the back end and is active (see {@link sap.ui.model.odata.v2.Context#isInactive}),
     *
     * 	 - {@link #callFunction}.
     * See:
     * 	#getPendingChanges
     * 	#hasPendingChanges
     *
     *
     * @returns Resolves when all regarded changes have been reset.
     */
    resetChanges(
      /**
       * Paths to be reset; if no array is passed, all changes are reset
       */
      aPath?: string[],
      /**
       * Whether also deferred requests are taken into account so that they are aborted
       */
      bAll?: boolean,
      /**
       * Whether to delete the entities created via {@link #createEntry} or {@link #callFunction}; since 1.95.0
       */
      bDeleteCreatedEntities?: boolean
    ): Promise<any>;
    /**
     * Returns a promise, which will resolve with the security token as soon as it is available.
     *
     *
     * @returns A promise on the security token
     */
    securityTokenAvailable(): Promise<any>;
    /**
     * Definition of batch groups per entity type for two-way binding changes.
     *
     * @deprecated As of version 1.32. Use {@link #setChangeGroups} instead
     */
    setChangeBatchGroups(
      /**
       * A map containing the definition of batch groups for two-way binding changes, see {@link #setChangeGroups}
       */
      mGroups: Record<string, ChangeGroupDefinition>
    ): void;
    /**
     * Definition of groups per entity type for two-way binding changes. **Note:** This will overwrite the existing
     * change group definition, including the default `{"*":{groupId: "changes"}}`.
     */
    setChangeGroups(
      /**
       * Maps an entity name to the definition of the batch group for two-way binding changes; use "*" as entity
       * name to define a default for all entities not contained in the map
       */
      mGroups: Record<string, ChangeGroupDefinition>
    ): void;
    /**
     * Sets the default mode how to retrieve the item count for a collection in this model.
     *
     * The count can be determined in the following ways
     * 	 - by sending a separate `$count` request
     * 	 - by adding parameter `$inlinecount=allpages` to one or all data requests
     * 	 - a combination of the previous two
     * 	 - not at all (questions about the size of the collection can't be answered then)  See {@link sap.ui.model.odata.CountMode }
     *     for all enumeration values and more details.
     *
     * Note that a call to this method does not modify the count mode for existing list bindings, only bindings
     * that are created afterwards will use the new mode when no mode is defined at their creation.
     *
     * If no default count mode is set for an `ODataModel` (v2), the mode `Request` will be used.
     *
     * @since 1.20
     */
    setDefaultCountMode(
      /**
       * The new default count mode for this model
       */
      sCountMode: CountMode | keyof typeof CountMode
    ): void;
    /**
     * Setting batch groups as deferred.
     *
     * Requests that belong to a deferred batch group have to be sent by explicitly calling {@link #submitChanges}.
     *
     * @deprecated As of version 1.32. use {@link #setDeferredGroups} instead
     */
    setDeferredBatchGroups(
      /**
       * Array of batch group IDs that should be set as deferred
       */
      aGroupIds: any[]
    ): void;
    /**
     * Setting request groups as deferred. **Note:** This will overwrite existing deferred groups, including
     * the default deferred group "changes".
     *
     * Requests that belong to a deferred group will be sent by explicitly calling {@link #submitChanges}.
     */
    setDeferredGroups(
      /**
       * The array of deferred group IDs; the default is: `["changes"]`
       */
      aGroupIds: any[]
    ): void;
    /**
     * Set custom headers which are provided in a key/value map.
     *
     * These headers are used for requests against the OData backend. Private headers which are set in the ODataModel
     * cannot be modified. These private headers are: `accept, accept-language, x-csrf-token, MaxDataServiceVersion,
     * DataServiceVersion`.
     *
     * To remove these custom headers simply set the `mHeaders` parameter to null. Please also note that when
     * calling this method again, all previous custom headers are removed unless they are specified again in
     * the `mHeaders` parameter.
     */
    setHeaders(
      /**
       * The header name/value map.
       */
      mHeaders: object
    ): void;
    /**
     * Sets this model's message scope.
     * See:
     * 	sap.ui.model.odata.MessageScope
     *
     * @since 1.76.0
     */
    setMessageScope(
      /**
       * The message scope
       */
      sMessageScope: MessageScope | keyof typeof MessageScope
    ): void;
    /**
     * Sets a new value for the given property `sPath` in the model.
     *
     * If the `changeBatchGroup` for the changed entity type is set to {@link #setDeferredGroups deferred},
     * changes could be submitted with {@link #submitChanges}. Otherwise the change will be submitted directly.
     *
     *
     * @returns `true` if the value was set correctly and `false` if errors occurred like the entry was not
     * found or another entry was already updated.
     */
    setProperty(
      /**
       * Path of the property to set
       */
      sPath: string,
      /**
       * Value to set the property to
       */
      oValue: any,
      /**
       * The context which will be used to set the property
       */
      oContext?: Context,
      /**
       * Whether to update other bindings dependent on this property asynchronously
       */
      bAsyncUpdate?: boolean
    ): boolean;
    /**
     * Defines whether all affected bindings are refreshed after a change operation.
     *
     * This flag can be overruled on request level by providing the `refreshAfterChange` parameter to the corresponding
     * function (for example {@link #update}).
     *
     * @since 1.16.3
     */
    setRefreshAfterChange(
      /**
       * Whether to automatically refresh after changes
       */
      bRefreshAfterChange: boolean
    ): void;
    /**
     * Sets a "Retry-After" handler, which is called when an OData request fails with HTTP status 503 (Service
     * Unavailable) and the response has a "Retry-After" header.
     *
     * The handler is called with an `Error` having a property `retryAfter` of type `Date`, which is the earliest
     * point in time when the request should be repeated. The handler has to return a promise. With this promise,
     * you can control the repetition of all pending requests including the failed HTTP request. If the promise
     * is resolved, the requests are repeated; if it is rejected, the requests are not repeated. If it is rejected
     * with the same `Error` reason as previously passed to the handler, then this reason is reported to the
     * message model.
     *
     * **Note:** For APIs, like e.g. {@link #submitChanges}, which return an object having an `abort` function
     * to abort the request triggered by the API, this abort function must not be called as long as the above
     * promise is pending. Otherwise an error will be thrown.
     *
     * @since 1.134.0
     */
    setRetryAfterHandler(
      /**
       * A "Retry-After" handler
       */
      fnRetryAfter: (p1: RetryAfterError) => Promise<undefined>
    ): void;
    /**
     * Enable/Disable security token handling.
     */
    setTokenHandlingEnabled(
      /**
       * Whether to use token handling or not
       */
      bTokenHandling?: boolean
    ): void;
    /**
     * Enable or disable batch mode for future requests.
     */
    setUseBatch(
      /**
       * Whether the requests should be encapsulated in a batch request
       */
      bUseBatch?: boolean
    ): void;
    /**
     * Submits the collected changes which were collected by the {@link #setProperty} method and other deferred
     * requests.
     *
     * The update method is defined by the global `defaultUpdateMethod` parameter which is `sap.ui.model.odata.UpdateMethod.MERGE`
     * by default. In case of a `sap.ui.model.odata.UpdateMethod.MERGE` request only the changed properties
     * will be updated. If a URI with a `$expand` query option was used then the expand entries will be removed
     * from the collected changes. Changes to this entries should be done on the entry itself. So no deep updates
     * are supported.
     *
     * **Important**: The success/error handler will only be called if batch support is enabled. If multiple
     * batch groups are submitted the handlers will be called for every batch group. If there are no changes/requests
     * or all contained requests are aborted before a batch request returns, the success handler will be called
     * with an empty response object. If the abort method on the return object is called, all contained batch
     * requests will be aborted and the error handler will be called for each of them.
     *
     *
     * @returns An object which has an `abort` function to abort the current request or requests
     */
    submitChanges(
      /**
       * A map which contains the following parameter properties:
       */
      mParameters?: {
        /**
         * Defines the group that should be submitted. If not specified, all deferred groups will be submitted.
         * Requests belonging to the same group will be bundled in one batch request.
         */
        groupId?: string;
        /**
         * A callback function which is called when the data has been successfully updated. The handler can have
         * the following parameters: `oData`. `oData` contains the parsed response data as a Javascript object.
         * The batch response is in the `__batchResponses` property which may contain further `__changeResponses`
         * in an array depending on the amount of changes and change sets of the actual batch request which was
         * sent to the backend. The changeResponses contain the actual response of that change set in the `response`
         * property. For each change set there is also a `__changeResponse` property.
         */
        success?: Function;
        /**
         * A callback function which is called when the request failed. The handler can have the parameter: `oError`
         * which contains additional error information. If all contained requests have been aborted, the error has
         * an `aborted` flag set to `true`.
         */
        error?: Function;
        /**
         * **Deprecated**, use `groupId` instead
         */
        batchGroupId?: string;
        /**
         * **Deprecated** since 1.38.0; use the `defaultUpdateMethod` constructor parameter instead. If unset, the
         * update method is determined from the `defaultUpdateMethod` constructor parameter. If `true`, `sap.ui.model.odata.UpdateMethod.MERGE`
         * is used for update operations; if set to `false`, `sap.ui.model.odata.UpdateMethod.PUT` is used.
         */
        merge?: boolean;
      }
    ): object;
    /**
     * Trigger a `PUT/MERGE` request to the OData service that was specified in the model constructor.
     *
     * The update method used is defined by the global `defaultUpdateMethod` parameter which is `sap.ui.model.odata.UpdateMethod.MERGE`
     * by default. Please note that deep updates are not supported and may not work. These should be done separately
     * and directly on the corresponding entry.
     *
     *
     * @returns An object which has an `abort` function to abort the current request.
     */
    update(
      /**
       * A string containing the path to the data that should be updated. The path is concatenated to the sServiceUrl
       * which was specified in the model constructor.
       */
      sPath: string,
      /**
       * Data of the entry that should be updated.
       */
      oData: object,
      /**
       * Optional, can contain the following attributes:
       */
      mParameters?: {
        /**
         * If specified the sPath has to be is relative to the path given with the context.
         */
        context?: object;
        /**
         * A callback function which is called when the data has been successfully updated.
         */
        success?: Function;
        /**
         * A callback function which is called when the request failed. The handler can have the parameter `oError`
         * which contains additional error information. If the `PUT/MERGE` request has been aborted, the error has
         * an `aborted` flag set to `true`.
         */
        error?: Function;
        /**
         * If specified, the `If-Match` header will be set to this ETag. Caution: This feature in not officially
         * supported as using asynchronous requests can lead to data inconsistencies. If you decide to use this
         * feature nevertheless, you have to make sure that the request is completed before the data is processed
         * any further.
         */
        eTag?: string;
        /**
         * A map containing the parameters that will be passed as query strings
         */
        urlParameters?: Record<string, string>;
        /**
         * A map of headers for this request
         */
        headers?: Record<string, string>;
        /**
         * Deprecated - use `groupId` instead
         */
        batchGroupId?: string;
        /**
         * ID of a request group; requests belonging to the same group will be bundled in one batch request
         */
        groupId?: string;
        /**
         * ID of the `ChangeSet` that this request should belong to
         */
        changeSetId?: string;
        /**
         * Since 1.46; defines whether to update all bindings after submitting this change operation. See {@link #setRefreshAfterChange}.
         * If given, this overrules the model-wide `refreshAfterChange` flag for this operation only.
         */
        refreshAfterChange?: boolean;
      }
    ): object;
    /**
     * Update all bindings.
     */
    updateBindings(
      /**
       * If set to `false`, an update will only be done when the value of a binding changed.
       */
      bForceUpdate?: boolean
    ): void;
  }
  /**
   * Definition of a change group.
   */
  export type ChangeGroupDefinition = {
    /**
     * The ID of the batch group
     */
    groupId: string;
    /**
     * The ID of a change set which bundles the changes for the entity type
     */
    changeSetId?: string;
    /**
     * Defines whether every change is put in a change set of its own
     */
    single?: boolean;
  };

  /**
   * Parameters of the ODataModel#annotationsFailed event.
   */
  export interface ODataModel$AnnotationsFailedEventParameters {
    /**
     * An array of Errors
     */
    result?: Error[];
  }

  /**
   * Event object of the ODataModel#annotationsFailed event.
   */
  export type ODataModel$AnnotationsFailedEvent = Event<
    ODataModel$AnnotationsFailedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#annotationsLoaded event.
   */
  export interface ODataModel$AnnotationsLoadedEventParameters {
    /**
     * An array consisting of one or several annotation sources and/or errors containing a source property and
     * error details.
     */
    result?: Source[];
  }

  /**
   * Event object of the ODataModel#annotationsLoaded event.
   */
  export type ODataModel$AnnotationsLoadedEvent = Event<
    ODataModel$AnnotationsLoadedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#batchRequestCompleted event.
   */
  export interface ODataModel$BatchRequestCompletedEventParameters {
    /**
     * The request ID
     */
    ID?: string;

    /**
     * The URL which is sent to the backend
     */
    url?: string;

    /**
     * The HTTP method
     */
    method?: string;

    /**
     * The request headers
     */
    headers?: Record<string, string>;

    /**
     * Request was successful or not
     */
    success?: boolean;

    /**
     * If the request is synchronous or asynchronous (if available)
     */
    async?: boolean;

    /**
     * Array of embedded requests ($batch) Each request object within the array contains the following properties:
     * url, method, headers, response object
     */
    requests?: any[];

    /**
     * The response object - empty object if no response: The response object contains the following properties:
     * message, success, headers, statusCode, statusText, responseText
     */
    response?: object;
  }

  /**
   * Event object of the ODataModel#batchRequestCompleted event.
   */
  export type ODataModel$BatchRequestCompletedEvent = Event<
    ODataModel$BatchRequestCompletedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#batchRequestFailed event.
   */
  export interface ODataModel$BatchRequestFailedEventParameters {
    /**
     * The request ID
     */
    ID?: string;

    /**
     * The URL which is sent to the backend
     */
    url?: string;

    /**
     * The HTTP method
     */
    method?: string;

    /**
     * The request headers
     */
    headers?: Record<string, string>;

    /**
     * If the request is synchronous or asynchronous (if available)
     */
    async?: boolean;

    /**
     * Request was successful or not
     */
    success?: boolean;

    /**
     * The response object - empty object if no response The response object contains the following properties:
     * message, success, headers, statusCode, statusText, responseText
     */
    response?: object;

    /**
     * Array of embedded requests ($batch) Each request object within the array contains the following properties:
     * url, method, headers, response object
     */
    requests?: any[];
  }

  /**
   * Event object of the ODataModel#batchRequestFailed event.
   */
  export type ODataModel$BatchRequestFailedEvent = Event<
    ODataModel$BatchRequestFailedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#batchRequestSent event.
   */
  export interface ODataModel$BatchRequestSentEventParameters {
    /**
     * The URL which is sent to the backend
     */
    url?: string;

    /**
     * The type of the request (if available)
     */
    type?: string;

    /**
     * If the request is synchronous or asynchronous (if available)
     */
    async?: boolean;

    /**
     * Array of embedded requests ($batch) Each request object within the array contains the following properties:
     * url, method, headers
     */
    requests?: any[];
  }

  /**
   * Event object of the ODataModel#batchRequestSent event.
   */
  export type ODataModel$BatchRequestSentEvent = Event<
    ODataModel$BatchRequestSentEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#metadataFailed event.
   */
  export interface ODataModel$MetadataFailedEventParameters {
    /**
     * The parsed metadata
     */
    metadata?: string;

    /**
     * A text that describes the failure.
     */
    message?: string;

    /**
     * HTTP status code returned by the request (if available)
     */
    statusCode?: string;

    /**
     * The status as a text, details not specified, intended only for diagnosis output
     */
    statusText?: string;

    /**
     * Response that has been received for the request, as a text string
     */
    responseText?: string;

    /**
     * The response object - empty object if no response
     */
    response?: object;
  }

  /**
   * Event object of the ODataModel#metadataFailed event.
   */
  export type ODataModel$MetadataFailedEvent = Event<
    ODataModel$MetadataFailedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#metadataLoaded event.
   */
  export interface ODataModel$MetadataLoadedEventParameters {
    /**
     * The parsed metadata
     */
    metadata?: string;
  }

  /**
   * Event object of the ODataModel#metadataLoaded event.
   */
  export type ODataModel$MetadataLoadedEvent = Event<
    ODataModel$MetadataLoadedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#requestCompleted event.
   */
  export interface ODataModel$RequestCompletedEventParameters
    extends Model$RequestCompletedEventParameters {
    /**
     * The request ID
     */
    ID?: string;

    /**
     * The HTTP method
     */
    method?: string;

    /**
     * The request headers
     */
    headers?: Record<string, string>;

    /**
     * The response object - empty object if no response: The response object contains the following properties:
     * message, success, headers, statusCode, statusText, responseText
     */
    response?: object;
  }

  /**
   * Event object of the ODataModel#requestCompleted event.
   */
  export type ODataModel$RequestCompletedEvent = Event<
    ODataModel$RequestCompletedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#requestFailed event.
   */
  export interface ODataModel$RequestFailedEventParameters
    extends Model$RequestFailedEventParameters {
    /**
     * The request ID
     */
    ID?: string;

    /**
     * The URL which is sent to the backend
     */
    url?: string;

    /**
     * The HTTP method
     */
    method?: string;

    /**
     * The request headers
     */
    headers?: Record<string, string>;

    /**
     * If the request is synchronous or asynchronous (if available)
     */
    async?: boolean;

    /**
     * Request was successful or not
     */
    success?: boolean;

    /**
     * The response object - empty object if no response The response object contains the following properties:
     * message, success, headers, statusCode, statusText, responseText
     */
    response?: object;
  }

  /**
   * Event object of the ODataModel#requestFailed event.
   */
  export type ODataModel$RequestFailedEvent = Event<
    ODataModel$RequestFailedEventParameters,
    ODataModel
  >;

  /**
   * Parameters of the ODataModel#requestSent event.
   */
  export interface ODataModel$RequestSentEventParameters
    extends Model$RequestSentEventParameters {
    /**
     * The request ID
     */
    ID?: string;

    /**
     * The HTTP method
     */
    method?: string;

    /**
     * The request headers
     */
    headers?: Record<string, string>;
  }

  /**
   * Event object of the ODataModel#requestSent event.
   */
  export type ODataModel$RequestSentEvent = Event<
    ODataModel$RequestSentEventParameters,
    ODataModel
  >;
}

declare module "sap/ui/performance/Measurement" {
  /**
   * Performance Measurement API.
   *
   * @since 1.58
   */
  interface Measurement {
    /**
     * Adds a performance measurement with all data.
     *
     * This is useful to add external measurements (e.g. from a backend) to the common measurement UI
     *
     *
     * @returns current measurement containing id, info and start-timestamp, end-timestamp, time, duration,
     * categories (false if error)
     */
    add(
      /**
       * ID of the measurement
       */
      sId: string,
      /**
       * Info for the measurement
       */
      sInfo: string,
      /**
       * start timestamp
       */
      iStart: int,
      /**
       * end timestamp
       */
      iEnd: int,
      /**
       * time in milliseconds
       */
      iTime: int,
      /**
       * effective time in milliseconds
       */
      iDuration: int,
      /**
       * An optional list of categories for the measure
       */
      aCategories?: string | string[]
    ): Entry | boolean | undefined;
    /**
     * Starts an average performance measure.
     *
     * The duration of this measure is an avarage of durations measured for each call. Optionally a category
     * or list of categories can be passed to allow filtering of measurements.
     *
     *
     * @returns current measurement containing id, info and start-timestamp (false if error)
     */
    average(
      /**
       * ID of the measurement
       */
      sId: string,
      /**
       * Info for the measurement
       */
      sInfo: string,
      /**
       * An optional list of categories for the measure
       */
      aCategories?: string | string[]
    ): Entry | boolean | undefined;
    /**
     * Clears all performance measurements.
     */
    clear(): void;
    /**
     * Ends a performance measure.
     *
     *
     * @returns current measurement containing id, info and start-timestamp, end-timestamp, time, duration (false
     * if error)
     */
    end(
      /**
       * ID of the measurement
       */
      sId: string
    ): Entry | boolean | undefined;
    /**
     * Gets all performance measurements where a provided filter function returns a truthy value.
     *
     * If neither a filter function nor a category is provided an empty array is returned. To filter for certain
     * properties of measurements a fnFilter can be implemented like this ` function(oMeasurement) { return
     * oMeasurement.duration > 50; }`
     *
     *
     * @returns filtered array with measurements containing id, info and start-timestamp, end-timestamp, time,
     * duration, categories (false if error)
     */
    filterMeasurements(
      /**
       * a filter function that returns true if the passed measurement should be added to the result
       */
      fnFilter?: (p1: Entry) => void,
      /**
       * Optional parameter to determine if either completed or incomplete measurements should be returned (both
       * if not set or undefined)
       */
      bCompleted?: boolean,
      /**
       * The function returns only measurements which match these specified categories
       */
      aCategories?: string[]
    ): Entry[];
    /**
     * Gets the current state of the performance measurement functionality.
     *
     *
     * @returns current state of the performance measurement functionality
     */
    getActive(): boolean;
    /**
     * Gets all performance measurements.
     *
     *
     * @returns current array with measurements containing id, info and start-timestamp, end-timestamp, time,
     * duration, categories
     */
    getAllMeasurements(
      /**
       * Whether only completed measurements should be returned, if explicitly set to false only incomplete measurements
       * are returned
       */
      bCompleted?: boolean
    ): Entry[];
    /**
     * Gets a performance measure.
     *
     *
     * @returns current measurement containing id, info and start-timestamp, end-timestamp, time, duration (false
     * if error)
     */
    getMeasurement(
      /**
       * ID of the measurement
       */
      sId: string
    ): Entry | boolean;
    /**
     * Pauses a performance measure.
     *
     *
     * @returns current measurement containing id, info and start-timestamp, pause-timestamp (false if error)
     */
    pause(
      /**
       * ID of the measurement
       */
      sId: string
    ): Entry | boolean | undefined;
    /**
     * Registers an average measurement for a given objects method.
     *
     *
     * @returns true if the registration was successful
     */
    registerMethod(
      /**
       * the id of the measurement
       */
      sId: string,
      /**
       * the object of the method
       */
      oObject: object,
      /**
       * the name of the method
       */
      sMethod: string,
      /**
       * An optional categories list for the measurement
       */
      aCategories?: string[]
    ): boolean;
    /**
     * Removes a performance measure.
     */
    remove(
      /**
       * ID of the measurement
       */
      sId: string
    ): void;
    /**
     * Resumes a performance measure.
     *
     *
     * @returns current measurement containing id, info and start-timestamp, resume-timestamp (false if error)
     */
    resume(
      /**
       * ID of the measurement
       */
      sId: string
    ): Entry | boolean | undefined;
    /**
     * Activates or deactivates the performance measure functionality.
     *
     * Optionally a category or list of categories can be passed to restrict measurements to certain categories
     * like "javascript", "require", "xmlhttprequest", "render"
     *
     *
     * @returns current state of the performance measurement functionality
     */
    setActive(
      /**
       * state of the performance measurement functionality to set
       */
      bOn: boolean,
      /**
       * An optional list of categories that should be measured
       */
      aCategories: string | string[]
    ): boolean;
    /**
     * Starts a performance measure.
     *
     * Optionally a category or list of categories can be passed to allow filtering of measurements.
     *
     *
     * @returns current measurement containing id, info and start-timestamp (false if error)
     */
    start(
      /**
       * ID of the measurement
       */
      sId: string,
      /**
       * Info for the measurement
       */
      sInfo: string,
      /**
       * An optional list of categories for the measure
       */
      aCategories?: string | string[]
    ): Entry | boolean | undefined;
    /**
     * Unregisters all average measurements.
     */
    unregisterAllMethods(): void;
    /**
     * Unregisters an average measurement for a given objects method.
     *
     *
     * @returns true if the unregistration was successful
     */
    unregisterMethod(
      /**
       * the id of the measurement
       */
      sId: string,
      /**
       * the object of the method
       */
      oObject: object,
      /**
       * the name of the method
       */
      sMethod: string
    ): boolean;
  }
  const Measurement: Measurement;
  export default Measurement;

  /**
   * Single Measurement Entry.
   */
  export type Entry = {
    /**
     * ID of the measurement
     */
    sId: string;
    /**
     * Info for the measurement
     */
    sInfo: string;
    /**
     * Start time
     */
    iStart: int;
    /**
     * End time
     */
    iEnd: int;
    /**
     * An optional list of categories for the measure
     */
    aCategories?: string | string[];
  };
}

declare module "sap/ui/performance/trace/FESRHelper" {
  import UI5Element from "sap/ui/core/Element";

  /**
   * FESRHelper API Provides helper functionality for FESR and consumers of FESR
   *
   * @since 1.100
   */
  interface FESRHelper {
    /**
     * Get semantic stepname for an event of a given element used for FESR.
     *
     * @since 1.100
     *
     * @returns The semantic stepname for the given event ID
     */
    getSemanticStepname(
      /**
       * The element conatining the semantic stepname
       */
      oElement: UI5Element,
      /**
       * The event ID of the semantic stepname
       */
      sEventId: string
    ): string;
    /**
     * Add semantic stepname for an event of a given element used for FESR.
     *
     * @since 1.100
     */
    setSemanticStepname(
      /**
       * The element the semantic stepname should be applied to
       */
      oElement: UI5Element,
      /**
       * The event ID the semantic stepname is valid for
       */
      sEventId: string,
      /**
       * The semantic stepname
       */
      sStepname: string
    ): void;
  }
  const FESRHelper: FESRHelper;
  export default FESRHelper;
}

declare module "sap/ui/performance/trace/Interaction" {
  import { Entry as Entry1 } from "sap/ui/performance/Measurement";

  /**
   * Provides base functionality for interaction detection heuristics & API. Interaction detection works through
   * the detection of relevant events and tracking of rendering activities.
   *  An example:
   *  The user clicks on a button
   *
   * 	 - "click" event gets detected via notification (`var notifyEventStart`)
   * 	 - a click handler is registered on the button, so this is an interaction start (`var notifyStepStart`)
   *
   * 	 - some requests are made and rendering has finished (`var notifyStepEnd`)  All measurement takes
   *     place in {@link module:sap/ui/performance/Measurement}.
   *
   * @since 1.76
   */
  interface Interaction {
    /**
     * Gets all interaction measurements for which a provided filter function returns a truthy value.
     *
     * To filter for certain categories of measurements a fnFilter can be implemented like this ` function(InteractionMeasurement)
     * { return InteractionMeasurement.duration > 0 }`
     *
     * @since 1.76
     *
     * @returns all interaction measurements passing the filter function successfully
     */
    filter(
      /**
       * a filter function that returns true if the passed measurement should be added to the result
       */
      fnFilter: Function
    ): Entry[];
    /**
     * Returns true if the interaction detection was enabled explicitly, or implicitly along with fesr.
     *
     * @since 1.76
     *
     * @returns bActive State of the interaction detection
     */
    getActive(): boolean;
    /**
     * Gets all interaction measurements.
     *
     * @since 1.76
     *
     * @returns all interaction measurements
     */
    getAll(
      /**
       * finalize the current pending interaction so that it is contained in the returned array
       */
      bFinalize: boolean
    ): Entry[];
    /**
     * Enables the interaction tracking.
     *
     * @since 1.76
     *
     * @returns Resolves when FESR is active
     */
    setActive(
      /**
       * State of the interaction detection
       */
      bActive: boolean
    ): Promise<any>;
  }
  const Interaction: Interaction;
  export default Interaction;

  /**
   * Interaction Entry
   */
  export type Entry = {
    /**
     * The event which triggered the interaction. The default value is "startup".
     */
    event: string;
    /**
     * The control which triggered the interaction.
     */
    trigger: string;
    /**
     * The identifier of the component or app that is associated with the interaction.
     */
    component: string;
    /**
     * The application version as from app descriptor
     */
    appVersion: string;
    /**
     * The start timestamp of the interaction which is initially set to the `fetchStart`
     */
    start: float;
    /**
     * The end timestamp of the interaction
     */
    end: float;
    /**
     * The sum over all navigation times
     */
    navigation: float;
    /**
     * The time from first request sent to last received response end - without gaps and ignored overlap
     */
    roundtrip: float;
    /**
     * The client processing time
     */
    processing: float;
    /**
     * The interaction duration
     */
    duration: float;
    /**
     * The Performance API requests during interaction
     */
    requests: PerformanceResourceTiming[];
    /**
     * The Performance measurements
     */
    measurements: Entry1[];
    /**
     * The SAP Statistics for OData
     */
    sapStatistics: SAPStatistics[];
    /**
     * The sum over all requests in the interaction
     */
    requestTime: float;
    /**
     * The request time minus server time from the header
     */
    networkTime: float;
    /**
     * The sum over all requests bytes
     */
    bytesSent: int;
    /**
     * The sum over all responses bytes
     */
    bytesReceived: int;
    /**
     * It's set with value "X" by default When compression does not match SAP rules, we report an empty string.
     */
    requestCompression: "X" | "";
    /**
     * The sum of the global busy indicator duration during the interaction
     */
    busyDuration: float;
    /**
     * The ID of the interaction
     */
    id: string;
    /**
     * The default PassportAction for startup
     */
    passportAction: string;
    /**
     * The root context ID
     */
    rootId: string;
  };

  /**
   * The SAP Statistics for OData
   */
  export type SAPStatistics = {
    /**
     * The url of the response
     */
    url: string;
    /**
     * The response header under the key "sap-statistics"
     */
    statistics: string;
    /**
     * The last performance resource timing
     */
    timing: PerformanceResourceTiming;
  };
}

declare module "sap/ui/security/Security" {
  import { URI } from "sap/ui/core/library";

  /**
   * Provides security related API
   *
   * @since 1.120.0
   */
  interface Security {
    /**
     * URL of the allowlist service.
     *
     * @since 1.120.0
     *
     * @returns allowlist service URL
     */
    getAllowlistService(): string;
    /**
     * frameOptions mode (allow/deny/trusted).
     *
     * @since 1.120.0
     *
     * @returns frameOptions mode
     */
    getFrameOptions(): string;
    /**
     * Returns the security token handlers of an OData V4 model.
     * See:
     * 	#setSecurityTokenHandlers
     *
     * @since 1.120.0
     *
     * @returns the security token handlers (an empty array if there are none)
     */
    getSecurityTokenHandlers(): Array<(p1: URI) => Promise<any>>;
    /**
     * Sets the security token handlers for an OData V4 model. See chapter {@link https://ui5.sap.com/#/topic/9613f1f2d88747cab21896f7216afdac/section_STH Security Token Handling}.
     * See:
     * 	#getSecurityTokenHandlers
     *
     * @since 1.120.0
     */
    setSecurityTokenHandlers(
      /**
       * The security token handlers
       */
      aSecurityTokenHandlers: Array<(p1: URI) => Promise<any>>
    ): void;
  }
  const Security: Security;
  export default Security;
}

declare module "sap/ui/test/OpaBuilder" {
  import {
    default as Opa5,
    SingleControlSelector,
    MultiControlSelector,
    Matcher,
    Action,
  } from "sap/ui/test/Opa5";

  import UI5Element from "sap/ui/core/Element";

  /**
   * Builder pattern for {@link sap.ui.test.Opa5#waitFor} options object - a function driven API supporting
   * easy test definition and execution.
   *
   * Sample usage:
   * ```javascript
   *
   * // {
   * //    id: "myButton",
   * //    press: new Press()
   * // }
   * OpaBuilder.create()
   *     .hasId("myButton")
   *     .doPress()
   *     .build();
   * ```
   *
   *
   * Replace `this.waitFor` call completely:
   * ```javascript
   *
   * // return this.waitFor({
   * //    controlType: "sap.m.StandardListItem",
   * //    matchers: [
   * //       {
   * //           properties: { text: "my test text" }
   * //       }
   * //    ],
   * //    press: new Press(),
   * //    success: function () {
   * //        Opa5.assert.ok(true, "Item selected - OK");
   * //    },
   * //    errorMessage: "Item selected - FAILURE"
   * // });
   * return OpaBuilder.create(this)
   *     .hasType("sap.m.StandardListItem")
   *     .hasProperties({ text: "my test text" })
   *     .doPress()
   *     .description("Item selected")
   *     .execute();
   * ```
   */
  export default class OpaBuilder {
    /**
     * Constructor for a new OpaBuilder.
     */
    constructor(
      /**
       * the Opa5 instance to operate on
       */
      oOpaInstance?: Opa5,
      /**
       * the initial {@link sap.ui.test.Opa5#waitFor} options
       */
      oOptions?: SingleControlSelector | MultiControlSelector
    );

    /**
     * Convenience creation and initialization of a new OpaBuilder.
     *
     *
     * @returns a new OpaBuilder instance
     */
    static create(
      /**
       * the Opa5 instance to operate on
       */
      oOpaInstance?: Opa5,
      /**
       * the id of the target control(s)
       */
      vId?: string | RegExp,
      /**
       * the type of the target control(s)
       */
      sControlType?: string,
      /**
       * if true, only popover and dialogs are searched for
       */
      bDialogElement?: boolean,
      /**
       * additional matchers to filter target control(s)
       */
      vMatchers?: Matcher | Matcher[],
      /**
       * the actions to be performed on target control(s)
       */
      vActions?: Action | Action[],
      /**
       * oOptions the {@link sap.ui.test.Opa5#waitFor} options to apply
       */
      oOptions?: SingleControlSelector | MultiControlSelector
    ): OpaBuilder;
    /**
     * Set or get the default options to be used as the builder base. If no options are provided, the current
     * default options are returned.
     *
     *
     * @returns the default {@link sap.ui.test.Opa5#waitFor} options
     */
    static defaultOptions(
      /**
       * the new default options to be used
       */
      oOptions?: SingleControlSelector | MultiControlSelector
    ): SingleControlSelector | MultiControlSelector;
    /**
     * Build the final {@link sap.ui.test.Opa5#waitFor} options object and returns it.
     *
     *
     * @returns the final options object
     */
    build(): SingleControlSelector | MultiControlSelector;
    /**
     * Add a check function. If another check function already exists, the functions are chained.
     *
     *
     * @returns this OpaBuilder instance
     */
    check(
      /**
       * the check that is executed on matched controls
       */
      fnCheck: (p1: UI5Element | UI5Element[]) => boolean,
      /**
       * true to replace all previous defined matchers, false to add it (default)
       */
      bReplace?: boolean
    ): this;
    /**
     * Adds a check for the expected number of matching controls.
     *
     *
     * @returns this OpaBuilder instance
     */
    checkNumberOfMatches(
      /**
       * the number of expected matching controls
       */
      iExpectedNumber: number
    ): this;
    /**
     * Sets the `debugTimeout` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    debugTimeout(
      /**
       * the debug timeout in seconds
       */
      iDebugTimeout: int
    ): this;
    /**
     * Set a output text that will be used as success and error message base message.
     *
     *
     * @returns this OpaBuilder instance
     */
    description(
      /**
       * a descriptive text
       */
      sDescription: string
    ): this;
    /**
     * Add an action to be performed on all matched controls.
     *
     *
     * @returns this OpaBuilder instance
     */
    do(
      /**
       * the action(s) to be performed on matched controls
       */
      vActions: Action | Action[],
      /**
       * true to replace all previous defined actions, false to add it (default)
       */
      bReplace?: boolean
    ): this;
    /**
     * Add an action that is only performed if target control fulfills the conditions. It is internally using
     * {@link module:sap/ui/test/OpaBuilder#Actions.conditional}.
     *
     *
     * @returns this OpaBuilder instance
     */
    doConditional(
      /**
       * target control is checked against these given conditions
       */
      vConditions: Matcher | Matcher[] | boolean,
      /**
       * the actions to be performed when conditions are fulfilled
       */
      vSuccessActions: Action | Action[],
      /**
       * the action(s) to be performed when conditions are not fulfilled
       */
      vElseActions?: Action | Action[]
    ): this;
    /**
     * Performs a {@link sap.ui.test.actions.EnterText} on target control(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    doEnterText(
      /**
       * defines the {@link sap.ui.test.actions.EnterText#setText} setting
       */
      sText: string,
      /**
       * defines the {@link sap.ui.test.actions.EnterText#setClearTextFirst} setting
       */
      bClearTextFirst?: boolean,
      /**
       * defines the {@link sap.ui.test.actions.EnterText#setKeepFocus} setting
       */
      bKeepFocus?: boolean,
      /**
       * defines the {@link sap.ui.test.actions.EnterText#setPressEnterKey} setting
       */
      bPressEnterKey?: boolean,
      /**
       * defines the {@link sap.ui.test.actions.Action#setIdSuffix} setting
       */
      sIdSuffix?: string
    ): this;
    /**
     * Performs given actions on all items of an aggregation fulfilling the matchers.
     *
     *
     * @returns this OpaBuilder instance
     */
    doOnAggregation(
      /**
       * the aggregation name
       */
      sAggregationName: string,
      /**
       * the matchers to filter aggregation items
       */
      vMatchers: Matcher | Matcher[],
      /**
       * the actions to be performed on matching aggregation items
       */
      vActions: Action | Action[]
    ): this;
    /**
     * Performs given actions on all items of an aggregation fulfilling the matchers.
     *
     *
     * @returns this OpaBuilder instance
     */
    doOnAggregation(
      /**
       * the aggregation name
       */
      sAggregationName: string,
      /**
       * the actions to be performed on matching aggregation items
       */
      vActions: Action | Action[]
    ): this;
    /**
     * Executes a builder with matching controls being descendants of matching target control(s). Children are
     * any controls in the control tree beneath this target control(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    doOnChildren(
      /**
       * the child builder or child matcher
       */
      vChildBuilderOrMatcher?: Matcher | Matcher[] | OpaBuilder,
      /**
       * the actions to be performed on matching child items
       */
      vActions?: Action | Action[],
      /**
       * specifies if the ancestor should be a direct ancestor (parent)
       */
      bDirect?: boolean
    ): this;
    /**
     * Executes a {@link sap.ui.test.actions.Press} action on target control(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    doPress(
      /**
       * the id suffix of the DOM Element the press action will be executed on
       */
      sIdSuffix?: string
    ): this;
    /**
     * Adds an error message or function.
     *
     *
     * @returns this OpaBuilder instance
     */
    error(
      /**
       * the message to be shown (or function executed) on failure
       */
      vErrorMessage: string | Function,
      /**
       * true to replace all previous defined error functions, false to add it (default)
       */
      bReplace?: boolean
    ): this;
    /**
     * Executes the definition on the given or previously defined Opa5 instance.
     *
     *
     * @returns an object extending a jQuery promise, corresponding to the result of {@link sap.ui.test.Opa5#waitFor}
     */
    execute(
      /**
       * the Opa5 instance to call {@link sap.ui.test.Opa5#waitFor} on
       */
      oOpaInstance?: Opa5
    ): Opa5;
    /**
     * Sets the `fragmentId` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    fragmentId(
      /**
       * the fragment id
       */
      sFragmentId: string
    ): this;
    /**
     * Get the Opa5 instance that will be used for {@link module:sap/ui/test/OpaBuilder#execute}. If no {sap.ui.test.Opa5 }
     * instance was set before, this function creates a new one lazily.
     *
     *
     * @returns the Opa5 instance
     */
    getOpaInstance(): Opa5;
    /**
     * Defines additional matchers for the target control(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    has(
      /**
       * additional matchers to filter target control(s)
       */
      vMatchers: Matcher | Matcher[],
      /**
       * true to replace all previous defined matchers, false to add it (default)
       */
      bReplace?: boolean
    ): this;
    /**
     * Adds matchers to aggregation items, that at least one aggregation item must match.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasAggregation(
      /**
       * the aggregation name
       */
      sAggregationName: string,
      /**
       * matchers to filter aggregation items
       */
      vMatchers?: Matcher | Matcher[]
    ): this;
    /**
     * Adds a matcher that checks for a certain number of aggregation items.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasAggregationLength(
      /**
       * the aggregation name
       */
      sAggregationName: string,
      /**
       * length to check against
       */
      iNumber: int
    ): this;
    /**
     * Adds a matcher to aggregation items checking for certain properties. At least one item must match the
     * properties.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasAggregationProperties(
      /**
       * the aggregation name
       */
      sAggregationName: string,
      /**
       * map of properties that aggregation item must match
       */
      oProperties: Record<string, any>
    ): this;
    /**
     * Adds a matcher that checks whether at least one child fulfilling given matcher(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    hasChildren(
      /**
       * the matchers to filter child items
       */
      vBuilderOrMatcher?: Matcher | Matcher[] | OpaBuilder,
      /**
       * specifies if the ancestor should be a direct ancestor (parent)
       */
      bDirect?: boolean
    ): this;
    /**
     * Adds a matcher that checks states for given conditions. It is internally using {@link module:sap/ui/test/OpaBuilder#Matchers.conditional}.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasConditional(
      /**
       * conditions to pre-check
       */
      vConditions: Matcher | Matcher[] | boolean,
      /**
       * actual matcher that is executed if conditions are met
       */
      vSuccessMatcher: Matcher | Matcher[] | Object,
      /**
       * actual matcher that is executed if conditions are not met
       */
      vElseMatcher?: Matcher | Matcher[] | Object
    ): this;
    /**
     * Adds a matcher for given properties.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasI18NText(
      /**
       * the name of the property to check for i18n text
       */
      sPropertyName: string,
      /**
       * the path to the I18N text. If model is omitted, `i18n` is used as model name.
       */
      sModelTokenPath: string,
      /**
       * the values to be used instead of the placeholders
       */
      aParameters?: any[]
    ): this;
    /**
     * Defines the id of the target control(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    hasId(
      /**
       * the id of the target control(s)
       */
      vId: string | RegExp
    ): this;
    /**
     * Adds a matcher for given properties.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasProperties(
      /**
       * map of properties that target control(s) must match
       */
      oProperties: Record<string, any>
    ): this;
    /**
     * Adds a group of matchers that requires only one of them to actually match. It is internally using {@link module:sap/ui/test/OpaBuilder#Matchers.some}.
     *
     *
     * @returns this OpaBuilder instance
     */
    hasSome(
      /**
       * aMatchers list of matchers where one must be met
       */
      aMatchers?: Matcher | Matcher[]
    ): this;
    /**
     * Defines the control type of the target control(s).
     *
     *
     * @returns this OpaBuilder instance
     */
    hasType(
      /**
       * the type of the target control(s)
       */
      sControlType: string
    ): this;
    /**
     * Defines whether target control is part of a popover or dialog (sets `searchOpenDialogs` property).
     *
     *
     * @returns this OpaBuilder instance
     */
    isDialogElement(
      /**
       * can be set to false to disable `searchOpenDialogs`, set to true if omitted
       */
      bDialog?: boolean
    ): this;
    /**
     * Sets the `enabled` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    mustBeEnabled(
      /**
       * can be set to false to prevent `enabled` check, set to true if omitted
       */
      bEnabled?: boolean
    ): this;
    /**
     * Sets the `autoWait` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    mustBeReady(
      /**
       * can be set to false to prevent `autoWait`, set to true if omitted
       */
      bReady?: boolean
    ): this;
    /**
     * Sets the `visible` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    mustBeVisible(
      /**
       * can be set to false to prevent `visible` check, set to true if omitted
       */
      bVisible?: boolean
    ): this;
    /**
     * Apply custom options. The options might override previously defined options of the OpaBuilder.
     *
     *
     * @returns this OpaBuilder instance
     */
    options(
      /**
       * the {@link sap.ui.test.Opa5#waitFor} options to apply
       */
      oOptions?: SingleControlSelector | MultiControlSelector
    ): this;
    /**
     * Sets the `pollingInterval` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    pollingInterval(
      /**
       * the polling interval in milliseconds
       */
      iPollingInterval: int
    ): this;
    /**
     * Set the Opa5 instance to be used for {@link module:sap/ui/test/OpaBuilder#execute}. Please note that
     * this function does not return the OpaBuilder instance and can therefore not be chained. Use the `oOpaInstance`
     * argument of {@link module:sap/ui/test/OpaBuilder#create}, {@link module:sap/ui/test/OpaBuilder#constructor }
     * or {@link module:sap/ui/test/OpaBuilder#execute} to provide the Opa5 instance within the builder chain.
     */
    setOpaInstance(
      /**
       * the Opa5 instance to operate on
       */
      oOpaInstance?: Opa5
    ): void;
    /**
     * Adds a success message or function. When providing an OpaBuilder, the action will execute it.
     *
     *
     * @returns this OpaBuilder instance
     */
    success(
      /**
       * the message that will be shown (or function executed) on success
       */
      vSuccess: string | Function | OpaBuilder,
      /**
       * true to replace all previous defined success functions, false to add it (default)
       */
      bReplace?: boolean
    ): this;
    /**
     * Sets the `timeout` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    timeout(
      /**
       * the timeout in seconds
       */
      iTimeout: int
    ): this;
    /**
     * Sets the `viewId` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    viewId(
      /**
       * the viewId
       */
      sViewId: string
    ): this;
    /**
     * Sets the `viewName` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    viewName(
      /**
       * the viewName
       */
      sViewName: string
    ): this;
    /**
     * Sets the `viewNamespace` parameter.
     *
     *
     * @returns this OpaBuilder instance
     */
    viewNamespace(
      /**
       * the viewNamespace
       */
      sViewNamespace: string
    ): this;
  }
  /**
   * a declarative matcher definition for {@link sap.ui.test.matchers.Ancestor}
   */
  export type AncestorDefinition = Array<object | string | boolean>;

  /**
   * a declarative matcher definition for {@link sap.ui.test.matchers.Descendant}
   */
  export type DescendantDefinition = Array<object | string | boolean>;

  /**
   * a declarative matcher definition for {@link sap.ui.test.matchers.I18NText}
   */
  export type I18NTextDefinition = {
    propertyName: string;

    modelName: string;

    key: string;

    parameters?: string[];
  };

  /**
   * a declarative matcher definition for {@link sap.ui.test.matchers.LabelFor}
   */
  export type LabelForDefinition = {
    propertyName: string;

    text?: string;

    modelName?: string;

    key?: string;

    parameters?: any[];
  };
}

declare module "sap/ui/test/opaQunit" {
  /**
   * QUnit test adapter for OPA: add a test to be executed by QUnit Has the same signature as QUnit.test (QUnit
   * version is also considered) Suggested usage:
   * ```javascript
   *
   * sap.ui.require(["sap/ui/test/Opa5", "sap/ui/test/opaQunit"], function (Opa5, opaTest) {
   *
   *    Opa5.extendConfig({
   *        assertions: new Opa5({
   *            checkIfSomethingIsOk : function () {
   *                this.waitFor({
   *                    success: function () {
   *                        Opa5.assert.ok(true, "Everything is fine");
   *                    }
   *                });
   *            }
   *        })
   *    });
   *
   *    opaTest("Should test something", function (Given, When, Then) {
   *       // Implementation of the test
   *       Then.checkIfSomethingIsOk();
   *    });
   *
   * });
   * ```
   */
  export default function opaQunit(
    /**
     * name of the QUnit test.
     */
    testName: string,
    /**
     * only supported in QUnit v1.x: denotes how many assertions are expected to be made in the test.
     */
    expected: int,
    /**
     * the test function. Expects 3 arguments, in order: {@link sap.ui.test.Opa.config}.arrangements, {@link sap.ui.test.Opa.config}.actions,
     * {@link sap.ui.test.Opa.config}.assertions. These arguments will be prefilled by OPA
     */
    callback: Function,
    /**
     * available only in QUnit v1.x. Indicates whether the test is asynchronous. False by default.
     */
    async?: boolean
  ): void;
  /**
   * QUnit test adapter for OPA: add a test to be executed by QUnit Has the same signature as QUnit.test (QUnit
   * version is also considered) Suggested usage:
   * ```javascript
   *
   * sap.ui.require(["sap/ui/test/Opa5", "sap/ui/test/opaQunit"], function (Opa5, opaTest) {
   *
   *    Opa5.extendConfig({
   *        assertions: new Opa5({
   *            checkIfSomethingIsOk : function () {
   *                this.waitFor({
   *                    success: function () {
   *                        Opa5.assert.ok(true, "Everything is fine");
   *                    }
   *                });
   *            }
   *        })
   *    });
   *
   *    opaTest("Should test something", function (Given, When, Then) {
   *       // Implementation of the test
   *       Then.checkIfSomethingIsOk();
   *    });
   *
   * });
   * ```
   */
  export default function opaQunit(
    /**
     * name of the QUnit test.
     */
    testName: string,
    /**
     * the test function. Expects 3 arguments, in order: {@link sap.ui.test.Opa.config}.arrangements, {@link sap.ui.test.Opa.config}.actions,
     * {@link sap.ui.test.Opa.config}.assertions. These arguments will be prefilled by OPA
     */
    callback: Function,
    /**
     * available only in QUnit v1.x. Indicates whether the test is asynchronous. False by default.
     */
    async?: boolean
  ): void;
}

declare module "sap/ui/test/utils/nextUIUpdate" {
  /**
   * Return a Promise that resolves when the next Rendering is ready. If no rendering is sheduled it resolves
   * immediately.
   *
   * **Note:** No module from `sap/ui/test` should be used for productive coding!
   *
   * @since 1.127
   *
   * @returns A promise resolving when the next UI update is finished or rejecting when the next update fails.
   */
  export default function nextUIUpdate(
    /**
     * An optional sinon clock. When using sinon faketimers the clock must be ticked to ensure async rendering.
     * Async rendering is done with a setTimeout(0) so, when given, we tick the clock by 1.
     */
    clock?: {
      tick: Function;
    }
  ): Promise<undefined>;
}

declare module "sap/ui/test/utils/waitForThemeApplied" {
  /**
   * Checks whether the theme has already been applied and if not, waits for the 'applied' event to be fired.
   *
   * Returns a rejected promise if the Core is not ready yet.
   *
   * **Note:** No module from `sap/ui/test` should be used for productive coding!
   *
   * @since 1.127
   *
   * @returns Promise that resolves when the theme has been applied
   */
  export default function waitForThemeApplied(): Promise<undefined>;
}

declare module "sap/ui/util/Mobile" {
  /**
   * @since 1.58
   */
  interface Mobile {
    /**
     * Does some basic modifications to the HTML page that make it more suitable for mobile apps. Only the first
     * call to this method is executed, subsequent calls are ignored. Note that this method is also called by
     * the constructor of toplevel controls like sap.m.App, sap.m.SplitApp and sap.m.Shell. Exception: if `homeIcon`
     * or `mobileWebAppCapable` were not set, subsequent calls have the chance to set them.
     *
     * The "options" parameter configures what exactly should be done.
     *
     * It can have the following properties:
     * 	 - viewport: whether to set the viewport in a way that disables zooming (default: true). This does not
     *     work in case there is already a meta tag with name 'viewport'.
     * 	 - statusBar: the iOS status bar color, "default", "black" or "black-translucent" (default: "default")
     *
     * 	 - hideBrowser: whether the browser UI should be hidden as far as possible to make the app feel more
     *     native (default: true)
     * 	 - preventScroll: whether native scrolling should be disabled in order to prevent the "rubber-band"
     *     effect where the whole window is moved (default: true)
     * 	 - preventPhoneNumberDetection: whether Safari Mobile should be prevented from transforming any numbers
     *     that look like phone numbers into clickable links; this should be left as "true", otherwise it might
     *     break controls because Safari actually changes the DOM. This only affects all page content which is created
     *     after init() is called and only in case there is not already a meta tag with name 'format-detection'.
     *
     * 	 - rootId: the ID of the root element that should be made fullscreen; only used when hideBrowser is
     *     set (default: the document.body)
     * 	 - useFullScreenHeight: a boolean that defines whether the height of the html root element should be
     *     set to 100%, which is required for other elements to cover the full height (default: true) homeIcon:
     *     deprecated since 1.12, use sap/ui/util/Mobile.setIcons instead.
     */
    init(
      /**
       * configures what exactly should be done
       */
      options?: {
        /**
         * whether to set the viewport in a way that disables zooming
         */
        viewport?: boolean;
        /**
         * the iOS status bar color, "default", "black" or "black-translucent"
         */
        statusBar?: string;
        /**
         * whether the browser UI should be hidden as far as possible to make the app feel more native
         */
        hideBrowser?: boolean;
        /**
         * whether native scrolling should be disabled in order to prevent the "rubber-band" effect where the whole
         * window is moved
         */
        preventScroll?: boolean;
        /**
         * whether Safari mobile should be prevented from transforming any numbers that look like phone numbers
         * into clickable links
         */
        preventPhoneNumberDetection?: boolean;
        /**
         * the ID of the root element that should be made fullscreen; only used when hideBrowser is set. If not
         * set, the body is used
         */
        rootId?: string;
        /**
         * whether the height of the html root element should be set to 100%, which is required for other elements
         * to cover the full height
         */
        useFullScreenHeight?: boolean;
        /**
         * deprecated since 1.12, use sap/ui/util/Mobile.setIcons instead.
         */
        homeIcon?: string;
        /**
         * deprecated since 1.12, use sap/ui/util/Mobile.setIcons instead.
         */
        homeIconPrecomposed?: boolean;
        /**
         * whether the Application will be loaded in full screen mode after added to home screen on mobile devices.
         * The default value for this property only enables the full screen mode when runs on iOS device.
         */
        mobileWebAppCapable?: boolean;
      }
    ): void;
    /**
     * Sets the bookmark icon for desktop browsers and the icon to be displayed on the home screen of iOS devices
     * after the user does "add to home screen".
     *
     * Only call this method once and call it early when the page is loading: browsers behave differently when
     * the favicon is modified while the page is alive. Some update the displayed icon inside the browser but
     * use an old icon for bookmarks. When a favicon is given, any other existing favicon in the document will
     * be removed. When at least one home icon is given, all existing home icons will be removed and new home
     * icon tags for all four resolutions will be created.
     *
     * The home icons must be in PNG format and given in different sizes for iPad/iPhone with low and high pixel
     * density display. The favicon is used in the browser and for desktop shortcuts and should optimally be
     * in ICO format: ICO files can contain different image sizes for different usage locations. E.g. a 16x16px
     * version is used inside browsers.
     *
     * All icons are given in an an object holding icon URLs and other settings. The properties of this object
     * are:
     * 	 - phone: a 120x120 pixel version for iPhones with low pixel density
     * 	 - tablet: a 152x152 pixel version for iPads with low pixel density
     * 	 - phone@2: a 180x180 pixel version for iPhones with high pixel density
     * 	 - tablet@2: a 167x167 pixel version for iPads with high pixel density
     * 	 - precomposed: whether the home icons already have some glare effect (otherwise iOS will add it) (default:
     *     false)
     * 	 - favicon: the ICO file to be used inside the browser and for desktop shortcuts
     *
     * One example is:
     * ```javascript
     *
     * {
     *    'phone':'phone-icon_120x120.png',
     *    'phone@2':'phone-retina_180x180.png',
     *    'tablet':'tablet-icon_152x152.png',
     *    'tablet@2':'tablet-retina_167x167.png',
     *    'precomposed':true,
     *    'favicon':'desktop.ico'
     * }
     * ```
     *  If one of the sizes is not given, the largest available alternative image will be used instead for this
     * size. On Android these icons may or may not be used by the device. Apparently chances can be improved
     * by using icons with glare effect, so the "precomposed" property can be set to "true". Some Android devices
     * may also use the favicon for bookmarks instead of the home icons.
     */
    setIcons(
      /**
       * Icon settings
       */
      oIcons: {
        /**
         * a 120x120 pixel version for iPhones with low pixel density
         */
        phone?: string;
        /**
         * a 152x152 pixel version for iPads with low pixel density
         */
        tablet?: string;
        /**
         * a 180x180 pixel version for iPhones with high pixel density
         */
        "phone@2"?: string;
        /**
         * a 167x167 pixel version for iPads with high pixel density
         */
        "tablet@2"?: string;
        /**
         * whether the home icons already have some glare effect (otherwise iOS will add it)
         */
        precomposed?: boolean;
        /**
         * the ICO file to be used inside the browser and for desktop shortcuts
         */
        favicon?: string;
      }
    ): void;
    /**
     * Sets the "apple-mobile-web-app-capable" and "mobile-web-app-capable" meta information which defines whether
     * the application is loaded in full screen mode (browser address bar and toolbar are hidden) after the
     * user does "add to home screen" on mobile devices. Currently this meta tag is only supported by iOS Safari
     * and mobile Chrome from version 31.
     *
     * If the application opens new tabs because of attachments, url and so on, setting this to false will let
     * the user be able to go from the new tab back to the application tab after the application is added to
     * home screen.
     *
     * Note: this function only has effect when the application runs on iOS Safari and mobile Chrome from version
     * 31.
     */
    setWebAppCapable(
      /**
       * whether the Application will be loaded in full screen mode after added to home screen from iOS Safari
       * or mobile Chrome from version 31.
       */
      bValue: boolean
    ): void;
  }
  const Mobile: Mobile;
  export default Mobile;
}

declare module "sap/ui/util/Storage" {
  /**
   * A Storage API for JavaScript.
   *
   * **Note:** The Web Storage API stores the data on the client. Therefore, you must not use this API for
   * confidential information.
   *
   * Provides a unified interface and methods to store data on the client using the Web Storage API or a custom
   * implementation. By default, data can be persisted inside localStorage or sessionStorage.
   *
   * A typical intended usage of this API is the storage of a string representing the state of a control.
   * In this case, the data is stored in the browser session, and the methods to be used are {@link #put }
   * and {@link #get}. The method {@link #remove} can be used to delete the previously saved state.
   *
   * The `Storage` class allows a simple scoping by prefixing the keys of all `put/get/remove` operations
   * with a fixed prefix given when constructing a storage instance. By choosing unique prefixes, different
   * instances can write/read/delete data to the same underlying storage implementation without interfering
   * with each other.
   *
   * For the sake of completeness, the method {@link #clear} is available. However, it does not honor the
   * scoping and therefore should only be called when a global erasing of data is required. If only keys with
   * certain prefix should be deleted, the method {@link #removeAll} should be used.
   *
   * Besides creating an own storage instance, callers can use the static methods of the `Storage` class to
   * access a default session storage instance. All calls will use the same scope (same prefix).
   *
   * @since 1.58
   */
  export default class Storage {
    constructor(
      /**
       * The type of native storage implementation that this `Storage` instance should use internally. Alternatively,
       * this can be a custom implementation of the {@link https://developer.mozilla.org/en-US/docs/Web/API/Storage Storage Web API).
       */
      vStorage?: (Type | keyof typeof Type) | Storage,
      /**
       * The scope prefix to be used by this storage instance
       */
      sStorageKeyPrefix?: string
    );
    /**
     * Enumeration of the storage types supported by {@link module:sap/ui/util/Storage}.
     *
     * This enum is part of the 'sap/ui/util/Storage' module export and must be accessed by the property 'Type'.
     */
    static Type: typeof Type;

    /**
     * Clears the whole storage (Independent of the current Storage instance!).
     *
     * **CAUTION** This method should be called only in very particular situations, when a global erasing of
     * data is required. Given that the method deletes the data saved under any key, it should not be called
     * when managing data for specific controls.
     *
     *
     * @returns Whether clearing the storage was successful
     */
    static clear(): boolean;
    /**
     * Retrieves the value for the given key or `null` if the key does not exist in this storage.
     *
     * The returned value will be of a type that `JSON.parse` could return, too.
     *
     *
     * @returns The key's value or `null` if the key does not exist in the storage.
     */
    static get(
      /**
       * Key to retrieve the value for; will be prefixed with the prefix given when constructing this `Storage`
       */
      sKey: string
    ): any;
    /**
     * Returns the type of this storage.
     *
     *
     * @returns Type of this storage or "unknown" when the Storage was created with an unknown type or implementation
     */
    static getType(): Type | string;
    /**
     * Returns whether the given storage is supported.
     *
     *
     * @returns true if storage is supported, false otherwise (e.g. due to browser security settings)
     */
    static isSupported(): boolean;
    /**
     * Stores the given value under the given key in the storage or updates the value if the key already exists.
     *
     * This method supports the same types of values as `JSON.stringify`.
     *
     *
     * @returns Whether the data was successfully stored
     */
    static put(
      /**
       * Key to store the given value under; will be prefixed with the prefix given when constructing this `Storage`
       */
      sKey: string,
      /**
       * Value to store/update under the given key
       */
      vValue: any
    ): boolean;
    /**
     * Removes the key and its value from storage, if the key exists.
     *
     *
     * @returns Whether the deletion succeeded; if the key didn't exists, the method also reports a success
     */
    static remove(
      /**
       * Key to remove; will be prefixed with the prefix given when constructing this `Storage`
       */
      sKey: string
    ): boolean;
    /**
     * Removes all key/value pairs form the storage where the key starts with the given `sKeyPrefix`.
     *
     *
     * @returns Whether the deletion was successful; if no key matches the prefix, this is also a success
     */
    static removeAll(
      /**
       * Key prefix for the keys/values to delete; will be additionally prefixed with the prefix given when constructing
       * this `Storage`
       */
      sKeyPrefix?: string
    ): boolean;
    /**
     * Clears the whole storage (Independent of the current Storage instance!).
     *
     * **CAUTION** This method should be called only in very particular situations, when a global erasing of
     * data is required. Given that the method deletes the data saved under any key, it should not be called
     * when managing data for specific controls.
     *
     *
     * @returns Whether clearing the storage was successful
     */
    clear(): boolean;
    /**
     * Retrieves the value for the given key or `null` if the key does not exist in this storage.
     *
     * The returned value will be of a type that `JSON.parse` could return, too.
     *
     *
     * @returns The key's value or `null` if the key does not exist in the storage.
     */
    get(
      /**
       * Key to retrieve the value for; will be prefixed with the prefix given when constructing this `Storage`
       */
      sKey: string
    ): any;
    /**
     * Returns the type of this storage.
     *
     *
     * @returns Type of this storage or "unknown" when the Storage was created with an unknown type or implementation
     */
    getType(): Type | string;
    /**
     * Returns whether the given storage is supported.
     *
     *
     * @returns true if storage is supported, false otherwise (e.g. due to browser security settings)
     */
    isSupported(): boolean;
    /**
     * Stores the given value under the given key in the storage or updates the value if the key already exists.
     *
     * This method supports the same types of values as `JSON.stringify`.
     *
     *
     * @returns Whether the data was successfully stored
     */
    put(
      /**
       * Key to store the given value under; will be prefixed with the prefix given when constructing this `Storage`
       */
      sKey: string,
      /**
       * Value to store/update under the given key
       */
      vValue: any
    ): boolean;
    /**
     * Removes the key and its value from storage, if the key exists.
     *
     *
     * @returns Whether the deletion succeeded; if the key didn't exists, the method also reports a success
     */
    remove(
      /**
       * Key to remove; will be prefixed with the prefix given when constructing this `Storage`
       */
      sKey: string
    ): boolean;
    /**
     * Removes all key/value pairs form the storage where the key starts with the given `sKeyPrefix`.
     *
     *
     * @returns Whether the deletion was successful; if no key matches the prefix, this is also a success
     */
    removeAll(
      /**
       * Key prefix for the keys/values to delete; will be additionally prefixed with the prefix given when constructing
       * this `Storage`
       */
      sKeyPrefix?: string
    ): boolean;
  }
  /**
   * Enumeration of the storage types supported by {@link module:sap/ui/util/Storage}.
   *
   * This enum is part of the 'sap/ui/util/Storage' module export and must be accessed by the property 'Type'.
   */
  enum Type {
    /**
     * Indicates usage of the browser's localStorage feature
     */
    local = "local",
    /**
     * Indicates usage of the browser's sessionStorage feature
     */
    session = "session",
  }
}

declare module "sap/ui/VersionInfo" {
  /**
   * @since 1.56.0
   */
  interface VersionInfo {
    /**
     * Loads the version info asynchronously from resource "sap-ui-version.json".
     *
     * By default, the returned promise will resolve with the whole version info file's content. If a library
     * name is specified in the options, then the promise will resolve with the version info for that library
     * only or with `undefined`, if the named library is not listed in the version info file.
     *
     * If loading the version info file fails, the promise will be rejected with the corresponding error.
     *
     * @since 1.56.0
     *
     * @returns A promise which resolves with the full version info or with the library specific version info
     * or `undefined` if the library is not listed; if an error occurred during loading, then the promise is
     * rejected.
     */
    load(
      /**
       * Map of options
       */
      mOptions?: {
        /**
         * Name of a library (e.g. "sap.ui.core")
         */
        library?: string;
      }
    ): Promise<object | undefined>;
  }
  const VersionInfo: VersionInfo;
  export default VersionInfo;
}

declare module "sap/ui/app/Application" {
  import {
    default as Component,
    $ComponentSettings,
  } from "sap/ui/core/Component";

  import UIComponent from "sap/ui/core/UIComponent";

  import ComponentMetadata from "sap/ui/core/ComponentMetadata";

  import Model from "sap/ui/model/Model";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Abstract application class. Extend this class to create a central application class.
   *
   * @deprecated As of version 1.15.1. The Component class is enhanced to take care about the Application
   * code.
   * @experimental As of version 1.11.1. The Application class is still under construction, so some implementation
   * details can be changed in future.
   */
  export default abstract class Application extends Component {
    /**
     * Creates an application instance, only one instance is allowed (singleton).
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new application instance
       */
      mSettings?: $ApplicationSettings
    );
    /**
     * Creates an application instance, only one instance is allowed (singleton).
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Optional ID for the application; generated automatically if no non-empty ID is given. **Note:** this
       * can be omitted, no matter whether `mSettings` will be given or not
       */
      sId?: string,
      /**
       * Initial settings for the new application instance
       */
      mSettings?: $ApplicationSettings
    );

    /**
     * Creates a new subclass of class sap.ui.app.Application with name `sClassName` and enriches it with the
     * information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Component.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Application>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.app.Application.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ComponentMetadata;
    /**
     * Creates and returns the root component. Override this method in your application implementation, if you
     * want to override the default creation by metadata.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the root component
     */
    createRootComponent(): UIComponent;
    /**
     * See:
     * 	sap.ui.core.Component#destroy
     */
    destroy(): void;
    /**
     * Destroys the rootComponent in the aggregation {@link #getRootComponent rootComponent}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyRootComponent(): this;
    /**
     * Gets current value of property {@link #getConfig config}.
     *
     *
     * @returns Value of property `config`
     */
    getConfig(): any;
    /**
     * Gets current value of property {@link #getRoot root}.
     *
     *
     * @returns Value of property `root`
     */
    getRoot(): string;
    /**
     * Gets content of aggregation {@link #getRootComponent rootComponent}.
     */
    getRootComponent(): UIComponent;
    /**
     * Returns the application root component.
     *
     * @since 1.13.1
     * @deprecated As of version 1.14.
     *
     * @returns The root component
     */
    getView(): UIComponent;
    /**
     * The main method is called when the DOM and UI5 is completely loaded. Override this method in your Application
     * class implementation to execute code which relies on a loaded DOM / UI5.
     */
    main(): void;
    /**
     * On before exit application hook. Override this method in your Application class implementation, to handle
     * cleanup before the real exit or to prompt a question to the user, if the application should be exited.
     *
     *
     * @returns return a string if a prompt should be displayed to the user confirming closing the application
     * (e.g. when the application is not yet saved).
     */
    onBeforeExit(): string;
    /**
     * On error hook. Override this method in your Application class implementation to listen to unhandled errors.
     */
    onError(
      /**
       * The error message.
       */
      sMessage: string,
      /**
       * The file where the error occurred
       */
      sFile: string,
      /**
       * The line number of the error
       */
      iLine: number
    ): void;
    /**
     * On exit application hook. Override this method in your Application class implementation, to handle cleanup
     * of the application.
     */
    onExit(): void;
    /**
     * Sets the configuration model.
     *
     * @since 1.13.1
     */
    setConfig(
      /**
       * the configuration model, the configuration object or a URI string to load a JSON configuration file.
       */
      vConfig: string | object | Model
    ): void;
    /**
     * Sets a new value for property {@link #getRoot root}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRoot(
      /**
       * New value for property `root`
       */
      sRoot: string
    ): this;
    /**
     * Sets the aggregated {@link #getRootComponent rootComponent}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRootComponent(
      /**
       * The rootComponent to set
       */
      oRootComponent: UIComponent
    ): this;
  }
  /**
   * Describes the settings that can be provided to the Application constructor.
   *
   * @deprecated As of version 1.15.1. The Component class is enhanced to take care about the Application
   * code.
   * @experimental As of version 1.11.1. The Application class is still under construction, so some implementation
   * details can be changed in future.
   */
  export interface $ApplicationSettings extends $ComponentSettings {
    root?: string | PropertyBindingInfo;

    config?: any | PropertyBindingInfo | `{${string}}`;

    rootComponent?: UIComponent;
  }
}

declare module "sap/ui/app/MockServer" {
  import {
    default as ManagedObject,
    $ManagedObjectSettings,
  } from "sap/ui/base/ManagedObject";

  /**
   * Class to mock a server.
   *
   * @deprecated As of version 1.15.1. The mock server code has been moved to sap.ui.core.util - see {@link sap.ui.core.util.MockServer}
   * @experimental As of version 1.13.0. The mock server is still under construction, so some implementation
   * details can be changed in future.
   */
  export default abstract class MockServer extends ManagedObject {
    /**
     * Creates a mocked server. This helps to mock all or some backend calls, e.g. for OData/JSON Models or
     * simple XHR calls, without changing the application code. This class can also be used for qunit tests.
     */
    constructor(
      /**
       * optional map/JSON-object with initial property values, aggregated objects etc. for the new object
       */
      mSettings?: $MockServerSettings,
      /**
       * scope object for resolving string based type and formatter references in bindings
       */
      oScope?: object
    );
    /**
     * Creates a mocked server. This helps to mock all or some backend calls, e.g. for OData/JSON Models or
     * simple XHR calls, without changing the application code. This class can also be used for qunit tests.
     */
    constructor(
      /**
       * id for the new server object; generated automatically if no non-empty id is given Note: this can be omitted,
       * no matter whether `mSettings` will be given or not!
       */
      sId?: string,
      /**
       * optional map/JSON-object with initial property values, aggregated objects etc. for the new object
       */
      mSettings?: $MockServerSettings,
      /**
       * scope object for resolving string based type and formatter references in bindings
       */
      oScope?: object
    );
  }
  /**
   * Describes the settings that can be provided to the MockServer constructor.
   *
   * @deprecated As of version 1.15.1. The mock server code has been moved to sap.ui.core.util - see {@link sap.ui.core.util.MockServer}
   * @experimental As of version 1.13.0. The mock server is still under construction, so some implementation
   * details can be changed in future.
   */
  export interface $MockServerSettings extends $ManagedObjectSettings {}
}

declare module "sap/ui/base/DataType" {
  /**
   * Represents the type of properties in a `ManagedObject` class.
   *
   * Each type provides some metadata like its {@link #getName qualified name} or its {@link #getBaseType base type }
   * in case of a derived type. Array types provide information about the allowed {@link #getComponentType type of components }
   * in an array, enumeration types inform about the set of their allowed {@link #getEnumValues keys and values}.
   *
   * Each type has a method to {@link #isValid check whether a value is valid} for a property of that type.
   *
   * Already defined types can be looked up by calling {@link #.getType DataType.getType}, new types can only
   * be created by calling the factory method {@link #.createType DataType.createType}, calling the constructor
   * will throw an error.
   *
   * @since 0.9.0
   */
  export default class DataType {
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor();

    /**
     * Derives a new type from a given base type.
     *
     * Example:
     *
     * ```javascript
     *
     *
     *   var fooType = DataType.createType('foo', {
     *       isValid : function(vValue) {
     *           return /^(foo(bar)?)$/.test(vValue);
     *       }
     *   }, DataType.getType('string'));
     *
     *   fooType.isValid('foo'); // true
     *   fooType.isValid('foobar'); // true
     *   fooType.isValid('==foobar=='); // false
     *
     * ```
     *
     *
     * If `mSettings` contains an implementation for `isValid`, then the validity check of the newly created
     * type will first execute the check of the base type and then call the given `isValid` function.
     *
     * Array types and enumeration types cannot be created with this method. They're created on-the-fly by {@link #.getType DataType.getType }
     * when such a type is looked up.
     *
     * **Note:** The creation of new primitive types is not supported. When a type is created without a base
     * type, it is automatically derived from the primitive type `any`.
     *
     * **Note:** If a type has to be used in classes, then the implementation of `isValid` must exactly have
     * the structure shown in the example above (single return statement, regular expression literal of the
     * form `/^(...)$/`, calling `/regex/.test()` on the given value). Only the inner part of the regular expression
     * literal can be different.
     *
     *
     * @returns The newly created type object
     */
    static createType(
      /**
       * Unique qualified name of the new type
       */
      sName: string,
      /**
       * Settings for the new type
       */
      mSettings?: {
        /**
         * Default value for the type (inherited if not given)
         */
        defaultValue?: any;
        /**
         * Additional validity check function for values of the type (inherited if not given)
         */
        isValid?: Function;
        /**
         * Parse function that converts a locale independent string into a value of the type (inherited if not given)
         */
        parseValue?: Function;
      },
      /**
       * Base type for the new type
       */
      vBase?: DataType | string
    ): DataType;
    /**
     * Looks up the type with the given name and returns it.
     *
     * See {@link https://ui5.sap.com/#/topic/ac56d92162ed47ff858fdf1ce26c18c4 Defining Control Properties }
     * for a list of the built-in primitive types and their semantics.
     *
     * The lookup consists of the following steps:
     * 	 - When a type with the given name is already known, it will be returned
     * 	 - When the name ends with a pair of brackets (`[]`), a type with the name in front of the brackets
     *     (`name.slice(0,-2)`) will be looked up and an array type will be created with the looked-up type as its
     *     component type. If the component type is `undefined`, `undefined` will be returned
     * 	 - When a global property exists with the same name as the type and when the value of that property
     *     is an instance of `DataType`, that instance will be returned
     * 	 - When a global property exists with the same name as the type and when the value of that property
     *     is a plain object (its prototype is `Object`), then an enum type will be created, based on the keys and
     *     values in that object. The `parseValue` method of the type will accept any of the keys in the plain object
     *     and convert them to the corresponding value; `isValid` will accept any of the values from the plain object's
     *     keys. The `defaultValue` will be the value of the first key found in the plain object
     * 	 - When a global property exist with any other, non-falsy value, a warning is logged and the primitive
     *     type 'any' is returned
     * 	 - If no such global property exist, an error is logged and `undefined` is returned
     *
     *  UI Libraries and even components can introduce additional types. This method only checks
     * for types that either have been defined already, or that describe arrays of values of an already defined
     * type or types whose name matches the global name of a plain object (containing enum keys and values).
     * This method doesn't try to load modules that might contain type definitions. So before being able to
     * lookup and use a specific type, the module containing its definition has to be loaded. For that reason
     * it is suggested that controls (or `ManagedObject` classes in general) declare a dependency to all modules
     * (typically `some/lib/library.js` modules) that contain the type definitions needed by the specific control
     * or class definition.
     *
     *
     * @returns Type object or `undefined` when no such type has been defined yet
     */
    static getType(
      /**
       * Qualified name of the type to retrieve
       */
      sTypeName: string,
      /**
       * Metadata of the property
       */
      oProperty?: /* was: sap.ui.base.ManagedObject.MetaOptions.Property */ any
    ): DataType | undefined;
    /**
     * Registers an enum under the given name. With version 2.0, registering an enum becomes mandatory when
     * said enum is to be used in properties of a {@link sap.ui.base.ManagedObject ManagedObject} subclass.
     *
     * Example:
     *
     * ```javascript
     *
     *    DataType.registerEnum("my.enums.Sample", {
     *       "A": "A",
     *       "B": "B",
     *       ...
     *    });
     * ```
     *
     *
     * @since 1.120.0
     */
    static registerEnum(
      /**
       * the type name in dot syntax, e.g. sap.ui.my.EnumType
       */
      sTypeName: string,
      /**
       * the enum content
       */
      mContent: object
    ): void;
    /**
     * The base type of this type or undefined if this is a primitive type.
     *
     *
     * @returns Base type or `undefined`
     */
    getBaseType(): DataType | undefined;
    /**
     * Returns the component type of this type or `undefined` if this is not an array type.
     *
     *
     * @returns Component type or `undefined`
     */
    getComponentType(): DataType | undefined;
    /**
     * The default value for this type. Each type must define a default value.
     *
     *
     * @returns Default value of the data type. The type of the returned value must match the JavaScript type
     * of the data type (a string for string types etc.)
     */
    getDefaultValue(): any;
    /**
     * Returns the object with keys and values from which this enum type was created or `undefined` if this
     * is not an enum type.
     *
     *
     * @returns Object with enum keys and values or `undefined`
     */
    getEnumValues(): Record<string, string> | undefined;
    /**
     * The qualified name of the data type.
     *
     *
     * @returns Name of the data type
     */
    getName(): string;
    /**
     * Returns the most basic (primitive) type that this type has been derived from.
     *
     * If the type is a primitive type by itself, `this` is returned.
     *
     *
     * @returns Primitive type of this type
     */
    getPrimitiveType(): DataType;
    /**
     * Whether this type is an array type.
     *
     *
     * @returns Whether this type is an array type
     */
    isArrayType(): boolean;
    /**
     * Whether this type is an enumeration type.
     *
     *
     * @returns Whether this type is an enum type
     */
    isEnumType(): boolean;
    /**
     * Checks whether the given value is valid for this type.
     *
     * To be implemented by concrete types.
     *
     *
     * @returns Whether the given value is valid for this data type (without conversion)
     */
    isValid(
      /**
       * Value to be checked
       */
      vValue: any
    ): boolean;
    /**
     * Normalizes the given value using the specified normalizer for this data type.
     *
     * If no normalizer has been set, the original value is returned.
     *
     *
     * @returns Normalized value
     */
    normalize(
      /**
       * Value to be normalized
       */
      oValue: any
    ): any;
    /**
     * Parses the given string value and converts it into the specific data type.
     *
     *
     * @returns Value in the correct internal format
     */
    parseValue(
      /**
       * String representation for a value of this type
       */
      sValue: string
    ): any;
    /**
     * Set or unset a normalizer function to be used for values of this data type.
     *
     * When a normalizer function has been set, it will be applied to values of this type whenever {@link #normalize }
     * is called. `ManagedObject.prototype.setProperty` calls the `normalize` method before setting a new value
     * to a property (normalization is applied on-write, not on-read).
     *
     * The `fnNormalize` function has the signature
     * ```javascript
     *
     *   fnNormalize(value:any) : any
     * ```
     *  It will be called with a value for this type and should return a normalized value (which also must be
     * valid for the this type). There's no mean to reject a value. The `this` context of the function will
     * be this type.
     *
     * This method allows applications or application frameworks to plug-in a generic value normalization for
     * a type, e.g. to convert all URLs in some app-specific way before they are applied to controls. It is
     * not intended to break-out of the value range defined by a type.
     */
    setNormalizer(
      /**
       * Function to apply for normalizing
       */
      fnNormalizer: (p1: any) => any
    ): void;
  }
}

declare module "sap/ui/base/Event" {
  import EventProvider from "sap/ui/base/EventProvider";

  import BaseObject from "sap/ui/base/Object";

  import { Poolable } from "sap/ui/base/ObjectPool";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * An Event object consisting of an ID, a source and a map of parameters. Implements {@link sap.ui.base.Poolable }
   * and therefore an event object in the event handler will be reset by {@link sap.ui.base.ObjectPool} after
   * the event handler is done.
   */
  export default class Event<
      ParamsType extends Record<string, any> = object,
      SourceType extends EventProvider = EventProvider,
    >
    extends BaseObject
    implements Poolable
  {
    __implements__sap_ui_base_Poolable: boolean;
    /**
     * Creates an event with the given `sId`, linked to the provided `oSource` and enriched with the `mParameters`.
     */
    constructor(
      /**
       * The ID of the event
       */
      sId: string,
      /**
       * Source of the event
       */
      oSource: SourceType,
      /**
       * Parameters for this event
       */
      oParameters: ParamsType
    );

    /**
     * Creates a new subclass of class sap.ui.base.Event with name `sClassName` and enriches it with the information
     * contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Event>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.base.Event.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Cancel bubbling of the event.
     *
     * **Note:** This function only has an effect if the bubbling of the event is supported by the event source.
     */
    cancelBubble(): void;
    /**
     * Returns the id of the event.
     *
     *
     * @returns The ID of the event
     */
    getId(): string;
    /**
     * Returns the value of the parameter with the given name.
     *
     *
     * @returns Value of the named parameter
     */
    getParameter<ParamName extends keyof ParamsType>(
      /**
       * Name of the parameter to return
       */
      sName: ParamName
    ): ParamsType[ParamName];
    /**
     * Returns an object with all parameter values of the event.
     *
     *
     * @returns All parameters of the event
     */
    getParameters(): ParamsType;
    /**
     * Returns the event provider on which the event was fired.
     *
     *
     * @returns The source of the event
     */
    getSource<T extends SourceType>(): T;
    /**
     * Init this event with its data.
     *
     * The `init` method is called by an object pool when the object is (re-)activated for a new caller.
     *
     * When no `oParameters` are given, an empty object is used instead.
     * See:
     * 	sap.ui.base.Poolable.prototype#init
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    init(
      /**
       * ID of the event
       */
      sId: string,
      /**
       * Source of the event
       */
      oSource: SourceType,
      /**
       * The event parameters
       */
      oParameters?: ParamsType
    ): void;
    /**
     * Prevent the default action of this event.
     *
     * **Note:** This function only has an effect if preventing the default action of the event is supported
     * by the event source.
     */
    preventDefault(): void;
    /**
     * Reset event data, needed for pooling.
     * See:
     * 	sap.ui.base.Poolable.prototype#reset
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    reset(): void;
  }
}

declare module "sap/ui/base/EventProvider" {
  import BaseObject from "sap/ui/base/Object";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Provides eventing capabilities for objects like attaching or detaching event handlers for events which
   * are notified when events are fired.
   */
  export default class EventProvider extends BaseObject {
    /**
     * Creates an instance of EventProvider.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.ui.base.EventProvider with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, EventProvider>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.base.EventProvider.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Attaches an event handler to the event with the given identifier.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    attachEvent(
      /**
       * The identifier of the event to listen for
       */
      sEventId: string,
      /**
       * An object that will be passed to the handler along with the event object when the event is fired
       */
      oData: object,
      /**
       * The handler function to call when the event occurs. This function will be called in the context of the
       * `oListener` instance (if present) or on the event provider instance. The event object ({@link sap.ui.base.Event})
       * is provided as first argument of the handler. Handlers must not change the content of the event. The
       * second argument is the specified `oData` instance (if present).
       */
      fnFunction: Function,
      /**
       * The object that wants to be notified when the event occurs (`this` context within the handler function).
       * If it is not specified, the handler function is called in the context of the event provider.
       */
      oListener?: object
    ): this;
    /**
     * Attaches an event handler to the event with the given identifier.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    attachEvent(
      /**
       * The identifier of the event to listen for
       */
      sEventId: string,
      /**
       * The handler function to call when the event occurs. This function will be called in the context of the
       * `oListener` instance (if present) or on the event provider instance. The event object ({@link sap.ui.base.Event})
       * is provided as first argument of the handler. Handlers must not change the content of the event. The
       * second argument is the specified `oData` instance (if present).
       */
      fnFunction: Function,
      /**
       * The object that wants to be notified when the event occurs (`this` context within the handler function).
       * If it is not specified, the handler function is called in the context of the event provider.
       */
      oListener?: object
    ): this;
    /**
     * Attaches an event handler, called one time only, to the event with the given identifier.
     *
     * When the event occurs, the handler function is called and the handler registration is automatically removed
     * afterwards.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    attachEventOnce(
      /**
       * The identifier of the event to listen for
       */
      sEventId: string,
      /**
       * An object that will be passed to the handler along with the event object when the event is fired
       */
      oData: object,
      /**
       * The handler function to call when the event occurs. This function will be called in the context of the
       * `oListener` instance (if present) or on the event provider instance. The event object ({@link sap.ui.base.Event})
       * is provided as first argument of the handler. Handlers must not change the content of the event. The
       * second argument is the specified `oData` instance (if present).
       */
      fnFunction: Function,
      /**
       * The object that wants to be notified when the event occurs (`this` context within the handler function).
       * If it is not specified, the handler function is called in the context of the event provider.
       */
      oListener?: object
    ): this;
    /**
     * Attaches an event handler, called one time only, to the event with the given identifier.
     *
     * When the event occurs, the handler function is called and the handler registration is automatically removed
     * afterwards.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    attachEventOnce(
      /**
       * The identifier of the event to listen for
       */
      sEventId: string,
      /**
       * The handler function to call when the event occurs. This function will be called in the context of the
       * `oListener` instance (if present) or on the event provider instance. The event object ({@link sap.ui.base.Event})
       * is provided as first argument of the handler. Handlers must not change the content of the event. The
       * second argument is the specified `oData` instance (if present).
       */
      fnFunction: Function,
      /**
       * The object that wants to be notified when the event occurs (`this` context within the handler function).
       * If it is not specified, the handler function is called in the context of the event provider.
       */
      oListener?: object
    ): this;
    /**
     * Cleans up the internal structures and removes all event handlers.
     *
     * The object must not be used anymore after destroy was called.
     * See:
     * 	sap.ui.base.Object#destroy
     */
    destroy(): void;
    /**
     * Removes a previously attached event handler from the event with the given identifier.
     *
     * The passed parameters must match those used for registration with {@link #attachEvent} beforehand.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    detachEvent(
      /**
       * The identifier of the event to detach from
       */
      sEventId: string,
      /**
       * The handler function to detach from the event
       */
      fnFunction: Function,
      /**
       * The object that wanted to be notified when the event occurred
       */
      oListener?: object
    ): this;
    /**
     * Fires an {@link sap.ui.base.Event event} with the given settings and notifies all attached event handlers.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining. When `preventDefault` is supported on the fired event
     * the function returns `true` if the default action should be executed, `false` otherwise.
     */
    fireEvent(
      /**
       * The identifier of the event to fire
       */
      sEventId: string,
      /**
       * Parameters which should be carried by the event
       */
      oParameters?: object,
      /**
       * Defines whether function `preventDefault` is supported on the fired event
       */
      bAllowPreventDefault?: boolean,
      /**
       * Defines whether event bubbling is enabled on the fired event. Set to `true` the event is also forwarded
       * to the parent(s) of the event provider ({@link #getEventingParent}) until the bubbling of the event is
       * stopped or no parent is available anymore.
       */
      bEnableEventBubbling?: boolean
    ): this | boolean;
    /**
     * Returns the parent in the eventing hierarchy of this object.
     *
     * Per default this returns null, but if eventing is used in objects, which are hierarchically structured,
     * this can be overwritten to make the object hierarchy visible to the eventing and enables the use of event
     * bubbling within this object hierarchy.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The parent event provider
     */
    getEventingParent(): EventProvider | null;
    /**
     * Returns whether there are any registered event handlers for the event with the given identifier.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether there are any registered event handlers
     */
    hasListeners(
      /**
       * The identifier of the event
       */
      sEventId: string
    ): boolean;
    /**
     * Returns a string representation of this object.
     *
     * In case there is no class or id information, a simple static string is returned. Subclasses should override
     * this method.
     *
     *
     * @returns A string description of this event provider
     */
    toString(): string;
  }
}

declare module "sap/ui/base/Interface" {
  import BaseObject from "sap/ui/base/Object";

  /**
   * A class whose instances act as a facade for other objects.
   *
   * **Note:** If a class returns a facade in its constructor, only the defined functions will be visible,
   * no internals of the class can be accessed.
   */
  export default class Interface {
    /**
     * Constructs a facade for the given object, containing only the named methods.
     *
     * For each method named in `aMethods`, a wrapper function will be added to the facade. When called, the
     * wrapper function calls the method with the same name in the original `oObject`, passing all its call
     * parameters to it without modification. A return value of the original method will be returned to the
     * caller. Before returning it, values of type `sap.ui.base.Object` will be replaced by their facades, calling
     * {@link sap.ui.base.Object#getInterface getInterface} on them.
     *
     * It is possible to create different facades exposing different sets of methods for the same object, but
     * as `getInterface` can only return one of those interfaces, the special handling of the return values
     * doesn't support multiple facades per object.
     */
    constructor(
      /**
       * Object for which a facade should be created
       */
      oObject: BaseObject,
      /**
       * Names of the methods, that should be available in the new facade
       */
      aMethods: string[]
    );
  }
}

declare module "sap/ui/base/ManagedObject" {
  import EventProvider from "sap/ui/base/EventProvider";

  import Event from "sap/ui/base/Event";

  import Context from "sap/ui/model/Context";

  import Binding from "sap/ui/model/Binding";

  import ManagedObjectMetadata from "sap/ui/base/ManagedObjectMetadata";

  import Model from "sap/ui/model/Model";

  import ContextBinding from "sap/ui/model/ContextBinding";

  import ChangeReason from "sap/ui/model/ChangeReason";

  import Sorter from "sap/ui/model/Sorter";

  import Filter from "sap/ui/model/Filter";

  import { MetadataOptions as MetadataOptions1 } from "sap/ui/base/Object";

  import Type from "sap/ui/model/Type";

  import BindingMode from "sap/ui/model/BindingMode";

  import { ID } from "sap/ui/core/library";

  /**
   * Base Class that introduces some basic concepts, such as, state management and data binding.
   *
   * New subclasses of ManagedObject are created with a call to {@link #.extend ManagedObject.extend} and
   * can make use of the following managed features:
   *
   * Properties: Managed properties represent the state of a ManagedObject. They can store a single value
   * of a simple data type (like 'string' or 'int'). They have a name (e.g. 'size') and methods to
   * get the current value (`getSize`), or to set a new value (`setSize`). When a property is modified by
   * calling the setter, the ManagedObject is marked as invalidated. A managed property can be bound against
   * a property in a {@link sap.ui.model.Model} by using the {@link #bindProperty} method. Updates to the
   * model property will be automatically reflected in the managed property and - if TwoWay databinding is
   * active, changes to the managed property will be reflected in the model. An existing binding can be removed
   * by calling {@link #unbindProperty}.
   *
   * If a ManagedObject is cloned, the clone will have the same values for its managed properties as the source
   * of the clone - if the property wasn't bound. If it is bound, the property in the clone will be bound
   * to the same model property as in the source.
   *
   * Details about the declaration of a managed property, the metadata that describes it and the set of methods
   * that are automatically generated to access it, can be found in the documentation of the {@link sap.ui.base.ManagedObject.extend extend  }
   * method.
   *
   * Aggregations: Managed aggregations can store one or more references to other ManagedObjects. They are
   * a mean to control the lifecycle of the aggregated objects: one ManagedObject can be aggregated by at
   * most one parent ManagedObject at any time. When a ManagedObject is destroyed, all aggregated objects
   * are destroyed as well and the object itself is removed from its parent. That is, aggregations won't contain
   * destroyed objects or null/undefined.
   *
   * Aggregations have a name ('e.g 'header' or 'items'), a cardinality ('0..1' or '0..n') and
   * are of a specific type (which must be a subclass of ManagedObject as well or a UI5 interface).
   * A ManagedObject will provide methods to set or get the aggregated object for a specific aggregation of
   * cardinality 0..1 (e.g. `setHeader`, `getHeader` for an aggregation named 'header'). For an aggregation
   * of cardinality 0..n, there are methods to get all aggregated objects (`getItems`), to locate an object
   * in the aggregation (e.g. `indexOfItem`), to add, insert or remove a single aggregated object (`addItem`,
   * `insertItem`, `removeItem`) or to remove or destroy all objects from an aggregation (`removeAllItems`,
   * `destroyItems`).
   *
   * Details about the declaration of a managed aggregation, the metadata that describes the aggregation,
   * and the set of methods that are automatically generated to access it, can be found in the documentation
   * of the {@link sap.ui.base.ManagedObject.extend extend} method.
   *
   * Aggregations of cardinality 0..n can be bound to a collection in a model by using {@link #bindAggregation }
   * (and unbound again using {@link #unbindAggregation}). For each context in the model collection, a corresponding
   * object will be created in the managed aggregation, either by cloning a template object or by calling
   * a factory function.
   *
   * Aggregations also control the databinding context of bound objects: by default, aggregated objects inherit
   * all models and binding contexts from their parent object.
   *
   * When a ManagedObject is cloned, all aggregated objects will be cloned as well - but only if they haven't
   * been added by databinding. In that case, the aggregation in the clone will be bound to the same model
   * collection.
   *
   * Associations: Managed associations also form a relationship between objects, but they don't define a
   * lifecycle for the associated objects. They even can 'break' in the sense that an associated object might
   * have been destroyed already although it is still referenced in an association. For the same reason, the
   * internal storage for associations are not direct object references but only the IDs of the associated
   * target objects.
   *
   * Associations have a name ('e.g 'initialFocus'), a cardinality ('0..1' or '0..n') and are
   * of a specific type (which must be a subclass of ManagedObject as well or a UI5 interface). A ManagedObject
   * will provide methods to set or get the associated object for a specific association of cardinality 0..1
   * (e.g. `setInitialFocus`, `getInitialFocus`). For an association of cardinality 0..n, there are methods
   * to get all associated objects (`getRefItems`), to add, insert or remove a single associated object (`addRefItem`,
   * `insertRefItem`, `removeRefItem`) or to remove all objects from an association (`removeAllRefItems`).
   *
   * Details about the declaration of a managed association, the metadata that describes it and the set of
   * methods that are automatically generated to access it, can be found in the documentation of the {@link sap.ui.base.ManagedObject.extend extend }
   * method.
   *
   * Associations can't be bound to the model.
   *
   * When a ManagedObject is cloned, the result for an association depends on the relationship between the
   * associated target object and the root of the clone operation. If the associated object is part of the
   * to-be-cloned object tree (reachable via aggregations from the root of the clone operation), then the
   * cloned association will reference the clone of the associated object. Otherwise the association will
   * reference the same object as in the original tree. When a ManagedObject is destroyed, other objects that
   * are only associated, are not affected by the destroy operation.
   *
   * Events: Managed events provide a mean for communicating important state changes to an arbitrary number
   * of 'interested' listeners. Events have a name and (optionally) a set of parameters. For
   * each event there will be methods to add or remove an event listener as well as a method to fire the event.
   * (e.g. `attachChange`, `detachChange`, `fireChange` for an event named 'change').
   *
   * Details about the declaration of managed events, the metadata that describes the event, and the set of
   * methods that are automatically generated to access it, can be found in the documentation of the {@link sap.ui.base.ManagedObject.extend extend }
   * method.
   *
   * When a ManagedObject is cloned, all listeners registered for any event in the clone source are also registered
   * to the clone. Later changes are not reflected in any direction (neither from source to clone, nor vice
   * versa).
   *
   * Low Level APIs:: The prototype of ManagedObject provides several generic, low
   * level APIs to manage properties, aggregations, associations, and events. These generic methods are solely
   * intended for implementing higher level, non-generic methods that manage a single managed property etc.
   * (e.g. a function `setSize(value)` that sets a new value for property 'size'). {@link sap.ui.base.ManagedObject.extend }
   * creates default implementations of those higher level APIs for all managed aspects. The implementation
   * of a subclass then can override those default implementations with a more specific implementation, e.g.
   * to implement a side effect when a specific property is set or retrieved. It is therefore important to
   * understand that the generic low-level methods ARE NOT SUITABLE FOR GENERIC ACCESS to the state of a managed
   * object, as that would bypass the overriding higher level methods and their side effects.
   */
  export default abstract class ManagedObject extends EventProvider {
    /**
     * Constructs and initializes a managed object with the given `sId` and settings.
     *
     * If the optional `mSettings` are given, they must be a simple object that defines values for properties,
     * aggregations, associations or events keyed by their name.
     *
     * **Valid Names and Value Ranges:**
     *
     * The property (key) names supported in the object literal are exactly the (case sensitive) names documented
     * in the JSDoc for the properties, aggregations, associations and events of the current class and its base
     * classes. Note that for 0..n aggregations and associations this name usually is the plural name, whereas
     * it is the singular name in case of 0..1 relations.
     *
     * The possible values for a setting depend on its kind:
     * 	 - for simple properties, the value has to match the documented type of the property (no type conversion
     *     occurs)
     * 	 - for 0..1 aggregations, the value has to be an instance of the aggregated type, or an object literal
     *     from which, the default class of the aggregation (or the corresponding aggregation type as fallback)
     *     will be instantiated.
     * 	 - for 0..n aggregations, the value has to be an array of instances of the aggregated type, a single
     *     instance or an object literal from which the default class will be instantiated.
     * 	 - for 0..1 associations, an instance of the associated type or an id (string) is accepted
     * 	 - for 0..n associations, an array of instances of the associated type or of IDs is accepted
     * 	 - for events, either a function (event handler) is accepted or an array of length 2 where the first
     *     element is a function and the 2nd element is an object to invoke the method on; or an array of length
     *     3, where the first element is an arbitrary payload object, the second one is a function and the 3rd one
     *     is an object to invoke the method on; or an array of arrays where each nested array has the 2 or 3 element
     *     structure described before (multiple listeners).
     *
     * Each subclass should document the name and type of its supported settings in its constructor documentation.
     *
     * Example usage:
     * ```javascript
     *
     * new Dialog({
     *    title: "Some title text",            // property of type "string"
     *    showHeader: true,                    // property of type "boolean"
     *    endButton: new Button(...),          // 0..1 aggregation
     *    content: [                           // 0..n aggregation
     *       new Input(...),
     *       new Input(...)
     *    ],
     *    afterClose: function(oEvent) { ... } // event handler function
     * });
     * ```
     *
     *
     * Instead of static values and object instances, data binding expressions can be used, either embedded
     * in a string or as a binding info object as described in {@link #bindProperty} or {@link #bindAggregation}.
     *
     * Example usage:
     * ```javascript
     *
     * new Dialog({
     *    title: "{/title}",       // embedded binding expression, points to a string property in the data model
     *    ...
     *    content: {               // binding info object
     *       path : "/inputItems", // points to a collection in the data model
     *       template : new Input(...)
     *    }
     * });
     * ```
     *
     *
     * Note that when setting string values, any curly braces in those values need to be escaped, so they are
     * not interpreted as binding expressions. Use {@link #escapeSettingsValue} to do so.
     *
     * **Note:** As of version 1.120, providing aggregation content via an object literal is deprecated, in
     * case the object's type is given via the property 'Type' as a string, or is derived via the defined type
     * of the aggregation. Additionally, as of version 1.120, a ManagedObject subclass can specify a `defaultClass`,
     * e.g. for cases where only a single class is valid. Please refer to the {@link sap.ui.base.ManagedObject.MetadataOptions.Aggregation Aggregation }
     * documentation for more details on the `defaultClass`.
     *
     * Besides the settings documented below, ManagedObject itself supports the following special settings:
     *
     * 	 - `id : sap.ui.core.ID` an ID for the new instance. Some subclasses (Element, Component) require
     *     the id to be unique in a specific scope (e.g. an Element Id must be unique across all Elements, a Component
     *     id must be unique across all Components). `models : object` a map of {@link sap.ui.model.Model }
     *     instances keyed by their model name (alias). Each entry with key k in this object has the same
     *     effect as a call `this.setModel(models[k], k);`.
     * 	 - `bindingContexts : object` a map of {@link sap.ui.model.Context} instances keyed by their
     *     model name. Each entry with key k in this object has the same effect as a call `this.setBindingContext(bindingContexts[k],
     *     k);`
     * 	 - `objectBindings : object` a map of binding paths keyed by the corresponding model name. Each
     *     entry with key k in this object has the same effect as a call `this.bindObject(objectBindings[k],
     *     k);` `metadataContexts : object` an array of single binding contexts keyed by the corresponding
     *     model or context name. The purpose of the `metadataContexts` special setting is to deduce as much information
     *     as possible from the binding context of the control in order to be able to predefine certain standard
     *     properties like e.g. visible, enabled, tooltip,...
     *
     * The structure is an array of single contexts, where a single context is a map containing the following
     * keys:
     * 	 - `path: string (mandatory)` The path to the corresponding model property or object, e.g. '/Customers/Name'.
     *     A path can also be relative, e.g. 'Name'
     * 	 - `model: string (optional)` The name of the model, in case there is no name then the undefined
     *     model is taken
     * 	 - `name: string (optional)` A name for the context to used in templating phase
     * 	 - `kind: string (optional)` The kind of the adapter, either `field` for single properties or
     *     `object` for structured contexts. `adapter: string (optional)` The path to an interpretion
     *     class that dilivers control relevant data depending on the context, e.g. enabled, visible. If not supplied
     *     the OData meta data is interpreted.  The syntax for providing the `metadataContexts` is as follows:
     *     `{SINGLE_CONTEXT1},...,{SINGLE_CONTEXTn}` or for simplicity in case there is only one context `{SINGLE_CONTEXT}`.
     *
     * Examples for such metadataContexts are:
     * 	 - `{/Customers/Name}` a single part with an absolute path to the property Name of the Customers
     *     entity set in the default model
     * 	 - `{path: 'Customers/Name', model:'json'}` a single part with an absolute path to the property Name
     *     of the Customers entity set in a named model
     * 	 - `{parts: [{path: 'Customers/Name'},{path: 'editable', model: 'viewModel'}]}` a combination of single
     *     binding contexts, one context from the default model and one from the viewModel
     */
    constructor(
      /**
       * Optional map/JSON-object with initial property values, aggregated objects etc. for the new object
       */
      mSettings?: $ManagedObjectSettings,
      /**
       * Scope object for resolving string based type and formatter references in bindings. When a scope object
       * is given, `mSettings` cannot be omitted, at least `null` or an empty object literal must be given.
       */
      oScope?: object
    );
    /**
     * Constructs and initializes a managed object with the given `sId` and settings.
     *
     * If the optional `mSettings` are given, they must be a simple object that defines values for properties,
     * aggregations, associations or events keyed by their name.
     *
     * **Valid Names and Value Ranges:**
     *
     * The property (key) names supported in the object literal are exactly the (case sensitive) names documented
     * in the JSDoc for the properties, aggregations, associations and events of the current class and its base
     * classes. Note that for 0..n aggregations and associations this name usually is the plural name, whereas
     * it is the singular name in case of 0..1 relations.
     *
     * The possible values for a setting depend on its kind:
     * 	 - for simple properties, the value has to match the documented type of the property (no type conversion
     *     occurs)
     * 	 - for 0..1 aggregations, the value has to be an instance of the aggregated type, or an object literal
     *     from which, the default class of the aggregation (or the corresponding aggregation type as fallback)
     *     will be instantiated.
     * 	 - for 0..n aggregations, the value has to be an array of instances of the aggregated type, a single
     *     instance or an object literal from which the default class will be instantiated.
     * 	 - for 0..1 associations, an instance of the associated type or an id (string) is accepted
     * 	 - for 0..n associations, an array of instances of the associated type or of IDs is accepted
     * 	 - for events, either a function (event handler) is accepted or an array of length 2 where the first
     *     element is a function and the 2nd element is an object to invoke the method on; or an array of length
     *     3, where the first element is an arbitrary payload object, the second one is a function and the 3rd one
     *     is an object to invoke the method on; or an array of arrays where each nested array has the 2 or 3 element
     *     structure described before (multiple listeners).
     *
     * Each subclass should document the name and type of its supported settings in its constructor documentation.
     *
     * Example usage:
     * ```javascript
     *
     * new Dialog({
     *    title: "Some title text",            // property of type "string"
     *    showHeader: true,                    // property of type "boolean"
     *    endButton: new Button(...),          // 0..1 aggregation
     *    content: [                           // 0..n aggregation
     *       new Input(...),
     *       new Input(...)
     *    ],
     *    afterClose: function(oEvent) { ... } // event handler function
     * });
     * ```
     *
     *
     * Instead of static values and object instances, data binding expressions can be used, either embedded
     * in a string or as a binding info object as described in {@link #bindProperty} or {@link #bindAggregation}.
     *
     * Example usage:
     * ```javascript
     *
     * new Dialog({
     *    title: "{/title}",       // embedded binding expression, points to a string property in the data model
     *    ...
     *    content: {               // binding info object
     *       path : "/inputItems", // points to a collection in the data model
     *       template : new Input(...)
     *    }
     * });
     * ```
     *
     *
     * Note that when setting string values, any curly braces in those values need to be escaped, so they are
     * not interpreted as binding expressions. Use {@link #escapeSettingsValue} to do so.
     *
     * **Note:** As of version 1.120, providing aggregation content via an object literal is deprecated, in
     * case the object's type is given via the property 'Type' as a string, or is derived via the defined type
     * of the aggregation. Additionally, as of version 1.120, a ManagedObject subclass can specify a `defaultClass`,
     * e.g. for cases where only a single class is valid. Please refer to the {@link sap.ui.base.ManagedObject.MetadataOptions.Aggregation Aggregation }
     * documentation for more details on the `defaultClass`.
     *
     * Besides the settings documented below, ManagedObject itself supports the following special settings:
     *
     * 	 - `id : sap.ui.core.ID` an ID for the new instance. Some subclasses (Element, Component) require
     *     the id to be unique in a specific scope (e.g. an Element Id must be unique across all Elements, a Component
     *     id must be unique across all Components). `models : object` a map of {@link sap.ui.model.Model }
     *     instances keyed by their model name (alias). Each entry with key k in this object has the same
     *     effect as a call `this.setModel(models[k], k);`.
     * 	 - `bindingContexts : object` a map of {@link sap.ui.model.Context} instances keyed by their
     *     model name. Each entry with key k in this object has the same effect as a call `this.setBindingContext(bindingContexts[k],
     *     k);`
     * 	 - `objectBindings : object` a map of binding paths keyed by the corresponding model name. Each
     *     entry with key k in this object has the same effect as a call `this.bindObject(objectBindings[k],
     *     k);` `metadataContexts : object` an array of single binding contexts keyed by the corresponding
     *     model or context name. The purpose of the `metadataContexts` special setting is to deduce as much information
     *     as possible from the binding context of the control in order to be able to predefine certain standard
     *     properties like e.g. visible, enabled, tooltip,...
     *
     * The structure is an array of single contexts, where a single context is a map containing the following
     * keys:
     * 	 - `path: string (mandatory)` The path to the corresponding model property or object, e.g. '/Customers/Name'.
     *     A path can also be relative, e.g. 'Name'
     * 	 - `model: string (optional)` The name of the model, in case there is no name then the undefined
     *     model is taken
     * 	 - `name: string (optional)` A name for the context to used in templating phase
     * 	 - `kind: string (optional)` The kind of the adapter, either `field` for single properties or
     *     `object` for structured contexts. `adapter: string (optional)` The path to an interpretion
     *     class that dilivers control relevant data depending on the context, e.g. enabled, visible. If not supplied
     *     the OData meta data is interpreted.  The syntax for providing the `metadataContexts` is as follows:
     *     `{SINGLE_CONTEXT1},...,{SINGLE_CONTEXTn}` or for simplicity in case there is only one context `{SINGLE_CONTEXT}`.
     *
     * Examples for such metadataContexts are:
     * 	 - `{/Customers/Name}` a single part with an absolute path to the property Name of the Customers
     *     entity set in the default model
     * 	 - `{path: 'Customers/Name', model:'json'}` a single part with an absolute path to the property Name
     *     of the Customers entity set in a named model
     * 	 - `{parts: [{path: 'Customers/Name'},{path: 'editable', model: 'viewModel'}]}` a combination of single
     *     binding contexts, one context from the default model and one from the viewModel
     */
    constructor(
      /**
       * ID for the new managed object; generated automatically if no non-empty ID is given **Note:** this can
       * be omitted, no matter whether `mSettings` will be given or not!
       */
      sId?: string,
      /**
       * Optional map/JSON-object with initial property values, aggregated objects etc. for the new object
       */
      mSettings?: $ManagedObjectSettings,
      /**
       * Scope object for resolving string based type and formatter references in bindings. When a scope object
       * is given, `mSettings` cannot be omitted, at least `null` or an empty object literal must be given.
       */
      oScope?: object
    );

    /**
     * Escapes the given value so it can be used in the constructor's settings object. Should be used when property
     * values are initialized with static string values which could contain binding characters (curly braces).
     *
     * @since 1.52
     *
     * @returns The given value, escaped for usage as static property value in the constructor's settings object
     * (or unchanged, if not of type string)
     */
    static escapeSettingsValue(
      /**
       * Value to escape; only needs to be done for string values, but the call will work for all types
       */
      vValue: any
    ): any;
    /**
     * Defines a new subclass of ManagedObject with name `sClassName` and enriches it with the information contained
     * in `oClassInfo`.
     *
     * `oClassInfo` can contain the same information that {@link sap.ui.base.Object.extend} already accepts,
     * plus the following new properties in the 'metadata' object literal (see {@link sap.ui.base.ManagedObject.MetadataOptions MetadataOptions }
     * for details on each of them):
     * 	 - `library : string`
     * 	 - `properties : object`
     * 	 - `defaultProperty : string`
     * 	 - `aggregations : object`
     * 	 - `defaultAggregation : string`
     * 	 - `associations : object`
     * 	 - `events : object`
     * 	 - `specialSettings : object` // this one is still experimental and not for public usage!
     *
     * Example:
     * ```javascript
     *
     * ManagedObject.extend('sap.mylib.MyClass', {
     *   metadata : {
     *     library: 'sap.mylib',
     *     properties : {
     *       value: 'string',
     *       width: 'sap.ui.core.CSSSize',
     *       height: { type: 'sap.ui.core.CSSSize', defaultValue: '100%'}
     *       description: { type: 'string', defaultValue: '', selector: '#{id}-desc'}
     *     },
     *     defaultProperty : 'value',
     *     aggregations : {
     *       header : { type: 'sap.mylib.FancyHeader', multiple : false }
     *       items : 'sap.ui.core.Control',
     *       buttons: { type: 'sap.mylib.Button', multiple : true, selector: '#{id} > .sapMLButtonsSection'}
     *     },
     *     defaultAggregation : 'items',
     *     associations : {
     *       initiallyFocused : { type: 'sap.ui.core.Control' }
     *     },
     *     events: {
     *       beforeOpen : {
     *         parameters : {
     *           opener : { type: 'sap.ui.core.Control' }
     *         }
     *       }
     *     },
     *   },
     *
     *   init: function() {
     *   }
     *
     * }); // end of 'extend' call
     * ```
     *
     *
     *
     * @returns The created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class to be created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ManagedObject>,
      /**
       * Constructor function for the metadata object. If not given, it defaults to `sap.ui.base.ManagedObjectMetadata`.
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns the metadata for the ManagedObject class.
     *
     *
     * @returns Metadata for the ManagedObject class.
     */
    static getMetadata(): ManagedObjectMetadata;
    /**
     * Adds some entity `oObject` to the aggregation identified by `sAggregationName`.
     *
     * If the given object is not valid with regard to the aggregation (if it is not an instance of the type
     * specified for that aggregation) or when the method is called for an aggregation of cardinality 0..1,
     * then an Error is thrown (see {@link #validateAggregation}.
     *
     * If the aggregation already has content, the new object will be added after the current content. If the
     * new object was already contained in the aggregation, it will be moved to the end.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically add an object to an aggregation. Use
     * the concrete method addXYZ for aggregation 'XYZ' or the generic {@link #applySettings} instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    addAggregation(
      /**
       * the string identifying the aggregation that `oObject` should be added to.
       */
      sAggregationName: string,
      /**
       * the object to add; if empty, nothing is added
       */
      oObject: ManagedObject,
      /**
       * if true, this ManagedObject as well as the added child are not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Adds some object with the ID `sId` to the association identified by `sAssociationName` and marks this
     * ManagedObject as changed.
     *
     * This method does not avoid duplicates.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically add an object to an association. Use
     * the concrete method addXYZ for association 'XYZ' or the generic {@link #applySettings} instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    addAssociation(
      /**
       * the string identifying the association the object should be added to.
       */
      sAssociationName: string,
      /**
       * the ID of the ManagedObject object to add; if empty, nothing is added; if a `sap.ui.base.ManagedObject`
       * is given, its ID is added
       */
      sId: string | ManagedObject,
      /**
       * if true, this managed object as well as the newly associated object are not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Sets all the properties, aggregations, associations and event handlers as given in the object literal
     * `mSettings`. If a property, aggregation, etc. is not listed in `mSettings`, then its value is not changed
     * by this method.
     *
     * For properties and 0..1 aggregations/associations, any given setting overwrites the current value. For
     * 0..n aggregations, the given values are appended; event listeners are registered in addition to existing
     * ones.
     *
     * For the possible keys and values in `mSettings` see the general documentation in {@link sap.ui.base.ManagedObject }
     * or the specific documentation of the constructor of the concrete managed object class.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    applySettings(
      /**
       * the settings to apply to this managed object
       */
      mSettings: $ManagedObjectSettings,
      /**
       * Scope object to resolve types and formatters
       */
      oScope?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:formatError formatError} event of this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when a new value for a bound property should have been propagated from the model, but formatting
     * the value failed with an exception.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachFormatError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$FormatErrorEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:formatError formatError} event of this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when a new value for a bound property should have been propagated from the model, but formatting
     * the value failed with an exception.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachFormatError(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$FormatErrorEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:modelContextChange modelContextChange} event
     * of this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when models or contexts are changed on this object (either by calling setModel/setBindingContext
     * or due to propagation)
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachModelContextChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:modelContextChange modelContextChange} event
     * of this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when models or contexts are changed on this object (either by calling setModel/setBindingContext
     * or due to propagation)
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachModelContextChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:parseError parseError} event of this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when a new value for a bound property should have been propagated to the model, but parsing the
     * value failed with an exception.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachParseError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$ParseErrorEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:parseError parseError} event of this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when a new value for a bound property should have been propagated to the model, but parsing the
     * value failed with an exception.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachParseError(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$ParseErrorEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationError validationError} event of this
     * `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when a new value for a bound property should have been propagated to the model, but validating
     * the value failed with an exception.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$ValidationErrorEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationError validationError} event of this
     * `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired when a new value for a bound property should have been propagated to the model, but validating
     * the value failed with an exception.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationError(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$ValidationErrorEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationSuccess validationSuccess} event of
     * this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired after a new value for a bound property has been propagated to the model. Only fired, when the binding
     * uses a data type.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationSuccess(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$ValidationSuccessEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationSuccess validationSuccess} event of
     * this `sap.ui.base.ManagedObject`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.base.ManagedObject` itself.
     *
     * Fired after a new value for a bound property has been propagated to the model. Only fired, when the binding
     * uses a data type.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationSuccess(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ManagedObject$ValidationSuccessEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.base.ManagedObject` itself
       */
      oListener?: object
    ): this;
    /**
     * Bind an aggregation to the model.
     *
     * Whenever the corresponding model becomes available or changes (either via a call to {@link #setModel setModel }
     * or propagated from a {@link #getParent parent}), its {@link sap.ui.model.Model#bindList bindList} method
     * will be called to create a new {@link sap.ui.model.ListBinding ListBinding} with the configured binding
     * options.
     *
     * The bound aggregation will use the given template, clone it for each item which exists in the bound list
     * and set the appropriate binding context.
     *
     * This is a generic method which can be used to bind any aggregation to the model. A class may flag aggregations
     * in its metadata with `bindable: "bindable"` to get typed `bindSomething` and `unbindSomething`
     * methods for those aggregations.
     *
     * Also see {@link https://ui5.sap.com/#/topic/91f057786f4d1014b6dd926db0e91070 List Binding (Aggregation Binding) }
     * in the documentation.
     *
     * For more information on the `oBindingInfo.key` property and its usage, see {@link https://ui5.sap.com/#/topic/7cdff73f308b4b10bdf7d83b7aba72e7 Extended Change Detection}.
     *
     * Providing sorters and/or filters as positional parameters is deprecated as of 1.135.0. Provide them as
     * part of a `BindingInfo` object instead.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    bindAggregation(
      /**
       * Name of a public aggregation to bind
       */
      sName: string,
      /**
       * A `BindingInfo` object or just the path, if no further properties are required
       */
      vBindingInfo: AggregationBindingInfo | string,
      /**
       * The template to clone for each item in the aggregation; either a template `Element` or a factory function
       * must be given
       */
      vTemplate?: ManagedObject | ((p1: string, p2: Context) => ManagedObject)
    ): this;
    /**
     * Bind the object to the referenced entity in the model, which is used as the binding context to resolve
     * bound properties or aggregations of the object itself and all of its children relatively to the given
     * path.
     *
     * @deprecated As of version 1.11.1. please use {@link #bindObject} instead.
     *
     * @returns reference to the instance itself
     */
    bindContext(
      /**
       * the binding path
       */
      sPath: string
    ): this;
    /**
     * Bind the object to the referenced entity in the model.
     *
     * The entity is used as the binding context to resolve bound properties or aggregations of the object itself
     * and all of its children relatively to the given path. If a relative binding path is used, it will be
     * evaluated anew whenever the parent context changes.
     *
     * Whenever the corresponding model becomes available or changes (either via a call to {@link #setModel setModel }
     * or propagated from a {@link #getParent parent}), its {@link sap.ui.model.Model#bindContext bindContext }
     * method will be called to create a new {@link sap.ui.model.ContextBinding ContextBinding} with the configured
     * binding options.
     *
     * There's no difference between `bindObject` and {@link sap.ui.core.Element#bindElement bindElement}. Method
     * `bindObject` was introduced together with `ManagedObject` to make context bindings also available on
     * `ManagedObject`s. The new name was chosen to reflect that the binding is not necessarily applied to an
     * `Element`, it also could be applied to a component or some other `ManagedObject`.
     *
     * Also see {@link https://ui5.sap.com/#/topic/91f05e8b6f4d1014b6dd926db0e91070 Context Binding} in the
     * documentation.
     *
     * As of 1.135, providing 'parameters' as positional parameter is deprecated. Provide them as part of a
     * `BindingInfo` object instead.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    bindObject(
      /**
       * A `BindingInfo` object or just the path, if no further properties are required
       */
      vBindingInfo: ObjectBindingInfo | string
    ): this;
    /**
     * Binds a property to the model.
     *
     * Whenever the corresponding model becomes available or changes (either via a call to {@link #setModel setModel }
     * or propagated from a {@link #getParent parent}), its {@link sap.ui.model.Model#bindProperty bindProperty }
     * method will be called to create a new {@link sap.ui.model.PropertyBinding PropertyBinding} with the configured
     * binding options.
     *
     * The Setter for the given property will be called by the binding with the value retrieved from the data
     * model. When the binding mode is `OneTime`, the property will be set only once. When it is `OneWay`, the
     * property will be updated whenever the corresponding data in the model changes. In mode `TwoWay`, changes
     * to the property (not originating in the model) will be reported back to the model (typical use case:
     * user interaction changes the value of a control).
     *
     * This is a generic method which can be used to bind any property to the model. A managed object may flag
     * any property in its metadata with `bindable: "bindable"` to additionally provide named methods to bind
     * and unbind the corresponding property.
     *
     * **Composite Binding**
     *  A composite property binding which combines data from multiple model paths can be declared using the
     * `parts` parameter instead of `path`. The `formatter` function or a {@link sap.ui.model.CompositeType composite type }
     * then can be used to combine the parts, Properties with a composite binding are also known as "calculated
     * fields".
     *
     * Example:
     * ```javascript
     *
     *   oTxt.bindValue({
     *     parts: [
     *       {path: "/firstName", type: "sap.ui.model.type.String"},
     *       {path: "myModel2>/lastName"}
     *     ]
     *   });
     * ```
     *
     *
     * Note that a composite binding will be forced into mode `OneWay` when one of the binding parts is not
     * in mode `TwoWay`.
     *
     * **Static Binding**
     *  A StaticBinding allows to define static values within a `sap.ui.model.CompositeBinding`. It behaves
     * like a property binding but always returns the value that is stored in the binding itself. The binding
     * does not have a `sap.ui.model.Context`, a `sap.ui.model.Model` or a `oBindingInfo.path`. A StaticBinding
     * is created when a `oBindingInfo.value` is passed instead of a `oBindingInfo.path` or `oBindingInfo.parts[i].path`.
     *
     * Also see {@link sap.ui.model.StaticBinding StaticBinding} in the documentation.
     *
     * **Formatter Functions**
     *  When a formatter function is specified for the binding or for a binding part, it will be called with
     * the value of the bound model property. After setting the initial property value, the formatter function
     * will only be called again when the bound model property changes (simple property binding) or when at
     * least one of the bound model properties changes (formatter function of a composite binding). Note that
     * a binding only monitors the bound model data for changes. Dependencies of the formatter implementation
     * to other model data is not known to the binding and changes won't be detected.
     *
     * When the formatter for a property binding (simple or composite) is called, the managed object will be
     * given as `this` context. For formatters of binding parts in a composite binding, this is not the case.
     *
     * Also see {@link https://ui5.sap.com/#/topic/91f0652b6f4d1014b6dd926db0e91070 Property Binding} in the
     * documentation.
     *
     * Providing a type, formatter, or bindingMode as a positional parameter is deprecated as of 1.135.0. Provide
     * them as part of a `BindingInfo` object instead.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    bindProperty(
      /**
       * Name of a public property to bind; public aggregations of cardinality 0..1 that have an alternative,
       * simple type (e.g. "string" or "int") can also be bound with this method
       */
      sName: string,
      /**
       * A `BindingInfo` object or just the path, if no further properties are required
       */
      vBindingInfo: PropertyBindingInfo | string
    ): this;
    /**
     * Clones a tree of objects starting with the object on which clone is called first (root object).
     *
     * The IDs within the newly created clone tree are derived from the original IDs by appending the given
     * `sIdSuffix` (if no suffix is given, one will be created; it will be unique across multiple clone calls).
     *
     * The `oOptions` configuration object can have the following properties:
     * 	 - The boolean value `cloneChildren` specifies whether associations/aggregations will be cloned
     * 	 - The boolean value `cloneBindings` specifies if bindings will be cloned  Note: In case the configuration
     *     `oOptions` is specified, the default values `true` no longer apply, which means in case `cloneChildren`
     *     or `cloneBindings` is not specified, then this ia assumed to be `false` and associations/aggregations
     *     or bindings are not cloned.
     *
     * For each cloned object, the following settings are cloned based on the metadata of the object and the
     * defined options:
     * 	 - All properties that are not bound. If `cloneBindings` is `false`, also the bound properties will
     *     be cloned; in general, values are referenced 1:1, not cloned. For some property types, however, the getters
     *     or setters might clone the value (e.g. array types and properties using metadata option `byValue`)
     * 	 - All aggregated objects that are not bound. If `cloneBindings` is `false`, also the ones that are
     *     bound will be cloned; they are all cloned recursively using the same `sIdSuffix`
     * 	 - All associated controls; when an association points to an object inside the cloned object tree, then
     *     the cloned association will be modified so that it points to the clone of the target object. When the
     *     association points to a managed object outside of the cloned object tree, then its target won't be changed.
     *
     * 	 - All models set via `setModel()`; used by reference.
     * 	 - All property and aggregation bindings (if `cloneBindings` is `true`); the pure binding information
     *     (path, model name) is cloned, but all other information like template control or factory function, data
     *     type or formatter function are copied by reference. The bindings themselves are created anew as they
     *     are specific for the combination (object, property, model). As a result, any later changes to a binding
     *     of the original object are not reflected in the clone, but changes to e.g the type or template etc. are.
     *
     *
     * Each clone is created by first collecting the above mentioned settings and then creating a new instance
     * with the normal constructor function. As a result, any side effects of mutator methods (`setProperty`
     * etc.) or init hooks are repeated during clone creation. There is no need to override `clone()` just to
     * reproduce these internal settings!
     *
     * Custom controls however can override `clone()` to implement additional clone steps. They usually will
     * first call `clone()` on the super class and then modify the returned clone accordingly.
     *
     * Applications **must never provide** the second parameter `aLocaleIds`. It is determined automatically
     * for the root object (and its non-existence also serves as an indicator for the root object). Specifying
     * it will break the implementation of `clone()`.
     *
     *
     * @returns Reference to the newly created clone
     */
    clone(
      /**
       * a suffix to be appended to the cloned object ID
       */
      sIdSuffix?: string,
      /**
       * an array of local IDs within the cloned hierarchy (internally used)
       */
      aLocalIds?: string[],
      /**
       * Configuration object; when omitted, both properties default to `true`; when specified, undefined properties
       * default to `false`
       */
      oOptions?: {
        /**
         * Whether associations and aggregations will be cloned
         */
        cloneChildren?: boolean;
        /**
         * Whether bindings will be cloned
         */
        cloneBindings?: boolean;
      }
    ): this;
    /**
     * Cleans up the resources associated with this object and all its aggregated children.
     *
     * After an object has been destroyed, it can no longer be used!
     *
     * Applications should call this method if they don't need the object any longer.
     */
    destroy(
      /**
       * If `true`, this ManagedObject and all its ancestors won't be invalidated.
       * This flag should be used only during control development to optimize invalidation procedures. It should
       * not be used by any application code.
       */
      bSuppressInvalidate?: boolean
    ): void;
    /**
     * Destroys (all) the managed object(s) in the aggregation named `sAggregationName` and empties the aggregation.
     * If the aggregation did contain any object, this ManagedObject is marked as changed.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically destroy all objects in an aggregation.
     * Use the concrete method destroyXYZ for aggregation 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    destroyAggregation(
      /**
       * the name of the aggregation
       */
      sAggregationName: string,
      /**
       * if true, this ManagedObject is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:formatError formatError} event of this `sap.ui.base.ManagedObject`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachFormatError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ManagedObject$FormatErrorEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:modelContextChange modelContextChange} event
     * of this `sap.ui.base.ManagedObject`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachModelContextChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:parseError parseError} event of this `sap.ui.base.ManagedObject`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachParseError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ManagedObject$ParseErrorEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:validationError validationError} event of
     * this `sap.ui.base.ManagedObject`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachValidationError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ManagedObject$ValidationErrorEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:validationSuccess validationSuccess} event
     * of this `sap.ui.base.ManagedObject`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachValidationSuccess(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ManagedObject$ValidationSuccessEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Searches and returns all aggregated objects that pass the given check function.
     *
     * When the search is done recursively (`bRecursive === true`), it will be executed depth-first and ancestors
     * will be added to the result array before their descendants.
     *
     * If no check function is given, all aggregated objects will pass the check and be added to the result
     * array.
     *
     * When setting `bIncludeBindingTemplates` to `true`, binding templates will be included in the search.
     *
     * **Take care:** this operation might be expensive.
     *
     *
     * @returns Array of aggregated objects that passed the check
     */
    findAggregatedObjects(
      /**
       * Whether the whole aggregation tree should be searched
       */
      bRecursive?: boolean,
      /**
       * Objects for which this function returns a falsy value will not be added to the result array
       */
      fnCondition?: (p1: ManagedObject) => boolean,
      /**
       * Whether binding templates should be included
       */
      bIncludeBindingTemplates?: boolean
    ): ManagedObject[];
    /**
     * Fires event {@link #event:formatError formatError} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireFormatError(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ManagedObject$FormatErrorEventParameters
    ): this;
    /**
     * Fires event {@link #event:modelContextChange modelContextChange} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireModelContextChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:parseError parseError} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireParseError(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ManagedObject$ParseErrorEventParameters
    ): this;
    /**
     * Fires event {@link #event:validationError validationError} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireValidationError(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ManagedObject$ValidationErrorEventParameters
    ): this;
    /**
     * Fires event {@link #event:validationSuccess validationSuccess} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireValidationSuccess(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ManagedObject$ValidationSuccessEventParameters
    ): this;
    /**
     * Returns the aggregated object(s) for the named aggregation of this ManagedObject.
     *
     * If the aggregation does not contain any objects(s), the given `oDefaultForCreation` (or `null`) is set
     * as new value of the aggregation and returned to the caller.
     *
     * **Note:** the need to specify a default value and the fact that it is stored as new value of a so far
     * empty aggregation is recognized as a shortcoming of this API but can no longer be changed for compatibility
     * reasons.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically read the content of an aggregation.
     * Use the concrete method getXYZ for aggregation 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Aggregation array in case of 0..n-aggregations or the managed object or `null` in case of 0..1-aggregations
     */
    getAggregation(
      /**
       * Name of the aggregation
       */
      sAggregationName: string,
      /**
       * Object that is used in case the current aggregation is empty. If provided, it must be null for 0..1 aggregations
       * or an empty array for 0..n aggregations. If not provided, `null` is used.
       *
       * **Note:** When an empty array is given and used because the aggregation was not set before, then this
       * array will be used for the aggregation from thereon. Sharing the same empty array between different calls
       * to this method therefore is not possible and will result in inconsistencies.
       */
      oDefaultForCreation?: ManagedObject | any[]
    ): ManagedObject | ManagedObject[] | null;
    /**
     * Returns the content of the association with the given name.
     *
     * For associations of cardinality 0..1, a single string with the ID of an associated object is returned
     * (if any). For cardinality 0..n, an array with the IDs of the associated objects is returned.
     *
     * If the association does not contain any objects(s), the given `oDefaultForCreation` is set as new value
     * of the association and returned to the caller. The only supported values for `oDefaultForCreation` are
     * `null` and `undefined` in the case of cardinality 0..1 and `null`, `undefined` or an empty array (`[]`)
     * in case of cardinality 0..n. If the argument is omitted, `null` is used independently from the cardinality.
     *
     * **Note:** the need to specify a default value and the fact that it is stored as new value of a so far
     * empty association is recognized as a shortcoming of this API but can no longer be changed for compatibility
     * reasons.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically retrieve the content of an association.
     * Use the concrete method getXYZ for association 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the ID of the associated managed object or an array of such IDs; may be null if the association
     * has not been populated
     */
    getAssociation(
      /**
       * the name of the association
       */
      sAssociationName: string,
      /**
       * the value that is used in case the current aggregation is empty (only null or empty array is allowed)
       */
      oDefaultForCreation: null | any[]
    ): string | string[] | null;
    /**
     * Get the binding object for a specific aggregation/property.
     *
     *
     * @returns the binding for the given name
     */
    getBinding(
      /**
       * the name of the property or aggregation
       */
      sName: string
    ): Binding | undefined;
    /**
     * Get the binding context of this object for the given model name.
     *
     * If the object does not have a binding context set on itself and has no own model set, it will use the
     * first binding context defined in its parent hierarchy.
     *
     * **Note:** to be compatible with future versions of this API, you must not use the following model names:
     *
     * 	 - `null`
     * 	 - empty string `""`
     * 	 - string literals `"null"` or `"undefined"`  Omitting the model name (or using the value `undefined`)
     *     is explicitly allowed and refers to the default model.
     *
     * **Note:** A ManagedObject inherits binding contexts from the Core only when it is a descendant of a UIArea.
     *
     *
     * @returns The binding context of this object
     */
    getBindingContext(
      /**
       * the name of the model or `undefined`
       */
      sModelName?: string
    ): Context | null | undefined;
    /**
     * Returns the binding info for the given property or aggregation.
     *
     * The binding info contains information about path, binding object, format options, sorter, filter etc.
     * for the property or aggregation. As the binding object is only created when the model becomes available,
     * the `binding` property may be undefined.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns A binding info object, containing at least a `path` or `parts` property and, depending on the
     * binding type, additional properties
     */
    getBindingInfo(
      /**
       * Name of the property or aggregation
       */
      sName: string
    ): PropertyBindingInfo | AggregationBindingInfo;
    /**
     * Get the binding path for a specific aggregation/property.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the binding path for the given name
     */
    getBindingPath(
      /**
       * the name of the property or aggregation
       */
      sName: string
    ): string | undefined;
    /**
     * Returns the parent managed object as new eventing parent to enable control event bubbling or `null` if
     * this object hasn't been added to a parent yet.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the parent event provider
     */
    getEventingParent(): EventProvider | null;
    /**
     * Returns the object's ID.
     *
     * There is no guarantee or check or requirement for the ID of a `ManagedObject` to be unique. Only some
     * subclasses of `ManagedObject` introduce this as a requirement, e.g. `Component` or `Element`. All elements
     * existing in the same window at the same time must have different IDs. A new element will fail during
     * construction when the given ID is already used by another element. But there might be a component with
     * the same ID as an element or another `ManagedObject`.
     *
     * For the same reason, there is no general lookup for `ManagedObject`s via their ID. Only for subclasses
     * that enforce unique IDs, there might be lookup mechanisms (e.g. {@link sap.ui.core.Element#getElementById sap.ui.core.Element.getElementById }
     * for elements).
     *
     *
     * @returns The objects's ID.
     */
    getId(): string;
    /**
     * Returns the metadata for the class that this object belongs to.
     *
     *
     * @returns Metadata for the class of the object
     */
    getMetadata(): ManagedObjectMetadata;
    /**
     * Get the model to be used for data bindings with the given model name. If the object does not have a model
     * set on itself, it will use the first model defined in its parent hierarchy.
     *
     * The name can be omitted to reference the default model or it must be a non-empty string.
     *
     * **Note:** to be compatible with future versions of this API, you must not use the following model names:
     *
     * 	 - `null`
     * 	 - empty string `""`
     * 	 - string literals `"null"` or `"undefined"`  Omitting the model name (or using the value `undefined`)
     *     is explicitly allowed and refers to the default model.
     *
     *
     * @returns oModel or undefined when there is no such model
     */
    getModel(
      /**
       * name of the model to be retrieved
       */
      sModelName?: string
    ): Model | undefined;
    /**
     * Get the object binding object for a specific model.
     *
     * **Note:** to be compatible with future versions of this API, you must not use the following model names:
     *
     * 	 - `null`
     * 	 - empty string `""`
     * 	 - string literals `"null"` or `"undefined"`  Omitting the model name (or using the value `undefined`)
     *     is explicitly allowed and refers to the default model.
     *
     *
     * @returns Context binding for the given model name or `undefined`
     */
    getObjectBinding(
      /**
       * Non-empty name of the model or `undefined`
       */
      sModelName?: string
    ): ContextBinding | undefined;
    /**
     * Returns the origin info for the value of the given property.
     *
     * The origin info might contain additional information for translatable texts. The bookkeeping of this
     * information is not active by default and must be activated by configuration. Even then, it might not
     * be present for all properties and their values depending on where the value came form.
     *
     * If no origin info is available, `null` will be returned.
     *
     *
     * @returns |null} An object describing the origin of this property's value or `null`
     */
    getOriginInfo(
      /**
       * Name of the property
       */
      sPropertyName: string
    ): {
      source: string;

      locale: string;
    };
    /**
     * Returns a map of all models assigned to this ManagedObject.
     *
     * The default model is available on key `undefined`.
     *
     * **Note:** Models propagated from the parent are not included.
     *
     * @since 1.88.0
     *
     * @returns The models
     */
    getOwnModels(): Record<string, Model>;
    /**
     * Returns the parent managed object or `null` if this object hasn't been added to a parent yet.
     *
     * The parent returned by this method is the technical parent used for data binding, invalidation, rendering
     * etc. It might differ from the object on which the application originally added this object (the so called
     * 'API parent'): some composite controls internally use hidden controls or containers to store their children.
     * This method will return the innermost container that technically contains this object as a child.
     *
     * **Example:**
     *
     * Assume that a `Dialog` internally uses a (hidden) `VerticalLayout` to store its content:
     *
     *
     * ```javascript
     *
     *   Dialog (API parent)
     *    \__ VerticalLayout (hidden composite part)
     *       \__ Text (API child)
     * ```
     *
     *
     * If you add some content by calling the `Dialog.prototype.addContent` API, this will lead to the following
     * observations:
     *
     *
     * ```javascript
     *
     *   oDialog.addContent(oText);
     *   console.log(oText.getParent() === oDialog);  // false
     *   console.log(oText.getParent() instanceof VerticalLayout); // true
     *   console.log(oText.getParent().getParent() === oDialog); // true now, but might fail with later versions
     * ```
     *
     *
     * Technically, from API perspective, `oText` is added as a child to `Dialog`. But internally, the `Dialog`
     * adds the child to the hidden `VerticalLayout` container. If you now call the `getParent` method of the
     * child, you will get the internal `VerticalLayout` object and not the `Dialog` API parent.
     *
     * **Note: ** The internal (hidden) structure of a composite control is not fixed and may be changed (see
     * also our "Compatibility Rules"). Therefore, you should **never** rely on a specific structure or object
     * being returned by `getParent`.
     *
     * **Note: ** There is no API to determine the original API parent.
     *
     *
     * @returns The technical parent managed object or `null`
     */
    getParent(): ManagedObject | null;
    /**
     * Returns the value for the property with the given `sPropertyName`.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically retrieve the value of a property.
     * Use the concrete method getXYZ for property 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the value of the property
     */
    getProperty(
      /**
       * the name of the property
       */
      sPropertyName: string
    ): any;
    /**
     * Check if any model is set to the ManagedObject or to one of its parents (including UIArea and Core).
     *
     * **Note:** A ManagedObject inherits models from the Core only when it is a descendant of a UIArea.
     *
     *
     * @returns whether a model reference exists or not
     */
    hasModel(): boolean;
    /**
     * Searches for the provided ManagedObject in the named aggregation and returns its 0-based index if found,
     * or -1 otherwise. Returns -2 if the given named aggregation is of cardinality 0..1 and doesn't reference
     * the given object.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically determine the position of an object
     * in an aggregation. Use the concrete method indexOfXYZ for aggregation 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the index of the provided managed object in the aggregation.
     */
    indexOfAggregation(
      /**
       * the name of the aggregation
       */
      sAggregationName: string,
      /**
       * the ManagedObject whose index is looked for.
       */
      oObject: ManagedObject
    ): int;
    /**
     * Inserts managed object `oObject` to the aggregation named `sAggregationName` at position `iIndex`.
     *
     * If the given object is not valid with regard to the aggregation (if it is not an instance of the type
     * specified for that aggregation) or when the method is called for an aggregation of cardinality 0..1,
     * then an Error is thrown (see {@link #validateAggregation}.
     *
     * If the given index is out of range with respect to the current content of the aggregation, it is clipped
     * to that range (0 for iIndex < 0, n for iIndex > n).
     *
     * Please note that this method does not work as expected when an object is added that is already part of
     * the aggregation. In order to change the index of an object inside an aggregation, first remove it, then
     * insert it again.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically insert an object into an aggregation.
     * Use the concrete method insertXYZ for aggregation 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    insertAggregation(
      /**
       * the string identifying the aggregation the managed object `oObject` should be inserted into.
       */
      sAggregationName: string,
      /**
       * the ManagedObject to add; if empty, nothing is inserted.
       */
      oObject: ManagedObject,
      /**
       * the `0`-based index the managed object should be inserted at; for a negative value `iIndex`, `oObject`
       * is inserted at position 0; for a value greater than the current size of the aggregation, `oObject` is
       * inserted at the last position
       */
      iIndex: int,
      /**
       * if true, this ManagedObject as well as the added child are not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Marks this object and its aggregated children as 'invalid'.
     *
     * The term 'invalid' originally was introduced by controls where a change to the object's state made the
     * rendered DOM invalid. Later, the concept of invalidation was moved up in the inheritance hierarchy
     * to `ManagedObject`, but the term was kept for compatibility reasons.
     *
     * Managed settings (properties, aggregations, associations) invalidate the corresponding object automatically.
     * Changing the state via the standard mutators, therefore, does not require an explicit call to `invalidate`.
     * The same applies to changes made via data binding, as it internally uses the standard mutators.
     *
     * By default, a `ManagedObject` propagates any invalidation to its parent, unless the invalidation is suppressed
     * on the parent. Controls or UIAreas handle invalidation on their own by triggering a re-rendering.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    invalidate(): void;
    /**
     * Find out whether a property or aggregation is bound
     *
     *
     * @returns whether a binding exists for the given name
     */
    isBound(
      /**
       * the name of the property or aggregation
       */
      sName: string
    ): boolean;
    /**
     * Returns whether this object is destroyed or not. A destroyed object cannot be used anymore.
     *
     * @since 1.93
     *
     * @returns Whether the object is destroyed
     */
    isDestroyed(): boolean;
    /**
     * Checks if an object's destruction has been started. During the descruction of an object its ID is still
     * registered, and child objects could be still aggregated. Creating another object with the same ID would
     * lead to duplicate ID issues. To check if the destruction is finished, call `isDestroyed`.
     *
     * @since 1.93
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether an object's destruction has been started
     */
    isDestroyStarted(): boolean;
    /**
     * Returns whether re-rendering is currently suppressed on this ManagedObject.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether re-rendering is suppressed
     */
    isInvalidateSuppressed(): boolean;
    /**
     * Returns whether the given property value is initial and has not been explicitly set or bound. Even after
     * setting the default value or setting null/undefined (which also causes the default value to be set),
     * the property is no longer initial. A property can be reset to initial state by calling `resetProperty(sPropertyName)`.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns true if the property is initial
     */
    isPropertyInitial(
      /**
       * the name of the property
       */
      sPropertyName: string
    ): boolean;
    /**
     * This method is used internally and should only be overridden by a tree managed object which utilizes
     * the tree binding. In this case and if the aggregation is a tree node the overridden method should then
     * return true. If true is returned the tree binding will be used instead of the list binding.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns whether tree binding should be used or list binding. Default is false. Override method to change
     * this behavior.
     */
    isTreeBinding(
      /**
       * the aggregation to bind (e.g. nodes for a tree managed object)
       */
      sName: string
    ): boolean;
    /**
     * Generic method which is called, whenever messages for this object exist.
     *
     * @since 1.28
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    propagateMessages(
      /**
       * The property name
       */
      sName: string,
      /**
       * The messages
       */
      aMessages: any[]
    ): void;
    /**
     * Generic method which can be called, when an aggregation needs to be refreshed. This method does not make
     * any change on the aggregation, but just calls the `getContexts` method of the binding to trigger fetching
     * of new data.
     *
     * Subclasses should call this method only in the implementation of a named refresh method and for no other
     * purposes. The framework might change the conditions under which the method is called and the method implementation
     * might rely on those conditions.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    refreshAggregation(
      /**
       * name of the aggregation to refresh
       */
      sName: string
    ): void;
    /**
     * Removes an object from the aggregation named `sAggregationName` with cardinality 0..n.
     *
     * The removed object is not destroyed nor is it marked as changed.
     *
     * If the given object is found in the aggregation, it is removed, it's parent relationship is unset and
     * this ManagedObject is marked as changed. The removed object is returned as result of this method. If
     * the object could not be found, `null` is returned.
     *
     * This method must only be called for aggregations of cardinality 0..n. The only way to remove objects
     * from a 0..1 aggregation is to set a `null` value for them.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically remove an object from an aggregation.
     * Use the concrete method removeXYZ for aggregation 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the removed object or `null`
     */
    removeAggregation(
      /**
       * the string identifying the aggregation that the given object should be removed from
       */
      sAggregationName: string,
      /**
       * the position or ID of the ManagedObject that should be removed or that ManagedObject itself; if `vObject`
       * is invalid, a negative value or a value greater or equal than the current size of the aggregation, nothing
       * is removed.
       */
      vObject: int | string | ManagedObject,
      /**
       * if true, this ManagedObject is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): ManagedObject | null;
    /**
     * Removes all objects from the 0..n-aggregation named `sAggregationName`.
     *
     * The removed objects are not destroyed nor are they marked as changed.
     *
     * Additionally, it clears the parent relationship of all removed objects, marks this ManagedObject as changed
     * and returns an array with the removed objects.
     *
     * If the aggregation did not contain any objects, an empty array is returned and this ManagedObject is
     * not marked as changed.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically remove all objects from an aggregation.
     * Use the concrete method removeAllXYZ for aggregation 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAggregation(
      /**
       * Name of the aggregation to remove all objects from
       */
      sAggregationName: string,
      /**
       * If true, this `ManagedObject` is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): ManagedObject[];
    /**
     * Removes all the objects in the 0..n-association named `sAssociationName` and returns an array with their
     * IDs. This ManagedObject is marked as changed, if the association contained any objects.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically remove all object from an association.
     * Use the concrete method removeAllXYZ for association 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns an array with the IDs of the removed objects (might be empty)
     */
    removeAllAssociation(
      /**
       * the name of the association
       */
      sAssociationName: string,
      /**
       * if true, this ManagedObject is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): any[];
    /**
     * Removes a `ManagedObject` from the association named `sAssociationName`.
     *
     * If an object is removed, the ID of that object is returned and this `ManagedObject` is marked as changed.
     * Otherwise `null` is returned.
     *
     * If the same object was added multiple times to the same association, only a single occurrence of it will
     * be removed by this method. If the object is not found or if the parameter can't be interpreted neither
     * as a `ManagedObject` (or ID) nor as an index in the association, nothing will be removed. The same is
     * true if an index is given and if that index is out of range for the association.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically remove an object from an association.
     * Use the concrete method removeXYZ for association 'XYZ' instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns ID of the removed `ManagedObject` or `null`
     */
    removeAssociation(
      /**
       * the string identifying the association the `ManagedObject` should be removed from.
       */
      sAssociationName: string,
      /**
       * the position or ID of the `ManagedObject` to remove or the `ManagedObject` itself; if `vObject` is invalid
       * input, a negative value or a value greater or equal than the current size of the association, nothing
       * is removed
       */
      vObject: int | string | ManagedObject,
      /**
       * if `true`, the managed object is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): string | null;
    /**
     * Resets the given property to the default value and also restores the "initial" state (like it has never
     * been set).
     *
     * As subclasses might have implemented side effects in the named setter `setXYZ` for property 'xyz', that
     * setter is called with a value of `null`, which by convention restores the default value of the property.
     * This is only done to notify subclasses, the internal state is anyhow reset.
     *
     * When the property has not been modified so far, nothing will be done.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    resetProperty(
      /**
       * Name of the property
       */
      sPropertyName: string
    ): this;
    /**
     * Sets a new object in the named 0..1 aggregation of this ManagedObject and marks this ManagedObject as
     * changed.
     *
     * If the given object is not valid with regard to the aggregation (if it is not an instance of the type
     * specified for that aggregation) or when the method is called for an aggregation of cardinality 0..n,
     * then an Error is thrown (see {@link #validateAggregation}.
     *
     * If the new object is the same as the currently aggregated object, then the internal state is not modified
     * and this ManagedObject is not marked as changed.
     *
     * If the given object is different, the parent of a previously aggregated object is cleared (it must have
     * been this ManagedObject before), the parent of the given object is set to this ManagedObject and {@link #invalidate }
     * is called for this object.
     *
     * Note that this method does neither return nor destroy the previously aggregated object. This behavior
     * is inherited by named set methods (see below) in subclasses. To avoid memory leaks, applications therefore
     * should first get the aggregated object, keep a reference to it or destroy it, depending on their needs,
     * and only then set a new object.
     *
     * Note that ManagedObject only implements a single level of change tracking: if a first call to setAggregation
     * recognizes a change, 'invalidate' is called. If another call to setAggregation reverts that change, invalidate()
     * will be called again, the new status is not recognized as being 'clean' again.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically set an object in an aggregation. Use
     * the concrete method setXYZ for aggregation 'XYZ' or the generic {@link #applySettings} instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    setAggregation(
      /**
       * name of an 0..1 aggregation
       */
      sAggregationName: string,
      /**
       * the managed object that is set as aggregated object
       */
      oObject: ManagedObject,
      /**
       * if true, this ManagedObject is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Sets the associated object for the given managed association of cardinality '0..1' and marks this ManagedObject
     * as changed.
     *
     * The associated object can either be given by itself or by its id. If `null` or `undefined` is given,
     * the association is cleared.
     *
     * **Note:** This method is a low-level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically set an object in an association. Use
     * the concrete method setXYZ for association 'XYZ' or the generic {@link #applySettings} instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    setAssociation(
      /**
       * name of the association
       */
      sAssociationName: string,
      /**
       * the ID of the managed object that is set as an association, or the managed object itself or null
       */
      sId: string | ManagedObject,
      /**
       * if true, the managed objects invalidate method is not called
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Set the binding context for this ManagedObject for the model with the given name.
     *
     * **Note:** to be compatible with future versions of this API, you must not use the following model names:
     *
     * 	 - `null`
     * 	 - empty string `""`
     * 	 - string literals `"null"` or `"undefined"`  Omitting the model name (or using the value `undefined`)
     *     is explicitly allowed and refers to the default model.
     *
     * A value of `null` for `oContext` hides the parent context. The parent context will no longer be propagated
     * to aggregated child controls. A value of `undefined` removes a currently active context or a `null` context
     * and the parent context gets visible and propagated again.
     *
     * **Note:** A ManagedObject inherits binding contexts from the Core only when it is a descendant of a UIArea.
     *
     *
     * @returns reference to the instance itself
     */
    setBindingContext(
      /**
       * the new binding context for this object
       */
      oContext: Context,
      /**
       * the name of the model to set the context for or `undefined`
       */
      sModelName?: string
    ): this;
    /**
     * Sets or unsets a model for the given model name for this ManagedObject.
     *
     * The `sName` must either be `undefined` (or omitted) or a non-empty string. When the name is omitted,
     * the default model is set/unset. To be compatible with future versions of this API, you must not use the
     * following model names:
     * 	 - `null`
     * 	 - empty string `""`
     * 	 - string literals `"null"` or `"undefined"`
     *
     * When `oModel` is `null` or `undefined`, a previously set model with that name is removed from this ManagedObject.
     * If an ancestor (parent, UIArea or Core) has a model with that name, this ManagedObject will immediately
     * inherit that model from its ancestor.
     *
     * All local bindings that depend on the given model name are updated (created if the model references became
     * complete now; updated, if any model reference has changed; removed if the model references became incomplete
     * now).
     *
     * Any change (new model, removed model, inherited model) is also applied to all aggregated descendants
     * as long as a descendant doesn't have its own model set for the given name.
     *
     * **Note:** By design, it is not possible to hide an inherited model by setting a `null` or `undefined`
     * model. Applications can set an empty model to achieve the same.
     *
     * **Note:** A ManagedObject inherits models from the Core only when it is a descendant of a UIArea.
     *
     *
     * @returns `this` to allow method chaining
     */
    setModel(
      /**
       * Model to be set or `null` or `undefined`
       */
      oModel: Model | null | undefined,
      /**
       * the name of the model or `undefined`
       */
      sName?: string
    ): this;
    /**
     * Sets the given value for the given property after validating and normalizing it, marks this object as
     * changed.
     *
     * If the value is not valid with regard to the declared data type of the property, an Error is thrown.
     * In case `null` or `undefined` is passed, the default value for this property is used (see {@link #validateProperty}).
     * To fully reset the property to initial state, use {@link #resetProperty} instead. If the validated and
     * normalized `oValue` equals the current value of the property, the internal state of this object is not
     * changed (apart from the result of {@link #isPropertyInitial}). If the value changes, it is stored internally
     * and the {@link #invalidate} method is called on this object. In the case of TwoWay databinding, the bound
     * model is informed about the property change.
     *
     * Note that ManagedObject only implements a single level of change tracking: if a first call to setProperty
     * recognizes a change, 'invalidate' is called. If another call to setProperty reverts that change, invalidate()
     * will be called again, the new status is not recognized as being 'clean' again.
     *
     * **Note:** This method is a low level API as described in the class documentation.
     * Applications or frameworks must not use this method to generically set a property. Use the concrete method
     * setXYZ for property 'XYZ' or the generic {@link #applySettings} instead.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Returns `this` to allow method chaining
     */
    setProperty(
      /**
       * name of the property to set
       */
      sPropertyName: string,
      /**
       * value to set the property to
       */
      oValue: any,
      /**
       * if true, the managed object is not marked as changed
       */
      bSuppressInvalidate?: boolean
    ): this;
    /**
     * Returns a simple string representation of this managed object.
     *
     * Mainly useful for tracing purposes.
     *
     *
     * @returns a string description of this managed object
     */
    toString(): string;
    /**
     * Unbind the aggregation from the model.
     *
     * After unbinding, the current content of the aggregation is destroyed by default. When the `bSuppressReset`
     * parameter is set, it is however retained.
     *
     *
     * @returns Reference to this instance itself
     */
    unbindAggregation(
      /**
       * Name of the aggregation
       */
      sName: string,
      /**
       * Indicates whether destroying the content of the aggregation is skipped
       */
      bSuppressReset: boolean
    ): this;
    /**
     * Removes the defined binding context of this object, all bindings will now resolve relative to the parent
     * context again.
     *
     * @deprecated As of version 1.11.1. please use {@link #unbindObject} instead.
     *
     * @returns reference to the instance itself
     */
    unbindContext(
      /**
       * name of the model to remove the context for.
       */
      sModelName?: string
    ): this;
    /**
     * Removes the defined binding context of this object, all bindings will now resolve relative to the parent
     * context again.
     *
     *
     * @returns Reference to the instance itself
     */
    unbindObject(
      /**
       * Name of the model to remove the context for.
       */
      sModelName?: string
    ): this;
    /**
     * Unbind the property from the model
     *
     *
     * @returns reference to the instance itself
     */
    unbindProperty(
      /**
       * the name of the property
       */
      sName: string,
      /**
       * whether the reset to the default value when unbinding should be suppressed
       */
      bSuppressReset: boolean
    ): this;
    /**
     * Generic method which is called whenever an aggregation binding has changed.
     *
     * Depending on the type of the list binding and on additional configuration, this method either destroys
     * all elements in the aggregation `sName` and recreates them anew or tries to reuse as many existing objects
     * as possible. It is up to the method which strategy it uses.
     *
     * In case a managed object needs special handling for an aggregation binding, it can create a named update
     * method (e.g. `updateRows` for an aggregation `rows`) which then will be called by the framework
     * instead of this generic method. THe method will be called with two arguments `sChangeReason` and `oEventInfo`.
     *
     * Subclasses should call this method only in the implementation of such a named update method and for no
     * other purposes. The framework might change the conditions under which the method is called and the method
     * implementation might rely on those conditions.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    updateAggregation(
      /**
       * Name of the aggregation to update
       */
      sName: string,
      /**
       * One of the predefined reasons for the change event
       */
      sChangeReason: ChangeReason,
      /**
       * Additional information about the change event
       */
      oEventInfo: {
        /**
         * A non-standardized string that further classifies the change event. Model implementations should document
         * any value that they might provide as detailed reason, and describe under what circumstances each value
         * will be used.
         */
        detailedReason?: string;
      }
    ): void;
    /**
     * Checks whether the given value is of the proper type for the given aggregation name.
     *
     * This method is already called by {@link #setAggregation}, {@link #addAggregation} and {@link #insertAggregation}.
     * In many cases, subclasses of ManagedObject don't need to call it again in their mutator methods.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns the passed object
     */
    validateAggregation(
      /**
       * the name of the aggregation
       */
      sAggregationName: string,
      /**
       * the aggregated object or a primitive value
       */
      oObject: ManagedObject | any,
      /**
       * whether the caller assumes the aggregation to have cardinality 0..n
       */
      bMultiple: boolean
    ): ManagedObject | any;
    /**
     * Checks whether the given value is of the proper type for the given property name.
     *
     * In case `null` or `undefined` is passed, the default value for this property is used as value. If no
     * default value is defined for the property, the default value of the type of the property is used.
     *
     * If the property has a data type that is an instance of sap.ui.base.DataType and if a `normalize` function
     * is defined for that type, that function will be called with the resulting value as only argument. The
     * result of the function call is then used instead of the raw value.
     *
     * This method is called by {@link #setProperty}. In many cases, subclasses of ManagedObject don't need
     * to call it themselves.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The normalized value for the passed value or for the default value if `null` or `undefined`
     * was passed
     */
    validateProperty(
      /**
       * Name of the property
       */
      sPropertyName: string,
      /**
       * Value to be set
       */
      oValue: any
    ): any;
  }
  /**
   * Configuration for the binding of a managed aggregation of cardinality 0..n.
   *
   * `path` is the only mandatory property, all others are optional.
   */
  export type AggregationBindingInfo = {
    /**
     * Path in the model to bind to, either an absolute path or relative to the binding context for the corresponding
     * model; when the path contains a '>' sign, the string preceding it will override the `model` property
     * and the remainder after the '>' will be used as binding path
     */
    path: string;
    /**
     * Name of the model to bind against; when `undefined` or omitted, the default model is used
     */
    model?: string;
    /**
     * The template to clone for each item in the aggregation; either a template or a factory must be given
     */
    template?: ManagedObject;
    /**
     * Whether the framework should assume that the application takes care of the lifecycle of the given template;
     * when set to `true`, the template can be used in multiple bindings, either in parallel or over time, and
     * the framework won't clone it when this `ManagedObject` is cloned; when set to `false`, the lifecycle
     * of the template is bound to the lifecycle of the binding, when the aggregation is unbound or when this
     * `ManagedObject` is destroyed, the template also will be destroyed, and when this `ManagedObject` is cloned,
     * the template will be cloned as well; the third option (`undefined`) only exists for compatibility reasons,
     * its behavior is not fully reliable and it may leak the template
     */
    templateShareable?: boolean;
    /**
     * A factory function that will be called to create an object for each item in the aggregation; this is
     * an alternative to providing a template object and can be used when the objects should differ depending
     * on the binding context; the factory function will be called with two parameters: an ID that should be
     * used for the created object and the binding context for which the object has to be created; the function
     * must return an object appropriate for the bound aggregation
     */
    factory?: (p1: string, p2: Context) => ManagedObject;
    /**
     * Whether the binding should be suspended initially
     */
    suspended?: boolean;
    /**
     * the first entry of the list to be created
     */
    startIndex?: int;
    /**
     * The amount of entries to be created (may exceed the size limit of the model)
     */
    length?: int;
    /**
     * The initial sort order (optional)
     */
    sorter?: Sorter | Sorter[];
    /**
     * The predefined filters for this aggregation (optional)
     */
    filters?: Filter | Filter[];
    /**
     * Name of the key property or a function getting the context as only parameter to calculate a key for entries.
     * This can be used to improve update behaviour in models, where a key is not already available.
     */
    key?: string | ((p1: Context) => string);
    /**
     * Map of additional parameters for this binding; the names and value ranges of the supported parameters
     * depend on the model implementation, they should be documented with the `bindList` method of the corresponding
     * model class or with the model specific subclass of `sap.ui.model.ListBinding`
     */
    parameters?: object;
    /**
     * A factory function to generate custom group visualization (optional). It should return a control suitable
     * to visualize a group header (e.g. a `sap.m.GroupHeaderListItem` for a `sap.m.List`).
     */
    groupHeaderFactory?: (p1: { key: string }) => ManagedObject;
    /**
     * Map of event handler functions keyed by the name of the binding events that they should be attached to.
     * The names and value ranges of the supported events depend on the model implementation and should be documented
     * with the model-specific subclass of `sap.ui.model.ListBinding`.
     */
    events?: Record<string, Function>;
  };

  /**
   * The structure of the "metadata" object which is passed when inheriting from sap.ui.base.ManagedObject
   * using its static "extend" method. See {@link sap.ui.base.ManagedObject.extend} for details on its usage.
   */
  export type MetadataOptions = MetadataOptions1 & {
    /**
     * Name of the library that the new subclass should belong to. If the subclass is a control or element,
     * it will automatically register with that library so that authoring tools can discover it. By convention,
     * the name of the subclass should have the library name as a prefix, but subfolders are allowed, e.g. `sap.ui.layout.form.Form`
     * belongs to library `sap.ui.layout`.
     */
    library?: string;
    /**
     * An object literal whose properties each define a new managed property in the ManagedObject subclass.
     * The value can either be a simple string which then will be assumed to be the type of the new property
     * or it can be an object literal with the following properties (see {@link sap.ui.base.ManagedObject.MetadataOptions.Property Property }
     * for details): type, visibility, byValue, group, defaultValue, bindable, selector Property names should
     * use camelCase notation, start with a lowercase letter and only use characters from the set [a-zA-Z0-9_$].
     * If an aggregation in the literal is preceded by a JSDoc comment (doclet) and if the UI5 plugin and template
     * are used for JSDoc3 generation, the doclet will be used as generic documentation of the aggregation.
     *
     * For each public property 'foo', the following methods will be created by the "extend" method and will
     * be added to the prototype of the subclass:
     * 	 - getFoo() - returns the current value of property 'foo'. Internally calls {@link #getProperty}
     * 	 - setFoo(v) - sets 'v' as the new value of property 'foo'. Internally calls {@link #setProperty}
     * 	 - bindFoo(c) - (only if property was defined to be 'bindable'): convenience function that wraps {@link #bindProperty }
     *
     * 	 - unbindFoo() - (only if property was defined to be 'bindable'): convenience function that wraps {@link #unbindProperty }
     *      For hidden properties, no methods are generated.
     */
    properties?: Record<string, string | MetadataOptions.Property>;
    /**
     * When specified, the default property must match the name of one of the properties defined for the new
     * subclass (either own or inherited). The named property can be used to identify the main property to be
     * used for bound data. E.g. the value property of a field control.
     */
    defaultProperty?: string;
    /**
     * An object literal whose properties each define a new aggregation in the ManagedObject subclass. The value
     * can either be a simple string which then will be assumed to be the type of the new aggregation or it
     * can be an object literal with the following properties (see {@link sap.ui.base.ManagedObject.MetadataOptions.Aggregation Aggregation }
     * for details): type, multiple, singularName, visibility, bindable, forwarding, selector. Aggregation names
     * should use camelCase notation, start with a lowercase letter and only use characters from the set [a-zA-Z0-9_$].
     * The name for a hidden aggregations might start with an underscore. If an aggregation in the literal is
     * preceded by a JSDoc comment (doclet) and if the UI5 plugin and template are used for JSDoc3 generation,
     * the doclet will be used as generic documentation of the aggregation.
     *
     * For each public aggregation 'item' of cardinality 0..1, the following methods will be created by the
     * "extend" method and will be added to the prototype of the subclass:
     * 	 - getItem() - returns the current value of aggregation 'item'. Internally calls {@link #getAggregation }
     *     with a default value of `undefined`
     * 	 - setItem(o) - sets 'o' as the new aggregated object in aggregation 'item'. Internally calls {@link #setAggregation }
     *
     * 	 - destroyItem(o) - destroy a currently aggregated object in aggregation 'item' and clears the aggregation.
     *     Internally calls {@link #destroyAggregation}
     * 	 - bindItem(c) - (only if aggregation was defined to be 'bindable'): convenience function that wraps
     *     {@link #bindAggregation}
     * 	 - unbindItem() - (only if aggregation was defined to be 'bindable'): convenience function that wraps
     *     {@link #unbindAggregation}  For a public aggregation 'items' of cardinality 0..n, the following
     *     methods will be created:
     * 	 - getItems() - returns an array with the objects contained in aggregation 'items'. Internally calls
     *     {@link #getAggregation} with a default value of `[]`
     * 	 - addItem(o) - adds an object as last element in the aggregation 'items'. Internally calls {@link #addAggregation }
     *
     * 	 - insertItem(o,p) - inserts an object into the aggregation 'items'. Internally calls {@link #insertAggregation }
     *
     * 	 - indexOfItem(o) - returns the position of the given object within the aggregation 'items'. Internally
     *     calls {@link #indexOfAggregation}
     * 	 - removeItem(v) - removes an object from the aggregation 'items'. Internally calls {@link #removeAggregation }
     *
     * 	 - removeAllItems() - removes all objects from the aggregation 'items'. Internally calls {@link #removeAllAggregation }
     *
     * 	 - destroyItems() - destroy all currently aggregated objects in aggregation 'items' and clears the aggregation.
     *     Internally calls {@link #destroyAggregation}
     * 	 - bindItems(c) - (only if aggregation was defined to be 'bindable'): convenience function that wraps
     *     {@link #bindAggregation}
     * 	 - unbindItems() - (only if aggregation was defined to be 'bindable'): convenience function that wraps
     *     {@link #unbindAggregation}  For hidden aggregations, no methods are generated.
     */
    aggregations?: Record<string, string | MetadataOptions.Aggregation>;
    /**
     * When specified, the default aggregation must match the name of one of the aggregations defined for the
     * new subclass (either own or inherited). The named aggregation will be used in contexts where no aggregation
     * is specified. E,g. when an object in an XMLView embeds other objects without naming an aggregation, as
     * in the following example:
     * ```javascript
     *
     *      <!-- assuming the defaultAggregation for Dialog is 'content' -->
     *      <Dialog>
     *        <Text/>
     *        <Button/>
     *      </Dialog>
     *     ```
     */
    defaultAggregation?: string;
    /**
     * An object literal whose properties each define a new association of the ManagedObject subclass. The value
     * can either be a simple string which then will be assumed to be the type of the new association or it
     * can be an object literal with the following properties (see {@link sap.ui.base.ManagedObject.MetadataOptions.Association Association }
     * for details): type, multiple, singularName, visibility Association names should use camelCase notation,
     * start with a lowercase letter and only use characters from the set [a-zA-Z0-9_$]. If an association in
     * the literal is preceded by a JSDoc comment (doclet) and if the UI5 plugin and template are used for JSDoc3
     * generation, the doclet will be used as generic documentation of the association.
     *
     * For each association 'ref' of cardinality 0..1, the following methods will be created by the "extend"
     * method and will be added to the prototype of the subclass:
     * 	 - getRef() - returns the current value of association 'item'. Internally calls {@link #getAssociation }
     *     with a default value of `undefined`
     * 	 - setRef(o) - sets 'o' as the new associated object in association 'item'. Internally calls {@link #setAssociation }
     *      For a public association 'refs' of cardinality 0..n, the following methods will be created:
     *
     * 	 - getRefs() - returns an array with the objects contained in association 'items'. Internally calls
     *     {@link #getAssociation} with a default value of `[]`
     * 	 - addRef(o) - adds an object as last element in the association 'items'. Internally calls {@link #addAssociation }
     *
     * 	 - removeRef(v) - removes an object from the association 'items'. Internally calls {@link #removeAssociation }
     *
     * 	 - removeAllRefs() - removes all objects from the association 'items'. Internally calls {@link #removeAllAssociation }
     *      For hidden associations, no methods are generated.
     */
    associations?: Record<string, string | MetadataOptions.Association>;
    /**
     * An object literal whose properties each define a new event of the ManagedObject subclass. In this literal,
     * the property names are used as event names and the values are object literals describing the respective
     * event which can have the following properties (see {@link sap.ui.base.ManagedObject.MetadataOptions.Event Event }
     * for details): allowPreventDefault, parameters Event names should use camelCase notation, start with a
     * lower-case letter and only use characters from the set [a-zA-Z0-9_$]. If an event in the literal is preceded
     * by a JSDoc comment (doclet) and if the UI5 plugin and template are used for JSDoc3 generation, the doclet
     * will be used as generic documentation of the event.
     *
     * For each event 'Some' the following methods will be created by the "extend" method and will be added
     * to the prototype of the subclass:
     * 	 - attachSome(fn,o) - registers a listener for the event. Internally calls {@link #attachEvent}
     * 	 - detachSome(fn,o) - deregisters a listener for the event. Internally calls {@link #detachEvent}
     * 	 - fireSome() - fire the event. Internally calls {@link #fireEvent}
     */
    events?: Record<string, string | MetadataOptions.Event>;
    /**
     * Name of a module that implements the designtime part. Alternatively `true` to indicate that the module's
     * file is named *.designtime.js with the same base name as the class itself.
     */
    designtime?: string | boolean;
    /**
     * Special settings are an experimental feature and MUST NOT BE DEFINED in controls or applications outside
     * of the `sap.ui.core` library. There's no generic or general way how to set or get the values for special
     * settings. For the same reason, they cannot be bound against a model. If there's a way for consumers to
     * define a value for a special setting, it must be documented in the class that introduces the setting.
     */
    specialSettings?: Record<string, any>;
  };

  /**
   * Configuration for the binding of a managed object
   *
   * `path` is the only mandatory property, all others are optional.
   */
  export type ObjectBindingInfo = {
    /**
     * Path in the model to bind to, either an absolute path or relative to the binding context for the corresponding
     * model. If the path contains a '>' sign, the string preceding it will override the `model` property,
     * and the remainder after the '>' sign will be used as binding path
     */
    path: string;
    /**
     * Name of the model to bind against; when `undefined` or omitted, the default model is used
     */
    model?: string;
    /**
     * Whether the binding is initially suspended
     */
    suspended?: boolean;
    /**
     * Map of additional parameters for this binding; the names and value ranges of the supported parameters
     * depend on the model implementation and should be documented with the `bindContext` method of the corresponding
     * model class or with the model-specific subclass of `sap.ui.model.ContextBinding`
     */
    parameters?: object;
    /**
     * Map of event handler functions keyed by the name of the binding events that they are attached to. The
     * names and value ranges of the supported events depend on the model implementation and should be documented
     * with the model-specific subclass of `sap.ui.model.ContextBinding`.
     */
    events?: Record<string, Function>;
  };

  /**
   * Configuration for the binding of a managed property.
   *
   * Exactly one of `path`, `value` or `parts` must be specified. The same configuration can be provided for
   * the parts of a composite binding, but nesting composite bindings in composite bindings is not yet supported.
   *
   * Aggregations with cardinality 0..1 that have a simple, alternative type (aka `altType`), can be bound
   * with the same kind of configuration, e.g. the `tooltip` aggregation of elements.
   */
  export type PropertyBindingInfo = {
    /**
     * Path in the model to bind to, either an absolute path or relative to the binding context for the corresponding
     * model; when the path contains a '>' sign, the string preceding it will override the `model` property
     * and the remainder after the '>' will be used as binding path
     */
    path?: string;
    /**
     * Since 1.61, defines a static binding with the given value.
     */
    value?: string;
    /**
     * Name of the model to bind against; when `undefined` or omitted, the default model is used
     */
    model?: string;
    /**
     * Whether the binding should be suspended initially
     */
    suspended?: boolean;
    /**
     * Function to convert model data into a property value
     */
    formatter?: Function;
    /**
     * Whether the parameters to the formatter function should be passed as raw values. In this case the specified
     * types for the binding parts are not used and the values are not formatted.
     *
     * **Note**: use this flag only when using multiple bindings. If you use only one binding and want raw values
     * then simply don't specify a type for that binding.
     */
    useRawValues?: boolean;
    /**
     * Whether the parameters to the formatter function should be passed as the related JavaScript primitive
     * values. In this case the values of the model are parsed by the {@link sap.ui.model.SimpleType#getModelFormat model format }
     * of the specified types from the binding parts.
     *
     * **Note**: use this flag only when using multiple bindings.
     */
    useInternalValues?: boolean;
    /**
     * A type object, or the name or constructor of a type class which is used to create a type object.
     *
     * The type will be used for converting model data to a property value (aka "formatting") and vice versa
     * (in binding mode `TwoWay`, aka "parsing")
     */
    type?: Type | string | (new (p1?: object, p2?: object) => Type);
    /**
     * Format options to be used when creating the type instance; the structure of the options depends on the
     * given type class. If a type object is given as `type`, it won't be modified - the formatOptions
     * will be ignored
     */
    formatOptions?: object;
    /**
     * Additional constraints to be used when constructing a type instance. Their structure depends on the given
     * type class. If a type object is given as `type`, it won't be modified - the `constraints` will be ignored
     */
    constraints?: object;
    /**
     * Target type to be used by the type when formatting model data, for example "boolean" or "string" or "any";
     * defaults to the property's type
     */
    targetType?: string;
    /**
     * Binding mode to be used for this property binding (e.g. one way)
     */
    mode?: BindingMode | keyof typeof BindingMode;
    /**
     * Map of additional parameters for this binding; the names and value ranges of the supported parameters
     * depend on the model implementation, they should be documented with the `bindProperty` method of the corresponding
     * model class or with the model specific subclass of `sap.ui.model.PropertyBinding`
     */
    parameters?: object;
    /**
     * Map of event handler functions keyed by the name of the binding events that they should be attached to
     */
    events?: Record<string, Function>;
    /**
     * Array of binding info objects for the parts of a composite binding; the structure of each binding info
     * is the same as described for the `oBindingInfo` as a whole.
     *
     * If a part is not specified as a binding info object but as a simple string, a binding info object will
     * be created with that string as `path`. The string may start with a model name prefix (see property `path`).
     *
     * **Note**: recursive composite bindings are currently not supported. Therefore, a part must not contain
     * a `parts` property.
     */
    parts?: Array<string | PropertyBindingInfo>;
  };

  /**
   * Describes the settings that can be provided to the ManagedObject constructor.
   */
  export interface $ManagedObjectSettings {
    /**
     * Unique ID of this instance. If not given, a so called autoID will be generated by the framework. AutoIDs
     * use a unique prefix that must not be used for Ids that the application (or other code) creates. It can
     * be configured option 'autoIDPrefix', see {@link https://ui5.sap.com/#/topic/91f2d03b6f4d1014b6dd926db0e91070 Configuration Options and URL Parameters}.
     */
    id?: ID;

    /**
     * A map of model instances to which the object should be attached. The models are keyed by their model
     * name. For the default model, String(undefined) is expected.
     */
    models?: object;

    /**
     * A map of model instances to which the object should be attached. The models are keyed by their model
     * name. For the default model, String(undefined) is expected.
     */
    bindingContexts?: object;

    /**
     * A map of model instances to which the object should be attached. The models are keyed by their model
     * name. For the default model, String(undefined) is expected.
     */
    objectBindings?: object;

    /**
     * A map of model instances to which the object should be attached. The models are keyed by their model
     * name. For the default model, String(undefined) is expected. The special setting is only for internal
     * use.
     */
    metadataContexts?: object;

    /**
     * Fired after a new value for a bound property has been propagated to the model. Only fired, when the binding
     * uses a data type.
     */
    validationSuccess?: (oEvent: ManagedObject$ValidationSuccessEvent) => void;

    /**
     * Fired when a new value for a bound property should have been propagated to the model, but validating
     * the value failed with an exception.
     */
    validationError?: (oEvent: ManagedObject$ValidationErrorEvent) => void;

    /**
     * Fired when a new value for a bound property should have been propagated to the model, but parsing the
     * value failed with an exception.
     */
    parseError?: (oEvent: ManagedObject$ParseErrorEvent) => void;

    /**
     * Fired when a new value for a bound property should have been propagated from the model, but formatting
     * the value failed with an exception.
     */
    formatError?: (oEvent: ManagedObject$FormatErrorEvent) => void;

    /**
     * Fired when models or contexts are changed on this object (either by calling setModel/setBindingContext
     * or due to propagation)
     */
    modelContextChange?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the ManagedObject#formatError event.
   */
  export interface ManagedObject$FormatErrorEventParameters {
    /**
     * ManagedObject instance whose property should have received the model update.
     */
    element?: ManagedObject;

    /**
     * Name of the property for which the binding should have been updated.
     */
    property?: string;

    /**
     * Data type used in the binding (if any).
     */
    type?: Type;

    /**
     * New value (model representation) as propagated from the model.
     */
    newValue?: any;

    /**
     * Old value (external representation) as previously stored in the ManagedObject.
     */
    oldValue?: any;
  }

  /**
   * Event object of the ManagedObject#formatError event.
   */
  export type ManagedObject$FormatErrorEvent = Event<
    ManagedObject$FormatErrorEventParameters,
    ManagedObject
  >;

  /**
   * Parameters of the ManagedObject#modelContextChange event.
   */
  export interface ManagedObject$ModelContextChangeEventParameters {}

  /**
   * Event object of the ManagedObject#modelContextChange event.
   */
  export type ManagedObject$ModelContextChangeEvent = Event<
    ManagedObject$ModelContextChangeEventParameters,
    ManagedObject
  >;

  /**
   * Parameters of the ManagedObject#parseError event.
   */
  export interface ManagedObject$ParseErrorEventParameters {
    /**
     * ManagedObject instance whose property initiated the model update.
     */
    element?: ManagedObject;

    /**
     * Name of the property for which the bound model property should have been been updated.
     */
    property?: string;

    /**
     * Data type used in the binding.
     */
    type?: Type;

    /**
     * New value (external representation) as parsed by the binding.
     */
    newValue?: any;

    /**
     * Old value (external representation) as previously stored in the ManagedObject.
     */
    oldValue?: any;

    /**
     * Localized message describing the parse error
     */
    message?: string;
  }

  /**
   * Event object of the ManagedObject#parseError event.
   */
  export type ManagedObject$ParseErrorEvent = Event<
    ManagedObject$ParseErrorEventParameters,
    ManagedObject
  >;

  /**
   * Parameters of the ManagedObject#validationError event.
   */
  export interface ManagedObject$ValidationErrorEventParameters {
    /**
     * ManagedObject instance whose property initiated the model update.
     */
    element?: ManagedObject;

    /**
     * Name of the property for which the bound model property should have been been updated.
     */
    property?: string;

    /**
     * Data type used in the binding.
     */
    type?: Type;

    /**
     * New value (external representation) as parsed and validated by the binding.
     */
    newValue?: any;

    /**
     * Old value (external representation) as previously stored in the ManagedObject.
     */
    oldValue?: any;

    /**
     * Localized message describing the validation issues
     */
    message?: string;
  }

  /**
   * Event object of the ManagedObject#validationError event.
   */
  export type ManagedObject$ValidationErrorEvent = Event<
    ManagedObject$ValidationErrorEventParameters,
    ManagedObject
  >;

  /**
   * Parameters of the ManagedObject#validationSuccess event.
   */
  export interface ManagedObject$ValidationSuccessEventParameters {
    /**
     * ManagedObject instance whose property initiated the model update.
     */
    element?: ManagedObject;

    /**
     * Name of the property for which the bound model property has been updated.
     */
    property?: string;

    /**
     * Data type used in the binding.
     */
    type?: Type;

    /**
     * New value (external representation) as propagated to the model.
     *
     * **Note: **the model might modify (normalize) the value again and this modification will be stored in
     * the ManagedObject. The 'newValue' parameter of this event contains the value **before** such a normalization.
     */
    newValue?: any;

    /**
     * Old value (external representation) as previously stored in the ManagedObject.
     */
    oldValue?: any;
  }

  /**
   * Event object of the ManagedObject#validationSuccess event.
   */
  export type ManagedObject$ValidationSuccessEvent = Event<
    ManagedObject$ValidationSuccessEventParameters,
    ManagedObject
  >;

  export namespace MetadataOptions {
    /**
     * An object literal describing an aggregation of a class derived from `sap.ui.base.ManagedObject`. See
     * {@link sap.ui.base.ManagedObject.MetadataOptions MetadataOptions} for details on its usage.
     */
    type Aggregation = {
      /**
       * Type of the new aggregation. Must be the full global name of a ManagedObject subclass or a UI5 interface
       * (in dot notation, e.g. 'sap.m.Button').
       */
      type?: string;
      /**
       * The default class for the aggregation. If aggregation content is created from a plain object and no explicit
       * 'Type' is given (capital 'T'), the default class will be instantiated.
       */
      defaultClass?: Function;
      /**
       * Whether the aggregation is a 0..1 (false) or a 0..n aggregation (true), defaults to true
       */
      multiple?: boolean;
      /**
       * Singular name for 0..n aggregations. For 0..n aggregations the name by convention should be the plural
       * name. Methods affecting multiple objects in an aggregation will use the plural name (e.g. getItems(),
       * whereas methods that deal with a single object will use the singular name (e.g. addItem). The framework
       * knows a set of common rules for building the plural form of English nouns and uses these rules to determine
       * a singular name on its own. If that name is wrong, a singluarName can be specified with this property.
       */
      singularName?: string;
      /**
       * Either 'hidden' or 'public', defaults to 'public'. Aggregations that belong to the API of a class must
       * be 'public' whereas 'hidden' aggregations typically are used for the implementation of composite classes
       * (e.g. composite controls). Only public aggregations are accepted by the constructor or by `applySettings`
       * or in declarative representations like an `XMLView`. Equally, only public aggregations are cloned.
       */
      visibility?: "hidden" | "public";
      /**
       * (Either can be omitted or set to the boolean value `true` or the magic string 'bindable'.) If set to
       * `true` or 'bindable', additional named methods `bindName` and `unbindName` are generated
       * as convenience. Despite its name, setting this flag is not mandatory to make the managed aggregation
       * bindable. The generic methods {@link #bindAggregation} and {@link #unbindAggregation} can always be used.
       */
      bindable?: boolean | "bindable";
      /**
       * If set, this defines a forwarding of objects added to this aggregation into an aggregation of another
       * ManagedObject - typically to an inner control within a composite control. This means that all adding,
       * removal, or other operations happening on the source aggregation are actually called on the target instance.
       * All elements added to the source aggregation will be located at the target aggregation (this means the
       * target instance is their parent). Both, source and target element will return the added elements when
       * asked for the content of the respective aggregation. If present, the named (non-generic) aggregation
       * methods will be called for the target aggregation. Aggregations can only be forwarded to non-hidden aggregations
       * of the same or higher multiplicity (i.e. an aggregation with multiplicity "0..n" cannot be forwarded
       * to an aggregation with multiplicity "0..1"). The target aggregation must also be "compatible" to the
       * source aggregation in the sense that any items given to the source aggregation must also be valid in
       * the target aggregation (otherwise the target element will throw a validation error). If the forwarded
       * elements use data binding, the target element must be properly aggregated by the source element to make
       * sure all models are available there as well. The aggregation target must remain the same instance across
       * the entire lifetime of the source control. Aggregation forwarding will behave unexpectedly when the content
       * in the target aggregation is modified by other actors (e.g. by the target element or by another forwarding
       * from a different source aggregation). Hence, this is not allowed.
       */
      forwarding?: {
        /**
         * The name of the aggregation on the target into which the objects shall be forwarded. The multiplicity
         * of the target aggregation must be the same as the one of the source aggregation for which forwarding
         * is defined.
         */
        aggregation: string;
        /**
         * A string which is appended to the ID of this ManagedObject to construct the ID of the target ManagedObject.
         * This is one of the two options to specify the target. This option requires the target instance to be
         * created in the init() method of this ManagedObject and to be always available.
         */
        idSuffix?: string;
        /**
         * The name of the function on instances of this ManagedObject which returns the target instance. This second
         * option to specify the target can be used for lazy instantiation of the target. Note that either idSuffix
         * or getter must be given. Also note that the target instance returned by the getter must remain the same
         * over the entire lifetime of this ManagedObject and the implementation assumes that all instances return
         * the same type of object (at least the target aggregation must always be defined in the same class).
         */
        getter?: string;
        /**
         * Whether any binding should happen on the forwarding target or not. Default if omitted is `false`, which
         * means any bindings happen on the outer ManagedObject. When the binding is forwarded, all binding methods
         * like updateAggregation, getBindingInfo, refreshAggregation etc. are called on the target element of the
         * forwarding instead of being called on this element. The basic aggregation mutator methods (add/remove
         * etc.) are only called on the forwarding target element. Without forwardBinding, they are called on this
         * element, but forwarded to the forwarding target, where they actually modify the aggregation.
         */
        forwardBinding?: boolean;
      };
      /**
       * Can be set to a valid CSS selector (as accepted by the {@link https://developer.mozilla.org/en-US/docs/Web/API/Element/querySelector Element.prototype.querySelector }
       * method). When set, it locates the DOM element that surrounds the aggregation's content. It should only
       * be set for aggregations that have a visual representation in the DOM. A DOM element surrounding the aggregation's
       * rendered content should be available in the DOM, even if the aggregation is empty or not rendered for
       * some reason. In cases where this is not possible or not intended, `getDomRefForSetting` can be overridden,
       * see below.
       *
       * The purpose of the selector is to allow other framework parts like drag and drop or design time tooling
       * to identify those DOM parts of a control or element that represent a specific aggregation without knowing
       * the control or element implementation in detail.
       *
       * As an extension to the standard CSS selector syntax, the selector string can contain the placeholder
       * `{id}` (multiple times). Before evaluating the selector in the context of an element or control, all
       * occurrences of the placeholder have to be replaced by the (potentially escaped) ID of that element or
       * control. In fact, any selector should start with `#{id}` to ensure that the query result is limited to
       * the desired element or control.
       *
       * **Note**: there is a convenience method {@link sap.ui.core.Element#getDomRefForSetting} that evaluates
       * the selector in the context of a concrete element or control instance. It also handles the placeholder
       * `{id}`. Only selected framework features may use that private method, it is not yet a public API and
       * might be changed or removed in future versions of UI5. However, instead of maintaining the `selector`
       * in the metadata, element and control classes can overwrite `getDomRefForSetting` to calculate or add
       * the appropriate DOM Element dynamically.
       */
      selector?: string;
      /**
       * Flag that marks the aggregation as deprecated (defaults to false). May lead to an additional warning
       * log message at runtime when the aggregation is still used. For the documentation, also add a `@deprecated`
       * tag in the JSDoc, describing since when it is deprecated and what any alternatives are.
       */
      deprecated?: boolean;
      /**
       * An optional list of alternative types that may be given instead of the main type. Alternative types may
       * only be simple types, no descendants of ManagedObject. An example of altTypes being used is the 'tooltip'
       * aggregation of `sap.ui.core.Element`, which accepts tooltip controls extending `sap.ui.core.TooltipBase`
       * with their own renderer and design, as well as plain strings, which will simply be displayed using the
       * browser's built-in tooltip functionality.
       */
      altTypes?: string[];
      /**
       * Only available for aggregations of a class extending `sap.ui.core.Element`, which is a subclass of `sap.ui.base.ManagedObject`!
       * Defines draggable and droppable configuration of the aggregation. If the `dnd` property is of type Boolean,
       * then the `draggable` and `droppable` configuration are both set to this Boolean value and the layout
       * (in case of enabled dnd) is set to default ("Vertical").
       */
      dnd?:
        | boolean
        | {
            /**
             * Defines whether elements from this aggregation are draggable or not. The default value is `false`.
             */
            draggable?: boolean;
            /**
             * Defines whether the element is droppable (it allows being dropped on by a draggable element) or not.
             * The default value is `false`.
             */
            droppable?: boolean;
            /**
             * The arrangement of the items in this aggregation. This setting is recommended for the aggregation with
             * multiplicity 0..n (`multiple: true`). Possible values are `Vertical` (e.g. rows in a table) and `Horizontal`
             * (e.g. columns in a table). It is recommended to use `Horizontal` layout if the visual arrangement of
             * the aggregation is two-dimensional.
             */
            layout?: "Vertical" | "Horizontal";
          };
    };

    /**
     * An object literal describing an association of a class derived from `sap.ui.base.ManagedObject`. See
     * {@link sap.ui.base.ManagedObject.MetadataOptions MetadataOptions} for details on its usage.
     */
    type Association = {
      /**
       * Type of the new association
       */
      type?: string;
      /**
       * Whether the association is a 0..1 (false) or a 0..n association (true), defaults to false (0..1) for
       * associations
       */
      multiple?: boolean;
      /**
       * Custom singular name. This is only relevant for 0..n associations where the association name should be
       * defined in plural form and the framework tries to generate the singular form of it for certain places
       * where it is needed. To do so, the framework knows a set of common rules for building the plural form
       * of English nouns and uses these rules to determine a singular name on its own. If that name is wrong,
       * a singularName can be specified with this property. E.g. for an association named `items`, methods affecting
       * multiple objects in an association will use the plural name (`getItems()`), whereas methods that deal
       * with a single object will automatically use the generated singular name (e.g. `addItem(...)`). However,
       * the generated singular form for an association `news` would be `new`, which is wrong, so the singular
       * name "news" would need to be set.
       */
      singularName?: string;
      /**
       * Either 'hidden' or 'public', defaults to 'public'. Associations that belong to the API of a class must
       * be 'public' whereas 'hidden' associations can only be used internally. Only public associations are accepted
       * by the constructor or by `applySettings` or in declarative representations like an `XMLView`. Equally,
       * only public associations are cloned.
       */
      visibility?: "hidden" | "public";
      /**
       * Flag that marks the association as deprecated (defaults to false). May lead to an additional warning
       * log message at runtime when the association is still used. For the documentation, also add a `@deprecated`
       * tag in the JSDoc, describing since when it is deprecated and what any alternatives are.
       */
      deprecated?: boolean;
    };

    /**
     * An object literal describing an event of a class derived from `sap.ui.base.ManagedObject`. See {@link sap.ui.base.ManagedObject.MetadataOptions MetadataOptions }
     * for details on its usage.
     */
    type Event = {
      /**
       * Whether the event allows to prevented the default behavior of the event source
       */
      allowPreventDefault?: boolean;
      /**
       * An object literal that describes the parameters of this event; the keys are the parameter names and the
       * values are objects with a 'type' property that specifies the type of the respective parameter.
       */
      parameters?: Record<
        string,
        | {
            type: string;
          }
        | string
      >;
      /**
       * whether event bubbling is enabled on this event. When `true` the event is also forwarded to the parent(s)
       * of the object (see {@link sap.ui.base.EventProvider#getEventingParent}) until the bubbling of the event
       * is stopped or no parent is available anymore.
       */
      enableEventBubbling?: boolean;
      /**
       * Flag that marks the event as deprecated (defaults to false). May lead to an additional warning log message
       * at runtime when the event is still used. For the documentation, also add a `@deprecated` tag in the JSDoc,
       * describing since when it is deprecated and what any alternatives are.
       */
      deprecated?: boolean;
    };

    /**
     * An object literal describing a property of a class derived from `sap.ui.base.ManagedObject`. See {@link sap.ui.base.ManagedObject.MetadataOptions MetadataOptions }
     * for details on its usage.
     */
    type Property = {
      /**
       * Type of the new property. Must either be one of the built-in types 'string', 'boolean', 'int', 'float',
       * 'object', 'function' or 'any', or a type created and registered with {@link sap.ui.base.DataType.createType }
       * or an array type based on one of the previous types (e.g. 'int[]' or 'string[]', but not just 'array').
       */
      type: string;
      /**
       * Either 'hidden' or 'public', defaults to 'public'. Properties that belong to the API of a class must
       * be 'public' whereas 'hidden' properties can only be used internally. Only public properties are accepted
       * by the constructor or by `applySettings` or in declarative representations like an `XMLView`. Equally,
       * only public properties are cloned.
       */
      visibility?: "hidden" | "public";
      /**
       * If set to `true`, the property value will be {@link module:sap/base/util/deepClone deep cloned} on write
       * and read operations to ensure that the internal value can't be modified by the outside. The property
       * `byValue` is currently restricted to a `boolean` value. Other types are reserved for future use. Class
       * definitions must only use boolean values for the flag (or omit it), but readers of ManagedObject metadata
       * should handle any truthy value as `true` to be future safe. Note that using `byValue:true` has a performance
       * impact on property access and therefore should be used carefully. It also doesn't make sense to set this
       * option for properties with a primitive type (they have value semantic anyhow) or for properties with
       * arrays of primitive types (they are already cloned with a less expensive implementation). Defaults to
       * 'false'.
       */
      byValue?: boolean;
      /**
       * A semantic grouping of the properties, intended to be used in design time tools. Allowed values are (case
       * sensitive): Accessibility, Appearance, Behavior, Data, Designtime, Dimension, Identification, Misc
       */
      group?:
        | "Accessibility"
        | "Appearance"
        | "Behavior"
        | "Data"
        | "Designtime"
        | "Dimension"
        | "Identification"
        | "Misc";
      /**
       * The default value for the property or null if there is no specific default value defined (the data type's
       * default becomes the default value in this case, e.g. `false` for boolean and the empty string for type
       * string). Omitting this property means the default value is `undefined`.
       */
      defaultValue?: any;
      /**
       * (Either can be omitted or set to the boolean value `true` or the magic string 'bindable'.) If set to
       * `true` or 'bindable', additional named methods `bindName` and `unbindName` are generated
       * as convenience. Despite its name, setting this flag is not mandatory to make the managed property bindable.
       * The generic methods {@link #bindProperty} and {@link #unbindProperty} can always be used.
       */
      bindable?: boolean | "bindable";
      /**
       * Can be set to a valid CSS selector (as accepted by the {@link https://developer.mozilla.org/en-US/docs/Web/API/Element/querySelector Element.prototype.querySelector }
       * method). When set, it locates the DOM element that represents this property's value. It should only be
       * set for properties that have a visual text representation in the DOM.
       *
       * The purpose of the selector is to allow other framework parts or design time tooling to identify the
       * DOM parts of a control or element that represent a specific property without knowing the control or element
       * implementation in detail.
       *
       * As an extension to the standard CSS selector syntax, the selector string can contain the placeholder
       * `{id}` (multiple times). Before evaluating the selector in the context of an element or control, all
       * occurrences of the placeholder have to be replaced by the (potentially escaped) ID of that element or
       * control. In fact, any selector should start with `#{id}` to ensure that the query result is limited to
       * the desired element or control.
       *
       * **Note**: there is a convenience method {@link sap.ui.core.Element#getDomRefForSetting} that evaluates
       * the selector in the context of a concrete element or control instance. It also handles the placeholder
       * `{id}`. Only selected framework features may use that private method, it is not yet a public API and
       * might be changed or removed in future versions of UI5. However, instead of maintaining the `selector`
       * in the metadata, element and control classes can overwrite `getDomRefForSetting` and determine the DOM
       * element dynamically.
       */
      selector?: string;
      /**
       * Flag that marks the property as deprecated (defaults to false). May lead to an additional warning log
       * message at runtime when the property is still used. For the documentation, also add a `@deprecated` tag
       * in the JSDoc, describing since when it is deprecated and what any alternatives are.
       */
      deprecated?: boolean;
    };
  }
}

declare module "sap/ui/base/ManagedObjectMetadata" {
  import Metadata from "sap/ui/base/Metadata";

  import {
    MetadataOptions,
    default as ManagedObject,
    MetadataOptions as MetadataOptions1,
  } from "sap/ui/base/ManagedObject";

  /**
   * **Note about Info Objects**
   *
   * Several methods in this class return info objects that describe a property, aggregation, association
   * or event of the class described by this metadata object. The type, structure and behavior of these info
   * objects is not yet documented and not part of the stable, public API.
   *
   * Code using such methods and the returned info objects therefore needs to be aware of the following restrictions:
   *
   *
   * 	the set of properties exposed by each info object, their type and value might change as well as the
   * class of the info object itself.
   *
   * Properties that represent settings provided during class definition (in the oClassInfo parameter of the
   * 'extend' call, e.g. 'type', 'multiple' of an aggregation) are more likely to stay the same than additional,
   * derived properties like '_iKind'.
   *
   *
   * 	 - info objects must not be modified / enriched although they technically could.
   *
   *
   * 	 - the period of validity of info objects is not defined. They should be referenced only for a short
   *     time and not be kept as members of long living objects or closures.
   *
   *
   *
   * @since 0.8.6
   */
  export default class ManagedObjectMetadata extends Metadata {
    /**
     * Creates a new metadata object that describes a subclass of ManagedObject.
     *
     * **Note:** Code outside the `sap.ui.base` namespace must not call this constructor directly. Instances
     * will be created automatically when a new class is defined with one of the {@link sap.ui.base.ManagedObject.extend SomeClass.extend }
     * methods.
     *
     * **Note**: throughout this class documentation, the described subclass of ManagedObject is referenced
     * as the described class.
     */
    constructor(
      /**
       * fully qualified name of the described class
       */
      sClassName: string,
      /**
       * static info to construct the metadata from
       */
      oClassInfo: {
        /**
         * The metadata object describing the class
         */
        metadata?: MetadataOptions;
      }
    );

    /**
     * Adds information to the given oAggregatedObject about its original API parent (or a subsequent API parent
     * in case of multiple forwarding). MUST be called before an element is forwarded to another internal aggregation
     * (in case forwarding is done explicitly/manually without using the declarative mechanism introduced in
     * UI5 1.56).
     *
     * CAUTION: ManagedObjectMetadata.addAPIParentInfoEnd(...) MUST be called AFTER the element has been forwarded
     * (set to an aggregation of an internal control). These two calls must wrap the forwarding.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    static addAPIParentInfoBegin(
      /**
       * Object to which the new API parent info should be added
       */
      oAggregatedObject: ManagedObject,
      /**
       * Object that is a new API parent
       */
      oParent: ManagedObject,
      /**
       * the name of the aggregation under which oAggregatedObject is aggregated by the API parent
       */
      sAggregationName: string
    ): void;
    /**
     * Completes the information about the original API parent of the given element. MUST be called after an
     * element is forwarded to another internal aggregation. For every call to ManagedObjectMetadata.addAPIParentInfoBegin(...)
     * this method here must be called as well.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    static addAPIParentInfoEnd(
      /**
       * Object to which the new API parent info should be added
       */
      oAggregatedObject: ManagedObject
    ): void;
    /**
     * Get the prefix used for the generated IDs from configuration
     *
     * @since 1.119.0
     *
     * @returns The prefix for the generated IDs
     */
    static getUIDPrefix(): string;
    /**
     * Test whether a given ID looks like it was automatically generated.
     *
     * Examples:
     * ```javascript
     *
     * True for:
     *   "foo--__bar04--baz"
     *   "foo--__bar04"
     *   "__bar04--baz"
     *   "__bar04"
     *   "__bar04--"
     *   "__bar04--foo"
     * False for:
     *   "foo__bar04"
     *   "foo__bar04--baz"
     * ```
     *
     *
     * See {@link sap.ui.base.ManagedObjectMetadata.prototype.uid} for details on ID generation.
     *
     *
     * @returns whether the ID is likely to be generated
     */
    static isGeneratedId(
      /**
       * the ID that should be tested
       */
      sId: string
    ): boolean;
    /**
     * Calculates a new ID based on a prefix.
     *
     * To guarantee uniqueness of the generated IDs across all ID prefixes, prefixes must not end with digits.
     *
     *
     * @returns A (hopefully unique) control id
     */
    static uid(
      /**
       * prefix for the new ID
       */
      sIdPrefix: string
    ): string;
    /**
     * Returns an info object for the named public aggregation of the described class no matter whether the
     * aggregation was defined by the class itself or by one of its ancestor classes.
     *
     * If neither the class nor its ancestor classes define a public aggregation with the given name, `undefined`
     * is returned.
     *
     * If the name is not given (or has a falsy value), then it is substituted by the name of the default aggregation
     * of the 'described class' (if any).
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @since 1.27.0
     *
     * @returns An info object describing the aggregation or `undefined`
     */
    getAggregation(
      /**
       * name of the aggregation or empty
       */
      sName?: string
    ): MetadataOptions1.Aggregation | undefined;
    /**
     * Returns a map of info objects for the public aggregations of the described class. Aggregations declared
     * by ancestor classes are not included.
     *
     * The returned map keys the aggregation info objects by their name. In case of 0..1 aggregations this is
     * the singular name, otherwise it is the plural name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of aggregation info objects keyed by aggregation names
     */
    getAggregations(): Record<string, MetadataOptions1.Aggregation>;
    /**
     * Returns a map of info objects for all public aggregations of the described class, including public aggregations
     * form the ancestor classes.
     *
     * The returned map keys the aggregation info objects by their name. In case of 0..1 aggregations this is
     * the singular name, otherwise it is the plural name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of aggregation info objects keyed by aggregation names
     */
    getAllAggregations(): Record<string, MetadataOptions1.Aggregation>;
    /**
     * Returns a map of info objects for all public associations of the described class, including public associations
     * form the ancestor classes.
     *
     * The returned map keys the association info objects by their name. In case of 0..1 associations this is
     * the singular name, otherwise it is the plural name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of association info objects keyed by association names
     */
    getAllAssociations(): Record<string, MetadataOptions1.Association>;
    /**
     * Returns a map of info objects for all public events of the described class, including public events form
     * the ancestor classes.
     *
     * The returned map keys the event info objects by their name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of event info objects keyed by event names
     */
    getAllEvents(): Record<string, MetadataOptions1.Event>;
    /**
     * Returns a map of info objects for all private (hidden) aggregations of the described class, including
     * private aggregations from the ancestor classes.
     *
     * The returned map contains aggregation info objects keyed by the aggregation name. In case of 0..1 aggregations
     * this is the singular name, otherwise it is the plural name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Map of aggregation info objects keyed by aggregation names
     */
    getAllPrivateAggregations(): Record<string, MetadataOptions1.Aggregation>;
    /**
     * Returns a map of info objects for all private (hidden) associations of the described class, including
     * private associations from the ancestor classes.
     *
     * The returned map contains association info objects keyed by the association name. In case of 0..1 associations
     * this is the singular name, otherwise it is the plural name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Map of association info objects keyed by association names
     */
    getAllPrivateAssociations(): Record<string, MetadataOptions1.Association>;
    /**
     * Returns a map of info objects for all private (hidden) properties of the described class, including private
     * properties from the ancestor classes.
     *
     * The returned map contains property info objects keyed by the property name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Map of property info objects keyed by property names
     */
    getAllPrivateProperties(): Record<string, MetadataOptions1.Property>;
    /**
     * Returns a map of info objects for all public properties of the described class, including public properties
     * from the ancestor classes.
     *
     * The returned map keys the property info objects by their name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of property info objects keyed by the property names
     */
    getAllProperties(): Record<string, MetadataOptions1.Property>;
    /**
     * Returns an info object for the named public association of the described class, no matter whether the
     * association was defined by the class itself or by one of its ancestor classes.
     *
     * If neither the described class nor its ancestor classes define an association with the given name, `undefined`
     * is returned.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @since 1.27.0
     *
     * @returns An info object describing the association or `undefined`
     */
    getAssociation(
      /**
       * name of the association
       */
      sName: string
    ): MetadataOptions1.Association | undefined;
    /**
     * Returns a map of info objects for all public associations of the described class. Associations declared
     * by ancestor classes are not included.
     *
     * The returned map keys the association info objects by their name. In case of 0..1 associations this is
     * the singular name, otherwise it is the plural name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of association info objects keyed by association names
     */
    getAssociations(): Record<string, MetadataOptions1.Association>;
    /**
     * Returns an info object for the default aggregation of the described class.
     *
     * If the class itself does not define a default aggregation, then the info object for the default aggregation
     * of the parent class is returned.
     *
     * @since 1.73
     *
     * @returns An info object for the default aggregation
     */
    getDefaultAggregation(): MetadataOptions1.Aggregation;
    /**
     * Returns the name of the default aggregation of the described class.
     *
     * If the class itself does not define a default aggregation, then the default aggregation of the parent
     * is returned. If no class in the hierarchy defines a default aggregation, `undefined` is returned.
     *
     * @since 1.73
     *
     * @returns Name of the default aggregation
     */
    getDefaultAggregationName(): string;
    /**
     * Returns an info object for the named public event of the described class, no matter whether the event
     * was defined by the class itself or by one of its ancestor classes.
     *
     * If neither the described class nor its ancestor classes define an event with the given name, `undefined`
     * is returned.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @since 1.27.0
     *
     * @returns An info object describing the event or `undefined`
     */
    getEvent(
      /**
       * name of the event
       */
      sName: string
    ): MetadataOptions1.Event | undefined;
    /**
     * Returns a map of info objects for the public events of the described class. Events declared by ancestor
     * classes are not included.
     *
     * The returned map keys the event info objects by their name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of event info objects keyed by event names
     */
    getEvents(): Record<string, MetadataOptions1.Event>;
    /**
     * Returns the name of the library that contains the described UIElement.
     *
     *
     * @returns the name of the library
     */
    getLibraryName(): string;
    /**
     * Returns the info object for the named public or private aggregation declared by the described class or
     * by any of its ancestors.
     *
     * If the name is not given (or has a falsy value), then it is substituted by the name of the default aggregation
     * of the described class (if it is defined).
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns aggregation info object or `undefined`
     */
    getManagedAggregation(
      /**
       * name of the aggregation to be retrieved or empty
       */
      sAggregationName: string
    ): MetadataOptions1.Aggregation | undefined;
    /**
     * Returns the info object for the named public or private association declared by the described class or
     * by any of its ancestors.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns association info object or `undefined`
     */
    getManagedAssociation(
      /**
       * name of the association to be retrieved
       */
      sName: string
    ): MetadataOptions1.Association | undefined;
    /**
     * Returns the info object for the named public or private property declared by the described class or by
     * any of its ancestors.
     *
     * If the name is not given (or has a falsy value), then it is substituted by the name of the default property
     * of the described class (if it is defined).
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns property info object or `undefined`
     */
    getManagedProperty(
      /**
       * name of the property to be retrieved or empty
       */
      sName: string
    ): MetadataOptions1.Property | undefined;
    /**
     * Returns a map of info objects for the public properties of the described class. Properties declared by
     * ancestor classes are not included.
     *
     * The returned map keys the property info objects by their name.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     *
     * @returns Map of property info objects keyed by the property names
     */
    getProperties(): Record<string, MetadataOptions1.Property>;
    /**
     * Returns an info object for the named public property of the described class, no matter whether the property
     * was defined by the class itself or by one of its ancestor classes.
     *
     * If neither the described class nor its ancestor classes define a property with the given name, `undefined`
     * is returned.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @since 1.27.0
     *
     * @returns An info object describing the property or `undefined`
     */
    getProperty(
      /**
       * name of the property
       */
      sName: string
    ): MetadataOptions1.Property | undefined;
    /**
     * Returns a map of default values for all properties declared by the described class and its ancestors,
     * keyed by the property name.
     *
     *
     * @returns Map of default values keyed by property names
     */
    getPropertyDefaults(): Record<string, any>;
    /**
     * Returns an info object for a public setting with the given name that either is a public property or a
     * public aggregation of cardinality 0..1 and with at least one simple alternative type. The setting can
     * be defined by the class itself or by one of its ancestor classes.
     *
     * If neither the described class nor its ancestor classes define a suitable setting with the given name,
     * `undefined` is returned.
     *
     * **Warning:** Type, structure and behavior of the returned info objects is not documented and therefore
     * not part of the API. See the {@link #constructor Notes about Info objects} in the constructor documentation
     * of this class.
     *
     * @since 1.27.0
     *
     * @returns An info object describing the property or aggregation or `undefined`
     */
    getPropertyLikeSetting(
      /**
       * name of the property like setting
       */
      sName: string
    ): MetadataOptions1.Property | MetadataOptions1.Aggregation | undefined;
    /**
     * Checks the existence of the given public aggregation by its name.
     *
     *
     * @returns true, if the aggregation exists
     */
    hasAggregation(
      /**
       * name of the aggregation
       */
      sName: string
    ): boolean;
    /**
     * Checks the existence of the given public association by its name
     *
     *
     * @returns true, if the association exists
     */
    hasAssociation(
      /**
       * name of the association
       */
      sName: string
    ): boolean;
    /**
     * Checks the existence of the given event by its name
     *
     *
     * @returns true, if the event exists
     */
    hasEvent(
      /**
       * name of the event
       */
      sName: string
    ): boolean;
    /**
     * Checks the existence of the given public property by its name
     *
     *
     * @returns true, if the property exists
     */
    hasProperty(
      /**
       * name of the property
       */
      sName: string
    ): boolean;
    /**
     * Calculates a new ID for an instance of this class.
     *
     * Note that the calculated short name part is usually not unique across all classes, but doesn't have to
     * be. It might even be empty when the class name consists of invalid characters only.
     *
     *
     * @returns A (hopefully unique) control ID
     */
    uid(): string;
  }
}

declare module "sap/ui/base/Metadata" {
  import { MetadataOptions, default as BaseObject } from "sap/ui/base/Object";

  /**
   * Metadata for a class.
   *
   * @since 0.8.6
   */
  export default class Metadata {
    /**
     * Creates a new metadata object from the given static infos.
     *
     * **Note:** Throughout this class documentation, the described subclass of Object is referenced as the
     * described class.
     */
    constructor(
      /**
       * Fully qualified name of the described class
       */
      sClassName: string,
      /**
       * Info to construct the class and its metadata from
       */
      oClassInfo: {
        /**
         * The metadata object describing the class
         */
        metadata?: MetadataOptions;
      }
    );

    /**
     * Returns an array with the names of all public methods declared by the described class and all its ancestors
     * classes.
     *
     * @deprecated As of version 1.58. this method should not be used for productive code. The accuracy of the
     * returned information highly depends on the concrete class and is not actively monitored. There might
     * be more public methods or some of the returned methods might not really be intended for public use. In
     * general, pure visibility information should not be exposed in runtime metadata but be part of the documentation.
     * Subclasses of `sap.ui.base.Object` might decide to provide runtime metadata describing their public API,
     * but this then should not be backed by this method. See {@link sap.ui.core.mvc.ControllerMetadata#getAllMethods }
     * for an example.
     *
     * @returns array with names of all public methods provided by the described class and its ancestors
     */
    getAllPublicMethods(): string[];
    /**
     * Returns the (constructor of the) described class
     *
     *
     * @returns class described by this metadata
     */
    getClass(): new () => BaseObject;
    /**
     * Returns the fully qualified name of the described class
     *
     *
     * @returns name of the described class
     */
    getName(): string;
    /**
     * Returns the metadata object of the base class of the described class or undefined if the class has no
     * (documented) base class.
     *
     *
     * @returns metadata of the base class
     */
    getParent(): Metadata | undefined;
    /**
     * Returns an array with the names of the public methods declared by the described class, methods of ancestors
     * are not listed.
     *
     * @deprecated As of version 1.58. this method should not be used for productive code. The accuracy of the
     * returned information highly depends on the concrete class and is not actively monitored. There might
     * be more public methods or some of the returned methods might not really be intended for public use. In
     * general, pure visibility information should not be exposed in runtime metadata but be part of the documentation.
     * Subclasses of `sap.ui.base.Object` might decide to provide runtime metadata describing their public API,
     * but this then should not be backed by this method. See {@link sap.ui.core.mvc.ControllerMetadata#getAllMethods }
     * for an example.
     *
     * @returns array with names of public methods declared by the described class
     */
    getPublicMethods(): string[];
    /**
     * Checks whether the class described by this metadata object is of the named type.
     *
     * This check is solely based on the type names as declared in the class metadata. It compares the given
     * `vTypeName` with the name of this class, with the names of any base class of this class and with the
     * names of all interfaces implemented by any of the aforementioned classes.
     *
     * Instead of a single type name, an array of type names can be given and the method will check if this
     * class is of any of the listed types (logical or).
     *
     * Should the UI5 class system in future implement additional means of associating classes with type names
     * (e.g. by introducing mixins), then this method might detect matches for those names as well.
     *
     * @since 1.56
     *
     * @returns Whether this class is of the given type or of any of the given types
     */
    isA(
      /**
       * Type or types to check for
       */
      vTypeName: string | string[]
    ): boolean;
    /**
     * Returns whether the described class is abstract
     *
     *
     * @returns whether the class is abstract
     */
    isAbstract(): boolean;
    /**
     * Whether the described class is deprecated and should not be used any more
     *
     * @since 1.26.4
     *
     * @returns whether the class is considered deprecated
     */
    isDeprecated(): boolean;
    /**
     * Returns whether the described class is final
     *
     *
     * @returns whether the class is final
     */
    isFinal(): boolean;
    /**
     * Checks whether the described class or one of its ancestor classes implements the given interface.
     *
     *
     * @returns whether this class implements the interface
     */
    isInstanceOf(
      /**
       * name of the interface to test for (in dot notation)
       */
      sInterface: string
    ): boolean;
  }
}

declare module "sap/ui/base/Object" {
  import Metadata from "sap/ui/base/Metadata";

  /**
   * Base class for all SAPUI5 Objects.
   */
  export default abstract class BaseObject {
    /**
     * Constructor for an `sap.ui.base.Object`.
     *
     * Subclasses of this class should always call the constructor of their base class.
     */
    constructor();

    /**
     * Creates metadata for a given class and attaches it to the constructor and prototype of that class.
     *
     * After creation, metadata can be retrieved with getMetadata().
     *
     * The static info can at least contain the following entries:
     * 	 - baseType: {string} fully qualified name of a base class or empty
     * 	 - publicMethods: {string} an array of method names that will be visible in the interface proxy returned
     *     by {@link #getInterface}
     *
     * @deprecated As of version 1.3.1. Use the static `extend` method of the desired base class (e.g. {@link sap.ui.base.Object.extend})
     *
     * @returns the created metadata object
     */
    static defineClass(
      /**
       * name of an (already declared) constructor function
       */
      sClassName: string,
      /**
       * static info used to create the metadata object
       */
      oStaticInfo: {
        /**
         * qualified name of a base class
         */
        baseType: string;
        /**
         * array of names of public methods
         */
        publicMethods: string[];
      },
      /**
       * constructor function for the metadata object. If not given, it defaults to sap.ui.base.Metadata.
       */
      FNMetaImpl?: Function
    ): Metadata;
    /**
     * Creates a subclass of class sap.ui.base.Object with name `sClassName` and enriches it with the information
     * contained in `oClassInfo`.
     *
     * `oClassInfo` might contain three kinds of information:
     * 	 - `metadata:` an (optional) object literal with metadata about the class like implemented interfaces,
     *     see {@link sap.ui.base.Object.MetadataOptions MetadataOptions} for details. The information in the object
     *     literal will be wrapped by an instance of {@link sap.ui.base.Metadata Metadata}. Subclasses of sap.ui.base.Object
     *     can enrich the set of supported metadata (e.g. see {@link sap.ui.core.Element.extend}).
     *
     *
     * 	 - `constructor:` a function that serves as a constructor function for the new class. If no constructor
     *     function is given, the framework creates a default implementation that delegates all its arguments to
     *     the constructor function of the base class.
     *
     * any-other-name: any other property in the `oClassInfo` is copied into the prototype object
     * of the newly created class. Callers can thereby add methods or properties to all instances of the class.
     * But be aware that the given values are shared between all instances of the class. Usually, it doesn't
     * make sense to use primitive values here other than to declare public constants.
     *
     * If such a property has a function as its value, and if the property name does not start with an underscore
     * or with the prefix "on", the property name will be automatically added to the list of public methods
     * of the class (see property `publicMethods` in the `metadata` section). If a method's name matches that
     * pattern, but is not meant to be public, it shouldn't be included in the class info object, but be assigned
     * to the prototype instead.
     *
     *
     *
     * The prototype object of the newly created class uses the same prototype as instances of the base class
     * (prototype chaining).
     *
     * A metadata object is always created, even if there is no `metadata` entry in the `oClassInfo` object.
     * A getter for the metadata is always attached to the prototype and to the class (constructor function)
     * itself.
     *
     * Last but not least, with the third argument `FNMetaImpl` the constructor of a metadata class can be specified.
     * Instances of that class will be used to represent metadata for the newly created class and for any subclass
     * created from it. Typically, only frameworks will use this parameter to enrich the metadata for a new
     * class hierarchy they introduce (e.g. {@link sap.ui.core.Element.extend Element}).
     *
     * @since 1.3.1
     *
     * @returns the created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * name of the class to be created
       */
      sClassName: string,
      /**
       * structured object with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BaseObject>,
      /**
       * constructor function for the metadata object. If not given, it defaults to sap.ui.base.Metadata.
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Checks whether the given object is an instance of the named type. This function is a short-hand convenience
     * for {@link sap.ui.base.Object#isA}.
     *
     * Please see the API documentation of {@link sap.ui.base.Object#isA} for more details.
     *
     * @since 1.56
     * @deprecated As of version 1.120. please use {@link sap.ui.base.Object.isObjectA}.
     *
     * @returns Whether the given object is an instance of the given type or of any of the given types
     */
    static isA<T extends BaseObject = BaseObject>(
      /**
       * Object which will be checked whether it is an instance of the given type
       */
      oObject: any,
      /**
       * Type or types to check for
       */
      vTypeName: string | string[]
    ): oObject is T;
    /**
     * Checks whether the given object is an instance of the named type. This function is a short-hand convenience
     * for {@link sap.ui.base.Object#isA}.
     *
     * Please see the API documentation of {@link sap.ui.base.Object#isA} for more details.
     *
     * @since 1.120
     *
     * @returns Whether the given object is an instance of the given type or of any of the given types
     */
    static isObjectA<T extends BaseObject = BaseObject>(
      /**
       * Object which will be checked whether it is an instance of the given type
       */
      oObject: any,
      /**
       * Type or types to check for
       */
      vTypeName: string | string[]
    ): oObject is T;
    /**
     * Destructor method for objects.
     */
    destroy(): void;
    /**
     * Returns the public facade of this object.
     *
     * By default, the public facade is implemented as an instance of {@link sap.ui.base.Interface}, exposing
     * the `publicMethods` as defined in the metadata of the class of this object.
     *
     * See the documentation of the {@link #.extend extend} method for an explanation of `publicMethods`.
     *
     * The facade is created on the first call of `getInterface` and reused for all later calls.
     *
     *
     * @returns A facade for this object, with at least the public methods of the class of this.
     */
    getInterface(): BaseObject;
    /**
     * Returns the metadata for the class that this object belongs to.
     *
     * This method is only defined when metadata has been declared by using {@link sap.ui.base.Object.defineClass }
     * or {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns metadata for the class of the object
     */
    getMetadata(): Metadata;
    /**
     * Checks whether this object is an instance of the named type.
     *
     * This check is solely based on the type names as declared in the class metadata. It compares the given
     * `vTypeName` with the name of the class of this object, with the names of any base class of that class
     * and with the names of all interfaces implemented by any of the aforementioned classes.
     *
     * Instead of a single type name, an array of type names can be given and the method will check if this
     * object is an instance of any of the listed types (logical or).
     *
     * Should the UI5 class system in future implement additional means of associating classes with type names
     * (e.g. by introducing mixins), then this method might detect matches for those names as well.
     *
     * @since 1.56
     *
     * @returns Whether this object is an instance of the given type or of any of the given types
     */
    isA<T extends BaseObject = BaseObject>(
      /**
       * Type or types to check for
       */
      vTypeName: string | string[]
    ): this is T;
  }
  /**
   * The structure of the "metadata" object which is passed when inheriting from sap.ui.base.Object using
   * its static "extend" method. See {@link sap.ui.base.Object.extend} for details on its usage.
   */
  export type MetadataOptions = {
    /**
     * set of names of implemented interfaces (defaults to no interfaces)
     */
    interfaces?: string[];
    /**
     * flag that marks the class as abstract (purely informational, defaults to false)
     */
    abstract?: boolean;
    /**
     * flag that marks the class as final (defaults to false)
     */
    final?: boolean;
    /**
     * flag that marks the class as deprecated (defaults to false). May lead to an additional warning log message
     * at runtime when the object is still used. For the documentation, also add a `@deprecated` tag in the
     * JSDoc, describing since when it is deprecated and what any alternatives are.
     */
    deprecated?: boolean;
  };
}

declare module "sap/ui/base/ObjectPool" {
  import BaseObject from "sap/ui/base/Object";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Manages a pool of objects for reuse, all of the same type; the type has to be specified at construction
   * time.
   *
   * Each pool maintains a list of free objects of the given type. If {@link sap.ui.base.ObjectPool.prototype.borrowObject }
   * is called, an existing free object is taken from the pool. When no free object is available, a new instance
   * is created by calling the constructor without any arguments. In either case, the {@link sap.ui.base.Poolable#init }
   * method is called on the object to initialize it with the data for the current caller.
   *
   * When the object is no longer needed, it has to be returned to the pool by calling {@link #returnObject}.
   * At that point in time, {@link sap.ui.base.Poolable#reset} is called on the object to remove all data
   * from it. Then it is is added back to the list of free objects for future reuse.
   *
   * See {@link sap.ui.base.Poolable} for a description of the contract for poolable objects.
   *
   * Example:
   * ```javascript
   *
   *   sap.ui.define([
   *     "sap/ui/base/Event",
   *     "sap/ui/base/ObjectPool"
   *   ], function(Event, ObjectPool) {
   *
   *     // create a pool for events
   *     var oEventPool = new ObjectPool(Event);
   *
   *     ...
   *
   *     // borrow an instance and initialize it at the same time
   *     var oEvent = oEventPool.borrowObject('myEvent', this, {foo: 'bar'});
   *     // this internally calls oEvent.init('myEvent', this, {foo: 'bar'})
   *
   *     // return the borrowed object
   *     oEventPool.returnObject(oEvent);
   *     // this internally calls oEvent.reset()
   *
   *     ...
   *
   *   }});
   * ```
   */
  export default class ObjectPool extends BaseObject {
    /**
     * Creates an `ObjectPool` for maintaining instances of the given class `oObjectClass`.
     *
     * `oObjectClass` must implement the {@link sap.ui.base.Poolable} interface.
     */
    constructor(
      /**
       * Constructor for the class of objects that this pool should manage
       */
      oObjectClass: new () => Object
    );

    /**
     * Creates a new subclass of class sap.ui.base.ObjectPool with name `sClassName` and enriches it with the
     * information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ObjectPool>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.base.ObjectPool.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Borrows a free object from the pool. Any arguments to this method are forwarded to the init method of
     * the borrowed object.
     *
     *
     * @returns The borrowed object of the same type that has been specified for this pool
     */
    borrowObject(
      /**
       * optional initialization parameters for the borrowed object
       */
      ...args: any[]
    ): Object;
    /**
     * Returns an object to the pool. The object must have been borrowed from this pool beforehand. The reset
     * method is called on the object before it is added to the set of free objects.
     */
    returnObject(
      /**
       * The object to return to the pool
       */
      oObject: Object
    ): void;
  }
  /**
   * Contract for objects that can be pooled by an `ObjectPool`.
   *
   * Poolable objects must provide a no-arg constructor which is used by the pool to construct new, unused
   * objects.
   *
   * To be more convenient to use, poolable objects should implement their constructor in a way that it either
   * can be called with no arguments (used by the pool) or with the same signature as their {@link #init }
   * method (to be used by applications).
   */
  export interface Poolable {
    __implements__sap_ui_base_Poolable: boolean;

    /**
     * Called by the `ObjectPool` when this instance will be activated for a caller.
     *
     * The same method will be called after a new instance has been created by an otherwise exhausted pool.
     *
     * If the caller provided any arguments to {@link sap.ui.base.ObjectPool#borrowObject}, all arguments will
     * be propagated to this method.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    init(
      /**
       * the arguments which were given to {@link sap.ui.base.ObjectPool#borrowObject}
       */
      ...args: any[]
    ): void;
    /**
     * Called by the object pool when an instance is returned to the pool.
     *
     * While no specific implementation is required, poolable objects in general should clean all caller specific
     * state (set to null) in this method to avoid memory leaks and to enforce garbage collection of the caller
     * state.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    reset(): void;
  }
}

declare module "sap/ui/core/library" {
  import UI5Element from "sap/ui/core/Element";

  import Control from "sap/ui/core/Control";

  import MessageType1 from "sap/ui/core/message/MessageType";

  /**
   * A string type that represents non-relative CSS size values.
   *
   * This is a subtype of the `'<length> type'` defined in the CSS specifications. Allowed values are
   * only absolute CSS sizes like "1px" or "2em". Percentage sizes like "50%"
   * and the special values "auto" and "inherit" are NOT allowed. Mathematical expressions
   * using the CSS3 `calc(expression)` operator are allowed as long as they do not use percentage sizes.
   *
   * Note that CSS might not allow all these values for every CSS property representing a size. So even if
   * a value is accepted by `sap.ui.core.AbsoluteCSSSize`, it still might have no effect in a specific context.
   * In other words: UI5 controls usually don't extend the range of allowed values in CSS.
   *
   * **Units**
   *
   * Valid font-relative units are `em, ex` and `rem`. Supported absolute units are `cm, mm, in, pc, pt` and
   * `px`. Other units are not supported.
   *
   * **Mathematical Expressions**
   *
   * Expressions inside the `calc()` operator are only roughly checked for validity. Not every value that
   * this type accepts is a valid expression in the sense of the CSS spec. But vice versa, any expression
   * that is valid according to the spec should be accepted by this type. The current implementation is based
   * on the {@link http://dev.w3.org/csswg/css-values-3/#calc-syntax CSS3 Draft specification from 22 April 2015}.
   *
   * Noteworthy details:
   * 	 - whitespace is mandatory around a '-' or '+' operator and optional otherwise
   * 	 - parentheses are accepted but not checked for being balanced (a restriction of regexp based checks)
   *
   * 	 - semantic constraints like type restrictions are not checked
   *
   * Future versions of UI5 might check `calc()` expressions in more detail, so applications should not assume
   * that a value, that is invalid according to the CSS spec but currently accepted by this type still will
   * be accepted by future versions of this type.
   */
  export type AbsoluteCSSSize = string;

  /**
   * The object contains accessibility information for a control.
   *
   * @since 1.110
   * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
   */
  export type AccessibilityInfo = {
    /**
     * The WAI-ARIA role which is implemented by the control.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    role?: string;
    /**
     * A translated text that represents the control type. Might correlate with the role.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    type?: string;
    /**
     * Describes the most relevant control state (e.g. the input's value) - it should be a translated text.
     * **Note:** The type and the enabled/editable state shouldn`t be handled here.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    description?: string;
    /**
     * Whether the control can get the focus.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    focusable?: boolean;
    /**
     * Whether the control is enabled. If not relevant, it shouldn`t be set or `null` can be provided.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    enabled?: boolean | null;
    /**
     * Whether the control is editable. If not relevant, it shouldn`t be set or `null` can be provided.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    editable?: boolean | null;
    /**
     * Whether the control is readonly. If not relevant, it shouldn`t be set or `null` can be provided.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    readonly?: boolean | null;
    /**
     * A list of elements or controls that are aggregated by the given control (e.g. when the control is a layout).
     * Primitive values in the list will be ignored. **Note:** Children should only be provided when it is helpful
     * to understand the accessibility context (e.g. a form control shouldn`t provide details of its internals
     * (fields, labels, ...) but a layout should).
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    children?: UI5Element[];
  };

  /**
   * Defines the accessible landmark roles for ARIA support. This enumeration is used with the AccessibleRole
   * control property. For more information, go to "Roles for Accessible Rich Internet Applications (WAI-ARIA
   * Roles)" at the www.w3.org homepage.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'AccessibleLandmarkRole'.
   */
  export enum AccessibleLandmarkRole {
    /**
     * Represents the ARIA role `banner`.
     *
     * A banner usually appears at the top of the page and typically spans the full width.
     */
    Banner = "Banner",
    /**
     * Represents the ARIA role `complementary`.
     *
     * A section of the page, designed to be complementary to the main content at a similar level in the DOM
     * hierarchy.
     */
    Complementary = "Complementary",
    /**
     * Represents the ARIA role `contentinfo`.
     *
     * A region that contains information about the content on the page.
     */
    ContentInfo = "ContentInfo",
    /**
     * Represents the ARIA role `form`.
     *
     * A region that contains a collection of items and objects that, as a whole, combine to create a form.
     */
    Form = "Form",
    /**
     * Represents the ARIA role `main`.
     *
     * The main content of a page.
     */
    Main = "Main",
    /**
     * Represents the ARIA role `navigation`.
     *
     * A region that contains a collection of items and objects that, as a whole, combine to create a navigation
     * facility.
     */
    Navigation = "Navigation",
    /**
     * No explicit role is applicable.
     *
     * The interpretation of this value depends on the control / element which defines a property with this
     * type. Normally this value means that no accessible landmark should be written.
     */
    None = "None",
    /**
     * Represents the ARIA role `region`.
     *
     * A section of a page, that is important enough to be included in a page summary or table of contents.
     */
    Region = "Region",
    /**
     * Represents the ARIA role `search`.
     *
     * A region that contains a collection of items and objects that, as a whole, combine to create a search
     * facility.
     */
    Search = "Search",
  }
  /**
   * Defines the accessible roles for ARIA support. This enumeration is used with the AccessibleRole control
   * property. For more information, goto "Roles for Accessible Rich Internet Applications (WAI-ARIA Roles)"
   * at the www.w3.org homepage.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'AccessibleRole'.
   */
  export enum AccessibleRole {
    /**
     * A message with an alert or error information.
     */
    Alert = "Alert",
    /**
     * A separate window with an alert or error information.
     */
    AlertDialog = "AlertDialog",
    /**
     * A software unit executing a set of tasks for the user.
     */
    Application = "Application",
    /**
     * Usually defined as the advertisement at the top of a web page. The banner content typically contains
     * the site or company logo, or other key advertisements.
     */
    Banner = "Banner",
    /**
     * Allows user-triggered actions.
     */
    Button = "Button",
    /**
     * A control that has three possible values: true, false, mixed.
     */
    Checkbox = "Checkbox",
    /**
     * A table cell containing header information for a column.
     */
    ColumnHeader = "ColumnHeader",
    /**
     * Allows selecting an item from a list, or to enter data directly in the input field.
     */
    Combobox = "Combobox",
    /**
     * Information about the content on the page. Examples are footnotes, copyrights, or links to privacy statements.
     */
    ContentInfo = "ContentInfo",
    /**
     * The content of the associated element represents a definition. If there is a definition element within
     * the content, this one represents the term being defined.
     */
    Definition = "Definition",
    /**
     * Descriptive content for a page element.
     */
    Description = "Description",
    /**
     * A small window that is designed to interrupt the current application processing in order to inform the
     * user and to get some response.
     */
    Dialog = "Dialog",
    /**
     * A list of references to members of a single group.
     */
    Directory = "Directory",
    /**
     * Content that contains related information, such as a book.
     */
    Document = "Document",
    /**
     * Contains cells of tabular data arranged in rows and columns, for example in a table.
     */
    Grid = "Grid",
    /**
     * A table cell in a grid where the cells can be active, editable, and selectable. Cells may have functional
     * relationships to controls, for example.
     */
    GridCell = "GridCell",
    /**
     * A section of user interface objects.
     */
    Group = "Group",
    /**
     * A heading for a section of the page.
     */
    Heading = "Heading",
    /**
     * A container for a collection of elements that form an image.
     */
    Img = "Img",
    /**
     * An interactive reference to a resource.
     */
    Link = "Link",
    /**
     * A container for non-interactive list items which are the children of the list.
     */
    List = "List",
    /**
     * A widget that allows the user to select one or more items from a list. The items within the list are
     * static and can contain images.
     */
    Listbox = "Listbox",
    /**
     * A single item in a list.
     */
    ListItem = "ListItem",
    /**
     * An area where new information is added, or old information disappears. Information types are chat logs,
     * messaging, or error logs, for example. The log contains a sequence: New information is always added to
     * the end of the log.
     */
    Log = "Log",
    /**
     * Defines the main content of a document.
     */
    Main = "Main",
    /**
     * Is used to scroll text across the page.
     */
    Marquee = "Marquee",
    /**
     * Offers a list of choices to the user.
     */
    Menu = "Menu",
    /**
     * A container for menu items where each item may activate a submenu.
     */
    Menubar = "Menubar",
    /**
     * A child in a menu.
     */
    MenuItem = "MenuItem",
    /**
     * A checkable menu item (tri-state).
     */
    MenuItemCheckbox = "MenuItemCheckbox",
    /**
     * A menu item which is part of a group of menuitemradio roles.
     */
    MenuItemRadio = "MenuItemRadio",
    /**
     * A collection of links suitable for use when navigating the document or related documents.
     */
    Navigation = "Navigation",
    /**
     * The content is parenthetic or ancillary to the main content of the resource.
     */
    Note = "Note",
    /**
     * A selectable item in a list represented by a select.
     */
    Option = "Option",
    /**
     * An element whose role is presentational does not need to be mapped to the accessibility API.
     */
    Presentation = "Presentation",
    /**
     * Shows the execution progress in applications providing functions that take a long time.
     */
    ProgressBar = "ProgressBar",
    /**
     * An option in single-select list. Only one radio control in a radiogroup can be selected at the same time.
     */
    Radio = "Radio",
    /**
     * A group of radio controls.
     */
    RadioGroup = "RadioGroup",
    /**
     * A large section on the web page.
     */
    Region = "Region",
    /**
     * A row of table cells.
     */
    Row = "Row",
    /**
     * A table cell containing header information for a row.
     */
    RowHeader = "RowHeader",
    /**
     * A search section of a web document. In many cases, this is a form used to submit search requests about
     * the site, or a more general Internet wide search service.
     */
    Search = "Search",
    /**
     * A unique section of the document. In the case of a portal, this may include time display, weather forecast,
     * or stock price.
     */
    Secondary = "Secondary",
    /**
     * Indicates that the element contains content that is related to the main content of the page.
     */
    SeeAlso = "SeeAlso",
    /**
     * A line or bar that separates sections of content.
     */
    Separator = "Separator",
    /**
     * A user input where the user selects an input in a given range. This form of range expects an analogous
     * keyboard interface.
     */
    Slider = "Slider",
    /**
     * Allows users to select a value from a list of given entries where exactly one value is displayed at runtime,
     * and the other ones can be displayed by scrolling using the arrow up and arrow down key.
     */
    SpinButton = "SpinButton",
    /**
     * A container for processing advisory information.
     */
    Status = "Status",
    /**
     * A header for a tab panel.
     */
    Tab = "Tab",
    /**
     * A list of tabs which are references to tab panels.
     */
    Tablist = "Tablist",
    /**
     * A container for the resources associated with a tab.
     */
    Tabpanel = "Tabpanel",
    /**
     * Inputs that allow free-form text as their value.
     */
    Textbox = "Textbox",
    /**
     * A numerical counter which indicates an amount of elapsed time from a start point, or of the time remaining
     * until a certain end point.
     */
    Timer = "Timer",
    /**
     * A collection of commonly used functions represented in compact visual form.
     */
    Toolbar = "Toolbar",
    /**
     * A popup that displays a description for an element when the user passes over or rests on that element.
     * Supplement to the normal tooltip processing of the user agent.
     */
    Tooltip = "Tooltip",
    /**
     * A form of a list (tree) having groups (subtrees) inside groups (subtrees), where the sub trees can be
     * collapsed and expanded.
     */
    Tree = "Tree",
    /**
     * A grid whose rows are expandable and collapsable in the same manner as the ones of trees.
     */
    TreeGrid = "TreeGrid",
    /**
     * A tree node
     */
    TreeItem = "TreeItem",
  }
  /**
   * Configuration options for the colors of a progress bar.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'BarColor'.
   */
  export enum BarColor {
    /**
     * Color: yellow (#faf2b0)
     */
    CRITICAL = "CRITICAL",
    /**
     * Color: red (#ff9a90)
     */
    NEGATIVE = "NEGATIVE",
    /**
     * Color: blue (#b8d0e8)
     */
    NEUTRAL = "NEUTRAL",
    /**
     * Color: green (#b5e7a8)
     */
    POSITIVE = "POSITIVE",
  }
  /**
   * Configuration options for the `BusyIndicator` size.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'BusyIndicatorSize'.
   */
  export enum BusyIndicatorSize {
    /**
     * Type: automatic size detection
     */
    Auto = "Auto",
    /**
     * Type: Large size
     */
    Large = "Large",
    /**
     * Type: Medium size
     */
    Medium = "Medium",
    /**
     * Type: Medium size, specifically if the BusyIndicator is displayed over a page section
     */
    Section = "Section",
    /**
     * Type: small size
     */
    Small = "Small",
  }
  /**
   * Collision behavior: horizontal/vertical.
   *
   * Defines how the position of an element should be adjusted in case it overflows the window in some direction.
   * For both directions this can be "flip", "fit", "flipfit" or "none". If only one behavior is provided
   * it is applied to both directions.
   *
   * Examples: "flip", "fit none", "flipfit fit"
   */
  export type Collision = string;

  /**
   * Enumeration for different lifecycle behaviors of components created by the `ComponentContainer`.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'ComponentLifecycle'.
   */
  export enum ComponentLifecycle {
    /**
     * Application managed lifecycle means that the application takes care to destroy the components associated
     * with the `ComponentContainer`.
     */
    Application = "Application",
    /**
     * Container managed lifecycle means that the `ComponentContainer` takes care to destroy the components
     * associated with the `ComponentContainer` once the `ComponentContainer` is destroyed or a new component
     * is associated.
     */
    Container = "Container",
    /**
     * Legacy lifecycle means that the `ComponentContainer` takes care to destroy the component which is associated
     * with the `ComponentContainer` once the `ComponentContainer` is destroyed, but not when a new component
     * is associated.
     */
    Legacy = "Legacy",
  }
  /**
   * A string type that represents CSS color values (CSS Color Level 3).
   *
   * **Allowed values are:**
   * 	 - Hex colors like `#666666` or `#fff`,
   * 	 - HSL/RGB values with or without transparency, like `hsla(90,10%,30%,0.5)` or `rgb(0,0,0)`,
   * 	 - CSS color names like `darkblue`, or special values like `inherit` and `transparent`,
   * 	 - an empty string, which has the same effect as setting no color.  For more information about
   *     the CSS Level 3 color specification, see {@link https://www.w3.org/TR/css-color-3/#css-system}.
   */
  export type CSSColor = string;

  /**
   * A string type that represents a short hand CSS gap.
   * See:
   * 	{@link https://developer.mozilla.org/en-US/docs/Web/CSS/gap}
   *
   * @since 1.134
   */
  export type CSSGapShortHand = string;

  /**
   * A string type that represents CSS size values.
   *
   * The CSS specifications calls this the `'<length> type'`. Allowed values are CSS sizes like "1px"
   * or "2em" or "50%". The special values `auto` and `inherit` are also accepted as well as mathematical
   * expressions using the CSS3 `calc(expression)` operator. Furthermore, length units representing
   * a percentage of the current viewport dimensions: width (vw), height (vh), the smaller of the two (vmin),
   * or the larger of the two (vmax) can also be defined as a CSS size.
   *
   * Note that CSS does not allow all these values for every CSS property representing a size. E.g. `padding-left`
   * doesn't allow the value `auto`. So even if a value is accepted by `sap.ui.core.CSSSize`, it still might
   * have no effect in a specific context. In other words: UI5 controls usually don't extend the range of
   * allowed values in CSS.
   *
   * **Units**
   *
   * Valid font-relative units are `em, ex` and `rem`. Viewport relative units `vw, vh, vmin, vmax` are also
   * valid. Supported absolute units are `cm, mm, in, pc, pt` and `px`.Other units are not supported yet.
   *
   * **Mathematical Expressions**
   *
   * Expressions inside the `calc()` operator are only roughly checked for validity. Not every value that
   * this type accepts might be a valid expression in the sense of the CSS spec. But vice versa, any expression
   * that is valid according to the spec should be accepted by this type. The current implementation is based
   * on the {@link http://dev.w3.org/csswg/css-values-3/#calc-syntax CSS3 Draft specification from 22 April 2015}.
   *
   * Noteworthy details:
   * 	 - whitespace is mandatory around a '-' or '+' operator and optional otherwise
   * 	 - parentheses are accepted but not checked for being balanced (a restriction of regexp based checks)
   *
   * 	 - semantic constraints like type restrictions are not checked
   *
   * Future versions of UI5 might check `calc()` expressions in more detail, so applications should not assume
   * that a value, that is invalid according to the CSS spec but currently accepted by this type still will
   * be accepted by future versions of this type.
   */
  export type CSSSize = string;

  /**
   * This type checks the short hand form of a margin or padding definition.
   *
   * E.g. "1px 1px" or up to four CSSSize values are allowed or tHe special keyword `inherit`.
   *
   * @since 1.11.0
   */
  export type CSSSizeShortHand = string;

  /**
   * Font design for texts.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'Design'.
   */
  export enum Design {
    /**
     * Mono space font
     */
    Monospace = "Monospace",
    /**
     * Standard font
     */
    Standard = "Standard",
  }
  /**
   * Docking position: horizontal/vertical.
   *
   * Defines a position on the element which is used for aligned positioning of another element (e.g. the
   * left top corner of a popup is positioned at the left bottom corner of the input field). For the horizontal
   * position possible values are "begin", "left", "center", "right" and "end", where left/right always are
   * left and right, or begin/end which are dependent on the text direction. For the vertical position possible
   * values are "top", "center" and "bottom". Examples: "left top", "end bottom", "center center".
   */
  export type Dock = string;

  /**
   * The object contains focus information for input controls.
   *
   * @since 1.111
   * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
   */
  export type FocusInfo = {
    /**
     * The ID of the focused control.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    id?: string;
    /**
     * The position of the cursor.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    cursorPos?: int;
    /**
     * The start position of selection.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    selectionStart?: int;
    /**
     * The end position of selection.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    selectionEnd?: int;
    /**
     * Prevents scrolling.
     *
     * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
     */
    preventScroll?: boolean | undefined;
  };

  /**
   * Configuration options for horizontal alignments of controls.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'HorizontalAlign'.
   */
  export enum HorizontalAlign {
    /**
     * Locale-specific positioning at the beginning of the line
     */
    Begin = "Begin",
    /**
     * Centered alignment of text
     */
    Center = "Center",
    /**
     * Locale-specific positioning at the end of the line
     */
    End = "End",
    /**
     * Hard option for left alignment
     */
    Left = "Left",
    /**
     * Hard option for right alignment
     */
    Right = "Right",
  }
  /**
   * Implementing this interface allows a control to be accessible via access keys.
   *
   * @since 1.104
   * @experimental As of version 1.104.
   */
  export interface IAccessKeySupport {
    __implements__sap_ui_core_IAccessKeySupport: boolean;

    /**
     * Returns a refence to DOM element to be focused during Access key navigation. If not implemented getFocusDomRef()
     * method is used.
     *
     * @since 1.104
     * @experimental As of version 1.104.
     */
    getAccessKeysFocusTarget?(): void;
    /**
     * If implemented called when access keys feature is enabled and highlighting is over
     *
     * @since 1.104
     * @experimental As of version 1.104.
     */
    onAccKeysHighlightEnd?(): void;
    /**
     * If implemented called when access keys feature is enabled and highlighting is ongoing
     *
     * @since 1.104
     * @experimental As of version 1.104.
     */
    onAccKeysHighlightStart?(): void;
  }

  /**
   * Marker interface for subclasses of `sap.ui.core.UIComponent`.
   *
   * Implementing this interface allows a {@link sap.ui.core.UIComponent} to be created fully asynchronously.
   * This interface will implicitly set the component's rootView and router configuration to async. Nested
   * views will also be handled asynchronously. Additionally the error handling during the processing of views
   * is stricter and will fail if a view definition contains errors, e.g. broken binding strings.
   *
   * **Note:** Nested components (via {@link sap.ui.core.ComponentContainer}) are not handled asynchronously
   * by default.
   *
   * When implementing this interface the {@link sap.ui.core.Component.create Component.create} factory's
   * result Promise will resolve once the defined `rootView` is fully processed.
   *
   * An asynchronous component can also return a Promise in its {@link sap.ui.core.UIComponent#createContent createContent }
   * function. This Promise will also be chained into the {@link sap.ui.core.Component.create Component.create }
   * factory's result Promise.
   *
   * See {@link sap.ui.core.UIComponent#createContent} for more details and usage samples.
   *
   * @since 1.89.0
   */
  export interface IAsyncContentCreation {
    __implements__sap_ui_core_IAsyncContentCreation: boolean;
  }

  /**
   * Marker interface for controls that can serve as a menu for a table column header.
   *
   * Implementation of this interface should include the `openBy`, `close`, `isOpen` and `getAriaHasPopupType`
   * methods and fire the `beforeOpen` and `afterClose` events.
   *
   * Refer to the base class {@link sap.m.table.columnmenu.MenuBase} for a detailed API description.
   *
   * @since 1.98
   */
  export interface IColumnHeaderMenu {
    __implements__sap_ui_core_IColumnHeaderMenu: boolean;

    /**
     * Closes the menu.
     *
     * @since 1.126
     */
    close(): void;
    /**
     * Returns the `sap.ui.core.aria.HasPopup` type of the menu.
     *
     * @since 1.98.0
     *
     * @returns `sap.ui.core.aria.HasPopup` type of the menu
     */
    getAriaHasPopupType(): aria.HasPopup;
    /**
     * Determines whether the menu is open.
     *
     * @since 1.126
     *
     * @returns `true` if the menu is open, `false` otherwise
     */
    isOpen(
      /**
       * The element for which the menu is opened. If it is an `HTMLElement`, the closest control is passed for
       * this event (if it exists).
       */
      openBy: UI5Element
    ): boolean;
    /**
     * Opens the menu using the column header.
     *
     * @since 1.98
     */
    openBy(
      /**
       * Specifies the element where the menu is placed.
       */
      oAnchor: Control | HTMLElement
    ): void;
  }

  /**
   * Semantic Colors of an icon.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'IconColor'.
   */
  export enum IconColor {
    /**
     * Contrast color.
     */
    Contrast = "Contrast",
    /**
     * Critical color
     */
    Critical = "Critical",
    /**
     * Default color (brand color)
     */
    Default = "Default",
    /**
     * Color for icon used as a marker
     *
     * @since 1.76
     */
    Marker = "Marker",
    /**
     * Negative color
     */
    Negative = "Negative",
    /**
     * Neutral color.
     */
    Neutral = "Neutral",
    /**
     * Color that indicates an icon which isn't interactive
     *
     * @since 1.76
     */
    NonInteractive = "NonInteractive",
    /**
     * Positive color
     */
    Positive = "Positive",
    /**
     * Color for icon used in a Tile
     *
     * @since 1.76
     */
    Tile = "Tile",
  }
  /**
   * Marker interface for controls that can serve as a context menu.
   *
   * Implementation of this interface should implement the `openAsContextMenu` method.
   */
  export interface IContextMenu {
    __implements__sap_ui_core_IContextMenu: boolean;

    /**
     * Opens the control by given opener ref.
     */
    openAsContextMenu(
      /**
       * An `oncontextmenu` event object or an object with properties left, top, offsetX, offsetY
       */
      oEvent:
        | jQuery.Event
        | {
            left: float;

            top: float;

            offsetX: float;

            offsetY: float;
          },
      /**
       * The element which will get the focus back again after the menu was closed
       */
      oOpenerRef: UI5Element | HTMLElement
    ): void;
  }

  /**
   * A string type representing an ID or a name.
   *
   * Allowed is a sequence of characters (capital/lowercase), digits, underscores, dashes, points and/or colons.
   * It may start with a character or underscore only.
   */
  export type ID = string;

  /**
   * Marker interface for controls that can be used as content of `sap.ui.layout.form.Form` or `sap.ui.layout.form.SimpleForm`.
   *
   * If the control's width must not be adjusted by the `Form` control to meet the cell's width, the control
   * must implement the `getFormDoNotAdjustWidth` function and return `true`.
   *
   * @since 1.48.0
   */
  export interface IFormContent {
    __implements__sap_ui_core_IFormContent: boolean;

    /**
     * Whether a control wants to keep its original width even when used in a `Form`.
     *
     * In the `Form` control, all content controls are positioned on a grid cell base. By default, the controls
     * use the full width of the used grid cell. But for some controls (like image controls), this is not the
     * desired behavior. In this case the control must keep its original width.
     *
     * This is an optional method. When not defined, the width of the control might be adjusted.
     *
     * @since 1.48.0
     *
     * @returns true if the `Form` is not allowed to adjust the width of the control to use the cell's width
     */
    getFormDoNotAdjustWidth?(): boolean;
  }

  /**
   * Defines a control, which can specify if it can be bound to a label
   *
   * @since 1.121.0
   */
  export interface ILabelable {
    __implements__sap_ui_core_ILabelable: boolean;

    /**
     * Returns if the control can be bound to a label
     *
     * @since 1.121.0
     *
     * @returns `true` if the control can be bound to a label
     */
    hasLabelableHTMLElement(): boolean;
  }

  /**
   * State of the Input Method Editor (IME) for the control.
   *
   * Depending on its value, it allows users to enter and edit for example Chinese characters.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'ImeMode'.
   */
  export enum ImeMode {
    /**
     * IME is used for entering characters.
     */
    Active = "Active",
    /**
     * The value is automatically computed by the user agent.
     */
    Auto = "Auto",
    /**
     * IME is disabled.
     */
    Disabled = "Disabled",
    /**
     * IME is not used for entering characters.
     */
    Inactive = "Inactive",
  }
  /**
   * Colors to highlight certain UI elements.
   *
   * In contrast to the `ValueState`, the semantic meaning must be defined by the application.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'IndicationColor'.
   * See:
   * 	{@link fiori:/how-to-use-semantic-colors/ Semantic Colors}
   *
   * @since 1.62.0
   */
  export enum IndicationColor {
    /**
     * Indication Color 1
     */
    Indication01 = "Indication01",
    /**
     * Indication Color 2
     */
    Indication02 = "Indication02",
    /**
     * Indication Color 3
     */
    Indication03 = "Indication03",
    /**
     * Indication Color 4
     */
    Indication04 = "Indication04",
    /**
     * Indication Color 5
     */
    Indication05 = "Indication05",
    /**
     * Indication Color 6
     *
     * @since 1.75
     */
    Indication06 = "Indication06",
    /**
     * Indication Color 7
     *
     * @since 1.75
     */
    Indication07 = "Indication07",
    /**
     * Indication Color 8
     *
     * @since 1.75
     */
    Indication08 = "Indication08",
    /**
     * Indication Color 9
     *
     * @since 1.120
     */
    Indication09 = "Indication09",
    /**
     * Indication Color 10
     *
     * @since 1.120
     */
    Indication10 = "Indication10",
    /**
     * Indication Color 11
     *
     * @since 1.120
     */
    Indication11 = "Indication11",
    /**
     * Indication Color 12
     *
     * @since 1.120
     */
    Indication12 = "Indication12",
    /**
     * Indication Color 13
     *
     * @since 1.120
     */
    Indication13 = "Indication13",
    /**
     * Indication Color 14
     *
     * @since 1.120
     */
    Indication14 = "Indication14",
    /**
     * Indication Color 15
     *
     * @since 1.120
     */
    Indication15 = "Indication15",
    /**
     * Indication Color 16
     *
     * @since 1.120
     */
    Indication16 = "Indication16",
    /**
     * Indication Color 17
     *
     * @since 1.120
     */
    Indication17 = "Indication17",
    /**
     * Indication Color 18
     *
     * @since 1.120
     */
    Indication18 = "Indication18",
    /**
     * Indication Color 19
     *
     * @since 1.120
     */
    Indication19 = "Indication19",
    /**
     * Indication Color 20
     *
     * @since 1.120
     */
    Indication20 = "Indication20",
  }
  /**
   * Enumeration for different mode behaviors of the `InvisibleMessage`.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'InvisibleMessageMode'.
   *
   * @since 1.78
   */
  export enum InvisibleMessageMode {
    /**
     * Indicates that updates to the region have the highest priority and should be presented to the user immediately.
     */
    Assertive = "Assertive",
    /**
     * Indicates that updates to the region should be presented at the next graceful opportunity, such as at
     * the end of reading the current sentence, or when the user pauses typing.
     */
    Polite = "Polite",
  }
  /**
   * Marker interface for container controls.
   *
   * Implementing this interface allows a container control to display a {@link sap.ui.core.Placeholder}.
   * This requires the container control to implement the `showPlaceholder` and `hidePlaceholder` methods.
   *
   * Optionally, the `needPlaceholder` method can be implemented to defined, whether a placeholder is needed
   * or not. If implemented, this method must return a `boolean`. Depending on the return value, `showPlaceholder`
   * will be called or not.
   *
   * @since 1.92.0
   */
  export interface IPlaceholderSupport {
    __implements__sap_ui_core_IPlaceholderSupport: boolean;
  }

  /**
   * Marker interface for controls that can be used as content of {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement}.
   *
   * If the value-holding property of the control is not `valuetext`, the name of the value-holding
   * property must be returned in the {@link sap.ui.core.ISemanticFormContent.getFormValueProperty getFormValueProperty }
   * function.
   *
   * If the value of the control needs some special output formatting (to show a description instead of a
   * key), this formatted text needs to be returned in the {@link sap.ui.core.ISemanticFormContent.getFormFormattedValue getFormFormattedValue }
   * function.
   *
   * @since 1.86.0
   */
  export interface ISemanticFormContent {
    __implements__sap_ui_core_ISemanticFormContent: boolean;

    /**
     * Returns the formatted value of a control used in a {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement}.
     *
     * In the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement} element, the assigned fields
     * are rendered in edit mode. In display mode, a text is rendered that concatenates the values of all assigned
     * fields. In some cases the displayed text does not match the value of the field and needs some formatting.
     * In other cases the control does not have a `value` property, so the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement }
     * element cannot determine the value.
     *
     * This is an optional method. If not defined, the `value` property or the `text` property is used to determine
     * the value.
     *
     * @since 1.86.0
     *
     * @returns Formatted value or a `Promise` returning the formatted value if resolved
     */
    getFormFormattedValue?(): string | Promise<string>;
    /**
     * Returns the names of the properties of a control that might update the rendering in a {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement}.
     *
     * In the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement} element, the assigned fields
     * are rendered in edit mode. In display mode, depending on {@link sap.ui.core.ISemanticFormContent.getFormRenderAsControl getFormRenderAsControl},
     * either a text is rendered, which concatenates the values of all assigned fields, or the control is rendered.
     * So if a property of the control changes that might lead to a different rendering (some controls have
     * a special rendering in display mode), the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement }
     * needs to check the rendering.
     *
     * This is an optional method. If not defined, no check for updates (only for property defined in {@link sap.ui.core.ISemanticFormContent.getFormValueProperty getFormValueProperty})
     * is done once the control has been assigned.
     *
     * @since 1.117.0
     *
     * @returns Name of the properties
     */
    getFormObservingProperties?(): string[];
    /**
     * If set to `true`, the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement} also renders
     * the control in display mode, if the used {@link sap.ui.layout.form.FormLayout FormLayout} supports this.
     *
     * This is an optional method. If not defined, just the text is rendered.
     *
     * @since 1.117.0
     *
     * @returns Name of the value-holding property
     */
    getFormRenderAsControl?(): string;
    /**
     * Returns the name of the value-holding property of a control used in a {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement}.
     *
     * In the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement} element, the assigned fields
     * are rendered in edit mode. In display mode, a text is rendered that concatenates the values of all assigned
     * fields. So the concatenated text needs to be updated if the value of a control changes. If a control
     * does not have a `value` property, the {@link sap.ui.layout.form.SemanticFormElement SemanticFormElement }
     * element needs to know the property it has to listen to for changes.
     *
     * This is an optional method. If not defined, the `value` property or the `text` property is used to determine
     * the value.
     *
     * @since 1.86.0
     *
     * @returns Name of the value-holding property
     */
    getFormValueProperty?(): string;
  }

  /**
   * Interface for the controls which are suitable to shrink.
   *
   * This means the control should still look fine when it gets smaller than its normal size, e.g. Text controls
   * which can show ellipsis in case of shrink.
   *
   * Note: This marker interface can be implemented by controls to give a hint to the container. The control
   * itself does not need to implement anything. A parent control that respects this interface will apply
   * the "flex-shrink" as a CSS property which determines how much the item will shrink relative to the rest
   * of the items in the container when negative free space is distributed.
   *
   * @since 1.26
   */
  export interface IShrinkable {
    __implements__sap_ui_core_IShrinkable: boolean;
  }

  /**
   * Defines the selection mode of the menu items.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'ItemSelectionMode'.
   *
   * @since 1.127.0
   */
  export enum ItemSelectionMode {
    /**
     * Multi selection mode (more than one menu item can be selected).
     */
    MultiSelect = "undefined",
    /**
     * No selection mode.
     */
    None = "undefined",
    /**
     * Single selection mode (only one menu item can be selected).
     */
    SingleSelect = "undefined",
  }
  /**
   * Marker interface for controls that can be used in `content` aggregation of the `sap.m.Title` control.
   *
   * @since 1.87
   */
  export interface ITitleContent {
    __implements__sap_ui_core_ITitleContent: boolean;
  }

  /**
   * Marker interface for controls which are suitable for use as label.
   */
  export interface Label {
    __implements__sap_ui_core_Label: boolean;
  }

  export {
    /**
     * Specifies possible message types.
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'MessageType'.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/core/message/MessageType} instead.
     */
    MessageType1 as MessageType,
  };

  /**
   * Defines the different possible states of an element that can be open or closed and does not only toggle
   * between these states, but also spends some time in between (e.g. because of an animation).
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'OpenState'.
   */
  export enum OpenState {
    /**
     * Closed and currently not changing states.
     */
    CLOSED = "CLOSED",
    /**
     * Still open, but in the process of going to the CLOSED state.
     */
    CLOSING = "CLOSING",
    /**
     * Open and currently not changing states.
     */
    OPEN = "OPEN",
    /**
     * Already left the CLOSED state, is not OPEN yet, but in the process of getting OPEN.
     */
    OPENING = "OPENING",
  }
  /**
   * Orientation of a UI element.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'Orientation'.
   *
   * @since 1.22
   */
  export enum Orientation {
    /**
     * Arrange Horizontally
     */
    Horizontal = "Horizontal",
    /**
     * Arrange Vertically
     */
    Vertical = "Vertical",
  }
  /**
   * A string type that represents a percentage value.
   */
  export type Percentage = string;

  /**
   * Marker interface for controls that are not rendered "embedded" into other controls but need to be opened/closed.
   *
   * Such controls are handled differently during rendering.
   *
   * @since 1.19.0
   */
  export interface PopupInterface {
    __implements__sap_ui_core_PopupInterface: boolean;
  }

  /**
   * Priorities for general use.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'Priority'.
   */
  export enum Priority {
    /**
     * High priority
     */
    High = "High",
    /**
     * Low priority
     */
    Low = "Low",
    /**
     * Medium priority
     */
    Medium = "Medium",
    /**
     * Default, none priority
     */
    None = "None",
  }
  /**
   * Actions are: Click on track, button, drag of thumb, or mouse wheel click.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'ScrollBarAction'.
   */
  export enum ScrollBarAction {
    /**
     * Scrolling done by dragging the scroll bar's paint thumb
     */
    Drag = "Drag",
    /**
     * Scrolling done by mouse wheel
     */
    MouseWheel = "MouseWheel",
    /**
     * Range scrolling caused by clicking track area or using page up or page down key.
     */
    Page = "Page",
    /**
     * Single step scrolling caused by clicking an arrow button or arrow key.
     */
    Step = "Step",
  }
  /**
   * Defines the possible values for horizontal and vertical scrolling behavior.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'Scrolling'.
   */
  export enum Scrolling {
    /**
     * A scroll bar is shown if the content requires more space than the given space (rectangle) provides.
     */
    Auto = "Auto",
    /**
     * No scroll bar is shown, and the content stays in the given rectangle.
     */
    Hidden = "Hidden",
    /**
     * No scroll bar provided even if the content is larger than the available space.
     */
    None = "None",
    /**
     * A scroll bar is always shown even if the space is large enough for the current content.
     */
    Scroll = "Scroll",
  }
  /**
   * Sort order of a column.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'SortOrder'.
   *
   * @since 1.61.0
   */
  export enum SortOrder {
    /**
     * Sorting is done in ascending order.
     */
    Ascending = "Ascending",
    /**
     * Sorting is done in descending order.
     */
    Descending = "Descending",
    /**
     * Sorting is not applied.
     */
    None = "None",
  }
  /**
   * Configuration options for text alignments.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'TextAlign'.
   */
  export enum TextAlign {
    /**
     * Locale-specific positioning at the beginning of the line.
     */
    Begin = "Begin",
    /**
     * Centered text alignment.
     */
    Center = "Center",
    /**
     * Locale-specific positioning at the end of the line.
     */
    End = "End",
    /**
     * Sets no text align, so the browser default is used.
     *
     * @since 1.26.0
     */
    Initial = "Initial",
    /**
     * Hard option for left alignment.
     */
    Left = "Left",
    /**
     * Hard option for right alignment.
     */
    Right = "Right",
  }
  /**
   * Configuration options for the direction of texts.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'TextDirection'.
   */
  export enum TextDirection {
    /**
     * Inherits the direction from its parent control/container.
     */
    Inherit = "Inherit",
    /**
     * Specifies left-to-right text direction.
     */
    LTR = "LTR",
    /**
     * Specifies right-to-left text direction.
     */
    RTL = "RTL",
  }
  /**
   * Level of a title.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'TitleLevel'.
   *
   * @since 1.9.1
   */
  export enum TitleLevel {
    /**
     * The level of the title is choosen by the control rendering the title.
     */
    Auto = "Auto",
    /**
     * The Title is of level 1.
     */
    H1 = "H1",
    /**
     * The Title is of level 2
     */
    H2 = "H2",
    /**
     * The Title is of level 3
     */
    H3 = "H3",
    /**
     * The Title is of level 4
     */
    H4 = "H4",
    /**
     * The Title is of level 5
     */
    H5 = "H5",
    /**
     * The Title is of level 6
     */
    H6 = "H6",
  }
  /**
   * Marker interface for toolbar controls.
   *
   * @since 1.21.0
   */
  export interface Toolbar {
    __implements__sap_ui_core_Toolbar: boolean;
  }

  /**
   * A string type that represents an RFC 3986 conformant URI.
   */
  export type URI = string;

  /**
   * Marker for the correctness of the current value.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'ValueState'.
   * See:
   * 	{@link fiori:/how-to-use-semantic-colors/ Semantic Colors}
   *
   * @since 1.0
   */
  export enum ValueState {
    /**
     * State is not valid.
     */
    Error = "Error",
    /**
     * State is informative.
     *
     * @since 1.61
     */
    Information = "Information",
    /**
     * State is not specified.
     */
    None = "None",
    /**
     * State is valid.
     */
    Success = "Success",
    /**
     * State is valid but with a warning.
     */
    Warning = "Warning",
  }
  /**
   * Configuration options for vertical alignments, for example of a layout cell content within the borders.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'VerticalAlign'.
   */
  export enum VerticalAlign {
    /**
     * Content is aligned at the bottom.
     */
    Bottom = "Bottom",
    /**
     * Content respect the parent's vertical alignment.
     */
    Inherit = "Inherit",
    /**
     * Content is centered vertically .
     */
    Middle = "Middle",
    /**
     * Content is aligned at the top.
     */
    Top = "Top",
  }
  /**
   * Configuration options for text wrapping.
   *
   * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'Wrapping'.
   */
  export enum Wrapping {
    /**
     * Inserts actual line breaks in the text at the wrap point.
     */
    Hard = "Hard",
    /**
     * The standard browser behavior is considered for wrapping.
     */
    None = "None",
    /**
     * Wrapping shall not be allowed.
     */
    Off = "Off",
    /**
     * The text is actually on the same line but displayed within several lines.
     */
    Soft = "Soft",
  }

  export namespace aria {
    /**
     * Types of popups to set as aria-haspopup attribute. Most of the values (except "None") of the enumeration
     * are taken from the ARIA specification: https://www.w3.org/TR/wai-aria/#aria-haspopup
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'aria.HasPopup'.
     *
     * @since 1.84
     */
    enum HasPopup {
      /**
       * Dialog popup type.
       */
      Dialog = "Dialog",
      /**
       * Grid popup type.
       */
      Grid = "Grid",
      /**
       * ListBox popup type.
       */
      ListBox = "ListBox",
      /**
       * Menu popup type.
       */
      Menu = "Menu",
      /**
       * None - the aria-haspopup attribute will not be rendered.
       */
      None = "None",
      /**
       * Tree popup type.
       */
      Tree = "Tree",
    }
  }

  export namespace dnd {
    /**
     * Marker interface for drag configuration providing information about the source of the drag operation.
     *
     * @since 1.52.0
     */
    interface IDragInfo {
      __implements__sap_ui_core_dnd_IDragInfo: boolean;
    }

    /**
     * Marker interface for drop configuration providing information about the target of the drop operation.
     *
     * @since 1.52.0
     */
    interface IDropInfo {
      __implements__sap_ui_core_dnd_IDropInfo: boolean;
    }

    /**
     * Configuration options for visual drop effects that are given during a drag and drop operation.
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'dnd.DropEffect'.
     *
     * @since 1.52.0
     */
    enum DropEffect {
      /**
       * A copy of the source item is made at the new location.
       */
      Copy = "Copy",
      /**
       * A link is established to the source at the new location.
       */
      Link = "Link",
      /**
       * An item is moved to a new location.
       */
      Move = "Move",
      /**
       * The item cannot be dropped.
       */
      None = "None",
    }
    /**
     * Configuration options for the layout of the droppable controls.
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'dnd.DropLayout'.
     *
     * @since 1.52.0
     */
    enum DropLayout {
      /**
       * Default {@link sap.ui.core.Element.extend layout} definition of the aggregations.
       */
      Default = "Default",
      /**
       * Droppable controls are arranged horizontally.
       */
      Horizontal = "Horizontal",
      /**
       * Droppable controls are arranged vertically.
       */
      Vertical = "Vertical",
    }
    /**
     * Configuration options for drop positions.
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'dnd.DropPosition'.
     *
     * @since 1.52.0
     */
    enum DropPosition {
      /**
       * Drop between the controls.
       */
      Between = "Between",
      /**
       * Drop on the control.
       */
      On = "On",
      /**
       * Drop on the control or between the controls.
       */
      OnOrBetween = "OnOrBetween",
    }
    /**
     * Drop positions relative to a dropped element.
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'dnd.RelativeDropPosition'.
     *
     * @since 1.100.0
     */
    enum RelativeDropPosition {
      /**
       * Drop after the control.
       */
      After = "After",
      /**
       * Drop before the control.
       */
      Before = "Before",
      /**
       * Drop on the control.
       */
      On = "On",
    }
  }

  export namespace mvc {
    /**
     * Marker interface for a ControllerExtension.
     *
     * @since 1.56.0
     */
    interface IControllerExtension {
      __implements__sap_ui_core_mvc_IControllerExtension: boolean;
    }
  }

  export namespace routing {
    /**
     * Enumeration for different HistoryDirections.
     *
     * This enum is part of the 'sap/ui/core/library' module export and must be accessed by the property 'routing.HistoryDirection'.
     */
    enum HistoryDirection {
      /**
       * The page has already been navigated to and it was the predecessor of the previous page.
       */
      Backwards = "Backwards",
      /**
       * The page has already been navigated to and it was the successor of the previous page.
       */
      Forwards = "Forwards",
      /**
       * A new entry is added to the history.
       */
      NewEntry = "NewEntry",
      /**
       * A navigation took place, but it could be any of the other three states.
       */
      Unknown = "Unknown",
    }
  }
}

declare module "sap/ui/core/CustomStyleClassSupport" {
  /**
   * Applies the support for custom style classes on the prototype of a `sap.ui.core.Element`.
   *
   * All controls (subclasses of `sap.ui.core.Control`) provide the support custom style classes. The control
   * API provides functions to the application which allow it to add, remove or change style classes for the
   * control. In general, this option is not available for elements because elements do not necessarily have
   * a representation in the DOM.
   *
   * This function can be used by a control developer to explicitly enrich the API of his/her element implementation
   * with the API functions for the custom style class support. It must be called on the prototype of the
   * element.
   *
   * **Usage Example:**
   * ```javascript
   *
   * sap.ui.define(['sap/ui/core/Element', 'sap/ui/core/CustomStyleClassSupport'], function(Element, CustomStyleClassSupport) {
   *    "use strict";
   *    var MyElement = Element.extend("my.MyElement", {
   *       metadata : {
   *          //...
   *       }
   *       //...
   *    });
   *
   *    CustomStyleClassSupport.apply(MyElement.prototype);
   *
   *    return MyElement;
   * }, true);
   * ```
   *
   *
   * The classes are written to the HTML automatically when using the {@link sap.ui.core.RenderManager Semantic Rendering API}.
   * To ensure that the classes are written to the HTML with the traditional string-based rendering, when
   * writing the root tag of the element you must call the function `oRenderManager.writeClasses(oElement);`
   * ({@link sap.ui.core.RenderManager#writeClasses}) within the renderer of the control to which the element
   * belongs.
   *
   * This function adds the following functions to the elements prototype:
   * 	 - `addStyleClass`: {@link sap.ui.core.Control#addStyleClass}
   * 	 - `removeStyleClass`: {@link sap.ui.core.Control#removeStyleClass}
   * 	 - `toggleStyleClass`: {@link sap.ui.core.Control#toggleStyleClass}
   * 	 - `hasStyleClass`: {@link sap.ui.core.Control#hasStyleClass}  In addition the clone function of
   *     the element is extended to ensure that the custom style classes are also available on the cloned element.
   *
   * **Note:** This function can only be used within control development. An application cannot add
   * style class support on existing elements by calling this function.
   */
  export default function CustomStyleClassSupport(): void;
}

declare module "sap/ui/core/AppCacheBuster" {
  /**
   * The AppCacheBuster is used to hook into URL relevant functions in jQuery and SAPUI5 and rewrite the URLs
   * with a timestamp segment. The timestamp information is fetched from the server and used later on for
   * the URL rewriting.
   */
  interface AppCacheBuster {
    /**
     * Converts the given URL if it matches a URL in the cachebuster index. If not then the same URL will be
     * returned. To prevent URLs from being modified by the application cachebuster you can implement the function
     * `sap.ui.core.AppCacheBuster.handleURL`.
     *
     *
     * @returns modified URL when matching the index or unmodified when not
     */
    convertURL(
      /**
       * any URL
       */
      sUrl: string
    ): string;
    /**
     * Callback function which can be overwritten to programmatically decide whether to rewrite the given URL
     * or not.
     *
     *
     * @returns `true` to rewrite or `false` to ignore
     */
    handleURL(
      /**
       * any URL
       */
      sUrl: string
    ): boolean;
    /**
     * Normalizes the given URL and make it absolute.
     *
     *
     * @returns normalized URL
     */
    normalizeURL(
      /**
       * any URL
       */
      sUrl: string
    ): string;
    /**
     * Registers an application. Loads the cachebuster index file from this locations. All registered files
     * will be considered by the cachebuster and the URLs will be prefixed with the timestamp of the index file.
     */
    register(
      /**
       * base URL of an application providing a cachebuster index file
       */
      sBaseUrl: string
    ): void;
  }
  const AppCacheBuster: AppCacheBuster;
  export default AppCacheBuster;
}

declare module "sap/ui/core/BusyIndicator" {
  /**
   * Provides methods to show or hide a waiting animation covering the whole page and blocking user interaction.
   */
  interface BusyIndicator {
    /**
     * Registers a handler for the {@link #event:Close Close} event.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to `sap.ui.core.BusyIndicator`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachClose(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with; defaults to `sap.ui.core.BusyIndicator`
       */
      oListener?: object
    ): this;
    /**
     * Registers a handler for the {@link #event:Open Open} event.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to `sap.ui.core.BusyIndicator`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachOpen(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with; defaults to `sap.ui.core.BusyIndicator`
       */
      oListener?: object
    ): this;
    /**
     * Unregisters a handler from the {@link #event:Close Close} event.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachClose(
      /**
       * The callback function to unregister
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Unregisters a handler from the {@link #event:Open Open} event.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachOpen(
      /**
       * The callback function to unregister
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Removes the BusyIndicator from the screen.
     */
    hide(): void;
    /**
     * Displays the `BusyIndicator` and starts blocking all user input. This only happens after some delay,
     * and if, after that delay, the `BusyIndicator.hide()` has not yet been called in the meantime.
     *
     * There is a certain default value for the delay, which can be overridden.
     */
    show(
      /**
       * The delay in milliseconds before opening the `BusyIndicator`; It is not opened if `hide()` is called
       * before the delay ends. If no delay (or no valid delay) is given, a delay of 1000 milliseconds is used.
       */
      iDelay?: int
    ): void;
  }
  const BusyIndicator: BusyIndicator;
  export default BusyIndicator;
}

declare module "sap/ui/core/CalendarType" {
  import CalendarType1 from "sap/base/i18n/date/CalendarType";

  export {
    /**
     * The types of `Calendar`.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/date/CalendarType} instead.
     */
    CalendarType1 as default,
  };
}

declare module "sap/ui/core/CommandExecution" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * @since 1.70
   */
  export default class CommandExecution extends UI5Element {
    /**
     * Creates and initializes a new CommandExecution.
     *
     * The CommandExecution registers a shortcut when it is added to the dependent aggregation of a control.
     * The shortcut information is retrieved from the owner components manifest (`/sap.ui5/commands/<command>`).
     *
     * You can use a CommandExecution instead of an event handler in XMLViews by using `cmd:` plus the command
     * name.
     *
     * Example for `sap.m.Button`:
     *
     *
     * ```javascript
     *
     * <Button press="cmd:MyCommand" />
     * ```
     *
     *
     * When the press event is fired, the CommandExecution will be triggered and the `execute` event is fired.
     *
     * When using commands, the component will create a model named `$cmd`. The model data provides the enabled
     * and visible state of all CommandExecutions. With that, action-triggering controls (e.g. a button) can
     * be bound to the enable/visible property of the CommandExecution to centrally control their state.
     *
     * **Note: The usage of the `$cmd` model is restricted to `sap.suite.ui.generic`**
     *
     * When binding a button's enabled state to this model, it follows the enabled state of the CommandExecution.
     * The binding path must be relative like `myCommand/enabled`:
     *
     *
     * ```javascript
     *
     * <Button press="cmd:MyCommand" enabled="$cmd>MyCommand/enabled" />
     * ```
     *
     *
     * A CommandExecution can have three states:
     * 	 - the CommandExecution is visible and enabled. If the configured shortcut is executed, the configured
     *     event handler of this CommandExecution is called
     * 	 - the CommandExecution is visible but not enabled. If the configured shortcut is executed, neither
     *     the configured event handler of this CommandExecution nor any event handler configured on CommandExecutions
     *     in the ancestor chain is called
     * 	 - the CommandExecution is not visible. If the configured shortcut is executed, the configured event
     *     handler of this CommandExecution is not called, but the event is propagated to its parent, which can
     *     then handle the event by a configured CommandExecution or propagate the event to its parent, until no
     *     parent exits anymore and the browser can handle the executed shortcut
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.ui.core.CommandExecution with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CommandExecution>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.CommandExecution.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:execute execute} event of this `sap.ui.core.CommandExecution`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.CommandExecution` itself.
     *
     * Execute will be fired when the CommandExecution will be triggered.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachExecute(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.CommandExecution` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:execute execute} event of this `sap.ui.core.CommandExecution`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.CommandExecution` itself.
     *
     * Execute will be fired when the CommandExecution will be triggered.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachExecute(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.CommandExecution` itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:execute execute} event of this `sap.ui.core.CommandExecution`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachExecute(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:execute execute} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireExecute(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getCommand command}.
     *
     * The command's name, that has to be defined in the manifest. This property can only be applied initially.
     *
     *
     * @returns Value of property `command`
     */
    getCommand(): string;
    /**
     * Gets current value of property {@link #getEnabled enabled}.
     *
     * Whether the CommandExecution is enabled or not. By default, it is enabled. If the CommandExecution is
     * disabled, the CommandExecution processes the event, but the event handler for it will not be called.
     * Therefore, also no event handler configured on CommandExecutions on ancestors is called.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `enabled`
     */
    getEnabled(): boolean;
    /**
     * Gets current value of property {@link #getVisible visible}.
     *
     * Whether the CommandExecution is visible, or not. By default, it is visible. If not visible, the CommandExecution
     * won't process the event, and the event handler for it will not be called, regardless of the enabled state.
     * Therefore, the configured event handler on the next CommandExecution in the ancestor chain or, if no
     * ancestor in the ancestor chain has any CommandExecutions configured for this shortcut, the event handler
     * of the browser is called.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `visible`
     */
    getVisible(): boolean;
    /**
     * Sets whether the `CommandExecution` is enabled, or not. If set to false, the `CommandExecution` will
     * still register the shortcut. This will block any configured CommandExecutions on any ancestors for that
     * shortcut.
     *
     *
     * @returns The CommandExecution
     */
    setEnabled(
      /**
       * Whether the CommandExecution is enabled, or not.
       */
      bValue: boolean
    ): UI5Element;
    /**
     * Sets whether the `CommandExecution` is visible, or not. If set to false, the `CommandExecution` will
     * unregister the shortcut. If not visible, the CommandExecution will not be triggered even if it is enabled.
     *
     *
     * @returns The CommandExecution
     */
    setVisible(
      /**
       * Whether the CommandExecution is visible, or not.
       */
      bValue: boolean
    ): UI5Element;
    /**
     * Fires the execute event and triggers the attached handler. If the CommandExecution is disabled, the handler
     * will not be triggered.
     */
    trigger(): void;
  }
  /**
   * Describes the settings that can be provided to the CommandExecution constructor.
   */
  export interface $CommandExecutionSettings extends $ElementSettings {
    /**
     * The command's name, that has to be defined in the manifest. This property can only be applied initially.
     */
    command?: string | PropertyBindingInfo;

    /**
     * Whether the CommandExecution is enabled or not. By default, it is enabled. If the CommandExecution is
     * disabled, the CommandExecution processes the event, but the event handler for it will not be called.
     * Therefore, also no event handler configured on CommandExecutions on ancestors is called.
     */
    enabled?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Whether the CommandExecution is visible, or not. By default, it is visible. If not visible, the CommandExecution
     * won't process the event, and the event handler for it will not be called, regardless of the enabled state.
     * Therefore, the configured event handler on the next CommandExecution in the ancestor chain or, if no
     * ancestor in the ancestor chain has any CommandExecutions configured for this shortcut, the event handler
     * of the browser is called.
     */
    visible?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Execute will be fired when the CommandExecution will be triggered.
     */
    execute?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the CommandExecution#execute event.
   */
  export interface CommandExecution$ExecuteEventParameters {}

  /**
   * Event object of the CommandExecution#execute event.
   */
  export type CommandExecution$ExecuteEvent = Event<
    CommandExecution$ExecuteEventParameters,
    CommandExecution
  >;
}

declare module "sap/ui/core/Component" {
  import {
    default as ManagedObject,
    MetadataOptions as MetadataOptions1,
    $ManagedObjectSettings,
  } from "sap/ui/base/ManagedObject";

  import { ID } from "sap/ui/core/library";

  import EventBus from "sap/ui/core/EventBus";

  import Manifest from "sap/ui/core/Manifest";

  import ComponentMetadata from "sap/ui/core/ComponentMetadata";

  /**
   * Base Class for Components. Components are independent and reusable parts of UI5 applications. They facilitate
   * the encapsulation of closely related parts of an application, thus enabling developers to structure and
   * maintain their applications more easily.
   *
   * @since 1.9.2
   */
  export default abstract class Component extends ManagedObject {
    /**
     * As `Component` is an abstract base class for components, applications should not call the constructor.
     * For many use cases the static {@link #.create Component.create} factory can be used to instantiate a
     * `Component`. Depending on the requirements, the framework also provides other ways to instantiate a `Component`,
     * documented under the {@link https://ui5.sap.com/#/topic/958ead51e2e94ab8bcdc90fb7e9d53d0 "Component" }
     * chapter.
     *
     * The set of allowed entries in the `mSettings` object depends on the concrete subclass and is described
     * there. See {@link sap.ui.core.Component} for a general description of this argument.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.ui.base.ManagedObject#constructor sap.ui.base.ManagedObject }
     * can be used.
     */
    constructor(
      /**
       * Optional object with initial settings for the new Component instance
       */
      mSettings?: $ComponentSettings
    );
    /**
     * As `Component` is an abstract base class for components, applications should not call the constructor.
     * For many use cases the static {@link #.create Component.create} factory can be used to instantiate a
     * `Component`. Depending on the requirements, the framework also provides other ways to instantiate a `Component`,
     * documented under the {@link https://ui5.sap.com/#/topic/958ead51e2e94ab8bcdc90fb7e9d53d0 "Component" }
     * chapter.
     *
     * The set of allowed entries in the `mSettings` object depends on the concrete subclass and is described
     * there. See {@link sap.ui.core.Component} for a general description of this argument.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.ui.base.ManagedObject#constructor sap.ui.base.ManagedObject }
     * can be used.
     */
    constructor(
      /**
       * Optional ID for the new control; generated automatically if no non-empty ID is given. Note: this can
       * be omitted, no matter whether `mSettings` are given or not!
       */
      sId?: string,
      /**
       * Optional object with initial settings for the new Component instance
       */
      mSettings?: $ComponentSettings
    );
    /**
     * Registry of all `Component`s that currently exist.
     *
     * @since 1.67
     * @deprecated As of version 1.120. Use {@link module:sap/ui/core/ComponentRegistry} instead.
     */
    static registry: registry;

    /**
     * Asynchronously creates a new component instance from the given configuration.
     *
     * If the component class does not already exists, the component class is loaded and afterwards the new
     * component instance is created.
     *
     * To optimize the loading process, additional `asyncHints` can be provided. The structure of these hints
     * and how they impact the loading of components is an internal feature of this API and reserved for UI5
     * internal use only. Code that wants to be safe wrt. version updates, should not use the `asyncHints` property.
     *
     * If Components and/or libraries are listed in the `asyncHints`, all the corresponding preload files will
     * be requested in parallel, loading errors (404s) will be ignored. The constructor class will only be required
     * after all preloads have been rejected or resolved. Only then, the new instance will be created.
     *
     * A component can implement the {@link sap.ui.core.IAsyncContentCreation} interface. Please see the respective
     * documentation for more information.
     *
     * @since 1.56.0
     *
     * @returns A Promise that resolves with the newly created component instance
     */
    static create(
      /**
       * Configuration options
       */
      mOptions: {
        /**
         * Name of the component to load, this is the dot-separated name of the package that contains the Component.js
         * module; Even when an alternative location is specified from which the manifest should be loaded (`mOptions.manifest`
         * is set to a non-empty string), then the name specified in that manifest will be ignored and this name
         * will be used instead to determine the module to be loaded.
         */
        name: string;
        /**
         * Alternative location from where to load the Component. If `mOptions.manifest` is set to a non-empty string,
         * this URL specifies the location of the final component defined via that manifest, otherwise it specifies
         * the location of the component defined via its name `mOptions.name`.
         */
        url?: string;
        /**
         * Initial data of the Component, see {@link sap.ui.core.Component#getComponentData}.
         */
        componentData?: object;
        /**
         * ID of the new Component
         */
        id?: ID;
        /**
         * List of active terminologies. The order of the given active terminologies is significant. The {@link module:sap/base/i18n/ResourceBundle ResourceBundle }
         * API documentation describes the processing behavior in more detail. Please have a look at this dev-guide
         * chapter for general usage instructions: {@link https://ui5.sap.com/#/topic/eba8d25a31ef416ead876e091e67824e Text Verticalization}.
         */
        activeTerminologies?: string[];
        /**
         * Settings of the new Component
         */
        settings?: $ComponentSettings;
        /**
         * Whether and from where to load the manifest.json for the Component. When set to any truthy value, the
         * manifest will be loaded and evaluated before the Component controller. If it is set to a falsy value,
         * the manifest will not be evaluated before the controller. It might still be loaded synchronously if declared
         * in the Component metadata. A non-empty string value will be interpreted as the URL to load the manifest
         * from. If the manifest could not be loaded from a given URL, the Promise returned by the Component.create
         * factory rejects. A non-null object value will be interpreted as manifest content. **Note:** If a manifest
         * is provided as URL or plain object, it must use the same major schema version as the original manifest
         * to avoid incompatible changes in the behavior of the component.
         */
        manifest?: boolean | string | object;
        /**
         * @since 1.61.0 Alternative URL for the manifest.json. If `mOptions.manifest` is set to an object value,
         * this URL specifies the location to which the manifest object should resolve the relative URLs to.
         */
        altManifestUrl?: string;
        /**
         * If set to `true` validation of the component is handled by the `Messaging`
         */
        handleValidation?: string;
        /**
         * Hints for asynchronous loading. **Beware:** This parameter is only used internally by the UI5 framework
         * and compatibility cannot be guaranteed. The parameter must not be used in productive code, except in
         * code delivered by the UI5 teams.
         */
        asyncHints?: {
          /**
           * a list of components needed by the current component and its subcomponents The framework will try to
           * preload these components (their Component-preload.js) asynchronously, errors will be ignored. Please
           * note that the framework has no knowledge about whether a Component provides a preload file or whether
           * it is bundled in some library preload. If Components are listed in the hints section, they will be preloaded.
           * Instead of specifying just the names of components, an object might be given that contains a mandatory
           * `name` property and optionally, an `url` that will be used for a `registerModulePath`, and/or a `lazy`
           * property. When `lazy` is set to a truthy value, only a necessary `registerModulePath` will be executed,
           * but the corresponding component won't be preloaded.
           */
          components?: string[] | object[];
          /**
           * libraries needed by the Component and its subcomponents These libraries should be (pre-)loaded before
           * the Component. The framework will asynchronously load those libraries, if they're not loaded yet. Instead
           * of specifying just the names of libraries, an object might be given that contains a mandatory `name`
           * property and optionally, an `url` that will be used for a `registerModulePath`, and/or a `lazy` property.
           * When `lazy` is set to a truthy value, only a necessary `registerModulePath` will be executed, but the
           * corresponding library won't be preloaded.
           */
          libs?: string[] | object[];
          /**
           * a list of additional preload bundles The framework will try to load these bundles asynchronously before
           * requiring the Component, errors will be ignored. The named modules must only represent preload bundles.
           * If they are normal modules, their dependencies will be loaded with the normal synchronous request mechanism
           * and performance might degrade. Instead of specifying just the names of preload bundles, an object might
           * be given that contains a mandatory `name` property and optionally, an `url` that will be used for a `registerModulePath`.
           */
          preloadBundles?: string[] | object[];
          /**
           * `Promise` or array of `Promise`s for which the Component instantiation should wait
           */
          waitFor?: Promise<any> | Array<Promise<any>>;
        };
      }
    ): Promise<Component>;
    /**
     * Creates a new subclass of class `sap.ui.core.Component` with name `sClassName` and enriches it with the
     * information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.ManagedObject.extend},
     * plus the `manifest` property in the 'metadata' object literal, indicating that the component configuration
     * should be read from a manifest.json file.
     *
     *
     * @returns The created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Qualified name of the newly created class
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Component>,
      /**
       * Constructor function for the metadata object. If not given, it defaults to an internal subclass of `sap.ui.core.ComponentMetadata`.
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns an existing component instance, identified by its ID.
     *
     * @since 1.56.0
     * @deprecated As of version 1.120. please use the static {@link sap.ui.core.Component.getComponentById getComponentById }
     * instead.
     *
     * @returns Component instance or `undefined` when no component with the given ID exists.
     */
    static get(
      /**
       * ID of the component.
       */
      sId: string
    ): Component | undefined;
    /**
     * Returns an existing component instance, identified by its ID.
     *
     * @since 1.120
     *
     * @returns Component instance or `undefined` when no component with the given ID exists.
     */
    static getComponentById(
      /**
       * ID of the component.
       */
      sId: string
    ): Component | undefined;
    /**
     * Returns the metadata for the Component class.
     *
     *
     * @returns Metadata for the Component class.
     */
    static getMetadata(): ComponentMetadata;
    /**
     * Returns the Component instance in whose "context" the given ManagedObject has been created or `undefined`.
     *
     * This is a convenience wrapper around {@link sap.ui.core.Component.getOwnerIdFor Component.getOwnerIdFor}.
     * If the owner ID cannot be determined for reasons documented on `getOwnerForId` or when the Component
     * for the determined ID no longer exists, `undefined` will be returned.
     *
     * @since 1.25.1
     *
     * @returns the owner Component or `undefined`.
     */
    static getOwnerComponentFor(
      /**
       * Object to retrieve the owner Component for
       */
      oObject: ManagedObject
    ): Component | undefined;
    /**
     * Returns the ID of the object in whose "context" the given ManagedObject has been created.
     *
     * For objects that are not ManagedObjects or for which the owner is unknown, `undefined` will be returned
     * as owner ID.
     *
     * **Note**: Ownership for objects is only checked by the framework at the time when they are created. It
     * is not checked or updated afterwards. And it can only be detected while the {@link sap.ui.core.Component#runAsOwner Component.runAsOwner }
     * function is executing. Without further action, this is only the case while the content of a UIComponent
     * is {@link sap.ui.core.UIComponent#createContent constructed} or when a {@link sap.ui.core.routing.Router Router }
     * creates a new View and its content.
     *
     * **Note**: This method does not guarantee that the returned owner ID belongs to a Component. Currently,
     * it always does. But future versions of UI5 might introduce a more fine grained ownership concept, e.g.
     * taking Views into account. Callers that want to deal only with components as owners, should use the following
     * method: {@link sap.ui.core.Component.getOwnerComponentFor Component.getOwnerComponentFor}. It guarantees
     * that the returned object (if any) will be a Component.
     *
     * **Further note** that only the ID of the owner is recorded. In rare cases, when the lifecycle of a ManagedObject
     * is not bound to the lifecycle of its owner, (e.g. by the means of aggregations), then the owner might
     * have been destroyed already whereas the ManagedObject is still alive. So even the existence of an owner
     * ID is not a guarantee for the existence of the corresponding owner.
     *
     * @since 1.15.1
     *
     * @returns ID of the owner or `undefined`
     */
    static getOwnerIdFor(
      /**
       * Object to retrieve the owner ID for
       */
      oObject: ManagedObject
    ): string | undefined;
    /**
     * Asynchronously loads a component class without instantiating it; returns a promise on the loaded class.
     *
     * Beware: "Asynchronous component loading" doesn't necessarily mean that no more synchronous loading occurs.
     * Both the framework as well as component implementations might still execute synchronous requests. `Component.load`
     * just allows to use async calls internally.
     *
     * When a manifest is referenced in `mOptions`, this manifest is not automatically used for instances of
     * the Component class that are created after loading. The manifest or the manifest url must be provided
     * for every instance explicitly.
     *
     * To optimize the loading process, additional `asyncHints` can be provided. If components and/or libraries
     * are listed in the `asyncHints`, all the corresponding preload files will be requested in parallel, loading
     * errors (404s) will be ignored. The constructor class will only be required after all preloads have been
     * rejected or resolved. The structure of the hints and how they impact the loading of components is an
     * internal feature of this API and reserved for UI5 internal use only. Code that wants to be safe wrt.
     * version updates, should not use the `asyncHints` property.
     *
     * @since 1.56.0
     *
     * @returns A Promise that resolves with the loaded component class or `undefined` in case `mOptions.asyncHints.preloadOnly`
     * is set to `true`
     */
    static load(
      /**
       * Configuration options
       */
      mOptions: {
        /**
         * Name of the Component to load, as a dot-separated name; Even when an alternative location is specified
         * from which the manifest should be loaded (`mOptions.manifest` is set to a non-empty string), then the
         * name specified in that manifest will be ignored and this name will be used instead to determine the module
         * to be loaded.
         */
        name: string;
        /**
         * Alternative location from where to load the Component. If `mOptions.manifest` is set to a non-empty string,
         * this URL specifies the location of the final component defined via that manifest, otherwise it specifies
         * the location of the component defined via its name `mOptions.name`.
         */
        url?: string;
        /**
         * Whether and from where to load the manifest.json for the Component. When set to a truthy value, the manifest
         * will be loaded and evaluated **before** the Component controller. When set to a falsy value, the manifest
         * will be loaded and evaluated **after** the Component controller. A non-empty string value will be interpreted
         * as the URL to load the manifest from. This implies that the manifest is loaded and evaluated **before**
         * the Component controller. A non-null object value will be interpreted as manifest content.
         */
        manifest?: boolean | string | object;
        /**
         * @since 1.61.0 Alternative URL for the manifest.json. If `mOptions.manifest` is set to an object value,
         * this URL specifies the location to which the manifest object should resolve the relative URLs to.
         */
        altManifestUrl?: string;
        /**
         * Hints for asynchronous loading. **Beware:** This parameter is only used internally by the UI5 framework
         * and compatibility cannot be guaranteed. The parameter must not be used in productive code, except in
         * code delivered by the UI5 teams.
         */
        asyncHints?: {
          /**
           * a list of components needed by the current component and its subcomponents The framework will try to
           * preload these components (their Component-preload.js) asynchronously, errors will be ignored. Please
           * note that the framework has no knowledge about whether a Component provides a preload file or whether
           * it is bundled in some library preload. If Components are listed in the hints section, they will be preloaded.
           * Instead of specifying just the names of components, an object might be given that contains a mandatory
           * `name` property and optionally, an `url` that will be used for a `registerModulePath`, and/or a `lazy`
           * property. When `lazy` is set to a truthy value, only a necessary `registerModulePath` will be executed,
           * but the corresponding component won't be preloaded.
           */
          components?: string[] | object[];
          /**
           * libraries needed by the Component and its subcomponents These libraries should be (pre-)loaded before
           * the Component. The framework will asynchronously load those libraries, if they're not loaded yet. Instead
           * of specifying just the names of libraries, an object might be given that contains a mandatory `name`
           * property and optionally, an `url` that will be used for a `registerModulePath`, and/or a `lazy` property.
           * When `lazy` is set to a truthy value, only a necessary `registerModulePath` will be executed, but the
           * corresponding library won't be preloaded.
           */
          libs?: string[] | object[];
          /**
           * a list of additional preload bundles The framework will try to load these bundles asynchronously before
           * requiring the component, errors will be ignored. The named modules must only represent preload bundles.
           * If they are normal modules, their dependencies will be loaded with the standard module loading mechanism
           * and performance might degrade. Instead of specifying just the names of preload bundles, an object might
           * be given that contains a mandatory `name` property and, optionally, a `url` that will be used for a `registerModulePath`.
           */
          preloadBundles?: string[] | object[];
          /**
           * Whether only the preloads should be done, but not the loading of the Component controller class itself.
           */
          preloadOnly?: boolean;
        };
      }
    ): Promise<Function>;
    /**
     * Creates a nested component that is declared in the `sap.ui5/componentUsages` section of the descriptor
     * (manifest.json). The following snippet shows the declaration:
     * ```javascript
     *
     * {
     *   [...]
     *   "sap.ui5": {
     *     "componentUsages": {
     *       "myUsage": {
     *         "name": "my.useful.Component"
     *       }
     *     }
     *   }
     *   [...]
     * }
     * ```
     *  The syntax of the configuration object of the component usage matches the configuration object of the
     * {#link sap.ui.component} factory function.
     *
     * This is an example of how the `createComponent` function can be used for asynchronous scenarios:
     * ```javascript
     *
     * oComponent.createComponent("myUsage").then(function(oComponent) {
     *   oComponent.doSomething();
     * }).catch(function(oError) {
     *   Log.error(oError);
     * });
     * ```
     *
     *
     * The following example shows how `createComponent` can be used to create a nested component by providing
     * specific properties like `id`, `async`, `settings`, or `componentData`:
     * ```javascript
     *
     * var oComponent = oComponent.createComponent({
     *   usage: "myUsage",
     *   id: "myId",
     *   settings: { ... },
     *   componentData: { ... }
     * });
     * ```
     *  The allowed list of properties are defined in the parameter documentation of this function.
     *
     * The properties can also be defined in the descriptor. These properties can be overwritten by the local
     * properties of that function.
     *
     * Synchronous Component creation is deprecated as of 1.135.0.
     *
     * @since 1.47.0
     *
     * @returns Component instance or Promise which will be resolved with the component instance (defaults to
     * Promise / asynchronous behavior)
     */
    createComponent(
      /**
       * ID of the component usage or the configuration object that creates the component
       */
      vUsage:
        | string
        | {
            /**
             * ID of component usage
             */
            usage: string;
            /**
             * ID of the nested component that is prefixed with `autoPrefixId`
             */
            id?: string;
            /**
             * Indicates whether the component creation is done asynchronously (You should use synchronous creation
             * only if really necessary, because this has a negative impact on performance.)
             */
            async?: boolean;
            /**
             * Settings for the nested component like for {#link sap.ui.component} or the component constructor
             */
            settings?: $ComponentSettings;
            /**
             * Initial data of the component, see {@link sap.ui.core.Component#getComponentData}
             */
            componentData?: object;
          }
    ): Component | Promise<Component>;
    /**
     * Cleans up the Component instance before destruction.
     *
     * Applications must not call this hook method directly, it is called by the framework when the element
     * is {@link #destroy destroyed}.
     *
     * Subclasses of Component should override this hook to implement any necessary cleanup.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns This hook method must not have a return value. Return value `void` is deprecated since 1.120,
     * as it does not force functions to **not** return something. This implies that, for instance, no async
     * function returning a Promise should be used.
     *
     * **Note:** While the return type is currently `void|undefined`, any implementation of this hook must not
     * return anything but undefined. Any other return value will cause an error log in this version of UI5
     * and will fail in future major versions of UI5.
     */
    exit(): void | undefined;
    /**
     * Returns the list of active terminologies. See the {@link sap.ui.core.Component.create Component.create }
     * factory API documentation for more detail.
     *
     * @since 1.76
     *
     * @returns List of active terminologies
     */
    getActiveTerminologies(): string[] | undefined;
    /**
     * Returns user specific data object
     *
     * @since 1.15.0
     *
     * @returns componentData
     */
    getComponentData(): object;
    /**
     * Returns the event bus of this component.
     *
     * @since 1.20.0
     *
     * @returns the event bus
     */
    getEventBus(): EventBus;
    /**
     * Components don't have a facade and therefore return themselves as their interface.
     * See:
     * 	sap.ui.base.Object#getInterface
     *
     *
     * @returns `this` as there's no facade for components
     */
    getInterface(): this;
    /**
     * Returns the manifest defined in the metadata of the component. If not specified, the return value is
     * null.
     *
     * @since 1.33.0
     *
     * @returns manifest.
     */
    getManifest(): object;
    /**
     * Returns the configuration of a manifest section or the value for a specific path. If no section or key
     * is specified, the return value is null.
     *
     * Example: ` { "sap.ui5": { "dependencies": { "libs": { "sap.m": {} }, "components": { "my.component.a":
     * {} } } }); `
     *
     * The configuration above can be accessed in the following ways:
     * 	 - **By section/namespace**: `oComponent.getManifestEntry("sap.ui5")`
     * 	 - **By path**: `oComponent.getManifestEntry("/sap.ui5/dependencies/libs")`
     *
     * By section/namespace returns the configuration for the specified manifest section and by path allows
     * to specify a concrete path to a dedicated entry inside the manifest. The path syntax always starts with
     * a slash (/).
     *
     * @since 1.33.0
     *
     * @returns Value of the manifest section or the key (could be any kind of value)
     */
    getManifestEntry(
      /**
       * Either the manifest section name (namespace) or a concrete path
       */
      sKey: string
    ): any | null;
    /**
     * Returns the manifest object.
     *
     * @since 1.33.0
     *
     * @returns manifest.
     */
    getManifestObject(): Manifest;
    /**
     * Returns the metadata for the specific class of the current instance.
     *
     *
     * @returns Metadata for the specific class of the current instance.
     */
    getMetadata(): ComponentMetadata;
    /**
     * Returns a service interface for the {@link sap.ui.core.service.Service Service} declared in the descriptor
     * for components (manifest.json). The declaration needs to be done in the `sap.ui5/services` section as
     * follows:
     * ```javascript
     *
     * {
     *   [...]
     *   "sap.ui5": {
     *     "services": {
     *       "myLocalServiceAlias": {
     *         "factoryName": "my.ServiceFactory",
     *         ["optional": true]
     *       }
     *     }
     *   }
     *   [...]
     * }
     * ```
     *  The service declaration is used to define a mapping between the local alias for the service that can
     * be used in the Component and the name of the service factory which will be used to create a service instance.
     *
     * The `getService` function will look up the service factory and will create a new instance by using the
     * service factory function {@link sap.ui.core.service.ServiceFactory#createInstance createInstance} The
     * optional property defines that the service is not mandatory and the usage will not depend on the availability
     * of this service. When requesting an optional service the `getService` function will reject but there
     * will be no error logged in the console.
     *
     * When creating a new instance of the service the Component context will be passed as `oServiceContext`
     * as follows:
     * ```javascript
     *
     * {
     *   "scopeObject": this,     // the Component instance
     *   "scopeType": "component" // the stereotype of the scopeObject
     * }
     * ```
     *
     *
     * The service will be created only once per Component and reused in future calls to the `getService` function.
     *  This function will return a `Promise` which provides the service interface when resolved. If the
     * `factoryName` could not be found in the {@link sap.ui.core.service.ServiceFactoryRegistry Service Factory Registry }
     * or the service declaration in the descriptor for components (manifest.json) is missing the Promise will
     * reject.
     *
     * This is an example of how the `getService` function can be used:
     * ```javascript
     *
     * oComponent.getService("myLocalServiceAlias").then(function(oService) {
     *   oService.doSomething();
     * }).catch(function(oError) {
     *   Log.error(oError);
     * });
     * ```
     *
     *
     * @since 1.37.0
     *
     * @returns Promise which will be resolved with the Service interface
     */
    getService(
      /**
       * Local service alias as defined in the manifest.json
       */
      sLocalServiceAlias: string
    ): Promise</* was: sap.ui.core.service.Service */ any>;
    /**
     * Initializes the Component instance after creation.
     *
     * Applications must not call this hook method directly, it is called by the framework while the constructor
     * of a Component is executed.
     *
     * Subclasses of Component should override this hook to implement any necessary initialization.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns This hook method must not have a return value. Return value `void` is deprecated since 1.120,
     * as it does not force functions to **not** return something. This implies that, for instance, no async
     * function returning a Promise should be used.
     *
     * **Note:** While the return type is currently `void|undefined`, any implementation of this hook must not
     * return anything but undefined. Any other return value will cause an error log in this version of UI5
     * and will fail in future major versions of UI5.
     */
    init(): void | undefined;
    /**
     * This method is called after the component is activated
     *
     * @since 1.88
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns This lifecycle hook must not have a return value.
     *
     * **Note:** While the return type is currently `void|undefined`, any implementation of this hook must not
     * return anything but undefined. Any other return value will cause an error log in this version of UI5
     * and will fail in future major versions of UI5.
     */
    onActivate(): void | undefined;
    /**
     * The hook which gets called when the static configuration of the component has been changed by some configuration
     * extension.
     *
     * @since 1.15.1
     */
    onConfigChange(
      /**
       * Error message.
       */
      sConfigKey: string
    ): void;
    /**
     * This method is called after the component is deactivated
     *
     * @since 1.88
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns This lifecycle hook must not have a return value.
     *
     * **Note:** While the return type is currently `void|undefined`, any implementation of this hook must not
     * return anything but undefined. Any other return value will cause an error log in this version of UI5
     * and will fail in future major versions of UI5.
     */
    onDeactivate(): void | undefined;
    /**
     * The window before unload hook. Override this method in your Component class implementation, to handle
     * cleanup before the real unload or to prompt a question to the user, if the component should be exited.
     *
     * @since 1.15.1
     * @deprecated As of version 1.119. recommended to use the browser-native page lifecycle API, providing
     * events such as 'pageshow' and 'pagehide'
     *
     * @returns a string if a prompt should be displayed to the user confirming closing the Component (e.g.
     * when the Component is not yet saved), or `undefined` if no prompt should be shown.
     */
    onWindowBeforeUnload(): string | undefined;
    /**
     * The window error hook. Override this method in your Component class implementation to listen to unhandled
     * errors.
     *
     * @since 1.15.1
     * @deprecated As of version 1.119. recommended to use the browser-native API to listen for errors: window.addEventListener("error",
     * function() { ... })
     */
    onWindowError(
      /**
       * The error message.
       */
      sMessage: string,
      /**
       * File where the error occurred
       */
      sFile: string,
      /**
       * Line number of the error
       */
      iLine: int
    ): void;
    /**
     * The window unload hook. Override this method in your Component class implementation, to handle cleanup
     * of the component once the window will be unloaded (e.g. closed).
     *
     * @since 1.15.1
     * @deprecated As of version 1.119. Newer browser versions deprecate the browser-native 'unload' event.
     * Therefore, the former API won't reliably work anymore. Please have a look at the browser-native page
     * lifecycle API, e.g. its events 'pageshow' and 'pagehide'.
     */
    onWindowUnload(): void;
    /**
     * Calls the function `fn` once and marks all ManagedObjects created during that call as "owned" by this
     * Component.
     *
     * Nested calls of this method are supported (e.g. inside a newly created, nested component). The currently
     * active owner Component will be remembered before executing `fn` and restored afterwards.
     *
     * @since 1.25.1
     *
     * @returns result of function `fn`
     */
    runAsOwner(
      /**
       * Function to execute
       */
      fn: Function
    ): any;
  }
  /**
   * The structure of the "metadata" object which is passed when inheriting from sap.ui.core.Component using
   * its static "extend" method. See {@link sap.ui.core.Component.extend} and {@link sap.ui.core.Component.create }
   * for additional details on its usage.
   */
  export type MetadataOptions = MetadataOptions1 & {
    /**
     * The manifest option determines how a component manifest should be evaluated. Default is `undefined`.
     *
     * When set to `false` or `undefined`, no manifest.json is present for this Component, however the Component
     * can still be started with a manifest given as an argument of the factory function, see {@link sap.ui.core.Component.create}.
     * When set to an object, this object will be interpreted as a manifest and must adhere to the {@link https://ui5.sap.com/#/topic/be0cf40f61184b358b5faedaec98b2da descriptor schema for components}.
     * When set to the string literal `"json"`, this property indicates that the component configuration should
     * be read from a manifest.json file which is assumed to exist next to the Component.js file.
     */
    manifest?: undefined | false | object | "json";
  };

  /**
   * Registry of all `Component`s that currently exist.
   *
   * @since 1.67
   * @deprecated As of version 1.120. Use {@link module:sap/ui/core/ComponentRegistry} instead.
   */
  interface registry {
    /**
     * Number of existing components.
     */
    size: int;

    /**
     * Return an object with all instances of `sap.ui.core.Component`, keyed by their ID.
     *
     * Each call creates a new snapshot object. Depending on the size of the UI, this operation therefore might
     * be expensive. Consider to use the `forEach` or `filter` method instead of executing similar operations
     * on the returned object.
     *
     * **Note**: The returned object is created by a call to `Object.create(null)`, and therefore lacks all
     * methods of `Object.prototype`, e.g. `toString` etc.
     *
     *
     * @returns Object with all components, keyed by their ID
     */
    all(): Record<ID, Component>;
    /**
     * Returns an array with components for which the given `callback` returns a value that coerces to `true`.
     *
     * The expected signature of the callback is
     * ```javascript
     *
     *    function callback(oComponent, sID)
     * ```
     *  where `oComponent` is the currently visited component instance and `sID` is the ID of that instance.
     *
     * If components are created or destroyed within the `callback`, then the behavior is not specified. Newly
     * added objects might or might not be visited. When a component is destroyed during the filtering and was
     * not visited yet, it might or might not be visited. As the behavior for such concurrent modifications
     * is not specified, it may change in newer releases.
     *
     * If a `thisArg` is given, it will be provided as `this` context when calling `callback`. The `this` value
     * that the implementation of `callback` sees, depends on the usual resolution mechanism. E.g. when `callback`
     * was bound to some context object, that object wins over the given `thisArg`.
     *
     * This function returns an array with all components matching the given predicate. The order of the components
     * in the array is not specified and might change between calls (over time and across different versions
     * of UI5).
     *
     *
     * @returns Array of components matching the predicate; order is undefined and might change in newer versions
     * of UI5
     */
    filter(
      /**
       * predicate against which each Component is tested
       */
      callback: (p1: Component, p2: ID) => boolean,
      /**
       * context object to provide as `this` in each call of `callback`
       */
      thisArg?: Object
    ): Component[];
    /**
     * Calls the given `callback` for each existing component.
     *
     * The expected signature of the callback is
     * ```javascript
     *
     *    function callback(oComponent, sID)
     * ```
     *  where `oComponent` is the currently visited component instance and `sID` is the ID of that instance.
     *
     * The order in which the callback is called for components is not specified and might change between calls
     * (over time and across different versions of UI5).
     *
     * If components are created or destroyed within the `callback`, then the behavior is not specified. Newly
     * added objects might or might not be visited. When a component is destroyed during the filtering and was
     * not visited yet, it might or might not be visited. As the behavior for such concurrent modifications
     * is not specified, it may change in newer releases.
     *
     * If a `thisArg` is given, it will be provided as `this` context when calling `callback`. The `this` value
     * that the implementation of `callback` sees, depends on the usual resolution mechanism. E.g. when `callback`
     * was bound to some context object, that object wins over the given `thisArg`.
     */
    forEach(
      /**
       * Function to call for each Component
       */
      callback: (p1: Component, p2: ID) => void,
      /**
       * Context object to provide as `this` in each call of `callback`
       */
      thisArg?: Object
    ): void;
    /**
     * Retrieves a Component by its ID.
     *
     * When the ID is `null` or `undefined` or when there's no Component with the given ID, then `undefined`
     * is returned.
     *
     *
     * @returns Component with the given ID or `undefined`
     */
    get(
      /**
       * ID of the Component to retrieve
       */
      id: ID
    ): Component | undefined;
  }
  export const registry: registry;

  /**
   * Describes the settings that can be provided to the Component constructor.
   */
  export interface $ComponentSettings extends $ManagedObjectSettings {
    componentData?: any;
  }
}

declare module "sap/ui/core/ComponentContainer" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import {
    IPlaceholderSupport,
    ID,
    CSSSize,
    ComponentLifecycle,
    URI,
  } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { $ComponentSettings } from "sap/ui/core/Component";

  import UIComponent from "sap/ui/core/UIComponent";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * Container that embeds a `sap/ui/core/UIComponent` in a control tree.
   *
   * **Concerning asynchronous component loading:**
   *
   * To activate a fully asynchronous loading behavior of components and their dependencies, the property
   * `async` needs to be set to `true` and the `manifest` property needs to be set to a 'truthy' value, e.g.
   * `true` or a URL to the manifest location. If both options are correctly set, the component factory will
   * load and evaluate the component manifest first. In this way, the additional dependencies of the Component
   * are already known before the Component preload/controller is loaded. Both the component preload/controller
   * and the additional dependencies can thus be loaded asynchronously and in parallel.
   *
   * Sample usage of the ComponentContainer:
   *
   *
   * ```javascript
   *
   *     <!-- inside XML view -->
   *     ...
   *     <core:ComponentContainer
   *         usage="someComponent"
   *         manifest="true"
   *         async="true"
   *     />
   * ```
   *
   *
   * See also {@link module:sap/ui/core/ComponentSupport}.
   */
  export default class ComponentContainer
    extends Control
    implements IPlaceholderSupport
  {
    __implements__sap_ui_core_IPlaceholderSupport: boolean;
    /**
     * Constructor for a new ComponentContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $ComponentContainerSettings
    );
    /**
     * Constructor for a new ComponentContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $ComponentContainerSettings
    );

    /**
     * Creates a new subclass of class sap.ui.core.ComponentContainer with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ComponentContainer>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.ComponentContainer.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:componentCreated componentCreated} event of
     * this `sap.ui.core.ComponentContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.ComponentContainer` itself.
     *
     * Fired when the component instance has been created by the ComponentContainer.
     *
     * @since 1.50
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachComponentCreated(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ComponentContainer$ComponentCreatedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.ComponentContainer` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:componentCreated componentCreated} event of
     * this `sap.ui.core.ComponentContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.ComponentContainer` itself.
     *
     * Fired when the component instance has been created by the ComponentContainer.
     *
     * @since 1.50
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachComponentCreated(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ComponentContainer$ComponentCreatedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.ComponentContainer` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:componentFailed componentFailed} event of this
     * `sap.ui.core.ComponentContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.ComponentContainer` itself.
     *
     * Fired when the creation of the component instance has failed.
     *
     * By default, the `ComponentContainer` also logs the error that occurred. Since 1.83, this default behavior
     * can be prevented by calling `preventDefault()` on the event object.
     *
     * @since 1.60
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachComponentFailed(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ComponentContainer$ComponentFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.ComponentContainer` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:componentFailed componentFailed} event of this
     * `sap.ui.core.ComponentContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.ComponentContainer` itself.
     *
     * Fired when the creation of the component instance has failed.
     *
     * By default, the `ComponentContainer` also logs the error that occurred. Since 1.83, this default behavior
     * can be prevented by calling `preventDefault()` on the event object.
     *
     * @since 1.60
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachComponentFailed(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ComponentContainer$ComponentFailedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.ComponentContainer` itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:componentCreated componentCreated} event of
     * this `sap.ui.core.ComponentContainer`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.50
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachComponentCreated(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ComponentContainer$ComponentCreatedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:componentFailed componentFailed} event of
     * this `sap.ui.core.ComponentContainer`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.60
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachComponentFailed(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ComponentContainer$ComponentFailedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:componentCreated componentCreated} to attached listeners.
     *
     * @since 1.50
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireComponentCreated(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ComponentContainer$ComponentCreatedEventParameters
    ): this;
    /**
     * Fires event {@link #event:componentFailed componentFailed} to attached listeners.
     *
     * Listeners may prevent the default action of this event by calling the `preventDefault` method on the
     * event object. The return value of this method indicates whether the default action should be executed.
     *
     * @since 1.60
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether or not to prevent the default action
     */
    fireComponentFailed(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ComponentContainer$ComponentFailedEventParameters
    ): boolean;
    /**
     * Gets current value of property {@link #getAsync async}.
     *
     * Flag whether the component should be created sync (default) or async. The default will be async when
     * initially the property `manifest` is set to a truthy value and for the property `async` no value has
     * been specified. This property can only be applied initially.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `async`
     */
    getAsync(): boolean;
    /**
     * Gets current value of property {@link #getAutoPrefixId autoPrefixId}.
     *
     * Flag, whether to auto-prefix the ID of the nested Component or not. If this property is set to true the
     * ID of the Component will be prefixed with the ID of the ComponentContainer followed by a single dash.
     * This property can only be applied initially.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `autoPrefixId`
     */
    getAutoPrefixId(): boolean;
    /**
     * ID of the element which is the current target of the association {@link #getComponent component}, or
     * `null`.
     */
    getComponent(): ID | null;
    /**
     * Gets current value of property {@link #getHandleValidation handleValidation}.
     *
     * Enable/disable validation handling by Messaging for this component. The resulting Messages will be propagated
     * to the controls. This property can only be applied initially.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `handleValidation`
     */
    getHandleValidation(): boolean;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Container height in CSS size
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Gets current value of property {@link #getLifecycle lifecycle}.
     *
     * Lifecycle behavior for the Component associated by the `ComponentContainer`. The default behavior is
     * `Legacy`. This means that the `ComponentContainer` takes care that the Component is destroyed when the
     * `ComponentContainer` is destroyed, but it is **not** destroyed when a new Component is associated. If
     * you use the `usage` property to create the Component, the default behavior is `Container`. This means
     * that the Component is destroyed when the `ComponentContainer` is destroyed or a new Component is associated.
     * This property must only be applied before a component instance is created.
     *
     * Default value is `Legacy`.
     *
     *
     * @returns Value of property `lifecycle`
     */
    getLifecycle(): ComponentLifecycle;
    /**
     * Gets current value of property {@link #getManifest manifest}.
     *
     * Controls when and from where to load the manifest for the Component. When set to any truthy value, the
     * manifest will be loaded asynchronously by default and evaluated before the Component controller, if it
     * is set to a falsy value other than `undefined`, the manifest will be loaded after the controller. A non-empty
     * string value will be interpreted as the URL location from where to load the manifest. A non-null object
     * value will be interpreted as manifest content. This property can only be applied initially.
     *
     *
     * @returns Value of property `manifest`
     */
    getManifest(): any;
    /**
     * Gets current value of property {@link #getName name}.
     *
     * Component name, the package where the component is contained. This property can only be applied initially.
     *
     *
     * @returns Value of property `name`
     */
    getName(): string;
    /**
     * Gets current value of property {@link #getPropagateModel propagateModel}.
     *
     * Defines whether binding information is propagated to the component.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `propagateModel`
     */
    getPropagateModel(): boolean;
    /**
     * Gets current value of property {@link #getSettings settings}.
     *
     * The settings object passed to the component when created. This property can only be applied initially.
     *
     *
     * @returns Value of property `settings`
     */
    getSettings(): $ComponentSettings;
    /**
     * Gets current value of property {@link #getUrl url}.
     *
     * The URL of the component. This property can only be applied initially.
     *
     *
     * @returns Value of property `url`
     */
    getUrl(): URI;
    /**
     * Gets current value of property {@link #getUsage usage}.
     *
     * The name of a component usage as configured in the app descriptor of the component owning this `ComponentContainer`.
     *
     * The configuration from the named usage will be used to create a component instance for this `ComponentContainer`.
     * If there's no owning component or if its app descriptor does not contain a usage with the given name,
     * an error will be logged.
     *
     * See {@link https://ui5.sap.com/#/topic/346599f0890d4dfaaa11c6b4ffa96312 Using and Nesting Components }
     * for more information about component usages.
     *
     * This property can only be applied initially.
     *
     *
     * @returns Value of property `usage`
     */
    getUsage(): string;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Container width in CSS size
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Sets a new value for property {@link #getAsync async}.
     *
     * Flag whether the component should be created sync (default) or async. The default will be async when
     * initially the property `manifest` is set to a truthy value and for the property `async` no value has
     * been specified. This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAsync(
      /**
       * New value for property `async`
       */
      bAsync?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getAutoPrefixId autoPrefixId}.
     *
     * Flag, whether to auto-prefix the ID of the nested Component or not. If this property is set to true the
     * ID of the Component will be prefixed with the ID of the ComponentContainer followed by a single dash.
     * This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAutoPrefixId(
      /**
       * New value for property `autoPrefixId`
       */
      bAutoPrefixId?: boolean
    ): this;
    /**
     * Sets the component of the container. Depending on the ComponentContainer's lifecycle this might destroy
     * the old associated Component.
     *
     * Once the component is associated with the container the cross connection to the component will be set
     * and the models will be propagated if defined. If the `usage` property is set the ComponentLifecycle is
     * processed like a "Container" lifecycle.
     *
     *
     * @returns the reference to `this` in order to allow method chaining
     */
    setComponent(
      /**
       * ID of an element which becomes the new target of this component association. Alternatively, an element
       * instance may be given.
       */
      vComponent: ID | UIComponent
    ): this;
    /**
     * Sets a new value for property {@link #getHandleValidation handleValidation}.
     *
     * Enable/disable validation handling by Messaging for this component. The resulting Messages will be propagated
     * to the controls. This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHandleValidation(
      /**
       * New value for property `handleValidation`
       */
      bHandleValidation?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Container height in CSS size
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getLifecycle lifecycle}.
     *
     * Lifecycle behavior for the Component associated by the `ComponentContainer`. The default behavior is
     * `Legacy`. This means that the `ComponentContainer` takes care that the Component is destroyed when the
     * `ComponentContainer` is destroyed, but it is **not** destroyed when a new Component is associated. If
     * you use the `usage` property to create the Component, the default behavior is `Container`. This means
     * that the Component is destroyed when the `ComponentContainer` is destroyed or a new Component is associated.
     * This property must only be applied before a component instance is created.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Legacy`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLifecycle(
      /**
       * New value for property `lifecycle`
       */
      sLifecycle?: ComponentLifecycle | keyof typeof ComponentLifecycle
    ): this;
    /**
     * Sets a new value for property {@link #getManifest manifest}.
     *
     * Controls when and from where to load the manifest for the Component. When set to any truthy value, the
     * manifest will be loaded asynchronously by default and evaluated before the Component controller, if it
     * is set to a falsy value other than `undefined`, the manifest will be loaded after the controller. A non-empty
     * string value will be interpreted as the URL location from where to load the manifest. A non-null object
     * value will be interpreted as manifest content. This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setManifest(
      /**
       * New value for property `manifest`
       */
      oManifest?: any
    ): this;
    /**
     * Sets a new value for property {@link #getName name}.
     *
     * Component name, the package where the component is contained. This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setName(
      /**
       * New value for property `name`
       */
      sName?: string
    ): this;
    /**
     * Sets a new value for property {@link #getPropagateModel propagateModel}.
     *
     * Defines whether binding information is propagated to the component.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPropagateModel(
      /**
       * New value for property `propagateModel`
       */
      bPropagateModel?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getSettings settings}.
     *
     * The settings object passed to the component when created. This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSettings(
      /**
       * New value for property `settings`
       */
      oSettings?: $ComponentSettings
    ): this;
    /**
     * Sets a new value for property {@link #getUrl url}.
     *
     * The URL of the component. This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUrl(
      /**
       * New value for property `url`
       */
      sUrl?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getUsage usage}.
     *
     * The name of a component usage as configured in the app descriptor of the component owning this `ComponentContainer`.
     *
     * The configuration from the named usage will be used to create a component instance for this `ComponentContainer`.
     * If there's no owning component or if its app descriptor does not contain a usage with the given name,
     * an error will be logged.
     *
     * See {@link https://ui5.sap.com/#/topic/346599f0890d4dfaaa11c6b4ffa96312 Using and Nesting Components }
     * for more information about component usages.
     *
     * This property can only be applied initially.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUsage(
      /**
       * New value for property `usage`
       */
      sUsage?: string
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Container width in CSS size
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ComponentContainer constructor.
   */
  export interface $ComponentContainerSettings extends $ControlSettings {
    /**
     * Component name, the package where the component is contained. This property can only be applied initially.
     */
    name?: string | PropertyBindingInfo;

    /**
     * The URL of the component. This property can only be applied initially.
     */
    url?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * Flag whether the component should be created sync (default) or async. The default will be async when
     * initially the property `manifest` is set to a truthy value and for the property `async` no value has
     * been specified. This property can only be applied initially.
     */
    async?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Enable/disable validation handling by Messaging for this component. The resulting Messages will be propagated
     * to the controls. This property can only be applied initially.
     */
    handleValidation?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The settings object passed to the component when created. This property can only be applied initially.
     */
    settings?: $ComponentSettings | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether binding information is propagated to the component.
     */
    propagateModel?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Container width in CSS size
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Container height in CSS size
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Lifecycle behavior for the Component associated by the `ComponentContainer`. The default behavior is
     * `Legacy`. This means that the `ComponentContainer` takes care that the Component is destroyed when the
     * `ComponentContainer` is destroyed, but it is **not** destroyed when a new Component is associated. If
     * you use the `usage` property to create the Component, the default behavior is `Container`. This means
     * that the Component is destroyed when the `ComponentContainer` is destroyed or a new Component is associated.
     * This property must only be applied before a component instance is created.
     */
    lifecycle?:
      | (ComponentLifecycle | keyof typeof ComponentLifecycle)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Flag, whether to auto-prefix the ID of the nested Component or not. If this property is set to true the
     * ID of the Component will be prefixed with the ID of the ComponentContainer followed by a single dash.
     * This property can only be applied initially.
     */
    autoPrefixId?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The name of a component usage as configured in the app descriptor of the component owning this `ComponentContainer`.
     *
     * The configuration from the named usage will be used to create a component instance for this `ComponentContainer`.
     * If there's no owning component or if its app descriptor does not contain a usage with the given name,
     * an error will be logged.
     *
     * See {@link https://ui5.sap.com/#/topic/346599f0890d4dfaaa11c6b4ffa96312 Using and Nesting Components }
     * for more information about component usages.
     *
     * This property can only be applied initially.
     */
    usage?: string | PropertyBindingInfo;

    /**
     * Controls when and from where to load the manifest for the Component. When set to any truthy value, the
     * manifest will be loaded asynchronously by default and evaluated before the Component controller, if it
     * is set to a falsy value other than `undefined`, the manifest will be loaded after the controller. A non-empty
     * string value will be interpreted as the URL location from where to load the manifest. A non-null object
     * value will be interpreted as manifest content. This property can only be applied initially.
     */
    manifest?: any | PropertyBindingInfo | `{${string}}`;

    /**
     * The component displayed in this ComponentContainer.
     */
    component?: UIComponent | string;

    /**
     * Fired when the component instance has been created by the ComponentContainer.
     *
     * @since 1.50
     */
    componentCreated?: (
      oEvent: ComponentContainer$ComponentCreatedEvent
    ) => void;

    /**
     * Fired when the creation of the component instance has failed.
     *
     * By default, the `ComponentContainer` also logs the error that occurred. Since 1.83, this default behavior
     * can be prevented by calling `preventDefault()` on the event object.
     *
     * @since 1.60
     */
    componentFailed?: (oEvent: ComponentContainer$ComponentFailedEvent) => void;
  }

  /**
   * Parameters of the ComponentContainer#componentCreated event.
   */
  export interface ComponentContainer$ComponentCreatedEventParameters {
    /**
     * Reference to the created component instance
     */
    component?: UIComponent;
  }

  /**
   * Event object of the ComponentContainer#componentCreated event.
   */
  export type ComponentContainer$ComponentCreatedEvent = Event<
    ComponentContainer$ComponentCreatedEventParameters,
    ComponentContainer
  >;

  /**
   * Parameters of the ComponentContainer#componentFailed event.
   */
  export interface ComponentContainer$ComponentFailedEventParameters {
    /**
     * The reason object as returned by the component promise
     */
    reason?: object;
  }

  /**
   * Event object of the ComponentContainer#componentFailed event.
   */
  export type ComponentContainer$ComponentFailedEvent = Event<
    ComponentContainer$ComponentFailedEventParameters,
    ComponentContainer
  >;
}

declare module "sap/ui/core/ComponentMetadata" {
  import ManagedObjectMetadata from "sap/ui/base/ManagedObjectMetadata";

  import { MetadataOptions } from "sap/ui/core/Component";

  import Manifest from "sap/ui/core/Manifest";

  /**
   * @since 1.9.2
   */
  export default class ComponentMetadata extends ManagedObjectMetadata {
    /**
     * Creates a new metadata object for a Component subclass.
     */
    constructor(
      /**
       * Fully qualified name of the class that is described by this metadata object
       */
      sClassName: string,
      /**
       * Static info to construct the metadata from
       */
      oClassInfo: {
        /**
         * The metadata object describing the class
         */
        metadata?: MetadataOptions;
      }
    );

    /**
     * Returns the name of the Component (which is the namespace only with the module name)
     *
     *
     * @returns Component name
     */
    getComponentName(): string;
    /**
     * Returns array of components specified in the metadata of the Component. If not specified or the array
     * is empty, the return value is null.  **Important:**
     *  If a Component is loaded using the manifest URL (or according the "manifest first" strategy), this function
     * ignores the entries of the manifest file! It returns only the entries which have been defined in the
     * Component metadata or in the proper Component manifest.
     *
     * @deprecated As of version 1.27.1. Please use {@link sap.ui.core.Component#getManifestEntry}("/sap.ui5/dependencies/components")
     *
     * @returns Required Components.
     */
    getComponents(): string[];
    /**
     * Returns a copy of the configuration property to disallow modifications. If no key is specified it returns
     * the complete configuration property
     *
     * @since 1.15.1
     * @deprecated As of version 1.27.1. Please use {@link sap.ui.core.Component#getManifestEntry}("/sap.ui5/config")
     *
     * @returns the value of the configuration property
     */
    getConfig(
      /**
       * Key of the configuration property
       */
      sKey?: string,
      /**
       * If set to `true`, only the local configuration is returned
       */
      bDoNotMerge?: boolean
    ): object;
    /**
     * Returns the custom Component configuration entry with the specified key (this must be a JSON object).
     * If no key is specified, the return value is null.
     *
     * Example: ` sap.ui.core.Component.extend("sample.Component", { metadata: { "my.custom.config" : { "property1"
     * : true, "property2" : "Something else" } } }); `
     *
     * The configuration above can be accessed via `sample.Component.getMetadata().getCustomEntry("my.custom.config")`.
     *
     * @deprecated As of version 1.27.1. Please use the sap.ui.core.ComponentMetadata#getManifestEntry
     *
     * @returns custom Component configuration with the specified key.
     */
    getCustomEntry(
      /**
       * Key of the custom configuration (must be prefixed with a namespace)
       */
      sKey: string,
      /**
       * Indicates whether the custom configuration is merged with the parent custom configuration of the Component.
       */
      bMerged?: boolean
    ): Object;
    /**
     * Returns the dependencies defined in the metadata of the Component. If not specified, the return value
     * is null.  **Important:**
     *  If a Component is loaded using the manifest URL (or according the "manifest first" strategy), this function
     * ignores the entries of the manifest file! It returns only the entries which have been defined in the
     * Component metadata or in the proper Component manifest.
     *
     * @deprecated As of version 1.27.1. Please use {@link sap.ui.core.Component#getManifestEntry}("/sap.ui5/dependencies")
     *
     * @returns Component dependencies.
     */
    getDependencies(): Object;
    /**
     * Returns the array of the included files that the Component requires such as CSS and JavaScript. If not
     * specified or the array is empty, the return value is null.  **Important:**
     *  If a Component is loaded using the manifest URL (or according the "manifest first" strategy), this function
     * ignores the entries of the manifest file! It returns only the entries which have been defined in the
     * Component metadata or in the proper Component manifest.
     *
     * @deprecated As of version 1.27.1. For CSS, please use {@link sap.ui.core.Component#getManifestEntry}("/sap.ui5/resources/css").
     *
     * @returns Included files.
     */
    getIncludes(): string[];
    /**
     * Returns array of libraries specified in metadata of the Component, that are automatically loaded when
     * an instance of the component is created. If not specified or the array is empty, the return value is
     * null.  **Important:**
     *  If a Component is loaded using the manifest URL (or according the "manifest first" strategy), this function
     * ignores the entries of the manifest file! It returns only the entries which have been defined in the
     * Component metadata or in the proper Component manifest.
     *
     * @deprecated As of version 1.27.1. Please use {@link sap.ui.core.Component#getManifestEntry}("/sap.ui5/dependencies/libs")
     *
     * @returns Required libraries.
     */
    getLibs(): string[];
    /**
     * Returns the manifest defined in the metadata of the Component. If not specified, the return value is
     * null.
     *
     * @since 1.27.1
     * @deprecated As of version 1.33.0. Please use the sap.ui.core.Component#getManifest
     *
     * @returns manifest.
     */
    getManifest(): Object | null;
    /**
     * Returns the configuration of a manifest section or the value for a specific path. If no section or key
     * is specified, the return value is null.
     *
     * Example: ` { "sap.ui5": { "dependencies": { "libs": { "sap.m": {} }, "components": { "my.component.a":
     * {} } } }); `
     *
     * The configuration above can be accessed in the following ways:
     * 	 - **By section/namespace**: `oComponent.getMetadata().getManifestEntry("sap.ui5")`
     * 	 - **By path**: `oComponent.getMetadata().getManifestEntry("/sap.ui5/dependencies/libs")`
     *
     * By section/namespace returns the configuration for the specified manifest section and by path allows
     * to specify a concrete path to a dedicated entry inside the manifest. The path syntax always starts with
     * a slash (/).
     *
     * @since 1.27.1
     * @deprecated As of version 1.33.0. Please use the sap.ui.core.Component#getManifest
     *
     * @returns Value of the manifest section or the key (could be any kind of value)
     */
    getManifestEntry(
      /**
       * Either the manifest section name (namespace) or a concrete path
       */
      sKey: string,
      /**
       * Indicates whether the custom configuration is merged with the parent custom configuration of the Component.
       */
      bMerged?: boolean
    ): any | null;
    /**
     * Returns the manifest object.
     *
     * @since 1.33.0
     *
     * @returns manifest.
     */
    getManifestObject(): Manifest;
    /**
     * Returns the version of the metadata which could be 1 or 2. 1 is for legacy metadata whereas 2 is for
     * the manifest.
     *
     * @since 1.27.1
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns metadata version (1: legacy metadata, 2: manifest)
     */
    getMetadataVersion(): int;
    /**
     * Returns the raw manifest defined in the metadata of the Component. If not specified, the return value
     * is null.
     *
     * @since 1.29.0
     * @deprecated As of version 1.33.0. Please use the sap.ui.core.Component#getManifest
     *
     * @returns manifest
     */
    getRawManifest(): Object;
    /**
     * Returns the required version of SAPUI5 defined in the metadata of the Component. If returned value is
     * null, then no special UI5 version is required.  **Important:**
     *  If a Component is loaded using the manifest URL (or according the "manifest first" strategy), this function
     * ignores the entries of the manifest file! It returns only the entries which have been defined in the
     * Component metadata or in the proper Component manifest.
     *
     * @deprecated As of version 1.27.1. Please use {@link sap.ui.core.Component#getManifestEntry}("/sap.ui5/dependencies/minUI5Version")
     *
     * @returns Required version of UI5 or if not specified then `null`.
     */
    getUI5Version(): string | null;
    /**
     * Returns the version of the component. If not specified, the return value is null.  **Important:**
     *  If a Component is loaded using the manifest URL (or according the "manifest first" strategy), this function
     * ignores the entries of the manifest file! It returns only the entries which have been defined in the
     * Component metadata or in the proper Component manifest.
     *
     * @deprecated As of version 1.34.2. Please use {@link sap.ui.core.Component#getManifestEntry}("/sap.app/applicationVersion/version")
     *
     * @returns The version of the component.
     */
    getVersion(): string;
    /**
     * Returns whether the class of this metadata is a component base class or not.
     *
     * @since 1.33.0
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns true if it is sap.ui.core.Component or sap.ui.core.UIComponent
     */
    isBaseClass(): boolean;
  }
}

declare module "sap/ui/core/Configuration" {
  import CalendarType from "sap/ui/core/CalendarType";

  import CalendarWeekNumbering from "sap/ui/core/date/CalendarWeekNumbering";

  import Version from "sap/base/util/Version";

  import Locale from "sap/ui/core/Locale";

  import Metadata from "sap/ui/base/Metadata";

  import { URI } from "sap/ui/core/library";

  import BaseObject from "sap/ui/base/Object";

  import {
    CustomCurrency,
    CustomUnit,
    CustomIslamicCalendarData,
  } from "sap/base/i18n/Formatting";

  /**
   * Collects and stores the configuration of the current environment.
   *
   * The Configuration is initialized once when the {@link sap.ui.core.Core} is created. There are different
   * ways to set the environment configuration (in ascending priority):
   * 	 - System defined defaults
   * 	 - Server wide defaults, read from /sap-ui-config.json
   * 	 - Properties of the global configuration object window["sap-ui-config"]
   * 	 - A configuration string in the data-sap-ui-config attribute of the bootstrap tag.
   * 	 - Individual data-sap-ui-xyz attributes of the bootstrap tag
   * 	 - Using URL parameters
   * 	 - Setters in this Configuration object (only for some parameters)
   *
   * That is, attributes of the DOM reference override the system defaults, URL parameters override the DOM
   * attributes (where empty URL parameters set the parameter back to its system default). Calling setters
   * at runtime will override any previous settings calculated during object creation.
   *
   * The naming convention for parameters is:
   * 	 - in the URL : sap-ui-PARAMETER-NAME="value"
   * 	 - in the DOM : data-sap-ui-PARAMETER-NAME="value"  where PARAMETER-NAME is the name
   *     of the parameter in lower case.
   *
   * Values of boolean parameters are case insensitive where "true" and "x" are interpreted as true.
   *
   * @deprecated As of version 1.120. There's no single replacement for this class. Instead, several facades
   * have been created for different topics, e.g. {@link module:sap/base/i18n/Localization Localization} for
   * settings related to localization, {@link module:sap/base/i18n/Formatting Formatting} for settings related
   * to data formatting, {@link module:sap/ui/core/Theming Theming} for theming related settings, {@link module:sap/ui/core/ControlBehavior ControlBehavior }
   * for settings that are typically required when implementing the behavior of a control, {@link module:sap/ui/security/Security Security }
   * for settings around security.
   *
   * Please check the individual methods of this class for their replacements, if any.
   */
  interface Configuration {
    /**
     * Enumerable list with available animation modes.
     *
     * This enumerable is used to validate the animation mode. Animation modes allow to specify different animation
     * scenarios or levels. The implementation of the Control (JavaScript or CSS) has to be done differently
     * for each animation mode.
     *
     * This enum is part of the 'sap/ui/core/Configuration' module export and must be accessed by the property
     * 'AnimationMode'.
     *
     * @since 1.50.0
     * @deprecated As of version 1.120. Please use module {@link module:sap/ui/core/AnimationMode AnimationMode }
     * instead.
     */
    AnimationMode: typeof AnimationMode;

    /**
     * Applies multiple changes to the configuration at once.
     *
     * If the changed settings contain localization related settings like `language` or calendarType,
     * then only a single `localizationChanged` event will be fired. As the framework has to inform all existing
     * components, elements, models etc. about localization changes, using `applySettings` can significantly
     * reduce the overhead for multiple changes, esp. when they occur after the UI has been created already.
     *
     * The `mSettings` can contain any property `xyz` for which a setter method `setXYZ` exists
     * in the API of this class. Similarly, values for the {@link sap.ui.core.Configuration.FormatSettings format settings }
     * API can be provided in a nested object with name `formatSettings`.
     *
     * @since 1.38.6
     * @deprecated As of version 1.120. without a replacement. As the different settings have been spread across
     * multiple new APIs (see {@link sap.ui.core.Configuration the deprecation hint for this class}), a common
     * API for changing multiple settings no longer made sense.
     *
     * Please check the individual methods of this class for their replacements and call those replacement methods
     * one by one.
     *
     * @returns Returns `this` to allow method chaining
     */
    applySettings(
      /**
       * Configuration options to apply
       */
      mSettings: object
    ): this;
    /**
     * Creates a new subclass of class sap.ui.core.Configuration with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    extend(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: object,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns whether the accessibility mode is enabled or not.
     *
     * @since 1.20
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/core/ControlBehavior.isAccessibilityEnabled ControlBehavior.isAccessibilityEnabled }
     * instead.
     *
     * @returns whether the accessibility mode is enabled or not
     */
    getAccessibility(): boolean;
    /**
     * Returns the list of active terminologies defined via the Configuration.
     *
     * @since 1.77.0
     * @deprecated As of version 1.118. Please use {@link module:sap/base/i18n/Localization.getActiveTerminologies Localization.getActiveTerminologies }
     * instead.
     *
     * @returns if no active terminologies are set, the default value `undefined` is returned.
     */
    getActiveTerminologies(): string[] | undefined;
    /**
     * URL of the allowlist service.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/security/Security.getAllowlistService Security.getAllowlistService }
     * instead.
     *
     * @returns allowlist service URL
     */
    getAllowlistService(): string;
    /**
     * Returns whether the animations are globally used.
     *
     * @deprecated As of version 1.50. Since 1.20, please use {@link module:sap/ui/core/ControlBehavior.getAnimationMode ControlBehavior.getAnimationMode }
     * instead.
     *
     * @returns whether the animations are globally used
     */
    getAnimation(): boolean;
    /**
     * Returns the current animation mode.
     *
     * @since 1.50.0
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/core/ControlBehavior.getAnimationMode ControlBehavior.getAnimationMode }
     * instead.
     *
     * @returns The current animationMode
     */
    getAnimationMode(): AnimationMode;
    /**
     * Base URLs to AppCacheBuster ETag-Index files.
     *
     * @deprecated As of version 1.120. without a replacement. Was only used within the implementation of the
     * `AppCacheBuster`.
     *
     * @returns array of base URLs
     */
    getAppCacheBuster(): string[];
    /**
     * The loading mode (sync|async|batch) of the AppCacheBuster (sync is default)
     *
     * @deprecated As of version 1.120. without a replacement. Was only used within the implementation of the
     * `AppCacheBuster`.
     *
     * @returns "sync" | "async" | "batch"
     */
    getAppCacheBusterMode(): string;
    /**
     * The name of the application to start or empty.
     *
     * @deprecated As of version 1.15.1. Please use {@link module:sap/ui/core/ComponentSupport ComponentSupport }
     * instead. See also {@link topic:82a0fcecc3cb427c91469bc537ebdddf Declarative API for Initial Components}.
     *
     * @returns name of the application
     */
    getApplication(): string;
    /**
     * Returns whether the framework automatically adds the ARIA role 'application' to the HTML body or not.
     *
     * @since 1.27.0
     * @deprecated As of version 1.120. without a replacement. Was only used in the implementation of the framework
     * itself.
     *
     * @returns Wether the ARIA role 'application' should be added to the HTML body or not
     */
    getAutoAriaBodyRole(): boolean;
    /**
     * Returns the calendar type which is being used in locale dependent functionality.
     *
     * When it's explicitly set by calling `setCalendar`, the set calendar type is returned. Otherwise, the
     * calendar type is determined by checking the format settings and current locale.
     *
     * @since 1.28.6
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.getCalendarType Formatting.getCalendarType }
     * instead.
     *
     * @returns the current calendar type, e.g. `Gregorian`
     */
    getCalendarType(): CalendarType;
    /**
     * Returns the calendar week numbering algorithm used to determine the first day of the week and the first
     * calendar week of the year, see {@link sap.ui.core.date.CalendarWeekNumbering}.
     *
     * @since 1.113.0
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.getCalendarWeekNumbering Formatting.getCalendarWeekNumbering }
     * instead.
     *
     * @returns The calendar week numbering algorithm
     */
    getCalendarWeekNumbering(): CalendarWeekNumbering;
    /**
     * Returns the used compatibility version for the given feature.
     *
     * @deprecated As of version 1.119. without a replacement. All features that have been controlled by a compatibility
     * version in UI5 1.x will abandon their legacy behavior, starting with the next major version. In other
     * words, they will behave as if compatibility version "edge" was configured. Due to this, no more access
     * to the compatibility version will be required starting with the next major version.
     *
     * @returns the used compatibility version
     */
    getCompatibilityVersion(
      /**
       * the key of desired feature
       */
      sFeature: string
    ): Version;
    /**
     * Returns whether the page runs in full debug mode.
     *
     * @deprecated As of version 1.120. without a replacement. All debug mode functionality is implemented within
     * the framework and should be transparent for application code. There's no need for a public accessor method.
     *
     * @returns Whether the page runs in full debug mode
     */
    getDebug(): boolean;
    /**
     * Name (ID) of a UI5 module that implements file share support.
     *
     * If no implementation is known, `undefined` is returned.
     *
     * The contract of the module is not defined by the configuration API.
     *
     * @since 1.102
     * @deprecated As of version 1.120. without a replacement. The configuration is only relevant for the `sap.ui.export`
     * library.
     *
     * @returns Module name (ID) of a file share support module
     */
    getFileShareSupport(): string | undefined;
    /**
     * Returns whether the Fiori2Adaptation is on.
     *
     * @deprecated As of version 1.120. without a replacement. All the Fiori 2.0 adaptation logic is handled
     * by the framework, there should be no need for apps to know about it.
     *
     * @returns false - no adaptation, true - full adaptation, comma-separated list - partial adaptation Possible
     * values: style, collapse, title, back, hierarchy
     */
    getFiori2Adaptation(): boolean | string;
    /**
     * Returns the URL from where the UI5 flexibility services are called; if empty, the flexibility services
     * are not called.
     *
     * @since 1.60.0
     * @deprecated As of version 1.120. without a replacement. Was only used by the Flexibility feature which
     * now resolves this on its own.
     *
     * @returns Flexibility services configuration
     */
    getFlexibilityServices(): object[];
    /**
     * Returns the format locale string with language and region code. Falls back to language configuration,
     * in case it has not been explicitly defined.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.getLanguageTag Formatting.getLanguageTag }
     * instead.
     *
     * @returns the format locale string with language and country code
     */
    getFormatLocale(): string;
    /**
     * Returns a configuration object that bundles the format settings of UI5.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting Formatting} instead.
     *
     * @returns A FormatSettings object.
     */
    getFormatSettings(): FormatSettings;
    /**
     * frameOptions mode (allow/deny/trusted).
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/security/Security.getFrameOptions Security.getFrameOptions }
     * instead.
     *
     * @returns frameOptions mode
     */
    getFrameOptions(): string;
    /**
     * Returns whether the UI5 control inspector is displayed.
     *
     * Has only an effect when the sap-ui-debug module has been loaded.
     *
     * @deprecated As of version 1.120. without a replacement. The inspect option is related to the very old
     * sap-ui-debug module. As there are now much better development tools (e.g. the UI5 Inpsector browser extension),
     * this option has been deprecated and the sap-ui-debug feature might be removed in future major versions
     * of UI5.
     *
     * @returns whether the UI5 control inspector is displayed
     */
    getInspect(): boolean;
    /**
     * Returns a string that identifies the current language.
     *
     * The value returned by config method in most cases corresponds to the exact value that has been configured
     * by the user or application or that has been determined from the user agent settings. It has not been
     * normalized, but has been validated against a relaxed version of {@link http://www.ietf.org/rfc/bcp/bcp47.txt BCP47},
     * allowing underscores ('_') instead of the suggested dashes ('-') and not taking the case of letters into
     * account.
     *
     * The exceptions mentioned above affect languages that have been specified via the URL parameter `sap-language`.
     * That parameter by definition represents an SAP logon language code ('ABAP language'). Most but not all
     * of these language codes are valid ISO639 two-letter languages and as such are valid BCP47 language tags.
     * For better BCP47 compliance, the framework maps the following non-BCP47 SAP logon codes to a BCP47 substitute:
     *
     * ```javascript
     *
     *    "ZH"  -->  "zh-Hans"         // script 'Hans' added to distinguish it from zh-Hant
     *    "ZF"  -->  "zh-Hant"         // ZF is not a valid ISO639 code, use the compliant language + script 'Hant'
     *    "1Q"  -->  "en-US-x-saptrc"  // special language code for supportability (tracing),
     *                                    represented as en-US with a private extension
     *    "2Q"  -->  "en-US-x-sappsd"  // special language code for supportability (pseudo translation),
     *                                    represented as en-US with a private extension
     *    "3Q"  -->  "en-US-x-saprigi" // special language code for the Rigi pseudo language,
     *                                    represented as en-US with a private extension
     * ```
     *
     *
     * For a normalized BCP47 tag, call {@link #.getLanguageTag Configuration.getLanguageTag} or call {@link #.getLocale Configuration.getLocale }
     * to get a {@link sap.ui.core.Locale Locale} object matching the language.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.getLanguage Localization.getLanguage }
     * instead.
     *
     * @returns Language string as configured
     */
    getLanguage(): string;
    /**
     * Returns a BCP47-compliant language tag for the current language.
     *
     * The return value of config method is especially useful for an HTTP `Accept-Language` header.
     *
     * Retrieves the modern locale, e.g. sr-Latn (Serbian (Latin)), he (Hebrew), yi (Yiddish)
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.getLanguageTag Localization.getLanguageTag }
     * instead.
     *
     * @returns The language tag for the current language, conforming to BCP47
     */
    getLanguageTag(): string;
    /**
     * Returns a Locale object for the current language.
     *
     * The Locale is derived from the {@link #.getLanguage language} property.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.getLanguageTag Localization.getLanguageTag }
     * instead.
     *
     * @returns The locale
     */
    getLocale(): Locale;
    /**
     * Flag whether a Component should load the manifest first.
     *
     * @since 1.33.0
     * @deprecated As of version 1.120. 'manifest-first' is the default for the {@link sap.ui.core.Component.create Component.create }
     * factory.
     *
     * @returns true if a Component should load the manifest first
     */
    getManifestFirst(): boolean;
    /**
     * Returns a metadata object for class sap.ui.core.Configuration.
     *
     *
     * @returns Metadata object describing this class
     */
    getMetadata(): Metadata;
    /**
     * Returns whether there should be an exception on any duplicate element IDs.
     *
     * @deprecated As of version 1.120. without a replacement. Future major versions of UI5 will always report
     * duplicate IDs as an error as the framework heavily relies on their uniqueness.
     *
     * @returns whether there should be an exception on any duplicate element IDs
     */
    getNoDuplicateIds(): boolean;
    /**
     * Returns whether the text origin information is collected.
     *
     * @deprecated As of version 1.120. The origin info was intended as a mean to track down translation issues
     * to the origin of a translated text. Meanwhile, with the pseudo logon language 3Q, a similar, more lightweight
     * feature exists for use with OpenUI5 or SAPUI5 libraries.
     *
     * @returns whether the text info is collected
     */
    getOriginInfo(): boolean;
    /**
     * The name of the root component to start or empty.
     *
     * @deprecated As of version 1.95. Please use {@link module:sap/ui/core/ComponentSupport ComponentSupport }
     * instead. See also {@link topic:82a0fcecc3cb427c91469bc537ebdddf Declarative API for Initial Components}.
     *
     * @returns name of the root component
     */
    getRootComponent(): string;
    /**
     * Returns whether the page uses the RTL text direction.
     *
     * If no mode has been explicitly set (neither `true` nor `false`), the mode is derived from the current
     * language setting.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.getRTL Localization.getRTL }
     * instead.
     *
     * @returns whether the page uses the RTL text direction
     */
    getRTL(): boolean;
    /**
     * Returns an SAP logon language for the current language.
     *
     * It will be returned in uppercase. e.g. "EN", "DE"
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.getSAPLogonLanguage Localization.getSAPLogonLanguage }
     * instead.
     *
     * @returns The SAP logon language code for the current language
     */
    getSAPLogonLanguage(): string;
    /**
     * Returns the security token handlers of an OData V4 model.
     * See:
     * 	{@link #.setSecurityTokenHandlers}
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/security/Security.getSecurityTokenHandlers Security.getSecurityTokenHandlers }
     * instead.
     *
     * @returns the security token handlers (an empty array if there are none)
     */
    getSecurityTokenHandlers(): Array<(p1: URI) => Promise<any>>;
    /**
     * Flag if statistics are requested.
     *
     * Flag set by TechnicalInfo Popup will also be checked. So it's active if set by URL parameter or manually
     * via TechnicalInfo.
     *
     * @since 1.106.0
     * @deprecated As of version 1.120. without a replacment. The configuration is only relevant within the
     * framework.
     *
     * @returns Whether statistics are enabled
     */
    getStatisticsEnabled(): boolean;
    /**
     * Returns the theme name
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/ui/core/Theming.getTheme Theming.getTheme }
     * instead.
     *
     * @returns the theme name
     */
    getTheme(): string;
    /**
     * Retrieves the configured IANA timezone ID.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.getTimezone Localization.getTimezone }
     * instead.
     *
     * @returns The configured IANA timezone ID, e.g. "America/New_York"
     */
    getTimezone(): string;
    /**
     * Prefix to be used for automatically generated control IDs. Default is a double underscore "__".
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.base.ManagedObjectMetadata.getUIDPrefix ManagedObjectMetadata.getUIDPrefix }
     * instead.
     *
     * @returns the prefix to be used
     */
    getUIDPrefix(): string;
    /**
     * Returns the version of the framework.
     *
     * Similar to `sap.ui.version`.
     *
     * @deprecated As of version 1.120. Please use the async {@link module:sap/ui/VersionInfo.load VersionInfo.load }
     * instead.
     *
     * @returns the version
     */
    getVersion(): Version;
    /**
     * URL of the whitelist service.
     *
     * @deprecated As of version 1.85. Use {@link module:sap/ui/security/Security.getAllowlistService Security.getAllowlistService }
     * instead. SAP strives to replace insensitive terms with inclusive language. Since APIs cannot be renamed
     * or immediately removed for compatibility reasons, this API has been deprecated.
     *
     * @returns whitelist service URL
     */
    getWhitelistService(): string;
    /**
     * Sets the current animation mode.
     *
     * Expects an animation mode as string and validates it. If a wrong animation mode was set, an error is
     * thrown. If the mode is valid it is set, then the attributes `data-sap-ui-animation` and `data-sap-ui-animation-mode`
     * of the HTML document root element are also updated. If the `animationMode` is `Configuration.AnimationMode.none`
     * the old `animation` property is set to `false`, otherwise it is set to `true`.
     *
     * @since 1.50.0
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/core/ControlBehavior.setAnimationMode ControlBehavior.setAnimationMode }
     * instead.
     */
    setAnimationMode(
      /**
       * A valid animation mode
       */
      sAnimationMode: AnimationMode
    ): void;
    /**
     * Sets the new calendar type to be used from now on in locale dependent functionality (for example, formatting,
     * translation texts, etc.).
     *
     * @since 1.28.6
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.setCalendarType Formatting.setCalendarType }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setCalendarType(
      /**
       * the new calendar type. Set it with null to clear the calendar type and the calendar type is calculated
       * based on the format settings and current locale.
       */
      sCalendarType: (CalendarType | keyof typeof CalendarType) | null
    ): this;
    /**
     * Sets the calendar week numbering algorithm which is used to determine the first day of the week and the
     * first calendar week of the year, see {@link sap.ui.core.date.CalendarWeekNumbering}.
     *
     * @since 1.113.0
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.setCalendarWeekNumbering Formatting.setCalendarWeekNumbering }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setCalendarWeekNumbering(
      /**
       * The calendar week numbering algorithm
       */
      sCalendarWeekNumbering:
        | CalendarWeekNumbering
        | keyof typeof CalendarWeekNumbering
    ): this;
    /**
     * Sets a new format locale to be used from now on for retrieving locale specific formatters. Modifying
     * this setting does not have an impact on the retrieval of translated texts!
     *
     * Can either be set to a concrete value (a BCP47 or Java locale compliant language tag) or to `null`. When
     * set to `null` (default value) then locale specific formatters are retrieved for the current language.
     *
     * After changing the format locale, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link #.setLanguage} for details and restrictions.
     *
     * **Note**: When a format locale is set, it has higher priority than a number, date or time format defined
     * with a call to `setLegacyNumberFormat`, `setLegacyDateFormat` or `setLegacyTimeFormat`.
     *
     * **Note**: See documentation of {@link #.setLanguage} for restrictions.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.setLanguageTag Formatting.setLanguageTag }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setFormatLocale(
      /**
       * the new format locale as a BCP47 compliant language tag; case doesn't matter and underscores can be used
       * instead of dashes to separate components (compatibility with Java Locale IDs)
       */
      sFormatLocale: string | null
    ): this;
    /**
     * Sets a new language to be used from now on for language/region dependent functionality (e.g. formatting,
     * data types, translated texts, ...).
     *
     * When the language can't be interpreted as a BCP47 language (using the relaxed syntax described in {@link #.getLanguage Configuration.getLanguage},
     * an error will be thrown.
     *
     * When the language has changed, the Core will fire its {@link sap.ui.core.Core#event:localizationChanged localizationChanged }
     * event.
     *
     * Restrictions:
     *
     * The framework **does not** guarantee that already created, language dependent objects will be updated
     * by config call. It therefore remains best practice for applications to switch the language early, e.g.
     * before any language dependent objects are created. Applications that need to support more dynamic changes
     * of the language should listen to the `localizationChanged` event and adapt all language dependent objects
     * that they use (e.g. by rebuilding their UI).
     *
     * Currently, the framework notifies the following objects about a change of the localization settings before
     * it fires the `localizationChanged` event:
     *
     *
     * 	 - date and number data types that are used in property bindings or composite bindings in existing Elements,
     *     Controls, UIAreas or Components
     * 	 - ResourceModels currently assigned to the Core, a UIArea, Component, Element or Control
     * 	 - Elements or Controls that implement the `onLocalizationChanged` hook
     *
     * It furthermore derives the RTL mode from the new language, if no explicit RTL mode has been set. If the
     * RTL mode changes, the following additional actions will be taken:
     *
     *
     * 	 - the URLs of already loaded library theme files will be changed
     * 	 - the `dir` attribute of the page will be changed to reflect the new mode.
     * 	 - all UIAreas will be invalidated (which results in a rendering of the whole UI5 UI)
     *
     * config method does not accept SAP language codes for `sLanguage`. Instead, a second parameter `sSAPLogonLanguage`
     * can be provided with an SAP language code corresponding to the given language. A given value will be
     * returned by the {@link #.getSAPLogonLanguage Configuration.getSAPLogonLanguage} method. It is up to the
     * caller to provide a consistent pair of BCP47 language and SAP language code. The SAP language code is
     * only checked to be of length 2 and must consist of letters or digits only.
     *
     * **Note**: When using config method please take note of and respect the above mentioned restrictions.
     * See:
     * 	http://scn.sap.com/docs/DOC-14377
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.setLanguage Localization.setLanguage }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setLanguage(
      /**
       * the new language as a BCP47 compliant language tag; case doesn't matter and underscores can be used instead
       * of dashes to separate components (compatibility with Java Locale IDs)
       */
      sLanguage: string,
      /**
       * SAP language code that corresponds to the `sLanguage`; if a value is specified, future calls to `getSAPLogonLanguage`
       * will return that value; if no value is specified, the framework will use the ISO639 language part of
       * `sLanguage` as SAP Logon language.
       */
      sSAPLogonLanguage?: string
    ): this;
    /**
     * Sets the character orientation mode to be used from now on.
     *
     * Can either be set to a concrete value (true meaning right-to-left, false meaning left-to-right) or to
     * `null` which means that the character orientation mode should be derived from the current language (incl.
     * region) setting.
     *
     * After changing the character orientation mode, the framework tries to update localization specific parts
     * of the UI. See the documentation of {@link #.setLanguage} for details and restrictions.
     *
     * **Note**: See documentation of {@link #.setLanguage} for restrictions.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.setRTL Localization.setRTL }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setRTL(
      /**
       * new character orientation mode or `null`
       */
      bRTL: boolean | null
    ): this;
    /**
     * Sets the security token handlers for an OData V4 model. See chapter {@link https://ui5.sap.com/#/topic/9613f1f2d88747cab21896f7216afdac/section_STH Security Token Handling}.
     * See:
     * 	{@link #.getSecurityTokenHandlers}
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/ui/security/Security.setSecurityTokenHandlers Security.setSecurityTokenHandlers }
     * instead.
     */
    setSecurityTokenHandlers(
      /**
       * The security token handlers
       */
      aSecurityTokenHandlers: Array<(p1: URI) => Promise<any>>
    ): void;
    /**
     * Allows setting the theme name
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/ui/core/Theming.setTheme Theming.setTheme }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setTheme(
      /**
       * the theme name
       */
      sTheme: string
    ): this;
    /**
     * Sets the timezone such that all date and time based calculations use config timezone.
     *
     * **Important:** It is strongly recommended to only use config API at the earliest point of time while
     * initializing a UI5 app. A later adjustment of the time zone should be avoided. It can lead to unexpected
     * data inconsistencies in a running application, because date objects could still be related to a previously
     * configured time zone. Instead, the app should be completely restarted with the new time zone. For more
     * information, see {@link https://ui5.sap.com/#/topic/6c9e61dc157a40c19460660ece8368bc Dates, Times, Timestamps, and Time Zones}.
     *
     * When the timezone has changed, the Core will fire its {@link sap.ui.core.Core#event:localizationChanged localizationChanged }
     * event.
     *
     * @since 1.99.0
     * @deprecated As of version 1.119. Please use {@link module:sap/base/i18n/Localization.setTimezone Localization.setTimezone }
     * instead.
     *
     * @returns `this` to allow method chaining
     */
    setTimezone(
      /**
       * IANA timezone ID, e.g. "America/New_York". Use `null` to reset the timezone to the browser's local timezone.
       * An invalid IANA timezone ID will fall back to the browser's timezone.
       */
      sTimezone?: string | null
    ): this;
  }
  const Configuration: Configuration;
  /**
   * @deprecated As of version 1.120. There's no single replacement for this class. Instead, several facades
   * have been created for different topics, e.g. {@link module:sap/base/i18n/Localization Localization} for
   * settings related to localization, {@link module:sap/base/i18n/Formatting Formatting} for settings related
   * to data formatting, {@link module:sap/ui/core/Theming Theming} for theming related settings, {@link module:sap/ui/core/ControlBehavior ControlBehavior }
   * for settings that are typically required when implementing the behavior of a control, {@link module:sap/ui/security/Security Security }
   * for settings around security.
   *
   * Please check the individual methods of this class for their replacements, if any.
   */
  export default Configuration;

  /**
   * Enumerable list with available animation modes.
   *
   * This enumerable is used to validate the animation mode. Animation modes allow to specify different animation
   * scenarios or levels. The implementation of the Control (JavaScript or CSS) has to be done differently
   * for each animation mode.
   *
   * This enum is part of the 'sap/ui/core/Configuration' module export and must be accessed by the property
   * 'AnimationMode'.
   *
   * @since 1.50.0
   * @deprecated As of version 1.120. Please use module {@link module:sap/ui/core/AnimationMode AnimationMode }
   * instead.
   */
  enum AnimationMode {
    /**
     * `basic` can be used for a reduced, more light-weight set of animations.
     */
    basic = "undefined",
    /**
     * `full` represents a mode with unrestricted animation capabilities.
     */
    full = "undefined",
    /**
     * `minimal` includes animations of fundamental functionality.
     */
    minimal = "undefined",
    /**
     * `none` deactivates the animation completely.
     */
    none = "undefined",
  }
  /**
   * Encapsulates configuration settings that are related to data formatting/parsing.
   *
   * **Note:** When format configuration settings are modified through this class, UI5 only ensures that formatter
   * objects created after that point in time will honor the modifications. To be on the safe side, applications
   * should do any modifications early in their lifecycle or recreate any model/UI that is locale dependent.
   *
   * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting Formatting} instead.
   */
  export class FormatSettings extends BaseObject {
    constructor();

    /**
     * Creates a new subclass of class sap.ui.core.Configuration.FormatSettings with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FormatSettings>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.Configuration.FormatSettings.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Adds custom currencies. There is a special currency code named "DEFAULT" that is optional. In case it
     * is set it is used for all currencies not contained in the list, otherwise currency digits as defined
     * by the CLDR are used as a fallback.
     *
     * **Note:** Adding custom currencies affects all applications running with the current UI5 core instance.
     *
     * @since 1.120
     */
    addCustomCurrencies(
      /**
       * A map with the currency code as key and a custom currency definition as value; already existing custom
       * currencies are replaced, new ones are added; the custom currency code must contain at least one non-digit
       * character, so that the currency part can be distinguished from the amount part; see {@link #.getCustomCurrencies Formatting.getCustomCurrencies }
       * for an example
       */
      mCurrencies?: Record<string, CustomCurrency>
    ): void;
    /**
     * Adds custom units.
     *
     * **Note:** Adding custom units affects all applications running with the current UI5 core instance.
     *
     * @since 1.123
     */
    addCustomUnits(
      /**
       * A map with the unit code as key and a custom unit definition as value; already existing custom units
       * are replaced, new ones are added; see {@link #.getCustomUnits Formatting.getCustomUnits} for an example
       */
      mUnits: Record<string, CustomUnit>
    ): void;
    /**
     * Gets the custom currencies that have been set via {@link #.addCustomCurrencies Formatting.addCustomCurrencies }
     * or {@link #.setCustomCurrencies Formatting.setCustomCurrencies}. There is a special currency code named
     * "DEFAULT" that is optional. If it is set it is used for all currencies not contained in the list, otherwise
     * currency digits as defined by the CLDR are used as a fallback.
     *
     * @since 1.120
     *
     * @returns A map with the currency code as key and a custom currency definition containing the number of
     * decimals as value; or `undefined` if there are no custom currencies
     */
    getCustomCurrencies(): Record<string, CustomCurrency> | undefined;
    /**
     * Gets the custom units that have been set via {@link #.addCustomUnits Formatting.addCustomUnits} or {@link #.setCustomUnits Formatting.setCustomUnits}.
     *
     * @since 1.123
     *
     * @returns A map with the unit code as key and a custom unit definition containing a display name and different
     * unit patterns as value; or `undefined` if there are no custom units
     */
    getCustomUnits(): Record<string, CustomUnit> | undefined;
    /**
     * Returns the currently set date pattern or undefined if no pattern has been defined.
     *
     * @since 1.120
     *
     * @returns The resulting date pattern
     */
    getDatePattern(
      /**
       * The date style (short, medium, long or full)
       */
      sStyle: "short" | "medium" | "long" | "full"
    ): string;
    /**
     * Returns the locale to be used for formatting.
     *
     * If no such locale has been defined, this method falls back to the language, see {@link sap.ui.core.Configuration.getLanguage Configuration.getLanguage()}.
     *
     * If any user preferences for date, time or number formatting have been set, and if no format locale has
     * been specified, then a special private use subtag is added to the locale, indicating to the framework
     * that these user preferences should be applied.
     *
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/Formatting.getLanguageTag Formatting.getLanguageTag }
     * instead.
     *
     * @returns the format locale
     */
    getFormatLocale(): Locale;
    /**
     * Returns the currently set customizing data for Islamic calendar support.
     *
     * See: {@link module:sap/base/i18n/Formatting.CustomIslamicCalendarData}
     *
     * @since 1.120
     *
     * @returns Returns an array that contains the customizing data. Each element in the array has properties:
     * dateFormat, islamicMonthStart, gregDate. For details, please see {@link #.setCustomIslamicCalendarData}
     */
    getLegacyDateCalendarCustomizing(): CustomIslamicCalendarData[] | undefined;
    /**
     * Returns the currently set number symbol of the given type or undefined if no symbol has been defined.
     *
     * @since 1.120
     *
     * @returns A non-numerical symbol used as part of a number for the given type, e.g. for locale de_DE:
     *
     * 	 - "group": "." (grouping separator)
     * 	 - "decimal": "," (decimal separator)
     * 	 - "plusSign": "+" (plus sign)
     * 	 - "minusSign": "-" (minus sign)
     */
    getNumberSymbol(
      /**
       * the type of symbol
       */
      sType: "group" | "decimal" | "plusSign" | "minusSign"
    ): string;
    /**
     * Returns the currently set time pattern or undefined if no pattern has been defined.
     *
     * @since 1.120
     *
     * @returns The resulting time pattern
     */
    getTimePattern(
      /**
       * The time style (short, medium, long or full)
       */
      sStyle: "short" | "medium" | "long" | "full"
    ): string;
    /**
     * Returns current trailingCurrencyCode configuration for new NumberFormatter instances
     *
     * @since 1.120
     *
     * @returns Whether currency codes shall always be placed after the numeric value
     */
    getTrailingCurrencyCode(): boolean;
    /**
     * Replaces existing custom currencies by the given custom currencies. There is a special currency code
     * named "DEFAULT" that is optional. In case it is set, it is used for all currencies not contained in the
     * list, otherwise currency digits as defined by the CLDR are used as a fallback.
     *
     * **Note:** Setting custom units affects all applications running with the current UI5 core instance.
     * See:
     * 	{@link module:sap/base/i18n/Formatting.addCustomCurrencies Formatting.addCustomCurrencies}
     *
     * @since 1.120
     */
    setCustomCurrencies(
      /**
       * A map with the currency code as key and a custom currency definition as value; the custom currency code
       * must contain at least one non-digit character, so that the currency part can be distinguished from the
       * amount part; `mCurrencies` replaces the current custom currencies; if not given, all custom currencies
       * are deleted; see {@link #.getCustomCurrencies Formatting.getCustomCurrencies} for an example
       */
      mCurrencies?: Record<string, CustomCurrency>
    ): void;
    /**
     * Replaces existing custom units by the given custom units.
     *
     * **Note:** Setting custom units affects all applications running with the current UI5 core instance.
     * See:
     * 	{@link module:sap/base/i18n/Formatting.addCustomUnits Formatting.addCustomUnits}
     *
     * @since 1.123
     */
    setCustomUnits(
      /**
       * A map with the unit code as key and a custom unit definition as value; `mUnits` replaces the current
       * custom units; if not given, all custom units are deleted; see {@link #.getCustomUnits Formatting.getCustomUnits }
       * for an example
       */
      mUnits?: Record<string, CustomUnit>
    ): void;
    /**
     * Defines the preferred format pattern for the given date format style.
     *
     * Calling this method with a null or undefined pattern removes a previously set pattern.
     *
     * If a pattern is defined, it will be preferred over patterns derived from the current locale.
     *
     * See class {@link sap.ui.core.format.DateFormat DateFormat} for details about the pattern syntax.
     *
     * After changing the date pattern, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage} for
     * details and restrictions.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    setDatePattern(
      /**
       * must be one of short, medium, long or full.
       */
      sStyle: string,
      /**
       * the format pattern to be used in LDML syntax.
       */
      sPattern: string
    ): this;
    /**
     * Defines the day used as the first day of the week.
     *
     * The day is set as an integer value between 0 (Sunday) and 6 (Saturday). Calling this method with a null
     * or undefined symbol removes a previously set value.
     *
     * If a value is defined, it will be preferred over values derived from the current locale.
     *
     * Usually in the US the week starts on Sunday while in most European countries on Monday. There are special
     * cases where you want to have the first day of week set independent of the user locale.
     *
     * After changing the first day of week, the framework tries to update localization specific parts of the
     * UI. See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage }
     * for details and restrictions.
     *
     * @deprecated As of version 1.113. Use {@link sap.ui.core.Configuration.FormatSettings#setCalendarWeekNumbering FormatSettings#setCalendarWeekNumbering }
     * instead.
     *
     * @returns Returns `this` to allow method chaining
     */
    setFirstDayOfWeek(
      /**
       * must be an integer value between 0 and 6
       */
      iValue: int
    ): this;
    /**
     * Allows to specify the customizing data for Islamic calendar support
     *
     * See: {@link module:sap/base/i18n/Formatting.CustomIslamicCalendarData}
     *
     * @since 1.120
     */
    setLegacyDateCalendarCustomizing(
      /**
       * Contains the customizing data for the support of Islamic calendar. One JSON object in the array represents
       * one row of data from Table TISLCAL
       */
      aCustomCalendarData: CustomIslamicCalendarData[]
    ): void;
    /**
     * Allows to specify one of the legacy ABAP date formats.
     *
     * This method modifies the date patterns for 'short' and 'medium' style with the corresponding ABAP format.
     * When called with a null or undefined format id, any previously applied format will be removed.
     *
     * After changing the legacy date format, the framework tries to update localization specific parts of the
     * UI. See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage }
     * for details and restrictions.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    setLegacyDateFormat(
      /**
       * ID of the ABAP date format, `""` will reset the date patterns for 'short' and 'medium' style to the locale-specific
       * ones.
       */
      sFormatId?:
        | ""
        | "1"
        | "2"
        | "3"
        | "4"
        | "5"
        | "6"
        | "7"
        | "8"
        | "9"
        | "A"
        | "B"
        | "C"
    ): this;
    /**
     * Allows to specify one of the legacy ABAP number format.
     *
     * This method will modify the 'group' and 'decimal' symbols. When called with a null or undefined format
     * id, any previously applied format will be removed.
     *
     * After changing the legacy number format, the framework tries to update localization specific parts of
     * the UI. See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage }
     * for details and restrictions.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    setLegacyNumberFormat(
      /**
       * ID of the ABAP number format set, `""` will reset the 'group' and 'decimal' symbols to the locale-specific
       * ones.
       */
      sFormatId?: "" | " " | "X" | "Y"
    ): this;
    /**
     * Allows to specify one of the legacy ABAP time formats.
     *
     * This method sets the time patterns for 'short' and 'medium' style to the corresponding ABAP formats and
     * sets the day period texts to "AM"/"PM" or "am"/"pm" respectively. When called with a null or undefined
     * format id, any previously applied format will be removed.
     *
     * After changing the legacy time format, the framework tries to update localization specific parts of the
     * UI. See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage }
     * for details and restrictions.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    setLegacyTimeFormat(
      /**
       * ID of the ABAP time format, `""` will reset the time patterns for 'short' and 'medium' style and the
       * day period texts to the locale-specific ones.
       */
      sFormatId?: "" | "0" | "1" | "2" | "3" | "4"
    ): this;
    /**
     * Defines the string to be used for the given number symbol.
     *
     * Calling this method with a null or undefined symbol removes a previously set symbol string. Note that
     * an empty string is explicitly allowed.
     *
     * If a symbol is defined, it will be preferred over symbols derived from the current locale.
     *
     * See class {@link sap.ui.core.format.NumberFormat NumberFormat} for details about the symbols.
     *
     * After changing the number symbol, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage} for
     * details and restrictions.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    setNumberSymbol(
      /**
       * the type of symbol
       */
      sType: "group" | "decimal" | "plusSign" | "minusSign",
      /**
       * will be used to represent the given symbol type
       */
      sSymbol: string
    ): this;
    /**
     * Defines the preferred format pattern for the given time format style.
     *
     * Calling this method with a null or undefined pattern removes a previously set pattern.
     *
     * If a pattern is defined, it will be preferred over patterns derived from the current locale.
     *
     * See class {@link sap.ui.core.format.DateFormat DateFormat} for details about the pattern syntax.
     *
     * After changing the time pattern, the framework tries to update localization specific parts of the UI.
     * See the documentation of {@link sap.ui.core.Configuration.setLanguage Configuration.setLanguage} for
     * details and restrictions.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    setTimePattern(
      /**
       * must be one of short, medium, long or full.
       */
      sStyle: string,
      /**
       * the format pattern to be used in LDML syntax.
       */
      sPattern: string
    ): this;
    /**
     * Define whether the NumberFormatter shall always place the currency code after the numeric value, with
     * the only exception of right-to-left locales, where the currency code shall be placed before the numeric
     * value. Default configuration setting is `true`.
     *
     * When set to `false` the placement of the currency code is done dynamically, depending on the configured
     * locale using data provided by the Unicode Common Locale Data Repository (CLDR).
     *
     * Each currency instance ({@link sap.ui.core.format.NumberFormat.getCurrencyInstance NumberFormat.getCurrencyInstance})
     * will be created with this setting unless overwritten on instance level.
     *
     * @since 1.120
     */
    setTrailingCurrencyCode(
      /**
       * Whether currency codes shall always be placed after the numeric value
       */
      bTrailingCurrencyCode: boolean
    ): void;
  }
}

declare module "sap/ui/core/Control" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import { AccessibilityInfo, BusyIndicatorSize } from "sap/ui/core/library";

  import {
    default as ElementMetadata,
    ControlRenderer,
  } from "sap/ui/core/ElementMetadata";

  import {
    default as ManagedObject,
    PropertyBindingInfo,
  } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * Base Class for Controls.
   *
   * Controls provide the following features:
   * 	 - **Rendering**: the `RenderManager` only expects instances of class `Control` in its {@link sap.ui.core.RenderManager#renderControl renderControl }
   *     method. By convention, each control class has an associated static class that takes care of rendering
   *     the control (its 'Renderer').
   * 	 - **show / hide**: a control can be hidden, although it is still part of the control tree, see property
   *     {@link #getVisible visible}
   * 	 - **local busy indicator**: marks a control visually as 'busy', see properties {@link #getBusy busy }
   *     and {@link #getBusyIndicatorDelay busyIndicatorDelay}
   * 	 - **field groups**: by assigning the same group ID to a set of editable controls, they form a group
   *     which can be validated together. See property {@link #getFieldGroupIds fieldGroupIds} and event {@link #event:validateFieldGroup validateFieldGroup}.
   *     The term field was chosen as most often this feature will be used to group editable fields in
   *     a form. See the documentation for {@link https://ui5.sap.com/#/topic/5b0775397e394b1fb973fa207554003e Field Groups }
   *     for more details.
   * 	 - **custom style classes**: all controls allow to add custom CSS classes to their rendered DOM without
   *     modifying their renderer code. See methods {@link #addStyleClass addStyleClass}, {@link #removeStyleClass removeStyleClass},
   *     {@link #toggleStyleClass toggleStyleClass} and {@link #hasStyleClass hasStyleClass}.
   *  The necessary implementation is encapsulated in {@link sap.ui.core.CustomStyleClassSupport CustomStyleClassSupport }
   * and can be applied to selected element classes as well.
   * 	 - **browser events**: by calling the methods {@link #attachBrowserEvent attachBrowserEvent} and {@link #detachBrowserEvent detachBrowserEvent},
   *     consumers can let the control class take care of registering / de-registering a given set of event listeners
   *     to the control's root DOM node. The framework will adapt the registration whenever the DOM node changes
   *     (e.g. before or after rendering or when the control is destroyed).
   *
   * See section "{@link https://ui5.sap.com/#/topic/8dcab0011d274051808f959800cabf9f Developing Controls}"
   * in the documentation for an introduction to control development.
   */
  export default abstract class Control extends UI5Element {
    /**
     * Creates and initializes a new control with the given `sId` and settings.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Object with initial settings for the new control
       */
      mSettings?: $ControlSettings
    );
    /**
     * Creates and initializes a new control with the given `sId` and settings.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Optional ID for the new control; generated automatically if no non-empty ID is given Note: this can be
       * omitted, no matter whether `mSettings` will be given or not!
       */
      sId?: string,
      /**
       * Object with initial settings for the new control
       */
      mSettings?: $ControlSettings
    );

    /**
     * Defines a new subclass of Control with the name `sClassName` and enriches it with the information contained
     * in `oClassInfo`.
     *
     * `oClassInfo` can contain the same information that {@link sap.ui.core.Element.extend} already accepts,
     * plus the following `renderer` property:
     *
     * Example:
     * ```javascript
     *
     * Control.extend("sap.mylib.MyControl", {
     *   metadata : {
     *     library : "sap.mylib",
     *     properties : {
     *       text : "string",
     *       width : "sap.ui.core.CSSSize"
     *     }
     *   },
     *   renderer: {
     *     apiVersion: 2,
     *     render: function(oRM, oControl) {
     *       oRM.openStart("div", oControl);
     *       oRM.style("width", oControl.getWidth());
     *       oRM.openEnd();
     *       oRM.text(oControl.getText());
     *       oRM.close("div");
     *     }
     *   }
     * });
     * ```
     *
     *
     * There are multiple ways how a renderer can be specified:
     * 	 - As a **plain object**: The object will be used to create a new renderer by using {@link sap.ui.core.Renderer.extend }
     *     to extend the renderer of the base class of this control. The new renderer will have the same global
     *     name as this control class with the additional suffix 'Renderer'.
     *  **Note:** The `Renderer.extend` method expects a plain object (no prototype chain).
     * 	 - As a **function**: The given function will be used as `render` function of a new renderer; the renderer
     *     will be created in the same way as described for the plain object case.
     * 	 - As a **ready-made renderer**, e.g. imported from the corresponding renderer module. As renderers
     *     are simple objects (not instances of a specific class), some heuristic is used to distinguish renderers
     *     from the plain object case above: An object is assumed to be a ready-made renderer when it has
     *     a `render` function and either is already exposed under the expected global name or has an `extend` method.
     *
     * 	 - As a **fully qualified name**: The name will be looked up as a global property. If not defined, a
     *     module name will be derived from the global name (dots replaced by slashes), the module will be required
     *     and provides the renderer, either as AMD export or via the named global property.
     * 	 - **Omitting the `renderer` property** or setting it to `undefined`: The fully qualified name of the
     *     renderer will be derived from the fully qualified name of the control by adding the suffix "Renderer".
     *     The renderer then is retrieved in the same way as described for the fully qualified name case.
     *
     * 	 - **`null` or empty string**: The control will have no renderer, a call to `oControl.getMetadata().getRenderer()`
     *     will return `undefined`.
     *
     * If the resulting renderer is incomplete (has no `render` function) or if it cannot be found at all, rendering
     * of the control will be skipped.
     *
     * **Note:** The `apiVersion: 2` flag is required to enable in-place rendering technology. Before setting
     * this property, please ensure that the constraints documented in section "Contract for Renderer.apiVersion
     * 2" of the {@link sap.ui.core.RenderManager RenderManager} API documentation are fulfilled.
     *
     *
     * @returns Constructor of the newly created class
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class to be created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Control>,
      /**
       * Constructor function for the metadata object. If not given, it defaults to `sap.ui.core.ElementMetadata`.
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a list of all controls with a field group ID. See {@link sap.ui.core.Control#checkFieldGroupIds Control.prototype.checkFieldGroupIds }
     * for a description of the `vFieldGroupIds` parameter.
     *
     * If possible please use the respective method on a Control instance (see {@link sap.ui.core.Control#getControlsByFieldGroupId}).
     * The control method only respects aggregated child controls, which is more effective and should be sufficient
     * for most use-cases.
     *
     * @since 1.118
     *
     * @returns The list of controls with matching field group IDs
     */
    static getControlsByFieldGroupId(
      /**
       * ID of the field group or an array of field group IDs to match
       */
      vFieldGroupIds?: string | string[]
    ): Control[];
    /**
     * Returns a metadata object for class sap.ui.core.Control.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * The string given as "sStyleClass" will be added to the "class" attribute of this control's root HTML
     * element.
     *
     * This method is intended to be used to mark controls as being of a special type for which special styling
     * can be provided using CSS selectors that reference this style class name.
     *
     *
     * ```javascript
     *
     * Example:
     *    myButton.addStyleClass("myRedTextButton"); // add a CSS class to one button instance
     *
     * ...and in CSS:
     *    .myRedTextButton {
     *       color: red;
     *    }
     * ```
     *
     *
     * This will add the CSS class "myRedTextButton" to the Button HTML and the CSS code above will then make
     * the text in this particular button red.
     *
     * Only characters allowed inside HTML attributes are allowed. Quotes are not allowed and this method will
     * ignore any strings containing quotes. Strings containing spaces are interpreted as multiple custom style
     * classes which are split by space and can be removed individually later by calling removeStyleClass. Multiple
     * calls with the same sStyleClass will have no different effect than calling once. If sStyleClass is null,
     * empty string or it contains quotes, the call is ignored.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    addStyleClass(
      /**
       * the CSS class name to be added
       */
      sStyleClass: string
    ): this;
    /**
     * Defines whether the user can select text inside this control. Defaults to `true` as long as this method
     * has not been called.
     *
     * **Note:**This only works in Safari; for Firefox the element's style must be set to:
     * ```javascript
     *
     *   -moz-user-select: none;
     * ```
     *  in order to prevent text selection.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    allowTextSelection(
      /**
       * whether to allow text selection or not
       */
      bAllow: boolean
    ): this;
    /**
     * Allows binding handlers for any native browser event to the root HTML element of this Control. This internally
     * handles DOM element replacements caused by re-rendering.
     *
     * **IMPORTANT:**
     *  This should be only used as FALLBACK when the Control events do not cover a specific use-case! Always
     * try using SAPUI5 control events, as e.g. accessibility-related functionality is then provided automatically.
     * E.g. when working with a `sap.m.Button`, always use the Button's "press" event, not the native "click"
     * event, because "press" is also guaranteed to be fired when certain keyboard activity is supposed to trigger
     * the Button.
     *
     * In the event handler, `this` refers to the Control - not to the root DOM element like in jQuery. While
     * the DOM element can be used and modified, the general caveats for working with SAPUI5 control DOM elements
     * apply. In particular the DOM element may be destroyed and replaced by a new one at any time, so modifications
     * that are required to have permanent effect may not be done. E.g. use {@link #addStyleClass} instead if
     * the modification is of visual nature.
     *
     * Use {@link #detachBrowserEvent} to remove the event handler(s) again.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    attachBrowserEvent(
      /**
       * A string containing one or more JavaScript event types, such as "click" or "blur".
       */
      sEventType: string,
      /**
       * A function to execute each time the event is triggered.
       */
      fnHandler: Function,
      /**
       * The object, that wants to be notified, when the event occurs
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validateFieldGroup validateFieldGroup} event
     * of this `sap.ui.core.Control`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.Control` itself.
     *
     * Event is fired if a logical field group defined by `fieldGroupIds` of a control was left or when the
     * user explicitly pressed the key combination that triggers validation.
     *
     * By default, the `RETURN` key without any modifier keys triggers validation, see {@link #triggerValidateFieldGroup}.
     *
     * Listen to this event to validate data of the controls belonging to a field group. See {@link #setFieldGroupIds},
     * or consult the {@link https://ui5.sap.com/#/topic/5b0775397e394b1fb973fa207554003e Field Group} documentation.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidateFieldGroup(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Control$ValidateFieldGroupEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.Control` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validateFieldGroup validateFieldGroup} event
     * of this `sap.ui.core.Control`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.Control` itself.
     *
     * Event is fired if a logical field group defined by `fieldGroupIds` of a control was left or when the
     * user explicitly pressed the key combination that triggers validation.
     *
     * By default, the `RETURN` key without any modifier keys triggers validation, see {@link #triggerValidateFieldGroup}.
     *
     * Listen to this event to validate data of the controls belonging to a field group. See {@link #setFieldGroupIds},
     * or consult the {@link https://ui5.sap.com/#/topic/5b0775397e394b1fb973fa207554003e Field Group} documentation.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidateFieldGroup(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Control$ValidateFieldGroupEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.Control` itself
       */
      oListener?: object
    ): this;
    /**
     * Returns whether this control belongs to a given combination of field groups.
     *
     * If the `vFieldGroupIds` parameter is not specified, the method checks whether this control belongs to
     * **any** field group, that is, whether any field group ID is defined for it.
     *
     * If a list of field group IDs is specified, either as an array of strings or as a single string (interpreted
     * as a comma separated list of IDs), then the method will check whether this control belongs to **all**
     * given field groups. Accordingly, an empty list of IDs (empty array or empty string) will always return
     * true.
     *
     * Note that a string value for `vFieldGroupIds` (comma separated list) will not be trimmed. All whitespace
     * characters are significant, but in general not recommended in field group IDs.
     * See:
     * 	{@link #setFieldGroupIds}
     *
     *
     * @returns Whether the field group IDs defined for the control match the given ones
     */
    checkFieldGroupIds(
      /**
       * An array of field group IDs or a single string with a comma separated list of IDs to match
       */
      vFieldGroupIds?: string | string[]
    ): boolean;
    /**
     * Removes event handlers which have been previously attached using {@link #attachBrowserEvent}.
     *
     * Note: listeners are only removed, if the same combination of event type, callback function and context
     * object is given as in the call to `attachBrowserEvent`.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    detachBrowserEvent(
      /**
       * A string containing one or more JavaScript event types, such as "click" or "blur".
       */
      sEventType: string,
      /**
       * The function that is to be no longer executed.
       */
      fnHandler: Function,
      /**
       * The context object that was given in the call to `attachBrowserEvent`.
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:validateFieldGroup validateFieldGroup} event
     * of this `sap.ui.core.Control`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachValidateFieldGroup(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Control$ValidateFieldGroupEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:validateFieldGroup validateFieldGroup} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireValidateFieldGroup(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Control$ValidateFieldGroupEventParameters
    ): this;
    /**
     * This function (if available on the concrete control) provides the current accessibility state of the
     * control.
     *
     * Applications must not call this hook method directly, it is called by the framework.
     *
     * Subclasses of Control should implement this hook to provide any necessary accessibility information:
     *
     *
     * ```javascript
     *
     * MyControl.prototype.getAccessibilityInfo = function() {
     *    return {
     *      role: "textbox",
     *      type: "date input",
     *      description: "value",
     *      focusable: true,
     *      enabled: true,
     *      editable: true,
     *      required: true,
     *      children: []
     *    };
     * };
     * ```
     *
     *
     * Note: The returned object provides the accessibility state of the control at the point in time when this
     * function is called.
     *
     * @since 1.37.0
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Current accessibility state of the control.
     */
    getAccessibilityInfo?(): AccessibilityInfo;
    /**
     * Gets current value of property blocked.
     *
     * @deprecated As of version 1.69. the blocked property is deprecated. There is no accessibility support
     * for this property. Blocked controls should not be used inside Controls, which rely on keyboard navigation,
     * e.g. List controls.
     *
     * @returns Whether the control is currently in blocked state. Default is 'false'.
     */
    getBlocked(): boolean;
    /**
     * Gets current value of property {@link #getBusy busy}.
     *
     * Whether the control is currently in busy state.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `busy`
     */
    getBusy(): boolean;
    /**
     * Gets current value of property {@link #getBusyIndicatorDelay busyIndicatorDelay}.
     *
     * The delay in milliseconds, after which the busy indicator will show up for this control.
     *
     * Default value is `1000`.
     *
     *
     * @returns Value of property `busyIndicatorDelay`
     */
    getBusyIndicatorDelay(): int;
    /**
     * Gets current value of property {@link #getBusyIndicatorSize busyIndicatorSize}.
     *
     * The size of the BusyIndicator. For controls with a width smaller 3rem a `sap.ui.core.BusyIndicatorSize.Small`
     * should be used. If the size could vary in width and the width could get smaller than 3rem, the `sap.ui.core.BusyIndicatorSize.Auto`
     * option could be used. The default is set to `sap.ui.core.BusyIndicatorSize.Medium` For a full screen
     * BusyIndicator use `sap.ui.core.BusyIndicatorSize.Large`.
     *
     * Default value is `'Medium'`.
     *
     * @since 1.54
     *
     * @returns Value of property `busyIndicatorSize`
     */
    getBusyIndicatorSize(): BusyIndicatorSize;
    /**
     * Returns a list of all child controls with a field group ID. See {@link #checkFieldGroupIds checkFieldGroupIds }
     * for a description of the `vFieldGroupIds` parameter. Associated controls are not taken into account.
     *
     *
     * @returns The list of controls with a field group ID
     */
    getControlsByFieldGroupId(
      /**
       * ID of the field group or an array of field group IDs to match
       */
      vFieldGroupIds?: string | string[]
    ): Control[];
    /**
     * Returns a copy of the field group IDs array. Modification of the field group IDs need to call {@link #setFieldGroupIds setFieldGroupIds }
     * to apply the changes.
     *
     *
     * @returns copy of the field group IDs
     */
    getFieldGroupIds(): string[];
    /**
     * Returns the DOMNode Id to be used for the "labelFor" attribute of the label.
     *
     * By default, this is the Id of the control itself.
     *
     *
     * @returns Id to be used for the `labelFor`
     */
    getIdForLabel(): string;
    /**
     * Returns a renderer for this control instance.
     *
     * It is retrieved using the RenderManager as done during rendering.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns a Renderer suitable for this Control instance.
     */
    getRenderer(): ControlRenderer;
    /**
     * Gets current value of property {@link #getVisible visible}.
     *
     * Whether the control should be visible on the screen.
     *
     * If set to false, a placeholder will be rendered to mark the location of the invisible control in the
     * DOM of the current page. The placeholder will be hidden and have zero dimensions (`display: none`).
     *
     * Also see {@link module:sap/ui/core/InvisibleRenderer InvisibleRenderer}.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `visible`
     */
    getVisible(): boolean;
    /**
     * Returns true if the given style class or all multiple style classes which are generated by splitting
     * the given string with space are already set on the control via previous call(s) to addStyleClass().
     *
     *
     * @returns Whether the given style(s) has been set before
     */
    hasStyleClass(
      /**
       * the style to check for
       */
      sStyleClass: string
    ): boolean;
    /**
     * Marks this control and its children for a re-rendering, usually because its state has changed and now
     * differs from the rendered DOM.
     *
     * Managed settings (properties, aggregations, associations) automatically invalidate the corresponding
     * object. Changing the state via the standard mutators, therefore, does not require an explicit call to
     * `invalidate`.
     *
     * By default, all invalidations are buffered and processed together (asynchronously) in a new browser task.
     *
     * The `oOrigin` parameter was introduced to allow parent controls to limit their re-rendering to certain
     * areas that have been invalidated by their children. As there is no strong guideline for control developers
     * whether or not to provide the parameter, it is not a reliable source of information. It is, therefore,
     * not recommended in general to use it, only in scenarios where a control and its descendants know each
     * other very well (e.g. complex controls where parent and children have the same code owner).
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    invalidate(
      /**
       * Child control for which the method was called
       */
      oOrigin?: ManagedObject
    ): void;
    /**
     * Check if the control is currently in busy state.
     *
     * @deprecated As of version 1.15. use {@link #getBusy} instead
     */
    isBusy(): boolean;
    /**
     * Function is called when the rendering of the control is completed.
     *
     * Applications must not call this hook method directly, it is called by the framework.
     *
     * Subclasses of Control should override this hook to implement any necessary actions after the rendering.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns This hook method must not have a return value. Return value `void` is deprecated since 1.120,
     * as it does not force functions to **not** return something. This implies that, for instance, no async
     * function returning a Promise should be used.
     *
     * **Note:** While the return type is currently `void|undefined`, any implementation of this hook must not
     * return anything but undefined. Any other return value will cause an error log in this version of UI5
     * and will fail in future major versions of UI5.
     */
    onAfterRendering(
      /**
       * onAfterRendering event object
       */
      oEvent: jQuery.Event
    ): void | undefined;
    /**
     * Function is called before the rendering of the control is started.
     *
     * Applications must not call this hook method directly, it is called by the framework.
     *
     * Subclasses of Control should override this hook to implement any necessary actions before the rendering.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns This hook method must not have a return value. Return value `void` is deprecated since 1.120,
     * as it does not force functions to **not** return something. This implies that, for instance, no async
     * function returning a Promise should be used.
     *
     * **Note:** While the return type is currently `void|undefined`, any implementation of this hook must not
     * return anything but undefined. Any other return value will cause an error log in this version of UI5
     * and will fail in future major versions of UI5.
     */
    onBeforeRendering(
      /**
       * onBeforeRendering event object
       */
      oEvent: jQuery.Event
    ): void | undefined;
    /**
     * Puts `this` control into the specified container (`oRef`) at the given position (`oPosition`).
     *
     * First it is checked whether `oRef` is a container element / control (has a multiple aggregation with
     * type `sap.ui.core.Control` and name 'content') or is an Id String of such a container. If this is not
     * the case `oRef` can either be a Dom Reference or Id String of the UIArea (if it does not yet exist implicitly
     * a new UIArea is created),
     *
     * The `oPosition` can be one of the following:
     *
     *
     * 	 - "first": The control is added as the first element to the container.
     * 	 - "last": The control is added as the last element to the container (default).
     * 	 - "only": All existing children of the container are removed (not destroyed!) and the control is added
     *     as new child.
     * 	 - index: The control is added at the specified index to the container.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    placeAt(
      /**
       * container into which the control should be put
       */
      oRef: string | Element | Control,
      /**
       * Describes the position where the control should be put into the container
       */
      vPosition?: string | int
    ): this;
    /**
     * Removes the given string from the list of custom style classes that have been set previously. Regular
     * style classes like "sapUiBtn" cannot be removed.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    removeStyleClass(
      /**
       * the style to be removed
       */
      sStyleClass: string
    ): this;
    /**
     * Synchronously updates the DOM of this control to reflect the current object state.
     *
     * Note that this method can only be called when the control already has a DOM representation (it has been
     * rendered before) and when the control still is assigned to a UIArea.
     *
     * @deprecated As of version 1.70. using this method is no longer recommended, but calling it still causes
     * a re-rendering of the control. Synchronous DOM updates via this method have several drawbacks: they only
     * work when the control has been rendered before (no initial rendering possible), multiple state changes
     * won't be combined automatically into a single re-rendering, they might cause additional layout thrashing,
     * standard invalidation might cause another async re-rendering.
     *
     * The recommended alternative is to rely on invalidation and standard re-rendering.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    rerender(): void;
    /**
     * Set the controls busy state.
     *
     * **Note:** The busy state can't be set on controls (e.g. sap.m.ColumnListItem) which renderings have the
     * following tags as DOM root element: area|base|br|col|embed|hr|img|input|keygen|link|menuitem|meta|param|source|track|wbr|tr
     *
     *
     * @returns `this` to allow method chaining
     */
    setBusy(
      /**
       * The new busy state to be set
       */
      bBusy: boolean
    ): this;
    /**
     * Define the delay, after which the busy indicator will show up.
     *
     *
     * @returns `this` to allow method chaining
     */
    setBusyIndicatorDelay(
      /**
       * The delay in ms
       */
      iDelay: int
    ): this;
    /**
     * Sets a new value for property {@link #getBusyIndicatorSize busyIndicatorSize}.
     *
     * The size of the BusyIndicator. For controls with a width smaller 3rem a `sap.ui.core.BusyIndicatorSize.Small`
     * should be used. If the size could vary in width and the width could get smaller than 3rem, the `sap.ui.core.BusyIndicatorSize.Auto`
     * option could be used. The default is set to `sap.ui.core.BusyIndicatorSize.Medium` For a full screen
     * BusyIndicator use `sap.ui.core.BusyIndicatorSize.Large`.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'Medium'`.
     *
     * @since 1.54
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBusyIndicatorSize(
      /**
       * New value for property `busyIndicatorSize`
       */
      sBusyIndicatorSize?: BusyIndicatorSize | keyof typeof BusyIndicatorSize
    ): this;
    /**
     * Sets a new value for property {@link #getFieldGroupIds fieldGroupIds}.
     *
     * The IDs of a logical field group that this control belongs to.
     *
     * All fields in a logical field group should share the same `fieldGroupId`. Once a logical field group
     * is left, the `validateFieldGroup` event is fired.
     *
     * For backward compatibility with older releases, field group IDs are syntactically not limited, but it
     * is suggested to use only valid {@link sap.ui.core.ID}s.
     *
     * See {@link #attachValidateFieldGroup} or consult the {@link https://ui5.sap.com/#/topic/5b0775397e394b1fb973fa207554003e Field Group }
     * documentation.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `[]`.
     *
     * @since 1.31
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFieldGroupIds(
      /**
       * New value for property `fieldGroupIds`
       */
      sFieldGroupIds?: string[]
    ): this;
    /**
     * Sets a new value for property {@link #getVisible visible}.
     *
     * Whether the control should be visible on the screen.
     *
     * If set to false, a placeholder will be rendered to mark the location of the invisible control in the
     * DOM of the current page. The placeholder will be hidden and have zero dimensions (`display: none`).
     *
     * Also see {@link module:sap/ui/core/InvisibleRenderer InvisibleRenderer}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setVisible(
      /**
       * New value for property `visible`
       */
      bVisible?: boolean
    ): this;
    /**
     * The string given as "sStyleClass" will be be either added to or removed from the "class" attribute of
     * this control's root HTML element, depending on the value of "bAdd": if bAdd is true, sStyleClass will
     * be added. If bAdd is not given, sStyleClass will be removed if it is currently present and will be added
     * if not present. If sStyleClass is null or empty string, the call is ignored.
     *
     * See addStyleClass and removeStyleClass for further documentation.
     *
     *
     * @returns Returns `this` to allow method chaining
     */
    toggleStyleClass(
      /**
       * the CSS class name to be added or removed
       */
      sStyleClass: string,
      /**
       * whether sStyleClass should be added (or removed); when this parameter is not given, sStyleClass will
       * be toggled (removed, if present, and added if not present)
       */
      bAdd?: boolean
    ): this;
    /**
     * Triggers the `validateFieldGroup` event for this control.
     *
     * Called by `sap.ui.core.UIArea` if a field group should be validated after it lost the focus or when the
     * key combination was pressed that was configured to trigger validation.
     *
     * By default, the `RETURN` key without any modifier keys triggers validation. There's no public API to
     * change that key combination.
     *
     * See {@link #attachValidateFieldGroup}.
     */
    triggerValidateFieldGroup(
      /**
       * IDs of the field groups that should be validated
       */
      aFieldGroupIds: string[]
    ): void;
  }
  /**
   * Describes the settings that can be provided to the Control constructor.
   */
  export interface $ControlSettings extends $ElementSettings {
    /**
     * Whether the control is currently in blocked state.
     *
     * @deprecated As of version 1.69. The blocked property is deprecated. There is no accessibility support
     * for this property. Blocked controls should not be used inside Controls, which rely on keyboard navigation,
     * e.g. List controls.
     */
    blocked?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Whether the control is currently in busy state.
     */
    busy?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The delay in milliseconds, after which the busy indicator will show up for this control.
     */
    busyIndicatorDelay?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * The size of the BusyIndicator. For controls with a width smaller 3rem a `sap.ui.core.BusyIndicatorSize.Small`
     * should be used. If the size could vary in width and the width could get smaller than 3rem, the `sap.ui.core.BusyIndicatorSize.Auto`
     * option could be used. The default is set to `sap.ui.core.BusyIndicatorSize.Medium` For a full screen
     * BusyIndicator use `sap.ui.core.BusyIndicatorSize.Large`.
     *
     * @since 1.54
     */
    busyIndicatorSize?:
      | (BusyIndicatorSize | keyof typeof BusyIndicatorSize)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Whether the control should be visible on the screen.
     *
     * If set to false, a placeholder will be rendered to mark the location of the invisible control in the
     * DOM of the current page. The placeholder will be hidden and have zero dimensions (`display: none`).
     *
     * Also see {@link module:sap/ui/core/InvisibleRenderer InvisibleRenderer}.
     */
    visible?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The IDs of a logical field group that this control belongs to.
     *
     * All fields in a logical field group should share the same `fieldGroupId`. Once a logical field group
     * is left, the `validateFieldGroup` event is fired.
     *
     * For backward compatibility with older releases, field group IDs are syntactically not limited, but it
     * is suggested to use only valid {@link sap.ui.core.ID}s.
     *
     * See {@link #attachValidateFieldGroup} or consult the {@link https://ui5.sap.com/#/topic/5b0775397e394b1fb973fa207554003e Field Group }
     * documentation.
     *
     * @since 1.31
     */
    fieldGroupIds?: string[] | PropertyBindingInfo | `{${string}}`;

    /**
     * Event is fired if a logical field group defined by `fieldGroupIds` of a control was left or when the
     * user explicitly pressed the key combination that triggers validation.
     *
     * By default, the `RETURN` key without any modifier keys triggers validation, see {@link #triggerValidateFieldGroup}.
     *
     * Listen to this event to validate data of the controls belonging to a field group. See {@link #setFieldGroupIds},
     * or consult the {@link https://ui5.sap.com/#/topic/5b0775397e394b1fb973fa207554003e Field Group} documentation.
     */
    validateFieldGroup?: (oEvent: Control$ValidateFieldGroupEvent) => void;
  }

  /**
   * Parameters of the Control#validateFieldGroup event.
   */
  export interface Control$ValidateFieldGroupEventParameters {
    /**
     * field group IDs of the logical field groups to validate
     */
    fieldGroupIds?: string[];
  }

  /**
   * Event object of the Control#validateFieldGroup event.
   */
  export type Control$ValidateFieldGroupEvent = Event<
    Control$ValidateFieldGroupEventParameters,
    Control
  >;
}

declare module "sap/ui/core/ElementMetadata" {
  import RenderManager from "sap/ui/core/RenderManager";

  import {
    default as UI5Element,
    MetadataOptions,
    MetadataOptions as MetadataOptions1,
  } from "sap/ui/core/Element";

  import ManagedObjectMetadata from "sap/ui/base/ManagedObjectMetadata";

  /**
   * Control Renderer
   */
  export type ControlRenderer = {
    /**
     * The function that renders the control
     */
    render: (p1: RenderManager, p2: UI5Element) => void;
    /**
     * The API version of the RenderManager that are used in this renderer. See {@link sap.ui.core.RenderManager RenderManager }
     * API documentation for detailed information
     */
    apiVersion?: 1 | 2 | 4;
  };

  /**
   * @since 0.8.6
   */
  export default class ElementMetadata extends ManagedObjectMetadata {
    /**
     * Creates a new metadata object for a UIElement subclass.
     */
    constructor(
      /**
       * fully qualified name of the class that is described by this metadata object
       */
      sClassName: string,
      /**
       * static info to construct the metadata from
       */
      oClassInfo: {
        /**
         * The metadata object describing the class
         */
        metadata?: MetadataOptions;
      }
    );

    /**
     * Calculates a new id based on a prefix.
     *
     *
     * @returns A (hopefully unique) control id
     */
    static uid(): string;
    /**
     * Returns an info object describing the drag-and-drop behavior.
     *
     * @since 1.56
     *
     * @returns An info object about the drag-and-drop behavior.
     */
    getDragDropInfo(
      /**
       * name of the aggregation or empty.
       */
      sAggregationName?: string
    ): MetadataOptions1.DnD;
    /**
     * By default, the element name is equal to the class name
     *
     *
     * @returns the qualified name of the UIElement class
     */
    getElementName(): string;
  }
}

declare module "sap/ui/core/Core" {
  import Control from "sap/ui/core/Control";

  import { ID } from "sap/ui/core/library";

  import UI5Element from "sap/ui/core/Element";

  import { default as Component, MetadataOptions } from "sap/ui/core/Component";

  import RenderManager from "sap/ui/core/RenderManager";

  import UIArea from "sap/ui/core/UIArea";

  import Type from "sap/ui/model/Type";

  import Application from "sap/ui/app/Application";

  import Configuration from "sap/ui/core/Configuration";

  import EventBus from "sap/ui/core/EventBus";

  import ResourceBundle from "sap/base/i18n/ResourceBundle";

  import Messaging from "sap/ui/core/Messaging";

  import Metadata from "sap/ui/base/Metadata";

  import Model from "sap/ui/model/Model";

  import Interface from "sap/ui/base/Interface";

  /**
   * Singleton Core instance of the SAP UI Library.
   *
   * The module export of `sap/ui/core/Core` is **not** a class, but the singleton Core instance itself. The
   * `sap.ui.core.Core` class itself must not be instantiated, except by the framework itself.
   *
   * The Core provides a {@link #ready ready function} to execute code after the Core was booted.
   *
   * Example:
   * ```javascript
   *
   *
   *   sap.ui.require(["sap/ui/core/Core"], async function(Core) {
   *
   *     // Usage of a callback function
   *     Core.ready(function() {
   *       ...
   *     });
   *
   *     // Usage of Core.ready() as a Promise
   *     await Core.ready();
   *     ...
   *   });
   *
   * ```
   */
  interface Core {
    /**
     * Creates a new subclass of class `sap.ui.core.Core` with name `sClassName` and enriches it with the information
     * contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.ManagedObject.extend}.
     *
     * @deprecated As of version 1.119. without replacement. In future major versions, the Core no longer has
     * a class nature and therefore can't be extended.
     *
     * @returns The created class / constructor function
     */
    extend(
      /**
       * Qualified name of the newly created class
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: {
        /**
         * The metadata object describing the class. See {@link sap.ui.core.Component.MetadataOptions MetadataOptions }
         * for the values allowed in every extend.
         */
        metadata?: MetadataOptions;
      }
    ): Function;
    /**
     * Returns the metadata for class `sap.ui.core.Core`.
     *
     * @deprecated As of version 1.119. without replacement. In future major versions, the Core no longer has
     * a class nature and no longer inherits from sap.ui.base.Object and therefore no longer has metadata.
     *
     * @returns Metadata object describing this class
     */
    getMetadata(): Metadata;
    /**
     * Enforces an immediate update of the visible UI (aka "rendering").
     *
     * In general, applications and Controls should avoid calling this method and instead let the framework
     * manage any necessary rendering.
     *
     * @deprecated As of version 1.118. without replacement. In the next major version, synchronously rendering
     * UI updates is no longer supported as it can lead to unnecessary intermediate DOM updates or layout shifting
     * etc. Controls should rather use invalidation and apps should not trigger rendering at all but rather
     * rely on the framework's automatic update mechanisms. Test code can use the test module `sap/ui/test/utils/nextUIUpdate`
     * as a convenient way to wait for the next asynchronous rendering.
     */
    applyChanges(): void;
    /**
     * Applies the theme with the given name (by loading the respective style sheets, which does not disrupt
     * the application).
     *
     * By default, the theme files are expected to be located at path relative to the respective control library
     * ([libraryLocation]/themes/[themeName]). Different locations can be configured by using the method setThemePath()
     * or by using the second parameter "sThemeBaseUrl" of applyTheme(). Usage of this second parameter is a
     * shorthand for setThemePath and internally calls setThemePath, so the theme location is then known.
     *
     * sThemeBaseUrl is a single URL to specify the default location of all theme files. This URL is the base
     * folder below which the control library folders are located. E.g. if the CSS files are not located relative
     * to the root location of UI5, but instead they are at locations like http://my.server/myapp/resources/sap/ui/core/themes/my_theme/library.css
     * then the URL that needs to be given is: http://my.server/myapp/resources All theme resources are then
     * loaded from below this folder - except if for a certain library a different location has been registered.
     *
     * If the theme resources are not all either below this base location or with their respective libraries,
     * then setThemePath must be used to configure individual locations.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/ui/core/Theming.setTheme Theming.setTheme }
     * instead.
     */
    applyTheme(
      /**
       * the name of the theme to be loaded
       */
      sThemeName: string,
      /**
       * the (optional) base location of the theme
       */
      sThemeBaseUrl?: string
    ): void;
    /**
     * Registers a listener for control events.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * @deprecated As of version 1.119. without a replacement. Applications should not have the need to intercept
     * all control events.
     */
    attachControlEvent(
      /**
       * Callback to be called for each control event
       */
      fnFunction: Function,
      /**
       * Optional context object to call the callback on
       */
      oListener?: object
    ): void;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:formatError formatError} event of `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachFormatError ManagedObject#attachFormatError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachFormatError(
      /**
       * An object that will be passed to the handler along with the event object when the event is fired
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:formatError formatError} event of `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachFormatError ManagedObject#attachFormatError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachFormatError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Registers a given function that is executed after the framework has been initialized.
     *
     * The given function will either be called as soon as the framework has been initialized or, if it has
     * been initialized already, it will be called immediately.
     *
     * More information about the initialization process and the steps it consists of can be found in the documentation
     * topic "{@link https://ui5.sap.com/#/topic/91f2c9076f4d1014b6dd926db0e91070 Initialization Process}".
     *
     * @since 1.13.2
     * @deprecated As of version 1.118. Please use {@link sap.ui.core.Core.ready Core.ready} instead.
     */
    attachInit(
      /**
       * Function to be after initialization of the framework
       */
      fnFunction: Function
    ): void;
    /**
     * Registers a given function that is executed after the framework has been initialized.
     *
     * The method is executed only once and only if the framework has not been initialized already. This could
     * be checked by calling {@link #isInitialized}, but in most cases it is more convenient to use {@link #attachInit }
     * instead. This guarantees that the given function is executed exactly once, independent of the state of
     * the framework.
     *
     * @deprecated As of version 1.13.2. Register with the more convenient {@link #attachInit} function instead
     */
    attachInitEvent(
      /**
       * Function that is called after initialization of the framework
       */
      fnFunction: Function
    ): void;
    /**
     * Registers a listener to the central interval timer.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to the interval timer instance.
     *
     * @since 1.16.0
     * @deprecated As of version 1.61. Use `IntervalTrigger.addListener()` from "sap/ui/core/IntervalTrigger"
     * module.
     */
    attachIntervalTimer(
      /**
       * Callback to be called periodically
       */
      fnFunction: Function,
      /**
       * Optional context object to call the callback on
       */
      oListener?: object
    ): void;
    /**
     * Register a listener for the {@link #event:localizationChanged localizationChanged} event.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * @deprecated As of version 1.118. Please use {@link module:sap/base/i18n/Localization.attachChange Localization.attachChange }
     * instead.
     */
    attachLocalizationChanged(
      /**
       * Callback to be called when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the function on
       */
      oListener?: object
    ): void;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:parseError parseError} event of `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachParseError ManagedObject#attachParseError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachParseError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:parseError parseError} event of `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachParseError ManagedObject#attachParseError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachParseError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:ThemeChanged ThemeChanged} event of this `sap.ui.core.Core`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to a dummy event provider object.
     *
     * @deprecated As of version 1.118. See {@link module:sap/ui/core/Theming.attachApplied Theming.attachApplied }
     * instead.
     */
    attachThemeChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): void;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationError validationError} event of `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachValidationError ManagedObject#attachValidationError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationError validationError} event of `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachValidationError ManagedObject#attachValidationError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationSuccess validationSuccess} event of
     * `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachValidationSuccess ManagedObject#attachValidationSuccess }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationSuccess(
      /**
       * The object, that should be passed along with the event-object when firing the event
       */
      oData: object,
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validationSuccess validationSuccess} event of
     * `sap.ui.core.Core`.
     *
     * When called, the context of the listener (its `this`) will be bound to `oListener` if specified, otherwise
     * it will be bound to a dummy event provider object.
     *
     * Please note that this event is a bubbling event and may already be canceled before reaching the core.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#attachValidationSuccess ManagedObject#attachValidationSuccess }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidationSuccess(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object to call the event handler with. Defaults to a dummy event provider object
       */
      oListener?: object
    ): this;
    /**
     * Returns a list of all controls with a field group ID. See {@link sap.ui.core.Control#checkFieldGroupIds Control.prototype.checkFieldGroupIds }
     * for a description of the `vFieldGroupIds` parameter.
     *
     * @deprecated As of version 1.118. use {@link sap.ui.core.Control.getControlsByFieldGroupId Control.getControlsByFieldGroupId }
     * instead.
     *
     * @returns The list of controls with matching field group IDs
     */
    byFieldGroupId(
      /**
       * ID of the field group or an array of field group IDs to match
       */
      vFieldGroupIds?: string | string[]
    ): Control[];
    /**
     * Returns the registered element with the given ID, if any.
     *
     * The ID must be the globally unique ID of an element, the same as returned by `oElement.getId()`.
     *
     * When the element has been created from a declarative source (e.g. XMLView), that source might have used
     * a shorter, non-unique local ID. A search for such a local ID cannot be executed with this method. It
     * can only be executed on the corresponding scope (e.g. on an XMLView instance), by using the {@link sap.ui.core.mvc.View#byId View#byId }
     * method of that scope.
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.core.Element.getElementById Element.getElementById }
     * instead.
     *
     * @returns Element with the given ID or `undefined`
     */
    byId(
      /**
       * ID of the element to search for
       */
      sId: ID | null | undefined
    ): UI5Element | undefined;
    /**
     * Creates a component with the provided id and settings.
     *
     * When the optional parameter `sUrl` is given, then all request for resources of the library will be redirected
     * to the given URL. This is convenience for a call to
     * ```javascript
     *
     *   sap.ui.loader.config({
     *       paths: {
     *         "lib/with/slashes": vUrl
     *       }
     *   });
     * ```
     *
     *
     * @deprecated As of version 1.95. Please use {@link sap.ui.core.Component.create Component.create} instead.
     *
     * @returns The created component instance or a promise on it in the async use case
     */
    createComponent(
      /**
       * name of the component to import or object containing all needed parameters
       */
      vComponent:
        | string
        | {
            /**
             * name of the component to import
             */
            name?: string;
            /**
             * URL to load the component from
             */
            url?: string;
            /**
             * ID for the component instance
             */
            id?: string;
            /**
             * settings object for the component
             */
            settings?: object;
            /**
             * user specific data which is available during the whole lifecycle of the component
             */
            componentData?: any;
          },
      /**
       * the URL to load the component from
       */
      sUrl?: string,
      /**
       * the ID for the component instance
       */
      sId?: string,
      /**
       * the settings object for the component
       */
      mSettings?: object
    ): Component | Promise<Component>;
    /**
     * Returns a new instance of the RenderManager for exclusive use by the caller.
     *
     * The caller must take care to destroy the render manager when it is no longer needed. Calling this method
     * before the Core has been {@link #isInitialized initialized}, is not recommended.
     *
     * @deprecated As of version 1.119. without replacement. In the next major version, synchronously rendering
     * UI updates is no longer supported as it can lead to unnecessary intermediate DOM updates or layout shifting
     * etc. Controls should rather use invalidation.
     *
     * @returns New instance of the RenderManager
     */
    createRenderManager(): RenderManager;
    /**
     * Creates a new {@link sap.ui.core.UIArea UIArea}.
     *
     * @deprecated As of version 1.1. use {@link sap.ui.core.Control#placeAt Control#placeAt} instead!
     *
     * @returns a new UIArea
     */
    createUIArea(
      /**
       * a DOM Element or ID string of the UIArea
       */
      oDomRef: string | Element
    ): UIArea;
    /**
     * Unregisters a listener for control events.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @deprecated As of version 1.119. without a replacement. Applications should not have the need to intercept
     * all control events.
     */
    detachControlEvent(
      /**
       * Function to unregister
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:formatError formatError} event of `sap.ui.core.Core`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#detachFormatError ManagedObject#detachFormatError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachFormatError(
      /**
       * The callback function to unregister
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Unregisters a listener for the central interval timer.
     *
     * A listener will only be unregistered if the same function/context combination is given as in the `attachIntervalTimer`
     * call.
     *
     * @since 1.16.0
     * @deprecated As of version 1.61. Use `IntervalTrigger.removeListener()` from "sap/ui/core/IntervalTrigger"
     * module.
     */
    detachIntervalTimer(
      /**
       * function to unregister
       */
      fnFunction: Function,
      /**
       * context object given during registration
       */
      oListener?: object
    ): void;
    /**
     * Unregister a listener from the {@link #event:localizationChanged localizationChanged} event.
     *
     * The listener will only be unregistered if the same function/context combination is given as in the call
     * to `attachLocalizationListener`.
     *
     * @deprecated As of version 1.118. Please use {@link module:sap/base/i18n/Localization.detachChange Localization.detachChange }
     * instead.
     */
    detachLocalizationChanged(
      /**
       * Callback to be deregistered
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:parseError parseError} event of `sap.ui.core.Core`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#detachParseError ManagedObject#detachParseError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachParseError(
      /**
       * The callback function to unregister.
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:ThemeChanged ThemeChanged} event of this `sap.ui.core.Core`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @deprecated As of version 1.118. See {@link module:sap/ui/core/Theming.detachApplied Theming#detachApplied }
     * instead.
     */
    detachThemeChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Object on which the given function had to be called.
       */
      oListener?: object
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:validationError validationError} event of
     * `sap.ui.core.Core`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#detachValidationError ManagedObject#detachValidationError }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachValidationError(
      /**
       * The callback function to unregister
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:validationSuccess validationSuccess} event
     * of `sap.ui.core.Core`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#detachValidationSuccess ManagedObject#detachValidationSuccess }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachValidationSuccess(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: Function,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:formatError formatError} to attached listeners.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#fireFormatError ManagedObject#fireFormatError }
     * instead.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireFormatError(
      /**
       * Parameters to pass along with the event.
       */
      oParameters?: {
        /**
         * Element where the format error occurred
         */
        element: UI5Element;
        /**
         * Name of the property of the element where the format error occurred
         */
        property: string;
        /**
         * Type of the property
         */
        type: Type;
        /**
         * Value of the property which was entered when the format error occurred
         */
        newValue: any;
        /**
         * Value of the property which was present before a new value was entered (before the format error)
         */
        oldValue: any;
        /**
         * Exception object which occurred and has more information about the format error
         */
        exception: object;
      }
    ): this;
    /**
     * Fires event {@link #event:parseError parseError} to attached listeners.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#fireParseError ManagedObject#fireParseError }
     * instead.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireParseError(
      /**
       * Parameters to pass along with the event.
       */
      oParameters?: {
        /**
         * Element where the parse error occurred
         */
        element: UI5Element;
        /**
         * Name of the property of the element where the parse error occurred
         */
        property: string;
        /**
         * Type of the property
         */
        type: Type;
        /**
         * Value of the property which was entered when the parse error occurred
         */
        newValue: object;
        /**
         * Value of the property which was present before a new value was entered (before the parse error)
         */
        oldValue: object;
        /**
         * Exception object which occurred and has more information about the parse error
         */
        exception: object;
      }
    ): this;
    /**
     * Fires event {@link #event:validationError validationError} to attached listeners.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#fireValidationError ManagedObject.fireValidationError }
     * instead.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireValidationError(
      /**
       * Parameters to pass along with the event.
       */
      oParameters?: {
        /**
         * The Element where the validation error occurred
         */
        element: UI5Element;
        /**
         * Property name of the element where the validation error occurred
         */
        property: string;
        /**
         * Type of the property
         */
        type: Type;
        /**
         * Value of the property which was entered when the validation error occurred
         */
        newValue: object;
        /**
         * Value of the property which was present before a new value was entered (before the validation error)
         */
        oldValue: object;
        /**
         * Exception object which occurred and has more information about the validation error
         */
        exception: object;
      }
    ): this;
    /**
     * Fires event {@link #event:validationSuccess validationSuccess} to attached listeners.
     *
     * Expects following event parameters:
     * 	 - 'element' of type `sap.ui.core.Element`
     * 	 - 'property' of type `string`
     * 	 - 'type' of type `string`
     * 	 - 'newValue' of type `object`
     * 	 - 'oldValue' of type `object`
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#fireValidationSuccess ManagedObject#fireValidationSuccess }
     * instead.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireValidationSuccess(
      /**
       * Parameters to pass along with the event
       */
      oParameters?: object
    ): this;
    /**
     * Returns the instance of the application (if exists).
     *
     * @deprecated As of version 1.15.1. The Component class is enhanced to take care about the Application
     * code.
     *
     * @returns instance of the current application
     */
    getApplication(): Application;
    /**
     * Returns the registered component for the given id, if any.
     *
     * @deprecated As of version 1.95. Please use {@link sap.ui.core.Component.get Component.get} instead.
     *
     * @returns the component for the given id
     */
    getComponent(sId: string): Component;
    /**
     * Returns the Configuration of the Core.
     *
     * @deprecated As of version 1.120. Please see {@link sap.ui.core.Configuration Configuration} for the corresponding
     * replacements.
     *
     * @returns the Configuration of the current Core.
     */
    getConfiguration(): Configuration;
    /**
     * Returns the registered element for the given ID, if any.
     *
     * @deprecated As of version 1.1. use `sap.ui.core.Core.byId` instead!
     *
     * @returns Element for the given ID or `undefined`
     */
    getControl(
      /**
       * ID of the control to retrieve
       */
      sId: ID | null | undefined
    ): UI5Element | undefined;
    /**
     * Returns the ID of the control/element currently in focus.
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.core.Element.getActiveElement Element.getActiveElement }
     * to get the currently focused element. You can then retrieve the ID of that element with {@link sap.ui.core.Element#getId Element#getId}.
     * Please be aware, {@link sap.ui.core.Element.getActiveElement Element.getActiveElement} can return 'undefined'.
     *
     * @returns the ID of the control/element currently in focus.
     */
    getCurrentFocusedControlId(): string;
    /**
     * Returns the registered element for the given ID, if any.
     *
     * @deprecated As of version 1.1. use `sap.ui.core.Core.byId` instead!
     *
     * @returns Element for the given ID or `undefined`
     */
    getElementById(
      /**
       * ID of the element to retrieve
       */
      sId: ID | null | undefined
    ): UI5Element | undefined;
    /**
     * Returns the event bus.
     *
     * @since 1.8.0
     * @deprecated As of version 1.119.0. Please use {@link sap.ui.core.EventBus.getInstance EventBus.getInstance }
     * for global usage instead. Creating an own local instance is the preferred usage.
     *
     * @returns the event bus
     */
    getEventBus(): EventBus;
    /**
     * Retrieves a resource bundle for the given library and locale.
     *
     * If only one argument is given, it is assumed to be the libraryName. The locale then falls back to the
     * current {@link module:sap/base/i18n/Localization.getLanguage session locale}. If no argument is given,
     * the library also falls back to a default: "sap.ui.core".
     *
     * Configuration via App Descriptor: When the App Descriptor for the library is available without further
     * request (manifest.json has been preloaded) and when the App Descriptor is at least of version 1.9.0 or
     * higher, then this method will evaluate the App Descriptor entry `"sap.ui5" / "library" / "i18n"`.
     *
     * 	 - When the entry is `true`, a bundle with the default name "messagebundle.properties" will be loaded
     *
     * 	 - If it is a string, then that string will be used as name of the bundle
     * 	 - If it is `false`, no bundle will be loaded and the result will be `undefined`
     *
     * Caching: Once a resource bundle for a library has been loaded, it will be cached by this method. Further
     * calls for the same library and locale won't create new requests, but return the already loaded bundle.
     * There's therefore no need for control code to cache the returned bundle for a longer period of time.
     * Not further caching the result also prevents stale texts after a locale change.
     *
     * Asynchronous Loading: The asynchronous variant of {@link #loadLibrary} will evaluate the same descriptor
     * entry as described above. If it is not `false`, loading the main resource bundle of the library will
     * become a subtask of the asynchronous loading of the library.
     *
     * Due to this preload of the main bundle and the caching behavior of this method, controls in such a library
     * still can use the synchronous variant of `getLibraryResourceBundle` in their API, behavior and rendering
     * code. Only when the bundle is needed at module execution time (by top level code in a control module),
     * then the asynchronous variant of this method should be preferred.
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.core.Lib.getResourceBundleFor Lib.getResourceBundleFor }
     * instead.
     *
     * @returns The best matching resource bundle for the given parameters or `undefined`; in asynchronous case
     * a Promise on that bundle is returned
     */
    getLibraryResourceBundle(
      /**
       * Name of the library to retrieve the bundle for
       */
      sLibraryName?: string,
      /**
       * Locale to retrieve the resource bundle for
       */
      sLocale?: string,
      /**
       * Whether the resource bundle is loaded asynchronously
       */
      bAsync?: boolean
    ): ResourceBundle | undefined | Promise<ResourceBundle | undefined>;
    /**
     * Retrieves a resource bundle for the given library and locale.
     *
     * If only one argument is given, it is assumed to be the libraryName. The locale then falls back to the
     * current {@link module:sap/base/i18n/Localization.getLanguage session locale}. If no argument is given,
     * the library also falls back to a default: "sap.ui.core".
     *
     * Configuration via App Descriptor: When the App Descriptor for the library is available without further
     * request (manifest.json has been preloaded) and when the App Descriptor is at least of version 1.9.0 or
     * higher, then this method will evaluate the App Descriptor entry `"sap.ui5" / "library" / "i18n"`.
     *
     * 	 - When the entry is `true`, a bundle with the default name "messagebundle.properties" will be loaded
     *
     * 	 - If it is a string, then that string will be used as name of the bundle
     * 	 - If it is `false`, no bundle will be loaded and the result will be `undefined`
     *
     * Caching: Once a resource bundle for a library has been loaded, it will be cached by this method. Further
     * calls for the same library and locale won't create new requests, but return the already loaded bundle.
     * There's therefore no need for control code to cache the returned bundle for a longer period of time.
     * Not further caching the result also prevents stale texts after a locale change.
     *
     * Asynchronous Loading: The asynchronous variant of {@link #loadLibrary} will evaluate the same descriptor
     * entry as described above. If it is not `false`, loading the main resource bundle of the library will
     * become a subtask of the asynchronous loading of the library.
     *
     * Due to this preload of the main bundle and the caching behavior of this method, controls in such a library
     * still can use the synchronous variant of `getLibraryResourceBundle` in their API, behavior and rendering
     * code. Only when the bundle is needed at module execution time (by top level code in a control module),
     * then the asynchronous variant of this method should be preferred.
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.core.Lib.getResourceBundleFor Lib.getResourceBundleFor }
     * instead.
     *
     * @returns The best matching resource bundle for the given parameters or `undefined`; in asynchronous case
     * a Promise on that bundle is returned
     */
    getLibraryResourceBundle(
      /**
       * Name of the library to retrieve the bundle for
       */
      sLibraryName?: string,
      /**
       * Whether the resource bundle is loaded asynchronously
       */
      bAsync?: boolean
    ): ResourceBundle | undefined | Promise<ResourceBundle | undefined>;
    /**
     * Returns a map of library info objects for all currently loaded libraries, keyed by their names.
     *
     * The structure of the library info objects matches the structure of the info object that the {@link #initLibrary }
     * method expects. Only property names documented with `initLibrary` should be accessed, any additional
     * properties might change or disappear in future. When a property does not exists, its default value (as
     * documented with `initLibrary`) should be assumed.
     *
     * **Note:** The returned info objects must not be modified. They might be a living copy of the internal
     * data (for efficiency reasons) and the framework is not prepared to handle modifications to these objects.
     *
     * @deprecated As of version 1.119. without a 1:1 replacement. Callers that used `getLoadedLibraries` to
     * check whether a certain library is loaded, should rather use {@link sap.ui.core.Lib#isLoaded Library#isLoaded}.
     *
     * @returns Map of library info objects keyed by the library names.
     */
    getLoadedLibraries(): Record<string, Object>;
    /**
     * Returns the `Messaging` module.
     *
     * @since 1.33.0
     * @deprecated As of version 1.118. Please use {@link module:sap/ui/core/Messaging Messaging} instead.
     */
    getMessageManager(): Messaging;
    /**
     * Get the model with the given model name.
     *
     * The name can be omitted to reference the default model or it must be a non-empty string.
     *
     * Note: to be compatible with future versions of this API, applications must not use the value `null`,
     * the empty string `""` or the string literals `"null"` or `"undefined"` as model name.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#getModel ManagedObject#getModel }
     * instead.
     *
     * @returns oModel
     */
    getModel(
      /**
       * name of the model to be retrieved
       */
      sName?: string
    ): Model;
    /**
     * Creates a new `RenderManager` instance for use by the caller.
     *
     * @deprecated As of version 0.15.0. Replaced by `createRenderManager()`
     *
     * @returns A newly created RenderManager
     */
    getRenderManager(): RenderManager;
    /**
     * Returns the instance of the root component (if exists).
     *
     * @deprecated As of version 1.95. Please use {@link module:sap/ui/core/ComponentSupport} instead. See also
     * {@link topic:82a0fcecc3cb427c91469bc537ebdddf Declarative API for Initial Components}.
     *
     * @returns instance of the current root component
     */
    getRootComponent(): Component;
    /**
     * Returns the static, hidden area DOM element belonging to this core instance.
     *
     * It can be used e.g. for hiding elements like Popups, Shadow, Blocklayer etc.
     *
     * If it is not yet available, a DIV is created and appended to the body.
     *
     * @deprecated As of version 1.119.0. Please use {@link module:sap/ui/core/StaticArea.getDomRef StaticArea.getDomRef }
     * instead.
     *
     * @returns the static, hidden area DOM element belonging to this core instance.
     */
    getStaticAreaRef(): Element;
    /**
     * Returns the registered template for the given id, if any.
     *
     * @deprecated As of version 1.29.1. Require 'sap/ui/core/tmpl/Template' and use {@link sap.ui.core.tmpl.Template.byId Template.byId }
     * instead.
     *
     * @returns the template for the given id
     */
    getTemplate(sId: string): Component;
    /**
     * Returns the {@link sap.ui.core.UIArea UIArea} with the given ID or that belongs to the given DOM element.
     *
     * @deprecated As of version 1.107. For access to the static UIArea, please use the {@link module:sap/ui/core/StaticArea StaticArea }
     * instead.
     *
     * @returns UIArea with the given ID or DOM element or `null` or `undefined`.
     */
    getUIArea(
      /**
       * DOM element or ID of the UIArea
       */
      o: string | Element
    ): UIArea | null | undefined;
    /**
     * Returns `true` if there are any pending rendering tasks or when such rendering tasks are currently being
     * executed.
     *
     * @deprecated As of version 1.118. without replacement. The known use cases in testing environments are
     * covered by other APIs or features, e.g. OPA's waitFor mechanism.
     *
     * @returns true if there are pending (or executing) rendering tasks.
     */
    getUIDirty(): boolean;
    /**
     * Check if a Model is set to the core
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#hasModel ManagedObject#hasModel }
     * instead.
     *
     * @returns true or false
     */
    hasModel(): boolean;
    /**
     * Includes a library theme into the current page (if a variant is specified it will include the variant
     * library theme)
     *
     * @deprecated As of version 1.119. without replacement. There's no known use case that would require a
     * public API.
     */
    includeLibraryTheme(
      /**
       * the name of the UI library
       */
      sLibName: string,
      /**
       * the variant to include (optional)
       */
      sVariant?: string,
      /**
       * to be used only by the Core
       */
      sQuery?: string
    ): void;
    /**
     * Provides the framework with information about a library.
     *
     * This method is intended to be called exactly once while the main module of a library (its `library.js`
     * module) is executing, typically at its begin. The single parameter `oLibInfo` is an info object that
     * describes the content of the library.
     *
     * When the `oLibInfo` has been processed, a normalized version of it will be kept and will be returned
     * as library information in later calls to {@link #getLoadedLibraries}. Finally, `initLibrary` fires (the
     * currently private) {@link #event:LibraryChanged} event with operation 'add' for the newly loaded library.
     *
     * Side Effects:
     *
     * While analyzing the `oLibInfo`, the framework takes some additional actions:
     *
     *
     * 	 - If the info object contains a list of `interfaces`, they will be registered with the {@link sap.ui.base.DataType }
     *     class to make them available as aggregation types in managed objects.
     *
     *
     * 	 - If the object contains a list of `controls` or `elements`, {@link sap.ui.lazyRequire lazy stubs }
     *     will be created for their constructor as well as for their static `extend` and `getMetadata` methods.
     *  **Note:** Future versions might abandon the concept of lazy stubs as it requires synchronous XMLHttpRequests
     * which have been deprecated (see {@link http://xhr.spec.whatwg.org}). To be on the safe side, productive
     * applications should always require any modules that they directly depend on.
     *
     *
     * 	 - With the `noLibraryCSS` property, the library can be marked as 'theming-free'. Otherwise, the framework
     *     will add a <link> tag to the page's head, pointing to the library's theme-specific stylesheet.
     *     The creation of such a <link> tag can be suppressed with the {@link https://ui5.sap.com/#/topic/91f2d03b6f4d1014b6dd926db0e91070 global configuration option }
     *     `preloadLibCss`. It can contain a list of library names for which no stylesheet should be included. This
     *     is e.g. useful when an application merges the CSS for multiple libraries and already loaded the resulting
     *     stylesheet.
     *
     *
     * 	 - If a list of library `dependencies` is specified in the info object, those libraries will be loaded
     *     synchronously by `initLibrary`.
     *  **Note:** Dependencies between libraries don't have to be modeled as AMD dependencies. Only when enums
     * or types from an additional library are used in the coding of the `library.js` module, the library should
     * be additionally listed in the AMD dependencies.
     *
     * Last but not least, higher layer frameworks might want to include their own metadata for libraries. The
     * property `extensions` might contain such additional metadata. Its structure is not defined by the framework,
     * but it is strongly suggested that each extension only occupies a single property in the `extensions`
     * object and that the name of that property contains some namespace information (e.g. library name that
     * introduces the feature) to avoid conflicts with other extensions. The framework won't touch the content
     * of `extensions` but will make it available in the library info objects returned by {@link #getLoadedLibraries}.
     *
     * Relationship to Descriptor for Libraries (manifest.json):
     *
     * The information contained in `oLibInfo` is partially redundant to the content of the descriptor for the
     * same library (its `manifest.json` file). Future versions of UI5 might ignore the information provided
     * in `oLibInfo` and might evaluate the descriptor file instead. Library developers therefore should keep
     * the information in both files in sync.
     *
     * When the `manifest.json` is generated from the `.library` file (which is the default for UI5 libraries
     * built with Maven), then the content of the `.library` and `library.js` files must be kept in sync.
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.core.Lib.init Lib.init} instead.
     *
     * @returns As of version 1.101; returns the library namespace, based on the given library name. Returns
     * 'undefined' if no library name is provided.
     */
    initLibrary(
      /**
       * Info object for the library
       */
      oLibInfo: LibraryInfo
    ): object | undefined;
    /**
     * Returns true if the Core has already been initialized. This means that instances of RenderManager etc.
     * do already exist and the init event has already been fired (and will not be fired again).
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.core.Core.ready Core.ready} instead.
     *
     * @returns whether the Core has already been initialized
     */
    isInitialized(): boolean;
    /**
     * Returns the locked state of the `sap.ui.core.Core`.
     *
     * @deprecated As of version 1.118. without a replacement. The ability to prevent the re-rendering of all
     * `UIArea`s wasn't really used in the past and did not provide a meaningful feature. It therefore has been
     * abandoned.
     *
     * @returns locked state
     */
    isLocked(): boolean;
    /**
     * Check if the script is running on mobile
     *
     * @deprecated As of version 1.118. use {@link sap.ui.Device.browser.mobile Device.browser.mobile} instead.
     *
     * @returns true or false
     */
    isMobile(): boolean;
    /**
     * Checks whether the given DOM element is the root of the static area.
     *
     * @deprecated As of version 1.119.0. Please use {@link module:sap/ui/core/StaticArea.contains StaticArea.contains }
     * instead.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether the given DOM element is the root of the static area
     */
    isStaticAreaRef(
      /**
       * DOM element to check
       */
      oDomRef: Element
    ): boolean;
    /**
     * Returns true, if the styles of the current theme are already applied, false otherwise.
     *
     * This function must not be used before the init event of the Core. If the styles are not yet applied a
     * theme changed event will follow when the styles will be applied.
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/ui/core/Theming.attachApplied Theming.attachApplied }
     * instead.
     *
     * @returns whether the styles of the current theme are already applied
     */
    isThemeApplied(): boolean;
    /**
     * Loads the given library and its dependencies and makes its content available to the application.
     *
     * What it does:
     *
     * When library preloads are not suppressed for the given library, then a library-preload bundle will be
     * loaded for it. By default, the bundle will be loaded synchronously (for compatibility reasons). Only
     * when the optional parameter `vUrl` is given as `true` or as a configuration object with a property of
     * `async:true`, then the bundle will be loaded asynchronously and a `Promise` will be returned (preferred
     * usage).
     *
     * After preloading the bundle, dependency information from the bundle is evaluated and any missing libraries
     * are also preloaded.
     *
     * Only then the library entry module (named `your/lib/library.js`) will be required and executed.
     * The module is supposed to call `sap.ui.getCore().initLibrary(...)` providing the framework with additional
     * metadata about the library, e.g. its version, the set of contained enums, types, interfaces, controls
     * and elements and whether the library requires CSS. If the library requires CSS, a <link> will be
     * added to the page referring to the corresponding `library.css` stylesheet for the library and the current
     * theme.
     *
     * When the optional parameter `vUrl` is given as a string or when a configuration object is given with
     * a non-empty, string-valued property `url`, then that URL will be registered for the namespace of the
     * library and all resources will be loaded from that location. This is convenience for a call like
     * ```javascript
     *
     *   sap.ui.loader.config({
     *     paths: {
     *       "lib/with/slashes": vUrl
     *     }
     *   });
     * ```
     *
     *
     * When the given library has been loaded already, no further action will be taken, especially, a given
     * URL will not be registered! In the case of asynchronous loading, a Promise will be returned, but will
     * be resolved immediately.
     *
     * When to use:
     *
     * For applications that follow the best practices and use components with component descriptors (manifest.json),
     * the framework will load all declared mandatory libraries and their dependencies automatically before
     * instantiating the application component.
     *
     * The same is true for libraries that are listed in the bootstrap configuration (e.g. with the attribute
     * `data-sap-ui-libs`). They will be loaded before the `init` event of the UI5 Core is fired.
     *
     * Only when an app declares a library to be a lazy library dependency or when code does not use descriptors
     * at all, then an explicit call to `loadLibrary` becomes necessary. The call should be made before artifacts
     * (controls, elements, types, helpers, modules etc.) from the library are used or required. This allows
     * the framework to optimize access to those artifacts.
     *
     * For example, when an app uses a heavy-weight charting library that shouldn't be loaded during startup,
     * it can declare it as "lazy" and load it just before it loads and displays a view that uses the charting
     * library:
     * ```javascript
     *
     *   sap.ui.getCore().loadLibrary("heavy.charting", {async: true})
     *     .then(function() {
     *       View.create({
     *         name: "myapp.views.HeavyChartingView",
     *         type: ViewType.XML
     *       });
     *     });
     * ```
     *
     *
     * @deprecated As of version 1.119. Please use {@link sap.ui.core.Lib.load Lib.load} instead.
     *
     * @returns An info object for the library (sync) or a Promise on it (async).
     */
    loadLibrary(
      /**
       * Name of the library to load
       */
      sLibrary: string,
      /**
       * URL to load the library from or the async flag or a complex configuration object
       */
      vUrl?:
        | string
        | boolean
        | {
            /**
             * Whether to load the library asynchronously
             */
            async?: boolean;
            /**
             * URL to load the library from
             */
            url?: string;
          }
    ): LibraryInfo | Promise<LibraryInfo>;
    /**
     * Locks the Core. No browser events are dispatched to the controls.
     *
     * Lock should be called before and after the DOM is modified for rendering, roundtrips... Exceptions might
     * be the case for asynchronous UI behavior
     *
     * @deprecated As of version 1.118. without a replacement. The ability to prevent the re-rendering of all
     * `UIArea`s wasn't really used in the past and did not provide a meaningful feature. It therefore has been
     * abandoned.
     */
    lock(): void;
    /**
     * Triggers a realignment of controls
     *
     * This method should be called after changing the cozy/compact CSS class of a DOM element at runtime, for
     * example at the `<body>` tag. Controls can listen to the themeChanged event to realign their appearance
     * after changing the theme. Changing the cozy/compact CSS class should then also be handled as a theme
     * change. In more simple scenarios where the cozy/compact CSS class is added to a DOM element which contains
     * only a few controls it might not be necessary to trigger the realignment of all controls placed in the
     * DOM, for example changing the cozy/compact CSS class at a single control
     *
     * @deprecated As of version 1.119. Please use {@link module:sap/ui/core/Theming.notifyContentDensityChanged Theming.notifyContentDensityChanged }
     * instead.
     */
    notifyContentDensityChanged(): void;
    /**
     * Returns a Promise that resolves if the Core is initialized. Additionally, a callback function can be
     * passed, for use cases where using Promises is not an option.
     *
     * @since 1.118.0
     *
     * @returns The ready promise
     */
    ready(
      /**
       * If the Core is ready the function will be called immediately, otherwise when the ready Promise resolves.
       */
      fnReady?: () => void
    ): Promise<undefined>;
    /**
     * Registers a Plugin to the `sap.ui.core.Core`, which lifecycle will be managed (start and stop).
     *
     * Plugin object need to implement two methods:
     * 	 - `startPlugin(oCore)`: will be invoked, when the Plugin should start (as parameter the reference to
     *     the Core will be provided
     * 	 - `stopPlugin()`: will be invoked, when the Plugin should stop
     *
     * @deprecated As of version 1.73. Plugins never have been meant as a public offering, but were intended
     * for internal usage only. They unfortunately allow access to all internals of the Core and therefore break
     * encapsulation and hinder evolution of the Core. The most common use case of accessing the set of all
     * controls/elements or all components can now be addressed by using the APIs {@link sap.ui.core.Element.registry }
     * or {@link sap.ui.core.Component.registry}, respectively. Future refactoring of the Core will only take
     * existing plugins in the OpenUI5 repository into account.
     */
    registerPlugin(
      /**
       * reference to a Plugin object
       */
      oPlugin: object
    ): void;
    /**
     * Sets or unsets a model for the given model name.
     *
     * The `sName` must either be `undefined` (or omitted) or a non-empty string. When the name is omitted,
     * the default model is set/unset.
     *
     * When `oModel` is `null` or `undefined`, a previously set model with that name is removed from the Core.
     *
     * Any change (new model, removed model) is propagated to all existing UIAreas and their descendants as
     * long as a descendant doesn't have its own model set for the given name.
     *
     * Note: to be compatible with future versions of this API, applications must not use the value `null`,
     * the empty string `""` or the string literals `"null"` or `"undefined"` as model name.
     *
     * @deprecated As of version 1.118. Please use {@link sap.ui.base.ManagedObject#setModel ManagedObject#setModel }
     * instead.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setModel(
      /**
       * the model to be set or `null` or `undefined`
       */
      oModel: Model,
      /**
       * the name of the model or `undefined`
       */
      sName?: string
    ): this;
    /**
     * Implicitly creates a new `UIArea` (or reuses an exiting one) for the given DOM reference and adds the
     * given control reference to the UIAreas content (existing content will be removed).
     *
     * @deprecated As of version 1.1. use {@link sap.ui.core.Control#placeAt oControl.placeAt(oDomRef, "only") }
     * instead.
     */
    setRoot(
      /**
       * a DOM Element or Id (string) of the UIArea
       */
      oDomRef: string | Element,
      /**
       * the Control that should be the added to the `UIArea`.
       */
      oControl: Interface | Control
    ): void;
    /**
     * Defines the root directory from below which UI5 should load the theme with the given name. Optionally
     * allows restricting the setting to parts of a theme covering specific control libraries.
     *
     * Example:
     * ```javascript
     *
     *   sap.ui.getCore().setThemeRoot("my_theme", "https://mythemeserver.com/allThemes");
     *   sap.ui.getCore().applyTheme("my_theme");
     * ```
     *
     *
     * will cause the following file to be loaded (assuming that the bootstrap is configured to load libraries
     * `sap.m` and `sap.ui.layout`):
     * ```javascript
     *
     *   https://mythemeserver.com/allThemes/sap/ui/core/themes/my_theme/library.css
     *   https://mythemeserver.com/allThemes/sap/ui/layout/themes/my_theme/library.css
     *   https://mythemeserver.com/allThemes/sap/m/themes/my_theme/library.css
     * ```
     *
     *
     * If parts of the theme are at different locations (e.g. because you provide a standard theme like "sap_belize"
     * for a custom control library and this self-made part of the standard theme is at a different location
     * than the UI5 resources), you can also specify for which control libraries the setting should be used,
     * by giving an array with the names of the respective control libraries as second parameter:
     * ```javascript
     *
     *   sap.ui.getCore().setThemeRoot("sap_belize", ["my.own.library"], "https://mythemeserver.com/allThemes");
     * ```
     *
     *
     * This will cause the Belize theme to be loaded from the UI5 location for all standard libraries. Resources
     * for styling the `my.own.library` controls will be loaded from the configured location:
     * ```javascript
     *
     *   https://sdk.openui5.org/resources/sap/ui/core/themes/sap_belize/library.css
     *   https://sdk.openui5.org/resources/sap/ui/layout/themes/sap_belize/library.css
     *   https://sdk.openui5.org/resources/sap/m/themes/sap_belize/library.css
     *   https://mythemeserver.com/allThemes/my/own/library/themes/sap_belize/library.css
     * ```
     *
     *
     * If the custom theme should be loaded initially (via bootstrap attribute), the `themeRoots` property of
     * the `window["sap-ui-config"]` object must be used instead of calling `sap.ui.getCore().setThemeRoot(...)`
     * in order to configure the theme location early enough.
     *
     * @since 1.10
     * @deprecated As of version 1.119. without replacement. The need to define the location for a theme should
     * be fully covered with the capabilities of the {@link sap/base/config base configuration}.
     *
     * @returns the Core, to allow method chaining
     */
    setThemeRoot(
      /**
       * Name of the theme for which to configure the location
       */
      sThemeName: string,
      /**
       * Optional library names to which the configuration should be restricted
       */
      aLibraryNames: string[],
      /**
       * Base URL below which the CSS file(s) will be loaded from
       */
      sThemeBaseUrl: string,
      /**
       * Force updating URLs of currently loaded theme
       */
      bForceUpdate?: boolean
    ): this;
    /**
     * Defines the root directory from below which UI5 should load the theme with the given name. Optionally
     * allows restricting the setting to parts of a theme covering specific control libraries.
     *
     * Example:
     * ```javascript
     *
     *   sap.ui.getCore().setThemeRoot("my_theme", "https://mythemeserver.com/allThemes");
     *   sap.ui.getCore().applyTheme("my_theme");
     * ```
     *
     *
     * will cause the following file to be loaded (assuming that the bootstrap is configured to load libraries
     * `sap.m` and `sap.ui.layout`):
     * ```javascript
     *
     *   https://mythemeserver.com/allThemes/sap/ui/core/themes/my_theme/library.css
     *   https://mythemeserver.com/allThemes/sap/ui/layout/themes/my_theme/library.css
     *   https://mythemeserver.com/allThemes/sap/m/themes/my_theme/library.css
     * ```
     *
     *
     * If parts of the theme are at different locations (e.g. because you provide a standard theme like "sap_belize"
     * for a custom control library and this self-made part of the standard theme is at a different location
     * than the UI5 resources), you can also specify for which control libraries the setting should be used,
     * by giving an array with the names of the respective control libraries as second parameter:
     * ```javascript
     *
     *   sap.ui.getCore().setThemeRoot("sap_belize", ["my.own.library"], "https://mythemeserver.com/allThemes");
     * ```
     *
     *
     * This will cause the Belize theme to be loaded from the UI5 location for all standard libraries. Resources
     * for styling the `my.own.library` controls will be loaded from the configured location:
     * ```javascript
     *
     *   https://sdk.openui5.org/resources/sap/ui/core/themes/sap_belize/library.css
     *   https://sdk.openui5.org/resources/sap/ui/layout/themes/sap_belize/library.css
     *   https://sdk.openui5.org/resources/sap/m/themes/sap_belize/library.css
     *   https://mythemeserver.com/allThemes/my/own/library/themes/sap_belize/library.css
     * ```
     *
     *
     * If the custom theme should be loaded initially (via bootstrap attribute), the `themeRoots` property of
     * the `window["sap-ui-config"]` object must be used instead of calling `sap.ui.getCore().setThemeRoot(...)`
     * in order to configure the theme location early enough.
     *
     * @since 1.10
     * @deprecated As of version 1.119. without replacement. The need to define the location for a theme should
     * be fully covered with the capabilities of the {@link sap/base/config base configuration}.
     *
     * @returns the Core, to allow method chaining
     */
    setThemeRoot(
      /**
       * Name of the theme for which to configure the location
       */
      sThemeName: string,
      /**
       * Base URL below which the CSS file(s) will be loaded from
       */
      sThemeBaseUrl: string,
      /**
       * Force updating URLs of currently loaded theme
       */
      bForceUpdate?: boolean
    ): this;
    /**
     * Unlocks the Core.
     *
     * Browser events are dispatched to the controls again after this method is called.
     *
     * @deprecated As of version 1.118. without a replacement. The ability to prevent the re-rendering of all
     * `UIArea`s wasn't really used in the past and did not provide a meaningful feature. It therefore has been
     * abandoned.
     */
    unlock(): void;
    /**
     * Unregisters a Plugin out of the `sap.ui.core.Core`
     *
     * @deprecated As of version 1.73. Plugins never have been meant as a public offering, but were intended
     * for internal usage only. They unfortunately allow access to all internals of the Core and therefore break
     * encapsulation and hinder evolution of the Core. The most common use case of accessing the set of all
     * controls/elements or all components can now be addressed by using the APIs {@link sap.ui.core.Element.registry }
     * or {@link sap.ui.core.Component.registry}, respectively. Future refactoring of the Core will only take
     * existing plugins in the OpenUI5 repository into account.
     */
    unregisterPlugin(
      /**
       * reference to a Plugin object
       */
      oPlugin: object
    ): void;
  }
  const Core: Core;
  export default Core;

  /**
   * Info object for the library
   */
  export type LibraryInfo = {
    /**
     * Version of the library
     */
    version: string;
    /**
     * Name of the library; when given it must match the name by which the library has been loaded
     */
    name?: string;
    /**
     * List of libraries that this library depends on; names are in dot notation (e.g. "sap.ui.core")
     */
    dependencies?: string[];
    /**
     * List of names of types that this library provides; names are in dot notation (e.g. "sap.ui.core.CSSSize")
     */
    types?: string[];
    /**
     * List of names of interface types that this library provides; names are in dot notation (e.g. "sap.ui.core.PopupInterface")
     */
    interfaces?: string[];
    /**
     * Names of control types that this library provides; names are in dot notation (e.g. "sap.ui.core.ComponentContainer")
     */
    controls?: string[];
    /**
     * Names of element types that this library provides (excluding controls); names are in dot notation (e.g.
     * "sap.ui.core.Item")
     */
    elements?: string[];
    /**
     * Indicates whether the library doesn't provide/use theming. When set to true, no library.css will be loaded
     * for this library
     */
    noLibraryCSS?: boolean;
    /**
     * Potential extensions of the library metadata; structure not defined by the UI5 core framework.
     */
    extensions?: Record<string, any>;
  };
}

declare module "sap/ui/core/CustomData" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Contains a single key/value pair of custom data attached to an `Element`.
   *
   * See method {@link sap.ui.core.Element#data Element.prototype.data} and the chapter {@link https://ui5.sap.com/#/topic/91f0c3ee6f4d1014b6dd926db0e91070 Custom Data - Attaching Data Objects to Controls }
   * in the documentation.
   */
  export default class CustomData extends UI5Element {
    /**
     * Constructor for a new `CustomData` element.
     */
    constructor(
      /**
       * initial settings for the new element
       */
      mSettings?: $CustomDataSettings
    );
    /**
     * Constructor for a new `CustomData` element.
     */
    constructor(
      /**
       * ID for the new element, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * initial settings for the new element
       */
      mSettings?: $CustomDataSettings
    );

    /**
     * Creates a new subclass of `CustomData` with name `sClassName` and enriches it with the information contained
     * in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CustomData>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to `sap.ui.core.ElementMetadata`
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class `sap.ui.core.CustomData`.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * The key of the data in this CustomData object. When the data is just stored, it can be any string, but
     * when it is to be written to HTML (`writeToDom == true`) then it must also be a valid HTML attribute name.
     * It must conform to the {@link sap.ui.core.ID} type and may contain no colon. To avoid collisions, it
     * also may not start with "sap-ui". When written to HTML, the key is prefixed with "data-". If any restriction
     * is violated, a warning will be logged and nothing will be written to the DOM.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * The data stored in this CustomData object. When the data is just stored, it can be any JS type, but when
     * it is to be written to HTML (`writeToDom == true`) then it must be a string. If this restriction is violated,
     * a warning will be logged and nothing will be written to the DOM.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): any;
    /**
     * Gets current value of property {@link #getWriteToDom writeToDom}.
     *
     * If set to "true" and the value is of type "string" and the key conforms to the documented restrictions,
     * this custom data is written to the HTML root element of the control as a "data-*" attribute. If the key
     * is "abc" and the value is "cde", the HTML will look as follows:
     *
     *
     * ```javascript
     *
     *   <SomeTag ... data-abc="cde" ... >
     * ```
     *
     *
     * Thus the application can provide stable attributes by data binding which can be used for styling or identification
     * purposes.
     *
     * **ATTENTION:** use carefully to not create huge attributes or a large number of them.
     *
     * Default value is `false`.
     *
     * @since 1.9.0
     *
     * @returns Value of property `writeToDom`
     */
    getWriteToDom(): boolean;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * The key of the data in this CustomData object. When the data is just stored, it can be any string, but
     * when it is to be written to HTML (`writeToDom == true`) then it must also be a valid HTML attribute name.
     * It must conform to the {@link sap.ui.core.ID} type and may contain no colon. To avoid collisions, it
     * also may not start with "sap-ui". When written to HTML, the key is prefixed with "data-". If any restriction
     * is violated, a warning will be logged and nothing will be written to the DOM.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * The data stored in this CustomData object. When the data is just stored, it can be any JS type, but when
     * it is to be written to HTML (`writeToDom == true`) then it must be a string. If this restriction is violated,
     * a warning will be logged and nothing will be written to the DOM.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      oValue?: any
    ): this;
    /**
     * Sets a new value for property {@link #getWriteToDom writeToDom}.
     *
     * If set to "true" and the value is of type "string" and the key conforms to the documented restrictions,
     * this custom data is written to the HTML root element of the control as a "data-*" attribute. If the key
     * is "abc" and the value is "cde", the HTML will look as follows:
     *
     *
     * ```javascript
     *
     *   <SomeTag ... data-abc="cde" ... >
     * ```
     *
     *
     * Thus the application can provide stable attributes by data binding which can be used for styling or identification
     * purposes.
     *
     * **ATTENTION:** use carefully to not create huge attributes or a large number of them.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.9.0
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWriteToDom(
      /**
       * New value for property `writeToDom`
       */
      bWriteToDom?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CustomData constructor.
   */
  export interface $CustomDataSettings extends $ElementSettings {
    /**
     * The key of the data in this CustomData object. When the data is just stored, it can be any string, but
     * when it is to be written to HTML (`writeToDom == true`) then it must also be a valid HTML attribute name.
     * It must conform to the {@link sap.ui.core.ID} type and may contain no colon. To avoid collisions, it
     * also may not start with "sap-ui". When written to HTML, the key is prefixed with "data-". If any restriction
     * is violated, a warning will be logged and nothing will be written to the DOM.
     */
    key?: string | PropertyBindingInfo;

    /**
     * The data stored in this CustomData object. When the data is just stored, it can be any JS type, but when
     * it is to be written to HTML (`writeToDom == true`) then it must be a string. If this restriction is violated,
     * a warning will be logged and nothing will be written to the DOM.
     */
    value?: any | PropertyBindingInfo | `{${string}}`;

    /**
     * If set to "true" and the value is of type "string" and the key conforms to the documented restrictions,
     * this custom data is written to the HTML root element of the control as a "data-*" attribute. If the key
     * is "abc" and the value is "cde", the HTML will look as follows:
     *
     *
     * ```javascript
     *
     *   <SomeTag ... data-abc="cde" ... >
     * ```
     *
     *
     * Thus the application can provide stable attributes by data binding which can be used for styling or identification
     * purposes.
     *
     * **ATTENTION:** use carefully to not create huge attributes or a large number of them.
     *
     * @since 1.9.0
     */
    writeToDom?: boolean | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/ui/core/date/CalendarWeekNumbering" {
  import CalendarWeekNumbering1 from "sap/base/i18n/date/CalendarWeekNumbering";

  export {
    /**
     * The `CalendarWeekNumbering` enum defines how to calculate calendar weeks. Each value defines:
     * 	 - The first day of the week,
     * 	 - the first week of the year.
     *
     * @since 1.108.0
     * @deprecated As of version 1.120. Please use {@link module:sap/base/i18n/date/CalendarWeekNumbering} instead.
     */
    CalendarWeekNumbering1 as default,
  };
}

declare module "sap/ui/core/DeclarativeSupport" {
  import HTMLView from "sap/ui/core/mvc/HTMLView";

  /**
   * Static class for enabling declarative UI support.
   *
   * @since 1.7.0
   * @deprecated As of version 1.120. Please consider using {@link sap.ui.core.mvc.XMLView XMLViews} or {@link topic:e6bb33d076dc4f23be50c082c271b9f0 Typed Views }
   * instead. For more information, see the documentation on {@link topic:91f27e3e6f4d1014b6dd926db0e91070 View types}.
   */
  export default class DeclarativeSupport {
    constructor();

    /**
     * Enhances the given DOM element by parsing the Control and Elements info and creating the SAPUI5 controls
     * for them.
     */
    static compile(
      /**
       * the DOM element to compile
       */
      oElement: Element,
      /**
       * The view instance to use
       */
      oView?: HTMLView,
      /**
       * Whether the call of the function is recursive.
       */
      isRecursive?: boolean
    ): void;
  }
}

declare module "sap/ui/core/delegate/ItemNavigation" {
  import EventProvider from "sap/ui/base/EventProvider";

  import Metadata from "sap/ui/base/Metadata";

  import Event from "sap/ui/base/Event";

  /**
   * Delegate for the navigation between DOM nodes with the keyboard.
   *
   * The `ItemNavigation` provides keyboard and mouse navigation between DOM nodes representing items. This
   * means that controls rendering a list of items can attach this delegate to get a common keyboard and mouse
   * support to navigate between these items. It is possible to navigate between the items via the arrow keys.
   * If needed, paging using the Page Up and Page Down keys is possible. (To activate call `setPageSize` with
   * a value > 0.) HOME and END keys are also supported. Focusing an item via mouse also is also supported.
   * For mouse navigation, the `mousedown` event is used.
   *
   * As the `ItemNavigation` works with DOM nodes, the items and the control area must be provided as DOM
   * references. There is one root DOM reference (set via `setRootDomRef`). All item DOM references (set via
   * `setItemDomRefs`) must be places somewhere inside of this root DOM reference. Only focusable items are
   * used for the navigation, meaning disabled items or separator items are just ignored by navigating through
   * the items. In some cases however, it makes sense to put the non-focusable items in the array of the DOM
   * references to keep the indexes stable or like in the calling control. **Hint:** To make a DOM reference
   * focusable a `tabindex` of -1 can be set.
   *
   * **Note** After re-rendering of the control or changing the DOM nodes of the control, the DOM references
   * of the `ItemNavigation` must be updated. Then the same item (corresponding to the index) will get the
   * focus.
   *
   * The `ItemNavigation` adjusts the `tabindex` of all DOM references relating to the current focused item.
   * So if the control containing the items gets the focus (e.g. via tab navigation), it is always the focused
   * items which will be focused.
   *
   * **Note:** If the `ItemNavigation` is nested in another `ItemNavigation` (e.g. `SegmentedButton` in `Toolbar`),
   * the `RootDomRef` will always have `tabindex` -1.
   *
   * Per default the `ItemNavigation` cycles over the items. It navigates again to the first item if the Arrow
   * Down or Arrow Right key is pressed while the last item has the focus. It navigates to the last item if
   * arrow up or arrow left is pressed while the first item has the focus. If you want to stop the navigation
   * at the first and last item, call the `setCycling` method with a value of `false`.
   *
   * It is possible to have multiple columns in the item navigation. If multiple columns are used, the keyboard
   * navigation changes. The Arrow Right and Arrow Left keys will take the user to the next or previous item,
   * and the Arrow Up and Arrow Down keys will navigate the same way but in a vertical direction.
   *
   * The `ItemNavigation` also allows setting a selected index that is used to identify the selected item.
   * Initially, if no other focus is set, the selected item will be the focused one. Note that navigating
   * through the items will not change the selected item, only the focus. (For example a radio group has one
   * selected item.)
   */
  export default class ItemNavigation extends EventProvider {
    /**
     * Creates an `ItemNavigation` delegate that can be attached to controls requiring capabilities for keyboard
     * navigation between items.
     */
    constructor(
      /**
       * The root DOM reference that includes all items
       */
      oDomRef: Element,
      /**
       * Array of DOM references representing the items for the navigation
       */
      aItemDomRefs: Element[],
      /**
       * Whether the selected element should be in the tab chain or not
       */
      bNotInTabChain?: boolean
    );

    /**
     * Creates a new subclass of class sap.ui.core.delegate.ItemNavigation with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.EventProvider.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ItemNavigation>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.delegate.ItemNavigation.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Returns disabled modifiers These modifiers will not be handled by the `ItemNavigation`
     *
     *
     * @returns Object that includes event type with disabled keys as an array
     */
    getDisabledModifiers(
      /**
       * Object that includes event type with disabled keys as an array
       */
      oDisabledModifiers: object
    ): object;
    /**
     * Returns the array of item DOM references
     *
     *
     * @returns Array of item DOM references
     */
    getItemDomRefs(): Element[];
    /**
     * Returns the root DOM reference surrounding the items
     *
     *
     * @returns Root DOM reference
     */
    getRootDomRef(): Element;
    /**
     * Check whether given event has disabled modifier or not
     *
     *
     * @returns Flag if disabled modifiers are set
     */
    hasDisabledModifier(
      /**
       * jQuery event
       */
      oEvent: jQuery.Event
    ): boolean;
    /**
     * Sets whether the items are displayed in columns.
     *
     * If columns are used, the Arrow Up and Arrow Down keys navigate to the next or previous item of the column.
     * If the first or last item of the column is reached, the next focused item is then in the next or previous
     * column.
     *
     *
     * @returns `this` to allow method chaining
     */
    setColumns(
      /**
       * Count of columns for the table mode or cycling mode
       */
      iColumns: int,
      /**
       * Forbids jumping to an other column with Arrow Up and Arrow Down keys
       */
      bNoColumnChange: boolean
    ): this;
    /**
     * Sets whether the `ItemNavigation` should cycle through the items.
     *
     * If cycling is disabled the navigation stops at the first and last item, if the corresponding arrow keys
     * are used.
     *
     *
     * @returns `this` to allow method chaining
     */
    setCycling(
      /**
       * Set to true if cycling should be done, else false
       */
      bCycling: boolean
    ): this;
    /**
     * Sets the disabled modifiers These modifiers will not be handled by the `ItemNavigation`
     *
     *
     * ```javascript
     *
     * Example: Disable shift + up handling of the `ItemNavigation`
     *
     * oItemNavigation.setDisabledModifiers({
     *     sapnext : ["shift"]
     * });
     *
     * Possible keys are : "shift", "alt", "ctrl", "meta"
     * Possible events are : "sapnext", "sapprevious", "saphome", "sapend"
     * ```
     *
     *
     *
     * @returns `this` to allow method chaining
     */
    setDisabledModifiers(
      /**
       * Object that includes event type with disabled keys as an array
       */
      oDisabledModifiers: Object
    ): this;
    /**
     * Sets behavior of HOME and END keys if columns are used.
     *
     *
     * @returns `this` to allow method chaining
     */
    setHomeEndColumnMode(
      /**
       * HOME -> go to first item in row; END -> go to last item in row
       */
      bStayInRow: boolean,
      /**
       * HOME/END with CTRL -> go to first/last item of all
       */
      bCtrlEnabled: boolean
    ): this;
    /**
     * Sets the item DOM references as an array for the items
     *
     *
     * @returns `this` to allow method chaining
     */
    setItemDomRefs(
      /**
       * Array of DOM references or DOM node list object, representing the items
       */
      aItemDomRefs: Element[]
    ): this;
    /**
     * Sets the page size of the item navigation to allow Page Up and Page Down keys.
     *
     *
     * @returns `this` to allow method chaining
     */
    setPageSize(
      /**
       * The page size, needs to be at least 1
       */
      iPageSize: int
    ): this;
    /**
     * Sets the root DOM reference surrounding the items
     *
     *
     * @returns `this` to allow method chaining
     */
    setRootDomRef(
      /**
       * Root DOM reference
       */
      oDomRef: Element
    ): this;
    /**
     * Sets the selected index if the used control supports selection.
     *
     *
     * @returns `this` to allow method chaining
     */
    setSelectedIndex(
      /**
       * Index of the first selected item
       */
      iIndex: int
    ): this;
    /**
     * Sets whether the `ItemNavigation` should use the table mode to navigate through the items (navigation
     * in a grid).
     *
     *
     * @returns `this` to allow method chaining
     */
    setTableMode(
      /**
       * Set to true if table mode should be used, else false
       */
      bTableMode: boolean,
      /**
       * This sets a different behavior for table mode. In this mode we keep using table navigation but there
       * are some differences. e.g.
       * 	 - Page-up moves focus to the first row, not to the first cell like in table mode
       * 	 - Page-down moves focus to the last row, not to the last cell like in table mode
       */
      bTableList?: boolean
    ): this;
  }
  /**
   * Parameters of the ItemNavigation#AfterFocus event.
   */
  export interface ItemNavigation$AfterFocusEventParameters {}

  /**
   * Event object of the ItemNavigation#AfterFocus event.
   */
  export type ItemNavigation$AfterFocusEvent = Event<
    ItemNavigation$AfterFocusEventParameters,
    ItemNavigation
  >;

  /**
   * Parameters of the ItemNavigation#BeforeFocus event.
   */
  export interface ItemNavigation$BeforeFocusEventParameters {}

  /**
   * Event object of the ItemNavigation#BeforeFocus event.
   */
  export type ItemNavigation$BeforeFocusEvent = Event<
    ItemNavigation$BeforeFocusEventParameters,
    ItemNavigation
  >;

  /**
   * Parameters of the ItemNavigation#BorderReached event.
   */
  export interface ItemNavigation$BorderReachedEventParameters {}

  /**
   * Event object of the ItemNavigation#BorderReached event.
   */
  export type ItemNavigation$BorderReachedEvent = Event<
    ItemNavigation$BorderReachedEventParameters,
    ItemNavigation
  >;

  /**
   * Parameters of the ItemNavigation#FocusAgain event.
   */
  export interface ItemNavigation$FocusAgainEventParameters {}

  /**
   * Event object of the ItemNavigation#FocusAgain event.
   */
  export type ItemNavigation$FocusAgainEvent = Event<
    ItemNavigation$FocusAgainEventParameters,
    ItemNavigation
  >;

  /**
   * Parameters of the ItemNavigation#FocusLeave event.
   */
  export interface ItemNavigation$FocusLeaveEventParameters {}

  /**
   * Event object of the ItemNavigation#FocusLeave event.
   */
  export type ItemNavigation$FocusLeaveEvent = Event<
    ItemNavigation$FocusLeaveEventParameters,
    ItemNavigation
  >;
}

declare module "sap/ui/core/delegate/ScrollEnablement" {
  import BaseObject from "sap/ui/base/Object";

  import Control from "sap/ui/core/Control";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Delegate for touch scrolling on mobile devices.
   *
   * This delegate uses native scrolling of mobile and desktop browsers. Third party scrolling libraries are
   * not supported.
   *
   * Controls that implement ScrollEnablement should additionally provide the getScrollDelegate method that
   * returns the current instance of this delegate object
   *
   * @ui5-protected DO NOT USE IN APPLICATIONS (only for related classes in the framework)
   */
  export default class ScrollEnablement extends BaseObject {
    /**
     * Creates a ScrollEnablement delegate that can be attached to Controls requiring capabilities for scrolling
     * of a certain part of their DOM.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    protected constructor(
      /**
       * the Control of which this Scroller is the delegate
       */
      oControl: Control,
      /**
       * the Id of the element within the DOM of the Control which should be scrollable
       */
      sScrollContentDom: string,
      /**
       * the configuration of the scroll delegate
       */
      oConfig: {
        /**
         * Whether the element should be scrollable horizontally
         */
        horizontal?: boolean;
        /**
         * Whether the element should be scrollable vertically
         */
        vertical?: boolean;
        /**
         * Deprecated since 1.42, the parameter has no effect
         */
        zynga?: boolean;
        /**
         * Deprecated since 1.42, the parameter has no effect
         */
        iscroll?: boolean;
        /**
         * Deprecated since 1.42, the parameter has no effect
         */
        preventDefault?: boolean;
        /**
         * If true, the delegate will also be active to allow touch like scrolling with the mouse on non-touch platforms.
         */
        nonTouchScrolling?: boolean;
        /**
         * Native scrolling does not need content wrapper. In this case, ID of the container element should be provided.
         */
        scrollContainerId?: string;
        /**
         * if true, the delegate event listeners are called before the event listeners of the element; default is
         * "false".
         */
        callBefore?: boolean;
      }
    );

    /**
     * Creates a new subclass of class sap.ui.core.delegate.ScrollEnablement with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ScrollEnablement>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.delegate.ScrollEnablement.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Destroys this Scrolling delegate.
     *
     * This function must be called by the control which uses this delegate in the `exit` function.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    destroy(): void;
    /**
     * Calculates scroll position of a child of a container.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Position object.
     */
    getChildPosition(
      /**
       * An element(DOM or jQuery) for which the scroll position will be calculated.
       */
      vElement: HTMLElement | jQuery
    ): object;
    /**
     * Get current setting for horizontal scrolling.
     *
     * @since 1.9.1
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns true if horizontal scrolling is enabled
     */
    getHorizontal(): boolean;
    /**
     * Get current setting for vertical scrolling.
     *
     * @since 1.9.1
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns true if vertical scrolling is enabled
     */
    getVertical(): boolean;
    /**
     * Refreshes this Scrolling delegate.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    refresh(): void;
    /**
     * Scrolls to a specific position in scroll container.
     */
    scrollTo(
      /**
       * Horizontal position of the scrollbar
       */
      x: int,
      /**
       * Vertical position of the scrollbar
       */
      y: int,
      /**
       * The duration of animated scrolling in milliseconds. To scroll immediately without animation, give 0 as
       * value.
       */
      time: int,
      /**
       * Called when the scroll completes or stops without completing
       */
      fnScrollEndCallback: Function
    ): this;
    /**
     * Scrolls to a specific position in scroll container.
     */
    scrollTo(
      /**
       * Horizontal position of the scrollbar
       */
      x: int,
      /**
       * Vertical position of the scrollbar
       */
      y: int,
      /**
       * Called when the scroll completes or stops without completing
       */
      fnScrollEndCallback: Function
    ): this;
    /**
     * Scrolls to an element within a container.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    scrollToElement(
      /**
       * A DOM element.
       */
      oElement: HTMLElement,
      /**
       * The duration of animated scrolling in milliseconds. To scroll immediately without animation, give 0 as
       * value.
       */
      iTime?: int,
      /**
       * Specifies an additional left and top offset of the target scroll position, relative to the upper left
       * corner of the DOM element
       */
      aOffset?: int[],
      /**
       * The configuration of the parameter for scrolling only if the element is not in the view port - i.e. if
       * bSkipElementsInScrollport is set to true, there will be no scrolling if the element is already in the
       * view port
       */
      bSkipElementsInScrollport?: boolean
    ): this;
    /**
     * Setter for property `bounce`.
     *
     * @since 1.17
     * @deprecated As of version 1.42. without replacement.
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setBounce(
      /**
       * new value for property `bounce`.
       */
      bBounce: boolean
    ): void;
    /**
     * Sets GrowingList control to scroll container
     *
     * @since 1.11.0
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setGrowingList(
      /**
       * Scrolling callback
       */
      fnScrollLoadCallback: Function,
      /**
       * Scrolling direction
       */
      sScrollLoadDirection: /* was: sap.m.ListGrowingDirection */ any,
      /**
       * listener for the `overflowChange` event
       */
      fnOverflowChange: Function
    ): void;
    /**
     * Enable or disable horizontal scrolling.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setHorizontal(
      /**
       * set true to enable horizontal scrolling, false - to disable
       */
      bHorizontal: boolean
    ): void;
    /**
     * Sets IconTabBar control to scroll container
     *
     * @since 1.16.1
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setIconTabBar(
      /**
       * instance
       */
      oIconTabBar: /* was: sap.m.IconTabBar */ any,
      /**
       * callback function for the scroll end event
       */
      fnScrollEndCallback: Function,
      /**
       * callback function for the scroll start event
       */
      fnScrollStartCallback: Function
    ): void;
    /**
     * Set overflow control on top of scroll container.
     *
     * @since 1.9.2
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setPullDown(
      /**
       * Top control that should be normally hidden over the top border of the scroll container (pull-down content).
       */
      oControl: Control
    ): void;
    /**
     * Enable or disable vertical scrolling.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setVertical(
      /**
       * set true to enable vertical scrolling, false - to disable
       */
      bVertical: boolean
    ): void;
  }
}

declare module "sap/ui/core/dnd/DragDropBase" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Provides the base class for all drag-and-drop configurations. This feature enables a native HTML5 drag-and-drop
   * API for the controls, therefore it is limited to browser support. Restrictions:
   * 	 - There is no accessible alternative for drag and drop. Applications which use the drag-and-drop functionality
   *     must provide an accessible alternative UI (for example, action buttons or menus) to perform the same
   *     operations.
   * 	 - Transparency of the drag ghost element and the cursor during drag-and-drop operations depends on
   *     the browser implementation.
   * 	 - Constraining a drag position is not possible, therefore there is no snap-to-grid or snap-to-element
   *     feature possible.
   * 	 - Texts in draggable controls cannot be selected.
   * 	 - The text of input fields in draggable controls can be selected, but not dragged.
   *
   * @since 1.52
   */
  export default abstract class DragDropBase extends UI5Element {
    /**
     * Constructor for a new DragDropBase.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link https://ui5.sap.com/#/topic/3ddb6cde6a8d416598ac8ced3f5d82d5 Drag and Drop}
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $DragDropBaseSettings
    );
    /**
     * Constructor for a new DragDropBase.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link https://ui5.sap.com/#/topic/3ddb6cde6a8d416598ac8ced3f5d82d5 Drag and Drop}
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $DragDropBaseSettings
    );

    /**
     * Creates a new subclass of class sap.ui.core.dnd.DragDropBase with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DragDropBase>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.dnd.DragDropBase.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getEnabled enabled}.
     *
     * Indicates whether this configuration is active or not.
     *
     * Default value is `true`.
     *
     * @since 1.56
     *
     * @returns Value of property `enabled`
     */
    getEnabled(): boolean;
    /**
     * Gets current value of property {@link #getGroupName groupName}.
     *
     * Defines the name of the group to which this object belongs. If `groupName` is specified, then this object
     * will only interacts with other drag-and-drop objects within the same group.
     *
     *
     * @returns Value of property `groupName`
     */
    getGroupName(): string;
    /**
     * Gets current value of property {@link #getKeyboardHandling keyboardHandling}.
     *
     * Indicates limited keyboard handling support for drag-and-drop configurations defined for aggregation
     * reordering.
     *
     * **Note:** If the drag-and-drop configuration is defined for the aggregation reordering of a control (only
     * if the `dropPosition` property is `Between`), the `Ctrl/Cmd + Left/Right` keys for horizontal move or
     * the `Ctrl/Cmd + Up/Down` keys for vertical move trigger a series of pseudo drag-and-drop events, such
     * as `dragstart, dragenter, drop, dragend`, to create an artificial drag-and-drop action. This keyboard
     * handling might not be suitable for every control where aggregation reordering is defined, and in such
     * cases, this property must not be set to `true`.
     *
     * Default value is `false`.
     *
     * @since 1.126
     *
     * @returns Value of property `keyboardHandling`
     */
    getKeyboardHandling(): boolean;
    /**
     * Sets a new value for property {@link #getEnabled enabled}.
     *
     * Indicates whether this configuration is active or not.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnabled(
      /**
       * New value for property `enabled`
       */
      bEnabled?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getGroupName groupName}.
     *
     * Defines the name of the group to which this object belongs. If `groupName` is specified, then this object
     * will only interacts with other drag-and-drop objects within the same group.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setGroupName(
      /**
       * New value for property `groupName`
       */
      sGroupName?: string
    ): this;
    /**
     * Sets a new value for property {@link #getKeyboardHandling keyboardHandling}.
     *
     * Indicates limited keyboard handling support for drag-and-drop configurations defined for aggregation
     * reordering.
     *
     * **Note:** If the drag-and-drop configuration is defined for the aggregation reordering of a control (only
     * if the `dropPosition` property is `Between`), the `Ctrl/Cmd + Left/Right` keys for horizontal move or
     * the `Ctrl/Cmd + Up/Down` keys for vertical move trigger a series of pseudo drag-and-drop events, such
     * as `dragstart, dragenter, drop, dragend`, to create an artificial drag-and-drop action. This keyboard
     * handling might not be suitable for every control where aggregation reordering is defined, and in such
     * cases, this property must not be set to `true`.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.126
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKeyboardHandling(
      /**
       * New value for property `keyboardHandling`
       */
      bKeyboardHandling?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the DragDropBase constructor.
   */
  export interface $DragDropBaseSettings extends $ElementSettings {
    /**
     * Defines the name of the group to which this object belongs. If `groupName` is specified, then this object
     * will only interacts with other drag-and-drop objects within the same group.
     */
    groupName?: string | PropertyBindingInfo;

    /**
     * Indicates whether this configuration is active or not.
     *
     * @since 1.56
     */
    enabled?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Indicates limited keyboard handling support for drag-and-drop configurations defined for aggregation
     * reordering.
     *
     * **Note:** If the drag-and-drop configuration is defined for the aggregation reordering of a control (only
     * if the `dropPosition` property is `Between`), the `Ctrl/Cmd + Left/Right` keys for horizontal move or
     * the `Ctrl/Cmd + Up/Down` keys for vertical move trigger a series of pseudo drag-and-drop events, such
     * as `dragstart, dragenter, drop, dragend`, to create an artificial drag-and-drop action. This keyboard
     * handling might not be suitable for every control where aggregation reordering is defined, and in such
     * cases, this property must not be set to `true`.
     *
     * @since 1.126
     */
    keyboardHandling?: boolean | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/ui/core/dnd/DragDropInfo" {
  import {
    default as DropInfo,
    $DropInfoSettings,
  } from "sap/ui/core/dnd/DropInfo";

  import { dnd, ID } from "sap/ui/core/library";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import UI5Element from "sap/ui/core/Element";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import { DragSession } from "sap/ui/core/dnd/DragAndDrop";

  /**
   * Provides the configuration for drag-and-drop operations.
   *
   * **Note:** This configuration might be ignored due to control {@link sap.ui.core.Element.extend metadata }
   * restrictions.
   *
   * @since 1.52
   */
  export default class DragDropInfo
    extends DropInfo
    implements dnd.IDragInfo, dnd.IDropInfo
  {
    __implements__sap_ui_core_dnd_IDragInfo: boolean;
    __implements__sap_ui_core_dnd_IDropInfo: boolean;
    /**
     * Constructor for a new DragDropInfo.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the DragDropInfo
       */
      mSettings?: $DragDropInfoSettings
    );
    /**
     * Constructor for a new DragDropInfo.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new DragDropInfo, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the DragDropInfo
       */
      mSettings?: $DragDropInfoSettings
    );

    /**
     * Creates a new subclass of class sap.ui.core.dnd.DragDropInfo with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.dnd.DropInfo.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DragDropInfo>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.dnd.DragDropInfo.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragEnd dragEnd} event of this `sap.ui.core.dnd.DragDropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragDropInfo` itself.
     *
     * This event is fired when a drag operation is being ended.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragEnd(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragDropInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragEnd dragEnd} event of this `sap.ui.core.dnd.DragDropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragDropInfo` itself.
     *
     * This event is fired when a drag operation is being ended.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragEnd(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragDropInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragStart dragStart} event of this `sap.ui.core.dnd.DragDropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragDropInfo` itself.
     *
     * This event is fired when the user starts dragging an element.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragStart(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DragDropInfo$DragStartEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragDropInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragStart dragStart} event of this `sap.ui.core.dnd.DragDropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragDropInfo` itself.
     *
     * This event is fired when the user starts dragging an element.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragStart(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DragDropInfo$DragStartEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragDropInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:dragEnd dragEnd} event of this `sap.ui.core.dnd.DragDropInfo`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachDragEnd(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:dragStart dragStart} event of this `sap.ui.core.dnd.DragDropInfo`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachDragStart(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: DragDropInfo$DragStartEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:dragEnd dragEnd} to attached listeners.
     *
     * @since 1.56
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireDragEnd(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:dragStart dragStart} to attached listeners.
     *
     * Listeners may prevent the default action of this event by calling the `preventDefault` method on the
     * event object. The return value of this method indicates whether the default action should be executed.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether or not to prevent the default action
     */
    fireDragStart(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: DragDropInfo$DragStartEventParameters
    ): boolean;
    /**
     * Gets current value of property {@link #getSourceAggregation sourceAggregation}.
     *
     * The name of the aggregation from which all children can be dragged. If undefined, the control itself
     * can be dragged.
     *
     *
     * @returns Value of property `sourceAggregation`
     */
    getSourceAggregation(): string;
    /**
     * ID of the element which is the current target of the association {@link #getTargetElement targetElement},
     * or `null`.
     */
    getTargetElement(): ID | null;
    /**
     * Sets a new value for property {@link #getSourceAggregation sourceAggregation}.
     *
     * The name of the aggregation from which all children can be dragged. If undefined, the control itself
     * can be dragged.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSourceAggregation(
      /**
       * New value for property `sourceAggregation`
       */
      sSourceAggregation?: string
    ): this;
    /**
     * Sets the associated {@link #getTargetElement targetElement}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTargetElement(
      /**
       * ID of an element which becomes the new target of this targetElement association; alternatively, an element
       * instance may be given
       */
      oTargetElement: ID | UI5Element
    ): this;
  }
  /**
   * Describes the settings that can be provided to the DragDropInfo constructor.
   */
  export interface $DragDropInfoSettings extends $DropInfoSettings {
    /**
     * The name of the aggregation from which all children can be dragged. If undefined, the control itself
     * can be dragged.
     */
    sourceAggregation?: string | PropertyBindingInfo;

    /**
     * The target element for this drag and drop action. If undefined, the control with this drag and drop configuration
     * itself is the target. Leaving this empty, but defining source and target aggregation, allows you to reorder
     * the children within a control, for example.
     */
    targetElement?: UI5Element | string;

    /**
     * This event is fired when the user starts dragging an element.
     */
    dragStart?: (oEvent: DragDropInfo$DragStartEvent) => void;

    /**
     * This event is fired when a drag operation is being ended.
     *
     * @since 1.56
     */
    dragEnd?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the DragDropInfo#dragEnd event.
   */
  export interface DragDropInfo$DragEndEventParameters {}

  /**
   * Event object of the DragDropInfo#dragEnd event.
   */
  export type DragDropInfo$DragEndEvent = Event<
    DragDropInfo$DragEndEventParameters,
    DragDropInfo
  >;

  /**
   * Parameters of the DragDropInfo#dragStart event.
   */
  export interface DragDropInfo$DragStartEventParameters {
    /**
     * The target element that will be dragged
     */
    target?: UI5Element;

    /**
     * The UI5 `dragSession` object that exists only during drag and drop
     */
    dragSession?: DragSession;

    /**
     * The underlying browser event
     */
    browserEvent?: DragEvent;
  }

  /**
   * Event object of the DragDropInfo#dragStart event.
   */
  export type DragDropInfo$DragStartEvent = Event<
    DragDropInfo$DragStartEventParameters,
    DragDropInfo
  >;
}

declare module "sap/ui/core/dnd/DragInfo" {
  import {
    default as DragDropBase,
    $DragDropBaseSettings,
  } from "sap/ui/core/dnd/DragDropBase";

  import { dnd } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import UI5Element from "sap/ui/core/Element";

  import { DragSession } from "sap/ui/core/dnd/DragAndDrop";

  import Event from "sap/ui/base/Event";

  /**
   * Provides the configuration for drag operations.
   *
   * **Note:** This configuration might be ignored due to control {@link sap.ui.core.Element.extend metadata }
   * restrictions.
   *
   * @since 1.56
   */
  export default class DragInfo extends DragDropBase implements dnd.IDragInfo {
    __implements__sap_ui_core_dnd_IDragInfo: boolean;
    /**
     * Constructor for a new DragInfo.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the DragInfo
       */
      mSettings?: $DragInfoSettings
    );
    /**
     * Constructor for a new DragInfo.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new DragInfo, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the DragInfo
       */
      mSettings?: $DragInfoSettings
    );

    /**
     * Creates a new subclass of class sap.ui.core.dnd.DragInfo with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.dnd.DragDropBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DragInfo>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.dnd.DragInfo.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragEnd dragEnd} event of this `sap.ui.core.dnd.DragInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragInfo` itself.
     *
     * This event is fired when a drag operation is ended.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragEnd(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DragInfo$DragEndEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragEnd dragEnd} event of this `sap.ui.core.dnd.DragInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragInfo` itself.
     *
     * This event is fired when a drag operation is ended.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragEnd(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DragInfo$DragEndEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragStart dragStart} event of this `sap.ui.core.dnd.DragInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragInfo` itself.
     *
     * This event is fired when the user starts dragging an element.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragStart(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DragInfo$DragStartEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragStart dragStart} event of this `sap.ui.core.dnd.DragInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DragInfo` itself.
     *
     * This event is fired when the user starts dragging an element.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragStart(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DragInfo$DragStartEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DragInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:dragEnd dragEnd} event of this `sap.ui.core.dnd.DragInfo`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachDragEnd(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: DragInfo$DragEndEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:dragStart dragStart} event of this `sap.ui.core.dnd.DragInfo`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachDragStart(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: DragInfo$DragStartEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:dragEnd dragEnd} to attached listeners.
     *
     * @since 1.56
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireDragEnd(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: DragInfo$DragEndEventParameters
    ): this;
    /**
     * Fires event {@link #event:dragStart dragStart} to attached listeners.
     *
     * Listeners may prevent the default action of this event by calling the `preventDefault` method on the
     * event object. The return value of this method indicates whether the default action should be executed.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether or not to prevent the default action
     */
    fireDragStart(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: DragInfo$DragStartEventParameters
    ): boolean;
    /**
     * Gets current value of property {@link #getSourceAggregation sourceAggregation}.
     *
     * The name of the aggregation from which all children can be dragged. If undefined, the control itself
     * can be dragged.
     *
     *
     * @returns Value of property `sourceAggregation`
     */
    getSourceAggregation(): string;
    /**
     * Sets a new value for property {@link #getSourceAggregation sourceAggregation}.
     *
     * The name of the aggregation from which all children can be dragged. If undefined, the control itself
     * can be dragged.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSourceAggregation(
      /**
       * New value for property `sourceAggregation`
       */
      sSourceAggregation?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the DragInfo constructor.
   */
  export interface $DragInfoSettings extends $DragDropBaseSettings {
    /**
     * The name of the aggregation from which all children can be dragged. If undefined, the control itself
     * can be dragged.
     */
    sourceAggregation?: string | PropertyBindingInfo;

    /**
     * This event is fired when the user starts dragging an element.
     */
    dragStart?: (oEvent: DragInfo$DragStartEvent) => void;

    /**
     * This event is fired when a drag operation is ended.
     *
     * @since 1.56
     */
    dragEnd?: (oEvent: DragInfo$DragEndEvent) => void;
  }

  /**
   * Parameters of the DragInfo#dragEnd event.
   */
  export interface DragInfo$DragEndEventParameters {
    /**
     * The target element that is being dragged
     */
    target?: UI5Element;

    /**
     * The UI5 `dragSession` object that exists only during drag and drop
     */
    dragSession?: DragSession;

    /**
     * The underlying browser event
     */
    browserEvent?: DragEvent;
  }

  /**
   * Event object of the DragInfo#dragEnd event.
   */
  export type DragInfo$DragEndEvent = Event<
    DragInfo$DragEndEventParameters,
    DragInfo
  >;

  /**
   * Parameters of the DragInfo#dragStart event.
   */
  export interface DragInfo$DragStartEventParameters {
    /**
     * The target element that will be dragged
     */
    target?: UI5Element;

    /**
     * The UI5 `dragSession` object that exists only during drag and drop
     */
    dragSession?: DragSession;

    /**
     * The underlying browser event
     */
    browserEvent?: DragEvent;
  }

  /**
   * Event object of the DragInfo#dragStart event.
   */
  export type DragInfo$DragStartEvent = Event<
    DragInfo$DragStartEventParameters,
    DragInfo
  >;
}

declare module "sap/ui/core/dnd/DragAndDrop" {
  import UI5Element from "sap/ui/core/Element";

  import DropInfo from "sap/ui/core/dnd/DropInfo";

  import { dnd } from "sap/ui/core/library";

  /**
   * When a user requests to drag some controls that can be dragged, a drag session is started. The drag session
   * can be used to transfer data between applications or between dragged and dropped controls. Please see
   * provided APIs for more details.
   *
   * **Note:** An implementation of this interface is provided by the framework during drag-and-drop operations
   * and is exposed as `dragSession` parameter of the different drag and drop events.
   *
   * **Note:** This interface is not intended to be implemented by controls, applications or test code. Extending
   * it with additional methods in future versions will be regarded as a compatible change.
   */
  export interface DragSession {
    __implements__sap_ui_core_dnd_DragSession: boolean;

    /**
     * Returns the data that has been set via `setComplexData` method.
     *
     *
     * @returns The previously set data or undefined
     */
    getComplexData(
      /**
       * The key of the data
       */
      sKey: string
    ): any | undefined;
    /**
     * Returns the data that has been set via `setData` method.
     *
     *
     * @returns Data
     */
    getData(
      /**
       * The key of the data
       */
      sKey: string
    ): string;
    /**
     * Returns the dragged control, if available within the same UI5 application frame.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The dragged control
     */
    getDragControl(): UI5Element | null;
    /**
     * The valid drop target underneath the dragged control.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The drop target
     */
    getDropControl(): UI5Element | null;
    /**
     * Returns the drop configuration corresponding to the drop control.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The drop configuration
     */
    getDropInfo(): DropInfo | null;
    /**
     * Returns the calculated position of the drop action relative to the valid dropped control.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The calculated position
     */
    getDropPosition(): dnd.RelativeDropPosition;
    /**
     * Returns the drop indicator.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Drop indicator's DOM reference
     */
    getIndicator(): HTMLElement | null;
    /**
     * Returns the visual configuration of the drop indicator.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Drop indicator configuration
     */
    getIndicatorConfig(): object;
    /**
     * Returns the data that has been set via `setTextData` method.
     *
     *
     * @returns Data
     */
    getTextData(): string;
    /**
     * Sets any type of data (even functions, pointers, anything non-serializable) with any MIME type. **Note:**
     * This works only within a UI5 application within the same window/frame.
     */
    setComplexData(
      /**
       * The key of the data
       */
      sKey: string,
      /**
       * Data
       */
      vData: any
    ): void;
    /**
     * Sets string data with any MIME type. **Note:** This works if you navigate between different windows.
     */
    setData(
      /**
       * The key of the data
       */
      sKey: string,
      /**
       * Data
       */
      sData: string
    ): void;
    /**
     * Set the valid target.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setDropControl(
      /**
       * The dropped target
       */
      oControl: UI5Element
    ): void;
    /**
     * Defines the visual configuration of the drop indicator for the current `DropInfo`.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    setIndicatorConfig(
      /**
       * Custom styles of the drop indicator.
       */
      mConfig: object
    ): void;
    /**
     * Sets string data with plain text MIME type. **Note:** This works if you navigate between different windows.
     */
    setTextData(
      /**
       * Data
       */
      sData: string
    ): void;
  }
}

declare module "sap/ui/core/dnd/DropInfo" {
  import {
    default as DragDropBase,
    $DragDropBaseSettings,
  } from "sap/ui/core/dnd/DragDropBase";

  import { dnd } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import UI5Element from "sap/ui/core/Element";

  import { DragSession } from "sap/ui/core/dnd/DragAndDrop";

  import Event from "sap/ui/base/Event";

  /**
   * Provides the configuration for drop operations. **Note:** This configuration might be ignored due to
   * control {@link sap.ui.core.Element.extend metadata} restrictions.
   *
   * @since 1.56
   */
  export default class DropInfo extends DragDropBase implements dnd.IDropInfo {
    __implements__sap_ui_core_dnd_IDropInfo: boolean;
    /**
     * Constructor for a new DropInfo.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the DropInfo
       */
      mSettings?: $DropInfoSettings
    );
    /**
     * Constructor for a new DropInfo.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new DropInfo, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the DropInfo
       */
      mSettings?: $DropInfoSettings
    );

    /**
     * Creates a new subclass of class sap.ui.core.dnd.DropInfo with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.dnd.DragDropBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DropInfo>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.core.dnd.DropInfo.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragEnter dragEnter} event of this `sap.ui.core.dnd.DropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DropInfo` itself.
     *
     * This event is fired when a dragged element enters a drop target.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragEnter(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DropInfo$DragEnterEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DropInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragEnter dragEnter} event of this `sap.ui.core.dnd.DropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DropInfo` itself.
     *
     * This event is fired when a dragged element enters a drop target.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragEnter(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DropInfo$DragEnterEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.core.dnd.DropInfo` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:dragOver dragOver} event of this `sap.ui.core.dnd.DropInfo`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.core.dnd.DropInfo` itself.
     *
     * This event is fired when an element is being dragged over a valid drop target.
     *
     * @since 1.56
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDragOver(
      /**
       * An applica