'use strict';

var utils$1 = require('@typescript-eslint/utils');
var astUtils = require('@typescript-eslint/utils/ast-utils');
var utils = require('./utils.js');

function createAllConfigs(plugin, name, flat, filter) {
  const rules = Object.fromEntries(
    Object.entries(plugin.rules).filter(
      ([key, rule]) => (
        // Only include fixable rules
        rule.meta.fixable && !rule.meta.deprecated && key === rule.meta.docs.url.split("/").pop() && (!filter || filter(key, rule))
      )
    ).map(([key]) => [`${name}/${key}`, 2])
  );
  if (flat) {
    return {
      plugins: {
        [name]: plugin
      },
      rules
    };
  } else {
    return {
      plugins: [name],
      rules
    };
  }
}

const baseRule$f = utils.getESLintCoreRule("block-spacing");
var blockSpacing = utils.createRule({
  name: "block-spacing",
  meta: {
    type: "layout",
    docs: {
      description: "Disallow or enforce spaces inside of blocks after opening block and before closing block",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    hasSuggestions: baseRule$f.meta.hasSuggestions,
    schema: baseRule$f.meta.schema,
    messages: baseRule$f.meta.messages
  },
  defaultOptions: ["always"],
  create(context, [whenToApplyOption]) {
    const sourceCode = context.sourceCode;
    const baseRules = baseRule$f.create(context);
    const always = whenToApplyOption !== "never";
    const messageId = always ? "missing" : "extra";
    function getOpenBrace(node) {
      return sourceCode.getFirstToken(node, {
        filter: (token) => token.type === utils$1.AST_TOKEN_TYPES.Punctuator && token.value === "{"
      });
    }
    function isValid(left, right) {
      return !astUtils.isTokenOnSameLine(left, right) || sourceCode.isSpaceBetween(left, right) === always;
    }
    function checkSpacingInsideBraces(node) {
      const openBrace = getOpenBrace(node);
      const closeBrace = sourceCode.getLastToken(node);
      const firstToken = sourceCode.getTokenAfter(openBrace, {
        includeComments: true
      });
      const lastToken = sourceCode.getTokenBefore(closeBrace, {
        includeComments: true
      });
      if (openBrace.type !== utils$1.AST_TOKEN_TYPES.Punctuator || openBrace.value !== "{" || closeBrace.type !== utils$1.AST_TOKEN_TYPES.Punctuator || closeBrace.value !== "}" || firstToken === closeBrace)
        return;
      if (!always && firstToken.type === utils$1.AST_TOKEN_TYPES.Line)
        return;
      if (!isValid(openBrace, firstToken)) {
        let loc = openBrace.loc;
        if (messageId === "extra") {
          loc = {
            start: openBrace.loc.end,
            end: firstToken.loc.start
          };
        }
        context.report({
          node,
          loc,
          messageId,
          data: {
            location: "after",
            token: openBrace.value
          },
          fix(fixer) {
            if (always)
              return fixer.insertTextBefore(firstToken, " ");
            return fixer.removeRange([openBrace.range[1], firstToken.range[0]]);
          }
        });
      }
      if (!isValid(lastToken, closeBrace)) {
        let loc = closeBrace.loc;
        if (messageId === "extra") {
          loc = {
            start: lastToken.loc.end,
            end: closeBrace.loc.start
          };
        }
        context.report({
          node,
          loc,
          messageId,
          data: {
            location: "before",
            token: closeBrace.value
          },
          fix(fixer) {
            if (always)
              return fixer.insertTextAfter(lastToken, " ");
            return fixer.removeRange([lastToken.range[1], closeBrace.range[0]]);
          }
        });
      }
    }
    return {
      ...baseRules,
      // This code worked "out of the box" for interface and type literal
      // Enums were very close to match as well, the only reason they are not is that was that enums don't have a body node in the parser
      // So the opening brace punctuator starts in the middle of the node - `getFirstToken` in
      // the base rule did not filter for the first opening brace punctuator
      TSInterfaceBody: baseRules.BlockStatement,
      TSTypeLiteral: baseRules.BlockStatement,
      TSEnumDeclaration: checkSpacingInsideBraces
    };
  }
});

const baseRule$e = utils.getESLintCoreRule("brace-style");
var braceStyle = utils.createRule({
  name: "brace-style",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent brace style for blocks",
      extendsBaseRule: true
    },
    messages: baseRule$e.meta.messages,
    fixable: baseRule$e.meta.fixable,
    hasSuggestions: baseRule$e.meta.hasSuggestions,
    schema: baseRule$e.meta.schema
  },
  defaultOptions: ["1tbs"],
  create(context) {
    const [style, { allowSingleLine } = { allowSingleLine: false }] = context.options;
    const isAllmanStyle = style === "allman";
    const sourceCode = context.sourceCode;
    const rules = baseRule$e.create(context);
    function validateCurlyPair(openingCurlyToken, closingCurlyToken) {
      if (allowSingleLine && astUtils.isTokenOnSameLine(openingCurlyToken, closingCurlyToken))
        return;
      const tokenBeforeOpeningCurly = sourceCode.getTokenBefore(openingCurlyToken);
      const tokenBeforeClosingCurly = sourceCode.getTokenBefore(closingCurlyToken);
      const tokenAfterOpeningCurly = sourceCode.getTokenAfter(openingCurlyToken);
      if (!isAllmanStyle && !astUtils.isTokenOnSameLine(tokenBeforeOpeningCurly, openingCurlyToken)) {
        context.report({
          node: openingCurlyToken,
          messageId: "nextLineOpen",
          fix: (fixer) => {
            const textRange = [
              tokenBeforeOpeningCurly.range[1],
              openingCurlyToken.range[0]
            ];
            const textBetween = sourceCode.text.slice(
              textRange[0],
              textRange[1]
            );
            if (textBetween.trim())
              return null;
            return fixer.replaceTextRange(textRange, " ");
          }
        });
      }
      if (isAllmanStyle && astUtils.isTokenOnSameLine(tokenBeforeOpeningCurly, openingCurlyToken)) {
        context.report({
          node: openingCurlyToken,
          messageId: "sameLineOpen",
          fix: (fixer) => fixer.insertTextBefore(openingCurlyToken, "\n")
        });
      }
      if (astUtils.isTokenOnSameLine(openingCurlyToken, tokenAfterOpeningCurly) && tokenAfterOpeningCurly !== closingCurlyToken) {
        context.report({
          node: openingCurlyToken,
          messageId: "blockSameLine",
          fix: (fixer) => fixer.insertTextAfter(openingCurlyToken, "\n")
        });
      }
      if (astUtils.isTokenOnSameLine(tokenBeforeClosingCurly, closingCurlyToken) && tokenBeforeClosingCurly !== openingCurlyToken) {
        context.report({
          node: closingCurlyToken,
          messageId: "singleLineClose",
          fix: (fixer) => fixer.insertTextBefore(closingCurlyToken, "\n")
        });
      }
    }
    return {
      ...rules,
      "TSInterfaceBody, TSModuleBlock": function(node) {
        const openingCurly = sourceCode.getFirstToken(node);
        const closingCurly = sourceCode.getLastToken(node);
        validateCurlyPair(openingCurly, closingCurly);
      },
      TSEnumDeclaration(node) {
        const closingCurly = sourceCode.getLastToken(node);
        const openingCurly = sourceCode.getTokenBefore(
          node.members.length ? node.members[0] : closingCurly
        );
        validateCurlyPair(openingCurly, closingCurly);
      }
    };
  }
});

const baseRule$d = utils.getESLintCoreRule("comma-dangle");
const OPTION_VALUE_SCHEME = [
  "always-multiline",
  "always",
  "never",
  "only-multiline"
];
const DEFAULT_OPTION_VALUE = "never";
function normalizeOptions(options = {}) {
  if (typeof options === "string") {
    return {
      enums: options,
      generics: options,
      tuples: options
    };
  }
  return {
    enums: options.enums ?? DEFAULT_OPTION_VALUE,
    generics: options.generics ?? DEFAULT_OPTION_VALUE,
    tuples: options.tuples ?? DEFAULT_OPTION_VALUE
  };
}
var commaDangle = utils.createRule({
  name: "comma-dangle",
  meta: {
    type: "layout",
    docs: {
      description: "Require or disallow trailing commas",
      extendsBaseRule: true
    },
    schema: {
      $defs: {
        value: {
          type: "string",
          enum: OPTION_VALUE_SCHEME
        },
        valueWithIgnore: {
          type: "string",
          enum: [...OPTION_VALUE_SCHEME, "ignore"]
        }
      },
      type: "array",
      items: [
        {
          oneOf: [
            {
              $ref: "#/$defs/value"
            },
            {
              type: "object",
              properties: {
                arrays: { $ref: "#/$defs/valueWithIgnore" },
                objects: { $ref: "#/$defs/valueWithIgnore" },
                imports: { $ref: "#/$defs/valueWithIgnore" },
                exports: { $ref: "#/$defs/valueWithIgnore" },
                functions: { $ref: "#/$defs/valueWithIgnore" },
                enums: { $ref: "#/$defs/valueWithIgnore" },
                generics: { $ref: "#/$defs/valueWithIgnore" },
                tuples: { $ref: "#/$defs/valueWithIgnore" }
              },
              additionalProperties: false
            }
          ]
        }
      ],
      additionalItems: false
    },
    fixable: "code",
    hasSuggestions: baseRule$d.meta.hasSuggestions,
    messages: baseRule$d.meta.messages
  },
  defaultOptions: ["never"],
  create(context, [options]) {
    const rules = baseRule$d.create(context);
    const sourceCode = context.sourceCode;
    const normalizedOptions = normalizeOptions(options);
    const isTSX = context.parserOptions?.ecmaFeatures?.jsx && context.filename?.endsWith(".tsx");
    const predicate = {
      "always": forceComma,
      "always-multiline": forceCommaIfMultiline,
      "only-multiline": allowCommaIfMultiline,
      "never": forbidComma,
      // https://github.com/typescript-eslint/typescript-eslint/issues/7220
      "ignore": () => {
      }
    };
    function last(nodes) {
      return nodes[nodes.length - 1] ?? null;
    }
    function getLastItem(node) {
      switch (node.type) {
        case utils$1.AST_NODE_TYPES.TSEnumDeclaration:
          return last(node.members);
        case utils$1.AST_NODE_TYPES.TSTypeParameterDeclaration:
          return last(node.params);
        case utils$1.AST_NODE_TYPES.TSTupleType:
          return last(node.elementTypes);
        default:
          return null;
      }
    }
    function getTrailingToken(node) {
      const last2 = getLastItem(node);
      const trailing = last2 && sourceCode.getTokenAfter(last2);
      return trailing;
    }
    function isMultiline(node) {
      const last2 = getLastItem(node);
      const lastToken = sourceCode.getLastToken(node);
      return last2?.loc.end.line !== lastToken?.loc.end.line;
    }
    function forbidComma(node) {
      if (isTSX && node.type === utils$1.AST_NODE_TYPES.TSTypeParameterDeclaration && node.params.length === 1)
        return;
      const last2 = getLastItem(node);
      const trailing = getTrailingToken(node);
      if (last2 && trailing && astUtils.isCommaToken(trailing)) {
        context.report({
          node,
          messageId: "unexpected",
          fix(fixer) {
            return fixer.remove(trailing);
          }
        });
      }
    }
    function forceComma(node) {
      const last2 = getLastItem(node);
      const trailing = getTrailingToken(node);
      if (last2 && trailing && !astUtils.isCommaToken(trailing)) {
        context.report({
          node,
          messageId: "missing",
          fix(fixer) {
            return fixer.insertTextAfter(last2, ",");
          }
        });
      }
    }
    function allowCommaIfMultiline(node) {
      if (!isMultiline(node))
        forbidComma(node);
    }
    function forceCommaIfMultiline(node) {
      if (isMultiline(node))
        forceComma(node);
      else
        forbidComma(node);
    }
    return {
      ...rules,
      TSEnumDeclaration: predicate[normalizedOptions.enums],
      TSTypeParameterDeclaration: predicate[normalizedOptions.generics],
      TSTupleType: predicate[normalizedOptions.tuples]
    };
  }
});

var commaSpacing = utils.createRule({
  name: "comma-spacing",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent spacing before and after commas",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    schema: [
      {
        type: "object",
        properties: {
          before: {
            type: "boolean",
            default: false
          },
          after: {
            type: "boolean",
            default: true
          }
        },
        additionalProperties: false
      }
    ],
    messages: {
      unexpected: `There should be no space {{loc}} ','.`,
      missing: `A space is required {{loc}} ','.`
    }
  },
  defaultOptions: [
    {
      before: false,
      after: true
    }
  ],
  create(context, [options = {}]) {
    const { before: spaceBefore, after: spaceAfter } = options;
    const sourceCode = context.sourceCode;
    const tokensAndComments = sourceCode.tokensAndComments;
    const ignoredTokens = /* @__PURE__ */ new Set();
    function addNullElementsToIgnoreList(node) {
      let previousToken = sourceCode.getFirstToken(node);
      for (const element of node.elements) {
        let token;
        if (element == null) {
          token = sourceCode.getTokenAfter(previousToken);
          if (token && astUtils.isCommaToken(token))
            ignoredTokens.add(token);
        } else {
          token = sourceCode.getTokenAfter(element);
        }
        previousToken = token;
      }
    }
    function addTypeParametersTrailingCommaToIgnoreList(node) {
      const paramLength = node.params.length;
      if (paramLength) {
        const param = node.params[paramLength - 1];
        const afterToken = sourceCode.getTokenAfter(param);
        if (afterToken && astUtils.isCommaToken(afterToken))
          ignoredTokens.add(afterToken);
      }
    }
    function validateCommaSpacing(commaToken, prevToken, nextToken) {
      if (prevToken && astUtils.isTokenOnSameLine(prevToken, commaToken) && spaceBefore !== sourceCode.isSpaceBetweenTokens(prevToken, commaToken)) {
        context.report({
          node: commaToken,
          data: {
            loc: "before"
          },
          messageId: spaceBefore ? "missing" : "unexpected",
          fix: (fixer) => spaceBefore ? fixer.insertTextBefore(commaToken, " ") : fixer.replaceTextRange(
            [prevToken.range[1], commaToken.range[0]],
            ""
          )
        });
      }
      if (nextToken && astUtils.isClosingParenToken(nextToken))
        return;
      if (spaceAfter && nextToken && (astUtils.isClosingBraceToken(nextToken) || astUtils.isClosingBracketToken(nextToken)))
        return;
      if (!spaceAfter && nextToken && nextToken.type === utils$1.AST_TOKEN_TYPES.Line)
        return;
      if (nextToken && astUtils.isTokenOnSameLine(commaToken, nextToken) && spaceAfter !== sourceCode.isSpaceBetweenTokens(commaToken, nextToken)) {
        context.report({
          node: commaToken,
          data: {
            loc: "after"
          },
          messageId: spaceAfter ? "missing" : "unexpected",
          fix: (fixer) => spaceAfter ? fixer.insertTextAfter(commaToken, " ") : fixer.replaceTextRange(
            [commaToken.range[1], nextToken.range[0]],
            ""
          )
        });
      }
    }
    return {
      "TSTypeParameterDeclaration": addTypeParametersTrailingCommaToIgnoreList,
      "ArrayExpression": addNullElementsToIgnoreList,
      "ArrayPattern": addNullElementsToIgnoreList,
      "Program:exit": function() {
        tokensAndComments.forEach((token, i) => {
          if (!astUtils.isCommaToken(token))
            return;
          const prevToken = tokensAndComments[i - 1];
          const nextToken = tokensAndComments[i + 1];
          validateCommaSpacing(
            token,
            astUtils.isCommaToken(prevToken) || ignoredTokens.has(token) ? null : prevToken,
            nextToken && astUtils.isCommaToken(nextToken) || ignoredTokens.has(token) ? null : nextToken
          );
        });
      }
    };
  }
});

var functionCallSpacing = utils.createRule({
  name: "function-call-spacing",
  meta: {
    type: "layout",
    docs: {
      description: "Require or disallow spacing between function identifiers and their invocations",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    schema: {
      anyOf: [
        {
          type: "array",
          items: [
            {
              type: "string",
              enum: ["never"]
            }
          ],
          minItems: 0,
          maxItems: 1
        },
        {
          type: "array",
          items: [
            {
              type: "string",
              enum: ["always"]
            },
            {
              type: "object",
              properties: {
                allowNewlines: {
                  type: "boolean"
                }
              },
              additionalProperties: false
            }
          ],
          minItems: 0,
          maxItems: 2
        }
      ]
    },
    messages: {
      unexpectedWhitespace: "Unexpected whitespace between function name and paren.",
      unexpectedNewline: "Unexpected newline between function name and paren.",
      missing: "Missing space between function name and paren."
    }
  },
  defaultOptions: ["never", {}],
  create(context, [option, config]) {
    const sourceCode = context.sourceCode;
    const text = sourceCode.getText();
    function checkSpacing(node) {
      const isOptionalCall = astUtils.isOptionalCallExpression(node);
      const closingParenToken = sourceCode.getLastToken(node);
      const lastCalleeTokenWithoutPossibleParens = sourceCode.getLastToken(
        node.typeArguments ?? node.callee
      );
      const openingParenToken = sourceCode.getFirstTokenBetween(
        lastCalleeTokenWithoutPossibleParens,
        closingParenToken,
        astUtils.isOpeningParenToken
      );
      if (!openingParenToken || openingParenToken.range[1] >= node.range[1]) {
        return;
      }
      const lastCalleeToken = sourceCode.getTokenBefore(
        openingParenToken,
        astUtils.isNotOptionalChainPunctuator
      );
      const textBetweenTokens = text.slice(lastCalleeToken.range[1], openingParenToken.range[0]).replace(/\/\*.*?\*\//gu, "");
      const hasWhitespace = /\s/u.test(textBetweenTokens);
      const hasNewline = hasWhitespace && astUtils.LINEBREAK_MATCHER.test(textBetweenTokens);
      if (option === "never") {
        if (hasWhitespace) {
          return context.report({
            node,
            loc: lastCalleeToken.loc.start,
            messageId: "unexpectedWhitespace",
            fix(fixer) {
              if (!hasNewline && !isOptionalCall) {
                return fixer.removeRange([
                  lastCalleeToken.range[1],
                  openingParenToken.range[0]
                ]);
              }
              return null;
            }
          });
        }
      } else if (isOptionalCall) {
        if (hasWhitespace || hasNewline) {
          context.report({
            node,
            loc: lastCalleeToken.loc.start,
            messageId: "unexpectedWhitespace"
          });
        }
      } else {
        if (!hasWhitespace) {
          context.report({
            node,
            loc: lastCalleeToken.loc.start,
            messageId: "missing",
            fix(fixer) {
              return fixer.insertTextBefore(openingParenToken, " ");
            }
          });
        } else if (!config.allowNewlines && hasNewline) {
          context.report({
            node,
            loc: lastCalleeToken.loc.start,
            messageId: "unexpectedNewline",
            fix(fixer) {
              return fixer.replaceTextRange(
                [lastCalleeToken.range[1], openingParenToken.range[0]],
                " "
              );
            }
          });
        }
      }
    }
    return {
      CallExpression: checkSpacing,
      NewExpression: checkSpacing
    };
  }
});

const baseRule$c = utils.getESLintCoreRule("indent");
const KNOWN_NODES = /* @__PURE__ */ new Set([
  // Class properties aren't yet supported by eslint...
  utils$1.AST_NODE_TYPES.PropertyDefinition,
  // ts keywords
  utils$1.AST_NODE_TYPES.TSAbstractKeyword,
  utils$1.AST_NODE_TYPES.TSAnyKeyword,
  utils$1.AST_NODE_TYPES.TSBooleanKeyword,
  utils$1.AST_NODE_TYPES.TSNeverKeyword,
  utils$1.AST_NODE_TYPES.TSNumberKeyword,
  utils$1.AST_NODE_TYPES.TSStringKeyword,
  utils$1.AST_NODE_TYPES.TSSymbolKeyword,
  utils$1.AST_NODE_TYPES.TSUndefinedKeyword,
  utils$1.AST_NODE_TYPES.TSUnknownKeyword,
  utils$1.AST_NODE_TYPES.TSVoidKeyword,
  utils$1.AST_NODE_TYPES.TSNullKeyword,
  // ts specific nodes we want to support
  utils$1.AST_NODE_TYPES.TSAbstractPropertyDefinition,
  utils$1.AST_NODE_TYPES.TSAbstractMethodDefinition,
  utils$1.AST_NODE_TYPES.TSArrayType,
  utils$1.AST_NODE_TYPES.TSAsExpression,
  utils$1.AST_NODE_TYPES.TSCallSignatureDeclaration,
  utils$1.AST_NODE_TYPES.TSConditionalType,
  utils$1.AST_NODE_TYPES.TSConstructorType,
  utils$1.AST_NODE_TYPES.TSConstructSignatureDeclaration,
  utils$1.AST_NODE_TYPES.TSDeclareFunction,
  utils$1.AST_NODE_TYPES.TSEmptyBodyFunctionExpression,
  utils$1.AST_NODE_TYPES.TSEnumDeclaration,
  utils$1.AST_NODE_TYPES.TSEnumMember,
  utils$1.AST_NODE_TYPES.TSExportAssignment,
  utils$1.AST_NODE_TYPES.TSExternalModuleReference,
  utils$1.AST_NODE_TYPES.TSFunctionType,
  utils$1.AST_NODE_TYPES.TSImportType,
  utils$1.AST_NODE_TYPES.TSIndexedAccessType,
  utils$1.AST_NODE_TYPES.TSIndexSignature,
  utils$1.AST_NODE_TYPES.TSInferType,
  utils$1.AST_NODE_TYPES.TSInterfaceBody,
  utils$1.AST_NODE_TYPES.TSInterfaceDeclaration,
  utils$1.AST_NODE_TYPES.TSInterfaceHeritage,
  utils$1.AST_NODE_TYPES.TSImportEqualsDeclaration,
  utils$1.AST_NODE_TYPES.TSLiteralType,
  utils$1.AST_NODE_TYPES.TSMappedType,
  utils$1.AST_NODE_TYPES.TSMethodSignature,
  "TSMinusToken",
  utils$1.AST_NODE_TYPES.TSModuleBlock,
  utils$1.AST_NODE_TYPES.TSModuleDeclaration,
  utils$1.AST_NODE_TYPES.TSNonNullExpression,
  utils$1.AST_NODE_TYPES.TSParameterProperty,
  "TSPlusToken",
  utils$1.AST_NODE_TYPES.TSPropertySignature,
  utils$1.AST_NODE_TYPES.TSQualifiedName,
  "TSQuestionToken",
  utils$1.AST_NODE_TYPES.TSRestType,
  utils$1.AST_NODE_TYPES.TSThisType,
  utils$1.AST_NODE_TYPES.TSTupleType,
  utils$1.AST_NODE_TYPES.TSTypeAnnotation,
  utils$1.AST_NODE_TYPES.TSTypeLiteral,
  utils$1.AST_NODE_TYPES.TSTypeOperator,
  utils$1.AST_NODE_TYPES.TSTypeParameter,
  utils$1.AST_NODE_TYPES.TSTypeParameterDeclaration,
  utils$1.AST_NODE_TYPES.TSTypeParameterInstantiation,
  utils$1.AST_NODE_TYPES.TSTypeReference,
  utils$1.AST_NODE_TYPES.Decorator
  // These are took care by `indent-binary-ops` rule
  // AST_NODE_TYPES.TSIntersectionType,
  // AST_NODE_TYPES.TSUnionType,
]);
var indent = utils.createRule({
  name: "indent",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent indentation",
      // too opinionated to be recommended
      extendsBaseRule: true
    },
    fixable: "whitespace",
    hasSuggestions: baseRule$c.meta.hasSuggestions,
    schema: baseRule$c.meta.schema,
    messages: baseRule$c.meta.messages
  },
  defaultOptions: [
    // typescript docs and playground use 4 space indent
    4,
    {
      // typescript docs indent the case from the switch
      // https://www.typescriptlang.org/docs/handbook/release-notes/typescript-1-8.html#example-4
      SwitchCase: 1,
      flatTernaryExpressions: false,
      ignoredNodes: []
    }
  ],
  create(context, optionsWithDefaults) {
    const contextWithDefaults = Object.create(context, {
      options: {
        writable: false,
        configurable: false,
        value: optionsWithDefaults
      }
    });
    const rules = baseRule$c.create(contextWithDefaults);
    function TSPropertySignatureToProperty(node, type = utils$1.AST_NODE_TYPES.Property) {
      const base = {
        // indent doesn't actually use these
        key: null,
        value: null,
        // Property flags
        computed: false,
        method: false,
        kind: "init",
        // this will stop eslint from interrogating the type literal
        shorthand: true,
        // location data
        parent: node.parent,
        range: node.range,
        loc: node.loc
      };
      if (type === utils$1.AST_NODE_TYPES.Property) {
        return {
          type,
          ...base
        };
      }
      return {
        type,
        accessibility: void 0,
        declare: false,
        decorators: [],
        definite: false,
        optional: false,
        override: false,
        readonly: false,
        static: false,
        typeAnnotation: void 0,
        ...base
      };
    }
    return Object.assign({}, rules, {
      // overwrite the base rule here so we can use our KNOWN_NODES list instead
      "*:exit": function(node) {
        if (!KNOWN_NODES.has(node.type))
          rules["*:exit"](node);
      },
      VariableDeclaration(node) {
        if (node.declarations.length === 0)
          return;
        return rules.VariableDeclaration(node);
      },
      TSAsExpression(node) {
        return rules["BinaryExpression, LogicalExpression"]({
          type: utils$1.AST_NODE_TYPES.BinaryExpression,
          operator: "as",
          left: node.expression,
          // the first typeAnnotation includes the as token
          right: node.typeAnnotation,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSConditionalType(node) {
        return rules.ConditionalExpression({
          type: utils$1.AST_NODE_TYPES.ConditionalExpression,
          test: {
            parent: node,
            type: utils$1.AST_NODE_TYPES.BinaryExpression,
            operator: "extends",
            left: node.checkType,
            right: node.extendsType,
            // location data
            range: [node.checkType.range[0], node.extendsType.range[1]],
            loc: {
              start: node.checkType.loc.start,
              end: node.extendsType.loc.end
            }
          },
          consequent: node.trueType,
          alternate: node.falseType,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      "TSEnumDeclaration, TSTypeLiteral": function(node) {
        return rules["ObjectExpression, ObjectPattern"]({
          type: utils$1.AST_NODE_TYPES.ObjectExpression,
          properties: node.members.map(
            (member) => TSPropertySignatureToProperty(member)
          ),
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSImportEqualsDeclaration(node) {
        const { id, moduleReference } = node;
        return rules.VariableDeclaration({
          type: utils$1.AST_NODE_TYPES.VariableDeclaration,
          kind: "const",
          declarations: [
            {
              type: utils$1.AST_NODE_TYPES.VariableDeclarator,
              range: [id.range[0], moduleReference.range[1]],
              loc: {
                start: id.loc.start,
                end: moduleReference.loc.end
              },
              id,
              init: {
                type: utils$1.AST_NODE_TYPES.CallExpression,
                callee: {
                  type: utils$1.AST_NODE_TYPES.Identifier,
                  name: "require",
                  range: [
                    moduleReference.range[0],
                    moduleReference.range[0] + "require".length
                  ],
                  loc: {
                    start: moduleReference.loc.start,
                    end: {
                      line: moduleReference.loc.end.line,
                      column: moduleReference.loc.start.line + "require".length
                    }
                  }
                },
                arguments: "expression" in moduleReference ? [moduleReference.expression] : [],
                // location data
                range: moduleReference.range,
                loc: moduleReference.loc
              }
            }
          ],
          declare: false,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSIndexedAccessType(node) {
        return rules["MemberExpression, JSXMemberExpression, MetaProperty"]({
          type: utils$1.AST_NODE_TYPES.MemberExpression,
          object: node.objectType,
          property: node.indexType,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc,
          optional: false,
          computed: true
        });
      },
      TSInterfaceBody(node) {
        return rules["BlockStatement, ClassBody"]({
          type: utils$1.AST_NODE_TYPES.ClassBody,
          body: node.body.map(
            (p) => TSPropertySignatureToProperty(
              p,
              utils$1.AST_NODE_TYPES.PropertyDefinition
            )
          ),
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      "TSInterfaceDeclaration[extends.length > 0]": function(node) {
        return rules["ClassDeclaration[superClass], ClassExpression[superClass]"](
          {
            type: utils$1.AST_NODE_TYPES.ClassDeclaration,
            body: node.body,
            id: null,
            // TODO: This is invalid, there can be more than one extends in interface
            superClass: node.extends[0].expression,
            abstract: false,
            declare: false,
            decorators: [],
            implements: [],
            superTypeArguments: void 0,
            superTypeParameters: void 0,
            typeParameters: void 0,
            // location data
            parent: node.parent,
            range: node.range,
            loc: node.loc
          }
        );
      },
      TSMappedType(node) {
        const sourceCode = context.sourceCode;
        const squareBracketStart = sourceCode.getTokenBefore(
          node.typeParameter
        );
        return rules["ObjectExpression, ObjectPattern"]({
          type: utils$1.AST_NODE_TYPES.ObjectExpression,
          properties: [
            {
              parent: node,
              type: utils$1.AST_NODE_TYPES.Property,
              key: node.typeParameter,
              value: node.typeAnnotation,
              // location data
              range: [
                squareBracketStart.range[0],
                node.typeAnnotation ? node.typeAnnotation.range[1] : squareBracketStart.range[0]
              ],
              loc: {
                start: squareBracketStart.loc.start,
                end: node.typeAnnotation ? node.typeAnnotation.loc.end : squareBracketStart.loc.end
              },
              kind: "init",
              computed: false,
              method: false,
              optional: false,
              shorthand: false
            }
          ],
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSModuleBlock(node) {
        return rules["BlockStatement, ClassBody"]({
          type: utils$1.AST_NODE_TYPES.BlockStatement,
          body: node.body,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSQualifiedName(node) {
        return rules["MemberExpression, JSXMemberExpression, MetaProperty"]({
          type: utils$1.AST_NODE_TYPES.MemberExpression,
          object: node.left,
          property: node.right,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc,
          optional: false,
          computed: false
        });
      },
      TSTupleType(node) {
        return rules["ArrayExpression, ArrayPattern"]({
          type: utils$1.AST_NODE_TYPES.ArrayExpression,
          elements: node.elementTypes,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSTypeParameterDeclaration(node) {
        if (!node.params.length)
          return;
        const [name, ...attributes] = node.params;
        return rules.JSXOpeningElement({
          type: utils$1.AST_NODE_TYPES.JSXOpeningElement,
          selfClosing: false,
          name,
          attributes,
          typeArguments: void 0,
          typeParameters: void 0,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      },
      TSTypeParameterInstantiation(node) {
        if (!node.params.length)
          return;
        const [name, ...attributes] = node.params;
        return rules.JSXOpeningElement({
          type: utils$1.AST_NODE_TYPES.JSXOpeningElement,
          selfClosing: false,
          name,
          attributes,
          typeArguments: void 0,
          typeParameters: void 0,
          // location data
          parent: node.parent,
          range: node.range,
          loc: node.loc
        });
      }
    });
  }
});

const baseRule$b = utils.getESLintCoreRule("key-spacing");
const baseSchema$1 = Array.isArray(baseRule$b.meta.schema) ? baseRule$b.meta.schema[0] : baseRule$b.meta.schema;
function at(arr, position) {
  if (position < 0)
    return arr[arr.length + position];
  return arr[position];
}
var keySpacing = utils.createRule({
  name: "key-spacing",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent spacing between property names and type annotations in types and interfaces",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    hasSuggestions: baseRule$b.meta.hasSuggestions,
    schema: [baseSchema$1],
    messages: baseRule$b.meta.messages
  },
  defaultOptions: [{}],
  create(context, [_options]) {
    const options = _options || {};
    const sourceCode = context.sourceCode;
    const baseRules = baseRule$b.create(context);
    function adjustedColumn(position) {
      const line = position.line - 1;
      return utils.getStringLength(
        at(sourceCode.lines, line).slice(0, position.column)
      );
    }
    function getLastTokenBeforeColon(node) {
      const colonToken = sourceCode.getTokenAfter(node, astUtils.isColonToken);
      return sourceCode.getTokenBefore(colonToken);
    }
    function isKeyTypeNode(node) {
      return (node.type === utils$1.AST_NODE_TYPES.TSPropertySignature || node.type === utils$1.AST_NODE_TYPES.TSIndexSignature || node.type === utils$1.AST_NODE_TYPES.PropertyDefinition) && !!node.typeAnnotation;
    }
    function isApplicable(node) {
      return isKeyTypeNode(node) && node.typeAnnotation.loc.start.line === node.loc.end.line;
    }
    function getKeyText(node) {
      if (node.type !== utils$1.AST_NODE_TYPES.TSIndexSignature)
        return sourceCode.getText(node.key);
      const code = sourceCode.getText(node);
      return code.slice(
        0,
        sourceCode.getTokenAfter(
          at(node.parameters, -1),
          astUtils.isClosingBracketToken
        ).range[1] - node.range[0]
      );
    }
    function getKeyLocEnd(node) {
      return getLastTokenBeforeColon(
        node.type !== utils$1.AST_NODE_TYPES.TSIndexSignature ? node.key : at(node.parameters, -1)
      ).loc.end;
    }
    function checkBeforeColon(node, expectedWhitespaceBeforeColon, mode) {
      const { typeAnnotation } = node;
      const colon = typeAnnotation.loc.start.column;
      const keyEnd = getKeyLocEnd(node);
      const difference = colon - keyEnd.column - expectedWhitespaceBeforeColon;
      if (mode === "strict" ? difference : difference < 0) {
        context.report({
          node,
          messageId: difference > 0 ? "extraKey" : "missingKey",
          fix: (fixer) => {
            if (difference > 0) {
              return fixer.removeRange([
                typeAnnotation.range[0] - difference,
                typeAnnotation.range[0]
              ]);
            }
            return fixer.insertTextBefore(
              typeAnnotation,
              " ".repeat(-difference)
            );
          },
          data: {
            computed: "",
            key: getKeyText(node)
          }
        });
      }
    }
    function checkAfterColon(node, expectedWhitespaceAfterColon, mode) {
      const { typeAnnotation } = node;
      const colonToken = sourceCode.getFirstToken(typeAnnotation);
      const typeStart = sourceCode.getTokenAfter(colonToken, {
        includeComments: true
      }).loc.start.column;
      const difference = typeStart - colonToken.loc.start.column - 1 - expectedWhitespaceAfterColon;
      if (mode === "strict" ? difference : difference < 0) {
        context.report({
          node,
          messageId: difference > 0 ? "extraValue" : "missingValue",
          fix: (fixer) => {
            if (difference > 0) {
              return fixer.removeRange([
                colonToken.range[1],
                colonToken.range[1] + difference
              ]);
            }
            return fixer.insertTextAfter(colonToken, " ".repeat(-difference));
          },
          data: {
            computed: "",
            key: getKeyText(node)
          }
        });
      }
    }
    function continuesAlignGroup(lastMember, candidate) {
      const groupEndLine = lastMember.loc.start.line;
      const candidateValueStartLine = (isKeyTypeNode(candidate) ? candidate.typeAnnotation : candidate).loc.start.line;
      if (candidateValueStartLine === groupEndLine)
        return false;
      if (candidateValueStartLine - groupEndLine === 1)
        return true;
      const leadingComments = sourceCode.getCommentsBefore(candidate);
      if (leadingComments.length && leadingComments[0].loc.start.line - groupEndLine <= 1 && candidateValueStartLine - at(leadingComments, -1).loc.end.line <= 1) {
        for (let i = 1; i < leadingComments.length; i++) {
          if (leadingComments[i].loc.start.line - leadingComments[i - 1].loc.end.line > 1)
            return false;
        }
        return true;
      }
      return false;
    }
    function checkAlignGroup(group) {
      let alignColumn = 0;
      const align = (typeof options.align === "object" ? options.align.on : typeof options.multiLine?.align === "object" ? options.multiLine.align.on : options.multiLine?.align ?? options.align) ?? "colon";
      const beforeColon = (typeof options.align === "object" ? options.align.beforeColon : options.multiLine ? typeof options.multiLine.align === "object" ? options.multiLine.align.beforeColon : options.multiLine.beforeColon : options.beforeColon) ?? false;
      const expectedWhitespaceBeforeColon = beforeColon ? 1 : 0;
      const afterColon = (typeof options.align === "object" ? options.align.afterColon : options.multiLine ? typeof options.multiLine.align === "object" ? options.multiLine.align.afterColon : options.multiLine.afterColon : options.afterColon) ?? true;
      const expectedWhitespaceAfterColon = afterColon ? 1 : 0;
      const mode = (typeof options.align === "object" ? options.align.mode : options.multiLine ? typeof options.multiLine.align === "object" ? options.multiLine.align.mode ?? options.multiLine.mode : options.multiLine.mode : options.mode) ?? "strict";
      for (const node of group) {
        if (isKeyTypeNode(node)) {
          const keyEnd = adjustedColumn(getKeyLocEnd(node));
          alignColumn = Math.max(
            alignColumn,
            align === "colon" ? keyEnd + expectedWhitespaceBeforeColon : keyEnd + ":".length + expectedWhitespaceAfterColon + expectedWhitespaceBeforeColon
          );
        }
      }
      for (const node of group) {
        if (!isApplicable(node))
          continue;
        const { typeAnnotation } = node;
        const toCheck = align === "colon" ? typeAnnotation : typeAnnotation.typeAnnotation;
        const difference = adjustedColumn(toCheck.loc.start) - alignColumn;
        if (difference) {
          context.report({
            node,
            messageId: difference > 0 ? align === "colon" ? "extraKey" : "extraValue" : align === "colon" ? "missingKey" : "missingValue",
            fix: (fixer) => {
              if (difference > 0) {
                return fixer.removeRange([
                  toCheck.range[0] - difference,
                  toCheck.range[0]
                ]);
              }
              return fixer.insertTextBefore(toCheck, " ".repeat(-difference));
            },
            data: {
              computed: "",
              key: getKeyText(node)
            }
          });
        }
        if (align === "colon")
          checkAfterColon(node, expectedWhitespaceAfterColon, mode);
        else
          checkBeforeColon(node, expectedWhitespaceBeforeColon, mode);
      }
    }
    function checkIndividualNode(node, { singleLine }) {
      const beforeColon = (singleLine ? options.singleLine ? options.singleLine.beforeColon : options.beforeColon : options.multiLine ? options.multiLine.beforeColon : options.beforeColon) ?? false;
      const expectedWhitespaceBeforeColon = beforeColon ? 1 : 0;
      const afterColon = (singleLine ? options.singleLine ? options.singleLine.afterColon : options.afterColon : options.multiLine ? options.multiLine.afterColon : options.afterColon) ?? true;
      const expectedWhitespaceAfterColon = afterColon ? 1 : 0;
      const mode = (singleLine ? options.singleLine ? options.singleLine.mode : options.mode : options.multiLine ? options.multiLine.mode : options.mode) ?? "strict";
      if (isApplicable(node)) {
        checkBeforeColon(node, expectedWhitespaceBeforeColon, mode);
        checkAfterColon(node, expectedWhitespaceAfterColon, mode);
      }
    }
    function validateBody(body) {
      const isSingleLine = body.loc.start.line === body.loc.end.line;
      const members = body.type === utils$1.AST_NODE_TYPES.TSTypeLiteral ? body.members : body.body;
      let alignGroups = [];
      let unalignedElements = [];
      if (options.align || options.multiLine?.align) {
        let currentAlignGroup = [];
        alignGroups.push(currentAlignGroup);
        let prevNode;
        for (const node of members) {
          let prevAlignedNode = at(currentAlignGroup, -1);
          if (prevAlignedNode !== prevNode)
            prevAlignedNode = void 0;
          if (prevAlignedNode && continuesAlignGroup(prevAlignedNode, node)) {
            currentAlignGroup.push(node);
          } else if (prevNode?.loc.start.line === node.loc.start.line) {
            if (prevAlignedNode) {
              unalignedElements.push(prevAlignedNode);
              currentAlignGroup.pop();
            }
            unalignedElements.push(node);
          } else {
            currentAlignGroup = [node];
            alignGroups.push(currentAlignGroup);
          }
          prevNode = node;
        }
        unalignedElements = unalignedElements.concat(
          ...alignGroups.filter((group) => group.length === 1)
        );
        alignGroups = alignGroups.filter((group) => group.length >= 2);
      } else {
        unalignedElements = members;
      }
      for (const group of alignGroups)
        checkAlignGroup(group);
      for (const node of unalignedElements)
        checkIndividualNode(node, { singleLine: isSingleLine });
    }
    return {
      ...baseRules,
      TSTypeLiteral: validateBody,
      TSInterfaceBody: validateBody,
      ClassBody: validateBody
    };
  }
});

const baseRule$a = utils.getESLintCoreRule("keyword-spacing");
const baseSchema = Array.isArray(baseRule$a.meta.schema) ? baseRule$a.meta.schema[0] : baseRule$a.meta.schema;
const schema$1 = utils.deepMerge(
  baseSchema,
  {
    properties: {
      overrides: {
        properties: {
          type: baseSchema.properties.overrides.properties.import
        }
      }
    }
  }
);
var keywordSpacing = utils.createRule({
  name: "keyword-spacing",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent spacing before and after keywords",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    hasSuggestions: baseRule$a.meta.hasSuggestions,
    schema: [schema$1],
    messages: baseRule$a.meta.messages
  },
  defaultOptions: [{}],
  create(context, [options]) {
    const { after, overrides } = options ?? {};
    const sourceCode = context.sourceCode;
    const baseRules = baseRule$a.create(context);
    return {
      ...baseRules,
      TSAsExpression(node) {
        const asToken = utils.nullThrows(
          sourceCode.getTokenAfter(
            node.expression,
            (token) => token.value === "as"
          ),
          utils.NullThrowsReasons.MissingToken("as", node.type)
        );
        const oldTokenType = asToken.type;
        asToken.type = utils$1.AST_TOKEN_TYPES.Keyword;
        baseRules.DebuggerStatement(asToken);
        asToken.type = oldTokenType;
      },
      "ImportDeclaration[importKind=type]": function(node) {
        const { type: typeOptionOverride = {} } = overrides ?? {};
        const typeToken = sourceCode.getFirstToken(node, { skip: 1 });
        const punctuatorToken = sourceCode.getTokenAfter(typeToken);
        if (node.specifiers?.[0]?.type === utils$1.AST_NODE_TYPES.ImportDefaultSpecifier)
          return;
        const spacesBetweenTypeAndPunctuator = punctuatorToken.range[0] - typeToken.range[1];
        if ((typeOptionOverride.after ?? after) === true && spacesBetweenTypeAndPunctuator === 0) {
          context.report({
            loc: typeToken.loc,
            messageId: "expectedAfter",
            data: { value: "type" },
            fix(fixer) {
              return fixer.insertTextAfter(typeToken, " ");
            }
          });
        }
        if ((typeOptionOverride.after ?? after) === false && spacesBetweenTypeAndPunctuator > 0) {
          context.report({
            loc: typeToken.loc,
            messageId: "unexpectedAfter",
            data: { value: "type" },
            fix(fixer) {
              return fixer.removeRange([
                typeToken.range[1],
                typeToken.range[1] + spacesBetweenTypeAndPunctuator
              ]);
            }
          });
        }
      }
    };
  }
});

const baseRule$9 = utils.getESLintCoreRule("lines-around-comment");
const COMMENTS_IGNORE_PATTERN = /^\s*(?:eslint|jshint\s+|jslint\s+|istanbul\s+|globals?\s+|exported\s+|jscs)/u;
function getEmptyLineNums(lines) {
  const emptyLines = lines.map((line, i) => ({
    code: line.trim(),
    num: i + 1
  })).filter((line) => !line.code).map((line) => line.num);
  return emptyLines;
}
function getCommentLineNums(comments) {
  const lines = [];
  comments.forEach((token) => {
    const start = token.loc.start.line;
    const end = token.loc.end.line;
    lines.push(start, end);
  });
  return lines;
}
var linesAroundComment = utils.createRule({
  name: "lines-around-comment",
  meta: {
    type: "layout",
    docs: {
      description: "Require empty lines around comments",
      extendsBaseRule: true
    },
    schema: [
      {
        type: "object",
        properties: {
          beforeBlockComment: {
            type: "boolean",
            default: true
          },
          afterBlockComment: {
            type: "boolean",
            default: false
          },
          beforeLineComment: {
            type: "boolean",
            default: false
          },
          afterLineComment: {
            type: "boolean",
            default: false
          },
          allowBlockStart: {
            type: "boolean",
            default: false
          },
          allowBlockEnd: {
            type: "boolean",
            default: false
          },
          allowClassStart: {
            type: "boolean"
          },
          allowClassEnd: {
            type: "boolean"
          },
          allowObjectStart: {
            type: "boolean"
          },
          allowObjectEnd: {
            type: "boolean"
          },
          allowArrayStart: {
            type: "boolean"
          },
          allowArrayEnd: {
            type: "boolean"
          },
          allowInterfaceStart: {
            type: "boolean"
          },
          allowInterfaceEnd: {
            type: "boolean"
          },
          allowTypeStart: {
            type: "boolean"
          },
          allowTypeEnd: {
            type: "boolean"
          },
          allowEnumStart: {
            type: "boolean"
          },
          allowEnumEnd: {
            type: "boolean"
          },
          allowModuleStart: {
            type: "boolean"
          },
          allowModuleEnd: {
            type: "boolean"
          },
          ignorePattern: {
            type: "string"
          },
          applyDefaultIgnorePatterns: {
            type: "boolean"
          },
          afterHashbangComment: {
            type: "boolean"
          }
        },
        additionalProperties: false
      }
    ],
    fixable: baseRule$9.meta.fixable,
    hasSuggestions: baseRule$9.meta.hasSuggestions,
    messages: baseRule$9.meta.messages
  },
  defaultOptions: [
    {
      beforeBlockComment: true
    }
  ],
  create(context, [_options]) {
    const options = _options;
    const defaultIgnoreRegExp = COMMENTS_IGNORE_PATTERN;
    const customIgnoreRegExp = new RegExp(options.ignorePattern ?? "", "u");
    const sourceCode = context.sourceCode;
    const comments = sourceCode.getAllComments();
    const lines = sourceCode.lines;
    const commentLines = getCommentLineNums(comments);
    const emptyLines = getEmptyLineNums(lines);
    const commentAndEmptyLines = new Set(commentLines.concat(emptyLines));
    function codeAroundComment(token) {
      let currentToken = token;
      do {
        currentToken = sourceCode.getTokenBefore(currentToken, {
          includeComments: true
        });
      } while (currentToken && astUtils.isCommentToken(currentToken));
      if (currentToken && astUtils.isTokenOnSameLine(currentToken, token))
        return true;
      currentToken = token;
      do {
        currentToken = sourceCode.getTokenAfter(currentToken, {
          includeComments: true
        });
      } while (currentToken && astUtils.isCommentToken(currentToken));
      if (currentToken && astUtils.isTokenOnSameLine(token, currentToken))
        return true;
      return false;
    }
    function isParentNodeType(parent, nodeType) {
      return parent.type === nodeType;
    }
    function getParentNodeOfToken(token) {
      const node = sourceCode.getNodeByRangeIndex(token.range[0]);
      return node;
    }
    function isCommentAtParentStart(token, nodeType) {
      const parent = getParentNodeOfToken(token);
      if (parent && isParentNodeType(parent, nodeType)) {
        const parentStartNodeOrToken = parent;
        return token.loc.start.line - parentStartNodeOrToken.loc.start.line === 1;
      }
      return false;
    }
    function isCommentAtParentEnd(token, nodeType) {
      const parent = getParentNodeOfToken(token);
      return !!parent && isParentNodeType(parent, nodeType) && parent.loc.end.line - token.loc.end.line === 1;
    }
    function isCommentAtInterfaceStart(token) {
      return isCommentAtParentStart(token, utils$1.AST_NODE_TYPES.TSInterfaceBody);
    }
    function isCommentAtInterfaceEnd(token) {
      return isCommentAtParentEnd(token, utils$1.AST_NODE_TYPES.TSInterfaceBody);
    }
    function isCommentAtTypeStart(token) {
      return isCommentAtParentStart(token, utils$1.AST_NODE_TYPES.TSTypeLiteral);
    }
    function isCommentAtTypeEnd(token) {
      return isCommentAtParentEnd(token, utils$1.AST_NODE_TYPES.TSTypeLiteral);
    }
    function isCommentAtEnumStart(token) {
      return isCommentAtParentStart(token, utils$1.AST_NODE_TYPES.TSEnumDeclaration);
    }
    function isCommentAtEnumEnd(token) {
      return isCommentAtParentEnd(token, utils$1.AST_NODE_TYPES.TSEnumDeclaration);
    }
    function isCommentAtModuleStart(token) {
      return isCommentAtParentStart(token, utils$1.AST_NODE_TYPES.TSModuleBlock);
    }
    function isCommentAtModuleEnd(token) {
      return isCommentAtParentEnd(token, utils$1.AST_NODE_TYPES.TSModuleBlock);
    }
    function isCommentNearTSConstruct(token) {
      return isCommentAtInterfaceStart(token) || isCommentAtInterfaceEnd(token) || isCommentAtTypeStart(token) || isCommentAtTypeEnd(token) || isCommentAtEnumStart(token) || isCommentAtEnumEnd(token) || isCommentAtModuleStart(token) || isCommentAtModuleEnd(token);
    }
    function checkForEmptyLine(token, { before, after }) {
      if (!isCommentNearTSConstruct(token))
        return;
      if (options.applyDefaultIgnorePatterns !== false && defaultIgnoreRegExp.test(token.value))
        return;
      if (options.ignorePattern && customIgnoreRegExp.test(token.value))
        return;
      const prevLineNum = token.loc.start.line - 1;
      const nextLineNum = token.loc.end.line + 1;
      if (codeAroundComment(token))
        return;
      const interfaceStartAllowed = Boolean(options.allowInterfaceStart) && isCommentAtInterfaceStart(token);
      const interfaceEndAllowed = Boolean(options.allowInterfaceEnd) && isCommentAtInterfaceEnd(token);
      const typeStartAllowed = Boolean(options.allowTypeStart) && isCommentAtTypeStart(token);
      const typeEndAllowed = Boolean(options.allowTypeEnd) && isCommentAtTypeEnd(token);
      const enumStartAllowed = Boolean(options.allowEnumStart) && isCommentAtEnumStart(token);
      const enumEndAllowed = Boolean(options.allowEnumEnd) && isCommentAtEnumEnd(token);
      const moduleStartAllowed = Boolean(options.allowModuleStart) && isCommentAtModuleStart(token);
      const moduleEndAllowed = Boolean(options.allowModuleEnd) && isCommentAtModuleEnd(token);
      const exceptionStartAllowed = interfaceStartAllowed || typeStartAllowed || enumStartAllowed || moduleStartAllowed;
      const exceptionEndAllowed = interfaceEndAllowed || typeEndAllowed || enumEndAllowed || moduleEndAllowed;
      const previousTokenOrComment = sourceCode.getTokenBefore(token, {
        includeComments: true
      });
      const nextTokenOrComment = sourceCode.getTokenAfter(token, {
        includeComments: true
      });
      if (!exceptionStartAllowed && before && !commentAndEmptyLines.has(prevLineNum) && !(astUtils.isCommentToken(previousTokenOrComment) && astUtils.isTokenOnSameLine(previousTokenOrComment, token))) {
        const lineStart = token.range[0] - token.loc.start.column;
        const range = [lineStart, lineStart];
        context.report({
          node: token,
          messageId: "before",
          fix(fixer) {
            return fixer.insertTextBeforeRange(range, "\n");
          }
        });
      }
      if (!exceptionEndAllowed && after && !commentAndEmptyLines.has(nextLineNum) && !(astUtils.isCommentToken(nextTokenOrComment) && astUtils.isTokenOnSameLine(token, nextTokenOrComment))) {
        context.report({
          node: token,
          messageId: "after",
          fix(fixer) {
            return fixer.insertTextAfter(token, "\n");
          }
        });
      }
    }
    const customReport = (descriptor) => {
      if ("node" in descriptor) {
        if (descriptor.node.type === utils$1.AST_TOKEN_TYPES.Line || descriptor.node.type === utils$1.AST_TOKEN_TYPES.Block) {
          if (isCommentNearTSConstruct(descriptor.node))
            return;
        }
      }
      return context.report(descriptor);
    };
    const customContext = { report: customReport };
    const proxiedContext = new Proxy(
      customContext,
      {
        get(target, path, receiver) {
          if (path !== "report")
            return Reflect.get(context, path, receiver);
          return Reflect.get(target, path, receiver);
        }
      }
    );
    const rules = baseRule$9.create(proxiedContext);
    return {
      Program(node) {
        rules.Program(node);
        comments.forEach((token) => {
          if (token.type === utils$1.AST_TOKEN_TYPES.Line) {
            if (options.beforeLineComment || options.afterLineComment) {
              checkForEmptyLine(token, {
                after: options.afterLineComment,
                before: options.beforeLineComment
              });
            }
          } else if (token.type === utils$1.AST_TOKEN_TYPES.Block) {
            if (options.beforeBlockComment || options.afterBlockComment) {
              checkForEmptyLine(token, {
                after: options.afterBlockComment,
                before: options.beforeBlockComment
              });
            }
          }
        });
      }
    };
  }
});

const baseRule$8 = utils.getESLintCoreRule("lines-between-class-members");
const schema = Object.values(
  utils.deepMerge(
    { ...baseRule$8.meta.schema },
    {
      1: {
        properties: {
          exceptAfterOverload: {
            type: "boolean",
            default: true
          }
        }
      }
    }
  )
);
var linesBetweenClassMembers = utils.createRule({
  name: "lines-between-class-members",
  meta: {
    type: "layout",
    docs: {
      description: "Require or disallow an empty line between class members",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    hasSuggestions: baseRule$8.meta.hasSuggestions,
    schema,
    messages: baseRule$8.meta.messages
  },
  defaultOptions: [
    "always",
    {
      exceptAfterOverload: true,
      exceptAfterSingleLine: false
    }
  ],
  create(context, [firstOption, secondOption]) {
    const rules = baseRule$8.create(context);
    const exceptAfterOverload = secondOption?.exceptAfterOverload && firstOption === "always";
    function isOverload(node) {
      return (node.type === utils$1.AST_NODE_TYPES.TSAbstractMethodDefinition || node.type === utils$1.AST_NODE_TYPES.MethodDefinition) && node.value.type === utils$1.AST_NODE_TYPES.TSEmptyBodyFunctionExpression;
    }
    return {
      ClassBody(node) {
        const body = exceptAfterOverload ? node.body.filter((node2) => !isOverload(node2)) : node.body;
        rules.ClassBody({ ...node, body });
      }
    };
  }
});

function isLastTokenEndOfLine(token, line) {
  const positionInLine = token.loc.start.column;
  return positionInLine === line.length - 1;
}
function isCommentsEndOfLine(token, comments, line) {
  if (!comments)
    return false;
  if (comments.loc.end.line > token.loc.end.line)
    return true;
  const positionInLine = comments.loc.end.column;
  return positionInLine === line.length;
}
function makeFixFunction({
  optsNone,
  optsSemi,
  lastToken,
  commentsAfterLastToken,
  missingDelimiter,
  lastTokenLine,
  isSingleLine
}) {
  if (optsNone && !isLastTokenEndOfLine(lastToken, lastTokenLine) && !isCommentsEndOfLine(lastToken, commentsAfterLastToken, lastTokenLine) && !isSingleLine)
    return null;
  return (fixer) => {
    if (optsNone) {
      return fixer.remove(lastToken);
    }
    const token = optsSemi ? ";" : ",";
    if (missingDelimiter) {
      return fixer.insertTextAfter(lastToken, token);
    }
    return fixer.replaceText(lastToken, token);
  };
}
const BASE_SCHEMA = {
  type: "object",
  properties: {
    multiline: {
      type: "object",
      properties: {
        delimiter: { $ref: "#/items/0/$defs/multiLineOption" },
        requireLast: { type: "boolean" }
      },
      additionalProperties: false
    },
    singleline: {
      type: "object",
      properties: {
        delimiter: { $ref: "#/items/0/$defs/singleLineOption" },
        requireLast: { type: "boolean" }
      },
      additionalProperties: false
    }
  },
  additionalProperties: false
};
var memberDelimiterStyle = utils.createRule({
  name: "member-delimiter-style",
  meta: {
    type: "layout",
    docs: {
      description: "Require a specific member delimiter style for interfaces and type literals"
    },
    fixable: "whitespace",
    messages: {
      unexpectedComma: "Unexpected separator (,).",
      unexpectedSemi: "Unexpected separator (;).",
      expectedComma: "Expected a comma.",
      expectedSemi: "Expected a semicolon."
    },
    schema: [
      {
        $defs: {
          multiLineOption: {
            type: "string",
            enum: ["none", "semi", "comma"]
          },
          // note can't have "none" for single line delimiter as it's invalid syntax
          singleLineOption: {
            type: "string",
            enum: ["semi", "comma"]
          },
          // note - need to define this last as it references the enums
          delimiterConfig: BASE_SCHEMA
        },
        type: "object",
        properties: {
          ...BASE_SCHEMA.properties,
          overrides: {
            type: "object",
            properties: {
              interface: {
                $ref: "#/items/0/$defs/delimiterConfig"
              },
              typeLiteral: {
                $ref: "#/items/0/$defs/delimiterConfig"
              }
            },
            additionalProperties: false
          },
          multilineDetection: {
            type: "string",
            enum: ["brackets", "last-member"]
          }
        },
        additionalProperties: false
      }
    ]
  },
  defaultOptions: [
    {
      multiline: {
        delimiter: "semi",
        requireLast: true
      },
      singleline: {
        delimiter: "semi",
        requireLast: false
      },
      multilineDetection: "brackets"
    }
  ],
  create(context, [options]) {
    const sourceCode = context.sourceCode;
    const baseOptions = options;
    const overrides = baseOptions.overrides ?? {};
    const interfaceOptions = utils.deepMerge(
      baseOptions,
      overrides.interface
    );
    const typeLiteralOptions = utils.deepMerge(
      baseOptions,
      overrides.typeLiteral
    );
    function checkLastToken(member, opts, isLast) {
      function getOption(type) {
        if (isLast && !opts.requireLast) {
          return type === "none";
        }
        return opts.delimiter === type;
      }
      let messageId = null;
      let missingDelimiter = false;
      const lastToken = sourceCode.getLastToken(member, {
        includeComments: false
      });
      if (!lastToken)
        return;
      const commentsAfterLastToken = sourceCode.getCommentsAfter(lastToken).pop();
      const sourceCodeLines = sourceCode.getLines();
      const lastTokenLine = sourceCodeLines[lastToken?.loc.start.line - 1];
      const optsSemi = getOption("semi");
      const optsComma = getOption("comma");
      const optsNone = getOption("none");
      if (lastToken.value === ";") {
        if (optsComma) {
          messageId = "expectedComma";
        } else if (optsNone) {
          missingDelimiter = true;
          messageId = "unexpectedSemi";
        }
      } else if (lastToken.value === ",") {
        if (optsSemi) {
          messageId = "expectedSemi";
        } else if (optsNone) {
          missingDelimiter = true;
          messageId = "unexpectedComma";
        }
      } else {
        if (optsSemi) {
          missingDelimiter = true;
          messageId = "expectedSemi";
        } else if (optsComma) {
          missingDelimiter = true;
          messageId = "expectedComma";
        }
      }
      if (messageId) {
        context.report({
          node: lastToken,
          loc: {
            start: {
              line: lastToken.loc.end.line,
              column: lastToken.loc.end.column
            },
            end: {
              line: lastToken.loc.end.line,
              column: lastToken.loc.end.column
            }
          },
          messageId,
          fix: makeFixFunction({
            optsNone,
            optsSemi,
            lastToken,
            commentsAfterLastToken,
            missingDelimiter,
            lastTokenLine,
            isSingleLine: opts.type === "single-line"
          })
        });
      }
    }
    function checkMemberSeparatorStyle(node) {
      const members = node.type === utils$1.AST_NODE_TYPES.TSInterfaceBody ? node.body : node.members;
      let isSingleLine = node.loc.start.line === node.loc.end.line;
      if (options.multilineDetection === "last-member" && !isSingleLine && members.length > 0) {
        const lastMember = members[members.length - 1];
        if (lastMember.loc.end.line === node.loc.end.line)
          isSingleLine = true;
      }
      const typeOpts = node.type === utils$1.AST_NODE_TYPES.TSInterfaceBody ? interfaceOptions : typeLiteralOptions;
      const opts = isSingleLine ? { ...typeOpts.singleline, type: "single-line" } : { ...typeOpts.multiline, type: "multi-line" };
      members.forEach((member, index) => {
        checkLastToken(member, opts ?? {}, index === members.length - 1);
      });
    }
    return {
      TSInterfaceBody: checkMemberSeparatorStyle,
      TSTypeLiteral: checkMemberSeparatorStyle
    };
  }
});

const baseRule$7 = utils.getESLintCoreRule("no-extra-parens");
var noExtraParens = utils.createRule({
  name: "no-extra-parens",
  meta: {
    type: "layout",
    docs: {
      description: "Disallow unnecessary parentheses",
      extendsBaseRule: true
    },
    fixable: "code",
    hasSuggestions: baseRule$7.meta.hasSuggestions,
    schema: baseRule$7.meta.schema,
    messages: baseRule$7.meta.messages
  },
  defaultOptions: ["all"],
  create(context) {
    const sourceCode = context.sourceCode;
    const rules = baseRule$7.create(context);
    function binaryExp(node) {
      const rule = rules.BinaryExpression;
      const isLeftTypeAssertion = astUtils.isTypeAssertion(node.left);
      const isRightTypeAssertion = astUtils.isTypeAssertion(node.right);
      if (isLeftTypeAssertion && isRightTypeAssertion)
        return;
      if (isLeftTypeAssertion) {
        return rule({
          ...node,
          left: {
            ...node.left,
            type: utils$1.AST_NODE_TYPES.SequenceExpression
          }
        });
      }
      if (isRightTypeAssertion) {
        return rule({
          ...node,
          right: {
            ...node.right,
            type: utils$1.AST_NODE_TYPES.SequenceExpression
          }
        });
      }
      return rule(node);
    }
    function callExp(node) {
      const rule = rules.CallExpression;
      if (astUtils.isTypeAssertion(node.callee)) {
        return rule({
          ...node,
          callee: {
            ...node.callee,
            type: utils$1.AST_NODE_TYPES.SequenceExpression
          }
        });
      }
      if (node.arguments.length === 1 && sourceCode.getTokenAfter(node.callee, astUtils.isOpeningParenToken) !== sourceCode.getTokenBefore(node.arguments[0], astUtils.isOpeningParenToken)) {
        return rule({
          ...node,
          arguments: [
            {
              ...node.arguments[0],
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          ]
        });
      }
      return rule(node);
    }
    function unaryUpdateExpression(node) {
      const rule = rules.UnaryExpression;
      if (astUtils.isTypeAssertion(node.argument)) {
        return rule({
          ...node,
          argument: {
            ...node.argument,
            type: utils$1.AST_NODE_TYPES.SequenceExpression
          }
        });
      }
      return rule(node);
    }
    const overrides = {
      ArrayExpression(node) {
        node.elements.forEach((element, index) => {
          if (astUtils.isTypeAssertion(element)) {
            return rules.ArrayExpression({
              ...node,
              elements: [
                ...node.elements.slice(0, index),
                {
                  ...element,
                  type: utils$1.AST_NODE_TYPES.FunctionExpression
                },
                ...node.elements.slice(index + 1)
              ]
            });
          }
        });
      },
      ArrowFunctionExpression(node) {
        if (!astUtils.isTypeAssertion(node.body))
          return rules.ArrowFunctionExpression(node);
      },
      // AssignmentExpression
      AwaitExpression(node) {
        if (astUtils.isTypeAssertion(node.argument)) {
          return rules.AwaitExpression({
            ...node,
            argument: {
              ...node.argument,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        return rules.AwaitExpression(node);
      },
      "BinaryExpression": binaryExp,
      "CallExpression": callExp,
      ClassDeclaration(node) {
        if (node.superClass?.type === utils$1.AST_NODE_TYPES.TSAsExpression) {
          return rules.ClassDeclaration({
            ...node,
            superClass: {
              ...node.superClass,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        return rules.ClassDeclaration(node);
      },
      ClassExpression(node) {
        if (node.superClass?.type === utils$1.AST_NODE_TYPES.TSAsExpression) {
          return rules.ClassExpression({
            ...node,
            superClass: {
              ...node.superClass,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        return rules.ClassExpression(node);
      },
      ConditionalExpression(node) {
        if (astUtils.isTypeAssertion(node.test)) {
          return rules.ConditionalExpression({
            ...node,
            test: {
              ...node.test,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        if (astUtils.isTypeAssertion(node.consequent)) {
          return rules.ConditionalExpression({
            ...node,
            consequent: {
              ...node.consequent,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        if (astUtils.isTypeAssertion(node.alternate)) {
          return rules.ConditionalExpression({
            ...node,
            alternate: {
              ...node.alternate,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        return rules.ConditionalExpression(node);
      },
      // DoWhileStatement
      // ForIn and ForOf are guarded by eslint version
      ForStatement(node) {
        if (node.init && astUtils.isTypeAssertion(node.init)) {
          return rules.ForStatement({
            ...node,
            init: null
          });
        }
        if (node.test && astUtils.isTypeAssertion(node.test)) {
          return rules.ForStatement({
            ...node,
            test: null
          });
        }
        if (node.update && astUtils.isTypeAssertion(node.update)) {
          return rules.ForStatement({
            ...node,
            update: null
          });
        }
        return rules.ForStatement(node);
      },
      "ForStatement > *.init:exit": function(node) {
        if (!astUtils.isTypeAssertion(node))
          return rules["ForStatement > *.init:exit"](node);
      },
      // IfStatement
      "LogicalExpression": binaryExp,
      MemberExpression(node) {
        if (astUtils.isTypeAssertion(node.object)) {
          return rules.MemberExpression({
            ...node,
            object: {
              ...node.object,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        if (astUtils.isTypeAssertion(node.property)) {
          return rules.MemberExpression({
            ...node,
            property: {
              ...node.property,
              type: utils$1.AST_NODE_TYPES.FunctionExpression
            }
          });
        }
        return rules.MemberExpression(node);
      },
      "NewExpression": callExp,
      // ObjectExpression
      // ReturnStatement
      // SequenceExpression
      SpreadElement(node) {
        if (!astUtils.isTypeAssertion(node.argument))
          return rules.SpreadElement(node);
      },
      SwitchCase(node) {
        if (node.test && !astUtils.isTypeAssertion(node.test))
          return rules.SwitchCase(node);
      },
      // SwitchStatement
      ThrowStatement(node) {
        if (node.argument && !astUtils.isTypeAssertion(node.argument))
          return rules.ThrowStatement(node);
      },
      "UnaryExpression": unaryUpdateExpression,
      "UpdateExpression": unaryUpdateExpression,
      // VariableDeclarator
      VariableDeclarator(node) {
        if (astUtils.isTypeAssertion(node.init)) {
          return rules.VariableDeclarator({
            ...node,
            type: utils$1.AST_NODE_TYPES.VariableDeclarator,
            init: {
              ...node.init,
              type: utils$1.AST_NODE_TYPES.FunctionExpression
            }
          });
        }
        return rules.VariableDeclarator(node);
      },
      // WhileStatement
      // WithStatement - i'm not going to even bother implementing this terrible and never used feature
      YieldExpression(node) {
        if (node.argument && !astUtils.isTypeAssertion(node.argument))
          return rules.YieldExpression(node);
      },
      ForInStatement(node) {
        if (astUtils.isTypeAssertion(node.right)) {
          return;
        }
        return rules.ForInStatement(node);
      },
      ForOfStatement(node) {
        if (astUtils.isTypeAssertion(node.right)) {
          return rules.ForOfStatement({
            ...node,
            type: utils$1.AST_NODE_TYPES.ForOfStatement,
            right: {
              ...node.right,
              type: utils$1.AST_NODE_TYPES.SequenceExpression
            }
          });
        }
        return rules.ForOfStatement(node);
      },
      TSStringKeyword(node) {
        return rules.TSStringKeyword({
          ...node,
          type: utils$1.AST_NODE_TYPES.FunctionExpression
        });
      }
    };
    return Object.assign({}, rules, overrides);
  }
});

const baseRule$6 = utils.getESLintCoreRule("no-extra-semi");
var noExtraSemi = utils.createRule({
  name: "no-extra-semi",
  meta: {
    type: "layout",
    docs: {
      description: "Disallow unnecessary semicolons",
      extendsBaseRule: true
    },
    fixable: "code",
    hasSuggestions: baseRule$6.meta.hasSuggestions,
    schema: baseRule$6.meta.schema,
    messages: baseRule$6.meta.messages
  },
  defaultOptions: [],
  create(context) {
    const rules = baseRule$6.create(context);
    return {
      ...rules,
      "TSAbstractMethodDefinition, TSAbstractPropertyDefinition": function(node) {
        rules["MethodDefinition, PropertyDefinition, StaticBlock"]?.(node);
      }
    };
  }
});

const baseRule$5 = utils.getESLintCoreRule("object-curly-spacing");
var objectCurlySpacing = utils.createRule({
  name: "object-curly-spacing",
  //
  meta: {
    ...baseRule$5.meta,
    docs: {
      description: "Enforce consistent spacing inside braces",
      extendsBaseRule: true
    }
  },
  defaultOptions: ["never"],
  create(context) {
    const [firstOption, secondOption] = context.options;
    const spaced = firstOption === "always";
    const sourceCode = context.sourceCode;
    function isOptionSet(option) {
      return secondOption ? secondOption[option] === !spaced : false;
    }
    const options = {
      spaced,
      arraysInObjectsException: isOptionSet("arraysInObjects"),
      objectsInObjectsException: isOptionSet("objectsInObjects")
    };
    function reportNoBeginningSpace(node, token) {
      const nextToken = context.getSourceCode().getTokenAfter(token, { includeComments: true });
      context.report({
        node,
        loc: { start: token.loc.end, end: nextToken.loc.start },
        messageId: "unexpectedSpaceAfter",
        data: {
          token: token.value
        },
        fix(fixer) {
          return fixer.removeRange([token.range[1], nextToken.range[0]]);
        }
      });
    }
    function reportNoEndingSpace(node, token) {
      const previousToken = context.getSourceCode().getTokenBefore(token, { includeComments: true });
      context.report({
        node,
        loc: { start: previousToken.loc.end, end: token.loc.start },
        messageId: "unexpectedSpaceBefore",
        data: {
          token: token.value
        },
        fix(fixer) {
          return fixer.removeRange([previousToken.range[1], token.range[0]]);
        }
      });
    }
    function reportRequiredBeginningSpace(node, token) {
      context.report({
        node,
        loc: token.loc,
        messageId: "requireSpaceAfter",
        data: {
          token: token.value
        },
        fix(fixer) {
          return fixer.insertTextAfter(token, " ");
        }
      });
    }
    function reportRequiredEndingSpace(node, token) {
      context.report({
        node,
        loc: token.loc,
        messageId: "requireSpaceBefore",
        data: {
          token: token.value
        },
        fix(fixer) {
          return fixer.insertTextBefore(token, " ");
        }
      });
    }
    function validateBraceSpacing(node, first, second, penultimate, last) {
      if (astUtils.isTokenOnSameLine(first, second)) {
        const firstSpaced = sourceCode.isSpaceBetween(first, second);
        const secondType = sourceCode.getNodeByRangeIndex(
          second.range[0]
        ).type;
        const openingCurlyBraceMustBeSpaced = options.arraysInObjectsException && [
          utils$1.AST_NODE_TYPES.TSMappedType,
          utils$1.AST_NODE_TYPES.TSIndexSignature
        ].includes(secondType) ? !options.spaced : options.spaced;
        if (openingCurlyBraceMustBeSpaced && !firstSpaced)
          reportRequiredBeginningSpace(node, first);
        if (!openingCurlyBraceMustBeSpaced && firstSpaced && second.type !== utils$1.AST_TOKEN_TYPES.Line)
          reportNoBeginningSpace(node, first);
      }
      if (astUtils.isTokenOnSameLine(penultimate, last)) {
        const shouldCheckPenultimate = options.arraysInObjectsException && astUtils.isClosingBracketToken(penultimate) || options.objectsInObjectsException && astUtils.isClosingBraceToken(penultimate);
        const penultimateType = shouldCheckPenultimate ? sourceCode.getNodeByRangeIndex(penultimate.range[0]).type : void 0;
        const closingCurlyBraceMustBeSpaced = options.arraysInObjectsException && penultimateType === utils$1.AST_NODE_TYPES.TSTupleType || options.objectsInObjectsException && penultimateType !== void 0 && [
          utils$1.AST_NODE_TYPES.TSMappedType,
          utils$1.AST_NODE_TYPES.TSTypeLiteral
        ].includes(penultimateType) ? !options.spaced : options.spaced;
        const lastSpaced = sourceCode.isSpaceBetween(penultimate, last);
        if (closingCurlyBraceMustBeSpaced && !lastSpaced)
          reportRequiredEndingSpace(node, last);
        if (!closingCurlyBraceMustBeSpaced && lastSpaced)
          reportNoEndingSpace(node, last);
      }
    }
    function getClosingBraceOfObject(node) {
      const lastProperty = node.members[node.members.length - 1];
      return sourceCode.getTokenAfter(lastProperty, astUtils.isClosingBraceToken);
    }
    const rules = baseRule$5.create(context);
    return {
      ...rules,
      TSMappedType(node) {
        const first = sourceCode.getFirstToken(node);
        const last = sourceCode.getLastToken(node);
        const second = sourceCode.getTokenAfter(first, {
          includeComments: true
        });
        const penultimate = sourceCode.getTokenBefore(last, {
          includeComments: true
        });
        validateBraceSpacing(node, first, second, penultimate, last);
      },
      TSTypeLiteral(node) {
        if (node.members.length === 0)
          return;
        const first = sourceCode.getFirstToken(node);
        const last = getClosingBraceOfObject(node);
        const second = sourceCode.getTokenAfter(first, {
          includeComments: true
        });
        const penultimate = sourceCode.getTokenBefore(last, {
          includeComments: true
        });
        validateBraceSpacing(node, first, second, penultimate, last);
      }
    };
  }
});

const CJS_EXPORT = /^(?:module\s*\.\s*)?exports(?:\s*\.|\s*\[|$)/u;
const CJS_IMPORT = /^require\(/u;
const LT = `[${Array.from(
  /* @__PURE__ */ new Set(["\r\n", "\r", "\n", "\u2028", "\u2029"])
).join("")}]`;
const PADDING_LINE_SEQUENCE = new RegExp(
  String.raw`^(\s*?${LT})\s*${LT}(\s*;?)$`,
  "u"
);
function newKeywordTester(type, keyword) {
  return {
    test(node, sourceCode) {
      const isSameKeyword = sourceCode.getFirstToken(node)?.value === keyword;
      const isSameType = Array.isArray(type) ? type.includes(node.type) : type === node.type;
      return isSameKeyword && isSameType;
    }
  };
}
function newSinglelineKeywordTester(keyword) {
  return {
    test(node, sourceCode) {
      return node.loc.start.line === node.loc.end.line && sourceCode.getFirstToken(node).value === keyword;
    }
  };
}
function newMultilineKeywordTester(keyword) {
  return {
    test(node, sourceCode) {
      return node.loc.start.line !== node.loc.end.line && sourceCode.getFirstToken(node).value === keyword;
    }
  };
}
function newNodeTypeTester(type) {
  return {
    test: (node) => node.type === type
  };
}
function skipChainExpression(node) {
  return node && node.type === utils$1.AST_NODE_TYPES.ChainExpression ? node.expression : node;
}
function isIIFEStatement(node) {
  if (node.type === utils$1.AST_NODE_TYPES.ExpressionStatement) {
    let expression = skipChainExpression(node.expression);
    if (expression.type === utils$1.AST_NODE_TYPES.UnaryExpression)
      expression = skipChainExpression(expression.argument);
    if (expression.type === utils$1.AST_NODE_TYPES.CallExpression) {
      let node2 = expression.callee;
      while (node2.type === utils$1.AST_NODE_TYPES.SequenceExpression)
        node2 = node2.expressions[node2.expressions.length - 1];
      return astUtils.isFunction(node2);
    }
  }
  return false;
}
function isCJSRequire(node) {
  if (node.type === utils$1.AST_NODE_TYPES.VariableDeclaration) {
    const declaration = node.declarations[0];
    if (declaration?.init) {
      let call = declaration?.init;
      while (call.type === utils$1.AST_NODE_TYPES.MemberExpression)
        call = call.object;
      if (call.type === utils$1.AST_NODE_TYPES.CallExpression && call.callee.type === utils$1.AST_NODE_TYPES.Identifier)
        return call.callee.name === "require";
    }
  }
  return false;
}
function isBlockLikeStatement(node, sourceCode) {
  if (node.type === utils$1.AST_NODE_TYPES.DoWhileStatement && node.body.type === utils$1.AST_NODE_TYPES.BlockStatement)
    return true;
  if (isIIFEStatement(node))
    return true;
  const lastToken = sourceCode.getLastToken(node, astUtils.isNotSemicolonToken);
  const belongingNode = lastToken && astUtils.isClosingBraceToken(lastToken) ? sourceCode.getNodeByRangeIndex(lastToken.range[0]) : null;
  return !!belongingNode && (belongingNode.type === utils$1.AST_NODE_TYPES.BlockStatement || belongingNode.type === utils$1.AST_NODE_TYPES.SwitchStatement);
}
function isDirective(node, sourceCode) {
  return node.type === utils$1.AST_NODE_TYPES.ExpressionStatement && (node.parent?.type === utils$1.AST_NODE_TYPES.Program || node.parent?.type === utils$1.AST_NODE_TYPES.BlockStatement && astUtils.isFunction(node.parent.parent)) && node.expression.type === utils$1.AST_NODE_TYPES.Literal && typeof node.expression.value === "string" && !astUtils.isParenthesized(node.expression, sourceCode);
}
function isDirectivePrologue(node, sourceCode) {
  if (isDirective(node, sourceCode) && node.parent && "body" in node.parent && Array.isArray(node.parent.body)) {
    for (const sibling of node.parent.body) {
      if (sibling === node)
        break;
      if (!isDirective(sibling, sourceCode))
        return false;
    }
    return true;
  }
  return false;
}
function isCJSExport(node) {
  if (node.type === utils$1.AST_NODE_TYPES.ExpressionStatement) {
    const expression = node.expression;
    if (expression.type === utils$1.AST_NODE_TYPES.AssignmentExpression) {
      let left = expression.left;
      if (left.type === utils$1.AST_NODE_TYPES.MemberExpression) {
        while (left.object.type === utils$1.AST_NODE_TYPES.MemberExpression)
          left = left.object;
        return left.object.type === utils$1.AST_NODE_TYPES.Identifier && (left.object.name === "exports" || left.object.name === "module" && left.property.type === utils$1.AST_NODE_TYPES.Identifier && left.property.name === "exports");
      }
    }
  }
  return false;
}
function isExpression(node, sourceCode) {
  return node.type === utils$1.AST_NODE_TYPES.ExpressionStatement && !isDirectivePrologue(node, sourceCode);
}
function getActualLastToken(node, sourceCode) {
  const semiToken = sourceCode.getLastToken(node);
  const prevToken = sourceCode.getTokenBefore(semiToken);
  const nextToken = sourceCode.getTokenAfter(semiToken);
  const isSemicolonLessStyle = prevToken && nextToken && prevToken.range[0] >= node.range[0] && astUtils.isSemicolonToken(semiToken) && semiToken.loc.start.line !== prevToken.loc.end.line && semiToken.loc.end.line === nextToken.loc.start.line;
  return isSemicolonLessStyle ? prevToken : semiToken;
}
function replacerToRemovePaddingLines(_, trailingSpaces, indentSpaces) {
  return trailingSpaces + indentSpaces;
}
function verifyForAny() {
}
function verifyForNever(context, _, nextNode, paddingLines) {
  if (paddingLines.length === 0)
    return;
  context.report({
    node: nextNode,
    messageId: "unexpectedBlankLine",
    fix(fixer) {
      if (paddingLines.length >= 2)
        return null;
      const prevToken = paddingLines[0][0];
      const nextToken = paddingLines[0][1];
      const start = prevToken.range[1];
      const end = nextToken.range[0];
      const text = context.getSourceCode().text.slice(start, end).replace(PADDING_LINE_SEQUENCE, replacerToRemovePaddingLines);
      return fixer.replaceTextRange([start, end], text);
    }
  });
}
function verifyForAlways(context, prevNode, nextNode, paddingLines) {
  if (paddingLines.length > 0)
    return;
  context.report({
    node: nextNode,
    messageId: "expectedBlankLine",
    fix(fixer) {
      const sourceCode = context.sourceCode;
      let prevToken = getActualLastToken(prevNode, sourceCode);
      const nextToken = sourceCode.getFirstTokenBetween(prevToken, nextNode, {
        includeComments: true,
        /**
         * Skip the trailing comments of the previous node.
         * This inserts a blank line after the last trailing comment.
         *
         * For example:
         *
         *     foo(); // trailing comment.
         *     // comment.
         *     bar();
         *
         * Get fixed to:
         *
         *     foo(); // trailing comment.
         *
         *     // comment.
         *     bar();
         * @param token The token to check.
         * @returns `true` if the token is not a trailing comment.
         * @private
         */
        filter(token) {
          if (astUtils.isTokenOnSameLine(prevToken, token)) {
            prevToken = token;
            return false;
          }
          return true;
        }
      }) || nextNode;
      const insertText = astUtils.isTokenOnSameLine(prevToken, nextToken) ? "\n\n" : "\n";
      return fixer.insertTextAfter(prevToken, insertText);
    }
  });
}
const PaddingTypes = {
  any: { verify: verifyForAny },
  never: { verify: verifyForNever },
  always: { verify: verifyForAlways }
};
const StatementTypes = {
  "*": { test: () => true },
  "block-like": { test: isBlockLikeStatement },
  "exports": { test: isCJSExport },
  "require": { test: isCJSRequire },
  "directive": { test: isDirectivePrologue },
  "expression": { test: isExpression },
  "iife": { test: isIIFEStatement },
  "multiline-block-like": {
    test: (node, sourceCode) => node.loc.start.line !== node.loc.end.line && isBlockLikeStatement(node, sourceCode)
  },
  "multiline-expression": {
    test: (node, sourceCode) => node.loc.start.line !== node.loc.end.line && node.type === utils$1.AST_NODE_TYPES.ExpressionStatement && !isDirectivePrologue(node, sourceCode)
  },
  "multiline-const": newMultilineKeywordTester("const"),
  "multiline-let": newMultilineKeywordTester("let"),
  "multiline-var": newMultilineKeywordTester("var"),
  "singleline-const": newSinglelineKeywordTester("const"),
  "singleline-let": newSinglelineKeywordTester("let"),
  "singleline-var": newSinglelineKeywordTester("var"),
  "block": newNodeTypeTester(utils$1.AST_NODE_TYPES.BlockStatement),
  "empty": newNodeTypeTester(utils$1.AST_NODE_TYPES.EmptyStatement),
  "function": newNodeTypeTester(utils$1.AST_NODE_TYPES.FunctionDeclaration),
  "break": newKeywordTester(utils$1.AST_NODE_TYPES.BreakStatement, "break"),
  "case": newKeywordTester(utils$1.AST_NODE_TYPES.SwitchCase, "case"),
  "class": newKeywordTester(utils$1.AST_NODE_TYPES.ClassDeclaration, "class"),
  "const": newKeywordTester(utils$1.AST_NODE_TYPES.VariableDeclaration, "const"),
  "continue": newKeywordTester(utils$1.AST_NODE_TYPES.ContinueStatement, "continue"),
  "debugger": newKeywordTester(utils$1.AST_NODE_TYPES.DebuggerStatement, "debugger"),
  "default": newKeywordTester(
    [utils$1.AST_NODE_TYPES.SwitchCase, utils$1.AST_NODE_TYPES.ExportDefaultDeclaration],
    "default"
  ),
  "do": newKeywordTester(utils$1.AST_NODE_TYPES.DoWhileStatement, "do"),
  "export": newKeywordTester(
    [
      utils$1.AST_NODE_TYPES.ExportAllDeclaration,
      utils$1.AST_NODE_TYPES.ExportDefaultDeclaration,
      utils$1.AST_NODE_TYPES.ExportNamedDeclaration
    ],
    "export"
  ),
  "for": newKeywordTester(
    [
      utils$1.AST_NODE_TYPES.ForStatement,
      utils$1.AST_NODE_TYPES.ForInStatement,
      utils$1.AST_NODE_TYPES.ForOfStatement
    ],
    "for"
  ),
  "if": newKeywordTester(utils$1.AST_NODE_TYPES.IfStatement, "if"),
  "import": newKeywordTester(utils$1.AST_NODE_TYPES.ImportDeclaration, "import"),
  "let": newKeywordTester(utils$1.AST_NODE_TYPES.VariableDeclaration, "let"),
  "return": newKeywordTester(utils$1.AST_NODE_TYPES.ReturnStatement, "return"),
  "switch": newKeywordTester(utils$1.AST_NODE_TYPES.SwitchStatement, "switch"),
  "throw": newKeywordTester(utils$1.AST_NODE_TYPES.ThrowStatement, "throw"),
  "try": newKeywordTester(utils$1.AST_NODE_TYPES.TryStatement, "try"),
  "var": newKeywordTester(utils$1.AST_NODE_TYPES.VariableDeclaration, "var"),
  "while": newKeywordTester(
    [utils$1.AST_NODE_TYPES.WhileStatement, utils$1.AST_NODE_TYPES.DoWhileStatement],
    "while"
  ),
  "with": newKeywordTester(utils$1.AST_NODE_TYPES.WithStatement, "with"),
  "cjs-export": {
    test: (node, sourceCode) => node.type === "ExpressionStatement" && node.expression.type === "AssignmentExpression" && CJS_EXPORT.test(sourceCode.getText(node.expression.left))
  },
  "cjs-import": {
    test: (node, sourceCode) => node.type === "VariableDeclaration" && node.declarations.length > 0 && Boolean(node.declarations[0].init) && CJS_IMPORT.test(sourceCode.getText(node.declarations[0].init))
  },
  // Additional Typescript constructs
  "interface": newKeywordTester(
    utils$1.AST_NODE_TYPES.TSInterfaceDeclaration,
    "interface"
  ),
  "type": newKeywordTester(
    utils$1.AST_NODE_TYPES.TSTypeAliasDeclaration,
    "type"
  ),
  "function-overload": {
    test: (node) => node.type === "TSDeclareFunction"
  }
};
var paddingLineBetweenStatements = utils.createRule({
  name: "padding-line-between-statements",
  meta: {
    type: "layout",
    docs: {
      description: "Require or disallow padding lines between statements",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    hasSuggestions: false,
    // This is intentionally an array schema as you can pass 0..n config objects
    schema: {
      $defs: {
        paddingType: {
          type: "string",
          enum: Object.keys(PaddingTypes)
        },
        statementType: {
          anyOf: [
            {
              type: "string",
              enum: Object.keys(StatementTypes)
            },
            {
              type: "array",
              items: {
                type: "string",
                enum: Object.keys(StatementTypes)
              },
              minItems: 1,
              uniqueItems: true,
              additionalItems: false
            }
          ]
        }
      },
      type: "array",
      additionalItems: false,
      items: {
        type: "object",
        properties: {
          blankLine: { $ref: "#/$defs/paddingType" },
          prev: { $ref: "#/$defs/statementType" },
          next: { $ref: "#/$defs/statementType" }
        },
        additionalProperties: false,
        required: ["blankLine", "prev", "next"]
      }
    },
    messages: {
      unexpectedBlankLine: "Unexpected blank line before this statement.",
      expectedBlankLine: "Expected blank line before this statement."
    }
  },
  defaultOptions: [],
  create(context) {
    const sourceCode = context.sourceCode;
    const configureList = context.options || [];
    let scopeInfo = null;
    function enterScope() {
      scopeInfo = {
        upper: scopeInfo,
        prevNode: null
      };
    }
    function exitScope() {
      if (scopeInfo)
        scopeInfo = scopeInfo.upper;
    }
    function match(node, type) {
      let innerStatementNode = node;
      while (innerStatementNode.type === utils$1.AST_NODE_TYPES.LabeledStatement)
        innerStatementNode = innerStatementNode.body;
      if (Array.isArray(type))
        return type.some(match.bind(null, innerStatementNode));
      return StatementTypes[type].test(innerStatementNode, sourceCode);
    }
    function getPaddingType(prevNode, nextNode) {
      for (let i = configureList.length - 1; i >= 0; --i) {
        const configure = configureList[i];
        if (match(prevNode, configure.prev) && match(nextNode, configure.next))
          return PaddingTypes[configure.blankLine];
      }
      return PaddingTypes.any;
    }
    function getPaddingLineSequences(prevNode, nextNode) {
      const pairs = [];
      let prevToken = getActualLastToken(prevNode, sourceCode);
      if (nextNode.loc.start.line - prevToken.loc.end.line >= 2) {
        do {
          const token = sourceCode.getTokenAfter(prevToken, {
            includeComments: true
          });
          if (token.loc.start.line - prevToken.loc.end.line >= 2)
            pairs.push([prevToken, token]);
          prevToken = token;
        } while (prevToken.range[0] < nextNode.range[0]);
      }
      return pairs;
    }
    function verify(node) {
      if (!node.parent || ![
        utils$1.AST_NODE_TYPES.BlockStatement,
        utils$1.AST_NODE_TYPES.Program,
        utils$1.AST_NODE_TYPES.SwitchCase,
        utils$1.AST_NODE_TYPES.SwitchStatement,
        utils$1.AST_NODE_TYPES.TSModuleBlock
      ].includes(node.parent.type))
        return;
      const prevNode = scopeInfo.prevNode;
      if (prevNode) {
        const type = getPaddingType(prevNode, node);
        const paddingLines = getPaddingLineSequences(prevNode, node);
        type.verify(context, prevNode, node, paddingLines);
      }
      scopeInfo.prevNode = node;
    }
    function verifyThenEnterScope(node) {
      verify(node);
      enterScope();
    }
    return {
      "Program": enterScope,
      "BlockStatement": enterScope,
      "SwitchStatement": enterScope,
      "TSModuleBlock": enterScope,
      "Program:exit": exitScope,
      "BlockStatement:exit": exitScope,
      "SwitchStatement:exit": exitScope,
      "TSModuleBlock:exit": exitScope,
      ":statement": verify,
      "SwitchCase": verifyThenEnterScope,
      "TSDeclareFunction": verifyThenEnterScope,
      "SwitchCase:exit": exitScope,
      "TSDeclareFunction:exit": exitScope
    };
  }
});

const baseRule$4 = utils.getESLintCoreRule("quote-props");
var quoteProps = utils.createRule({
  name: "quote-props",
  meta: {
    ...baseRule$4.meta,
    docs: {
      description: "Require quotes around object literal, type literal, interfaces and enums property names",
      extendsBaseRule: true
    }
  },
  defaultOptions: ["always"],
  create(context) {
    const rules = baseRule$4.create(context);
    return {
      ...rules,
      TSPropertySignature(node) {
        return rules.Property({
          ...node,
          type: utils$1.AST_NODE_TYPES.Property,
          shorthand: false,
          method: false,
          kind: "init",
          value: null
        });
      },
      TSMethodSignature(node) {
        return rules.Property({
          ...node,
          type: utils$1.AST_NODE_TYPES.Property,
          shorthand: false,
          method: true,
          kind: "init",
          value: null
        });
      },
      TSEnumMember(node) {
        return rules.Property({
          ...node,
          type: utils$1.AST_NODE_TYPES.Property,
          key: node.id,
          optional: false,
          shorthand: false,
          method: false,
          kind: "init",
          value: null
        });
      },
      TSTypeLiteral(node) {
        return rules.ObjectExpression({
          ...node,
          type: utils$1.AST_NODE_TYPES.ObjectExpression,
          properties: node.members
        });
      },
      TSInterfaceBody(node) {
        return rules.ObjectExpression({
          ...node,
          type: utils$1.AST_NODE_TYPES.ObjectExpression,
          properties: node.body
        });
      },
      TSEnumDeclaration(node) {
        return rules.ObjectExpression({
          ...node,
          type: utils$1.AST_NODE_TYPES.ObjectExpression,
          properties: node.members.map((member) => ({ ...member, key: member.id }))
        });
      }
    };
  }
});

const baseRule$3 = utils.getESLintCoreRule("quotes");
var quotes = utils.createRule({
  name: "quotes",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce the consistent use of either backticks, double, or single quotes",
      extendsBaseRule: true
    },
    fixable: "code",
    hasSuggestions: baseRule$3.meta.hasSuggestions,
    messages: baseRule$3.meta.messages,
    schema: baseRule$3.meta.schema
  },
  defaultOptions: [
    "double",
    {
      allowTemplateLiterals: false,
      avoidEscape: false
    }
  ],
  create(context, [option]) {
    const rules = baseRule$3.create(context);
    function isAllowedAsNonBacktick(node) {
      const parent = node.parent;
      switch (parent?.type) {
        case utils$1.AST_NODE_TYPES.TSAbstractMethodDefinition:
        case utils$1.AST_NODE_TYPES.TSMethodSignature:
        case utils$1.AST_NODE_TYPES.TSPropertySignature:
        case utils$1.AST_NODE_TYPES.TSModuleDeclaration:
        case utils$1.AST_NODE_TYPES.TSExternalModuleReference:
          return true;
        case utils$1.AST_NODE_TYPES.TSEnumMember:
          return node === parent.id;
        case utils$1.AST_NODE_TYPES.TSAbstractPropertyDefinition:
        case utils$1.AST_NODE_TYPES.PropertyDefinition:
          return node === parent.key;
        default:
          return false;
      }
    }
    return {
      Literal(node) {
        if (option === "backtick" && isAllowedAsNonBacktick(node))
          return;
        rules.Literal(node);
      },
      TemplateLiteral(node) {
        rules.TemplateLiteral(node);
      }
    };
  }
});

const baseRule$2 = utils.getESLintCoreRule("semi");
var semi = utils.createRule({
  name: "semi",
  meta: {
    type: "layout",
    docs: {
      description: "Require or disallow semicolons instead of ASI",
      // too opinionated to be recommended
      extendsBaseRule: true
    },
    fixable: "code",
    hasSuggestions: baseRule$2.meta.hasSuggestions,
    schema: baseRule$2.meta.schema,
    messages: baseRule$2.meta.messages
  },
  defaultOptions: [
    "always",
    {
      omitLastInOneLineBlock: false,
      beforeStatementContinuationChars: "any"
    }
  ],
  create(context) {
    const rules = baseRule$2.create(context);
    const checkForSemicolon = rules.ExpressionStatement;
    const nodesToCheck = [
      utils$1.AST_NODE_TYPES.PropertyDefinition,
      utils$1.AST_NODE_TYPES.TSAbstractPropertyDefinition,
      utils$1.AST_NODE_TYPES.TSDeclareFunction,
      utils$1.AST_NODE_TYPES.TSExportAssignment,
      utils$1.AST_NODE_TYPES.TSImportEqualsDeclaration,
      utils$1.AST_NODE_TYPES.TSTypeAliasDeclaration,
      utils$1.AST_NODE_TYPES.TSEmptyBodyFunctionExpression
    ].reduce((acc, node) => {
      acc[node] = checkForSemicolon;
      return acc;
    }, {});
    return {
      ...rules,
      ...nodesToCheck,
      ExportDefaultDeclaration(node) {
        if (node.declaration.type !== utils$1.AST_NODE_TYPES.TSInterfaceDeclaration)
          rules.ExportDefaultDeclaration(node);
      }
    };
  }
});

const baseRule$1 = utils.getESLintCoreRule("space-before-blocks");
var spaceBeforeBlocks = utils.createRule({
  name: "space-before-blocks",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent spacing before blocks",
      extendsBaseRule: true
    },
    fixable: baseRule$1.meta.fixable,
    hasSuggestions: baseRule$1.meta.hasSuggestions,
    schema: baseRule$1.meta.schema,
    messages: {
      unexpectedSpace: "Unexpected space before opening brace.",
      missingSpace: "Missing space before opening brace.",
      ...baseRule$1.meta.messages
    }
  },
  defaultOptions: ["always"],
  create(context, [config]) {
    const rules = baseRule$1.create(context);
    const sourceCode = context.sourceCode;
    let requireSpace = true;
    if (typeof config === "object")
      requireSpace = config.classes === "always";
    else if (config === "never")
      requireSpace = false;
    function checkPrecedingSpace(node) {
      const precedingToken = sourceCode.getTokenBefore(node);
      if (precedingToken && astUtils.isTokenOnSameLine(precedingToken, node)) {
        const hasSpace = sourceCode.isSpaceBetweenTokens(
          precedingToken,
          node
        );
        if (requireSpace && !hasSpace) {
          context.report({
            node,
            messageId: "missingSpace",
            fix(fixer) {
              return fixer.insertTextBefore(node, " ");
            }
          });
        } else if (!requireSpace && hasSpace) {
          context.report({
            node,
            messageId: "unexpectedSpace",
            fix(fixer) {
              return fixer.removeRange([
                precedingToken.range[1],
                node.range[0]
              ]);
            }
          });
        }
      }
    }
    function checkSpaceAfterEnum(node) {
      const punctuator = sourceCode.getTokenAfter(node.id);
      if (punctuator)
        checkPrecedingSpace(punctuator);
    }
    return {
      ...rules,
      TSEnumDeclaration: checkSpaceAfterEnum,
      TSInterfaceBody: checkPrecedingSpace
    };
  }
});

var spaceBeforeFunctionParen = utils.createRule({
  name: "space-before-function-paren",
  meta: {
    type: "layout",
    docs: {
      description: "Enforce consistent spacing before function parenthesis",
      extendsBaseRule: true
    },
    fixable: "whitespace",
    schema: [
      {
        oneOf: [
          {
            type: "string",
            enum: ["always", "never"]
          },
          {
            type: "object",
            properties: {
              anonymous: {
                type: "string",
                enum: ["always", "never", "ignore"]
              },
              named: {
                type: "string",
                enum: ["always", "never", "ignore"]
              },
              asyncArrow: {
                type: "string",
                enum: ["always", "never", "ignore"]
              }
            },
            additionalProperties: false
          }
        ]
      }
    ],
    messages: {
      unexpected: "Unexpected space before function parentheses.",
      missing: "Missing space before function parentheses."
    }
  },
  defaultOptions: ["always"],
  create(context, [firstOption]) {
    const sourceCode = context.sourceCode;
    const baseConfig = typeof firstOption === "string" ? firstOption : "always";
    const overrideConfig = typeof firstOption === "object" ? firstOption : {};
    function isNamedFunction(node) {
      if (node.id != null)
        return true;
      const parent = node.parent;
      return parent.type === utils$1.AST_NODE_TYPES.MethodDefinition || parent.type === utils$1.AST_NODE_TYPES.TSAbstractMethodDefinition || parent.type === utils$1.AST_NODE_TYPES.Property && (parent.kind === "get" || parent.kind === "set" || parent.method);
    }
    function getConfigForFunction(node) {
      if (node.type === utils$1.AST_NODE_TYPES.ArrowFunctionExpression) {
        if (node.async && astUtils.isOpeningParenToken(sourceCode.getFirstToken(node, { skip: 1 })))
          return overrideConfig.asyncArrow ?? baseConfig;
      } else if (isNamedFunction(node)) {
        return overrideConfig.named ?? baseConfig;
      } else if (!node.generator) {
        return overrideConfig.anonymous ?? baseConfig;
      }
      return "ignore";
    }
    function checkFunction(node) {
      const functionConfig = getConfigForFunction(node);
      if (functionConfig === "ignore")
        return;
      let leftToken;
      let rightToken;
      if (node.typeParameters) {
        leftToken = sourceCode.getLastToken(node.typeParameters);
        rightToken = sourceCode.getTokenAfter(leftToken);
      } else {
        rightToken = sourceCode.getFirstToken(node, astUtils.isOpeningParenToken);
        leftToken = sourceCode.getTokenBefore(rightToken);
      }
      const hasSpacing = sourceCode.isSpaceBetweenTokens(leftToken, rightToken);
      if (hasSpacing && functionConfig === "never") {
        context.report({
          node,
          loc: {
            start: leftToken.loc.end,
            end: rightToken.loc.start
          },
          messageId: "unexpected",
          fix: (fixer) => fixer.removeRange([leftToken.range[1], rightToken.range[0]])
        });
      } else if (!hasSpacing && functionConfig === "always" && (!node.typeParameters || node.id)) {
        context.report({
          node,
          loc: rightToken.loc,
          messageId: "missing",
          fix: (fixer) => fixer.insertTextAfter(leftToken, " ")
        });
      }
    }
    return {
      ArrowFunctionExpression: checkFunction,
      FunctionDeclaration: checkFunction,
      FunctionExpression: checkFunction,
      TSEmptyBodyFunctionExpression: checkFunction,
      TSDeclareFunction: checkFunction
    };
  }
});

const baseRule = utils.getESLintCoreRule("space-infix-ops");
const UNIONS = ["|", "&"];
var spaceInfixOps = utils.createRule({
  name: "space-infix-ops",
  meta: {
    type: "layout",
    docs: {
      description: "Require spacing around infix operators",
      extendsBaseRule: true
    },
    fixable: baseRule.meta.fixable,
    hasSuggestions: baseRule.meta.hasSuggestions,
    schema: baseRule.meta.schema,
    messages: {
      missingSpace: "Operator '{{operator}}' must be spaced.",
      ...baseRule.meta.messages
    }
  },
  defaultOptions: [
    {
      int32Hint: false
    }
  ],
  create(context) {
    const rules = baseRule.create(context);
    const sourceCode = context.sourceCode;
    function report(operator) {
      context.report({
        node: operator,
        messageId: "missingSpace",
        data: {
          operator: operator.value
        },
        fix(fixer) {
          const previousToken = sourceCode.getTokenBefore(operator);
          const afterToken = sourceCode.getTokenAfter(operator);
          let fixString = "";
          if (operator.range[0] - previousToken.range[1] === 0)
            fixString = " ";
          fixString += operator.value;
          if (afterToken.range[0] - operator.range[1] === 0)
            fixString += " ";
          return fixer.replaceText(operator, fixString);
        }
      });
    }
    function isSpaceChar(token) {
      return token.type === utils$1.AST_TOKEN_TYPES.Punctuator && /^[=?:]$/.test(token.value);
    }
    function checkAndReportAssignmentSpace(leftNode, rightNode) {
      if (!rightNode || !leftNode)
        return;
      const operator = sourceCode.getFirstTokenBetween(
        leftNode,
        rightNode,
        isSpaceChar
      );
      const prev = sourceCode.getTokenBefore(operator);
      const next = sourceCode.getTokenAfter(operator);
      if (!sourceCode.isSpaceBetween(prev, operator) || !sourceCode.isSpaceBetween(operator, next))
        report(operator);
    }
    function checkForEnumAssignmentSpace(node) {
      checkAndReportAssignmentSpace(node.id, node.initializer);
    }
    function checkForPropertyDefinitionAssignmentSpace(node) {
      const leftNode = node.optional && !node.typeAnnotation ? sourceCode.getTokenAfter(node.key) : node.typeAnnotation ?? node.key;
      checkAndReportAssignmentSpace(leftNode, node.value);
    }
    function checkForTypeAnnotationSpace(typeAnnotation) {
      const types = typeAnnotation.types;
      types.forEach((type) => {
        const skipFunctionParenthesis = type.type === utils$1.AST_NODE_TYPES.TSFunctionType ? astUtils.isNotOpeningParenToken : 0;
        const operator = sourceCode.getTokenBefore(
          type,
          skipFunctionParenthesis
        );
        if (operator != null && UNIONS.includes(operator.value)) {
          const prev = sourceCode.getTokenBefore(operator);
          const next = sourceCode.getTokenAfter(operator);
          if (!sourceCode.isSpaceBetween(prev, operator) || !sourceCode.isSpaceBetween(operator, next))
            report(operator);
        }
      });
    }
    function checkForTypeAliasAssignment(node) {
      checkAndReportAssignmentSpace(
        node.typeParameters ?? node.id,
        node.typeAnnotation
      );
    }
    function checkForTypeConditional(node) {
      checkAndReportAssignmentSpace(node.extendsType, node.trueType);
      checkAndReportAssignmentSpace(node.trueType, node.falseType);
    }
    return {
      ...rules,
      TSEnumMember: checkForEnumAssignmentSpace,
      PropertyDefinition: checkForPropertyDefinitionAssignmentSpace,
      TSTypeAliasDeclaration: checkForTypeAliasAssignment,
      TSUnionType: checkForTypeAnnotationSpace,
      TSIntersectionType: checkForTypeAnnotationSpace,
      TSConditionalType: checkForTypeConditional
    };
  }
});

function createRules(options) {
  const globals = {
    ...options?.before !== void 0 ? { before: options.before } : {},
    ...options?.after !== void 0 ? { after: options.after } : {}
  };
  const override = options?.overrides ?? {};
  const colon = {
    ...{ before: false, after: true },
    ...globals,
    ...override?.colon
  };
  const arrow = {
    ...{ before: true, after: true },
    ...globals,
    ...override?.arrow
  };
  return {
    colon,
    arrow,
    variable: { ...colon, ...override?.variable },
    property: { ...colon, ...override?.property },
    parameter: { ...colon, ...override?.parameter },
    returnType: { ...colon, ...override?.returnType }
  };
}
function getIdentifierRules(rules, node) {
  const scope = node?.parent;
  if (astUtils.isVariableDeclarator(scope))
    return rules.variable;
  else if (astUtils.isFunctionOrFunctionType(scope))
    return rules.parameter;
  return rules.colon;
}
function getRules(rules, node) {
  const scope = node?.parent?.parent;
  if (astUtils.isTSFunctionType(scope) || astUtils.isTSConstructorType(scope))
    return rules.arrow;
  else if (astUtils.isIdentifier(scope))
    return getIdentifierRules(rules, scope);
  else if (astUtils.isClassOrTypeElement(scope))
    return rules.property;
  else if (astUtils.isFunction(scope))
    return rules.returnType;
  return rules.colon;
}
var typeAnnotationSpacing = utils.createRule({
  name: "type-annotation-spacing",
  meta: {
    type: "layout",
    docs: {
      description: "Require consistent spacing around type annotations"
    },
    fixable: "whitespace",
    messages: {
      expectedSpaceAfter: "Expected a space after the '{{type}}'.",
      expectedSpaceBefore: "Expected a space before the '{{type}}'.",
      unexpectedSpaceAfter: "Unexpected space after the '{{type}}'.",
      unexpectedSpaceBefore: "Unexpected space before the '{{type}}'.",
      unexpectedSpaceBetween: "Unexpected space between the '{{previousToken}}' and the '{{type}}'."
    },
    schema: [
      {
        $defs: {
          spacingConfig: {
            type: "object",
            properties: {
              before: { type: "boolean" },
              after: { type: "boolean" }
            },
            additionalProperties: false
          }
        },
        type: "object",
        properties: {
          before: { type: "boolean" },
          after: { type: "boolean" },
          overrides: {
            type: "object",
            properties: {
              colon: { $ref: "#/items/0/$defs/spacingConfig" },
              arrow: { $ref: "#/items/0/$defs/spacingConfig" },
              variable: { $ref: "#/items/0/$defs/spacingConfig" },
              parameter: { $ref: "#/items/0/$defs/spacingConfig" },
              property: { $ref: "#/items/0/$defs/spacingConfig" },
              returnType: { $ref: "#/items/0/$defs/spacingConfig" }
            },
            additionalProperties: false
          }
        },
        additionalProperties: false
      }
    ]
  },
  defaultOptions: [
    // technically there is a default, but the overrides mean
    // that if we apply them here, it will break the no override case.
    {}
  ],
  create(context, [options]) {
    const punctuators = [":", "=>"];
    const sourceCode = context.sourceCode;
    const ruleSet = createRules(options);
    function checkTypeAnnotationSpacing(typeAnnotation) {
      const nextToken = typeAnnotation;
      const punctuatorTokenEnd = sourceCode.getTokenBefore(nextToken);
      let punctuatorTokenStart = punctuatorTokenEnd;
      let previousToken = sourceCode.getTokenBefore(punctuatorTokenEnd);
      let type = punctuatorTokenEnd.value;
      if (!punctuators.includes(type))
        return;
      const { before, after } = getRules(ruleSet, typeAnnotation);
      if (type === ":" && previousToken.value === "?") {
        if (
          //  -- TODO - switch once our min ESLint version is 6.7.0
          sourceCode.isSpaceBetweenTokens(previousToken, punctuatorTokenStart)
        ) {
          context.report({
            node: punctuatorTokenStart,
            messageId: "unexpectedSpaceBetween",
            data: {
              type,
              previousToken: previousToken.value
            },
            fix(fixer) {
              return fixer.removeRange([
                previousToken.range[1],
                punctuatorTokenStart.range[0]
              ]);
            }
          });
        }
        type = "?:";
        punctuatorTokenStart = previousToken;
        previousToken = sourceCode.getTokenBefore(previousToken);
        if (previousToken.value === "+" || previousToken.value === "-") {
          type = `${previousToken.value}?:`;
          punctuatorTokenStart = previousToken;
          previousToken = sourceCode.getTokenBefore(previousToken);
        }
      }
      const previousDelta = punctuatorTokenStart.range[0] - previousToken.range[1];
      const nextDelta = nextToken.range[0] - punctuatorTokenEnd.range[1];
      if (after && nextDelta === 0) {
        context.report({
          node: punctuatorTokenEnd,
          messageId: "expectedSpaceAfter",
          data: {
            type
          },
          fix(fixer) {
            return fixer.insertTextAfter(punctuatorTokenEnd, " ");
          }
        });
      } else if (!after && nextDelta > 0) {
        context.report({
          node: punctuatorTokenEnd,
          messageId: "unexpectedSpaceAfter",
          data: {
            type
          },
          fix(fixer) {
            return fixer.removeRange([
              punctuatorTokenEnd.range[1],
              nextToken.range[0]
            ]);
          }
        });
      }
      if (before && previousDelta === 0) {
        context.report({
          node: punctuatorTokenStart,
          messageId: "expectedSpaceBefore",
          data: {
            type
          },
          fix(fixer) {
            return fixer.insertTextAfter(previousToken, " ");
          }
        });
      } else if (!before && previousDelta > 0) {
        context.report({
          node: punctuatorTokenStart,
          messageId: "unexpectedSpaceBefore",
          data: {
            type
          },
          fix(fixer) {
            return fixer.removeRange([
              previousToken.range[1],
              punctuatorTokenStart.range[0]
            ]);
          }
        });
      }
    }
    return {
      TSMappedType(node) {
        if (node.typeAnnotation)
          checkTypeAnnotationSpacing(node.typeAnnotation);
      },
      TSTypeAnnotation(node) {
        checkTypeAnnotationSpacing(node.typeAnnotation);
      }
    };
  }
});

var rules = {
  "block-spacing": blockSpacing,
  "brace-style": braceStyle,
  "comma-dangle": commaDangle,
  "comma-spacing": commaSpacing,
  "func-call-spacing": functionCallSpacing,
  "function-call-spacing": functionCallSpacing,
  "indent": indent,
  "key-spacing": keySpacing,
  "keyword-spacing": keywordSpacing,
  "lines-around-comment": linesAroundComment,
  "lines-between-class-members": linesBetweenClassMembers,
  "member-delimiter-style": memberDelimiterStyle,
  "no-extra-parens": noExtraParens,
  "no-extra-semi": noExtraSemi,
  "object-curly-spacing": objectCurlySpacing,
  "padding-line-between-statements": paddingLineBetweenStatements,
  "quote-props": quoteProps,
  "quotes": quotes,
  "semi": semi,
  "space-before-blocks": spaceBeforeBlocks,
  "space-before-function-paren": spaceBeforeFunctionParen,
  "space-infix-ops": spaceInfixOps,
  "type-annotation-spacing": typeAnnotationSpacing
};

var plugin = {
  rules
};

const config = {
  rules: {
    "@typescript-eslint/block-spacing": 0,
    "@typescript-eslint/brace-style": 0,
    "@typescript-eslint/comma-dangle": 0,
    "@typescript-eslint/comma-spacing": 0,
    "@typescript-eslint/func-call-spacing": 0,
    "@typescript-eslint/indent": 0,
    "@typescript-eslint/key-spacing": 0,
    "@typescript-eslint/keyword-spacing": 0,
    "@typescript-eslint/lines-around-comment": 0,
    "@typescript-eslint/lines-between-class-members": 0,
    "@typescript-eslint/member-delimiter-style": 0,
    "@typescript-eslint/no-extra-parens": 0,
    "@typescript-eslint/no-extra-semi": 0,
    "@typescript-eslint/object-curly-spacing": 0,
    "@typescript-eslint/padding-line-between-statements": 0,
    "@typescript-eslint/quote-props": 0,
    "@typescript-eslint/quotes": 0,
    "@typescript-eslint/semi": 0,
    "@typescript-eslint/space-before-blocks": 0,
    "@typescript-eslint/space-before-function-paren": 0,
    "@typescript-eslint/space-infix-ops": 0,
    "@typescript-eslint/type-annotation-spacing": 0
  }
};

const configs = {
  "disable-legacy": config,
  "all-flat": createAllConfigs(plugin, "@stylistic/ts", true),
  "all-extends": createAllConfigs(plugin, "@stylistic/ts", false)
};

exports.blockSpacing = blockSpacing;
exports.braceStyle = braceStyle;
exports.commaDangle = commaDangle;
exports.commaSpacing = commaSpacing;
exports.configs = configs;
exports.functionCallSpacing = functionCallSpacing;
exports.indent = indent;
exports.keySpacing = keySpacing;
exports.keywordSpacing = keywordSpacing;
exports.linesAroundComment = linesAroundComment;
exports.linesBetweenClassMembers = linesBetweenClassMembers;
exports.memberDelimiterStyle = memberDelimiterStyle;
exports.noExtraParens = noExtraParens;
exports.noExtraSemi = noExtraSemi;
exports.objectCurlySpacing = objectCurlySpacing;
exports.paddingLineBetweenStatements = paddingLineBetweenStatements;
exports.plugin = plugin;
exports.quoteProps = quoteProps;
exports.quotes = quotes;
exports.semi = semi;
exports.spaceBeforeBlocks = spaceBeforeBlocks;
exports.spaceBeforeFunctionParen = spaceBeforeFunctionParen;
exports.spaceInfixOps = spaceInfixOps;
exports.typeAnnotationSpacing = typeAnnotationSpacing;
