"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fromYaml = void 0;
const yaml_ast_parser_1 = require("yaml-ast-parser");
const keys_1 = require("./keys");
// The following code is inspired by
// https://github.com/yldio/pseudo-yaml-ast
const isBetween = (start, pos, end) => pos <= end && pos >= start;
const calculatePosition = (input, { start, end }) => {
    const lines = input.split(/\n/);
    const position = {
        start: {
            line: -1,
            column: -1,
        },
        end: {
            line: -1,
            column: -1,
        },
    };
    let sum = 0;
    for (const i of lines.keys()) {
        const line = lines[i];
        const ls = sum;
        const le = sum + line.length;
        if (position.start.line === -1 && isBetween(ls, start, le)) {
            position.start.line = i + 1;
            position.start.column = start - ls + 1;
        }
        if (position.end.line === -1 && isBetween(ls, end, le)) {
            position.end.line = i + 1;
            position.end.column = end - ls + 1;
        }
        sum = le + 1; // +1 because the break is also a char
    }
    return position;
};
const wrappedScalar = (Constructor, kind, value, position) => {
    const v = new Constructor(value);
    v[keys_1.positionKey] = position;
    v[keys_1.kindKey] = kind;
    return v;
};
const returnNull = () => null;
const wrappedNull = (position) => ({
    [keys_1.positionKey]: position,
    [keys_1.kindKey]: "null",
    valueOf: returnNull,
});
const visitorByNodeKind = {};
const walk = (nodes, input, ctx = {}) => {
    const onNode = (node, ctx2, fallback) => {
        const visitor = node
            ? visitorByNodeKind[node.kind]
            : visitorByNodeKind[yaml_ast_parser_1.Kind.SCALAR];
        return visitor ? visitor(node, input, ctx2) : fallback;
    };
    const walkObj = () => nodes.reduce((sum, node) => {
        return onNode(node, sum, sum);
    }, ctx);
    const walkArr = () => nodes.map((node) => onNode(node, ctx, null), ctx).filter(Boolean);
    return Array.isArray(ctx) ? walkArr() : walkObj();
};
visitorByNodeKind[yaml_ast_parser_1.Kind.MAP] = (node, input, ctx) => {
    return Object.assign(walk(node.mappings, input), {
        [keys_1.positionKey]: calculatePosition(input, {
            start: node.startPosition,
            end: node.endPosition,
        }),
        [keys_1.kindKey]: "object",
    });
};
visitorByNodeKind[yaml_ast_parser_1.Kind.MAPPING] = (node, input, ctx) => {
    const value = walk([node.value], input);
    if (node.value === null) {
        return Object.assign(ctx, {
            [node.key.value]: wrappedNull(calculatePosition(input, {
                start: node.startPosition,
                end: node.endPosition,
            })),
        });
    }
    value[keys_1.positionKey] = calculatePosition(input, {
        start: node.startPosition,
        end: node.endPosition,
    });
    return Object.assign(ctx, {
        [node.key.value]: value,
    });
};
visitorByNodeKind[yaml_ast_parser_1.Kind.SCALAR] = (node, input) => {
    if (!node) {
        return;
    }
    const position = calculatePosition(input, {
        start: node.startPosition,
        end: node.endPosition,
    });
    if (typeof node.valueObject === "boolean") {
        return wrappedScalar(Boolean, "boolean", node.valueObject, position);
    }
    else if (typeof node.valueObject === "number") {
        return wrappedScalar(Number, "number", node.valueObject, position);
    }
    else if (node.valueObject === null || node.value === null) {
        return wrappedNull(position);
    }
    return wrappedScalar(String, "string", node.value, position);
};
visitorByNodeKind[yaml_ast_parser_1.Kind.SEQ] = (node, input) => {
    const items = walk(node.items, input, []);
    items[keys_1.positionKey] = calculatePosition(input, {
        start: node.startPosition,
        end: node.endPosition,
    });
    items[keys_1.kindKey] = "array";
    return items;
};
const fromYaml = (input) => walk([(0, yaml_ast_parser_1.load)(input)], input);
exports.fromYaml = fromYaml;
//# sourceMappingURL=fromYaml.js.map