import lintXml from "./xmlTemplate/linter.js";
import lintJson from "./manifestJson/linter.js";
import lintHtml from "./html/linter.js";
import lintUI5Yaml from "./yaml/linter.js";
import lintDotLibrary from "./dotLibrary/linter.js";
import lintFileTypes from "./fileTypes/linter.js";
import { taskStart } from "../utils/perf.js";
import TypeLinter from "./ui5Types/TypeLinter.js";
import LinterContext from "./LinterContext.js";
import { createReader, createResource } from "@ui5/fs/resourceFactory";
import { mergeIgnorePatterns, resolveReader } from "./linter.js";
import autofix from "../autofix/autofix.js";
import { writeFile } from "node:fs/promises";
import { transformVirtualPathToFilePath } from "../utils/virtualPathToFilePath.js";
import { getLogger } from "@ui5/logger";
import path from "node:path";
const log = getLogger("linter:lintWorkspace");
export default async function lintWorkspace(workspace, filePathsWorkspace, options, config, patternsMatch, sharedLanguageService) {
    let context = await runLintWorkspace(workspace, filePathsWorkspace, options, config, patternsMatch, sharedLanguageService);
    if (options.fix) {
        const rawLintResults = context.generateRawLintResults();
        const autofixResources = new Map();
        for (const { filePath, rawMessages } of rawLintResults) {
            const resource = await workspace.byPath(filePath);
            if (!resource) {
                // This might happen in case a file with an existing source map was linted and the referenced
                // file is not available in the workspace.
                log.verbose(`Resource '${filePath}' not found. Skipping autofix for this file.`);
                continue;
            }
            autofixResources.set(filePath, {
                resource,
                messages: rawMessages,
            });
        }
        log.verbose(`Autofixing ${autofixResources.size} files...`);
        const doneAutofix = taskStart("Autofix");
        const autofixResult = await autofix({
            rootDir: options.rootDir,
            namespace: options.namespace,
            resources: autofixResources,
            context,
        });
        doneAutofix();
        log.verbose(`Autofix provided solutions for ${autofixResult.size} files`);
        if (autofixResult.size > 0) {
            for (const [filePath, content] of autofixResult.entries()) {
                const newResource = createResource({
                    path: filePath,
                    string: content,
                });
                await workspace.write(newResource);
                await filePathsWorkspace.write(newResource);
            }
            log.verbose("Linting again after applying fixes...");
            // Run lint again after fixes are applied, but without fixing
            const optionsAfterFix = {
                ...options,
                fix: false,
            };
            context = await runLintWorkspace(workspace, filePathsWorkspace, optionsAfterFix, config, patternsMatch, sharedLanguageService);
            // Update fixed files on the filesystem
            if (process.env.UI5LINT_FIX_DRY_RUN) {
                log.verbose("UI5LINT_FIX_DRY_RUN: Not updating files on the filesystem");
            }
            else {
                const autofixFiles = Array.from(autofixResult.entries());
                await Promise.all(autofixFiles.map(async ([filePath, content]) => {
                    const realFilePath = path.join(optionsAfterFix.rootDir, transformVirtualPathToFilePath(filePath, optionsAfterFix));
                    log.verbose(`Writing fixed file '${filePath}' to '${realFilePath}'`);
                    await writeFile(realFilePath, content);
                }));
            }
        }
    }
    return context.generateLintResults();
}
async function runLintWorkspace(workspace, filePathsWorkspace, options, config, patternsMatch, sharedLanguageService) {
    const done = taskStart("Linting Workspace");
    const fsToVirtualPathOptions = {
        relFsBasePath: options.relFsBasePath ?? "",
        virBasePath: options.virBasePath ?? "/",
        relFsBasePathTest: options.relFsBasePathTest,
        virBasePathTest: options.virBasePathTest,
    };
    const context = new LinterContext(options);
    let reader = resolveReader({
        patterns: options.filePatterns ?? config.files ?? [],
        fsToVirtualPathOptions,
        resourceReader: createReader({
            fsBasePath: options.rootDir,
            virBasePath: "/",
        }),
        inverseResult: true,
        patternsMatch,
    });
    reader = resolveReader({
        patterns: mergeIgnorePatterns(options, config),
        resourceReader: reader,
        patternsMatch,
        fsToVirtualPathOptions,
    });
    context.setRootReader(reader);
    const params = {
        workspace, filePathsWorkspace, context,
    };
    await Promise.all([
        lintXml(params),
        lintJson(params),
        lintHtml(params),
        lintUI5Yaml(params),
        lintDotLibrary(params),
        lintFileTypes(params),
    ]);
    const typeLinter = new TypeLinter(params, sharedLanguageService);
    await typeLinter.lint();
    done();
    return context;
}
//# sourceMappingURL=lintWorkspace.js.map