'use strict';

var ts = require('./typescript');
var minimatch = require('minimatch');
var fs$1 = require('fs');
var fsp = require('fs/promises');
var os = require('os');
var path$2 = require('path');
var tinyglobby = require('tinyglobby');

function _interopNamespaceCompat(e) {
    if (e && typeof e === 'object' && 'default' in e) return e;
    var n = Object.create(null);
    if (e) {
        Object.keys(e).forEach(function (k) {
            if (k !== 'default') {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(n, k, d.get ? d : {
                    enumerable: true,
                    get: function () { return e[k]; }
                });
            }
        });
    }
    n.default = e;
    return Object.freeze(n);
}

var ts__namespace = /*#__PURE__*/_interopNamespaceCompat(ts);
var minimatch__namespace = /*#__PURE__*/_interopNamespaceCompat(minimatch);
var fs__namespace = /*#__PURE__*/_interopNamespaceCompat(fs$1);
var fsp__namespace = /*#__PURE__*/_interopNamespaceCompat(fsp);
var os__namespace = /*#__PURE__*/_interopNamespaceCompat(os);
var path__namespace = /*#__PURE__*/_interopNamespaceCompat(path$2);

class KeyValueCache {
    #cacheItems = new Map();
    getSize() {
        return this.#cacheItems.size;
    }
    getValues() {
        return this.#cacheItems.values();
    }
    getValuesAsArray() {
        return Array.from(this.getValues());
    }
    getKeys() {
        return this.#cacheItems.keys();
    }
    getEntries() {
        return this.#cacheItems.entries();
    }
    getOrCreate(key, createFunc) {
        let item = this.get(key);
        if (item == null) {
            item = createFunc();
            this.set(key, item);
        }
        return item;
    }
    has(key) {
        return this.#cacheItems.has(key);
    }
    get(key) {
        return this.#cacheItems.get(key);
    }
    set(key, value) {
        this.#cacheItems.set(key, value);
    }
    replaceKey(key, newKey) {
        if (!this.#cacheItems.has(key))
            throw new Error("Key not found.");
        const value = this.#cacheItems.get(key);
        this.#cacheItems.delete(key);
        this.#cacheItems.set(newKey, value);
    }
    removeByKey(key) {
        this.#cacheItems.delete(key);
    }
    clear() {
        this.#cacheItems.clear();
    }
}

class ComparerToStoredComparer {
    #comparer;
    #storedValue;
    constructor(comparer, storedValue) {
        this.#comparer = comparer;
        this.#storedValue = storedValue;
    }
    compareTo(value) {
        return this.#comparer.compareTo(this.#storedValue, value);
    }
}

class LocaleStringComparer {
    static instance = new LocaleStringComparer();
    compareTo(a, b) {
        const comparisonResult = a.localeCompare(b, "en-us-u-kf-upper");
        if (comparisonResult < 0)
            return -1;
        else if (comparisonResult === 0)
            return 0;
        return 1;
    }
}

class PropertyComparer {
    #comparer;
    #getProperty;
    constructor(getProperty, comparer) {
        this.#getProperty = getProperty;
        this.#comparer = comparer;
    }
    compareTo(a, b) {
        return this.#comparer.compareTo(this.#getProperty(a), this.#getProperty(b));
    }
}

class PropertyStoredComparer {
    #comparer;
    #getProperty;
    constructor(getProperty, comparer) {
        this.#getProperty = getProperty;
        this.#comparer = comparer;
    }
    compareTo(value) {
        return this.#comparer.compareTo(this.#getProperty(value));
    }
}

class ArrayUtils {
    constructor() {
    }
    static isReadonlyArray(a) {
        return a instanceof Array;
    }
    static isNullOrEmpty(a) {
        return !(a instanceof Array) || a.length === 0;
    }
    static getUniqueItems(a) {
        return a.filter((item, index) => a.indexOf(item) === index);
    }
    static removeFirst(a, item) {
        const index = a.indexOf(item);
        if (index === -1)
            return false;
        a.splice(index, 1);
        return true;
    }
    static removeAll(a, isMatch) {
        const removedItems = [];
        for (let i = a.length - 1; i >= 0; i--) {
            if (isMatch(a[i])) {
                removedItems.push(a[i]);
                a.splice(i, 1);
            }
        }
        return removedItems;
    }
    static *toIterator(items) {
        for (const item of items)
            yield item;
    }
    static sortByProperty(items, getProp) {
        items.sort((a, b) => getProp(a) <= getProp(b) ? -1 : 1);
        return items;
    }
    static groupBy(items, getGroup) {
        const result = [];
        const groups = {};
        for (const item of items) {
            const group = getGroup(item).toString();
            if (groups[group] == null) {
                groups[group] = [];
                result.push(groups[group]);
            }
            groups[group].push(item);
        }
        return result;
    }
    static binaryInsertWithOverwrite(items, newItem, comparer) {
        let top = items.length - 1;
        let bottom = 0;
        while (bottom <= top) {
            const mid = Math.floor((top + bottom) / 2);
            if (comparer.compareTo(newItem, items[mid]) < 0)
                top = mid - 1;
            else
                bottom = mid + 1;
        }
        if (items[top] != null && comparer.compareTo(newItem, items[top]) === 0)
            items[top] = newItem;
        else
            items.splice(top + 1, 0, newItem);
    }
    static binarySearch(items, storedComparer) {
        let top = items.length - 1;
        let bottom = 0;
        while (bottom <= top) {
            const mid = Math.floor((top + bottom) / 2);
            const comparisonResult = storedComparer.compareTo(items[mid]);
            if (comparisonResult === 0)
                return mid;
            else if (comparisonResult < 0)
                top = mid - 1;
            else
                bottom = mid + 1;
        }
        return -1;
    }
    static containsSubArray(items, subArray) {
        let findIndex = 0;
        for (const item of items) {
            if (subArray[findIndex] === item) {
                findIndex++;
                if (findIndex === subArray.length)
                    return true;
            }
            else {
                findIndex = 0;
            }
        }
        return false;
    }
}

function deepClone(objToClone) {
    return clone(objToClone);
    function clone(obj) {
        const newObj = Object.create(obj.constructor.prototype);
        for (const propName of Object.keys(obj))
            newObj[propName] = cloneItem(obj[propName]);
        return newObj;
    }
    function cloneArray(array) {
        return array.map(cloneItem);
    }
    function cloneItem(item) {
        if (item instanceof Array)
            return cloneArray(item);
        else if (typeof item === "object")
            return item === null ? item : clone(item);
        return item;
    }
}

class EventContainer {
    #subscriptions = [];
    subscribe(subscription) {
        const index = this.#getIndex(subscription);
        if (index === -1)
            this.#subscriptions.push(subscription);
    }
    unsubscribe(subscription) {
        const index = this.#getIndex(subscription);
        if (index >= 0)
            this.#subscriptions.splice(index, 1);
    }
    fire(arg) {
        for (const subscription of this.#subscriptions)
            subscription(arg);
    }
    #getIndex(subscription) {
        return this.#subscriptions.indexOf(subscription);
    }
}

class IterableUtils {
    static find(items, condition) {
        for (const item of items) {
            if (condition(item))
                return item;
        }
        return undefined;
    }
}

function nameof(key1, key2) {
    return key2 ?? key1;
}

class ObjectUtils {
    constructor() {
    }
    static clone(obj) {
        if (obj == null)
            return undefined;
        if (obj instanceof Array)
            return cloneArray(obj);
        return Object.assign({}, obj);
        function cloneArray(a) {
            return a.map(item => ObjectUtils.clone(item));
        }
    }
}

function matchFiles(path, extensions, excludes, includes, useCaseSensitiveFileNames, currentDirectory, depth, getEntries, realpath, directoryExists) {
    return ts__namespace.matchFiles.apply(this, arguments);
}
function getFileMatcherPatterns(path, excludes, includes, useCaseSensitiveFileNames, currentDirectory) {
    return ts__namespace.getFileMatcherPatterns.apply(this, arguments);
}
function getEmitModuleResolutionKind(compilerOptions) {
    return ts__namespace.getEmitModuleResolutionKind.apply(this, arguments);
}

function getSyntaxKindName(kind) {
    return getKindCache()[kind];
}
let kindCache = undefined;
function getKindCache() {
    if (kindCache != null)
        return kindCache;
    kindCache = {};
    for (const name of Object.keys(ts__namespace.SyntaxKind).filter(k => isNaN(parseInt(k, 10)))) {
        const value = ts__namespace.SyntaxKind[name];
        if (kindCache[value] == null)
            kindCache[value] = name;
    }
    return kindCache;
}

exports.errors = void 0;
(function (errors) {
    class BaseError extends Error {
        constructor(message, node) {
            const nodeLocation = node && getPrettyNodeLocation(node);
            const messageWithLocation = nodeLocation ? `${message}\n\n${nodeLocation}` : message;
            super(messageWithLocation);
            this.message = messageWithLocation;
        }
    }
    errors.BaseError = BaseError;
    class ArgumentError extends BaseError {
        constructor(argName, message, node) {
            super(`Argument Error (${argName}): ${message}`, node);
        }
    }
    errors.ArgumentError = ArgumentError;
    class ArgumentNullOrWhitespaceError extends ArgumentError {
        constructor(argName, node) {
            super(argName, "Cannot be null or whitespace.", node);
        }
    }
    errors.ArgumentNullOrWhitespaceError = ArgumentNullOrWhitespaceError;
    class ArgumentOutOfRangeError extends ArgumentError {
        constructor(argName, value, range, node) {
            super(argName, `Range is ${range[0]} to ${range[1]}, but ${value} was provided.`, node);
        }
    }
    errors.ArgumentOutOfRangeError = ArgumentOutOfRangeError;
    class ArgumentTypeError extends ArgumentError {
        constructor(argName, expectedType, actualType, node) {
            super(argName, `Expected type '${expectedType}', but was '${actualType}'.`, node);
        }
    }
    errors.ArgumentTypeError = ArgumentTypeError;
    class PathNotFoundError extends BaseError {
        path;
        constructor(path, prefix = "Path") {
            super(`${prefix} not found: ${path}`);
            this.path = path;
        }
        code = "ENOENT";
    }
    errors.PathNotFoundError = PathNotFoundError;
    class DirectoryNotFoundError extends PathNotFoundError {
        constructor(dirPath) {
            super(dirPath, "Directory");
        }
    }
    errors.DirectoryNotFoundError = DirectoryNotFoundError;
    class FileNotFoundError extends PathNotFoundError {
        constructor(filePath) {
            super(filePath, "File");
        }
    }
    errors.FileNotFoundError = FileNotFoundError;
    class InvalidOperationError extends BaseError {
        constructor(message, node) {
            super(message, node);
        }
    }
    errors.InvalidOperationError = InvalidOperationError;
    class NotImplementedError extends BaseError {
        constructor(message = "Not implemented.", node) {
            super(message, node);
        }
    }
    errors.NotImplementedError = NotImplementedError;
    class NotSupportedError extends BaseError {
        constructor(message) {
            super(message);
        }
    }
    errors.NotSupportedError = NotSupportedError;
    function throwIfNotType(value, expectedType, argName) {
        if (typeof value !== expectedType)
            throw new ArgumentTypeError(argName, expectedType, typeof value);
    }
    errors.throwIfNotType = throwIfNotType;
    function throwIfNotString(value, argName) {
        if (typeof value !== "string")
            throw new ArgumentTypeError(argName, "string", typeof value);
    }
    errors.throwIfNotString = throwIfNotString;
    function throwIfWhitespaceOrNotString(value, argName) {
        throwIfNotString(value, argName);
        if (value.trim().length === 0)
            throw new ArgumentNullOrWhitespaceError(argName);
    }
    errors.throwIfWhitespaceOrNotString = throwIfWhitespaceOrNotString;
    function throwIfOutOfRange(value, range, argName) {
        if (value < range[0] || value > range[1])
            throw new ArgumentOutOfRangeError(argName, value, range);
    }
    errors.throwIfOutOfRange = throwIfOutOfRange;
    function throwIfRangeOutOfRange(actualRange, range, argName) {
        if (actualRange[0] > actualRange[1])
            throw new ArgumentError(argName, `The start of a range must not be greater than the end: [${actualRange[0]}, ${actualRange[1]}]`);
        throwIfOutOfRange(actualRange[0], range, argName);
        throwIfOutOfRange(actualRange[1], range, argName);
    }
    errors.throwIfRangeOutOfRange = throwIfRangeOutOfRange;
    function throwNotImplementedForSyntaxKindError(kind, node) {
        throw new NotImplementedError(`Not implemented feature for syntax kind '${getSyntaxKindName(kind)}'.`, node);
    }
    errors.throwNotImplementedForSyntaxKindError = throwNotImplementedForSyntaxKindError;
    function throwIfNegative(value, argName) {
        if (value < 0)
            throw new ArgumentError(argName, "Expected a non-negative value.");
    }
    errors.throwIfNegative = throwIfNegative;
    function throwIfNullOrUndefined(value, errorMessage, node) {
        if (value == null)
            throw new InvalidOperationError(typeof errorMessage === "string" ? errorMessage : errorMessage(), node);
        return value;
    }
    errors.throwIfNullOrUndefined = throwIfNullOrUndefined;
    function throwNotImplementedForNeverValueError(value, sourceNode) {
        const node = value;
        if (node != null && typeof node.kind === "number")
            return throwNotImplementedForSyntaxKindError(node.kind, sourceNode);
        throw new NotImplementedError(`Not implemented value: ${JSON.stringify(value)}`, sourceNode);
    }
    errors.throwNotImplementedForNeverValueError = throwNotImplementedForNeverValueError;
    function throwIfNotEqual(actual, expected, description) {
        if (actual !== expected)
            throw new InvalidOperationError(`Expected ${actual} to equal ${expected}. ${description}`);
    }
    errors.throwIfNotEqual = throwIfNotEqual;
    function throwIfTrue(value, errorMessage) {
        if (value === true)
            throw new InvalidOperationError(errorMessage);
    }
    errors.throwIfTrue = throwIfTrue;
})(exports.errors || (exports.errors = {}));
function getPrettyNodeLocation(node) {
    const source = getSourceLocation(node);
    if (!source)
        return undefined;
    return `${source.filePath}:${source.loc.line}:${source.loc.character}\n`
        + `> ${source.loc.line} | ${source.lineText}`;
}
function getSourceLocation(node) {
    if (!isNode(node))
        return;
    const sourceFile = node.getSourceFile();
    const sourceCode = sourceFile.getFullText();
    const start = node.getStart();
    const lineStart = sourceCode.lastIndexOf("\n", start) + 1;
    const nextNewLinePos = sourceCode.indexOf("\n", start);
    const lineEnd = nextNewLinePos === -1 ? sourceCode.length : nextNewLinePos;
    const textStart = (start - lineStart > 40) ? start - 37 : lineStart;
    const textEnd = (lineEnd - textStart > 80) ? textStart + 77 : lineEnd;
    let lineText = "";
    if (textStart !== lineStart)
        lineText += "...";
    lineText += sourceCode.substring(textStart, textEnd);
    if (textEnd !== lineEnd)
        lineText += "...";
    return {
        filePath: sourceFile.getFilePath(),
        loc: {
            line: StringUtils.getLineNumberAtPos(sourceCode, start),
            character: start - lineStart + 1,
        },
        lineText,
    };
}
function isNode(node) {
    return typeof node === "object" && node !== null && ("getSourceFile" in node) && ("getStart" in node);
}

const CharCodes = {
    ASTERISK: "*".charCodeAt(0),
    NEWLINE: "\n".charCodeAt(0),
    CARRIAGE_RETURN: "\r".charCodeAt(0),
    SPACE: " ".charCodeAt(0),
    TAB: "\t".charCodeAt(0),
    CLOSE_BRACE: "}".charCodeAt(0),
};

const regExWhitespaceSet = new Set([" ", "\f", "\n", "\r", "\t", "\v", "\u00A0", "\u2028", "\u2029"].map(c => c.charCodeAt(0)));
class StringUtils {
    constructor() {
    }
    static isWhitespaceCharCode(charCode) {
        return regExWhitespaceSet.has(charCode);
    }
    static isSpaces(text) {
        if (text == null || text.length === 0)
            return false;
        for (let i = 0; i < text.length; i++) {
            if (text.charCodeAt(i) !== CharCodes.SPACE)
                return false;
        }
        return true;
    }
    static hasBom(text) {
        return text.charCodeAt(0) === 0xFEFF;
    }
    static stripBom(text) {
        if (StringUtils.hasBom(text))
            return text.slice(1);
        return text;
    }
    static stripQuotes(text) {
        if (StringUtils.isQuoted(text))
            return text.substring(1, text.length - 1);
        return text;
    }
    static isQuoted(text) {
        return text.startsWith("'") && text.endsWith("'") || text.startsWith("\"") && text.endsWith("\"");
    }
    static isNullOrWhitespace(str) {
        return typeof str !== "string" || StringUtils.isWhitespace(str);
    }
    static isNullOrEmpty(str) {
        return typeof str !== "string" || str.length === 0;
    }
    static isWhitespace(text) {
        if (text == null)
            return true;
        for (let i = 0; i < text.length; i++) {
            if (!StringUtils.isWhitespaceCharCode(text.charCodeAt(i)))
                return false;
        }
        return true;
    }
    static startsWithNewLine(str) {
        if (str == null)
            return false;
        return str.charCodeAt(0) === CharCodes.NEWLINE || str.charCodeAt(0) === CharCodes.CARRIAGE_RETURN && str.charCodeAt(1) === CharCodes.NEWLINE;
    }
    static endsWithNewLine(str) {
        if (str == null)
            return false;
        return str.charCodeAt(str.length - 1) === CharCodes.NEWLINE;
    }
    static insertAtLastNonWhitespace(str, insertText) {
        let i = str.length;
        while (i > 0 && StringUtils.isWhitespaceCharCode(str.charCodeAt(i - 1)))
            i--;
        return str.substring(0, i) + insertText + str.substring(i);
    }
    static getLineNumberAtPos(str, pos) {
        exports.errors.throwIfOutOfRange(pos, [0, str.length], "pos");
        let count = 0;
        for (let i = 0; i < pos; i++) {
            if (str.charCodeAt(i) === CharCodes.NEWLINE)
                count++;
        }
        return count + 1;
    }
    static getLengthFromLineStartAtPos(str, pos) {
        exports.errors.throwIfOutOfRange(pos, [0, str.length], "pos");
        return pos - StringUtils.getLineStartFromPos(str, pos);
    }
    static getLineStartFromPos(str, pos) {
        exports.errors.throwIfOutOfRange(pos, [0, str.length], "pos");
        while (pos > 0) {
            const previousCharCode = str.charCodeAt(pos - 1);
            if (previousCharCode === CharCodes.NEWLINE || previousCharCode === CharCodes.CARRIAGE_RETURN)
                break;
            pos--;
        }
        return pos;
    }
    static getLineEndFromPos(str, pos) {
        exports.errors.throwIfOutOfRange(pos, [0, str.length], "pos");
        while (pos < str.length) {
            const currentChar = str.charCodeAt(pos);
            if (currentChar === CharCodes.NEWLINE || currentChar === CharCodes.CARRIAGE_RETURN)
                break;
            pos++;
        }
        return pos;
    }
    static escapeForWithinString(str, quoteKind) {
        return StringUtils.escapeChar(str, quoteKind).replace(/(\r?\n)/g, "\\$1");
    }
    static escapeChar(str, char) {
        if (char.length !== 1)
            throw new exports.errors.InvalidOperationError(`Specified char must be one character long.`);
        let result = "";
        for (const currentChar of str) {
            if (currentChar === char)
                result += "\\";
            result += currentChar;
        }
        return result;
    }
    static removeIndentation(str, opts) {
        const { isInStringAtPos, indentSizeInSpaces } = opts;
        const positions = [];
        let minIndentWidth;
        analyze();
        return buildString();
        function analyze() {
            let isAtStartOfLine = str.charCodeAt(0) === CharCodes.SPACE || str.charCodeAt(0) === CharCodes.TAB;
            for (let i = 0; i < str.length; i++) {
                if (!isAtStartOfLine) {
                    if (str.charCodeAt(i) === CharCodes.NEWLINE && !isInStringAtPos(i + 1))
                        isAtStartOfLine = true;
                    continue;
                }
                let startPosition = i;
                let spacesCount = 0;
                let tabsCount = 0;
                while (true) {
                    let charCode = str.charCodeAt(i);
                    if (charCode === CharCodes.SPACE)
                        spacesCount++;
                    else if (charCode === CharCodes.TAB)
                        tabsCount++;
                    else if (charCode === CharCodes.NEWLINE || charCode === CharCodes.CARRIAGE_RETURN && str.charCodeAt(i + 1) === CharCodes.NEWLINE) {
                        spacesCount = 0;
                        tabsCount = 0;
                        positions.push([startPosition, i]);
                        if (charCode === CharCodes.CARRIAGE_RETURN) {
                            startPosition = i + 2;
                            i++;
                        }
                        else {
                            startPosition = i + 1;
                        }
                    }
                    else if (charCode == null)
                        break;
                    else {
                        const indentWidth = Math.ceil(spacesCount / indentSizeInSpaces) * indentSizeInSpaces + tabsCount * indentSizeInSpaces;
                        if (minIndentWidth == null || indentWidth < minIndentWidth)
                            minIndentWidth = indentWidth;
                        positions.push([startPosition, i]);
                        isAtStartOfLine = false;
                        break;
                    }
                    i++;
                }
            }
        }
        function buildString() {
            if (positions.length === 0)
                return str;
            if (minIndentWidth == null || minIndentWidth === 0)
                return str;
            const deindentWidth = minIndentWidth;
            let result = "";
            result += str.substring(0, positions[0][0]);
            for (let i = 0; i < positions.length; i++) {
                const [startPosition, endPosition] = positions[i];
                let indentCount = 0;
                let pos;
                for (pos = startPosition; pos < endPosition; pos++) {
                    if (indentCount >= deindentWidth)
                        break;
                    if (str.charCodeAt(pos) === CharCodes.SPACE)
                        indentCount++;
                    else if (str.charCodeAt(pos) === CharCodes.TAB)
                        indentCount += indentSizeInSpaces;
                }
                result += str.substring(pos, positions[i + 1]?.[0] ?? str.length);
            }
            return result;
        }
    }
    static indent(str, times, options) {
        if (times === 0)
            return str;
        const { indentText, indentSizeInSpaces, isInStringAtPos } = options;
        const fullIndentationText = times > 0 ? indentText.repeat(times) : undefined;
        const totalIndentSpaces = Math.abs(times * indentSizeInSpaces);
        let result = "";
        let lineStart = 0;
        let lineEnd = 0;
        for (let i = 0; i < str.length; i++) {
            lineStart = i;
            while (i < str.length && str.charCodeAt(i) !== CharCodes.NEWLINE)
                i++;
            lineEnd = i === str.length ? i : i + 1;
            appendLine();
        }
        return result;
        function appendLine() {
            if (isInStringAtPos(lineStart))
                result += str.substring(lineStart, lineEnd);
            else if (times > 0)
                result += fullIndentationText + str.substring(lineStart, lineEnd);
            else {
                let start = lineStart;
                let indentSpaces = 0;
                for (start = lineStart; start < str.length; start++) {
                    if (indentSpaces >= totalIndentSpaces)
                        break;
                    if (str.charCodeAt(start) === CharCodes.SPACE)
                        indentSpaces++;
                    else if (str.charCodeAt(start) === CharCodes.TAB)
                        indentSpaces += indentSizeInSpaces;
                    else
                        break;
                }
                result += str.substring(start, lineEnd);
            }
        }
    }
}

class SortedKeyValueArray {
    #array = [];
    #getKey;
    #comparer;
    constructor(getKey, comparer) {
        this.#getKey = getKey;
        this.#comparer = comparer;
    }
    set(value) {
        ArrayUtils.binaryInsertWithOverwrite(this.#array, value, new PropertyComparer(this.#getKey, this.#comparer));
    }
    removeByValue(value) {
        this.removeByKey(this.#getKey(value));
    }
    removeByKey(key) {
        const storedComparer = new ComparerToStoredComparer(this.#comparer, key);
        const index = ArrayUtils.binarySearch(this.#array, new PropertyStoredComparer(this.#getKey, storedComparer));
        if (index >= 0)
            this.#array.splice(index, 1);
    }
    getArrayCopy() {
        return [...this.#array];
    }
    hasItems() {
        return this.#array.length > 0;
    }
    *entries() {
        yield* this.#array;
    }
}

class WeakCache {
    #cacheItems = new WeakMap();
    getOrCreate(key, createFunc) {
        let item = this.get(key);
        if (item == null) {
            item = createFunc();
            this.set(key, item);
        }
        return item;
    }
    has(key) {
        return this.#cacheItems.has(key);
    }
    get(key) {
        return this.#cacheItems.get(key);
    }
    set(key, value) {
        this.#cacheItems.set(key, value);
    }
    removeByKey(key) {
        this.#cacheItems.delete(key);
    }
}

function createCompilerSourceFile(filePath, scriptSnapshot, optionsOrScriptTarget, version, setParentNodes, scriptKind) {
    return ts__namespace.createLanguageServiceSourceFile(filePath, scriptSnapshot, optionsOrScriptTarget ?? ts__namespace.ScriptTarget.Latest, version, setParentNodes, scriptKind);
}

function createDocumentCache(files) {
    const cache = new InternalDocumentCache();
    cache._addFiles(files);
    return cache;
}
class FileSystemDocumentCache {
    #documentCache;
    #absoluteToOriginalPath = new Map();
    constructor(fileSystem, documentCache) {
        for (const filePath of documentCache._getFilePaths())
            this.#absoluteToOriginalPath.set(fileSystem.getStandardizedAbsolutePath(filePath), filePath);
        this.#documentCache = documentCache;
    }
    getDocumentIfMatch(filePath, scriptSnapshot, scriptTarget, scriptKind) {
        const originalFilePath = this.#absoluteToOriginalPath.get(filePath);
        if (originalFilePath == null)
            return;
        return this.#documentCache._getDocumentIfMatch(originalFilePath, filePath, scriptSnapshot, scriptTarget, scriptKind);
    }
}
class InternalDocumentCache {
    __documentCacheBrand;
    #fileTexts = new Map();
    #documents = new Map();
    _addFiles(files) {
        for (const file of files)
            this.#fileTexts.set(file.fileName, file.text);
    }
    _getFilePaths() {
        return this.#fileTexts.keys();
    }
    _getCacheForFileSystem(fileSystem) {
        return new FileSystemDocumentCache(fileSystem, this);
    }
    _getDocumentIfMatch(filePath, absoluteFilePath, scriptSnapshot, scriptTarget, scriptKind) {
        const fileText = this.#fileTexts.get(filePath);
        if (fileText == null)
            return undefined;
        if (fileText !== scriptSnapshot.getText(0, scriptSnapshot.getLength()))
            return undefined;
        return this.#getDocument(filePath, absoluteFilePath, scriptSnapshot, scriptTarget, scriptKind);
    }
    #getDocument(filePath, absoluteFilePath, scriptSnapshot, scriptTarget, scriptKind) {
        const documentKey = this.#getKey(filePath, scriptTarget, scriptKind);
        let document = this.#documents.get(documentKey);
        if (document == null) {
            document = createCompilerSourceFile(absoluteFilePath, scriptSnapshot, scriptTarget, "-1", false, scriptKind);
            this.#documents.set(documentKey, document);
        }
        document = deepClone(document);
        document.fileName = absoluteFilePath;
        return document;
    }
    #getKey(filePath, scriptTarget, scriptKind) {
        return (filePath + (scriptTarget?.toString() ?? "-1") + (scriptKind?.toString() ?? "-1"));
    }
}

const libFiles = [{
        fileName: "lib.es2019.object.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.iterable\" />\ninterface ObjectConstructor{fromEntries<T=any>(entries:Iterable<readonly[PropertyKey,T]>):{[k:string]:T;};fromEntries(entries:Iterable<readonly any[]>):any;}"
    }, {
        fileName: "lib.dom.iterable.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface AbortSignal{any(signals:Iterable<AbortSignal>):AbortSignal;}interface AudioParam{setValueCurveAtTime(values:Iterable<number>,startTime:number,duration:number):AudioParam;}interface AudioParamMap extends ReadonlyMap<string,AudioParam>{}interface BaseAudioContext{createIIRFilter(feedforward:Iterable<number>,feedback:Iterable<number>):IIRFilterNode;createPeriodicWave(real:Iterable<number>,imag:Iterable<number>,constraints?:PeriodicWaveConstraints):PeriodicWave;}interface CSSKeyframesRule{[Symbol.iterator]():ArrayIterator<CSSKeyframeRule>;}interface CSSNumericArray{[Symbol.iterator]():ArrayIterator<CSSNumericValue>;entries():ArrayIterator<[number,CSSNumericValue]>;keys():ArrayIterator<number>;values():ArrayIterator<CSSNumericValue>;}interface CSSRuleList{[Symbol.iterator]():ArrayIterator<CSSRule>;}interface CSSStyleDeclaration{[Symbol.iterator]():ArrayIterator<string>;}interface CSSTransformValue{[Symbol.iterator]():ArrayIterator<CSSTransformComponent>;entries():ArrayIterator<[number,CSSTransformComponent]>;keys():ArrayIterator<number>;values():ArrayIterator<CSSTransformComponent>;}interface CSSUnparsedValue{[Symbol.iterator]():ArrayIterator<CSSUnparsedSegment>;entries():ArrayIterator<[number,CSSUnparsedSegment]>;keys():ArrayIterator<number>;values():ArrayIterator<CSSUnparsedSegment>;}interface Cache{addAll(requests:Iterable<RequestInfo>):Promise<void>;}interface CanvasPath{roundRect(x:number,y:number,w:number,h:number,radii?:number|DOMPointInit|Iterable<number|DOMPointInit>):void;}interface CanvasPathDrawingStyles{setLineDash(segments:Iterable<number>):void;}interface CustomStateSet extends Set<string>{}interface DOMRectList{[Symbol.iterator]():ArrayIterator<DOMRect>;}interface DOMStringList{[Symbol.iterator]():ArrayIterator<string>;}interface DOMTokenList{[Symbol.iterator]():ArrayIterator<string>;entries():ArrayIterator<[number,string]>;keys():ArrayIterator<number>;values():ArrayIterator<string>;}interface DataTransferItemList{[Symbol.iterator]():ArrayIterator<DataTransferItem>;}interface EventCounts extends ReadonlyMap<string,number>{}interface FileList{[Symbol.iterator]():ArrayIterator<File>;}interface FontFaceSet extends Set<FontFace>{}interface FormDataIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():FormDataIterator<T>;}interface FormData{[Symbol.iterator]():FormDataIterator<[string,FormDataEntryValue]>;entries():FormDataIterator<[string,FormDataEntryValue]>;keys():FormDataIterator<string>;values():FormDataIterator<FormDataEntryValue>;}interface HTMLAllCollection{[Symbol.iterator]():ArrayIterator<Element>;}interface HTMLCollectionBase{[Symbol.iterator]():ArrayIterator<Element>;}interface HTMLCollectionOf<T extends Element>{[Symbol.iterator]():ArrayIterator<T>;}interface HTMLFormElement{[Symbol.iterator]():ArrayIterator<Element>;}interface HTMLSelectElement{[Symbol.iterator]():ArrayIterator<HTMLOptionElement>;}interface HeadersIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():HeadersIterator<T>;}interface Headers{[Symbol.iterator]():HeadersIterator<[string,string]>;entries():HeadersIterator<[string,string]>;keys():HeadersIterator<string>;values():HeadersIterator<string>;}interface Highlight extends Set<AbstractRange>{}interface HighlightRegistry extends Map<string,Highlight>{}interface IDBDatabase{transaction(storeNames:string|Iterable<string>,mode?:IDBTransactionMode,options?:IDBTransactionOptions):IDBTransaction;}interface IDBObjectStore{createIndex(name:string,keyPath:string|Iterable<string>,options?:IDBIndexParameters):IDBIndex;}interface MIDIInputMap extends ReadonlyMap<string,MIDIInput>{}interface MIDIOutput{send(data:Iterable<number>,timestamp?:DOMHighResTimeStamp):void;}interface MIDIOutputMap extends ReadonlyMap<string,MIDIOutput>{}interface MediaKeyStatusMapIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():MediaKeyStatusMapIterator<T>;}interface MediaKeyStatusMap{[Symbol.iterator]():MediaKeyStatusMapIterator<[BufferSource,MediaKeyStatus]>;entries():MediaKeyStatusMapIterator<[BufferSource,MediaKeyStatus]>;keys():MediaKeyStatusMapIterator<BufferSource>;values():MediaKeyStatusMapIterator<MediaKeyStatus>;}interface MediaList{[Symbol.iterator]():ArrayIterator<string>;}interface MessageEvent<T=any>{initMessageEvent(type:string,bubbles?:boolean,cancelable?:boolean,data?:any,origin?:string,lastEventId?:string,source?:MessageEventSource|null,ports?:Iterable<MessagePort>):void;}interface MimeTypeArray{[Symbol.iterator]():ArrayIterator<MimeType>;}interface NamedNodeMap{[Symbol.iterator]():ArrayIterator<Attr>;}interface Navigator{requestMediaKeySystemAccess(keySystem:string,supportedConfigurations:Iterable<MediaKeySystemConfiguration>):Promise<MediaKeySystemAccess>;vibrate(pattern:Iterable<number>):boolean;}interface NodeList{[Symbol.iterator]():ArrayIterator<Node>;entries():ArrayIterator<[number,Node]>;keys():ArrayIterator<number>;values():ArrayIterator<Node>;}interface NodeListOf<TNode extends Node>{[Symbol.iterator]():ArrayIterator<TNode>;entries():ArrayIterator<[number,TNode]>;keys():ArrayIterator<number>;values():ArrayIterator<TNode>;}interface Plugin{[Symbol.iterator]():ArrayIterator<MimeType>;}interface PluginArray{[Symbol.iterator]():ArrayIterator<Plugin>;}interface RTCRtpTransceiver{setCodecPreferences(codecs:Iterable<RTCRtpCodec>):void;}interface RTCStatsReport extends ReadonlyMap<string,any>{}interface SVGLengthList{[Symbol.iterator]():ArrayIterator<SVGLength>;}interface SVGNumberList{[Symbol.iterator]():ArrayIterator<SVGNumber>;}interface SVGPointList{[Symbol.iterator]():ArrayIterator<DOMPoint>;}interface SVGStringList{[Symbol.iterator]():ArrayIterator<string>;}interface SVGTransformList{[Symbol.iterator]():ArrayIterator<SVGTransform>;}interface SourceBufferList{[Symbol.iterator]():ArrayIterator<SourceBuffer>;}interface SpeechRecognitionResult{[Symbol.iterator]():ArrayIterator<SpeechRecognitionAlternative>;}interface SpeechRecognitionResultList{[Symbol.iterator]():ArrayIterator<SpeechRecognitionResult>;}interface StylePropertyMapReadOnlyIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():StylePropertyMapReadOnlyIterator<T>;}interface StylePropertyMapReadOnly{[Symbol.iterator]():StylePropertyMapReadOnlyIterator<[string,Iterable<CSSStyleValue>]>;entries():StylePropertyMapReadOnlyIterator<[string,Iterable<CSSStyleValue>]>;keys():StylePropertyMapReadOnlyIterator<string>;values():StylePropertyMapReadOnlyIterator<Iterable<CSSStyleValue>>;}interface StyleSheetList{[Symbol.iterator]():ArrayIterator<CSSStyleSheet>;}interface SubtleCrypto{deriveKey(algorithm:AlgorithmIdentifier|EcdhKeyDeriveParams|HkdfParams|Pbkdf2Params,baseKey:CryptoKey,derivedKeyType:AlgorithmIdentifier|AesDerivedKeyParams|HmacImportParams|HkdfParams|Pbkdf2Params,extractable:boolean,keyUsages:Iterable<KeyUsage>):Promise<CryptoKey>;generateKey(algorithm:\"Ed25519\",extractable:boolean,keyUsages:ReadonlyArray<\"sign\"|\"verify\">):Promise<CryptoKeyPair>;generateKey(algorithm:RsaHashedKeyGenParams|EcKeyGenParams,extractable:boolean,keyUsages:ReadonlyArray<KeyUsage>):Promise<CryptoKeyPair>;generateKey(algorithm:AesKeyGenParams|HmacKeyGenParams|Pbkdf2Params,extractable:boolean,keyUsages:ReadonlyArray<KeyUsage>):Promise<CryptoKey>;generateKey(algorithm:AlgorithmIdentifier,extractable:boolean,keyUsages:Iterable<KeyUsage>):Promise<CryptoKeyPair|CryptoKey>;importKey(format:\"jwk\",keyData:JsonWebKey,algorithm:AlgorithmIdentifier|RsaHashedImportParams|EcKeyImportParams|HmacImportParams|AesKeyAlgorithm,extractable:boolean,keyUsages:ReadonlyArray<KeyUsage>):Promise<CryptoKey>;importKey(format:Exclude<KeyFormat,\"jwk\">,keyData:BufferSource,algorithm:AlgorithmIdentifier|RsaHashedImportParams|EcKeyImportParams|HmacImportParams|AesKeyAlgorithm,extractable:boolean,keyUsages:Iterable<KeyUsage>):Promise<CryptoKey>;unwrapKey(format:KeyFormat,wrappedKey:BufferSource,unwrappingKey:CryptoKey,unwrapAlgorithm:AlgorithmIdentifier|RsaOaepParams|AesCtrParams|AesCbcParams|AesGcmParams,unwrappedKeyAlgorithm:AlgorithmIdentifier|RsaHashedImportParams|EcKeyImportParams|HmacImportParams|AesKeyAlgorithm,extractable:boolean,keyUsages:Iterable<KeyUsage>):Promise<CryptoKey>;}interface TextTrackCueList{[Symbol.iterator]():ArrayIterator<TextTrackCue>;}interface TextTrackList{[Symbol.iterator]():ArrayIterator<TextTrack>;}interface TouchList{[Symbol.iterator]():ArrayIterator<Touch>;}interface URLSearchParamsIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():URLSearchParamsIterator<T>;}interface URLSearchParams{[Symbol.iterator]():URLSearchParamsIterator<[string,string]>;entries():URLSearchParamsIterator<[string,string]>;keys():URLSearchParamsIterator<string>;values():URLSearchParamsIterator<string>;}interface WEBGL_draw_buffers{drawBuffersWEBGL(buffers:Iterable<GLenum>):void;}interface WEBGL_multi_draw{multiDrawArraysInstancedWEBGL(mode:GLenum,firstsList:Int32Array|Iterable<GLint>,firstsOffset:number,countsList:Int32Array|Iterable<GLsizei>,countsOffset:number,instanceCountsList:Int32Array|Iterable<GLsizei>,instanceCountsOffset:number,drawcount:GLsizei):void;multiDrawArraysWEBGL(mode:GLenum,firstsList:Int32Array|Iterable<GLint>,firstsOffset:number,countsList:Int32Array|Iterable<GLsizei>,countsOffset:number,drawcount:GLsizei):void;multiDrawElementsInstancedWEBGL(mode:GLenum,countsList:Int32Array|Iterable<GLsizei>,countsOffset:number,type:GLenum,offsetsList:Int32Array|Iterable<GLsizei>,offsetsOffset:number,instanceCountsList:Int32Array|Iterable<GLsizei>,instanceCountsOffset:number,drawcount:GLsizei):void;multiDrawElementsWEBGL(mode:GLenum,countsList:Int32Array|Iterable<GLsizei>,countsOffset:number,type:GLenum,offsetsList:Int32Array|Iterable<GLsizei>,offsetsOffset:number,drawcount:GLsizei):void;}interface WebGL2RenderingContextBase{clearBufferfv(buffer:GLenum,drawbuffer:GLint,values:Iterable<GLfloat>,srcOffset?:number):void;clearBufferiv(buffer:GLenum,drawbuffer:GLint,values:Iterable<GLint>,srcOffset?:number):void;clearBufferuiv(buffer:GLenum,drawbuffer:GLint,values:Iterable<GLuint>,srcOffset?:number):void;drawBuffers(buffers:Iterable<GLenum>):void;getActiveUniforms(program:WebGLProgram,uniformIndices:Iterable<GLuint>,pname:GLenum):any;getUniformIndices(program:WebGLProgram,uniformNames:Iterable<string>):Iterable<GLuint>|null;invalidateFramebuffer(target:GLenum,attachments:Iterable<GLenum>):void;invalidateSubFramebuffer(target:GLenum,attachments:Iterable<GLenum>,x:GLint,y:GLint,width:GLsizei,height:GLsizei):void;transformFeedbackVaryings(program:WebGLProgram,varyings:Iterable<string>,bufferMode:GLenum):void;uniform1uiv(location:WebGLUniformLocation|null,data:Iterable<GLuint>,srcOffset?:number,srcLength?:GLuint):void;uniform2uiv(location:WebGLUniformLocation|null,data:Iterable<GLuint>,srcOffset?:number,srcLength?:GLuint):void;uniform3uiv(location:WebGLUniformLocation|null,data:Iterable<GLuint>,srcOffset?:number,srcLength?:GLuint):void;uniform4uiv(location:WebGLUniformLocation|null,data:Iterable<GLuint>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix2x3fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix2x4fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix3x2fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix3x4fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix4x2fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix4x3fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;vertexAttribI4iv(index:GLuint,values:Iterable<GLint>):void;vertexAttribI4uiv(index:GLuint,values:Iterable<GLuint>):void;}interface WebGL2RenderingContextOverloads{uniform1fv(location:WebGLUniformLocation|null,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniform1iv(location:WebGLUniformLocation|null,data:Iterable<GLint>,srcOffset?:number,srcLength?:GLuint):void;uniform2fv(location:WebGLUniformLocation|null,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniform2iv(location:WebGLUniformLocation|null,data:Iterable<GLint>,srcOffset?:number,srcLength?:GLuint):void;uniform3fv(location:WebGLUniformLocation|null,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniform3iv(location:WebGLUniformLocation|null,data:Iterable<GLint>,srcOffset?:number,srcLength?:GLuint):void;uniform4fv(location:WebGLUniformLocation|null,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniform4iv(location:WebGLUniformLocation|null,data:Iterable<GLint>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix2fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix3fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;uniformMatrix4fv(location:WebGLUniformLocation|null,transpose:GLboolean,data:Iterable<GLfloat>,srcOffset?:number,srcLength?:GLuint):void;}interface WebGLRenderingContextBase{vertexAttrib1fv(index:GLuint,values:Iterable<GLfloat>):void;vertexAttrib2fv(index:GLuint,values:Iterable<GLfloat>):void;vertexAttrib3fv(index:GLuint,values:Iterable<GLfloat>):void;vertexAttrib4fv(index:GLuint,values:Iterable<GLfloat>):void;}interface WebGLRenderingContextOverloads{uniform1fv(location:WebGLUniformLocation|null,v:Iterable<GLfloat>):void;uniform1iv(location:WebGLUniformLocation|null,v:Iterable<GLint>):void;uniform2fv(location:WebGLUniformLocation|null,v:Iterable<GLfloat>):void;uniform2iv(location:WebGLUniformLocation|null,v:Iterable<GLint>):void;uniform3fv(location:WebGLUniformLocation|null,v:Iterable<GLfloat>):void;uniform3iv(location:WebGLUniformLocation|null,v:Iterable<GLint>):void;uniform4fv(location:WebGLUniformLocation|null,v:Iterable<GLfloat>):void;uniform4iv(location:WebGLUniformLocation|null,v:Iterable<GLint>):void;uniformMatrix2fv(location:WebGLUniformLocation|null,transpose:GLboolean,value:Iterable<GLfloat>):void;uniformMatrix3fv(location:WebGLUniformLocation|null,transpose:GLboolean,value:Iterable<GLfloat>):void;uniformMatrix4fv(location:WebGLUniformLocation|null,transpose:GLboolean,value:Iterable<GLfloat>):void;}"
    }, {
        fileName: "lib.es2018.full.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2018\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n/// <reference lib=\"dom.iterable\" />\n/// <reference lib=\"dom.asynciterable\" />\n"
    }, {
        fileName: "lib.esnext.collection.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface MapConstructor{groupBy<K,T>(items:Iterable<T>,keySelector:(item:T,index:number)=>K,):Map<K,T[]>;}interface ReadonlySetLike<T>{keys():Iterator<T>;has(value:T):boolean;readonly size:number;}interface Set<T>{union<U>(other:ReadonlySetLike<U>):Set<T|U>;intersection<U>(other:ReadonlySetLike<U>):Set<T&U>;difference<U>(other:ReadonlySetLike<U>):Set<T>;symmetricDifference<U>(other:ReadonlySetLike<U>):Set<T|U>;isSubsetOf(other:ReadonlySetLike<unknown>):boolean;isSupersetOf(other:ReadonlySetLike<unknown>):boolean;isDisjointFrom(other:ReadonlySetLike<unknown>):boolean;}interface ReadonlySet<T>{union<U>(other:ReadonlySetLike<U>):Set<T|U>;intersection<U>(other:ReadonlySetLike<U>):Set<T&U>;difference<U>(other:ReadonlySetLike<U>):Set<T>;symmetricDifference<U>(other:ReadonlySetLike<U>):Set<T|U>;isSubsetOf(other:ReadonlySetLike<unknown>):boolean;isSupersetOf(other:ReadonlySetLike<unknown>):boolean;isDisjointFrom(other:ReadonlySetLike<unknown>):boolean;}"
    }, {
        fileName: "lib.es2015.reflect.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Reflect{function apply<T,A extends readonly any[],R>(target:(this:T,...args:A)=>R,thisArgument:T,argumentsList:Readonly<A>,):R;function apply(target:Function,thisArgument:any,argumentsList:ArrayLike<any>):any;function construct<A extends readonly any[],R>(target:new(...args:A)=>R,argumentsList:Readonly<A>,newTarget?:new(...args:any)=>any,):R;function construct(target:Function,argumentsList:ArrayLike<any>,newTarget?:Function):any;function defineProperty(target:object,propertyKey:PropertyKey,attributes:PropertyDescriptor&ThisType<any>):boolean;function deleteProperty(target:object,propertyKey:PropertyKey):boolean;function get<T extends object,P extends PropertyKey>(target:T,propertyKey:P,receiver?:unknown,):P extends keyof T?T[P]:any;function getOwnPropertyDescriptor<T extends object,P extends PropertyKey>(target:T,propertyKey:P,):TypedPropertyDescriptor<P extends keyof T?T[P]:any>|undefined;function getPrototypeOf(target:object):object|null;function has(target:object,propertyKey:PropertyKey):boolean;function isExtensible(target:object):boolean;function ownKeys(target:object):(string|symbol)[];function preventExtensions(target:object):boolean;function set<T extends object,P extends PropertyKey>(target:T,propertyKey:P,value:P extends keyof T?T[P]:any,receiver?:any,):boolean;function set(target:object,propertyKey:PropertyKey,value:any,receiver?:any):boolean;function setPrototypeOf(target:object,proto:object|null):boolean;}"
    }, {
        fileName: "lib.es2019.string.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface String{trimEnd():string;trimStart():string;trimLeft():string;trimRight():string;}"
    }, {
        fileName: "lib.es2017.date.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface DateConstructor{UTC(year:number,monthIndex?:number,date?:number,hours?:number,minutes?:number,seconds?:number,ms?:number):number;}"
    }, {
        fileName: "lib.es2021.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Intl{interface DateTimeFormatPartTypesRegistry{fractionalSecond:any;}interface DateTimeFormatOptions{formatMatcher?:\"basic\"|\"best fit\"|\"best fit\"|undefined;dateStyle?:\"full\"|\"long\"|\"medium\"|\"short\"|undefined;timeStyle?:\"full\"|\"long\"|\"medium\"|\"short\"|undefined;dayPeriod?:\"narrow\"|\"short\"|\"long\"|undefined;fractionalSecondDigits?:1|2|3|undefined;}interface DateTimeRangeFormatPart extends DateTimeFormatPart{source:\"startRange\"|\"endRange\"|\"shared\";}interface DateTimeFormat{formatRange(startDate:Date|number|bigint,endDate:Date|number|bigint):string;formatRangeToParts(startDate:Date|number|bigint,endDate:Date|number|bigint):DateTimeRangeFormatPart[];}interface ResolvedDateTimeFormatOptions{formatMatcher?:\"basic\"|\"best fit\"|\"best fit\";dateStyle?:\"full\"|\"long\"|\"medium\"|\"short\";timeStyle?:\"full\"|\"long\"|\"medium\"|\"short\";hourCycle?:\"h11\"|\"h12\"|\"h23\"|\"h24\";dayPeriod?:\"narrow\"|\"short\"|\"long\";fractionalSecondDigits?:1|2|3;}type ListFormatLocaleMatcher=\"lookup\"|\"best fit\";type ListFormatType=\"conjunction\"|\"disjunction\"|\"unit\";type ListFormatStyle=\"long\"|\"short\"|\"narrow\";interface ListFormatOptions{localeMatcher?:ListFormatLocaleMatcher|undefined;type?:ListFormatType|undefined;style?:ListFormatStyle|undefined;}interface ResolvedListFormatOptions{locale:string;style:ListFormatStyle;type:ListFormatType;}interface ListFormat{format(list:Iterable<string>):string;formatToParts(list:Iterable<string>):{type:\"element\"|\"literal\";value:string;}[];resolvedOptions():ResolvedListFormatOptions;}const ListFormat:{prototype:ListFormat;new(locales?:LocalesArgument,options?:ListFormatOptions):ListFormat;supportedLocalesOf(locales:LocalesArgument,options?:Pick<ListFormatOptions,\"localeMatcher\">):UnicodeBCP47LocaleIdentifier[];};}"
    }, {
        fileName: "lib.es2016.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015\" />\n/// <reference lib=\"es2016.array.include\" />\n/// <reference lib=\"es2016.intl\" />\n"
    }, {
        fileName: "lib.es2015.core.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface Array<T>{find<S extends T>(predicate:(value:T,index:number,obj:T[])=>value is S,thisArg?:any):S|undefined;find(predicate:(value:T,index:number,obj:T[])=>unknown,thisArg?:any):T|undefined;findIndex(predicate:(value:T,index:number,obj:T[])=>unknown,thisArg?:any):number;fill(value:T,start?:number,end?:number):this;copyWithin(target:number,start:number,end?:number):this;toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions&Intl.DateTimeFormatOptions):string;}interface ArrayConstructor{from<T>(arrayLike:ArrayLike<T>):T[];from<T,U>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>U,thisArg?:any):U[];of<T>(...items:T[]):T[];}interface DateConstructor{new(value:number|string|Date):Date;}interface Function{readonly name:string;}interface Math{clz32(x:number):number;imul(x:number,y:number):number;sign(x:number):number;log10(x:number):number;log2(x:number):number;log1p(x:number):number;expm1(x:number):number;cosh(x:number):number;sinh(x:number):number;tanh(x:number):number;acosh(x:number):number;asinh(x:number):number;atanh(x:number):number;hypot(...values:number[]):number;trunc(x:number):number;fround(x:number):number;cbrt(x:number):number;}interface NumberConstructor{readonly EPSILON:number;isFinite(number:unknown):boolean;isInteger(number:unknown):boolean;isNaN(number:unknown):boolean;isSafeInteger(number:unknown):boolean;readonly MAX_SAFE_INTEGER:number;readonly MIN_SAFE_INTEGER:number;parseFloat(string:string):number;parseInt(string:string,radix?:number):number;}interface ObjectConstructor{assign<T extends{},U>(target:T,source:U):T&U;assign<T extends{},U,V>(target:T,source1:U,source2:V):T&U&V;assign<T extends{},U,V,W>(target:T,source1:U,source2:V,source3:W):T&U&V&W;assign(target:object,...sources:any[]):any;getOwnPropertySymbols(o:any):symbol[];keys(o:{}):string[];is(value1:any,value2:any):boolean;setPrototypeOf(o:any,proto:object|null):any;}interface ReadonlyArray<T>{find<S extends T>(predicate:(value:T,index:number,obj:readonly T[])=>value is S,thisArg?:any):S|undefined;find(predicate:(value:T,index:number,obj:readonly T[])=>unknown,thisArg?:any):T|undefined;findIndex(predicate:(value:T,index:number,obj:readonly T[])=>unknown,thisArg?:any):number;toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions&Intl.DateTimeFormatOptions):string;}interface RegExp{readonly flags:string;readonly sticky:boolean;readonly unicode:boolean;}interface RegExpConstructor{new(pattern:RegExp|string,flags?:string):RegExp;(pattern:RegExp|string,flags?:string):RegExp;}interface String{codePointAt(pos:number):number|undefined;includes(searchString:string,position?:number):boolean;endsWith(searchString:string,endPosition?:number):boolean;normalize(form:\"NFC\"|\"NFD\"|\"NFKC\"|\"NFKD\"):string;normalize(form?:string):string;repeat(count:number):string;startsWith(searchString:string,position?:number):boolean;anchor(name:string):string;big():string;blink():string;bold():string;fixed():string;fontcolor(color:string):string;fontsize(size:number):string;fontsize(size:string):string;italics():string;link(url:string):string;small():string;strike():string;sub():string;sup():string;}interface StringConstructor{fromCodePoint(...codePoints:number[]):string;raw(template:{raw:readonly string[]|ArrayLike<string>;},...substitutions:any[]):string;}interface Int8Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Uint8Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Uint8ClampedArray{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Int16Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Uint16Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Int32Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Uint32Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Float32Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}interface Float64Array{toLocaleString(locales:string|string[],options?:Intl.NumberFormatOptions):string;}"
    }, {
        fileName: "lib.dom.asynciterable.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface FileSystemDirectoryHandleAsyncIterator<T>extends AsyncIteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.asyncIterator]():FileSystemDirectoryHandleAsyncIterator<T>;}interface FileSystemDirectoryHandle{[Symbol.asyncIterator]():FileSystemDirectoryHandleAsyncIterator<[string,FileSystemHandle]>;entries():FileSystemDirectoryHandleAsyncIterator<[string,FileSystemHandle]>;keys():FileSystemDirectoryHandleAsyncIterator<string>;values():FileSystemDirectoryHandleAsyncIterator<FileSystemHandle>;}interface ReadableStreamAsyncIterator<T>extends AsyncIteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.asyncIterator]():ReadableStreamAsyncIterator<T>;}interface ReadableStream<R=any>{[Symbol.asyncIterator](options?:ReadableStreamIteratorOptions):ReadableStreamAsyncIterator<R>;values(options?:ReadableStreamIteratorOptions):ReadableStreamAsyncIterator<R>;}"
    }, {
        fileName: "lib.es2019.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Intl{interface DateTimeFormatPartTypesRegistry{unknown:never;}}"
    }, {
        fileName: "lib.es2022.string.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface String{at(index:number):string|undefined;}"
    }, {
        fileName: "lib.es2020.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2019\" />\n/// <reference lib=\"es2020.bigint\" />\n/// <reference lib=\"es2020.date\" />\n/// <reference lib=\"es2020.number\" />\n/// <reference lib=\"es2020.promise\" />\n/// <reference lib=\"es2020.sharedmemory\" />\n/// <reference lib=\"es2020.string\" />\n/// <reference lib=\"es2020.symbol.wellknown\" />\n/// <reference lib=\"es2020.intl\" />\n"
    }, {
        fileName: "lib.es2020.full.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2020\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n/// <reference lib=\"dom.iterable\" />\n/// <reference lib=\"dom.asynciterable\" />\n"
    }, {
        fileName: "lib.es2022.object.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface ObjectConstructor{hasOwn(o:object,v:PropertyKey):boolean;}"
    }, {
        fileName: "lib.esnext.iterator.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.iterable\" />\nexport{};declare abstract class Iterator<T,TResult=undefined,TNext=unknown>{abstract next(value?:TNext):IteratorResult<T,TResult>;}interface Iterator<T,TResult,TNext>extends globalThis.IteratorObject<T,TResult,TNext>{}type IteratorObjectConstructor=typeof Iterator;declare global{interface IteratorObject<T,TReturn,TNext>{[Symbol.iterator]():IteratorObject<T,TReturn,TNext>;map<U>(callbackfn:(value:T,index:number)=>U):IteratorObject<U,undefined,unknown>;filter<S extends T>(predicate:(value:T,index:number)=>value is S):IteratorObject<S,undefined,unknown>;filter(predicate:(value:T,index:number)=>unknown):IteratorObject<T,undefined,unknown>;take(limit:number):IteratorObject<T,undefined,unknown>;drop(count:number):IteratorObject<T,undefined,unknown>;flatMap<U>(callback:(value:T,index:number)=>Iterator<U,unknown,undefined>|Iterable<U,unknown,undefined>):IteratorObject<U,undefined,unknown>;reduce(callbackfn:(previousValue:T,currentValue:T,currentIndex:number)=>T):T;reduce(callbackfn:(previousValue:T,currentValue:T,currentIndex:number)=>T,initialValue:T):T;reduce<U>(callbackfn:(previousValue:U,currentValue:T,currentIndex:number)=>U,initialValue:U):U;toArray():T[];forEach(callbackfn:(value:T,index:number)=>void):void;some(predicate:(value:T,index:number)=>unknown):boolean;every(predicate:(value:T,index:number)=>unknown):boolean;find<S extends T>(predicate:(value:T,index:number)=>value is S):S|undefined;find(predicate:(value:T,index:number)=>unknown):T|undefined;readonly[Symbol.toStringTag]:string;}interface IteratorConstructor extends IteratorObjectConstructor{from<T>(value:Iterator<T,unknown,undefined>|Iterable<T,unknown,undefined>):IteratorObject<T,undefined,unknown>;}var Iterator:IteratorConstructor;}"
    }, {
        fileName: "lib.es2015.proxy.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface ProxyHandler<T extends object>{apply?(target:T,thisArg:any,argArray:any[]):any;construct?(target:T,argArray:any[],newTarget:Function):object;defineProperty?(target:T,property:string|symbol,attributes:PropertyDescriptor):boolean;deleteProperty?(target:T,p:string|symbol):boolean;get?(target:T,p:string|symbol,receiver:any):any;getOwnPropertyDescriptor?(target:T,p:string|symbol):PropertyDescriptor|undefined;getPrototypeOf?(target:T):object|null;has?(target:T,p:string|symbol):boolean;isExtensible?(target:T):boolean;ownKeys?(target:T):ArrayLike<string|symbol>;preventExtensions?(target:T):boolean;set?(target:T,p:string|symbol,newValue:any,receiver:any):boolean;setPrototypeOf?(target:T,v:object|null):boolean;}interface ProxyConstructor{revocable<T extends object>(target:T,handler:ProxyHandler<T>):{proxy:T;revoke:()=>void;};new<T extends object>(target:T,handler:ProxyHandler<T>):T;}declare var Proxy:ProxyConstructor;"
    }, {
        fileName: "lib.es2015.generator.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.iterable\" />\ninterface Generator<T=unknown,TReturn=any,TNext=any>extends IteratorObject<T,TReturn,TNext>{next(...[value]:[]|[TNext]):IteratorResult<T,TReturn>;return(value:TReturn):IteratorResult<T,TReturn>;throw(e:any):IteratorResult<T,TReturn>;[Symbol.iterator]():Generator<T,TReturn,TNext>;}interface GeneratorFunction{new(...args:any[]):Generator;(...args:any[]):Generator;readonly length:number;readonly name:string;readonly prototype:Generator;}interface GeneratorFunctionConstructor{new(...args:string[]):GeneratorFunction;(...args:string[]):GeneratorFunction;readonly length:number;readonly name:string;readonly prototype:GeneratorFunction;}"
    }, {
        fileName: "lib.esnext.promise.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface PromiseWithResolvers<T>{promise:Promise<T>;resolve:(value:T|PromiseLike<T>)=>void;reject:(reason?:any)=>void;}interface PromiseConstructor{withResolvers<T>():PromiseWithResolvers<T>;}"
    }, {
        fileName: "lib.es2021.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2020\" />\n/// <reference lib=\"es2021.promise\" />\n/// <reference lib=\"es2021.string\" />\n/// <reference lib=\"es2021.weakref\" />\n/// <reference lib=\"es2021.intl\" />\n"
    }, {
        fileName: "lib.esnext.disposable.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.symbol\" />\n/// <reference lib=\"es2015.iterable\" />\n/// <reference lib=\"es2018.asynciterable\" />\ninterface SymbolConstructor{readonly dispose:unique symbol;readonly asyncDispose:unique symbol;}interface Disposable{[Symbol.dispose]():void;}interface AsyncDisposable{[Symbol.asyncDispose]():PromiseLike<void>;}interface SuppressedError extends Error{error:any;suppressed:any;}interface SuppressedErrorConstructor{new(error:any,suppressed:any,message?:string):SuppressedError;(error:any,suppressed:any,message?:string):SuppressedError;readonly prototype:SuppressedError;}declare var SuppressedError:SuppressedErrorConstructor;interface DisposableStack{readonly disposed:boolean;dispose():void;use<T extends Disposable|null|undefined>(value:T):T;adopt<T>(value:T,onDispose:(value:T)=>void):T;defer(onDispose:()=>void):void;move():DisposableStack;[Symbol.dispose]():void;readonly[Symbol.toStringTag]:string;}interface DisposableStackConstructor{new():DisposableStack;readonly prototype:DisposableStack;}declare var DisposableStack:DisposableStackConstructor;interface AsyncDisposableStack{readonly disposed:boolean;disposeAsync():Promise<void>;use<T extends AsyncDisposable|Disposable|null|undefined>(value:T):T;adopt<T>(value:T,onDisposeAsync:(value:T)=>PromiseLike<void>|void):T;defer(onDisposeAsync:()=>PromiseLike<void>|void):void;move():AsyncDisposableStack;[Symbol.asyncDispose]():Promise<void>;readonly[Symbol.toStringTag]:string;}interface AsyncDisposableStackConstructor{new():AsyncDisposableStack;readonly prototype:AsyncDisposableStack;}declare var AsyncDisposableStack:AsyncDisposableStackConstructor;interface IteratorObject<T,TReturn,TNext>extends Disposable{}interface AsyncIteratorObject<T,TReturn,TNext>extends AsyncDisposable{}"
    }, {
        fileName: "lib.es2020.number.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2020.intl\" />\ninterface Number{toLocaleString(locales?:Intl.LocalesArgument,options?:Intl.NumberFormatOptions):string;}"
    }, {
        fileName: "lib.es2017.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2016\" />\n/// <reference lib=\"es2017.object\" />\n/// <reference lib=\"es2017.sharedmemory\" />\n/// <reference lib=\"es2017.string\" />\n/// <reference lib=\"es2017.intl\" />\n/// <reference lib=\"es2017.typedarrays\" />\n/// <reference lib=\"es2017.date\" />\n"
    }, {
        fileName: "lib.es2018.promise.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface Promise<T>{finally(onfinally?:(()=>void)|undefined|null):Promise<T>;}"
    }, {
        fileName: "lib.es2017.typedarrays.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface Int8ArrayConstructor{new():Int8Array;}interface Uint8ArrayConstructor{new():Uint8Array;}interface Uint8ClampedArrayConstructor{new():Uint8ClampedArray;}interface Int16ArrayConstructor{new():Int16Array;}interface Uint16ArrayConstructor{new():Uint16Array;}interface Int32ArrayConstructor{new():Int32Array;}interface Uint32ArrayConstructor{new():Uint32Array;}interface Float32ArrayConstructor{new():Float32Array;}interface Float64ArrayConstructor{new():Float64Array;}"
    }, {
        fileName: "lib.scripthost.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface ActiveXObject{new(s:string):any;}declare var ActiveXObject:ActiveXObject;interface ITextWriter{Write(s:string):void;WriteLine(s:string):void;Close():void;}interface TextStreamBase{Column:number;Line:number;Close():void;}interface TextStreamWriter extends TextStreamBase{Write(s:string):void;WriteBlankLines(intLines:number):void;WriteLine(s:string):void;}interface TextStreamReader extends TextStreamBase{Read(characters:number):string;ReadAll():string;ReadLine():string;Skip(characters:number):void;SkipLine():void;AtEndOfLine:boolean;AtEndOfStream:boolean;}declare var WScript:{Echo(s:any):void;StdErr:TextStreamWriter;StdOut:TextStreamWriter;Arguments:{length:number;Item(n:number):string;};ScriptFullName:string;Quit(exitCode?:number):number;BuildVersion:number;FullName:string;Interactive:boolean;Name:string;Path:string;ScriptName:string;StdIn:TextStreamReader;Version:string;ConnectObject(objEventSource:any,strPrefix:string):void;CreateObject(strProgID:string,strPrefix?:string):any;DisconnectObject(obj:any):void;GetObject(strPathname:string,strProgID?:string,strPrefix?:string):any;Sleep(intTime:number):void;};declare var WSH:typeof WScript;declare class SafeArray<T=any>{private constructor();private SafeArray_typekey:SafeArray<T>;}interface Enumerator<T=any>{atEnd():boolean;item():T;moveFirst():void;moveNext():void;}interface EnumeratorConstructor{new<T=any>(safearray:SafeArray<T>):Enumerator<T>;new<T=any>(collection:{Item(index:any):T;}):Enumerator<T>;new<T=any>(collection:any):Enumerator<T>;}declare var Enumerator:EnumeratorConstructor;interface VBArray<T=any>{dimensions():number;getItem(dimension1Index:number,...dimensionNIndexes:number[]):T;lbound(dimension?:number):number;ubound(dimension?:number):number;toArray():T[];}interface VBArrayConstructor{new<T=any>(safeArray:SafeArray<T>):VBArray<T>;}declare var VBArray:VBArrayConstructor;declare class VarDate{private constructor();private VarDate_typekey:VarDate;}interface DateConstructor{new(vd:VarDate):Date;}interface Date{getVarDate:()=>VarDate;}"
    }, {
        fileName: "lib.es2022.array.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface Array<T>{at(index:number):T|undefined;}interface ReadonlyArray<T>{at(index:number):T|undefined;}interface Int8Array{at(index:number):number|undefined;}interface Uint8Array{at(index:number):number|undefined;}interface Uint8ClampedArray{at(index:number):number|undefined;}interface Int16Array{at(index:number):number|undefined;}interface Uint16Array{at(index:number):number|undefined;}interface Int32Array{at(index:number):number|undefined;}interface Uint32Array{at(index:number):number|undefined;}interface Float32Array{at(index:number):number|undefined;}interface Float64Array{at(index:number):number|undefined;}interface BigInt64Array{at(index:number):bigint|undefined;}interface BigUint64Array{at(index:number):bigint|undefined;}"
    }, {
        fileName: "lib.esnext.decorators.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.symbol\" />\n/// <reference lib=\"decorators\" />\ninterface SymbolConstructor{readonly metadata:unique symbol;}interface Function{[Symbol.metadata]:DecoratorMetadata|null;}"
    }, {
        fileName: "lib.es2019.full.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2019\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n/// <reference lib=\"dom.iterable\" />\n/// <reference lib=\"dom.asynciterable\" />\n"
    }, {
        fileName: "lib.es2020.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2018.intl\" />\ndeclare namespace Intl{type UnicodeBCP47LocaleIdentifier=string;type RelativeTimeFormatUnit=|\"year\"|\"years\"|\"quarter\"|\"quarters\"|\"month\"|\"months\"|\"week\"|\"weeks\"|\"day\"|\"days\"|\"hour\"|\"hours\"|\"minute\"|\"minutes\"|\"second\"|\"seconds\";type RelativeTimeFormatUnitSingular=|\"year\"|\"quarter\"|\"month\"|\"week\"|\"day\"|\"hour\"|\"minute\"|\"second\";type RelativeTimeFormatLocaleMatcher=\"lookup\"|\"best fit\";type RelativeTimeFormatNumeric=\"always\"|\"auto\";type RelativeTimeFormatStyle=\"long\"|\"short\"|\"narrow\";type LocalesArgument=UnicodeBCP47LocaleIdentifier|Locale|readonly(UnicodeBCP47LocaleIdentifier|Locale)[]|undefined;interface RelativeTimeFormatOptions{localeMatcher?:RelativeTimeFormatLocaleMatcher;numeric?:RelativeTimeFormatNumeric;style?:RelativeTimeFormatStyle;}interface ResolvedRelativeTimeFormatOptions{locale:UnicodeBCP47LocaleIdentifier;style:RelativeTimeFormatStyle;numeric:RelativeTimeFormatNumeric;numberingSystem:string;}type RelativeTimeFormatPart=|{type:\"literal\";value:string;}|{type:Exclude<NumberFormatPartTypes,\"literal\">;value:string;unit:RelativeTimeFormatUnitSingular;};interface RelativeTimeFormat{format(value:number,unit:RelativeTimeFormatUnit):string;formatToParts(value:number,unit:RelativeTimeFormatUnit):RelativeTimeFormatPart[];resolvedOptions():ResolvedRelativeTimeFormatOptions;}const RelativeTimeFormat:{new(locales?:LocalesArgument,options?:RelativeTimeFormatOptions,):RelativeTimeFormat;supportedLocalesOf(locales?:LocalesArgument,options?:RelativeTimeFormatOptions,):UnicodeBCP47LocaleIdentifier[];};interface NumberFormatOptionsStyleRegistry{unit:never;}interface NumberFormatOptionsCurrencyDisplayRegistry{narrowSymbol:never;}interface NumberFormatOptionsSignDisplayRegistry{auto:never;never:never;always:never;exceptZero:never;}type NumberFormatOptionsSignDisplay=keyof NumberFormatOptionsSignDisplayRegistry;interface NumberFormatOptions{numberingSystem?:string|undefined;compactDisplay?:\"short\"|\"long\"|undefined;notation?:\"standard\"|\"scientific\"|\"engineering\"|\"compact\"|undefined;signDisplay?:NumberFormatOptionsSignDisplay|undefined;unit?:string|undefined;unitDisplay?:\"short\"|\"long\"|\"narrow\"|undefined;currencySign?:\"standard\"|\"accounting\"|undefined;}interface ResolvedNumberFormatOptions{compactDisplay?:\"short\"|\"long\";notation:\"standard\"|\"scientific\"|\"engineering\"|\"compact\";signDisplay:NumberFormatOptionsSignDisplay;unit?:string;unitDisplay?:\"short\"|\"long\"|\"narrow\";currencySign?:\"standard\"|\"accounting\";}interface NumberFormatPartTypeRegistry{compact:never;exponentInteger:never;exponentMinusSign:never;exponentSeparator:never;unit:never;unknown:never;}interface DateTimeFormatOptions{calendar?:string|undefined;dayPeriod?:\"narrow\"|\"short\"|\"long\"|undefined;numberingSystem?:string|undefined;dateStyle?:\"full\"|\"long\"|\"medium\"|\"short\"|undefined;timeStyle?:\"full\"|\"long\"|\"medium\"|\"short\"|undefined;hourCycle?:\"h11\"|\"h12\"|\"h23\"|\"h24\"|undefined;}type LocaleHourCycleKey=\"h12\"|\"h23\"|\"h11\"|\"h24\";type LocaleCollationCaseFirst=\"upper\"|\"lower\"|\"false\";interface LocaleOptions{baseName?:string;calendar?:string;caseFirst?:LocaleCollationCaseFirst;collation?:string;hourCycle?:LocaleHourCycleKey;language?:string;numberingSystem?:string;numeric?:boolean;region?:string;script?:string;}interface Locale extends LocaleOptions{baseName:string;language:string;maximize():Locale;minimize():Locale;toString():UnicodeBCP47LocaleIdentifier;}const Locale:{new(tag:UnicodeBCP47LocaleIdentifier|Locale,options?:LocaleOptions):Locale;};type DisplayNamesFallback=|\"code\"|\"none\";type DisplayNamesType=|\"language\"|\"region\"|\"script\"|\"calendar\"|\"dateTimeField\"|\"currency\";type DisplayNamesLanguageDisplay=|\"dialect\"|\"standard\";interface DisplayNamesOptions{localeMatcher?:RelativeTimeFormatLocaleMatcher;style?:RelativeTimeFormatStyle;type:DisplayNamesType;languageDisplay?:DisplayNamesLanguageDisplay;fallback?:DisplayNamesFallback;}interface ResolvedDisplayNamesOptions{locale:UnicodeBCP47LocaleIdentifier;style:RelativeTimeFormatStyle;type:DisplayNamesType;fallback:DisplayNamesFallback;languageDisplay?:DisplayNamesLanguageDisplay;}interface DisplayNames{of(code:string):string|undefined;resolvedOptions():ResolvedDisplayNamesOptions;}const DisplayNames:{prototype:DisplayNames;new(locales:LocalesArgument,options:DisplayNamesOptions):DisplayNames;supportedLocalesOf(locales?:LocalesArgument,options?:{localeMatcher?:RelativeTimeFormatLocaleMatcher;}):UnicodeBCP47LocaleIdentifier[];};interface CollatorConstructor{new(locales?:LocalesArgument,options?:CollatorOptions):Collator;(locales?:LocalesArgument,options?:CollatorOptions):Collator;supportedLocalesOf(locales:LocalesArgument,options?:CollatorOptions):string[];}interface DateTimeFormatConstructor{new(locales?:LocalesArgument,options?:DateTimeFormatOptions):DateTimeFormat;(locales?:LocalesArgument,options?:DateTimeFormatOptions):DateTimeFormat;supportedLocalesOf(locales:LocalesArgument,options?:DateTimeFormatOptions):string[];}interface NumberFormatConstructor{new(locales?:LocalesArgument,options?:NumberFormatOptions):NumberFormat;(locales?:LocalesArgument,options?:NumberFormatOptions):NumberFormat;supportedLocalesOf(locales:LocalesArgument,options?:NumberFormatOptions):string[];}interface PluralRulesConstructor{new(locales?:LocalesArgument,options?:PluralRulesOptions):PluralRules;(locales?:LocalesArgument,options?:PluralRulesOptions):PluralRules;supportedLocalesOf(locales:LocalesArgument,options?:{localeMatcher?:\"lookup\"|\"best fit\";}):string[];}}"
    }, {
        fileName: "lib.decorators.legacy.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare type ClassDecorator=<TFunction extends Function>(target:TFunction)=>TFunction|void;declare type PropertyDecorator=(target:Object,propertyKey:string|symbol)=>void;declare type MethodDecorator=<T>(target:Object,propertyKey:string|symbol,descriptor:TypedPropertyDescriptor<T>)=>TypedPropertyDescriptor<T>|void;declare type ParameterDecorator=(target:Object,propertyKey:string|symbol|undefined,parameterIndex:number)=>void;"
    }, {
        fileName: "lib.es2015.promise.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface PromiseConstructor{readonly prototype:Promise<any>;new<T>(executor:(resolve:(value:T|PromiseLike<T>)=>void,reject:(reason?:any)=>void)=>void):Promise<T>;all<T extends readonly unknown[]|[]>(values:T):Promise<{-readonly[P in keyof T]:Awaited<T[P]>;}>;race<T extends readonly unknown[]|[]>(values:T):Promise<Awaited<T[number]>>;reject<T=never>(reason?:any):Promise<T>;resolve():Promise<void>;resolve<T>(value:T):Promise<Awaited<T>>;resolve<T>(value:T|PromiseLike<T>):Promise<Awaited<T>>;}declare var Promise:PromiseConstructor;"
    }, {
        fileName: "lib.es2018.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Intl{type LDMLPluralRule=\"zero\"|\"one\"|\"two\"|\"few\"|\"many\"|\"other\";type PluralRuleType=\"cardinal\"|\"ordinal\";interface PluralRulesOptions{localeMatcher?:\"lookup\"|\"best fit\"|undefined;type?:PluralRuleType|undefined;minimumIntegerDigits?:number|undefined;minimumFractionDigits?:number|undefined;maximumFractionDigits?:number|undefined;minimumSignificantDigits?:number|undefined;maximumSignificantDigits?:number|undefined;}interface ResolvedPluralRulesOptions{locale:string;pluralCategories:LDMLPluralRule[];type:PluralRuleType;minimumIntegerDigits:number;minimumFractionDigits:number;maximumFractionDigits:number;minimumSignificantDigits?:number;maximumSignificantDigits?:number;}interface PluralRules{resolvedOptions():ResolvedPluralRulesOptions;select(n:number):LDMLPluralRule;}interface PluralRulesConstructor{new(locales?:string|readonly string[],options?:PluralRulesOptions):PluralRules;(locales?:string|readonly string[],options?:PluralRulesOptions):PluralRules;supportedLocalesOf(locales:string|readonly string[],options?:{localeMatcher?:\"lookup\"|\"best fit\";}):string[];}const PluralRules:PluralRulesConstructor;interface NumberFormatPartTypeRegistry{literal:never;nan:never;infinity:never;percent:never;integer:never;group:never;decimal:never;fraction:never;plusSign:never;minusSign:never;percentSign:never;currency:never;}type NumberFormatPartTypes=keyof NumberFormatPartTypeRegistry;interface NumberFormatPart{type:NumberFormatPartTypes;value:string;}interface NumberFormat{formatToParts(number?:number|bigint):NumberFormatPart[];}}"
    }, {
        fileName: "lib.es2015.symbol.wellknown.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.symbol\" />\ninterface SymbolConstructor{readonly hasInstance:unique symbol;readonly isConcatSpreadable:unique symbol;readonly match:unique symbol;readonly replace:unique symbol;readonly search:unique symbol;readonly species:unique symbol;readonly split:unique symbol;readonly toPrimitive:unique symbol;readonly toStringTag:unique symbol;readonly unscopables:unique symbol;}interface Symbol{[Symbol.toPrimitive](hint:string):symbol;readonly[Symbol.toStringTag]:string;}interface Array<T>{readonly[Symbol.unscopables]:{[K in keyof any[]]?:boolean;};}interface ReadonlyArray<T>{readonly[Symbol.unscopables]:{[K in keyof readonly any[]]?:boolean;};}interface Date{[Symbol.toPrimitive](hint:\"default\"):string;[Symbol.toPrimitive](hint:\"string\"):string;[Symbol.toPrimitive](hint:\"number\"):number;[Symbol.toPrimitive](hint:string):string|number;}interface Map<K,V>{readonly[Symbol.toStringTag]:string;}interface WeakMap<K extends WeakKey,V>{readonly[Symbol.toStringTag]:string;}interface Set<T>{readonly[Symbol.toStringTag]:string;}interface WeakSet<T extends WeakKey>{readonly[Symbol.toStringTag]:string;}interface JSON{readonly[Symbol.toStringTag]:string;}interface Function{[Symbol.hasInstance](value:any):boolean;}interface GeneratorFunction{readonly[Symbol.toStringTag]:string;}interface Math{readonly[Symbol.toStringTag]:string;}interface Promise<T>{readonly[Symbol.toStringTag]:string;}interface PromiseConstructor{readonly[Symbol.species]:PromiseConstructor;}interface RegExp{[Symbol.match](string:string):RegExpMatchArray|null;[Symbol.replace](string:string,replaceValue:string):string;[Symbol.replace](string:string,replacer:(substring:string,...args:any[])=>string):string;[Symbol.search](string:string):number;[Symbol.split](string:string,limit?:number):string[];}interface RegExpConstructor{readonly[Symbol.species]:RegExpConstructor;}interface String{match(matcher:{[Symbol.match](string:string):RegExpMatchArray|null;}):RegExpMatchArray|null;replace(searchValue:{[Symbol.replace](string:string,replaceValue:string):string;},replaceValue:string):string;replace(searchValue:{[Symbol.replace](string:string,replacer:(substring:string,...args:any[])=>string):string;},replacer:(substring:string,...args:any[])=>string):string;search(searcher:{[Symbol.search](string:string):number;}):number;split(splitter:{[Symbol.split](string:string,limit?:number):string[];},limit?:number):string[];}interface ArrayBuffer{readonly[Symbol.toStringTag]:string;}interface DataView{readonly[Symbol.toStringTag]:string;}interface Int8Array{readonly[Symbol.toStringTag]:\"Int8Array\";}interface Uint8Array{readonly[Symbol.toStringTag]:\"Uint8Array\";}interface Uint8ClampedArray{readonly[Symbol.toStringTag]:\"Uint8ClampedArray\";}interface Int16Array{readonly[Symbol.toStringTag]:\"Int16Array\";}interface Uint16Array{readonly[Symbol.toStringTag]:\"Uint16Array\";}interface Int32Array{readonly[Symbol.toStringTag]:\"Int32Array\";}interface Uint32Array{readonly[Symbol.toStringTag]:\"Uint32Array\";}interface Float32Array{readonly[Symbol.toStringTag]:\"Float32Array\";}interface Float64Array{readonly[Symbol.toStringTag]:\"Float64Array\";}interface ArrayConstructor{readonly[Symbol.species]:ArrayConstructor;}interface MapConstructor{readonly[Symbol.species]:MapConstructor;}interface SetConstructor{readonly[Symbol.species]:SetConstructor;}interface ArrayBufferConstructor{readonly[Symbol.species]:ArrayBufferConstructor;}"
    }, {
        fileName: "lib.esnext.regexp.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface RegExp{readonly unicodeSets:boolean;}"
    }, {
        fileName: "lib.es2021.full.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2021\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n/// <reference lib=\"dom.iterable\" />\n/// <reference lib=\"dom.asynciterable\" />\n"
    }, {
        fileName: "lib.es2015.iterable.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.symbol\" />\ninterface SymbolConstructor{readonly iterator:unique symbol;}interface IteratorYieldResult<TYield>{done?:false;value:TYield;}interface IteratorReturnResult<TReturn>{done:true;value:TReturn;}type IteratorResult<T,TReturn=any>=IteratorYieldResult<T>|IteratorReturnResult<TReturn>;interface Iterator<T,TReturn=any,TNext=any>{next(...[value]:[]|[TNext]):IteratorResult<T,TReturn>;return?(value?:TReturn):IteratorResult<T,TReturn>;throw?(e?:any):IteratorResult<T,TReturn>;}interface Iterable<T,TReturn=any,TNext=any>{[Symbol.iterator]():Iterator<T,TReturn,TNext>;}interface IterableIterator<T,TReturn=any,TNext=any>extends Iterator<T,TReturn,TNext>{[Symbol.iterator]():IterableIterator<T,TReturn,TNext>;}interface IteratorObject<T,TReturn=unknown,TNext=unknown>extends Iterator<T,TReturn,TNext>{[Symbol.iterator]():IteratorObject<T,TReturn,TNext>;}type BuiltinIteratorReturn=intrinsic;interface ArrayIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():ArrayIterator<T>;}interface Array<T>{[Symbol.iterator]():ArrayIterator<T>;entries():ArrayIterator<[number,T]>;keys():ArrayIterator<number>;values():ArrayIterator<T>;}interface ArrayConstructor{from<T>(iterable:Iterable<T>|ArrayLike<T>):T[];from<T,U>(iterable:Iterable<T>|ArrayLike<T>,mapfn:(v:T,k:number)=>U,thisArg?:any):U[];}interface ReadonlyArray<T>{[Symbol.iterator]():ArrayIterator<T>;entries():ArrayIterator<[number,T]>;keys():ArrayIterator<number>;values():ArrayIterator<T>;}interface IArguments{[Symbol.iterator]():ArrayIterator<any>;}interface MapIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():MapIterator<T>;}interface Map<K,V>{[Symbol.iterator]():MapIterator<[K,V]>;entries():MapIterator<[K,V]>;keys():MapIterator<K>;values():MapIterator<V>;}interface ReadonlyMap<K,V>{[Symbol.iterator]():MapIterator<[K,V]>;entries():MapIterator<[K,V]>;keys():MapIterator<K>;values():MapIterator<V>;}interface MapConstructor{new():Map<any,any>;new<K,V>(iterable?:Iterable<readonly[K,V]>|null):Map<K,V>;}interface WeakMap<K extends WeakKey,V>{}interface WeakMapConstructor{new<K extends WeakKey,V>(iterable:Iterable<readonly[K,V]>):WeakMap<K,V>;}interface SetIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():SetIterator<T>;}interface Set<T>{[Symbol.iterator]():SetIterator<T>;entries():SetIterator<[T,T]>;keys():SetIterator<T>;values():SetIterator<T>;}interface ReadonlySet<T>{[Symbol.iterator]():SetIterator<T>;entries():SetIterator<[T,T]>;keys():SetIterator<T>;values():SetIterator<T>;}interface SetConstructor{new<T>(iterable?:Iterable<T>|null):Set<T>;}interface WeakSet<T extends WeakKey>{}interface WeakSetConstructor{new<T extends WeakKey=WeakKey>(iterable:Iterable<T>):WeakSet<T>;}interface Promise<T>{}interface PromiseConstructor{all<T>(values:Iterable<T|PromiseLike<T>>):Promise<Awaited<T>[]>;race<T>(values:Iterable<T|PromiseLike<T>>):Promise<Awaited<T>>;}interface StringIterator<T>extends IteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.iterator]():StringIterator<T>;}interface String{[Symbol.iterator]():StringIterator<string>;}interface Int8Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Int8ArrayConstructor{new(elements:Iterable<number>):Int8Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Int8Array;}interface Uint8Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Uint8ArrayConstructor{new(elements:Iterable<number>):Uint8Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Uint8Array;}interface Uint8ClampedArray{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Uint8ClampedArrayConstructor{new(elements:Iterable<number>):Uint8ClampedArray;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Uint8ClampedArray;}interface Int16Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Int16ArrayConstructor{new(elements:Iterable<number>):Int16Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Int16Array;}interface Uint16Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Uint16ArrayConstructor{new(elements:Iterable<number>):Uint16Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Uint16Array;}interface Int32Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Int32ArrayConstructor{new(elements:Iterable<number>):Int32Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Int32Array;}interface Uint32Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Uint32ArrayConstructor{new(elements:Iterable<number>):Uint32Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Uint32Array;}interface Float32Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Float32ArrayConstructor{new(elements:Iterable<number>):Float32Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Float32Array;}interface Float64Array{[Symbol.iterator]():ArrayIterator<number>;entries():ArrayIterator<[number,number]>;keys():ArrayIterator<number>;values():ArrayIterator<number>;}interface Float64ArrayConstructor{new(elements:Iterable<number>):Float64Array;from(arrayLike:Iterable<number>,mapfn?:(v:number,k:number)=>number,thisArg?:any):Float64Array;}"
    }, {
        fileName: "lib.es2018.regexp.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface RegExpMatchArray{groups?:{[key:string]:string;};}interface RegExpExecArray{groups?:{[key:string]:string;};}interface RegExp{readonly dotAll:boolean;}"
    }, {
        fileName: "lib.decorators.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ntype ClassMemberDecoratorContext=|ClassMethodDecoratorContext|ClassGetterDecoratorContext|ClassSetterDecoratorContext|ClassFieldDecoratorContext|ClassAccessorDecoratorContext;type DecoratorContext=|ClassDecoratorContext|ClassMemberDecoratorContext;type DecoratorMetadataObject=Record<PropertyKey,unknown>&object;type DecoratorMetadata=typeof globalThis extends{Symbol:{readonly metadata:symbol;};}?DecoratorMetadataObject:DecoratorMetadataObject|undefined;interface ClassDecoratorContext<\nClass extends abstract new(...args:any)=>any=abstract new(...args:any)=>any,>{readonly kind:\"class\";readonly name:string|undefined;addInitializer(initializer:(this:Class)=>void):void;readonly metadata:DecoratorMetadata;}interface ClassMethodDecoratorContext<\nThis=unknown,Value extends(this:This,...args:any)=>any=(this:This,...args:any)=>any,>{readonly kind:\"method\";readonly name:string|symbol;readonly static:boolean;readonly private:boolean;readonly access:{has(object:This):boolean;get(object:This):Value;};addInitializer(initializer:(this:This)=>void):void;readonly metadata:DecoratorMetadata;}interface ClassGetterDecoratorContext<\nThis=unknown,Value=unknown,>{readonly kind:\"getter\";readonly name:string|symbol;readonly static:boolean;readonly private:boolean;readonly access:{has(object:This):boolean;get(object:This):Value;};addInitializer(initializer:(this:This)=>void):void;readonly metadata:DecoratorMetadata;}interface ClassSetterDecoratorContext<\nThis=unknown,Value=unknown,>{readonly kind:\"setter\";readonly name:string|symbol;readonly static:boolean;readonly private:boolean;readonly access:{has(object:This):boolean;set(object:This,value:Value):void;};addInitializer(initializer:(this:This)=>void):void;readonly metadata:DecoratorMetadata;}interface ClassAccessorDecoratorContext<\nThis=unknown,Value=unknown,>{readonly kind:\"accessor\";readonly name:string|symbol;readonly static:boolean;readonly private:boolean;readonly access:{has(object:This):boolean;get(object:This):Value;set(object:This,value:Value):void;};addInitializer(initializer:(this:This)=>void):void;readonly metadata:DecoratorMetadata;}interface ClassAccessorDecoratorTarget<This,Value>{get(this:This):Value;set(this:This,value:Value):void;}interface ClassAccessorDecoratorResult<This,Value>{get?(this:This):Value;set?(this:This,value:Value):void;init?(this:This,value:Value):Value;}interface ClassFieldDecoratorContext<\nThis=unknown,Value=unknown,>{readonly kind:\"field\";readonly name:string|symbol;readonly static:boolean;readonly private:boolean;readonly access:{has(object:This):boolean;get(object:This):Value;set(object:This,value:Value):void;};addInitializer(initializer:(this:This)=>void):void;readonly metadata:DecoratorMetadata;}"
    }, {
        fileName: "lib.es2021.string.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface String{replaceAll(searchValue:string|RegExp,replaceValue:string):string;replaceAll(searchValue:string|RegExp,replacer:(substring:string,...args:any[])=>string):string;}"
    }, {
        fileName: "lib.es2018.asynciterable.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015.symbol\" />\n/// <reference lib=\"es2015.iterable\" />\ninterface SymbolConstructor{readonly asyncIterator:unique symbol;}interface AsyncIterator<T,TReturn=any,TNext=any>{next(...[value]:[]|[TNext]):Promise<IteratorResult<T,TReturn>>;return?(value?:TReturn|PromiseLike<TReturn>):Promise<IteratorResult<T,TReturn>>;throw?(e?:any):Promise<IteratorResult<T,TReturn>>;}interface AsyncIterable<T,TReturn=any,TNext=any>{[Symbol.asyncIterator]():AsyncIterator<T,TReturn,TNext>;}interface AsyncIterableIterator<T,TReturn=any,TNext=any>extends AsyncIterator<T,TReturn,TNext>{[Symbol.asyncIterator]():AsyncIterableIterator<T,TReturn,TNext>;}interface AsyncIteratorObject<T,TReturn=unknown,TNext=unknown>extends AsyncIterator<T,TReturn,TNext>{[Symbol.asyncIterator]():AsyncIteratorObject<T,TReturn,TNext>;}"
    }, {
        fileName: "lib.es6.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2015\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"dom.iterable\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n"
    }, {
        fileName: "lib.es2023.collection.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface WeakKeyTypes{symbol:symbol;}"
    }, {
        fileName: "lib.es2022.error.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface ErrorOptions{cause?:unknown;}interface Error{cause?:unknown;}interface ErrorConstructor{new(message?:string,options?:ErrorOptions):Error;(message?:string,options?:ErrorOptions):Error;}interface EvalErrorConstructor{new(message?:string,options?:ErrorOptions):EvalError;(message?:string,options?:ErrorOptions):EvalError;}interface RangeErrorConstructor{new(message?:string,options?:ErrorOptions):RangeError;(message?:string,options?:ErrorOptions):RangeError;}interface ReferenceErrorConstructor{new(message?:string,options?:ErrorOptions):ReferenceError;(message?:string,options?:ErrorOptions):ReferenceError;}interface SyntaxErrorConstructor{new(message?:string,options?:ErrorOptions):SyntaxError;(message?:string,options?:ErrorOptions):SyntaxError;}interface TypeErrorConstructor{new(message?:string,options?:ErrorOptions):TypeError;(message?:string,options?:ErrorOptions):TypeError;}interface URIErrorConstructor{new(message?:string,options?:ErrorOptions):URIError;(message?:string,options?:ErrorOptions):URIError;}interface AggregateErrorConstructor{new(errors:Iterable<any>,message?:string,options?:ErrorOptions,):AggregateError;(errors:Iterable<any>,message?:string,options?:ErrorOptions,):AggregateError;}"
    }, {
        fileName: "lib.es2016.array.include.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface Array<T>{includes(searchElement:T,fromIndex?:number):boolean;}interface ReadonlyArray<T>{includes(searchElement:T,fromIndex?:number):boolean;}interface Int8Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Uint8Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Uint8ClampedArray{includes(searchElement:number,fromIndex?:number):boolean;}interface Int16Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Uint16Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Int32Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Uint32Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Float32Array{includes(searchElement:number,fromIndex?:number):boolean;}interface Float64Array{includes(searchElement:number,fromIndex?:number):boolean;}"
    }, {
        fileName: "lib.esnext.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Intl{}"
    }, {
        fileName: "lib.es5.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"decorators\" />\n/// <reference lib=\"decorators.legacy\" />\ndeclare var NaN:number;declare var Infinity:number;declare function eval(x:string):any;declare function parseInt(string:string,radix?:number):number;declare function parseFloat(string:string):number;declare function isNaN(number:number):boolean;declare function isFinite(number:number):boolean;declare function decodeURI(encodedURI:string):string;declare function decodeURIComponent(encodedURIComponent:string):string;declare function encodeURI(uri:string):string;declare function encodeURIComponent(uriComponent:string|number|boolean):string;declare function escape(string:string):string;declare function unescape(string:string):string;interface Symbol{toString():string;valueOf():symbol;}declare type PropertyKey=string|number|symbol;interface PropertyDescriptor{configurable?:boolean;enumerable?:boolean;value?:any;writable?:boolean;get?():any;set?(v:any):void;}interface PropertyDescriptorMap{[key:PropertyKey]:PropertyDescriptor;}interface Object{constructor:Function;toString():string;toLocaleString():string;valueOf():Object;hasOwnProperty(v:PropertyKey):boolean;isPrototypeOf(v:Object):boolean;propertyIsEnumerable(v:PropertyKey):boolean;}interface ObjectConstructor{new(value?:any):Object;():any;(value:any):any;readonly prototype:Object;getPrototypeOf(o:any):any;getOwnPropertyDescriptor(o:any,p:PropertyKey):PropertyDescriptor|undefined;getOwnPropertyNames(o:any):string[];create(o:object|null):any;create(o:object|null,properties:PropertyDescriptorMap&ThisType<any>):any;defineProperty<T>(o:T,p:PropertyKey,attributes:PropertyDescriptor&ThisType<any>):T;defineProperties<T>(o:T,properties:PropertyDescriptorMap&ThisType<any>):T;seal<T>(o:T):T;freeze<T extends Function>(f:T):T;freeze<T extends{[idx:string]:U|null|undefined|object;},U extends string|bigint|number|boolean|symbol>(o:T):Readonly<T>;freeze<T>(o:T):Readonly<T>;preventExtensions<T>(o:T):T;isSealed(o:any):boolean;isFrozen(o:any):boolean;isExtensible(o:any):boolean;keys(o:object):string[];}declare var Object:ObjectConstructor;interface Function{apply(this:Function,thisArg:any,argArray?:any):any;call(this:Function,thisArg:any,...argArray:any[]):any;bind(this:Function,thisArg:any,...argArray:any[]):any;toString():string;prototype:any;readonly length:number;arguments:any;caller:Function;}interface FunctionConstructor{new(...args:string[]):Function;(...args:string[]):Function;readonly prototype:Function;}declare var Function:FunctionConstructor;type ThisParameterType<T>=T extends(this:infer U,...args:never)=>any?U:unknown;type OmitThisParameter<T>=unknown extends ThisParameterType<T>?T:T extends(...args:infer A)=>infer R?(...args:A)=>R:T;interface CallableFunction extends Function{apply<T,R>(this:(this:T)=>R,thisArg:T):R;apply<T,A extends any[],R>(this:(this:T,...args:A)=>R,thisArg:T,args:A):R;call<T,A extends any[],R>(this:(this:T,...args:A)=>R,thisArg:T,...args:A):R;bind<T>(this:T,thisArg:ThisParameterType<T>):OmitThisParameter<T>;bind<T,A extends any[],B extends any[],R>(this:(this:T,...args:[...A,...B])=>R,thisArg:T,...args:A):(...args:B)=>R;}interface NewableFunction extends Function{apply<T>(this:new()=>T,thisArg:T):void;apply<T,A extends any[]>(this:new(...args:A)=>T,thisArg:T,args:A):void;call<T,A extends any[]>(this:new(...args:A)=>T,thisArg:T,...args:A):void;bind<T>(this:T,thisArg:any):T;bind<A extends any[],B extends any[],R>(this:new(...args:[...A,...B])=>R,thisArg:any,...args:A):new(...args:B)=>R;}interface IArguments{[index:number]:any;length:number;callee:Function;}interface String{toString():string;charAt(pos:number):string;charCodeAt(index:number):number;concat(...strings:string[]):string;indexOf(searchString:string,position?:number):number;lastIndexOf(searchString:string,position?:number):number;localeCompare(that:string):number;match(regexp:string|RegExp):RegExpMatchArray|null;replace(searchValue:string|RegExp,replaceValue:string):string;replace(searchValue:string|RegExp,replacer:(substring:string,...args:any[])=>string):string;search(regexp:string|RegExp):number;slice(start?:number,end?:number):string;split(separator:string|RegExp,limit?:number):string[];substring(start:number,end?:number):string;toLowerCase():string;toLocaleLowerCase(locales?:string|string[]):string;toUpperCase():string;toLocaleUpperCase(locales?:string|string[]):string;trim():string;readonly length:number;substr(from:number,length?:number):string;valueOf():string;readonly[index:number]:string;}interface StringConstructor{new(value?:any):String;(value?:any):string;readonly prototype:String;fromCharCode(...codes:number[]):string;}declare var String:StringConstructor;interface Boolean{valueOf():boolean;}interface BooleanConstructor{new(value?:any):Boolean;<T>(value?:T):boolean;readonly prototype:Boolean;}declare var Boolean:BooleanConstructor;interface Number{toString(radix?:number):string;toFixed(fractionDigits?:number):string;toExponential(fractionDigits?:number):string;toPrecision(precision?:number):string;valueOf():number;}interface NumberConstructor{new(value?:any):Number;(value?:any):number;readonly prototype:Number;readonly MAX_VALUE:number;readonly MIN_VALUE:number;readonly NaN:number;readonly NEGATIVE_INFINITY:number;readonly POSITIVE_INFINITY:number;}declare var Number:NumberConstructor;interface TemplateStringsArray extends ReadonlyArray<string>{readonly raw:readonly string[];}interface ImportMeta{}interface ImportCallOptions{assert?:ImportAssertions;with?:ImportAttributes;}interface ImportAssertions{[key:string]:string;}interface ImportAttributes{[key:string]:string;}interface Math{readonly E:number;readonly LN10:number;readonly LN2:number;readonly LOG2E:number;readonly LOG10E:number;readonly PI:number;readonly SQRT1_2:number;readonly SQRT2:number;abs(x:number):number;acos(x:number):number;asin(x:number):number;atan(x:number):number;atan2(y:number,x:number):number;ceil(x:number):number;cos(x:number):number;exp(x:number):number;floor(x:number):number;log(x:number):number;max(...values:number[]):number;min(...values:number[]):number;pow(x:number,y:number):number;random():number;round(x:number):number;sin(x:number):number;sqrt(x:number):number;tan(x:number):number;}declare var Math:Math;interface Date{toString():string;toDateString():string;toTimeString():string;toLocaleString():string;toLocaleDateString():string;toLocaleTimeString():string;valueOf():number;getTime():number;getFullYear():number;getUTCFullYear():number;getMonth():number;getUTCMonth():number;getDate():number;getUTCDate():number;getDay():number;getUTCDay():number;getHours():number;getUTCHours():number;getMinutes():number;getUTCMinutes():number;getSeconds():number;getUTCSeconds():number;getMilliseconds():number;getUTCMilliseconds():number;getTimezoneOffset():number;setTime(time:number):number;setMilliseconds(ms:number):number;setUTCMilliseconds(ms:number):number;setSeconds(sec:number,ms?:number):number;setUTCSeconds(sec:number,ms?:number):number;setMinutes(min:number,sec?:number,ms?:number):number;setUTCMinutes(min:number,sec?:number,ms?:number):number;setHours(hours:number,min?:number,sec?:number,ms?:number):number;setUTCHours(hours:number,min?:number,sec?:number,ms?:number):number;setDate(date:number):number;setUTCDate(date:number):number;setMonth(month:number,date?:number):number;setUTCMonth(month:number,date?:number):number;setFullYear(year:number,month?:number,date?:number):number;setUTCFullYear(year:number,month?:number,date?:number):number;toUTCString():string;toISOString():string;toJSON(key?:any):string;}interface DateConstructor{new():Date;new(value:number|string):Date;new(year:number,monthIndex:number,date?:number,hours?:number,minutes?:number,seconds?:number,ms?:number):Date;():string;readonly prototype:Date;parse(s:string):number;UTC(year:number,monthIndex:number,date?:number,hours?:number,minutes?:number,seconds?:number,ms?:number):number;now():number;}declare var Date:DateConstructor;interface RegExpMatchArray extends Array<string>{index?:number;input?:string;0:string;}interface RegExpExecArray extends Array<string>{index:number;input:string;0:string;}interface RegExp{exec(string:string):RegExpExecArray|null;test(string:string):boolean;readonly source:string;readonly global:boolean;readonly ignoreCase:boolean;readonly multiline:boolean;lastIndex:number;compile(pattern:string,flags?:string):this;}interface RegExpConstructor{new(pattern:RegExp|string):RegExp;new(pattern:string,flags?:string):RegExp;(pattern:RegExp|string):RegExp;(pattern:string,flags?:string):RegExp;readonly\"prototype\":RegExp;\"$1\":string;\"$2\":string;\"$3\":string;\"$4\":string;\"$5\":string;\"$6\":string;\"$7\":string;\"$8\":string;\"$9\":string;\"input\":string;\"$_\":string;\"lastMatch\":string;\"$&\":string;\"lastParen\":string;\"$+\":string;\"leftContext\":string;\"$`\":string;\"rightContext\":string;\"$'\":string;}declare var RegExp:RegExpConstructor;interface Error{name:string;message:string;stack?:string;}interface ErrorConstructor{new(message?:string):Error;(message?:string):Error;readonly prototype:Error;}declare var Error:ErrorConstructor;interface EvalError extends Error{}interface EvalErrorConstructor extends ErrorConstructor{new(message?:string):EvalError;(message?:string):EvalError;readonly prototype:EvalError;}declare var EvalError:EvalErrorConstructor;interface RangeError extends Error{}interface RangeErrorConstructor extends ErrorConstructor{new(message?:string):RangeError;(message?:string):RangeError;readonly prototype:RangeError;}declare var RangeError:RangeErrorConstructor;interface ReferenceError extends Error{}interface ReferenceErrorConstructor extends ErrorConstructor{new(message?:string):ReferenceError;(message?:string):ReferenceError;readonly prototype:ReferenceError;}declare var ReferenceError:ReferenceErrorConstructor;interface SyntaxError extends Error{}interface SyntaxErrorConstructor extends ErrorConstructor{new(message?:string):SyntaxError;(message?:string):SyntaxError;readonly prototype:SyntaxError;}declare var SyntaxError:SyntaxErrorConstructor;interface TypeError extends Error{}interface TypeErrorConstructor extends ErrorConstructor{new(message?:string):TypeError;(message?:string):TypeError;readonly prototype:TypeError;}declare var TypeError:TypeErrorConstructor;interface URIError extends Error{}interface URIErrorConstructor extends ErrorConstructor{new(message?:string):URIError;(message?:string):URIError;readonly prototype:URIError;}declare var URIError:URIErrorConstructor;interface JSON{parse(text:string,reviver?:(this:any,key:string,value:any)=>any):any;stringify(value:any,replacer?:(this:any,key:string,value:any)=>any,space?:string|number):string;stringify(value:any,replacer?:(number|string)[]|null,space?:string|number):string;}declare var JSON:JSON;interface ReadonlyArray<T>{readonly length:number;toString():string;toLocaleString():string;concat(...items:ConcatArray<T>[]):T[];concat(...items:(T|ConcatArray<T>)[]):T[];join(separator?:string):string;slice(start?:number,end?:number):T[];indexOf(searchElement:T,fromIndex?:number):number;lastIndexOf(searchElement:T,fromIndex?:number):number;every<S extends T>(predicate:(value:T,index:number,array:readonly T[])=>value is S,thisArg?:any):this is readonly S[];every(predicate:(value:T,index:number,array:readonly T[])=>unknown,thisArg?:any):boolean;some(predicate:(value:T,index:number,array:readonly T[])=>unknown,thisArg?:any):boolean;forEach(callbackfn:(value:T,index:number,array:readonly T[])=>void,thisArg?:any):void;map<U>(callbackfn:(value:T,index:number,array:readonly T[])=>U,thisArg?:any):U[];filter<S extends T>(predicate:(value:T,index:number,array:readonly T[])=>value is S,thisArg?:any):S[];filter(predicate:(value:T,index:number,array:readonly T[])=>unknown,thisArg?:any):T[];reduce(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:readonly T[])=>T):T;reduce(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:readonly T[])=>T,initialValue:T):T;reduce<U>(callbackfn:(previousValue:U,currentValue:T,currentIndex:number,array:readonly T[])=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:readonly T[])=>T):T;reduceRight(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:readonly T[])=>T,initialValue:T):T;reduceRight<U>(callbackfn:(previousValue:U,currentValue:T,currentIndex:number,array:readonly T[])=>U,initialValue:U):U;readonly[n:number]:T;}interface ConcatArray<T>{readonly length:number;readonly[n:number]:T;join(separator?:string):string;slice(start?:number,end?:number):T[];}interface Array<T>{length:number;toString():string;toLocaleString():string;pop():T|undefined;push(...items:T[]):number;concat(...items:ConcatArray<T>[]):T[];concat(...items:(T|ConcatArray<T>)[]):T[];join(separator?:string):string;reverse():T[];shift():T|undefined;slice(start?:number,end?:number):T[];sort(compareFn?:(a:T,b:T)=>number):this;splice(start:number,deleteCount?:number):T[];splice(start:number,deleteCount:number,...items:T[]):T[];unshift(...items:T[]):number;indexOf(searchElement:T,fromIndex?:number):number;lastIndexOf(searchElement:T,fromIndex?:number):number;every<S extends T>(predicate:(value:T,index:number,array:T[])=>value is S,thisArg?:any):this is S[];every(predicate:(value:T,index:number,array:T[])=>unknown,thisArg?:any):boolean;some(predicate:(value:T,index:number,array:T[])=>unknown,thisArg?:any):boolean;forEach(callbackfn:(value:T,index:number,array:T[])=>void,thisArg?:any):void;map<U>(callbackfn:(value:T,index:number,array:T[])=>U,thisArg?:any):U[];filter<S extends T>(predicate:(value:T,index:number,array:T[])=>value is S,thisArg?:any):S[];filter(predicate:(value:T,index:number,array:T[])=>unknown,thisArg?:any):T[];reduce(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:T[])=>T):T;reduce(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:T[])=>T,initialValue:T):T;reduce<U>(callbackfn:(previousValue:U,currentValue:T,currentIndex:number,array:T[])=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:T[])=>T):T;reduceRight(callbackfn:(previousValue:T,currentValue:T,currentIndex:number,array:T[])=>T,initialValue:T):T;reduceRight<U>(callbackfn:(previousValue:U,currentValue:T,currentIndex:number,array:T[])=>U,initialValue:U):U;[n:number]:T;}interface ArrayConstructor{new(arrayLength?:number):any[];new<T>(arrayLength:number):T[];new<T>(...items:T[]):T[];(arrayLength?:number):any[];<T>(arrayLength:number):T[];<T>(...items:T[]):T[];isArray(arg:any):arg is any[];readonly prototype:any[];}declare var Array:ArrayConstructor;interface TypedPropertyDescriptor<T>{enumerable?:boolean;configurable?:boolean;writable?:boolean;value?:T;get?:()=>T;set?:(value:T)=>void;}declare type PromiseConstructorLike=new<T>(executor:(resolve:(value:T|PromiseLike<T>)=>void,reject:(reason?:any)=>void)=>void)=>PromiseLike<T>;interface PromiseLike<T>{then<TResult1=T,TResult2=never>(onfulfilled?:((value:T)=>TResult1|PromiseLike<TResult1>)|undefined|null,onrejected?:((reason:any)=>TResult2|PromiseLike<TResult2>)|undefined|null):PromiseLike<TResult1|TResult2>;}interface Promise<T>{then<TResult1=T,TResult2=never>(onfulfilled?:((value:T)=>TResult1|PromiseLike<TResult1>)|undefined|null,onrejected?:((reason:any)=>TResult2|PromiseLike<TResult2>)|undefined|null):Promise<TResult1|TResult2>;catch<TResult=never>(onrejected?:((reason:any)=>TResult|PromiseLike<TResult>)|undefined|null):Promise<T|TResult>;}type Awaited<T>=T extends null|undefined?T:T extends object&{then(onfulfilled:infer F,...args:infer _):any;}?\nF extends((value:infer V,...args:infer _)=>any)?\nAwaited<V>:never:T;interface ArrayLike<T>{readonly length:number;readonly[n:number]:T;}type Partial<T>={[P in keyof T]?:T[P];};type Required<T>={[P in keyof T]-?:T[P];};type Readonly<T>={readonly[P in keyof T]:T[P];};type Pick<T,K extends keyof T>={[P in K]:T[P];};type Record<K extends keyof any,T>={[P in K]:T;};type Exclude<T,U>=T extends U?never:T;type Extract<T,U>=T extends U?T:never;type Omit<T,K extends keyof any>=Pick<T,Exclude<keyof T,K>>;type NonNullable<T>=T&{};type Parameters<T extends(...args:any)=>any>=T extends(...args:infer P)=>any?P:never;type ConstructorParameters<T extends abstract new(...args:any)=>any>=T extends abstract new(...args:infer P)=>any?P:never;type ReturnType<T extends(...args:any)=>any>=T extends(...args:any)=>infer R?R:any;type InstanceType<T extends abstract new(...args:any)=>any>=T extends abstract new(...args:any)=>infer R?R:any;type Uppercase<S extends string>=intrinsic;type Lowercase<S extends string>=intrinsic;type Capitalize<S extends string>=intrinsic;type Uncapitalize<S extends string>=intrinsic;type NoInfer<T>=intrinsic;interface ThisType<T>{}interface WeakKeyTypes{object:object;}type WeakKey=WeakKeyTypes[keyof WeakKeyTypes];interface ArrayBuffer{readonly byteLength:number;slice(begin:number,end?:number):ArrayBuffer;}interface ArrayBufferTypes{ArrayBuffer:ArrayBuffer;}type ArrayBufferLike=ArrayBufferTypes[keyof ArrayBufferTypes];interface ArrayBufferConstructor{readonly prototype:ArrayBuffer;new(byteLength:number):ArrayBuffer;isView(arg:any):arg is ArrayBufferView;}declare var ArrayBuffer:ArrayBufferConstructor;interface ArrayBufferView{buffer:ArrayBufferLike;byteLength:number;byteOffset:number;}interface DataView{readonly buffer:ArrayBuffer;readonly byteLength:number;readonly byteOffset:number;getFloat32(byteOffset:number,littleEndian?:boolean):number;getFloat64(byteOffset:number,littleEndian?:boolean):number;getInt8(byteOffset:number):number;getInt16(byteOffset:number,littleEndian?:boolean):number;getInt32(byteOffset:number,littleEndian?:boolean):number;getUint8(byteOffset:number):number;getUint16(byteOffset:number,littleEndian?:boolean):number;getUint32(byteOffset:number,littleEndian?:boolean):number;setFloat32(byteOffset:number,value:number,littleEndian?:boolean):void;setFloat64(byteOffset:number,value:number,littleEndian?:boolean):void;setInt8(byteOffset:number,value:number):void;setInt16(byteOffset:number,value:number,littleEndian?:boolean):void;setInt32(byteOffset:number,value:number,littleEndian?:boolean):void;setUint8(byteOffset:number,value:number):void;setUint16(byteOffset:number,value:number,littleEndian?:boolean):void;setUint32(byteOffset:number,value:number,littleEndian?:boolean):void;}interface DataViewConstructor{readonly prototype:DataView;new(buffer:ArrayBufferLike&{BYTES_PER_ELEMENT?:never;},byteOffset?:number,byteLength?:number):DataView;}declare var DataView:DataViewConstructor;interface Int8Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Int8Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Int8Array)=>any,thisArg?:any):Int8Array;find(predicate:(value:number,index:number,obj:Int8Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Int8Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Int8Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Int8Array)=>number,thisArg?:any):Int8Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int8Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int8Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Int8Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int8Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int8Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Int8Array)=>U,initialValue:U):U;reverse():Int8Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Int8Array;some(predicate:(value:number,index:number,array:Int8Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Int8Array;toLocaleString():string;toString():string;valueOf():Int8Array;[index:number]:number;}interface Int8ArrayConstructor{readonly prototype:Int8Array;new(length:number):Int8Array;new(array:ArrayLike<number>|ArrayBufferLike):Int8Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Int8Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Int8Array;from(arrayLike:ArrayLike<number>):Int8Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Int8Array;}declare var Int8Array:Int8ArrayConstructor;interface Uint8Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Uint8Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Uint8Array)=>any,thisArg?:any):Uint8Array;find(predicate:(value:number,index:number,obj:Uint8Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Uint8Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Uint8Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Uint8Array)=>number,thisArg?:any):Uint8Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint8Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint8Array)=>U,initialValue:U):U;reverse():Uint8Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Uint8Array;some(predicate:(value:number,index:number,array:Uint8Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Uint8Array;toLocaleString():string;toString():string;valueOf():Uint8Array;[index:number]:number;}interface Uint8ArrayConstructor{readonly prototype:Uint8Array;new(length:number):Uint8Array;new(array:ArrayLike<number>|ArrayBufferLike):Uint8Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Uint8Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Uint8Array;from(arrayLike:ArrayLike<number>):Uint8Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Uint8Array;}declare var Uint8Array:Uint8ArrayConstructor;interface Uint8ClampedArray{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Uint8ClampedArray)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Uint8ClampedArray)=>any,thisArg?:any):Uint8ClampedArray;find(predicate:(value:number,index:number,obj:Uint8ClampedArray)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Uint8ClampedArray)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Uint8ClampedArray)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Uint8ClampedArray)=>number,thisArg?:any):Uint8ClampedArray;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8ClampedArray)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8ClampedArray)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint8ClampedArray)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8ClampedArray)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint8ClampedArray)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint8ClampedArray)=>U,initialValue:U):U;reverse():Uint8ClampedArray;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Uint8ClampedArray;some(predicate:(value:number,index:number,array:Uint8ClampedArray)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Uint8ClampedArray;toLocaleString():string;toString():string;valueOf():Uint8ClampedArray;[index:number]:number;}interface Uint8ClampedArrayConstructor{readonly prototype:Uint8ClampedArray;new(length:number):Uint8ClampedArray;new(array:ArrayLike<number>|ArrayBufferLike):Uint8ClampedArray;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Uint8ClampedArray;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Uint8ClampedArray;from(arrayLike:ArrayLike<number>):Uint8ClampedArray;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Uint8ClampedArray;}declare var Uint8ClampedArray:Uint8ClampedArrayConstructor;interface Int16Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Int16Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Int16Array)=>any,thisArg?:any):Int16Array;find(predicate:(value:number,index:number,obj:Int16Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Int16Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Int16Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Int16Array)=>number,thisArg?:any):Int16Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int16Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int16Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Int16Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int16Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int16Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Int16Array)=>U,initialValue:U):U;reverse():Int16Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Int16Array;some(predicate:(value:number,index:number,array:Int16Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Int16Array;toLocaleString():string;toString():string;valueOf():Int16Array;[index:number]:number;}interface Int16ArrayConstructor{readonly prototype:Int16Array;new(length:number):Int16Array;new(array:ArrayLike<number>|ArrayBufferLike):Int16Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Int16Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Int16Array;from(arrayLike:ArrayLike<number>):Int16Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Int16Array;}declare var Int16Array:Int16ArrayConstructor;interface Uint16Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Uint16Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Uint16Array)=>any,thisArg?:any):Uint16Array;find(predicate:(value:number,index:number,obj:Uint16Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Uint16Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Uint16Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Uint16Array)=>number,thisArg?:any):Uint16Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint16Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint16Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint16Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint16Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint16Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint16Array)=>U,initialValue:U):U;reverse():Uint16Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Uint16Array;some(predicate:(value:number,index:number,array:Uint16Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Uint16Array;toLocaleString():string;toString():string;valueOf():Uint16Array;[index:number]:number;}interface Uint16ArrayConstructor{readonly prototype:Uint16Array;new(length:number):Uint16Array;new(array:ArrayLike<number>|ArrayBufferLike):Uint16Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Uint16Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Uint16Array;from(arrayLike:ArrayLike<number>):Uint16Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Uint16Array;}declare var Uint16Array:Uint16ArrayConstructor;interface Int32Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Int32Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Int32Array)=>any,thisArg?:any):Int32Array;find(predicate:(value:number,index:number,obj:Int32Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Int32Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Int32Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Int32Array)=>number,thisArg?:any):Int32Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int32Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int32Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Int32Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int32Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Int32Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Int32Array)=>U,initialValue:U):U;reverse():Int32Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Int32Array;some(predicate:(value:number,index:number,array:Int32Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Int32Array;toLocaleString():string;toString():string;valueOf():Int32Array;[index:number]:number;}interface Int32ArrayConstructor{readonly prototype:Int32Array;new(length:number):Int32Array;new(array:ArrayLike<number>|ArrayBufferLike):Int32Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Int32Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Int32Array;from(arrayLike:ArrayLike<number>):Int32Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Int32Array;}declare var Int32Array:Int32ArrayConstructor;interface Uint32Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Uint32Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Uint32Array)=>any,thisArg?:any):Uint32Array;find(predicate:(value:number,index:number,obj:Uint32Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Uint32Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Uint32Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Uint32Array)=>number,thisArg?:any):Uint32Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint32Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint32Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint32Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint32Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Uint32Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Uint32Array)=>U,initialValue:U):U;reverse():Uint32Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Uint32Array;some(predicate:(value:number,index:number,array:Uint32Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Uint32Array;toLocaleString():string;toString():string;valueOf():Uint32Array;[index:number]:number;}interface Uint32ArrayConstructor{readonly prototype:Uint32Array;new(length:number):Uint32Array;new(array:ArrayLike<number>|ArrayBufferLike):Uint32Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Uint32Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Uint32Array;from(arrayLike:ArrayLike<number>):Uint32Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Uint32Array;}declare var Uint32Array:Uint32ArrayConstructor;interface Float32Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Float32Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Float32Array)=>any,thisArg?:any):Float32Array;find(predicate:(value:number,index:number,obj:Float32Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Float32Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Float32Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Float32Array)=>number,thisArg?:any):Float32Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float32Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float32Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Float32Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float32Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float32Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Float32Array)=>U,initialValue:U):U;reverse():Float32Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Float32Array;some(predicate:(value:number,index:number,array:Float32Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Float32Array;toLocaleString():string;toString():string;valueOf():Float32Array;[index:number]:number;}interface Float32ArrayConstructor{readonly prototype:Float32Array;new(length:number):Float32Array;new(array:ArrayLike<number>|ArrayBufferLike):Float32Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Float32Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Float32Array;from(arrayLike:ArrayLike<number>):Float32Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Float32Array;}declare var Float32Array:Float32ArrayConstructor;interface Float64Array{readonly BYTES_PER_ELEMENT:number;readonly buffer:ArrayBufferLike;readonly byteLength:number;readonly byteOffset:number;copyWithin(target:number,start:number,end?:number):this;every(predicate:(value:number,index:number,array:Float64Array)=>unknown,thisArg?:any):boolean;fill(value:number,start?:number,end?:number):this;filter(predicate:(value:number,index:number,array:Float64Array)=>any,thisArg?:any):Float64Array;find(predicate:(value:number,index:number,obj:Float64Array)=>boolean,thisArg?:any):number|undefined;findIndex(predicate:(value:number,index:number,obj:Float64Array)=>boolean,thisArg?:any):number;forEach(callbackfn:(value:number,index:number,array:Float64Array)=>void,thisArg?:any):void;indexOf(searchElement:number,fromIndex?:number):number;join(separator?:string):string;lastIndexOf(searchElement:number,fromIndex?:number):number;readonly length:number;map(callbackfn:(value:number,index:number,array:Float64Array)=>number,thisArg?:any):Float64Array;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float64Array)=>number):number;reduce(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float64Array)=>number,initialValue:number):number;reduce<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Float64Array)=>U,initialValue:U):U;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float64Array)=>number):number;reduceRight(callbackfn:(previousValue:number,currentValue:number,currentIndex:number,array:Float64Array)=>number,initialValue:number):number;reduceRight<U>(callbackfn:(previousValue:U,currentValue:number,currentIndex:number,array:Float64Array)=>U,initialValue:U):U;reverse():Float64Array;set(array:ArrayLike<number>,offset?:number):void;slice(start?:number,end?:number):Float64Array;some(predicate:(value:number,index:number,array:Float64Array)=>unknown,thisArg?:any):boolean;sort(compareFn?:(a:number,b:number)=>number):this;subarray(begin?:number,end?:number):Float64Array;toLocaleString():string;toString():string;valueOf():Float64Array;[index:number]:number;}interface Float64ArrayConstructor{readonly prototype:Float64Array;new(length:number):Float64Array;new(array:ArrayLike<number>|ArrayBufferLike):Float64Array;new(buffer:ArrayBufferLike,byteOffset?:number,length?:number):Float64Array;readonly BYTES_PER_ELEMENT:number;of(...items:number[]):Float64Array;from(arrayLike:ArrayLike<number>):Float64Array;from<T>(arrayLike:ArrayLike<T>,mapfn:(v:T,k:number)=>number,thisArg?:any):Float64Array;}declare var Float64Array:Float64ArrayConstructor;declare namespace Intl{interface CollatorOptions{usage?:\"sort\"|\"search\"|undefined;localeMatcher?:\"lookup\"|\"best fit\"|undefined;numeric?:boolean|undefined;caseFirst?:\"upper\"|\"lower\"|\"false\"|undefined;sensitivity?:\"base\"|\"accent\"|\"case\"|\"variant\"|undefined;collation?:\"big5han\"|\"compat\"|\"dict\"|\"direct\"|\"ducet\"|\"emoji\"|\"eor\"|\"gb2312\"|\"phonebk\"|\"phonetic\"|\"pinyin\"|\"reformed\"|\"searchjl\"|\"stroke\"|\"trad\"|\"unihan\"|\"zhuyin\"|undefined;ignorePunctuation?:boolean|undefined;}interface ResolvedCollatorOptions{locale:string;usage:string;sensitivity:string;ignorePunctuation:boolean;collation:string;caseFirst:string;numeric:boolean;}interface Collator{compare(x:string,y:string):number;resolvedOptions():ResolvedCollatorOptions;}interface CollatorConstructor{new(locales?:string|string[],options?:CollatorOptions):Collator;(locales?:string|string[],options?:CollatorOptions):Collator;supportedLocalesOf(locales:string|string[],options?:CollatorOptions):string[];}var Collator:CollatorConstructor;interface NumberFormatOptionsStyleRegistry{decimal:never;percent:never;currency:never;}type NumberFormatOptionsStyle=keyof NumberFormatOptionsStyleRegistry;interface NumberFormatOptionsCurrencyDisplayRegistry{code:never;symbol:never;name:never;}type NumberFormatOptionsCurrencyDisplay=keyof NumberFormatOptionsCurrencyDisplayRegistry;interface NumberFormatOptionsUseGroupingRegistry{}type NumberFormatOptionsUseGrouping={}extends NumberFormatOptionsUseGroupingRegistry?boolean:keyof NumberFormatOptionsUseGroupingRegistry|\"true\"|\"false\"|boolean;type ResolvedNumberFormatOptionsUseGrouping={}extends NumberFormatOptionsUseGroupingRegistry?boolean:keyof NumberFormatOptionsUseGroupingRegistry|false;interface NumberFormatOptions{localeMatcher?:\"lookup\"|\"best fit\"|undefined;style?:NumberFormatOptionsStyle|undefined;currency?:string|undefined;currencyDisplay?:NumberFormatOptionsCurrencyDisplay|undefined;useGrouping?:NumberFormatOptionsUseGrouping|undefined;minimumIntegerDigits?:number|undefined;minimumFractionDigits?:number|undefined;maximumFractionDigits?:number|undefined;minimumSignificantDigits?:number|undefined;maximumSignificantDigits?:number|undefined;}interface ResolvedNumberFormatOptions{locale:string;numberingSystem:string;style:NumberFormatOptionsStyle;currency?:string;currencyDisplay?:NumberFormatOptionsCurrencyDisplay;minimumIntegerDigits:number;minimumFractionDigits?:number;maximumFractionDigits?:number;minimumSignificantDigits?:number;maximumSignificantDigits?:number;useGrouping:ResolvedNumberFormatOptionsUseGrouping;}interface NumberFormat{format(value:number):string;resolvedOptions():ResolvedNumberFormatOptions;}interface NumberFormatConstructor{new(locales?:string|string[],options?:NumberFormatOptions):NumberFormat;(locales?:string|string[],options?:NumberFormatOptions):NumberFormat;supportedLocalesOf(locales:string|string[],options?:NumberFormatOptions):string[];readonly prototype:NumberFormat;}var NumberFormat:NumberFormatConstructor;interface DateTimeFormatOptions{localeMatcher?:\"best fit\"|\"lookup\"|undefined;weekday?:\"long\"|\"short\"|\"narrow\"|undefined;era?:\"long\"|\"short\"|\"narrow\"|undefined;year?:\"numeric\"|\"2-digit\"|undefined;month?:\"numeric\"|\"2-digit\"|\"long\"|\"short\"|\"narrow\"|undefined;day?:\"numeric\"|\"2-digit\"|undefined;hour?:\"numeric\"|\"2-digit\"|undefined;minute?:\"numeric\"|\"2-digit\"|undefined;second?:\"numeric\"|\"2-digit\"|undefined;timeZoneName?:\"short\"|\"long\"|\"shortOffset\"|\"longOffset\"|\"shortGeneric\"|\"longGeneric\"|undefined;formatMatcher?:\"best fit\"|\"basic\"|undefined;hour12?:boolean|undefined;timeZone?:string|undefined;}interface ResolvedDateTimeFormatOptions{locale:string;calendar:string;numberingSystem:string;timeZone:string;hour12?:boolean;weekday?:string;era?:string;year?:string;month?:string;day?:string;hour?:string;minute?:string;second?:string;timeZoneName?:string;}interface DateTimeFormat{format(date?:Date|number):string;resolvedOptions():ResolvedDateTimeFormatOptions;}interface DateTimeFormatConstructor{new(locales?:string|string[],options?:DateTimeFormatOptions):DateTimeFormat;(locales?:string|string[],options?:DateTimeFormatOptions):DateTimeFormat;supportedLocalesOf(locales:string|string[],options?:DateTimeFormatOptions):string[];readonly prototype:DateTimeFormat;}var DateTimeFormat:DateTimeFormatConstructor;}interface String{localeCompare(that:string,locales?:string|string[],options?:Intl.CollatorOptions):number;}interface Number{toLocaleString(locales?:string|string[],options?:Intl.NumberFormatOptions):string;}interface Date{toLocaleString(locales?:string|string[],options?:Intl.DateTimeFormatOptions):string;toLocaleDateString(locales?:string|string[],options?:Intl.DateTimeFormatOptions):string;toLocaleTimeString(locales?:string|string[],options?:Intl.DateTimeFormatOptions):string;}"
    }, {
        fileName: "lib.esnext.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2023\" />\n/// <reference lib=\"esnext.intl\" />\n/// <reference lib=\"esnext.decorators\" />\n/// <reference lib=\"esnext.disposable\" />\n/// <reference lib=\"esnext.promise\" />\n/// <reference lib=\"esnext.object\" />\n/// <reference lib=\"esnext.collection\" />\n/// <reference lib=\"esnext.array\" />\n/// <reference lib=\"esnext.regexp\" />\n/// <reference lib=\"esnext.string\" />\n/// <reference lib=\"esnext.iterator\" />\n"
    }, {
        fileName: "lib.es2023.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Intl{interface NumberFormatOptionsUseGroupingRegistry{min2:never;auto:never;always:never;}interface NumberFormatOptionsSignDisplayRegistry{negative:never;}interface NumberFormatOptions{roundingPriority?:\"auto\"|\"morePrecision\"|\"lessPrecision\"|undefined;roundingIncrement?:1|2|5|10|20|25|50|100|200|250|500|1000|2000|2500|5000|undefined;roundingMode?:\"ceil\"|\"floor\"|\"expand\"|\"trunc\"|\"halfCeil\"|\"halfFloor\"|\"halfExpand\"|\"halfTrunc\"|\"halfEven\"|undefined;trailingZeroDisplay?:\"auto\"|\"stripIfInteger\"|undefined;}interface ResolvedNumberFormatOptions{roundingPriority:\"auto\"|\"morePrecision\"|\"lessPrecision\";roundingMode:\"ceil\"|\"floor\"|\"expand\"|\"trunc\"|\"halfCeil\"|\"halfFloor\"|\"halfExpand\"|\"halfTrunc\"|\"halfEven\";roundingIncrement:1|2|5|10|20|25|50|100|200|250|500|1000|2000|2500|5000;trailingZeroDisplay:\"auto\"|\"stripIfInteger\";}interface NumberRangeFormatPart extends NumberFormatPart{source:\"startRange\"|\"endRange\"|\"shared\";}type StringNumericLiteral=`${number}` | \"Infinity\" | \"-Infinity\" | \"+Infinity\";\n\n    interface NumberFormat {\n        format(value: number | bigint | StringNumericLiteral): string;\n        formatToParts(value: number | bigint | StringNumericLiteral): NumberFormatPart[];\n        formatRange(start: number | bigint | StringNumericLiteral, end: number | bigint | StringNumericLiteral): string;\n        formatRangeToParts(start: number | bigint | StringNumericLiteral, end: number | bigint | StringNumericLiteral): NumberRangeFormatPart[];\n    }\n}\n"
    }, {
        fileName: "lib.esnext.object.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface ObjectConstructor{groupBy<K extends PropertyKey,T>(items:Iterable<T>,keySelector:(item:T,index:number)=>K,):Partial<Record<K,T[]>>;}"
    }, {
        fileName: "lib.es2019.array.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ntype FlatArray<Arr,Depth extends number>={done:Arr;recur:Arr extends ReadonlyArray<infer InnerArr>?FlatArray<InnerArr,[-1,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20][Depth]>:Arr;}[Depth extends-1?\"done\":\"recur\"];interface ReadonlyArray<T>{flatMap<U,This=undefined>(callback:(this:This,value:T,index:number,array:T[])=>U|ReadonlyArray<U>,thisArg?:This,):U[];flat<A,D extends number=1>(this:A,depth?:D,):FlatArray<A,D>[];}interface Array<T>{flatMap<U,This=undefined>(callback:(this:This,value:T,index:number,array:T[])=>U|ReadonlyArray<U>,thisArg?:This,):U[];flat<A,D extends number=1>(this:A,depth?:D,):FlatArray<A,D>[];}"
    }, {
        fileName: "lib.esnext.string.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface String{isWellFormed():boolean;toWellFormed():string;}"
    }, {
        fileName: "lib.es2016.intl.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ndeclare namespace Intl{function getCanonicalLocales(locale?:string|readonly string[]):string[];}"
    }, {
        fileName: "lib.es2020.date.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2020.intl\" />\ninterface Date{toLocaleString(locales?:Intl.LocalesArgument,options?:Intl.DateTimeFormatOptions):string;toLocaleDateString(locales?:Intl.LocalesArgument,options?:Intl.DateTimeFormatOptions):string;toLocaleTimeString(locales?:Intl.LocalesArgument,options?:Intl.DateTimeFormatOptions):string;}"
    }, {
        fileName: "lib.webworker.asynciterable.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface FileSystemDirectoryHandleAsyncIterator<T>extends AsyncIteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.asyncIterator]():FileSystemDirectoryHandleAsyncIterator<T>;}interface FileSystemDirectoryHandle{[Symbol.asyncIterator]():FileSystemDirectoryHandleAsyncIterator<[string,FileSystemHandle]>;entries():FileSystemDirectoryHandleAsyncIterator<[string,FileSystemHandle]>;keys():FileSystemDirectoryHandleAsyncIterator<string>;values():FileSystemDirectoryHandleAsyncIterator<FileSystemHandle>;}interface ReadableStreamAsyncIterator<T>extends AsyncIteratorObject<T,BuiltinIteratorReturn,unknown>{[Symbol.asyncIterator]():ReadableStreamAsyncIterator<T>;}interface ReadableStream<R=any>{[Symbol.asyncIterator](options?:ReadableStreamIteratorOptions):ReadableStreamAsyncIterator<R>;values(options?:ReadableStreamIteratorOptions):ReadableStreamAsyncIterator<R>;}"
    }, {
        fileName: "lib.es2022.full.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2022\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n/// <reference lib=\"dom.iterable\" />\n/// <reference lib=\"dom.asynciterable\" />\n"
    }, {
        fileName: "lib.es2017.full.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\n/// <reference lib=\"es2017\" />\n/// <reference lib=\"dom\" />\n/// <reference lib=\"webworker.importscripts\" />\n/// <reference lib=\"scripthost\" />\n/// <reference lib=\"dom.iterable\" />\n"
    }, {
        fileName: "lib.es2020.promise.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface PromiseFulfilledResult<T>{status:\"fulfilled\";value:T;}interface PromiseRejectedResult{status:\"rejected\";reason:any;}type PromiseSettledResult<T>=PromiseFulfilledResult<T>|PromiseRejectedResult;interface PromiseConstructor{allSettled<T extends readonly unknown[]|[]>(values:T):Promise<{-readonly[P in keyof T]:PromiseSettledResult<Awaited<T[P]>>;}>;allSettled<T>(values:Iterable<T|PromiseLike<T>>):Promise<PromiseSettledResult<Awaited<T>>[]>;}"
    }, {
        fileName: "lib.dom.d.ts",
        text: "/// <reference no-default-lib=\"true\"/>\ninterface AddEventListenerOptions extends EventListenerOptions{once?:boolean;passive?:boolean;signal?:AbortSignal;}interface AesCbcParams extends Algorithm{iv:BufferSource;}interface AesCtrParams extends Algorithm{counter:BufferSource;length:number;}interface AesDerivedKeyParams extends Algorithm{length:number;}interface AesGcmParams extends Algorithm{additionalData?:BufferSource;iv:BufferSource;tagLength?:number;}interface AesKeyAlgorithm extends KeyAlgorithm{length:number;}interface AesKeyGenParams extends Algorithm{length:number;}interface Algorithm{name:string;}interface AnalyserOptions extends AudioNodeOptions{fftSize?:number;maxDecibels?:number;minDecibels?:number;smoothingTimeConstant?:number;}interface AnimationEventInit extends EventInit{animationName?:string;elapsedTime?:number;pseudoElement?:string;}interface AnimationPlaybackEventInit extends EventInit{currentTime?:CSSNumberish|null;timelineTime?:CSSNumberish|null;}interface AssignedNodesOptions{flatten?:boolean;}interface AudioBufferOptions{length:number;numberOfChannels?:number;sampleRate:number;}interface AudioBufferSourceOptions{buffer?:AudioBuffer|null;detune?:number;loop?:boolean;loopEnd?:number;loopStart?:number;playbackRate?:number;}interface AudioConfiguration{bitrate?:number;channels?:string;contentType:string;samplerate?:number;spatialRendering?:boolean;}interface AudioContextOptions{latencyHint?:AudioContextLatencyCategory|number;sampleRate?:number;}interface AudioNodeOptions{channelCount?:number;channelCountMode?:ChannelCountMode;channelInterpretation?:ChannelInterpretation;}interface AudioProcessingEventInit extends EventInit{inputBuffer:AudioBuffer;outputBuffer:AudioBuffer;playbackTime:number;}interface AudioTimestamp{contextTime?:number;performanceTime?:DOMHighResTimeStamp;}interface AudioWorkletNodeOptions extends AudioNodeOptions{numberOfInputs?:number;numberOfOutputs?:number;outputChannelCount?:number[];parameterData?:Record<string,number>;processorOptions?:any;}interface AuthenticationExtensionsClientInputs{appid?:string;credProps?:boolean;hmacCreateSecret?:boolean;minPinLength?:boolean;}interface AuthenticationExtensionsClientOutputs{appid?:boolean;credProps?:CredentialPropertiesOutput;hmacCreateSecret?:boolean;}interface AuthenticatorSelectionCriteria{authenticatorAttachment?:AuthenticatorAttachment;requireResidentKey?:boolean;residentKey?:ResidentKeyRequirement;userVerification?:UserVerificationRequirement;}interface AvcEncoderConfig{format?:AvcBitstreamFormat;}interface BiquadFilterOptions extends AudioNodeOptions{Q?:number;detune?:number;frequency?:number;gain?:number;type?:BiquadFilterType;}interface BlobEventInit{data:Blob;timecode?:DOMHighResTimeStamp;}interface BlobPropertyBag{endings?:EndingType;type?:string;}interface CSSMatrixComponentOptions{is2D?:boolean;}interface CSSNumericType{angle?:number;flex?:number;frequency?:number;length?:number;percent?:number;percentHint?:CSSNumericBaseType;resolution?:number;time?:number;}interface CSSStyleSheetInit{baseURL?:string;disabled?:boolean;media?:MediaList|string;}interface CacheQueryOptions{ignoreMethod?:boolean;ignoreSearch?:boolean;ignoreVary?:boolean;}interface CanvasRenderingContext2DSettings{alpha?:boolean;colorSpace?:PredefinedColorSpace;desynchronized?:boolean;willReadFrequently?:boolean;}interface ChannelMergerOptions extends AudioNodeOptions{numberOfInputs?:number;}interface ChannelSplitterOptions extends AudioNodeOptions{numberOfOutputs?:number;}interface CheckVisibilityOptions{checkOpacity?:boolean;checkVisibilityCSS?:boolean;contentVisibilityAuto?:boolean;opacityProperty?:boolean;visibilityProperty?:boolean;}interface ClientQueryOptions{includeUncontrolled?:boolean;type?:ClientTypes;}interface ClipboardEventInit extends EventInit{clipboardData?:DataTransfer|null;}interface ClipboardItemOptions{presentationStyle?:PresentationStyle;}interface CloseEventInit extends EventInit{code?:number;reason?:string;wasClean?:boolean;}interface CompositionEventInit extends UIEventInit{data?:string;}interface ComputedEffectTiming extends EffectTiming{activeDuration?:CSSNumberish;currentIteration?:number|null;endTime?:CSSNumberish;localTime?:CSSNumberish|null;progress?:number|null;startTime?:CSSNumberish;}interface ComputedKeyframe{composite:CompositeOperationOrAuto;computedOffset:number;easing:string;offset:number|null;[property:string]:string|number|null|undefined;}interface ConstantSourceOptions{offset?:number;}interface ConstrainBooleanParameters{exact?:boolean;ideal?:boolean;}interface ConstrainDOMStringParameters{exact?:string|string[];ideal?:string|string[];}interface ConstrainDoubleRange extends DoubleRange{exact?:number;ideal?:number;}interface ConstrainULongRange extends ULongRange{exact?:number;ideal?:number;}interface ContentVisibilityAutoStateChangeEventInit extends EventInit{skipped?:boolean;}interface ConvolverOptions extends AudioNodeOptions{buffer?:AudioBuffer|null;disableNormalization?:boolean;}interface CredentialCreationOptions{publicKey?:PublicKeyCredentialCreationOptions;signal?:AbortSignal;}interface CredentialPropertiesOutput{rk?:boolean;}interface CredentialRequestOptions{mediation?:CredentialMediationRequirement;publicKey?:PublicKeyCredentialRequestOptions;signal?:AbortSignal;}interface CryptoKeyPair{privateKey:CryptoKey;publicKey:CryptoKey;}interface CustomEventInit<T=any>extends EventInit{detail?:T;}interface DOMMatrix2DInit{a?:number;b?:number;c?:number;d?:number;e?:number;f?:number;m11?:number;m12?:number;m21?:number;m22?:number;m41?:number;m42?:number;}interface DOMMatrixInit extends DOMMatrix2DInit{is2D?:boolean;m13?:number;m14?:number;m23?:number;m24?:number;m31?:number;m32?:number;m33?:number;m34?:number;m43?:number;m44?:number;}interface DOMPointInit{w?:number;x?:number;y?:number;z?:number;}interface DOMQuadInit{p1?:DOMPointInit;p2?:DOMPointInit;p3?:DOMPointInit;p4?:DOMPointInit;}interface DOMRectInit{height?:number;width?:number;x?:number;y?:number;}interface DelayOptions extends AudioNodeOptions{delayTime?:number;maxDelayTime?:number;}interface DeviceMotionEventAccelerationInit{x?:number|null;y?:number|null;z?:number|null;}interface DeviceMotionEventInit extends EventInit{acceleration?:DeviceMotionEventAccelerationInit;accelerationIncludingGravity?:DeviceMotionEventAccelerationInit;interval?:number;rotationRate?:DeviceMotionEventRotationRateInit;}interface DeviceMotionEventRotationRateInit{alpha?:number|null;beta?:number|null;gamma?:number|null;}interface DeviceOrientationEventInit extends EventInit{absolute?:boolean;alpha?:number|null;beta?:number|null;gamma?:number|null;}interface DisplayMediaStreamOptions{audio?:boolean|MediaTrackConstraints;video?:boolean|MediaTrackConstraints;}interface DocumentTimelineOptions{originTime?:DOMHighResTimeStamp;}interface DoubleRange{max?:number;min?:number;}interface DragEventInit extends MouseEventInit{dataTransfer?:DataTransfer|null;}interface DynamicsCompressorOptions extends AudioNodeOptions{attack?:number;knee?:number;ratio?:number;release?:number;threshold?:number;}interface EcKeyAlgorithm extends KeyAlgorithm{namedCurve:NamedCurve;}interface EcKeyGenParams extends Algorithm{namedCurve:NamedCurve;}interface EcKeyImportParams extends Algorithm{namedCurve:NamedCurve;}interface EcdhKeyDeriveParams extends Algorithm{public:CryptoKey;}interface EcdsaParams extends Algorithm{hash:HashAlgorithmIdentifier;}interface EffectTiming{delay?:number;direction?:PlaybackDirection;duration?:number|CSSNumericValue|string;easing?:string;endDelay?:number;fill?:FillMode;iterationStart?:number;iterations?:number;playbackRate?:number;}interface ElementCreationOptions{is?:string;}interface ElementDefinitionOptions{extends?:string;}interface EncodedVideoChunkInit{data:AllowSharedBufferSource;duration?:number;timestamp:number;type:EncodedVideoChunkType;}interface EncodedVideoChunkMetadata{decoderConfig?:VideoDecoderConfig;}interface ErrorEventInit extends EventInit{colno?:number;error?:any;filename?:string;lineno?:number;message?:string;}interface EventInit{bubbles?:boolean;cancelable?:boolean;composed?:boolean;}interface EventListenerOptions{capture?:boolean;}interface EventModifierInit extends UIEventInit{altKey?:boolean;ctrlKey?:boolean;metaKey?:boolean;modifierAltGraph?:boolean;modifierCapsLock?:boolean;modifierFn?:boolean;modifierFnLock?:boolean;modifierHyper?:boolean;modifierNumLock?:boolean;modifierScrollLock?:boolean;modifierSuper?:boolean;modifierSymbol?:boolean;modifierSymbolLock?:boolean;shiftKey?:boolean;}interface EventSourceInit{withCredentials?:boolean;}interface FilePropertyBag extends BlobPropertyBag{lastModified?:number;}interface FileSystemCreateWritableOptions{keepExistingData?:boolean;}interface FileSystemFlags{create?:boolean;exclusive?:boolean;}interface FileSystemGetDirectoryOptions{create?:boolean;}interface FileSystemGetFileOptions{create?:boolean;}interface FileSystemRemoveOptions{recursive?:boolean;}interface FocusEventInit extends UIEventInit{relatedTarget?:EventTarget|null;}interface FocusOptions{preventScroll?:boolean;}interface FontFaceDescriptors{ascentOverride?:string;descentOverride?:string;display?:FontDisplay;featureSettings?:string;lineGapOverride?:string;stretch?:string;style?:string;unicodeRange?:string;weight?:string;}interface FontFaceSetLoadEventInit extends EventInit{fontfaces?:FontFace[];}interface FormDataEventInit extends EventInit{formData:FormData;}interface FullscreenOptions{navigationUI?:FullscreenNavigationUI;}interface GainOptions extends AudioNodeOptions{gain?:number;}interface GamepadEffectParameters{duration?:number;leftTrigger?:number;rightTrigger?:number;startDelay?:number;strongMagnitude?:number;weakMagnitude?:number;}interface GamepadEventInit extends EventInit{gamepad:Gamepad;}interface GetAnimationsOptions{subtree?:boolean;}interface GetHTMLOptions{serializableShadowRoots?:boolean;shadowRoots?:ShadowRoot[];}interface GetNotificationOptions{tag?:string;}interface GetRootNodeOptions{composed?:boolean;}interface HashChangeEventInit extends EventInit{newURL?:string;oldURL?:string;}interface HkdfParams extends Algorithm{hash:HashAlgorithmIdentifier;info:BufferSource;salt:BufferSource;}interface HmacImportParams extends Algorithm{hash:HashAlgorithmIdentifier;length?:number;}interface HmacKeyAlgorithm extends KeyAlgorithm{hash:KeyAlgorithm;length:number;}interface HmacKeyGenParams extends Algorithm{hash:HashAlgorithmIdentifier;length?:number;}interface IDBDatabaseInfo{name?:string;version?:number;}interface IDBIndexParameters{multiEntry?:boolean;unique?:boolean;}interface IDBObjectStoreParameters{autoIncrement?:boolean;keyPath?:string|string[]|null;}interface IDBTransactionOptions{durability?:IDBTransactionDurability;}interface IDBVersionChangeEventInit extends EventInit{newVersion?:number|null;oldVersion?:number;}interface IIRFilterOptions extends AudioNodeOptions{feedback:number[];feedforward:number[];}interface IdleRequestOptions{timeout?:number;}interface ImageBitmapOptions{colorSpaceConversion?:ColorSpaceConversion;imageOrientation?:ImageOrientation;premultiplyAlpha?:PremultiplyAlpha;resizeHeight?:number;resizeQuality?:ResizeQuality;resizeWidth?:number;}interface ImageBitmapRenderingContextSettings{alpha?:boolean;}interface ImageDataSettings{colorSpace?:PredefinedColorSpace;}interface ImageEncodeOptions{quality?:number;type?:string;}interface InputEventInit extends UIEventInit{data?:string|null;dataTransfer?:DataTransfer|null;inputType?:string;isComposing?:boolean;targetRanges?:StaticRange[];}interface IntersectionObserverEntryInit{boundingClientRect:DOMRectInit;intersectionRatio:number;intersectionRect:DOMRectInit;isIntersecting:boolean;rootBounds:DOMRectInit|null;target:Element;time:DOMHighResTimeStamp;}interface IntersectionObserverInit{root?:Element|Document|null;rootMargin?:string;threshold?:number|number[];}interface JsonWebKey{alg?:string;crv?:string;d?:string;dp?:string;dq?:string;e?:string;ext?:boolean;k?:string;key_ops?:string[];kty?:string;n?:string;oth?:RsaOtherPrimesInfo[];p?:string;q?:string;qi?:string;use?:string;x?:string;y?:string;}interface KeyAlgorithm{name:string;}interface KeyboardEventInit extends EventModifierInit{charCode?:number;code?:string;isComposing?:boolean;key?:string;keyCode?:number;location?:number;repeat?:boolean;}interface Keyframe{composite?:CompositeOperationOrAuto;easing?:string;offset?:number|null;[property:string]:string|number|null|undefined;}interface KeyframeAnimationOptions extends KeyframeEffectOptions{id?:string;timeline?:AnimationTimeline|null;}interface KeyframeEffectOptions extends EffectTiming{composite?:CompositeOperation;iterationComposite?:IterationCompositeOperation;pseudoElement?:string|null;}interface LockInfo{clientId?:string;mode?:LockMode;name?:string;}interface LockManagerSnapshot{held?:LockInfo[];pending?:LockInfo[];}interface LockOptions{ifAvailable?:boolean;mode?:LockMode;signal?:AbortSignal;steal?:boolean;}interface MIDIConnectionEventInit extends EventInit{port?:MIDIPort;}interface MIDIMessageEventInit extends EventInit{data?:Uint8Array;}interface MIDIOptions{software?:boolean;sysex?:boolean;}interface MediaCapabilitiesDecodingInfo extends MediaCapabilitiesInfo{configuration?:MediaDecodingConfiguration;}interface MediaCapabilitiesEncodingInfo extends MediaCapabilitiesInfo{configuration?:MediaEncodingConfiguration;}interface MediaCapabilitiesInfo{powerEfficient:boolean;smooth:boolean;supported:boolean;}interface MediaConfiguration{audio?:AudioConfiguration;video?:VideoConfiguration;}interface MediaDecodingConfiguration extends MediaConfiguration{type:MediaDecodingType;}interface MediaElementAudioSourceOptions{mediaElement:HTMLMediaElement;}interface MediaEncodingConfiguration extends MediaConfiguration{type:MediaEncodingType;}interface MediaEncryptedEventInit extends EventInit{initData?:ArrayBuffer|null;initDataType?:string;}interface MediaImage{sizes?:string;src:string;type?:string;}interface MediaKeyMessageEventInit extends EventInit{message:ArrayBuffer;messageType:MediaKeyMessageType;}interface MediaKeySystemConfiguration{audioCapabilities?:MediaKeySystemMediaCapability[];distinctiveIdentifier?:MediaKeysRequirement;initDataTypes?:string[];label?:string;persistentState?:MediaKeysRequirement;sessionTypes?:string[];videoCapabilities?:MediaKeySystemMediaCapability[];}interface MediaKeySystemMediaCapability{contentType?:string;encryptionScheme?:string|null;robustness?:string;}interface MediaKeysPolicy{minHdcpVersion?:string;}interface MediaMetadataInit{album?:string;artist?:string;artwork?:MediaImage[];title?:string;}interface MediaPositionState{duration?:number;playbackRate?:number;position?:number;}interface MediaQueryListEventInit extends EventInit{matches?:boolean;media?:string;}interface MediaRecorderOptions{audioBitsPerSecond?:number;bitsPerSecond?:number;mimeType?:string;videoBitsPerSecond?:number;}interface MediaSessionActionDetails{action:MediaSessionAction;fastSeek?:boolean;seekOffset?:number;seekTime?:number;}interface MediaStreamAudioSourceOptions{mediaStream:MediaStream;}interface MediaStreamConstraints{audio?:boolean|MediaTrackConstraints;peerIdentity?:string;preferCurrentTab?:boolean;video?:boolean|MediaTrackConstraints;}interface MediaStreamTrackEventInit extends EventInit{track:MediaStreamTrack;}interface MediaTrackCapabilities{aspectRatio?:DoubleRange;autoGainControl?:boolean[];channelCount?:ULongRange;deviceId?:string;displaySurface?:string;echoCancellation?:boolean[];facingMode?:string[];frameRate?:DoubleRange;groupId?:string;height?:ULongRange;noiseSuppression?:boolean[];sampleRate?:ULongRange;sampleSize?:ULongRange;width?:ULongRange;}interface MediaTrackConstraintSet{aspectRatio?:ConstrainDouble;autoGainControl?:ConstrainBoolean;channelCount?:ConstrainULong;deviceId?:ConstrainDOMString;displaySurface?:ConstrainDOMString;echoCancellation?:ConstrainBoolean;facingMode?:ConstrainDOMString;frameRate?:ConstrainDouble;groupId?:ConstrainDOMString;height?:ConstrainULong;noiseSuppression?:ConstrainBoolean;sampleRate?:ConstrainULong;sampleSize?:ConstrainULong;width?:ConstrainULong;}interface MediaTrackConstraints extends MediaTrackConstraintSet{advanced?:MediaTrackConstraintSet[];}interface MediaTrackSettings{aspectRatio?:number;autoGainControl?:boolean;channelCount?:number;deviceId?:string;displaySurface?:string;echoCancellation?:boolean;facingMode?:string;frameRate?:number;groupId?:string;height?:number;noiseSuppression?:boolean;sampleRate?:number;sampleSize?:number;width?:number;}interface MediaTrackSupportedConstraints{aspectRatio?:boolean;autoGainControl?:boolean;channelCount?:boolean;deviceId?:boolean;displaySurface?:boolean;echoCancellation?:boolean;facingMode?:boolean;frameRate?:boolean;groupId?:boolean;height?:boolean;noiseSuppression?:boolean;sampleRate?:boolean;sampleSize?:boolean;width?:boolean;}interface MessageEventInit<T=any>extends EventInit{data?:T;lastEventId?:string;origin?:string;ports?:MessagePort[];source?:MessageEventSource|null;}interface MouseEventInit extends EventModifierInit{button?:number;buttons?:number;clientX?:number;clientY?:number;movementX?:number;movementY?:number;relatedTarget?:EventTarget|null;screenX?:number;screenY?:number;}interface MultiCacheQueryOptions extends CacheQueryOptions{cacheName?:string;}interface MutationObserverInit{attributeFilter?:string[];attributeOldValue?:boolean;attributes?:boolean;characterData?:boolean;characterDataOldValue?:boolean;childList?:boolean;subtree?:boolean;}interface NavigationPreloadState{enabled?:boolean;headerValue?:string;}interface NotificationOptions{badge?:string;body?:string;data?:any;dir?:NotificationDirection;icon?:string;lang?:string;requireInteraction?:boolean;silent?:boolean|null;tag?:string;}interface OfflineAudioCompletionEventInit extends EventInit{renderedBuffer:AudioBuffer;}interface OfflineAudioContextOptions{length:number;numberOfChannels?:number;sampleRate:number;}interface OptionalEffectTiming{delay?:number;direction?:PlaybackDirection;duration?:number|string;easing?:string;endDelay?:number;fill?:FillMode;iterationStart?:number;iterations?:number;playbackRate?:number;}interface OscillatorOptions extends AudioNodeOptions{detune?:number;frequency?:number;periodicWave?:PeriodicWave;type?:OscillatorType;}interface PageTransitionEventInit extends EventInit{persisted?:boolean;}interface PannerOptions extends AudioNodeOptions{coneInnerAngle?:number;coneOuterAngle?:number;coneOuterGain?:number;distanceModel?:DistanceModelType;maxDistance?:number;orientationX?:number;orientationY?:number;orientationZ?:number;panningModel?:PanningModelType;positionX?:number;positionY?:number;positionZ?:number;refDistance?:number;rolloffFactor?:number;}interface PaymentCurrencyAmount{currency:string;value:string;}interface PaymentDetailsBase{displayItems?:PaymentItem[];modifiers?:PaymentDetailsModifier[];}interface PaymentDetailsInit extends PaymentDetailsBase{id?:string;total:PaymentItem;}interface PaymentDetailsModifier{additionalDisplayItems?:PaymentItem[];data?:any;supportedMethods:string;total?:PaymentItem;}interface PaymentDetailsUpdate extends PaymentDetailsBase{paymentMethodErrors?:any;total?:PaymentItem;}interface PaymentItem{amount:PaymentCurrencyAmount;label:string;pending?:boolean;}interface PaymentMethodChangeEventInit extends PaymentRequestUpdateEventInit{methodDetails?:any;methodName?:string;}interface PaymentMethodData{data?:any;supportedMethods:string;}interface PaymentRequestUpdateEventInit extends EventInit{}interface PaymentValidationErrors{error?:string;paymentMethod?:any;}interface Pbkdf2Params extends Algorithm{hash:HashAlgorithmIdentifier;iterations:number;salt:BufferSource;}interface PerformanceMarkOptions{detail?:any;startTime?:DOMHighResTimeStamp;}interface PerformanceMeasureOptions{detail?:any;duration?:DOMHighResTimeStamp;end?:string|DOMHighResTimeStamp;start?:string|DOMHighResTimeStamp;}interface PerformanceObserverInit{buffered?:boolean;entryTypes?:string[];type?:string;}interface PeriodicWaveConstraints{disableNormalization?:boolean;}interface PeriodicWaveOptions extends PeriodicWaveConstraints{imag?:number[]|Float32Array;real?:number[]|Float32Array;}interface PermissionDescriptor{name:PermissionName;}interface PictureInPictureEventInit extends EventInit{pictureInPictureWindow:PictureInPictureWindow;}interface PlaneLayout{offset:number;stride:number;}interface PointerEventInit extends MouseEventInit{coalescedEvents?:PointerEvent[];height?:number;isPrimary?:boolean;pointerId?:number;pointerType?:string;predictedEvents?:PointerEvent[];pressure?:number;tangentialPressure?:number;tiltX?:number;tiltY?:number;twist?:number;width?:number;}interface PointerLockOptions{unadjustedMovement?:boolean;}interface PopStateEventInit extends EventInit{state?:any;}interface PositionOptions{enableHighAccuracy?:boolean;maximumAge?:number;timeout?:number;}interface ProgressEventInit extends EventInit{lengthComputable?:boolean;loaded?:number;total?:number;}interface PromiseRejectionEventInit extends EventInit{promise:Promise<any>;reason?:any;}interface PropertyDefinition{inherits:boolean;initialValue?:string;name:string;syntax?:string;}interface PropertyIndexedKeyframes{composite?:CompositeOperationOrAuto|CompositeOperationOrAuto[];easing?:string|string[];offset?:number|(number|null)[];[property:string]:string|string[]|number|null|(number|null)[]|undefined;}interface PublicKeyCredentialCreationOptions{attestation?:AttestationConveyancePreference;authenticatorSelection?:AuthenticatorSelectionCriteria;challenge:BufferSource;excludeCredentials?:PublicKeyCredentialDescriptor[];extensions?:AuthenticationExtensionsClientInputs;pubKeyCredParams:PublicKeyCredentialParameters[];rp:PublicKeyCredentialRpEntity;timeout?:number;user:PublicKeyCredentialUserEntity;}interface PublicKeyCredentialDescriptor{id:BufferSource;transports?:AuthenticatorTransport[];type:PublicKeyCredentialType;}interface PublicKeyCredentialEntity{name:string;}interface PublicKeyCredentialParameters{alg:COSEAlgorithmIdentifier;type:PublicKeyCredentialType;}interface PublicKeyCredentialRequestOptions{allowCredentials?:PublicKeyCredentialDescriptor[];challenge:BufferSource;extensions?:AuthenticationExtensionsClientInputs;rpId?:string;timeout?:number;userVerification?:UserVerificationRequirement;}interface PublicKeyCredentialRpEntity extends PublicKeyCredentialEntity{id?:string;}interface PublicKeyCredentialUserEntity extends PublicKeyCredentialEntity{displayName:string;id:BufferSource;}interface PushSubscriptionJSON{endpoint?:string;expirationTime?:EpochTimeStamp|null;keys?:Record<string,string>;}interface PushSubscriptionOptionsInit{applicationServerKey?:BufferSource|string|null;userVisibleOnly?:boolean;}interface QueuingStrategy<T=any>{highWaterMark?:number;size?:QueuingStrategySize<T>;}interface QueuingStrategyInit{highWaterMark:number;}interface RTCAnswerOptions extends RTCOfferAnswerOptions{}interface RTCCertificateExpiration{expires?:number;}interface RTCConfiguration{bundlePolicy?:RTCBundlePolicy;certificates?:RTCCertificate[];iceCandidatePoolSize?:number;iceServers?:RTCIceServer[];iceTransportPolicy?:RTCIceTransportPolicy;rtcpMuxPolicy?:RTCRtcpMuxPolicy;}interface RTCDTMFToneChangeEventInit extends EventInit{tone?:string;}interface RTCDataChannelEventInit extends EventInit{channel:RTCDataChannel;}interface RTCDataChannelInit{id?:number;maxPacketLifeTime?:number;maxRetransmits?:number;negotiated?:boolean;ordered?:boolean;protocol?:string;}interface RTCDtlsFingerprint{algorithm?:string;value?:string;}interface RTCEncodedAudioFrameMetadata{contributingSources?:number[];payloadType?:number;sequenceNumber?:number;synchronizationSource?:number;}interface RTCEncodedVideoFrameMetadata{contributingSources?:number[];dependencies?:number[];frameId?:number;height?:number;payloadType?:number;spatialIndex?:number;synchronizationSource?:number;temporalIndex?:number;timestamp?:number;width?:number;}interface RTCErrorEventInit extends EventInit{error:RTCError;}interface RTCErrorInit{errorDetail:RTCErrorDetailType;httpRequestStatusCode?:number;receivedAlert?:number;sctpCauseCode?:number;sdpLineNumber?:number;sentAlert?:number;}interface RTCIceCandidateInit{candidate?:string;sdpMLineIndex?:number|null;sdpMid?:string|null;usernameFragment?:string|null;}interface RTCIceCandidatePairStats extends RTCStats{availableIncomingBitrate?:number;availableOutgoingBitrate?:number;bytesReceived?:number;bytesSent?:number;currentRoundTripTime?:number;lastPacketReceivedTimestamp?:DOMHighResTimeStamp;lastPacketSentTimestamp?:DOMHighResTimeStamp;localCandidateId:string;nominated?:boolean;remoteCandidateId:string;requestsReceived?:number;requestsSent?:number;responsesReceived?:number;responsesSent?:number;state:RTCStatsIceCandidatePairState;totalRoundTripTime?:number;transportId:string;}interface RTCIceServer{credential?:string;urls:string|string[];username?:string;}interface RTCInboundRtpStreamStats extends RTCReceivedRtpStreamStats{audioLevel?:number;bytesReceived?:number;concealedSamples?:number;concealmentEvents?:number;decoderImplementation?:string;estimatedPlayoutTimestamp?:DOMHighResTimeStamp;fecPacketsDiscarded?:number;fecPacketsReceived?:number;firCount?:number;frameHeight?:number;frameWidth?:number;framesDecoded?:number;framesDropped?:number;framesPerSecond?:number;framesReceived?:number;headerBytesReceived?:number;insertedSamplesForDeceleration?:number;jitterBufferDelay?:number;jitterBufferEmittedCount?:number;keyFramesDecoded?:number;lastPacketReceivedTimestamp?:DOMHighResTimeStamp;mid?:string;nackCount?:number;packetsDiscarded?:number;pliCount?:number;qpSum?:number;remoteId?:string;removedSamplesForAcceleration?:number;silentConcealedSamples?:number;totalAudioEnergy?:number;totalDecodeTime?:number;totalInterFrameDelay?:number;totalProcessingDelay?:number;totalSamplesDuration?:number;totalSamplesReceived?:number;totalSquaredInterFrameDelay?:number;trackIdentifier:string;}interface RTCLocalSessionDescriptionInit{sdp?:string;type?:RTCSdpType;}interface RTCOfferAnswerOptions{}interface RTCOfferOptions extends RTCOfferAnswerOptions{iceRestart?:boolean;offerToReceiveAudio?:boolean;offerToReceiveVideo?:boolean;}interface RTCOutboundRtpStreamStats extends RTCSentRtpStreamStats{firCount?:number;frameHeight?:number;frameWidth?:number;framesEncoded?:number;framesPerSecond?:number;framesSent?:number;headerBytesSent?:number;hugeFramesSent?:number;keyFramesEncoded?:number;mediaSourceId?:string;nackCount?:number;pliCount?:number;qpSum?:number;qualityLimitationResolutionChanges?:number;remoteId?:string;retransmittedBytesSent?:number;retransmittedPacketsSent?:number;rid?:string;rtxSsrc?:number;targetBitrate?:number;totalEncodeTime?:number;totalEncodedBytesTarget?:number;totalPacketSendDelay?:number;}interface RTCPeerConnectionIceErrorEventInit extends EventInit{address?:string|null;errorCode:number;errorText?:string;port?:number|null;url?:string;}interface RTCPeerConnectionIceEventInit extends EventInit{candidate?:RTCIceCandidate|null;url?:string|null;}interface RTCReceivedRtpStreamStats extends RTCRtpStreamStats{jitter?:number;packetsLost?:number;packetsReceived?:number;}interface RTCRtcpParameters{cname?:string;reducedSize?:boolean;}interface RTCRtpCapabilities{codecs:RTCRtpCodec[];headerExtensions:RTCRtpHeaderExtensionCapability[];}interface RTCRtpCodec{channels?:number;clockRate:number;mimeType:string;sdpFmtpLine?:string;}interface RTCRtpCodecParameters extends RTCRtpCodec{payloadType:number;}interface RTCRtpCodingParameters{rid?:string;}interface RTCRtpContributingSource{audioLevel?:number;rtpTimestamp:number;source:number;timestamp:DOMHighResTimeStamp;}interface RTCRtpEncodingParameters extends RTCRtpCodingParameters{active?:boolean;maxBitrate?:number;maxFramerate?:number;networkPriority?:RTCPriorityType;priority?:RTCPriorityType;scaleResolutionDownBy?:number;}interface RTCRtpHeaderExtensionCapability{uri:string;}interface RTCRtpHeaderExtensionParameters{encrypted?:boolean;id:number;uri:string;}interface RTCRtpParameters{codecs:RTCRtpCodecParameters[];headerExtensions:RTCRtpHeaderExtensionParameters[];rtcp:RTCRtcpParameters;}interface RTCRtpReceiveParameters extends RTCRtpParameters{}interface RTCRtpSendParameters extends RTCRtpParameters{degradationPreference?:RTCDegradationPreference;encodings:RTCRtpEncodingParameters[];transactionId:string;}interface RTCRtpStreamStats extends RTCStats{codecId?:string;kind:string;ssrc:number;transportId?:string;}interface RTCRtpSynchronizationSource extends RTCRtpContributingSource{}interface RTCRtpTransceiverInit{direction?:RTCRtpTransceiverDirection;sendEncodings?:RTCRtpEncodingParameters[];streams?:MediaStream[];}interface RTCSentRtpStreamStats extends RTCRtpStreamStats{bytesSent?:number;packetsSent?:number;}interface RTCSessionDescriptionInit{sdp?:string;type:RTCSdpType;}interface RTCSetParameterOptions{}interface RTCStats{id:string;timestamp:DOMHighResTimeStamp;type:RTCStatsType;}interface RTCTrackEventInit extends EventInit{receiver:RTCRtpReceiver;streams?:MediaStream[];track:MediaStreamTrack;transceiver:RTCRtpTransceiver;}interface RTCTransportStats extends RTCStats{bytesReceived?:number;bytesSent?:number;dtlsCipher?:string;dtlsState:RTCDtlsTransportState;localCertificateId?:string;remoteCertificateId?:string;selectedCandidatePairId?:string;srtpCipher?:string;tlsVersion?:string;}interface ReadableStreamGetReaderOptions{mode?:ReadableStreamReaderMode;}interface ReadableStreamIteratorOptions{preventCancel?:boolean;}interface ReadableStreamReadDoneResult<T>{done:true;value?:T;}interface ReadableStreamReadValueResult<T>{done:false;value:T;}interface ReadableWritablePair<R=any,W=any>{readable:ReadableStream<R>;writable:WritableStream<W>;}interface RegistrationOptions{scope?:string;type?:WorkerType;updateViaCache?:ServiceWorkerUpdateViaCache;}interface ReportingObserverOptions{buffered?:boolean;types?:string[];}interface RequestInit{body?:BodyInit|null;cache?:RequestCache;credentials?:RequestCredentials;headers?:HeadersInit;integrity?:string;keepalive?:boolean;method?:string;mode?:RequestMode;priority?:RequestPriority;redirect?:RequestRedirect;referrer?:string;referrerPolicy?:ReferrerPolicy;signal?:AbortSignal|null;window?:null;}interface ResizeObserverOptions{box?:ResizeObserverBoxOptions;}interface ResponseInit{headers?:HeadersInit;status?:number;statusText?:string;}interface RsaHashedImportParams extends Algorithm{hash:HashAlgorithmIdentifier;}interface RsaHashedKeyAlgorithm extends RsaKeyAlgorithm{hash:KeyAlgorithm;}interface RsaHashedKeyGenParams extends RsaKeyGenParams{hash:HashAlgorithmIdentifier;}interface RsaKeyAlgorithm extends KeyAlgorithm{modulusLength:number;publicExponent:BigInteger;}interface RsaKeyGenParams extends Algorithm{modulusLength:number;publicExponent:BigInteger;}interface RsaOaepParams extends Algorithm{label?:BufferSource;}interface RsaOtherPrimesInfo{d?:string;r?:string;t?:string;}interface RsaPssParams extends Algorithm{saltLength:number;}interface SVGBoundingBoxOptions{clipped?:boolean;fill?:boolean;markers?:boolean;stroke?:boolean;}interface ScrollIntoViewOptions extends ScrollOptions{block?:ScrollLogicalPosition;inline?:ScrollLogicalPosition;}interface ScrollOptions{behavior?:ScrollBehavior;}interface ScrollToOptions extends ScrollOptions{left?:number;top?:number;}interface SecurityPolicyViolationEventInit extends EventInit{blockedURI?:string;columnNumber?:number;disposition?:SecurityPolicyViolationEventDisposition;documentURI?:string;effectiveDirective?:string;lineNumber?:number;originalPolicy?:string;referrer?:string;sample?:string;sourceFile?:string;statusCode?:number;violatedDirective?:string;}interface ShadowRootInit{delegatesFocus?:boolean;mode:ShadowRootMode;serializable?:boolean;slotAssignment?:SlotAssignmentMode;}interface ShareData{files?:File[];text?:string;title?:string;url?:string;}interface SpeechSynthesisErrorEventInit extends SpeechSynthesisEventInit{error:SpeechSynthesisErrorCode;}interface SpeechSynthesisEventInit extends EventInit{charIndex?:number;charLength?:number;elapsedTime?:number;name?:string;utterance:SpeechSynthesisUtterance;}interface StaticRangeInit{endContainer:Node;endOffset:number;startContainer:Node;startOffset:number;}interface StereoPannerOptions extends AudioNodeOptions{pan?:number;}interface StorageEstimate{quota?:number;usage?:number;}interface StorageEventInit extends EventInit{key?:string|null;newValue?:string|null;oldValue?:string|null;storageArea?:Storage|null;url?:string;}interface StreamPipeOptions{preventAbort?:boolean;preventCancel?:boolean;preventClose?:boolean;signal?:AbortSignal;}interface StructuredSerializeOptions{transfer?:Transferable[];}interface SubmitEventInit extends EventInit{submitter?:HTMLElement|null;}interface TextDecodeOptions{stream?:boolean;}interface TextDecoderOptions{fatal?:boolean;ignoreBOM?:boolean;}interface TextEncoderEncodeIntoResult{read:number;written:number;}interface ToggleEventInit extends EventInit{newState?:string;oldState?:string;}interface TouchEventInit extends EventModifierInit{changedTouches?:Touch[];targetTouches?:Touch[];touches?:Touch[];}interface TouchInit{altitudeAngle?:number;azimuthAngle?:number;clientX?:number;clientY?:number;force?:number;identifier:number;pageX?:number;pageY?:number;radiusX?:number;radiusY?:number;rotationAngle?:number;screenX?:number;screenY?:number;target:EventTarget;touchType?:TouchType;}interface TrackEventInit extends EventInit{track?:TextTrack|null;}interface Transformer<I=any,O=any>{flush?:TransformerFlushCallback<O>;readableType?:undefined;start?:TransformerStartCallback<O>;transform?:TransformerTransformCallback<I,O>;writableType?:undefined;}interface TransitionEventInit extends EventInit{elapsedTime?:number;propertyName?:string;pseudoElement?:string;}interface UIEventInit extends EventInit{detail?:number;view?:Window|null;which?:number;}interface ULongRange{max?:number;min?:number;}interface UnderlyingByteSource{autoAllocateChunkSize?:number;cancel?:UnderlyingSourceCancelCallback;pull?:(controller:ReadableByteStreamController)=>void|PromiseLike<void>;start?:(controller:ReadableByteStreamController)=>any;type:\"bytes\";}interface UnderlyingDefaultSource<R=any>{cancel?:UnderlyingSourceCancelCallback;pull?:(controller:ReadableStreamDefaultController<R>)=>void|PromiseLike<void>;start?:(controller:ReadableStreamDefaultController<R>)=>any;type?:undefined;}interface UnderlyingSink<W=any>{abort?:UnderlyingSinkAbortCallback;close?:UnderlyingSinkCloseCallback;start?:UnderlyingSinkStartCallback;type?:undefined;write?:UnderlyingSinkWriteCallback<W>;}interface UnderlyingSource<R=any>{autoAllocateChunkSize?:number;cancel?:UnderlyingSourceCancelCallback;pull?:UnderlyingSourcePullCallback<R>;start?:UnderlyingSourceStartCallback<R>;type?:ReadableStreamType;}interface ValidityStateFlags{badInput?:boolean;customError?:boolean;patternMismatch?:boolean;rangeOverflow?:boolean;rangeUnderflow?:boolean;stepMismatch?:boolean;tooLong?:boolean;tooShort?:boolean;typeMismatch?:boolean;valueMissing?:boolean;}interface VideoColorSpaceInit{fullRange?:boolean|null;matrix?:VideoMatrixCoefficients|null;primaries?:VideoColorPrimaries|null;transfer?:VideoTransferCharacteristics|null;}interface VideoConfiguration{bitrate:number;colorGamut?:ColorGamut;contentType:string;framerate:number;hdrMetadataType?:HdrMetadataType;height:number;scalabilityMode?:string;transferFunction?:TransferFunction;width:number;}interface VideoDecoderConfig{codec:string;codedHeight?:number;codedWidth?:number;colorSpace?:VideoColorSpaceInit;description?:AllowSharedBufferSource;displayAspectHeight?:number;displayAspectWidth?:number;hardwareAcceleration?:HardwareAcceleration;optimizeForLatency?:boolean;}interface VideoDecoderInit{error:WebCodecsErrorCallback;output:VideoFrameOutputCallback;}interface VideoDecoderSupport{config?:VideoDecoderConfig;supported?:boolean;}interface VideoEncoderConfig{alpha?:AlphaOption;avc?:AvcEncoderConfig;bitrate?:number;bitrateMode?:VideoEncoderBitrateMode;codec:string;displayHeight?:number;displayWidth?:number;framerate?:number;hardwareAcceleration?:HardwareAcceleration;height:number;latencyMode?:LatencyMode;scalabilityMode?:string;width:number;}interface VideoEncoderEncodeOptions{keyFrame?:boolean;}interface VideoEncoderInit{error:WebCodecsErrorCallback;output:EncodedVideoChunkOutputCallback;}interface VideoEncoderSupport{config?:VideoEncoderConfig;supported?:boolean;}interface VideoFrameBufferInit{codedHeight:number;codedWidth:number;colorSpace?:VideoColorSpaceInit;displayHeight?:number;displayWidth?:number;duration?:number;format:VideoPixelFormat;layout?:PlaneLayout[];timestamp:number;visibleRect?:DOMRectInit;}interface VideoFrameCallbackMetadata{captureTime?:DOMHighResTimeStamp;expectedDisplayTime:DOMHighResTimeStamp;height:number;mediaTime:number;presentationTime:DOMHighResTimeStamp;presentedFrames:number;processingDuration?:number;receiveTime?:DOMHighResTimeStamp;rtpTimestamp?:number;width:number;}interface VideoFrameCopyToOptions{layout?:PlaneLayout[];rect?:DOMRectInit;}interface VideoFrameInit{alpha?:AlphaOption;displayHeight?:number;displayWidth?:number;duration?:number;timestamp?:number;visibleRect?:DOMRectInit;}interface WaveShaperOptions extends AudioNodeOptions{curve?:number[]|Float32Array;oversample?:OverSampleType;}interface WebGLContextAttributes{alpha?:boolean;antialias?:boolean;depth?:boolean;desynchronized?:boolean;failIfMajorPerformanceCaveat?:boolean;powerPreference?:WebGLPowerPreference;premultipliedAlpha?:boolean;preserveDrawingBuffer?:boolean;stencil?:boolean;}interface WebGLContextEventInit extends EventInit{statusMessage?:string;}interface WebTransportCloseInfo{closeCode?:number;reason?:string;}interface WebTransportErrorOptions{source?:WebTransportErrorSource;streamErrorCode?:number|null;}interface WebTransportHash{algorithm?:string;value?:BufferSource;}interface WebTransportOptions{allowPooling?:boolean;congestionControl?:WebTransportCongestionControl;requireUnreliable?:boolean;serverCertificateHashes?:WebTransportHash[];}interface WebTransportSendStreamOptions{sendOrder?:number;}interface WheelEventInit extends MouseEventInit{deltaMode?:number;deltaX?:number;deltaY?:number;deltaZ?:number;}interface WindowPostMessageOptions extends StructuredSerializeOptions{targetOrigin?:string;}interface WorkerOptions{credentials?:RequestCredentials;name?:string;type?:WorkerType;}interface WorkletOptions{credentials?:RequestCredentials;}interface WriteParams{data?:BufferSource|Blob|string|null;position?:number|null;size?:number|null;type:WriteCommandType;}type NodeFilter=((node:Node)=>number)|{acceptNode(node:Node):number;};declare var NodeFilter:{readonly FILTER_ACCEPT:1;readonly FILTER_REJECT:2;readonly FILTER_SKIP:3;readonly SHOW_ALL:0xFFFFFFFF;readonly SHOW_ELEMENT:0x1;readonly SHOW_ATTRIBUTE:0x2;readonly SHOW_TEXT:0x4;readonly SHOW_CDATA_SECTION:0x8;readonly SHOW_ENTITY_REFERENCE:0x10;readonly SHOW_ENTITY:0x20;readonly SHOW_PROCESSING_INSTRUCTION:0x40;readonly SHOW_COMMENT:0x80;readonly SHOW_DOCUMENT:0x100;readonly SHOW_DOCUMENT_TYPE:0x200;readonly SHOW_DOCUMENT_FRAGMENT:0x400;readonly SHOW_NOTATION:0x800;};type XPathNSResolver=((prefix:string|null)=>string|null)|{lookupNamespaceURI(prefix:string|null):string|null;};interface ANGLE_instanced_arrays{drawArraysInstancedANGLE(mode:GLenum,first:GLint,count:GLsizei,primcount:GLsizei):void;drawElementsInstancedANGLE(mode:GLenum,count:GLsizei,type:GLenum,offset:GLintptr,primcount:GLsizei):void;vertexAttribDivisorANGLE(index:GLuint,divisor:GLuint):void;readonly VERTEX_ATTRIB_ARRAY_DIVISOR_ANGLE:0x88FE;}interface ARIAMixin{ariaAtomic:string|null;ariaAutoComplete:string|null;ariaBrailleLabel:string|null;ariaBrailleRoleDescription:string|null;ariaBusy:string|null;ariaChecked:string|null;ariaColCount:string|null;ariaColIndex:string|null;ariaColSpan:string|null;ariaCurrent:string|null;ariaDescription:string|null;ariaDisabled:string|null;ariaExpanded:string|null;ariaHasPopup:string|null;ariaHidden:string|null;ariaInvalid:string|null;ariaKeyShortcuts:string|null;ariaLabel:string|null;ariaLevel:string|null;ariaLive:string|null;ariaModal:string|null;ariaMultiLine:string|null;ariaMultiSelectable:string|null;ariaOrientation:string|null;ariaPlaceholder:string|null;ariaPosInSet:string|null;ariaPressed:string|null;ariaReadOnly:string|null;ariaRequired:string|null;ariaRoleDescription:string|null;ariaRowCount:string|null;ariaRowIndex:string|null;ariaRowSpan:string|null;ariaSelected:string|null;ariaSetSize:string|null;ariaSort:string|null;ariaValueMax:string|null;ariaValueMin:string|null;ariaValueNow:string|null;ariaValueText:string|null;role:string|null;}interface AbortController{readonly signal:AbortSignal;abort(reason?:any):void;}declare var AbortController:{prototype:AbortController;new():AbortController;};interface AbortSignalEventMap{\"abort\":Event;}interface AbortSignal extends EventTarget{readonly aborted:boolean;onabort:((this:AbortSignal,ev:Event)=>any)|null;readonly reason:any;throwIfAborted():void;addEventListener<K extends keyof AbortSignalEventMap>(type:K,listener:(this:AbortSignal,ev:AbortSignalEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AbortSignalEventMap>(type:K,listener:(this:AbortSignal,ev:AbortSignalEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var AbortSignal:{prototype:AbortSignal;new():AbortSignal;abort(reason?:any):AbortSignal;any(signals:AbortSignal[]):AbortSignal;timeout(milliseconds:number):AbortSignal;};interface AbstractRange{readonly collapsed:boolean;readonly endContainer:Node;readonly endOffset:number;readonly startContainer:Node;readonly startOffset:number;}declare var AbstractRange:{prototype:AbstractRange;new():AbstractRange;};interface AbstractWorkerEventMap{\"error\":ErrorEvent;}interface AbstractWorker{onerror:((this:AbstractWorker,ev:ErrorEvent)=>any)|null;addEventListener<K extends keyof AbstractWorkerEventMap>(type:K,listener:(this:AbstractWorker,ev:AbstractWorkerEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AbstractWorkerEventMap>(type:K,listener:(this:AbstractWorker,ev:AbstractWorkerEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}interface AnalyserNode extends AudioNode{fftSize:number;readonly frequencyBinCount:number;maxDecibels:number;minDecibels:number;smoothingTimeConstant:number;getByteFrequencyData(array:Uint8Array):void;getByteTimeDomainData(array:Uint8Array):void;getFloatFrequencyData(array:Float32Array):void;getFloatTimeDomainData(array:Float32Array):void;}declare var AnalyserNode:{prototype:AnalyserNode;new(context:BaseAudioContext,options?:AnalyserOptions):AnalyserNode;};interface Animatable{animate(keyframes:Keyframe[]|PropertyIndexedKeyframes|null,options?:number|KeyframeAnimationOptions):Animation;getAnimations(options?:GetAnimationsOptions):Animation[];}interface AnimationEventMap{\"cancel\":AnimationPlaybackEvent;\"finish\":AnimationPlaybackEvent;\"remove\":Event;}interface Animation extends EventTarget{currentTime:CSSNumberish|null;effect:AnimationEffect|null;readonly finished:Promise<Animation>;id:string;oncancel:((this:Animation,ev:AnimationPlaybackEvent)=>any)|null;onfinish:((this:Animation,ev:AnimationPlaybackEvent)=>any)|null;onremove:((this:Animation,ev:Event)=>any)|null;readonly pending:boolean;readonly playState:AnimationPlayState;playbackRate:number;readonly ready:Promise<Animation>;readonly replaceState:AnimationReplaceState;startTime:CSSNumberish|null;timeline:AnimationTimeline|null;cancel():void;commitStyles():void;finish():void;pause():void;persist():void;play():void;reverse():void;updatePlaybackRate(playbackRate:number):void;addEventListener<K extends keyof AnimationEventMap>(type:K,listener:(this:Animation,ev:AnimationEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AnimationEventMap>(type:K,listener:(this:Animation,ev:AnimationEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var Animation:{prototype:Animation;new(effect?:AnimationEffect|null,timeline?:AnimationTimeline|null):Animation;};interface AnimationEffect{getComputedTiming():ComputedEffectTiming;getTiming():EffectTiming;updateTiming(timing?:OptionalEffectTiming):void;}declare var AnimationEffect:{prototype:AnimationEffect;new():AnimationEffect;};interface AnimationEvent extends Event{readonly animationName:string;readonly elapsedTime:number;readonly pseudoElement:string;}declare var AnimationEvent:{prototype:AnimationEvent;new(type:string,animationEventInitDict?:AnimationEventInit):AnimationEvent;};interface AnimationFrameProvider{cancelAnimationFrame(handle:number):void;requestAnimationFrame(callback:FrameRequestCallback):number;}interface AnimationPlaybackEvent extends Event{readonly currentTime:CSSNumberish|null;readonly timelineTime:CSSNumberish|null;}declare var AnimationPlaybackEvent:{prototype:AnimationPlaybackEvent;new(type:string,eventInitDict?:AnimationPlaybackEventInit):AnimationPlaybackEvent;};interface AnimationTimeline{readonly currentTime:CSSNumberish|null;}declare var AnimationTimeline:{prototype:AnimationTimeline;new():AnimationTimeline;};interface Attr extends Node{readonly localName:string;readonly name:string;readonly namespaceURI:string|null;readonly ownerDocument:Document;readonly ownerElement:Element|null;readonly prefix:string|null;readonly specified:boolean;value:string;}declare var Attr:{prototype:Attr;new():Attr;};interface AudioBuffer{readonly duration:number;readonly length:number;readonly numberOfChannels:number;readonly sampleRate:number;copyFromChannel(destination:Float32Array,channelNumber:number,bufferOffset?:number):void;copyToChannel(source:Float32Array,channelNumber:number,bufferOffset?:number):void;getChannelData(channel:number):Float32Array;}declare var AudioBuffer:{prototype:AudioBuffer;new(options:AudioBufferOptions):AudioBuffer;};interface AudioBufferSourceNode extends AudioScheduledSourceNode{buffer:AudioBuffer|null;readonly detune:AudioParam;loop:boolean;loopEnd:number;loopStart:number;readonly playbackRate:AudioParam;start(when?:number,offset?:number,duration?:number):void;addEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type:K,listener:(this:AudioBufferSourceNode,ev:AudioScheduledSourceNodeEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type:K,listener:(this:AudioBufferSourceNode,ev:AudioScheduledSourceNodeEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var AudioBufferSourceNode:{prototype:AudioBufferSourceNode;new(context:BaseAudioContext,options?:AudioBufferSourceOptions):AudioBufferSourceNode;};interface AudioContext extends BaseAudioContext{readonly baseLatency:number;readonly outputLatency:number;close():Promise<void>;createMediaElementSource(mediaElement:HTMLMediaElement):MediaElementAudioSourceNode;createMediaStreamDestination():MediaStreamAudioDestinationNode;createMediaStreamSource(mediaStream:MediaStream):MediaStreamAudioSourceNode;getOutputTimestamp():AudioTimestamp;resume():Promise<void>;suspend():Promise<void>;addEventListener<K extends keyof BaseAudioContextEventMap>(type:K,listener:(this:AudioContext,ev:BaseAudioContextEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof BaseAudioContextEventMap>(type:K,listener:(this:AudioContext,ev:BaseAudioContextEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var AudioContext:{prototype:AudioContext;new(contextOptions?:AudioContextOptions):AudioContext;};interface AudioDestinationNode extends AudioNode{readonly maxChannelCount:number;}declare var AudioDestinationNode:{prototype:AudioDestinationNode;new():AudioDestinationNode;};interface AudioListener{readonly forwardX:AudioParam;readonly forwardY:AudioParam;readonly forwardZ:AudioParam;readonly positionX:AudioParam;readonly positionY:AudioParam;readonly positionZ:AudioParam;readonly upX:AudioParam;readonly upY:AudioParam;readonly upZ:AudioParam;setOrientation(x:number,y:number,z:number,xUp:number,yUp:number,zUp:number):void;setPosition(x:number,y:number,z:number):void;}declare var AudioListener:{prototype:AudioListener;new():AudioListener;};interface AudioNode extends EventTarget{channelCount:number;channelCountMode:ChannelCountMode;channelInterpretation:ChannelInterpretation;readonly context:BaseAudioContext;readonly numberOfInputs:number;readonly numberOfOutputs:number;connect(destinationNode:AudioNode,output?:number,input?:number):AudioNode;connect(destinationParam:AudioParam,output?:number):void;disconnect():void;disconnect(output:number):void;disconnect(destinationNode:AudioNode):void;disconnect(destinationNode:AudioNode,output:number):void;disconnect(destinationNode:AudioNode,output:number,input:number):void;disconnect(destinationParam:AudioParam):void;disconnect(destinationParam:AudioParam,output:number):void;}declare var AudioNode:{prototype:AudioNode;new():AudioNode;};interface AudioParam{automationRate:AutomationRate;readonly defaultValue:number;readonly maxValue:number;readonly minValue:number;value:number;cancelAndHoldAtTime(cancelTime:number):AudioParam;cancelScheduledValues(cancelTime:number):AudioParam;exponentialRampToValueAtTime(value:number,endTime:number):AudioParam;linearRampToValueAtTime(value:number,endTime:number):AudioParam;setTargetAtTime(target:number,startTime:number,timeConstant:number):AudioParam;setValueAtTime(value:number,startTime:number):AudioParam;setValueCurveAtTime(values:number[]|Float32Array,startTime:number,duration:number):AudioParam;}declare var AudioParam:{prototype:AudioParam;new():AudioParam;};interface AudioParamMap{forEach(callbackfn:(value:AudioParam,key:string,parent:AudioParamMap)=>void,thisArg?:any):void;}declare var AudioParamMap:{prototype:AudioParamMap;new():AudioParamMap;};interface AudioProcessingEvent extends Event{readonly inputBuffer:AudioBuffer;readonly outputBuffer:AudioBuffer;readonly playbackTime:number;}declare var AudioProcessingEvent:{prototype:AudioProcessingEvent;new(type:string,eventInitDict:AudioProcessingEventInit):AudioProcessingEvent;};interface AudioScheduledSourceNodeEventMap{\"ended\":Event;}interface AudioScheduledSourceNode extends AudioNode{onended:((this:AudioScheduledSourceNode,ev:Event)=>any)|null;start(when?:number):void;stop(when?:number):void;addEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type:K,listener:(this:AudioScheduledSourceNode,ev:AudioScheduledSourceNodeEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type:K,listener:(this:AudioScheduledSourceNode,ev:AudioScheduledSourceNodeEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var AudioScheduledSourceNode:{prototype:AudioScheduledSourceNode;new():AudioScheduledSourceNode;};interface AudioWorklet extends Worklet{}declare var AudioWorklet:{prototype:AudioWorklet;new():AudioWorklet;};interface AudioWorkletNodeEventMap{\"processorerror\":Event;}interface AudioWorkletNode extends AudioNode{onprocessorerror:((this:AudioWorkletNode,ev:Event)=>any)|null;readonly parameters:AudioParamMap;readonly port:MessagePort;addEventListener<K extends keyof AudioWorkletNodeEventMap>(type:K,listener:(this:AudioWorkletNode,ev:AudioWorkletNodeEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AudioWorkletNodeEventMap>(type:K,listener:(this:AudioWorkletNode,ev:AudioWorkletNodeEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var AudioWorkletNode:{prototype:AudioWorkletNode;new(context:BaseAudioContext,name:string,options?:AudioWorkletNodeOptions):AudioWorkletNode;};interface AuthenticatorAssertionResponse extends AuthenticatorResponse{readonly authenticatorData:ArrayBuffer;readonly signature:ArrayBuffer;readonly userHandle:ArrayBuffer|null;}declare var AuthenticatorAssertionResponse:{prototype:AuthenticatorAssertionResponse;new():AuthenticatorAssertionResponse;};interface AuthenticatorAttestationResponse extends AuthenticatorResponse{readonly attestationObject:ArrayBuffer;getAuthenticatorData():ArrayBuffer;getPublicKey():ArrayBuffer|null;getPublicKeyAlgorithm():COSEAlgorithmIdentifier;getTransports():string[];}declare var AuthenticatorAttestationResponse:{prototype:AuthenticatorAttestationResponse;new():AuthenticatorAttestationResponse;};interface AuthenticatorResponse{readonly clientDataJSON:ArrayBuffer;}declare var AuthenticatorResponse:{prototype:AuthenticatorResponse;new():AuthenticatorResponse;};interface BarProp{readonly visible:boolean;}declare var BarProp:{prototype:BarProp;new():BarProp;};interface BaseAudioContextEventMap{\"statechange\":Event;}interface BaseAudioContext extends EventTarget{readonly audioWorklet:AudioWorklet;readonly currentTime:number;readonly destination:AudioDestinationNode;readonly listener:AudioListener;onstatechange:((this:BaseAudioContext,ev:Event)=>any)|null;readonly sampleRate:number;readonly state:AudioContextState;createAnalyser():AnalyserNode;createBiquadFilter():BiquadFilterNode;createBuffer(numberOfChannels:number,length:number,sampleRate:number):AudioBuffer;createBufferSource():AudioBufferSourceNode;createChannelMerger(numberOfInputs?:number):ChannelMergerNode;createChannelSplitter(numberOfOutputs?:number):ChannelSplitterNode;createConstantSource():ConstantSourceNode;createConvolver():ConvolverNode;createDelay(maxDelayTime?:number):DelayNode;createDynamicsCompressor():DynamicsCompressorNode;createGain():GainNode;createIIRFilter(feedforward:number[],feedback:number[]):IIRFilterNode;createOscillator():OscillatorNode;createPanner():PannerNode;createPeriodicWave(real:number[]|Float32Array,imag:number[]|Float32Array,constraints?:PeriodicWaveConstraints):PeriodicWave;createScriptProcessor(bufferSize?:number,numberOfInputChannels?:number,numberOfOutputChannels?:number):ScriptProcessorNode;createStereoPanner():StereoPannerNode;createWaveShaper():WaveShaperNode;decodeAudioData(audioData:ArrayBuffer,successCallback?:DecodeSuccessCallback|null,errorCallback?:DecodeErrorCallback|null):Promise<AudioBuffer>;addEventListener<K extends keyof BaseAudioContextEventMap>(type:K,listener:(this:BaseAudioContext,ev:BaseAudioContextEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof BaseAudioContextEventMap>(type:K,listener:(this:BaseAudioContext,ev:BaseAudioContextEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var BaseAudioContext:{prototype:BaseAudioContext;new():BaseAudioContext;};interface BeforeUnloadEvent extends Event{returnValue:any;}declare var BeforeUnloadEvent:{prototype:BeforeUnloadEvent;new():BeforeUnloadEvent;};interface BiquadFilterNode extends AudioNode{readonly Q:AudioParam;readonly detune:AudioParam;readonly frequency:AudioParam;readonly gain:AudioParam;type:BiquadFilterType;getFrequencyResponse(frequencyHz:Float32Array,magResponse:Float32Array,phaseResponse:Float32Array):void;}declare var BiquadFilterNode:{prototype:BiquadFilterNode;new(context:BaseAudioContext,options?:BiquadFilterOptions):BiquadFilterNode;};interface Blob{readonly size:number;readonly type:string;arrayBuffer():Promise<ArrayBuffer>;slice(start?:number,end?:number,contentType?:string):Blob;stream():ReadableStream<Uint8Array>;text():Promise<string>;}declare var Blob:{prototype:Blob;new(blobParts?:BlobPart[],options?:BlobPropertyBag):Blob;};interface BlobEvent extends Event{readonly data:Blob;readonly timecode:DOMHighResTimeStamp;}declare var BlobEvent:{prototype:BlobEvent;new(type:string,eventInitDict:BlobEventInit):BlobEvent;};interface Body{readonly body:ReadableStream<Uint8Array>|null;readonly bodyUsed:boolean;arrayBuffer():Promise<ArrayBuffer>;blob():Promise<Blob>;formData():Promise<FormData>;json():Promise<any>;text():Promise<string>;}interface BroadcastChannelEventMap{\"message\":MessageEvent;\"messageerror\":MessageEvent;}interface BroadcastChannel extends EventTarget{readonly name:string;onmessage:((this:BroadcastChannel,ev:MessageEvent)=>any)|null;onmessageerror:((this:BroadcastChannel,ev:MessageEvent)=>any)|null;close():void;postMessage(message:any):void;addEventListener<K extends keyof BroadcastChannelEventMap>(type:K,listener:(this:BroadcastChannel,ev:BroadcastChannelEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof BroadcastChannelEventMap>(type:K,listener:(this:BroadcastChannel,ev:BroadcastChannelEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var BroadcastChannel:{prototype:BroadcastChannel;new(name:string):BroadcastChannel;};interface ByteLengthQueuingStrategy extends QueuingStrategy<ArrayBufferView>{readonly highWaterMark:number;readonly size:QueuingStrategySize<ArrayBufferView>;}declare var ByteLengthQueuingStrategy:{prototype:ByteLengthQueuingStrategy;new(init:QueuingStrategyInit):ByteLengthQueuingStrategy;};interface CDATASection extends Text{}declare var CDATASection:{prototype:CDATASection;new():CDATASection;};interface CSSAnimation extends Animation{readonly animationName:string;addEventListener<K extends keyof AnimationEventMap>(type:K,listener:(this:CSSAnimation,ev:AnimationEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AnimationEventMap>(type:K,listener:(this:CSSAnimation,ev:AnimationEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var CSSAnimation:{prototype:CSSAnimation;new():CSSAnimation;};interface CSSConditionRule extends CSSGroupingRule{readonly conditionText:string;}declare var CSSConditionRule:{prototype:CSSConditionRule;new():CSSConditionRule;};interface CSSContainerRule extends CSSConditionRule{readonly containerName:string;readonly containerQuery:string;}declare var CSSContainerRule:{prototype:CSSContainerRule;new():CSSContainerRule;};interface CSSCounterStyleRule extends CSSRule{additiveSymbols:string;fallback:string;name:string;negative:string;pad:string;prefix:string;range:string;speakAs:string;suffix:string;symbols:string;system:string;}declare var CSSCounterStyleRule:{prototype:CSSCounterStyleRule;new():CSSCounterStyleRule;};interface CSSFontFaceRule extends CSSRule{readonly style:CSSStyleDeclaration;}declare var CSSFontFaceRule:{prototype:CSSFontFaceRule;new():CSSFontFaceRule;};interface CSSFontFeatureValuesRule extends CSSRule{fontFamily:string;}declare var CSSFontFeatureValuesRule:{prototype:CSSFontFeatureValuesRule;new():CSSFontFeatureValuesRule;};interface CSSFontPaletteValuesRule extends CSSRule{readonly basePalette:string;readonly fontFamily:string;readonly name:string;readonly overrideColors:string;}declare var CSSFontPaletteValuesRule:{prototype:CSSFontPaletteValuesRule;new():CSSFontPaletteValuesRule;};interface CSSGroupingRule extends CSSRule{readonly cssRules:CSSRuleList;deleteRule(index:number):void;insertRule(rule:string,index?:number):number;}declare var CSSGroupingRule:{prototype:CSSGroupingRule;new():CSSGroupingRule;};interface CSSImageValue extends CSSStyleValue{}declare var CSSImageValue:{prototype:CSSImageValue;new():CSSImageValue;};interface CSSImportRule extends CSSRule{readonly href:string;readonly layerName:string|null;readonly media:MediaList;readonly styleSheet:CSSStyleSheet|null;readonly supportsText:string|null;}declare var CSSImportRule:{prototype:CSSImportRule;new():CSSImportRule;};interface CSSKeyframeRule extends CSSRule{keyText:string;readonly style:CSSStyleDeclaration;}declare var CSSKeyframeRule:{prototype:CSSKeyframeRule;new():CSSKeyframeRule;};interface CSSKeyframesRule extends CSSRule{readonly cssRules:CSSRuleList;readonly length:number;name:string;appendRule(rule:string):void;deleteRule(select:string):void;findRule(select:string):CSSKeyframeRule|null;[index:number]:CSSKeyframeRule;}declare var CSSKeyframesRule:{prototype:CSSKeyframesRule;new():CSSKeyframesRule;};interface CSSKeywordValue extends CSSStyleValue{value:string;}declare var CSSKeywordValue:{prototype:CSSKeywordValue;new(value:string):CSSKeywordValue;};interface CSSLayerBlockRule extends CSSGroupingRule{readonly name:string;}declare var CSSLayerBlockRule:{prototype:CSSLayerBlockRule;new():CSSLayerBlockRule;};interface CSSLayerStatementRule extends CSSRule{readonly nameList:ReadonlyArray<string>;}declare var CSSLayerStatementRule:{prototype:CSSLayerStatementRule;new():CSSLayerStatementRule;};interface CSSMathClamp extends CSSMathValue{readonly lower:CSSNumericValue;readonly upper:CSSNumericValue;readonly value:CSSNumericValue;}declare var CSSMathClamp:{prototype:CSSMathClamp;new(lower:CSSNumberish,value:CSSNumberish,upper:CSSNumberish):CSSMathClamp;};interface CSSMathInvert extends CSSMathValue{readonly value:CSSNumericValue;}declare var CSSMathInvert:{prototype:CSSMathInvert;new(arg:CSSNumberish):CSSMathInvert;};interface CSSMathMax extends CSSMathValue{readonly values:CSSNumericArray;}declare var CSSMathMax:{prototype:CSSMathMax;new(...args:CSSNumberish[]):CSSMathMax;};interface CSSMathMin extends CSSMathValue{readonly values:CSSNumericArray;}declare var CSSMathMin:{prototype:CSSMathMin;new(...args:CSSNumberish[]):CSSMathMin;};interface CSSMathNegate extends CSSMathValue{readonly value:CSSNumericValue;}declare var CSSMathNegate:{prototype:CSSMathNegate;new(arg:CSSNumberish):CSSMathNegate;};interface CSSMathProduct extends CSSMathValue{readonly values:CSSNumericArray;}declare var CSSMathProduct:{prototype:CSSMathProduct;new(...args:CSSNumberish[]):CSSMathProduct;};interface CSSMathSum extends CSSMathValue{readonly values:CSSNumericArray;}declare var CSSMathSum:{prototype:CSSMathSum;new(...args:CSSNumberish[]):CSSMathSum;};interface CSSMathValue extends CSSNumericValue{readonly operator:CSSMathOperator;}declare var CSSMathValue:{prototype:CSSMathValue;new():CSSMathValue;};interface CSSMatrixComponent extends CSSTransformComponent{matrix:DOMMatrix;}declare var CSSMatrixComponent:{prototype:CSSMatrixComponent;new(matrix:DOMMatrixReadOnly,options?:CSSMatrixComponentOptions):CSSMatrixComponent;};interface CSSMediaRule extends CSSConditionRule{readonly media:MediaList;}declare var CSSMediaRule:{prototype:CSSMediaRule;new():CSSMediaRule;};interface CSSNamespaceRule extends CSSRule{readonly namespaceURI:string;readonly prefix:string;}declare var CSSNamespaceRule:{prototype:CSSNamespaceRule;new():CSSNamespaceRule;};interface CSSNumericArray{readonly length:number;forEach(callbackfn:(value:CSSNumericValue,key:number,parent:CSSNumericArray)=>void,thisArg?:any):void;[index:number]:CSSNumericValue;}declare var CSSNumericArray:{prototype:CSSNumericArray;new():CSSNumericArray;};interface CSSNumericValue extends CSSStyleValue{add(...values:CSSNumberish[]):CSSNumericValue;div(...values:CSSNumberish[]):CSSNumericValue;equals(...value:CSSNumberish[]):boolean;max(...values:CSSNumberish[]):CSSNumericValue;min(...values:CSSNumberish[]):CSSNumericValue;mul(...values:CSSNumberish[]):CSSNumericValue;sub(...values:CSSNumberish[]):CSSNumericValue;to(unit:string):CSSUnitValue;toSum(...units:string[]):CSSMathSum;type():CSSNumericType;}declare var CSSNumericValue:{prototype:CSSNumericValue;new():CSSNumericValue;parse(cssText:string):CSSNumericValue;};interface CSSPageRule extends CSSGroupingRule{selectorText:string;readonly style:CSSStyleDeclaration;}declare var CSSPageRule:{prototype:CSSPageRule;new():CSSPageRule;};interface CSSPerspective extends CSSTransformComponent{length:CSSPerspectiveValue;}declare var CSSPerspective:{prototype:CSSPerspective;new(length:CSSPerspectiveValue):CSSPerspective;};interface CSSPropertyRule extends CSSRule{readonly inherits:boolean;readonly initialValue:string|null;readonly name:string;readonly syntax:string;}declare var CSSPropertyRule:{prototype:CSSPropertyRule;new():CSSPropertyRule;};interface CSSRotate extends CSSTransformComponent{angle:CSSNumericValue;x:CSSNumberish;y:CSSNumberish;z:CSSNumberish;}declare var CSSRotate:{prototype:CSSRotate;new(angle:CSSNumericValue):CSSRotate;new(x:CSSNumberish,y:CSSNumberish,z:CSSNumberish,angle:CSSNumericValue):CSSRotate;};interface CSSRule{cssText:string;readonly parentRule:CSSRule|null;readonly parentStyleSheet:CSSStyleSheet|null;readonly type:number;readonly STYLE_RULE:1;readonly CHARSET_RULE:2;readonly IMPORT_RULE:3;readonly MEDIA_RULE:4;readonly FONT_FACE_RULE:5;readonly PAGE_RULE:6;readonly NAMESPACE_RULE:10;readonly KEYFRAMES_RULE:7;readonly KEYFRAME_RULE:8;readonly SUPPORTS_RULE:12;readonly COUNTER_STYLE_RULE:11;readonly FONT_FEATURE_VALUES_RULE:14;}declare var CSSRule:{prototype:CSSRule;new():CSSRule;readonly STYLE_RULE:1;readonly CHARSET_RULE:2;readonly IMPORT_RULE:3;readonly MEDIA_RULE:4;readonly FONT_FACE_RULE:5;readonly PAGE_RULE:6;readonly NAMESPACE_RULE:10;readonly KEYFRAMES_RULE:7;readonly KEYFRAME_RULE:8;readonly SUPPORTS_RULE:12;readonly COUNTER_STYLE_RULE:11;readonly FONT_FEATURE_VALUES_RULE:14;};interface CSSRuleList{readonly length:number;item(index:number):CSSRule|null;[index:number]:CSSRule;}declare var CSSRuleList:{prototype:CSSRuleList;new():CSSRuleList;};interface CSSScale extends CSSTransformComponent{x:CSSNumberish;y:CSSNumberish;z:CSSNumberish;}declare var CSSScale:{prototype:CSSScale;new(x:CSSNumberish,y:CSSNumberish,z?:CSSNumberish):CSSScale;};interface CSSScopeRule extends CSSGroupingRule{readonly end:string|null;readonly start:string|null;}declare var CSSScopeRule:{prototype:CSSScopeRule;new():CSSScopeRule;};interface CSSSkew extends CSSTransformComponent{ax:CSSNumericValue;ay:CSSNumericValue;}declare var CSSSkew:{prototype:CSSSkew;new(ax:CSSNumericValue,ay:CSSNumericValue):CSSSkew;};interface CSSSkewX extends CSSTransformComponent{ax:CSSNumericValue;}declare var CSSSkewX:{prototype:CSSSkewX;new(ax:CSSNumericValue):CSSSkewX;};interface CSSSkewY extends CSSTransformComponent{ay:CSSNumericValue;}declare var CSSSkewY:{prototype:CSSSkewY;new(ay:CSSNumericValue):CSSSkewY;};interface CSSStartingStyleRule extends CSSGroupingRule{}declare var CSSStartingStyleRule:{prototype:CSSStartingStyleRule;new():CSSStartingStyleRule;};interface CSSStyleDeclaration{accentColor:string;alignContent:string;alignItems:string;alignSelf:string;alignmentBaseline:string;all:string;animation:string;animationComposition:string;animationDelay:string;animationDirection:string;animationDuration:string;animationFillMode:string;animationIterationCount:string;animationName:string;animationPlayState:string;animationTimingFunction:string;appearance:string;aspectRatio:string;backdropFilter:string;backfaceVisibility:string;background:string;backgroundAttachment:string;backgroundBlendMode:string;backgroundClip:string;backgroundColor:string;backgroundImage:string;backgroundOrigin:string;backgroundPosition:string;backgroundPositionX:string;backgroundPositionY:string;backgroundRepeat:string;backgroundSize:string;baselineShift:string;baselineSource:string;blockSize:string;border:string;borderBlock:string;borderBlockColor:string;borderBlockEnd:string;borderBlockEndColor:string;borderBlockEndStyle:string;borderBlockEndWidth:string;borderBlockStart:string;borderBlockStartColor:string;borderBlockStartStyle:string;borderBlockStartWidth:string;borderBlockStyle:string;borderBlockWidth:string;borderBottom:string;borderBottomColor:string;borderBottomLeftRadius:string;borderBottomRightRadius:string;borderBottomStyle:string;borderBottomWidth:string;borderCollapse:string;borderColor:string;borderEndEndRadius:string;borderEndStartRadius:string;borderImage:string;borderImageOutset:string;borderImageRepeat:string;borderImageSlice:string;borderImageSource:string;borderImageWidth:string;borderInline:string;borderInlineColor:string;borderInlineEnd:string;borderInlineEndColor:string;borderInlineEndStyle:string;borderInlineEndWidth:string;borderInlineStart:string;borderInlineStartColor:string;borderInlineStartStyle:string;borderInlineStartWidth:string;borderInlineStyle:string;borderInlineWidth:string;borderLeft:string;borderLeftColor:string;borderLeftStyle:string;borderLeftWidth:string;borderRadius:string;borderRight:string;borderRightColor:string;borderRightStyle:string;borderRightWidth:string;borderSpacing:string;borderStartEndRadius:string;borderStartStartRadius:string;borderStyle:string;borderTop:string;borderTopColor:string;borderTopLeftRadius:string;borderTopRightRadius:string;borderTopStyle:string;borderTopWidth:string;borderWidth:string;bottom:string;boxShadow:string;boxSizing:string;breakAfter:string;breakBefore:string;breakInside:string;captionSide:string;caretColor:string;clear:string;clip:string;clipPath:string;clipRule:string;color:string;colorInterpolation:string;colorInterpolationFilters:string;colorScheme:string;columnCount:string;columnFill:string;columnGap:string;columnRule:string;columnRuleColor:string;columnRuleStyle:string;columnRuleWidth:string;columnSpan:string;columnWidth:string;columns:string;contain:string;containIntrinsicBlockSize:string;containIntrinsicHeight:string;containIntrinsicInlineSize:string;containIntrinsicSize:string;containIntrinsicWidth:string;container:string;containerName:string;containerType:string;content:string;contentVisibility:string;counterIncrement:string;counterReset:string;counterSet:string;cssFloat:string;cssText:string;cursor:string;cx:string;cy:string;d:string;direction:string;display:string;dominantBaseline:string;emptyCells:string;fill:string;fillOpacity:string;fillRule:string;filter:string;flex:string;flexBasis:string;flexDirection:string;flexFlow:string;flexGrow:string;flexShrink:string;flexWrap:string;float:string;floodColor:string;floodOpacity:string;font:string;fontFamily:string;fontFeatureSettings:string;fontKerning:string;fontOpticalSizing:string;fontPalette:string;fontSize:string;fontSizeAdjust:string;fontStretch:string;fontStyle:string;fontSynthesis:string;fontSynthesisSmallCaps:string;fontSynthesisStyle:string;fontSynthesisWeight:string;fontVariant:string;fontVariantAlternates:string;fontVariantCaps:string;fontVariantEastAsian:string;fontVariantLigatures:string;fontVariantNumeric:string;fontVariantPosition:string;fontVariationSettings:string;fontWeight:string;forcedColorAdjust:string;gap:string;grid:string;gridArea:string;gridAutoColumns:string;gridAutoFlow:string;gridAutoRows:string;gridColumn:string;gridColumnEnd:string;gridColumnGap:string;gridColumnStart:string;gridGap:string;gridRow:string;gridRowEnd:string;gridRowGap:string;gridRowStart:string;gridTemplate:string;gridTemplateAreas:string;gridTemplateColumns:string;gridTemplateRows:string;height:string;hyphenateCharacter:string;hyphens:string;imageOrientation:string;imageRendering:string;inlineSize:string;inset:string;insetBlock:string;insetBlockEnd:string;insetBlockStart:string;insetInline:string;insetInlineEnd:string;insetInlineStart:string;isolation:string;justifyContent:string;justifyItems:string;justifySelf:string;left:string;readonly length:number;letterSpacing:string;lightingColor:string;lineBreak:string;lineHeight:string;listStyle:string;listStyleImage:string;listStylePosition:string;listStyleType:string;margin:string;marginBlock:string;marginBlockEnd:string;marginBlockStart:string;marginBottom:string;marginInline:string;marginInlineEnd:string;marginInlineStart:string;marginLeft:string;marginRight:string;marginTop:string;marker:string;markerEnd:string;markerMid:string;markerStart:string;mask:string;maskClip:string;maskComposite:string;maskImage:string;maskMode:string;maskOrigin:string;maskPosition:string;maskRepeat:string;maskSize:string;maskType:string;mathDepth:string;mathStyle:string;maxBlockSize:string;maxHeight:string;maxInlineSize:string;maxWidth:string;minBlockSize:string;minHeight:string;minInlineSize:string;minWidth:string;mixBlendMode:string;objectFit:string;objectPosition:string;offset:string;offsetAnchor:string;offsetDistance:string;offsetPath:string;offsetPosition:string;offsetRotate:string;opacity:string;order:string;orphans:string;outline:string;outlineColor:string;outlineOffset:string;outlineStyle:string;outlineWidth:string;overflow:string;overflowAnchor:string;overflowClipMargin:string;overflowWrap:string;overflowX:string;overflowY:string;overscrollBehavior:string;overscrollBehaviorBlock:string;overscrollBehaviorInline:string;overscrollBehaviorX:string;overscrollBehaviorY:string;padding:string;paddingBlock:string;paddingBlockEnd:string;paddingBlockStart:string;paddingBottom:string;paddingInline:string;paddingInlineEnd:string;paddingInlineStart:string;paddingLeft:string;paddingRight:string;paddingTop:string;page:string;pageBreakAfter:string;pageBreakBefore:string;pageBreakInside:string;paintOrder:string;readonly parentRule:CSSRule|null;perspective:string;perspectiveOrigin:string;placeContent:string;placeItems:string;placeSelf:string;pointerEvents:string;position:string;printColorAdjust:string;quotes:string;r:string;resize:string;right:string;rotate:string;rowGap:string;rubyPosition:string;rx:string;ry:string;scale:string;scrollBehavior:string;scrollMargin:string;scrollMarginBlock:string;scrollMarginBlockEnd:string;scrollMarginBlockStart:string;scrollMarginBottom:string;scrollMarginInline:string;scrollMarginInlineEnd:string;scrollMarginInlineStart:string;scrollMarginLeft:string;scrollMarginRight:string;scrollMarginTop:string;scrollPadding:string;scrollPaddingBlock:string;scrollPaddingBlockEnd:string;scrollPaddingBlockStart:string;scrollPaddingBottom:string;scrollPaddingInline:string;scrollPaddingInlineEnd:string;scrollPaddingInlineStart:string;scrollPaddingLeft:string;scrollPaddingRight:string;scrollPaddingTop:string;scrollSnapAlign:string;scrollSnapStop:string;scrollSnapType:string;scrollbarColor:string;scrollbarGutter:string;scrollbarWidth:string;shapeImageThreshold:string;shapeMargin:string;shapeOutside:string;shapeRendering:string;stopColor:string;stopOpacity:string;stroke:string;strokeDasharray:string;strokeDashoffset:string;strokeLinecap:string;strokeLinejoin:string;strokeMiterlimit:string;strokeOpacity:string;strokeWidth:string;tabSize:string;tableLayout:string;textAlign:string;textAlignLast:string;textAnchor:string;textCombineUpright:string;textDecoration:string;textDecorationColor:string;textDecorationLine:string;textDecorationSkipInk:string;textDecorationStyle:string;textDecorationThickness:string;textEmphasis:string;textEmphasisColor:string;textEmphasisPosition:string;textEmphasisStyle:string;textIndent:string;textOrientation:string;textOverflow:string;textRendering:string;textShadow:string;textTransform:string;textUnderlineOffset:string;textUnderlinePosition:string;textWrap:string;textWrapMode:string;textWrapStyle:string;top:string;touchAction:string;transform:string;transformBox:string;transformOrigin:string;transformStyle:string;transition:string;transitionBehavior:string;transitionDelay:string;transitionDuration:string;transitionProperty:string;transitionTimingFunction:string;translate:string;unicodeBidi:string;userSelect:string;vectorEffect:string;verticalAlign:string;viewTransitionName:string;visibility:string;webkitAlignContent:string;webkitAlignItems:string;webkitAlignSelf:string;webkitAnimation:string;webkitAnimationDelay:string;webkitAnimationDirection:string;webkitAnimationDuration:string;webkitAnimationFillMode:string;webkitAnimationIterationCount:string;webkitAnimationName:string;webkitAnimationPlayState:string;webkitAnimationTimingFunction:string;webkitAppearance:string;webkitBackfaceVisibility:string;webkitBackgroundClip:string;webkitBackgroundOrigin:string;webkitBackgroundSize:string;webkitBorderBottomLeftRadius:string;webkitBorderBottomRightRadius:string;webkitBorderRadius:string;webkitBorderTopLeftRadius:string;webkitBorderTopRightRadius:string;webkitBoxAlign:string;webkitBoxFlex:string;webkitBoxOrdinalGroup:string;webkitBoxOrient:string;webkitBoxPack:string;webkitBoxShadow:string;webkitBoxSizing:string;webkitFilter:string;webkitFlex:string;webkitFlexBasis:string;webkitFlexDirection:string;webkitFlexFlow:string;webkitFlexGrow:string;webkitFlexShrink:string;webkitFlexWrap:string;webkitJustifyContent:string;webkitLineClamp:string;webkitMask:string;webkitMaskBoxImage:string;webkitMaskBoxImageOutset:string;webkitMaskBoxImageRepeat:string;webkitMaskBoxImageSlice:string;webkitMaskBoxImageSource:string;webkitMaskBoxImageWidth:string;webkitMaskClip:string;webkitMaskComposite:string;webkitMaskImage:string;webkitMaskOrigin:string;webkitMaskPosition:string;webkitMaskRepeat:string;webkitMaskSize:string;webkitOrder:string;webkitPerspective:string;webkitPerspectiveOrigin:string;webkitTextFillColor:string;webkitTextSizeAdjust:string;webkitTextStroke:string;webkitTextStrokeColor:string;webkitTextStrokeWidth:string;webkitTransform:string;webkitTransformOrigin:string;webkitTransformStyle:string;webkitTransition:string;webkitTransitionDelay:string;webkitTransitionDuration:string;webkitTransitionProperty:string;webkitTransitionTimingFunction:string;webkitUserSelect:string;whiteSpace:string;whiteSpaceCollapse:string;widows:string;width:string;willChange:string;wordBreak:string;wordSpacing:string;wordWrap:string;writingMode:string;x:string;y:string;zIndex:string;zoom:string;getPropertyPriority(property:string):string;getPropertyValue(property:string):string;item(index:number):string;removeProperty(property:string):string;setProperty(property:string,value:string|null,priority?:string):void;[index:number]:string;}declare var CSSStyleDeclaration:{prototype:CSSStyleDeclaration;new():CSSStyleDeclaration;};interface CSSStyleRule extends CSSGroupingRule{selectorText:string;readonly style:CSSStyleDeclaration;readonly styleMap:StylePropertyMap;}declare var CSSStyleRule:{prototype:CSSStyleRule;new():CSSStyleRule;};interface CSSStyleSheet extends StyleSheet{readonly cssRules:CSSRuleList;readonly ownerRule:CSSRule|null;readonly rules:CSSRuleList;addRule(selector?:string,style?:string,index?:number):number;deleteRule(index:number):void;insertRule(rule:string,index?:number):number;removeRule(index?:number):void;replace(text:string):Promise<CSSStyleSheet>;replaceSync(text:string):void;}declare var CSSStyleSheet:{prototype:CSSStyleSheet;new(options?:CSSStyleSheetInit):CSSStyleSheet;};interface CSSStyleValue{toString():string;}declare var CSSStyleValue:{prototype:CSSStyleValue;new():CSSStyleValue;parse(property:string,cssText:string):CSSStyleValue;parseAll(property:string,cssText:string):CSSStyleValue[];};interface CSSSupportsRule extends CSSConditionRule{}declare var CSSSupportsRule:{prototype:CSSSupportsRule;new():CSSSupportsRule;};interface CSSTransformComponent{is2D:boolean;toMatrix():DOMMatrix;toString():string;}declare var CSSTransformComponent:{prototype:CSSTransformComponent;new():CSSTransformComponent;};interface CSSTransformValue extends CSSStyleValue{readonly is2D:boolean;readonly length:number;toMatrix():DOMMatrix;forEach(callbackfn:(value:CSSTransformComponent,key:number,parent:CSSTransformValue)=>void,thisArg?:any):void;[index:number]:CSSTransformComponent;}declare var CSSTransformValue:{prototype:CSSTransformValue;new(transforms:CSSTransformComponent[]):CSSTransformValue;};interface CSSTransition extends Animation{readonly transitionProperty:string;addEventListener<K extends keyof AnimationEventMap>(type:K,listener:(this:CSSTransition,ev:AnimationEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AnimationEventMap>(type:K,listener:(this:CSSTransition,ev:AnimationEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var CSSTransition:{prototype:CSSTransition;new():CSSTransition;};interface CSSTranslate extends CSSTransformComponent{x:CSSNumericValue;y:CSSNumericValue;z:CSSNumericValue;}declare var CSSTranslate:{prototype:CSSTranslate;new(x:CSSNumericValue,y:CSSNumericValue,z?:CSSNumericValue):CSSTranslate;};interface CSSUnitValue extends CSSNumericValue{readonly unit:string;value:number;}declare var CSSUnitValue:{prototype:CSSUnitValue;new(value:number,unit:string):CSSUnitValue;};interface CSSUnparsedValue extends CSSStyleValue{readonly length:number;forEach(callbackfn:(value:CSSUnparsedSegment,key:number,parent:CSSUnparsedValue)=>void,thisArg?:any):void;[index:number]:CSSUnparsedSegment;}declare var CSSUnparsedValue:{prototype:CSSUnparsedValue;new(members:CSSUnparsedSegment[]):CSSUnparsedValue;};interface CSSVariableReferenceValue{readonly fallback:CSSUnparsedValue|null;variable:string;}declare var CSSVariableReferenceValue:{prototype:CSSVariableReferenceValue;new(variable:string,fallback?:CSSUnparsedValue|null):CSSVariableReferenceValue;};interface Cache{add(request:RequestInfo|URL):Promise<void>;addAll(requests:RequestInfo[]):Promise<void>;delete(request:RequestInfo|URL,options?:CacheQueryOptions):Promise<boolean>;keys(request?:RequestInfo|URL,options?:CacheQueryOptions):Promise<ReadonlyArray<Request>>;match(request:RequestInfo|URL,options?:CacheQueryOptions):Promise<Response|undefined>;matchAll(request?:RequestInfo|URL,options?:CacheQueryOptions):Promise<ReadonlyArray<Response>>;put(request:RequestInfo|URL,response:Response):Promise<void>;}declare var Cache:{prototype:Cache;new():Cache;};interface CacheStorage{delete(cacheName:string):Promise<boolean>;has(cacheName:string):Promise<boolean>;keys():Promise<string[]>;match(request:RequestInfo|URL,options?:MultiCacheQueryOptions):Promise<Response|undefined>;open(cacheName:string):Promise<Cache>;}declare var CacheStorage:{prototype:CacheStorage;new():CacheStorage;};interface CanvasCaptureMediaStreamTrack extends MediaStreamTrack{readonly canvas:HTMLCanvasElement;requestFrame():void;addEventListener<K extends keyof MediaStreamTrackEventMap>(type:K,listener:(this:CanvasCaptureMediaStreamTrack,ev:MediaStreamTrackEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof MediaStreamTrackEventMap>(type:K,listener:(this:CanvasCaptureMediaStreamTrack,ev:MediaStreamTrackEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var CanvasCaptureMediaStreamTrack:{prototype:CanvasCaptureMediaStreamTrack;new():CanvasCaptureMediaStreamTrack;};interface CanvasCompositing{globalAlpha:number;globalCompositeOperation:GlobalCompositeOperation;}interface CanvasDrawImage{drawImage(image:CanvasImageSource,dx:number,dy:number):void;drawImage(image:CanvasImageSource,dx:number,dy:number,dw:number,dh:number):void;drawImage(image:CanvasImageSource,sx:number,sy:number,sw:number,sh:number,dx:number,dy:number,dw:number,dh:number):void;}interface CanvasDrawPath{beginPath():void;clip(fillRule?:CanvasFillRule):void;clip(path:Path2D,fillRule?:CanvasFillRule):void;fill(fillRule?:CanvasFillRule):void;fill(path:Path2D,fillRule?:CanvasFillRule):void;isPointInPath(x:number,y:number,fillRule?:CanvasFillRule):boolean;isPointInPath(path:Path2D,x:number,y:number,fillRule?:CanvasFillRule):boolean;isPointInStroke(x:number,y:number):boolean;isPointInStroke(path:Path2D,x:number,y:number):boolean;stroke():void;stroke(path:Path2D):void;}interface CanvasFillStrokeStyles{fillStyle:string|CanvasGradient|CanvasPattern;strokeStyle:string|CanvasGradient|CanvasPattern;createConicGradient(startAngle:number,x:number,y:number):CanvasGradient;createLinearGradient(x0:number,y0:number,x1:number,y1:number):CanvasGradient;createPattern(image:CanvasImageSource,repetition:string|null):CanvasPattern|null;createRadialGradient(x0:number,y0:number,r0:number,x1:number,y1:number,r1:number):CanvasGradient;}interface CanvasFilters{filter:string;}interface CanvasGradient{addColorStop(offset:number,color:string):void;}declare var CanvasGradient:{prototype:CanvasGradient;new():CanvasGradient;};interface CanvasImageData{createImageData(sw:number,sh:number,settings?:ImageDataSettings):ImageData;createImageData(imagedata:ImageData):ImageData;getImageData(sx:number,sy:number,sw:number,sh:number,settings?:ImageDataSettings):ImageData;putImageData(imagedata:ImageData,dx:number,dy:number):void;putImageData(imagedata:ImageData,dx:number,dy:number,dirtyX:number,dirtyY:number,dirtyWidth:number,dirtyHeight:number):void;}interface CanvasImageSmoothing{imageSmoothingEnabled:boolean;imageSmoothingQuality:ImageSmoothingQuality;}interface CanvasPath{arc(x:number,y:number,radius:number,startAngle:number,endAngle:number,counterclockwise?:boolean):void;arcTo(x1:number,y1:number,x2:number,y2:number,radius:number):void;bezierCurveTo(cp1x:number,cp1y:number,cp2x:number,cp2y:number,x:number,y:number):void;closePath():void;ellipse(x:number,y:number,radiusX:number,radiusY:number,rotation:number,startAngle:number,endAngle:number,counterclockwise?:boolean):void;lineTo(x:number,y:number):void;moveTo(x:number,y:number):void;quadraticCurveTo(cpx:number,cpy:number,x:number,y:number):void;rect(x:number,y:number,w:number,h:number):void;roundRect(x:number,y:number,w:number,h:number,radii?:number|DOMPointInit|(number|DOMPointInit)[]):void;}interface CanvasPathDrawingStyles{lineCap:CanvasLineCap;lineDashOffset:number;lineJoin:CanvasLineJoin;lineWidth:number;miterLimit:number;getLineDash():number[];setLineDash(segments:number[]):void;}interface CanvasPattern{setTransform(transform?:DOMMatrix2DInit):void;}declare var CanvasPattern:{prototype:CanvasPattern;new():CanvasPattern;};interface CanvasRect{clearRect(x:number,y:number,w:number,h:number):void;fillRect(x:number,y:number,w:number,h:number):void;strokeRect(x:number,y:number,w:number,h:number):void;}interface CanvasRenderingContext2D extends CanvasCompositing,CanvasDrawImage,CanvasDrawPath,CanvasFillStrokeStyles,CanvasFilters,CanvasImageData,CanvasImageSmoothing,CanvasPath,CanvasPathDrawingStyles,CanvasRect,CanvasShadowStyles,CanvasState,CanvasText,CanvasTextDrawingStyles,CanvasTransform,CanvasUserInterface{readonly canvas:HTMLCanvasElement;getContextAttributes():CanvasRenderingContext2DSettings;}declare var CanvasRenderingContext2D:{prototype:CanvasRenderingContext2D;new():CanvasRenderingContext2D;};interface CanvasShadowStyles{shadowBlur:number;shadowColor:string;shadowOffsetX:number;shadowOffsetY:number;}interface CanvasState{isContextLost():boolean;reset():void;restore():void;save():void;}interface CanvasText{fillText(text:string,x:number,y:number,maxWidth?:number):void;measureText(text:string):TextMetrics;strokeText(text:string,x:number,y:number,maxWidth?:number):void;}interface CanvasTextDrawingStyles{direction:CanvasDirection;font:string;fontKerning:CanvasFontKerning;fontStretch:CanvasFontStretch;fontVariantCaps:CanvasFontVariantCaps;letterSpacing:string;textAlign:CanvasTextAlign;textBaseline:CanvasTextBaseline;textRendering:CanvasTextRendering;wordSpacing:string;}interface CanvasTransform{getTransform():DOMMatrix;resetTransform():void;rotate(angle:number):void;scale(x:number,y:number):void;setTransform(a:number,b:number,c:number,d:number,e:number,f:number):void;setTransform(transform?:DOMMatrix2DInit):void;transform(a:number,b:number,c:number,d:number,e:number,f:number):void;translate(x:number,y:number):void;}interface CanvasUserInterface{drawFocusIfNeeded(element:Element):void;drawFocusIfNeeded(path:Path2D,element:Element):void;}interface ChannelMergerNode extends AudioNode{}declare var ChannelMergerNode:{prototype:ChannelMergerNode;new(context:BaseAudioContext,options?:ChannelMergerOptions):ChannelMergerNode;};interface ChannelSplitterNode extends AudioNode{}declare var ChannelSplitterNode:{prototype:ChannelSplitterNode;new(context:BaseAudioContext,options?:ChannelSplitterOptions):ChannelSplitterNode;};interface CharacterData extends Node,ChildNode,NonDocumentTypeChildNode{data:string;readonly length:number;readonly ownerDocument:Document;appendData(data:string):void;deleteData(offset:number,count:number):void;insertData(offset:number,data:string):void;replaceData(offset:number,count:number,data:string):void;substringData(offset:number,count:number):string;}declare var CharacterData:{prototype:CharacterData;new():CharacterData;};interface ChildNode extends Node{after(...nodes:(Node|string)[]):void;before(...nodes:(Node|string)[]):void;remove():void;replaceWith(...nodes:(Node|string)[]):void;}interface ClientRect extends DOMRect{}interface Clipboard extends EventTarget{read():Promise<ClipboardItems>;readText():Promise<string>;write(data:ClipboardItems):Promise<void>;writeText(data:string):Promise<void>;}declare var Clipboard:{prototype:Clipboard;new():Clipboard;};interface ClipboardEvent extends Event{readonly clipboardData:DataTransfer|null;}declare var ClipboardEvent:{prototype:ClipboardEvent;new(type:string,eventInitDict?:ClipboardEventInit):ClipboardEvent;};interface ClipboardItem{readonly presentationStyle:PresentationStyle;readonly types:ReadonlyArray<string>;getType(type:string):Promise<Blob>;}declare var ClipboardItem:{prototype:ClipboardItem;new(items:Record<string,string|Blob|PromiseLike<string|Blob>>,options?:ClipboardItemOptions):ClipboardItem;supports(type:string):boolean;};interface CloseEvent extends Event{readonly code:number;readonly reason:string;readonly wasClean:boolean;}declare var CloseEvent:{prototype:CloseEvent;new(type:string,eventInitDict?:CloseEventInit):CloseEvent;};interface Comment extends CharacterData{}declare var Comment:{prototype:Comment;new(data?:string):Comment;};interface CompositionEvent extends UIEvent{readonly data:string;initCompositionEvent(typeArg:string,bubblesArg?:boolean,cancelableArg?:boolean,viewArg?:WindowProxy|null,dataArg?:string):void;}declare var CompositionEvent:{prototype:CompositionEvent;new(type:string,eventInitDict?:CompositionEventInit):CompositionEvent;};interface CompressionStream extends GenericTransformStream{}declare var CompressionStream:{prototype:CompressionStream;new(format:CompressionFormat):CompressionStream;};interface ConstantSourceNode extends AudioScheduledSourceNode{readonly offset:AudioParam;addEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type:K,listener:(this:ConstantSourceNode,ev:AudioScheduledSourceNodeEventMap[K])=>any,options?:boolean|AddEventListenerOptions):void;addEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|AddEventListenerOptions):void;removeEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type:K,listener:(this:ConstantSourceNode,ev:AudioScheduledSourceNodeEventMap[K])=>any,options?:boolean|EventListenerOptions):void;removeEventListener(type:string,listener:EventListenerOrEventListenerObject,options?:boolean|EventListenerOptions):void;}declare var ConstantSourceNode:{prototype:ConstantSourceNode;new(context:BaseAudioContext,options?:ConstantSourceOptions):ConstantSourceNode;};interface ContentVisibilityAutoStateChangeEvent extends Event{readonly skipped:boolean;}declare var ContentVisibilityAutoStateChangeEvent:{prototype:ContentVisibilityAutoStateChangeEvent;new(type:string,eventInitDict?:ContentVisibilityAutoStateChangeEventInit):ContentVisibilityAutoStateChangeEvent;};interface ConvolverNode extends AudioNode{buffer:AudioBuffer|null;normalize:boolean;}declare var ConvolverNode:{prototype:ConvolverNode;new(context:BaseAudioContext,options?:ConvolverOptions):ConvolverNode;};interface CountQueuingStrategy extends QueuingStrategy{readonly highWaterMark:number;readonly size:QueuingStrategySize;}declare var CountQueuingStrategy:{prototype:CountQueuingStrategy;new(init:QueuingStrategyInit):CountQueuingStrategy;};interface Credential{readonly id:string;readonly type:string;}declare var Credential:{prototype:Credential;new():Credential;};interface CredentialsContainer{create(options?:CredentialCreationOptions):Promise<Credential|null>;get(options?:CredentialRequestOptions):Promise<Credential|null>;preventSilentAccess():Promise<void>;store(credential:Credential):Promise<void>;}declare var CredentialsContainer:{prototype:CredentialsContainer;new():CredentialsContainer;};interface Crypto{readonly subtle:SubtleCrypto;getRandomValues<T extends ArrayBufferView|null>(array:T):T;randomUUID():`${string}-${string}-${string}-${string}-${string}`;\n}\n\ndeclare var Crypto: {\n    prototype: Crypto;\n    new(): Crypto;\n};\n\n/**\n * The CryptoKey dictionary of the Web Crypto API represents a cryptographic key.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CryptoKey)\n */\ninterface CryptoKey {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CryptoKey/algorithm) */\n    readonly algorithm: KeyAlgorithm;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CryptoKey/extractable) */\n    readonly extractable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CryptoKey/type) */\n    readonly type: KeyType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CryptoKey/usages) */\n    readonly usages: KeyUsage[];\n}\n\ndeclare var CryptoKey: {\n    prototype: CryptoKey;\n    new(): CryptoKey;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomElementRegistry) */\ninterface CustomElementRegistry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomElementRegistry/define) */\n    define(name: string, constructor: CustomElementConstructor, options?: ElementDefinitionOptions): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomElementRegistry/get) */\n    get(name: string): CustomElementConstructor | undefined;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomElementRegistry/getName) */\n    getName(constructor: CustomElementConstructor): string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomElementRegistry/upgrade) */\n    upgrade(root: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomElementRegistry/whenDefined) */\n    whenDefined(name: string): Promise<CustomElementConstructor>;\n}\n\ndeclare var CustomElementRegistry: {\n    prototype: CustomElementRegistry;\n    new(): CustomElementRegistry;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomEvent) */\ninterface CustomEvent<T = any> extends Event {\n    /**\n     * Returns any custom data event was created with. Typically used for synthetic events.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomEvent/detail)\n     */\n    readonly detail: T;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomEvent/initCustomEvent)\n     */\n    initCustomEvent(type: string, bubbles?: boolean, cancelable?: boolean, detail?: T): void;\n}\n\ndeclare var CustomEvent: {\n    prototype: CustomEvent;\n    new<T>(type: string, eventInitDict?: CustomEventInit<T>): CustomEvent<T>;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CustomStateSet) */\ninterface CustomStateSet {\n    forEach(callbackfn: (value: string, key: string, parent: CustomStateSet) => void, thisArg?: any): void;\n}\n\ndeclare var CustomStateSet: {\n    prototype: CustomStateSet;\n    new(): CustomStateSet;\n};\n\n/**\n * An abnormal event (called an exception) which occurs as a result of calling a method or accessing a property of a web API.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMException)\n */\ninterface DOMException extends Error {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMException/code)\n     */\n    readonly code: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMException/message) */\n    readonly message: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMException/name) */\n    readonly name: string;\n    readonly INDEX_SIZE_ERR: 1;\n    readonly DOMSTRING_SIZE_ERR: 2;\n    readonly HIERARCHY_REQUEST_ERR: 3;\n    readonly WRONG_DOCUMENT_ERR: 4;\n    readonly INVALID_CHARACTER_ERR: 5;\n    readonly NO_DATA_ALLOWED_ERR: 6;\n    readonly NO_MODIFICATION_ALLOWED_ERR: 7;\n    readonly NOT_FOUND_ERR: 8;\n    readonly NOT_SUPPORTED_ERR: 9;\n    readonly INUSE_ATTRIBUTE_ERR: 10;\n    readonly INVALID_STATE_ERR: 11;\n    readonly SYNTAX_ERR: 12;\n    readonly INVALID_MODIFICATION_ERR: 13;\n    readonly NAMESPACE_ERR: 14;\n    readonly INVALID_ACCESS_ERR: 15;\n    readonly VALIDATION_ERR: 16;\n    readonly TYPE_MISMATCH_ERR: 17;\n    readonly SECURITY_ERR: 18;\n    readonly NETWORK_ERR: 19;\n    readonly ABORT_ERR: 20;\n    readonly URL_MISMATCH_ERR: 21;\n    readonly QUOTA_EXCEEDED_ERR: 22;\n    readonly TIMEOUT_ERR: 23;\n    readonly INVALID_NODE_TYPE_ERR: 24;\n    readonly DATA_CLONE_ERR: 25;\n}\n\ndeclare var DOMException: {\n    prototype: DOMException;\n    new(message?: string, name?: string): DOMException;\n    readonly INDEX_SIZE_ERR: 1;\n    readonly DOMSTRING_SIZE_ERR: 2;\n    readonly HIERARCHY_REQUEST_ERR: 3;\n    readonly WRONG_DOCUMENT_ERR: 4;\n    readonly INVALID_CHARACTER_ERR: 5;\n    readonly NO_DATA_ALLOWED_ERR: 6;\n    readonly NO_MODIFICATION_ALLOWED_ERR: 7;\n    readonly NOT_FOUND_ERR: 8;\n    readonly NOT_SUPPORTED_ERR: 9;\n    readonly INUSE_ATTRIBUTE_ERR: 10;\n    readonly INVALID_STATE_ERR: 11;\n    readonly SYNTAX_ERR: 12;\n    readonly INVALID_MODIFICATION_ERR: 13;\n    readonly NAMESPACE_ERR: 14;\n    readonly INVALID_ACCESS_ERR: 15;\n    readonly VALIDATION_ERR: 16;\n    readonly TYPE_MISMATCH_ERR: 17;\n    readonly SECURITY_ERR: 18;\n    readonly NETWORK_ERR: 19;\n    readonly ABORT_ERR: 20;\n    readonly URL_MISMATCH_ERR: 21;\n    readonly QUOTA_EXCEEDED_ERR: 22;\n    readonly TIMEOUT_ERR: 23;\n    readonly INVALID_NODE_TYPE_ERR: 24;\n    readonly DATA_CLONE_ERR: 25;\n};\n\n/**\n * An object providing methods which are not dependent on any particular document. Such an object is returned by the Document.implementation property.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMImplementation)\n */\ninterface DOMImplementation {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMImplementation/createDocument) */\n    createDocument(namespace: string | null, qualifiedName: string | null, doctype?: DocumentType | null): XMLDocument;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMImplementation/createDocumentType) */\n    createDocumentType(qualifiedName: string, publicId: string, systemId: string): DocumentType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMImplementation/createHTMLDocument) */\n    createHTMLDocument(title?: string): Document;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMImplementation/hasFeature)\n     */\n    hasFeature(...args: any[]): true;\n}\n\ndeclare var DOMImplementation: {\n    prototype: DOMImplementation;\n    new(): DOMImplementation;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMMatrix) */\ninterface DOMMatrix extends DOMMatrixReadOnly {\n    a: number;\n    b: number;\n    c: number;\n    d: number;\n    e: number;\n    f: number;\n    m11: number;\n    m12: number;\n    m13: number;\n    m14: number;\n    m21: number;\n    m22: number;\n    m23: number;\n    m24: number;\n    m31: number;\n    m32: number;\n    m33: number;\n    m34: number;\n    m41: number;\n    m42: number;\n    m43: number;\n    m44: number;\n    invertSelf(): DOMMatrix;\n    multiplySelf(other?: DOMMatrixInit): DOMMatrix;\n    preMultiplySelf(other?: DOMMatrixInit): DOMMatrix;\n    rotateAxisAngleSelf(x?: number, y?: number, z?: number, angle?: number): DOMMatrix;\n    rotateFromVectorSelf(x?: number, y?: number): DOMMatrix;\n    rotateSelf(rotX?: number, rotY?: number, rotZ?: number): DOMMatrix;\n    scale3dSelf(scale?: number, originX?: number, originY?: number, originZ?: number): DOMMatrix;\n    scaleSelf(scaleX?: number, scaleY?: number, scaleZ?: number, originX?: number, originY?: number, originZ?: number): DOMMatrix;\n    setMatrixValue(transformList: string): DOMMatrix;\n    skewXSelf(sx?: number): DOMMatrix;\n    skewYSelf(sy?: number): DOMMatrix;\n    translateSelf(tx?: number, ty?: number, tz?: number): DOMMatrix;\n}\n\ndeclare var DOMMatrix: {\n    prototype: DOMMatrix;\n    new(init?: string | number[]): DOMMatrix;\n    fromFloat32Array(array32: Float32Array): DOMMatrix;\n    fromFloat64Array(array64: Float64Array): DOMMatrix;\n    fromMatrix(other?: DOMMatrixInit): DOMMatrix;\n};\n\ntype SVGMatrix = DOMMatrix;\ndeclare var SVGMatrix: typeof DOMMatrix;\n\ntype WebKitCSSMatrix = DOMMatrix;\ndeclare var WebKitCSSMatrix: typeof DOMMatrix;\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMMatrixReadOnly) */\ninterface DOMMatrixReadOnly {\n    readonly a: number;\n    readonly b: number;\n    readonly c: number;\n    readonly d: number;\n    readonly e: number;\n    readonly f: number;\n    readonly is2D: boolean;\n    readonly isIdentity: boolean;\n    readonly m11: number;\n    readonly m12: number;\n    readonly m13: number;\n    readonly m14: number;\n    readonly m21: number;\n    readonly m22: number;\n    readonly m23: number;\n    readonly m24: number;\n    readonly m31: number;\n    readonly m32: number;\n    readonly m33: number;\n    readonly m34: number;\n    readonly m41: number;\n    readonly m42: number;\n    readonly m43: number;\n    readonly m44: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMMatrixReadOnly/flipX) */\n    flipX(): DOMMatrix;\n    flipY(): DOMMatrix;\n    inverse(): DOMMatrix;\n    multiply(other?: DOMMatrixInit): DOMMatrix;\n    rotate(rotX?: number, rotY?: number, rotZ?: number): DOMMatrix;\n    rotateAxisAngle(x?: number, y?: number, z?: number, angle?: number): DOMMatrix;\n    rotateFromVector(x?: number, y?: number): DOMMatrix;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMMatrixReadOnly/scale) */\n    scale(scaleX?: number, scaleY?: number, scaleZ?: number, originX?: number, originY?: number, originZ?: number): DOMMatrix;\n    scale3d(scale?: number, originX?: number, originY?: number, originZ?: number): DOMMatrix;\n    /** @deprecated */\n    scaleNonUniform(scaleX?: number, scaleY?: number): DOMMatrix;\n    skewX(sx?: number): DOMMatrix;\n    skewY(sy?: number): DOMMatrix;\n    toFloat32Array(): Float32Array;\n    toFloat64Array(): Float64Array;\n    toJSON(): any;\n    transformPoint(point?: DOMPointInit): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMMatrixReadOnly/translate) */\n    translate(tx?: number, ty?: number, tz?: number): DOMMatrix;\n    toString(): string;\n}\n\ndeclare var DOMMatrixReadOnly: {\n    prototype: DOMMatrixReadOnly;\n    new(init?: string | number[]): DOMMatrixReadOnly;\n    fromFloat32Array(array32: Float32Array): DOMMatrixReadOnly;\n    fromFloat64Array(array64: Float64Array): DOMMatrixReadOnly;\n    fromMatrix(other?: DOMMatrixInit): DOMMatrixReadOnly;\n};\n\n/**\n * Provides the ability to parse XML or HTML source code from a string into a DOM Document.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMParser)\n */\ninterface DOMParser {\n    /**\n     * Parses string using either the HTML or XML parser, according to type, and returns the resulting Document. type can be \"text/html\" (which will invoke the HTML parser), or any of \"text/xml\", \"application/xml\", \"application/xhtml+xml\", or \"image/svg+xml\" (which will invoke the XML parser).\n     *\n     * For the XML parser, if string cannot be parsed, then the returned Document will contain elements describing the resulting error.\n     *\n     * Note that script elements are not evaluated during parsing, and the resulting document's encoding will always be UTF-8.\n     *\n     * Values other than the above for type will cause a TypeError exception to be thrown.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMParser/parseFromString)\n     */\n    parseFromString(string: string, type: DOMParserSupportedType): Document;\n}\n\ndeclare var DOMParser: {\n    prototype: DOMParser;\n    new(): DOMParser;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPoint) */\ninterface DOMPoint extends DOMPointReadOnly {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPoint/w) */\n    w: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPoint/x) */\n    x: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPoint/y) */\n    y: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPoint/z) */\n    z: number;\n}\n\ndeclare var DOMPoint: {\n    prototype: DOMPoint;\n    new(x?: number, y?: number, z?: number, w?: number): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPoint/fromPoint_static) */\n    fromPoint(other?: DOMPointInit): DOMPoint;\n};\n\ntype SVGPoint = DOMPoint;\ndeclare var SVGPoint: typeof DOMPoint;\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly) */\ninterface DOMPointReadOnly {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly/w) */\n    readonly w: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly/x) */\n    readonly x: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly/y) */\n    readonly y: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly/z) */\n    readonly z: number;\n    matrixTransform(matrix?: DOMMatrixInit): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var DOMPointReadOnly: {\n    prototype: DOMPointReadOnly;\n    new(x?: number, y?: number, z?: number, w?: number): DOMPointReadOnly;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMPointReadOnly/fromPoint_static) */\n    fromPoint(other?: DOMPointInit): DOMPointReadOnly;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMQuad) */\ninterface DOMQuad {\n    readonly p1: DOMPoint;\n    readonly p2: DOMPoint;\n    readonly p3: DOMPoint;\n    readonly p4: DOMPoint;\n    getBounds(): DOMRect;\n    toJSON(): any;\n}\n\ndeclare var DOMQuad: {\n    prototype: DOMQuad;\n    new(p1?: DOMPointInit, p2?: DOMPointInit, p3?: DOMPointInit, p4?: DOMPointInit): DOMQuad;\n    fromQuad(other?: DOMQuadInit): DOMQuad;\n    fromRect(other?: DOMRectInit): DOMQuad;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRect) */\ninterface DOMRect extends DOMRectReadOnly {\n    height: number;\n    width: number;\n    x: number;\n    y: number;\n}\n\ndeclare var DOMRect: {\n    prototype: DOMRect;\n    new(x?: number, y?: number, width?: number, height?: number): DOMRect;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRect/fromRect_static) */\n    fromRect(other?: DOMRectInit): DOMRect;\n};\n\ntype SVGRect = DOMRect;\ndeclare var SVGRect: typeof DOMRect;\n\ninterface DOMRectList {\n    readonly length: number;\n    item(index: number): DOMRect | null;\n    [index: number]: DOMRect;\n}\n\ndeclare var DOMRectList: {\n    prototype: DOMRectList;\n    new(): DOMRectList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly) */\ninterface DOMRectReadOnly {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/bottom) */\n    readonly bottom: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/height) */\n    readonly height: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/left) */\n    readonly left: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/right) */\n    readonly right: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/top) */\n    readonly top: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/width) */\n    readonly width: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/x) */\n    readonly x: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/y) */\n    readonly y: number;\n    toJSON(): any;\n}\n\ndeclare var DOMRectReadOnly: {\n    prototype: DOMRectReadOnly;\n    new(x?: number, y?: number, width?: number, height?: number): DOMRectReadOnly;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMRectReadOnly/fromRect_static) */\n    fromRect(other?: DOMRectInit): DOMRectReadOnly;\n};\n\n/**\n * A type returned by some APIs which contains a list of DOMString (strings).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMStringList)\n */\ninterface DOMStringList {\n    /**\n     * Returns the number of strings in strings.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMStringList/length)\n     */\n    readonly length: number;\n    /**\n     * Returns true if strings contains string, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMStringList/contains)\n     */\n    contains(string: string): boolean;\n    /**\n     * Returns the string with index index from strings.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMStringList/item)\n     */\n    item(index: number): string | null;\n    [index: number]: string;\n}\n\ndeclare var DOMStringList: {\n    prototype: DOMStringList;\n    new(): DOMStringList;\n};\n\n/**\n * Used by the dataset HTML attribute to represent data for custom attributes added to elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMStringMap)\n */\ninterface DOMStringMap {\n    [name: string]: string | undefined;\n}\n\ndeclare var DOMStringMap: {\n    prototype: DOMStringMap;\n    new(): DOMStringMap;\n};\n\n/**\n * A set of space-separated tokens. Such a set is returned by Element.classList, HTMLLinkElement.relList, HTMLAnchorElement.relList, HTMLAreaElement.relList, HTMLIframeElement.sandbox, or HTMLOutputElement.htmlFor. It is indexed beginning with 0 as with JavaScript Array objects. DOMTokenList is always case-sensitive.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList)\n */\ninterface DOMTokenList {\n    /**\n     * Returns the number of tokens.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/length)\n     */\n    readonly length: number;\n    /**\n     * Returns the associated set as string.\n     *\n     * Can be set, to change the associated attribute.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/value)\n     */\n    value: string;\n    toString(): string;\n    /**\n     * Adds all arguments passed, except those already present.\n     *\n     * Throws a \"SyntaxError\" DOMException if one of the arguments is the empty string.\n     *\n     * Throws an \"InvalidCharacterError\" DOMException if one of the arguments contains any ASCII whitespace.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/add)\n     */\n    add(...tokens: string[]): void;\n    /**\n     * Returns true if token is present, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/contains)\n     */\n    contains(token: string): boolean;\n    /**\n     * Returns the token with index index.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/item)\n     */\n    item(index: number): string | null;\n    /**\n     * Removes arguments passed, if they are present.\n     *\n     * Throws a \"SyntaxError\" DOMException if one of the arguments is the empty string.\n     *\n     * Throws an \"InvalidCharacterError\" DOMException if one of the arguments contains any ASCII whitespace.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/remove)\n     */\n    remove(...tokens: string[]): void;\n    /**\n     * Replaces token with newToken.\n     *\n     * Returns true if token was replaced with newToken, and false otherwise.\n     *\n     * Throws a \"SyntaxError\" DOMException if one of the arguments is the empty string.\n     *\n     * Throws an \"InvalidCharacterError\" DOMException if one of the arguments contains any ASCII whitespace.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/replace)\n     */\n    replace(token: string, newToken: string): boolean;\n    /**\n     * Returns true if token is in the associated attribute's supported tokens. Returns false otherwise.\n     *\n     * Throws a TypeError if the associated attribute has no supported tokens defined.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/supports)\n     */\n    supports(token: string): boolean;\n    /**\n     * If force is not given, \"toggles\" token, removing it if it's present and adding it if it's not present. If force is true, adds token (same as add()). If force is false, removes token (same as remove()).\n     *\n     * Returns true if token is now present, and false otherwise.\n     *\n     * Throws a \"SyntaxError\" DOMException if token is empty.\n     *\n     * Throws an \"InvalidCharacterError\" DOMException if token contains any spaces.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DOMTokenList/toggle)\n     */\n    toggle(token: string, force?: boolean): boolean;\n    forEach(callbackfn: (value: string, key: number, parent: DOMTokenList) => void, thisArg?: any): void;\n    [index: number]: string;\n}\n\ndeclare var DOMTokenList: {\n    prototype: DOMTokenList;\n    new(): DOMTokenList;\n};\n\n/**\n * Used to hold the data that is being dragged during a drag and drop operation. It may hold one or more data items, each of one or more data types. For more information about drag and drop, see HTML Drag and Drop API.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer)\n */\ninterface DataTransfer {\n    /**\n     * Returns the kind of operation that is currently selected. If the kind of operation isn't one of those that is allowed by the effectAllowed attribute, then the operation will fail.\n     *\n     * Can be set, to change the selected operation.\n     *\n     * The possible values are \"none\", \"copy\", \"link\", and \"move\".\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/dropEffect)\n     */\n    dropEffect: \"none\" | \"copy\" | \"link\" | \"move\";\n    /**\n     * Returns the kinds of operations that are to be allowed.\n     *\n     * Can be set (during the dragstart event), to change the allowed operations.\n     *\n     * The possible values are \"none\", \"copy\", \"copyLink\", \"copyMove\", \"link\", \"linkMove\", \"move\", \"all\", and \"uninitialized\",\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/effectAllowed)\n     */\n    effectAllowed: \"none\" | \"copy\" | \"copyLink\" | \"copyMove\" | \"link\" | \"linkMove\" | \"move\" | \"all\" | \"uninitialized\";\n    /**\n     * Returns a FileList of the files being dragged, if any.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/files)\n     */\n    readonly files: FileList;\n    /**\n     * Returns a DataTransferItemList object, with the drag data.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/items)\n     */\n    readonly items: DataTransferItemList;\n    /**\n     * Returns a frozen array listing the formats that were set in the dragstart event. In addition, if any files are being dragged, then one of the types will be the string \"Files\".\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/types)\n     */\n    readonly types: ReadonlyArray<string>;\n    /**\n     * Removes the data of the specified formats. Removes all data if the argument is omitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/clearData)\n     */\n    clearData(format?: string): void;\n    /**\n     * Returns the specified data. If there is no such data, returns the empty string.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/getData)\n     */\n    getData(format: string): string;\n    /**\n     * Adds the specified data.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/setData)\n     */\n    setData(format: string, data: string): void;\n    /**\n     * Uses the given element to update the drag feedback, replacing any previously specified feedback.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransfer/setDragImage)\n     */\n    setDragImage(image: Element, x: number, y: number): void;\n}\n\ndeclare var DataTransfer: {\n    prototype: DataTransfer;\n    new(): DataTransfer;\n};\n\n/**\n * One drag data item. During a drag operation, each drag event has a dataTransfer property which contains a list of drag data items. Each item in the list is a DataTransferItem object.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItem)\n */\ninterface DataTransferItem {\n    /**\n     * Returns the drag data item kind, one of: \"string\", \"file\".\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItem/kind)\n     */\n    readonly kind: string;\n    /**\n     * Returns the drag data item type string.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItem/type)\n     */\n    readonly type: string;\n    /**\n     * Returns a File object, if the drag data item kind is File.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItem/getAsFile)\n     */\n    getAsFile(): File | null;\n    /**\n     * Invokes the callback with the string data as the argument, if the drag data item kind is text.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItem/getAsString)\n     */\n    getAsString(callback: FunctionStringCallback | null): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItem/webkitGetAsEntry) */\n    webkitGetAsEntry(): FileSystemEntry | null;\n}\n\ndeclare var DataTransferItem: {\n    prototype: DataTransferItem;\n    new(): DataTransferItem;\n};\n\n/**\n * A list of DataTransferItem objects representing items being dragged. During a drag operation, each DragEvent has a dataTransfer property and that property is a DataTransferItemList.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItemList)\n */\ninterface DataTransferItemList {\n    /**\n     * Returns the number of items in the drag data store.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItemList/length)\n     */\n    readonly length: number;\n    /**\n     * Adds a new entry for the given data to the drag data store. If the data is plain text then a type string has to be provided also.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItemList/add)\n     */\n    add(data: string, type: string): DataTransferItem | null;\n    add(data: File): DataTransferItem | null;\n    /**\n     * Removes all the entries in the drag data store.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItemList/clear)\n     */\n    clear(): void;\n    /**\n     * Removes the indexth entry in the drag data store.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DataTransferItemList/remove)\n     */\n    remove(index: number): void;\n    [index: number]: DataTransferItem;\n}\n\ndeclare var DataTransferItemList: {\n    prototype: DataTransferItemList;\n    new(): DataTransferItemList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DecompressionStream) */\ninterface DecompressionStream extends GenericTransformStream {\n}\n\ndeclare var DecompressionStream: {\n    prototype: DecompressionStream;\n    new(format: CompressionFormat): DecompressionStream;\n};\n\n/**\n * A delay-line; an AudioNode audio-processing module that causes a delay between the arrival of an input data and its propagation to the output.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DelayNode)\n */\ninterface DelayNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DelayNode/delayTime) */\n    readonly delayTime: AudioParam;\n}\n\ndeclare var DelayNode: {\n    prototype: DelayNode;\n    new(context: BaseAudioContext, options?: DelayOptions): DelayNode;\n};\n\n/**\n * The DeviceMotionEvent provides web developers with information about the speed of changes for the device's position and orientation.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEvent)\n */\ninterface DeviceMotionEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEvent/acceleration) */\n    readonly acceleration: DeviceMotionEventAcceleration | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEvent/accelerationIncludingGravity) */\n    readonly accelerationIncludingGravity: DeviceMotionEventAcceleration | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEvent/interval) */\n    readonly interval: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEvent/rotationRate) */\n    readonly rotationRate: DeviceMotionEventRotationRate | null;\n}\n\ndeclare var DeviceMotionEvent: {\n    prototype: DeviceMotionEvent;\n    new(type: string, eventInitDict?: DeviceMotionEventInit): DeviceMotionEvent;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventAcceleration)\n */\ninterface DeviceMotionEventAcceleration {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventAcceleration/x) */\n    readonly x: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventAcceleration/y) */\n    readonly y: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventAcceleration/z) */\n    readonly z: number | null;\n}\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventRotationRate)\n */\ninterface DeviceMotionEventRotationRate {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventRotationRate/alpha) */\n    readonly alpha: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventRotationRate/beta) */\n    readonly beta: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceMotionEventRotationRate/gamma) */\n    readonly gamma: number | null;\n}\n\n/**\n * The DeviceOrientationEvent provides web developers with information from the physical orientation of the device running the web page.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceOrientationEvent)\n */\ninterface DeviceOrientationEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceOrientationEvent/absolute) */\n    readonly absolute: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceOrientationEvent/alpha) */\n    readonly alpha: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceOrientationEvent/beta) */\n    readonly beta: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DeviceOrientationEvent/gamma) */\n    readonly gamma: number | null;\n}\n\ndeclare var DeviceOrientationEvent: {\n    prototype: DeviceOrientationEvent;\n    new(type: string, eventInitDict?: DeviceOrientationEventInit): DeviceOrientationEvent;\n};\n\ninterface DocumentEventMap extends GlobalEventHandlersEventMap {\n    \"DOMContentLoaded\": Event;\n    \"fullscreenchange\": Event;\n    \"fullscreenerror\": Event;\n    \"pointerlockchange\": Event;\n    \"pointerlockerror\": Event;\n    \"readystatechange\": Event;\n    \"visibilitychange\": Event;\n}\n\n/**\n * Any web page loaded in the browser and serves as an entry point into the web page's content, which is the DOM tree.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document)\n */\ninterface Document extends Node, DocumentOrShadowRoot, FontFaceSource, GlobalEventHandlers, NonElementParentNode, ParentNode, XPathEvaluatorBase {\n    /**\n     * Sets or gets the URL for the current document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/URL)\n     */\n    readonly URL: string;\n    /**\n     * Sets or gets the color of all active links in the document.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/alinkColor)\n     */\n    alinkColor: string;\n    /**\n     * Returns a reference to the collection of elements contained by the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/all)\n     */\n    readonly all: HTMLAllCollection;\n    /**\n     * Retrieves a collection of all a objects that have a name and/or id property. Objects in this collection are in HTML source order.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/anchors)\n     */\n    readonly anchors: HTMLCollectionOf<HTMLAnchorElement>;\n    /**\n     * Retrieves a collection of all applet objects in the document.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/applets)\n     */\n    readonly applets: HTMLCollection;\n    /**\n     * Deprecated. Sets or retrieves a value that indicates the background color behind the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/bgColor)\n     */\n    bgColor: string;\n    /**\n     * Specifies the beginning and end of the document body.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/body)\n     */\n    body: HTMLElement;\n    /**\n     * Returns document's encoding.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/characterSet)\n     */\n    readonly characterSet: string;\n    /**\n     * Gets or sets the character set used to encode the object.\n     * @deprecated This is a legacy alias of `characterSet`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/characterSet)\n     */\n    readonly charset: string;\n    /**\n     * Gets a value that indicates whether standards-compliant mode is switched on for the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/compatMode)\n     */\n    readonly compatMode: string;\n    /**\n     * Returns document's content type.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/contentType)\n     */\n    readonly contentType: string;\n    /**\n     * Returns the HTTP cookies that apply to the Document. If there are no cookies or cookies can't be applied to this resource, the empty string will be returned.\n     *\n     * Can be set, to add a new cookie to the element's set of HTTP cookies.\n     *\n     * If the contents are sandboxed into a unique origin (e.g. in an iframe with the sandbox attribute), a \"SecurityError\" DOMException will be thrown on getting and setting.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/cookie)\n     */\n    cookie: string;\n    /**\n     * Returns the script element, or the SVG script element, that is currently executing, as long as the element represents a classic script. In the case of reentrant script execution, returns the one that most recently started executing amongst those that have not yet finished executing.\n     *\n     * Returns null if the Document is not currently executing a script or SVG script element (e.g., because the running script is an event handler, or a timeout), or if the currently executing script or SVG script element represents a module script.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/currentScript)\n     */\n    readonly currentScript: HTMLOrSVGScriptElement | null;\n    /**\n     * Returns the Window object of the active document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/defaultView)\n     */\n    readonly defaultView: (WindowProxy & typeof globalThis) | null;\n    /**\n     * Sets or gets a value that indicates whether the document can be edited.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/designMode)\n     */\n    designMode: string;\n    /**\n     * Sets or retrieves a value that indicates the reading order of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/dir)\n     */\n    dir: string;\n    /**\n     * Gets an object representing the document type declaration associated with the current document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/doctype)\n     */\n    readonly doctype: DocumentType | null;\n    /**\n     * Gets a reference to the root node of the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/documentElement)\n     */\n    readonly documentElement: HTMLElement;\n    /**\n     * Returns document's URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/documentURI)\n     */\n    readonly documentURI: string;\n    /**\n     * Sets or gets the security domain of the document.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/domain)\n     */\n    domain: string;\n    /**\n     * Retrieves a collection of all embed objects in the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/embeds)\n     */\n    readonly embeds: HTMLCollectionOf<HTMLEmbedElement>;\n    /**\n     * Sets or gets the foreground (text) color of the document.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fgColor)\n     */\n    fgColor: string;\n    /**\n     * Retrieves a collection, in source order, of all form objects in the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/forms)\n     */\n    readonly forms: HTMLCollectionOf<HTMLFormElement>;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fullscreen)\n     */\n    readonly fullscreen: boolean;\n    /**\n     * Returns true if document has the ability to display elements fullscreen and fullscreen is supported, or false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fullscreenEnabled)\n     */\n    readonly fullscreenEnabled: boolean;\n    /**\n     * Returns the head element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/head)\n     */\n    readonly head: HTMLHeadElement;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/hidden) */\n    readonly hidden: boolean;\n    /**\n     * Retrieves a collection, in source order, of img objects in the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/images)\n     */\n    readonly images: HTMLCollectionOf<HTMLImageElement>;\n    /**\n     * Gets the implementation object of the current document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/implementation)\n     */\n    readonly implementation: DOMImplementation;\n    /**\n     * Returns the character encoding used to create the webpage that is loaded into the document object.\n     * @deprecated This is a legacy alias of `characterSet`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/characterSet)\n     */\n    readonly inputEncoding: string;\n    /**\n     * Gets the date that the page was last modified, if the page supplies one.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/lastModified)\n     */\n    readonly lastModified: string;\n    /**\n     * Sets or gets the color of the document links.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/linkColor)\n     */\n    linkColor: string;\n    /**\n     * Retrieves a collection of all a objects that specify the href property and all area objects in the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/links)\n     */\n    readonly links: HTMLCollectionOf<HTMLAnchorElement | HTMLAreaElement>;\n    /**\n     * Contains information about the current URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/location)\n     */\n    get location(): Location;\n    set location(href: string | Location);\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fullscreenchange_event) */\n    onfullscreenchange: ((this: Document, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fullscreenerror_event) */\n    onfullscreenerror: ((this: Document, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/pointerlockchange_event) */\n    onpointerlockchange: ((this: Document, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/pointerlockerror_event) */\n    onpointerlockerror: ((this: Document, ev: Event) => any) | null;\n    /**\n     * Fires when the state of the object has changed.\n     * @param ev The event\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/readystatechange_event)\n     */\n    onreadystatechange: ((this: Document, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/visibilitychange_event) */\n    onvisibilitychange: ((this: Document, ev: Event) => any) | null;\n    readonly ownerDocument: null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/pictureInPictureEnabled) */\n    readonly pictureInPictureEnabled: boolean;\n    /**\n     * Return an HTMLCollection of the embed elements in the Document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/plugins)\n     */\n    readonly plugins: HTMLCollectionOf<HTMLEmbedElement>;\n    /**\n     * Retrieves a value that indicates the current state of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/readyState)\n     */\n    readonly readyState: DocumentReadyState;\n    /**\n     * Gets the URL of the location that referred the user to the current page.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/referrer)\n     */\n    readonly referrer: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/rootElement)\n     */\n    readonly rootElement: SVGSVGElement | null;\n    /**\n     * Retrieves a collection of all script objects in the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/scripts)\n     */\n    readonly scripts: HTMLCollectionOf<HTMLScriptElement>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/scrollingElement) */\n    readonly scrollingElement: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/timeline) */\n    readonly timeline: DocumentTimeline;\n    /**\n     * Contains the title of the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/title)\n     */\n    title: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/visibilityState) */\n    readonly visibilityState: DocumentVisibilityState;\n    /**\n     * Sets or gets the color of the links that the user has visited.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/vlinkColor)\n     */\n    vlinkColor: string;\n    /**\n     * Moves node from another document and returns it.\n     *\n     * If node is a document, throws a \"NotSupportedError\" DOMException or, if node is a shadow root, throws a \"HierarchyRequestError\" DOMException.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/adoptNode)\n     */\n    adoptNode<T extends Node>(node: T): T;\n    /** @deprecated */\n    captureEvents(): void;\n    /** @deprecated */\n    caretRangeFromPoint(x: number, y: number): Range | null;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/clear)\n     */\n    clear(): void;\n    /**\n     * Closes an output stream and forces the sent data to display.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/close)\n     */\n    close(): void;\n    /**\n     * Creates an attribute object with a specified name.\n     * @param name String that sets the attribute object's name.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createAttribute)\n     */\n    createAttribute(localName: string): Attr;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createAttributeNS) */\n    createAttributeNS(namespace: string | null, qualifiedName: string): Attr;\n    /**\n     * Returns a CDATASection node whose data is data.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createCDATASection)\n     */\n    createCDATASection(data: string): CDATASection;\n    /**\n     * Creates a comment object with the specified data.\n     * @param data Sets the comment object's data.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createComment)\n     */\n    createComment(data: string): Comment;\n    /**\n     * Creates a new document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createDocumentFragment)\n     */\n    createDocumentFragment(): DocumentFragment;\n    /**\n     * Creates an instance of the element for the specified tag.\n     * @param tagName The name of an element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createElement)\n     */\n    createElement<K extends keyof HTMLElementTagNameMap>(tagName: K, options?: ElementCreationOptions): HTMLElementTagNameMap[K];\n    /** @deprecated */\n    createElement<K extends keyof HTMLElementDeprecatedTagNameMap>(tagName: K, options?: ElementCreationOptions): HTMLElementDeprecatedTagNameMap[K];\n    createElement(tagName: string, options?: ElementCreationOptions): HTMLElement;\n    /**\n     * Returns an element with namespace namespace. Its namespace prefix will be everything before \":\" (U+003E) in qualifiedName or null. Its local name will be everything after \":\" (U+003E) in qualifiedName or qualifiedName.\n     *\n     * If localName does not match the Name production an \"InvalidCharacterError\" DOMException will be thrown.\n     *\n     * If one of the following conditions is true a \"NamespaceError\" DOMException will be thrown:\n     *\n     * localName does not match the QName production.\n     * Namespace prefix is not null and namespace is the empty string.\n     * Namespace prefix is \"xml\" and namespace is not the XML namespace.\n     * qualifiedName or namespace prefix is \"xmlns\" and namespace is not the XMLNS namespace.\n     * namespace is the XMLNS namespace and neither qualifiedName nor namespace prefix is \"xmlns\".\n     *\n     * When supplied, options's is can be used to create a customized built-in element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createElementNS)\n     */\n    createElementNS(namespaceURI: \"http://www.w3.org/1999/xhtml\", qualifiedName: string): HTMLElement;\n    createElementNS<K extends keyof SVGElementTagNameMap>(namespaceURI: \"http://www.w3.org/2000/svg\", qualifiedName: K): SVGElementTagNameMap[K];\n    createElementNS(namespaceURI: \"http://www.w3.org/2000/svg\", qualifiedName: string): SVGElement;\n    createElementNS<K extends keyof MathMLElementTagNameMap>(namespaceURI: \"http://www.w3.org/1998/Math/MathML\", qualifiedName: K): MathMLElementTagNameMap[K];\n    createElementNS(namespaceURI: \"http://www.w3.org/1998/Math/MathML\", qualifiedName: string): MathMLElement;\n    createElementNS(namespaceURI: string | null, qualifiedName: string, options?: ElementCreationOptions): Element;\n    createElementNS(namespace: string | null, qualifiedName: string, options?: string | ElementCreationOptions): Element;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createEvent) */\n    createEvent(eventInterface: \"AnimationEvent\"): AnimationEvent;\n    createEvent(eventInterface: \"AnimationPlaybackEvent\"): AnimationPlaybackEvent;\n    createEvent(eventInterface: \"AudioProcessingEvent\"): AudioProcessingEvent;\n    createEvent(eventInterface: \"BeforeUnloadEvent\"): BeforeUnloadEvent;\n    createEvent(eventInterface: \"BlobEvent\"): BlobEvent;\n    createEvent(eventInterface: \"ClipboardEvent\"): ClipboardEvent;\n    createEvent(eventInterface: \"CloseEvent\"): CloseEvent;\n    createEvent(eventInterface: \"CompositionEvent\"): CompositionEvent;\n    createEvent(eventInterface: \"ContentVisibilityAutoStateChangeEvent\"): ContentVisibilityAutoStateChangeEvent;\n    createEvent(eventInterface: \"CustomEvent\"): CustomEvent;\n    createEvent(eventInterface: \"DeviceMotionEvent\"): DeviceMotionEvent;\n    createEvent(eventInterface: \"DeviceOrientationEvent\"): DeviceOrientationEvent;\n    createEvent(eventInterface: \"DragEvent\"): DragEvent;\n    createEvent(eventInterface: \"ErrorEvent\"): ErrorEvent;\n    createEvent(eventInterface: \"Event\"): Event;\n    createEvent(eventInterface: \"Events\"): Event;\n    createEvent(eventInterface: \"FocusEvent\"): FocusEvent;\n    createEvent(eventInterface: \"FontFaceSetLoadEvent\"): FontFaceSetLoadEvent;\n    createEvent(eventInterface: \"FormDataEvent\"): FormDataEvent;\n    createEvent(eventInterface: \"GamepadEvent\"): GamepadEvent;\n    createEvent(eventInterface: \"HashChangeEvent\"): HashChangeEvent;\n    createEvent(eventInterface: \"IDBVersionChangeEvent\"): IDBVersionChangeEvent;\n    createEvent(eventInterface: \"InputEvent\"): InputEvent;\n    createEvent(eventInterface: \"KeyboardEvent\"): KeyboardEvent;\n    createEvent(eventInterface: \"MIDIConnectionEvent\"): MIDIConnectionEvent;\n    createEvent(eventInterface: \"MIDIMessageEvent\"): MIDIMessageEvent;\n    createEvent(eventInterface: \"MediaEncryptedEvent\"): MediaEncryptedEvent;\n    createEvent(eventInterface: \"MediaKeyMessageEvent\"): MediaKeyMessageEvent;\n    createEvent(eventInterface: \"MediaQueryListEvent\"): MediaQueryListEvent;\n    createEvent(eventInterface: \"MediaStreamTrackEvent\"): MediaStreamTrackEvent;\n    createEvent(eventInterface: \"MessageEvent\"): MessageEvent;\n    createEvent(eventInterface: \"MouseEvent\"): MouseEvent;\n    createEvent(eventInterface: \"MouseEvents\"): MouseEvent;\n    createEvent(eventInterface: \"MutationEvent\"): MutationEvent;\n    createEvent(eventInterface: \"MutationEvents\"): MutationEvent;\n    createEvent(eventInterface: \"OfflineAudioCompletionEvent\"): OfflineAudioCompletionEvent;\n    createEvent(eventInterface: \"PageTransitionEvent\"): PageTransitionEvent;\n    createEvent(eventInterface: \"PaymentMethodChangeEvent\"): PaymentMethodChangeEvent;\n    createEvent(eventInterface: \"PaymentRequestUpdateEvent\"): PaymentRequestUpdateEvent;\n    createEvent(eventInterface: \"PictureInPictureEvent\"): PictureInPictureEvent;\n    createEvent(eventInterface: \"PointerEvent\"): PointerEvent;\n    createEvent(eventInterface: \"PopStateEvent\"): PopStateEvent;\n    createEvent(eventInterface: \"ProgressEvent\"): ProgressEvent;\n    createEvent(eventInterface: \"PromiseRejectionEvent\"): PromiseRejectionEvent;\n    createEvent(eventInterface: \"RTCDTMFToneChangeEvent\"): RTCDTMFToneChangeEvent;\n    createEvent(eventInterface: \"RTCDataChannelEvent\"): RTCDataChannelEvent;\n    createEvent(eventInterface: \"RTCErrorEvent\"): RTCErrorEvent;\n    createEvent(eventInterface: \"RTCPeerConnectionIceErrorEvent\"): RTCPeerConnectionIceErrorEvent;\n    createEvent(eventInterface: \"RTCPeerConnectionIceEvent\"): RTCPeerConnectionIceEvent;\n    createEvent(eventInterface: \"RTCTrackEvent\"): RTCTrackEvent;\n    createEvent(eventInterface: \"SecurityPolicyViolationEvent\"): SecurityPolicyViolationEvent;\n    createEvent(eventInterface: \"SpeechSynthesisErrorEvent\"): SpeechSynthesisErrorEvent;\n    createEvent(eventInterface: \"SpeechSynthesisEvent\"): SpeechSynthesisEvent;\n    createEvent(eventInterface: \"StorageEvent\"): StorageEvent;\n    createEvent(eventInterface: \"SubmitEvent\"): SubmitEvent;\n    createEvent(eventInterface: \"TextEvent\"): TextEvent;\n    createEvent(eventInterface: \"ToggleEvent\"): ToggleEvent;\n    createEvent(eventInterface: \"TouchEvent\"): TouchEvent;\n    createEvent(eventInterface: \"TrackEvent\"): TrackEvent;\n    createEvent(eventInterface: \"TransitionEvent\"): TransitionEvent;\n    createEvent(eventInterface: \"UIEvent\"): UIEvent;\n    createEvent(eventInterface: \"UIEvents\"): UIEvent;\n    createEvent(eventInterface: \"WebGLContextEvent\"): WebGLContextEvent;\n    createEvent(eventInterface: \"WheelEvent\"): WheelEvent;\n    createEvent(eventInterface: string): Event;\n    /**\n     * Creates a NodeIterator object that you can use to traverse filtered lists of nodes or elements in a document.\n     * @param root The root element or node to start traversing on.\n     * @param whatToShow The type of nodes or elements to appear in the node list\n     * @param filter A custom NodeFilter function to use. For more information, see filter. Use null for no filter.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createNodeIterator)\n     */\n    createNodeIterator(root: Node, whatToShow?: number, filter?: NodeFilter | null): NodeIterator;\n    /**\n     * Returns a ProcessingInstruction node whose target is target and data is data. If target does not match the Name production an \"InvalidCharacterError\" DOMException will be thrown. If data contains \"?>\" an \"InvalidCharacterError\" DOMException will be thrown.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createProcessingInstruction)\n     */\n    createProcessingInstruction(target: string, data: string): ProcessingInstruction;\n    /**\n     *  Returns an empty range object that has both of its boundary points positioned at the beginning of the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createRange)\n     */\n    createRange(): Range;\n    /**\n     * Creates a text string from the specified value.\n     * @param data String that specifies the nodeValue property of the text node.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createTextNode)\n     */\n    createTextNode(data: string): Text;\n    /**\n     * Creates a TreeWalker object that you can use to traverse filtered lists of nodes or elements in a document.\n     * @param root The root element or node to start traversing on.\n     * @param whatToShow The type of nodes or elements to appear in the node list. For more information, see whatToShow.\n     * @param filter A custom NodeFilter function to use.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/createTreeWalker)\n     */\n    createTreeWalker(root: Node, whatToShow?: number, filter?: NodeFilter | null): TreeWalker;\n    /**\n     * Executes a command on the current document, current selection, or the given range.\n     * @param commandId String that specifies the command to execute. This command can be any of the command identifiers that can be executed in script.\n     * @param showUI Display the user interface, defaults to false.\n     * @param value Value to assign.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/execCommand)\n     */\n    execCommand(commandId: string, showUI?: boolean, value?: string): boolean;\n    /**\n     * Stops document's fullscreen element from being displayed fullscreen and resolves promise when done.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/exitFullscreen)\n     */\n    exitFullscreen(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/exitPictureInPicture) */\n    exitPictureInPicture(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/exitPointerLock) */\n    exitPointerLock(): void;\n    /**\n     * Returns a reference to the first object with the specified value of the ID attribute.\n     * @param elementId String that specifies the ID value.\n     */\n    getElementById(elementId: string): HTMLElement | null;\n    /**\n     * Returns a HTMLCollection of the elements in the object on which the method was invoked (a document or an element) that have all the classes given by classNames. The classNames argument is interpreted as a space-separated list of classes.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByClassName)\n     */\n    getElementsByClassName(classNames: string): HTMLCollectionOf<Element>;\n    /**\n     * Gets a collection of objects based on the value of the NAME or ID attribute.\n     * @param elementName Gets a collection of objects based on the value of the NAME or ID attribute.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByName)\n     */\n    getElementsByName(elementName: string): NodeListOf<HTMLElement>;\n    /**\n     * Retrieves a collection of objects based on the specified element name.\n     * @param name Specifies the name of an element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByTagName)\n     */\n    getElementsByTagName<K extends keyof HTMLElementTagNameMap>(qualifiedName: K): HTMLCollectionOf<HTMLElementTagNameMap[K]>;\n    getElementsByTagName<K extends keyof SVGElementTagNameMap>(qualifiedName: K): HTMLCollectionOf<SVGElementTagNameMap[K]>;\n    getElementsByTagName<K extends keyof MathMLElementTagNameMap>(qualifiedName: K): HTMLCollectionOf<MathMLElementTagNameMap[K]>;\n    /** @deprecated */\n    getElementsByTagName<K extends keyof HTMLElementDeprecatedTagNameMap>(qualifiedName: K): HTMLCollectionOf<HTMLElementDeprecatedTagNameMap[K]>;\n    getElementsByTagName(qualifiedName: string): HTMLCollectionOf<Element>;\n    /**\n     * If namespace and localName are \"*\" returns a HTMLCollection of all descendant elements.\n     *\n     * If only namespace is \"*\" returns a HTMLCollection of all descendant elements whose local name is localName.\n     *\n     * If only localName is \"*\" returns a HTMLCollection of all descendant elements whose namespace is namespace.\n     *\n     * Otherwise, returns a HTMLCollection of all descendant elements whose namespace is namespace and local name is localName.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementsByTagNameNS)\n     */\n    getElementsByTagNameNS(namespaceURI: \"http://www.w3.org/1999/xhtml\", localName: string): HTMLCollectionOf<HTMLElement>;\n    getElementsByTagNameNS(namespaceURI: \"http://www.w3.org/2000/svg\", localName: string): HTMLCollectionOf<SVGElement>;\n    getElementsByTagNameNS(namespaceURI: \"http://www.w3.org/1998/Math/MathML\", localName: string): HTMLCollectionOf<MathMLElement>;\n    getElementsByTagNameNS(namespace: string | null, localName: string): HTMLCollectionOf<Element>;\n    /**\n     * Returns an object representing the current selection of the document that is loaded into the object displaying a webpage.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getSelection)\n     */\n    getSelection(): Selection | null;\n    /**\n     * Gets a value indicating whether the object currently has focus.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/hasFocus)\n     */\n    hasFocus(): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/hasStorageAccess) */\n    hasStorageAccess(): Promise<boolean>;\n    /**\n     * Returns a copy of node. If deep is true, the copy also includes the node's descendants.\n     *\n     * If node is a document or a shadow root, throws a \"NotSupportedError\" DOMException.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/importNode)\n     */\n    importNode<T extends Node>(node: T, deep?: boolean): T;\n    /**\n     * Opens a new window and loads a document specified by a given URL. Also, opens a new window that uses the url parameter and the name parameter to collect the output of the write method and the writeln method.\n     * @param url Specifies a MIME type for the document.\n     * @param name Specifies the name of the window. This name is used as the value for the TARGET attribute on a form or an anchor element.\n     * @param features Contains a list of items separated by commas. Each item consists of an option and a value, separated by an equals sign (for example, \"fullscreen=yes, toolbar=yes\"). The following values are supported.\n     * @param replace Specifies whether the existing entry for the document is replaced in the history list.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/open)\n     */\n    open(unused1?: string, unused2?: string): Document;\n    open(url: string | URL, name: string, features: string): WindowProxy | null;\n    /**\n     * Returns a Boolean value that indicates whether a specified command can be successfully executed using execCommand, given the current state of the document.\n     * @param commandId Specifies a command identifier.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/queryCommandEnabled)\n     */\n    queryCommandEnabled(commandId: string): boolean;\n    /**\n     * Returns a Boolean value that indicates whether the specified command is in the indeterminate state.\n     * @param commandId String that specifies a command identifier.\n     * @deprecated\n     */\n    queryCommandIndeterm(commandId: string): boolean;\n    /**\n     * Returns a Boolean value that indicates the current state of the command.\n     * @param commandId String that specifies a command identifier.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/queryCommandState)\n     */\n    queryCommandState(commandId: string): boolean;\n    /**\n     * Returns a Boolean value that indicates whether the current command is supported on the current range.\n     * @param commandId Specifies a command identifier.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/queryCommandSupported)\n     */\n    queryCommandSupported(commandId: string): boolean;\n    /**\n     * Returns the current value of the document, range, or current selection for the given command.\n     * @param commandId String that specifies a command identifier.\n     * @deprecated\n     */\n    queryCommandValue(commandId: string): string;\n    /** @deprecated */\n    releaseEvents(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/requestStorageAccess) */\n    requestStorageAccess(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/startViewTransition) */\n    startViewTransition(callbackOptions?: UpdateCallback): ViewTransition;\n    /**\n     * Writes one or more HTML expressions to a document in the specified window.\n     * @param content Specifies the text and HTML tags to write.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/write)\n     */\n    write(...text: string[]): void;\n    /**\n     * Writes one or more HTML expressions, followed by a carriage return, to a document in the specified window.\n     * @param content The text and HTML tags to write.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/writeln)\n     */\n    writeln(...text: string[]): void;\n    addEventListener<K extends keyof DocumentEventMap>(type: K, listener: (this: Document, ev: DocumentEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof DocumentEventMap>(type: K, listener: (this: Document, ev: DocumentEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var Document: {\n    prototype: Document;\n    new(): Document;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/parseHTMLUnsafe_static) */\n    parseHTMLUnsafe(html: string): Document;\n};\n\n/**\n * A minimal document object that has no parent. It is used as a lightweight version of Document that stores a segment of a document structure comprised of nodes just like a standard document. The key difference is that because the document fragment isn't part of the active document tree structure, changes made to the fragment don't affect the document, cause reflow, or incur any performance impact that can occur when changes are made.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DocumentFragment)\n */\ninterface DocumentFragment extends Node, NonElementParentNode, ParentNode {\n    readonly ownerDocument: Document;\n    getElementById(elementId: string): HTMLElement | null;\n}\n\ndeclare var DocumentFragment: {\n    prototype: DocumentFragment;\n    new(): DocumentFragment;\n};\n\ninterface DocumentOrShadowRoot {\n    /**\n     * Returns the deepest element in the document through which or to which key events are being routed. This is, roughly speaking, the focused element in the document.\n     *\n     * For the purposes of this API, when a child browsing context is focused, its container is focused in the parent browsing context. For example, if the user moves the focus to a text control in an iframe, the iframe is the element returned by the activeElement API in the iframe's node document.\n     *\n     * Similarly, when the focused element is in a different node tree than documentOrShadowRoot, the element returned will be the host that's located in the same node tree as documentOrShadowRoot if documentOrShadowRoot is a shadow-including inclusive ancestor of the focused element, and null if not.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/activeElement)\n     */\n    readonly activeElement: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/adoptedStyleSheets) */\n    adoptedStyleSheets: CSSStyleSheet[];\n    /**\n     * Returns document's fullscreen element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fullscreenElement)\n     */\n    readonly fullscreenElement: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/pictureInPictureElement) */\n    readonly pictureInPictureElement: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/pointerLockElement) */\n    readonly pointerLockElement: Element | null;\n    /**\n     * Retrieves a collection of styleSheet objects representing the style sheets that correspond to each instance of a link or style object in the document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/styleSheets)\n     */\n    readonly styleSheets: StyleSheetList;\n    /**\n     * Returns the element for the specified x coordinate and the specified y coordinate.\n     * @param x The x-offset\n     * @param y The y-offset\n     */\n    elementFromPoint(x: number, y: number): Element | null;\n    elementsFromPoint(x: number, y: number): Element[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getAnimations) */\n    getAnimations(): Animation[];\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DocumentTimeline) */\ninterface DocumentTimeline extends AnimationTimeline {\n}\n\ndeclare var DocumentTimeline: {\n    prototype: DocumentTimeline;\n    new(options?: DocumentTimelineOptions): DocumentTimeline;\n};\n\n/**\n * A Node containing a doctype.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DocumentType)\n */\ninterface DocumentType extends Node, ChildNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DocumentType/name) */\n    readonly name: string;\n    readonly ownerDocument: Document;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DocumentType/publicId) */\n    readonly publicId: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DocumentType/systemId) */\n    readonly systemId: string;\n}\n\ndeclare var DocumentType: {\n    prototype: DocumentType;\n    new(): DocumentType;\n};\n\n/**\n * A DOM event that represents a drag and drop interaction. The user initiates a drag by placing a pointer device (such as a mouse) on the touch surface and then dragging the pointer to a new location (such as another DOM element). Applications are free to interpret a drag and drop interaction in an application-specific way.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DragEvent)\n */\ninterface DragEvent extends MouseEvent {\n    /**\n     * Returns the DataTransfer object for the event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DragEvent/dataTransfer)\n     */\n    readonly dataTransfer: DataTransfer | null;\n}\n\ndeclare var DragEvent: {\n    prototype: DragEvent;\n    new(type: string, eventInitDict?: DragEventInit): DragEvent;\n};\n\n/**\n * Inherits properties from its parent, AudioNode.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode)\n */\ninterface DynamicsCompressorNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode/attack) */\n    readonly attack: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode/knee) */\n    readonly knee: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode/ratio) */\n    readonly ratio: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode/reduction) */\n    readonly reduction: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode/release) */\n    readonly release: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/DynamicsCompressorNode/threshold) */\n    readonly threshold: AudioParam;\n}\n\ndeclare var DynamicsCompressorNode: {\n    prototype: DynamicsCompressorNode;\n    new(context: BaseAudioContext, options?: DynamicsCompressorOptions): DynamicsCompressorNode;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_blend_minmax) */\ninterface EXT_blend_minmax {\n    readonly MIN_EXT: 0x8007;\n    readonly MAX_EXT: 0x8008;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_color_buffer_float) */\ninterface EXT_color_buffer_float {\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_color_buffer_half_float) */\ninterface EXT_color_buffer_half_float {\n    readonly RGBA16F_EXT: 0x881A;\n    readonly RGB16F_EXT: 0x881B;\n    readonly FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE_EXT: 0x8211;\n    readonly UNSIGNED_NORMALIZED_EXT: 0x8C17;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_float_blend) */\ninterface EXT_float_blend {\n}\n\n/**\n * The EXT_frag_depth extension is part of the WebGL API and enables to set a depth value of a fragment from within the fragment shader.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_frag_depth)\n */\ninterface EXT_frag_depth {\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_sRGB) */\ninterface EXT_sRGB {\n    readonly SRGB_EXT: 0x8C40;\n    readonly SRGB_ALPHA_EXT: 0x8C42;\n    readonly SRGB8_ALPHA8_EXT: 0x8C43;\n    readonly FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING_EXT: 0x8210;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_shader_texture_lod) */\ninterface EXT_shader_texture_lod {\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_texture_compression_bptc) */\ninterface EXT_texture_compression_bptc {\n    readonly COMPRESSED_RGBA_BPTC_UNORM_EXT: 0x8E8C;\n    readonly COMPRESSED_SRGB_ALPHA_BPTC_UNORM_EXT: 0x8E8D;\n    readonly COMPRESSED_RGB_BPTC_SIGNED_FLOAT_EXT: 0x8E8E;\n    readonly COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT_EXT: 0x8E8F;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_texture_compression_rgtc) */\ninterface EXT_texture_compression_rgtc {\n    readonly COMPRESSED_RED_RGTC1_EXT: 0x8DBB;\n    readonly COMPRESSED_SIGNED_RED_RGTC1_EXT: 0x8DBC;\n    readonly COMPRESSED_RED_GREEN_RGTC2_EXT: 0x8DBD;\n    readonly COMPRESSED_SIGNED_RED_GREEN_RGTC2_EXT: 0x8DBE;\n}\n\n/**\n * The EXT_texture_filter_anisotropic extension is part of the WebGL API and exposes two constants for anisotropic filtering (AF).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_texture_filter_anisotropic)\n */\ninterface EXT_texture_filter_anisotropic {\n    readonly TEXTURE_MAX_ANISOTROPY_EXT: 0x84FE;\n    readonly MAX_TEXTURE_MAX_ANISOTROPY_EXT: 0x84FF;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EXT_texture_norm16) */\ninterface EXT_texture_norm16 {\n    readonly R16_EXT: 0x822A;\n    readonly RG16_EXT: 0x822C;\n    readonly RGB16_EXT: 0x8054;\n    readonly RGBA16_EXT: 0x805B;\n    readonly R16_SNORM_EXT: 0x8F98;\n    readonly RG16_SNORM_EXT: 0x8F99;\n    readonly RGB16_SNORM_EXT: 0x8F9A;\n    readonly RGBA16_SNORM_EXT: 0x8F9B;\n}\n\ninterface ElementEventMap {\n    \"fullscreenchange\": Event;\n    \"fullscreenerror\": Event;\n}\n\n/**\n * Element is the most general base class from which all objects in a Document inherit. It only has methods and properties common to all kinds of elements. More specific classes inherit from Element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element)\n */\ninterface Element extends Node, ARIAMixin, Animatable, ChildNode, NonDocumentTypeChildNode, ParentNode, Slottable {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/attributes) */\n    readonly attributes: NamedNodeMap;\n    /**\n     * Allows for manipulation of element's class content attribute as a set of whitespace-separated tokens through a DOMTokenList object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/classList)\n     */\n    readonly classList: DOMTokenList;\n    /**\n     * Returns the value of element's class content attribute. Can be set to change it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/className)\n     */\n    className: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/clientHeight) */\n    readonly clientHeight: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/clientLeft) */\n    readonly clientLeft: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/clientTop) */\n    readonly clientTop: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/clientWidth) */\n    readonly clientWidth: number;\n    /**\n     * Returns the value of element's id content attribute. Can be set to change it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/id)\n     */\n    id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/innerHTML) */\n    innerHTML: string;\n    /**\n     * Returns the local name.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/localName)\n     */\n    readonly localName: string;\n    /**\n     * Returns the namespace.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/namespaceURI)\n     */\n    readonly namespaceURI: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/fullscreenchange_event) */\n    onfullscreenchange: ((this: Element, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/fullscreenerror_event) */\n    onfullscreenerror: ((this: Element, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/outerHTML) */\n    outerHTML: string;\n    readonly ownerDocument: Document;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/part) */\n    readonly part: DOMTokenList;\n    /**\n     * Returns the namespace prefix.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/prefix)\n     */\n    readonly prefix: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollHeight) */\n    readonly scrollHeight: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollLeft) */\n    scrollLeft: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollTop) */\n    scrollTop: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollWidth) */\n    readonly scrollWidth: number;\n    /**\n     * Returns element's shadow root, if any, and if shadow root's mode is \"open\", and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/shadowRoot)\n     */\n    readonly shadowRoot: ShadowRoot | null;\n    /**\n     * Returns the value of element's slot content attribute. Can be set to change it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/slot)\n     */\n    slot: string;\n    /**\n     * Returns the HTML-uppercased qualified name.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/tagName)\n     */\n    readonly tagName: string;\n    /**\n     * Creates a shadow root for element and returns it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/attachShadow)\n     */\n    attachShadow(init: ShadowRootInit): ShadowRoot;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/checkVisibility) */\n    checkVisibility(options?: CheckVisibilityOptions): boolean;\n    /**\n     * Returns the first (starting at element) inclusive ancestor that matches selectors, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/closest)\n     */\n    closest<K extends keyof HTMLElementTagNameMap>(selector: K): HTMLElementTagNameMap[K] | null;\n    closest<K extends keyof SVGElementTagNameMap>(selector: K): SVGElementTagNameMap[K] | null;\n    closest<K extends keyof MathMLElementTagNameMap>(selector: K): MathMLElementTagNameMap[K] | null;\n    closest<E extends Element = Element>(selectors: string): E | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/computedStyleMap) */\n    computedStyleMap(): StylePropertyMapReadOnly;\n    /**\n     * Returns element's first attribute whose qualified name is qualifiedName, and null if there is no such attribute otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttribute)\n     */\n    getAttribute(qualifiedName: string): string | null;\n    /**\n     * Returns element's attribute whose namespace is namespace and local name is localName, and null if there is no such attribute otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNS)\n     */\n    getAttributeNS(namespace: string | null, localName: string): string | null;\n    /**\n     * Returns the qualified names of all element's attributes. Can contain duplicates.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNames)\n     */\n    getAttributeNames(): string[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNode) */\n    getAttributeNode(qualifiedName: string): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getAttributeNodeNS) */\n    getAttributeNodeNS(namespace: string | null, localName: string): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getBoundingClientRect) */\n    getBoundingClientRect(): DOMRect;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getClientRects) */\n    getClientRects(): DOMRectList;\n    /**\n     * Returns a HTMLCollection of the elements in the object on which the method was invoked (a document or an element) that have all the classes given by classNames. The classNames argument is interpreted as a space-separated list of classes.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getElementsByClassName)\n     */\n    getElementsByClassName(classNames: string): HTMLCollectionOf<Element>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getElementsByTagName) */\n    getElementsByTagName<K extends keyof HTMLElementTagNameMap>(qualifiedName: K): HTMLCollectionOf<HTMLElementTagNameMap[K]>;\n    getElementsByTagName<K extends keyof SVGElementTagNameMap>(qualifiedName: K): HTMLCollectionOf<SVGElementTagNameMap[K]>;\n    getElementsByTagName<K extends keyof MathMLElementTagNameMap>(qualifiedName: K): HTMLCollectionOf<MathMLElementTagNameMap[K]>;\n    /** @deprecated */\n    getElementsByTagName<K extends keyof HTMLElementDeprecatedTagNameMap>(qualifiedName: K): HTMLCollectionOf<HTMLElementDeprecatedTagNameMap[K]>;\n    getElementsByTagName(qualifiedName: string): HTMLCollectionOf<Element>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getElementsByTagNameNS) */\n    getElementsByTagNameNS(namespaceURI: \"http://www.w3.org/1999/xhtml\", localName: string): HTMLCollectionOf<HTMLElement>;\n    getElementsByTagNameNS(namespaceURI: \"http://www.w3.org/2000/svg\", localName: string): HTMLCollectionOf<SVGElement>;\n    getElementsByTagNameNS(namespaceURI: \"http://www.w3.org/1998/Math/MathML\", localName: string): HTMLCollectionOf<MathMLElement>;\n    getElementsByTagNameNS(namespace: string | null, localName: string): HTMLCollectionOf<Element>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/getHTML) */\n    getHTML(options?: GetHTMLOptions): string;\n    /**\n     * Returns true if element has an attribute whose qualified name is qualifiedName, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/hasAttribute)\n     */\n    hasAttribute(qualifiedName: string): boolean;\n    /**\n     * Returns true if element has an attribute whose namespace is namespace and local name is localName.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/hasAttributeNS)\n     */\n    hasAttributeNS(namespace: string | null, localName: string): boolean;\n    /**\n     * Returns true if element has attributes, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/hasAttributes)\n     */\n    hasAttributes(): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/hasPointerCapture) */\n    hasPointerCapture(pointerId: number): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/insertAdjacentElement) */\n    insertAdjacentElement(where: InsertPosition, element: Element): Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/insertAdjacentHTML) */\n    insertAdjacentHTML(position: InsertPosition, string: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/insertAdjacentText) */\n    insertAdjacentText(where: InsertPosition, data: string): void;\n    /**\n     * Returns true if matching selectors against element's root yields element, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/matches)\n     */\n    matches(selectors: string): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/releasePointerCapture) */\n    releasePointerCapture(pointerId: number): void;\n    /**\n     * Removes element's first attribute whose qualified name is qualifiedName.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/removeAttribute)\n     */\n    removeAttribute(qualifiedName: string): void;\n    /**\n     * Removes element's attribute whose namespace is namespace and local name is localName.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/removeAttributeNS)\n     */\n    removeAttributeNS(namespace: string | null, localName: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/removeAttributeNode) */\n    removeAttributeNode(attr: Attr): Attr;\n    /**\n     * Displays element fullscreen and resolves promise when done.\n     *\n     * When supplied, options's navigationUI member indicates whether showing navigation UI while in fullscreen is preferred or not. If set to \"show\", navigation simplicity is preferred over screen space, and if set to \"hide\", more screen space is preferred. User agents are always free to honor user preference over the application's. The default value \"auto\" indicates no application preference.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/requestFullscreen)\n     */\n    requestFullscreen(options?: FullscreenOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/requestPointerLock) */\n    requestPointerLock(options?: PointerLockOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scroll) */\n    scroll(options?: ScrollToOptions): void;\n    scroll(x: number, y: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollBy) */\n    scrollBy(options?: ScrollToOptions): void;\n    scrollBy(x: number, y: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollIntoView) */\n    scrollIntoView(arg?: boolean | ScrollIntoViewOptions): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/scrollTo) */\n    scrollTo(options?: ScrollToOptions): void;\n    scrollTo(x: number, y: number): void;\n    /**\n     * Sets the value of element's first attribute whose qualified name is qualifiedName to value.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttribute)\n     */\n    setAttribute(qualifiedName: string, value: string): void;\n    /**\n     * Sets the value of element's attribute whose namespace is namespace and local name is localName to value.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttributeNS)\n     */\n    setAttributeNS(namespace: string | null, qualifiedName: string, value: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttributeNode) */\n    setAttributeNode(attr: Attr): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setAttributeNodeNS) */\n    setAttributeNodeNS(attr: Attr): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setHTMLUnsafe) */\n    setHTMLUnsafe(html: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/setPointerCapture) */\n    setPointerCapture(pointerId: number): void;\n    /**\n     * If force is not given, \"toggles\" qualifiedName, removing it if it is present and adding it if it is not present. If force is true, adds qualifiedName. If force is false, removes qualifiedName.\n     *\n     * Returns true if qualifiedName is now present, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/toggleAttribute)\n     */\n    toggleAttribute(qualifiedName: string, force?: boolean): boolean;\n    /**\n     * @deprecated This is a legacy alias of `matches`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/matches)\n     */\n    webkitMatchesSelector(selectors: string): boolean;\n    addEventListener<K extends keyof ElementEventMap>(type: K, listener: (this: Element, ev: ElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ElementEventMap>(type: K, listener: (this: Element, ev: ElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var Element: {\n    prototype: Element;\n    new(): Element;\n};\n\ninterface ElementCSSInlineStyle {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/attributeStyleMap) */\n    readonly attributeStyleMap: StylePropertyMap;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/style) */\n    readonly style: CSSStyleDeclaration;\n}\n\ninterface ElementContentEditable {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/contentEditable) */\n    contentEditable: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/enterKeyHint) */\n    enterKeyHint: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/inputMode) */\n    inputMode: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/isContentEditable) */\n    readonly isContentEditable: boolean;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals) */\ninterface ElementInternals extends ARIAMixin {\n    /**\n     * Returns the form owner of internals's target element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/form)\n     */\n    readonly form: HTMLFormElement | null;\n    /**\n     * Returns a NodeList of all the label elements that internals's target element is associated with.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/labels)\n     */\n    readonly labels: NodeList;\n    /**\n     * Returns the ShadowRoot for internals's target element, if the target element is a shadow host, or null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/shadowRoot)\n     */\n    readonly shadowRoot: ShadowRoot | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/states) */\n    readonly states: CustomStateSet;\n    /**\n     * Returns the error message that would be shown to the user if internals's target element was to be checked for validity.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/validationMessage)\n     */\n    readonly validationMessage: string;\n    /**\n     * Returns the ValidityState object for internals's target element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/validity)\n     */\n    readonly validity: ValidityState;\n    /**\n     * Returns true if internals's target element will be validated when the form is submitted; false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/willValidate)\n     */\n    readonly willValidate: boolean;\n    /**\n     * Returns true if internals's target element has no validity problems; false otherwise. Fires an invalid event at the element in the latter case.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/checkValidity)\n     */\n    checkValidity(): boolean;\n    /**\n     * Returns true if internals's target element has no validity problems; otherwise, returns false, fires an invalid event at the element, and (if the event isn't canceled) reports the problem to the user.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/reportValidity)\n     */\n    reportValidity(): boolean;\n    /**\n     * Sets both the state and submission value of internals's target element to value.\n     *\n     * If value is null, the element won't participate in form submission.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/setFormValue)\n     */\n    setFormValue(value: File | string | FormData | null, state?: File | string | FormData | null): void;\n    /**\n     * Marks internals's target element as suffering from the constraints indicated by the flags argument, and sets the element's validation message to message. If anchor is specified, the user agent might use it to indicate problems with the constraints of internals's target element when the form owner is validated interactively or reportValidity() is called.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ElementInternals/setValidity)\n     */\n    setValidity(flags?: ValidityStateFlags, message?: string, anchor?: HTMLElement): void;\n}\n\ndeclare var ElementInternals: {\n    prototype: ElementInternals;\n    new(): ElementInternals;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EncodedVideoChunk) */\ninterface EncodedVideoChunk {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EncodedVideoChunk/byteLength) */\n    readonly byteLength: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EncodedVideoChunk/duration) */\n    readonly duration: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EncodedVideoChunk/timestamp) */\n    readonly timestamp: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EncodedVideoChunk/type) */\n    readonly type: EncodedVideoChunkType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EncodedVideoChunk/copyTo) */\n    copyTo(destination: AllowSharedBufferSource): void;\n}\n\ndeclare var EncodedVideoChunk: {\n    prototype: EncodedVideoChunk;\n    new(init: EncodedVideoChunkInit): EncodedVideoChunk;\n};\n\n/**\n * Events providing information related to errors in scripts or in files.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ErrorEvent)\n */\ninterface ErrorEvent extends Event {\n    readonly colno: number;\n    readonly error: any;\n    readonly filename: string;\n    readonly lineno: number;\n    readonly message: string;\n}\n\ndeclare var ErrorEvent: {\n    prototype: ErrorEvent;\n    new(type: string, eventInitDict?: ErrorEventInit): ErrorEvent;\n};\n\n/**\n * An event which takes place in the DOM.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event)\n */\ninterface Event {\n    /**\n     * Returns true or false depending on how event was initialized. True if event goes through its target's ancestors in reverse tree order, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/bubbles)\n     */\n    readonly bubbles: boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/cancelBubble)\n     */\n    cancelBubble: boolean;\n    /**\n     * Returns true or false depending on how event was initialized. Its return value does not always carry meaning, but true can indicate that part of the operation during which event was dispatched, can be canceled by invoking the preventDefault() method.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/cancelable)\n     */\n    readonly cancelable: boolean;\n    /**\n     * Returns true or false depending on how event was initialized. True if event invokes listeners past a ShadowRoot node that is the root of its target, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/composed)\n     */\n    readonly composed: boolean;\n    /**\n     * Returns the object whose event listener's callback is currently being invoked.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/currentTarget)\n     */\n    readonly currentTarget: EventTarget | null;\n    /**\n     * Returns true if preventDefault() was invoked successfully to indicate cancelation, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/defaultPrevented)\n     */\n    readonly defaultPrevented: boolean;\n    /**\n     * Returns the event's phase, which is one of NONE, CAPTURING_PHASE, AT_TARGET, and BUBBLING_PHASE.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/eventPhase)\n     */\n    readonly eventPhase: number;\n    /**\n     * Returns true if event was dispatched by the user agent, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/isTrusted)\n     */\n    readonly isTrusted: boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/returnValue)\n     */\n    returnValue: boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/srcElement)\n     */\n    readonly srcElement: EventTarget | null;\n    /**\n     * Returns the object to which event is dispatched (its target).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/target)\n     */\n    readonly target: EventTarget | null;\n    /**\n     * Returns the event's timestamp as the number of milliseconds measured relative to the time origin.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/timeStamp)\n     */\n    readonly timeStamp: DOMHighResTimeStamp;\n    /**\n     * Returns the type of event, e.g. \"click\", \"hashchange\", or \"submit\".\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/type)\n     */\n    readonly type: string;\n    /**\n     * Returns the invocation target objects of event's path (objects on which listeners will be invoked), except for any nodes in shadow trees of which the shadow root's mode is \"closed\" that are not reachable from event's currentTarget.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/composedPath)\n     */\n    composedPath(): EventTarget[];\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/initEvent)\n     */\n    initEvent(type: string, bubbles?: boolean, cancelable?: boolean): void;\n    /**\n     * If invoked when the cancelable attribute value is true, and while executing a listener for the event with passive set to false, signals to the operation that caused event to be dispatched that it needs to be canceled.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/preventDefault)\n     */\n    preventDefault(): void;\n    /**\n     * Invoking this method prevents event from reaching any registered event listeners after the current one finishes running and, when dispatched in a tree, also prevents event from reaching any other objects.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/stopImmediatePropagation)\n     */\n    stopImmediatePropagation(): void;\n    /**\n     * When dispatched in a tree, invoking this method prevents event from reaching any objects other than the current object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Event/stopPropagation)\n     */\n    stopPropagation(): void;\n    readonly NONE: 0;\n    readonly CAPTURING_PHASE: 1;\n    readonly AT_TARGET: 2;\n    readonly BUBBLING_PHASE: 3;\n}\n\ndeclare var Event: {\n    prototype: Event;\n    new(type: string, eventInitDict?: EventInit): Event;\n    readonly NONE: 0;\n    readonly CAPTURING_PHASE: 1;\n    readonly AT_TARGET: 2;\n    readonly BUBBLING_PHASE: 3;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventCounts) */\ninterface EventCounts {\n    forEach(callbackfn: (value: number, key: string, parent: EventCounts) => void, thisArg?: any): void;\n}\n\ndeclare var EventCounts: {\n    prototype: EventCounts;\n    new(): EventCounts;\n};\n\ninterface EventListener {\n    (evt: Event): void;\n}\n\ninterface EventListenerObject {\n    handleEvent(object: Event): void;\n}\n\ninterface EventSourceEventMap {\n    \"error\": Event;\n    \"message\": MessageEvent;\n    \"open\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource) */\ninterface EventSource extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/error_event) */\n    onerror: ((this: EventSource, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/message_event) */\n    onmessage: ((this: EventSource, ev: MessageEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/open_event) */\n    onopen: ((this: EventSource, ev: Event) => any) | null;\n    /**\n     * Returns the state of this EventSource object's connection. It can have the values described below.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/readyState)\n     */\n    readonly readyState: number;\n    /**\n     * Returns the URL providing the event stream.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/url)\n     */\n    readonly url: string;\n    /**\n     * Returns true if the credentials mode for connection requests to the URL providing the event stream is set to \"include\", and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/withCredentials)\n     */\n    readonly withCredentials: boolean;\n    /**\n     * Aborts any instances of the fetch algorithm started for this EventSource object, and sets the readyState attribute to CLOSED.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventSource/close)\n     */\n    close(): void;\n    readonly CONNECTING: 0;\n    readonly OPEN: 1;\n    readonly CLOSED: 2;\n    addEventListener<K extends keyof EventSourceEventMap>(type: K, listener: (this: EventSource, ev: EventSourceEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: (this: EventSource, event: MessageEvent) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof EventSourceEventMap>(type: K, listener: (this: EventSource, ev: EventSourceEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: (this: EventSource, event: MessageEvent) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var EventSource: {\n    prototype: EventSource;\n    new(url: string | URL, eventSourceInitDict?: EventSourceInit): EventSource;\n    readonly CONNECTING: 0;\n    readonly OPEN: 1;\n    readonly CLOSED: 2;\n};\n\n/**\n * EventTarget is a DOM interface implemented by objects that can receive events and may have listeners for them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventTarget)\n */\ninterface EventTarget {\n    /**\n     * Appends an event listener for events whose type attribute value is type. The callback argument sets the callback that will be invoked when the event is dispatched.\n     *\n     * The options argument sets listener-specific options. For compatibility this can be a boolean, in which case the method behaves exactly as if the value was specified as options's capture.\n     *\n     * When set to true, options's capture prevents callback from being invoked when the event's eventPhase attribute value is BUBBLING_PHASE. When false (or not present), callback will not be invoked when event's eventPhase attribute value is CAPTURING_PHASE. Either way, callback will be invoked if event's eventPhase attribute value is AT_TARGET.\n     *\n     * When set to true, options's passive indicates that the callback will not cancel the event by invoking preventDefault(). This is used to enable performance optimizations described in § 2.8 Observing event listeners.\n     *\n     * When set to true, options's once indicates that the callback will only be invoked once after which the event listener will be removed.\n     *\n     * If an AbortSignal is passed for options's signal, then the event listener will be removed when signal is aborted.\n     *\n     * The event listener is appended to target's event listener list and is not appended if it has the same type, callback, and capture.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventTarget/addEventListener)\n     */\n    addEventListener(type: string, callback: EventListenerOrEventListenerObject | null, options?: AddEventListenerOptions | boolean): void;\n    /**\n     * Dispatches a synthetic event event to target and returns true if either event's cancelable attribute value is false or its preventDefault() method was not invoked, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventTarget/dispatchEvent)\n     */\n    dispatchEvent(event: Event): boolean;\n    /**\n     * Removes the event listener in target's event listener list with the same type, callback, and options.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/EventTarget/removeEventListener)\n     */\n    removeEventListener(type: string, callback: EventListenerOrEventListenerObject | null, options?: EventListenerOptions | boolean): void;\n}\n\ndeclare var EventTarget: {\n    prototype: EventTarget;\n    new(): EventTarget;\n};\n\n/** @deprecated */\ninterface External {\n    /** @deprecated */\n    AddSearchProvider(): void;\n    /** @deprecated */\n    IsSearchProviderInstalled(): void;\n}\n\n/** @deprecated */\ndeclare var External: {\n    prototype: External;\n    new(): External;\n};\n\n/**\n * Provides information about files and allows JavaScript in a web page to access their content.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/File)\n */\ninterface File extends Blob {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/File/lastModified) */\n    readonly lastModified: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/File/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/File/webkitRelativePath) */\n    readonly webkitRelativePath: string;\n}\n\ndeclare var File: {\n    prototype: File;\n    new(fileBits: BlobPart[], fileName: string, options?: FilePropertyBag): File;\n};\n\n/**\n * An object of this type is returned by the files property of the HTML <input> element; this lets you access the list of files selected with the <input type=\"file\"> element. It's also used for a list of files dropped into web content when using the drag and drop API; see the DataTransfer object for details on this usage.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileList)\n */\ninterface FileList {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileList/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileList/item) */\n    item(index: number): File | null;\n    [index: number]: File;\n}\n\ndeclare var FileList: {\n    prototype: FileList;\n    new(): FileList;\n};\n\ninterface FileReaderEventMap {\n    \"abort\": ProgressEvent<FileReader>;\n    \"error\": ProgressEvent<FileReader>;\n    \"load\": ProgressEvent<FileReader>;\n    \"loadend\": ProgressEvent<FileReader>;\n    \"loadstart\": ProgressEvent<FileReader>;\n    \"progress\": ProgressEvent<FileReader>;\n}\n\n/**\n * Lets web applications asynchronously read the contents of files (or raw data buffers) stored on the user's computer, using File or Blob objects to specify the file or data to read.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader)\n */\ninterface FileReader extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/error) */\n    readonly error: DOMException | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/abort_event) */\n    onabort: ((this: FileReader, ev: ProgressEvent<FileReader>) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/error_event) */\n    onerror: ((this: FileReader, ev: ProgressEvent<FileReader>) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/load_event) */\n    onload: ((this: FileReader, ev: ProgressEvent<FileReader>) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/loadend_event) */\n    onloadend: ((this: FileReader, ev: ProgressEvent<FileReader>) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/loadstart_event) */\n    onloadstart: ((this: FileReader, ev: ProgressEvent<FileReader>) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/progress_event) */\n    onprogress: ((this: FileReader, ev: ProgressEvent<FileReader>) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/readyState) */\n    readonly readyState: typeof FileReader.EMPTY | typeof FileReader.LOADING | typeof FileReader.DONE;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/result) */\n    readonly result: string | ArrayBuffer | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/abort) */\n    abort(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/readAsArrayBuffer) */\n    readAsArrayBuffer(blob: Blob): void;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/readAsBinaryString)\n     */\n    readAsBinaryString(blob: Blob): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/readAsDataURL) */\n    readAsDataURL(blob: Blob): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileReader/readAsText) */\n    readAsText(blob: Blob, encoding?: string): void;\n    readonly EMPTY: 0;\n    readonly LOADING: 1;\n    readonly DONE: 2;\n    addEventListener<K extends keyof FileReaderEventMap>(type: K, listener: (this: FileReader, ev: FileReaderEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof FileReaderEventMap>(type: K, listener: (this: FileReader, ev: FileReaderEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var FileReader: {\n    prototype: FileReader;\n    new(): FileReader;\n    readonly EMPTY: 0;\n    readonly LOADING: 1;\n    readonly DONE: 2;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystem) */\ninterface FileSystem {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystem/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystem/root) */\n    readonly root: FileSystemDirectoryEntry;\n}\n\ndeclare var FileSystem: {\n    prototype: FileSystem;\n    new(): FileSystem;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryEntry) */\ninterface FileSystemDirectoryEntry extends FileSystemEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryEntry/createReader) */\n    createReader(): FileSystemDirectoryReader;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryEntry/getDirectory) */\n    getDirectory(path?: string | null, options?: FileSystemFlags, successCallback?: FileSystemEntryCallback, errorCallback?: ErrorCallback): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryEntry/getFile) */\n    getFile(path?: string | null, options?: FileSystemFlags, successCallback?: FileSystemEntryCallback, errorCallback?: ErrorCallback): void;\n}\n\ndeclare var FileSystemDirectoryEntry: {\n    prototype: FileSystemDirectoryEntry;\n    new(): FileSystemDirectoryEntry;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryHandle)\n */\ninterface FileSystemDirectoryHandle extends FileSystemHandle {\n    readonly kind: \"directory\";\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryHandle/getDirectoryHandle) */\n    getDirectoryHandle(name: string, options?: FileSystemGetDirectoryOptions): Promise<FileSystemDirectoryHandle>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryHandle/getFileHandle) */\n    getFileHandle(name: string, options?: FileSystemGetFileOptions): Promise<FileSystemFileHandle>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryHandle/removeEntry) */\n    removeEntry(name: string, options?: FileSystemRemoveOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryHandle/resolve) */\n    resolve(possibleDescendant: FileSystemHandle): Promise<string[] | null>;\n}\n\ndeclare var FileSystemDirectoryHandle: {\n    prototype: FileSystemDirectoryHandle;\n    new(): FileSystemDirectoryHandle;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryReader) */\ninterface FileSystemDirectoryReader {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemDirectoryReader/readEntries) */\n    readEntries(successCallback: FileSystemEntriesCallback, errorCallback?: ErrorCallback): void;\n}\n\ndeclare var FileSystemDirectoryReader: {\n    prototype: FileSystemDirectoryReader;\n    new(): FileSystemDirectoryReader;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry) */\ninterface FileSystemEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry/filesystem) */\n    readonly filesystem: FileSystem;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry/fullPath) */\n    readonly fullPath: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry/isDirectory) */\n    readonly isDirectory: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry/isFile) */\n    readonly isFile: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemEntry/getParent) */\n    getParent(successCallback?: FileSystemEntryCallback, errorCallback?: ErrorCallback): void;\n}\n\ndeclare var FileSystemEntry: {\n    prototype: FileSystemEntry;\n    new(): FileSystemEntry;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemFileEntry) */\ninterface FileSystemFileEntry extends FileSystemEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemFileEntry/file) */\n    file(successCallback: FileCallback, errorCallback?: ErrorCallback): void;\n}\n\ndeclare var FileSystemFileEntry: {\n    prototype: FileSystemFileEntry;\n    new(): FileSystemFileEntry;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemFileHandle)\n */\ninterface FileSystemFileHandle extends FileSystemHandle {\n    readonly kind: \"file\";\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemFileHandle/createWritable) */\n    createWritable(options?: FileSystemCreateWritableOptions): Promise<FileSystemWritableFileStream>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemFileHandle/getFile) */\n    getFile(): Promise<File>;\n}\n\ndeclare var FileSystemFileHandle: {\n    prototype: FileSystemFileHandle;\n    new(): FileSystemFileHandle;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemHandle)\n */\ninterface FileSystemHandle {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemHandle/kind) */\n    readonly kind: FileSystemHandleKind;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemHandle/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemHandle/isSameEntry) */\n    isSameEntry(other: FileSystemHandle): Promise<boolean>;\n}\n\ndeclare var FileSystemHandle: {\n    prototype: FileSystemHandle;\n    new(): FileSystemHandle;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemWritableFileStream)\n */\ninterface FileSystemWritableFileStream extends WritableStream {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemWritableFileStream/seek) */\n    seek(position: number): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemWritableFileStream/truncate) */\n    truncate(size: number): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FileSystemWritableFileStream/write) */\n    write(data: FileSystemWriteChunkType): Promise<void>;\n}\n\ndeclare var FileSystemWritableFileStream: {\n    prototype: FileSystemWritableFileStream;\n    new(): FileSystemWritableFileStream;\n};\n\n/**\n * Focus-related events like focus, blur, focusin, or focusout.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FocusEvent)\n */\ninterface FocusEvent extends UIEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FocusEvent/relatedTarget) */\n    readonly relatedTarget: EventTarget | null;\n}\n\ndeclare var FocusEvent: {\n    prototype: FocusEvent;\n    new(type: string, eventInitDict?: FocusEventInit): FocusEvent;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace) */\ninterface FontFace {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/ascentOverride) */\n    ascentOverride: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/descentOverride) */\n    descentOverride: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/display) */\n    display: FontDisplay;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/family) */\n    family: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/featureSettings) */\n    featureSettings: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/lineGapOverride) */\n    lineGapOverride: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/loaded) */\n    readonly loaded: Promise<FontFace>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/status) */\n    readonly status: FontFaceLoadStatus;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/stretch) */\n    stretch: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/style) */\n    style: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/unicodeRange) */\n    unicodeRange: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/weight) */\n    weight: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFace/load) */\n    load(): Promise<FontFace>;\n}\n\ndeclare var FontFace: {\n    prototype: FontFace;\n    new(family: string, source: string | BinaryData, descriptors?: FontFaceDescriptors): FontFace;\n};\n\ninterface FontFaceSetEventMap {\n    \"loading\": Event;\n    \"loadingdone\": Event;\n    \"loadingerror\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet) */\ninterface FontFaceSet extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/loading_event) */\n    onloading: ((this: FontFaceSet, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/loadingdone_event) */\n    onloadingdone: ((this: FontFaceSet, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/loadingerror_event) */\n    onloadingerror: ((this: FontFaceSet, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/ready) */\n    readonly ready: Promise<FontFaceSet>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/status) */\n    readonly status: FontFaceSetLoadStatus;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/check) */\n    check(font: string, text?: string): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSet/load) */\n    load(font: string, text?: string): Promise<FontFace[]>;\n    forEach(callbackfn: (value: FontFace, key: FontFace, parent: FontFaceSet) => void, thisArg?: any): void;\n    addEventListener<K extends keyof FontFaceSetEventMap>(type: K, listener: (this: FontFaceSet, ev: FontFaceSetEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof FontFaceSetEventMap>(type: K, listener: (this: FontFaceSet, ev: FontFaceSetEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var FontFaceSet: {\n    prototype: FontFaceSet;\n    new(initialFaces: FontFace[]): FontFaceSet;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSetLoadEvent) */\ninterface FontFaceSetLoadEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FontFaceSetLoadEvent/fontfaces) */\n    readonly fontfaces: ReadonlyArray<FontFace>;\n}\n\ndeclare var FontFaceSetLoadEvent: {\n    prototype: FontFaceSetLoadEvent;\n    new(type: string, eventInitDict?: FontFaceSetLoadEventInit): FontFaceSetLoadEvent;\n};\n\ninterface FontFaceSource {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/fonts) */\n    readonly fonts: FontFaceSet;\n}\n\n/**\n * Provides a way to easily construct a set of key/value pairs representing form fields and their values, which can then be easily sent using the XMLHttpRequest.send() method. It uses the same format a form would use if the encoding type were set to \"multipart/form-data\".\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData)\n */\ninterface FormData {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData/append) */\n    append(name: string, value: string | Blob): void;\n    append(name: string, value: string): void;\n    append(name: string, blobValue: Blob, filename?: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData/delete) */\n    delete(name: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData/get) */\n    get(name: string): FormDataEntryValue | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData/getAll) */\n    getAll(name: string): FormDataEntryValue[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData/has) */\n    has(name: string): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormData/set) */\n    set(name: string, value: string | Blob): void;\n    set(name: string, value: string): void;\n    set(name: string, blobValue: Blob, filename?: string): void;\n    forEach(callbackfn: (value: FormDataEntryValue, key: string, parent: FormData) => void, thisArg?: any): void;\n}\n\ndeclare var FormData: {\n    prototype: FormData;\n    new(form?: HTMLFormElement, submitter?: HTMLElement | null): FormData;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormDataEvent) */\ninterface FormDataEvent extends Event {\n    /**\n     * Returns a FormData object representing names and values of elements associated to the target form. Operations on the FormData object will affect form data to be submitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/FormDataEvent/formData)\n     */\n    readonly formData: FormData;\n}\n\ndeclare var FormDataEvent: {\n    prototype: FormDataEvent;\n    new(type: string, eventInitDict: FormDataEventInit): FormDataEvent;\n};\n\n/**\n * A change in volume. It is an AudioNode audio-processing module that causes a given gain to be applied to the input data before its propagation to the output. A GainNode always has exactly one input and one output, both with the same number of channels.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/GainNode)\n */\ninterface GainNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GainNode/gain) */\n    readonly gain: AudioParam;\n}\n\ndeclare var GainNode: {\n    prototype: GainNode;\n    new(context: BaseAudioContext, options?: GainOptions): GainNode;\n};\n\n/**\n * This Gamepad API interface defines an individual gamepad or other controller, allowing access to information such as button presses, axis positions, and id.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad)\n */\ninterface Gamepad {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/axes) */\n    readonly axes: ReadonlyArray<number>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/buttons) */\n    readonly buttons: ReadonlyArray<GamepadButton>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/connected) */\n    readonly connected: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/id) */\n    readonly id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/index) */\n    readonly index: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/mapping) */\n    readonly mapping: GamepadMappingType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/timestamp) */\n    readonly timestamp: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Gamepad/vibrationActuator) */\n    readonly vibrationActuator: GamepadHapticActuator;\n}\n\ndeclare var Gamepad: {\n    prototype: Gamepad;\n    new(): Gamepad;\n};\n\n/**\n * An individual button of a gamepad or other controller, allowing access to the current state of different types of buttons available on the control device.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadButton)\n */\ninterface GamepadButton {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadButton/pressed) */\n    readonly pressed: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadButton/touched) */\n    readonly touched: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadButton/value) */\n    readonly value: number;\n}\n\ndeclare var GamepadButton: {\n    prototype: GamepadButton;\n    new(): GamepadButton;\n};\n\n/**\n * This Gamepad API interface contains references to gamepads connected to the system, which is what the gamepad events Window.gamepadconnected and Window.gamepaddisconnected are fired in response to.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadEvent)\n */\ninterface GamepadEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadEvent/gamepad) */\n    readonly gamepad: Gamepad;\n}\n\ndeclare var GamepadEvent: {\n    prototype: GamepadEvent;\n    new(type: string, eventInitDict: GamepadEventInit): GamepadEvent;\n};\n\n/**\n * This Gamepad API interface represents hardware in the controller designed to provide haptic feedback to the user (if available), most commonly vibration hardware.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadHapticActuator)\n */\ninterface GamepadHapticActuator {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GamepadHapticActuator/playEffect) */\n    playEffect(type: GamepadHapticEffectType, params?: GamepadEffectParameters): Promise<GamepadHapticsResult>;\n    reset(): Promise<GamepadHapticsResult>;\n}\n\ndeclare var GamepadHapticActuator: {\n    prototype: GamepadHapticActuator;\n    new(): GamepadHapticActuator;\n};\n\ninterface GenericTransformStream {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CompressionStream/readable) */\n    readonly readable: ReadableStream;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/CompressionStream/writable) */\n    readonly writable: WritableStream;\n}\n\n/**\n * An object able to programmatically obtain the position of the device. It gives Web content access to the location of the device. This allows a Web site or app to offer customized results based on the user's location.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Geolocation)\n */\ninterface Geolocation {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Geolocation/clearWatch) */\n    clearWatch(watchId: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Geolocation/getCurrentPosition) */\n    getCurrentPosition(successCallback: PositionCallback, errorCallback?: PositionErrorCallback | null, options?: PositionOptions): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Geolocation/watchPosition) */\n    watchPosition(successCallback: PositionCallback, errorCallback?: PositionErrorCallback | null, options?: PositionOptions): number;\n}\n\ndeclare var Geolocation: {\n    prototype: Geolocation;\n    new(): Geolocation;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates)\n */\ninterface GeolocationCoordinates {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/accuracy) */\n    readonly accuracy: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/altitude) */\n    readonly altitude: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/altitudeAccuracy) */\n    readonly altitudeAccuracy: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/heading) */\n    readonly heading: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/latitude) */\n    readonly latitude: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/longitude) */\n    readonly longitude: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/speed) */\n    readonly speed: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationCoordinates/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var GeolocationCoordinates: {\n    prototype: GeolocationCoordinates;\n    new(): GeolocationCoordinates;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPosition)\n */\ninterface GeolocationPosition {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPosition/coords) */\n    readonly coords: GeolocationCoordinates;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPosition/timestamp) */\n    readonly timestamp: EpochTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPosition/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var GeolocationPosition: {\n    prototype: GeolocationPosition;\n    new(): GeolocationPosition;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPositionError) */\ninterface GeolocationPositionError {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPositionError/code) */\n    readonly code: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/GeolocationPositionError/message) */\n    readonly message: string;\n    readonly PERMISSION_DENIED: 1;\n    readonly POSITION_UNAVAILABLE: 2;\n    readonly TIMEOUT: 3;\n}\n\ndeclare var GeolocationPositionError: {\n    prototype: GeolocationPositionError;\n    new(): GeolocationPositionError;\n    readonly PERMISSION_DENIED: 1;\n    readonly POSITION_UNAVAILABLE: 2;\n    readonly TIMEOUT: 3;\n};\n\ninterface GlobalEventHandlersEventMap {\n    \"abort\": UIEvent;\n    \"animationcancel\": AnimationEvent;\n    \"animationend\": AnimationEvent;\n    \"animationiteration\": AnimationEvent;\n    \"animationstart\": AnimationEvent;\n    \"auxclick\": MouseEvent;\n    \"beforeinput\": InputEvent;\n    \"beforetoggle\": Event;\n    \"blur\": FocusEvent;\n    \"cancel\": Event;\n    \"canplay\": Event;\n    \"canplaythrough\": Event;\n    \"change\": Event;\n    \"click\": MouseEvent;\n    \"close\": Event;\n    \"compositionend\": CompositionEvent;\n    \"compositionstart\": CompositionEvent;\n    \"compositionupdate\": CompositionEvent;\n    \"contextlost\": Event;\n    \"contextmenu\": MouseEvent;\n    \"contextrestored\": Event;\n    \"copy\": ClipboardEvent;\n    \"cuechange\": Event;\n    \"cut\": ClipboardEvent;\n    \"dblclick\": MouseEvent;\n    \"drag\": DragEvent;\n    \"dragend\": DragEvent;\n    \"dragenter\": DragEvent;\n    \"dragleave\": DragEvent;\n    \"dragover\": DragEvent;\n    \"dragstart\": DragEvent;\n    \"drop\": DragEvent;\n    \"durationchange\": Event;\n    \"emptied\": Event;\n    \"ended\": Event;\n    \"error\": ErrorEvent;\n    \"focus\": FocusEvent;\n    \"focusin\": FocusEvent;\n    \"focusout\": FocusEvent;\n    \"formdata\": FormDataEvent;\n    \"gotpointercapture\": PointerEvent;\n    \"input\": Event;\n    \"invalid\": Event;\n    \"keydown\": KeyboardEvent;\n    \"keypress\": KeyboardEvent;\n    \"keyup\": KeyboardEvent;\n    \"load\": Event;\n    \"loadeddata\": Event;\n    \"loadedmetadata\": Event;\n    \"loadstart\": Event;\n    \"lostpointercapture\": PointerEvent;\n    \"mousedown\": MouseEvent;\n    \"mouseenter\": MouseEvent;\n    \"mouseleave\": MouseEvent;\n    \"mousemove\": MouseEvent;\n    \"mouseout\": MouseEvent;\n    \"mouseover\": MouseEvent;\n    \"mouseup\": MouseEvent;\n    \"paste\": ClipboardEvent;\n    \"pause\": Event;\n    \"play\": Event;\n    \"playing\": Event;\n    \"pointercancel\": PointerEvent;\n    \"pointerdown\": PointerEvent;\n    \"pointerenter\": PointerEvent;\n    \"pointerleave\": PointerEvent;\n    \"pointermove\": PointerEvent;\n    \"pointerout\": PointerEvent;\n    \"pointerover\": PointerEvent;\n    \"pointerup\": PointerEvent;\n    \"progress\": ProgressEvent;\n    \"ratechange\": Event;\n    \"reset\": Event;\n    \"resize\": UIEvent;\n    \"scroll\": Event;\n    \"scrollend\": Event;\n    \"securitypolicyviolation\": SecurityPolicyViolationEvent;\n    \"seeked\": Event;\n    \"seeking\": Event;\n    \"select\": Event;\n    \"selectionchange\": Event;\n    \"selectstart\": Event;\n    \"slotchange\": Event;\n    \"stalled\": Event;\n    \"submit\": SubmitEvent;\n    \"suspend\": Event;\n    \"timeupdate\": Event;\n    \"toggle\": Event;\n    \"touchcancel\": TouchEvent;\n    \"touchend\": TouchEvent;\n    \"touchmove\": TouchEvent;\n    \"touchstart\": TouchEvent;\n    \"transitioncancel\": TransitionEvent;\n    \"transitionend\": TransitionEvent;\n    \"transitionrun\": TransitionEvent;\n    \"transitionstart\": TransitionEvent;\n    \"volumechange\": Event;\n    \"waiting\": Event;\n    \"webkitanimationend\": Event;\n    \"webkitanimationiteration\": Event;\n    \"webkitanimationstart\": Event;\n    \"webkittransitionend\": Event;\n    \"wheel\": WheelEvent;\n}\n\ninterface GlobalEventHandlers {\n    /**\n     * Fires when the user aborts the download.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/abort_event)\n     */\n    onabort: ((this: GlobalEventHandlers, ev: UIEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationcancel_event) */\n    onanimationcancel: ((this: GlobalEventHandlers, ev: AnimationEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationend_event) */\n    onanimationend: ((this: GlobalEventHandlers, ev: AnimationEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationiteration_event) */\n    onanimationiteration: ((this: GlobalEventHandlers, ev: AnimationEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationstart_event) */\n    onanimationstart: ((this: GlobalEventHandlers, ev: AnimationEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/auxclick_event) */\n    onauxclick: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/beforeinput_event) */\n    onbeforeinput: ((this: GlobalEventHandlers, ev: InputEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/beforetoggle_event) */\n    onbeforetoggle: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the object loses the input focus.\n     * @param ev The focus event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/blur_event)\n     */\n    onblur: ((this: GlobalEventHandlers, ev: FocusEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/cancel_event) */\n    oncancel: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when playback is possible, but would require further buffering.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/canplay_event)\n     */\n    oncanplay: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/canplaythrough_event) */\n    oncanplaythrough: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the contents of the object or selection have changed.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/change_event)\n     */\n    onchange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the user clicks the left mouse button on the object\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/click_event)\n     */\n    onclick: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement/close_event) */\n    onclose: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/webglcontextlost_event) */\n    oncontextlost: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the user clicks the right mouse button in the client area, opening the context menu.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/contextmenu_event)\n     */\n    oncontextmenu: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/contextrestored_event) */\n    oncontextrestored: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/copy_event) */\n    oncopy: ((this: GlobalEventHandlers, ev: ClipboardEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTrackElement/cuechange_event) */\n    oncuechange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/cut_event) */\n    oncut: ((this: GlobalEventHandlers, ev: ClipboardEvent) => any) | null;\n    /**\n     * Fires when the user double-clicks the object.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/dblclick_event)\n     */\n    ondblclick: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /**\n     * Fires on the source object continuously during a drag operation.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/drag_event)\n     */\n    ondrag: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /**\n     * Fires on the source object when the user releases the mouse at the close of a drag operation.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dragend_event)\n     */\n    ondragend: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /**\n     * Fires on the target element when the user drags the object to a valid drop target.\n     * @param ev The drag event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dragenter_event)\n     */\n    ondragenter: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /**\n     * Fires on the target object when the user moves the mouse out of a valid drop target during a drag operation.\n     * @param ev The drag event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dragleave_event)\n     */\n    ondragleave: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /**\n     * Fires on the target element continuously while the user drags the object over a valid drop target.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dragover_event)\n     */\n    ondragover: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /**\n     * Fires on the source object when the user starts to drag a text selection or selected object.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dragstart_event)\n     */\n    ondragstart: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/drop_event) */\n    ondrop: ((this: GlobalEventHandlers, ev: DragEvent) => any) | null;\n    /**\n     * Occurs when the duration attribute is updated.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/durationchange_event)\n     */\n    ondurationchange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the media element is reset to its initial state.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/emptied_event)\n     */\n    onemptied: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the end of playback is reached.\n     * @param ev The event\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/ended_event)\n     */\n    onended: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when an error occurs during object loading.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/error_event)\n     */\n    onerror: OnErrorEventHandler;\n    /**\n     * Fires when the object receives focus.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/focus_event)\n     */\n    onfocus: ((this: GlobalEventHandlers, ev: FocusEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/formdata_event) */\n    onformdata: ((this: GlobalEventHandlers, ev: FormDataEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/gotpointercapture_event) */\n    ongotpointercapture: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/input_event) */\n    oninput: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/invalid_event) */\n    oninvalid: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the user presses a key.\n     * @param ev The keyboard event\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/keydown_event)\n     */\n    onkeydown: ((this: GlobalEventHandlers, ev: KeyboardEvent) => any) | null;\n    /**\n     * Fires when the user presses an alphanumeric key.\n     * @param ev The event.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/keypress_event)\n     */\n    onkeypress: ((this: GlobalEventHandlers, ev: KeyboardEvent) => any) | null;\n    /**\n     * Fires when the user releases a key.\n     * @param ev The keyboard event\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/keyup_event)\n     */\n    onkeyup: ((this: GlobalEventHandlers, ev: KeyboardEvent) => any) | null;\n    /**\n     * Fires immediately after the browser loads the object.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGElement/load_event)\n     */\n    onload: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when media data is loaded at the current playback position.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/loadeddata_event)\n     */\n    onloadeddata: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the duration and dimensions of the media have been determined.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/loadedmetadata_event)\n     */\n    onloadedmetadata: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when Internet Explorer begins looking for media data.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/loadstart_event)\n     */\n    onloadstart: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/lostpointercapture_event) */\n    onlostpointercapture: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /**\n     * Fires when the user clicks the object with either mouse button.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mousedown_event)\n     */\n    onmousedown: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mouseenter_event) */\n    onmouseenter: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mouseleave_event) */\n    onmouseleave: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /**\n     * Fires when the user moves the mouse over the object.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mousemove_event)\n     */\n    onmousemove: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /**\n     * Fires when the user moves the mouse pointer outside the boundaries of the object.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mouseout_event)\n     */\n    onmouseout: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /**\n     * Fires when the user moves the mouse pointer into the object.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mouseover_event)\n     */\n    onmouseover: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /**\n     * Fires when the user releases a mouse button while the mouse is over the object.\n     * @param ev The mouse event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/mouseup_event)\n     */\n    onmouseup: ((this: GlobalEventHandlers, ev: MouseEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/paste_event) */\n    onpaste: ((this: GlobalEventHandlers, ev: ClipboardEvent) => any) | null;\n    /**\n     * Occurs when playback is paused.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/pause_event)\n     */\n    onpause: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the play method is requested.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/play_event)\n     */\n    onplay: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the audio or video has started playing.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/playing_event)\n     */\n    onplaying: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointercancel_event) */\n    onpointercancel: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointerdown_event) */\n    onpointerdown: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointerenter_event) */\n    onpointerenter: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointerleave_event) */\n    onpointerleave: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointermove_event) */\n    onpointermove: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointerout_event) */\n    onpointerout: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointerover_event) */\n    onpointerover: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/pointerup_event) */\n    onpointerup: ((this: GlobalEventHandlers, ev: PointerEvent) => any) | null;\n    /**\n     * Occurs to indicate progress while downloading media data.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/progress_event)\n     */\n    onprogress: ((this: GlobalEventHandlers, ev: ProgressEvent) => any) | null;\n    /**\n     * Occurs when the playback rate is increased or decreased.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/ratechange_event)\n     */\n    onratechange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the user resets a form.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/reset_event)\n     */\n    onreset: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/resize_event) */\n    onresize: ((this: GlobalEventHandlers, ev: UIEvent) => any) | null;\n    /**\n     * Fires when the user repositions the scroll box in the scroll bar on the object.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/scroll_event)\n     */\n    onscroll: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/scrollend_event) */\n    onscrollend: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/securitypolicyviolation_event) */\n    onsecuritypolicyviolation: ((this: GlobalEventHandlers, ev: SecurityPolicyViolationEvent) => any) | null;\n    /**\n     * Occurs when the seek operation ends.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/seeked_event)\n     */\n    onseeked: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the current playback position is moved.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/seeking_event)\n     */\n    onseeking: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Fires when the current selection changes.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/select_event)\n     */\n    onselect: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/selectionchange_event) */\n    onselectionchange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/selectstart_event) */\n    onselectstart: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSlotElement/slotchange_event) */\n    onslotchange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when the download has stopped.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/stalled_event)\n     */\n    onstalled: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/submit_event) */\n    onsubmit: ((this: GlobalEventHandlers, ev: SubmitEvent) => any) | null;\n    /**\n     * Occurs if the load operation has been intentionally halted.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/suspend_event)\n     */\n    onsuspend: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs to indicate the current playback position.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/timeupdate_event)\n     */\n    ontimeupdate: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDetailsElement/toggle_event) */\n    ontoggle: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/touchcancel_event) */\n    ontouchcancel?: ((this: GlobalEventHandlers, ev: TouchEvent) => any) | null | undefined;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/touchend_event) */\n    ontouchend?: ((this: GlobalEventHandlers, ev: TouchEvent) => any) | null | undefined;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/touchmove_event) */\n    ontouchmove?: ((this: GlobalEventHandlers, ev: TouchEvent) => any) | null | undefined;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/touchstart_event) */\n    ontouchstart?: ((this: GlobalEventHandlers, ev: TouchEvent) => any) | null | undefined;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/transitioncancel_event) */\n    ontransitioncancel: ((this: GlobalEventHandlers, ev: TransitionEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/transitionend_event) */\n    ontransitionend: ((this: GlobalEventHandlers, ev: TransitionEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/transitionrun_event) */\n    ontransitionrun: ((this: GlobalEventHandlers, ev: TransitionEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/transitionstart_event) */\n    ontransitionstart: ((this: GlobalEventHandlers, ev: TransitionEvent) => any) | null;\n    /**\n     * Occurs when the volume is changed, or playback is muted or unmuted.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/volumechange_event)\n     */\n    onvolumechange: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * Occurs when playback stops because the next frame of a video resource is not available.\n     * @param ev The event.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/waiting_event)\n     */\n    onwaiting: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * @deprecated This is a legacy alias of `onanimationend`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationend_event)\n     */\n    onwebkitanimationend: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * @deprecated This is a legacy alias of `onanimationiteration`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationiteration_event)\n     */\n    onwebkitanimationiteration: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * @deprecated This is a legacy alias of `onanimationstart`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/animationstart_event)\n     */\n    onwebkitanimationstart: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /**\n     * @deprecated This is a legacy alias of `ontransitionend`.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/transitionend_event)\n     */\n    onwebkittransitionend: ((this: GlobalEventHandlers, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/wheel_event) */\n    onwheel: ((this: GlobalEventHandlers, ev: WheelEvent) => any) | null;\n    addEventListener<K extends keyof GlobalEventHandlersEventMap>(type: K, listener: (this: GlobalEventHandlers, ev: GlobalEventHandlersEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof GlobalEventHandlersEventMap>(type: K, listener: (this: GlobalEventHandlers, ev: GlobalEventHandlersEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAllCollection) */\ninterface HTMLAllCollection {\n    /**\n     * Returns the number of elements in the collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAllCollection/length)\n     */\n    readonly length: number;\n    /**\n     * Returns the item with index index from the collection (determined by tree order).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAllCollection/item)\n     */\n    item(nameOrIndex?: string): HTMLCollection | Element | null;\n    /**\n     * Returns the item with ID or name name from the collection.\n     *\n     * If there are multiple matching items, then an HTMLCollection object containing all those elements is returned.\n     *\n     * Only button, form, iframe, input, map, meta, object, select, and textarea elements can have a name for the purpose of this method; their name is given by the value of their name attribute.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAllCollection/namedItem)\n     */\n    namedItem(name: string): HTMLCollection | Element | null;\n    [index: number]: Element;\n}\n\ndeclare var HTMLAllCollection: {\n    prototype: HTMLAllCollection;\n    new(): HTMLAllCollection;\n};\n\n/**\n * Hyperlink elements and provides special properties and methods (beyond those of the regular HTMLElement object interface that they inherit from) for manipulating the layout and presentation of such elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement)\n */\ninterface HTMLAnchorElement extends HTMLElement, HTMLHyperlinkElementUtils {\n    /**\n     * Sets or retrieves the character set used to encode the object.\n     * @deprecated\n     */\n    charset: string;\n    /**\n     * Sets or retrieves the coordinates of the object.\n     * @deprecated\n     */\n    coords: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/download) */\n    download: string;\n    /**\n     * Sets or retrieves the language code of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/hreflang)\n     */\n    hreflang: string;\n    /**\n     * Sets or retrieves the shape of the object.\n     * @deprecated\n     */\n    name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/ping) */\n    ping: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/referrerPolicy) */\n    referrerPolicy: string;\n    /**\n     * Sets or retrieves the relationship between the object and the destination of the link.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/rel)\n     */\n    rel: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/relList) */\n    readonly relList: DOMTokenList;\n    /**\n     * Sets or retrieves the relationship between the object and the destination of the link.\n     * @deprecated\n     */\n    rev: string;\n    /**\n     * Sets or retrieves the shape of the object.\n     * @deprecated\n     */\n    shape: string;\n    /**\n     * Sets or retrieves the window or frame at which to target content.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/target)\n     */\n    target: string;\n    /**\n     * Retrieves or sets the text of the object as a string.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/text)\n     */\n    text: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/type) */\n    type: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLAnchorElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLAnchorElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLAnchorElement: {\n    prototype: HTMLAnchorElement;\n    new(): HTMLAnchorElement;\n};\n\n/**\n * Provides special properties and methods (beyond those of the regular object HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of <area> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAreaElement)\n */\ninterface HTMLAreaElement extends HTMLElement, HTMLHyperlinkElementUtils {\n    /** Sets or retrieves a text alternative to the graphic. */\n    alt: string;\n    /** Sets or retrieves the coordinates of the object. */\n    coords: string;\n    download: string;\n    /**\n     * Sets or gets whether clicks in this region cause action.\n     * @deprecated\n     */\n    noHref: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAreaElement/ping) */\n    ping: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAreaElement/referrerPolicy) */\n    referrerPolicy: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAreaElement/rel) */\n    rel: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAreaElement/relList) */\n    readonly relList: DOMTokenList;\n    /** Sets or retrieves the shape of the object. */\n    shape: string;\n    /**\n     * Sets or retrieves the window or frame at which to target content.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAreaElement/target)\n     */\n    target: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLAreaElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLAreaElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLAreaElement: {\n    prototype: HTMLAreaElement;\n    new(): HTMLAreaElement;\n};\n\n/**\n * Provides access to the properties of <audio> elements, as well as methods to manipulate them. It derives from the HTMLMediaElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAudioElement)\n */\ninterface HTMLAudioElement extends HTMLMediaElement {\n    addEventListener<K extends keyof HTMLMediaElementEventMap>(type: K, listener: (this: HTMLAudioElement, ev: HTMLMediaElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLMediaElementEventMap>(type: K, listener: (this: HTMLAudioElement, ev: HTMLMediaElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLAudioElement: {\n    prototype: HTMLAudioElement;\n    new(): HTMLAudioElement;\n};\n\n/**\n * A HTML line break element (<br>). It inherits from HTMLElement.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLBRElement)\n */\ninterface HTMLBRElement extends HTMLElement {\n    /**\n     * Sets or retrieves the side on which floating objects are not to be positioned when any IHTMLBlockElement is inserted into the document.\n     * @deprecated\n     */\n    clear: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLBRElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLBRElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLBRElement: {\n    prototype: HTMLBRElement;\n    new(): HTMLBRElement;\n};\n\n/**\n * Contains the base URI for a document. This object inherits all of the properties and methods as described in the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLBaseElement)\n */\ninterface HTMLBaseElement extends HTMLElement {\n    /** Gets or sets the baseline URL on which relative links are based. */\n    href: string;\n    /**\n     * Sets or retrieves the window or frame at which to target content.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLBaseElement/target)\n     */\n    target: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLBaseElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLBaseElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLBaseElement: {\n    prototype: HTMLBaseElement;\n    new(): HTMLBaseElement;\n};\n\ninterface HTMLBodyElementEventMap extends HTMLElementEventMap, WindowEventHandlersEventMap {\n}\n\n/**\n * Provides special properties (beyond those inherited from the regular HTMLElement interface) for manipulating <body> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLBodyElement)\n */\ninterface HTMLBodyElement extends HTMLElement, WindowEventHandlers {\n    /** @deprecated */\n    aLink: string;\n    /** @deprecated */\n    background: string;\n    /** @deprecated */\n    bgColor: string;\n    /** @deprecated */\n    link: string;\n    /** @deprecated */\n    text: string;\n    /** @deprecated */\n    vLink: string;\n    addEventListener<K extends keyof HTMLBodyElementEventMap>(type: K, listener: (this: HTMLBodyElement, ev: HTMLBodyElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLBodyElementEventMap>(type: K, listener: (this: HTMLBodyElement, ev: HTMLBodyElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLBodyElement: {\n    prototype: HTMLBodyElement;\n    new(): HTMLBodyElement;\n};\n\n/**\n * Provides properties and methods (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating <button> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLButtonElement)\n */\ninterface HTMLButtonElement extends HTMLElement, PopoverInvokerElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLButtonElement/disabled) */\n    disabled: boolean;\n    /** Retrieves a reference to the form that the object is embedded in. */\n    readonly form: HTMLFormElement | null;\n    /** Overrides the action attribute (where the data on a form is sent) on the parent form element. */\n    formAction: string;\n    /** Used to override the encoding (formEnctype attribute) specified on the form element. */\n    formEnctype: string;\n    /** Overrides the submit method attribute previously specified on a form element. */\n    formMethod: string;\n    /** Overrides any validation or required attributes on a form or form elements to allow it to be submitted without validation. This can be used to create a \"save draft\"-type submit option. */\n    formNoValidate: boolean;\n    /** Overrides the target attribute on a form element. */\n    formTarget: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLButtonElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement>;\n    /** Sets or retrieves the name of the object. */\n    name: string;\n    /**\n     * Gets the classification and default behavior of the button.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLButtonElement/type)\n     */\n    type: \"submit\" | \"reset\" | \"button\";\n    /** Returns the error message that would be displayed if the user submits the form, or an empty string if no error message. It also triggers the standard error message, such as \"this is a required field\". The result is that the user sees validation messages without actually submitting. */\n    readonly validationMessage: string;\n    /** Returns a  ValidityState object that represents the validity states of an element. */\n    readonly validity: ValidityState;\n    /** Sets or retrieves the default or selected value of the control. */\n    value: string;\n    /** Returns whether an element will successfully validate based on forms validation rules and constraints. */\n    readonly willValidate: boolean;\n    /** Returns whether a form will validate when it is submitted, without having to submit it. */\n    checkValidity(): boolean;\n    reportValidity(): boolean;\n    /**\n     * Sets a custom error message that is displayed when a form is submitted.\n     * @param error Sets a custom error message that is displayed when a form is submitted.\n     */\n    setCustomValidity(error: string): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLButtonElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLButtonElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLButtonElement: {\n    prototype: HTMLButtonElement;\n    new(): HTMLButtonElement;\n};\n\n/**\n * Provides properties and methods for manipulating the layout and presentation of <canvas> elements. The HTMLCanvasElement interface also inherits the properties and methods of the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement)\n */\ninterface HTMLCanvasElement extends HTMLElement {\n    /**\n     * Gets or sets the height of a canvas element on a document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/height)\n     */\n    height: number;\n    /**\n     * Gets or sets the width of a canvas element on a document.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/width)\n     */\n    width: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/captureStream) */\n    captureStream(frameRequestRate?: number): MediaStream;\n    /**\n     * Returns an object that provides methods and properties for drawing and manipulating images and graphics on a canvas element in a document. A context object includes information about colors, line widths, fonts, and other graphic parameters that can be drawn on a canvas.\n     * @param contextId The identifier (ID) of the type of canvas to create. Internet Explorer 9 and Internet Explorer 10 support only a 2-D context using canvas.getContext(\"2d\"); IE11 Preview also supports 3-D or WebGL context using canvas.getContext(\"experimental-webgl\");\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/getContext)\n     */\n    getContext(contextId: \"2d\", options?: CanvasRenderingContext2DSettings): CanvasRenderingContext2D | null;\n    getContext(contextId: \"bitmaprenderer\", options?: ImageBitmapRenderingContextSettings): ImageBitmapRenderingContext | null;\n    getContext(contextId: \"webgl\", options?: WebGLContextAttributes): WebGLRenderingContext | null;\n    getContext(contextId: \"webgl2\", options?: WebGLContextAttributes): WebGL2RenderingContext | null;\n    getContext(contextId: string, options?: any): RenderingContext | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/toBlob) */\n    toBlob(callback: BlobCallback, type?: string, quality?: any): void;\n    /**\n     * Returns the content of the current canvas as an image that you can use as a source for another canvas or an HTML element.\n     * @param type The standard MIME type for the image format to return. If you do not specify this parameter, the default value is a PNG format image.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/toDataURL)\n     */\n    toDataURL(type?: string, quality?: any): string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCanvasElement/transferControlToOffscreen) */\n    transferControlToOffscreen(): OffscreenCanvas;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLCanvasElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLCanvasElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLCanvasElement: {\n    prototype: HTMLCanvasElement;\n    new(): HTMLCanvasElement;\n};\n\n/**\n * A generic collection (array-like object similar to arguments) of elements (in document order) and offers methods and properties for selecting from the list.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCollection)\n */\ninterface HTMLCollectionBase {\n    /**\n     * Sets or retrieves the number of objects in a collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCollection/length)\n     */\n    readonly length: number;\n    /**\n     * Retrieves an object from various collections.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCollection/item)\n     */\n    item(index: number): Element | null;\n    [index: number]: Element;\n}\n\ninterface HTMLCollection extends HTMLCollectionBase {\n    /**\n     * Retrieves a select object or an object from an options collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLCollection/namedItem)\n     */\n    namedItem(name: string): Element | null;\n}\n\ndeclare var HTMLCollection: {\n    prototype: HTMLCollection;\n    new(): HTMLCollection;\n};\n\ninterface HTMLCollectionOf<T extends Element> extends HTMLCollectionBase {\n    item(index: number): T | null;\n    namedItem(name: string): T | null;\n    [index: number]: T;\n}\n\n/**\n * Provides special properties (beyond those of the regular HTMLElement interface it also has available to it by inheritance) for manipulating definition list (<dl>) elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDListElement)\n */\ninterface HTMLDListElement extends HTMLElement {\n    /** @deprecated */\n    compact: boolean;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLDListElement: {\n    prototype: HTMLDListElement;\n    new(): HTMLDListElement;\n};\n\n/**\n * Provides special properties (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating <data> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDataElement)\n */\ninterface HTMLDataElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDataElement/value) */\n    value: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDataElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDataElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLDataElement: {\n    prototype: HTMLDataElement;\n    new(): HTMLDataElement;\n};\n\n/**\n * Provides special properties (beyond the HTMLElement object interface it also has available to it by inheritance) to manipulate <datalist> elements and their content.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDataListElement)\n */\ninterface HTMLDataListElement extends HTMLElement {\n    /** Returns an HTMLCollection of the option elements of the datalist element. */\n    readonly options: HTMLCollectionOf<HTMLOptionElement>;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDataListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDataListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLDataListElement: {\n    prototype: HTMLDataListElement;\n    new(): HTMLDataListElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDetailsElement) */\ninterface HTMLDetailsElement extends HTMLElement {\n    name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDetailsElement/open) */\n    open: boolean;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDetailsElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDetailsElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLDetailsElement: {\n    prototype: HTMLDetailsElement;\n    new(): HTMLDetailsElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement) */\ninterface HTMLDialogElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement/open) */\n    open: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement/returnValue) */\n    returnValue: string;\n    /**\n     * Closes the dialog element.\n     *\n     * The argument, if provided, provides a return value.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement/close)\n     */\n    close(returnValue?: string): void;\n    /**\n     * Displays the dialog element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement/show)\n     */\n    show(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDialogElement/showModal) */\n    showModal(): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDialogElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDialogElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLDialogElement: {\n    prototype: HTMLDialogElement;\n    new(): HTMLDialogElement;\n};\n\n/** @deprecated */\ninterface HTMLDirectoryElement extends HTMLElement {\n    /** @deprecated */\n    compact: boolean;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDirectoryElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDirectoryElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLDirectoryElement: {\n    prototype: HTMLDirectoryElement;\n    new(): HTMLDirectoryElement;\n};\n\n/**\n * Provides special properties (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating <div> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLDivElement)\n */\ninterface HTMLDivElement extends HTMLElement {\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     */\n    align: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDivElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLDivElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLDivElement: {\n    prototype: HTMLDivElement;\n    new(): HTMLDivElement;\n};\n\n/** @deprecated use Document */\ninterface HTMLDocument extends Document {\n    addEventListener<K extends keyof DocumentEventMap>(type: K, listener: (this: HTMLDocument, ev: DocumentEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof DocumentEventMap>(type: K, listener: (this: HTMLDocument, ev: DocumentEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLDocument: {\n    prototype: HTMLDocument;\n    new(): HTMLDocument;\n};\n\ninterface HTMLElementEventMap extends ElementEventMap, GlobalEventHandlersEventMap {\n}\n\n/**\n * Any HTML element. Some elements directly implement this interface, while others implement it via an interface that inherits it.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement)\n */\ninterface HTMLElement extends Element, ElementCSSInlineStyle, ElementContentEditable, GlobalEventHandlers, HTMLOrSVGElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/accessKey) */\n    accessKey: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/accessKeyLabel) */\n    readonly accessKeyLabel: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/autocapitalize) */\n    autocapitalize: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dir) */\n    dir: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/draggable) */\n    draggable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/hidden) */\n    hidden: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/inert) */\n    inert: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/innerText) */\n    innerText: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/lang) */\n    lang: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/offsetHeight) */\n    readonly offsetHeight: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/offsetLeft) */\n    readonly offsetLeft: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/offsetParent) */\n    readonly offsetParent: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/offsetTop) */\n    readonly offsetTop: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/offsetWidth) */\n    readonly offsetWidth: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/outerText) */\n    outerText: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/popover) */\n    popover: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/spellcheck) */\n    spellcheck: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/title) */\n    title: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/translate) */\n    translate: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/attachInternals) */\n    attachInternals(): ElementInternals;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/click) */\n    click(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/hidePopover) */\n    hidePopover(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/showPopover) */\n    showPopover(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/togglePopover) */\n    togglePopover(force?: boolean): boolean;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLElement: {\n    prototype: HTMLElement;\n    new(): HTMLElement;\n};\n\n/**\n * Provides special properties (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating <embed> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLEmbedElement)\n */\ninterface HTMLEmbedElement extends HTMLElement {\n    /** @deprecated */\n    align: string;\n    /**\n     * Sets or retrieves the height of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLEmbedElement/height)\n     */\n    height: string;\n    /**\n     * Sets or retrieves the name of the object.\n     * @deprecated\n     */\n    name: string;\n    /**\n     * Sets or retrieves a URL to be loaded by the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLEmbedElement/src)\n     */\n    src: string;\n    type: string;\n    /**\n     * Sets or retrieves the width of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLEmbedElement/width)\n     */\n    width: string;\n    getSVGDocument(): Document | null;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLEmbedElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLEmbedElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLEmbedElement: {\n    prototype: HTMLEmbedElement;\n    new(): HTMLEmbedElement;\n};\n\n/**\n * Provides special properties and methods (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of <fieldset> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFieldSetElement)\n */\ninterface HTMLFieldSetElement extends HTMLElement {\n    disabled: boolean;\n    /** Returns an HTMLCollection of the form controls in the element. */\n    readonly elements: HTMLCollection;\n    /** Retrieves a reference to the form that the object is embedded in. */\n    readonly form: HTMLFormElement | null;\n    name: string;\n    /** Returns the string \"fieldset\". */\n    readonly type: string;\n    /** Returns the error message that would be displayed if the user submits the form, or an empty string if no error message. It also triggers the standard error message, such as \"this is a required field\". The result is that the user sees validation messages without actually submitting. */\n    readonly validationMessage: string;\n    /** Returns a  ValidityState object that represents the validity states of an element. */\n    readonly validity: ValidityState;\n    /** Returns whether an element will successfully validate based on forms validation rules and constraints. */\n    readonly willValidate: boolean;\n    /** Returns whether a form will validate when it is submitted, without having to submit it. */\n    checkValidity(): boolean;\n    reportValidity(): boolean;\n    /**\n     * Sets a custom error message that is displayed when a form is submitted.\n     * @param error Sets a custom error message that is displayed when a form is submitted.\n     */\n    setCustomValidity(error: string): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFieldSetElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFieldSetElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLFieldSetElement: {\n    prototype: HTMLFieldSetElement;\n    new(): HTMLFieldSetElement;\n};\n\n/**\n * Implements the document object model (DOM) representation of the font element. The HTML Font Element <font> defines the font size, font face and color of text.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFontElement)\n */\ninterface HTMLFontElement extends HTMLElement {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFontElement/color)\n     */\n    color: string;\n    /**\n     * Sets or retrieves the current typeface family.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFontElement/face)\n     */\n    face: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFontElement/size)\n     */\n    size: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFontElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFontElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLFontElement: {\n    prototype: HTMLFontElement;\n    new(): HTMLFontElement;\n};\n\n/**\n * A collection of HTML form control elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormControlsCollection)\n */\ninterface HTMLFormControlsCollection extends HTMLCollectionBase {\n    /**\n     * Returns the item with ID or name name from the collection.\n     *\n     * If there are multiple matching items, then a RadioNodeList object containing all those elements is returned.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormControlsCollection/namedItem)\n     */\n    namedItem(name: string): RadioNodeList | Element | null;\n}\n\ndeclare var HTMLFormControlsCollection: {\n    prototype: HTMLFormControlsCollection;\n    new(): HTMLFormControlsCollection;\n};\n\n/**\n * A <form> element in the DOM; it allows access to and in some cases modification of aspects of the form, as well as access to its component elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement)\n */\ninterface HTMLFormElement extends HTMLElement {\n    /**\n     * Sets or retrieves a list of character encodings for input data that must be accepted by the server processing the form.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/acceptCharset)\n     */\n    acceptCharset: string;\n    /**\n     * Sets or retrieves the URL to which the form content is sent for processing.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/action)\n     */\n    action: string;\n    /** Specifies whether autocomplete is applied to an editable text field. */\n    autocomplete: AutoFillBase;\n    /**\n     * Retrieves a collection, in source order, of all controls in a given form.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/elements)\n     */\n    readonly elements: HTMLFormControlsCollection;\n    /**\n     * Sets or retrieves the MIME encoding for the form.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/encoding)\n     */\n    encoding: string;\n    /**\n     * Sets or retrieves the encoding type for the form.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/enctype)\n     */\n    enctype: string;\n    /**\n     * Sets or retrieves the number of objects in a collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/length)\n     */\n    readonly length: number;\n    /**\n     * Sets or retrieves how to send the form data to the server.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/method)\n     */\n    method: string;\n    /**\n     * Sets or retrieves the name of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/name)\n     */\n    name: string;\n    /** Designates a form that is not validated when submitted. */\n    noValidate: boolean;\n    rel: string;\n    readonly relList: DOMTokenList;\n    /**\n     * Sets or retrieves the window or frame at which to target content.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/target)\n     */\n    target: string;\n    /** Returns whether a form will validate when it is submitted, without having to submit it. */\n    checkValidity(): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/reportValidity) */\n    reportValidity(): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/requestSubmit) */\n    requestSubmit(submitter?: HTMLElement | null): void;\n    /**\n     * Fires when the user resets a form.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/reset)\n     */\n    reset(): void;\n    /**\n     * Fires when a FORM is about to be submitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFormElement/submit)\n     */\n    submit(): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFormElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFormElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n    [index: number]: Element;\n    [name: string]: any;\n}\n\ndeclare var HTMLFormElement: {\n    prototype: HTMLFormElement;\n    new(): HTMLFormElement;\n};\n\n/** @deprecated */\ninterface HTMLFrameElement extends HTMLElement {\n    /**\n     * Retrieves the document object of the page or frame.\n     * @deprecated\n     */\n    readonly contentDocument: Document | null;\n    /**\n     * Retrieves the object of the specified.\n     * @deprecated\n     */\n    readonly contentWindow: WindowProxy | null;\n    /**\n     * Sets or retrieves whether to display a border for the frame.\n     * @deprecated\n     */\n    frameBorder: string;\n    /**\n     * Sets or retrieves a URI to a long description of the object.\n     * @deprecated\n     */\n    longDesc: string;\n    /**\n     * Sets or retrieves the top and bottom margin heights before displaying the text in a frame.\n     * @deprecated\n     */\n    marginHeight: string;\n    /**\n     * Sets or retrieves the left and right margin widths before displaying the text in a frame.\n     * @deprecated\n     */\n    marginWidth: string;\n    /**\n     * Sets or retrieves the frame name.\n     * @deprecated\n     */\n    name: string;\n    /**\n     * Sets or retrieves whether the user can resize the frame.\n     * @deprecated\n     */\n    noResize: boolean;\n    /**\n     * Sets or retrieves whether the frame can be scrolled.\n     * @deprecated\n     */\n    scrolling: string;\n    /**\n     * Sets or retrieves a URL to be loaded by the object.\n     * @deprecated\n     */\n    src: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFrameElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLFrameElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLFrameElement: {\n    prototype: HTMLFrameElement;\n    new(): HTMLFrameElement;\n};\n\ninterface HTMLFrameSetElementEventMap extends HTMLElementEventMap, WindowEventHandlersEventMap {\n}\n\n/**\n * Provides special properties (beyond those of the regular HTMLElement interface they also inherit) for manipulating <frameset> elements.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLFrameSetElement)\n */\ninterface HTMLFrameSetElement extends HTMLElement, WindowEventHandlers {\n    /**\n     * Sets or retrieves the frame widths of the object.\n     * @deprecated\n     */\n    cols: string;\n    /**\n     * Sets or retrieves the frame heights of the object.\n     * @deprecated\n     */\n    rows: string;\n    addEventListener<K extends keyof HTMLFrameSetElementEventMap>(type: K, listener: (this: HTMLFrameSetElement, ev: HTMLFrameSetElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLFrameSetElementEventMap>(type: K, listener: (this: HTMLFrameSetElement, ev: HTMLFrameSetElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLFrameSetElement: {\n    prototype: HTMLFrameSetElement;\n    new(): HTMLFrameSetElement;\n};\n\n/**\n * Provides special properties (beyond those of the HTMLElement interface it also has available to it by inheritance) for manipulating <hr> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLHRElement)\n */\ninterface HTMLHRElement extends HTMLElement {\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     */\n    align: string;\n    /** @deprecated */\n    color: string;\n    /**\n     * Sets or retrieves whether the horizontal rule is drawn with 3-D shading.\n     * @deprecated\n     */\n    noShade: boolean;\n    /** @deprecated */\n    size: string;\n    /**\n     * Sets or retrieves the width of the object.\n     * @deprecated\n     */\n    width: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHRElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHRElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLHRElement: {\n    prototype: HTMLHRElement;\n    new(): HTMLHRElement;\n};\n\n/**\n * Contains the descriptive information, or metadata, for a document. This object inherits all of the properties and methods described in the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLHeadElement)\n */\ninterface HTMLHeadElement extends HTMLElement {\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHeadElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHeadElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLHeadElement: {\n    prototype: HTMLHeadElement;\n    new(): HTMLHeadElement;\n};\n\n/**\n * The different heading elements. It inherits methods and properties from the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLHeadingElement)\n */\ninterface HTMLHeadingElement extends HTMLElement {\n    /**\n     * Sets or retrieves a value that indicates the table alignment.\n     * @deprecated\n     */\n    align: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHeadingElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHeadingElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLHeadingElement: {\n    prototype: HTMLHeadingElement;\n    new(): HTMLHeadingElement;\n};\n\n/**\n * Serves as the root node for a given HTML document. This object inherits the properties and methods described in the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLHtmlElement)\n */\ninterface HTMLHtmlElement extends HTMLElement {\n    /**\n     * Sets or retrieves the DTD version that governs the current document.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLHtmlElement/version)\n     */\n    version: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHtmlElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLHtmlElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLHtmlElement: {\n    prototype: HTMLHtmlElement;\n    new(): HTMLHtmlElement;\n};\n\ninterface HTMLHyperlinkElementUtils {\n    /**\n     * Returns the hyperlink's URL's fragment (includes leading \"#\" if non-empty).\n     *\n     * Can be set, to change the URL's fragment (ignores leading \"#\").\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/hash)\n     */\n    hash: string;\n    /**\n     * Returns the hyperlink's URL's host and port (if different from the default port for the scheme).\n     *\n     * Can be set, to change the URL's host and port.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/host)\n     */\n    host: string;\n    /**\n     * Returns the hyperlink's URL's host.\n     *\n     * Can be set, to change the URL's host.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/hostname)\n     */\n    hostname: string;\n    /**\n     * Returns the hyperlink's URL.\n     *\n     * Can be set, to change the URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/href)\n     */\n    href: string;\n    toString(): string;\n    /**\n     * Returns the hyperlink's URL's origin.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/origin)\n     */\n    readonly origin: string;\n    /**\n     * Returns the hyperlink's URL's password.\n     *\n     * Can be set, to change the URL's password.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/password)\n     */\n    password: string;\n    /**\n     * Returns the hyperlink's URL's path.\n     *\n     * Can be set, to change the URL's path.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/pathname)\n     */\n    pathname: string;\n    /**\n     * Returns the hyperlink's URL's port.\n     *\n     * Can be set, to change the URL's port.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/port)\n     */\n    port: string;\n    /**\n     * Returns the hyperlink's URL's scheme.\n     *\n     * Can be set, to change the URL's scheme.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/protocol)\n     */\n    protocol: string;\n    /**\n     * Returns the hyperlink's URL's query (includes leading \"?\" if non-empty).\n     *\n     * Can be set, to change the URL's query (ignores leading \"?\").\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/search)\n     */\n    search: string;\n    /**\n     * Returns the hyperlink's URL's username.\n     *\n     * Can be set, to change the URL's username.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLAnchorElement/username)\n     */\n    username: string;\n}\n\n/**\n * Provides special properties and methods (beyond those of the HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of inline frame elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement)\n */\ninterface HTMLIFrameElement extends HTMLElement {\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     */\n    align: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/allow) */\n    allow: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/allowFullscreen) */\n    allowFullscreen: boolean;\n    /**\n     * Retrieves the document object of the page or frame.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/contentDocument)\n     */\n    readonly contentDocument: Document | null;\n    /**\n     * Retrieves the object of the specified.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/contentWindow)\n     */\n    readonly contentWindow: WindowProxy | null;\n    /**\n     * Sets or retrieves whether to display a border for the frame.\n     * @deprecated\n     */\n    frameBorder: string;\n    /**\n     * Sets or retrieves the height of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/height)\n     */\n    height: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/loading) */\n    loading: string;\n    /**\n     * Sets or retrieves a URI to a long description of the object.\n     * @deprecated\n     */\n    longDesc: string;\n    /**\n     * Sets or retrieves the top and bottom margin heights before displaying the text in a frame.\n     * @deprecated\n     */\n    marginHeight: string;\n    /**\n     * Sets or retrieves the left and right margin widths before displaying the text in a frame.\n     * @deprecated\n     */\n    marginWidth: string;\n    /**\n     * Sets or retrieves the frame name.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/name)\n     */\n    name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/referrerPolicy) */\n    referrerPolicy: ReferrerPolicy;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/sandbox) */\n    readonly sandbox: DOMTokenList;\n    /**\n     * Sets or retrieves whether the frame can be scrolled.\n     * @deprecated\n     */\n    scrolling: string;\n    /**\n     * Sets or retrieves a URL to be loaded by the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/src)\n     */\n    src: string;\n    /**\n     * Sets or retrives the content of the page that is to contain.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/srcdoc)\n     */\n    srcdoc: string;\n    /**\n     * Sets or retrieves the width of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLIFrameElement/width)\n     */\n    width: string;\n    getSVGDocument(): Document | null;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLIFrameElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLIFrameElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLIFrameElement: {\n    prototype: HTMLIFrameElement;\n    new(): HTMLIFrameElement;\n};\n\n/**\n * Provides special properties and methods for manipulating <img> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement)\n */\ninterface HTMLImageElement extends HTMLElement {\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/align)\n     */\n    align: string;\n    /**\n     * Sets or retrieves a text alternative to the graphic.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/alt)\n     */\n    alt: string;\n    /**\n     * Specifies the properties of a border drawn around an object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/border)\n     */\n    border: string;\n    /**\n     * Retrieves whether the object is fully loaded.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/complete)\n     */\n    readonly complete: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/crossOrigin) */\n    crossOrigin: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/currentSrc) */\n    readonly currentSrc: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/decoding) */\n    decoding: \"async\" | \"sync\" | \"auto\";\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/fetchPriority) */\n    fetchPriority: string;\n    /**\n     * Sets or retrieves the height of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/height)\n     */\n    height: number;\n    /**\n     * Sets or retrieves the width of the border to draw around the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/hspace)\n     */\n    hspace: number;\n    /**\n     * Sets or retrieves whether the image is a server-side image map.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/isMap)\n     */\n    isMap: boolean;\n    /**\n     * Sets or retrieves the policy for loading image elements that are outside the viewport.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/loading)\n     */\n    loading: \"eager\" | \"lazy\";\n    /**\n     * Sets or retrieves a Uniform Resource Identifier (URI) to a long description of the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/longDesc)\n     */\n    longDesc: string;\n    /** @deprecated */\n    lowsrc: string;\n    /**\n     * Sets or retrieves the name of the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/name)\n     */\n    name: string;\n    /**\n     * The original height of the image resource before sizing.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/naturalHeight)\n     */\n    readonly naturalHeight: number;\n    /**\n     * The original width of the image resource before sizing.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/naturalWidth)\n     */\n    readonly naturalWidth: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/referrerPolicy) */\n    referrerPolicy: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/sizes) */\n    sizes: string;\n    /**\n     * The address or URL of the a media resource that is to be considered.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/src)\n     */\n    src: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/srcset) */\n    srcset: string;\n    /**\n     * Sets or retrieves the URL, often with a bookmark extension (#name), to use as a client-side image map.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/useMap)\n     */\n    useMap: string;\n    /**\n     * Sets or retrieves the vertical margin for the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/vspace)\n     */\n    vspace: number;\n    /**\n     * Sets or retrieves the width of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/width)\n     */\n    width: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/x) */\n    readonly x: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/y) */\n    readonly y: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLImageElement/decode) */\n    decode(): Promise<void>;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLImageElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLImageElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLImageElement: {\n    prototype: HTMLImageElement;\n    new(): HTMLImageElement;\n};\n\n/**\n * Provides special properties and methods for manipulating the options, layout, and presentation of <input> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement)\n */\ninterface HTMLInputElement extends HTMLElement, PopoverInvokerElement {\n    /** Sets or retrieves a comma-separated list of content types. */\n    accept: string;\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     */\n    align: string;\n    /** Sets or retrieves a text alternative to the graphic. */\n    alt: string;\n    /** Specifies whether autocomplete is applied to an editable text field. */\n    autocomplete: AutoFill;\n    capture: string;\n    /** Sets or retrieves the state of the check box or radio button. */\n    checked: boolean;\n    /** Sets or retrieves the state of the check box or radio button. */\n    defaultChecked: boolean;\n    /** Sets or retrieves the initial contents of the object. */\n    defaultValue: string;\n    dirName: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/disabled) */\n    disabled: boolean;\n    /**\n     * Returns a FileList object on a file type input object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/files)\n     */\n    files: FileList | null;\n    /** Retrieves a reference to the form that the object is embedded in. */\n    readonly form: HTMLFormElement | null;\n    /** Overrides the action attribute (where the data on a form is sent) on the parent form element. */\n    formAction: string;\n    /** Used to override the encoding (formEnctype attribute) specified on the form element. */\n    formEnctype: string;\n    /** Overrides the submit method attribute previously specified on a form element. */\n    formMethod: string;\n    /** Overrides any validation or required attributes on a form or form elements to allow it to be submitted without validation. This can be used to create a \"save draft\"-type submit option. */\n    formNoValidate: boolean;\n    /** Overrides the target attribute on a form element. */\n    formTarget: string;\n    /** Sets or retrieves the height of the object. */\n    height: number;\n    /** When set, overrides the rendering of checkbox controls so that the current value is not visible. */\n    indeterminate: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement> | null;\n    /** Specifies the ID of a pre-defined datalist of options for an input element. */\n    readonly list: HTMLDataListElement | null;\n    /** Defines the maximum acceptable value for an input element with type=\"number\".When used with the min and step attributes, lets you control the range and increment (such as only even numbers) that the user can enter into an input field. */\n    max: string;\n    /** Sets or retrieves the maximum number of characters that the user can enter in a text control. */\n    maxLength: number;\n    /** Defines the minimum acceptable value for an input element with type=\"number\". When used with the max and step attributes, lets you control the range and increment (such as even numbers only) that the user can enter into an input field. */\n    min: string;\n    minLength: number;\n    /**\n     * Sets or retrieves the Boolean value indicating whether multiple items can be selected from a list.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/multiple)\n     */\n    multiple: boolean;\n    /** Sets or retrieves the name of the object. */\n    name: string;\n    /** Gets or sets a string containing a regular expression that the user's input must match. */\n    pattern: string;\n    /** Gets or sets a text string that is displayed in an input field as a hint or prompt to users as the format or type of information they need to enter.The text appears in an input field until the user puts focus on the field. */\n    placeholder: string;\n    readOnly: boolean;\n    /** When present, marks an element that can't be submitted without a value. */\n    required: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/selectionDirection) */\n    selectionDirection: \"forward\" | \"backward\" | \"none\" | null;\n    /**\n     * Gets or sets the end position or offset of a text selection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/selectionEnd)\n     */\n    selectionEnd: number | null;\n    /**\n     * Gets or sets the starting position or offset of a text selection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/selectionStart)\n     */\n    selectionStart: number | null;\n    size: number;\n    /** The address or URL of the a media resource that is to be considered. */\n    src: string;\n    /** Defines an increment or jump between values that you want to allow the user to enter. When used with the max and min attributes, lets you control the range and increment (for example, allow only even numbers) that the user can enter into an input field. */\n    step: string;\n    /**\n     * Returns the content type of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/type)\n     */\n    type: string;\n    /**\n     * Sets or retrieves the URL, often with a bookmark extension (#name), to use as a client-side image map.\n     * @deprecated\n     */\n    useMap: string;\n    /** Returns the error message that would be displayed if the user submits the form, or an empty string if no error message. It also triggers the standard error message, such as \"this is a required field\". The result is that the user sees validation messages without actually submitting. */\n    readonly validationMessage: string;\n    /** Returns a  ValidityState object that represents the validity states of an element. */\n    readonly validity: ValidityState;\n    /** Returns the value of the data at the cursor's current position. */\n    value: string;\n    /** Returns a Date object representing the form control's value, if applicable; otherwise, returns null. Can be set, to change the value. Throws an \"InvalidStateError\" DOMException if the control isn't date- or time-based. */\n    valueAsDate: Date | null;\n    /** Returns the input field value as a number. */\n    valueAsNumber: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/webkitEntries) */\n    readonly webkitEntries: ReadonlyArray<FileSystemEntry>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/webkitdirectory) */\n    webkitdirectory: boolean;\n    /** Sets or retrieves the width of the object. */\n    width: number;\n    /** Returns whether an element will successfully validate based on forms validation rules and constraints. */\n    readonly willValidate: boolean;\n    /**\n     * Returns whether a form will validate when it is submitted, without having to submit it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/checkValidity)\n     */\n    checkValidity(): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/reportValidity) */\n    reportValidity(): boolean;\n    /**\n     * Makes the selection equal to the current object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/select)\n     */\n    select(): void;\n    /**\n     * Sets a custom error message that is displayed when a form is submitted.\n     * @param error Sets a custom error message that is displayed when a form is submitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/setCustomValidity)\n     */\n    setCustomValidity(error: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/setRangeText) */\n    setRangeText(replacement: string): void;\n    setRangeText(replacement: string, start: number, end: number, selectionMode?: SelectionMode): void;\n    /**\n     * Sets the start and end positions of a selection in a text field.\n     * @param start The offset into the text field for the start of the selection.\n     * @param end The offset into the text field for the end of the selection.\n     * @param direction The direction in which the selection is performed.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/setSelectionRange)\n     */\n    setSelectionRange(start: number | null, end: number | null, direction?: \"forward\" | \"backward\" | \"none\"): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/showPicker) */\n    showPicker(): void;\n    /**\n     * Decrements a range input control's value by the value given by the Step attribute. If the optional parameter is used, it will decrement the input control's step value multiplied by the parameter's value.\n     * @param n Value to decrement the value by.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/stepDown)\n     */\n    stepDown(n?: number): void;\n    /**\n     * Increments a range input control's value by the value given by the Step attribute. If the optional parameter is used, will increment the input control's value by that value.\n     * @param n Value to increment the value by.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLInputElement/stepUp)\n     */\n    stepUp(n?: number): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLInputElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLInputElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLInputElement: {\n    prototype: HTMLInputElement;\n    new(): HTMLInputElement;\n};\n\n/**\n * Exposes specific properties and methods (beyond those defined by regular HTMLElement interface it also has available to it by inheritance) for manipulating list elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLIElement)\n */\ninterface HTMLLIElement extends HTMLElement {\n    /** @deprecated */\n    type: string;\n    /** Sets or retrieves the value of a list item. */\n    value: number;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLIElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLIElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLLIElement: {\n    prototype: HTMLLIElement;\n    new(): HTMLLIElement;\n};\n\n/**\n * Gives access to properties specific to <label> elements. It inherits methods and properties from the base HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLabelElement)\n */\ninterface HTMLLabelElement extends HTMLElement {\n    /**\n     * Returns the form control that is associated with this element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLabelElement/control)\n     */\n    readonly control: HTMLElement | null;\n    /**\n     * Retrieves a reference to the form that the object is embedded in.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLabelElement/form)\n     */\n    readonly form: HTMLFormElement | null;\n    /**\n     * Sets or retrieves the object to which the given label object is assigned.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLabelElement/htmlFor)\n     */\n    htmlFor: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLabelElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLabelElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLLabelElement: {\n    prototype: HTMLLabelElement;\n    new(): HTMLLabelElement;\n};\n\n/**\n * The HTMLLegendElement is an interface allowing to access properties of the <legend> elements. It inherits properties and methods from the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLegendElement)\n */\ninterface HTMLLegendElement extends HTMLElement {\n    /** @deprecated */\n    align: string;\n    /** Retrieves a reference to the form that the object is embedded in. */\n    readonly form: HTMLFormElement | null;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLegendElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLegendElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLLegendElement: {\n    prototype: HTMLLegendElement;\n    new(): HTMLLegendElement;\n};\n\n/**\n * Reference information for external resources and the relationship of those resources to a document and vice-versa. This object inherits all of the properties and methods of the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement)\n */\ninterface HTMLLinkElement extends HTMLElement, LinkStyle {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/as) */\n    as: string;\n    /**\n     * Sets or retrieves the character set used to encode the object.\n     * @deprecated\n     */\n    charset: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/crossOrigin) */\n    crossOrigin: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/disabled) */\n    disabled: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/fetchPriority) */\n    fetchPriority: string;\n    /** Sets or retrieves a destination URL or an anchor point. */\n    href: string;\n    /**\n     * Sets or retrieves the language code of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/hreflang)\n     */\n    hreflang: string;\n    imageSizes: string;\n    imageSrcset: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/integrity) */\n    integrity: string;\n    /** Sets or retrieves the media type. */\n    media: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/referrerPolicy) */\n    referrerPolicy: string;\n    /**\n     * Sets or retrieves the relationship between the object and the destination of the link.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/rel)\n     */\n    rel: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/relList) */\n    readonly relList: DOMTokenList;\n    /**\n     * Sets or retrieves the relationship between the object and the destination of the link.\n     * @deprecated\n     */\n    rev: string;\n    readonly sizes: DOMTokenList;\n    /**\n     * Sets or retrieves the window or frame at which to target content.\n     * @deprecated\n     */\n    target: string;\n    /**\n     * Sets or retrieves the MIME type of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/type)\n     */\n    type: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLinkElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLLinkElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLLinkElement: {\n    prototype: HTMLLinkElement;\n    new(): HTMLLinkElement;\n};\n\n/**\n * Provides special properties and methods (beyond those of the regular object HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of map elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMapElement)\n */\ninterface HTMLMapElement extends HTMLElement {\n    /** Retrieves a collection of the area objects defined for the given map object. */\n    readonly areas: HTMLCollection;\n    /**\n     * Sets or retrieves the name of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMapElement/name)\n     */\n    name: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMapElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMapElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLMapElement: {\n    prototype: HTMLMapElement;\n    new(): HTMLMapElement;\n};\n\n/**\n * Provides methods to manipulate <marquee> elements.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMarqueeElement)\n */\ninterface HTMLMarqueeElement extends HTMLElement {\n    /** @deprecated */\n    behavior: string;\n    /** @deprecated */\n    bgColor: string;\n    /** @deprecated */\n    direction: string;\n    /** @deprecated */\n    height: string;\n    /** @deprecated */\n    hspace: number;\n    /** @deprecated */\n    loop: number;\n    /** @deprecated */\n    scrollAmount: number;\n    /** @deprecated */\n    scrollDelay: number;\n    /** @deprecated */\n    trueSpeed: boolean;\n    /** @deprecated */\n    vspace: number;\n    /** @deprecated */\n    width: string;\n    /** @deprecated */\n    start(): void;\n    /** @deprecated */\n    stop(): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMarqueeElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMarqueeElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLMarqueeElement: {\n    prototype: HTMLMarqueeElement;\n    new(): HTMLMarqueeElement;\n};\n\ninterface HTMLMediaElementEventMap extends HTMLElementEventMap {\n    \"encrypted\": MediaEncryptedEvent;\n    \"waitingforkey\": Event;\n}\n\n/**\n * Adds to HTMLElement the properties and methods needed to support basic media-related capabilities that are common to audio and video.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement)\n */\ninterface HTMLMediaElement extends HTMLElement {\n    /**\n     * Gets or sets a value that indicates whether to start playing the media automatically.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/autoplay)\n     */\n    autoplay: boolean;\n    /**\n     * Gets a collection of buffered time ranges.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/buffered)\n     */\n    readonly buffered: TimeRanges;\n    /**\n     * Gets or sets a flag that indicates whether the client provides a set of controls for the media (in case the developer does not include controls for the player).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/controls)\n     */\n    controls: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/crossOrigin) */\n    crossOrigin: string | null;\n    /**\n     * Gets the address or URL of the current media resource that is selected by IHTMLMediaElement.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/currentSrc)\n     */\n    readonly currentSrc: string;\n    /**\n     * Gets or sets the current playback position, in seconds.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/currentTime)\n     */\n    currentTime: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/defaultMuted) */\n    defaultMuted: boolean;\n    /**\n     * Gets or sets the default playback rate when the user is not using fast forward or reverse for a video or audio resource.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/defaultPlaybackRate)\n     */\n    defaultPlaybackRate: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/disableRemotePlayback) */\n    disableRemotePlayback: boolean;\n    /**\n     * Returns the duration in seconds of the current media resource. A NaN value is returned if duration is not available, or Infinity if the media resource is streaming.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/duration)\n     */\n    readonly duration: number;\n    /**\n     * Gets information about whether the playback has ended or not.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/ended)\n     */\n    readonly ended: boolean;\n    /**\n     * Returns an object representing the current error state of the audio or video element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/error)\n     */\n    readonly error: MediaError | null;\n    /**\n     * Gets or sets a flag to specify whether playback should restart after it completes.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/loop)\n     */\n    loop: boolean;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/mediaKeys)\n     */\n    readonly mediaKeys: MediaKeys | null;\n    /**\n     * Gets or sets a flag that indicates whether the audio (either audio or the audio track on video media) is muted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/muted)\n     */\n    muted: boolean;\n    /**\n     * Gets the current network activity for the element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/networkState)\n     */\n    readonly networkState: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/encrypted_event) */\n    onencrypted: ((this: HTMLMediaElement, ev: MediaEncryptedEvent) => any) | null;\n    onwaitingforkey: ((this: HTMLMediaElement, ev: Event) => any) | null;\n    /**\n     * Gets a flag that specifies whether playback is paused.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/paused)\n     */\n    readonly paused: boolean;\n    /**\n     * Gets or sets the current rate of speed for the media resource to play. This speed is expressed as a multiple of the normal speed of the media resource.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/playbackRate)\n     */\n    playbackRate: number;\n    /** Gets TimeRanges for the current media resource that has been played. */\n    readonly played: TimeRanges;\n    /**\n     * Gets or sets a value indicating what data should be preloaded, if any.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/preload)\n     */\n    preload: \"none\" | \"metadata\" | \"auto\" | \"\";\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/preservesPitch) */\n    preservesPitch: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/readyState) */\n    readonly readyState: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/remote) */\n    readonly remote: RemotePlayback;\n    /**\n     * Returns a TimeRanges object that represents the ranges of the current media resource that can be seeked.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/seekable)\n     */\n    readonly seekable: TimeRanges;\n    /** Gets a flag that indicates whether the client is currently moving to a new playback position in the media resource. */\n    readonly seeking: boolean;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/sinkId)\n     */\n    readonly sinkId: string;\n    /**\n     * The address or URL of the a media resource that is to be considered.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/src)\n     */\n    src: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/srcObject) */\n    srcObject: MediaProvider | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/textTracks) */\n    readonly textTracks: TextTrackList;\n    /**\n     * Gets or sets the volume level for audio portions of the media element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/volume)\n     */\n    volume: number;\n    addTextTrack(kind: TextTrackKind, label?: string, language?: string): TextTrack;\n    /**\n     * Returns a string that specifies whether the client can play a given media resource type.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/canPlayType)\n     */\n    canPlayType(type: string): CanPlayTypeResult;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/fastSeek) */\n    fastSeek(time: number): void;\n    /**\n     * Resets the audio or video object and loads a new media resource.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/load)\n     */\n    load(): void;\n    /**\n     * Pauses the current playback and sets paused to TRUE.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/pause)\n     */\n    pause(): void;\n    /**\n     * Loads and starts playback of a media resource.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/play)\n     */\n    play(): Promise<void>;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/setMediaKeys)\n     */\n    setMediaKeys(mediaKeys: MediaKeys | null): Promise<void>;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMediaElement/setSinkId)\n     */\n    setSinkId(sinkId: string): Promise<void>;\n    readonly NETWORK_EMPTY: 0;\n    readonly NETWORK_IDLE: 1;\n    readonly NETWORK_LOADING: 2;\n    readonly NETWORK_NO_SOURCE: 3;\n    readonly HAVE_NOTHING: 0;\n    readonly HAVE_METADATA: 1;\n    readonly HAVE_CURRENT_DATA: 2;\n    readonly HAVE_FUTURE_DATA: 3;\n    readonly HAVE_ENOUGH_DATA: 4;\n    addEventListener<K extends keyof HTMLMediaElementEventMap>(type: K, listener: (this: HTMLMediaElement, ev: HTMLMediaElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLMediaElementEventMap>(type: K, listener: (this: HTMLMediaElement, ev: HTMLMediaElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLMediaElement: {\n    prototype: HTMLMediaElement;\n    new(): HTMLMediaElement;\n    readonly NETWORK_EMPTY: 0;\n    readonly NETWORK_IDLE: 1;\n    readonly NETWORK_LOADING: 2;\n    readonly NETWORK_NO_SOURCE: 3;\n    readonly HAVE_NOTHING: 0;\n    readonly HAVE_METADATA: 1;\n    readonly HAVE_CURRENT_DATA: 2;\n    readonly HAVE_FUTURE_DATA: 3;\n    readonly HAVE_ENOUGH_DATA: 4;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMenuElement) */\ninterface HTMLMenuElement extends HTMLElement {\n    /** @deprecated */\n    compact: boolean;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMenuElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMenuElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLMenuElement: {\n    prototype: HTMLMenuElement;\n    new(): HTMLMenuElement;\n};\n\n/**\n * Contains descriptive metadata about a document. It inherits all of the properties and methods described in the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMetaElement)\n */\ninterface HTMLMetaElement extends HTMLElement {\n    /**\n     * Gets or sets meta-information to associate with httpEquiv or name.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMetaElement/content)\n     */\n    content: string;\n    /**\n     * Gets or sets information used to bind the value of a content attribute of a meta element to an HTTP response header.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMetaElement/httpEquiv)\n     */\n    httpEquiv: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMetaElement/media) */\n    media: string;\n    /**\n     * Sets or retrieves the value specified in the content attribute of the meta object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMetaElement/name)\n     */\n    name: string;\n    /**\n     * Sets or retrieves a scheme to be used in interpreting the value of a property specified for the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMetaElement/scheme)\n     */\n    scheme: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMetaElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMetaElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLMetaElement: {\n    prototype: HTMLMetaElement;\n    new(): HTMLMetaElement;\n};\n\n/**\n * The HTML <meter> elements expose the HTMLMeterElement interface, which provides special properties and methods (beyond the HTMLElement object interface they also have available to them by inheritance) for manipulating the layout and presentation of <meter> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMeterElement)\n */\ninterface HTMLMeterElement extends HTMLElement {\n    high: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLMeterElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement>;\n    low: number;\n    max: number;\n    min: number;\n    optimum: number;\n    value: number;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMeterElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLMeterElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLMeterElement: {\n    prototype: HTMLMeterElement;\n    new(): HTMLMeterElement;\n};\n\n/**\n * Provides special properties (beyond the regular methods and properties available through the HTMLElement interface they also have available to them by inheritance) for manipulating modification elements, that is <del> and <ins>.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLModElement)\n */\ninterface HTMLModElement extends HTMLElement {\n    /** Sets or retrieves reference information about the object. */\n    cite: string;\n    /** Sets or retrieves the date and time of a modification to the object. */\n    dateTime: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLModElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLModElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLModElement: {\n    prototype: HTMLModElement;\n    new(): HTMLModElement;\n};\n\n/**\n * Provides special properties (beyond those defined on the regular HTMLElement interface it also has available to it by inheritance) for manipulating ordered list elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOListElement)\n */\ninterface HTMLOListElement extends HTMLElement {\n    /** @deprecated */\n    compact: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOListElement/reversed) */\n    reversed: boolean;\n    /**\n     * The starting number.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOListElement/start)\n     */\n    start: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOListElement/type) */\n    type: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLOListElement: {\n    prototype: HTMLOListElement;\n    new(): HTMLOListElement;\n};\n\n/**\n * Provides special properties and methods (beyond those on the HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of <object> element, representing external resources.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement)\n */\ninterface HTMLObjectElement extends HTMLElement {\n    /** @deprecated */\n    align: string;\n    /**\n     * Sets or retrieves a character string that can be used to implement your own archive functionality for the object.\n     * @deprecated\n     */\n    archive: string;\n    /** @deprecated */\n    border: string;\n    /**\n     * Sets or retrieves the URL of the file containing the compiled Java class.\n     * @deprecated\n     */\n    code: string;\n    /**\n     * Sets or retrieves the URL of the component.\n     * @deprecated\n     */\n    codeBase: string;\n    /**\n     * Sets or retrieves the Internet media type for the code associated with the object.\n     * @deprecated\n     */\n    codeType: string;\n    /**\n     * Retrieves the document object of the page or frame.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/contentDocument)\n     */\n    readonly contentDocument: Document | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/contentWindow) */\n    readonly contentWindow: WindowProxy | null;\n    /**\n     * Sets or retrieves the URL that references the data of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/data)\n     */\n    data: string;\n    /** @deprecated */\n    declare: boolean;\n    /**\n     * Retrieves a reference to the form that the object is embedded in.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/form)\n     */\n    readonly form: HTMLFormElement | null;\n    /**\n     * Sets or retrieves the height of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/height)\n     */\n    height: string;\n    /** @deprecated */\n    hspace: number;\n    /**\n     * Sets or retrieves the name of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/name)\n     */\n    name: string;\n    /**\n     * Sets or retrieves a message to be displayed while an object is loading.\n     * @deprecated\n     */\n    standby: string;\n    /**\n     * Sets or retrieves the MIME type of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/type)\n     */\n    type: string;\n    /**\n     * Sets or retrieves the URL, often with a bookmark extension (#name), to use as a client-side image map.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/useMap)\n     */\n    useMap: string;\n    /**\n     * Returns the error message that would be displayed if the user submits the form, or an empty string if no error message. It also triggers the standard error message, such as \"this is a required field\". The result is that the user sees validation messages without actually submitting.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/validationMessage)\n     */\n    readonly validationMessage: string;\n    /**\n     * Returns a  ValidityState object that represents the validity states of an element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/validity)\n     */\n    readonly validity: ValidityState;\n    /** @deprecated */\n    vspace: number;\n    /**\n     * Sets or retrieves the width of the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/width)\n     */\n    width: string;\n    /**\n     * Returns whether an element will successfully validate based on forms validation rules and constraints.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/willValidate)\n     */\n    readonly willValidate: boolean;\n    /**\n     * Returns whether a form will validate when it is submitted, without having to submit it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/checkValidity)\n     */\n    checkValidity(): boolean;\n    getSVGDocument(): Document | null;\n    reportValidity(): boolean;\n    /**\n     * Sets a custom error message that is displayed when a form is submitted.\n     * @param error Sets a custom error message that is displayed when a form is submitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLObjectElement/setCustomValidity)\n     */\n    setCustomValidity(error: string): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLObjectElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLObjectElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLObjectElement: {\n    prototype: HTMLObjectElement;\n    new(): HTMLObjectElement;\n};\n\n/**\n * Provides special properties and methods (beyond the regular HTMLElement object interface they also have available to them by inheritance) for manipulating the layout and presentation of <optgroup> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOptGroupElement)\n */\ninterface HTMLOptGroupElement extends HTMLElement {\n    disabled: boolean;\n    /** Sets or retrieves a value that you can use to implement your own label functionality for the object. */\n    label: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOptGroupElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOptGroupElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLOptGroupElement: {\n    prototype: HTMLOptGroupElement;\n    new(): HTMLOptGroupElement;\n};\n\n/**\n * <option> elements and inherits all classes and methods of the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOptionElement)\n */\ninterface HTMLOptionElement extends HTMLElement {\n    /** Sets or retrieves the status of an option. */\n    defaultSelected: boolean;\n    disabled: boolean;\n    /** Retrieves a reference to the form that the object is embedded in. */\n    readonly form: HTMLFormElement | null;\n    /** Sets or retrieves the ordinal position of an option in a list box. */\n    readonly index: number;\n    /** Sets or retrieves a value that you can use to implement your own label functionality for the object. */\n    label: string;\n    /** Sets or retrieves whether the option in the list box is the default item. */\n    selected: boolean;\n    /** Sets or retrieves the text string specified by the option tag. */\n    text: string;\n    /** Sets or retrieves the value which is returned to the server when the form control is submitted. */\n    value: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOptionElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOptionElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLOptionElement: {\n    prototype: HTMLOptionElement;\n    new(): HTMLOptionElement;\n};\n\n/**\n * HTMLOptionsCollection is an interface representing a collection of HTML option elements (in document order) and offers methods and properties for traversing the list as well as optionally altering its items. This type is returned solely by the \"options\" property of select.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOptionsCollection)\n */\ninterface HTMLOptionsCollection extends HTMLCollectionOf<HTMLOptionElement> {\n    /**\n     * Returns the number of elements in the collection.\n     *\n     * When set to a smaller number, truncates the number of option elements in the corresponding container.\n     *\n     * When set to a greater number, adds new blank option elements to that container.\n     */\n    length: number;\n    /**\n     * Returns the index of the first selected item, if any, or −1 if there is no selected item.\n     *\n     * Can be set, to change the selection.\n     */\n    selectedIndex: number;\n    /**\n     * Inserts element before the node given by before.\n     *\n     * The before argument can be a number, in which case element is inserted before the item with that number, or an element from the collection, in which case element is inserted before that element.\n     *\n     * If before is omitted, null, or a number out of range, then element will be added at the end of the list.\n     *\n     * This method will throw a \"HierarchyRequestError\" DOMException if element is an ancestor of the element into which it is to be inserted.\n     */\n    add(element: HTMLOptionElement | HTMLOptGroupElement, before?: HTMLElement | number | null): void;\n    /** Removes the item with index index from the collection. */\n    remove(index: number): void;\n}\n\ndeclare var HTMLOptionsCollection: {\n    prototype: HTMLOptionsCollection;\n    new(): HTMLOptionsCollection;\n};\n\ninterface HTMLOrSVGElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/autofocus) */\n    autofocus: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/dataset) */\n    readonly dataset: DOMStringMap;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/nonce) */\n    nonce?: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/tabIndex) */\n    tabIndex: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/blur) */\n    blur(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLElement/focus) */\n    focus(options?: FocusOptions): void;\n}\n\n/**\n * Provides properties and methods (beyond those inherited from HTMLElement) for manipulating the layout and presentation of <output> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOutputElement)\n */\ninterface HTMLOutputElement extends HTMLElement {\n    defaultValue: string;\n    readonly form: HTMLFormElement | null;\n    readonly htmlFor: DOMTokenList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLOutputElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement>;\n    name: string;\n    /** Returns the string \"output\". */\n    readonly type: string;\n    readonly validationMessage: string;\n    readonly validity: ValidityState;\n    /**\n     * Returns the element's current value.\n     *\n     * Can be set, to change the value.\n     */\n    value: string;\n    readonly willValidate: boolean;\n    checkValidity(): boolean;\n    reportValidity(): boolean;\n    setCustomValidity(error: string): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOutputElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLOutputElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLOutputElement: {\n    prototype: HTMLOutputElement;\n    new(): HTMLOutputElement;\n};\n\n/**\n * Provides special properties (beyond those of the regular HTMLElement object interface it inherits) for manipulating <p> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLParagraphElement)\n */\ninterface HTMLParagraphElement extends HTMLElement {\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     */\n    align: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLParagraphElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLParagraphElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLParagraphElement: {\n    prototype: HTMLParagraphElement;\n    new(): HTMLParagraphElement;\n};\n\n/**\n * Provides special properties (beyond those of the regular HTMLElement object interface it inherits) for manipulating <param> elements, representing a pair of a key and a value that acts as a parameter for an <object> element.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLParamElement)\n */\ninterface HTMLParamElement extends HTMLElement {\n    /**\n     * Sets or retrieves the name of an input parameter for an element.\n     * @deprecated\n     */\n    name: string;\n    /**\n     * Sets or retrieves the content type of the resource designated by the value attribute.\n     * @deprecated\n     */\n    type: string;\n    /**\n     * Sets or retrieves the value of an input parameter for an element.\n     * @deprecated\n     */\n    value: string;\n    /**\n     * Sets or retrieves the data type of the value attribute.\n     * @deprecated\n     */\n    valueType: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLParamElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLParamElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var HTMLParamElement: {\n    prototype: HTMLParamElement;\n    new(): HTMLParamElement;\n};\n\n/**\n * A <picture> HTML element. It doesn't implement specific properties or methods.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLPictureElement)\n */\ninterface HTMLPictureElement extends HTMLElement {\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLPictureElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLPictureElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLPictureElement: {\n    prototype: HTMLPictureElement;\n    new(): HTMLPictureElement;\n};\n\n/**\n * Exposes specific properties and methods (beyond those of the HTMLElement interface it also has available to it by inheritance) for manipulating a block of preformatted text (<pre>).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLPreElement)\n */\ninterface HTMLPreElement extends HTMLElement {\n    /**\n     * Sets or gets a value that you can use to implement your own width functionality for the object.\n     * @deprecated\n     */\n    width: number;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLPreElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLPreElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLPreElement: {\n    prototype: HTMLPreElement;\n    new(): HTMLPreElement;\n};\n\n/**\n * Provides special properties and methods (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of <progress> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLProgressElement)\n */\ninterface HTMLProgressElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLProgressElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement>;\n    /**\n     * Defines the maximum, or \"done\" value for a progress element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLProgressElement/max)\n     */\n    max: number;\n    /**\n     * Returns the quotient of value/max when the value attribute is set (determinate progress bar), or -1 when the value attribute is missing (indeterminate progress bar).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLProgressElement/position)\n     */\n    readonly position: number;\n    /**\n     * Sets or gets the current value of a progress element. The value must be a non-negative number between 0 and the max value.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLProgressElement/value)\n     */\n    value: number;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLProgressElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLProgressElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLProgressElement: {\n    prototype: HTMLProgressElement;\n    new(): HTMLProgressElement;\n};\n\n/**\n * Provides special properties and methods (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating quoting elements, like <blockquote> and <q>, but not the <cite> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLQuoteElement)\n */\ninterface HTMLQuoteElement extends HTMLElement {\n    /** Sets or retrieves reference information about the object. */\n    cite: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLQuoteElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLQuoteElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLQuoteElement: {\n    prototype: HTMLQuoteElement;\n    new(): HTMLQuoteElement;\n};\n\n/**\n * HTML <script> elements expose the HTMLScriptElement interface, which provides special properties and methods for manipulating the behavior and execution of <script> elements (beyond the inherited HTMLElement interface).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement)\n */\ninterface HTMLScriptElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/async) */\n    async: boolean;\n    /**\n     * Sets or retrieves the character set used to encode the object.\n     * @deprecated\n     */\n    charset: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/crossOrigin) */\n    crossOrigin: string | null;\n    /**\n     * Sets or retrieves the status of the script.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/defer)\n     */\n    defer: boolean;\n    /**\n     * Sets or retrieves the event for which the script is written.\n     * @deprecated\n     */\n    event: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/fetchPriority) */\n    fetchPriority: string;\n    /**\n     * Sets or retrieves the object that is bound to the event script.\n     * @deprecated\n     */\n    htmlFor: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/integrity) */\n    integrity: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/noModule) */\n    noModule: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/referrerPolicy) */\n    referrerPolicy: string;\n    /**\n     * Retrieves the URL to an external file that contains the source code or data.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/src)\n     */\n    src: string;\n    /**\n     * Retrieves or sets the text of the object as a string.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/text)\n     */\n    text: string;\n    /**\n     * Sets or retrieves the MIME type for the associated scripting engine.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/type)\n     */\n    type: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLScriptElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLScriptElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLScriptElement: {\n    prototype: HTMLScriptElement;\n    new(): HTMLScriptElement;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLScriptElement/supports_static) */\n    supports(type: string): boolean;\n};\n\n/**\n * A <select> HTML Element. These elements also share all of the properties and methods of other HTML elements via the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement)\n */\ninterface HTMLSelectElement extends HTMLElement {\n    autocomplete: AutoFill;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/disabled) */\n    disabled: boolean;\n    /**\n     * Retrieves a reference to the form that the object is embedded in.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/form)\n     */\n    readonly form: HTMLFormElement | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement>;\n    /** Sets or retrieves the number of objects in a collection. */\n    length: number;\n    /** Sets or retrieves the Boolean value indicating whether multiple items can be selected from a list. */\n    multiple: boolean;\n    /** Sets or retrieves the name of the object. */\n    name: string;\n    /**\n     * Returns an HTMLOptionsCollection of the list of options.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/options)\n     */\n    readonly options: HTMLOptionsCollection;\n    /** When present, marks an element that can't be submitted without a value. */\n    required: boolean;\n    /**\n     * Sets or retrieves the index of the selected option in a select object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/selectedIndex)\n     */\n    selectedIndex: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/selectedOptions) */\n    readonly selectedOptions: HTMLCollectionOf<HTMLOptionElement>;\n    /** Sets or retrieves the number of rows in the list box. */\n    size: number;\n    /**\n     * Retrieves the type of select control based on the value of the MULTIPLE attribute.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/type)\n     */\n    readonly type: \"select-one\" | \"select-multiple\";\n    /** Returns the error message that would be displayed if the user submits the form, or an empty string if no error message. It also triggers the standard error message, such as \"this is a required field\". The result is that the user sees validation messages without actually submitting. */\n    readonly validationMessage: string;\n    /** Returns a  ValidityState object that represents the validity states of an element. */\n    readonly validity: ValidityState;\n    /**\n     * Sets or retrieves the value which is returned to the server when the form control is submitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/value)\n     */\n    value: string;\n    /** Returns whether an element will successfully validate based on forms validation rules and constraints. */\n    readonly willValidate: boolean;\n    /**\n     * Adds an element to the areas, controlRange, or options collection.\n     * @param element Variant of type Number that specifies the index position in the collection where the element is placed. If no value is given, the method places the element at the end of the collection.\n     * @param before Variant of type Object that specifies an element to insert before, or null to append the object to the collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/add)\n     */\n    add(element: HTMLOptionElement | HTMLOptGroupElement, before?: HTMLElement | number | null): void;\n    /**\n     * Returns whether a form will validate when it is submitted, without having to submit it.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/checkValidity)\n     */\n    checkValidity(): boolean;\n    /**\n     * Retrieves a select object or an object from an options collection.\n     * @param name Variant of type Number or String that specifies the object or collection to retrieve. If this parameter is an integer, it is the zero-based index of the object. If this parameter is a string, all objects with matching name or id properties are retrieved, and a collection is returned if more than one match is made.\n     * @param index Variant of type Number that specifies the zero-based index of the object to retrieve when a collection is returned.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/item)\n     */\n    item(index: number): HTMLOptionElement | null;\n    /**\n     * Retrieves a select object or an object from an options collection.\n     * @param namedItem A String that specifies the name or id property of the object to retrieve. A collection is returned if more than one match is made.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/namedItem)\n     */\n    namedItem(name: string): HTMLOptionElement | null;\n    /**\n     * Removes an element from the collection.\n     * @param index Number that specifies the zero-based index of the element to remove from the collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/remove)\n     */\n    remove(): void;\n    remove(index: number): void;\n    reportValidity(): boolean;\n    /**\n     * Sets a custom error message that is displayed when a form is submitted.\n     * @param error Sets a custom error message that is displayed when a form is submitted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/setCustomValidity)\n     */\n    setCustomValidity(error: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSelectElement/showPicker) */\n    showPicker(): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSelectElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSelectElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n    [name: number]: HTMLOptionElement | HTMLOptGroupElement;\n}\n\ndeclare var HTMLSelectElement: {\n    prototype: HTMLSelectElement;\n    new(): HTMLSelectElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSlotElement) */\ninterface HTMLSlotElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSlotElement/name) */\n    name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSlotElement/assign) */\n    assign(...nodes: (Element | Text)[]): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSlotElement/assignedElements) */\n    assignedElements(options?: AssignedNodesOptions): Element[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSlotElement/assignedNodes) */\n    assignedNodes(options?: AssignedNodesOptions): Node[];\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSlotElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSlotElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLSlotElement: {\n    prototype: HTMLSlotElement;\n    new(): HTMLSlotElement;\n};\n\n/**\n * Provides special properties (beyond the regular HTMLElement object interface it also has available to it by inheritance) for manipulating <source> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSourceElement)\n */\ninterface HTMLSourceElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSourceElement/height) */\n    height: number;\n    /** Gets or sets the intended media type of the media source. */\n    media: string;\n    sizes: string;\n    /** The address or URL of the a media resource that is to be considered. */\n    src: string;\n    srcset: string;\n    /** Gets or sets the MIME type of a media resource. */\n    type: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSourceElement/width) */\n    width: number;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSourceElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSourceElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLSourceElement: {\n    prototype: HTMLSourceElement;\n    new(): HTMLSourceElement;\n};\n\n/**\n * A <span> element and derives from the HTMLElement interface, but without implementing any additional properties or methods.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLSpanElement)\n */\ninterface HTMLSpanElement extends HTMLElement {\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSpanElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLSpanElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLSpanElement: {\n    prototype: HTMLSpanElement;\n    new(): HTMLSpanElement;\n};\n\n/**\n * A <style> element. It inherits properties and methods from its parent, HTMLElement, and from LinkStyle.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLStyleElement)\n */\ninterface HTMLStyleElement extends HTMLElement, LinkStyle {\n    /**\n     * Enables or disables the style sheet.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLStyleElement/disabled)\n     */\n    disabled: boolean;\n    /**\n     * Sets or retrieves the media type.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLStyleElement/media)\n     */\n    media: string;\n    /**\n     * Retrieves the CSS language in which the style sheet is written.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLStyleElement/type)\n     */\n    type: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLStyleElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLStyleElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLStyleElement: {\n    prototype: HTMLStyleElement;\n    new(): HTMLStyleElement;\n};\n\n/**\n * Special properties (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating table caption elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCaptionElement)\n */\ninterface HTMLTableCaptionElement extends HTMLElement {\n    /**\n     * Sets or retrieves the alignment of the caption or legend.\n     * @deprecated\n     */\n    align: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableCaptionElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableCaptionElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTableCaptionElement: {\n    prototype: HTMLTableCaptionElement;\n    new(): HTMLTableCaptionElement;\n};\n\n/**\n * Provides special properties and methods (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of table cells, either header or data cells, in an HTML document.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement)\n */\ninterface HTMLTableCellElement extends HTMLElement {\n    /**\n     * Sets or retrieves abbreviated text for the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/abbr)\n     */\n    abbr: string;\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/align)\n     */\n    align: string;\n    /**\n     * Sets or retrieves a comma-delimited list of conceptual categories associated with the object.\n     * @deprecated\n     */\n    axis: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/bgColor)\n     */\n    bgColor: string;\n    /**\n     * Retrieves the position of the object in the cells collection of a row.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/cellIndex)\n     */\n    readonly cellIndex: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/ch)\n     */\n    ch: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/chOff)\n     */\n    chOff: string;\n    /**\n     * Sets or retrieves the number columns in the table that the object should span.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/colSpan)\n     */\n    colSpan: number;\n    /**\n     * Sets or retrieves a list of header cells that provide information for the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/headers)\n     */\n    headers: string;\n    /**\n     * Sets or retrieves the height of the object.\n     * @deprecated\n     */\n    height: string;\n    /**\n     * Sets or retrieves whether the browser automatically performs wordwrap.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/noWrap)\n     */\n    noWrap: boolean;\n    /**\n     * Sets or retrieves how many rows in a table the cell should span.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/rowSpan)\n     */\n    rowSpan: number;\n    /**\n     * Sets or retrieves the group of cells in a table to which the object's information applies.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/scope)\n     */\n    scope: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableCellElement/vAlign)\n     */\n    vAlign: string;\n    /**\n     * Sets or retrieves the width of the object.\n     * @deprecated\n     */\n    width: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableCellElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableCellElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTableCellElement: {\n    prototype: HTMLTableCellElement;\n    new(): HTMLTableCellElement;\n};\n\n/**\n * Provides special properties (beyond the HTMLElement interface it also has available to it inheritance) for manipulating single or grouped table column elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableColElement)\n */\ninterface HTMLTableColElement extends HTMLElement {\n    /**\n     * Sets or retrieves the alignment of the object relative to the display or table.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableColElement/align)\n     */\n    align: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableColElement/ch)\n     */\n    ch: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableColElement/chOff)\n     */\n    chOff: string;\n    /**\n     * Sets or retrieves the number of columns in the group.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableColElement/span)\n     */\n    span: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableColElement/vAlign)\n     */\n    vAlign: string;\n    /**\n     * Sets or retrieves the width of the object.\n     * @deprecated\n     */\n    width: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableColElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableColElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTableColElement: {\n    prototype: HTMLTableColElement;\n    new(): HTMLTableColElement;\n};\n\n/** @deprecated prefer HTMLTableCellElement */\ninterface HTMLTableDataCellElement extends HTMLTableCellElement {\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableDataCellElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableDataCellElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/**\n * Provides special properties and methods (beyond the regular HTMLElement object interface it also has available to it by inheritance) for manipulating the layout and presentation of tables in an HTML document.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement)\n */\ninterface HTMLTableElement extends HTMLElement {\n    /**\n     * Sets or retrieves a value that indicates the table alignment.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/align)\n     */\n    align: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/bgColor)\n     */\n    bgColor: string;\n    /**\n     * Sets or retrieves the width of the border to draw around the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/border)\n     */\n    border: string;\n    /**\n     * Retrieves the caption object of a table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/caption)\n     */\n    caption: HTMLTableCaptionElement | null;\n    /**\n     * Sets or retrieves the amount of space between the border of the cell and the content of the cell.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/cellPadding)\n     */\n    cellPadding: string;\n    /**\n     * Sets or retrieves the amount of space between cells in a table.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/cellSpacing)\n     */\n    cellSpacing: string;\n    /**\n     * Sets or retrieves the way the border frame around the table is displayed.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/frame)\n     */\n    frame: string;\n    /**\n     * Sets or retrieves the number of horizontal rows contained in the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/rows)\n     */\n    readonly rows: HTMLCollectionOf<HTMLTableRowElement>;\n    /**\n     * Sets or retrieves which dividing lines (inner borders) are displayed.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/rules)\n     */\n    rules: string;\n    /**\n     * Sets or retrieves a description and/or structure of the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/summary)\n     */\n    summary: string;\n    /**\n     * Retrieves a collection of all tBody objects in the table. Objects in this collection are in source order.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/tBodies)\n     */\n    readonly tBodies: HTMLCollectionOf<HTMLTableSectionElement>;\n    /**\n     * Retrieves the tFoot object of the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/tFoot)\n     */\n    tFoot: HTMLTableSectionElement | null;\n    /**\n     * Retrieves the tHead object of the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/tHead)\n     */\n    tHead: HTMLTableSectionElement | null;\n    /**\n     * Sets or retrieves the width of the object.\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/width)\n     */\n    width: string;\n    /**\n     * Creates an empty caption element in the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/createCaption)\n     */\n    createCaption(): HTMLTableCaptionElement;\n    /**\n     * Creates an empty tBody element in the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/createTBody)\n     */\n    createTBody(): HTMLTableSectionElement;\n    /**\n     * Creates an empty tFoot element in the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/createTFoot)\n     */\n    createTFoot(): HTMLTableSectionElement;\n    /**\n     * Returns the tHead element object if successful, or null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/createTHead)\n     */\n    createTHead(): HTMLTableSectionElement;\n    /**\n     * Deletes the caption element and its contents from the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/deleteCaption)\n     */\n    deleteCaption(): void;\n    /**\n     * Removes the specified row (tr) from the element and from the rows collection.\n     * @param index Number that specifies the zero-based position in the rows collection of the row to remove.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/deleteRow)\n     */\n    deleteRow(index: number): void;\n    /**\n     * Deletes the tFoot element and its contents from the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/deleteTFoot)\n     */\n    deleteTFoot(): void;\n    /**\n     * Deletes the tHead element and its contents from the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/deleteTHead)\n     */\n    deleteTHead(): void;\n    /**\n     * Creates a new row (tr) in the table, and adds the row to the rows collection.\n     * @param index Number that specifies where to insert the row in the rows collection. The default value is -1, which appends the new row to the end of the rows collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableElement/insertRow)\n     */\n    insertRow(index?: number): HTMLTableRowElement;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTableElement: {\n    prototype: HTMLTableElement;\n    new(): HTMLTableElement;\n};\n\n/** @deprecated prefer HTMLTableCellElement */\ninterface HTMLTableHeaderCellElement extends HTMLTableCellElement {\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableHeaderCellElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableHeaderCellElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/**\n * Provides special properties and methods (beyond the HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of rows in an HTML table.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement)\n */\ninterface HTMLTableRowElement extends HTMLElement {\n    /**\n     * Sets or retrieves how the object is aligned with adjacent text.\n     * @deprecated\n     */\n    align: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/bgColor)\n     */\n    bgColor: string;\n    /**\n     * Retrieves a collection of all cells in the table row.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/cells)\n     */\n    readonly cells: HTMLCollectionOf<HTMLTableCellElement>;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/ch)\n     */\n    ch: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/chOff)\n     */\n    chOff: string;\n    /**\n     * Retrieves the position of the object in the rows collection for the table.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/rowIndex)\n     */\n    readonly rowIndex: number;\n    /**\n     * Retrieves the position of the object in the collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/sectionRowIndex)\n     */\n    readonly sectionRowIndex: number;\n    /** @deprecated */\n    vAlign: string;\n    /**\n     * Removes the specified cell from the table row, as well as from the cells collection.\n     * @param index Number that specifies the zero-based position of the cell to remove from the table row. If no value is provided, the last cell in the cells collection is deleted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/deleteCell)\n     */\n    deleteCell(index: number): void;\n    /**\n     * Creates a new cell in the table row, and adds the cell to the cells collection.\n     * @param index Number that specifies where to insert the cell in the tr. The default value is -1, which appends the new cell to the end of the cells collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableRowElement/insertCell)\n     */\n    insertCell(index?: number): HTMLTableCellElement;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableRowElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableRowElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTableRowElement: {\n    prototype: HTMLTableRowElement;\n    new(): HTMLTableRowElement;\n};\n\n/**\n * Provides special properties and methods (beyond the HTMLElement interface it also has available to it by inheritance) for manipulating the layout and presentation of sections, that is headers, footers and bodies, in an HTML table.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableSectionElement)\n */\ninterface HTMLTableSectionElement extends HTMLElement {\n    /**\n     * Sets or retrieves a value that indicates the table alignment.\n     * @deprecated\n     */\n    align: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableSectionElement/ch)\n     */\n    ch: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableSectionElement/chOff)\n     */\n    chOff: string;\n    /**\n     * Sets or retrieves the number of horizontal rows contained in the object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableSectionElement/rows)\n     */\n    readonly rows: HTMLCollectionOf<HTMLTableRowElement>;\n    /** @deprecated */\n    vAlign: string;\n    /**\n     * Removes the specified row (tr) from the element and from the rows collection.\n     * @param index Number that specifies the zero-based position in the rows collection of the row to remove.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableSectionElement/deleteRow)\n     */\n    deleteRow(index: number): void;\n    /**\n     * Creates a new row (tr) in the table, and adds the row to the rows collection.\n     * @param index Number that specifies where to insert the row in the rows collection. The default value is -1, which appends the new row to the end of the rows collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTableSectionElement/insertRow)\n     */\n    insertRow(index?: number): HTMLTableRowElement;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableSectionElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTableSectionElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTableSectionElement: {\n    prototype: HTMLTableSectionElement;\n    new(): HTMLTableSectionElement;\n};\n\n/**\n * Enables access to the contents of an HTML <template> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTemplateElement)\n */\ninterface HTMLTemplateElement extends HTMLElement {\n    /**\n     * Returns the template contents (a DocumentFragment).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTemplateElement/content)\n     */\n    readonly content: DocumentFragment;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTemplateElement/shadowRootClonable) */\n    shadowRootClonable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTemplateElement/shadowRootDelegatesFocus) */\n    shadowRootDelegatesFocus: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTemplateElement/shadowRootMode) */\n    shadowRootMode: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTemplateElement/shadowRootSerializable) */\n    shadowRootSerializable: boolean;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTemplateElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTemplateElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTemplateElement: {\n    prototype: HTMLTemplateElement;\n    new(): HTMLTemplateElement;\n};\n\n/**\n * Provides special properties and methods for manipulating the layout and presentation of <textarea> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTextAreaElement)\n */\ninterface HTMLTextAreaElement extends HTMLElement {\n    autocomplete: AutoFill;\n    /** Sets or retrieves the width of the object. */\n    cols: number;\n    /** Sets or retrieves the initial contents of the object. */\n    defaultValue: string;\n    dirName: string;\n    disabled: boolean;\n    /** Retrieves a reference to the form that the object is embedded in. */\n    readonly form: HTMLFormElement | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTextAreaElement/labels) */\n    readonly labels: NodeListOf<HTMLLabelElement>;\n    /** Sets or retrieves the maximum number of characters that the user can enter in a text control. */\n    maxLength: number;\n    minLength: number;\n    /** Sets or retrieves the name of the object. */\n    name: string;\n    /** Gets or sets a text string that is displayed in an input field as a hint or prompt to users as the format or type of information they need to enter.The text appears in an input field until the user puts focus on the field. */\n    placeholder: string;\n    /** Sets or retrieves the value indicated whether the content of the object is read-only. */\n    readOnly: boolean;\n    /** When present, marks an element that can't be submitted without a value. */\n    required: boolean;\n    /** Sets or retrieves the number of horizontal rows contained in the object. */\n    rows: number;\n    selectionDirection: \"forward\" | \"backward\" | \"none\";\n    /** Gets or sets the end position or offset of a text selection. */\n    selectionEnd: number;\n    /** Gets or sets the starting position or offset of a text selection. */\n    selectionStart: number;\n    readonly textLength: number;\n    /**\n     * Retrieves the type of control.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTextAreaElement/type)\n     */\n    readonly type: string;\n    /** Returns the error message that would be displayed if the user submits the form, or an empty string if no error message. It also triggers the standard error message, such as \"this is a required field\". The result is that the user sees validation messages without actually submitting. */\n    readonly validationMessage: string;\n    /** Returns a  ValidityState object that represents the validity states of an element. */\n    readonly validity: ValidityState;\n    /** Retrieves or sets the text in the entry field of the textArea element. */\n    value: string;\n    /** Returns whether an element will successfully validate based on forms validation rules and constraints. */\n    readonly willValidate: boolean;\n    /** Sets or retrieves how to handle wordwrapping in the object. */\n    wrap: string;\n    /** Returns whether a form will validate when it is submitted, without having to submit it. */\n    checkValidity(): boolean;\n    reportValidity(): boolean;\n    /** Highlights the input area of a form element. */\n    select(): void;\n    /**\n     * Sets a custom error message that is displayed when a form is submitted.\n     * @param error Sets a custom error message that is displayed when a form is submitted.\n     */\n    setCustomValidity(error: string): void;\n    setRangeText(replacement: string): void;\n    setRangeText(replacement: string, start: number, end: number, selectionMode?: SelectionMode): void;\n    /**\n     * Sets the start and end positions of a selection in a text field.\n     * @param start The offset into the text field for the start of the selection.\n     * @param end The offset into the text field for the end of the selection.\n     * @param direction The direction in which the selection is performed.\n     */\n    setSelectionRange(start: number | null, end: number | null, direction?: \"forward\" | \"backward\" | \"none\"): void;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTextAreaElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTextAreaElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTextAreaElement: {\n    prototype: HTMLTextAreaElement;\n    new(): HTMLTextAreaElement;\n};\n\n/**\n * Provides special properties (beyond the regular HTMLElement interface it also has available to it by inheritance) for manipulating <time> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTimeElement)\n */\ninterface HTMLTimeElement extends HTMLElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTimeElement/dateTime) */\n    dateTime: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTimeElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTimeElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTimeElement: {\n    prototype: HTMLTimeElement;\n    new(): HTMLTimeElement;\n};\n\n/**\n * Contains the title for a document. This element inherits all of the properties and methods of the HTMLElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTitleElement)\n */\ninterface HTMLTitleElement extends HTMLElement {\n    /**\n     * Retrieves or sets the text of the object as a string.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTitleElement/text)\n     */\n    text: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTitleElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTitleElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTitleElement: {\n    prototype: HTMLTitleElement;\n    new(): HTMLTitleElement;\n};\n\n/**\n * The HTMLTrackElement\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTrackElement)\n */\ninterface HTMLTrackElement extends HTMLElement {\n    default: boolean;\n    kind: string;\n    label: string;\n    readonly readyState: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLTrackElement/src) */\n    src: string;\n    srclang: string;\n    /** Returns the TextTrack object corresponding to the text track of the track element. */\n    readonly track: TextTrack;\n    readonly NONE: 0;\n    readonly LOADING: 1;\n    readonly LOADED: 2;\n    readonly ERROR: 3;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTrackElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLTrackElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLTrackElement: {\n    prototype: HTMLTrackElement;\n    new(): HTMLTrackElement;\n    readonly NONE: 0;\n    readonly LOADING: 1;\n    readonly LOADED: 2;\n    readonly ERROR: 3;\n};\n\n/**\n * Provides special properties (beyond those defined on the regular HTMLElement interface it also has available to it by inheritance) for manipulating unordered list elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLUListElement)\n */\ninterface HTMLUListElement extends HTMLElement {\n    /** @deprecated */\n    compact: boolean;\n    /** @deprecated */\n    type: string;\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLUListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLUListElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLUListElement: {\n    prototype: HTMLUListElement;\n    new(): HTMLUListElement;\n};\n\n/**\n * An invalid HTML element and derives from the HTMLElement interface, but without implementing any additional properties or methods.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLUnknownElement)\n */\ninterface HTMLUnknownElement extends HTMLElement {\n    addEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLUnknownElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLElementEventMap>(type: K, listener: (this: HTMLUnknownElement, ev: HTMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLUnknownElement: {\n    prototype: HTMLUnknownElement;\n    new(): HTMLUnknownElement;\n};\n\ninterface HTMLVideoElementEventMap extends HTMLMediaElementEventMap {\n    \"enterpictureinpicture\": Event;\n    \"leavepictureinpicture\": Event;\n}\n\n/**\n * Provides special properties and methods for manipulating video objects. It also inherits properties and methods of HTMLMediaElement and HTMLElement.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement)\n */\ninterface HTMLVideoElement extends HTMLMediaElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/disablePictureInPicture) */\n    disablePictureInPicture: boolean;\n    /**\n     * Gets or sets the height of the video element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/height)\n     */\n    height: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/enterpictureinpicture_event) */\n    onenterpictureinpicture: ((this: HTMLVideoElement, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/leavepictureinpicture_event) */\n    onleavepictureinpicture: ((this: HTMLVideoElement, ev: Event) => any) | null;\n    /** Gets or sets the playsinline of the video element. for example, On iPhone, video elements will now be allowed to play inline, and will not automatically enter fullscreen mode when playback begins. */\n    playsInline: boolean;\n    /**\n     * Gets or sets a URL of an image to display, for example, like a movie poster. This can be a still frame from the video, or another image if no video data is available.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/poster)\n     */\n    poster: string;\n    /**\n     * Gets the intrinsic height of a video in CSS pixels, or zero if the dimensions are not known.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/videoHeight)\n     */\n    readonly videoHeight: number;\n    /**\n     * Gets the intrinsic width of a video in CSS pixels, or zero if the dimensions are not known.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/videoWidth)\n     */\n    readonly videoWidth: number;\n    /**\n     * Gets or sets the width of the video element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/width)\n     */\n    width: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/cancelVideoFrameCallback) */\n    cancelVideoFrameCallback(handle: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/getVideoPlaybackQuality) */\n    getVideoPlaybackQuality(): VideoPlaybackQuality;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/requestPictureInPicture) */\n    requestPictureInPicture(): Promise<PictureInPictureWindow>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLVideoElement/requestVideoFrameCallback) */\n    requestVideoFrameCallback(callback: VideoFrameRequestCallback): number;\n    addEventListener<K extends keyof HTMLVideoElementEventMap>(type: K, listener: (this: HTMLVideoElement, ev: HTMLVideoElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof HTMLVideoElementEventMap>(type: K, listener: (this: HTMLVideoElement, ev: HTMLVideoElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var HTMLVideoElement: {\n    prototype: HTMLVideoElement;\n    new(): HTMLVideoElement;\n};\n\n/**\n * Events that fire when the fragment identifier of the URL has changed.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HashChangeEvent)\n */\ninterface HashChangeEvent extends Event {\n    /**\n     * Returns the URL of the session history entry that is now current.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HashChangeEvent/newURL)\n     */\n    readonly newURL: string;\n    /**\n     * Returns the URL of the session history entry that was previously current.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/HashChangeEvent/oldURL)\n     */\n    readonly oldURL: string;\n}\n\ndeclare var HashChangeEvent: {\n    prototype: HashChangeEvent;\n    new(type: string, eventInitDict?: HashChangeEventInit): HashChangeEvent;\n};\n\n/**\n * This Fetch API interface allows you to perform various actions on HTTP request and response headers. These actions include retrieving, setting, adding to, and removing. A Headers object has an associated header list, which is initially empty and consists of zero or more name and value pairs.  You can add to this using methods like append() (see Examples.) In all methods of this interface, header names are matched by case-insensitive byte sequence.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers)\n */\ninterface Headers {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers/append) */\n    append(name: string, value: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers/delete) */\n    delete(name: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers/get) */\n    get(name: string): string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers/getSetCookie) */\n    getSetCookie(): string[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers/has) */\n    has(name: string): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Headers/set) */\n    set(name: string, value: string): void;\n    forEach(callbackfn: (value: string, key: string, parent: Headers) => void, thisArg?: any): void;\n}\n\ndeclare var Headers: {\n    prototype: Headers;\n    new(init?: HeadersInit): Headers;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Highlight) */\ninterface Highlight {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Highlight/priority) */\n    priority: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Highlight/type) */\n    type: HighlightType;\n    forEach(callbackfn: (value: AbstractRange, key: AbstractRange, parent: Highlight) => void, thisArg?: any): void;\n}\n\ndeclare var Highlight: {\n    prototype: Highlight;\n    new(...initialRanges: AbstractRange[]): Highlight;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HighlightRegistry) */\ninterface HighlightRegistry {\n    forEach(callbackfn: (value: Highlight, key: string, parent: HighlightRegistry) => void, thisArg?: any): void;\n}\n\ndeclare var HighlightRegistry: {\n    prototype: HighlightRegistry;\n    new(): HighlightRegistry;\n};\n\n/**\n * Allows manipulation of the browser session history, that is the pages visited in the tab or frame that the current page is loaded in.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/History)\n */\ninterface History {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/scrollRestoration) */\n    scrollRestoration: ScrollRestoration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/state) */\n    readonly state: any;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/back) */\n    back(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/forward) */\n    forward(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/go) */\n    go(delta?: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/pushState) */\n    pushState(data: any, unused: string, url?: string | URL | null): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/History/replaceState) */\n    replaceState(data: any, unused: string, url?: string | URL | null): void;\n}\n\ndeclare var History: {\n    prototype: History;\n    new(): History;\n};\n\n/**\n * This IndexedDB API interface represents a cursor for traversing or iterating over multiple records in a database.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor)\n */\ninterface IDBCursor {\n    /**\n     * Returns the direction (\"next\", \"nextunique\", \"prev\" or \"prevunique\") of the cursor.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/direction)\n     */\n    readonly direction: IDBCursorDirection;\n    /**\n     * Returns the key of the cursor. Throws a \"InvalidStateError\" DOMException if the cursor is advancing or is finished.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/key)\n     */\n    readonly key: IDBValidKey;\n    /**\n     * Returns the effective key of the cursor. Throws a \"InvalidStateError\" DOMException if the cursor is advancing or is finished.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/primaryKey)\n     */\n    readonly primaryKey: IDBValidKey;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/request) */\n    readonly request: IDBRequest;\n    /**\n     * Returns the IDBObjectStore or IDBIndex the cursor was opened from.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/source)\n     */\n    readonly source: IDBObjectStore | IDBIndex;\n    /**\n     * Advances the cursor through the next count records in range.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/advance)\n     */\n    advance(count: number): void;\n    /**\n     * Advances the cursor to the next record in range.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/continue)\n     */\n    continue(key?: IDBValidKey): void;\n    /**\n     * Advances the cursor to the next record in range matching or after key and primaryKey. Throws an \"InvalidAccessError\" DOMException if the source is not an index.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/continuePrimaryKey)\n     */\n    continuePrimaryKey(key: IDBValidKey, primaryKey: IDBValidKey): void;\n    /**\n     * Delete the record pointed at by the cursor with a new value.\n     *\n     * If successful, request's result will be undefined.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/delete)\n     */\n    delete(): IDBRequest<undefined>;\n    /**\n     * Updated the record pointed at by the cursor with a new value.\n     *\n     * Throws a \"DataError\" DOMException if the effective object store uses in-line keys and the key would have changed.\n     *\n     * If successful, request's result will be the record's key.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursor/update)\n     */\n    update(value: any): IDBRequest<IDBValidKey>;\n}\n\ndeclare var IDBCursor: {\n    prototype: IDBCursor;\n    new(): IDBCursor;\n};\n\n/**\n * This IndexedDB API interface represents a cursor for traversing or iterating over multiple records in a database. It is the same as the IDBCursor, except that it includes the value property.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursorWithValue)\n */\ninterface IDBCursorWithValue extends IDBCursor {\n    /**\n     * Returns the cursor's current value.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBCursorWithValue/value)\n     */\n    readonly value: any;\n}\n\ndeclare var IDBCursorWithValue: {\n    prototype: IDBCursorWithValue;\n    new(): IDBCursorWithValue;\n};\n\ninterface IDBDatabaseEventMap {\n    \"abort\": Event;\n    \"close\": Event;\n    \"error\": Event;\n    \"versionchange\": IDBVersionChangeEvent;\n}\n\n/**\n * This IndexedDB API interface provides a connection to a database; you can use an IDBDatabase object to open a transaction on your database then create, manipulate, and delete objects (data) in that database. The interface provides the only way to get and manage versions of the database.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase)\n */\ninterface IDBDatabase extends EventTarget {\n    /**\n     * Returns the name of the database.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/name)\n     */\n    readonly name: string;\n    /**\n     * Returns a list of the names of object stores in the database.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/objectStoreNames)\n     */\n    readonly objectStoreNames: DOMStringList;\n    onabort: ((this: IDBDatabase, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/close_event) */\n    onclose: ((this: IDBDatabase, ev: Event) => any) | null;\n    onerror: ((this: IDBDatabase, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/versionchange_event) */\n    onversionchange: ((this: IDBDatabase, ev: IDBVersionChangeEvent) => any) | null;\n    /**\n     * Returns the version of the database.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/version)\n     */\n    readonly version: number;\n    /**\n     * Closes the connection once all running transactions have finished.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/close)\n     */\n    close(): void;\n    /**\n     * Creates a new object store with the given name and options and returns a new IDBObjectStore.\n     *\n     * Throws a \"InvalidStateError\" DOMException if not called within an upgrade transaction.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/createObjectStore)\n     */\n    createObjectStore(name: string, options?: IDBObjectStoreParameters): IDBObjectStore;\n    /**\n     * Deletes the object store with the given name.\n     *\n     * Throws a \"InvalidStateError\" DOMException if not called within an upgrade transaction.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/deleteObjectStore)\n     */\n    deleteObjectStore(name: string): void;\n    /**\n     * Returns a new transaction with the given mode (\"readonly\" or \"readwrite\") and scope which can be a single object store name or an array of names.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBDatabase/transaction)\n     */\n    transaction(storeNames: string | string[], mode?: IDBTransactionMode, options?: IDBTransactionOptions): IDBTransaction;\n    addEventListener<K extends keyof IDBDatabaseEventMap>(type: K, listener: (this: IDBDatabase, ev: IDBDatabaseEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof IDBDatabaseEventMap>(type: K, listener: (this: IDBDatabase, ev: IDBDatabaseEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var IDBDatabase: {\n    prototype: IDBDatabase;\n    new(): IDBDatabase;\n};\n\n/**\n * In the following code snippet, we make a request to open a database, and include handlers for the success and error cases. For a full working example, see our To-do Notifications app (view example live.)\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBFactory)\n */\ninterface IDBFactory {\n    /**\n     * Compares two values as keys. Returns -1 if key1 precedes key2, 1 if key2 precedes key1, and 0 if the keys are equal.\n     *\n     * Throws a \"DataError\" DOMException if either input is not a valid key.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBFactory/cmp)\n     */\n    cmp(first: any, second: any): number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBFactory/databases) */\n    databases(): Promise<IDBDatabaseInfo[]>;\n    /**\n     * Attempts to delete the named database. If the database already exists and there are open connections that don't close in response to a versionchange event, the request will be blocked until all they close. If the request is successful request's result will be null.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBFactory/deleteDatabase)\n     */\n    deleteDatabase(name: string): IDBOpenDBRequest;\n    /**\n     * Attempts to open a connection to the named database with the current version, or 1 if it does not already exist. If the request is successful request's result will be the connection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBFactory/open)\n     */\n    open(name: string, version?: number): IDBOpenDBRequest;\n}\n\ndeclare var IDBFactory: {\n    prototype: IDBFactory;\n    new(): IDBFactory;\n};\n\n/**\n * IDBIndex interface of the IndexedDB API provides asynchronous access to an index in a database. An index is a kind of object store for looking up records in another object store, called the referenced object store. You use this interface to retrieve data.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex)\n */\ninterface IDBIndex {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/keyPath) */\n    readonly keyPath: string | string[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/multiEntry) */\n    readonly multiEntry: boolean;\n    /**\n     * Returns the name of the index.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/name)\n     */\n    name: string;\n    /**\n     * Returns the IDBObjectStore the index belongs to.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/objectStore)\n     */\n    readonly objectStore: IDBObjectStore;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/unique) */\n    readonly unique: boolean;\n    /**\n     * Retrieves the number of records matching the given key or key range in query.\n     *\n     * If successful, request's result will be the count.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/count)\n     */\n    count(query?: IDBValidKey | IDBKeyRange): IDBRequest<number>;\n    /**\n     * Retrieves the value of the first record matching the given key or key range in query.\n     *\n     * If successful, request's result will be the value, or undefined if there was no matching record.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/get)\n     */\n    get(query: IDBValidKey | IDBKeyRange): IDBRequest<any>;\n    /**\n     * Retrieves the values of the records matching the given key or key range in query (up to count if given).\n     *\n     * If successful, request's result will be an Array of the values.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/getAll)\n     */\n    getAll(query?: IDBValidKey | IDBKeyRange | null, count?: number): IDBRequest<any[]>;\n    /**\n     * Retrieves the keys of records matching the given key or key range in query (up to count if given).\n     *\n     * If successful, request's result will be an Array of the keys.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/getAllKeys)\n     */\n    getAllKeys(query?: IDBValidKey | IDBKeyRange | null, count?: number): IDBRequest<IDBValidKey[]>;\n    /**\n     * Retrieves the key of the first record matching the given key or key range in query.\n     *\n     * If successful, request's result will be the key, or undefined if there was no matching record.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/getKey)\n     */\n    getKey(query: IDBValidKey | IDBKeyRange): IDBRequest<IDBValidKey | undefined>;\n    /**\n     * Opens a cursor over the records matching query, ordered by direction. If query is null, all records in index are matched.\n     *\n     * If successful, request's result will be an IDBCursorWithValue, or null if there were no matching records.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/openCursor)\n     */\n    openCursor(query?: IDBValidKey | IDBKeyRange | null, direction?: IDBCursorDirection): IDBRequest<IDBCursorWithValue | null>;\n    /**\n     * Opens a cursor with key only flag set over the records matching query, ordered by direction. If query is null, all records in index are matched.\n     *\n     * If successful, request's result will be an IDBCursor, or null if there were no matching records.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBIndex/openKeyCursor)\n     */\n    openKeyCursor(query?: IDBValidKey | IDBKeyRange | null, direction?: IDBCursorDirection): IDBRequest<IDBCursor | null>;\n}\n\ndeclare var IDBIndex: {\n    prototype: IDBIndex;\n    new(): IDBIndex;\n};\n\n/**\n * A key range can be a single value or a range with upper and lower bounds or endpoints. If the key range has both upper and lower bounds, then it is bounded; if it has no bounds, it is unbounded. A bounded key range can either be open (the endpoints are excluded) or closed (the endpoints are included). To retrieve all keys within a certain range, you can use the following code constructs:\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange)\n */\ninterface IDBKeyRange {\n    /**\n     * Returns lower bound, or undefined if none.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/lower)\n     */\n    readonly lower: any;\n    /**\n     * Returns true if the lower open flag is set, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/lowerOpen)\n     */\n    readonly lowerOpen: boolean;\n    /**\n     * Returns upper bound, or undefined if none.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/upper)\n     */\n    readonly upper: any;\n    /**\n     * Returns true if the upper open flag is set, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/upperOpen)\n     */\n    readonly upperOpen: boolean;\n    /**\n     * Returns true if key is included in the range, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/includes)\n     */\n    includes(key: any): boolean;\n}\n\ndeclare var IDBKeyRange: {\n    prototype: IDBKeyRange;\n    new(): IDBKeyRange;\n    /**\n     * Returns a new IDBKeyRange spanning from lower to upper. If lowerOpen is true, lower is not included in the range. If upperOpen is true, upper is not included in the range.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/bound_static)\n     */\n    bound(lower: any, upper: any, lowerOpen?: boolean, upperOpen?: boolean): IDBKeyRange;\n    /**\n     * Returns a new IDBKeyRange starting at key with no upper bound. If open is true, key is not included in the range.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/lowerBound_static)\n     */\n    lowerBound(lower: any, open?: boolean): IDBKeyRange;\n    /**\n     * Returns a new IDBKeyRange spanning only key.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/only_static)\n     */\n    only(value: any): IDBKeyRange;\n    /**\n     * Returns a new IDBKeyRange with no lower bound and ending at key. If open is true, key is not included in the range.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBKeyRange/upperBound_static)\n     */\n    upperBound(upper: any, open?: boolean): IDBKeyRange;\n};\n\n/**\n * This example shows a variety of different uses of object stores, from updating the data structure with IDBObjectStore.createIndex inside an onupgradeneeded function, to adding a new item to our object store with IDBObjectStore.add. For a full working example, see our To-do Notifications app (view example live.)\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore)\n */\ninterface IDBObjectStore {\n    /**\n     * Returns true if the store has a key generator, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/autoIncrement)\n     */\n    readonly autoIncrement: boolean;\n    /**\n     * Returns a list of the names of indexes in the store.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/indexNames)\n     */\n    readonly indexNames: DOMStringList;\n    /**\n     * Returns the key path of the store, or null if none.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/keyPath)\n     */\n    readonly keyPath: string | string[];\n    /**\n     * Returns the name of the store.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/name)\n     */\n    name: string;\n    /**\n     * Returns the associated transaction.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/transaction)\n     */\n    readonly transaction: IDBTransaction;\n    /**\n     * Adds or updates a record in store with the given value and key.\n     *\n     * If the store uses in-line keys and key is specified a \"DataError\" DOMException will be thrown.\n     *\n     * If put() is used, any existing record with the key will be replaced. If add() is used, and if a record with the key already exists the request will fail, with request's error set to a \"ConstraintError\" DOMException.\n     *\n     * If successful, request's result will be the record's key.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/add)\n     */\n    add(value: any, key?: IDBValidKey): IDBRequest<IDBValidKey>;\n    /**\n     * Deletes all records in store.\n     *\n     * If successful, request's result will be undefined.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/clear)\n     */\n    clear(): IDBRequest<undefined>;\n    /**\n     * Retrieves the number of records matching the given key or key range in query.\n     *\n     * If successful, request's result will be the count.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/count)\n     */\n    count(query?: IDBValidKey | IDBKeyRange): IDBRequest<number>;\n    /**\n     * Creates a new index in store with the given name, keyPath and options and returns a new IDBIndex. If the keyPath and options define constraints that cannot be satisfied with the data already in store the upgrade transaction will abort with a \"ConstraintError\" DOMException.\n     *\n     * Throws an \"InvalidStateError\" DOMException if not called within an upgrade transaction.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/createIndex)\n     */\n    createIndex(name: string, keyPath: string | string[], options?: IDBIndexParameters): IDBIndex;\n    /**\n     * Deletes records in store with the given key or in the given key range in query.\n     *\n     * If successful, request's result will be undefined.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/delete)\n     */\n    delete(query: IDBValidKey | IDBKeyRange): IDBRequest<undefined>;\n    /**\n     * Deletes the index in store with the given name.\n     *\n     * Throws an \"InvalidStateError\" DOMException if not called within an upgrade transaction.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/deleteIndex)\n     */\n    deleteIndex(name: string): void;\n    /**\n     * Retrieves the value of the first record matching the given key or key range in query.\n     *\n     * If successful, request's result will be the value, or undefined if there was no matching record.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/get)\n     */\n    get(query: IDBValidKey | IDBKeyRange): IDBRequest<any>;\n    /**\n     * Retrieves the values of the records matching the given key or key range in query (up to count if given).\n     *\n     * If successful, request's result will be an Array of the values.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/getAll)\n     */\n    getAll(query?: IDBValidKey | IDBKeyRange | null, count?: number): IDBRequest<any[]>;\n    /**\n     * Retrieves the keys of records matching the given key or key range in query (up to count if given).\n     *\n     * If successful, request's result will be an Array of the keys.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/getAllKeys)\n     */\n    getAllKeys(query?: IDBValidKey | IDBKeyRange | null, count?: number): IDBRequest<IDBValidKey[]>;\n    /**\n     * Retrieves the key of the first record matching the given key or key range in query.\n     *\n     * If successful, request's result will be the key, or undefined if there was no matching record.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/getKey)\n     */\n    getKey(query: IDBValidKey | IDBKeyRange): IDBRequest<IDBValidKey | undefined>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/index) */\n    index(name: string): IDBIndex;\n    /**\n     * Opens a cursor over the records matching query, ordered by direction. If query is null, all records in store are matched.\n     *\n     * If successful, request's result will be an IDBCursorWithValue pointing at the first matching record, or null if there were no matching records.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/openCursor)\n     */\n    openCursor(query?: IDBValidKey | IDBKeyRange | null, direction?: IDBCursorDirection): IDBRequest<IDBCursorWithValue | null>;\n    /**\n     * Opens a cursor with key only flag set over the records matching query, ordered by direction. If query is null, all records in store are matched.\n     *\n     * If successful, request's result will be an IDBCursor pointing at the first matching record, or null if there were no matching records.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/openKeyCursor)\n     */\n    openKeyCursor(query?: IDBValidKey | IDBKeyRange | null, direction?: IDBCursorDirection): IDBRequest<IDBCursor | null>;\n    /**\n     * Adds or updates a record in store with the given value and key.\n     *\n     * If the store uses in-line keys and key is specified a \"DataError\" DOMException will be thrown.\n     *\n     * If put() is used, any existing record with the key will be replaced. If add() is used, and if a record with the key already exists the request will fail, with request's error set to a \"ConstraintError\" DOMException.\n     *\n     * If successful, request's result will be the record's key.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBObjectStore/put)\n     */\n    put(value: any, key?: IDBValidKey): IDBRequest<IDBValidKey>;\n}\n\ndeclare var IDBObjectStore: {\n    prototype: IDBObjectStore;\n    new(): IDBObjectStore;\n};\n\ninterface IDBOpenDBRequestEventMap extends IDBRequestEventMap {\n    \"blocked\": IDBVersionChangeEvent;\n    \"upgradeneeded\": IDBVersionChangeEvent;\n}\n\n/**\n * Also inherits methods from its parents IDBRequest and EventTarget.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBOpenDBRequest)\n */\ninterface IDBOpenDBRequest extends IDBRequest<IDBDatabase> {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBOpenDBRequest/blocked_event) */\n    onblocked: ((this: IDBOpenDBRequest, ev: IDBVersionChangeEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBOpenDBRequest/upgradeneeded_event) */\n    onupgradeneeded: ((this: IDBOpenDBRequest, ev: IDBVersionChangeEvent) => any) | null;\n    addEventListener<K extends keyof IDBOpenDBRequestEventMap>(type: K, listener: (this: IDBOpenDBRequest, ev: IDBOpenDBRequestEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof IDBOpenDBRequestEventMap>(type: K, listener: (this: IDBOpenDBRequest, ev: IDBOpenDBRequestEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var IDBOpenDBRequest: {\n    prototype: IDBOpenDBRequest;\n    new(): IDBOpenDBRequest;\n};\n\ninterface IDBRequestEventMap {\n    \"error\": Event;\n    \"success\": Event;\n}\n\n/**\n * The request object does not initially contain any information about the result of the operation, but once information becomes available, an event is fired on the request, and the information becomes available through the properties of the IDBRequest instance.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest)\n */\ninterface IDBRequest<T = any> extends EventTarget {\n    /**\n     * When a request is completed, returns the error (a DOMException), or null if the request succeeded. Throws a \"InvalidStateError\" DOMException if the request is still pending.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/error)\n     */\n    readonly error: DOMException | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/error_event) */\n    onerror: ((this: IDBRequest<T>, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/success_event) */\n    onsuccess: ((this: IDBRequest<T>, ev: Event) => any) | null;\n    /**\n     * Returns \"pending\" until a request is complete, then returns \"done\".\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/readyState)\n     */\n    readonly readyState: IDBRequestReadyState;\n    /**\n     * When a request is completed, returns the result, or undefined if the request failed. Throws a \"InvalidStateError\" DOMException if the request is still pending.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/result)\n     */\n    readonly result: T;\n    /**\n     * Returns the IDBObjectStore, IDBIndex, or IDBCursor the request was made against, or null if is was an open request.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/source)\n     */\n    readonly source: IDBObjectStore | IDBIndex | IDBCursor;\n    /**\n     * Returns the IDBTransaction the request was made within. If this as an open request, then it returns an upgrade transaction while it is running, or null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBRequest/transaction)\n     */\n    readonly transaction: IDBTransaction | null;\n    addEventListener<K extends keyof IDBRequestEventMap>(type: K, listener: (this: IDBRequest<T>, ev: IDBRequestEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof IDBRequestEventMap>(type: K, listener: (this: IDBRequest<T>, ev: IDBRequestEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var IDBRequest: {\n    prototype: IDBRequest;\n    new(): IDBRequest;\n};\n\ninterface IDBTransactionEventMap {\n    \"abort\": Event;\n    \"complete\": Event;\n    \"error\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction) */\ninterface IDBTransaction extends EventTarget {\n    /**\n     * Returns the transaction's connection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/db)\n     */\n    readonly db: IDBDatabase;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/durability) */\n    readonly durability: IDBTransactionDurability;\n    /**\n     * If the transaction was aborted, returns the error (a DOMException) providing the reason.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/error)\n     */\n    readonly error: DOMException | null;\n    /**\n     * Returns the mode the transaction was created with (\"readonly\" or \"readwrite\"), or \"versionchange\" for an upgrade transaction.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/mode)\n     */\n    readonly mode: IDBTransactionMode;\n    /**\n     * Returns a list of the names of object stores in the transaction's scope. For an upgrade transaction this is all object stores in the database.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/ObjectStoreNames)\n     */\n    readonly objectStoreNames: DOMStringList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/abort_event) */\n    onabort: ((this: IDBTransaction, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/complete_event) */\n    oncomplete: ((this: IDBTransaction, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/error_event) */\n    onerror: ((this: IDBTransaction, ev: Event) => any) | null;\n    /**\n     * Aborts the transaction. All pending requests will fail with a \"AbortError\" DOMException and all changes made to the database will be reverted.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/abort)\n     */\n    abort(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/commit) */\n    commit(): void;\n    /**\n     * Returns an IDBObjectStore in the transaction's scope.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBTransaction/objectStore)\n     */\n    objectStore(name: string): IDBObjectStore;\n    addEventListener<K extends keyof IDBTransactionEventMap>(type: K, listener: (this: IDBTransaction, ev: IDBTransactionEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof IDBTransactionEventMap>(type: K, listener: (this: IDBTransaction, ev: IDBTransactionEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var IDBTransaction: {\n    prototype: IDBTransaction;\n    new(): IDBTransaction;\n};\n\n/**\n * This IndexedDB API interface indicates that the version of the database has changed, as the result of an IDBOpenDBRequest.onupgradeneeded event handler function.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBVersionChangeEvent)\n */\ninterface IDBVersionChangeEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBVersionChangeEvent/newVersion) */\n    readonly newVersion: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IDBVersionChangeEvent/oldVersion) */\n    readonly oldVersion: number;\n}\n\ndeclare var IDBVersionChangeEvent: {\n    prototype: IDBVersionChangeEvent;\n    new(type: string, eventInitDict?: IDBVersionChangeEventInit): IDBVersionChangeEvent;\n};\n\n/**\n * The IIRFilterNode interface of the Web Audio API is a AudioNode processor which implements a general infinite impulse response (IIR)  filter; this type of filter can be used to implement tone control devices and graphic equalizers as well. It lets the parameters of the filter response be specified, so that it can be tuned as needed.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IIRFilterNode)\n */\ninterface IIRFilterNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IIRFilterNode/getFrequencyResponse) */\n    getFrequencyResponse(frequencyHz: Float32Array, magResponse: Float32Array, phaseResponse: Float32Array): void;\n}\n\ndeclare var IIRFilterNode: {\n    prototype: IIRFilterNode;\n    new(context: BaseAudioContext, options: IIRFilterOptions): IIRFilterNode;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IdleDeadline) */\ninterface IdleDeadline {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IdleDeadline/didTimeout) */\n    readonly didTimeout: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IdleDeadline/timeRemaining) */\n    timeRemaining(): DOMHighResTimeStamp;\n}\n\ndeclare var IdleDeadline: {\n    prototype: IdleDeadline;\n    new(): IdleDeadline;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageBitmap) */\ninterface ImageBitmap {\n    /**\n     * Returns the intrinsic height of the image, in CSS pixels.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageBitmap/height)\n     */\n    readonly height: number;\n    /**\n     * Returns the intrinsic width of the image, in CSS pixels.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageBitmap/width)\n     */\n    readonly width: number;\n    /**\n     * Releases imageBitmap's underlying bitmap data.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageBitmap/close)\n     */\n    close(): void;\n}\n\ndeclare var ImageBitmap: {\n    prototype: ImageBitmap;\n    new(): ImageBitmap;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageBitmapRenderingContext) */\ninterface ImageBitmapRenderingContext {\n    /** Returns the canvas element that the context is bound to. */\n    readonly canvas: HTMLCanvasElement | OffscreenCanvas;\n    /**\n     * Transfers the underlying bitmap data from imageBitmap to context, and the bitmap becomes the contents of the canvas element to which context is bound.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageBitmapRenderingContext/transferFromImageBitmap)\n     */\n    transferFromImageBitmap(bitmap: ImageBitmap | null): void;\n}\n\ndeclare var ImageBitmapRenderingContext: {\n    prototype: ImageBitmapRenderingContext;\n    new(): ImageBitmapRenderingContext;\n};\n\n/**\n * The underlying pixel data of an area of a <canvas> element. It is created using the ImageData() constructor or creator methods on the CanvasRenderingContext2D object associated with a canvas: createImageData() and getImageData(). It can also be used to set a part of the canvas by using putImageData().\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageData)\n */\ninterface ImageData {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageData/colorSpace) */\n    readonly colorSpace: PredefinedColorSpace;\n    /**\n     * Returns the one-dimensional array containing the data in RGBA order, as integers in the range 0 to 255.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageData/data)\n     */\n    readonly data: Uint8ClampedArray;\n    /**\n     * Returns the actual dimensions of the data in the ImageData object, in pixels.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageData/height)\n     */\n    readonly height: number;\n    /**\n     * Returns the actual dimensions of the data in the ImageData object, in pixels.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ImageData/width)\n     */\n    readonly width: number;\n}\n\ndeclare var ImageData: {\n    prototype: ImageData;\n    new(sw: number, sh: number, settings?: ImageDataSettings): ImageData;\n    new(data: Uint8ClampedArray, sw: number, sh?: number, settings?: ImageDataSettings): ImageData;\n};\n\ninterface ImportMeta {\n    url: string;\n    resolve(specifier: string): string;\n}\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputDeviceInfo)\n */\ninterface InputDeviceInfo extends MediaDeviceInfo {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputDeviceInfo/getCapabilities) */\n    getCapabilities(): MediaTrackCapabilities;\n}\n\ndeclare var InputDeviceInfo: {\n    prototype: InputDeviceInfo;\n    new(): InputDeviceInfo;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputEvent) */\ninterface InputEvent extends UIEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputEvent/data) */\n    readonly data: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputEvent/dataTransfer) */\n    readonly dataTransfer: DataTransfer | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputEvent/inputType) */\n    readonly inputType: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputEvent/isComposing) */\n    readonly isComposing: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/InputEvent/getTargetRanges) */\n    getTargetRanges(): StaticRange[];\n}\n\ndeclare var InputEvent: {\n    prototype: InputEvent;\n    new(type: string, eventInitDict?: InputEventInit): InputEvent;\n};\n\n/**\n * provides a way to asynchronously observe changes in the intersection of a target element with an ancestor element or with a top-level document's viewport.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver)\n */\ninterface IntersectionObserver {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/root) */\n    readonly root: Element | Document | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/rootMargin) */\n    readonly rootMargin: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/thresholds) */\n    readonly thresholds: ReadonlyArray<number>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/disconnect) */\n    disconnect(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/observe) */\n    observe(target: Element): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/takeRecords) */\n    takeRecords(): IntersectionObserverEntry[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserver/unobserve) */\n    unobserve(target: Element): void;\n}\n\ndeclare var IntersectionObserver: {\n    prototype: IntersectionObserver;\n    new(callback: IntersectionObserverCallback, options?: IntersectionObserverInit): IntersectionObserver;\n};\n\n/**\n * This Intersection Observer API interface describes the intersection between the target element and its root container at a specific moment of transition.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry)\n */\ninterface IntersectionObserverEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/boundingClientRect) */\n    readonly boundingClientRect: DOMRectReadOnly;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/intersectionRatio) */\n    readonly intersectionRatio: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/intersectionRect) */\n    readonly intersectionRect: DOMRectReadOnly;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/isIntersecting) */\n    readonly isIntersecting: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/rootBounds) */\n    readonly rootBounds: DOMRectReadOnly | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/target) */\n    readonly target: Element;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/IntersectionObserverEntry/time) */\n    readonly time: DOMHighResTimeStamp;\n}\n\ndeclare var IntersectionObserverEntry: {\n    prototype: IntersectionObserverEntry;\n    new(intersectionObserverEntryInit: IntersectionObserverEntryInit): IntersectionObserverEntry;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KHR_parallel_shader_compile) */\ninterface KHR_parallel_shader_compile {\n    readonly COMPLETION_STATUS_KHR: 0x91B1;\n}\n\n/**\n * KeyboardEvent objects describe a user interaction with the keyboard; each event describes a single interaction between the user and a key (or combination of a key with modifier keys) on the keyboard.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent)\n */\ninterface KeyboardEvent extends UIEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/altKey) */\n    readonly altKey: boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/charCode)\n     */\n    readonly charCode: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/code) */\n    readonly code: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/ctrlKey) */\n    readonly ctrlKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/isComposing) */\n    readonly isComposing: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/key) */\n    readonly key: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/keyCode)\n     */\n    readonly keyCode: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/location) */\n    readonly location: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/metaKey) */\n    readonly metaKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/repeat) */\n    readonly repeat: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/shiftKey) */\n    readonly shiftKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/getModifierState) */\n    getModifierState(keyArg: string): boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyboardEvent/initKeyboardEvent)\n     */\n    initKeyboardEvent(typeArg: string, bubblesArg?: boolean, cancelableArg?: boolean, viewArg?: Window | null, keyArg?: string, locationArg?: number, ctrlKey?: boolean, altKey?: boolean, shiftKey?: boolean, metaKey?: boolean): void;\n    readonly DOM_KEY_LOCATION_STANDARD: 0x00;\n    readonly DOM_KEY_LOCATION_LEFT: 0x01;\n    readonly DOM_KEY_LOCATION_RIGHT: 0x02;\n    readonly DOM_KEY_LOCATION_NUMPAD: 0x03;\n}\n\ndeclare var KeyboardEvent: {\n    prototype: KeyboardEvent;\n    new(type: string, eventInitDict?: KeyboardEventInit): KeyboardEvent;\n    readonly DOM_KEY_LOCATION_STANDARD: 0x00;\n    readonly DOM_KEY_LOCATION_LEFT: 0x01;\n    readonly DOM_KEY_LOCATION_RIGHT: 0x02;\n    readonly DOM_KEY_LOCATION_NUMPAD: 0x03;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect) */\ninterface KeyframeEffect extends AnimationEffect {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect/composite) */\n    composite: CompositeOperation;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect/iterationComposite) */\n    iterationComposite: IterationCompositeOperation;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect/pseudoElement) */\n    pseudoElement: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect/target) */\n    target: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect/getKeyframes) */\n    getKeyframes(): ComputedKeyframe[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/KeyframeEffect/setKeyframes) */\n    setKeyframes(keyframes: Keyframe[] | PropertyIndexedKeyframes | null): void;\n}\n\ndeclare var KeyframeEffect: {\n    prototype: KeyframeEffect;\n    new(target: Element | null, keyframes: Keyframe[] | PropertyIndexedKeyframes | null, options?: number | KeyframeEffectOptions): KeyframeEffect;\n    new(source: KeyframeEffect): KeyframeEffect;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint) */\ninterface LargestContentfulPaint extends PerformanceEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/element) */\n    readonly element: Element | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/id) */\n    readonly id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/loadTime) */\n    readonly loadTime: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/renderTime) */\n    readonly renderTime: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/size) */\n    readonly size: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/url) */\n    readonly url: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LargestContentfulPaint/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var LargestContentfulPaint: {\n    prototype: LargestContentfulPaint;\n    new(): LargestContentfulPaint;\n};\n\ninterface LinkStyle {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLLinkElement/sheet) */\n    readonly sheet: CSSStyleSheet | null;\n}\n\n/**\n * The location (URL) of the object it is linked to. Changes done on it are reflected on the object it relates to. Both the Document and Window interface have such a linked Location, accessible via Document.location and Window.location respectively.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location)\n */\ninterface Location {\n    /**\n     * Returns a DOMStringList object listing the origins of the ancestor browsing contexts, from the parent browsing context to the top-level browsing context.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/ancestorOrigins)\n     */\n    readonly ancestorOrigins: DOMStringList;\n    /**\n     * Returns the Location object's URL's fragment (includes leading \"#\" if non-empty).\n     *\n     * Can be set, to navigate to the same URL with a changed fragment (ignores leading \"#\").\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/hash)\n     */\n    hash: string;\n    /**\n     * Returns the Location object's URL's host and port (if different from the default port for the scheme).\n     *\n     * Can be set, to navigate to the same URL with a changed host and port.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/host)\n     */\n    host: string;\n    /**\n     * Returns the Location object's URL's host.\n     *\n     * Can be set, to navigate to the same URL with a changed host.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/hostname)\n     */\n    hostname: string;\n    /**\n     * Returns the Location object's URL.\n     *\n     * Can be set, to navigate to the given URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/href)\n     */\n    href: string;\n    toString(): string;\n    /**\n     * Returns the Location object's URL's origin.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/origin)\n     */\n    readonly origin: string;\n    /**\n     * Returns the Location object's URL's path.\n     *\n     * Can be set, to navigate to the same URL with a changed path.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/pathname)\n     */\n    pathname: string;\n    /**\n     * Returns the Location object's URL's port.\n     *\n     * Can be set, to navigate to the same URL with a changed port.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/port)\n     */\n    port: string;\n    /**\n     * Returns the Location object's URL's scheme.\n     *\n     * Can be set, to navigate to the same URL with a changed scheme.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/protocol)\n     */\n    protocol: string;\n    /**\n     * Returns the Location object's URL's query (includes leading \"?\" if non-empty).\n     *\n     * Can be set, to navigate to the same URL with a changed query (ignores leading \"?\").\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/search)\n     */\n    search: string;\n    /**\n     * Navigates to the given URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/assign)\n     */\n    assign(url: string | URL): void;\n    /**\n     * Reloads the current page.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/reload)\n     */\n    reload(): void;\n    /**\n     * Removes the current page from the session history and navigates to the given URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Location/replace)\n     */\n    replace(url: string | URL): void;\n}\n\ndeclare var Location: {\n    prototype: Location;\n    new(): Location;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Lock)\n */\ninterface Lock {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Lock/mode) */\n    readonly mode: LockMode;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Lock/name) */\n    readonly name: string;\n}\n\ndeclare var Lock: {\n    prototype: Lock;\n    new(): Lock;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/LockManager)\n */\ninterface LockManager {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LockManager/query) */\n    query(): Promise<LockManagerSnapshot>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/LockManager/request) */\n    request(name: string, callback: LockGrantedCallback): Promise<any>;\n    request(name: string, options: LockOptions, callback: LockGrantedCallback): Promise<any>;\n}\n\ndeclare var LockManager: {\n    prototype: LockManager;\n    new(): LockManager;\n};\n\ninterface MIDIAccessEventMap {\n    \"statechange\": Event;\n}\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIAccess)\n */\ninterface MIDIAccess extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIAccess/inputs) */\n    readonly inputs: MIDIInputMap;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIAccess/statechange_event) */\n    onstatechange: ((this: MIDIAccess, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIAccess/outputs) */\n    readonly outputs: MIDIOutputMap;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIAccess/sysexEnabled) */\n    readonly sysexEnabled: boolean;\n    addEventListener<K extends keyof MIDIAccessEventMap>(type: K, listener: (this: MIDIAccess, ev: MIDIAccessEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MIDIAccessEventMap>(type: K, listener: (this: MIDIAccess, ev: MIDIAccessEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MIDIAccess: {\n    prototype: MIDIAccess;\n    new(): MIDIAccess;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIConnectionEvent)\n */\ninterface MIDIConnectionEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIConnectionEvent/port) */\n    readonly port: MIDIPort | null;\n}\n\ndeclare var MIDIConnectionEvent: {\n    prototype: MIDIConnectionEvent;\n    new(type: string, eventInitDict?: MIDIConnectionEventInit): MIDIConnectionEvent;\n};\n\ninterface MIDIInputEventMap extends MIDIPortEventMap {\n    \"midimessage\": MIDIMessageEvent;\n}\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIInput)\n */\ninterface MIDIInput extends MIDIPort {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIInput/midimessage_event) */\n    onmidimessage: ((this: MIDIInput, ev: MIDIMessageEvent) => any) | null;\n    addEventListener<K extends keyof MIDIInputEventMap>(type: K, listener: (this: MIDIInput, ev: MIDIInputEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MIDIInputEventMap>(type: K, listener: (this: MIDIInput, ev: MIDIInputEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MIDIInput: {\n    prototype: MIDIInput;\n    new(): MIDIInput;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIInputMap)\n */\ninterface MIDIInputMap {\n    forEach(callbackfn: (value: MIDIInput, key: string, parent: MIDIInputMap) => void, thisArg?: any): void;\n}\n\ndeclare var MIDIInputMap: {\n    prototype: MIDIInputMap;\n    new(): MIDIInputMap;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIMessageEvent)\n */\ninterface MIDIMessageEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIMessageEvent/data) */\n    readonly data: Uint8Array | null;\n}\n\ndeclare var MIDIMessageEvent: {\n    prototype: MIDIMessageEvent;\n    new(type: string, eventInitDict?: MIDIMessageEventInit): MIDIMessageEvent;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIOutput)\n */\ninterface MIDIOutput extends MIDIPort {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIOutput/send) */\n    send(data: number[], timestamp?: DOMHighResTimeStamp): void;\n    addEventListener<K extends keyof MIDIPortEventMap>(type: K, listener: (this: MIDIOutput, ev: MIDIPortEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MIDIPortEventMap>(type: K, listener: (this: MIDIOutput, ev: MIDIPortEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MIDIOutput: {\n    prototype: MIDIOutput;\n    new(): MIDIOutput;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIOutputMap)\n */\ninterface MIDIOutputMap {\n    forEach(callbackfn: (value: MIDIOutput, key: string, parent: MIDIOutputMap) => void, thisArg?: any): void;\n}\n\ndeclare var MIDIOutputMap: {\n    prototype: MIDIOutputMap;\n    new(): MIDIOutputMap;\n};\n\ninterface MIDIPortEventMap {\n    \"statechange\": MIDIConnectionEvent;\n}\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort)\n */\ninterface MIDIPort extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/connection) */\n    readonly connection: MIDIPortConnectionState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/id) */\n    readonly id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/manufacturer) */\n    readonly manufacturer: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/name) */\n    readonly name: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/statechange_event) */\n    onstatechange: ((this: MIDIPort, ev: MIDIConnectionEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/state) */\n    readonly state: MIDIPortDeviceState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/type) */\n    readonly type: MIDIPortType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/version) */\n    readonly version: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/close) */\n    close(): Promise<MIDIPort>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MIDIPort/open) */\n    open(): Promise<MIDIPort>;\n    addEventListener<K extends keyof MIDIPortEventMap>(type: K, listener: (this: MIDIPort, ev: MIDIPortEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MIDIPortEventMap>(type: K, listener: (this: MIDIPort, ev: MIDIPortEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MIDIPort: {\n    prototype: MIDIPort;\n    new(): MIDIPort;\n};\n\ninterface MathMLElementEventMap extends ElementEventMap, GlobalEventHandlersEventMap {\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MathMLElement) */\ninterface MathMLElement extends Element, ElementCSSInlineStyle, GlobalEventHandlers, HTMLOrSVGElement {\n    addEventListener<K extends keyof MathMLElementEventMap>(type: K, listener: (this: MathMLElement, ev: MathMLElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MathMLElementEventMap>(type: K, listener: (this: MathMLElement, ev: MathMLElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MathMLElement: {\n    prototype: MathMLElement;\n    new(): MathMLElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaCapabilities) */\ninterface MediaCapabilities {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaCapabilities/decodingInfo) */\n    decodingInfo(configuration: MediaDecodingConfiguration): Promise<MediaCapabilitiesDecodingInfo>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaCapabilities/encodingInfo) */\n    encodingInfo(configuration: MediaEncodingConfiguration): Promise<MediaCapabilitiesEncodingInfo>;\n}\n\ndeclare var MediaCapabilities: {\n    prototype: MediaCapabilities;\n    new(): MediaCapabilities;\n};\n\n/**\n * The MediaDevicesInfo interface contains information that describes a single media input or output device.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDeviceInfo)\n */\ninterface MediaDeviceInfo {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDeviceInfo/deviceId) */\n    readonly deviceId: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDeviceInfo/groupId) */\n    readonly groupId: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDeviceInfo/kind) */\n    readonly kind: MediaDeviceKind;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDeviceInfo/label) */\n    readonly label: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDeviceInfo/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var MediaDeviceInfo: {\n    prototype: MediaDeviceInfo;\n    new(): MediaDeviceInfo;\n};\n\ninterface MediaDevicesEventMap {\n    \"devicechange\": Event;\n}\n\n/**\n * Provides access to connected media input devices like cameras and microphones, as well as screen sharing. In essence, it lets you obtain access to any hardware source of media data.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDevices)\n */\ninterface MediaDevices extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDevices/devicechange_event) */\n    ondevicechange: ((this: MediaDevices, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDevices/enumerateDevices) */\n    enumerateDevices(): Promise<MediaDeviceInfo[]>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDevices/getDisplayMedia) */\n    getDisplayMedia(options?: DisplayMediaStreamOptions): Promise<MediaStream>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDevices/getSupportedConstraints) */\n    getSupportedConstraints(): MediaTrackSupportedConstraints;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaDevices/getUserMedia) */\n    getUserMedia(constraints?: MediaStreamConstraints): Promise<MediaStream>;\n    addEventListener<K extends keyof MediaDevicesEventMap>(type: K, listener: (this: MediaDevices, ev: MediaDevicesEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaDevicesEventMap>(type: K, listener: (this: MediaDevices, ev: MediaDevicesEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaDevices: {\n    prototype: MediaDevices;\n    new(): MediaDevices;\n};\n\n/**\n * A MediaElementSourceNode has no inputs and exactly one output, and is created using the AudioContext.createMediaElementSource method. The amount of channels in the output equals the number of channels of the audio referenced by the HTMLMediaElement used in the creation of the node, or is 1 if the HTMLMediaElement has no audio.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaElementAudioSourceNode)\n */\ninterface MediaElementAudioSourceNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaElementAudioSourceNode/mediaElement) */\n    readonly mediaElement: HTMLMediaElement;\n}\n\ndeclare var MediaElementAudioSourceNode: {\n    prototype: MediaElementAudioSourceNode;\n    new(context: AudioContext, options: MediaElementAudioSourceOptions): MediaElementAudioSourceNode;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaEncryptedEvent) */\ninterface MediaEncryptedEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaEncryptedEvent/initData) */\n    readonly initData: ArrayBuffer | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaEncryptedEvent/initDataType) */\n    readonly initDataType: string;\n}\n\ndeclare var MediaEncryptedEvent: {\n    prototype: MediaEncryptedEvent;\n    new(type: string, eventInitDict?: MediaEncryptedEventInit): MediaEncryptedEvent;\n};\n\n/**\n * An error which occurred while handling media in an HTML media element based on HTMLMediaElement, such as <audio> or <video>.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaError)\n */\ninterface MediaError {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaError/code) */\n    readonly code: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaError/message) */\n    readonly message: string;\n    readonly MEDIA_ERR_ABORTED: 1;\n    readonly MEDIA_ERR_NETWORK: 2;\n    readonly MEDIA_ERR_DECODE: 3;\n    readonly MEDIA_ERR_SRC_NOT_SUPPORTED: 4;\n}\n\ndeclare var MediaError: {\n    prototype: MediaError;\n    new(): MediaError;\n    readonly MEDIA_ERR_ABORTED: 1;\n    readonly MEDIA_ERR_NETWORK: 2;\n    readonly MEDIA_ERR_DECODE: 3;\n    readonly MEDIA_ERR_SRC_NOT_SUPPORTED: 4;\n};\n\n/**\n * This EncryptedMediaExtensions API interface contains the content and related data when the content decryption module generates a message for the session.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyMessageEvent)\n */\ninterface MediaKeyMessageEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyMessageEvent/message) */\n    readonly message: ArrayBuffer;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyMessageEvent/messageType) */\n    readonly messageType: MediaKeyMessageType;\n}\n\ndeclare var MediaKeyMessageEvent: {\n    prototype: MediaKeyMessageEvent;\n    new(type: string, eventInitDict: MediaKeyMessageEventInit): MediaKeyMessageEvent;\n};\n\ninterface MediaKeySessionEventMap {\n    \"keystatuseschange\": Event;\n    \"message\": MediaKeyMessageEvent;\n}\n\n/**\n * This EncryptedMediaExtensions API interface represents a context for message exchange with a content decryption module (CDM).\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession)\n */\ninterface MediaKeySession extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/closed) */\n    readonly closed: Promise<MediaKeySessionClosedReason>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/expiration) */\n    readonly expiration: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/keyStatuses) */\n    readonly keyStatuses: MediaKeyStatusMap;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/keystatuseschange_event) */\n    onkeystatuseschange: ((this: MediaKeySession, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/message_event) */\n    onmessage: ((this: MediaKeySession, ev: MediaKeyMessageEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/sessionId) */\n    readonly sessionId: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/close) */\n    close(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/generateRequest) */\n    generateRequest(initDataType: string, initData: BufferSource): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/load) */\n    load(sessionId: string): Promise<boolean>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/remove) */\n    remove(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySession/update) */\n    update(response: BufferSource): Promise<void>;\n    addEventListener<K extends keyof MediaKeySessionEventMap>(type: K, listener: (this: MediaKeySession, ev: MediaKeySessionEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaKeySessionEventMap>(type: K, listener: (this: MediaKeySession, ev: MediaKeySessionEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaKeySession: {\n    prototype: MediaKeySession;\n    new(): MediaKeySession;\n};\n\n/**\n * This EncryptedMediaExtensions API interface is a read-only map of media key statuses by key IDs.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyStatusMap)\n */\ninterface MediaKeyStatusMap {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyStatusMap/size) */\n    readonly size: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyStatusMap/get) */\n    get(keyId: BufferSource): MediaKeyStatus | undefined;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeyStatusMap/has) */\n    has(keyId: BufferSource): boolean;\n    forEach(callbackfn: (value: MediaKeyStatus, key: BufferSource, parent: MediaKeyStatusMap) => void, thisArg?: any): void;\n}\n\ndeclare var MediaKeyStatusMap: {\n    prototype: MediaKeyStatusMap;\n    new(): MediaKeyStatusMap;\n};\n\n/**\n * This EncryptedMediaExtensions API interface provides access to a Key System for decryption and/or a content protection provider. You can request an instance of this object using the Navigator.requestMediaKeySystemAccess method.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySystemAccess)\n */\ninterface MediaKeySystemAccess {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySystemAccess/keySystem) */\n    readonly keySystem: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySystemAccess/createMediaKeys) */\n    createMediaKeys(): Promise<MediaKeys>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeySystemAccess/getConfiguration) */\n    getConfiguration(): MediaKeySystemConfiguration;\n}\n\ndeclare var MediaKeySystemAccess: {\n    prototype: MediaKeySystemAccess;\n    new(): MediaKeySystemAccess;\n};\n\n/**\n * This EncryptedMediaExtensions API interface the represents a set of keys that an associated HTMLMediaElement can use for decryption of media data during playback.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeys)\n */\ninterface MediaKeys {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeys/createSession) */\n    createSession(sessionType?: MediaKeySessionType): MediaKeySession;\n    getStatusForPolicy(policy?: MediaKeysPolicy): Promise<MediaKeyStatus>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaKeys/setServerCertificate) */\n    setServerCertificate(serverCertificate: BufferSource): Promise<boolean>;\n}\n\ndeclare var MediaKeys: {\n    prototype: MediaKeys;\n    new(): MediaKeys;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaList) */\ninterface MediaList {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaList/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaList/mediaText) */\n    mediaText: string;\n    toString(): string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaList/appendMedium) */\n    appendMedium(medium: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaList/deleteMedium) */\n    deleteMedium(medium: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaList/item) */\n    item(index: number): string | null;\n    [index: number]: string;\n}\n\ndeclare var MediaList: {\n    prototype: MediaList;\n    new(): MediaList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaMetadata) */\ninterface MediaMetadata {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaMetadata/album) */\n    album: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaMetadata/artist) */\n    artist: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaMetadata/artwork) */\n    artwork: ReadonlyArray<MediaImage>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaMetadata/title) */\n    title: string;\n}\n\ndeclare var MediaMetadata: {\n    prototype: MediaMetadata;\n    new(init?: MediaMetadataInit): MediaMetadata;\n};\n\ninterface MediaQueryListEventMap {\n    \"change\": MediaQueryListEvent;\n}\n\n/**\n * Stores information on a media query applied to a document, and handles sending notifications to listeners when the media query state change (i.e. when the media query test starts or stops evaluating to true).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryList)\n */\ninterface MediaQueryList extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryList/matches) */\n    readonly matches: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryList/media) */\n    readonly media: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryList/change_event) */\n    onchange: ((this: MediaQueryList, ev: MediaQueryListEvent) => any) | null;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryList/addListener)\n     */\n    addListener(callback: ((this: MediaQueryList, ev: MediaQueryListEvent) => any) | null): void;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryList/removeListener)\n     */\n    removeListener(callback: ((this: MediaQueryList, ev: MediaQueryListEvent) => any) | null): void;\n    addEventListener<K extends keyof MediaQueryListEventMap>(type: K, listener: (this: MediaQueryList, ev: MediaQueryListEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaQueryListEventMap>(type: K, listener: (this: MediaQueryList, ev: MediaQueryListEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaQueryList: {\n    prototype: MediaQueryList;\n    new(): MediaQueryList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryListEvent) */\ninterface MediaQueryListEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryListEvent/matches) */\n    readonly matches: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaQueryListEvent/media) */\n    readonly media: string;\n}\n\ndeclare var MediaQueryListEvent: {\n    prototype: MediaQueryListEvent;\n    new(type: string, eventInitDict?: MediaQueryListEventInit): MediaQueryListEvent;\n};\n\ninterface MediaRecorderEventMap {\n    \"dataavailable\": BlobEvent;\n    \"error\": Event;\n    \"pause\": Event;\n    \"resume\": Event;\n    \"start\": Event;\n    \"stop\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder) */\ninterface MediaRecorder extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/audioBitsPerSecond) */\n    readonly audioBitsPerSecond: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/mimeType) */\n    readonly mimeType: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/dataavailable_event) */\n    ondataavailable: ((this: MediaRecorder, ev: BlobEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/error_event) */\n    onerror: ((this: MediaRecorder, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/pause_event) */\n    onpause: ((this: MediaRecorder, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/resume_event) */\n    onresume: ((this: MediaRecorder, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/start_event) */\n    onstart: ((this: MediaRecorder, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/stop_event) */\n    onstop: ((this: MediaRecorder, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/state) */\n    readonly state: RecordingState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/stream) */\n    readonly stream: MediaStream;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/videoBitsPerSecond) */\n    readonly videoBitsPerSecond: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/pause) */\n    pause(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/requestData) */\n    requestData(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/resume) */\n    resume(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/start) */\n    start(timeslice?: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/stop) */\n    stop(): void;\n    addEventListener<K extends keyof MediaRecorderEventMap>(type: K, listener: (this: MediaRecorder, ev: MediaRecorderEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaRecorderEventMap>(type: K, listener: (this: MediaRecorder, ev: MediaRecorderEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaRecorder: {\n    prototype: MediaRecorder;\n    new(stream: MediaStream, options?: MediaRecorderOptions): MediaRecorder;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaRecorder/isTypeSupported_static) */\n    isTypeSupported(type: string): boolean;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSession) */\ninterface MediaSession {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSession/metadata) */\n    metadata: MediaMetadata | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSession/playbackState) */\n    playbackState: MediaSessionPlaybackState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSession/setActionHandler) */\n    setActionHandler(action: MediaSessionAction, handler: MediaSessionActionHandler | null): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSession/setPositionState) */\n    setPositionState(state?: MediaPositionState): void;\n}\n\ndeclare var MediaSession: {\n    prototype: MediaSession;\n    new(): MediaSession;\n};\n\ninterface MediaSourceEventMap {\n    \"sourceclose\": Event;\n    \"sourceended\": Event;\n    \"sourceopen\": Event;\n}\n\n/**\n * This Media Source Extensions API interface represents a source of media data for an HTMLMediaElement object. A MediaSource object can be attached to a HTMLMediaElement to be played in the user agent.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource)\n */\ninterface MediaSource extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/activeSourceBuffers) */\n    readonly activeSourceBuffers: SourceBufferList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/duration) */\n    duration: number;\n    onsourceclose: ((this: MediaSource, ev: Event) => any) | null;\n    onsourceended: ((this: MediaSource, ev: Event) => any) | null;\n    onsourceopen: ((this: MediaSource, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/readyState) */\n    readonly readyState: ReadyState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/sourceBuffers) */\n    readonly sourceBuffers: SourceBufferList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/addSourceBuffer) */\n    addSourceBuffer(type: string): SourceBuffer;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/clearLiveSeekableRange) */\n    clearLiveSeekableRange(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/endOfStream) */\n    endOfStream(error?: EndOfStreamError): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/removeSourceBuffer) */\n    removeSourceBuffer(sourceBuffer: SourceBuffer): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/setLiveSeekableRange) */\n    setLiveSeekableRange(start: number, end: number): void;\n    addEventListener<K extends keyof MediaSourceEventMap>(type: K, listener: (this: MediaSource, ev: MediaSourceEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaSourceEventMap>(type: K, listener: (this: MediaSource, ev: MediaSourceEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaSource: {\n    prototype: MediaSource;\n    new(): MediaSource;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/canConstructInDedicatedWorker_static) */\n    readonly canConstructInDedicatedWorker: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSource/isTypeSupported_static) */\n    isTypeSupported(type: string): boolean;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaSourceHandle) */\ninterface MediaSourceHandle {\n}\n\ndeclare var MediaSourceHandle: {\n    prototype: MediaSourceHandle;\n    new(): MediaSourceHandle;\n};\n\ninterface MediaStreamEventMap {\n    \"addtrack\": MediaStreamTrackEvent;\n    \"removetrack\": MediaStreamTrackEvent;\n}\n\n/**\n * A stream of media content. A stream consists of several tracks such as video or audio tracks. Each track is specified as an instance of MediaStreamTrack.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream)\n */\ninterface MediaStream extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/active) */\n    readonly active: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/id) */\n    readonly id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/addtrack_event) */\n    onaddtrack: ((this: MediaStream, ev: MediaStreamTrackEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/removetrack_event) */\n    onremovetrack: ((this: MediaStream, ev: MediaStreamTrackEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/addTrack) */\n    addTrack(track: MediaStreamTrack): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/clone) */\n    clone(): MediaStream;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/getAudioTracks) */\n    getAudioTracks(): MediaStreamTrack[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/getTrackById) */\n    getTrackById(trackId: string): MediaStreamTrack | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/getTracks) */\n    getTracks(): MediaStreamTrack[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/getVideoTracks) */\n    getVideoTracks(): MediaStreamTrack[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStream/removeTrack) */\n    removeTrack(track: MediaStreamTrack): void;\n    addEventListener<K extends keyof MediaStreamEventMap>(type: K, listener: (this: MediaStream, ev: MediaStreamEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaStreamEventMap>(type: K, listener: (this: MediaStream, ev: MediaStreamEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaStream: {\n    prototype: MediaStream;\n    new(): MediaStream;\n    new(stream: MediaStream): MediaStream;\n    new(tracks: MediaStreamTrack[]): MediaStream;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamAudioDestinationNode) */\ninterface MediaStreamAudioDestinationNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamAudioDestinationNode/stream) */\n    readonly stream: MediaStream;\n}\n\ndeclare var MediaStreamAudioDestinationNode: {\n    prototype: MediaStreamAudioDestinationNode;\n    new(context: AudioContext, options?: AudioNodeOptions): MediaStreamAudioDestinationNode;\n};\n\n/**\n * A type of AudioNode which operates as an audio source whose media is received from a MediaStream obtained using the WebRTC or Media Capture and Streams APIs.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamAudioSourceNode)\n */\ninterface MediaStreamAudioSourceNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamAudioSourceNode/mediaStream) */\n    readonly mediaStream: MediaStream;\n}\n\ndeclare var MediaStreamAudioSourceNode: {\n    prototype: MediaStreamAudioSourceNode;\n    new(context: AudioContext, options: MediaStreamAudioSourceOptions): MediaStreamAudioSourceNode;\n};\n\ninterface MediaStreamTrackEventMap {\n    \"ended\": Event;\n    \"mute\": Event;\n    \"unmute\": Event;\n}\n\n/**\n * A single media track within a stream; typically, these are audio or video tracks, but other track types may exist as well.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack)\n */\ninterface MediaStreamTrack extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/contentHint) */\n    contentHint: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/enabled) */\n    enabled: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/id) */\n    readonly id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/kind) */\n    readonly kind: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/label) */\n    readonly label: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/muted) */\n    readonly muted: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/ended_event) */\n    onended: ((this: MediaStreamTrack, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/mute_event) */\n    onmute: ((this: MediaStreamTrack, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/unmute_event) */\n    onunmute: ((this: MediaStreamTrack, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/readyState) */\n    readonly readyState: MediaStreamTrackState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/applyConstraints) */\n    applyConstraints(constraints?: MediaTrackConstraints): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/clone) */\n    clone(): MediaStreamTrack;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/getCapabilities) */\n    getCapabilities(): MediaTrackCapabilities;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/getConstraints) */\n    getConstraints(): MediaTrackConstraints;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/getSettings) */\n    getSettings(): MediaTrackSettings;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrack/stop) */\n    stop(): void;\n    addEventListener<K extends keyof MediaStreamTrackEventMap>(type: K, listener: (this: MediaStreamTrack, ev: MediaStreamTrackEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MediaStreamTrackEventMap>(type: K, listener: (this: MediaStreamTrack, ev: MediaStreamTrackEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MediaStreamTrack: {\n    prototype: MediaStreamTrack;\n    new(): MediaStreamTrack;\n};\n\n/**\n * Events which indicate that a MediaStream has had tracks added to or removed from the stream through calls to Media Stream API methods. These events are sent to the stream when these changes occur.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrackEvent)\n */\ninterface MediaStreamTrackEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MediaStreamTrackEvent/track) */\n    readonly track: MediaStreamTrack;\n}\n\ndeclare var MediaStreamTrackEvent: {\n    prototype: MediaStreamTrackEvent;\n    new(type: string, eventInitDict: MediaStreamTrackEventInit): MediaStreamTrackEvent;\n};\n\n/**\n * This Channel Messaging API interface allows us to create a new message channel and send data through it via its two MessagePort properties.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageChannel)\n */\ninterface MessageChannel {\n    /**\n     * Returns the first MessagePort object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageChannel/port1)\n     */\n    readonly port1: MessagePort;\n    /**\n     * Returns the second MessagePort object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageChannel/port2)\n     */\n    readonly port2: MessagePort;\n}\n\ndeclare var MessageChannel: {\n    prototype: MessageChannel;\n    new(): MessageChannel;\n};\n\n/**\n * A message received by a target object.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageEvent)\n */\ninterface MessageEvent<T = any> extends Event {\n    /**\n     * Returns the data of the message.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageEvent/data)\n     */\n    readonly data: T;\n    /**\n     * Returns the last event ID string, for server-sent events.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageEvent/lastEventId)\n     */\n    readonly lastEventId: string;\n    /**\n     * Returns the origin of the message, for server-sent events and cross-document messaging.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageEvent/origin)\n     */\n    readonly origin: string;\n    /**\n     * Returns the MessagePort array sent with the message, for cross-document messaging and channel messaging.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageEvent/ports)\n     */\n    readonly ports: ReadonlyArray<MessagePort>;\n    /**\n     * Returns the WindowProxy of the source window, for cross-document messaging, and the MessagePort being attached, in the connect event fired at SharedWorkerGlobalScope objects.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessageEvent/source)\n     */\n    readonly source: MessageEventSource | null;\n    /** @deprecated */\n    initMessageEvent(type: string, bubbles?: boolean, cancelable?: boolean, data?: any, origin?: string, lastEventId?: string, source?: MessageEventSource | null, ports?: MessagePort[]): void;\n}\n\ndeclare var MessageEvent: {\n    prototype: MessageEvent;\n    new<T>(type: string, eventInitDict?: MessageEventInit<T>): MessageEvent<T>;\n};\n\ninterface MessagePortEventMap {\n    \"message\": MessageEvent;\n    \"messageerror\": MessageEvent;\n}\n\n/**\n * This Channel Messaging API interface represents one of the two ports of a MessageChannel, allowing messages to be sent from one port and listening out for them arriving at the other.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessagePort)\n */\ninterface MessagePort extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessagePort/message_event) */\n    onmessage: ((this: MessagePort, ev: MessageEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessagePort/messageerror_event) */\n    onmessageerror: ((this: MessagePort, ev: MessageEvent) => any) | null;\n    /**\n     * Disconnects the port, so that it is no longer active.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessagePort/close)\n     */\n    close(): void;\n    /**\n     * Posts a message through the channel. Objects listed in transfer are transferred, not just cloned, meaning that they are no longer usable on the sending side.\n     *\n     * Throws a \"DataCloneError\" DOMException if transfer contains duplicate objects or port, or if message could not be cloned.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessagePort/postMessage)\n     */\n    postMessage(message: any, transfer: Transferable[]): void;\n    postMessage(message: any, options?: StructuredSerializeOptions): void;\n    /**\n     * Begins dispatching messages received on the port.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MessagePort/start)\n     */\n    start(): void;\n    addEventListener<K extends keyof MessagePortEventMap>(type: K, listener: (this: MessagePort, ev: MessagePortEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof MessagePortEventMap>(type: K, listener: (this: MessagePort, ev: MessagePortEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var MessagePort: {\n    prototype: MessagePort;\n    new(): MessagePort;\n};\n\n/**\n * Provides contains information about a MIME type associated with a particular plugin. NavigatorPlugins.mimeTypes returns an array of this object.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MimeType)\n */\ninterface MimeType {\n    /**\n     * Returns the MIME type's description.\n     * @deprecated\n     */\n    readonly description: string;\n    /**\n     * Returns the Plugin object that implements this MIME type.\n     * @deprecated\n     */\n    readonly enabledPlugin: Plugin;\n    /**\n     * Returns the MIME type's typical file extensions, in a comma-separated list.\n     * @deprecated\n     */\n    readonly suffixes: string;\n    /**\n     * Returns the MIME type.\n     * @deprecated\n     */\n    readonly type: string;\n}\n\n/** @deprecated */\ndeclare var MimeType: {\n    prototype: MimeType;\n    new(): MimeType;\n};\n\n/**\n * Returns an array of MimeType instances, each of which contains information about a supported browser plugins. This object is returned by NavigatorPlugins.mimeTypes.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MimeTypeArray)\n */\ninterface MimeTypeArray {\n    /** @deprecated */\n    readonly length: number;\n    /** @deprecated */\n    item(index: number): MimeType | null;\n    /** @deprecated */\n    namedItem(name: string): MimeType | null;\n    [index: number]: MimeType;\n}\n\n/** @deprecated */\ndeclare var MimeTypeArray: {\n    prototype: MimeTypeArray;\n    new(): MimeTypeArray;\n};\n\n/**\n * Events that occur due to the user interacting with a pointing device (such as a mouse). Common events using this interface include click, dblclick, mouseup, mousedown.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent)\n */\ninterface MouseEvent extends UIEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/altKey) */\n    readonly altKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/button) */\n    readonly button: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/buttons) */\n    readonly buttons: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/clientX) */\n    readonly clientX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/clientY) */\n    readonly clientY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/ctrlKey) */\n    readonly ctrlKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/layerX) */\n    readonly layerX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/layerY) */\n    readonly layerY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/metaKey) */\n    readonly metaKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/movementX) */\n    readonly movementX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/movementY) */\n    readonly movementY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/offsetX) */\n    readonly offsetX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/offsetY) */\n    readonly offsetY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/pageX) */\n    readonly pageX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/pageY) */\n    readonly pageY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/relatedTarget) */\n    readonly relatedTarget: EventTarget | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/screenX) */\n    readonly screenX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/screenY) */\n    readonly screenY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/shiftKey) */\n    readonly shiftKey: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/x) */\n    readonly x: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/y) */\n    readonly y: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/getModifierState) */\n    getModifierState(keyArg: string): boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MouseEvent/initMouseEvent)\n     */\n    initMouseEvent(typeArg: string, canBubbleArg: boolean, cancelableArg: boolean, viewArg: Window, detailArg: number, screenXArg: number, screenYArg: number, clientXArg: number, clientYArg: number, ctrlKeyArg: boolean, altKeyArg: boolean, shiftKeyArg: boolean, metaKeyArg: boolean, buttonArg: number, relatedTargetArg: EventTarget | null): void;\n}\n\ndeclare var MouseEvent: {\n    prototype: MouseEvent;\n    new(type: string, eventInitDict?: MouseEventInit): MouseEvent;\n};\n\n/**\n * Provides event properties that are specific to modifications to the Document Object Model (DOM) hierarchy and nodes.\n * @deprecated DOM4 [DOM] provides a new mechanism using a MutationObserver interface which addresses the use cases that mutation events solve, but in a more performant manner. Thus, this specification describes mutation events for reference and completeness of legacy behavior, but deprecates the use of the MutationEvent interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent)\n */\ninterface MutationEvent extends Event {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent/attrChange)\n     */\n    readonly attrChange: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent/attrName)\n     */\n    readonly attrName: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent/newValue)\n     */\n    readonly newValue: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent/prevValue)\n     */\n    readonly prevValue: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent/relatedNode)\n     */\n    readonly relatedNode: Node | null;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationEvent/initMutationEvent)\n     */\n    initMutationEvent(typeArg: string, bubblesArg?: boolean, cancelableArg?: boolean, relatedNodeArg?: Node | null, prevValueArg?: string, newValueArg?: string, attrNameArg?: string, attrChangeArg?: number): void;\n    readonly MODIFICATION: 1;\n    readonly ADDITION: 2;\n    readonly REMOVAL: 3;\n}\n\n/** @deprecated */\ndeclare var MutationEvent: {\n    prototype: MutationEvent;\n    new(): MutationEvent;\n    readonly MODIFICATION: 1;\n    readonly ADDITION: 2;\n    readonly REMOVAL: 3;\n};\n\n/**\n * Provides the ability to watch for changes being made to the DOM tree. It is designed as a replacement for the older Mutation Events feature which was part of the DOM3 Events specification.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationObserver)\n */\ninterface MutationObserver {\n    /**\n     * Stops observer from observing any mutations. Until the observe() method is used again, observer's callback will not be invoked.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationObserver/disconnect)\n     */\n    disconnect(): void;\n    /**\n     * Instructs the user agent to observe a given target (a node) and report any mutations based on the criteria given by options (an object).\n     *\n     * The options argument allows for setting mutation observation options via object members.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationObserver/observe)\n     */\n    observe(target: Node, options?: MutationObserverInit): void;\n    /**\n     * Empties the record queue and returns what was in there.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationObserver/takeRecords)\n     */\n    takeRecords(): MutationRecord[];\n}\n\ndeclare var MutationObserver: {\n    prototype: MutationObserver;\n    new(callback: MutationCallback): MutationObserver;\n};\n\n/**\n * A MutationRecord represents an individual DOM mutation. It is the object that is passed to MutationObserver's callback.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord)\n */\ninterface MutationRecord {\n    /**\n     * Return the nodes added and removed respectively.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/addedNodes)\n     */\n    readonly addedNodes: NodeList;\n    /**\n     * Returns the local name of the changed attribute, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/attributeName)\n     */\n    readonly attributeName: string | null;\n    /**\n     * Returns the namespace of the changed attribute, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/attributeNamespace)\n     */\n    readonly attributeNamespace: string | null;\n    /**\n     * Return the previous and next sibling respectively of the added or removed nodes, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/nextSibling)\n     */\n    readonly nextSibling: Node | null;\n    /**\n     * The return value depends on type. For \"attributes\", it is the value of the changed attribute before the change. For \"characterData\", it is the data of the changed node before the change. For \"childList\", it is null.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/oldValue)\n     */\n    readonly oldValue: string | null;\n    /**\n     * Return the previous and next sibling respectively of the added or removed nodes, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/previousSibling)\n     */\n    readonly previousSibling: Node | null;\n    /**\n     * Return the nodes added and removed respectively.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/removedNodes)\n     */\n    readonly removedNodes: NodeList;\n    /**\n     * Returns the node the mutation affected, depending on the type. For \"attributes\", it is the element whose attribute changed. For \"characterData\", it is the CharacterData node. For \"childList\", it is the node whose children changed.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/target)\n     */\n    readonly target: Node;\n    /**\n     * Returns \"attributes\" if it was an attribute mutation. \"characterData\" if it was a mutation to a CharacterData node. And \"childList\" if it was a mutation to the tree of nodes.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/MutationRecord/type)\n     */\n    readonly type: MutationRecordType;\n}\n\ndeclare var MutationRecord: {\n    prototype: MutationRecord;\n    new(): MutationRecord;\n};\n\n/**\n * A collection of Attr objects. Objects inside a NamedNodeMap are not in any particular order, unlike NodeList, although they may be accessed by an index as in an array.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap)\n */\ninterface NamedNodeMap {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/getNamedItem) */\n    getNamedItem(qualifiedName: string): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/getNamedItemNS) */\n    getNamedItemNS(namespace: string | null, localName: string): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/item) */\n    item(index: number): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/removeNamedItem) */\n    removeNamedItem(qualifiedName: string): Attr;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/removeNamedItemNS) */\n    removeNamedItemNS(namespace: string | null, localName: string): Attr;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/setNamedItem) */\n    setNamedItem(attr: Attr): Attr | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NamedNodeMap/setNamedItemNS) */\n    setNamedItemNS(attr: Attr): Attr | null;\n    [index: number]: Attr;\n}\n\ndeclare var NamedNodeMap: {\n    prototype: NamedNodeMap;\n    new(): NamedNodeMap;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NavigationPreloadManager)\n */\ninterface NavigationPreloadManager {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NavigationPreloadManager/disable) */\n    disable(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NavigationPreloadManager/enable) */\n    enable(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NavigationPreloadManager/getState) */\n    getState(): Promise<NavigationPreloadState>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NavigationPreloadManager/setHeaderValue) */\n    setHeaderValue(value: string): Promise<void>;\n}\n\ndeclare var NavigationPreloadManager: {\n    prototype: NavigationPreloadManager;\n    new(): NavigationPreloadManager;\n};\n\n/**\n * The state and the identity of the user agent. It allows scripts to query it and to register themselves to carry on some activities.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator)\n */\ninterface Navigator extends NavigatorAutomationInformation, NavigatorBadge, NavigatorConcurrentHardware, NavigatorContentUtils, NavigatorCookies, NavigatorID, NavigatorLanguage, NavigatorLocks, NavigatorOnLine, NavigatorPlugins, NavigatorStorage {\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/clipboard)\n     */\n    readonly clipboard: Clipboard;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/credentials)\n     */\n    readonly credentials: CredentialsContainer;\n    readonly doNotTrack: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/geolocation) */\n    readonly geolocation: Geolocation;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/maxTouchPoints) */\n    readonly maxTouchPoints: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/mediaCapabilities) */\n    readonly mediaCapabilities: MediaCapabilities;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/mediaDevices)\n     */\n    readonly mediaDevices: MediaDevices;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/mediaSession) */\n    readonly mediaSession: MediaSession;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/permissions) */\n    readonly permissions: Permissions;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/serviceWorker)\n     */\n    readonly serviceWorker: ServiceWorkerContainer;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/userActivation) */\n    readonly userActivation: UserActivation;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/wakeLock) */\n    readonly wakeLock: WakeLock;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/canShare)\n     */\n    canShare(data?: ShareData): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/getGamepads) */\n    getGamepads(): (Gamepad | null)[];\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/requestMIDIAccess)\n     */\n    requestMIDIAccess(options?: MIDIOptions): Promise<MIDIAccess>;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/requestMediaKeySystemAccess)\n     */\n    requestMediaKeySystemAccess(keySystem: string, supportedConfigurations: MediaKeySystemConfiguration[]): Promise<MediaKeySystemAccess>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/sendBeacon) */\n    sendBeacon(url: string | URL, data?: BodyInit | null): boolean;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/share)\n     */\n    share(data?: ShareData): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/vibrate) */\n    vibrate(pattern: VibratePattern): boolean;\n}\n\ndeclare var Navigator: {\n    prototype: Navigator;\n    new(): Navigator;\n};\n\ninterface NavigatorAutomationInformation {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/webdriver) */\n    readonly webdriver: boolean;\n}\n\n/** Available only in secure contexts. */\ninterface NavigatorBadge {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/clearAppBadge) */\n    clearAppBadge(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/setAppBadge) */\n    setAppBadge(contents?: number): Promise<void>;\n}\n\ninterface NavigatorConcurrentHardware {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/hardwareConcurrency) */\n    readonly hardwareConcurrency: number;\n}\n\ninterface NavigatorContentUtils {\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/registerProtocolHandler)\n     */\n    registerProtocolHandler(scheme: string, url: string | URL): void;\n}\n\ninterface NavigatorCookies {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/cookieEnabled) */\n    readonly cookieEnabled: boolean;\n}\n\ninterface NavigatorID {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/appCodeName)\n     */\n    readonly appCodeName: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/appName)\n     */\n    readonly appName: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/appVersion)\n     */\n    readonly appVersion: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/platform)\n     */\n    readonly platform: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/product)\n     */\n    readonly product: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/productSub)\n     */\n    readonly productSub: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/userAgent) */\n    readonly userAgent: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/vendor)\n     */\n    readonly vendor: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/vendorSub)\n     */\n    readonly vendorSub: string;\n}\n\ninterface NavigatorLanguage {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/language) */\n    readonly language: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/languages) */\n    readonly languages: ReadonlyArray<string>;\n}\n\n/** Available only in secure contexts. */\ninterface NavigatorLocks {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/locks) */\n    readonly locks: LockManager;\n}\n\ninterface NavigatorOnLine {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/onLine) */\n    readonly onLine: boolean;\n}\n\ninterface NavigatorPlugins {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/mimeTypes)\n     */\n    readonly mimeTypes: MimeTypeArray;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/pdfViewerEnabled) */\n    readonly pdfViewerEnabled: boolean;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/plugins)\n     */\n    readonly plugins: PluginArray;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/javaEnabled)\n     */\n    javaEnabled(): boolean;\n}\n\n/** Available only in secure contexts. */\ninterface NavigatorStorage {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Navigator/storage) */\n    readonly storage: StorageManager;\n}\n\n/**\n * Node is an interface from which a number of DOM API object types inherit. It allows those types to be treated similarly; for example, inheriting the same set of methods, or being tested in the same way.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node)\n */\ninterface Node extends EventTarget {\n    /**\n     * Returns node's node document's document base URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/baseURI)\n     */\n    readonly baseURI: string;\n    /**\n     * Returns the children.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/childNodes)\n     */\n    readonly childNodes: NodeListOf<ChildNode>;\n    /**\n     * Returns the first child.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/firstChild)\n     */\n    readonly firstChild: ChildNode | null;\n    /**\n     * Returns true if node is connected and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isConnected)\n     */\n    readonly isConnected: boolean;\n    /**\n     * Returns the last child.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/lastChild)\n     */\n    readonly lastChild: ChildNode | null;\n    /**\n     * Returns the next sibling.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nextSibling)\n     */\n    readonly nextSibling: ChildNode | null;\n    /**\n     * Returns a string appropriate for the type of node.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nodeName)\n     */\n    readonly nodeName: string;\n    /**\n     * Returns the type of node.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nodeType)\n     */\n    readonly nodeType: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/nodeValue) */\n    nodeValue: string | null;\n    /**\n     * Returns the node document. Returns null for documents.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/ownerDocument)\n     */\n    readonly ownerDocument: Document | null;\n    /**\n     * Returns the parent element.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/parentElement)\n     */\n    readonly parentElement: HTMLElement | null;\n    /**\n     * Returns the parent.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/parentNode)\n     */\n    readonly parentNode: ParentNode | null;\n    /**\n     * Returns the previous sibling.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/previousSibling)\n     */\n    readonly previousSibling: ChildNode | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/textContent) */\n    textContent: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/appendChild) */\n    appendChild<T extends Node>(node: T): T;\n    /**\n     * Returns a copy of node. If deep is true, the copy also includes the node's descendants.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/cloneNode)\n     */\n    cloneNode(deep?: boolean): Node;\n    /**\n     * Returns a bitmask indicating the position of other relative to node.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/compareDocumentPosition)\n     */\n    compareDocumentPosition(other: Node): number;\n    /**\n     * Returns true if other is an inclusive descendant of node, and false otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/contains)\n     */\n    contains(other: Node | null): boolean;\n    /**\n     * Returns node's root.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/getRootNode)\n     */\n    getRootNode(options?: GetRootNodeOptions): Node;\n    /**\n     * Returns whether node has children.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/hasChildNodes)\n     */\n    hasChildNodes(): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/insertBefore) */\n    insertBefore<T extends Node>(node: T, child: Node | null): T;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isDefaultNamespace) */\n    isDefaultNamespace(namespace: string | null): boolean;\n    /**\n     * Returns whether node and otherNode have the same properties.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isEqualNode)\n     */\n    isEqualNode(otherNode: Node | null): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/isSameNode) */\n    isSameNode(otherNode: Node | null): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/lookupNamespaceURI) */\n    lookupNamespaceURI(prefix: string | null): string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/lookupPrefix) */\n    lookupPrefix(namespace: string | null): string | null;\n    /**\n     * Removes empty exclusive Text nodes and concatenates the data of remaining contiguous exclusive Text nodes into the first of their nodes.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/normalize)\n     */\n    normalize(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/removeChild) */\n    removeChild<T extends Node>(child: T): T;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Node/replaceChild) */\n    replaceChild<T extends Node>(node: Node, child: T): T;\n    /** node is an element. */\n    readonly ELEMENT_NODE: 1;\n    readonly ATTRIBUTE_NODE: 2;\n    /** node is a Text node. */\n    readonly TEXT_NODE: 3;\n    /** node is a CDATASection node. */\n    readonly CDATA_SECTION_NODE: 4;\n    readonly ENTITY_REFERENCE_NODE: 5;\n    readonly ENTITY_NODE: 6;\n    /** node is a ProcessingInstruction node. */\n    readonly PROCESSING_INSTRUCTION_NODE: 7;\n    /** node is a Comment node. */\n    readonly COMMENT_NODE: 8;\n    /** node is a document. */\n    readonly DOCUMENT_NODE: 9;\n    /** node is a doctype. */\n    readonly DOCUMENT_TYPE_NODE: 10;\n    /** node is a DocumentFragment node. */\n    readonly DOCUMENT_FRAGMENT_NODE: 11;\n    readonly NOTATION_NODE: 12;\n    /** Set when node and other are not in the same tree. */\n    readonly DOCUMENT_POSITION_DISCONNECTED: 0x01;\n    /** Set when other is preceding node. */\n    readonly DOCUMENT_POSITION_PRECEDING: 0x02;\n    /** Set when other is following node. */\n    readonly DOCUMENT_POSITION_FOLLOWING: 0x04;\n    /** Set when other is an ancestor of node. */\n    readonly DOCUMENT_POSITION_CONTAINS: 0x08;\n    /** Set when other is a descendant of node. */\n    readonly DOCUMENT_POSITION_CONTAINED_BY: 0x10;\n    readonly DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC: 0x20;\n}\n\ndeclare var Node: {\n    prototype: Node;\n    new(): Node;\n    /** node is an element. */\n    readonly ELEMENT_NODE: 1;\n    readonly ATTRIBUTE_NODE: 2;\n    /** node is a Text node. */\n    readonly TEXT_NODE: 3;\n    /** node is a CDATASection node. */\n    readonly CDATA_SECTION_NODE: 4;\n    readonly ENTITY_REFERENCE_NODE: 5;\n    readonly ENTITY_NODE: 6;\n    /** node is a ProcessingInstruction node. */\n    readonly PROCESSING_INSTRUCTION_NODE: 7;\n    /** node is a Comment node. */\n    readonly COMMENT_NODE: 8;\n    /** node is a document. */\n    readonly DOCUMENT_NODE: 9;\n    /** node is a doctype. */\n    readonly DOCUMENT_TYPE_NODE: 10;\n    /** node is a DocumentFragment node. */\n    readonly DOCUMENT_FRAGMENT_NODE: 11;\n    readonly NOTATION_NODE: 12;\n    /** Set when node and other are not in the same tree. */\n    readonly DOCUMENT_POSITION_DISCONNECTED: 0x01;\n    /** Set when other is preceding node. */\n    readonly DOCUMENT_POSITION_PRECEDING: 0x02;\n    /** Set when other is following node. */\n    readonly DOCUMENT_POSITION_FOLLOWING: 0x04;\n    /** Set when other is an ancestor of node. */\n    readonly DOCUMENT_POSITION_CONTAINS: 0x08;\n    /** Set when other is a descendant of node. */\n    readonly DOCUMENT_POSITION_CONTAINED_BY: 0x10;\n    readonly DOCUMENT_POSITION_IMPLEMENTATION_SPECIFIC: 0x20;\n};\n\n/**\n * An iterator over the members of a list of the nodes in a subtree of the DOM. The nodes will be returned in document order.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator)\n */\ninterface NodeIterator {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/filter) */\n    readonly filter: NodeFilter | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/pointerBeforeReferenceNode) */\n    readonly pointerBeforeReferenceNode: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/referenceNode) */\n    readonly referenceNode: Node;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/root) */\n    readonly root: Node;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/whatToShow) */\n    readonly whatToShow: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/detach)\n     */\n    detach(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/nextNode) */\n    nextNode(): Node | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeIterator/previousNode) */\n    previousNode(): Node | null;\n}\n\ndeclare var NodeIterator: {\n    prototype: NodeIterator;\n    new(): NodeIterator;\n};\n\n/**\n * NodeList objects are collections of nodes, usually returned by properties such as Node.childNodes and methods such as document.querySelectorAll().\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeList)\n */\ninterface NodeList {\n    /**\n     * Returns the number of nodes in the collection.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeList/length)\n     */\n    readonly length: number;\n    /**\n     * Returns the node with index index from the collection. The nodes are sorted in tree order.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/NodeList/item)\n     */\n    item(index: number): Node | null;\n    /**\n     * Performs the specified action for each node in an list.\n     * @param callbackfn  A function that accepts up to three arguments. forEach calls the callbackfn function one time for each element in the list.\n     * @param thisArg  An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.\n     */\n    forEach(callbackfn: (value: Node, key: number, parent: NodeList) => void, thisArg?: any): void;\n    [index: number]: Node;\n}\n\ndeclare var NodeList: {\n    prototype: NodeList;\n    new(): NodeList;\n};\n\ninterface NodeListOf<TNode extends Node> extends NodeList {\n    item(index: number): TNode;\n    /**\n     * Performs the specified action for each node in an list.\n     * @param callbackfn  A function that accepts up to three arguments. forEach calls the callbackfn function one time for each element in the list.\n     * @param thisArg  An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.\n     */\n    forEach(callbackfn: (value: TNode, key: number, parent: NodeListOf<TNode>) => void, thisArg?: any): void;\n    [index: number]: TNode;\n}\n\ninterface NonDocumentTypeChildNode {\n    /**\n     * Returns the first following sibling that is an element, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/nextElementSibling)\n     */\n    readonly nextElementSibling: Element | null;\n    /**\n     * Returns the first preceding sibling that is an element, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/CharacterData/previousElementSibling)\n     */\n    readonly previousElementSibling: Element | null;\n}\n\ninterface NonElementParentNode {\n    /**\n     * Returns the first element within node's descendants whose ID is elementId.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/getElementById)\n     */\n    getElementById(elementId: string): Element | null;\n}\n\ninterface NotificationEventMap {\n    \"click\": Event;\n    \"close\": Event;\n    \"error\": Event;\n    \"show\": Event;\n}\n\n/**\n * This Notifications API interface is used to configure and display desktop notifications to the user.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification)\n */\ninterface Notification extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/badge) */\n    readonly badge: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/body) */\n    readonly body: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/data) */\n    readonly data: any;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/dir) */\n    readonly dir: NotificationDirection;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/icon) */\n    readonly icon: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/lang) */\n    readonly lang: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/click_event) */\n    onclick: ((this: Notification, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/close_event) */\n    onclose: ((this: Notification, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/error_event) */\n    onerror: ((this: Notification, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/show_event) */\n    onshow: ((this: Notification, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/requireInteraction) */\n    readonly requireInteraction: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/silent) */\n    readonly silent: boolean | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/tag) */\n    readonly tag: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/title) */\n    readonly title: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/close) */\n    close(): void;\n    addEventListener<K extends keyof NotificationEventMap>(type: K, listener: (this: Notification, ev: NotificationEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof NotificationEventMap>(type: K, listener: (this: Notification, ev: NotificationEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var Notification: {\n    prototype: Notification;\n    new(title: string, options?: NotificationOptions): Notification;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/permission_static) */\n    readonly permission: NotificationPermission;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Notification/requestPermission_static) */\n    requestPermission(deprecatedCallback?: NotificationPermissionCallback): Promise<NotificationPermission>;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed) */\ninterface OES_draw_buffers_indexed {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/blendEquationSeparateiOES) */\n    blendEquationSeparateiOES(buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/blendEquationiOES) */\n    blendEquationiOES(buf: GLuint, mode: GLenum): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/blendFuncSeparateiOES) */\n    blendFuncSeparateiOES(buf: GLuint, srcRGB: GLenum, dstRGB: GLenum, srcAlpha: GLenum, dstAlpha: GLenum): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/blendFunciOES) */\n    blendFunciOES(buf: GLuint, src: GLenum, dst: GLenum): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/colorMaskiOES) */\n    colorMaskiOES(buf: GLuint, r: GLboolean, g: GLboolean, b: GLboolean, a: GLboolean): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/disableiOES) */\n    disableiOES(target: GLenum, index: GLuint): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_draw_buffers_indexed/enableiOES) */\n    enableiOES(target: GLenum, index: GLuint): void;\n}\n\n/**\n * The OES_element_index_uint extension is part of the WebGL API and adds support for gl.UNSIGNED_INT types to WebGLRenderingContext.drawElements().\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_element_index_uint)\n */\ninterface OES_element_index_uint {\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_fbo_render_mipmap) */\ninterface OES_fbo_render_mipmap {\n}\n\n/**\n * The OES_standard_derivatives extension is part of the WebGL API and adds the GLSL derivative functions dFdx, dFdy, and fwidth.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_standard_derivatives)\n */\ninterface OES_standard_derivatives {\n    readonly FRAGMENT_SHADER_DERIVATIVE_HINT_OES: 0x8B8B;\n}\n\n/**\n * The OES_texture_float extension is part of the WebGL API and exposes floating-point pixel types for textures.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_texture_float)\n */\ninterface OES_texture_float {\n}\n\n/**\n * The OES_texture_float_linear extension is part of the WebGL API and allows linear filtering with floating-point pixel types for textures.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_texture_float_linear)\n */\ninterface OES_texture_float_linear {\n}\n\n/**\n * The OES_texture_half_float extension is part of the WebGL API and adds texture formats with 16- (aka half float) and 32-bit floating-point components.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_texture_half_float)\n */\ninterface OES_texture_half_float {\n    readonly HALF_FLOAT_OES: 0x8D61;\n}\n\n/**\n * The OES_texture_half_float_linear extension is part of the WebGL API and allows linear filtering with half floating-point pixel types for textures.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_texture_half_float_linear)\n */\ninterface OES_texture_half_float_linear {\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_vertex_array_object) */\ninterface OES_vertex_array_object {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_vertex_array_object/bindVertexArrayOES) */\n    bindVertexArrayOES(arrayObject: WebGLVertexArrayObjectOES | null): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_vertex_array_object/createVertexArrayOES) */\n    createVertexArrayOES(): WebGLVertexArrayObjectOES | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_vertex_array_object/deleteVertexArrayOES) */\n    deleteVertexArrayOES(arrayObject: WebGLVertexArrayObjectOES | null): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OES_vertex_array_object/isVertexArrayOES) */\n    isVertexArrayOES(arrayObject: WebGLVertexArrayObjectOES | null): GLboolean;\n    readonly VERTEX_ARRAY_BINDING_OES: 0x85B5;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OVR_multiview2) */\ninterface OVR_multiview2 {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OVR_multiview2/framebufferTextureMultiviewOVR) */\n    framebufferTextureMultiviewOVR(target: GLenum, attachment: GLenum, texture: WebGLTexture | null, level: GLint, baseViewIndex: GLint, numViews: GLsizei): void;\n    readonly FRAMEBUFFER_ATTACHMENT_TEXTURE_NUM_VIEWS_OVR: 0x9630;\n    readonly FRAMEBUFFER_ATTACHMENT_TEXTURE_BASE_VIEW_INDEX_OVR: 0x9632;\n    readonly MAX_VIEWS_OVR: 0x9631;\n    readonly FRAMEBUFFER_INCOMPLETE_VIEW_TARGETS_OVR: 0x9633;\n}\n\n/**\n * The Web Audio API OfflineAudioCompletionEvent interface represents events that occur when the processing of an OfflineAudioContext is terminated. The complete event implements this interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioCompletionEvent)\n */\ninterface OfflineAudioCompletionEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioCompletionEvent/renderedBuffer) */\n    readonly renderedBuffer: AudioBuffer;\n}\n\ndeclare var OfflineAudioCompletionEvent: {\n    prototype: OfflineAudioCompletionEvent;\n    new(type: string, eventInitDict: OfflineAudioCompletionEventInit): OfflineAudioCompletionEvent;\n};\n\ninterface OfflineAudioContextEventMap extends BaseAudioContextEventMap {\n    \"complete\": OfflineAudioCompletionEvent;\n}\n\n/**\n * An AudioContext interface representing an audio-processing graph built from linked together AudioNodes. In contrast with a standard AudioContext, an OfflineAudioContext doesn't render the audio to the device hardware; instead, it generates it, as fast as it can, and outputs the result to an AudioBuffer.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioContext)\n */\ninterface OfflineAudioContext extends BaseAudioContext {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioContext/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioContext/complete_event) */\n    oncomplete: ((this: OfflineAudioContext, ev: OfflineAudioCompletionEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioContext/resume) */\n    resume(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioContext/startRendering) */\n    startRendering(): Promise<AudioBuffer>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OfflineAudioContext/suspend) */\n    suspend(suspendTime: number): Promise<void>;\n    addEventListener<K extends keyof OfflineAudioContextEventMap>(type: K, listener: (this: OfflineAudioContext, ev: OfflineAudioContextEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof OfflineAudioContextEventMap>(type: K, listener: (this: OfflineAudioContext, ev: OfflineAudioContextEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var OfflineAudioContext: {\n    prototype: OfflineAudioContext;\n    new(contextOptions: OfflineAudioContextOptions): OfflineAudioContext;\n    new(numberOfChannels: number, length: number, sampleRate: number): OfflineAudioContext;\n};\n\ninterface OffscreenCanvasEventMap {\n    \"contextlost\": Event;\n    \"contextrestored\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas) */\ninterface OffscreenCanvas extends EventTarget {\n    /**\n     * These attributes return the dimensions of the OffscreenCanvas object's bitmap.\n     *\n     * They can be set, to replace the bitmap with a new, transparent black bitmap of the specified dimensions (effectively resizing it).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/height)\n     */\n    height: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/contextlost_event) */\n    oncontextlost: ((this: OffscreenCanvas, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/contextrestored_event) */\n    oncontextrestored: ((this: OffscreenCanvas, ev: Event) => any) | null;\n    /**\n     * These attributes return the dimensions of the OffscreenCanvas object's bitmap.\n     *\n     * They can be set, to replace the bitmap with a new, transparent black bitmap of the specified dimensions (effectively resizing it).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/width)\n     */\n    width: number;\n    /**\n     * Returns a promise that will fulfill with a new Blob object representing a file containing the image in the OffscreenCanvas object.\n     *\n     * The argument, if provided, is a dictionary that controls the encoding options of the image file to be created. The type field specifies the file format and has a default value of \"image/png\"; that type is also used if the requested type isn't supported. If the image format supports variable quality (such as \"image/jpeg\"), then the quality field is a number in the range 0.0 to 1.0 inclusive indicating the desired quality level for the resulting image.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/convertToBlob)\n     */\n    convertToBlob(options?: ImageEncodeOptions): Promise<Blob>;\n    /**\n     * Returns an object that exposes an API for drawing on the OffscreenCanvas object. contextId specifies the desired API: \"2d\", \"bitmaprenderer\", \"webgl\", or \"webgl2\". options is handled by that API.\n     *\n     * This specification defines the \"2d\" context below, which is similar but distinct from the \"2d\" context that is created from a canvas element. The WebGL specifications define the \"webgl\" and \"webgl2\" contexts. [WEBGL]\n     *\n     * Returns null if the canvas has already been initialized with another context type (e.g., trying to get a \"2d\" context after getting a \"webgl\" context).\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/getContext)\n     */\n    getContext(contextId: \"2d\", options?: any): OffscreenCanvasRenderingContext2D | null;\n    getContext(contextId: \"bitmaprenderer\", options?: any): ImageBitmapRenderingContext | null;\n    getContext(contextId: \"webgl\", options?: any): WebGLRenderingContext | null;\n    getContext(contextId: \"webgl2\", options?: any): WebGL2RenderingContext | null;\n    getContext(contextId: OffscreenRenderingContextId, options?: any): OffscreenRenderingContext | null;\n    /**\n     * Returns a newly created ImageBitmap object with the image in the OffscreenCanvas object. The image in the OffscreenCanvas object is replaced with a new blank image.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvas/transferToImageBitmap)\n     */\n    transferToImageBitmap(): ImageBitmap;\n    addEventListener<K extends keyof OffscreenCanvasEventMap>(type: K, listener: (this: OffscreenCanvas, ev: OffscreenCanvasEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof OffscreenCanvasEventMap>(type: K, listener: (this: OffscreenCanvas, ev: OffscreenCanvasEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var OffscreenCanvas: {\n    prototype: OffscreenCanvas;\n    new(width: number, height: number): OffscreenCanvas;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OffscreenCanvasRenderingContext2D) */\ninterface OffscreenCanvasRenderingContext2D extends CanvasCompositing, CanvasDrawImage, CanvasDrawPath, CanvasFillStrokeStyles, CanvasFilters, CanvasImageData, CanvasImageSmoothing, CanvasPath, CanvasPathDrawingStyles, CanvasRect, CanvasShadowStyles, CanvasState, CanvasText, CanvasTextDrawingStyles, CanvasTransform {\n    readonly canvas: OffscreenCanvas;\n}\n\ndeclare var OffscreenCanvasRenderingContext2D: {\n    prototype: OffscreenCanvasRenderingContext2D;\n    new(): OffscreenCanvasRenderingContext2D;\n};\n\n/**\n * The OscillatorNode interface represents a periodic waveform, such as a sine wave. It is an AudioScheduledSourceNode audio-processing module that causes a specified frequency of a given wave to be created—in effect, a constant tone.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/OscillatorNode)\n */\ninterface OscillatorNode extends AudioScheduledSourceNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OscillatorNode/detune) */\n    readonly detune: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OscillatorNode/frequency) */\n    readonly frequency: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OscillatorNode/type) */\n    type: OscillatorType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OscillatorNode/setPeriodicWave) */\n    setPeriodicWave(periodicWave: PeriodicWave): void;\n    addEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type: K, listener: (this: OscillatorNode, ev: AudioScheduledSourceNodeEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof AudioScheduledSourceNodeEventMap>(type: K, listener: (this: OscillatorNode, ev: AudioScheduledSourceNodeEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var OscillatorNode: {\n    prototype: OscillatorNode;\n    new(context: BaseAudioContext, options?: OscillatorOptions): OscillatorNode;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OverconstrainedError) */\ninterface OverconstrainedError extends DOMException {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/OverconstrainedError/constraint) */\n    readonly constraint: string;\n}\n\ndeclare var OverconstrainedError: {\n    prototype: OverconstrainedError;\n    new(constraint: string, message?: string): OverconstrainedError;\n};\n\n/**\n * The PageTransitionEvent is fired when a document is being loaded or unloaded.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PageTransitionEvent)\n */\ninterface PageTransitionEvent extends Event {\n    /**\n     * For the pageshow event, returns false if the page is newly being loaded (and the load event will fire). Otherwise, returns true.\n     *\n     * For the pagehide event, returns false if the page is going away for the last time. Otherwise, returns true, meaning that (if nothing conspires to make the page unsalvageable) the page might be reused if the user navigates back to this page.\n     *\n     * Things that can cause the page to be unsalvageable include:\n     *\n     * The user agent decided to not keep the Document alive in a session history entry after unload\n     * Having iframes that are not salvageable\n     * Active WebSocket objects\n     * Aborting a Document\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PageTransitionEvent/persisted)\n     */\n    readonly persisted: boolean;\n}\n\ndeclare var PageTransitionEvent: {\n    prototype: PageTransitionEvent;\n    new(type: string, eventInitDict?: PageTransitionEventInit): PageTransitionEvent;\n};\n\n/**\n * A PannerNode always has exactly one input and one output: the input can be mono or stereo but the output is always stereo (2 channels); you can't have panning effects without at least two audio channels!\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode)\n */\ninterface PannerNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/coneInnerAngle) */\n    coneInnerAngle: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/coneOuterAngle) */\n    coneOuterAngle: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/coneOuterGain) */\n    coneOuterGain: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/distanceModel) */\n    distanceModel: DistanceModelType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/maxDistance) */\n    maxDistance: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/orientationX) */\n    readonly orientationX: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/orientationY) */\n    readonly orientationY: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/orientationZ) */\n    readonly orientationZ: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/panningModel) */\n    panningModel: PanningModelType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/positionX) */\n    readonly positionX: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/positionY) */\n    readonly positionY: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/positionZ) */\n    readonly positionZ: AudioParam;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/refDistance) */\n    refDistance: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/rolloffFactor) */\n    rolloffFactor: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/setOrientation)\n     */\n    setOrientation(x: number, y: number, z: number): void;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PannerNode/setPosition)\n     */\n    setPosition(x: number, y: number, z: number): void;\n}\n\ndeclare var PannerNode: {\n    prototype: PannerNode;\n    new(context: BaseAudioContext, options?: PannerOptions): PannerNode;\n};\n\ninterface ParentNode extends Node {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/childElementCount) */\n    readonly childElementCount: number;\n    /**\n     * Returns the child elements.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/children)\n     */\n    readonly children: HTMLCollection;\n    /**\n     * Returns the first child that is an element, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/firstElementChild)\n     */\n    readonly firstElementChild: Element | null;\n    /**\n     * Returns the last child that is an element, and null otherwise.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/lastElementChild)\n     */\n    readonly lastElementChild: Element | null;\n    /**\n     * Inserts nodes after the last child of node, while replacing strings in nodes with equivalent Text nodes.\n     *\n     * Throws a \"HierarchyRequestError\" DOMException if the constraints of the node tree are violated.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/append)\n     */\n    append(...nodes: (Node | string)[]): void;\n    /**\n     * Inserts nodes before the first child of node, while replacing strings in nodes with equivalent Text nodes.\n     *\n     * Throws a \"HierarchyRequestError\" DOMException if the constraints of the node tree are violated.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/prepend)\n     */\n    prepend(...nodes: (Node | string)[]): void;\n    /**\n     * Returns the first element that is a descendant of node that matches selectors.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/querySelector)\n     */\n    querySelector<K extends keyof HTMLElementTagNameMap>(selectors: K): HTMLElementTagNameMap[K] | null;\n    querySelector<K extends keyof SVGElementTagNameMap>(selectors: K): SVGElementTagNameMap[K] | null;\n    querySelector<K extends keyof MathMLElementTagNameMap>(selectors: K): MathMLElementTagNameMap[K] | null;\n    /** @deprecated */\n    querySelector<K extends keyof HTMLElementDeprecatedTagNameMap>(selectors: K): HTMLElementDeprecatedTagNameMap[K] | null;\n    querySelector<E extends Element = Element>(selectors: string): E | null;\n    /**\n     * Returns all element descendants of node that match selectors.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/querySelectorAll)\n     */\n    querySelectorAll<K extends keyof HTMLElementTagNameMap>(selectors: K): NodeListOf<HTMLElementTagNameMap[K]>;\n    querySelectorAll<K extends keyof SVGElementTagNameMap>(selectors: K): NodeListOf<SVGElementTagNameMap[K]>;\n    querySelectorAll<K extends keyof MathMLElementTagNameMap>(selectors: K): NodeListOf<MathMLElementTagNameMap[K]>;\n    /** @deprecated */\n    querySelectorAll<K extends keyof HTMLElementDeprecatedTagNameMap>(selectors: K): NodeListOf<HTMLElementDeprecatedTagNameMap[K]>;\n    querySelectorAll<E extends Element = Element>(selectors: string): NodeListOf<E>;\n    /**\n     * Replace all children of node with nodes, while replacing strings in nodes with equivalent Text nodes.\n     *\n     * Throws a \"HierarchyRequestError\" DOMException if the constraints of the node tree are violated.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Document/replaceChildren)\n     */\n    replaceChildren(...nodes: (Node | string)[]): void;\n}\n\n/**\n * This Canvas 2D API interface is used to declare a path that can then be used on a CanvasRenderingContext2D object. The path methods of the CanvasRenderingContext2D interface are also present on this interface, which gives you the convenience of being able to retain and replay your path whenever desired.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Path2D)\n */\ninterface Path2D extends CanvasPath {\n    /**\n     * Adds to the path the path given by the argument.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Path2D/addPath)\n     */\n    addPath(path: Path2D, transform?: DOMMatrix2DInit): void;\n}\n\ndeclare var Path2D: {\n    prototype: Path2D;\n    new(path?: Path2D | string): Path2D;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentMethodChangeEvent)\n */\ninterface PaymentMethodChangeEvent extends PaymentRequestUpdateEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentMethodChangeEvent/methodDetails) */\n    readonly methodDetails: any;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentMethodChangeEvent/methodName) */\n    readonly methodName: string;\n}\n\ndeclare var PaymentMethodChangeEvent: {\n    prototype: PaymentMethodChangeEvent;\n    new(type: string, eventInitDict?: PaymentMethodChangeEventInit): PaymentMethodChangeEvent;\n};\n\ninterface PaymentRequestEventMap {\n    \"paymentmethodchange\": Event;\n}\n\n/**\n * This Payment Request API interface is the primary access point into the API, and lets web content and apps accept payments from the end user.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequest)\n */\ninterface PaymentRequest extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequest/id) */\n    readonly id: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequest/paymentmethodchange_event) */\n    onpaymentmethodchange: ((this: PaymentRequest, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequest/abort) */\n    abort(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequest/canMakePayment) */\n    canMakePayment(): Promise<boolean>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequest/show) */\n    show(detailsPromise?: PaymentDetailsUpdate | PromiseLike<PaymentDetailsUpdate>): Promise<PaymentResponse>;\n    addEventListener<K extends keyof PaymentRequestEventMap>(type: K, listener: (this: PaymentRequest, ev: PaymentRequestEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof PaymentRequestEventMap>(type: K, listener: (this: PaymentRequest, ev: PaymentRequestEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var PaymentRequest: {\n    prototype: PaymentRequest;\n    new(methodData: PaymentMethodData[], details: PaymentDetailsInit): PaymentRequest;\n};\n\n/**\n * This Payment Request API interface enables a web page to update the details of a PaymentRequest in response to a user action.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequestUpdateEvent)\n */\ninterface PaymentRequestUpdateEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentRequestUpdateEvent/updateWith) */\n    updateWith(detailsPromise: PaymentDetailsUpdate | PromiseLike<PaymentDetailsUpdate>): void;\n}\n\ndeclare var PaymentRequestUpdateEvent: {\n    prototype: PaymentRequestUpdateEvent;\n    new(type: string, eventInitDict?: PaymentRequestUpdateEventInit): PaymentRequestUpdateEvent;\n};\n\n/**\n * This Payment Request API interface is returned after a user selects a payment method and approves a payment request.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse)\n */\ninterface PaymentResponse extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse/details) */\n    readonly details: any;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse/methodName) */\n    readonly methodName: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse/requestId) */\n    readonly requestId: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse/complete) */\n    complete(result?: PaymentComplete): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse/retry) */\n    retry(errorFields?: PaymentValidationErrors): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PaymentResponse/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var PaymentResponse: {\n    prototype: PaymentResponse;\n    new(): PaymentResponse;\n};\n\ninterface PerformanceEventMap {\n    \"resourcetimingbufferfull\": Event;\n}\n\n/**\n * Provides access to performance-related information for the current page. It's part of the High Resolution Time API, but is enhanced by the Performance Timeline API, the Navigation Timing API, the User Timing API, and the Resource Timing API.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance)\n */\ninterface Performance extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/eventCounts) */\n    readonly eventCounts: EventCounts;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/navigation)\n     */\n    readonly navigation: PerformanceNavigation;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/resourcetimingbufferfull_event) */\n    onresourcetimingbufferfull: ((this: Performance, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/timeOrigin) */\n    readonly timeOrigin: DOMHighResTimeStamp;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/timing)\n     */\n    readonly timing: PerformanceTiming;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/clearMarks) */\n    clearMarks(markName?: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/clearMeasures) */\n    clearMeasures(measureName?: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/clearResourceTimings) */\n    clearResourceTimings(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/getEntries) */\n    getEntries(): PerformanceEntryList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/getEntriesByName) */\n    getEntriesByName(name: string, type?: string): PerformanceEntryList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/getEntriesByType) */\n    getEntriesByType(type: string): PerformanceEntryList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/mark) */\n    mark(markName: string, markOptions?: PerformanceMarkOptions): PerformanceMark;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/measure) */\n    measure(measureName: string, startOrMeasureOptions?: string | PerformanceMeasureOptions, endMark?: string): PerformanceMeasure;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/now) */\n    now(): DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/setResourceTimingBufferSize) */\n    setResourceTimingBufferSize(maxSize: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Performance/toJSON) */\n    toJSON(): any;\n    addEventListener<K extends keyof PerformanceEventMap>(type: K, listener: (this: Performance, ev: PerformanceEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof PerformanceEventMap>(type: K, listener: (this: Performance, ev: PerformanceEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var Performance: {\n    prototype: Performance;\n    new(): Performance;\n};\n\n/**\n * Encapsulates a single performance metric that is part of the performance timeline. A performance entry can be directly created by making a performance mark or measure (for example by calling the mark() method) at an explicit point in an application. Performance entries are also created in indirect ways such as loading a resource (such as an image).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEntry)\n */\ninterface PerformanceEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEntry/duration) */\n    readonly duration: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEntry/entryType) */\n    readonly entryType: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEntry/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEntry/startTime) */\n    readonly startTime: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEntry/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var PerformanceEntry: {\n    prototype: PerformanceEntry;\n    new(): PerformanceEntry;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEventTiming) */\ninterface PerformanceEventTiming extends PerformanceEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEventTiming/cancelable) */\n    readonly cancelable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEventTiming/processingEnd) */\n    readonly processingEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEventTiming/processingStart) */\n    readonly processingStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEventTiming/target) */\n    readonly target: Node | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceEventTiming/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var PerformanceEventTiming: {\n    prototype: PerformanceEventTiming;\n    new(): PerformanceEventTiming;\n};\n\n/**\n * PerformanceMark is an abstract interface for PerformanceEntry objects with an entryType of \"mark\". Entries of this type are created by calling performance.mark() to add a named DOMHighResTimeStamp (the mark) to the browser's performance timeline.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceMark)\n */\ninterface PerformanceMark extends PerformanceEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceMark/detail) */\n    readonly detail: any;\n}\n\ndeclare var PerformanceMark: {\n    prototype: PerformanceMark;\n    new(markName: string, markOptions?: PerformanceMarkOptions): PerformanceMark;\n};\n\n/**\n * PerformanceMeasure is an abstract interface for PerformanceEntry objects with an entryType of \"measure\". Entries of this type are created by calling performance.measure() to add a named DOMHighResTimeStamp (the measure) between two marks to the browser's performance timeline.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceMeasure)\n */\ninterface PerformanceMeasure extends PerformanceEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceMeasure/detail) */\n    readonly detail: any;\n}\n\ndeclare var PerformanceMeasure: {\n    prototype: PerformanceMeasure;\n    new(): PerformanceMeasure;\n};\n\n/**\n * The legacy PerformanceNavigation interface represents information about how the navigation to the current document was done.\n * @deprecated This interface is deprecated in the Navigation Timing Level 2 specification. Please use the PerformanceNavigationTiming interface instead.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigation)\n */\ninterface PerformanceNavigation {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigation/redirectCount)\n     */\n    readonly redirectCount: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigation/type)\n     */\n    readonly type: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigation/toJSON)\n     */\n    toJSON(): any;\n    readonly TYPE_NAVIGATE: 0;\n    readonly TYPE_RELOAD: 1;\n    readonly TYPE_BACK_FORWARD: 2;\n    readonly TYPE_RESERVED: 255;\n}\n\n/** @deprecated */\ndeclare var PerformanceNavigation: {\n    prototype: PerformanceNavigation;\n    new(): PerformanceNavigation;\n    readonly TYPE_NAVIGATE: 0;\n    readonly TYPE_RELOAD: 1;\n    readonly TYPE_BACK_FORWARD: 2;\n    readonly TYPE_RESERVED: 255;\n};\n\n/**\n * Provides methods and properties to store and retrieve metrics regarding the browser's document navigation events. For example, this interface can be used to determine how much time it takes to load or unload a document.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming)\n */\ninterface PerformanceNavigationTiming extends PerformanceResourceTiming {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/domComplete) */\n    readonly domComplete: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/domContentLoadedEventEnd) */\n    readonly domContentLoadedEventEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/domContentLoadedEventStart) */\n    readonly domContentLoadedEventStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/domInteractive) */\n    readonly domInteractive: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/loadEventEnd) */\n    readonly loadEventEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/loadEventStart) */\n    readonly loadEventStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/redirectCount) */\n    readonly redirectCount: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/type) */\n    readonly type: NavigationTimingType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/unloadEventEnd) */\n    readonly unloadEventEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/unloadEventStart) */\n    readonly unloadEventStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceNavigationTiming/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var PerformanceNavigationTiming: {\n    prototype: PerformanceNavigationTiming;\n    new(): PerformanceNavigationTiming;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserver) */\ninterface PerformanceObserver {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserver/disconnect) */\n    disconnect(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserver/observe) */\n    observe(options?: PerformanceObserverInit): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserver/takeRecords) */\n    takeRecords(): PerformanceEntryList;\n}\n\ndeclare var PerformanceObserver: {\n    prototype: PerformanceObserver;\n    new(callback: PerformanceObserverCallback): PerformanceObserver;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserver/supportedEntryTypes_static) */\n    readonly supportedEntryTypes: ReadonlyArray<string>;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserverEntryList) */\ninterface PerformanceObserverEntryList {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserverEntryList/getEntries) */\n    getEntries(): PerformanceEntryList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserverEntryList/getEntriesByName) */\n    getEntriesByName(name: string, type?: string): PerformanceEntryList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceObserverEntryList/getEntriesByType) */\n    getEntriesByType(type: string): PerformanceEntryList;\n}\n\ndeclare var PerformanceObserverEntryList: {\n    prototype: PerformanceObserverEntryList;\n    new(): PerformanceObserverEntryList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformancePaintTiming) */\ninterface PerformancePaintTiming extends PerformanceEntry {\n}\n\ndeclare var PerformancePaintTiming: {\n    prototype: PerformancePaintTiming;\n    new(): PerformancePaintTiming;\n};\n\n/**\n * Enables retrieval and analysis of detailed network timing data regarding the loading of an application's resources. An application can use the timing metrics to determine, for example, the length of time it takes to fetch a specific resource, such as an XMLHttpRequest, <SVG>, image, or script.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming)\n */\ninterface PerformanceResourceTiming extends PerformanceEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/connectEnd) */\n    readonly connectEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/connectStart) */\n    readonly connectStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/decodedBodySize) */\n    readonly decodedBodySize: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/domainLookupEnd) */\n    readonly domainLookupEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/domainLookupStart) */\n    readonly domainLookupStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/encodedBodySize) */\n    readonly encodedBodySize: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/fetchStart) */\n    readonly fetchStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/initiatorType) */\n    readonly initiatorType: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/nextHopProtocol) */\n    readonly nextHopProtocol: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/redirectEnd) */\n    readonly redirectEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/redirectStart) */\n    readonly redirectStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/requestStart) */\n    readonly requestStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/responseEnd) */\n    readonly responseEnd: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/responseStart) */\n    readonly responseStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/secureConnectionStart) */\n    readonly secureConnectionStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/serverTiming) */\n    readonly serverTiming: ReadonlyArray<PerformanceServerTiming>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/transferSize) */\n    readonly transferSize: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/workerStart) */\n    readonly workerStart: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceResourceTiming/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var PerformanceResourceTiming: {\n    prototype: PerformanceResourceTiming;\n    new(): PerformanceResourceTiming;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceServerTiming) */\ninterface PerformanceServerTiming {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceServerTiming/description) */\n    readonly description: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceServerTiming/duration) */\n    readonly duration: DOMHighResTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceServerTiming/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceServerTiming/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var PerformanceServerTiming: {\n    prototype: PerformanceServerTiming;\n    new(): PerformanceServerTiming;\n};\n\n/**\n * A legacy interface kept for backwards compatibility and contains properties that offer performance timing information for various events which occur during the loading and use of the current page. You get a PerformanceTiming object describing your page using the window.performance.timing property.\n * @deprecated This interface is deprecated in the Navigation Timing Level 2 specification. Please use the PerformanceNavigationTiming interface instead.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming)\n */\ninterface PerformanceTiming {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/connectEnd)\n     */\n    readonly connectEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/connectStart)\n     */\n    readonly connectStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domComplete)\n     */\n    readonly domComplete: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domContentLoadedEventEnd)\n     */\n    readonly domContentLoadedEventEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domContentLoadedEventStart)\n     */\n    readonly domContentLoadedEventStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domInteractive)\n     */\n    readonly domInteractive: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domLoading)\n     */\n    readonly domLoading: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domainLookupEnd)\n     */\n    readonly domainLookupEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/domainLookupStart)\n     */\n    readonly domainLookupStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/fetchStart)\n     */\n    readonly fetchStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/loadEventEnd)\n     */\n    readonly loadEventEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/loadEventStart)\n     */\n    readonly loadEventStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/navigationStart)\n     */\n    readonly navigationStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/redirectEnd)\n     */\n    readonly redirectEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/redirectStart)\n     */\n    readonly redirectStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/requestStart)\n     */\n    readonly requestStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/responseEnd)\n     */\n    readonly responseEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/responseStart)\n     */\n    readonly responseStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/secureConnectionStart)\n     */\n    readonly secureConnectionStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/unloadEventEnd)\n     */\n    readonly unloadEventEnd: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/unloadEventStart)\n     */\n    readonly unloadEventStart: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PerformanceTiming/toJSON)\n     */\n    toJSON(): any;\n}\n\n/** @deprecated */\ndeclare var PerformanceTiming: {\n    prototype: PerformanceTiming;\n    new(): PerformanceTiming;\n};\n\n/**\n * PeriodicWave has no inputs or outputs; it is used to define custom oscillators when calling OscillatorNode.setPeriodicWave(). The PeriodicWave itself is created/returned by AudioContext.createPeriodicWave().\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PeriodicWave)\n */\ninterface PeriodicWave {\n}\n\ndeclare var PeriodicWave: {\n    prototype: PeriodicWave;\n    new(context: BaseAudioContext, options?: PeriodicWaveOptions): PeriodicWave;\n};\n\ninterface PermissionStatusEventMap {\n    \"change\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PermissionStatus) */\ninterface PermissionStatus extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PermissionStatus/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PermissionStatus/change_event) */\n    onchange: ((this: PermissionStatus, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PermissionStatus/state) */\n    readonly state: PermissionState;\n    addEventListener<K extends keyof PermissionStatusEventMap>(type: K, listener: (this: PermissionStatus, ev: PermissionStatusEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof PermissionStatusEventMap>(type: K, listener: (this: PermissionStatus, ev: PermissionStatusEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var PermissionStatus: {\n    prototype: PermissionStatus;\n    new(): PermissionStatus;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Permissions) */\ninterface Permissions {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Permissions/query) */\n    query(permissionDesc: PermissionDescriptor): Promise<PermissionStatus>;\n}\n\ndeclare var Permissions: {\n    prototype: Permissions;\n    new(): Permissions;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PictureInPictureEvent) */\ninterface PictureInPictureEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PictureInPictureEvent/pictureInPictureWindow) */\n    readonly pictureInPictureWindow: PictureInPictureWindow;\n}\n\ndeclare var PictureInPictureEvent: {\n    prototype: PictureInPictureEvent;\n    new(type: string, eventInitDict: PictureInPictureEventInit): PictureInPictureEvent;\n};\n\ninterface PictureInPictureWindowEventMap {\n    \"resize\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PictureInPictureWindow) */\ninterface PictureInPictureWindow extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PictureInPictureWindow/height) */\n    readonly height: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PictureInPictureWindow/resize_event) */\n    onresize: ((this: PictureInPictureWindow, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PictureInPictureWindow/width) */\n    readonly width: number;\n    addEventListener<K extends keyof PictureInPictureWindowEventMap>(type: K, listener: (this: PictureInPictureWindow, ev: PictureInPictureWindowEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof PictureInPictureWindowEventMap>(type: K, listener: (this: PictureInPictureWindow, ev: PictureInPictureWindowEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var PictureInPictureWindow: {\n    prototype: PictureInPictureWindow;\n    new(): PictureInPictureWindow;\n};\n\n/**\n * Provides information about a browser plugin.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Plugin)\n */\ninterface Plugin {\n    /**\n     * Returns the plugin's description.\n     * @deprecated\n     */\n    readonly description: string;\n    /**\n     * Returns the plugin library's filename, if applicable on the current platform.\n     * @deprecated\n     */\n    readonly filename: string;\n    /**\n     * Returns the number of MIME types, represented by MimeType objects, supported by the plugin.\n     * @deprecated\n     */\n    readonly length: number;\n    /**\n     * Returns the plugin's name.\n     * @deprecated\n     */\n    readonly name: string;\n    /**\n     * Returns the specified MimeType object.\n     * @deprecated\n     */\n    item(index: number): MimeType | null;\n    /** @deprecated */\n    namedItem(name: string): MimeType | null;\n    [index: number]: MimeType;\n}\n\n/** @deprecated */\ndeclare var Plugin: {\n    prototype: Plugin;\n    new(): Plugin;\n};\n\n/**\n * Used to store a list of Plugin objects describing the available plugins; it's returned by the window.navigator.plugins property. The PluginArray is not a JavaScript array, but has the length property and supports accessing individual items using bracket notation (plugins[2]), as well as via item(index) and namedItem(\"name\") methods.\n * @deprecated\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PluginArray)\n */\ninterface PluginArray {\n    /** @deprecated */\n    readonly length: number;\n    /** @deprecated */\n    item(index: number): Plugin | null;\n    /** @deprecated */\n    namedItem(name: string): Plugin | null;\n    /** @deprecated */\n    refresh(): void;\n    [index: number]: Plugin;\n}\n\n/** @deprecated */\ndeclare var PluginArray: {\n    prototype: PluginArray;\n    new(): PluginArray;\n};\n\n/**\n * The state of a DOM event produced by a pointer such as the geometry of the contact point, the device type that generated the event, the amount of pressure that was applied on the contact surface, etc.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent)\n */\ninterface PointerEvent extends MouseEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/height) */\n    readonly height: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/isPrimary) */\n    readonly isPrimary: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/pointerId) */\n    readonly pointerId: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/pointerType) */\n    readonly pointerType: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/pressure) */\n    readonly pressure: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/tangentialPressure) */\n    readonly tangentialPressure: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/tiltX) */\n    readonly tiltX: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/tiltY) */\n    readonly tiltY: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/twist) */\n    readonly twist: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/width) */\n    readonly width: number;\n    /**\n     * Available only in secure contexts.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/getCoalescedEvents)\n     */\n    getCoalescedEvents(): PointerEvent[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PointerEvent/getPredictedEvents) */\n    getPredictedEvents(): PointerEvent[];\n}\n\ndeclare var PointerEvent: {\n    prototype: PointerEvent;\n    new(type: string, eventInitDict?: PointerEventInit): PointerEvent;\n};\n\n/**\n * PopStateEvent is an event handler for the popstate event on the window.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PopStateEvent)\n */\ninterface PopStateEvent extends Event {\n    readonly hasUAVisualTransition: boolean;\n    /**\n     * Returns a copy of the information that was provided to pushState() or replaceState().\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PopStateEvent/state)\n     */\n    readonly state: any;\n}\n\ndeclare var PopStateEvent: {\n    prototype: PopStateEvent;\n    new(type: string, eventInitDict?: PopStateEventInit): PopStateEvent;\n};\n\ninterface PopoverInvokerElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLButtonElement/popoverTargetAction) */\n    popoverTargetAction: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/HTMLButtonElement/popoverTargetElement) */\n    popoverTargetElement: Element | null;\n}\n\n/**\n * A processing instruction embeds application-specific instructions in XML which can be ignored by other applications that don't recognize them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ProcessingInstruction)\n */\ninterface ProcessingInstruction extends CharacterData, LinkStyle {\n    readonly ownerDocument: Document;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ProcessingInstruction/target) */\n    readonly target: string;\n}\n\ndeclare var ProcessingInstruction: {\n    prototype: ProcessingInstruction;\n    new(): ProcessingInstruction;\n};\n\n/**\n * Events measuring progress of an underlying process, like an HTTP request (for an XMLHttpRequest, or the loading of the underlying resource of an <img>, <audio>, <video>, <style> or <link>).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ProgressEvent)\n */\ninterface ProgressEvent<T extends EventTarget = EventTarget> extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ProgressEvent/lengthComputable) */\n    readonly lengthComputable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ProgressEvent/loaded) */\n    readonly loaded: number;\n    readonly target: T | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ProgressEvent/total) */\n    readonly total: number;\n}\n\ndeclare var ProgressEvent: {\n    prototype: ProgressEvent;\n    new(type: string, eventInitDict?: ProgressEventInit): ProgressEvent;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PromiseRejectionEvent) */\ninterface PromiseRejectionEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PromiseRejectionEvent/promise) */\n    readonly promise: Promise<any>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PromiseRejectionEvent/reason) */\n    readonly reason: any;\n}\n\ndeclare var PromiseRejectionEvent: {\n    prototype: PromiseRejectionEvent;\n    new(type: string, eventInitDict: PromiseRejectionEventInit): PromiseRejectionEvent;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential)\n */\ninterface PublicKeyCredential extends Credential {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential/authenticatorAttachment) */\n    readonly authenticatorAttachment: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential/rawId) */\n    readonly rawId: ArrayBuffer;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential/response) */\n    readonly response: AuthenticatorResponse;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential/getClientExtensionResults) */\n    getClientExtensionResults(): AuthenticationExtensionsClientOutputs;\n}\n\ndeclare var PublicKeyCredential: {\n    prototype: PublicKeyCredential;\n    new(): PublicKeyCredential;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential/isConditionalMediationAvailable) */\n    isConditionalMediationAvailable(): Promise<boolean>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PublicKeyCredential/isUserVerifyingPlatformAuthenticatorAvailable_static) */\n    isUserVerifyingPlatformAuthenticatorAvailable(): Promise<boolean>;\n};\n\n/**\n * This Push API interface provides a way to receive notifications from third-party servers as well as request URLs for push notifications.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushManager)\n */\ninterface PushManager {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushManager/getSubscription) */\n    getSubscription(): Promise<PushSubscription | null>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushManager/permissionState) */\n    permissionState(options?: PushSubscriptionOptionsInit): Promise<PermissionState>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushManager/subscribe) */\n    subscribe(options?: PushSubscriptionOptionsInit): Promise<PushSubscription>;\n}\n\ndeclare var PushManager: {\n    prototype: PushManager;\n    new(): PushManager;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushManager/supportedContentEncodings_static) */\n    readonly supportedContentEncodings: ReadonlyArray<string>;\n};\n\n/**\n * This Push API interface provides a subcription's URL endpoint and allows unsubscription from a push service.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription)\n */\ninterface PushSubscription {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription/endpoint) */\n    readonly endpoint: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription/expirationTime) */\n    readonly expirationTime: EpochTimeStamp | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription/options) */\n    readonly options: PushSubscriptionOptions;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription/getKey) */\n    getKey(name: PushEncryptionKeyName): ArrayBuffer | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription/toJSON) */\n    toJSON(): PushSubscriptionJSON;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscription/unsubscribe) */\n    unsubscribe(): Promise<boolean>;\n}\n\ndeclare var PushSubscription: {\n    prototype: PushSubscription;\n    new(): PushSubscription;\n};\n\n/**\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscriptionOptions)\n */\ninterface PushSubscriptionOptions {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscriptionOptions/applicationServerKey) */\n    readonly applicationServerKey: ArrayBuffer | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/PushSubscriptionOptions/userVisibleOnly) */\n    readonly userVisibleOnly: boolean;\n}\n\ndeclare var PushSubscriptionOptions: {\n    prototype: PushSubscriptionOptions;\n    new(): PushSubscriptionOptions;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCCertificate) */\ninterface RTCCertificate {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCCertificate/expires) */\n    readonly expires: EpochTimeStamp;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCCertificate/getFingerprints) */\n    getFingerprints(): RTCDtlsFingerprint[];\n}\n\ndeclare var RTCCertificate: {\n    prototype: RTCCertificate;\n    new(): RTCCertificate;\n};\n\ninterface RTCDTMFSenderEventMap {\n    \"tonechange\": RTCDTMFToneChangeEvent;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFSender) */\ninterface RTCDTMFSender extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFSender/canInsertDTMF) */\n    readonly canInsertDTMF: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFSender/tonechange_event) */\n    ontonechange: ((this: RTCDTMFSender, ev: RTCDTMFToneChangeEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFSender/toneBuffer) */\n    readonly toneBuffer: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFSender/insertDTMF) */\n    insertDTMF(tones: string, duration?: number, interToneGap?: number): void;\n    addEventListener<K extends keyof RTCDTMFSenderEventMap>(type: K, listener: (this: RTCDTMFSender, ev: RTCDTMFSenderEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RTCDTMFSenderEventMap>(type: K, listener: (this: RTCDTMFSender, ev: RTCDTMFSenderEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RTCDTMFSender: {\n    prototype: RTCDTMFSender;\n    new(): RTCDTMFSender;\n};\n\n/**\n * Events sent to indicate that DTMF tones have started or finished playing. This interface is used by the tonechange event.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFToneChangeEvent)\n */\ninterface RTCDTMFToneChangeEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDTMFToneChangeEvent/tone) */\n    readonly tone: string;\n}\n\ndeclare var RTCDTMFToneChangeEvent: {\n    prototype: RTCDTMFToneChangeEvent;\n    new(type: string, eventInitDict?: RTCDTMFToneChangeEventInit): RTCDTMFToneChangeEvent;\n};\n\ninterface RTCDataChannelEventMap {\n    \"bufferedamountlow\": Event;\n    \"close\": Event;\n    \"closing\": Event;\n    \"error\": Event;\n    \"message\": MessageEvent;\n    \"open\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel) */\ninterface RTCDataChannel extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/binaryType) */\n    binaryType: BinaryType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/bufferedAmount) */\n    readonly bufferedAmount: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/bufferedAmountLowThreshold) */\n    bufferedAmountLowThreshold: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/id) */\n    readonly id: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/label) */\n    readonly label: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/maxPacketLifeTime) */\n    readonly maxPacketLifeTime: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/maxRetransmits) */\n    readonly maxRetransmits: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/negotiated) */\n    readonly negotiated: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/bufferedamountlow_event) */\n    onbufferedamountlow: ((this: RTCDataChannel, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/close_event) */\n    onclose: ((this: RTCDataChannel, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/closing_event) */\n    onclosing: ((this: RTCDataChannel, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/error_event) */\n    onerror: ((this: RTCDataChannel, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/message_event) */\n    onmessage: ((this: RTCDataChannel, ev: MessageEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/open_event) */\n    onopen: ((this: RTCDataChannel, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/ordered) */\n    readonly ordered: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/protocol) */\n    readonly protocol: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/readyState) */\n    readonly readyState: RTCDataChannelState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/close) */\n    close(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannel/send) */\n    send(data: string): void;\n    send(data: Blob): void;\n    send(data: ArrayBuffer): void;\n    send(data: ArrayBufferView): void;\n    addEventListener<K extends keyof RTCDataChannelEventMap>(type: K, listener: (this: RTCDataChannel, ev: RTCDataChannelEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RTCDataChannelEventMap>(type: K, listener: (this: RTCDataChannel, ev: RTCDataChannelEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RTCDataChannel: {\n    prototype: RTCDataChannel;\n    new(): RTCDataChannel;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannelEvent) */\ninterface RTCDataChannelEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDataChannelEvent/channel) */\n    readonly channel: RTCDataChannel;\n}\n\ndeclare var RTCDataChannelEvent: {\n    prototype: RTCDataChannelEvent;\n    new(type: string, eventInitDict: RTCDataChannelEventInit): RTCDataChannelEvent;\n};\n\ninterface RTCDtlsTransportEventMap {\n    \"error\": Event;\n    \"statechange\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDtlsTransport) */\ninterface RTCDtlsTransport extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDtlsTransport/iceTransport) */\n    readonly iceTransport: RTCIceTransport;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDtlsTransport/error_event) */\n    onerror: ((this: RTCDtlsTransport, ev: Event) => any) | null;\n    onstatechange: ((this: RTCDtlsTransport, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCDtlsTransport/state) */\n    readonly state: RTCDtlsTransportState;\n    getRemoteCertificates(): ArrayBuffer[];\n    addEventListener<K extends keyof RTCDtlsTransportEventMap>(type: K, listener: (this: RTCDtlsTransport, ev: RTCDtlsTransportEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RTCDtlsTransportEventMap>(type: K, listener: (this: RTCDtlsTransport, ev: RTCDtlsTransportEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RTCDtlsTransport: {\n    prototype: RTCDtlsTransport;\n    new(): RTCDtlsTransport;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedAudioFrame) */\ninterface RTCEncodedAudioFrame {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedAudioFrame/data) */\n    data: ArrayBuffer;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedAudioFrame/timestamp) */\n    readonly timestamp: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedAudioFrame/getMetadata) */\n    getMetadata(): RTCEncodedAudioFrameMetadata;\n}\n\ndeclare var RTCEncodedAudioFrame: {\n    prototype: RTCEncodedAudioFrame;\n    new(): RTCEncodedAudioFrame;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedVideoFrame) */\ninterface RTCEncodedVideoFrame {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedVideoFrame/data) */\n    data: ArrayBuffer;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedVideoFrame/timestamp) */\n    readonly timestamp: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedVideoFrame/type) */\n    readonly type: RTCEncodedVideoFrameType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCEncodedVideoFrame/getMetadata) */\n    getMetadata(): RTCEncodedVideoFrameMetadata;\n}\n\ndeclare var RTCEncodedVideoFrame: {\n    prototype: RTCEncodedVideoFrame;\n    new(): RTCEncodedVideoFrame;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCError) */\ninterface RTCError extends DOMException {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCError/errorDetail) */\n    readonly errorDetail: RTCErrorDetailType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCError/receivedAlert) */\n    readonly receivedAlert: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCError/sctpCauseCode) */\n    readonly sctpCauseCode: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCError/sdpLineNumber) */\n    readonly sdpLineNumber: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCError/sentAlert) */\n    readonly sentAlert: number | null;\n}\n\ndeclare var RTCError: {\n    prototype: RTCError;\n    new(init: RTCErrorInit, message?: string): RTCError;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCErrorEvent) */\ninterface RTCErrorEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCErrorEvent/error) */\n    readonly error: RTCError;\n}\n\ndeclare var RTCErrorEvent: {\n    prototype: RTCErrorEvent;\n    new(type: string, eventInitDict: RTCErrorEventInit): RTCErrorEvent;\n};\n\n/**\n * The RTCIceCandidate interface—part of the WebRTC API—represents a candidate Internet Connectivity Establishment (ICE) configuration which may be used to establish an RTCPeerConnection.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate)\n */\ninterface RTCIceCandidate {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/address) */\n    readonly address: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/candidate) */\n    readonly candidate: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/component) */\n    readonly component: RTCIceComponent | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/foundation) */\n    readonly foundation: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/port) */\n    readonly port: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/priority) */\n    readonly priority: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/protocol) */\n    readonly protocol: RTCIceProtocol | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/relatedAddress) */\n    readonly relatedAddress: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/relatedPort) */\n    readonly relatedPort: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/sdpMLineIndex) */\n    readonly sdpMLineIndex: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/sdpMid) */\n    readonly sdpMid: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/tcpType) */\n    readonly tcpType: RTCIceTcpCandidateType | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/type) */\n    readonly type: RTCIceCandidateType | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/usernameFragment) */\n    readonly usernameFragment: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceCandidate/toJSON) */\n    toJSON(): RTCIceCandidateInit;\n}\n\ndeclare var RTCIceCandidate: {\n    prototype: RTCIceCandidate;\n    new(candidateInitDict?: RTCIceCandidateInit): RTCIceCandidate;\n};\n\ninterface RTCIceCandidatePair {\n    local: RTCIceCandidate;\n    remote: RTCIceCandidate;\n}\n\ninterface RTCIceTransportEventMap {\n    \"gatheringstatechange\": Event;\n    \"selectedcandidatepairchange\": Event;\n    \"statechange\": Event;\n}\n\n/**\n * Provides access to information about the ICE transport layer over which the data is being sent and received.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport)\n */\ninterface RTCIceTransport extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport/gatheringState) */\n    readonly gatheringState: RTCIceGathererState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport/gatheringstatechange_event) */\n    ongatheringstatechange: ((this: RTCIceTransport, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport/selectedcandidatepairchange_event) */\n    onselectedcandidatepairchange: ((this: RTCIceTransport, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport/statechange_event) */\n    onstatechange: ((this: RTCIceTransport, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport/state) */\n    readonly state: RTCIceTransportState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCIceTransport/getSelectedCandidatePair) */\n    getSelectedCandidatePair(): RTCIceCandidatePair | null;\n    addEventListener<K extends keyof RTCIceTransportEventMap>(type: K, listener: (this: RTCIceTransport, ev: RTCIceTransportEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RTCIceTransportEventMap>(type: K, listener: (this: RTCIceTransport, ev: RTCIceTransportEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RTCIceTransport: {\n    prototype: RTCIceTransport;\n    new(): RTCIceTransport;\n};\n\ninterface RTCPeerConnectionEventMap {\n    \"connectionstatechange\": Event;\n    \"datachannel\": RTCDataChannelEvent;\n    \"icecandidate\": RTCPeerConnectionIceEvent;\n    \"icecandidateerror\": RTCPeerConnectionIceErrorEvent;\n    \"iceconnectionstatechange\": Event;\n    \"icegatheringstatechange\": Event;\n    \"negotiationneeded\": Event;\n    \"signalingstatechange\": Event;\n    \"track\": RTCTrackEvent;\n}\n\n/**\n * A WebRTC connection between the local computer and a remote peer. It provides methods to connect to a remote peer, maintain and monitor the connection, and close the connection once it's no longer needed.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection)\n */\ninterface RTCPeerConnection extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/canTrickleIceCandidates) */\n    readonly canTrickleIceCandidates: boolean | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/connectionState) */\n    readonly connectionState: RTCPeerConnectionState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/currentLocalDescription) */\n    readonly currentLocalDescription: RTCSessionDescription | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/currentRemoteDescription) */\n    readonly currentRemoteDescription: RTCSessionDescription | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/iceConnectionState) */\n    readonly iceConnectionState: RTCIceConnectionState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/iceGatheringState) */\n    readonly iceGatheringState: RTCIceGatheringState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/localDescription) */\n    readonly localDescription: RTCSessionDescription | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/connectionstatechange_event) */\n    onconnectionstatechange: ((this: RTCPeerConnection, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/datachannel_event) */\n    ondatachannel: ((this: RTCPeerConnection, ev: RTCDataChannelEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/icecandidate_event) */\n    onicecandidate: ((this: RTCPeerConnection, ev: RTCPeerConnectionIceEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/icecandidateerror_event) */\n    onicecandidateerror: ((this: RTCPeerConnection, ev: RTCPeerConnectionIceErrorEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/iceconnectionstatechange_event) */\n    oniceconnectionstatechange: ((this: RTCPeerConnection, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/icegatheringstatechange_event) */\n    onicegatheringstatechange: ((this: RTCPeerConnection, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/negotiationneeded_event) */\n    onnegotiationneeded: ((this: RTCPeerConnection, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/signalingstatechange_event) */\n    onsignalingstatechange: ((this: RTCPeerConnection, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/track_event) */\n    ontrack: ((this: RTCPeerConnection, ev: RTCTrackEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/pendingLocalDescription) */\n    readonly pendingLocalDescription: RTCSessionDescription | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/pendingRemoteDescription) */\n    readonly pendingRemoteDescription: RTCSessionDescription | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/remoteDescription) */\n    readonly remoteDescription: RTCSessionDescription | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/sctp) */\n    readonly sctp: RTCSctpTransport | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/signalingState) */\n    readonly signalingState: RTCSignalingState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/addIceCandidate) */\n    addIceCandidate(candidate?: RTCIceCandidateInit): Promise<void>;\n    /** @deprecated */\n    addIceCandidate(candidate: RTCIceCandidateInit, successCallback: VoidFunction, failureCallback: RTCPeerConnectionErrorCallback): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/addTrack) */\n    addTrack(track: MediaStreamTrack, ...streams: MediaStream[]): RTCRtpSender;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/addTransceiver) */\n    addTransceiver(trackOrKind: MediaStreamTrack | string, init?: RTCRtpTransceiverInit): RTCRtpTransceiver;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/close) */\n    close(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/createAnswer) */\n    createAnswer(options?: RTCAnswerOptions): Promise<RTCSessionDescriptionInit>;\n    /** @deprecated */\n    createAnswer(successCallback: RTCSessionDescriptionCallback, failureCallback: RTCPeerConnectionErrorCallback): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/createDataChannel) */\n    createDataChannel(label: string, dataChannelDict?: RTCDataChannelInit): RTCDataChannel;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/createOffer) */\n    createOffer(options?: RTCOfferOptions): Promise<RTCSessionDescriptionInit>;\n    /** @deprecated */\n    createOffer(successCallback: RTCSessionDescriptionCallback, failureCallback: RTCPeerConnectionErrorCallback, options?: RTCOfferOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/getConfiguration) */\n    getConfiguration(): RTCConfiguration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/getReceivers) */\n    getReceivers(): RTCRtpReceiver[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/getSenders) */\n    getSenders(): RTCRtpSender[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/getStats) */\n    getStats(selector?: MediaStreamTrack | null): Promise<RTCStatsReport>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/getTransceivers) */\n    getTransceivers(): RTCRtpTransceiver[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/removeTrack) */\n    removeTrack(sender: RTCRtpSender): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/restartIce) */\n    restartIce(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/setConfiguration) */\n    setConfiguration(configuration?: RTCConfiguration): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/setLocalDescription) */\n    setLocalDescription(description?: RTCLocalSessionDescriptionInit): Promise<void>;\n    /** @deprecated */\n    setLocalDescription(description: RTCLocalSessionDescriptionInit, successCallback: VoidFunction, failureCallback: RTCPeerConnectionErrorCallback): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/setRemoteDescription) */\n    setRemoteDescription(description: RTCSessionDescriptionInit): Promise<void>;\n    /** @deprecated */\n    setRemoteDescription(description: RTCSessionDescriptionInit, successCallback: VoidFunction, failureCallback: RTCPeerConnectionErrorCallback): Promise<void>;\n    addEventListener<K extends keyof RTCPeerConnectionEventMap>(type: K, listener: (this: RTCPeerConnection, ev: RTCPeerConnectionEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RTCPeerConnectionEventMap>(type: K, listener: (this: RTCPeerConnection, ev: RTCPeerConnectionEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RTCPeerConnection: {\n    prototype: RTCPeerConnection;\n    new(configuration?: RTCConfiguration): RTCPeerConnection;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnection/generateCertificate_static) */\n    generateCertificate(keygenAlgorithm: AlgorithmIdentifier): Promise<RTCCertificate>;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnectionIceErrorEvent) */\ninterface RTCPeerConnectionIceErrorEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnectionIceErrorEvent/address) */\n    readonly address: string | null;\n    readonly errorCode: number;\n    readonly errorText: string;\n    readonly port: number | null;\n    readonly url: string;\n}\n\ndeclare var RTCPeerConnectionIceErrorEvent: {\n    prototype: RTCPeerConnectionIceErrorEvent;\n    new(type: string, eventInitDict: RTCPeerConnectionIceErrorEventInit): RTCPeerConnectionIceErrorEvent;\n};\n\n/**\n * Events that occurs in relation to ICE candidates with the target, usually an RTCPeerConnection. Only one event is of this type: icecandidate.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnectionIceEvent)\n */\ninterface RTCPeerConnectionIceEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCPeerConnectionIceEvent/candidate) */\n    readonly candidate: RTCIceCandidate | null;\n}\n\ndeclare var RTCPeerConnectionIceEvent: {\n    prototype: RTCPeerConnectionIceEvent;\n    new(type: string, eventInitDict?: RTCPeerConnectionIceEventInit): RTCPeerConnectionIceEvent;\n};\n\n/**\n * This WebRTC API interface manages the reception and decoding of data for a MediaStreamTrack on an RTCPeerConnection.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver)\n */\ninterface RTCRtpReceiver {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/jitterBufferTarget) */\n    jitterBufferTarget: DOMHighResTimeStamp | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/track) */\n    readonly track: MediaStreamTrack;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/transform) */\n    transform: RTCRtpTransform | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/transport) */\n    readonly transport: RTCDtlsTransport | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/getContributingSources) */\n    getContributingSources(): RTCRtpContributingSource[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/getParameters) */\n    getParameters(): RTCRtpReceiveParameters;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/getStats) */\n    getStats(): Promise<RTCStatsReport>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/getSynchronizationSources) */\n    getSynchronizationSources(): RTCRtpSynchronizationSource[];\n}\n\ndeclare var RTCRtpReceiver: {\n    prototype: RTCRtpReceiver;\n    new(): RTCRtpReceiver;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpReceiver/getCapabilities_static) */\n    getCapabilities(kind: string): RTCRtpCapabilities | null;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpScriptTransform) */\ninterface RTCRtpScriptTransform {\n}\n\ndeclare var RTCRtpScriptTransform: {\n    prototype: RTCRtpScriptTransform;\n    new(worker: Worker, options?: any, transfer?: any[]): RTCRtpScriptTransform;\n};\n\n/**\n * Provides the ability to control and obtain details about how a particular MediaStreamTrack is encoded and sent to a remote peer.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender)\n */\ninterface RTCRtpSender {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/dtmf) */\n    readonly dtmf: RTCDTMFSender | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/track) */\n    readonly track: MediaStreamTrack | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/transform) */\n    transform: RTCRtpTransform | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/transport) */\n    readonly transport: RTCDtlsTransport | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/getParameters) */\n    getParameters(): RTCRtpSendParameters;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/getStats) */\n    getStats(): Promise<RTCStatsReport>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/replaceTrack) */\n    replaceTrack(withTrack: MediaStreamTrack | null): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/setParameters) */\n    setParameters(parameters: RTCRtpSendParameters, setParameterOptions?: RTCSetParameterOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/setStreams) */\n    setStreams(...streams: MediaStream[]): void;\n}\n\ndeclare var RTCRtpSender: {\n    prototype: RTCRtpSender;\n    new(): RTCRtpSender;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpSender/getCapabilities_static) */\n    getCapabilities(kind: string): RTCRtpCapabilities | null;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver) */\ninterface RTCRtpTransceiver {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/currentDirection) */\n    readonly currentDirection: RTCRtpTransceiverDirection | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/direction) */\n    direction: RTCRtpTransceiverDirection;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/mid) */\n    readonly mid: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/receiver) */\n    readonly receiver: RTCRtpReceiver;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/sender) */\n    readonly sender: RTCRtpSender;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/setCodecPreferences) */\n    setCodecPreferences(codecs: RTCRtpCodec[]): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCRtpTransceiver/stop) */\n    stop(): void;\n}\n\ndeclare var RTCRtpTransceiver: {\n    prototype: RTCRtpTransceiver;\n    new(): RTCRtpTransceiver;\n};\n\ninterface RTCSctpTransportEventMap {\n    \"statechange\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSctpTransport) */\ninterface RTCSctpTransport extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSctpTransport/maxChannels) */\n    readonly maxChannels: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSctpTransport/maxMessageSize) */\n    readonly maxMessageSize: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSctpTransport/statechange_event) */\n    onstatechange: ((this: RTCSctpTransport, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSctpTransport/state) */\n    readonly state: RTCSctpTransportState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSctpTransport/transport) */\n    readonly transport: RTCDtlsTransport;\n    addEventListener<K extends keyof RTCSctpTransportEventMap>(type: K, listener: (this: RTCSctpTransport, ev: RTCSctpTransportEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RTCSctpTransportEventMap>(type: K, listener: (this: RTCSctpTransport, ev: RTCSctpTransportEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RTCSctpTransport: {\n    prototype: RTCSctpTransport;\n    new(): RTCSctpTransport;\n};\n\n/**\n * One end of a connection—or potential connection—and how it's configured. Each RTCSessionDescription consists of a description type indicating which part of the offer/answer negotiation process it describes and of the SDP descriptor of the session.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSessionDescription)\n */\ninterface RTCSessionDescription {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSessionDescription/sdp) */\n    readonly sdp: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSessionDescription/type) */\n    readonly type: RTCSdpType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCSessionDescription/toJSON) */\n    toJSON(): RTCSessionDescriptionInit;\n}\n\ndeclare var RTCSessionDescription: {\n    prototype: RTCSessionDescription;\n    new(descriptionInitDict: RTCSessionDescriptionInit): RTCSessionDescription;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCStatsReport) */\ninterface RTCStatsReport {\n    forEach(callbackfn: (value: any, key: string, parent: RTCStatsReport) => void, thisArg?: any): void;\n}\n\ndeclare var RTCStatsReport: {\n    prototype: RTCStatsReport;\n    new(): RTCStatsReport;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCTrackEvent) */\ninterface RTCTrackEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCTrackEvent/receiver) */\n    readonly receiver: RTCRtpReceiver;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCTrackEvent/streams) */\n    readonly streams: ReadonlyArray<MediaStream>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCTrackEvent/track) */\n    readonly track: MediaStreamTrack;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RTCTrackEvent/transceiver) */\n    readonly transceiver: RTCRtpTransceiver;\n}\n\ndeclare var RTCTrackEvent: {\n    prototype: RTCTrackEvent;\n    new(type: string, eventInitDict: RTCTrackEventInit): RTCTrackEvent;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RadioNodeList) */\ninterface RadioNodeList extends NodeList {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RadioNodeList/value) */\n    value: string;\n}\n\ndeclare var RadioNodeList: {\n    prototype: RadioNodeList;\n    new(): RadioNodeList;\n};\n\n/**\n * A fragment of a document that can contain nodes and parts of text nodes.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range)\n */\ninterface Range extends AbstractRange {\n    /**\n     * Returns the node, furthest away from the document, that is an ancestor of both range's start node and end node.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/commonAncestorContainer)\n     */\n    readonly commonAncestorContainer: Node;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/cloneContents) */\n    cloneContents(): DocumentFragment;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/cloneRange) */\n    cloneRange(): Range;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/collapse) */\n    collapse(toStart?: boolean): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/compareBoundaryPoints) */\n    compareBoundaryPoints(how: number, sourceRange: Range): number;\n    /**\n     * Returns −1 if the point is before the range, 0 if the point is in the range, and 1 if the point is after the range.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/comparePoint)\n     */\n    comparePoint(node: Node, offset: number): number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/createContextualFragment) */\n    createContextualFragment(string: string): DocumentFragment;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/deleteContents) */\n    deleteContents(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/detach) */\n    detach(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/extractContents) */\n    extractContents(): DocumentFragment;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/getBoundingClientRect) */\n    getBoundingClientRect(): DOMRect;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/getClientRects) */\n    getClientRects(): DOMRectList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/insertNode) */\n    insertNode(node: Node): void;\n    /**\n     * Returns whether range intersects node.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/intersectsNode)\n     */\n    intersectsNode(node: Node): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/isPointInRange) */\n    isPointInRange(node: Node, offset: number): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/selectNode) */\n    selectNode(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/selectNodeContents) */\n    selectNodeContents(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/setEnd) */\n    setEnd(node: Node, offset: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/setEndAfter) */\n    setEndAfter(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/setEndBefore) */\n    setEndBefore(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/setStart) */\n    setStart(node: Node, offset: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/setStartAfter) */\n    setStartAfter(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/setStartBefore) */\n    setStartBefore(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Range/surroundContents) */\n    surroundContents(newParent: Node): void;\n    toString(): string;\n    readonly START_TO_START: 0;\n    readonly START_TO_END: 1;\n    readonly END_TO_END: 2;\n    readonly END_TO_START: 3;\n}\n\ndeclare var Range: {\n    prototype: Range;\n    new(): Range;\n    readonly START_TO_START: 0;\n    readonly START_TO_END: 1;\n    readonly END_TO_END: 2;\n    readonly END_TO_START: 3;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableByteStreamController) */\ninterface ReadableByteStreamController {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableByteStreamController/byobRequest) */\n    readonly byobRequest: ReadableStreamBYOBRequest | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableByteStreamController/desiredSize) */\n    readonly desiredSize: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableByteStreamController/close) */\n    close(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableByteStreamController/enqueue) */\n    enqueue(chunk: ArrayBufferView): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableByteStreamController/error) */\n    error(e?: any): void;\n}\n\ndeclare var ReadableByteStreamController: {\n    prototype: ReadableByteStreamController;\n    new(): ReadableByteStreamController;\n};\n\n/**\n * This Streams API interface represents a readable stream of byte data. The Fetch API offers a concrete instance of a ReadableStream through the body property of a Response object.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream)\n */\ninterface ReadableStream<R = any> {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream/locked) */\n    readonly locked: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream/cancel) */\n    cancel(reason?: any): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream/getReader) */\n    getReader(options: { mode: \"byob\" }): ReadableStreamBYOBReader;\n    getReader(): ReadableStreamDefaultReader<R>;\n    getReader(options?: ReadableStreamGetReaderOptions): ReadableStreamReader<R>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream/pipeThrough) */\n    pipeThrough<T>(transform: ReadableWritablePair<T, R>, options?: StreamPipeOptions): ReadableStream<T>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream/pipeTo) */\n    pipeTo(destination: WritableStream<R>, options?: StreamPipeOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStream/tee) */\n    tee(): [ReadableStream<R>, ReadableStream<R>];\n}\n\ndeclare var ReadableStream: {\n    prototype: ReadableStream;\n    new(underlyingSource: UnderlyingByteSource, strategy?: { highWaterMark?: number }): ReadableStream<Uint8Array>;\n    new<R = any>(underlyingSource: UnderlyingDefaultSource<R>, strategy?: QueuingStrategy<R>): ReadableStream<R>;\n    new<R = any>(underlyingSource?: UnderlyingSource<R>, strategy?: QueuingStrategy<R>): ReadableStream<R>;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBReader) */\ninterface ReadableStreamBYOBReader extends ReadableStreamGenericReader {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBReader/read) */\n    read<T extends ArrayBufferView>(view: T): Promise<ReadableStreamReadResult<T>>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBReader/releaseLock) */\n    releaseLock(): void;\n}\n\ndeclare var ReadableStreamBYOBReader: {\n    prototype: ReadableStreamBYOBReader;\n    new(stream: ReadableStream): ReadableStreamBYOBReader;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBRequest) */\ninterface ReadableStreamBYOBRequest {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBRequest/view) */\n    readonly view: ArrayBufferView | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBRequest/respond) */\n    respond(bytesWritten: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBRequest/respondWithNewView) */\n    respondWithNewView(view: ArrayBufferView): void;\n}\n\ndeclare var ReadableStreamBYOBRequest: {\n    prototype: ReadableStreamBYOBRequest;\n    new(): ReadableStreamBYOBRequest;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultController) */\ninterface ReadableStreamDefaultController<R = any> {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultController/desiredSize) */\n    readonly desiredSize: number | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultController/close) */\n    close(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultController/enqueue) */\n    enqueue(chunk?: R): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultController/error) */\n    error(e?: any): void;\n}\n\ndeclare var ReadableStreamDefaultController: {\n    prototype: ReadableStreamDefaultController;\n    new(): ReadableStreamDefaultController;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultReader) */\ninterface ReadableStreamDefaultReader<R = any> extends ReadableStreamGenericReader {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultReader/read) */\n    read(): Promise<ReadableStreamReadResult<R>>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamDefaultReader/releaseLock) */\n    releaseLock(): void;\n}\n\ndeclare var ReadableStreamDefaultReader: {\n    prototype: ReadableStreamDefaultReader;\n    new<R = any>(stream: ReadableStream<R>): ReadableStreamDefaultReader<R>;\n};\n\ninterface ReadableStreamGenericReader {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBReader/closed) */\n    readonly closed: Promise<undefined>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReadableStreamBYOBReader/cancel) */\n    cancel(reason?: any): Promise<void>;\n}\n\ninterface RemotePlaybackEventMap {\n    \"connect\": Event;\n    \"connecting\": Event;\n    \"disconnect\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback) */\ninterface RemotePlayback extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/connect_event) */\n    onconnect: ((this: RemotePlayback, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/connecting_event) */\n    onconnecting: ((this: RemotePlayback, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/disconnect_event) */\n    ondisconnect: ((this: RemotePlayback, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/state) */\n    readonly state: RemotePlaybackState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/cancelWatchAvailability) */\n    cancelWatchAvailability(id?: number): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/prompt) */\n    prompt(): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/RemotePlayback/watchAvailability) */\n    watchAvailability(callback: RemotePlaybackAvailabilityCallback): Promise<number>;\n    addEventListener<K extends keyof RemotePlaybackEventMap>(type: K, listener: (this: RemotePlayback, ev: RemotePlaybackEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof RemotePlaybackEventMap>(type: K, listener: (this: RemotePlayback, ev: RemotePlaybackEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var RemotePlayback: {\n    prototype: RemotePlayback;\n    new(): RemotePlayback;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Report) */\ninterface Report {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Report/body) */\n    readonly body: ReportBody | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Report/type) */\n    readonly type: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Report/url) */\n    readonly url: string;\n    toJSON(): any;\n}\n\ndeclare var Report: {\n    prototype: Report;\n    new(): Report;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReportBody) */\ninterface ReportBody {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReportBody/toJSON) */\n    toJSON(): any;\n}\n\ndeclare var ReportBody: {\n    prototype: ReportBody;\n    new(): ReportBody;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReportingObserver) */\ninterface ReportingObserver {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReportingObserver/disconnect) */\n    disconnect(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReportingObserver/observe) */\n    observe(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ReportingObserver/takeRecords) */\n    takeRecords(): ReportList;\n}\n\ndeclare var ReportingObserver: {\n    prototype: ReportingObserver;\n    new(callback: ReportingObserverCallback, options?: ReportingObserverOptions): ReportingObserver;\n};\n\n/**\n * This Fetch API interface represents a resource request.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request)\n */\ninterface Request extends Body {\n    /**\n     * Returns the cache mode associated with request, which is a string indicating how the request will interact with the browser's cache when fetching.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/cache)\n     */\n    readonly cache: RequestCache;\n    /**\n     * Returns the credentials mode associated with request, which is a string indicating whether credentials will be sent with the request always, never, or only when sent to a same-origin URL.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/credentials)\n     */\n    readonly credentials: RequestCredentials;\n    /**\n     * Returns the kind of resource requested by request, e.g., \"document\" or \"script\".\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/destination)\n     */\n    readonly destination: RequestDestination;\n    /**\n     * Returns a Headers object consisting of the headers associated with request. Note that headers added in the network layer by the user agent will not be accounted for in this object, e.g., the \"Host\" header.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/headers)\n     */\n    readonly headers: Headers;\n    /**\n     * Returns request's subresource integrity metadata, which is a cryptographic hash of the resource being fetched. Its value consists of multiple hashes separated by whitespace. [SRI]\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/integrity)\n     */\n    readonly integrity: string;\n    /** Returns a boolean indicating whether or not request can outlive the global in which it was created. */\n    readonly keepalive: boolean;\n    /**\n     * Returns request's HTTP method, which is \"GET\" by default.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/method)\n     */\n    readonly method: string;\n    /**\n     * Returns the mode associated with request, which is a string indicating whether the request will use CORS, or will be restricted to same-origin URLs.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/mode)\n     */\n    readonly mode: RequestMode;\n    /**\n     * Returns the redirect mode associated with request, which is a string indicating how redirects for the request will be handled during fetching. A request will follow redirects by default.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/redirect)\n     */\n    readonly redirect: RequestRedirect;\n    /**\n     * Returns the referrer of request. Its value can be a same-origin URL if explicitly set in init, the empty string to indicate no referrer, and \"about:client\" when defaulting to the global's default. This is used during fetching to determine the value of the `Referer` header of the request being made.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/referrer)\n     */\n    readonly referrer: string;\n    /**\n     * Returns the referrer policy associated with request. This is used during fetching to compute the value of the request's referrer.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/referrerPolicy)\n     */\n    readonly referrerPolicy: ReferrerPolicy;\n    /**\n     * Returns the signal associated with request, which is an AbortSignal object indicating whether or not request has been aborted, and its abort event handler.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/signal)\n     */\n    readonly signal: AbortSignal;\n    /**\n     * Returns the URL of request as a string.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/url)\n     */\n    readonly url: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/clone) */\n    clone(): Request;\n}\n\ndeclare var Request: {\n    prototype: Request;\n    new(input: RequestInfo | URL, init?: RequestInit): Request;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserver) */\ninterface ResizeObserver {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserver/disconnect) */\n    disconnect(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserver/observe) */\n    observe(target: Element, options?: ResizeObserverOptions): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserver/unobserve) */\n    unobserve(target: Element): void;\n}\n\ndeclare var ResizeObserver: {\n    prototype: ResizeObserver;\n    new(callback: ResizeObserverCallback): ResizeObserver;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverEntry) */\ninterface ResizeObserverEntry {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverEntry/borderBoxSize) */\n    readonly borderBoxSize: ReadonlyArray<ResizeObserverSize>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverEntry/contentBoxSize) */\n    readonly contentBoxSize: ReadonlyArray<ResizeObserverSize>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverEntry/contentRect) */\n    readonly contentRect: DOMRectReadOnly;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverEntry/devicePixelContentBoxSize) */\n    readonly devicePixelContentBoxSize: ReadonlyArray<ResizeObserverSize>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverEntry/target) */\n    readonly target: Element;\n}\n\ndeclare var ResizeObserverEntry: {\n    prototype: ResizeObserverEntry;\n    new(): ResizeObserverEntry;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverSize) */\ninterface ResizeObserverSize {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverSize/blockSize) */\n    readonly blockSize: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ResizeObserverSize/inlineSize) */\n    readonly inlineSize: number;\n}\n\ndeclare var ResizeObserverSize: {\n    prototype: ResizeObserverSize;\n    new(): ResizeObserverSize;\n};\n\n/**\n * This Fetch API interface represents the response to a request.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response)\n */\ninterface Response extends Body {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/headers) */\n    readonly headers: Headers;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/ok) */\n    readonly ok: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/redirected) */\n    readonly redirected: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/status) */\n    readonly status: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/statusText) */\n    readonly statusText: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/type) */\n    readonly type: ResponseType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/url) */\n    readonly url: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/clone) */\n    clone(): Response;\n}\n\ndeclare var Response: {\n    prototype: Response;\n    new(body?: BodyInit | null, init?: ResponseInit): Response;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/error_static) */\n    error(): Response;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/json_static) */\n    json(data: any, init?: ResponseInit): Response;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/redirect_static) */\n    redirect(url: string | URL, status?: number): Response;\n};\n\n/**\n * Provides access to the properties of <a> element, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAElement)\n */\ninterface SVGAElement extends SVGGraphicsElement, SVGURIReference {\n    rel: string;\n    readonly relList: DOMTokenList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAElement/target) */\n    readonly target: SVGAnimatedString;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGAElement: {\n    prototype: SVGAElement;\n    new(): SVGAElement;\n};\n\n/**\n * Used to represent a value that can be an <angle> or <number> value. An SVGAngle reflected through the animVal attribute is always read only.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAngle)\n */\ninterface SVGAngle {\n    readonly unitType: number;\n    value: number;\n    valueAsString: string;\n    valueInSpecifiedUnits: number;\n    convertToSpecifiedUnits(unitType: number): void;\n    newValueSpecifiedUnits(unitType: number, valueInSpecifiedUnits: number): void;\n    readonly SVG_ANGLETYPE_UNKNOWN: 0;\n    readonly SVG_ANGLETYPE_UNSPECIFIED: 1;\n    readonly SVG_ANGLETYPE_DEG: 2;\n    readonly SVG_ANGLETYPE_RAD: 3;\n    readonly SVG_ANGLETYPE_GRAD: 4;\n}\n\ndeclare var SVGAngle: {\n    prototype: SVGAngle;\n    new(): SVGAngle;\n    readonly SVG_ANGLETYPE_UNKNOWN: 0;\n    readonly SVG_ANGLETYPE_UNSPECIFIED: 1;\n    readonly SVG_ANGLETYPE_DEG: 2;\n    readonly SVG_ANGLETYPE_RAD: 3;\n    readonly SVG_ANGLETYPE_GRAD: 4;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimateElement) */\ninterface SVGAnimateElement extends SVGAnimationElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimateElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimateElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGAnimateElement: {\n    prototype: SVGAnimateElement;\n    new(): SVGAnimateElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimateMotionElement) */\ninterface SVGAnimateMotionElement extends SVGAnimationElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimateMotionElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimateMotionElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGAnimateMotionElement: {\n    prototype: SVGAnimateMotionElement;\n    new(): SVGAnimateMotionElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimateTransformElement) */\ninterface SVGAnimateTransformElement extends SVGAnimationElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimateTransformElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimateTransformElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGAnimateTransformElement: {\n    prototype: SVGAnimateTransformElement;\n    new(): SVGAnimateTransformElement;\n};\n\n/**\n * Used for attributes of basic type <angle> which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedAngle)\n */\ninterface SVGAnimatedAngle {\n    readonly animVal: SVGAngle;\n    readonly baseVal: SVGAngle;\n}\n\ndeclare var SVGAnimatedAngle: {\n    prototype: SVGAnimatedAngle;\n    new(): SVGAnimatedAngle;\n};\n\n/**\n * Used for attributes of type boolean which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedBoolean)\n */\ninterface SVGAnimatedBoolean {\n    readonly animVal: boolean;\n    baseVal: boolean;\n}\n\ndeclare var SVGAnimatedBoolean: {\n    prototype: SVGAnimatedBoolean;\n    new(): SVGAnimatedBoolean;\n};\n\n/**\n * Used for attributes whose value must be a constant from a particular enumeration and which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedEnumeration)\n */\ninterface SVGAnimatedEnumeration {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedEnumeration/animVal) */\n    readonly animVal: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedEnumeration/baseVal) */\n    baseVal: number;\n}\n\ndeclare var SVGAnimatedEnumeration: {\n    prototype: SVGAnimatedEnumeration;\n    new(): SVGAnimatedEnumeration;\n};\n\n/**\n * Used for attributes of basic type <integer> which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedInteger)\n */\ninterface SVGAnimatedInteger {\n    readonly animVal: number;\n    baseVal: number;\n}\n\ndeclare var SVGAnimatedInteger: {\n    prototype: SVGAnimatedInteger;\n    new(): SVGAnimatedInteger;\n};\n\n/**\n * Used for attributes of basic type <length> which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedLength)\n */\ninterface SVGAnimatedLength {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedLength/animVal) */\n    readonly animVal: SVGLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedLength/baseVal) */\n    readonly baseVal: SVGLength;\n}\n\ndeclare var SVGAnimatedLength: {\n    prototype: SVGAnimatedLength;\n    new(): SVGAnimatedLength;\n};\n\n/**\n * Used for attributes of type SVGLengthList which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedLengthList)\n */\ninterface SVGAnimatedLengthList {\n    readonly animVal: SVGLengthList;\n    readonly baseVal: SVGLengthList;\n}\n\ndeclare var SVGAnimatedLengthList: {\n    prototype: SVGAnimatedLengthList;\n    new(): SVGAnimatedLengthList;\n};\n\n/**\n * Used for attributes of basic type <Number> which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedNumber)\n */\ninterface SVGAnimatedNumber {\n    readonly animVal: number;\n    baseVal: number;\n}\n\ndeclare var SVGAnimatedNumber: {\n    prototype: SVGAnimatedNumber;\n    new(): SVGAnimatedNumber;\n};\n\n/**\n * The SVGAnimatedNumber interface is used for attributes which take a list of numbers and which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedNumberList)\n */\ninterface SVGAnimatedNumberList {\n    readonly animVal: SVGNumberList;\n    readonly baseVal: SVGNumberList;\n}\n\ndeclare var SVGAnimatedNumberList: {\n    prototype: SVGAnimatedNumberList;\n    new(): SVGAnimatedNumberList;\n};\n\ninterface SVGAnimatedPoints {\n    readonly animatedPoints: SVGPointList;\n    readonly points: SVGPointList;\n}\n\n/**\n * Used for attributes of type SVGPreserveAspectRatio which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedPreserveAspectRatio)\n */\ninterface SVGAnimatedPreserveAspectRatio {\n    readonly animVal: SVGPreserveAspectRatio;\n    readonly baseVal: SVGPreserveAspectRatio;\n}\n\ndeclare var SVGAnimatedPreserveAspectRatio: {\n    prototype: SVGAnimatedPreserveAspectRatio;\n    new(): SVGAnimatedPreserveAspectRatio;\n};\n\n/**\n * Used for attributes of basic SVGRect which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedRect)\n */\ninterface SVGAnimatedRect {\n    readonly animVal: DOMRectReadOnly;\n    readonly baseVal: DOMRect;\n}\n\ndeclare var SVGAnimatedRect: {\n    prototype: SVGAnimatedRect;\n    new(): SVGAnimatedRect;\n};\n\n/**\n * The SVGAnimatedString interface represents string attributes which can be animated from each SVG declaration. You need to create SVG attribute before doing anything else, everything should be declared inside this.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedString)\n */\ninterface SVGAnimatedString {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedString/animVal) */\n    readonly animVal: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedString/baseVal) */\n    baseVal: string;\n}\n\ndeclare var SVGAnimatedString: {\n    prototype: SVGAnimatedString;\n    new(): SVGAnimatedString;\n};\n\n/**\n * Used for attributes which take a list of numbers and which can be animated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimatedTransformList)\n */\ninterface SVGAnimatedTransformList {\n    readonly animVal: SVGTransformList;\n    readonly baseVal: SVGTransformList;\n}\n\ndeclare var SVGAnimatedTransformList: {\n    prototype: SVGAnimatedTransformList;\n    new(): SVGAnimatedTransformList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimationElement) */\ninterface SVGAnimationElement extends SVGElement, SVGTests {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGAnimationElement/targetElement) */\n    readonly targetElement: SVGElement | null;\n    beginElement(): void;\n    beginElementAt(offset: number): void;\n    endElement(): void;\n    endElementAt(offset: number): void;\n    getCurrentTime(): number;\n    getSimpleDuration(): number;\n    getStartTime(): number;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimationElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGAnimationElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGAnimationElement: {\n    prototype: SVGAnimationElement;\n    new(): SVGAnimationElement;\n};\n\n/**\n * An interface for the <circle> element. The circle element is defined by the cx and cy attributes that denote the coordinates of the centre of the circle.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGCircleElement)\n */\ninterface SVGCircleElement extends SVGGeometryElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGCircleElement/cx) */\n    readonly cx: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGCircleElement/cy) */\n    readonly cy: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGCircleElement/r) */\n    readonly r: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGCircleElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGCircleElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGCircleElement: {\n    prototype: SVGCircleElement;\n    new(): SVGCircleElement;\n};\n\n/**\n * Provides access to the properties of <clipPath> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGClipPathElement)\n */\ninterface SVGClipPathElement extends SVGElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGClipPathElement/clipPathUnits) */\n    readonly clipPathUnits: SVGAnimatedEnumeration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGClipPathElement/transform) */\n    readonly transform: SVGAnimatedTransformList;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGClipPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGClipPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGClipPathElement: {\n    prototype: SVGClipPathElement;\n    new(): SVGClipPathElement;\n};\n\n/**\n * A base interface used by the component transfer function interfaces.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGComponentTransferFunctionElement)\n */\ninterface SVGComponentTransferFunctionElement extends SVGElement {\n    readonly amplitude: SVGAnimatedNumber;\n    readonly exponent: SVGAnimatedNumber;\n    readonly intercept: SVGAnimatedNumber;\n    readonly offset: SVGAnimatedNumber;\n    readonly slope: SVGAnimatedNumber;\n    readonly tableValues: SVGAnimatedNumberList;\n    readonly type: SVGAnimatedEnumeration;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_UNKNOWN: 0;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_IDENTITY: 1;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_TABLE: 2;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_DISCRETE: 3;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_LINEAR: 4;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_GAMMA: 5;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGComponentTransferFunctionElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGComponentTransferFunctionElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGComponentTransferFunctionElement: {\n    prototype: SVGComponentTransferFunctionElement;\n    new(): SVGComponentTransferFunctionElement;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_UNKNOWN: 0;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_IDENTITY: 1;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_TABLE: 2;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_DISCRETE: 3;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_LINEAR: 4;\n    readonly SVG_FECOMPONENTTRANSFER_TYPE_GAMMA: 5;\n};\n\n/**\n * Corresponds to the <defs> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGDefsElement)\n */\ninterface SVGDefsElement extends SVGGraphicsElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGDefsElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGDefsElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGDefsElement: {\n    prototype: SVGDefsElement;\n    new(): SVGDefsElement;\n};\n\n/**\n * Corresponds to the <desc> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGDescElement)\n */\ninterface SVGDescElement extends SVGElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGDescElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGDescElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGDescElement: {\n    prototype: SVGDescElement;\n    new(): SVGDescElement;\n};\n\ninterface SVGElementEventMap extends ElementEventMap, GlobalEventHandlersEventMap {\n}\n\n/**\n * All of the SVG DOM interfaces that correspond directly to elements in the SVG language derive from the SVGElement interface.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGElement)\n */\ninterface SVGElement extends Element, ElementCSSInlineStyle, GlobalEventHandlers, HTMLOrSVGElement {\n    /** @deprecated */\n    readonly className: any;\n    readonly ownerSVGElement: SVGSVGElement | null;\n    readonly viewportElement: SVGElement | null;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGElement: {\n    prototype: SVGElement;\n    new(): SVGElement;\n};\n\n/**\n * Provides access to the properties of <ellipse> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGEllipseElement)\n */\ninterface SVGEllipseElement extends SVGGeometryElement {\n    readonly cx: SVGAnimatedLength;\n    readonly cy: SVGAnimatedLength;\n    readonly rx: SVGAnimatedLength;\n    readonly ry: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGEllipseElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGEllipseElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGEllipseElement: {\n    prototype: SVGEllipseElement;\n    new(): SVGEllipseElement;\n};\n\n/**\n * Corresponds to the <feBlend> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEBlendElement)\n */\ninterface SVGFEBlendElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly in2: SVGAnimatedString;\n    readonly mode: SVGAnimatedEnumeration;\n    readonly SVG_FEBLEND_MODE_UNKNOWN: 0;\n    readonly SVG_FEBLEND_MODE_NORMAL: 1;\n    readonly SVG_FEBLEND_MODE_MULTIPLY: 2;\n    readonly SVG_FEBLEND_MODE_SCREEN: 3;\n    readonly SVG_FEBLEND_MODE_DARKEN: 4;\n    readonly SVG_FEBLEND_MODE_LIGHTEN: 5;\n    readonly SVG_FEBLEND_MODE_OVERLAY: 6;\n    readonly SVG_FEBLEND_MODE_COLOR_DODGE: 7;\n    readonly SVG_FEBLEND_MODE_COLOR_BURN: 8;\n    readonly SVG_FEBLEND_MODE_HARD_LIGHT: 9;\n    readonly SVG_FEBLEND_MODE_SOFT_LIGHT: 10;\n    readonly SVG_FEBLEND_MODE_DIFFERENCE: 11;\n    readonly SVG_FEBLEND_MODE_EXCLUSION: 12;\n    readonly SVG_FEBLEND_MODE_HUE: 13;\n    readonly SVG_FEBLEND_MODE_SATURATION: 14;\n    readonly SVG_FEBLEND_MODE_COLOR: 15;\n    readonly SVG_FEBLEND_MODE_LUMINOSITY: 16;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEBlendElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEBlendElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEBlendElement: {\n    prototype: SVGFEBlendElement;\n    new(): SVGFEBlendElement;\n    readonly SVG_FEBLEND_MODE_UNKNOWN: 0;\n    readonly SVG_FEBLEND_MODE_NORMAL: 1;\n    readonly SVG_FEBLEND_MODE_MULTIPLY: 2;\n    readonly SVG_FEBLEND_MODE_SCREEN: 3;\n    readonly SVG_FEBLEND_MODE_DARKEN: 4;\n    readonly SVG_FEBLEND_MODE_LIGHTEN: 5;\n    readonly SVG_FEBLEND_MODE_OVERLAY: 6;\n    readonly SVG_FEBLEND_MODE_COLOR_DODGE: 7;\n    readonly SVG_FEBLEND_MODE_COLOR_BURN: 8;\n    readonly SVG_FEBLEND_MODE_HARD_LIGHT: 9;\n    readonly SVG_FEBLEND_MODE_SOFT_LIGHT: 10;\n    readonly SVG_FEBLEND_MODE_DIFFERENCE: 11;\n    readonly SVG_FEBLEND_MODE_EXCLUSION: 12;\n    readonly SVG_FEBLEND_MODE_HUE: 13;\n    readonly SVG_FEBLEND_MODE_SATURATION: 14;\n    readonly SVG_FEBLEND_MODE_COLOR: 15;\n    readonly SVG_FEBLEND_MODE_LUMINOSITY: 16;\n};\n\n/**\n * Corresponds to the <feColorMatrix> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEColorMatrixElement)\n */\ninterface SVGFEColorMatrixElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly type: SVGAnimatedEnumeration;\n    readonly values: SVGAnimatedNumberList;\n    readonly SVG_FECOLORMATRIX_TYPE_UNKNOWN: 0;\n    readonly SVG_FECOLORMATRIX_TYPE_MATRIX: 1;\n    readonly SVG_FECOLORMATRIX_TYPE_SATURATE: 2;\n    readonly SVG_FECOLORMATRIX_TYPE_HUEROTATE: 3;\n    readonly SVG_FECOLORMATRIX_TYPE_LUMINANCETOALPHA: 4;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEColorMatrixElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEColorMatrixElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEColorMatrixElement: {\n    prototype: SVGFEColorMatrixElement;\n    new(): SVGFEColorMatrixElement;\n    readonly SVG_FECOLORMATRIX_TYPE_UNKNOWN: 0;\n    readonly SVG_FECOLORMATRIX_TYPE_MATRIX: 1;\n    readonly SVG_FECOLORMATRIX_TYPE_SATURATE: 2;\n    readonly SVG_FECOLORMATRIX_TYPE_HUEROTATE: 3;\n    readonly SVG_FECOLORMATRIX_TYPE_LUMINANCETOALPHA: 4;\n};\n\n/**\n * Corresponds to the <feComponentTransfer> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEComponentTransferElement)\n */\ninterface SVGFEComponentTransferElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEComponentTransferElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEComponentTransferElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEComponentTransferElement: {\n    prototype: SVGFEComponentTransferElement;\n    new(): SVGFEComponentTransferElement;\n};\n\n/**\n * Corresponds to the <feComposite> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFECompositeElement)\n */\ninterface SVGFECompositeElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly in2: SVGAnimatedString;\n    readonly k1: SVGAnimatedNumber;\n    readonly k2: SVGAnimatedNumber;\n    readonly k3: SVGAnimatedNumber;\n    readonly k4: SVGAnimatedNumber;\n    readonly operator: SVGAnimatedEnumeration;\n    readonly SVG_FECOMPOSITE_OPERATOR_UNKNOWN: 0;\n    readonly SVG_FECOMPOSITE_OPERATOR_OVER: 1;\n    readonly SVG_FECOMPOSITE_OPERATOR_IN: 2;\n    readonly SVG_FECOMPOSITE_OPERATOR_OUT: 3;\n    readonly SVG_FECOMPOSITE_OPERATOR_ATOP: 4;\n    readonly SVG_FECOMPOSITE_OPERATOR_XOR: 5;\n    readonly SVG_FECOMPOSITE_OPERATOR_ARITHMETIC: 6;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFECompositeElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFECompositeElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFECompositeElement: {\n    prototype: SVGFECompositeElement;\n    new(): SVGFECompositeElement;\n    readonly SVG_FECOMPOSITE_OPERATOR_UNKNOWN: 0;\n    readonly SVG_FECOMPOSITE_OPERATOR_OVER: 1;\n    readonly SVG_FECOMPOSITE_OPERATOR_IN: 2;\n    readonly SVG_FECOMPOSITE_OPERATOR_OUT: 3;\n    readonly SVG_FECOMPOSITE_OPERATOR_ATOP: 4;\n    readonly SVG_FECOMPOSITE_OPERATOR_XOR: 5;\n    readonly SVG_FECOMPOSITE_OPERATOR_ARITHMETIC: 6;\n};\n\n/**\n * Corresponds to the <feConvolveMatrix> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEConvolveMatrixElement)\n */\ninterface SVGFEConvolveMatrixElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly bias: SVGAnimatedNumber;\n    readonly divisor: SVGAnimatedNumber;\n    readonly edgeMode: SVGAnimatedEnumeration;\n    readonly in1: SVGAnimatedString;\n    readonly kernelMatrix: SVGAnimatedNumberList;\n    readonly kernelUnitLengthX: SVGAnimatedNumber;\n    readonly kernelUnitLengthY: SVGAnimatedNumber;\n    readonly orderX: SVGAnimatedInteger;\n    readonly orderY: SVGAnimatedInteger;\n    readonly preserveAlpha: SVGAnimatedBoolean;\n    readonly targetX: SVGAnimatedInteger;\n    readonly targetY: SVGAnimatedInteger;\n    readonly SVG_EDGEMODE_UNKNOWN: 0;\n    readonly SVG_EDGEMODE_DUPLICATE: 1;\n    readonly SVG_EDGEMODE_WRAP: 2;\n    readonly SVG_EDGEMODE_NONE: 3;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEConvolveMatrixElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEConvolveMatrixElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEConvolveMatrixElement: {\n    prototype: SVGFEConvolveMatrixElement;\n    new(): SVGFEConvolveMatrixElement;\n    readonly SVG_EDGEMODE_UNKNOWN: 0;\n    readonly SVG_EDGEMODE_DUPLICATE: 1;\n    readonly SVG_EDGEMODE_WRAP: 2;\n    readonly SVG_EDGEMODE_NONE: 3;\n};\n\n/**\n * Corresponds to the <feDiffuseLighting> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEDiffuseLightingElement)\n */\ninterface SVGFEDiffuseLightingElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly diffuseConstant: SVGAnimatedNumber;\n    readonly in1: SVGAnimatedString;\n    readonly kernelUnitLengthX: SVGAnimatedNumber;\n    readonly kernelUnitLengthY: SVGAnimatedNumber;\n    readonly surfaceScale: SVGAnimatedNumber;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDiffuseLightingElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDiffuseLightingElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEDiffuseLightingElement: {\n    prototype: SVGFEDiffuseLightingElement;\n    new(): SVGFEDiffuseLightingElement;\n};\n\n/**\n * Corresponds to the <feDisplacementMap> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEDisplacementMapElement)\n */\ninterface SVGFEDisplacementMapElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly in2: SVGAnimatedString;\n    readonly scale: SVGAnimatedNumber;\n    readonly xChannelSelector: SVGAnimatedEnumeration;\n    readonly yChannelSelector: SVGAnimatedEnumeration;\n    readonly SVG_CHANNEL_UNKNOWN: 0;\n    readonly SVG_CHANNEL_R: 1;\n    readonly SVG_CHANNEL_G: 2;\n    readonly SVG_CHANNEL_B: 3;\n    readonly SVG_CHANNEL_A: 4;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDisplacementMapElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDisplacementMapElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEDisplacementMapElement: {\n    prototype: SVGFEDisplacementMapElement;\n    new(): SVGFEDisplacementMapElement;\n    readonly SVG_CHANNEL_UNKNOWN: 0;\n    readonly SVG_CHANNEL_R: 1;\n    readonly SVG_CHANNEL_G: 2;\n    readonly SVG_CHANNEL_B: 3;\n    readonly SVG_CHANNEL_A: 4;\n};\n\n/**\n * Corresponds to the <feDistantLight> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEDistantLightElement)\n */\ninterface SVGFEDistantLightElement extends SVGElement {\n    readonly azimuth: SVGAnimatedNumber;\n    readonly elevation: SVGAnimatedNumber;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDistantLightElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDistantLightElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEDistantLightElement: {\n    prototype: SVGFEDistantLightElement;\n    new(): SVGFEDistantLightElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEDropShadowElement) */\ninterface SVGFEDropShadowElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly dx: SVGAnimatedNumber;\n    readonly dy: SVGAnimatedNumber;\n    readonly in1: SVGAnimatedString;\n    readonly stdDeviationX: SVGAnimatedNumber;\n    readonly stdDeviationY: SVGAnimatedNumber;\n    setStdDeviation(stdDeviationX: number, stdDeviationY: number): void;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDropShadowElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEDropShadowElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEDropShadowElement: {\n    prototype: SVGFEDropShadowElement;\n    new(): SVGFEDropShadowElement;\n};\n\n/**\n * Corresponds to the <feFlood> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEFloodElement)\n */\ninterface SVGFEFloodElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFloodElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFloodElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEFloodElement: {\n    prototype: SVGFEFloodElement;\n    new(): SVGFEFloodElement;\n};\n\n/**\n * Corresponds to the <feFuncA> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEFuncAElement)\n */\ninterface SVGFEFuncAElement extends SVGComponentTransferFunctionElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncAElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncAElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEFuncAElement: {\n    prototype: SVGFEFuncAElement;\n    new(): SVGFEFuncAElement;\n};\n\n/**\n * Corresponds to the <feFuncB> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEFuncBElement)\n */\ninterface SVGFEFuncBElement extends SVGComponentTransferFunctionElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncBElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncBElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEFuncBElement: {\n    prototype: SVGFEFuncBElement;\n    new(): SVGFEFuncBElement;\n};\n\n/**\n * Corresponds to the <feFuncG> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEFuncGElement)\n */\ninterface SVGFEFuncGElement extends SVGComponentTransferFunctionElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncGElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncGElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEFuncGElement: {\n    prototype: SVGFEFuncGElement;\n    new(): SVGFEFuncGElement;\n};\n\n/**\n * Corresponds to the <feFuncR> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEFuncRElement)\n */\ninterface SVGFEFuncRElement extends SVGComponentTransferFunctionElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncRElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEFuncRElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEFuncRElement: {\n    prototype: SVGFEFuncRElement;\n    new(): SVGFEFuncRElement;\n};\n\n/**\n * Corresponds to the <feGaussianBlur> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEGaussianBlurElement)\n */\ninterface SVGFEGaussianBlurElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly stdDeviationX: SVGAnimatedNumber;\n    readonly stdDeviationY: SVGAnimatedNumber;\n    setStdDeviation(stdDeviationX: number, stdDeviationY: number): void;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEGaussianBlurElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEGaussianBlurElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEGaussianBlurElement: {\n    prototype: SVGFEGaussianBlurElement;\n    new(): SVGFEGaussianBlurElement;\n};\n\n/**\n * Corresponds to the <feImage> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEImageElement)\n */\ninterface SVGFEImageElement extends SVGElement, SVGFilterPrimitiveStandardAttributes, SVGURIReference {\n    readonly preserveAspectRatio: SVGAnimatedPreserveAspectRatio;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEImageElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEImageElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEImageElement: {\n    prototype: SVGFEImageElement;\n    new(): SVGFEImageElement;\n};\n\n/**\n * Corresponds to the <feMerge> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEMergeElement)\n */\ninterface SVGFEMergeElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEMergeElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEMergeElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEMergeElement: {\n    prototype: SVGFEMergeElement;\n    new(): SVGFEMergeElement;\n};\n\n/**\n * Corresponds to the <feMergeNode> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEMergeNodeElement)\n */\ninterface SVGFEMergeNodeElement extends SVGElement {\n    readonly in1: SVGAnimatedString;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEMergeNodeElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEMergeNodeElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEMergeNodeElement: {\n    prototype: SVGFEMergeNodeElement;\n    new(): SVGFEMergeNodeElement;\n};\n\n/**\n * Corresponds to the <feMorphology> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEMorphologyElement)\n */\ninterface SVGFEMorphologyElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly operator: SVGAnimatedEnumeration;\n    readonly radiusX: SVGAnimatedNumber;\n    readonly radiusY: SVGAnimatedNumber;\n    readonly SVG_MORPHOLOGY_OPERATOR_UNKNOWN: 0;\n    readonly SVG_MORPHOLOGY_OPERATOR_ERODE: 1;\n    readonly SVG_MORPHOLOGY_OPERATOR_DILATE: 2;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEMorphologyElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEMorphologyElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEMorphologyElement: {\n    prototype: SVGFEMorphologyElement;\n    new(): SVGFEMorphologyElement;\n    readonly SVG_MORPHOLOGY_OPERATOR_UNKNOWN: 0;\n    readonly SVG_MORPHOLOGY_OPERATOR_ERODE: 1;\n    readonly SVG_MORPHOLOGY_OPERATOR_DILATE: 2;\n};\n\n/**\n * Corresponds to the <feOffset> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEOffsetElement)\n */\ninterface SVGFEOffsetElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly dx: SVGAnimatedNumber;\n    readonly dy: SVGAnimatedNumber;\n    readonly in1: SVGAnimatedString;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEOffsetElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEOffsetElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEOffsetElement: {\n    prototype: SVGFEOffsetElement;\n    new(): SVGFEOffsetElement;\n};\n\n/**\n * Corresponds to the <fePointLight> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFEPointLightElement)\n */\ninterface SVGFEPointLightElement extends SVGElement {\n    readonly x: SVGAnimatedNumber;\n    readonly y: SVGAnimatedNumber;\n    readonly z: SVGAnimatedNumber;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEPointLightElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFEPointLightElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFEPointLightElement: {\n    prototype: SVGFEPointLightElement;\n    new(): SVGFEPointLightElement;\n};\n\n/**\n * Corresponds to the <feSpecularLighting> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFESpecularLightingElement)\n */\ninterface SVGFESpecularLightingElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    readonly kernelUnitLengthX: SVGAnimatedNumber;\n    readonly kernelUnitLengthY: SVGAnimatedNumber;\n    readonly specularConstant: SVGAnimatedNumber;\n    readonly specularExponent: SVGAnimatedNumber;\n    readonly surfaceScale: SVGAnimatedNumber;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFESpecularLightingElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFESpecularLightingElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFESpecularLightingElement: {\n    prototype: SVGFESpecularLightingElement;\n    new(): SVGFESpecularLightingElement;\n};\n\n/**\n * Corresponds to the <feSpotLight> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFESpotLightElement)\n */\ninterface SVGFESpotLightElement extends SVGElement {\n    readonly limitingConeAngle: SVGAnimatedNumber;\n    readonly pointsAtX: SVGAnimatedNumber;\n    readonly pointsAtY: SVGAnimatedNumber;\n    readonly pointsAtZ: SVGAnimatedNumber;\n    readonly specularExponent: SVGAnimatedNumber;\n    readonly x: SVGAnimatedNumber;\n    readonly y: SVGAnimatedNumber;\n    readonly z: SVGAnimatedNumber;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFESpotLightElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFESpotLightElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFESpotLightElement: {\n    prototype: SVGFESpotLightElement;\n    new(): SVGFESpotLightElement;\n};\n\n/**\n * Corresponds to the <feTile> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFETileElement)\n */\ninterface SVGFETileElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly in1: SVGAnimatedString;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFETileElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFETileElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFETileElement: {\n    prototype: SVGFETileElement;\n    new(): SVGFETileElement;\n};\n\n/**\n * Corresponds to the <feTurbulence> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFETurbulenceElement)\n */\ninterface SVGFETurbulenceElement extends SVGElement, SVGFilterPrimitiveStandardAttributes {\n    readonly baseFrequencyX: SVGAnimatedNumber;\n    readonly baseFrequencyY: SVGAnimatedNumber;\n    readonly numOctaves: SVGAnimatedInteger;\n    readonly seed: SVGAnimatedNumber;\n    readonly stitchTiles: SVGAnimatedEnumeration;\n    readonly type: SVGAnimatedEnumeration;\n    readonly SVG_TURBULENCE_TYPE_UNKNOWN: 0;\n    readonly SVG_TURBULENCE_TYPE_FRACTALNOISE: 1;\n    readonly SVG_TURBULENCE_TYPE_TURBULENCE: 2;\n    readonly SVG_STITCHTYPE_UNKNOWN: 0;\n    readonly SVG_STITCHTYPE_STITCH: 1;\n    readonly SVG_STITCHTYPE_NOSTITCH: 2;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFETurbulenceElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFETurbulenceElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFETurbulenceElement: {\n    prototype: SVGFETurbulenceElement;\n    new(): SVGFETurbulenceElement;\n    readonly SVG_TURBULENCE_TYPE_UNKNOWN: 0;\n    readonly SVG_TURBULENCE_TYPE_FRACTALNOISE: 1;\n    readonly SVG_TURBULENCE_TYPE_TURBULENCE: 2;\n    readonly SVG_STITCHTYPE_UNKNOWN: 0;\n    readonly SVG_STITCHTYPE_STITCH: 1;\n    readonly SVG_STITCHTYPE_NOSTITCH: 2;\n};\n\n/**\n * Provides access to the properties of <filter> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGFilterElement)\n */\ninterface SVGFilterElement extends SVGElement, SVGURIReference {\n    readonly filterUnits: SVGAnimatedEnumeration;\n    readonly height: SVGAnimatedLength;\n    readonly primitiveUnits: SVGAnimatedEnumeration;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFilterElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGFilterElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGFilterElement: {\n    prototype: SVGFilterElement;\n    new(): SVGFilterElement;\n};\n\ninterface SVGFilterPrimitiveStandardAttributes {\n    readonly height: SVGAnimatedLength;\n    readonly result: SVGAnimatedString;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n}\n\ninterface SVGFitToViewBox {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/preserveAspectRatio) */\n    readonly preserveAspectRatio: SVGAnimatedPreserveAspectRatio;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/viewBox) */\n    readonly viewBox: SVGAnimatedRect;\n}\n\n/**\n * Provides access to the properties of <foreignObject> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGForeignObjectElement)\n */\ninterface SVGForeignObjectElement extends SVGGraphicsElement {\n    readonly height: SVGAnimatedLength;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGForeignObjectElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGForeignObjectElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGForeignObjectElement: {\n    prototype: SVGForeignObjectElement;\n    new(): SVGForeignObjectElement;\n};\n\n/**\n * Corresponds to the <g> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGElement)\n */\ninterface SVGGElement extends SVGGraphicsElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGGElement: {\n    prototype: SVGGElement;\n    new(): SVGGElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGeometryElement) */\ninterface SVGGeometryElement extends SVGGraphicsElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGeometryElement/pathLength) */\n    readonly pathLength: SVGAnimatedNumber;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGeometryElement/getPointAtLength) */\n    getPointAtLength(distance: number): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGeometryElement/getTotalLength) */\n    getTotalLength(): number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGeometryElement/isPointInFill) */\n    isPointInFill(point?: DOMPointInit): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGeometryElement/isPointInStroke) */\n    isPointInStroke(point?: DOMPointInit): boolean;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGeometryElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGeometryElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGGeometryElement: {\n    prototype: SVGGeometryElement;\n    new(): SVGGeometryElement;\n};\n\n/**\n * The SVGGradient interface is a base interface used by SVGLinearGradientElement and SVGRadialGradientElement.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGradientElement)\n */\ninterface SVGGradientElement extends SVGElement, SVGURIReference {\n    readonly gradientTransform: SVGAnimatedTransformList;\n    readonly gradientUnits: SVGAnimatedEnumeration;\n    readonly spreadMethod: SVGAnimatedEnumeration;\n    readonly SVG_SPREADMETHOD_UNKNOWN: 0;\n    readonly SVG_SPREADMETHOD_PAD: 1;\n    readonly SVG_SPREADMETHOD_REFLECT: 2;\n    readonly SVG_SPREADMETHOD_REPEAT: 3;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGradientElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGradientElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGGradientElement: {\n    prototype: SVGGradientElement;\n    new(): SVGGradientElement;\n    readonly SVG_SPREADMETHOD_UNKNOWN: 0;\n    readonly SVG_SPREADMETHOD_PAD: 1;\n    readonly SVG_SPREADMETHOD_REFLECT: 2;\n    readonly SVG_SPREADMETHOD_REPEAT: 3;\n};\n\n/**\n * SVG elements whose primary purpose is to directly render graphics into a group.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGraphicsElement)\n */\ninterface SVGGraphicsElement extends SVGElement, SVGTests {\n    readonly transform: SVGAnimatedTransformList;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGGraphicsElement/getBBox) */\n    getBBox(options?: SVGBoundingBoxOptions): DOMRect;\n    getCTM(): DOMMatrix | null;\n    getScreenCTM(): DOMMatrix | null;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGraphicsElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGGraphicsElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGGraphicsElement: {\n    prototype: SVGGraphicsElement;\n    new(): SVGGraphicsElement;\n};\n\n/**\n * Corresponds to the <image> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGImageElement)\n */\ninterface SVGImageElement extends SVGGraphicsElement, SVGURIReference {\n    crossOrigin: string | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGImageElement/height) */\n    readonly height: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGImageElement/preserveAspectRatio) */\n    readonly preserveAspectRatio: SVGAnimatedPreserveAspectRatio;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGImageElement/width) */\n    readonly width: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGImageElement/x) */\n    readonly x: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGImageElement/y) */\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGImageElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGImageElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGImageElement: {\n    prototype: SVGImageElement;\n    new(): SVGImageElement;\n};\n\n/**\n * Correspond to the <length> basic data type.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGLength)\n */\ninterface SVGLength {\n    readonly unitType: number;\n    value: number;\n    valueAsString: string;\n    valueInSpecifiedUnits: number;\n    convertToSpecifiedUnits(unitType: number): void;\n    newValueSpecifiedUnits(unitType: number, valueInSpecifiedUnits: number): void;\n    readonly SVG_LENGTHTYPE_UNKNOWN: 0;\n    readonly SVG_LENGTHTYPE_NUMBER: 1;\n    readonly SVG_LENGTHTYPE_PERCENTAGE: 2;\n    readonly SVG_LENGTHTYPE_EMS: 3;\n    readonly SVG_LENGTHTYPE_EXS: 4;\n    readonly SVG_LENGTHTYPE_PX: 5;\n    readonly SVG_LENGTHTYPE_CM: 6;\n    readonly SVG_LENGTHTYPE_MM: 7;\n    readonly SVG_LENGTHTYPE_IN: 8;\n    readonly SVG_LENGTHTYPE_PT: 9;\n    readonly SVG_LENGTHTYPE_PC: 10;\n}\n\ndeclare var SVGLength: {\n    prototype: SVGLength;\n    new(): SVGLength;\n    readonly SVG_LENGTHTYPE_UNKNOWN: 0;\n    readonly SVG_LENGTHTYPE_NUMBER: 1;\n    readonly SVG_LENGTHTYPE_PERCENTAGE: 2;\n    readonly SVG_LENGTHTYPE_EMS: 3;\n    readonly SVG_LENGTHTYPE_EXS: 4;\n    readonly SVG_LENGTHTYPE_PX: 5;\n    readonly SVG_LENGTHTYPE_CM: 6;\n    readonly SVG_LENGTHTYPE_MM: 7;\n    readonly SVG_LENGTHTYPE_IN: 8;\n    readonly SVG_LENGTHTYPE_PT: 9;\n    readonly SVG_LENGTHTYPE_PC: 10;\n};\n\n/**\n * The SVGLengthList defines a list of SVGLength objects.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGLengthList)\n */\ninterface SVGLengthList {\n    readonly length: number;\n    readonly numberOfItems: number;\n    appendItem(newItem: SVGLength): SVGLength;\n    clear(): void;\n    getItem(index: number): SVGLength;\n    initialize(newItem: SVGLength): SVGLength;\n    insertItemBefore(newItem: SVGLength, index: number): SVGLength;\n    removeItem(index: number): SVGLength;\n    replaceItem(newItem: SVGLength, index: number): SVGLength;\n    [index: number]: SVGLength;\n}\n\ndeclare var SVGLengthList: {\n    prototype: SVGLengthList;\n    new(): SVGLengthList;\n};\n\n/**\n * Provides access to the properties of <line> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGLineElement)\n */\ninterface SVGLineElement extends SVGGeometryElement {\n    readonly x1: SVGAnimatedLength;\n    readonly x2: SVGAnimatedLength;\n    readonly y1: SVGAnimatedLength;\n    readonly y2: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGLineElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGLineElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGLineElement: {\n    prototype: SVGLineElement;\n    new(): SVGLineElement;\n};\n\n/**\n * Corresponds to the <linearGradient> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGLinearGradientElement)\n */\ninterface SVGLinearGradientElement extends SVGGradientElement {\n    readonly x1: SVGAnimatedLength;\n    readonly x2: SVGAnimatedLength;\n    readonly y1: SVGAnimatedLength;\n    readonly y2: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGLinearGradientElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGLinearGradientElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGLinearGradientElement: {\n    prototype: SVGLinearGradientElement;\n    new(): SVGLinearGradientElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMPathElement) */\ninterface SVGMPathElement extends SVGElement, SVGURIReference {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGMPathElement: {\n    prototype: SVGMPathElement;\n    new(): SVGMPathElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement) */\ninterface SVGMarkerElement extends SVGElement, SVGFitToViewBox {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/markerHeight) */\n    readonly markerHeight: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/markerUnits) */\n    readonly markerUnits: SVGAnimatedEnumeration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/markerWidth) */\n    readonly markerWidth: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/orientAngle) */\n    readonly orientAngle: SVGAnimatedAngle;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/orientType) */\n    readonly orientType: SVGAnimatedEnumeration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/refX) */\n    readonly refX: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/refY) */\n    readonly refY: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/setOrientToAngle) */\n    setOrientToAngle(angle: SVGAngle): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMarkerElement/setOrientToAuto) */\n    setOrientToAuto(): void;\n    readonly SVG_MARKERUNITS_UNKNOWN: 0;\n    readonly SVG_MARKERUNITS_USERSPACEONUSE: 1;\n    readonly SVG_MARKERUNITS_STROKEWIDTH: 2;\n    readonly SVG_MARKER_ORIENT_UNKNOWN: 0;\n    readonly SVG_MARKER_ORIENT_AUTO: 1;\n    readonly SVG_MARKER_ORIENT_ANGLE: 2;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMarkerElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMarkerElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGMarkerElement: {\n    prototype: SVGMarkerElement;\n    new(): SVGMarkerElement;\n    readonly SVG_MARKERUNITS_UNKNOWN: 0;\n    readonly SVG_MARKERUNITS_USERSPACEONUSE: 1;\n    readonly SVG_MARKERUNITS_STROKEWIDTH: 2;\n    readonly SVG_MARKER_ORIENT_UNKNOWN: 0;\n    readonly SVG_MARKER_ORIENT_AUTO: 1;\n    readonly SVG_MARKER_ORIENT_ANGLE: 2;\n};\n\n/**\n * Provides access to the properties of <mask> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement)\n */\ninterface SVGMaskElement extends SVGElement {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement/height) */\n    readonly height: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement/maskContentUnits) */\n    readonly maskContentUnits: SVGAnimatedEnumeration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement/maskUnits) */\n    readonly maskUnits: SVGAnimatedEnumeration;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement/width) */\n    readonly width: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement/x) */\n    readonly x: SVGAnimatedLength;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMaskElement/y) */\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMaskElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMaskElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGMaskElement: {\n    prototype: SVGMaskElement;\n    new(): SVGMaskElement;\n};\n\n/**\n * Corresponds to the <metadata> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGMetadataElement)\n */\ninterface SVGMetadataElement extends SVGElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMetadataElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGMetadataElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGMetadataElement: {\n    prototype: SVGMetadataElement;\n    new(): SVGMetadataElement;\n};\n\n/**\n * Corresponds to the <number> basic data type.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGNumber)\n */\ninterface SVGNumber {\n    value: number;\n}\n\ndeclare var SVGNumber: {\n    prototype: SVGNumber;\n    new(): SVGNumber;\n};\n\n/**\n * The SVGNumberList defines a list of SVGNumber objects.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGNumberList)\n */\ninterface SVGNumberList {\n    readonly length: number;\n    readonly numberOfItems: number;\n    appendItem(newItem: SVGNumber): SVGNumber;\n    clear(): void;\n    getItem(index: number): SVGNumber;\n    initialize(newItem: SVGNumber): SVGNumber;\n    insertItemBefore(newItem: SVGNumber, index: number): SVGNumber;\n    removeItem(index: number): SVGNumber;\n    replaceItem(newItem: SVGNumber, index: number): SVGNumber;\n    [index: number]: SVGNumber;\n}\n\ndeclare var SVGNumberList: {\n    prototype: SVGNumberList;\n    new(): SVGNumberList;\n};\n\n/**\n * Corresponds to the <path> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPathElement)\n */\ninterface SVGPathElement extends SVGGeometryElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGPathElement: {\n    prototype: SVGPathElement;\n    new(): SVGPathElement;\n};\n\n/**\n * Corresponds to the <pattern> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPatternElement)\n */\ninterface SVGPatternElement extends SVGElement, SVGFitToViewBox, SVGURIReference {\n    readonly height: SVGAnimatedLength;\n    readonly patternContentUnits: SVGAnimatedEnumeration;\n    readonly patternTransform: SVGAnimatedTransformList;\n    readonly patternUnits: SVGAnimatedEnumeration;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPatternElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPatternElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGPatternElement: {\n    prototype: SVGPatternElement;\n    new(): SVGPatternElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList) */\ninterface SVGPointList {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/numberOfItems) */\n    readonly numberOfItems: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/appendItem) */\n    appendItem(newItem: DOMPoint): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/clear) */\n    clear(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/getItem) */\n    getItem(index: number): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/initialize) */\n    initialize(newItem: DOMPoint): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/insertItemBefore) */\n    insertItemBefore(newItem: DOMPoint, index: number): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/removeItem) */\n    removeItem(index: number): DOMPoint;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPointList/replaceItem) */\n    replaceItem(newItem: DOMPoint, index: number): DOMPoint;\n    [index: number]: DOMPoint;\n}\n\ndeclare var SVGPointList: {\n    prototype: SVGPointList;\n    new(): SVGPointList;\n};\n\n/**\n * Provides access to the properties of <polygon> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPolygonElement)\n */\ninterface SVGPolygonElement extends SVGGeometryElement, SVGAnimatedPoints {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPolygonElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPolygonElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGPolygonElement: {\n    prototype: SVGPolygonElement;\n    new(): SVGPolygonElement;\n};\n\n/**\n * Provides access to the properties of <polyline> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPolylineElement)\n */\ninterface SVGPolylineElement extends SVGGeometryElement, SVGAnimatedPoints {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPolylineElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGPolylineElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGPolylineElement: {\n    prototype: SVGPolylineElement;\n    new(): SVGPolylineElement;\n};\n\n/**\n * Corresponds to the preserveAspectRatio attribute, which is available for some of SVG's elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGPreserveAspectRatio)\n */\ninterface SVGPreserveAspectRatio {\n    align: number;\n    meetOrSlice: number;\n    readonly SVG_PRESERVEASPECTRATIO_UNKNOWN: 0;\n    readonly SVG_PRESERVEASPECTRATIO_NONE: 1;\n    readonly SVG_PRESERVEASPECTRATIO_XMINYMIN: 2;\n    readonly SVG_PRESERVEASPECTRATIO_XMIDYMIN: 3;\n    readonly SVG_PRESERVEASPECTRATIO_XMAXYMIN: 4;\n    readonly SVG_PRESERVEASPECTRATIO_XMINYMID: 5;\n    readonly SVG_PRESERVEASPECTRATIO_XMIDYMID: 6;\n    readonly SVG_PRESERVEASPECTRATIO_XMAXYMID: 7;\n    readonly SVG_PRESERVEASPECTRATIO_XMINYMAX: 8;\n    readonly SVG_PRESERVEASPECTRATIO_XMIDYMAX: 9;\n    readonly SVG_PRESERVEASPECTRATIO_XMAXYMAX: 10;\n    readonly SVG_MEETORSLICE_UNKNOWN: 0;\n    readonly SVG_MEETORSLICE_MEET: 1;\n    readonly SVG_MEETORSLICE_SLICE: 2;\n}\n\ndeclare var SVGPreserveAspectRatio: {\n    prototype: SVGPreserveAspectRatio;\n    new(): SVGPreserveAspectRatio;\n    readonly SVG_PRESERVEASPECTRATIO_UNKNOWN: 0;\n    readonly SVG_PRESERVEASPECTRATIO_NONE: 1;\n    readonly SVG_PRESERVEASPECTRATIO_XMINYMIN: 2;\n    readonly SVG_PRESERVEASPECTRATIO_XMIDYMIN: 3;\n    readonly SVG_PRESERVEASPECTRATIO_XMAXYMIN: 4;\n    readonly SVG_PRESERVEASPECTRATIO_XMINYMID: 5;\n    readonly SVG_PRESERVEASPECTRATIO_XMIDYMID: 6;\n    readonly SVG_PRESERVEASPECTRATIO_XMAXYMID: 7;\n    readonly SVG_PRESERVEASPECTRATIO_XMINYMAX: 8;\n    readonly SVG_PRESERVEASPECTRATIO_XMIDYMAX: 9;\n    readonly SVG_PRESERVEASPECTRATIO_XMAXYMAX: 10;\n    readonly SVG_MEETORSLICE_UNKNOWN: 0;\n    readonly SVG_MEETORSLICE_MEET: 1;\n    readonly SVG_MEETORSLICE_SLICE: 2;\n};\n\n/**\n * Corresponds to the <RadialGradient> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGRadialGradientElement)\n */\ninterface SVGRadialGradientElement extends SVGGradientElement {\n    readonly cx: SVGAnimatedLength;\n    readonly cy: SVGAnimatedLength;\n    readonly fr: SVGAnimatedLength;\n    readonly fx: SVGAnimatedLength;\n    readonly fy: SVGAnimatedLength;\n    readonly r: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGRadialGradientElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGRadialGradientElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGRadialGradientElement: {\n    prototype: SVGRadialGradientElement;\n    new(): SVGRadialGradientElement;\n};\n\n/**\n * Provides access to the properties of <rect> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGRectElement)\n */\ninterface SVGRectElement extends SVGGeometryElement {\n    readonly height: SVGAnimatedLength;\n    readonly rx: SVGAnimatedLength;\n    readonly ry: SVGAnimatedLength;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGRectElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGRectElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGRectElement: {\n    prototype: SVGRectElement;\n    new(): SVGRectElement;\n};\n\ninterface SVGSVGElementEventMap extends SVGElementEventMap, WindowEventHandlersEventMap {\n}\n\n/**\n * Provides access to the properties of <svg> elements, as well as methods to manipulate them. This interface contains also various miscellaneous commonly-used utility methods, such as matrix operations and the ability to control the time of redraw on visual rendering devices.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGSVGElement)\n */\ninterface SVGSVGElement extends SVGGraphicsElement, SVGFitToViewBox, WindowEventHandlers {\n    currentScale: number;\n    readonly currentTranslate: DOMPointReadOnly;\n    readonly height: SVGAnimatedLength;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n    animationsPaused(): boolean;\n    checkEnclosure(element: SVGElement, rect: DOMRectReadOnly): boolean;\n    checkIntersection(element: SVGElement, rect: DOMRectReadOnly): boolean;\n    createSVGAngle(): SVGAngle;\n    createSVGLength(): SVGLength;\n    createSVGMatrix(): DOMMatrix;\n    createSVGNumber(): SVGNumber;\n    createSVGPoint(): DOMPoint;\n    createSVGRect(): DOMRect;\n    createSVGTransform(): SVGTransform;\n    createSVGTransformFromMatrix(matrix?: DOMMatrix2DInit): SVGTransform;\n    deselectAll(): void;\n    /** @deprecated */\n    forceRedraw(): void;\n    getCurrentTime(): number;\n    getElementById(elementId: string): Element;\n    getEnclosureList(rect: DOMRectReadOnly, referenceElement: SVGElement | null): NodeListOf<SVGCircleElement | SVGEllipseElement | SVGImageElement | SVGLineElement | SVGPathElement | SVGPolygonElement | SVGPolylineElement | SVGRectElement | SVGTextElement | SVGUseElement>;\n    getIntersectionList(rect: DOMRectReadOnly, referenceElement: SVGElement | null): NodeListOf<SVGCircleElement | SVGEllipseElement | SVGImageElement | SVGLineElement | SVGPathElement | SVGPolygonElement | SVGPolylineElement | SVGRectElement | SVGTextElement | SVGUseElement>;\n    pauseAnimations(): void;\n    setCurrentTime(seconds: number): void;\n    /** @deprecated */\n    suspendRedraw(maxWaitMilliseconds: number): number;\n    unpauseAnimations(): void;\n    /** @deprecated */\n    unsuspendRedraw(suspendHandleID: number): void;\n    /** @deprecated */\n    unsuspendRedrawAll(): void;\n    addEventListener<K extends keyof SVGSVGElementEventMap>(type: K, listener: (this: SVGSVGElement, ev: SVGSVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGSVGElementEventMap>(type: K, listener: (this: SVGSVGElement, ev: SVGSVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGSVGElement: {\n    prototype: SVGSVGElement;\n    new(): SVGSVGElement;\n};\n\n/**\n * Corresponds to the SVG <script> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGScriptElement)\n */\ninterface SVGScriptElement extends SVGElement, SVGURIReference {\n    type: string;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGScriptElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGScriptElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGScriptElement: {\n    prototype: SVGScriptElement;\n    new(): SVGScriptElement;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGSetElement) */\ninterface SVGSetElement extends SVGAnimationElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGSetElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGSetElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGSetElement: {\n    prototype: SVGSetElement;\n    new(): SVGSetElement;\n};\n\n/**\n * Corresponds to the <stop> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGStopElement)\n */\ninterface SVGStopElement extends SVGElement {\n    readonly offset: SVGAnimatedNumber;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGStopElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGStopElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGStopElement: {\n    prototype: SVGStopElement;\n    new(): SVGStopElement;\n};\n\n/**\n * The SVGStringList defines a list of DOMString objects.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGStringList)\n */\ninterface SVGStringList {\n    readonly length: number;\n    readonly numberOfItems: number;\n    appendItem(newItem: string): string;\n    clear(): void;\n    getItem(index: number): string;\n    initialize(newItem: string): string;\n    insertItemBefore(newItem: string, index: number): string;\n    removeItem(index: number): string;\n    replaceItem(newItem: string, index: number): string;\n    [index: number]: string;\n}\n\ndeclare var SVGStringList: {\n    prototype: SVGStringList;\n    new(): SVGStringList;\n};\n\n/**\n * Corresponds to the SVG <style> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGStyleElement)\n */\ninterface SVGStyleElement extends SVGElement, LinkStyle {\n    disabled: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGStyleElement/media) */\n    media: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGStyleElement/title) */\n    title: string;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGStyleElement/type)\n     */\n    type: string;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGStyleElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGStyleElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGStyleElement: {\n    prototype: SVGStyleElement;\n    new(): SVGStyleElement;\n};\n\n/**\n * Corresponds to the <switch> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGSwitchElement)\n */\ninterface SVGSwitchElement extends SVGGraphicsElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGSwitchElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGSwitchElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGSwitchElement: {\n    prototype: SVGSwitchElement;\n    new(): SVGSwitchElement;\n};\n\n/**\n * Corresponds to the <symbol> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGSymbolElement)\n */\ninterface SVGSymbolElement extends SVGElement, SVGFitToViewBox {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGSymbolElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGSymbolElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGSymbolElement: {\n    prototype: SVGSymbolElement;\n    new(): SVGSymbolElement;\n};\n\n/**\n * A <tspan> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTSpanElement)\n */\ninterface SVGTSpanElement extends SVGTextPositioningElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTSpanElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTSpanElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGTSpanElement: {\n    prototype: SVGTSpanElement;\n    new(): SVGTSpanElement;\n};\n\ninterface SVGTests {\n    readonly requiredExtensions: SVGStringList;\n    readonly systemLanguage: SVGStringList;\n}\n\n/**\n * Implemented by elements that support rendering child text content. It is inherited by various text-related interfaces, such as SVGTextElement, SVGTSpanElement, SVGTRefElement, SVGAltGlyphElement and SVGTextPathElement.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTextContentElement)\n */\ninterface SVGTextContentElement extends SVGGraphicsElement {\n    readonly lengthAdjust: SVGAnimatedEnumeration;\n    readonly textLength: SVGAnimatedLength;\n    getCharNumAtPosition(point?: DOMPointInit): number;\n    getComputedTextLength(): number;\n    getEndPositionOfChar(charnum: number): DOMPoint;\n    getExtentOfChar(charnum: number): DOMRect;\n    getNumberOfChars(): number;\n    getRotationOfChar(charnum: number): number;\n    getStartPositionOfChar(charnum: number): DOMPoint;\n    getSubStringLength(charnum: number, nchars: number): number;\n    /** @deprecated */\n    selectSubString(charnum: number, nchars: number): void;\n    readonly LENGTHADJUST_UNKNOWN: 0;\n    readonly LENGTHADJUST_SPACING: 1;\n    readonly LENGTHADJUST_SPACINGANDGLYPHS: 2;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextContentElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextContentElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGTextContentElement: {\n    prototype: SVGTextContentElement;\n    new(): SVGTextContentElement;\n    readonly LENGTHADJUST_UNKNOWN: 0;\n    readonly LENGTHADJUST_SPACING: 1;\n    readonly LENGTHADJUST_SPACINGANDGLYPHS: 2;\n};\n\n/**\n * Corresponds to the <text> elements.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTextElement)\n */\ninterface SVGTextElement extends SVGTextPositioningElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGTextElement: {\n    prototype: SVGTextElement;\n    new(): SVGTextElement;\n};\n\n/**\n * Corresponds to the <textPath> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTextPathElement)\n */\ninterface SVGTextPathElement extends SVGTextContentElement, SVGURIReference {\n    readonly method: SVGAnimatedEnumeration;\n    readonly spacing: SVGAnimatedEnumeration;\n    readonly startOffset: SVGAnimatedLength;\n    readonly TEXTPATH_METHODTYPE_UNKNOWN: 0;\n    readonly TEXTPATH_METHODTYPE_ALIGN: 1;\n    readonly TEXTPATH_METHODTYPE_STRETCH: 2;\n    readonly TEXTPATH_SPACINGTYPE_UNKNOWN: 0;\n    readonly TEXTPATH_SPACINGTYPE_AUTO: 1;\n    readonly TEXTPATH_SPACINGTYPE_EXACT: 2;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextPathElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGTextPathElement: {\n    prototype: SVGTextPathElement;\n    new(): SVGTextPathElement;\n    readonly TEXTPATH_METHODTYPE_UNKNOWN: 0;\n    readonly TEXTPATH_METHODTYPE_ALIGN: 1;\n    readonly TEXTPATH_METHODTYPE_STRETCH: 2;\n    readonly TEXTPATH_SPACINGTYPE_UNKNOWN: 0;\n    readonly TEXTPATH_SPACINGTYPE_AUTO: 1;\n    readonly TEXTPATH_SPACINGTYPE_EXACT: 2;\n};\n\n/**\n * Implemented by elements that support attributes that position individual text glyphs. It is inherited by SVGTextElement, SVGTSpanElement, SVGTRefElement and SVGAltGlyphElement.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTextPositioningElement)\n */\ninterface SVGTextPositioningElement extends SVGTextContentElement {\n    readonly dx: SVGAnimatedLengthList;\n    readonly dy: SVGAnimatedLengthList;\n    readonly rotate: SVGAnimatedNumberList;\n    readonly x: SVGAnimatedLengthList;\n    readonly y: SVGAnimatedLengthList;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextPositioningElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTextPositioningElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGTextPositioningElement: {\n    prototype: SVGTextPositioningElement;\n    new(): SVGTextPositioningElement;\n};\n\n/**\n * Corresponds to the <title> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTitleElement)\n */\ninterface SVGTitleElement extends SVGElement {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTitleElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGTitleElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGTitleElement: {\n    prototype: SVGTitleElement;\n    new(): SVGTitleElement;\n};\n\n/**\n * SVGTransform is the interface for one of the component transformations within an SVGTransformList; thus, an SVGTransform object corresponds to a single component (e.g., scale(…) or matrix(…)) within a transform attribute.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTransform)\n */\ninterface SVGTransform {\n    readonly angle: number;\n    readonly matrix: DOMMatrix;\n    readonly type: number;\n    setMatrix(matrix?: DOMMatrix2DInit): void;\n    setRotate(angle: number, cx: number, cy: number): void;\n    setScale(sx: number, sy: number): void;\n    setSkewX(angle: number): void;\n    setSkewY(angle: number): void;\n    setTranslate(tx: number, ty: number): void;\n    readonly SVG_TRANSFORM_UNKNOWN: 0;\n    readonly SVG_TRANSFORM_MATRIX: 1;\n    readonly SVG_TRANSFORM_TRANSLATE: 2;\n    readonly SVG_TRANSFORM_SCALE: 3;\n    readonly SVG_TRANSFORM_ROTATE: 4;\n    readonly SVG_TRANSFORM_SKEWX: 5;\n    readonly SVG_TRANSFORM_SKEWY: 6;\n}\n\ndeclare var SVGTransform: {\n    prototype: SVGTransform;\n    new(): SVGTransform;\n    readonly SVG_TRANSFORM_UNKNOWN: 0;\n    readonly SVG_TRANSFORM_MATRIX: 1;\n    readonly SVG_TRANSFORM_TRANSLATE: 2;\n    readonly SVG_TRANSFORM_SCALE: 3;\n    readonly SVG_TRANSFORM_ROTATE: 4;\n    readonly SVG_TRANSFORM_SKEWX: 5;\n    readonly SVG_TRANSFORM_SKEWY: 6;\n};\n\n/**\n * The SVGTransformList defines a list of SVGTransform objects.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGTransformList)\n */\ninterface SVGTransformList {\n    readonly length: number;\n    readonly numberOfItems: number;\n    appendItem(newItem: SVGTransform): SVGTransform;\n    clear(): void;\n    consolidate(): SVGTransform | null;\n    createSVGTransformFromMatrix(matrix?: DOMMatrix2DInit): SVGTransform;\n    getItem(index: number): SVGTransform;\n    initialize(newItem: SVGTransform): SVGTransform;\n    insertItemBefore(newItem: SVGTransform, index: number): SVGTransform;\n    removeItem(index: number): SVGTransform;\n    replaceItem(newItem: SVGTransform, index: number): SVGTransform;\n    [index: number]: SVGTransform;\n}\n\ndeclare var SVGTransformList: {\n    prototype: SVGTransformList;\n    new(): SVGTransformList;\n};\n\ninterface SVGURIReference {\n    readonly href: SVGAnimatedString;\n}\n\n/**\n * A commonly used set of constants used for reflecting gradientUnits, patternContentUnits and other similar attributes.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGUnitTypes)\n */\ninterface SVGUnitTypes {\n    readonly SVG_UNIT_TYPE_UNKNOWN: 0;\n    readonly SVG_UNIT_TYPE_USERSPACEONUSE: 1;\n    readonly SVG_UNIT_TYPE_OBJECTBOUNDINGBOX: 2;\n}\n\ndeclare var SVGUnitTypes: {\n    prototype: SVGUnitTypes;\n    new(): SVGUnitTypes;\n    readonly SVG_UNIT_TYPE_UNKNOWN: 0;\n    readonly SVG_UNIT_TYPE_USERSPACEONUSE: 1;\n    readonly SVG_UNIT_TYPE_OBJECTBOUNDINGBOX: 2;\n};\n\n/**\n * Corresponds to the <use> element.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGUseElement)\n */\ninterface SVGUseElement extends SVGGraphicsElement, SVGURIReference {\n    readonly height: SVGAnimatedLength;\n    readonly width: SVGAnimatedLength;\n    readonly x: SVGAnimatedLength;\n    readonly y: SVGAnimatedLength;\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGUseElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGUseElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGUseElement: {\n    prototype: SVGUseElement;\n    new(): SVGUseElement;\n};\n\n/**\n * Provides access to the properties of <view> elements, as well as methods to manipulate them.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SVGViewElement)\n */\ninterface SVGViewElement extends SVGElement, SVGFitToViewBox {\n    addEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGViewElement, ev: SVGElementEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SVGElementEventMap>(type: K, listener: (this: SVGViewElement, ev: SVGElementEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SVGViewElement: {\n    prototype: SVGViewElement;\n    new(): SVGViewElement;\n};\n\n/**\n * A screen, usually the one on which the current window is being rendered, and is obtained using window.screen.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen)\n */\ninterface Screen {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/availHeight) */\n    readonly availHeight: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/availWidth) */\n    readonly availWidth: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/colorDepth) */\n    readonly colorDepth: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/height) */\n    readonly height: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/orientation) */\n    readonly orientation: ScreenOrientation;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/pixelDepth) */\n    readonly pixelDepth: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Screen/width) */\n    readonly width: number;\n}\n\ndeclare var Screen: {\n    prototype: Screen;\n    new(): Screen;\n};\n\ninterface ScreenOrientationEventMap {\n    \"change\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScreenOrientation) */\ninterface ScreenOrientation extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScreenOrientation/angle) */\n    readonly angle: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScreenOrientation/change_event) */\n    onchange: ((this: ScreenOrientation, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScreenOrientation/type) */\n    readonly type: OrientationType;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScreenOrientation/unlock) */\n    unlock(): void;\n    addEventListener<K extends keyof ScreenOrientationEventMap>(type: K, listener: (this: ScreenOrientation, ev: ScreenOrientationEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ScreenOrientationEventMap>(type: K, listener: (this: ScreenOrientation, ev: ScreenOrientationEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var ScreenOrientation: {\n    prototype: ScreenOrientation;\n    new(): ScreenOrientation;\n};\n\ninterface ScriptProcessorNodeEventMap {\n    \"audioprocess\": AudioProcessingEvent;\n}\n\n/**\n * Allows the generation, processing, or analyzing of audio using JavaScript.\n * @deprecated As of the August 29 2014 Web Audio API spec publication, this feature has been marked as deprecated, and was replaced by AudioWorklet (see AudioWorkletNode).\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScriptProcessorNode)\n */\ninterface ScriptProcessorNode extends AudioNode {\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScriptProcessorNode/bufferSize)\n     */\n    readonly bufferSize: number;\n    /**\n     * @deprecated\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ScriptProcessorNode/audioprocess_event)\n     */\n    onaudioprocess: ((this: ScriptProcessorNode, ev: AudioProcessingEvent) => any) | null;\n    addEventListener<K extends keyof ScriptProcessorNodeEventMap>(type: K, listener: (this: ScriptProcessorNode, ev: ScriptProcessorNodeEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ScriptProcessorNodeEventMap>(type: K, listener: (this: ScriptProcessorNode, ev: ScriptProcessorNodeEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\n/** @deprecated */\ndeclare var ScriptProcessorNode: {\n    prototype: ScriptProcessorNode;\n    new(): ScriptProcessorNode;\n};\n\n/**\n * Inherits from Event, and represents the event object of an event sent on a document or worker when its content security policy is violated.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent)\n */\ninterface SecurityPolicyViolationEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/blockedURI) */\n    readonly blockedURI: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/columnNumber) */\n    readonly columnNumber: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/disposition) */\n    readonly disposition: SecurityPolicyViolationEventDisposition;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/documentURI) */\n    readonly documentURI: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/effectiveDirective) */\n    readonly effectiveDirective: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/lineNumber) */\n    readonly lineNumber: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/originalPolicy) */\n    readonly originalPolicy: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/referrer) */\n    readonly referrer: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/sample) */\n    readonly sample: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/sourceFile) */\n    readonly sourceFile: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/statusCode) */\n    readonly statusCode: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SecurityPolicyViolationEvent/violatedDirective) */\n    readonly violatedDirective: string;\n}\n\ndeclare var SecurityPolicyViolationEvent: {\n    prototype: SecurityPolicyViolationEvent;\n    new(type: string, eventInitDict?: SecurityPolicyViolationEventInit): SecurityPolicyViolationEvent;\n};\n\n/**\n * A Selection object represents the range of text selected by the user or the current position of the caret. To obtain a Selection object for examination or modification, call Window.getSelection().\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection)\n */\ninterface Selection {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/anchorNode) */\n    readonly anchorNode: Node | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/anchorOffset) */\n    readonly anchorOffset: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/direction) */\n    readonly direction: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/focusNode) */\n    readonly focusNode: Node | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/focusOffset) */\n    readonly focusOffset: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/isCollapsed) */\n    readonly isCollapsed: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/rangeCount) */\n    readonly rangeCount: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/type) */\n    readonly type: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/addRange) */\n    addRange(range: Range): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/collapse) */\n    collapse(node: Node | null, offset?: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/collapseToEnd) */\n    collapseToEnd(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/collapseToStart) */\n    collapseToStart(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/containsNode) */\n    containsNode(node: Node, allowPartialContainment?: boolean): boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/deleteFromDocument) */\n    deleteFromDocument(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/removeAllRanges) */\n    empty(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/extend) */\n    extend(node: Node, offset?: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/getRangeAt) */\n    getRangeAt(index: number): Range;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/modify) */\n    modify(alter?: string, direction?: string, granularity?: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/removeAllRanges) */\n    removeAllRanges(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/removeRange) */\n    removeRange(range: Range): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/selectAllChildren) */\n    selectAllChildren(node: Node): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/setBaseAndExtent) */\n    setBaseAndExtent(anchorNode: Node, anchorOffset: number, focusNode: Node, focusOffset: number): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Selection/collapse) */\n    setPosition(node: Node | null, offset?: number): void;\n    toString(): string;\n}\n\ndeclare var Selection: {\n    prototype: Selection;\n    new(): Selection;\n};\n\ninterface ServiceWorkerEventMap extends AbstractWorkerEventMap {\n    \"statechange\": Event;\n}\n\n/**\n * This ServiceWorker API interface provides a reference to a service worker. Multiple browsing contexts (e.g. pages, workers, etc.) can be associated with the same service worker, each through a unique ServiceWorker object.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorker)\n */\ninterface ServiceWorker extends EventTarget, AbstractWorker {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorker/statechange_event) */\n    onstatechange: ((this: ServiceWorker, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorker/scriptURL) */\n    readonly scriptURL: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorker/state) */\n    readonly state: ServiceWorkerState;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorker/postMessage) */\n    postMessage(message: any, transfer: Transferable[]): void;\n    postMessage(message: any, options?: StructuredSerializeOptions): void;\n    addEventListener<K extends keyof ServiceWorkerEventMap>(type: K, listener: (this: ServiceWorker, ev: ServiceWorkerEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ServiceWorkerEventMap>(type: K, listener: (this: ServiceWorker, ev: ServiceWorkerEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var ServiceWorker: {\n    prototype: ServiceWorker;\n    new(): ServiceWorker;\n};\n\ninterface ServiceWorkerContainerEventMap {\n    \"controllerchange\": Event;\n    \"message\": MessageEvent;\n    \"messageerror\": MessageEvent;\n}\n\n/**\n * The ServiceWorkerContainer interface of the ServiceWorker API provides an object representing the service worker as an overall unit in the network ecosystem, including facilities to register, unregister and update service workers, and access the state of service workers and their registrations.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer)\n */\ninterface ServiceWorkerContainer extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/controller) */\n    readonly controller: ServiceWorker | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/controllerchange_event) */\n    oncontrollerchange: ((this: ServiceWorkerContainer, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/message_event) */\n    onmessage: ((this: ServiceWorkerContainer, ev: MessageEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/messageerror_event) */\n    onmessageerror: ((this: ServiceWorkerContainer, ev: MessageEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/ready) */\n    readonly ready: Promise<ServiceWorkerRegistration>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/getRegistration) */\n    getRegistration(clientURL?: string | URL): Promise<ServiceWorkerRegistration | undefined>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/getRegistrations) */\n    getRegistrations(): Promise<ReadonlyArray<ServiceWorkerRegistration>>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/register) */\n    register(scriptURL: string | URL, options?: RegistrationOptions): Promise<ServiceWorkerRegistration>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerContainer/startMessages) */\n    startMessages(): void;\n    addEventListener<K extends keyof ServiceWorkerContainerEventMap>(type: K, listener: (this: ServiceWorkerContainer, ev: ServiceWorkerContainerEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ServiceWorkerContainerEventMap>(type: K, listener: (this: ServiceWorkerContainer, ev: ServiceWorkerContainerEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var ServiceWorkerContainer: {\n    prototype: ServiceWorkerContainer;\n    new(): ServiceWorkerContainer;\n};\n\ninterface ServiceWorkerRegistrationEventMap {\n    \"updatefound\": Event;\n}\n\n/**\n * This ServiceWorker API interface represents the service worker registration. You register a service worker to control one or more pages that share the same origin.\n * Available only in secure contexts.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration)\n */\ninterface ServiceWorkerRegistration extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/active) */\n    readonly active: ServiceWorker | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/installing) */\n    readonly installing: ServiceWorker | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/navigationPreload) */\n    readonly navigationPreload: NavigationPreloadManager;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/updatefound_event) */\n    onupdatefound: ((this: ServiceWorkerRegistration, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/pushManager) */\n    readonly pushManager: PushManager;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/scope) */\n    readonly scope: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/updateViaCache) */\n    readonly updateViaCache: ServiceWorkerUpdateViaCache;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/waiting) */\n    readonly waiting: ServiceWorker | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/getNotifications) */\n    getNotifications(filter?: GetNotificationOptions): Promise<Notification[]>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/showNotification) */\n    showNotification(title: string, options?: NotificationOptions): Promise<void>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/unregister) */\n    unregister(): Promise<boolean>;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ServiceWorkerRegistration/update) */\n    update(): Promise<void>;\n    addEventListener<K extends keyof ServiceWorkerRegistrationEventMap>(type: K, listener: (this: ServiceWorkerRegistration, ev: ServiceWorkerRegistrationEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ServiceWorkerRegistrationEventMap>(type: K, listener: (this: ServiceWorkerRegistration, ev: ServiceWorkerRegistrationEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var ServiceWorkerRegistration: {\n    prototype: ServiceWorkerRegistration;\n    new(): ServiceWorkerRegistration;\n};\n\ninterface ShadowRootEventMap {\n    \"slotchange\": Event;\n}\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot) */\ninterface ShadowRoot extends DocumentFragment, DocumentOrShadowRoot {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/clonable) */\n    readonly clonable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/delegatesFocus) */\n    readonly delegatesFocus: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/host) */\n    readonly host: Element;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/innerHTML) */\n    innerHTML: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/mode) */\n    readonly mode: ShadowRootMode;\n    onslotchange: ((this: ShadowRoot, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/serializable) */\n    readonly serializable: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/slotAssignment) */\n    readonly slotAssignment: SlotAssignmentMode;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/getHTML) */\n    getHTML(options?: GetHTMLOptions): string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/ShadowRoot/setHTMLUnsafe) */\n    setHTMLUnsafe(html: string): void;\n    /** Throws a \"NotSupportedError\" DOMException if context object is a shadow root. */\n    addEventListener<K extends keyof ShadowRootEventMap>(type: K, listener: (this: ShadowRoot, ev: ShadowRootEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof ShadowRootEventMap>(type: K, listener: (this: ShadowRoot, ev: ShadowRootEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var ShadowRoot: {\n    prototype: ShadowRoot;\n    new(): ShadowRoot;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SharedWorker) */\ninterface SharedWorker extends EventTarget, AbstractWorker {\n    /**\n     * Returns sharedWorker's MessagePort object which can be used to communicate with the global environment.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SharedWorker/port)\n     */\n    readonly port: MessagePort;\n    addEventListener<K extends keyof AbstractWorkerEventMap>(type: K, listener: (this: SharedWorker, ev: AbstractWorkerEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof AbstractWorkerEventMap>(type: K, listener: (this: SharedWorker, ev: AbstractWorkerEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SharedWorker: {\n    prototype: SharedWorker;\n    new(scriptURL: string | URL, options?: string | WorkerOptions): SharedWorker;\n};\n\ninterface Slottable {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Element/assignedSlot) */\n    readonly assignedSlot: HTMLSlotElement | null;\n}\n\ninterface SourceBufferEventMap {\n    \"abort\": Event;\n    \"error\": Event;\n    \"update\": Event;\n    \"updateend\": Event;\n    \"updatestart\": Event;\n}\n\n/**\n * A chunk of media to be passed into an HTMLMediaElement and played, via a MediaSource object. This can be made up of one or several media segments.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer)\n */\ninterface SourceBuffer extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/appendWindowEnd) */\n    appendWindowEnd: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/appendWindowStart) */\n    appendWindowStart: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/buffered) */\n    readonly buffered: TimeRanges;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/mode) */\n    mode: AppendMode;\n    onabort: ((this: SourceBuffer, ev: Event) => any) | null;\n    onerror: ((this: SourceBuffer, ev: Event) => any) | null;\n    onupdate: ((this: SourceBuffer, ev: Event) => any) | null;\n    onupdateend: ((this: SourceBuffer, ev: Event) => any) | null;\n    onupdatestart: ((this: SourceBuffer, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/timestampOffset) */\n    timestampOffset: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/updating) */\n    readonly updating: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/abort) */\n    abort(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/appendBuffer) */\n    appendBuffer(data: BufferSource): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/changeType) */\n    changeType(type: string): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBuffer/remove) */\n    remove(start: number, end: number): void;\n    addEventListener<K extends keyof SourceBufferEventMap>(type: K, listener: (this: SourceBuffer, ev: SourceBufferEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SourceBufferEventMap>(type: K, listener: (this: SourceBuffer, ev: SourceBufferEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SourceBuffer: {\n    prototype: SourceBuffer;\n    new(): SourceBuffer;\n};\n\ninterface SourceBufferListEventMap {\n    \"addsourcebuffer\": Event;\n    \"removesourcebuffer\": Event;\n}\n\n/**\n * A simple container list for multiple SourceBuffer objects.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBufferList)\n */\ninterface SourceBufferList extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SourceBufferList/length) */\n    readonly length: number;\n    onaddsourcebuffer: ((this: SourceBufferList, ev: Event) => any) | null;\n    onremovesourcebuffer: ((this: SourceBufferList, ev: Event) => any) | null;\n    addEventListener<K extends keyof SourceBufferListEventMap>(type: K, listener: (this: SourceBufferList, ev: SourceBufferListEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SourceBufferListEventMap>(type: K, listener: (this: SourceBufferList, ev: SourceBufferListEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n    [index: number]: SourceBuffer;\n}\n\ndeclare var SourceBufferList: {\n    prototype: SourceBufferList;\n    new(): SourceBufferList;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionAlternative) */\ninterface SpeechRecognitionAlternative {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionAlternative/confidence) */\n    readonly confidence: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionAlternative/transcript) */\n    readonly transcript: string;\n}\n\ndeclare var SpeechRecognitionAlternative: {\n    prototype: SpeechRecognitionAlternative;\n    new(): SpeechRecognitionAlternative;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResult) */\ninterface SpeechRecognitionResult {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResult/isFinal) */\n    readonly isFinal: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResult/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResult/item) */\n    item(index: number): SpeechRecognitionAlternative;\n    [index: number]: SpeechRecognitionAlternative;\n}\n\ndeclare var SpeechRecognitionResult: {\n    prototype: SpeechRecognitionResult;\n    new(): SpeechRecognitionResult;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResultList) */\ninterface SpeechRecognitionResultList {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResultList/length) */\n    readonly length: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechRecognitionResultList/item) */\n    item(index: number): SpeechRecognitionResult;\n    [index: number]: SpeechRecognitionResult;\n}\n\ndeclare var SpeechRecognitionResultList: {\n    prototype: SpeechRecognitionResultList;\n    new(): SpeechRecognitionResultList;\n};\n\ninterface SpeechSynthesisEventMap {\n    \"voiceschanged\": Event;\n}\n\n/**\n * This Web Speech API interface is the controller interface for the speech service; this can be used to retrieve information about the synthesis voices available on the device, start and pause speech, and other commands besides.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis)\n */\ninterface SpeechSynthesis extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/voiceschanged_event) */\n    onvoiceschanged: ((this: SpeechSynthesis, ev: Event) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/paused) */\n    readonly paused: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/pending) */\n    readonly pending: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/speaking) */\n    readonly speaking: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/cancel) */\n    cancel(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/getVoices) */\n    getVoices(): SpeechSynthesisVoice[];\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/pause) */\n    pause(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/resume) */\n    resume(): void;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesis/speak) */\n    speak(utterance: SpeechSynthesisUtterance): void;\n    addEventListener<K extends keyof SpeechSynthesisEventMap>(type: K, listener: (this: SpeechSynthesis, ev: SpeechSynthesisEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SpeechSynthesisEventMap>(type: K, listener: (this: SpeechSynthesis, ev: SpeechSynthesisEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SpeechSynthesis: {\n    prototype: SpeechSynthesis;\n    new(): SpeechSynthesis;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisErrorEvent) */\ninterface SpeechSynthesisErrorEvent extends SpeechSynthesisEvent {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisErrorEvent/error) */\n    readonly error: SpeechSynthesisErrorCode;\n}\n\ndeclare var SpeechSynthesisErrorEvent: {\n    prototype: SpeechSynthesisErrorEvent;\n    new(type: string, eventInitDict: SpeechSynthesisErrorEventInit): SpeechSynthesisErrorEvent;\n};\n\n/**\n * This Web Speech API interface contains information about the current state of SpeechSynthesisUtterance objects that have been processed in the speech service.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisEvent)\n */\ninterface SpeechSynthesisEvent extends Event {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisEvent/charIndex) */\n    readonly charIndex: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisEvent/charLength) */\n    readonly charLength: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisEvent/elapsedTime) */\n    readonly elapsedTime: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisEvent/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisEvent/utterance) */\n    readonly utterance: SpeechSynthesisUtterance;\n}\n\ndeclare var SpeechSynthesisEvent: {\n    prototype: SpeechSynthesisEvent;\n    new(type: string, eventInitDict: SpeechSynthesisEventInit): SpeechSynthesisEvent;\n};\n\ninterface SpeechSynthesisUtteranceEventMap {\n    \"boundary\": SpeechSynthesisEvent;\n    \"end\": SpeechSynthesisEvent;\n    \"error\": SpeechSynthesisErrorEvent;\n    \"mark\": SpeechSynthesisEvent;\n    \"pause\": SpeechSynthesisEvent;\n    \"resume\": SpeechSynthesisEvent;\n    \"start\": SpeechSynthesisEvent;\n}\n\n/**\n * This Web Speech API interface represents a speech request. It contains the content the speech service should read and information about how to read it (e.g. language, pitch and volume.)\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance)\n */\ninterface SpeechSynthesisUtterance extends EventTarget {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/lang) */\n    lang: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/boundary_event) */\n    onboundary: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/end_event) */\n    onend: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/error_event) */\n    onerror: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisErrorEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/mark_event) */\n    onmark: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/pause_event) */\n    onpause: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/resume_event) */\n    onresume: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/start_event) */\n    onstart: ((this: SpeechSynthesisUtterance, ev: SpeechSynthesisEvent) => any) | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/pitch) */\n    pitch: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/rate) */\n    rate: number;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/text) */\n    text: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/voice) */\n    voice: SpeechSynthesisVoice | null;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisUtterance/volume) */\n    volume: number;\n    addEventListener<K extends keyof SpeechSynthesisUtteranceEventMap>(type: K, listener: (this: SpeechSynthesisUtterance, ev: SpeechSynthesisUtteranceEventMap[K]) => any, options?: boolean | AddEventListenerOptions): void;\n    addEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | AddEventListenerOptions): void;\n    removeEventListener<K extends keyof SpeechSynthesisUtteranceEventMap>(type: K, listener: (this: SpeechSynthesisUtterance, ev: SpeechSynthesisUtteranceEventMap[K]) => any, options?: boolean | EventListenerOptions): void;\n    removeEventListener(type: string, listener: EventListenerOrEventListenerObject, options?: boolean | EventListenerOptions): void;\n}\n\ndeclare var SpeechSynthesisUtterance: {\n    prototype: SpeechSynthesisUtterance;\n    new(text?: string): SpeechSynthesisUtterance;\n};\n\n/**\n * This Web Speech API interface represents a voice that the system supports. Every SpeechSynthesisVoice has its own relative speech service including information about language, name and URI.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisVoice)\n */\ninterface SpeechSynthesisVoice {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisVoice/default) */\n    readonly default: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisVoice/lang) */\n    readonly lang: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisVoice/localService) */\n    readonly localService: boolean;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisVoice/name) */\n    readonly name: string;\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/SpeechSynthesisVoice/voiceURI) */\n    readonly voiceURI: string;\n}\n\ndeclare var SpeechSynthesisVoice: {\n    prototype: SpeechSynthesisVoice;\n    new(): SpeechSynthesisVoice;\n};\n\n/** [MDN Reference](https://developer.mozilla.org/docs/Web/API/StaticRange) */\ninterface StaticRange extends AbstractRange {\n}\n\ndeclare var StaticRange: {\n    prototype: StaticRange;\n    new(init: StaticRangeInit): StaticRange;\n};\n\n/**\n * The pan property takes a unitless value between -1 (full left pan) and 1 (full right pan). This interface was introduced as a much simpler way to apply a simple panning effect than having to use a full PannerNode.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/StereoPannerNode)\n */\ninterface StereoPannerNode extends AudioNode {\n    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/StereoPannerNode/pan) */\n    readonly pan: AudioParam;\n}\n\ndeclare var StereoPannerNode: {\n    prototype: StereoPannerNode;\n    new(context: BaseAudioContext, options?: StereoPannerOptions): StereoPannerNode;\n};\n\n/**\n * This Web Storage API interface provides access to a particular domain's session or local storage. It allows, for example, the addition, modification, or deletion of stored data items.\n *\n * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage)\n */\ninterface Storage {\n    /**\n     * Returns the number of key/value pairs.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage/length)\n     */\n    readonly length: number;\n    /**\n     * Removes all key/value pairs, if there are any.\n     *\n     * Dispatches a storage event on Window objects holding an equivalent Storage object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage/clear)\n     */\n    clear(): void;\n    /**\n     * Returns the current value associated with the given key, or null if the given key does not exist.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage/getItem)\n     */\n    getItem(key: string): string | null;\n    /**\n     * Returns the name of the nth key, or null if n is greater than or equal to the number of key/value pairs.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage/key)\n     */\n    key(index: number): string | null;\n    /**\n     * Removes the key/value pair with the given key, if a key/value pair with the given key exists.\n     *\n     * Dispatches a storage event on Window objects holding an equivalent Storage object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage/removeItem)\n     */\n    removeItem(key: string): void;\n    /**\n     * Sets the value of the pair identified by key to value, creating a new key/value pair if none existed for key previously.\n     *\n     * Throws a \"QuotaExceededError\" DOMException exception if the new value couldn't be set. (Setting could fail if, e.g., the user has disabled storage for the site, or if the quota has been exceeded.)\n     *\n     * Dispatches a storage event on Window objects holding an equivalent Storage object.\n     *\n     * [MDN Reference](https://developer.mozilla.org/docs/Web/API/Storage/setItem)\n     */\n    setItem(key: string, value: string): void;\n    [name: string]: any;\n}\n\ndeclare var Storage: {\n    prototype: Storage;\n    new(): Storage;\n};\n\n/**\n * A StorageEvent is sent to a wind