/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { setActiveConsumer } from '@angular/core/primitives/signals';
import { findMatchingDehydratedView } from '../../hydration/views';
import { assertDefined } from '../../util/assert';
import { performanceMarkFeature } from '../../util/performance';
import { assertLContainer, assertLView, assertTNode } from '../assert';
import { bindingUpdated } from '../bindings';
import { CONTAINER_HEADER_OFFSET } from '../interfaces/container';
import { CONTEXT, DECLARATION_COMPONENT_VIEW, HEADER_OFFSET, HYDRATION, TVIEW } from '../interfaces/view';
import { LiveCollection, reconcile } from '../list_reconciliation';
import { destroyLView, detachView } from '../node_manipulation';
import { getLView, getSelectedIndex, nextBindingIndex } from '../state';
import { getTNode } from '../util/view_utils';
import { addLViewToLContainer, createAndRenderEmbeddedLView, getLViewFromLContainer, removeLViewFromLContainer, shouldAddViewToDom } from '../view_manipulation';
import { ɵɵtemplate } from './template';
/**
 * The conditional instruction represents the basic building block on the runtime side to support
 * built-in "if" and "switch". On the high level this instruction is responsible for adding and
 * removing views selected by a conditional expression.
 *
 * @param containerIndex index of a container in a host view (indexed from HEADER_OFFSET) where
 *     conditional views should be inserted.
 * @param matchingTemplateIndex index of a template TNode representing a conditional view to be
 *     inserted; -1 represents a special case when there is no view to insert.
 * @codeGenApi
 */
export function ɵɵconditional(containerIndex, matchingTemplateIndex, value) {
    performanceMarkFeature('NgControlFlow');
    const hostLView = getLView();
    const bindingIndex = nextBindingIndex();
    const lContainer = getLContainer(hostLView, HEADER_OFFSET + containerIndex);
    const viewInContainerIdx = 0;
    if (bindingUpdated(hostLView, bindingIndex, matchingTemplateIndex)) {
        const prevConsumer = setActiveConsumer(null);
        try {
            // The index of the view to show changed - remove the previously displayed one
            // (it is a noop if there are no active views in a container).
            removeLViewFromLContainer(lContainer, viewInContainerIdx);
            // Index -1 is a special case where none of the conditions evaluates to
            // a truthy value and as the consequence we've got no view to show.
            if (matchingTemplateIndex !== -1) {
                const templateTNode = getExistingTNode(hostLView[TVIEW], HEADER_OFFSET + matchingTemplateIndex);
                const dehydratedView = findMatchingDehydratedView(lContainer, templateTNode.tView.ssrId);
                const embeddedLView = createAndRenderEmbeddedLView(hostLView, templateTNode, value, { dehydratedView });
                addLViewToLContainer(lContainer, embeddedLView, viewInContainerIdx, shouldAddViewToDom(templateTNode, dehydratedView));
            }
        }
        finally {
            setActiveConsumer(prevConsumer);
        }
    }
    else {
        // We might keep displaying the same template but the actual value of the expression could have
        // changed - re-bind in context.
        const lView = getLViewFromLContainer(lContainer, viewInContainerIdx);
        if (lView !== undefined) {
            lView[CONTEXT] = value;
        }
    }
}
export class RepeaterContext {
    constructor(lContainer, $implicit, $index) {
        this.lContainer = lContainer;
        this.$implicit = $implicit;
        this.$index = $index;
    }
    get $count() {
        return this.lContainer.length - CONTAINER_HEADER_OFFSET;
    }
}
/**
 * A built-in trackBy function used for situations where users specified collection index as a
 * tracking expression. Having this function body in the runtime avoids unnecessary code generation.
 *
 * @param index
 * @returns
 */
export function ɵɵrepeaterTrackByIndex(index) {
    return index;
}
/**
 * A built-in trackBy function used for situations where users specified collection item reference
 * as a tracking expression. Having this function body in the runtime avoids unnecessary code
 * generation.
 *
 * @param index
 * @returns
 */
export function ɵɵrepeaterTrackByIdentity(_, value) {
    return value;
}
class RepeaterMetadata {
    constructor(hasEmptyBlock, trackByFn, liveCollection) {
        this.hasEmptyBlock = hasEmptyBlock;
        this.trackByFn = trackByFn;
        this.liveCollection = liveCollection;
    }
}
/**
 * The repeaterCreate instruction runs in the creation part of the template pass and initializes
 * internal data structures required by the update pass of the built-in repeater logic. Repeater
 * metadata are allocated in the data part of LView with the following layout:
 * - LView[HEADER_OFFSET + index] - metadata
 * - LView[HEADER_OFFSET + index + 1] - reference to a template function rendering an item
 * - LView[HEADER_OFFSET + index + 2] - optional reference to a template function rendering an empty
 * block
 *
 * @param index Index at which to store the metadata of the repeater.
 * @param templateFn Reference to the template of the main repeater block.
 * @param decls The number of nodes, local refs, and pipes for the main block.
 * @param vars The number of bindings for the main block.
 * @param tagName The name of the container element, if applicable
 * @param attrsIndex Index of template attributes in the `consts` array.
 * @param trackByFn Reference to the tracking function.
 * @param trackByUsesComponentInstance Whether the tracking function has any references to the
 *  component instance. If it doesn't, we can avoid rebinding it.
 * @param emptyTemplateFn Reference to the template function of the empty block.
 * @param emptyDecls The number of nodes, local refs, and pipes for the empty block.
 * @param emptyVars The number of bindings for the empty block.
 * @param emptyTagName The name of the empty block container element, if applicable
 * @param emptyAttrsIndex Index of the empty block template attributes in the `consts` array.
 *
 * @codeGenApi
 */
export function ɵɵrepeaterCreate(index, templateFn, decls, vars, tagName, attrsIndex, trackByFn, trackByUsesComponentInstance, emptyTemplateFn, emptyDecls, emptyVars, emptyTagName, emptyAttrsIndex) {
    performanceMarkFeature('NgControlFlow');
    const hasEmptyBlock = emptyTemplateFn !== undefined;
    const hostLView = getLView();
    const boundTrackBy = trackByUsesComponentInstance ?
        // We only want to bind when necessary, because it produces a
        // new function. For pure functions it's not necessary.
        trackByFn.bind(hostLView[DECLARATION_COMPONENT_VIEW][CONTEXT]) :
        trackByFn;
    const metadata = new RepeaterMetadata(hasEmptyBlock, boundTrackBy);
    hostLView[HEADER_OFFSET + index] = metadata;
    ɵɵtemplate(index + 1, templateFn, decls, vars, tagName, attrsIndex);
    if (hasEmptyBlock) {
        ngDevMode &&
            assertDefined(emptyDecls, 'Missing number of declarations for the empty repeater block.');
        ngDevMode &&
            assertDefined(emptyVars, 'Missing number of bindings for the empty repeater block.');
        ɵɵtemplate(index + 2, emptyTemplateFn, emptyDecls, emptyVars, emptyTagName, emptyAttrsIndex);
    }
}
class LiveCollectionLContainerImpl extends LiveCollection {
    constructor(lContainer, hostLView, templateTNode) {
        super();
        this.lContainer = lContainer;
        this.hostLView = hostLView;
        this.templateTNode = templateTNode;
        /**
         Property indicating if indexes in the repeater context need to be updated following the live
         collection changes. Index updates are necessary if and only if views are inserted / removed in
         the middle of LContainer. Adds and removals at the end don't require index updates.
       */
        this.needsIndexUpdate = false;
    }
    get length() {
        return this.lContainer.length - CONTAINER_HEADER_OFFSET;
    }
    at(index) {
        return this.getLView(index)[CONTEXT].$implicit;
    }
    attach(index, lView) {
        const dehydratedView = lView[HYDRATION];
        this.needsIndexUpdate ||= index !== this.length;
        addLViewToLContainer(this.lContainer, lView, index, shouldAddViewToDom(this.templateTNode, dehydratedView));
    }
    detach(index) {
        this.needsIndexUpdate ||= index !== this.length - 1;
        return detachExistingView(this.lContainer, index);
    }
    create(index, value) {
        const dehydratedView = findMatchingDehydratedView(this.lContainer, this.templateTNode.tView.ssrId);
        const embeddedLView = createAndRenderEmbeddedLView(this.hostLView, this.templateTNode, new RepeaterContext(this.lContainer, value, index), { dehydratedView });
        return embeddedLView;
    }
    destroy(lView) {
        destroyLView(lView[TVIEW], lView);
    }
    updateValue(index, value) {
        this.getLView(index)[CONTEXT].$implicit = value;
    }
    reset() {
        this.needsIndexUpdate = false;
    }
    updateIndexes() {
        if (this.needsIndexUpdate) {
            for (let i = 0; i < this.length; i++) {
                this.getLView(i)[CONTEXT].$index = i;
            }
        }
    }
    getLView(index) {
        return getExistingLViewFromLContainer(this.lContainer, index);
    }
}
/**
 * The repeater instruction does update-time diffing of a provided collection (against the
 * collection seen previously) and maps changes in the collection to views structure (by adding,
 * removing or moving views as needed).
 * @param collection - the collection instance to be checked for changes
 * @codeGenApi
 */
export function ɵɵrepeater(collection) {
    const prevConsumer = setActiveConsumer(null);
    const metadataSlotIdx = getSelectedIndex();
    try {
        const hostLView = getLView();
        const hostTView = hostLView[TVIEW];
        const metadata = hostLView[metadataSlotIdx];
        if (metadata.liveCollection === undefined) {
            const containerIndex = metadataSlotIdx + 1;
            const lContainer = getLContainer(hostLView, containerIndex);
            const itemTemplateTNode = getExistingTNode(hostTView, containerIndex);
            metadata.liveCollection =
                new LiveCollectionLContainerImpl(lContainer, hostLView, itemTemplateTNode);
        }
        else {
            metadata.liveCollection.reset();
        }
        const liveCollection = metadata.liveCollection;
        reconcile(liveCollection, collection, metadata.trackByFn);
        // moves in the container might caused context's index to get out of order, re-adjust if needed
        liveCollection.updateIndexes();
        // handle empty blocks
        if (metadata.hasEmptyBlock) {
            const bindingIndex = nextBindingIndex();
            const isCollectionEmpty = liveCollection.length === 0;
            if (bindingUpdated(hostLView, bindingIndex, isCollectionEmpty)) {
                const emptyTemplateIndex = metadataSlotIdx + 2;
                const lContainerForEmpty = getLContainer(hostLView, emptyTemplateIndex);
                if (isCollectionEmpty) {
                    const emptyTemplateTNode = getExistingTNode(hostTView, emptyTemplateIndex);
                    const dehydratedView = findMatchingDehydratedView(lContainerForEmpty, emptyTemplateTNode.tView.ssrId);
                    const embeddedLView = createAndRenderEmbeddedLView(hostLView, emptyTemplateTNode, undefined, { dehydratedView });
                    addLViewToLContainer(lContainerForEmpty, embeddedLView, 0, shouldAddViewToDom(emptyTemplateTNode, dehydratedView));
                }
                else {
                    removeLViewFromLContainer(lContainerForEmpty, 0);
                }
            }
        }
    }
    finally {
        setActiveConsumer(prevConsumer);
    }
}
function getLContainer(lView, index) {
    const lContainer = lView[index];
    ngDevMode && assertLContainer(lContainer);
    return lContainer;
}
function detachExistingView(lContainer, index) {
    const existingLView = detachView(lContainer, index);
    ngDevMode && assertLView(existingLView);
    return existingLView;
}
function getExistingLViewFromLContainer(lContainer, index) {
    const existingLView = getLViewFromLContainer(lContainer, index);
    ngDevMode && assertLView(existingLView);
    return existingLView;
}
function getExistingTNode(tView, index) {
    const tNode = getTNode(tView, index);
    ngDevMode && assertTNode(tNode);
    return tNode;
}
//# sourceMappingURL=data:application/json;base64,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