/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { producerAccessed, SIGNAL } from '@angular/core/primitives/signals';
import { RuntimeError } from '../../errors';
import { INPUT_SIGNAL_NODE, REQUIRED_UNSET_VALUE } from './input_signal_node';
export const ɵINPUT_SIGNAL_BRAND_READ_TYPE = /* @__PURE__ */ Symbol();
export const ɵINPUT_SIGNAL_BRAND_WRITE_TYPE = /* @__PURE__ */ Symbol();
/**
 * Creates an input signal.
 *
 * @param initialValue The initial value.
 *   Can be set to {@link REQUIRED_UNSET_VALUE} for required inputs.
 * @param options Additional options for the input. e.g. a transform, or an alias.
 */
export function createInputSignal(initialValue, options) {
    const node = Object.create(INPUT_SIGNAL_NODE);
    node.value = initialValue;
    // Perf note: Always set `transformFn` here to ensure that `node` always
    // has the same v8 class shape, allowing monomorphic reads on input signals.
    node.transformFn = options?.transform;
    function inputValueFn() {
        // Record that someone looked at this signal.
        producerAccessed(node);
        if (node.value === REQUIRED_UNSET_VALUE) {
            throw new RuntimeError(-950 /* RuntimeErrorCode.REQUIRED_INPUT_NO_VALUE */, ngDevMode && 'Input is required but no value is available yet.');
        }
        return node.value;
    }
    inputValueFn[SIGNAL] = node;
    if (ngDevMode) {
        inputValueFn.toString = () => `[Input Signal: ${inputValueFn()}]`;
    }
    return inputValueFn;
}
//# sourceMappingURL=data:application/json;base64,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