/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { CONTAINER_HEADER_OFFSET, DEHYDRATED_VIEWS } from '../render3/interfaces/container';
import { isLContainer, isLView } from '../render3/interfaces/type_checks';
import { HEADER_OFFSET, HOST, PARENT, RENDERER, TVIEW } from '../render3/interfaces/view';
import { nativeRemoveNode } from '../render3/node_manipulation';
import { EMPTY_ARRAY } from '../util/empty';
import { validateSiblingNodeExists } from './error_handling';
import { NUM_ROOT_NODES } from './interfaces';
import { getLNodeForHydration } from './utils';
/**
 * Removes all dehydrated views from a given LContainer:
 * both in internal data structure, as well as removing
 * corresponding DOM nodes that belong to that dehydrated view.
 */
export function removeDehydratedViews(lContainer) {
    const views = lContainer[DEHYDRATED_VIEWS] ?? [];
    const parentLView = lContainer[PARENT];
    const renderer = parentLView[RENDERER];
    for (const view of views) {
        removeDehydratedView(view, renderer);
        ngDevMode && ngDevMode.dehydratedViewsRemoved++;
    }
    // Reset the value to an empty array to indicate that no
    // further processing of dehydrated views is needed for
    // this view container (i.e. do not trigger the lookup process
    // once again in case a `ViewContainerRef` is created later).
    lContainer[DEHYDRATED_VIEWS] = EMPTY_ARRAY;
}
/**
 * Helper function to remove all nodes from a dehydrated view.
 */
function removeDehydratedView(dehydratedView, renderer) {
    let nodesRemoved = 0;
    let currentRNode = dehydratedView.firstChild;
    if (currentRNode) {
        const numNodes = dehydratedView.data[NUM_ROOT_NODES];
        while (nodesRemoved < numNodes) {
            ngDevMode && validateSiblingNodeExists(currentRNode);
            const nextSibling = currentRNode.nextSibling;
            nativeRemoveNode(renderer, currentRNode, false);
            currentRNode = nextSibling;
            nodesRemoved++;
        }
    }
}
/**
 * Walks over all views within this LContainer invokes dehydrated views
 * cleanup function for each one.
 */
function cleanupLContainer(lContainer) {
    removeDehydratedViews(lContainer);
    for (let i = CONTAINER_HEADER_OFFSET; i < lContainer.length; i++) {
        cleanupLView(lContainer[i]);
    }
}
/**
 * Walks over `LContainer`s and components registered within
 * this LView and invokes dehydrated views cleanup function for each one.
 */
function cleanupLView(lView) {
    const tView = lView[TVIEW];
    for (let i = HEADER_OFFSET; i < tView.bindingStartIndex; i++) {
        if (isLContainer(lView[i])) {
            const lContainer = lView[i];
            cleanupLContainer(lContainer);
        }
        else if (isLView(lView[i])) {
            // This is a component, enter the `cleanupLView` recursively.
            cleanupLView(lView[i]);
        }
    }
}
/**
 * Walks over all views registered within the ApplicationRef and removes
 * all dehydrated views from all `LContainer`s along the way.
 */
export function cleanupDehydratedViews(appRef) {
    const viewRefs = appRef._views;
    for (const viewRef of viewRefs) {
        const lNode = getLNodeForHydration(viewRef);
        // An `lView` might be `null` if a `ViewRef` represents
        // an embedded view (not a component view).
        if (lNode !== null && lNode[HOST] !== null) {
            if (isLView(lNode)) {
                cleanupLView(lNode);
            }
            else {
                // Cleanup in the root component view
                const componentLView = lNode[HOST];
                cleanupLView(componentLView);
                // Cleanup in all views within this view container
                cleanupLContainer(lNode);
            }
            ngDevMode && ngDevMode.dehydratedViewsCleanupRuns++;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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