/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { createSignal, SIGNAL, signalSetFn, signalUpdateFn } from '@angular/core/primitives/signals';
import { isSignal } from './api';
/** Symbol used distinguish `WritableSignal` from other non-writable signals and functions. */
const WRITABLE_SIGNAL = /* @__PURE__ */ Symbol('WRITABLE_SIGNAL');
/**
 * Utility function used during template type checking to extract the value from a `WritableSignal`.
 * @codeGenApi
 */
export function ɵunwrapWritableSignal(value) {
    // Note: needs to be kept in sync with the copy in `fake_core/index.ts`.
    // Note: the function uses `WRITABLE_SIGNAL` as a brand instead of `WritableSignal<T>`,
    // because the latter incorrectly unwraps non-signal getter functions.
    return null;
}
/**
 * Create a `Signal` that can be set or updated directly.
 */
export function signal(initialValue, options) {
    const signalFn = createSignal(initialValue);
    const node = signalFn[SIGNAL];
    if (options?.equal) {
        node.equal = options.equal;
    }
    signalFn.set = (newValue) => signalSetFn(node, newValue);
    signalFn.update = (updateFn) => signalUpdateFn(node, updateFn);
    signalFn.asReadonly = signalAsReadonlyFn.bind(signalFn);
    if (ngDevMode) {
        signalFn.toString = () => `[Signal: ${signalFn()}]`;
    }
    return signalFn;
}
function signalAsReadonlyFn() {
    const node = this[SIGNAL];
    if (node.readonlyFn === undefined) {
        const readonlyFn = () => this();
        readonlyFn[SIGNAL] = node;
        node.readonlyFn = readonlyFn;
    }
    return node.readonlyFn;
}
/**
 * Checks if the given `value` is a writeable signal.
 */
export function isWritableSignal(value) {
    return isSignal(value) && typeof value.set === 'function';
}
//# sourceMappingURL=data:application/json;base64,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