/*
 * Decompiled with CFR 0.152.
 */
package org.openimaj.math.geometry.shape;

import Jama.Matrix;
import java.util.ArrayList;
import java.util.List;
import org.openimaj.citation.annotation.Reference;
import org.openimaj.citation.annotation.ReferenceType;
import org.openimaj.citation.annotation.References;
import org.openimaj.math.geometry.point.Point2d;
import org.openimaj.math.geometry.point.Point2dImpl;
import org.openimaj.math.geometry.shape.PointList;
import org.openimaj.math.geometry.shape.algorithm.GeneralisedProcrustesAnalysis;
import org.openimaj.math.geometry.shape.algorithm.ProcrustesAnalysis;
import org.openimaj.math.matrix.algorithm.pca.PrincipalComponentAnalysis;
import org.openimaj.math.matrix.algorithm.pca.SvdPrincipalComponentAnalysis;
import org.openimaj.util.pair.IndependentPair;

@References(references={@Reference(author={"Cootes, T. F.", "Taylor, C. J."}, title="Statistical Models of Appearance for Computer Vision", type=ReferenceType.Unpublished, month="October", year="2001", url="http://isbe.man.ac.uk/~bim/Models/app_model.ps.gz"), @Reference(type=ReferenceType.Inproceedings, author={"Cj. Taylor", "D. H. Cooper", "J. Graham"}, title="Training models of shape from sets of examples", year="1992", booktitle="Proc. BMVC92, Springer-Verlag", pages={"9", "", "18"})})
public class PointDistributionModel {
    protected Constraint constraint;
    protected PrincipalComponentAnalysis pc;
    protected PointList mean;
    protected int numComponents;
    protected int maxIter = 100;

    public PointDistributionModel(List<PointList> data) {
        this(new NullConstraint(), data);
    }

    public PointDistributionModel(Constraint constraint, List<PointList> data) {
        this.constraint = constraint;
        this.mean = GeneralisedProcrustesAnalysis.alignPoints(data, 5.0f, 10);
        Matrix m = this.buildDataMatrix(data);
        this.pc = new SvdPrincipalComponentAnalysis();
        this.pc.learnBasis(m);
        this.numComponents = this.pc.getEigenValues().length;
    }

    private Matrix buildDataMatrix(PointList data) {
        ArrayList<PointList> pls = new ArrayList<PointList>(1);
        pls.add(data);
        return this.buildDataMatrix(pls);
    }

    private Matrix buildDataMatrix(List<PointList> data) {
        int nData = data.size();
        int nPoints = data.get(0).size();
        Matrix m = new Matrix(nData, nPoints * 2);
        double[][] mData = m.getArray();
        for (int i = 0; i < nData; ++i) {
            PointList pts = data.get(i);
            int j = 0;
            for (int k = 0; k < nPoints; ++k) {
                Point2d pt = pts.points.get(k);
                mData[i][j] = pt.getX();
                mData[i][j + 1] = pt.getY();
                j += 2;
            }
        }
        return m;
    }

    public PointList getMean() {
        return this.mean;
    }

    public void setNumComponents(int n) {
        this.pc.selectSubset(n);
        this.numComponents = this.pc.getEigenValues().length;
    }

    public void setNumComponents(PrincipalComponentAnalysis.ComponentSelector selector) {
        this.pc.selectSubset(selector);
        this.numComponents = this.pc.getEigenValues().length;
    }

    public PointList generateNewShape(double[] scaling) {
        PointList newShape = new PointList(new Point2d[0]);
        double[] pts = this.pc.generate(this.constraint.apply(scaling, this.pc.getEigenValues()));
        for (int i = 0; i < pts.length; i += 2) {
            float x = (float)pts[i];
            float y = (float)pts[i + 1];
            newShape.points.add(new Point2dImpl(x, y));
        }
        return newShape;
    }

    public double[] getStandardDeviations(double multiplier) {
        double[] rngs = this.pc.getStandardDeviations();
        for (int i = 0; i < rngs.length; ++i) {
            rngs[i] = rngs[i] * multiplier;
        }
        return rngs;
    }

    public IndependentPair<Matrix, double[]> fitModel(PointList observed) {
        double[] model = new double[this.numComponents];
        double delta = 1.0;
        Matrix pose = null;
        ProcrustesAnalysis pa = new ProcrustesAnalysis(observed);
        int count = 0;
        while (delta > 1.0E-6 && count++ < this.maxIter) {
            PointList instance = this.generateNewShape(model);
            pose = pa.align(instance);
            PointList projected = observed.transform(pose.inverse());
            Matrix y = this.buildDataMatrix(projected);
            Matrix xbar = new Matrix((double[][])new double[][]{this.pc.getMean()});
            double[] newModel = y.minus(xbar).times(this.pc.getBasis()).getArray()[0];
            newModel = this.constraint.apply(newModel, this.pc.getEigenValues());
            delta = 0.0;
            for (int i = 0; i < newModel.length; ++i) {
                delta += (newModel[i] - model[i]) * (newModel[i] - model[i]);
            }
            delta = Math.sqrt(delta);
            model = newModel;
        }
        return new IndependentPair(pose, (Object)model);
    }

    public static class EllipsoidConstraint
    implements Constraint {
        double dmax;

        public EllipsoidConstraint(double dmax) {
            this.dmax = dmax;
        }

        @Override
        public double[] apply(double[] in, double[] lamda) {
            double dmsq = 0.0;
            for (int i = 0; i < in.length; ++i) {
                dmsq += in[i] * in[i] / lamda[i];
            }
            if (dmsq < this.dmax * this.dmax) {
                return in;
            }
            double sc = this.dmax / Math.sqrt(dmsq);
            double[] out = new double[in.length];
            for (int i = 0; i < in.length; ++i) {
                out[i] = in[i] * sc;
            }
            return out;
        }
    }

    public static class BoxConstraint
    implements Constraint {
        double multiplier;

        public BoxConstraint(double multiplier) {
            this.multiplier = multiplier;
        }

        @Override
        public double[] apply(double[] in, double[] lamda) {
            double[] out = new double[in.length];
            for (int i = 0; i < in.length; ++i) {
                double w = this.multiplier * Math.sqrt(lamda[i]);
                out[i] = in[i] > w ? w : (in[i] < -w ? -w : in[i]);
            }
            return out;
        }
    }

    public static class NullConstraint
    implements Constraint {
        @Override
        public double[] apply(double[] in, double[] lamda) {
            return in;
        }
    }

    public static interface Constraint {
        public double[] apply(double[] var1, double[] var2);
    }
}

