/*
 * Copyright (c) 2019 OpenFTC Team
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package org.openftc.easyopencv;

import android.content.Context;
import android.support.annotation.IdRes;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;

import com.qualcomm.robotcore.eventloop.opmode.AnnotatedOpModeManager;
import com.qualcomm.robotcore.eventloop.opmode.OpMode;
import com.qualcomm.robotcore.eventloop.opmode.OpModeRegistrar;

import org.firstinspires.ftc.robotcore.external.ClassFactory;
import org.firstinspires.ftc.robotcore.external.hardware.camera.WebcamName;
import org.firstinspires.ftc.robotcore.internal.camera.delegating.SwitchableCameraName;
import org.firstinspires.ftc.robotcore.internal.system.AppUtil;

import java.util.ArrayList;
import java.util.concurrent.CountDownLatch;

class OpenCvCameraFactoryImpl extends OpenCvCameraFactory
{
    static void init()
    {
        OpenCvCameraFactory.theInstance = new OpenCvCameraFactoryImpl();
    }

    @OpModeRegistrar
    public static void initOnSdkBoot(Context context, AnnotatedOpModeManager manager)
    {
        init();
    }

    @Override
    public OpenCvInternalCamera createInternalCamera(OpenCvInternalCamera.CameraDirection direction)
    {
        return new OpenCvInternalCameraImpl(direction);
    }

    @Override
    public OpenCvInternalCamera createInternalCamera(OpenCvInternalCamera.CameraDirection direction, int containerId)
    {
        return new OpenCvInternalCameraImpl(direction, containerId);
    }

    @Override
    public OpenCvInternalCamera2 createInternalCamera2(OpenCvInternalCamera2.CameraDirection direction)
    {
        return new OpenCvInternalCamera2Impl(direction);
    }

    @Override
    public OpenCvInternalCamera2 createInternalCamera2(OpenCvInternalCamera2.CameraDirection direction, int containerId)
    {
        return new OpenCvInternalCamera2Impl(direction, containerId);
    }

    @Override
    public OpenCvWebcam createWebcam(WebcamName webcamName)
    {
        return new OpenCvWebcamImpl(webcamName);
    }

    @Override
    public OpenCvWebcam createWebcam(WebcamName webcamName, @IdRes int viewportContainerId)
    {
        return new OpenCvWebcamImpl(webcamName, viewportContainerId);
    }

    @Override
    public OpenCvSwitchableWebcam createSwitchableWebcam(WebcamName... names)
    {
        SwitchableCameraName cameraName = ClassFactory.getInstance().getCameraManager().nameForSwitchableCamera(names);

        return new OpenCvSwitchableWebcamImpl(cameraName);
    }

    @Override
    public OpenCvSwitchableWebcam createSwitchableWebcam(int viewportContainerId, WebcamName... names)
    {
        SwitchableCameraName cameraName = ClassFactory.getInstance().getCameraManager().nameForSwitchableCamera(names);

        return new OpenCvSwitchableWebcamImpl(cameraName, viewportContainerId);
    }

    @Override
    public int[] splitLayoutForMultipleViewports(final int containerId, final int numViewports, final ViewportSplitMethod viewportSplitMethod)
    {
        if(numViewports < 2)
        {
            throw new IllegalArgumentException("Layout requested to be split for <2 viewports!");
        }

        final int[] ids = new int[numViewports];
        final ArrayList<LinearLayout> layoutArrayList = new ArrayList<>(numViewports);

        final CountDownLatch latch = new CountDownLatch(1);

        //We do the viewport creation on the UI thread, but if there's an exception then
        //we need to catch it and rethrow it on the OpMode thread
        final RuntimeException[] exToRethrowOnOpModeThread = {null};

        AppUtil.getInstance().getActivity().runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    final LinearLayout containerLayout = (LinearLayout) AppUtil.getInstance().getActivity().findViewById(containerId);

                    if(containerLayout == null)
                    {
                        throw new OpenCvCameraException("Viewport container specified by user does not exist!");
                    }
                    else if(containerLayout.getChildCount() != 0)
                    {
                        throw new OpenCvCameraException("Viewport container specified by user is not empty!");
                    }

                    containerLayout.setVisibility(View.VISIBLE);

                    if(viewportSplitMethod == null)
                    {
                        throw new IllegalArgumentException("Viewport split method cannot be null!");
                    }
                    else if(viewportSplitMethod == ViewportSplitMethod.VERTICALLY)
                    {
                        containerLayout.setOrientation(LinearLayout.VERTICAL);
                    }
                    else
                    {
                        containerLayout.setOrientation(LinearLayout.HORIZONTAL);
                    }

                    for(int i = 0; i < numViewports; i++)
                    {
                        LinearLayout linearLayout = new LinearLayout(AppUtil.getInstance().getActivity());
                        LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
                        params.weight = 1;
                        linearLayout.setLayoutParams(params);
                        linearLayout.setId(View.generateViewId());
                        ids[i] = linearLayout.getId();
                        layoutArrayList.add(linearLayout);
                        containerLayout.addView(linearLayout);
                    }

                    LIFO_OpModeCallbackDelegate.getInstance().add(new LIFO_OpModeCallbackDelegate.OnOpModeStoppedListener()
                    {
                        @Override
                        public void onOpModePostStop(OpMode opMode)
                        {
                            AppUtil.getInstance().getActivity().runOnUiThread(new Runnable()
                            {
                                @Override
                                public void run()
                                {
                                    for(LinearLayout layout : layoutArrayList)
                                    {
                                        containerLayout.removeView(layout);
                                    }
                                    containerLayout.setVisibility(View.GONE);
                                    containerLayout.setOrientation(LinearLayout.VERTICAL);
                                }
                            });

                        }
                    });

                    latch.countDown();
                }
                catch (RuntimeException e)
                {
                    exToRethrowOnOpModeThread[0] = e;
                }

            }
        });

        if(exToRethrowOnOpModeThread[0] != null)
        {
            throw exToRethrowOnOpModeThread[0];
        }

        try
        {
            latch.await();

            return ids;
        }
        catch (InterruptedException e)
        {
            e.printStackTrace();
            return null;
        }
    }
}
