// $ANTLR 2.7.7 (20060906): "java.g" -> "JavaRecognizer.java"$

package com.atlassian.clover.instr.java;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;

import com.atlassian.clover.cfg.instr.java.JavaInstrumentationConfig;
import com.atlassian.clover.cfg.instr.java.LambdaInstrumentation;
import com.atlassian.clover.Contract;
import com.atlassian.clover.Logger;
import com.atlassian.clover.context.ContextStore;
import com.atlassian.clover.context.ContextSet;
import com.atlassian.clover.registry.*;
import com.atlassian.clover.registry.entities.*;


import clover.antlr.TokenBuffer;
import clover.antlr.TokenStreamException;
import clover.antlr.TokenStreamIOException;
import clover.antlr.ANTLRException;
import clover.antlr.LLkParser;
import clover.antlr.Token;
import clover.antlr.TokenStream;
import clover.antlr.RecognitionException;
import clover.antlr.NoViableAltException;
import clover.antlr.MismatchedTokenException;
import clover.antlr.SemanticException;
import clover.antlr.ParserSharedInputState;
import clover.antlr.collections.impl.BitSet;

/**
 * Java 1.5/JSR14 Recognizer
 * Based on the Public Domain Java 1.3 antlr grammar provided at
 * <a href="http://www.antlr.org">www.antlr.org</a>
 */
public class JavaRecognizer extends clover.antlr.LLkParser       implements JavaTokenTypes
 {


    /** use to log messages **/
    private static final Logger LOG = Logger.getInstance();

    private ContextTreeNode currentContextTreeNode;
    private boolean topLevelClass = true;
    private CloverTokenStreamFilter mFilter;
    /** A stack of nested classes. We compose a full class name as a concatenation of outer classes' names. */
    private LinkedList<String> classnameList = new LinkedList<String>();
    private boolean constExpr = false;
    
    private JavaInstrumentationConfig cfg;
    private FileStructureInfo fileInfo;
    private ClassEntryNode currentTopLevelClassEntry;
    private Emitter suppressWarningsInstr = null;
    private boolean existingFallthroughSuppression = false; 

    /**
     * Counts the number of LT seen in the typeArguments production.
     * It is used in semantic predicates to ensure we have seen
     * enough closing '>' characters; which actually may have been
     * either GT, SR or BSR tokens.
     */
    private int ltCounter = 0;

    /**
     * Counts number of lambdas present in the source file in the order of their appearance. This number is later
     * used as lambda ID. Such number is more stable than using a line number, for instance.
     */
    private int lambdaCounter = 0;

    private final Deque<Deque<String>> identifiersStack = new ArrayDeque<Deque<String>>();

    public JavaRecognizer(CloverTokenStreamFilter filter, JavaInstrumentationConfig cfg, FileStructureInfo fileInfo, ContextTreeNode contextTreeRoot) {
        this(filter);
        mFilter = filter;
        this.cfg = cfg;
        this.fileInfo = fileInfo;
        this.currentContextTreeNode = contextTreeRoot;
        pushHeadIdentifierStack();
    }

    private ClassEntryNode enterClass(Modifiers mods, CloverToken tok,
            boolean aIsInterface, boolean aIsEnum, boolean isAnnotation) {

        return enterClass(null, mods, tok, aIsInterface, aIsEnum, isAnnotation, null);
    }

    private ClassEntryNode enterClass(Map<String, List<String>> tags, Modifiers mods, CloverToken tok,
            boolean aIsInterface, boolean aIsEnum, boolean isAnnotation, String superclass) {

        String classname = tok.getText();
        int startLine = tok.getLine();
        int startCol = tok.getColumn();
        classnameList.add(classname);
        // the full name of this class is the concatenation of all containing class names
        String fullName = getClassname(classnameList);
        ClassEntryNode node = new ClassEntryNode(tags, mods, fullName, fileInfo.getPackageName(), superclass,
                getCurrentContext(), startLine, startCol, topLevelClass, aIsInterface, aIsEnum, isAnnotation);
        tok.addPreEmitter(node);
        if (topLevelClass) {
            currentTopLevelClassEntry = node;
        }
        return node;
    }

     private void pushIdentifierToHeadStack(String ident) {
         headIdentifiersStack().push(ident);
     }

     private Deque<String> headIdentifiersStack() {
         return identifiersStack.getFirst();
     }

     private void popHeadIdentifierStack() {
         identifiersStack.pollFirst();
     }

     private void pushHeadIdentifierStack() {
         identifiersStack.push(new ArrayDeque<String>());
     }

    private void exitClass(CloverToken t, ClassEntryNode entry) {
        Contract.pre(classnameList.size() > 0);
        classnameList.removeLast();
        t.addPreEmitter(new ClassExitNode(entry, getClassname(classnameList), t.getLine(), t.getColumn() + t.getText().length()));
    }

    private String getClassname(List<String> classList) {
        String fullName = "";
        String sep = "";
        for (String className : classList) {
            fullName += sep + className;
            sep = ".";
        }
        return fullName;
    }

    private void enterContext(int newContext) {
        currentContextTreeNode = currentContextTreeNode.enterContext(newContext);
    }

    private ContextSet getCurrentContext() {
        return currentContextTreeNode.getContext();
    }
    
    private void exitContext() {
        currentContextTreeNode = currentContextTreeNode.exitContext();
    }

    private void instrStaticRecorderMember(boolean isEnum) {
        if (currentTopLevelClassEntry != null && currentTopLevelClassEntry.getRecorderInsertPoint() != null) {

            RecorderInstrEmitter recorderInstr = new RecorderInstrEmitter(isEnum);
            currentTopLevelClassEntry.setRecorderInstrEmitter(recorderInstr);
            if (isEnum) {
                currentTopLevelClassEntry.getRecorderInsertPoint().addPreEmitter(new SimpleEmitter(";"));
                currentTopLevelClassEntry.getRecorderInsertPoint().addPreEmitter(recorderInstr);
            } else {
                currentTopLevelClassEntry.getRecorderInsertPoint().addPostEmitter(recorderInstr);
            }
        }
    }

    private void instrSuppressWarnings(CloverToken instrPoint) {
        if (!existingFallthroughSuppression && fileInfo.isSuppressFallthroughWarnings()) {
            if (suppressWarningsInstr == null) {
                // no existing SuppressWarnings annotation on the outermost type, so add our own
                instrPoint.addPreEmitter(new SimpleEmitter("@" + cfg.getJavaLangPrefix() + "SuppressWarnings({\"fallthrough\"}) "));
            }
        }
        else if (suppressWarningsInstr != null) {
            suppressWarningsInstr.setEnabled(false);
        }
    }

    private void setRecorderMemberInsertPoint(ClassEntryNode node, CloverToken insertPoint) {
        if (topLevelClass) {
            node.setRecorderInsertPoint(insertPoint);
            topLevelClass = false;
        }
    }

    private MethodEntryInstrEmitter instrEnterMethod(MethodSignature sig, CloverToken start,
                                                     CloverToken leftCurly, CloverToken skip) {
        CloverToken instrPoint = (skip == null ? leftCurly : skip);
        MethodRegistrationNode reg = new MethodRegistrationNode(getCurrentContext(), sig, start.getLine(), start.getColumn());
        leftCurly.addPreEmitter(reg);

        MethodEntryInstrEmitter ret = new MethodEntryInstrEmitter(reg);
        instrPoint.addPostEmitter(ret);
        instrPoint.addPostEmitter(new DirectedFlushEmitter());
        return ret;
    }

    private MethodEntryInstrEmitter instrEnterMethod(MethodSignature sig, CloverToken start, CloverToken leftCurly) {
        return instrEnterMethod(sig, start, leftCurly, null);
    }

    private void instrExitMethod(MethodEntryInstrEmitter entryEmitter, CloverToken tok) {
        tok.addPreEmitter(new MethodExitInstrEmitter(entryEmitter, tok.getLine(), tok.getColumn()+tok.getText().length()));
    }

    /**
     * Instrument beginning of a lambda body in a method-like way. We shall get a code like this:
     *   before:  { }
     *   after :  { RECORDER_INSTANCE_NAME.inc(777); }
     */
    private LambdaBlockEntryEmitter instrEnterLambdaBlock(MethodSignature lambdaSignature, CloverToken leftCurly) {
        if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL
                        || cfg.getInstrumentLambda() == LambdaInstrumentation.ALL_BUT_REFERENCE
                        || cfg.getInstrumentLambda() == LambdaInstrumentation.BLOCK) {
             LambdaBlockEntryEmitter emitter = new LambdaBlockEntryEmitter(lambdaSignature, leftCurly.getLine(),
                    leftCurly.getColumn());
             leftCurly.addPostEmitter(emitter);
             return emitter;
        }
        return null;
    }

    /**
     * Instrument ending of a lambda body. It does not do anything currently (we might add coverage flushing in future)
     */
    private void instrExitLambdaBlock(LambdaBlockEntryEmitter entryEmitter, CloverToken rightCurly) {
        if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL
                    || cfg.getInstrumentLambda() == LambdaInstrumentation.ALL_BUT_REFERENCE
                    || cfg.getInstrumentLambda() == LambdaInstrumentation.BLOCK) {
            rightCurly.addPreEmitter(new LambdaBlockExitEmitter(entryEmitter, rightCurly.getLine(),
                    rightCurly.getColumn() + rightCurly.getText().length()));
        }
    }

    /**
     * Instrument beginning of a lambda expression. We shall get a code like this:
     *   before:   [() -> 1 + 2]
     *   after :   [RECORDER_INSTANCE_NAME.lambdaInc(777, () -> 1 + 2]
     * <p/>
     * or in case when class cast is also present:
     *   before:   [(Integer)() -> 1 + 2]
     *   after :   [(Integer)RECORDER_INSTANCE_NAME.lambdaInc(777, (Integer)() -> 1 + 2]
     * <p/>
     * Tokens:
     * <pre>
     *   (Integer)(x, y) -> x + y
     *            ^ lambdaStart
     *                      ^ bodyStart
     *   ^ classCastStart
     *           ^ classCastEnd
     * </pre>
     */
    private LambdaExpressionEntryEmitter instrEnterLambdaExpression(MethodSignature lambdaSignature,
            CloverToken lambdaStart, CloverToken bodyStart,
            CloverToken classCastStart, CloverToken classCastEnd) {
        if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL
                     || cfg.getInstrumentLambda() == LambdaInstrumentation.ALL_BUT_REFERENCE
                     || cfg.getInstrumentLambda() == LambdaInstrumentation.EXPRESSION) {
            String classCast = "";
            if (classCastStart != null && classCastEnd != null && classCastStart != classCastEnd) {
                classCast = TokenListUtil.getNormalisedSequence(classCastStart, classCastEnd);
            }
            LambdaExpressionEntryEmitter emitter = new LambdaExpressionEntryEmitter(lambdaSignature, classCast,
                    lambdaStart.getLine(), lambdaStart.getColumn(), bodyStart.getLine(), bodyStart.getColumn());
            lambdaStart.addPreEmitter(emitter);
            return emitter;
        }
        return null;
    }

     /**
      * Instrument ending of a lambda expression. We shall get a code like this:
      *   before:   [() -> 1 + 2]
      *   after :   [() -> 1 + 2)]
      */
    private void instrExitLambdaExpression(LambdaExpressionEntryEmitter entryEmitter, CloverToken tok) {
         if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL
                         || cfg.getInstrumentLambda() == LambdaInstrumentation.ALL_BUT_REFERENCE
                         || cfg.getInstrumentLambda() == LambdaInstrumentation.EXPRESSION) {
             tok.addPostEmitter(new LambdaExpressionExitEmitter(entryEmitter, tok.getLine(), tok.getColumn()+tok.getText().length()));
         }
    }

    /**
     * Instrument beginning of a lambda method reference. We shall get a code like this:
     *   before:   [Math::sum]
     *   after :   [RECORDER_INSTANCE_NAME.lambdaInc(777, Math::abs]
     * <p/>
     * or in case when class cast is also present:
     *   before:   [(Function<Integer,Integer>)Math::abs]
     *   after :   [(Function<Integer,Integer>)RECORDER_INSTANCE_NAME.lambdaInc(777, (Function<Integer,Integer>)Math::abs]
     * <p/>
     * Tokens:
     * <pre>
     *   (Function<Integer,Integer>)Math::abs
     *                              ^ methodReferenceStart
     *
     *   ^ classCastStart
     *                             ^ classCastEnd
     * </pre>
     */
    private LambdaExpressionEntryEmitter instrEnterLambdaMethodReference(MethodSignature lambdaSignature,
            CloverToken methodReferenceStart, CloverToken classCastStart, CloverToken classCastEnd) {
        if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL) {
            String classCast = "";
            if (classCastStart != null && classCastEnd != null && classCastStart != classCastEnd) {
                classCast = TokenListUtil.getNormalisedSequence(classCastStart, classCastEnd);
            }
            LambdaExpressionEntryEmitter emitter = new LambdaExpressionEntryEmitter(lambdaSignature, classCast,
                    methodReferenceStart.getLine(), methodReferenceStart.getColumn(),
                    methodReferenceStart.getLine(), methodReferenceStart.getColumn());
            methodReferenceStart.addPreEmitter(emitter);
            return emitter;
        }
        return null;
    }

     /**
      * Instrument ending of a lambda expression. We shall get a code like this:
      *   before:   [Math::abs]
      *   after :   [Math::abs)]
      */
    private void instrExitMethodReference(LambdaExpressionEntryEmitter entryEmitter, CloverToken tok) {
         if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL) {
             tok.addPostEmitter(new LambdaExpressionExitEmitter(entryEmitter, tok.getLine(), tok.getColumn()+tok.getText().length()));
         }
    }

    /**
     * Instrument beginning of a lambda expression, which shall be transformed into lambda block. We shall get a code like this:
     *   before:   [() -> 1 + 2]
     *   after :   [() -> {RECORDER_INSTANCE_NAME.inc(777); return 1 + 2]
     * <p/>
     * or in case when class cast is also present:
     *   before:   [(Integer)() -> 1 + 2]
     *   after :   [(Integer)() -> {RECORDER_INSTANCE_NAME.inc(777); return 1 + 2]
     * <p/>
     *  Return key word will be skipped if lambda expression is noted as void return type by CLOVER:VOID directive, like this:
     *
     *  [() -> /*CLOVER:VOID"&#42;&#47;" System.out.println("Hello World!")]
     *
     * Tokens:
     * <pre>
     *   (x, y) -> x + y
     *   ^ lambdaStart
     *             ^ bodyStart
     * </pre>
     */
     private LambdaExprToBlockStartEntryEmitter instrEnterLambdaExprToBlockExpression(MethodSignature lambdaSignature,
            CloverToken lambdaStart, CloverToken bodyStart) {

        if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL
                || cfg.getInstrumentLambda() == LambdaInstrumentation.ALL_BUT_REFERENCE
                || cfg.getInstrumentLambda() == LambdaInstrumentation.EXPRESSION) {
             LambdaExprToBlockStartEntryEmitter startEmitter = new LambdaExprToBlockStartEntryEmitter(
                    lambdaSignature, lambdaStart.getLine(), lambdaStart.getColumn());
             LambdaExprToBlockBodyEntryEmitter bodyEmitter = new LambdaExprToBlockBodyEntryEmitter(
                    startEmitter, bodyStart.getLine(), bodyStart.getColumn());
             lambdaStart.addPreEmitter(startEmitter);
             bodyStart.addPreEmitter(bodyEmitter);
             return startEmitter;
         }
         return null;
     }


    /**
     * Instrument ending of a lambda expression, which shall be transformed into lambda block. We shall get a code like this:
     *   before:   [() -> 1 + 2]
     *   after :   [() -> 1 + 2;}]
     */
    private void instrExitLambdaExprToBlockExpression(LambdaExprToBlockStartEntryEmitter entryEmitter, CloverToken tok) {
        if (cfg.getInstrumentLambda() == LambdaInstrumentation.ALL
                     || cfg.getInstrumentLambda() == LambdaInstrumentation.ALL_BUT_REFERENCE
                     || cfg.getInstrumentLambda() == LambdaInstrumentation.EXPRESSION) {
             tok.addPostEmitter(
                    new LambdaExprToBlockExitEmitter(entryEmitter, tok.getLine(), tok.getColumn()+tok.getText().length()));
         }
    }

    private CaseExpressionEntryEmitter instrEnterCaseExpression(CloverToken insertionPoint, CloverToken endToken, ContextSet context, int complexity) {
        // we add "caseInc(123,()->" AFTER the "->"
        final CaseExpressionEntryEmitter entryEmitter = new CaseExpressionEntryEmitter(
                context,
                insertionPoint.getLine(),
                insertionPoint.getColumn(),
                endToken.getLine(),
                endToken.getColumn(),
                complexity);
        insertionPoint.addPostEmitter(entryEmitter);
        return entryEmitter;
    }

    private void instrExitCaseExpression(CaseExpressionEntryEmitter entryEmitter, CloverToken insertionPoint) {
        // we add closing ")" BEFORE the ";"
        insertionPoint.addPreEmitter(
                new CaseExpressionExitEmitter(entryEmitter));
    }

    private CaseThrowExpressionEntryEmitter instrEnterCaseThrowExpression(CloverToken insertionPoint, CloverToken endToken, ContextSet context, int complexity) {
        // we add "{ R.inc();" AFTER the "->"
        final CaseThrowExpressionEntryEmitter entryEmitter = new CaseThrowExpressionEntryEmitter(
                context,
                insertionPoint.getLine(),
                insertionPoint.getColumn(),
                endToken.getLine(),
                endToken.getColumn(),
                complexity);
        insertionPoint.addPostEmitter(entryEmitter);
        return entryEmitter;
    }

    private void instrExitCaseThrowExpression(CaseThrowExpressionEntryEmitter entryEmitter, CloverToken insertionPoint) {
        // we add closing "}" AFTER the ";"
        insertionPoint.addPostEmitter(
                new CaseThrowExpressionExitEmitter(entryEmitter));
    }

    private CloverToken maybeAddFlushInstr(CloverToken last) {
        last.addPostEmitter(new DirectedFlushEmitter());
        return last;
    }

    private FlagDeclEmitter declareFlagBefore(CloverToken tok) {
        FlagDeclEmitter flag = new FlagDeclEmitter();
        tok.addPreEmitter(flag);
        return flag;
    }

    private CloverToken instrInlineAfter(CloverToken instr, CloverToken start, CloverToken end, int complexity) {
        if (cfg.isStatementInstrEnabled()) {
            instr.addPostEmitter(
                    new StatementInstrEmitter(
                            getCurrentContext(), start.getLine(), start.getColumn(), end.getLine(),
                            end.getColumn() + end.getText().length(), complexity));
            instr.addPostEmitter(new DirectedFlushEmitter());
            fileInfo.addStatementMarker(start, end);
        }
        return instr;
    }

    // same as above, but protected by a flag check
    private CloverToken instrInlineAfter(CloverToken tok, CloverToken start, CloverToken end, FlagDeclEmitter flag, int complexity) {
        if (cfg.isStatementInstrEnabled()) {
            tok.addPostEmitter(
                    new FlaggedInstrEmitter(
                        flag,
                        new StatementInstrEmitter(
                                getCurrentContext(), start.getLine(), start.getColumn(),
                                end.getLine(), end.getColumn() + end.getText().length(), complexity)));
            fileInfo.addStatementMarker(start, end);
        }
        return tok;
    }

    private CloverToken instrInlineBefore(CloverToken start, CloverToken end, ContextSet context, int complexity) {
        if (cfg.isStatementInstrEnabled()) {
            start.addPreEmitter(
                    new StatementInstrEmitter(
                                context, start.getLine(), start.getColumn(), end.getLine(),
                                end.getColumn() + end.getText().length(), complexity));
        }
        return start;
    }

    private void insertAutoCloseableClassDecl(CloverToken tryBlock) {
        if (cfg.isStatementInstrEnabled()) {
            tryBlock.addPreEmitter(
                new AutoCloseableEmitter());
        }
    }

    private void instrArmDecl(CloverToken start, CloverToken end, ContextSet context) {
        if (cfg.isStatementInstrEnabled()) {
          start.addPreEmitter(
            new ArmInstrEmitter(
              context, start.getLine(), start.getColumn(), end.getLine(),
              end.getColumn() + end.getText().length(), 1 /*TODO*/));
        }
    }

    private void addCloseBraceAfter(CloverToken tok) {
        // inserts a top-level token into the token stream, rather than using instrumentation
        CloverToken closeBrace = new CloverToken(JavaTokenTypes.RCURLY, "}");
        closeBrace.setLine(tok.getLine());
        closeBrace.setColumn(tok.getColumn());
        closeBrace.setPrev(tok);
        closeBrace.setNext(tok.getNext());
        tok.setNext(closeBrace);
    }

    private void addOpenBraceAfter(CloverToken tok) {
        // inserts a top-level token into the token stream, rather than using instrumentation
        CloverToken openBrace = new CloverToken(JavaTokenTypes.LCURLY, "{");
        openBrace.setLine(tok.getLine());
        openBrace.setColumn(tok.getColumn());
        openBrace.setPrev(tok);
        openBrace.setNext(tok.getNext());
        tok.setNext(openBrace);
    }

    private CloverToken instrBoolExpr(CloverToken begin, CloverToken end) {
        if (cfg.isStatementInstrEnabled()) {
          // if there is an assignment in this boolean expr, then
          // instrumentation may cause a compilation failure if the variable
          // being assigned to is not initialised prior to this expression.
          ExpressionInfo expr = ExpressionInfo.fromTokens(begin, end);
          if (!expr.isConstant()) {
              begin.addPreEmitter(new StartBoolInstrEmitter(expr));
              end.addPreEmitter(
                    new EndBoolInstrEmitter(getCurrentContext(), begin.getLine(), begin.getColumn() - 1,
                            end.getLine(), end.getColumn() + end.getText().length(), expr));
          }
        }
        return end;
    }

    private boolean maybeEnterDeprecated(CloverToken startOfBlock) {
        Map<String, List<String>> tags  = TokenListUtil.getJDocTagsAndValuesOnBlock(startOfBlock);
        boolean deprecated = tags.containsKey("deprecated");
        if (deprecated) {
            enterContext(ContextStore.CONTEXT_DEPRECATED);
        }
        return deprecated;
    }

    private boolean maybeEnterDeprecated(Map tags, Modifiers mods) {
        boolean deprecated = tags.containsKey("deprecated") || mods.containsAnnotation("Deprecated");
        if (deprecated) {
            enterContext(ContextStore.CONTEXT_DEPRECATED);
        }
        return deprecated;
    }

    private void maybeExitDeprecated(boolean dep) {
       if (dep) {
            exitContext();
       }
    }

    /**
     * A helper method to check a currently matched token.
     * Usage: <pre>{isCurrentKeyword("abc")}?</pre>
     */
    private boolean isCurrentKeyword(String keyword) throws TokenStreamException {
        return LT(0).getText().equals(keyword);
    }

    /**
     * A helper method to lookup a next token, this way avoiding writing a string literal in a grammar rule,
     * which otherwise would have been interpreted by ANTLR as a token (a reserved keyword). Use it when a
     * "pseudo-keyword" (like "record", "sealed", "module") have to be checked in a given context.
     * Usage: <pre>{isNextKeyword("abc")}? IDENT</pre>
     */
    private boolean isNextKeyword(String keyword) throws TokenStreamException {
        return LT(1).getText().equals(keyword);
    }

    /**
     * Like above, but looks up two tokens.
     * Sample usage: <pre>{isNextKeyword("abc", "def")}? IDENT IDENT</pre>
     */
    private boolean isNextKeyword(String firstKeyword, String secondKeyword) throws TokenStreamException {
        return LT(1).getText().equals(firstKeyword) && LT(2).getText().equals(secondKeyword);
    }

    /**
     * Returns n-th token from LT(n) casted to CloverToken
     */
    private CloverToken lt(int i) throws TokenStreamException {
        return (CloverToken) LT(i);
    }

    /**
     * Casts Token into CloverToken
     */
    private CloverToken ct(Token token) {
        return (CloverToken) token;
    }

protected JavaRecognizer(TokenBuffer tokenBuf, int k) {
  super(tokenBuf,k);
  tokenNames = _tokenNames;
}

public JavaRecognizer(TokenBuffer tokenBuf) {
  this(tokenBuf,2);
}

protected JavaRecognizer(TokenStream lexer, int k) {
  super(lexer,k);
  tokenNames = _tokenNames;
}

public JavaRecognizer(TokenStream lexer) {
  this(lexer,2);
}

public JavaRecognizer(ParserSharedInputState state) {
  super(state,2);
  tokenNames = _tokenNames;
}

	public final void moduleDeclarationPredicate() throws RecognitionException, TokenStreamException {
		
		
		AnnotationImpl an;
		String moduleName;
		
		
		{
		_loop3:
		do {
			if ((LA(1)==AT)) {
				an=annotation();
			}
			else {
				break _loop3;
			}
			
		} while (true);
		}
		{
		if (((LA(1)==IDENT) && (LA(2)==IDENT))&&( isNextKeyword("open", "module") )) {
			match(IDENT);
			match(IDENT);
		}
		else if (((LA(1)==IDENT) && (LA(2)==IDENT))&&( isNextKeyword("module") )) {
			match(IDENT);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		moduleName=identifier();
		match(LCURLY);
	}
	
	protected final AnnotationImpl  annotation() throws RecognitionException, TokenStreamException {
		AnnotationImpl anno;
		
		
		anno=annotation2(false);
		return anno;
	}
	
	public final String  identifier() throws RecognitionException, TokenStreamException {
		String str;
		
		Token  i1 = null;
		Token  i2 = null;
		
		StringBuffer buf = new StringBuffer();
		str = null;
		
		
		i1 = LT(1);
		match(IDENT);
		if ( inputState.guessing==0 ) {
			buf.append(i1.getText());
		}
		{
		_loop103:
		do {
			if ((LA(1)==DOT)) {
				match(DOT);
				if ( inputState.guessing==0 ) {
					buf.append('.');
				}
				i2 = LT(1);
				match(IDENT);
				if ( inputState.guessing==0 ) {
					buf.append(i2.getText());
				}
			}
			else {
				break _loop103;
			}
			
		} while (true);
		}
		if ( inputState.guessing==0 ) {
			
			str = buf.toString();
			
		}
		return str;
	}
	
	public final void moduleDeclaration() throws RecognitionException, TokenStreamException {
		
		
		AnnotationImpl an;
		String moduleName;
		
		
		{
		_loop7:
		do {
			if ((LA(1)==AT)) {
				an=annotation();
			}
			else {
				break _loop7;
			}
			
		} while (true);
		}
		{
		if (((LA(1)==IDENT) && (LA(2)==IDENT))&&( isNextKeyword("open") )) {
			match(IDENT);
		}
		else if ((LA(1)==IDENT) && (LA(2)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if (!( isNextKeyword("module") ))
		  throw new SemanticException(" isNextKeyword(\"module\") ");
		match(IDENT);
		moduleName=identifier();
		match(LCURLY);
		{
		_loop10:
		do {
			if ((LA(1)==IDENT)) {
				moduleDirective();
			}
			else {
				break _loop10;
			}
			
		} while (true);
		}
		match(RCURLY);
	}
	
	public final void moduleDirective() throws RecognitionException, TokenStreamException {
		
		
		{
		boolean synPredMatched14 = false;
		if (((LA(1)==IDENT) && (LA(2)==STATIC||LA(2)==IDENT))) {
			int _m14 = mark();
			synPredMatched14 = true;
			inputState.guessing++;
			try {
				{
				requiresDirective();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched14 = false;
			}
			rewind(_m14);
inputState.guessing--;
		}
		if ( synPredMatched14 ) {
			requiresDirective();
		}
		else {
			boolean synPredMatched16 = false;
			if (((LA(1)==IDENT) && (LA(2)==IDENT))) {
				int _m16 = mark();
				synPredMatched16 = true;
				inputState.guessing++;
				try {
					{
					exportsDirective();
					}
				}
				catch (RecognitionException pe) {
					synPredMatched16 = false;
				}
				rewind(_m16);
inputState.guessing--;
			}
			if ( synPredMatched16 ) {
				exportsDirective();
			}
			else {
				boolean synPredMatched18 = false;
				if (((LA(1)==IDENT) && (LA(2)==IDENT))) {
					int _m18 = mark();
					synPredMatched18 = true;
					inputState.guessing++;
					try {
						{
						opensDirective();
						}
					}
					catch (RecognitionException pe) {
						synPredMatched18 = false;
					}
					rewind(_m18);
inputState.guessing--;
				}
				if ( synPredMatched18 ) {
					opensDirective();
				}
				else {
					boolean synPredMatched20 = false;
					if (((LA(1)==IDENT) && (LA(2)==IDENT))) {
						int _m20 = mark();
						synPredMatched20 = true;
						inputState.guessing++;
						try {
							{
							usesDirective();
							}
						}
						catch (RecognitionException pe) {
							synPredMatched20 = false;
						}
						rewind(_m20);
inputState.guessing--;
					}
					if ( synPredMatched20 ) {
						usesDirective();
					}
					else {
						boolean synPredMatched22 = false;
						if (((LA(1)==IDENT) && (LA(2)==IDENT))) {
							int _m22 = mark();
							synPredMatched22 = true;
							inputState.guessing++;
							try {
								{
								providesDirective();
								}
							}
							catch (RecognitionException pe) {
								synPredMatched22 = false;
							}
							rewind(_m22);
inputState.guessing--;
						}
						if ( synPredMatched22 ) {
							providesDirective();
						}
						else {
							throw new NoViableAltException(LT(1), getFilename());
						}
						}}}}
						}
					}
					
	public final void requiresDirective() throws RecognitionException, TokenStreamException {
		
		
		String requiredModule;
		
		
		if (!( isNextKeyword("requires") ))
		  throw new SemanticException(" isNextKeyword(\"requires\") ");
		match(IDENT);
		{
		if (((LA(1)==IDENT) && (LA(2)==IDENT))&&( isNextKeyword("transitive") )) {
			match(IDENT);
		}
		else if ((LA(1)==STATIC)) {
			match(STATIC);
		}
		else if ((LA(1)==IDENT) && (LA(2)==SEMI||LA(2)==DOT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		requiredModule=identifier();
		match(SEMI);
	}
	
	public final void exportsDirective() throws RecognitionException, TokenStreamException {
		
		
		String exportedPackage;
		String moduleName;
		
		
		if (!( isNextKeyword("exports") ))
		  throw new SemanticException(" isNextKeyword(\"exports\") ");
		match(IDENT);
		exportedPackage=identifier();
		{
		if (((LA(1)==IDENT))&&( isNextKeyword("to") )) {
			match(IDENT);
			moduleName=identifier();
			{
			_loop28:
			do {
				if ((LA(1)==COMMA)) {
					match(COMMA);
					moduleName=identifier();
				}
				else {
					break _loop28;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		match(SEMI);
	}
	
	public final void opensDirective() throws RecognitionException, TokenStreamException {
		
		
		String openedPackage;
		String moduleName;
		
		
		if (!( isNextKeyword("opens") ))
		  throw new SemanticException(" isNextKeyword(\"opens\") ");
		match(IDENT);
		openedPackage=identifier();
		{
		if (((LA(1)==IDENT))&&( isNextKeyword("to") )) {
			match(IDENT);
			moduleName=identifier();
			{
			_loop32:
			do {
				if ((LA(1)==COMMA)) {
					match(COMMA);
					moduleName=identifier();
				}
				else {
					break _loop32;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		match(SEMI);
	}
	
	public final void usesDirective() throws RecognitionException, TokenStreamException {
		
		
		String serviceName;
		
		
		if (!( isNextKeyword("uses") ))
		  throw new SemanticException(" isNextKeyword(\"uses\") ");
		match(IDENT);
		serviceName=identifier();
		match(SEMI);
	}
	
	public final void providesDirective() throws RecognitionException, TokenStreamException {
		
		
		String serviceName;
		String withType;
		
		
		if (!( isNextKeyword("provides") ))
		  throw new SemanticException(" isNextKeyword(\"provides\") ");
		match(IDENT);
		serviceName=identifier();
		if (!( isNextKeyword("with") ))
		  throw new SemanticException(" isNextKeyword(\"with\") ");
		match(IDENT);
		withType=identifier();
		{
		_loop36:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				withType=identifier();
			}
			else {
				break _loop36;
			}
			
		} while (true);
		}
		match(SEMI);
	}
	
	public final void compilationUnit() throws RecognitionException, TokenStreamException {
		
		
		{
		boolean synPredMatched40 = false;
		if (((LA(1)==PACKAGE||LA(1)==AT) && (LA(2)==IDENT))) {
			int _m40 = mark();
			synPredMatched40 = true;
			inputState.guessing++;
			try {
				{
				packageDefinition();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched40 = false;
			}
			rewind(_m40);
inputState.guessing--;
		}
		if ( synPredMatched40 ) {
			packageDefinition();
		}
		else if ((LA(1)==EOF||LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==ENUM||LA(1)==FINAL||LA(1)==IMPORT||LA(1)==INTERFACE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT) && (LA(2)==EOF||LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		_loop42:
		do {
			if ((LA(1)==IMPORT)) {
				importDefinition();
			}
			else {
				break _loop42;
			}
			
		} while (true);
		}
		{
		boolean synPredMatched45 = false;
		if (((LA(1)==AT||LA(1)==IDENT) && (LA(2)==IDENT))) {
			int _m45 = mark();
			synPredMatched45 = true;
			inputState.guessing++;
			try {
				{
				moduleDeclarationPredicate();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched45 = false;
			}
			rewind(_m45);
inputState.guessing--;
		}
		if ( synPredMatched45 ) {
			moduleDeclaration();
		}
		else if ((LA(1)==EOF||LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==ENUM||LA(1)==FINAL||LA(1)==INTERFACE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT) && (LA(2)==EOF||LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT)) {
			{
			_loop47:
			do {
				if ((LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==ENUM||LA(1)==FINAL||LA(1)==INTERFACE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
					typeDefinition(false);
					if ( inputState.guessing==0 ) {
						
						topLevelClass=true;
						existingFallthroughSuppression = false;
						
					}
				}
				else {
					break _loop47;
				}
				
			} while (true);
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		match(Token.EOF_TYPE);
	}
	
	public final void packageDefinition() throws RecognitionException, TokenStreamException {
		
		
		String pn = "";
		AnnotationImpl ann = null;
		
		
		{
		_loop50:
		do {
			if ((LA(1)==AT)) {
				ann=annotation();
			}
			else {
				break _loop50;
			}
			
		} while (true);
		}
		match(PACKAGE);
		pn=identifier();
		match(SEMI);
		if ( inputState.guessing==0 ) {
			fileInfo.setPackageName(pn);
		}
	}
	
	public final void importDefinition() throws RecognitionException, TokenStreamException {
		
		
		match(IMPORT);
		{
		if ((LA(1)==STATIC)) {
			match(STATIC);
		}
		else if ((LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		identifierStar();
		match(SEMI);
	}
	
	public final void typeDefinition(
		boolean nested
	) throws RecognitionException, TokenStreamException {
		
		
		CloverToken first = null;
		Modifiers mods = null;
		
		
		boolean synPredMatched56 = false;
		if (((LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==ENUM||LA(1)==FINAL||LA(1)==INTERFACE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==AT||LA(1)==IDENT))) {
			int _m56 = mark();
			synPredMatched56 = true;
			inputState.guessing++;
			try {
				{
				classOrInterfaceModifiers(false);
				{
				if ((LA(1)==CLASS)) {
					match(CLASS);
				}
				else if ((LA(1)==INTERFACE)) {
					match(INTERFACE);
				}
				else if ((LA(1)==AT)) {
					match(AT);
					match(INTERFACE);
				}
				else if ((LA(1)==ENUM)) {
					match(ENUM);
				}
				else if (((true))&&( isCurrentKeyword("record") )) {
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				}
			}
			catch (RecognitionException pe) {
				synPredMatched56 = false;
			}
			rewind(_m56);
inputState.guessing--;
		}
		if ( synPredMatched56 ) {
			if ( inputState.guessing==0 ) {
				
				pushHeadIdentifierStack(); // for the case when identifiers are recognized outside expression context
				// (i.e. annotations)
				// we don't want to fail on empty stack
				first = lt(1);
				
			}
			mods=classOrInterfaceModifiers(!nested);
			typeDefinition2(mods, first, nested);
		}
		else if ((LA(1)==SEMI)) {
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				popHeadIdentifierStack();
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
	}
	
	public final void identifierStar() throws RecognitionException, TokenStreamException {
		
		
		match(IDENT);
		{
		_loop106:
		do {
			if ((LA(1)==DOT) && (LA(2)==IDENT)) {
				match(DOT);
				match(IDENT);
			}
			else {
				break _loop106;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==DOT)) {
			match(DOT);
			match(STAR);
		}
		else if ((LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
/**
 * A list of zero or more modifiers (including annotations) in the class, enum or interface signature.
 */
	public final Modifiers  classOrInterfaceModifiers(
		boolean instrSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		Modifiers mods;
		
		
		mods = new Modifiers();
		long m = 0;
		AnnotationImpl ann = null;
		
		
		{
		_loop110:
		do {
			if ((LA(1)==ABSTRACT||LA(1)==FINAL||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==AT||LA(2)==IDENT)) {
				m=classOrInterfaceModifier();
				if ( inputState.guessing==0 ) {
					mods.setMask(mods.getMask() | m);
				}
			}
			else if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation2(instrSuppressWarnings);
				if ( inputState.guessing==0 ) {
					mods.addAnnotation(ann);
				}
			}
			else {
				break _loop110;
			}
			
		} while (true);
		}
		return mods;
	}
	
	public final void typeDefinition2(
		Modifiers mods, CloverToken first, boolean nested
	) throws RecognitionException, TokenStreamException {
		
		
		boolean isEnum = false;
		String name = null;
		
		
		{
		if ((LA(1)==CLASS)) {
			name=classDefinition(mods);
		}
		else {
			boolean synPredMatched60 = false;
			if (((LA(1)==IDENT))) {
				int _m60 = mark();
				synPredMatched60 = true;
				inputState.guessing++;
				try {
					{
					if (!( isCurrentKeyword("record") ))
					  throw new SemanticException(" isCurrentKeyword(\"record\") ");
					}
				}
				catch (RecognitionException pe) {
					synPredMatched60 = false;
				}
				rewind(_m60);
inputState.guessing--;
			}
			if ( synPredMatched60 ) {
				name=recordDefinition(mods);
			}
			else if ((LA(1)==INTERFACE)) {
				name=interfaceDefinition(mods);
			}
			else if ((LA(1)==ENUM)) {
				name=enumDefinition(mods);
				if ( inputState.guessing==0 ) {
					isEnum=true;
				}
			}
			else if ((LA(1)==AT)) {
				name=annotationTypeDeclaration(mods);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			}
			}
			if ( inputState.guessing==0 ) {
				
				if (!nested) {
				instrStaticRecorderMember(isEnum);
				instrSuppressWarnings(first);
				}
				
			}
		}
		
	public final String  classDefinition(
		Modifiers mods
	) throws RecognitionException, TokenStreamException {
		String classname;
		
		Token  id = null;
		
			CloverToken first = lt(0);
			Map<String, List<String>> tags = null;
		boolean deprecated = false;
		CloverToken endOfBlock = null;
		String superclass = null;
		ClassEntryNode classEntry = null;
		classname = null;
		String typeParam = null;
		
		
		match(CLASS);
		if ( inputState.guessing==0 ) {
			
			tags = TokenListUtil.getJDocTagsAndValuesOnBlock(first);
			deprecated = maybeEnterDeprecated(first);
			
		}
		id = LT(1);
		match(IDENT);
		{
		if ((LA(1)==LT)) {
			typeParam=typeParameters();
		}
		else if ((LA(1)==EXTENDS||LA(1)==IMPLEMENTS||LA(1)==LCURLY||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		superclass=superClassClause();
		implementsClause();
		permitsClause(mods);
		if ( inputState.guessing==0 ) {
			
			classEntry = enterClass(tags, mods, ct(id), false, false, false, superclass);
			
		}
		endOfBlock=classBlock(classEntry);
		if ( inputState.guessing==0 ) {
			
			exitClass(endOfBlock, classEntry); maybeExitDeprecated(deprecated);
			classname = id.getText();
			
		}
		return classname;
	}
	
	public final String  recordDefinition(
		Modifiers mods
	) throws RecognitionException, TokenStreamException {
		String recordName;
		
		Token  id = null;
		
		CloverToken first = lt(0);
		Map<String, List<String>> tags = null;
		boolean deprecated = false;
		CloverToken endOfBlock = null;
		String superclass = null;
		ClassEntryNode classEntry = null;
		recordName = null;
		String typeParam = null;
		Parameter [] parameters = null;
		
		
		if (!( (mods.getMask() & com.atlassian.clover.registry.entities.ModifierExt.RECORD) != 0 ))
		  throw new SemanticException(" (mods.getMask() & com.atlassian.clover.registry.entities.ModifierExt.RECORD) != 0 ");
		if ( inputState.guessing==0 ) {
			
			tags = TokenListUtil.getJDocTagsAndValuesOnBlock(first);
			deprecated = maybeEnterDeprecated(first);
			
		}
		id = LT(1);
		match(IDENT);
		match(LPAREN);
		parameters=parameterDeclarationList();
		match(RPAREN);
		superclass=superClassClause();
		implementsClause();
		if ( inputState.guessing==0 ) {
			
			classEntry = enterClass(tags, mods, ct(id), false, false, false, superclass);
			
		}
		endOfBlock=classBlock(classEntry);
		if ( inputState.guessing==0 ) {
			
			exitClass(endOfBlock, classEntry); maybeExitDeprecated(deprecated);
			recordName = id.getText();
			
		}
		return recordName;
	}
	
	public final String  interfaceDefinition(
		Modifiers mods
	) throws RecognitionException, TokenStreamException {
		String name;
		
		Token  id = null;
		
		boolean deprecated = false;
		CloverToken endOfBlock = null;
		ClassEntryNode classEntry = null;
		name = null;
		String typeParam = null;
		
		
		match(INTERFACE);
		if ( inputState.guessing==0 ) {
			
			deprecated = maybeEnterDeprecated(lt(0));
			
		}
		id = LT(1);
		match(IDENT);
		{
		if ((LA(1)==LT)) {
			typeParam=typeParameters();
		}
		else if ((LA(1)==EXTENDS||LA(1)==LCURLY||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			classEntry = enterClass(mods, ct(id), true, false, false);
			
		}
		interfaceExtends();
		permitsClause(mods);
		endOfBlock=classBlock(classEntry);
		if ( inputState.guessing==0 ) {
			
			exitClass(endOfBlock, classEntry);
			maybeExitDeprecated(deprecated);
			name = id.getText();
			
		}
		return name;
	}
	
	public final String  enumDefinition(
		Modifiers mods
	) throws RecognitionException, TokenStreamException {
		String name;
		
		Token  id = null;
		
		boolean deprecated = false;
		CloverToken endOfBlock = null;
		ClassEntryNode classEntry = null;
		name = null;
		
		
		match(ENUM);
		if ( inputState.guessing==0 ) {
			
			deprecated = maybeEnterDeprecated(lt(0));
			
		}
		id = LT(1);
		match(IDENT);
		if ( inputState.guessing==0 ) {
			
			classEntry = enterClass(mods, ct(id), false, true, false);
			
		}
		implementsClause();
		endOfBlock=enumBlock(classEntry);
		if ( inputState.guessing==0 ) {
			
			exitClass(endOfBlock, classEntry);
			maybeExitDeprecated(deprecated);
			name = id.getText();
			
		}
		return name;
	}
	
	public final String  annotationTypeDeclaration(
		Modifiers mods
	) throws RecognitionException, TokenStreamException {
		String name;
		
		Token  id = null;
		
		CloverToken endOfBlock = null;
		ClassEntryNode classEntry = null;
		name = null;
		
		
		match(AT);
		match(INTERFACE);
		id = LT(1);
		match(IDENT);
		if ( inputState.guessing==0 ) {
			
			classEntry = enterClass(mods, ct(id), false, false, true);
			
		}
		endOfBlock=annotationTypeBody(classEntry);
		if ( inputState.guessing==0 ) {
			
			exitClass(endOfBlock, classEntry);
			name = id.getText();
			
		}
		return name;
	}
	
/**
 * A declaration is the creation of a reference or primitive-type variable
 * Create a separate Type/Var tree for each var in the var list.
 */
	public final int  declaration() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		Modifiers mods = null;
		String type = null;
		
		
		mods=fieldModifiers(false);
		type=typeSpec();
		complexity=variableDefinitions();
		return complexity;
	}
	
/**
 * A list of zero or more modifiers in the field signature, including annotations.
 */
	public final Modifiers  fieldModifiers(
		boolean instrSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		Modifiers mods;
		
		
		mods = new Modifiers();
		long m = 0;
		AnnotationImpl ann = null;
		
		
		{
		_loop114:
		do {
			if ((LA(1)==FINAL||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOLATILE)) {
				m=fieldModifier();
				if ( inputState.guessing==0 ) {
					mods.setMask(mods.getMask() | m);
				}
			}
			else if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation2(instrSuppressWarnings);
				if ( inputState.guessing==0 ) {
					mods.addAnnotation(ann);
				}
			}
			else {
				break _loop114;
			}
			
		} while (true);
		}
		return mods;
	}
	
	public final String  typeSpec() throws RecognitionException, TokenStreamException {
		String spec;
		
		
		AnnotationImpl ann = null;
		
		
		{
		_loop64:
		do {
			if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation();
			}
			else {
				break _loop64;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==AT||LA(1)==IDENT)) {
			spec=classTypeSpec();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
			spec=builtInTypeSpec();
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return spec;
	}
	
	public final int  variableDefinitions() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		complexity = 0;
		
		
		complexity=variableDeclarator();
		{
		_loop220:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				tmpComplexity=variableDeclarator();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop220;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final String  classTypeSpec() throws RecognitionException, TokenStreamException {
		String typeSpec;
		
		
		String arrayOpt = "";
		
		
		typeSpec=classOrInterfaceType();
		arrayOpt=arraySpecOpt();
		if ( inputState.guessing==0 ) {
			
			typeSpec += arrayOpt;
			
		}
		return typeSpec;
	}
	
	public final String  builtInTypeSpec() throws RecognitionException, TokenStreamException {
		String spec;
		
		
		String arrayOpt = "";
		
		
		spec=builtInType();
		arrayOpt=arraySpecOpt();
		if ( inputState.guessing==0 ) {
			
			spec += arrayOpt;
			
		}
		return spec;
	}
	
	public final String  arraySpecOpt() throws RecognitionException, TokenStreamException {
		String brackets;
		
		
		brackets = "";
		AnnotationImpl ann = null;
		
		
		{
		_loop70:
		do {
			if ((LA(1)==LBRACK||LA(1)==AT) && (LA(2)==RBRACK||LA(2)==IDENT)) {
				{
				_loop69:
				do {
					if ((LA(1)==AT)) {
						ann=annotation();
					}
					else {
						break _loop69;
					}
					
				} while (true);
				}
				match(LBRACK);
				match(RBRACK);
				if ( inputState.guessing==0 ) {
					
					brackets += "[]";
					
				}
			}
			else {
				break _loop70;
			}
			
		} while (true);
		}
		return brackets;
	}
	
	public final String  classOrInterfaceType() throws RecognitionException, TokenStreamException {
		String type;
		
		
		CloverToken first = null;
		CloverToken last = null;
		type = null;
		AnnotationImpl ann = null;
		
		
		if ( inputState.guessing==0 ) {
			
			first = lt(1);
			
		}
		{
		_loop74:
		do {
			if ((LA(1)==AT)) {
				ann=annotation();
			}
			else {
				break _loop74;
			}
			
		} while (true);
		}
		match(IDENT);
		{
		if ((LA(1)==LT)) {
			typeArguments();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==VOID||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GT||LA(1)==SL_ASSIGN||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT||LA(1)==ELLIPSIS||LA(1)==DOT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		_loop78:
		do {
			if ((LA(1)==DOT) && (LA(2)==IDENT)) {
				match(DOT);
				match(IDENT);
				{
				if ((LA(1)==LT)) {
					typeArguments();
				}
				else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==VOID||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GT||LA(1)==SL_ASSIGN||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT||LA(1)==ELLIPSIS||LA(1)==DOT)) {
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
			}
			else {
				break _loop78;
			}
			
		} while (true);
		}
		if ( inputState.guessing==0 ) {
			
			last = lt(0);
			type = TokenListUtil.getNormalisedSequence(first, last);
			
		}
		return type;
	}
	
	public final void typeArguments() throws RecognitionException, TokenStreamException {
		
		
		int currentLtLevel = 0;
		
		
		if ( inputState.guessing==0 ) {
			
			currentLtLevel = ltCounter;
			
		}
		match(LT);
		if ( inputState.guessing==0 ) {
			
			ltCounter++;
			
		}
		{
		if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==QUESTION||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==EXTENDS||LA(2)==FLOAT||LA(2)==IMPLEMENTS||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==VOID||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==RBRACK||LA(2)==LCURLY||LA(2)==RCURLY||LA(2)==COLON||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==NOT_EQUAL||LA(2)==DIV_ASSIGN||LA(2)==PLUS_ASSIGN||LA(2)==MINUS_ASSIGN||LA(2)==STAR_ASSIGN||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GT||LA(2)==SL_ASSIGN||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT||LA(2)==ELLIPSIS||LA(2)==DOT)) {
			singleTypeArgument();
			{
			_loop82:
			do {
				if ((LA(1)==COMMA) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==QUESTION||LA(2)==AT||LA(2)==IDENT)) {
					match(COMMA);
					singleTypeArgument();
				}
				else {
					break _loop82;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==VOID||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GT||LA(1)==SL_ASSIGN||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT||LA(1)==ELLIPSIS||LA(1)==DOT) && (_tokenSet_0.member(LA(2)))) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==SR||LA(1)==BSR||LA(1)==GT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==EXTENDS||LA(2)==FLOAT||LA(2)==IMPLEMENTS||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==VOID||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==RBRACK||LA(2)==LCURLY||LA(2)==RCURLY||LA(2)==COLON||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==NOT_EQUAL||LA(2)==DIV_ASSIGN||LA(2)==PLUS_ASSIGN||LA(2)==MINUS_ASSIGN||LA(2)==STAR_ASSIGN||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GT||LA(2)==SL_ASSIGN||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT||LA(2)==ELLIPSIS||LA(2)==DOT)) {
			typeArgumentsEnd();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==VOID||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GT||LA(1)==SL_ASSIGN||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT||LA(1)==ELLIPSIS||LA(1)==DOT) && (_tokenSet_0.member(LA(2)))) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if (!((currentLtLevel != 0) || ltCounter == currentLtLevel))
		  throw new SemanticException("(currentLtLevel != 0) || ltCounter == currentLtLevel");
	}
	
	public final void singleTypeArgument() throws RecognitionException, TokenStreamException {
		
		
		String type = null;
		AnnotationImpl ann = null;
		
		
		{
		_loop86:
		do {
			if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation();
			}
			else {
				break _loop86;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==AT||LA(1)==IDENT)) {
			type=classTypeSpec();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
			type=builtInTypeSpec();
		}
		else if ((LA(1)==QUESTION)) {
			match(QUESTION);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==EXTENDS||LA(1)==SUPER) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==QUESTION||LA(2)==AT||LA(2)==IDENT)) {
			{
			if ((LA(1)==EXTENDS)) {
				match(EXTENDS);
			}
			else if ((LA(1)==SUPER)) {
				match(SUPER);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			{
			_loop91:
			do {
				if ((LA(1)==AT) && (LA(2)==IDENT)) {
					ann=annotation();
				}
				else {
					break _loop91;
				}
				
			} while (true);
			}
			{
			if ((LA(1)==AT||LA(1)==IDENT)) {
				type=classTypeSpec();
			}
			else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
				type=builtInTypeSpec();
			}
			else if ((LA(1)==QUESTION)) {
				match(QUESTION);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==VOID||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GT||LA(1)==SL_ASSIGN||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT||LA(1)==ELLIPSIS||LA(1)==DOT) && (_tokenSet_0.member(LA(2)))) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	protected final void typeArgumentsEnd() throws RecognitionException, TokenStreamException {
		
		
		if ((LA(1)==GT)) {
			match(GT);
			if ( inputState.guessing==0 ) {
				ltCounter-=1;
			}
		}
		else if ((LA(1)==SR)) {
			match(SR);
			if ( inputState.guessing==0 ) {
				ltCounter-=2;
			}
		}
		else if ((LA(1)==BSR)) {
			match(BSR);
			if ( inputState.guessing==0 ) {
				ltCounter-=3;
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
	}
	
	public final String  builtInType() throws RecognitionException, TokenStreamException {
		String type;
		
		
		type = "";
		
		
		{
		if ((LA(1)==VOID)) {
			match(VOID);
		}
		else if ((LA(1)==BOOLEAN)) {
			match(BOOLEAN);
		}
		else if ((LA(1)==BYTE)) {
			match(BYTE);
		}
		else if ((LA(1)==CHAR)) {
			match(CHAR);
		}
		else if ((LA(1)==SHORT)) {
			match(SHORT);
		}
		else if ((LA(1)==INT)) {
			match(INT);
		}
		else if ((LA(1)==FLOAT)) {
			match(FLOAT);
		}
		else if ((LA(1)==LONG)) {
			match(LONG);
		}
		else if ((LA(1)==DOUBLE)) {
			match(DOUBLE);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			type = LT(0).getText();
			
		}
		return type;
	}
	
	public final void type() throws RecognitionException, TokenStreamException {
		
		
		String spec = null;
		AnnotationImpl ann = null;
		
		
		{
		_loop97:
		do {
			if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation();
			}
			else {
				break _loop97;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==AT||LA(1)==IDENT)) {
			spec=classOrInterfaceType();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
			spec=builtInType();
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
/**
 * Keywords allowed for a class or interface modifier.
 */
	public final long  classOrInterfaceModifier() throws RecognitionException, TokenStreamException {
		long m;
		
		
		m = 0;
		
		
		if ((LA(1)==PRIVATE)) {
			match(PRIVATE);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PRIVATE;
			}
		}
		else if ((LA(1)==PROTECTED)) {
			match(PROTECTED);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PROTECTED;
			}
		}
		else if ((LA(1)==PUBLIC)) {
			match(PUBLIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PUBLIC;
			}
		}
		else if ((LA(1)==ABSTRACT)) {
			match(ABSTRACT);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.ABSTRACT;
			}
		}
		else if ((LA(1)==FINAL)) {
			match(FINAL);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.FINAL;
			}
		}
		else if ((LA(1)==STATIC)) {
			match(STATIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.STATIC;
			}
		}
		else if ((LA(1)==STRICTFP)) {
			match(STRICTFP);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.STRICT;
			}
		}
		else if ((LA(1)==NON_SEALED)) {
			match(NON_SEALED);
			if ( inputState.guessing==0 ) {
				
				m = com.atlassian.clover.registry.entities.ModifierExt.NON_SEALED;
				
			}
		}
		else if (((LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==AT||LA(2)==IDENT))&&( isNextKeyword("sealed") )) {
			match(IDENT);
			if ( inputState.guessing==0 ) {
				
				m = com.atlassian.clover.registry.entities.ModifierExt.SEALED;
				
			}
		}
		else if (((LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==AT||LA(2)==IDENT))&&( isNextKeyword("record") )) {
			match(IDENT);
			if ( inputState.guessing==0 ) {
				
				m = com.atlassian.clover.registry.entities.ModifierExt.RECORD;
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return m;
	}
	
	protected final AnnotationImpl  annotation2(
		boolean instrSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		AnnotationImpl anno;
		
		Token  lparen = null;
		
		boolean isSuppressWarnings = false;
		boolean ft = false;
		CloverToken ident = null; 
		boolean hasArgs = false;
		String annotationName = null;
		anno = new AnnotationImpl();
		
		
		match(AT);
		annotationName=identifier();
		if ( inputState.guessing==0 ) {
			ident=lt(0);isSuppressWarnings = instrSuppressWarnings && "SuppressWarnings".equals(annotationName);anno.setName(annotationName);
		}
		{
		if ((LA(1)==LPAREN) && (LA(2)==RPAREN)) {
			lparen = LT(1);
			match(LPAREN);
			match(RPAREN);
		}
		else {
			boolean synPredMatched530 = false;
			if (((LA(1)==LPAREN) && (LA(2)==IDENT))) {
				int _m530 = mark();
				synPredMatched530 = true;
				inputState.guessing++;
				try {
					{
					match(LPAREN);
					match(IDENT);
					match(ASSIGN);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched530 = false;
				}
				rewind(_m530);
inputState.guessing--;
			}
			if ( synPredMatched530 ) {
				match(LPAREN);
				annMemberValuePair(anno, isSuppressWarnings);
				{
				_loop532:
				do {
					if ((LA(1)==COMMA)) {
						match(COMMA);
						annMemberValuePair(anno, false);
					}
					else {
						break _loop532;
					}
					
				} while (true);
				}
				match(RPAREN);
				if ( inputState.guessing==0 ) {
					hasArgs=true;
				}
			}
			else if ((LA(1)==LPAREN) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LCURLY||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==AT||LA(2)==IDENT)) {
				match(LPAREN);
				ft=annMemberValue2(anno, "value", isSuppressWarnings, true);
				match(RPAREN);
				if ( inputState.guessing==0 ) {
					hasArgs=true;
				}
			}
			else if ((LA(1)==ABSTRACT||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==DEFAULT||LA(1)==DOUBLE||LA(1)==ENUM||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==INTERFACE||LA(1)==LONG||LA(1)==NATIVE||LA(1)==PACKAGE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SYNCHRONIZED||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==NON_SEALED||LA(1)==QUESTION||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RCURLY||LA(1)==COMMA||LA(1)==LT||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			}
			}
			if ( inputState.guessing==0 ) {
				
				if (isSuppressWarnings && !hasArgs) {
				suppressWarningsInstr = new SimpleEmitter("{\"fallthrough\"}");
				Emitter openParen = new SimpleEmitter("(");
				Emitter closeParen = new SimpleEmitter(")");
				CloverToken instrPoint = ct(lparen);
				if (instrPoint == null) {
				instrPoint = ident;
				suppressWarningsInstr.addDependent(openParen);
				suppressWarningsInstr.addDependent(closeParen);
				instrPoint.addPostEmitter(openParen);
				}
				instrPoint.addPostEmitter(suppressWarningsInstr);
				if (lparen == null) {
				instrPoint.addPostEmitter(closeParen);
				}
				}
				
			}
			return anno;
		}
		
/**
 * Keywords allowed for a field modifier
 */
	public final int  fieldModifier() throws RecognitionException, TokenStreamException {
		int m;
		
		
		m = 0;
		
		
		if ((LA(1)==PRIVATE)) {
			match(PRIVATE);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PRIVATE;
			}
		}
		else if ((LA(1)==PROTECTED)) {
			match(PROTECTED);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PROTECTED;
			}
		}
		else if ((LA(1)==PUBLIC)) {
			match(PUBLIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PUBLIC;
			}
		}
		else if ((LA(1)==FINAL)) {
			match(FINAL);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.FINAL;
			}
		}
		else if ((LA(1)==STATIC)) {
			match(STATIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.STATIC;
			}
		}
		else if ((LA(1)==TRANSIENT)) {
			match(TRANSIENT);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.TRANSIENT;
			}
		}
		else if ((LA(1)==VOLATILE)) {
			match(VOLATILE);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.VOLATILE;
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return m;
	}
	
	public final MethodSignatureExt  methodSignature(
		Map tags, CloverToken first, boolean isPredicate
	) throws RecognitionException, TokenStreamException {
		MethodSignatureExt signatureExt;
		
		Token  methodName = null;
		
		Modifiers mods = null;
		boolean deprecated = false;
		String typeParam = null;
		String returnType = "";
		Parameter [] parameters = null;
		String brackets = "";
		String [] throwsTypes = null;
		signatureExt = MethodSignatureExt.of(null, null, null);
		
		
		mods=methodModifiers(false);
		if ( inputState.guessing==0 ) {
			if (!isPredicate) { deprecated = maybeEnterDeprecated(tags, mods); }
		}
		{
		if ((LA(1)==LT)) {
			typeParam=typeParameters();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		returnType=typeSpec();
		methodName = LT(1);
		match(IDENT);
		match(LPAREN);
		parameters=parameterDeclarationList();
		match(RPAREN);
		brackets=declaratorBrackets();
		{
		if ((LA(1)==THROWS)) {
			throwsTypes=throwsClause();
		}
		else if ((LA(1)==LCURLY||LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			CloverToken endSig = lt(0);
			MethodSignature signature = new MethodSignature(first, ct(methodName), endSig, tags, mods,
			methodName.getText(), typeParam, returnType + brackets, parameters, throwsTypes);
			signatureExt = MethodSignatureExt.of(signature, endSig, deprecated);
			
		}
		return signatureExt;
	}
	
/**
 * A list of zero or more modifiers in the method signature, including annotations.
 */
	public final Modifiers  methodModifiers(
		boolean instrSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		Modifiers mods;
		
		
		mods = new Modifiers();
		long m = 0;
		AnnotationImpl ann = null;
		
		
		{
		_loop130:
		do {
			if ((LA(1)==ABSTRACT||LA(1)==DEFAULT||LA(1)==FINAL||LA(1)==NATIVE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SYNCHRONIZED)) {
				m=methodModifier();
				if ( inputState.guessing==0 ) {
					mods.setMask(mods.getMask() | m);
				}
			}
			else if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation2(instrSuppressWarnings);
				if ( inputState.guessing==0 ) {
					mods.addAnnotation(ann);
				}
			}
			else {
				break _loop130;
			}
			
		} while (true);
		}
		return mods;
	}
	
	public final String  typeParameters() throws RecognitionException, TokenStreamException {
		String asString;
		
		
		int currentLtLevel = 0;
		CloverToken start = lt(1);
		asString = null;
		
		
		if ( inputState.guessing==0 ) {
			
			currentLtLevel = ltCounter;
			
		}
		match(LT);
		if ( inputState.guessing==0 ) {
			ltCounter++;
		}
		typeParameter();
		{
		_loop143:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				typeParameter();
			}
			else {
				break _loop143;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==SR||LA(1)==BSR||LA(1)==GT)) {
			typeArgumentsEnd();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==LCURLY||LA(1)==AT||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if (!((currentLtLevel != 0) || ltCounter == currentLtLevel))
		  throw new SemanticException("(currentLtLevel != 0) || ltCounter == currentLtLevel");
		if ( inputState.guessing==0 ) {
			
			asString = TokenListUtil.getNormalisedSequence(start, lt(0));
			
		}
		return asString;
	}
	
	public final Parameter []  parameterDeclarationList() throws RecognitionException, TokenStreamException {
		Parameter [] params;
		
		
		List<Parameter> parameters = new ArrayList<Parameter>();
		Parameter param = null;
		params = new Parameter[0];
		
		
		{
		if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT)) {
			param=parameterDeclaration();
			if ( inputState.guessing==0 ) {
				
				parameters.add(param);
				
			}
			{
			_loop245:
			do {
				if ((LA(1)==COMMA)) {
					match(COMMA);
					param=parameterDeclaration();
					if ( inputState.guessing==0 ) {
						
						parameters.add(param);
						
					}
				}
				else {
					break _loop245;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==RPAREN)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			params = (Parameter[])parameters.toArray(new Parameter [parameters.size()]);
			
		}
		return params;
	}
	
	public final String  declaratorBrackets() throws RecognitionException, TokenStreamException {
		String brackets;
		
		
		brackets = "";
		AnnotationImpl ann = null;
		
		
		{
		_loop226:
		do {
			if ((LA(1)==LBRACK||LA(1)==AT)) {
				{
				_loop225:
				do {
					if ((LA(1)==AT)) {
						ann=annotation();
					}
					else {
						break _loop225;
					}
					
				} while (true);
				}
				match(LBRACK);
				match(RBRACK);
				if ( inputState.guessing==0 ) {
					brackets += "[]";
				}
			}
			else {
				break _loop226;
			}
			
		} while (true);
		}
		return brackets;
	}
	
	public final String []  throwsClause() throws RecognitionException, TokenStreamException {
		String [] throwsTypes;
		
		
		List<String> throwsList = new ArrayList<String>();
		throwsTypes = null;
		String id;
		AnnotationImpl ann = null;
		
		
		match(THROWS);
		{
		_loop237:
		do {
			if ((LA(1)==AT)) {
				ann=annotation();
			}
			else {
				break _loop237;
			}
			
		} while (true);
		}
		id=identifier();
		if ( inputState.guessing==0 ) {
			
			throwsList.add(id);
			
		}
		{
		_loop241:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				{
				_loop240:
				do {
					if ((LA(1)==AT)) {
						ann=annotation();
					}
					else {
						break _loop240;
					}
					
				} while (true);
				}
				id=identifier();
				if ( inputState.guessing==0 ) {
					
					throwsList.add(id);
					
				}
			}
			else {
				break _loop241;
			}
			
		} while (true);
		}
		if ( inputState.guessing==0 ) {
			
			throwsTypes = (String[])throwsList.toArray(new String[throwsList.size()]);
			
		}
		return throwsTypes;
	}
	
	public final MethodSignatureExt  standardConstructorSignature(
		Map tags, CloverToken first, boolean isPredicate
	) throws RecognitionException, TokenStreamException {
		MethodSignatureExt signatureExt;
		
		Token  constructorName = null;
		
		Modifiers mods = null;
		boolean deprecated = false;
		String typeParam = null;
		Parameter [] params = null;
		String [] throwsTypes = null;
		signatureExt = MethodSignatureExt.of(null, null, null);
		
		
		mods=constructorModifiers(false);
		if ( inputState.guessing==0 ) {
			if (!isPredicate) { deprecated = maybeEnterDeprecated(tags, mods); }
		}
		{
		if ((LA(1)==LT)) {
			typeParam=typeParameters();
		}
		else if ((LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		constructorName = LT(1);
		match(IDENT);
		match(LPAREN);
		params=parameterDeclarationList();
		match(RPAREN);
		{
		if ((LA(1)==THROWS)) {
			throwsTypes=throwsClause();
		}
		else if ((LA(1)==LCURLY)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			CloverToken endSig = lt(0);
			MethodSignature signature = new MethodSignature(first, ct(constructorName), endSig, tags, mods,
			constructorName.getText(), null, null, params, throwsTypes);
			signatureExt = MethodSignatureExt.of(signature, endSig, deprecated);
			
		}
		return signatureExt;
	}
	
/**
 * A list of zero or more modifiers in the constructor signature, including annotations.
 */
	public final Modifiers  constructorModifiers(
		boolean instrSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		Modifiers mods;
		
		
		mods = new Modifiers();
		long m = 0;
		AnnotationImpl ann = null;
		
		
		{
		_loop126:
		do {
			if (((LA(1) >= PRIVATE && LA(1) <= PUBLIC))) {
				m=constructorModifier();
				if ( inputState.guessing==0 ) {
					mods.setMask(mods.getMask() | m);
				}
			}
			else if ((LA(1)==AT)) {
				ann=annotation2(instrSuppressWarnings);
				if ( inputState.guessing==0 ) {
					mods.addAnnotation(ann);
				}
			}
			else {
				break _loop126;
			}
			
		} while (true);
		}
		return mods;
	}
	
	public final MethodSignatureExt  compactConstructorSignature(
		Map tags, CloverToken first, boolean isPredicate
	) throws RecognitionException, TokenStreamException {
		MethodSignatureExt signatureExt;
		
		Token  constructorName = null;
		
		Modifiers mods = null;
		boolean deprecated = false;
		String typeParam = null;
		signatureExt = MethodSignatureExt.of(null, null, null);
		
		
		mods=constructorModifiers(false);
		if ( inputState.guessing==0 ) {
			if (!isPredicate) { deprecated = maybeEnterDeprecated(tags, mods); }
		}
		{
		if ((LA(1)==LT)) {
			typeParam=typeParameters();
		}
		else if ((LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		constructorName = LT(1);
		match(IDENT);
		if ( inputState.guessing==0 ) {
			
			CloverToken endSig = lt(0);
			MethodSignature signature = new MethodSignature(first, ct(constructorName), endSig, tags, mods,
			constructorName.getText(), null, null, null, null);
			signatureExt = MethodSignatureExt.of(signature, endSig, deprecated);
			
		}
		return signatureExt;
	}
	
/**
 * Keywords allowed for a constructor modifier
 */
	public final int  constructorModifier() throws RecognitionException, TokenStreamException {
		int m;
		
		
		m = 0;
		
		
		if ((LA(1)==PRIVATE)) {
			match(PRIVATE);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PRIVATE;
			}
		}
		else if ((LA(1)==PROTECTED)) {
			match(PROTECTED);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PROTECTED;
			}
		}
		else if ((LA(1)==PUBLIC)) {
			match(PUBLIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PUBLIC;
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return m;
	}
	
/**
 * Keywords allowed for a method modifier
 */
	public final long  methodModifier() throws RecognitionException, TokenStreamException {
		long m;
		
		
		m = 0;
		
		
		if ((LA(1)==PRIVATE)) {
			match(PRIVATE);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PRIVATE;
			}
		}
		else if ((LA(1)==PROTECTED)) {
			match(PROTECTED);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PROTECTED;
			}
		}
		else if ((LA(1)==PUBLIC)) {
			match(PUBLIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.PUBLIC;
			}
		}
		else if ((LA(1)==ABSTRACT)) {
			match(ABSTRACT);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.ABSTRACT;
			}
		}
		else if ((LA(1)==FINAL)) {
			match(FINAL);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.FINAL;
			}
		}
		else if ((LA(1)==NATIVE)) {
			match(NATIVE);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.NATIVE;
			}
		}
		else if ((LA(1)==STATIC)) {
			match(STATIC);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.STATIC;
			}
		}
		else if ((LA(1)==STRICTFP)) {
			match(STRICTFP);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.STRICT;
			}
		}
		else if ((LA(1)==SYNCHRONIZED)) {
			match(SYNCHRONIZED);
			if ( inputState.guessing==0 ) {
				m=java.lang.reflect.Modifier.SYNCHRONIZED;
			}
		}
		else if ((LA(1)==DEFAULT)) {
			match(DEFAULT);
			if ( inputState.guessing==0 ) {
				m=com.atlassian.clover.registry.entities.ModifierExt.DEFAULT;
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return m;
	}
	
	public final String  superClassClause() throws RecognitionException, TokenStreamException {
		String superclass;
		
		
		superclass = null;
		
		
		{
		if ((LA(1)==EXTENDS)) {
			match(EXTENDS);
			superclass=classOrInterfaceType();
		}
		else if ((LA(1)==IMPLEMENTS||LA(1)==LCURLY||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return superclass;
	}
	
	public final void implementsClause() throws RecognitionException, TokenStreamException {
		
		
		String type = null;
		
		
		{
		if ((LA(1)==IMPLEMENTS)) {
			match(IMPLEMENTS);
			type=classOrInterfaceType();
			{
			_loop188:
			do {
				if ((LA(1)==COMMA)) {
					match(COMMA);
					type=classOrInterfaceType();
				}
				else {
					break _loop188;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==LCURLY||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	public final void permitsClause(
		Modifiers mods
	) throws RecognitionException, TokenStreamException {
		
		
		String type = null;
		
		
		{
		if (((LA(1)==IDENT))&&(
                (mods.getMask() & com.atlassian.clover.registry.entities.ModifierExt.SEALED) != 0
                    && isNextKeyword("permits")
            )) {
			match(IDENT);
			type=classOrInterfaceType();
			{
			_loop192:
			do {
				if ((LA(1)==COMMA)) {
					match(COMMA);
					type=classOrInterfaceType();
				}
				else {
					break _loop192;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==LCURLY)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	public final CloverToken  classBlock(
		ClassEntryNode classEntry
	) throws RecognitionException, TokenStreamException {
		CloverToken t;
		
		Token  ip = null;
		Token  rc = null;
		
		t = null;
		
		
		ip = LT(1);
		match(LCURLY);
		if ( inputState.guessing==0 ) {
			
			setRecorderMemberInsertPoint(classEntry, ct(ip));
			
		}
		{
		_loop154:
		do {
			if ((LA(1)==ABSTRACT||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==DEFAULT||LA(1)==DOUBLE||LA(1)==ENUM||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==INTERFACE||LA(1)==LONG||LA(1)==NATIVE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SYNCHRONIZED||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==NON_SEALED||LA(1)==LCURLY||LA(1)==LT||LA(1)==AT||LA(1)==IDENT)) {
				field(classEntry);
			}
			else if ((LA(1)==SEMI)) {
				match(SEMI);
			}
			else {
				break _loop154;
			}
			
		} while (true);
		}
		rc = LT(1);
		match(RCURLY);
		if ( inputState.guessing==0 ) {
			
			t = ct(rc);
			
		}
		return t;
	}
	
	public final void interfaceExtends() throws RecognitionException, TokenStreamException {
		
		
		String type = null;
		
		
		{
		if ((LA(1)==EXTENDS)) {
			match(EXTENDS);
			type=classOrInterfaceType();
			{
			_loop184:
			do {
				if ((LA(1)==COMMA)) {
					match(COMMA);
					type=classOrInterfaceType();
				}
				else {
					break _loop184;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==LCURLY||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	public final CloverToken  enumBlock(
		ClassEntryNode classEntry
	) throws RecognitionException, TokenStreamException {
		CloverToken t;
		
		Token  ip = null;
		
		boolean topLevelSave = topLevelClass;
		t = null;
		
		
		match(LCURLY);
		{
		if ((LA(1)==RCURLY||LA(1)==COMMA||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT) && (LA(2)==EOF||LA(2)==ABSTRACT||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==CLASS||LA(2)==DEFAULT||LA(2)==DOUBLE||LA(2)==ENUM||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==INTERFACE||LA(2)==LONG||LA(2)==NATIVE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SYNCHRONIZED||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LCURLY||LA(2)==RCURLY||LA(2)==COMMA||LA(2)==LT||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT)) {
			{
			if ((LA(1)==AT||LA(1)==IDENT)) {
				enumConstant();
				{
				_loop159:
				do {
					if ((LA(1)==COMMA) && (LA(2)==AT||LA(2)==IDENT)) {
						match(COMMA);
						enumConstant();
					}
					else {
						break _loop159;
					}
					
				} while (true);
				}
			}
			else if ((LA(1)==RCURLY||LA(1)==COMMA||LA(1)==SEMI)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			{
			if ((LA(1)==COMMA)) {
				match(COMMA);
			}
			else if ((LA(1)==RCURLY||LA(1)==SEMI)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			{
			if ((LA(1)==SEMI)) {
				if ( inputState.guessing==0 ) {
					
					topLevelSave = topLevelClass;
					topLevelClass = false;
					
				}
				match(SEMI);
				{
				_loop163:
				do {
					if ((LA(1)==ABSTRACT||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==DEFAULT||LA(1)==DOUBLE||LA(1)==ENUM||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==INTERFACE||LA(1)==LONG||LA(1)==NATIVE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SYNCHRONIZED||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==NON_SEALED||LA(1)==LCURLY||LA(1)==LT||LA(1)==AT||LA(1)==IDENT)) {
						field(null);
					}
					else if ((LA(1)==SEMI)) {
						match(SEMI);
					}
					else {
						break _loop163;
					}
					
				} while (true);
				}
				if ( inputState.guessing==0 ) {
					
					topLevelClass = topLevelSave;
					
				}
			}
			else if ((LA(1)==RCURLY)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
		}
		else if ((LA(1)==RCURLY) && (LA(2)==EOF||LA(2)==ABSTRACT||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==CLASS||LA(2)==DEFAULT||LA(2)==DOUBLE||LA(2)==ENUM||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==INTERFACE||LA(2)==LONG||LA(2)==NATIVE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SYNCHRONIZED||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==NON_SEALED||LA(2)==LCURLY||LA(2)==RCURLY||LA(2)==LT||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		ip = LT(1);
		match(RCURLY);
		if ( inputState.guessing==0 ) {
			
			t = ct(ip);
			setRecorderMemberInsertPoint(classEntry, t);
			
		}
		return t;
	}
	
	public final CloverToken  annotationTypeBody(
		ClassEntryNode classEntry
	) throws RecognitionException, TokenStreamException {
		CloverToken t;
		
		Token  ip = null;
		Token  endOfBlock = null;
		
		boolean topLevelSave = topLevelClass;
		t = null;
		Modifiers mods = null;
		String type = null;
		int ignoredComplexity;
		
		
		ip = LT(1);
		match(LCURLY);
		if ( inputState.guessing==0 ) {
			
			setRecorderMemberInsertPoint(classEntry, ct(ip));
			
		}
		{
		_loop179:
		do {
			boolean synPredMatched172 = false;
			if (((LA(1)==ABSTRACT||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DEFAULT||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NATIVE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SYNCHRONIZED||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DEFAULT||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NATIVE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SYNCHRONIZED||LA(2)==VOID||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
				int _m172 = mark();
				synPredMatched172 = true;
				inputState.guessing++;
				try {
					{
					methodModifiers(false);
					typeSpec();
					match(IDENT);
					match(LPAREN);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched172 = false;
				}
				rewind(_m172);
inputState.guessing--;
			}
			if ( synPredMatched172 ) {
				mods=methodModifiers(false);
				type=typeSpec();
				match(IDENT);
				match(LPAREN);
				match(RPAREN);
				{
				if ((LA(1)==DEFAULT)) {
					annDefaultValue();
				}
				else if ((LA(1)==SEMI)) {
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				match(SEMI);
			}
			else {
				boolean synPredMatched175 = false;
				if (((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
					int _m175 = mark();
					synPredMatched175 = true;
					inputState.guessing++;
					try {
						{
						fieldModifiers(false);
						typeSpec();
						variableDeclarator();
						}
					}
					catch (RecognitionException pe) {
						synPredMatched175 = false;
					}
					rewind(_m175);
inputState.guessing--;
				}
				if ( synPredMatched175 ) {
					mods=fieldModifiers(false);
					type=typeSpec();
					ignoredComplexity=variableDefinitions();
					match(SEMI);
				}
				else {
					boolean synPredMatched178 = false;
					if (((LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==ENUM||LA(1)==FINAL||LA(1)==INTERFACE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==AT||LA(2)==IDENT))) {
						int _m178 = mark();
						synPredMatched178 = true;
						inputState.guessing++;
						try {
							{
							classOrInterfaceModifiers(false);
							{
							if ((LA(1)==CLASS)) {
								match(CLASS);
							}
							else if ((LA(1)==INTERFACE)) {
								match(INTERFACE);
							}
							else if ((LA(1)==AT)) {
								match(AT);
								match(INTERFACE);
							}
							else if ((LA(1)==ENUM)) {
								match(ENUM);
							}
							else if (((true))&&( isCurrentKeyword("record") )) {
							}
							else {
								throw new NoViableAltException(LT(1), getFilename());
							}
							
							}
							}
						}
						catch (RecognitionException pe) {
							synPredMatched178 = false;
						}
						rewind(_m178);
inputState.guessing--;
					}
					if ( synPredMatched178 ) {
						if ( inputState.guessing==0 ) {
							
							topLevelSave = topLevelClass;
							topLevelClass = false;
							
						}
						mods=classOrInterfaceModifiers(false);
						typeDefinition2(mods, null, true);
						if ( inputState.guessing==0 ) {
							
							topLevelClass = topLevelSave;
							
						}
					}
					else if ((LA(1)==SEMI)) {
						match(SEMI);
					}
					else {
						break _loop179;
					}
					}}
				} while (true);
				}
				endOfBlock = LT(1);
				match(RCURLY);
				if ( inputState.guessing==0 ) {
					
					t = ct(endOfBlock);
					
				}
				return t;
			}
			
	public final void typeParameter() throws RecognitionException, TokenStreamException {
		
		
		String type = null;
		AnnotationImpl ann = null;
		
		
		{
		_loop147:
		do {
			if ((LA(1)==AT)) {
				ann=annotation();
			}
			else {
				break _loop147;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==IDENT)) {
			match(IDENT);
		}
		else if ((LA(1)==QUESTION)) {
			match(QUESTION);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==EXTENDS) && (LA(2)==AT||LA(2)==IDENT)) {
			match(EXTENDS);
			type=classOrInterfaceType();
			{
			_loop151:
			do {
				if ((LA(1)==BAND)) {
					match(BAND);
					type=classOrInterfaceType();
				}
				else {
					break _loop151;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==EXTENDS||LA(1)==FLOAT||LA(1)==IMPLEMENTS||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==LCURLY||LA(1)==COMMA||LA(1)==SR||LA(1)==BSR||LA(1)==GT||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==CLASS||LA(2)==DEFAULT||LA(2)==DOUBLE||LA(2)==ENUM||LA(2)==EXTENDS||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==IMPLEMENTS||LA(2)==INT||LA(2)==INTERFACE||LA(2)==LONG||LA(2)==NATIVE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SYNCHRONIZED||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==LBRACK||LA(2)==LCURLY||LA(2)==RCURLY||LA(2)==LT||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	public final void field(
		ClassEntryNode containingClass
	) throws RecognitionException, TokenStreamException {
		
		
		CloverToken tmp;
		CloverToken first = lt(1);
		CloverToken endSig = null;
		String typeParam = null;
		boolean deprecated = false;
		Modifiers mods = null;
		String name = null;
		MethodSignature signature = null;
		MethodSignatureExt signatureExt = null;
		String [] throwsTypes = null;
		String returnType = "";
		String brackets = "";
		Parameter [] parameters = null;
		Map<String, List<String>> tags = null;
		String typename = null;
		int ignoredComplexity;
		
		
		if ( inputState.guessing==0 ) {
			
			tags = TokenListUtil.getJDocTagsAndValuesOnBlock(first);
			
		}
		{
		boolean synPredMatched196 = false;
		if (((LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==LT||LA(1)==AT||LA(1)==IDENT) && (LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==QUESTION||LA(2)==LCURLY||LA(2)==LT||LA(2)==AT||LA(2)==IDENT))) {
			int _m196 = mark();
			synPredMatched196 = true;
			inputState.guessing++;
			try {
				{
				compactConstructorSignature(tags, first, true);
				match(LCURLY);
				}
			}
			catch (RecognitionException pe) {
				synPredMatched196 = false;
			}
			rewind(_m196);
inputState.guessing--;
		}
		if ( synPredMatched196 ) {
			signatureExt=compactConstructorSignature(tags, first, false);
			constructorBody(signatureExt.signature(), first, signatureExt.endToken());
			if ( inputState.guessing==0 ) {
				maybeExitDeprecated(signatureExt.isDeprecated());
			}
		}
		else {
			boolean synPredMatched198 = false;
			if (((LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==LT||LA(1)==AT||LA(1)==IDENT) && (LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==LT||LA(2)==AT||LA(2)==IDENT))) {
				int _m198 = mark();
				synPredMatched198 = true;
				inputState.guessing++;
				try {
					{
					standardConstructorSignature(tags, first, true);
					match(LCURLY);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched198 = false;
				}
				rewind(_m198);
inputState.guessing--;
			}
			if ( synPredMatched198 ) {
				signatureExt=standardConstructorSignature(tags, first, false);
				constructorBody(signatureExt.signature(), first, signatureExt.endToken());
				if ( inputState.guessing==0 ) {
					maybeExitDeprecated(signatureExt.isDeprecated());
				}
			}
			else {
				boolean synPredMatched200 = false;
				if (((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
					int _m200 = mark();
					synPredMatched200 = true;
					inputState.guessing++;
					try {
						{
						fieldModifiers(false);
						typeSpec();
						variableDefinitions();
						}
					}
					catch (RecognitionException pe) {
						synPredMatched200 = false;
					}
					rewind(_m200);
inputState.guessing--;
				}
				if ( synPredMatched200 ) {
					mods=fieldModifiers(false);
					if ( inputState.guessing==0 ) {
						deprecated = maybeEnterDeprecated(tags, mods);
					}
					returnType=typeSpec();
					ignoredComplexity=variableDefinitions();
					match(SEMI);
					if ( inputState.guessing==0 ) {
						maybeExitDeprecated(deprecated);
					}
				}
				else if ((LA(1)==STATIC) && (LA(2)==LCURLY)) {
					if ( inputState.guessing==0 ) {
						
						deprecated = maybeEnterDeprecated(first);
						signature = new MethodSignature(null, null, null, "<clinit>, line " + first.getLine(), null, null, null, null);
						
					}
					match(STATIC);
					outerCompoundStmt(signature, first, null, ContextStore.CONTEXT_STATIC);
					if ( inputState.guessing==0 ) {
						maybeExitDeprecated(deprecated);
					}
				}
				else if ((LA(1)==LCURLY)) {
					if ( inputState.guessing==0 ) {
						
						deprecated = maybeEnterDeprecated(first);
						signature = new MethodSignature(null, null, null, "<init>, line " + first.getLine(), null, null, null, null);
						
					}
					outerCompoundStmt(signature, first, null, ContextStore.CONTEXT_INSTANCE);
					if ( inputState.guessing==0 ) {
						maybeExitDeprecated(deprecated);
					}
				}
				else {
					boolean synPredMatched203 = false;
					if (((LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==ENUM||LA(1)==FINAL||LA(1)==INTERFACE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==ENUM||LA(2)==FINAL||LA(2)==INTERFACE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==AT||LA(2)==IDENT))) {
						int _m203 = mark();
						synPredMatched203 = true;
						inputState.guessing++;
						try {
							{
							classOrInterfaceModifiers(false);
							{
							if ((LA(1)==CLASS)) {
								match(CLASS);
							}
							else if ((LA(1)==INTERFACE)) {
								match(INTERFACE);
							}
							else if ((LA(1)==AT)) {
								match(AT);
								match(INTERFACE);
							}
							else if ((LA(1)==ENUM)) {
								match(ENUM);
							}
							else if (((LA(1)==IDENT))&&( isCurrentKeyword("record") )) {
							}
							else {
								throw new NoViableAltException(LT(1), getFilename());
							}
							
							}
							match(IDENT);
							}
						}
						catch (RecognitionException pe) {
							synPredMatched203 = false;
						}
						rewind(_m203);
inputState.guessing--;
					}
					if ( synPredMatched203 ) {
						mods=classOrInterfaceModifiers(false);
						if ( inputState.guessing==0 ) {
							deprecated = maybeEnterDeprecated(tags, mods);
						}
						{
						if ((LA(1)==CLASS)) {
							typename=classDefinition(mods);
						}
						else if ((LA(1)==IDENT)) {
							typename=recordDefinition(mods);
						}
						else if ((LA(1)==INTERFACE)) {
							typename=interfaceDefinition(mods);
						}
						else if ((LA(1)==ENUM)) {
							typename=enumDefinition(mods);
						}
						else if ((LA(1)==AT)) {
							typename=annotationTypeDeclaration(mods);
						}
						else {
							throw new NoViableAltException(LT(1), getFilename());
						}
						
						}
						if ( inputState.guessing==0 ) {
							maybeExitDeprecated(deprecated);
						}
					}
					else {
						boolean synPredMatched207 = false;
						if (((LA(1)==ABSTRACT||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DEFAULT||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NATIVE||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SYNCHRONIZED||LA(1)==VOID||LA(1)==LT||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DEFAULT||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NATIVE||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SYNCHRONIZED||LA(2)==VOID||LA(2)==QUESTION||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
							int _m207 = mark();
							synPredMatched207 = true;
							inputState.guessing++;
							try {
								{
								methodSignature(tags, first, true);
								{
								if ((LA(1)==LCURLY)) {
									match(LCURLY);
								}
								else if ((LA(1)==SEMI)) {
									match(SEMI);
								}
								else {
									throw new NoViableAltException(LT(1), getFilename());
								}
								
								}
								}
							}
							catch (RecognitionException pe) {
								synPredMatched207 = false;
							}
							rewind(_m207);
inputState.guessing--;
						}
						if ( synPredMatched207 ) {
							signatureExt=methodSignature(tags, first, false);
							{
							if ((LA(1)==LCURLY)) {
								outerCompoundStmt(signatureExt.signature(), first, signatureExt.endToken(), ContextStore.CONTEXT_METHOD);
							}
							else if ((LA(1)==SEMI)) {
								match(SEMI);
							}
							else {
								throw new NoViableAltException(LT(1), getFilename());
							}
							
							}
							if ( inputState.guessing==0 ) {
								maybeExitDeprecated(signatureExt.isDeprecated());
							}
						}
						else {
							throw new NoViableAltException(LT(1), getFilename());
						}
						}}}}
						}
					}
					
	public final void enumConstant() throws RecognitionException, TokenStreamException {
		
		
		boolean topLevelSave = topLevelClass;
		CloverToken endOfBlock = null;
		AnnotationImpl ann = null;
		int argListComplexity = 0; // ignored for enum constant
		
		
		if ( inputState.guessing==0 ) {
			
			topLevelSave = topLevelClass;
			topLevelClass = false;
			
		}
		{
		_loop166:
		do {
			if ((LA(1)==AT)) {
				ann=annotation();
			}
			else {
				break _loop166;
			}
			
		} while (true);
		}
		match(IDENT);
		{
		if ((LA(1)==LPAREN)) {
			match(LPAREN);
			argListComplexity=argList();
			match(RPAREN);
		}
		else if ((LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==COMMA||LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==LCURLY)) {
			endOfBlock=classBlock(null);
		}
		else if ((LA(1)==RCURLY||LA(1)==COMMA||LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			topLevelClass = topLevelSave;
			
		}
	}
	
	public final int  argList() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		complexity = 0;
		
		
		{
		if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
			complexity=expressionList();
		}
		else if ((LA(1)==RPAREN)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
	protected final void annDefaultValue() throws RecognitionException, TokenStreamException {
		
		
		boolean ft = false;
		
		
		match(DEFAULT);
		ft=annMemberValue(false,false);
	}
	
/**
 * Declaration of a variable.  This can be a class/instance variable, or a local variable in a method.
 * It can also include possible initialization.
 */
	public final int  variableDeclarator() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		String brackets = null;
		
		
		match(IDENT);
		brackets=declaratorBrackets();
		complexity=varInitializer();
		return complexity;
	}
	
	protected final boolean  annMemberValue(
		boolean isSuppressWarnings, boolean makeArrayAndAdd
	) throws RecognitionException, TokenStreamException {
		boolean wasFallthroughLiteral;
		
		
		wasFallthroughLiteral=annMemberValue2(null, null, isSuppressWarnings, makeArrayAndAdd);
		return wasFallthroughLiteral;
	}
	
	public final void constructorBody(
		MethodSignature signature, CloverToken start, CloverToken endSig
	) throws RecognitionException, TokenStreamException {
		
		Token  lc = null;
		Token  rc = null;
		
		CloverToken endOfInv = null;
		CloverToken tmp;
		
		
		if ( inputState.guessing==0 ) {
			
			enterContext(ContextStore.CONTEXT_CTOR);
			
		}
		lc = LT(1);
		match(LCURLY);
		{
		boolean synPredMatched212 = false;
		if (((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LBRACK||LA(2)==METHOD_REF||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
			int _m212 = mark();
			synPredMatched212 = true;
			inputState.guessing++;
			try {
				{
				explicitConstructorInvocation();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched212 = false;
			}
			rewind(_m212);
inputState.guessing--;
		}
		if ( synPredMatched212 ) {
			endOfInv=explicitConstructorInvocation();
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DO||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==ABSTRACT||LA(2)==ASSERT||LA(2)==BOOLEAN||LA(2)==BREAK||LA(2)==BYTE||LA(2)==CHAR||LA(2)==CLASS||LA(2)==CONTINUE||LA(2)==DEFAULT||LA(2)==DO||LA(2)==DOUBLE||LA(2)==ENUM||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==FOR||LA(2)==IF||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==INTERFACE||LA(2)==LONG||LA(2)==NATIVE||LA(2)==NEW||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==RETURN||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==SYNCHRONIZED||LA(2)==THIS||LA(2)==THROW||LA(2)==TRANSIENT||LA(2)==TRY||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==WHILE||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==LCURLY||LA(2)==RCURLY||LA(2)==COLON||LA(2)==METHOD_REF||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==DIV_ASSIGN||LA(2)==PLUS||LA(2)==PLUS_ASSIGN||LA(2)==INC||LA(2)==MINUS||LA(2)==MINUS_ASSIGN||LA(2)==DEC||LA(2)==STAR||LA(2)==STAR_ASSIGN||LA(2)==MOD||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==SL_ASSIGN||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		_loop214:
		do {
			if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DO||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
				tmp=statement(null);
			}
			else {
				break _loop214;
			}
			
		} while (true);
		}
		rc = LT(1);
		match(RCURLY);
		if ( inputState.guessing==0 ) {
			
			// special case for instrumenting entry to ctors - HACK add ctor sig for completeness
			MethodEntryInstrEmitter entry = instrEnterMethod(signature, start, ct(lc), endOfInv);
			instrExitMethod(entry, ct(rc));
			exitContext();
			fileInfo.addMethodMarker(entry, start, endSig, ct(rc));
			
		}
	}
	
	public final void outerCompoundStmt(
		MethodSignature sig, CloverToken start, CloverToken endSig, int context
	) throws RecognitionException, TokenStreamException {
		
		Token  lc = null;
		Token  rc = null;
		
		CloverToken tmp;
		
		
		if ( inputState.guessing==0 ) {
			
			enterContext(context);
			
		}
		lc = LT(1);
		match(LCURLY);
		{
		_loop263:
		do {
			if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DO||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
				tmp=statement(null);
			}
			else {
				break _loop263;
			}
			
		} while (true);
		}
		rc = LT(1);
		match(RCURLY);
		if ( inputState.guessing==0 ) {
			
			MethodEntryInstrEmitter entry = instrEnterMethod(sig, start, ct(lc));
			instrExitMethod(entry, ct(rc));
			exitContext();
			if (context == ContextStore.CONTEXT_METHOD) {
			fileInfo.addMethodMarker(entry, start, endSig, ct(rc));
			}
			
		}
	}
	
	public final CloverToken  explicitConstructorInvocation() throws RecognitionException, TokenStreamException {
		CloverToken t;
		
		Token  pos1 = null;
		Token  t1 = null;
		Token  pos2 = null;
		Token  lp2 = null;
		Token  t2 = null;
		Token  pos3 = null;
		Token  lp3 = null;
		Token  t3 = null;
		
		ContextSetAndComplexity cc = null;
		int argListComplexity = 0;
		t = null;
		
		
		{
		if ((LA(1)==THIS) && (LA(2)==LPAREN)) {
			pos1 = LT(1);
			match(THIS);
			match(LPAREN);
			argListComplexity=argList();
			match(RPAREN);
			t1 = LT(1);
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				t=instrInlineAfter(ct(t1), ct(pos1), ct(t1), argListComplexity);
				
			}
		}
		else if ((LA(1)==SUPER) && (LA(2)==LPAREN)) {
			pos2 = LT(1);
			match(SUPER);
			lp2 = LT(1);
			match(LPAREN);
			argListComplexity=argList();
			match(RPAREN);
			t2 = LT(1);
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				t=instrInlineAfter(ct(t2), ct(pos2), ct(t2), argListComplexity);
				
			}
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LBRACK||LA(2)==METHOD_REF||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
			cc=primaryExpressionPart();
			{
			if ((LA(1)==DOT) && (LA(2)==THIS)) {
				match(DOT);
				match(THIS);
			}
			else if ((LA(1)==DOT) && (LA(2)==SUPER)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			match(DOT);
			pos3 = LT(1);
			match(SUPER);
			lp3 = LT(1);
			match(LPAREN);
			argListComplexity=argList();
			match(RPAREN);
			t3 = LT(1);
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				t=instrInlineAfter(ct(t3), ct(pos3), ct(t3), argListComplexity);
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return t;
	}
	
	public final CloverToken  statement(
		CloverToken owningLabel
	) throws RecognitionException, TokenStreamException {
		CloverToken last;
		
		Token  colon = null;
		Token  semi = null;
		Token  se1 = null;
		Token  se2 = null;
		Token  rp1 = null;
		Token  el = null;
		Token  rp = null;
		Token  rp2 = null;
		Token  d1 = null;
		Token  rp3 = null;
		Token  sem = null;
		
		CloverToken first = null;
		CloverToken tmp = null;
		CloverToken labelTok = null;
		boolean assertColonPart = false;
		last = null;
		boolean labelled = (owningLabel != null);
		FlagDeclEmitter flag = null;
		boolean matchable = true;
		boolean instrumentable = !labelled;
		CloverToken instr = null; // instr point for statement
		CloverToken flushAfter = null; // if not null, add maybeFlush instr
		int complexity = 0; // complexity of the entire statement
		int declComplexity, exprComplexity = 0, forInitComp = 0, forCondComp = 0, forIterComp = 0;
		boolean wasDefault = false;
		Modifiers mods = null;
		Parameter parameter = null;
		String classname = null;
		ContextSet saveContext = getCurrentContext();
		ContextSetAndComplexity contextAndComplexity = null;
		
		
		if ( inputState.guessing==0 ) {
			
			first = lt(1);
			
		}
		{
		if ((LA(1)==ASSERT)) {
			match(ASSERT);
			if ( inputState.guessing==0 ) {
				
				enterContext(ContextStore.CONTEXT_ASSERT);
				instrumentable = false;
				saveContext = getCurrentContext();
				
			}
			if ( inputState.guessing==0 ) {
				
				tmp=lt(1);
				
			}
			exprComplexity=expression();
			{
			if ((LA(1)==COLON)) {
				colon = LT(1);
				match(COLON);
				if ( inputState.guessing==0 ) {
					
					instrBoolExpr(tmp, ct(colon));
					assertColonPart=true;
					
				}
				exprComplexity=expression();
			}
			else if ((LA(1)==SEMI)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			semi = LT(1);
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				if (!assertColonPart) {
				instrBoolExpr(tmp, ct(semi));
				}
				exitContext();
				
			}
		}
		else if ((LA(1)==LCURLY)) {
			tmp=compoundStatement();
			if ( inputState.guessing==0 ) {
				
				matchable = false;
				instrumentable = false;
				
			}
		}
		else {
			boolean synPredMatched268 = false;
			if (((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
				int _m268 = mark();
				synPredMatched268 = true;
				inputState.guessing++;
				try {
					{
					declaration();
					}
				}
				catch (RecognitionException pe) {
					synPredMatched268 = false;
				}
				rewind(_m268);
inputState.guessing--;
			}
			if ( synPredMatched268 ) {
				declComplexity=declaration();
				se1 = LT(1);
				match(SEMI);
				if ( inputState.guessing==0 ) {
					
					complexity += declComplexity;
					flushAfter = ct(se1);
					
				}
			}
			else {
				boolean synPredMatched270 = false;
				if ((((LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==IDENT))&&( isNextKeyword("yield") ))) {
					int _m270 = mark();
					synPredMatched270 = true;
					inputState.guessing++;
					try {
						{
						if (!( isNextKeyword("yield") ))
						  throw new SemanticException(" isNextKeyword(\"yield\") ");
						match(IDENT);
						expression();
						match(SEMI);
						}
					}
					catch (RecognitionException pe) {
						synPredMatched270 = false;
					}
					rewind(_m270);
inputState.guessing--;
				}
				if ( synPredMatched270 ) {
					match(IDENT);
					exprComplexity=expression();
					match(SEMI);
					if ( inputState.guessing==0 ) {
						
						complexity += exprComplexity;
						
					}
				}
				else {
					boolean synPredMatched272 = false;
					if (((LA(1)==ABSTRACT||LA(1)==FINAL||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==FINAL||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==AT||LA(2)==IDENT))) {
						int _m272 = mark();
						synPredMatched272 = true;
						inputState.guessing++;
						try {
							{
							classOrInterfaceModifiers(false);
							if (!( isCurrentKeyword("record") ))
							  throw new SemanticException(" isCurrentKeyword(\"record\") ");
							}
						}
						catch (RecognitionException pe) {
							synPredMatched272 = false;
						}
						rewind(_m272);
inputState.guessing--;
					}
					if ( synPredMatched272 ) {
						mods=classOrInterfaceModifiers(false);
						classname=recordDefinition(mods);
						if ( inputState.guessing==0 ) {
							
							instrumentable = false;
							//TODO - return last token
							
						}
					}
					else {
						boolean synPredMatched274 = false;
						if (((LA(1)==ABSTRACT||LA(1)==CLASS||LA(1)==FINAL||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==NON_SEALED||LA(1)==AT||LA(1)==IDENT) && (LA(2)==ABSTRACT||LA(2)==CLASS||LA(2)==FINAL||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==NON_SEALED||LA(2)==AT||LA(2)==IDENT))) {
							int _m274 = mark();
							synPredMatched274 = true;
							inputState.guessing++;
							try {
								{
								classOrInterfaceModifiers(false);
								match(CLASS);
								}
							}
							catch (RecognitionException pe) {
								synPredMatched274 = false;
							}
							rewind(_m274);
inputState.guessing--;
						}
						if ( synPredMatched274 ) {
							mods=classOrInterfaceModifiers(false);
							classname=classDefinition(mods);
							if ( inputState.guessing==0 ) {
								
								instrumentable = false;
								//TODO - return last token
								
							}
						}
						else {
							boolean synPredMatched276 = false;
							if (((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==METHOD_REF||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==DIV_ASSIGN||LA(2)==PLUS||LA(2)==PLUS_ASSIGN||LA(2)==INC||LA(2)==MINUS||LA(2)==MINUS_ASSIGN||LA(2)==DEC||LA(2)==STAR||LA(2)==STAR_ASSIGN||LA(2)==MOD||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==SL_ASSIGN||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
								int _m276 = mark();
								synPredMatched276 = true;
								inputState.guessing++;
								try {
									{
									expression();
									match(SEMI);
									}
								}
								catch (RecognitionException pe) {
									synPredMatched276 = false;
								}
								rewind(_m276);
inputState.guessing--;
							}
							if ( synPredMatched276 ) {
								exprComplexity=expression();
								se2 = LT(1);
								match(SEMI);
								if ( inputState.guessing==0 ) {
									
									flushAfter = ct(se2);
									complexity += exprComplexity;
									
								}
							}
							else if ((LA(1)==IDENT) && (LA(2)==COLON)) {
								match(IDENT);
								match(COLON);
								if ( inputState.guessing==0 ) {
									
									labelTok = owningLabel;
									if (!labelled) {
									labelTok = first;
									}
									
								}
								last=statement(labelTok);
							}
							else if ((LA(1)==IF)) {
								match(IF);
								if ( inputState.guessing==0 ) {
									
									enterContext(ContextStore.CONTEXT_IF);
									saveContext = getCurrentContext();
									
								}
								match(LPAREN);
								if ( inputState.guessing==0 ) {
									
									tmp=lt(1);
									
								}
								exprComplexity=expression();
								if ( inputState.guessing==0 ) {
									
									complexity += exprComplexity;
									
								}
								rp1 = LT(1);
								match(RPAREN);
								if ( inputState.guessing==0 ) {
									
									instrBoolExpr(tmp, ct(rp1));
									addOpenBraceAfter(ct(rp1));
									
								}
								last=statement(null);
								if ( inputState.guessing==0 ) {
									
									addCloseBraceAfter(last);
									exitContext();
									
								}
								{
								if ((LA(1)==ELSE) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==ABSTRACT||LA(2)==ASSERT||LA(2)==BOOLEAN||LA(2)==BREAK||LA(2)==BYTE||LA(2)==CHAR||LA(2)==CLASS||LA(2)==CONTINUE||LA(2)==DO||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==FOR||LA(2)==IF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==RETURN||LA(2)==SHORT||LA(2)==STATIC||LA(2)==STRICTFP||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==SYNCHRONIZED||LA(2)==THIS||LA(2)==THROW||LA(2)==TRANSIENT||LA(2)==TRY||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==WHILE||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LCURLY||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT)) {
									el = LT(1);
									match(ELSE);
									if ( inputState.guessing==0 ) {
										
										addOpenBraceAfter(ct(el));
										enterContext(ContextStore.CONTEXT_ELSE);
										saveContext = getCurrentContext();
										
									}
									last=statement(null);
									if ( inputState.guessing==0 ) {
										
										addCloseBraceAfter(last);
										exitContext();
										
									}
								}
								else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CASE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DEFAULT||LA(1)==DO||LA(1)==DOUBLE||LA(1)==ELSE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT) && (_tokenSet_1.member(LA(2)))) {
								}
								else {
									throw new NoViableAltException(LT(1), getFilename());
								}
								
								}
							}
							else if ((LA(1)==FOR)) {
								match(FOR);
								if ( inputState.guessing==0 ) {
									
									enterContext(ContextStore.CONTEXT_FOR);
									saveContext = getCurrentContext();
									
								}
								match(LPAREN);
								{
								boolean synPredMatched280 = false;
								if (((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==ELLIPSIS||LA(2)==DOT))) {
									int _m280 = mark();
									synPredMatched280 = true;
									inputState.guessing++;
									try {
										{
										parameterDeclaration();
										match(COLON);
										}
									}
									catch (RecognitionException pe) {
										synPredMatched280 = false;
									}
									rewind(_m280);
inputState.guessing--;
								}
								if ( synPredMatched280 ) {
									{
									parameter=parameterDeclaration();
									match(COLON);
									exprComplexity=expression();
									if ( inputState.guessing==0 ) {
										
										complexity += exprComplexity;
										
									}
									}
								}
								else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==DIV_ASSIGN||LA(2)==PLUS||LA(2)==PLUS_ASSIGN||LA(2)==INC||LA(2)==MINUS||LA(2)==MINUS_ASSIGN||LA(2)==DEC||LA(2)==STAR||LA(2)==STAR_ASSIGN||LA(2)==MOD||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==SL_ASSIGN||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
									{
									forInitComp=forInit();
									match(SEMI);
									forCondComp=forCond();
									forIterComp=forIter();
									if ( inputState.guessing==0 ) {
										
										complexity += forInitComp + forCondComp + forIterComp;
										
									}
									}
								}
								else {
									throw new NoViableAltException(LT(1), getFilename());
								}
								
								}
								rp = LT(1);
								match(RPAREN);
								if ( inputState.guessing==0 ) {
									
									addOpenBraceAfter(ct(rp));
									
								}
								last=statement(null);
								if ( inputState.guessing==0 ) {
									
									addCloseBraceAfter(last);
									exitContext();
									
								}
							}
							else if ((LA(1)==WHILE)) {
								match(WHILE);
								if ( inputState.guessing==0 ) {
									
									enterContext(ContextStore.CONTEXT_WHILE);
									saveContext = getCurrentContext();
									
								}
								match(LPAREN);
								if ( inputState.guessing==0 ) {
									
									tmp = lt(1);
									
								}
								exprComplexity=expression();
								if ( inputState.guessing==0 ) {
									
									complexity += exprComplexity;
									
								}
								rp2 = LT(1);
								match(RPAREN);
								if ( inputState.guessing==0 ) {
									
									instrBoolExpr(tmp, ct(rp2));
									addOpenBraceAfter(ct(rp2));
									
								}
								last=statement(null);
								if ( inputState.guessing==0 ) {
									
									addCloseBraceAfter(last);
									exitContext();
									
								}
							}
							else if ((LA(1)==DO)) {
								d1 = LT(1);
								match(DO);
								if ( inputState.guessing==0 ) {
									
									addOpenBraceAfter(ct(d1));
									enterContext(ContextStore.CONTEXT_DO);
									saveContext = getCurrentContext();
									
								}
								tmp=statement(null);
								if ( inputState.guessing==0 ) {
									
									addCloseBraceAfter(tmp);
									exitContext();
									
								}
								match(WHILE);
								match(LPAREN);
								if ( inputState.guessing==0 ) {
									
									tmp=lt(1);
									
								}
								exprComplexity=expression();
								if ( inputState.guessing==0 ) {
									
									complexity += exprComplexity;
									
								}
								rp3 = LT(1);
								match(RPAREN);
								if ( inputState.guessing==0 ) {
									
									instrBoolExpr(tmp, ct(rp3));
									
								}
								sem = LT(1);
								match(SEMI);
								if ( inputState.guessing==0 ) {
									
									flushAfter = ct(sem);
									
								}
							}
							else if ((LA(1)==BREAK)) {
								match(BREAK);
								{
								if ((LA(1)==IDENT)) {
									match(IDENT);
								}
								else if ((LA(1)==SEMI)) {
								}
								else {
									throw new NoViableAltException(LT(1), getFilename());
								}
								
								}
								match(SEMI);
							}
							else if ((LA(1)==CONTINUE)) {
								match(CONTINUE);
								{
								if ((LA(1)==IDENT)) {
									match(IDENT);
								}
								else if ((LA(1)==SEMI)) {
								}
								else {
									throw new NoViableAltException(LT(1), getFilename());
								}
								
								}
								match(SEMI);
							}
							else if ((LA(1)==RETURN)) {
								match(RETURN);
								{
								if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
									exprComplexity=expression();
									if ( inputState.guessing==0 ) {
										
										complexity += exprComplexity;
										
									}
								}
								else if ((LA(1)==SEMI)) {
								}
								else {
									throw new NoViableAltException(LT(1), getFilename());
								}
								
								}
								match(SEMI);
							}
							else {
								boolean synPredMatched288 = false;
								if (((LA(1)==SWITCH) && (LA(2)==LPAREN))) {
									int _m288 = mark();
									synPredMatched288 = true;
									inputState.guessing++;
									try {
										{
										match(SWITCH);
										match(LPAREN);
										expression();
										match(RPAREN);
										match(LCURLY);
										{
										if ((LA(1)==CASE)) {
											match(CASE);
											expression();
										}
										else if ((LA(1)==DEFAULT)) {
											match(DEFAULT);
										}
										else {
											throw new NoViableAltException(LT(1), getFilename());
										}
										
										}
										match(COLON);
										}
									}
									catch (RecognitionException pe) {
										synPredMatched288 = false;
									}
									rewind(_m288);
inputState.guessing--;
								}
								if ( synPredMatched288 ) {
									contextAndComplexity=colonSwitchExpression(owningLabel, false);
									if ( inputState.guessing==0 ) {
										
										saveContext = contextAndComplexity.getContext();
										complexity += contextAndComplexity.getComplexity();
										
									}
								}
								else {
									boolean synPredMatched291 = false;
									if (((LA(1)==SWITCH) && (LA(2)==LPAREN))) {
										int _m291 = mark();
										synPredMatched291 = true;
										inputState.guessing++;
										try {
											{
											match(SWITCH);
											match(LPAREN);
											expression();
											match(RPAREN);
											match(LCURLY);
											{
											if ((LA(1)==CASE)) {
												match(CASE);
												patternMatch();
											}
											else if ((LA(1)==DEFAULT)) {
												match(DEFAULT);
											}
											else {
												throw new NoViableAltException(LT(1), getFilename());
											}
											
											}
											match(LAMBDA);
											}
										}
										catch (RecognitionException pe) {
											synPredMatched291 = false;
										}
										rewind(_m291);
inputState.guessing--;
									}
									if ( synPredMatched291 ) {
										contextAndComplexity=lambdaSwitchExpression(owningLabel);
										if ( inputState.guessing==0 ) {
											
											saveContext = contextAndComplexity.getContext();
											complexity += contextAndComplexity.getComplexity();
											
										}
									}
									else {
										boolean synPredMatched293 = false;
										if (((LA(1)==TRY))) {
											int _m293 = mark();
											synPredMatched293 = true;
											inputState.guessing++;
											try {
												{
												tryCatchBlock(labelled);
												}
											}
											catch (RecognitionException pe) {
												synPredMatched293 = false;
											}
											rewind(_m293);
inputState.guessing--;
										}
										if ( synPredMatched293 ) {
											last=tryCatchBlock(labelled);
											if ( inputState.guessing==0 ) {
												
												instrumentable = false; /* instrumentation happens in the tryCatchBlock rule */
												
											}
										}
										else if ((LA(1)==THROW)) {
											match(THROW);
											exprComplexity=expression();
											match(SEMI);
											if ( inputState.guessing==0 ) {
												
												complexity += exprComplexity;
												
											}
										}
										else if ((LA(1)==SYNCHRONIZED)) {
											match(SYNCHRONIZED);
											match(LPAREN);
											exprComplexity=expression();
											match(RPAREN);
											if ( inputState.guessing==0 ) {
												
												complexity += exprComplexity;
												enterContext(ContextStore.CONTEXT_SYNC);
												saveContext = getCurrentContext();
												
											}
											last=compoundStatement();
											if ( inputState.guessing==0 ) {
												
												exitContext();
												
											}
										}
										else if ((LA(1)==SEMI)) {
											match(SEMI);
										}
										else {
											throw new NoViableAltException(LT(1), getFilename());
										}
										}}}}}}}}
										}
										if ( inputState.guessing==0 ) {
											
											if (last == null) {
											last = lt(0);
											}
											
											if (instrumentable) {
											instrInlineBefore(first, last, saveContext, complexity);
											}
											
											if (flushAfter != null) {
											maybeAddFlushInstr(flushAfter);
											}
											
											if (matchable) {
											fileInfo.addStatementMarker(first, last);
											}
											
										}
										return last;
									}
									
	public final ContextSetAndComplexity  primaryExpressionPart() throws RecognitionException, TokenStreamException {
		ContextSetAndComplexity ret;
		
		
		String type = null;
		int tmpComplexity = 0;
		ret = ContextSetAndComplexity.empty();
		
		
		if ((LA(1)==IDENT)) {
			match(IDENT);
			if ( inputState.guessing==0 ) {
				
				pushIdentifierToHeadStack(LT(0).getText());
				
			}
		}
		else if (((LA(1) >= NUM_INT && LA(1) <= NUM_DOUBLE))) {
			constant();
		}
		else if ((LA(1)==TRUE)) {
			match(TRUE);
		}
		else if ((LA(1)==FALSE)) {
			match(FALSE);
		}
		else if ((LA(1)==THIS)) {
			match(THIS);
			if ( inputState.guessing==0 ) {
				
				pushIdentifierToHeadStack(LT(0).getText());
				
			}
		}
		else if ((LA(1)==NULL)) {
			match(NULL);
		}
		else if ((LA(1)==NEW)) {
			tmpComplexity=newExpression();
			if ( inputState.guessing==0 ) {
				
				ret = ContextSetAndComplexity.ofComplexity(tmpComplexity);
				
			}
		}
		else if ((LA(1)==LPAREN)) {
			match(LPAREN);
			tmpComplexity=assignmentExpression();
			match(RPAREN);
			if ( inputState.guessing==0 ) {
				
				ret = ContextSetAndComplexity.ofComplexity(tmpComplexity);
				
			}
		}
		else if ((LA(1)==SUPER)) {
			match(SUPER);
			if ( inputState.guessing==0 ) {
				
				pushIdentifierToHeadStack(LT(0).getText());
				
			}
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
			type=builtInType();
			{
			_loop424:
			do {
				if ((LA(1)==LBRACK)) {
					match(LBRACK);
					match(RBRACK);
				}
				else {
					break _loop424;
				}
				
			} while (true);
			}
			{
			if ((LA(1)==DOT)) {
				match(DOT);
				match(CLASS);
			}
			else if ((LA(1)==METHOD_REF)) {
				match(METHOD_REF);
				match(NEW);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
		}
		else if ((LA(1)==NON_SEALED)) {
			match(NON_SEALED);
		}
		else {
			boolean synPredMatched428 = false;
			if (((LA(1)==SWITCH) && (LA(2)==LPAREN))) {
				int _m428 = mark();
				synPredMatched428 = true;
				inputState.guessing++;
				try {
					{
					match(SWITCH);
					match(LPAREN);
					expression();
					match(RPAREN);
					match(LCURLY);
					{
					if ((LA(1)==CASE)) {
						match(CASE);
						patternMatch();
					}
					else if ((LA(1)==DEFAULT)) {
						match(DEFAULT);
					}
					else {
						throw new NoViableAltException(LT(1), getFilename());
					}
					
					}
					match(LAMBDA);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched428 = false;
				}
				rewind(_m428);
inputState.guessing--;
			}
			if ( synPredMatched428 ) {
				ret=lambdaSwitchExpression(null);
			}
			else {
				boolean synPredMatched431 = false;
				if (((LA(1)==SWITCH) && (LA(2)==LPAREN))) {
					int _m431 = mark();
					synPredMatched431 = true;
					inputState.guessing++;
					try {
						{
						match(SWITCH);
						match(LPAREN);
						expression();
						match(RPAREN);
						match(LCURLY);
						{
						if ((LA(1)==CASE)) {
							match(CASE);
							expression();
						}
						else if ((LA(1)==DEFAULT)) {
							match(DEFAULT);
						}
						else {
							throw new NoViableAltException(LT(1), getFilename());
						}
						
						}
						match(COLON);
						}
					}
					catch (RecognitionException pe) {
						synPredMatched431 = false;
					}
					rewind(_m431);
inputState.guessing--;
				}
				if ( synPredMatched431 ) {
					ret=colonSwitchExpression(null, true);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				}}
				return ret;
			}
			
	public final int  varInitializer() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		complexity = 0;
		
		
		{
		if ((LA(1)==ASSIGN)) {
			match(ASSIGN);
			complexity=initializer();
		}
		else if ((LA(1)==RPAREN||LA(1)==COMMA||LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
	public final int  initializer() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		complexity = 0;
		
		
		if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
			complexity=expression();
		}
		else if ((LA(1)==LCURLY)) {
			complexity=arrayInitializer();
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return complexity;
	}
	
	public final int  arrayInitializer() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int initComplexity;
		complexity = 0;
		
		
		match(LCURLY);
		{
		if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
			initComplexity=initializer();
			if ( inputState.guessing==0 ) {
				
				complexity += initComplexity;
				
			}
			{
			_loop232:
			do {
				if ((LA(1)==COMMA) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LCURLY||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==IDENT)) {
					match(COMMA);
					initComplexity=initializer();
					if ( inputState.guessing==0 ) {
						
						complexity += initComplexity;
						
					}
				}
				else {
					break _loop232;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==RCURLY||LA(1)==COMMA)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==COMMA)) {
			match(COMMA);
		}
		else if ((LA(1)==RCURLY)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		match(RCURLY);
		return complexity;
	}
	
	public final int  expression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		if ( inputState.guessing==0 ) {
			
			pushHeadIdentifierStack();
			
		}
		complexity=assignmentExpression();
		if ( inputState.guessing==0 ) {
			
			popHeadIdentifierStack();
			
		}
		return complexity;
	}
	
	public final Parameter  parameterDeclaration() throws RecognitionException, TokenStreamException {
		Parameter parameter;
		
		Token  i = null;
		
		Parameter param = null;
		String brackets = "";
		String type=null;
		parameter=null;
		
		
		parameterModifier();
		type=typeSpec();
		{
		if ((LA(1)==ELLIPSIS)) {
			match(ELLIPSIS);
		}
		else if ((LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		i = LT(1);
		match(IDENT);
		brackets=declaratorBrackets();
		if ( inputState.guessing==0 ) {
			
			parameter = new Parameter(type+brackets,i.getText());
			
		}
		return parameter;
	}
	
	public final void parameterModifier() throws RecognitionException, TokenStreamException {
		
		Token  f = null;
		
		AnnotationImpl ann = null;
		
		
		{
		_loop250:
		do {
			if ((LA(1)==AT) && (LA(2)==IDENT)) {
				ann=annotation();
			}
			else {
				break _loop250;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==FINAL)) {
			f = LT(1);
			match(FINAL);
			{
			_loop253:
			do {
				if ((LA(1)==AT) && (LA(2)==IDENT)) {
					ann=annotation();
				}
				else {
					break _loop253;
				}
				
			} while (true);
			}
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
/**
 * A list of declared parameters (one or more) with unspecified types, used in argument list of a lambda function,
 * for example:
 *   <code>x, y, z</code>
 */
	public final Parameter[]  implicitParameterDeclarationList() throws RecognitionException, TokenStreamException {
		Parameter[] parameters;
		
		
		List<Parameter> parameterList = new ArrayList<Parameter>();
		Parameter param = null;
		parameters = new Parameter[0];
		
		
		param=implicitParameterDeclaration();
		if ( inputState.guessing==0 ) {
			
			parameterList.add(param);
			
		}
		{
		_loop256:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				param=implicitParameterDeclaration();
				if ( inputState.guessing==0 ) {
					
					parameterList.add(param);
					
				}
			}
			else {
				break _loop256;
			}
			
		} while (true);
		}
		if ( inputState.guessing==0 ) {
			
			parameters = parameterList.toArray(new Parameter[parameterList.size()]);
			
		}
		return parameters;
	}
	
/**
 * A parameter with an undefined type, used in lambda function argument list
 */
	public final Parameter  implicitParameterDeclaration() throws RecognitionException, TokenStreamException {
		Parameter parameter;
		
		Token  i = null;
		
		parameter = null;
		
		
		i = LT(1);
		match(IDENT);
		if ( inputState.guessing==0 ) {
			
			parameter = new Parameter(Parameter.INFERRED, i.getText());
			
		}
		return parameter;
	}
	
	public final CloverToken  compoundStatement() throws RecognitionException, TokenStreamException {
		CloverToken t;
		
		Token  rc = null;
		
		t = null;
		
		
		match(LCURLY);
		{
		_loop260:
		do {
			if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DO||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
				t=statement(null);
			}
			else {
				break _loop260;
			}
			
		} while (true);
		}
		rc = LT(1);
		match(RCURLY);
		if ( inputState.guessing==0 ) {
			
			t = ct(rc);
			
		}
		return t;
	}
	
	public final int  forInit() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		complexity = 0;
		
		
		{
		boolean synPredMatched305 = false;
		if (((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
			int _m305 = mark();
			synPredMatched305 = true;
			inputState.guessing++;
			try {
				{
				declaration();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched305 = false;
			}
			rewind(_m305);
inputState.guessing--;
		}
		if ( synPredMatched305 ) {
			complexity=declaration();
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==DIV_ASSIGN||LA(2)==PLUS||LA(2)==PLUS_ASSIGN||LA(2)==INC||LA(2)==MINUS||LA(2)==MINUS_ASSIGN||LA(2)==DEC||LA(2)==STAR||LA(2)==STAR_ASSIGN||LA(2)==MOD||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==SL_ASSIGN||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
			complexity=expressionList();
		}
		else if ((LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
	public final int  forCond() throws RecognitionException, TokenStreamException {
		int complexity;
		
		Token  se = null;
		
		CloverToken tmp = null;
		complexity = 0;
		
		
		if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
			{
			if ( inputState.guessing==0 ) {
				
				tmp=lt(1);
				
			}
			complexity=expression();
			se = LT(1);
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				instrBoolExpr(tmp, ct(se));
				
			}
			}
		}
		else if ((LA(1)==SEMI)) {
			match(SEMI);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return complexity;
	}
	
	public final int  forIter() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		complexity = 0;
		
		
		{
		if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
			complexity=expressionList();
		}
		else if ((LA(1)==RPAREN)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
/**
 * A switch statement or expression containing one or more "case :" or "default :" conditions.
 * @param owningLabel a label before switch or null if not present
 * @param isInsideExpression true if the switch is part of an expression, false if is a standalone statement
 */
	public final ContextSetAndComplexity  colonSwitchExpression(
		CloverToken owningLabel, boolean isInsideExpression
	) throws RecognitionException, TokenStreamException {
		ContextSetAndComplexity ret;
		
		Token  sw = null;
		Token  rc = null;
		
		CloverToken tmp = null;
		boolean labelled = (owningLabel != null);
		FlagDeclEmitter flag = null;
		ret = ContextSetAndComplexity.empty();
		int casesGroupComplexity, exprComplexity;
		
		
		sw = LT(1);
		match(SWITCH);
		if ( inputState.guessing==0 ) {
			
			tmp = ct(sw);
			if (labelled) {
			tmp = owningLabel;
			}
			if (!isInsideExpression) {
			flag = declareFlagBefore(tmp);
			}
			enterContext(ContextStore.CONTEXT_SWITCH);
			ret.setContext(getCurrentContext());
			
		}
		match(LPAREN);
		exprComplexity=expression();
		match(RPAREN);
		match(LCURLY);
		if ( inputState.guessing==0 ) {
			
			ret.addComplexity(exprComplexity);
			
		}
		{
		_loop434:
		do {
			if ((LA(1)==CASE||LA(1)==DEFAULT)) {
				casesGroupComplexity=colonCasesGroup(flag);
				if ( inputState.guessing==0 ) {
					
					ret.addComplexity(casesGroupComplexity);
					
				}
			}
			else {
				break _loop434;
			}
			
		} while (true);
		}
		if ( inputState.guessing==0 ) {
			
			exitContext();
			
		}
		rc = LT(1);
		match(RCURLY);
		return ret;
	}
	
	public final void patternMatch() throws RecognitionException, TokenStreamException {
		
		
		constantExpression();
		{
		_loop445:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				constantExpression();
			}
			else {
				break _loop445;
			}
			
		} while (true);
		}
	}
	
/**
 * A switch statement or expression containing one or more "case ->" or "default ->" conditions.
 */
	public final ContextSetAndComplexity  lambdaSwitchExpression(
		CloverToken owningLabel
	) throws RecognitionException, TokenStreamException {
		ContextSetAndComplexity ret;
		
		Token  sw = null;
		Token  rc = null;
		
		// every switch must have at least one case/default branch, cyclomatic complexity is number of branches minus 1
		// every case/default increases it by one, cyclomatic complexity of only one branch is zero
		int caseComplexity = -1;
		int exprComplexity = 0;
		ContextSet saveContext = getCurrentContext();
		CloverToken tmp = null;
		ret = ContextSetAndComplexity.ofComplexity(caseComplexity);
		
		
		sw = LT(1);
		match(SWITCH);
		if ( inputState.guessing==0 ) {
			
			tmp = ct(sw);
			enterContext(ContextStore.CONTEXT_SWITCH);
			ret.setContext(getCurrentContext());
			
		}
		match(LPAREN);
		exprComplexity=expression();
		match(RPAREN);
		match(LCURLY);
		if ( inputState.guessing==0 ) {
			
			ret.addComplexity(exprComplexity);
			
		}
		{
		int _cnt437=0;
		_loop437:
		do {
			if ((LA(1)==CASE||LA(1)==DEFAULT)) {
				caseComplexity=lambdaCase(saveContext);
				if ( inputState.guessing==0 ) {
					
					ret.addComplexity(caseComplexity);
					
				}
			}
			else {
				if ( _cnt437>=1 ) { break _loop437; } else {throw new NoViableAltException(LT(1), getFilename());}
			}
			
			_cnt437++;
		} while (true);
		}
		if ( inputState.guessing==0 ) {
			
			exitContext();
			
		}
		rc = LT(1);
		match(RCURLY);
		return ret;
	}
	
	public final CloverToken  tryCatchBlock(
		boolean labelled
	) throws RecognitionException, TokenStreamException {
		CloverToken last;
		
		Token  tr = null;
		Token  lp = null;
		Token  semi = null;
		Token  rp = null;
		
		last = null;
		int declComplexity;
		int complexity = 0;
		ContextSet saveContext = getCurrentContext();
		
		
		tr = LT(1);
		match(TRY);
		{
		if ((LA(1)==LPAREN)) {
			lp = LT(1);
			match(LPAREN);
			if ( inputState.guessing==0 ) {
				
				insertAutoCloseableClassDecl(ct(tr));
				
			}
			{
			boolean synPredMatched314 = false;
			if (((LA(1)==IDENT) && (LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==ASSIGN||LA(2)==SEMI||LA(2)==AT))) {
				int _m314 = mark();
				synPredMatched314 = true;
				inputState.guessing++;
				try {
					{
					match(IDENT);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched314 = false;
				}
				rewind(_m314);
inputState.guessing--;
			}
			if ( synPredMatched314 ) {
				declComplexity=variableDeclarator();
			}
			else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
				declComplexity=declaration();
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			if ( inputState.guessing==0 ) {
				
				complexity += declComplexity;
				complexity++;
				instrArmDecl((ct(lp)).getNext(), lt(0), saveContext);
				
			}
			{
			_loop319:
			do {
				if ((LA(1)==SEMI) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==AT||LA(2)==IDENT)) {
					semi = LT(1);
					match(SEMI);
					{
					boolean synPredMatched318 = false;
					if (((LA(1)==IDENT) && (LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==ASSIGN||LA(2)==SEMI||LA(2)==AT))) {
						int _m318 = mark();
						synPredMatched318 = true;
						inputState.guessing++;
						try {
							{
							match(IDENT);
							}
						}
						catch (RecognitionException pe) {
							synPredMatched318 = false;
						}
						rewind(_m318);
inputState.guessing--;
					}
					if ( synPredMatched318 ) {
						declComplexity=variableDeclarator();
					}
					else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==SHORT||LA(1)==STATIC||LA(1)==TRANSIENT||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==PRIVATE||LA(2)==PROTECTED||LA(2)==PUBLIC||LA(2)==SHORT||LA(2)==STATIC||LA(2)==TRANSIENT||LA(2)==VOID||LA(2)==VOLATILE||LA(2)==LBRACK||LA(2)==LT||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
						declComplexity=declaration();
					}
					else {
						throw new NoViableAltException(LT(1), getFilename());
					}
					
					}
					if ( inputState.guessing==0 ) {
						
						complexity += declComplexity;
						complexity++;
						instrArmDecl((ct(semi)).getNext(), lt(0), saveContext);
						
					}
				}
				else {
					break _loop319;
				}
				
			} while (true);
			}
			{
			if ((LA(1)==SEMI)) {
				match(SEMI);
			}
			else if ((LA(1)==RPAREN)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			rp = LT(1);
			match(RPAREN);
		}
		else if ((LA(1)==LCURLY)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			enterContext(ContextStore.CONTEXT_TRY);
			saveContext = getCurrentContext();
			
		}
		last=compoundStatement();
		if ( inputState.guessing==0 ) {
			
			exitContext();
			
		}
		{
		_loop322:
		do {
			if ((LA(1)==CATCH)) {
				last=handler();
				if ( inputState.guessing==0 ) {
					
					complexity++;
					
				}
			}
			else {
				break _loop322;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==FINALLY)) {
			match(FINALLY);
			if ( inputState.guessing==0 ) {
				
				enterContext(ContextStore.CONTEXT_FINALLY); saveContext = getCurrentContext();
				
			}
			last=compoundStatement();
			if ( inputState.guessing==0 ) {
				
				exitContext();
				
			}
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CASE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DEFAULT||LA(1)==DO||LA(1)==DOUBLE||LA(1)==ELSE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==RCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			if (!labelled) {
			instrInlineBefore(ct(tr), last, saveContext, complexity);
			}
			
		}
		return last;
	}
	
/**
 * A group of one or more "case x:" labels, followed by a list of statements.
 */
	public final int  colonCasesGroup(
		FlagDeclEmitter flag
	) throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmp = 0;
		complexity = 0;
		
		
		{
		int _cnt296=0;
		_loop296:
		do {
			if ((LA(1)==CASE||LA(1)==DEFAULT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==DEFAULT||LA(2)==SUPER||LA(2)==THIS||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==LPAREN||LA(2)==COLON||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==MINUS||LA(2)==IDENT)) {
				tmp=colonCase(flag);
				if ( inputState.guessing==0 ) {
					
					complexity += tmp;
					
				}
			}
			else {
				if ( _cnt296>=1 ) { break _loop296; } else {throw new NoViableAltException(LT(1), getFilename());}
			}
			
			_cnt296++;
		} while (true);
		}
		caseStatementsList();
		return complexity;
	}
	
/**
 * A single "case x:" or "default:" label.
 */
	public final int  colonCase(
		FlagDeclEmitter flag
	) throws RecognitionException, TokenStreamException {
		int complexity;
		
		Token  si1 = null;
		Token  si2 = null;
		Token  t = null;
		
		Token pos = null;
		complexity = 0;
		
		
		{
		if ((LA(1)==CASE)) {
			si1 = LT(1);
			match(CASE);
			if ( inputState.guessing==0 ) {
				
				constExpr = true;
				
			}
			constantExpression();
			if ( inputState.guessing==0 ) {
				
				constExpr = false;
				pos = si1;
				complexity++;
				
			}
		}
		else if ((LA(1)==DEFAULT)) {
			si2 = LT(1);
			match(DEFAULT);
			if ( inputState.guessing==0 ) {
				
				pos = si2;
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		t = LT(1);
		match(COLON);
		if ( inputState.guessing==0 ) {
			
			// 0 cyclomatic complexity here as we pass it up to the switch statement
			if (flag != null) {
			instrInlineAfter(ct(t), ct(pos), ct(t), flag, 0);
			} else {
			instrInlineAfter(ct(t), ct(pos), ct(t), 0);
			}
			fileInfo.setSuppressFallthroughWarnings(true);
			
		}
		return complexity;
	}
	
/**
 * A list of statements inside a single "case" or "default" block.
 */
	public final void caseStatementsList() throws RecognitionException, TokenStreamException {
		
		
		CloverToken tmp;
		
		
		{
		_loop301:
		do {
			if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==ABSTRACT||LA(1)==ASSERT||LA(1)==BOOLEAN||LA(1)==BREAK||LA(1)==BYTE||LA(1)==CHAR||LA(1)==CLASS||LA(1)==CONTINUE||LA(1)==DO||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==FOR||LA(1)==IF||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==PRIVATE||LA(1)==PROTECTED||LA(1)==PUBLIC||LA(1)==RETURN||LA(1)==SHORT||LA(1)==STATIC||LA(1)==STRICTFP||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==SYNCHRONIZED||LA(1)==THIS||LA(1)==THROW||LA(1)==TRANSIENT||LA(1)==TRY||LA(1)==VOID||LA(1)==VOLATILE||LA(1)==WHILE||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==SEMI||LA(1)==AT||LA(1)==IDENT)) {
				tmp=statement(null);
			}
			else {
				break _loop301;
			}
			
		} while (true);
		}
	}
	
	public final void constantExpression() throws RecognitionException, TokenStreamException {
		
		
		constantConditionalExpression();
	}
	
	public final int  expressionList() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		complexity = 0;
		
		
		complexity=expression();
		{
		_loop358:
		do {
			if ((LA(1)==COMMA)) {
				match(COMMA);
				tmpComplexity=expression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop358;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final CloverToken  handler() throws RecognitionException, TokenStreamException {
		CloverToken last;
		
		
		AnnotationImpl an;
		String ts;
		last = null;
		
		
		match(CATCH);
		match(LPAREN);
		{
		_loop326:
		do {
			if ((LA(1)==AT) && (LA(2)==IDENT)) {
				an=annotation2(false);
			}
			else {
				break _loop326;
			}
			
		} while (true);
		}
		{
		if ((LA(1)==FINAL)) {
			match(FINAL);
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		_loop329:
		do {
			if ((LA(1)==AT) && (LA(2)==IDENT)) {
				an=annotation2(false);
			}
			else {
				break _loop329;
			}
			
		} while (true);
		}
		ts=typeSpec();
		{
		_loop331:
		do {
			if ((LA(1)==BOR)) {
				match(BOR);
				ts=typeSpec();
			}
			else {
				break _loop331;
			}
			
		} while (true);
		}
		match(IDENT);
		match(RPAREN);
		if ( inputState.guessing==0 ) {
			
			enterContext(ContextStore.CONTEXT_CATCH);
			
		}
		last=compoundStatement();
		if ( inputState.guessing==0 ) {
			
			exitContext();
			
		}
		return last;
	}
	
/**
 * A predicate which recognizes whether we have a lambda function or not. Predicate tries to parse tokens
 * up to the LAMBDA symbol.
 */
	public final void lambdaFunctionPredicate() throws RecognitionException, TokenStreamException {
		
		
		Parameter[] la;
		String type, typeExt;
		
		
		boolean synPredMatched336 = false;
		if (((LA(1)==LPAREN) && (LA(2)==AT||LA(2)==IDENT))) {
			int _m336 = mark();
			synPredMatched336 = true;
			inputState.guessing++;
			try {
				{
				match(LPAREN);
				classTypeSpec();
				{
				_loop335:
				do {
					if ((LA(1)==BAND)) {
						match(BAND);
						classOrInterfaceType();
					}
					else {
						break _loop335;
					}
					
				} while (true);
				}
				match(RPAREN);
				la=lambdaArgs();
				match(LAMBDA);
				}
			}
			catch (RecognitionException pe) {
				synPredMatched336 = false;
			}
			rewind(_m336);
inputState.guessing--;
		}
		if ( synPredMatched336 ) {
			match(LPAREN);
			type=classTypeSpec();
			{
			_loop338:
			do {
				if ((LA(1)==BAND)) {
					match(BAND);
					typeExt=classOrInterfaceType();
				}
				else {
					break _loop338;
				}
				
			} while (true);
			}
			match(RPAREN);
			la=lambdaArgs();
			match(LAMBDA);
		}
		else {
			boolean synPredMatched340 = false;
			if (((LA(1)==LPAREN||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==RPAREN||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT))) {
				int _m340 = mark();
				synPredMatched340 = true;
				inputState.guessing++;
				try {
					{
					la=lambdaArgs();
					match(LAMBDA);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched340 = false;
				}
				rewind(_m340);
inputState.guessing--;
			}
			if ( synPredMatched340 ) {
				la=lambdaArgs();
				match(LAMBDA);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			}
		}
		
/**
 * List of formal arguments for lambda function definition. Possible forms are:
 * 1) Empty list of arguments for a lambda function, i.e.:
 *     <code>()</code>
 * <p/>
 * 2) List of explicitly declared arguments' types, for instance:
 *     <code>(Integer x, String s)</code>
 * <p/>
 * 3) List of arguments for where types are undefined, for example:
 *     <pre>
 *         (x, y, z)    // few identifiers separated by comma, enclosed in parentheses
 *         (abc)        // one identifier, enclosed in parentheses
 *         i            // exactly one identifier, without parentheses
 *     </pre>
 */
	public final Parameter[]  lambdaArgs() throws RecognitionException, TokenStreamException {
		Parameter[] parameters;
		
		
		Parameter p;
		Parameter[] ps;
		parameters = new Parameter[0]; // empty list by default
		
		
		{
		if ((LA(1)==LPAREN)) {
			match(LPAREN);
			{
			if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FINAL||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==RPAREN||LA(1)==AT||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==LBRACK||LA(2)==LT||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT||LA(2)==ELLIPSIS||LA(2)==DOT)) {
				ps=parameterDeclarationList();
				if ( inputState.guessing==0 ) {
					parameters = ps;
				}
			}
			else if ((LA(1)==IDENT) && (LA(2)==RPAREN||LA(2)==COMMA)) {
				ps=implicitParameterDeclarationList();
				if ( inputState.guessing==0 ) {
					parameters = ps;
				}
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			match(RPAREN);
		}
		else if ((LA(1)==IDENT)) {
			p=implicitParameterDeclaration();
			if ( inputState.guessing==0 ) {
				
				parameters = new Parameter[1];
				parameters[0] = p;
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return parameters;
	}
	
/**
 * Lambda function in a form like:
 *   <code>(arguments) -> body</code>
 * <p/>
 * where body is in a form of a single statement or expression, e.g.:
 *   <code>System.out.println("Hello")</code>
 *   <code>x + y</code>
 * <p/>
 * or in a form of the code block, e.g.:
 *   <pre>
 *   {
 *      the;
 *      code;
 *      block;
 *   }
 *   </pre>
 * <p/>
 * or in a form of the method reference, e.g:
 *   <code>Math::abs</code>
 */
	public final CloverToken  lambdaFunction() throws RecognitionException, TokenStreamException {
		CloverToken last;
		
		
		CloverToken startLambdaArgs = null; // remembers where lambda's argument list begins
		CloverToken startLambdaBody = null; // remembers where the expression begins
		CloverToken cs = null; // unused
		Parameter[] la = null; // list of lambda's type arguments
		String type;           // optional class cast
		String typeExt;        // .. with extra interfaces concatenated by &
		MethodSignature lambdaSignature = null;
		LambdaExpressionEntryEmitter expressionEntryEmitter = null;
		LambdaBlockEntryEmitter blockEntryEmitter = null;
		LambdaExprToBlockStartEntryEmitter exprToBlockStartEntryEmitter = null;
		boolean exprToBlockHeuristic = false;
		
		last = null;
		
		/*
		* A pair of marker tokens used to remember a class cast. This pair of markers is used to solve type-inference for
		* lambdaInc wrapper. For example:
		* <pre>
		*   Object o = (Runnable) () -> System.out.println("lambda expression with class cast");
		* </pre>
		* will be wrapped into lambdaInc like this:
		* <pre>
		*   Object o = (Runnable) lambdaInc(123, () -> System.out.println("lambda expression with class cast"));
		* </pre>
		* The problem is that javac is unable to infer proper type of lambdaInc - it sees Object.
		*
		* In such case when Clover finds a lambda with a class cast, it will memorize class cast start/end and copy
		* it into a lambda wrapper:
		* <pre>
		*   Object o = (Runnable) lambdaInc(123, (Runnable)() -> System.out.println("lambda expression with class cast"));
		*         start^     end^                ^^^copy^^^
		* </pre>
		*/
		CloverToken classCastStart = null;
		CloverToken classCastEnd = null;
		int tmpComplexity = 0;
		
		
		{
		boolean synPredMatched346 = false;
		if (((LA(1)==LPAREN) && (LA(2)==AT||LA(2)==IDENT))) {
			int _m346 = mark();
			synPredMatched346 = true;
			inputState.guessing++;
			try {
				{
				match(LPAREN);
				type=classTypeSpec();
				{
				_loop345:
				do {
					if ((LA(1)==BAND)) {
						match(BAND);
						classOrInterfaceType();
					}
					else {
						break _loop345;
					}
					
				} while (true);
				}
				match(RPAREN);
				la=lambdaArgs();
				match(LAMBDA);
				}
			}
			catch (RecognitionException pe) {
				synPredMatched346 = false;
			}
			rewind(_m346);
inputState.guessing--;
		}
		if ( synPredMatched346 ) {
			if ( inputState.guessing==0 ) {
				classCastStart = lt(1);
			}
			match(LPAREN);
			type=classTypeSpec();
			{
			_loop348:
			do {
				if ((LA(1)==BAND)) {
					match(BAND);
					typeExt=classOrInterfaceType();
					if ( inputState.guessing==0 ) {
						type += "&" + typeExt;
					}
				}
				else {
					break _loop348;
				}
				
			} while (true);
			}
			match(RPAREN);
			if ( inputState.guessing==0 ) {
				classCastEnd = lt(0);
			}
			if ( inputState.guessing==0 ) {
				startLambdaArgs = lt(1);
			}
			la=lambdaArgs();
			match(LAMBDA);
		}
		else {
			boolean synPredMatched350 = false;
			if (((LA(1)==LPAREN||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==RPAREN||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT))) {
				int _m350 = mark();
				synPredMatched350 = true;
				inputState.guessing++;
				try {
					{
					la=lambdaArgs();
					match(LAMBDA);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched350 = false;
				}
				rewind(_m350);
inputState.guessing--;
			}
			if ( synPredMatched350 ) {
				if ( inputState.guessing==0 ) {
					startLambdaArgs = lt(1);
				}
				la=lambdaArgs();
				match(LAMBDA);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			}
			}
			if ( inputState.guessing==0 ) {
				
				lambdaSignature = new MethodSignature(LambdaUtil.generateLambdaNameWithId(la, lambdaCounter++),
				null, null, la, null, new Modifiers());
				
			}
			{
			if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
				if ( inputState.guessing==0 ) {
					
					// wrap entire lambda, including argument list and not just an expression body
					// pass also a preceding class cast if present (non-null)
					startLambdaBody = lt(1);
					exprToBlockHeuristic = RewriteLambdaToBlockMatcher.shouldRewriteAsBlock(identifiersStack);
					if (exprToBlockHeuristic) {
					exprToBlockStartEntryEmitter = instrEnterLambdaExprToBlockExpression(lambdaSignature, startLambdaArgs, startLambdaBody);
					} else {
					expressionEntryEmitter = instrEnterLambdaExpression(lambdaSignature, startLambdaArgs, startLambdaBody,
					classCastStart, classCastEnd);
					}
					
				}
				tmpComplexity=expression();
				if ( inputState.guessing==0 ) {
					
					if (exprToBlockHeuristic) {
					instrExitLambdaExprToBlockExpression(exprToBlockStartEntryEmitter, lt(0));
					} else {
					instrExitLambdaExpression(expressionEntryEmitter, lt(0));
					}
					
				}
			}
			else if ((LA(1)==LCURLY)) {
				if ( inputState.guessing==0 ) {
					
					// expected LT(1)=LCURLY
					blockEntryEmitter = instrEnterLambdaBlock(lambdaSignature, lt(1));
					
				}
				cs=compoundStatement();
				if ( inputState.guessing==0 ) {
					
					// expected LT(0)=RCURLY
					instrExitLambdaBlock(blockEntryEmitter, lt(0));
					
				}
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			return last;
		}
		
	public final int  assignmentExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int complexity1 = 0, complexity2 = 0;
		complexity = 0;
		
		
		complexity1=conditionalExpression();
		{
		if ((LA(1)==ASSIGN||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR_ASSIGN||LA(1)==BSR_ASSIGN||LA(1)==SL_ASSIGN||LA(1)==BXOR_ASSIGN||LA(1)==BOR_ASSIGN||LA(1)==BAND_ASSIGN) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==IDENT)) {
			{
			if ((LA(1)==ASSIGN)) {
				match(ASSIGN);
			}
			else if ((LA(1)==PLUS_ASSIGN)) {
				match(PLUS_ASSIGN);
			}
			else if ((LA(1)==MINUS_ASSIGN)) {
				match(MINUS_ASSIGN);
			}
			else if ((LA(1)==STAR_ASSIGN)) {
				match(STAR_ASSIGN);
			}
			else if ((LA(1)==DIV_ASSIGN)) {
				match(DIV_ASSIGN);
			}
			else if ((LA(1)==MOD_ASSIGN)) {
				match(MOD_ASSIGN);
			}
			else if ((LA(1)==SR_ASSIGN)) {
				match(SR_ASSIGN);
			}
			else if ((LA(1)==BSR_ASSIGN)) {
				match(BSR_ASSIGN);
			}
			else if ((LA(1)==SL_ASSIGN)) {
				match(SL_ASSIGN);
			}
			else if ((LA(1)==BAND_ASSIGN)) {
				match(BAND_ASSIGN);
			}
			else if ((LA(1)==BXOR_ASSIGN)) {
				match(BXOR_ASSIGN);
			}
			else if ((LA(1)==BOR_ASSIGN)) {
				match(BOR_ASSIGN);
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			complexity2=assignmentExpression();
		}
		else if ((LA(1)==RPAREN||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR_ASSIGN||LA(1)==BSR_ASSIGN||LA(1)==SL_ASSIGN||LA(1)==BXOR_ASSIGN||LA(1)==BOR_ASSIGN||LA(1)==BAND_ASSIGN||LA(1)==SEMI) && (_tokenSet_2.member(LA(2)))) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			complexity = complexity1 + complexity2;
			
		}
		return complexity;
	}
	
	public final int  conditionalExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		Token  endOfCond = null;
		
		CloverToken startOfCond = null;
		CloverToken lf = null;
		int complexity1 = 0, complexity2 = 0, complexity3 = 0;
		complexity = 0;
		
		
		boolean synPredMatched364 = false;
		if (((LA(1)==LPAREN||LA(1)==IDENT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==RPAREN||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT))) {
			int _m364 = mark();
			synPredMatched364 = true;
			inputState.guessing++;
			try {
				{
				lambdaFunctionPredicate();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched364 = false;
			}
			rewind(_m364);
inputState.guessing--;
		}
		if ( synPredMatched364 ) {
			lf=lambdaFunction();
		}
		else {
			boolean synPredMatched367 = false;
			if (((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==RBRACK||LA(2)==RCURLY||LA(2)==COLON||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==DIV_ASSIGN||LA(2)==PLUS||LA(2)==PLUS_ASSIGN||LA(2)==INC||LA(2)==MINUS||LA(2)==MINUS_ASSIGN||LA(2)==DEC||LA(2)==STAR||LA(2)==STAR_ASSIGN||LA(2)==MOD||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==SL_ASSIGN||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT))) {
				int _m367 = mark();
				synPredMatched367 = true;
				inputState.guessing++;
				try {
					{
					logicalOrExpression();
					{
					if ((LA(1)==QUESTION)) {
						match(QUESTION);
					}
					else {
					}
					
					}
					}
				}
				catch (RecognitionException pe) {
					synPredMatched367 = false;
				}
				rewind(_m367);
inputState.guessing--;
			}
			if ( synPredMatched367 ) {
				if ( inputState.guessing==0 ) {
					startOfCond = lt(1);
				}
				complexity1=logicalOrExpression();
				{
				if ((LA(1)==QUESTION)) {
					endOfCond = LT(1);
					match(QUESTION);
					if ( inputState.guessing==0 ) {
						
						if (!constExpr)
						instrBoolExpr(startOfCond, ct(endOfCond));
						
					}
					complexity2=assignmentExpression();
					match(COLON);
					complexity3=conditionalExpression();
				}
				else if ((LA(1)==RPAREN||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR_ASSIGN||LA(1)==BSR_ASSIGN||LA(1)==SL_ASSIGN||LA(1)==BXOR_ASSIGN||LA(1)==BOR_ASSIGN||LA(1)==BAND_ASSIGN||LA(1)==SEMI)) {
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				if ( inputState.guessing==0 ) {
					
					// as ternary exists, it's complexity is 1, but this is already accounted for in ExpressionInfo.fromTokens
					// sum whatever we have in branches
					complexity = complexity1 + complexity2 + complexity3;
					
				}
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			}
			return complexity;
		}
		
	public final int  logicalOrExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=logicalAndExpression();
		{
		_loop371:
		do {
			if ((LA(1)==LOR)) {
				match(LOR);
				tmpComplexity=logicalAndExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop371;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  logicalAndExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=inclusiveOrExpression();
		{
		_loop374:
		do {
			if ((LA(1)==LAND)) {
				match(LAND);
				tmpComplexity=inclusiveOrExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop374;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  inclusiveOrExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=exclusiveOrExpression();
		{
		_loop377:
		do {
			if ((LA(1)==BOR)) {
				match(BOR);
				tmpComplexity=exclusiveOrExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop377;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  exclusiveOrExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=andExpression();
		{
		_loop380:
		do {
			if ((LA(1)==BXOR)) {
				match(BXOR);
				tmpComplexity=andExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop380;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  andExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=equalityExpression();
		{
		_loop383:
		do {
			if ((LA(1)==BAND)) {
				match(BAND);
				tmpComplexity=equalityExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop383;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  equalityExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=relationalExpression();
		{
		_loop387:
		do {
			if ((LA(1)==EQUAL||LA(1)==NOT_EQUAL)) {
				{
				if ((LA(1)==NOT_EQUAL)) {
					match(NOT_EQUAL);
				}
				else if ((LA(1)==EQUAL)) {
					match(EQUAL);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				tmpComplexity=relationalExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop387;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  relationalExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		String type = null;
		int tmpComplexity = 0;
		
		
		complexity=shiftExpression();
		{
		if ((LA(1)==QUESTION||LA(1)==RPAREN||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV_ASSIGN||LA(1)==PLUS_ASSIGN||LA(1)==MINUS_ASSIGN||LA(1)==STAR_ASSIGN||LA(1)==MOD_ASSIGN||LA(1)==SR_ASSIGN||LA(1)==BSR_ASSIGN||LA(1)==GE||LA(1)==GT||LA(1)==SL_ASSIGN||LA(1)==LE||LA(1)==LT||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI)) {
			{
			_loop392:
			do {
				if ((LA(1)==GE||LA(1)==GT||LA(1)==LE||LA(1)==LT)) {
					{
					if ((LA(1)==LT)) {
						match(LT);
					}
					else if ((LA(1)==GT)) {
						match(GT);
					}
					else if ((LA(1)==LE)) {
						match(LE);
					}
					else if ((LA(1)==GE)) {
						match(GE);
					}
					else {
						throw new NoViableAltException(LT(1), getFilename());
					}
					
					}
					tmpComplexity=shiftExpression();
					if ( inputState.guessing==0 ) {
						
						complexity += tmpComplexity;
						
					}
				}
				else {
					break _loop392;
				}
				
			} while (true);
			}
		}
		else {
			boolean synPredMatched394 = false;
			if (((LA(1)==INSTANCEOF) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==AT||LA(2)==IDENT))) {
				int _m394 = mark();
				synPredMatched394 = true;
				inputState.guessing++;
				try {
					{
					match(INSTANCEOF);
					type=typeSpec();
					match(IDENT);
					}
				}
				catch (RecognitionException pe) {
					synPredMatched394 = false;
				}
				rewind(_m394);
inputState.guessing--;
			}
			if ( synPredMatched394 ) {
				match(INSTANCEOF);
				type=typeSpec();
				match(IDENT);
			}
			else if ((LA(1)==INSTANCEOF) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID||LA(2)==AT||LA(2)==IDENT)) {
				match(INSTANCEOF);
				type=typeSpec();
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			}
			}
			return complexity;
		}
		
	public final int  shiftExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=additiveExpression();
		{
		_loop398:
		do {
			if ((LA(1)==SR||LA(1)==BSR||LA(1)==SL)) {
				{
				if ((LA(1)==SL)) {
					match(SL);
				}
				else if ((LA(1)==SR)) {
					match(SR);
				}
				else if ((LA(1)==BSR)) {
					match(BSR);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				tmpComplexity=additiveExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop398;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  additiveExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=multiplicativeExpression();
		{
		_loop402:
		do {
			if ((LA(1)==PLUS||LA(1)==MINUS)) {
				{
				if ((LA(1)==PLUS)) {
					match(PLUS);
				}
				else if ((LA(1)==MINUS)) {
					match(MINUS);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				tmpComplexity=multiplicativeExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop402;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  multiplicativeExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int tmpComplexity = 0;
		
		
		complexity=unaryExpression();
		{
		_loop406:
		do {
			if ((LA(1)==DIV||LA(1)==STAR||LA(1)==MOD)) {
				{
				if ((LA(1)==STAR)) {
					match(STAR);
				}
				else if ((LA(1)==DIV)) {
					match(DIV);
				}
				else if ((LA(1)==MOD)) {
					match(MOD);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				tmpComplexity=unaryExpression();
				if ( inputState.guessing==0 ) {
					
					complexity += tmpComplexity;
					
				}
			}
			else {
				break _loop406;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final int  unaryExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		if ((LA(1)==INC)) {
			match(INC);
			complexity=unaryExpression();
		}
		else if ((LA(1)==DEC)) {
			match(DEC);
			complexity=unaryExpression();
		}
		else if ((LA(1)==MINUS)) {
			match(MINUS);
			complexity=unaryExpression();
		}
		else if ((LA(1)==PLUS)) {
			match(PLUS);
			complexity=unaryExpression();
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==IDENT)) {
			complexity=unaryExpressionNotPlusMinus(null, null);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return complexity;
	}
	
/**
 * Unary expression which is not "+/- value".
 * <p/>
 * classCastStart/classCastEnd - a pair of marker tokens used to remember a class cast. This pair of markers is used to
 * solve type-inference for lambdaInc wrapper. For example:
 * <pre>
 *   interface Produce<T> { T produce(); }
 *   Object o = (Produce<String>)String::new;
 * </pre>
 * will be wrapped into lambdaInc like this:
 * <pre>
 *   Object o = (Produce<String>)lambdaInc(123, String::new);
 * </pre>
 * <p/>
 * The problem is that javac is unable to infer proper type of lambdaInc - it sees Object. In such case when Clover
 * finds a method reference with a class cast, it will memorize class cast start/end and copy it into a lambda wrapper:
 * <pre>
 *   Object o = (Produce<String>)lambdaInc(123, (Produce<String>)String::new);
 *         start^            end^               ^^^^^^copy^^^^^^^
 * </pre>
 *
 * The classCastStart and classCastEnd are being passed as arguments, because have a recursive call like this:
 * <pre>
 *    "(Produce<String>)String::new" is handled as
 *
 *    unaryExpressionNotPlusMinus[null, null] ->
 *    LPAREN classTypeSpec RPAREN unaryExpressionNotPlusMinus[start, end] ->
 *    postfixExpression[start, end]
 * </pre>
 */
	public final int  unaryExpressionNotPlusMinus(
		CloverToken classCastStart, CloverToken classCastEnd
	) throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		String type = null;
		complexity = 0;
		
		
		if ((LA(1)==BNOT)) {
			match(BNOT);
			complexity=unaryExpression();
		}
		else if ((LA(1)==LNOT)) {
			match(LNOT);
			complexity=unaryExpression();
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==IDENT)) {
			{
			boolean synPredMatched411 = false;
			if (((LA(1)==LPAREN) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==SHORT||LA(2)==VOID))) {
				int _m411 = mark();
				synPredMatched411 = true;
				inputState.guessing++;
				try {
					{
					match(LPAREN);
					type=builtInTypeSpec();
					match(RPAREN);
					unaryExpression();
					}
				}
				catch (RecognitionException pe) {
					synPredMatched411 = false;
				}
				rewind(_m411);
inputState.guessing--;
			}
			if ( synPredMatched411 ) {
				match(LPAREN);
				type=builtInTypeSpec();
				match(RPAREN);
				complexity=unaryExpression();
			}
			else {
				boolean synPredMatched415 = false;
				if (((LA(1)==LPAREN) && (LA(2)==AT||LA(2)==IDENT))) {
					int _m415 = mark();
					synPredMatched415 = true;
					inputState.guessing++;
					try {
						{
						match(LPAREN);
						type=classTypeSpec();
						{
						_loop414:
						do {
							if ((LA(1)==BAND)) {
								match(BAND);
								type=classOrInterfaceType();
							}
							else {
								break _loop414;
							}
							
						} while (true);
						}
						match(RPAREN);
						unaryExpressionNotPlusMinus(null, null);
						}
					}
					catch (RecognitionException pe) {
						synPredMatched415 = false;
					}
					rewind(_m415);
inputState.guessing--;
				}
				if ( synPredMatched415 ) {
					if ( inputState.guessing==0 ) {
						classCastStart = lt(1);
					}
					match(LPAREN);
					type=classTypeSpec();
					{
					_loop417:
					do {
						if ((LA(1)==BAND)) {
							match(BAND);
							type=classOrInterfaceType();
						}
						else {
							break _loop417;
						}
						
					} while (true);
					}
					match(RPAREN);
					if ( inputState.guessing==0 ) {
						classCastEnd = lt(0);
					}
					complexity=unaryExpressionNotPlusMinus(classCastStart, classCastEnd);
				}
				else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==RBRACK||LA(2)==RCURLY||LA(2)==COLON||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==ASSIGN||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==DIV_ASSIGN||LA(2)==PLUS||LA(2)==PLUS_ASSIGN||LA(2)==INC||LA(2)==MINUS||LA(2)==MINUS_ASSIGN||LA(2)==DEC||LA(2)==STAR||LA(2)==STAR_ASSIGN||LA(2)==MOD||LA(2)==MOD_ASSIGN||LA(2)==SR||LA(2)==SR_ASSIGN||LA(2)==BSR||LA(2)==BSR_ASSIGN||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==SL_ASSIGN||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BXOR_ASSIGN||LA(2)==BOR||LA(2)==BOR_ASSIGN||LA(2)==LOR||LA(2)==BAND||LA(2)==BAND_ASSIGN||LA(2)==LAND||LA(2)==SEMI||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
					complexity=postfixExpression(classCastStart, classCastEnd);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				}
				}
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			return complexity;
		}
		
/**
 * Qualified names, array expressions, method invocation, post inc/dec
 *
 * @param classCastStart - used for instrumentation of a method reference with type cast
 * @param classCastEnd - used for instrumentation of a method reference with type cast
 */
	public final int  postfixExpression(
		CloverToken classCastStart, CloverToken classCastEnd
	) throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		/*
		* A marker token to remember where the method reference starts (like "Math::abs" or "String::new" or "int[]::new"
		* This is declared as a field and not as a local variable of a rule, because it must be shared between
		* postfixExpression (start and end of a reference) and supplementaryExpressionPart (end of an array constructor
		* reference).
		*/
		CloverToken startMethodReference = null;
		ContextSetAndComplexity cc = null;
		int tmpComplexity = 0;
		complexity = 0;
		
		
		if ( inputState.guessing==0 ) {
			
			// we might start a method reference, remember this token
			startMethodReference = lt(1);
			
		}
		cc=primaryExpressionPart();
		if ( inputState.guessing==0 ) {
			
			complexity += cc.getComplexity();
			
		}
		tmpComplexity=supplementaryExpressionPart(classCastStart, classCastEnd, startMethodReference);
		if ( inputState.guessing==0 ) {
			
			complexity += tmpComplexity;
			
		}
		{
		boolean synPredMatched421 = false;
		if (((LA(1)==METHOD_REF||LA(1)==LT) && (LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==VOID||LA(2)==QUESTION||LA(2)==METHOD_REF||LA(2)==SR||LA(2)==BSR||LA(2)==GT||LA(2)==LT||LA(2)==AT||LA(2)==IDENT))) {
			int _m421 = mark();
			synPredMatched421 = true;
			inputState.guessing++;
			try {
				{
				methodReferencePredicate();
				}
			}
			catch (RecognitionException pe) {
				synPredMatched421 = false;
			}
			rewind(_m421);
inputState.guessing--;
		}
		if ( synPredMatched421 ) {
			methodReferencePart();
			if ( inputState.guessing==0 ) {
				
				// we don't have a method signature here, so we'll use just a basic name
				Parameter[] la = new Parameter[0];
				MethodSignature methodReferenceSignature = new MethodSignature(
				LambdaUtil.generateLambdaNameWithId(la, lambdaCounter++), null, null, la, null, new Modifiers());
				LambdaExpressionEntryEmitter expressionEntryEmitter = instrEnterLambdaMethodReference(
				methodReferenceSignature, startMethodReference, classCastStart, classCastEnd);
				instrExitMethodReference(expressionEntryEmitter, lt(0));
				
			}
		}
		else if ((LA(1)==INSTANCEOF||LA(1)==QUESTION||LA(1)==RPAREN||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV||LA(1)==DIV_ASSIGN||LA(1)==PLUS||LA(1)==PLUS_ASSIGN||LA(1)==INC||LA(1)==MINUS||LA(1)==MINUS_ASSIGN||LA(1)==DEC||LA(1)==STAR||LA(1)==STAR_ASSIGN||LA(1)==MOD||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GE||LA(1)==GT||LA(1)==SL||LA(1)==SL_ASSIGN||LA(1)==LE||LA(1)==LT||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI) && (_tokenSet_2.member(LA(2)))) {
			supplementaryPostIncDecPart();
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
/**
 * A supplementary part for the primaryExpressionPart, which allows us to use array indexes, dot-qualified names,
 * this/class/super calls etc. Shall be used in conjunction with the primaryExpressionPart.
 *
 * @param classCastStart - used for instrumentation of a method reference with type cast
 * @param classCastEnd - used for instrumentation of a method reference with type cast
 */
	public final int  supplementaryExpressionPart(
		CloverToken classCastStart, CloverToken classCastEnd, CloverToken startMethodReference
	) throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		int argListComplexity = 0;
		int exprComplexity = 0;
		int newComplexity = 0;
		complexity = 0;
		
		
		{
		_loop453:
		do {
			if ((LA(1)==DOT)) {
				match(DOT);
				{
				if ((LA(1)==LT||LA(1)==IDENT)) {
					{
					if ((LA(1)==LT)) {
						typeArguments();
					}
					else if ((LA(1)==IDENT)) {
					}
					else {
						throw new NoViableAltException(LT(1), getFilename());
					}
					
					}
					match(IDENT);
					if ( inputState.guessing==0 ) {
						
						pushIdentifierToHeadStack(LT(0).getText());
						
					}
				}
				else if ((LA(1)==THIS)) {
					match(THIS);
				}
				else if ((LA(1)==CLASS)) {
					match(CLASS);
				}
				else if ((LA(1)==NEW)) {
					newComplexity=newExpression();
					if ( inputState.guessing==0 ) {
						
						complexity += newComplexity;
						
					}
				}
				else if ((LA(1)==SUPER)) {
					match(SUPER);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
			}
			else if ((LA(1)==LBRACK) && (LA(2)==RBRACK)) {
				{
				int _cnt451=0;
				_loop451:
				do {
					if ((LA(1)==LBRACK)) {
						match(LBRACK);
						match(RBRACK);
					}
					else {
						if ( _cnt451>=1 ) { break _loop451; } else {throw new NoViableAltException(LT(1), getFilename());}
					}
					
					_cnt451++;
				} while (true);
				}
				{
				if ((LA(1)==DOT)) {
					match(DOT);
					match(CLASS);
				}
				else if ((LA(1)==METHOD_REF)) {
					match(METHOD_REF);
					match(NEW);
					if ( inputState.guessing==0 ) {
						
						// we don't have a method signature here, so well use just a basic name like
						Parameter[] la = new Parameter[0];
						MethodSignature methodReferenceSignature = new MethodSignature(
						LambdaUtil.generateLambdaNameWithId(la, lambdaCounter++), null, null, la, null, new Modifiers());
						LambdaExpressionEntryEmitter expressionEntryEmitter = instrEnterLambdaExpression(
						methodReferenceSignature, startMethodReference, startMethodReference,
						classCastStart, classCastEnd);
						instrExitLambdaExpression(expressionEntryEmitter, lt(0));
						
					}
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
			}
			else if ((LA(1)==LBRACK) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==IDENT)) {
				match(LBRACK);
				exprComplexity=expression();
				match(RBRACK);
				if ( inputState.guessing==0 ) {
					
					complexity += exprComplexity;
					
				}
			}
			else if ((LA(1)==LPAREN)) {
				match(LPAREN);
				argListComplexity=argList();
				match(RPAREN);
				if ( inputState.guessing==0 ) {
					
					complexity += argListComplexity;
					
				}
			}
			else {
				break _loop453;
			}
			
		} while (true);
		}
		return complexity;
	}
	
	public final void methodReferencePredicate() throws RecognitionException, TokenStreamException {
		
		
		{
		if ((LA(1)==LT)) {
			typeArguments();
		}
		else if ((LA(1)==METHOD_REF)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		match(METHOD_REF);
	}
	
	public final void methodReferencePart() throws RecognitionException, TokenStreamException {
		
		
		String tp;
		
		
		{
		if ((LA(1)==LT)) {
			typeArguments();
		}
		else if ((LA(1)==METHOD_REF)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		match(METHOD_REF);
		{
		if ((LA(1)==LT)) {
			typeArguments();
		}
		else if ((LA(1)==NEW||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==IDENT)) {
			match(IDENT);
		}
		else if ((LA(1)==NEW)) {
			match(NEW);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
/**
 * Possibly add on a post-increment or post-decrement.
 */
	public final void supplementaryPostIncDecPart() throws RecognitionException, TokenStreamException {
		
		
		{
		if ((LA(1)==INC)) {
			match(INC);
		}
		else if ((LA(1)==DEC)) {
			match(DEC);
		}
		else if ((LA(1)==INSTANCEOF||LA(1)==QUESTION||LA(1)==RPAREN||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV||LA(1)==DIV_ASSIGN||LA(1)==PLUS||LA(1)==PLUS_ASSIGN||LA(1)==MINUS||LA(1)==MINUS_ASSIGN||LA(1)==STAR||LA(1)==STAR_ASSIGN||LA(1)==MOD||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GE||LA(1)==GT||LA(1)==SL||LA(1)==SL_ASSIGN||LA(1)==LE||LA(1)==LT||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	public final void constant() throws RecognitionException, TokenStreamException {
		
		
		if ((LA(1)==NUM_INT)) {
			match(NUM_INT);
		}
		else if ((LA(1)==CHAR_LITERAL)) {
			match(CHAR_LITERAL);
		}
		else if ((LA(1)==STRING_LITERAL)) {
			match(STRING_LITERAL);
		}
		else if ((LA(1)==NUM_FLOAT)) {
			match(NUM_FLOAT);
		}
		else if ((LA(1)==NUM_LONG)) {
			match(NUM_LONG);
		}
		else if ((LA(1)==NUM_DOUBLE)) {
			match(NUM_DOUBLE);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
	}
	
/** object instantiation.
 *  Trees are built as illustrated by the following input/tree pairs:
 * <pre>
 *  new T()
 *
 *  new
 *   |
 *   T --  ELIST
 *           |
 *          arg1 -- arg2 -- .. -- argn
 *
 *  new int[]
 *
 *  new
 *   |
 *  int -- ARRAY_DECLARATOR
 *
 *  new int[] {1,2}
 *
 *  new
 *   |
 *  int -- ARRAY_DECLARATOR -- ARRAY_INIT
 *                                  |
 *                                EXPR -- EXPR
 *                                  |      |
 *                                  1      2
 *
 *  new int[3]
 *  new
 *   |
 *  int -- ARRAY_DECLARATOR
 *                |
 *              EXPR
 *                |
 *                3
 *
 *  new int[1][2]
 *
 *  new
 *   |
 *  int -- ARRAY_DECLARATOR
 *               |
 *         ARRAY_DECLARATOR -- EXPR
 *               |              |
 *             EXPR             1
 *               |
 *               2
 * </pre>
 */
	public final int  newExpression() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		CloverToken endOfBlock = null;
		String typeParam = null;
		int argListComplexity = 0;
		int declComplexity = 0;
		int initComplexity = 0;
		complexity = 0;
		
		
		match(NEW);
		{
		if ((LA(1)==LT)) {
			typeParam=typeParameters();
		}
		else if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID||LA(1)==AT||LA(1)==IDENT)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		type();
		{
		if ((LA(1)==LPAREN)) {
			match(LPAREN);
			argListComplexity=argList();
			match(RPAREN);
			{
			if ((LA(1)==LCURLY)) {
				endOfBlock=classBlock(null);
			}
			else if ((LA(1)==INSTANCEOF||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV||LA(1)==DIV_ASSIGN||LA(1)==PLUS||LA(1)==PLUS_ASSIGN||LA(1)==INC||LA(1)==MINUS||LA(1)==MINUS_ASSIGN||LA(1)==DEC||LA(1)==STAR||LA(1)==STAR_ASSIGN||LA(1)==MOD||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GE||LA(1)==GT||LA(1)==SL||LA(1)==SL_ASSIGN||LA(1)==LE||LA(1)==LT||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==DOT)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			if ( inputState.guessing==0 ) {
				
				complexity = argListComplexity;
				
			}
		}
		else if ((LA(1)==LBRACK||LA(1)==AT)) {
			declComplexity=newArrayDeclarator();
			{
			if ((LA(1)==LCURLY)) {
				initComplexity=arrayInitializer();
			}
			else if ((LA(1)==INSTANCEOF||LA(1)==QUESTION||LA(1)==LPAREN||LA(1)==RPAREN||LA(1)==LBRACK||LA(1)==RBRACK||LA(1)==RCURLY||LA(1)==COLON||LA(1)==METHOD_REF||LA(1)==COMMA||LA(1)==ASSIGN||LA(1)==EQUAL||LA(1)==NOT_EQUAL||LA(1)==DIV||LA(1)==DIV_ASSIGN||LA(1)==PLUS||LA(1)==PLUS_ASSIGN||LA(1)==INC||LA(1)==MINUS||LA(1)==MINUS_ASSIGN||LA(1)==DEC||LA(1)==STAR||LA(1)==STAR_ASSIGN||LA(1)==MOD||LA(1)==MOD_ASSIGN||LA(1)==SR||LA(1)==SR_ASSIGN||LA(1)==BSR||LA(1)==BSR_ASSIGN||LA(1)==GE||LA(1)==GT||LA(1)==SL||LA(1)==SL_ASSIGN||LA(1)==LE||LA(1)==LT||LA(1)==BXOR||LA(1)==BXOR_ASSIGN||LA(1)==BOR||LA(1)==BOR_ASSIGN||LA(1)==LOR||LA(1)==BAND||LA(1)==BAND_ASSIGN||LA(1)==LAND||LA(1)==SEMI||LA(1)==DOT)) {
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
			if ( inputState.guessing==0 ) {
				
				complexity = declComplexity + initComplexity;
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
/**
 * A single "case x ->" or "default ->" label, followed by an expression or a block statement.
 */
	public final int  lambdaCase(
		ContextSet context
	) throws RecognitionException, TokenStreamException {
		int complexity;
		
		Token  si1 = null;
		Token  si2 = null;
		Token  t = null;
		
		CloverToken endTok = null;
		Token pos = null;
		CaseExpressionEntryEmitter expressionEntryEmitter = null;
		CaseThrowExpressionEntryEmitter throwEntryEmitter = null;
		int exprComplexity = 0;
		complexity = 1;
		
		
		{
		if ((LA(1)==CASE)) {
			si1 = LT(1);
			match(CASE);
			patternMatch();
			if ( inputState.guessing==0 ) {
				
				pos = si1;
				
			}
		}
		else if ((LA(1)==DEFAULT)) {
			si2 = LT(1);
			match(DEFAULT);
			if ( inputState.guessing==0 ) {
				
				pos = si2;
				
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		t = LT(1);
		match(LAMBDA);
		{
		boolean synPredMatched442 = false;
		if (((LA(1)==THROW))) {
			int _m442 = mark();
			synPredMatched442 = true;
			inputState.guessing++;
			try {
				{
				match(THROW);
				expression();
				match(SEMI);
				}
			}
			catch (RecognitionException pe) {
				synPredMatched442 = false;
			}
			rewind(_m442);
inputState.guessing--;
		}
		if ( synPredMatched442 ) {
			match(THROW);
			exprComplexity=expression();
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				throwEntryEmitter = instrEnterCaseThrowExpression(ct(t), lt(0), context, exprComplexity);
				instrExitCaseThrowExpression(throwEntryEmitter, lt(0));
				
			}
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
			exprComplexity=expression();
			match(SEMI);
			if ( inputState.guessing==0 ) {
				
				expressionEntryEmitter = instrEnterCaseExpression(ct(t), lt(0), context, exprComplexity);
				instrExitCaseExpression(expressionEntryEmitter, lt(0));
				
			}
		}
		else if ((LA(1)==LCURLY)) {
			endTok=compoundStatement();
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		return complexity;
	}
	
	public final int  newArrayDeclarator() throws RecognitionException, TokenStreamException {
		int complexity;
		
		
		AnnotationImpl ann = null;
		complexity = 0;
		
		
		{
		int _cnt474=0;
		_loop474:
		do {
			if ((LA(1)==LBRACK||LA(1)==AT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==RBRACK||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==IDENT)) {
				{
				_loop472:
				do {
					if ((LA(1)==AT)) {
						ann=annotation();
					}
					else {
						break _loop472;
					}
					
				} while (true);
				}
				match(LBRACK);
				{
				if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT)) {
					complexity=expression();
				}
				else if ((LA(1)==RBRACK)) {
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				match(RBRACK);
			}
			else {
				if ( _cnt474>=1 ) { break _loop474; } else {throw new NoViableAltException(LT(1), getFilename());}
			}
			
			_cnt474++;
		} while (true);
		}
		return complexity;
	}
	
	public final void constantConditionalExpression() throws RecognitionException, TokenStreamException {
		
		
		constantLogicalOrExpression();
		{
		if ((LA(1)==QUESTION)) {
			match(QUESTION);
			constantConditionalExpression();
			match(COLON);
			constantConditionalExpression();
		}
		else if ((LA(1)==COLON||LA(1)==COMMA||LA(1)==LAMBDA)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
	}
	
	public final void constantLogicalOrExpression() throws RecognitionException, TokenStreamException {
		
		
		constantLogicalAndExpression();
		{
		_loop481:
		do {
			if ((LA(1)==LOR)) {
				match(LOR);
				constantLogicalAndExpression();
			}
			else {
				break _loop481;
			}
			
		} while (true);
		}
	}
	
	public final void constantLogicalAndExpression() throws RecognitionException, TokenStreamException {
		
		
		constantInclusiveOrExpression();
		{
		_loop484:
		do {
			if ((LA(1)==LAND)) {
				match(LAND);
				constantInclusiveOrExpression();
			}
			else {
				break _loop484;
			}
			
		} while (true);
		}
	}
	
	public final void constantInclusiveOrExpression() throws RecognitionException, TokenStreamException {
		
		
		constantExclusiveOrExpression();
		{
		_loop487:
		do {
			if ((LA(1)==BOR)) {
				match(BOR);
				constantExclusiveOrExpression();
			}
			else {
				break _loop487;
			}
			
		} while (true);
		}
	}
	
	public final void constantExclusiveOrExpression() throws RecognitionException, TokenStreamException {
		
		
		constantAndExpression();
		{
		_loop490:
		do {
			if ((LA(1)==BXOR)) {
				match(BXOR);
				constantAndExpression();
			}
			else {
				break _loop490;
			}
			
		} while (true);
		}
	}
	
	public final void constantAndExpression() throws RecognitionException, TokenStreamException {
		
		
		constantEqualityExpression();
		{
		_loop493:
		do {
			if ((LA(1)==BAND)) {
				match(BAND);
				constantEqualityExpression();
			}
			else {
				break _loop493;
			}
			
		} while (true);
		}
	}
	
	public final void constantEqualityExpression() throws RecognitionException, TokenStreamException {
		
		
		constantRelationalExpression();
		{
		_loop497:
		do {
			if ((LA(1)==EQUAL||LA(1)==NOT_EQUAL)) {
				{
				if ((LA(1)==NOT_EQUAL)) {
					match(NOT_EQUAL);
				}
				else if ((LA(1)==EQUAL)) {
					match(EQUAL);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				constantRelationalExpression();
			}
			else {
				break _loop497;
			}
			
		} while (true);
		}
	}
	
	public final void constantRelationalExpression() throws RecognitionException, TokenStreamException {
		
		
		constantShiftExpression();
		{
		_loop501:
		do {
			if ((LA(1)==GE||LA(1)==GT||LA(1)==LE||LA(1)==LT)) {
				{
				if ((LA(1)==LT)) {
					match(LT);
				}
				else if ((LA(1)==GT)) {
					match(GT);
				}
				else if ((LA(1)==LE)) {
					match(LE);
				}
				else if ((LA(1)==GE)) {
					match(GE);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				constantShiftExpression();
			}
			else {
				break _loop501;
			}
			
		} while (true);
		}
	}
	
	public final void constantShiftExpression() throws RecognitionException, TokenStreamException {
		
		
		constantAdditiveExpression();
		{
		_loop505:
		do {
			if ((LA(1)==SR||LA(1)==BSR||LA(1)==SL)) {
				{
				if ((LA(1)==SL)) {
					match(SL);
				}
				else if ((LA(1)==SR)) {
					match(SR);
				}
				else if ((LA(1)==BSR)) {
					match(BSR);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				constantAdditiveExpression();
			}
			else {
				break _loop505;
			}
			
		} while (true);
		}
	}
	
	public final void constantAdditiveExpression() throws RecognitionException, TokenStreamException {
		
		
		constantMultiplicativeExpression();
		{
		_loop509:
		do {
			if ((LA(1)==PLUS||LA(1)==MINUS)) {
				{
				if ((LA(1)==PLUS)) {
					match(PLUS);
				}
				else if ((LA(1)==MINUS)) {
					match(MINUS);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				constantMultiplicativeExpression();
			}
			else {
				break _loop509;
			}
			
		} while (true);
		}
	}
	
	public final void constantMultiplicativeExpression() throws RecognitionException, TokenStreamException {
		
		
		constantUnaryExpression();
		{
		_loop513:
		do {
			if ((LA(1)==DIV||LA(1)==STAR||LA(1)==MOD)) {
				{
				if ((LA(1)==STAR)) {
					match(STAR);
				}
				else if ((LA(1)==DIV)) {
					match(DIV);
				}
				else if ((LA(1)==MOD)) {
					match(MOD);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				constantUnaryExpression();
			}
			else {
				break _loop513;
			}
			
		} while (true);
		}
	}
	
	public final void constantUnaryExpression() throws RecognitionException, TokenStreamException {
		
		
		if ((LA(1)==MINUS)) {
			match(MINUS);
			constantUnaryExpression();
		}
		else if ((LA(1)==PLUS)) {
			match(PLUS);
			constantUnaryExpression();
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==DEFAULT||LA(1)==SUPER||LA(1)==THIS||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==IDENT)) {
			constantUnaryExpressionNotPlusMinus();
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
	}
	
	public final void constantUnaryExpressionNotPlusMinus() throws RecognitionException, TokenStreamException {
		
		
		String type = null;
		
		
		if ((LA(1)==BNOT)) {
			match(BNOT);
			constantUnaryExpression();
		}
		else if ((LA(1)==LNOT)) {
			match(LNOT);
			constantUnaryExpression();
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==DEFAULT||LA(1)==SUPER||LA(1)==THIS||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==LPAREN||LA(1)==IDENT)) {
			{
			boolean synPredMatched519 = false;
			if (((LA(1)==LPAREN))) {
				int _m519 = mark();
				synPredMatched519 = true;
				inputState.guessing++;
				try {
					{
					match(LPAREN);
					{
					if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
						builtInTypeSpec();
					}
					else if (((LA(1)==IDENT))&&(isNextKeyword("String"))) {
						match(IDENT);
					}
					else {
						throw new NoViableAltException(LT(1), getFilename());
					}
					
					}
					match(RPAREN);
					constantUnaryExpression();
					}
				}
				catch (RecognitionException pe) {
					synPredMatched519 = false;
				}
				rewind(_m519);
inputState.guessing--;
			}
			if ( synPredMatched519 ) {
				match(LPAREN);
				{
				if ((LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==SHORT||LA(1)==VOID)) {
					type=builtInTypeSpec();
				}
				else if (((LA(1)==IDENT))&&(isNextKeyword("String"))) {
					match(IDENT);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
				match(RPAREN);
				constantUnaryExpression();
			}
			else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==DEFAULT||LA(1)==SUPER||LA(1)==THIS||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==IDENT)) {
				constantPostfixExpression();
			}
			else {
				throw new NoViableAltException(LT(1), getFilename());
			}
			
			}
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
	}
	
	public final void constantPostfixExpression() throws RecognitionException, TokenStreamException {
		
		
		{
		if ((LA(1)==IDENT)) {
			match(IDENT);
		}
		else if (((LA(1) >= NUM_INT && LA(1) <= NUM_DOUBLE))) {
			constant();
		}
		else if ((LA(1)==TRUE)) {
			match(TRUE);
		}
		else if ((LA(1)==FALSE)) {
			match(FALSE);
		}
		else if ((LA(1)==THIS)) {
			match(THIS);
		}
		else if ((LA(1)==NULL)) {
			match(NULL);
		}
		else if ((LA(1)==SUPER)) {
			match(SUPER);
		}
		else if ((LA(1)==DEFAULT)) {
			match(DEFAULT);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		_loop525:
		do {
			if ((LA(1)==DOT)) {
				match(DOT);
				{
				if ((LA(1)==IDENT)) {
					match(IDENT);
				}
				else if ((LA(1)==THIS)) {
					match(THIS);
				}
				else if ((LA(1)==SUPER)) {
					match(SUPER);
				}
				else if ((LA(1)==CLASS)) {
					match(CLASS);
				}
				else {
					throw new NoViableAltException(LT(1), getFilename());
				}
				
				}
			}
			else {
				break _loop525;
			}
			
		} while (true);
		}
	}
	
	protected final void annMemberValuePair(
		AnnotationImpl anno, boolean isSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		
		Token  name = null;
		
		boolean ft = false;
		
		
		name = LT(1);
		match(IDENT);
		match(ASSIGN);
		ft=annMemberValue2(anno, name.getText(), isSuppressWarnings, true);
	}
	
	protected final boolean  annMemberValue2(
		AnnotationValueCollection anno, String key, boolean isSuppressWarnings, boolean makeArrayAndAdd
	) throws RecognitionException, TokenStreamException {
		boolean wasFallthroughLiteral;
		
		Token  t = null;
		
		AnnotationImpl innerAnno = null;
		wasFallthroughLiteral = false;
		String expr = null;
		
		
		boolean synPredMatched537 = false;
		if (((LA(1)==STRING_LITERAL) && (LA(2)==RPAREN||LA(2)==RCURLY||LA(2)==COMMA||LA(2)==SEMI))) {
			int _m537 = mark();
			synPredMatched537 = true;
			inputState.guessing++;
			try {
				{
				match(STRING_LITERAL);
				}
			}
			catch (RecognitionException pe) {
				synPredMatched537 = false;
			}
			rewind(_m537);
inputState.guessing--;
		}
		if ( synPredMatched537 ) {
			t = LT(1);
			match(STRING_LITERAL);
			if ( inputState.guessing==0 ) {
				wasFallthroughLiteral = "\"fallthrough\"".equalsIgnoreCase(t.getText());
				if (isSuppressWarnings && makeArrayAndAdd) {
				
				if (wasFallthroughLiteral) {
				// we don't need to set the suppression because one is already present
				existingFallthroughSuppression = true;
				}
				else {
				// put the string literal inside array initializer brackets, and add "fallthrough"
				suppressWarningsInstr = new SimpleEmitter("{");
				Emitter closeCurly = new SimpleEmitter("}");
				Emitter comma = new SimpleEmitter(",");
				Emitter fallthrough = new SimpleEmitter("\"fallthrough\"");
				ct(t).addPreEmitter(suppressWarningsInstr);
				ct(t).addPostEmitter(comma);
				ct(t).addPostEmitter(fallthrough);
				ct(t).addPostEmitter(closeCurly);
				// adding as dependents allows them all to be turned off as a group
				suppressWarningsInstr.addDependent(closeCurly);
				suppressWarningsInstr.addDependent(comma);
				suppressWarningsInstr.addDependent(fallthrough);
				}
				}
				if (anno != null) anno.put(key, new StringifiedAnnotationValue(t.getText()));
				
			}
		}
		else if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==IDENT) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FINAL||LA(2)==FLOAT||LA(2)==INSTANCEOF||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==QUESTION||LA(2)==LPAREN||LA(2)==RPAREN||LA(2)==LBRACK||LA(2)==RCURLY||LA(2)==METHOD_REF||LA(2)==COMMA||LA(2)==EQUAL||LA(2)==LNOT||LA(2)==BNOT||LA(2)==NOT_EQUAL||LA(2)==DIV||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==STAR||LA(2)==MOD||LA(2)==SR||LA(2)==BSR||LA(2)==GE||LA(2)==GT||LA(2)==SL||LA(2)==LE||LA(2)==LT||LA(2)==BXOR||LA(2)==BOR||LA(2)==LOR||LA(2)==BAND||LA(2)==LAND||LA(2)==SEMI||LA(2)==LAMBDA||LA(2)==AT||LA(2)==IDENT||LA(2)==DOT)) {
			expr=conditionalExpression2();
			if ( inputState.guessing==0 ) {
				if (anno != null) anno.put(key, new StringifiedAnnotationValue(expr));
			}
		}
		else if ((LA(1)==AT)) {
			innerAnno=annotation();
			if ( inputState.guessing==0 ) {
				if (anno != null) anno.put(key, innerAnno);
			}
		}
		else if ((LA(1)==LCURLY)) {
			annMemberValueArrayInitializer(anno, key, isSuppressWarnings);
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		return wasFallthroughLiteral;
	}
	
	protected final String  conditionalExpression2() throws RecognitionException, TokenStreamException {
		String asString;
		
		
			CloverToken start = lt(1);
			CloverToken end = null;
			int tmpComplexity;
			asString = null;
		
		
		tmpComplexity=conditionalExpression();
		if ( inputState.guessing==0 ) {
			
				        end = lt(0);
				        asString = TokenListUtil.getNormalisedSequence(start, end);
			
		}
		return asString;
	}
	
	protected final void annMemberValueArrayInitializer(
		AnnotationValueCollection anno, String key, boolean isSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		
		
		boolean emitComma = false;
		boolean seenFallthrough = false;
		CloverToken last = null;
		ArrayAnnotationValue annoArray = (anno == null) ? null : new ArrayAnnotationValue();
		if (anno != null) {
			    anno.put(key, annoArray);
		}
		
		
		match(LCURLY);
		{
		if ((LA(1)==NUM_INT||LA(1)==CHAR_LITERAL||LA(1)==STRING_LITERAL||LA(1)==NUM_FLOAT||LA(1)==NUM_LONG||LA(1)==NUM_DOUBLE||LA(1)==BOOLEAN||LA(1)==BYTE||LA(1)==CHAR||LA(1)==DOUBLE||LA(1)==FLOAT||LA(1)==INT||LA(1)==LONG||LA(1)==NEW||LA(1)==SHORT||LA(1)==SUPER||LA(1)==SWITCH||LA(1)==THIS||LA(1)==VOID||LA(1)==TRUE||LA(1)==FALSE||LA(1)==NULL||LA(1)==NON_SEALED||LA(1)==LPAREN||LA(1)==LCURLY||LA(1)==LNOT||LA(1)==BNOT||LA(1)==PLUS||LA(1)==INC||LA(1)==MINUS||LA(1)==DEC||LA(1)==AT||LA(1)==IDENT)) {
			seenFallthrough=annMemberValues(annoArray, isSuppressWarnings);
			if ( inputState.guessing==0 ) {
				emitComma = true;
			}
		}
		else if ((LA(1)==RCURLY||LA(1)==COMMA)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		{
		if ((LA(1)==COMMA)) {
			match(COMMA);
			if ( inputState.guessing==0 ) {
				emitComma=false;
			}
		}
		else if ((LA(1)==RCURLY)) {
		}
		else {
			throw new NoViableAltException(LT(1), getFilename());
		}
		
		}
		if ( inputState.guessing==0 ) {
			
			CloverToken t = lt(0);
			if (isSuppressWarnings) {
			
			if (seenFallthrough) {
			// we don't need to set the suppression because one is already present
			existingFallthroughSuppression = true;
			}
			else {
			// add "fallthrough" to existing array init
			if (emitComma) {
			suppressWarningsInstr = new SimpleEmitter(",");
			ct(t).addPostEmitter(suppressWarningsInstr);
			}
			Emitter fallthrough = new SimpleEmitter("\"fallthrough\"");
			ct(t).addPostEmitter(fallthrough);
			if (suppressWarningsInstr == null) {
			suppressWarningsInstr = fallthrough;
			}
			else {
			suppressWarningsInstr.addDependent(fallthrough);
			}
			}
			}
			
		}
		match(RCURLY);
	}
	
	protected final boolean  annMemberValues(
		ArrayAnnotationValue annoArray, boolean isSuppressWarnings
	) throws RecognitionException, TokenStreamException {
		boolean sawFallthrough;
		
		
		sawFallthrough = false;
		boolean tmp;
		
		
		tmp=annMemberValue2(annoArray, null, isSuppressWarnings, false);
		if ( inputState.guessing==0 ) {
			
			if (tmp) {
			sawFallthrough = true;
			}
			
		}
		{
		_loop544:
		do {
			if ((LA(1)==COMMA) && (LA(2)==NUM_INT||LA(2)==CHAR_LITERAL||LA(2)==STRING_LITERAL||LA(2)==NUM_FLOAT||LA(2)==NUM_LONG||LA(2)==NUM_DOUBLE||LA(2)==BOOLEAN||LA(2)==BYTE||LA(2)==CHAR||LA(2)==DOUBLE||LA(2)==FLOAT||LA(2)==INT||LA(2)==LONG||LA(2)==NEW||LA(2)==SHORT||LA(2)==SUPER||LA(2)==SWITCH||LA(2)==THIS||LA(2)==VOID||LA(2)==TRUE||LA(2)==FALSE||LA(2)==NULL||LA(2)==NON_SEALED||LA(2)==LPAREN||LA(2)==LCURLY||LA(2)==LNOT||LA(2)==BNOT||LA(2)==PLUS||LA(2)==INC||LA(2)==MINUS||LA(2)==DEC||LA(2)==AT||LA(2)==IDENT)) {
				match(COMMA);
				tmp=annMemberValue2(annoArray, null, isSuppressWarnings, false);
				if ( inputState.guessing==0 ) {
					
					if (tmp) {
					sawFallthrough = true;
					}
					
				}
			}
			else {
				break _loop544;
			}
			
		} while (true);
		}
		return sawFallthrough;
	}
	
	
	public static final String[] _tokenNames = {
		"<0>",
		"EOF",
		"<2>",
		"NULL_TREE_LOOKAHEAD",
		"NUM_INT",
		"CHAR_LITERAL",
		"STRING_LITERAL",
		"NUM_FLOAT",
		"NUM_LONG",
		"NUM_DOUBLE",
		"\"abstract\"",
		"\"assert\"",
		"\"boolean\"",
		"\"break\"",
		"\"byte\"",
		"\"case\"",
		"\"catch\"",
		"\"char\"",
		"\"class\"",
		"\"continue\"",
		"\"default\"",
		"\"do\"",
		"\"double\"",
		"\"else\"",
		"\"enum\"",
		"\"extends\"",
		"\"final\"",
		"\"finally\"",
		"\"float\"",
		"\"for\"",
		"\"if\"",
		"\"implements\"",
		"\"import\"",
		"\"instanceof\"",
		"\"int\"",
		"\"interface\"",
		"\"long\"",
		"\"native\"",
		"\"new\"",
		"\"package\"",
		"\"private\"",
		"\"protected\"",
		"\"public\"",
		"\"return\"",
		"\"short\"",
		"\"static\"",
		"\"strictfp\"",
		"\"super\"",
		"\"switch\"",
		"\"synchronized\"",
		"\"this\"",
		"\"throw\"",
		"\"throws\"",
		"\"transient\"",
		"\"try\"",
		"\"void\"",
		"\"volatile\"",
		"\"while\"",
		"\"const\"",
		"\"goto\"",
		"\"true\"",
		"\"false\"",
		"\"null\"",
		"NON_SEALED",
		"QUESTION",
		"LPAREN",
		"RPAREN",
		"LBRACK",
		"RBRACK",
		"LCURLY",
		"RCURLY",
		"COLON",
		"METHOD_REF",
		"COMMA",
		"ASSIGN",
		"EQUAL",
		"LNOT",
		"BNOT",
		"NOT_EQUAL",
		"DIV",
		"DIV_ASSIGN",
		"PLUS",
		"PLUS_ASSIGN",
		"INC",
		"MINUS",
		"MINUS_ASSIGN",
		"DEC",
		"STAR",
		"STAR_ASSIGN",
		"MOD",
		"MOD_ASSIGN",
		"SR",
		"SR_ASSIGN",
		"BSR",
		"BSR_ASSIGN",
		"GE",
		"GT",
		"SL",
		"SL_ASSIGN",
		"LE",
		"LT",
		"BXOR",
		"BXOR_ASSIGN",
		"BOR",
		"BOR_ASSIGN",
		"LOR",
		"BAND",
		"BAND_ASSIGN",
		"LAND",
		"SEMI",
		"LAMBDA",
		"AT",
		"WS",
		"SL_COMMENT",
		"ML_COMMENT",
		"STRING_LITERAL_SINGLE_LINE",
		"STRING_LITERAL_TEXT_BLOCK",
		"ESC",
		"BACKSLASH",
		"VOCAB",
		"IDENT",
		"INT_LITERAL",
		"LONG_LITERAL",
		"FLOAT_LITERAL",
		"DOUBLE_LITERAL",
		"HEX_FLOAT_LITERAL",
		"DEC_DIGIT",
		"BIN_DIGIT",
		"HEX_DIGIT",
		"DEC_INT",
		"HEX_INT",
		"BIN_INT",
		"ELLIPSIS",
		"DOT",
		"FLOAT_SUFFIX",
		"EXPONENT",
		"BINARY_EXPONENT",
		"IdentifierStart",
		"IdentifierPart",
		"BLOCK",
		"MODIFIERS",
		"OBJBLOCK",
		"SLIST",
		"CTOR_DEF",
		"METHOD_DEF",
		"VARIABLE_DEF",
		"INSTANCE_INIT",
		"STATIC_INIT",
		"TYPE",
		"CLASS_DEF",
		"INTERFACE_DEF",
		"PACKAGE_DEF",
		"ARRAY_DECLARATOR",
		"EXTENDS_CLAUSE",
		"IMPLEMENTS_CLAUSE",
		"PARAMETERS",
		"PARAMETER_DEF",
		"LABELED_STAT",
		"TYPECAST",
		"INDEX_OP",
		"POST_INC",
		"POST_DEC",
		"METHOD_CALL",
		"EXPR",
		"ARRAY_INIT",
		"UNARY_MINUS",
		"UNARY_PLUS",
		"CASE_GROUP",
		"ELIST",
		"FOR_INIT",
		"FOR_CONDITION",
		"FOR_ITERATOR",
		"EMPTY_STAT",
		"SUPER_CTOR_CALL",
		"CTOR_CALL"
	};
	
	private static final long[] mk_tokenSet_0() {
		long[] data = { -869194732511756302L, 72268700270460927L, 48L, 0L, 0L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_0 = new BitSet(mk_tokenSet_0());
	private static final long[] mk_tokenSet_1() {
		long[] data = { -869195284314325008L, 72339069014638591L, 32L, 0L, 0L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_1 = new BitSet(mk_tokenSet_1());
	private static final long[] mk_tokenSet_2() {
		long[] data = { -869194734692794384L, 72268700270460927L, 32L, 0L, 0L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_2 = new BitSet(mk_tokenSet_2());
	
	}
