/*
 * Decompiled with CFR 0.152.
 */
package org.opencastproject.assetmanager.aws.s3;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.HttpMethod;
import com.amazonaws.SdkClientException;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.BucketVersioningConfiguration;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.GeneratePresignedUrlRequest;
import com.amazonaws.services.s3.model.GetObjectTaggingRequest;
import com.amazonaws.services.s3.model.GetObjectTaggingResult;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.ObjectTagging;
import com.amazonaws.services.s3.model.RestoreObjectRequest;
import com.amazonaws.services.s3.model.SetBucketVersioningConfigurationRequest;
import com.amazonaws.services.s3.model.SetObjectTaggingRequest;
import com.amazonaws.services.s3.model.StorageClass;
import com.amazonaws.services.s3.model.Tag;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.amazonaws.services.s3.transfer.Upload;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Date;
import java.util.Dictionary;
import java.util.Optional;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.opencastproject.assetmanager.api.storage.AssetStoreException;
import org.opencastproject.assetmanager.api.storage.RemoteAssetStore;
import org.opencastproject.assetmanager.api.storage.StoragePath;
import org.opencastproject.assetmanager.aws.AwsAbstractArchive;
import org.opencastproject.assetmanager.aws.AwsUploadOperationResult;
import org.opencastproject.assetmanager.aws.persistence.AwsAssetDatabase;
import org.opencastproject.assetmanager.aws.persistence.AwsAssetDatabaseException;
import org.opencastproject.assetmanager.aws.persistence.AwsAssetMapping;
import org.opencastproject.util.ConfigurationException;
import org.opencastproject.util.MimeType;
import org.opencastproject.util.OsgiUtil;
import org.opencastproject.util.data.Option;
import org.opencastproject.workspace.api.Workspace;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(property={"service.description=Amazon S3 based asset store", "store.type=aws-s3"}, immediate=true, service={RemoteAssetStore.class, AwsS3AssetStore.class})
public class AwsS3AssetStore
extends AwsAbstractArchive
implements RemoteAssetStore {
    private static final Logger logger = LoggerFactory.getLogger(AwsS3AssetStore.class);
    private static final Tag freezable = new Tag("Freezable", "true");
    private static final Integer RESTORE_MIN_WAIT = 1080000;
    private static final Integer RESTORE_POLL = 900000;
    public static final String AWS_S3_ENABLED = "org.opencastproject.assetmanager.aws.s3.enabled";
    public static final String AWS_S3_ACCESS_KEY_ID_CONFIG = "org.opencastproject.assetmanager.aws.s3.access.id";
    public static final String AWS_S3_SECRET_ACCESS_KEY_CONFIG = "org.opencastproject.assetmanager.aws.s3.secret.key";
    public static final String AWS_S3_REGION_CONFIG = "org.opencastproject.assetmanager.aws.s3.region";
    public static final String AWS_S3_BUCKET_CONFIG = "org.opencastproject.assetmanager.aws.s3.bucket";
    public static final String AWS_S3_ENDPOINT_CONFIG = "org.opencastproject.assetmanager.aws.s3.endpoint";
    public static final String AWS_S3_PATH_STYLE_CONFIG = "org.opencastproject.assetmanager.aws.s3.path.style";
    public static final String AWS_S3_MAX_CONNECTIONS = "org.opencastproject.assetmanager.aws.s3.max.connections";
    public static final String AWS_S3_CONNECTION_TIMEOUT = "org.opencastproject.assetmanager.aws.s3.connection.timeout";
    public static final String AWS_S3_MAX_RETRIES = "org.opencastproject.assetmanager.aws.s3.max.retries";
    public static final String AWS_GLACIER_RESTORE_DAYS = "org.opencastproject.assetmanager.aws.s3.glacier.restore.days";
    public static final Integer AWS_S3_GLACIER_RESTORE_DAYS_DEFAULT = 2;
    public static final int DEFAULT_MAX_CONNECTIONS = 50;
    public static final int DEFAULT_CONNECTION_TIMEOUT = 10000;
    public static final int DEFAULT_MAX_RETRIES = 100;
    public static final long DOWNLOAD_URL_EXPIRATION_MS = 1800000L;
    private AmazonS3 s3 = null;
    private TransferManager s3TransferManager = null;
    private String bucketName = null;
    private String endpoint = null;
    private boolean pathStyle = false;
    private Integer restorePeriod;
    protected boolean bucketCreated = false;

    @Override
    @Reference
    public void setWorkspace(Workspace workspace) {
        super.setWorkspace(workspace);
    }

    @Override
    @Reference
    public void setDatabase(AwsAssetDatabase db) {
        super.setDatabase(db);
    }

    @Activate
    public void activate(ComponentContext cc) throws IllegalStateException, ConfigurationException {
        if (cc != null) {
            Dictionary properties = cc.getProperties();
            boolean enabled = Boolean.parseBoolean(StringUtils.trimToEmpty((String)((String)properties.get(AWS_S3_ENABLED))));
            if (!enabled) {
                logger.info("AWS S3 asset store is disabled");
                return;
            }
            this.storeType = StringUtils.trimToEmpty((String)((String)properties.get("store.type")));
            if (StringUtils.isEmpty((CharSequence)this.storeType)) {
                throw new ConfigurationException("Invalid store type value");
            }
            logger.info("{} is: {}", (Object)"store.type", (Object)this.storeType);
            this.bucketName = this.getAWSConfigKey(cc, AWS_S3_BUCKET_CONFIG);
            logger.info("AWS S3 bucket name is {}", (Object)this.bucketName);
            this.regionName = this.getAWSConfigKey(cc, AWS_S3_REGION_CONFIG);
            logger.info("AWS region is {}", (Object)this.regionName);
            this.endpoint = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)AWS_S3_ENDPOINT_CONFIG, (String)("s3." + this.regionName + ".amazonaws.com"));
            logger.info("AWS endpoint is {}", (Object)this.endpoint);
            this.pathStyle = BooleanUtils.toBoolean((String)OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)AWS_S3_PATH_STYLE_CONFIG, (String)"false"));
            logger.info("AWS path style is {}", (Object)this.pathStyle);
            this.restorePeriod = (Integer)OsgiUtil.getOptCfgAsInt((Dictionary)cc.getProperties(), (String)AWS_GLACIER_RESTORE_DAYS).getOrElse((Object)AWS_S3_GLACIER_RESTORE_DAYS_DEFAULT);
            Object provider = null;
            Option accessKeyIdOpt = OsgiUtil.getOptCfg((Dictionary)cc.getProperties(), (String)AWS_S3_ACCESS_KEY_ID_CONFIG);
            Option accessKeySecretOpt = OsgiUtil.getOptCfg((Dictionary)cc.getProperties(), (String)AWS_S3_SECRET_ACCESS_KEY_CONFIG);
            provider = accessKeyIdOpt.isNone() && accessKeySecretOpt.isNone() ? new DefaultAWSCredentialsProviderChain() : new AWSStaticCredentialsProvider((AWSCredentials)new BasicAWSCredentials((String)accessKeyIdOpt.get(), (String)accessKeySecretOpt.get()));
            ClientConfiguration clientConfiguration = new ClientConfiguration();
            int maxConnections = (Integer)OsgiUtil.getOptCfgAsInt((Dictionary)cc.getProperties(), (String)AWS_S3_MAX_CONNECTIONS).getOrElse((Object)50);
            logger.debug("Max Connections: {}", (Object)maxConnections);
            clientConfiguration.setMaxConnections(maxConnections);
            int connectionTimeout = (Integer)OsgiUtil.getOptCfgAsInt((Dictionary)cc.getProperties(), (String)AWS_S3_CONNECTION_TIMEOUT).getOrElse((Object)10000);
            logger.debug("Connection Output: {}", (Object)connectionTimeout);
            clientConfiguration.setConnectionTimeout(connectionTimeout);
            int maxRetries = (Integer)OsgiUtil.getOptCfgAsInt((Dictionary)cc.getProperties(), (String)AWS_S3_MAX_RETRIES).getOrElse((Object)100);
            logger.debug("Max Retry: {}", (Object)maxRetries);
            clientConfiguration.setMaxErrorRetry(maxRetries);
            this.s3 = (AmazonS3)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)AmazonS3ClientBuilder.standard().withEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(this.endpoint, this.regionName))).withClientConfiguration(clientConfiguration)).withPathStyleAccessEnabled(Boolean.valueOf(this.pathStyle))).withCredentials((AWSCredentialsProvider)provider)).build();
            this.s3TransferManager = TransferManagerBuilder.standard().withS3Client(this.s3).build();
            logger.info("AwsS3ArchiveAssetStore activated!");
        }
    }

    void createAWSBucket() {
        try {
            this.s3.listObjects(this.bucketName);
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                try {
                    this.s3.createBucket(this.bucketName);
                    BucketVersioningConfiguration configuration = new BucketVersioningConfiguration().withStatus("Enabled");
                    SetBucketVersioningConfigurationRequest configRequest = new SetBucketVersioningConfigurationRequest(this.bucketName, configuration);
                    this.s3.setBucketVersioningConfiguration(configRequest);
                    logger.info("AWS S3 ARCHIVE bucket {} created and versioning enabled", (Object)this.bucketName);
                }
                catch (Exception e2) {
                    throw new IllegalStateException("ARCHIVE bucket " + this.bucketName + " cannot be created: " + e2.getMessage(), e2);
                }
            }
            throw new IllegalStateException("ARCHIVE bucket " + this.bucketName + " exists, but we can't access it: " + e.getMessage(), e);
        }
        this.bucketCreated = true;
    }

    @Override
    protected AwsUploadOperationResult uploadObject(File origin, String objectName, Optional<MimeType> mimeType) throws AssetStoreException {
        if (!this.bucketCreated) {
            this.createAWSBucket();
        }
        logger.info("Uploading {} to archive bucket {}...", (Object)objectName, (Object)this.bucketName);
        try {
            String versionId;
            Upload upload = this.s3TransferManager.upload(this.bucketName, objectName, origin);
            long start = System.currentTimeMillis();
            upload.waitForCompletion();
            logger.info("Upload of {} to archive bucket {} completed in {} seconds", new Object[]{objectName, this.bucketName, (System.currentTimeMillis() - start) / 1000L});
            ObjectMetadata objMetadata = this.s3.getObjectMetadata(this.bucketName, objectName);
            logger.trace("Got object metadata for: {}, version is {}", (Object)objectName, (Object)objMetadata.getVersionId());
            if (mimeType.isPresent()) {
                switch (mimeType.get().getType()) {
                    case "audio": 
                    case "image": 
                    case "video": {
                        logger.debug("Tagging S3 object {} as Freezable", (Object)objectName);
                        ArrayList<Tag> tags = new ArrayList<Tag>();
                        tags.add(freezable);
                        this.s3.setObjectTagging(new SetObjectTaggingRequest(this.bucketName, objectName, new ObjectTagging(tags)));
                        break;
                    }
                }
            }
            if (null == (versionId = objMetadata.getVersionId())) {
                return new AwsUploadOperationResult(objectName, "-1");
            }
            return new AwsUploadOperationResult(objectName, versionId);
        }
        catch (InterruptedException e) {
            throw new AssetStoreException("Operation interrupted", (Throwable)e);
        }
        catch (Exception e) {
            throw new AssetStoreException("Upload failed", (Throwable)e);
        }
    }

    public String getAssetObjectKey(StoragePath storagePath) throws AssetStoreException {
        try {
            AwsAssetMapping map = this.database.findMapping(storagePath);
            return map.getObjectKey();
        }
        catch (AwsAssetDatabaseException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    public String getAssetStorageClass(StoragePath storagePath) throws AssetStoreException {
        if (!this.contains(storagePath)) {
            return "NONE";
        }
        return this.getObjectStorageClass(this.getAssetObjectKey(storagePath));
    }

    private String getObjectStorageClass(String objectName) throws AssetStoreException {
        try {
            String storageClass = this.s3.getObjectMetadata(this.bucketName, objectName).getStorageClass();
            return storageClass == null ? StorageClass.Standard.toString() : storageClass;
        }
        catch (SdkClientException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    public String modifyAssetStorageClass(StoragePath storagePath, String storageClassId) throws AssetStoreException {
        try {
            StorageClass storageClass = StorageClass.fromValue((String)storageClassId);
            AwsAssetMapping map = this.database.findMapping(storagePath);
            return this.modifyObjectStorageClass(map.getObjectKey(), storageClass).toString();
        }
        catch (IllegalArgumentException | AwsAssetDatabaseException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    private StorageClass modifyObjectStorageClass(String objectName, StorageClass storageClass) throws AssetStoreException {
        try {
            StorageClass objectStorageClass = StorageClass.fromValue((String)this.getObjectStorageClass(objectName));
            if (storageClass != objectStorageClass) {
                GetObjectTaggingRequest gotr;
                GetObjectTaggingResult objectTaggingRequest;
                if (objectStorageClass == StorageClass.Glacier || objectStorageClass == StorageClass.DeepArchive) {
                    boolean isRestored;
                    boolean isRestoring = this.isRestoring(objectName);
                    boolean bl = isRestored = null != this.s3.getObjectMetadata(this.bucketName, objectName).getRestoreExpirationTime();
                    if (!isRestoring && !isRestored) {
                        logger.warn("S3 Object {} can not be moved from storage class {} to {} without restoring the object first", (Object)objectStorageClass, (Object)storageClass);
                        return objectStorageClass;
                    }
                }
                if (!(storageClass != StorageClass.Glacier && objectStorageClass != StorageClass.DeepArchive || (objectTaggingRequest = this.s3.getObjectTagging(gotr = new GetObjectTaggingRequest(this.bucketName, objectName))).getTagSet().contains(freezable))) {
                    logger.info("S3 object {} is not suitable for storage class {}", (Object)objectName, (Object)storageClass);
                    return objectStorageClass;
                }
                CopyObjectRequest copyRequest = new CopyObjectRequest(this.bucketName, objectName, this.bucketName, objectName).withStorageClass(storageClass);
                this.s3.copyObject(copyRequest);
                logger.info("S3 object {} moved to storage class {}", (Object)objectName, (Object)storageClass);
            } else {
                logger.info("S3 object {} already in storage class {}", (Object)objectName, (Object)storageClass);
            }
            return storageClass;
        }
        catch (SdkClientException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    @Override
    protected InputStream getObject(AwsAssetMapping map) {
        String storageClassId = this.getObjectStorageClass(map.getObjectKey());
        if (StorageClass.Glacier.name().equals(storageClassId) || StorageClass.DeepArchive.name().equals(storageClassId)) {
            this.restoreGlacierObject(map.getObjectKey(), this.restorePeriod, true);
        }
        try {
            String objectKey = map.getObjectKey();
            Date expiration = new Date(System.currentTimeMillis() + 1800000L);
            GeneratePresignedUrlRequest generatePresignedUrlRequest = new GeneratePresignedUrlRequest(this.bucketName, objectKey).withMethod(HttpMethod.GET).withExpiration(expiration);
            URL signedUrl = this.s3.generatePresignedUrl(generatePresignedUrlRequest);
            logger.debug("Returning pre-signed URL stream for '{}': {}", (Object)map, (Object)signedUrl);
            return signedUrl.openStream();
        }
        catch (IOException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    public String getAssetRestoreStatusString(StoragePath storagePath) {
        try {
            AwsAssetMapping map = this.database.findMapping(storagePath);
            Date expirationTime = this.s3.getObjectMetadata(this.bucketName, map.getObjectKey()).getRestoreExpirationTime();
            if (expirationTime != null) {
                return String.format("RESTORED, expires in %s", expirationTime.toString());
            }
            Boolean prevOngoingRestore = this.s3.getObjectMetadata(this.bucketName, map.getObjectKey()).getOngoingRestore();
            if (prevOngoingRestore != null && prevOngoingRestore.booleanValue()) {
                return "RESTORING";
            }
            return "NONE";
        }
        catch (IllegalArgumentException | AwsAssetDatabaseException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    public void initiateRestoreAsset(StoragePath storagePath, Integer assetRestorePeriod) throws AssetStoreException {
        try {
            AwsAssetMapping map = this.database.findMapping(storagePath);
            this.restoreGlacierObject(map.getObjectKey(), assetRestorePeriod, false);
        }
        catch (IllegalArgumentException | AwsAssetDatabaseException e) {
            throw new AssetStoreException((Throwable)e);
        }
    }

    private boolean isRestoring(String objectName) {
        Boolean prevOngoingRestore = this.s3.getObjectMetadata(this.bucketName, objectName).getOngoingRestore();
        if (prevOngoingRestore != null && prevOngoingRestore.booleanValue()) {
            logger.info("Object {} is already being restored", (Object)objectName);
            return true;
        }
        logger.info("Object {} is not currently being restored", (Object)objectName);
        return false;
    }

    private void restoreGlacierObject(String objectName, Integer objectRestorePeriod, Boolean wait) {
        boolean newRestore = false;
        if (this.isRestoring(objectName)) {
            if (!wait.booleanValue()) {
                return;
            }
            logger.info("Waiting for object {}", (Object)objectName);
        } else {
            RestoreObjectRequest requestRestore = new RestoreObjectRequest(this.bucketName, objectName, objectRestorePeriod.intValue());
            this.s3.restoreObjectV2(requestRestore);
            newRestore = true;
        }
        if (this.s3.getObjectMetadata(this.bucketName, objectName).getRestoreExpirationTime() == null) {
            logger.info("Restoring object {} from Glacier class storage", (Object)objectName);
            if (!wait.booleanValue()) {
                return;
            }
            try {
                if (newRestore) {
                    Thread.sleep(RESTORE_MIN_WAIT.intValue());
                }
                while (this.s3.getObjectMetadata(this.bucketName, objectName).getOngoingRestore().booleanValue()) {
                    Thread.sleep(RESTORE_POLL.intValue());
                }
                logger.info("Object {} has been restored from Glacier class storage, for {} days", (Object)objectName, (Object)objectRestorePeriod);
            }
            catch (InterruptedException e) {
                logger.error("Object {} has not yet been restored from Glacier class storage", (Object)objectName);
            }
        } else {
            logger.info("Object {} has already been restored, further extended by {} days", (Object)objectName, (Object)objectRestorePeriod);
        }
    }

    @Override
    protected void deleteObject(AwsAssetMapping map) {
        this.s3.deleteObject(this.bucketName, map.getObjectKey());
    }

    public Integer getRestorePeriod() {
        return this.restorePeriod;
    }

    void setS3(AmazonS3 s3) {
        this.s3 = s3;
    }

    void setS3TransferManager(TransferManager s3TransferManager) {
        this.s3TransferManager = s3TransferManager;
    }

    void setBucketName(String bucketName) {
        this.bucketName = bucketName;
    }
}

