/*
 * Decompiled with CFR 0.152.
 */
package apoc.metrics;

import apoc.ApocConfig;
import apoc.Extended;
import apoc.export.util.CountingReader;
import apoc.load.CSVResult;
import apoc.load.LoadCsv;
import apoc.load.util.LoadCsvConfig;
import apoc.util.FileUtils;
import apoc.util.Util;
import java.io.File;
import java.io.FilenameFilter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;

@Extended
public class Metrics {
    @Context
    public Log log;
    private static final Predicate<CSVResult> duplicatedHeaderRows = new Predicate<CSVResult>(){

        @Override
        public boolean test(CSVResult o) {
            if (o == null) {
                return false;
            }
            Map map = o.map;
            if ("t".equals(map.get("t"))) {
                return false;
            }
            for (Object value : map.values()) {
                if (!(value instanceof Number)) continue;
                return true;
            }
            return false;
        }
    };
    private static final Map<String, Object> METRIC_TYPE_MAPPINGS = new HashMap<String, Object>();

    @Procedure(mode=Mode.DBMS)
    @Description(value="apoc.metrics.list() - get a list of available metrics")
    public Stream<Neo4jMeasuredMetric> list() {
        File metricsDir = FileUtils.getMetricsDirectory();
        FilenameFilter filter = (dir, name) -> name.toLowerCase().endsWith(".csv");
        return Arrays.asList(metricsDir.listFiles(filter)).stream().map(metricFile -> {
            String name = metricFile.getName();
            String metricName = name.substring(0, name.length() - 4);
            File f = new File(metricsDir, name);
            return new Neo4jMeasuredMetric(metricName, f.lastModified());
        });
    }

    public Stream<GenericMetric> loadCsvForMetric(String metricName, Map<String, Object> config) {
        config.put("sep", ",");
        config.put("header", true);
        File metricsDir = FileUtils.getMetricsDirectory();
        if (metricsDir == null) {
            throw new RuntimeException("Metrics directory either does not exist or is not readable.  To use this procedure please ensure CSV metrics are configured https://neo4j.com/docs/operations-manual/current/monitoring/metrics/expose/#metrics-csv");
        }
        String url = new File(metricsDir, metricName + ".csv").getAbsolutePath();
        CountingReader reader = null;
        try {
            reader = FileUtils.readFile((String)url);
            return new LoadCsv().streamCsv(url, new LoadCsvConfig(config), reader).filter(duplicatedHeaderRows).map(csvResult -> new GenericMetric(metricName, Util.toLong(csvResult.map.get("t")), csvResult.map));
        }
        catch (Exception e) {
            FileUtils.closeReaderSafely((CountingReader)reader);
            throw new RuntimeException(e);
        }
    }

    @Procedure(mode=Mode.DBMS)
    @Description(value="apoc.metrics.storage(directorySetting) - retrieve storage metrics about the devices Neo4j uses for data storage. directorySetting may be any valid neo4j directory setting name, such as 'dbms.directories.data'.  If null is provided as a directorySetting, you will get back all available directory settings.  For a list of available directory settings, see the Neo4j operations manual reference on configuration settings.   Directory settings are **not** paths, they are a neo4j.conf setting key name")
    public Stream<StorageMetric> storage(@Name(value="directorySetting") String directorySetting) {
        boolean validSetting;
        String input = directorySetting == null ? null : directorySetting.toLowerCase();
        boolean bl = validSetting = input == null || FileUtils.NEO4J_DIRECTORY_CONFIGURATION_SETTING_NAMES.contains(input);
        if (!validSetting) {
            String validOptions = String.join((CharSequence)", ", FileUtils.NEO4J_DIRECTORY_CONFIGURATION_SETTING_NAMES);
            throw new RuntimeException("Invalid directory setting specified.  Valid options are one of: " + validOptions);
        }
        return FileUtils.NEO4J_DIRECTORY_CONFIGURATION_SETTING_NAMES.stream().filter(dirSetting -> input == null || input.equals(dirSetting)).map(StoragePair::fromDirectorySetting).filter(sp -> {
            if (sp == null) {
                return false;
            }
            if (sp.dir.exists() && sp.dir.isDirectory() && sp.dir.canRead()) {
                return true;
            }
            this.log.warn("System directory " + sp.setting + " => " + sp.dir + " does not exist or is not readable.");
            return false;
        }).map(StorageMetric::fromStoragePair);
    }

    @Procedure(mode=Mode.DBMS)
    @Description(value="apoc.metrics.get(metricName, {}) - retrieve a system metric by its metric name. Additional configuration options may be passed matching the options available for apoc.load.csv.")
    public Stream<GenericMetric> get(@Name(value="metricName") String metricName, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        Map<String, Object> csvConfig = config;
        if (csvConfig == null) {
            csvConfig = new HashMap<String, Object>();
        }
        if (!csvConfig.containsKey("mapping")) {
            csvConfig.put("mapping", METRIC_TYPE_MAPPINGS);
        }
        return this.loadCsvForMetric(metricName, csvConfig);
    }

    static {
        HashMap<String, String> typeFloat = new HashMap<String, String>();
        typeFloat.put("type", "float");
        HashMap<String, String> typeLong = new HashMap<String, String>();
        typeLong.put("type", "long");
        METRIC_TYPE_MAPPINGS.put("t", typeLong);
        METRIC_TYPE_MAPPINGS.put("count", typeLong);
        METRIC_TYPE_MAPPINGS.put("value", typeFloat);
        METRIC_TYPE_MAPPINGS.put("max", typeFloat);
        METRIC_TYPE_MAPPINGS.put("mean", typeFloat);
        METRIC_TYPE_MAPPINGS.put("min", typeFloat);
        METRIC_TYPE_MAPPINGS.put("mean_rate", typeFloat);
        METRIC_TYPE_MAPPINGS.put("m1_rate", typeFloat);
        METRIC_TYPE_MAPPINGS.put("m5_rate", typeFloat);
        METRIC_TYPE_MAPPINGS.put("m15_rate", typeFloat);
        METRIC_TYPE_MAPPINGS.put("p50", typeFloat);
        METRIC_TYPE_MAPPINGS.put("p75", typeFloat);
        METRIC_TYPE_MAPPINGS.put("p95", typeFloat);
        METRIC_TYPE_MAPPINGS.put("p98", typeFloat);
        METRIC_TYPE_MAPPINGS.put("p99", typeFloat);
        METRIC_TYPE_MAPPINGS.put("p999", typeFloat);
    }

    public static class Neo4jMeasuredMetric {
        public final String name;
        public final long lastUpdated;

        public Neo4jMeasuredMetric(String name, long lastUpdated) {
            this.name = name;
            this.lastUpdated = lastUpdated;
        }
    }

    public static class GenericMetric {
        public final long timestamp;
        public final String metric;
        public final Map<String, Object> map;

        public GenericMetric(String metric, long t, Map<String, Object> map) {
            this.timestamp = t;
            this.metric = metric;
            this.map = map;
        }
    }

    public static class StorageMetric {
        public final String setting;
        public final long freeSpaceBytes;
        public final long totalSpaceBytes;
        public final long usableSpaceBytes;
        public final double percentFree;

        public StorageMetric(String setting, long freeSpaceBytes, long totalSpaceBytes, long usableSpaceBytes) {
            this.setting = setting;
            this.freeSpaceBytes = freeSpaceBytes;
            this.totalSpaceBytes = totalSpaceBytes;
            this.usableSpaceBytes = usableSpaceBytes;
            this.percentFree = totalSpaceBytes <= 0L ? 0.0 : (double)freeSpaceBytes / (double)totalSpaceBytes;
        }

        public static StorageMetric fromStoragePair(StoragePair storagePair) {
            long freeSpace = storagePair.dir.getFreeSpace();
            long usableSpace = storagePair.dir.getUsableSpace();
            long totalSpace = storagePair.dir.getTotalSpace();
            return new StorageMetric(storagePair.setting, freeSpace, totalSpace, usableSpace);
        }
    }

    public static class StoragePair {
        public final String setting;
        public final File dir;

        public StoragePair(String setting, File dir) {
            this.setting = setting;
            this.dir = dir;
        }

        public static StoragePair fromDirectorySetting(String dir) {
            if (dir == null) {
                return null;
            }
            String configLocation = ApocConfig.apocConfig().getString(dir, null);
            if (configLocation == null) {
                return null;
            }
            File f = new File(configLocation);
            return new StoragePair(dir, f);
        }
    }
}

