/*
 * Decompiled with CFR 0.152.
 */
package apoc.periodic;

import apoc.Pools;
import apoc.util.Util;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.ToLongFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.helpers.collection.Pair;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;
import org.neo4j.procedure.TerminationGuard;

public class Periodic {
    public static final Pattern RUNTIME_PATTERN = Pattern.compile("\\bruntime\\s*=", 2);
    public static final Pattern CYPHER_PREFIX_PATTERN = Pattern.compile("\\bcypher\\b", 2);
    public static final String CYPHER_RUNTIME_SLOTTED = "cypher runtime=slotted ";
    static final Pattern LIMIT_PATTERN = Pattern.compile("\\slimit\\s", 2);
    @Context
    public GraphDatabaseService db;
    @Context
    public TerminationGuard terminationGuard;
    @Context
    public Log log;
    @Context
    public Pools pools;
    @Context
    public Transaction tx;

    @Procedure
    @Description(value="apoc.periodic.list - list all jobs")
    public Stream<JobInfo> list() {
        return this.pools.getJobList().entrySet().stream().map(e -> ((JobInfo)e.getKey()).update((Future)e.getValue()));
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.commit(statement,params) - runs the given statement in separate transactions until it returns 0")
    public Stream<RundownResult> commit(@Name(value="statement") String statement, @Name(value="params", defaultValue="{}") Map<String, Object> parameters) throws ExecutionException, InterruptedException {
        this.validateQuery(statement);
        Map<Object, Object> params = parameters == null ? Collections.emptyMap() : parameters;
        long total = 0L;
        long executions = 0L;
        long updates = 0L;
        long start = System.nanoTime();
        if (!LIMIT_PATTERN.matcher(statement).find()) {
            throw new IllegalArgumentException("the statement sent to apoc.periodic.commit must contain a `limit`");
        }
        AtomicInteger batches = new AtomicInteger();
        AtomicInteger failedCommits = new AtomicInteger();
        ConcurrentHashMap<String, Long> commitErrors = new ConcurrentHashMap<String, Long>();
        AtomicInteger failedBatches = new AtomicInteger();
        ConcurrentHashMap<String, Long> batchErrors = new ConcurrentHashMap<String, Long>();
        do {
            Map<String, Object> window = Util.map("_count", updates, "_total", total);
            updates = Util.getFuture(this.pools.getScheduledExecutorService().submit(() -> {
                batches.incrementAndGet();
                try {
                    return this.executeNumericResultStatement(statement, Util.merge(window, params));
                }
                catch (Exception e) {
                    failedBatches.incrementAndGet();
                    this.recordError(batchErrors, e);
                    return 0L;
                }
            }), commitErrors, failedCommits, 0L);
            total += updates;
            if (updates <= 0L) continue;
            ++executions;
        } while (updates > 0L && !Util.transactionIsTerminated(this.terminationGuard));
        long timeTaken = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start);
        boolean wasTerminated = Util.transactionIsTerminated(this.terminationGuard);
        return Stream.of(new RundownResult(total, executions, timeTaken, batches.get(), failedBatches.get(), batchErrors, failedCommits.get(), commitErrors, wasTerminated));
    }

    private void recordError(Map<String, Long> executionErrors, Exception e) {
        String msg = ExceptionUtils.getRootCause((Throwable)e).getMessage();
        executionErrors.compute(msg, (s, i) -> i == null ? 1L : i + 1L);
    }

    private long executeNumericResultStatement(@Name(value="statement") String statement, @Name(value="params") Map<String, Object> parameters) {
        return (Long)this.db.executeTransactionally(statement, parameters, result -> {
            String column = (String)Iterables.single((Iterable)result.columns());
            return result.columnAs(column).stream().mapToLong(o -> (Long)o).sum();
        });
    }

    @Procedure
    @Description(value="apoc.periodic.cancel(name) - cancel job with the given name")
    public Stream<JobInfo> cancel(@Name(value="name") String name) {
        JobInfo info = new JobInfo(name);
        Future future = this.pools.getJobList().remove(info);
        if (future != null) {
            future.cancel(false);
            return Stream.of(info.update(future));
        }
        return Stream.empty();
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.submit('name',statement) - submit a one-off background statement")
    public Stream<JobInfo> submit(@Name(value="name") String name, @Name(value="statement") String statement) {
        this.validateQuery(statement);
        JobInfo info = this.submit(name, () -> {
            try {
                this.db.executeTransactionally(statement);
            }
            catch (Exception e) {
                this.log.warn("in background task via submit", (Throwable)e);
                throw new RuntimeException(e);
            }
        }, this.log);
        return Stream.of(info);
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.repeat('name',statement,repeat-rate-in-seconds, config) submit a repeatedly-called background statement. Fourth parameter 'config' is optional and can contain 'params' entry for nested statement.")
    public Stream<JobInfo> repeat(@Name(value="name") String name, @Name(value="statement") String statement, @Name(value="rate") long rate, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        this.validateQuery(statement);
        Map params = config.getOrDefault("params", Collections.emptyMap());
        JobInfo info = this.schedule(name, () -> this.db.executeTransactionally(statement, params), 0L, rate);
        return Stream.of(info);
    }

    private void validateQuery(String statement) {
        this.db.executeTransactionally("EXPLAIN " + statement);
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.countdown('name',statement,repeat-rate-in-seconds) submit a repeatedly-called background statement until it returns 0")
    public Stream<JobInfo> countdown(@Name(value="name") String name, @Name(value="statement") String statement, @Name(value="rate") long rate) {
        this.validateQuery(statement);
        JobInfo info = this.submit(name, new Countdown(name, statement, rate, this.log), this.log);
        info.rate = rate;
        return Stream.of(info);
    }

    public <T> JobInfo submit(String name, Runnable task, Log log) {
        JobInfo info = new JobInfo(name);
        Future future = this.pools.getJobList().remove(info);
        if (future != null && !future.isDone()) {
            future.cancel(false);
        }
        Runnable wrappingTask = Periodic.wrapTask(name, task, log);
        Future<?> newFuture = this.pools.getScheduledExecutorService().submit(wrappingTask);
        this.pools.getJobList().put(info, newFuture);
        return info;
    }

    public JobInfo schedule(String name, Runnable task, long delay, long repeat) {
        JobInfo info = new JobInfo(name, delay, repeat);
        Future future = this.pools.getJobList().remove(info);
        if (future != null && !future.isDone()) {
            future.cancel(false);
        }
        Runnable wrappingTask = Periodic.wrapTask(name, task, this.log);
        ScheduledFuture<?> newFuture = this.pools.getScheduledExecutorService().scheduleWithFixedDelay(wrappingTask, delay, repeat, TimeUnit.SECONDS);
        this.pools.getJobList().put(info, newFuture);
        return info;
    }

    private static Runnable wrapTask(String name, Runnable task, Log log) {
        return () -> {
            log.debug("Executing task " + name);
            try {
                task.run();
            }
            catch (Exception e) {
                log.error("Error while executing task " + name + " because of the following exception (the task will be killed):", (Throwable)e);
                throw e;
            }
            log.debug("Executed task " + name);
        };
    }

    @Procedure(mode=Mode.WRITE)
    @Deprecated
    @Description(value="apoc.periodic.rock_n_roll_while('some cypher for knowing when to stop', 'some cypher for iteration', 'some cypher as action on each iteration', 10000) YIELD batches, total - run the action statement in batches over the iterator statement's results in a separate thread. Returns number of batches and total processed rows")
    public Stream<LoopingBatchAndTotalResult> rock_n_roll_while(@Name(value="cypherLoop") String cypherLoop, @Name(value="cypherIterate") String cypherIterate, @Name(value="cypherAction") String cypherAction, @Name(value="batchSize") long batchSize) {
        Map<String, String> fieldStatement = Util.map("cypherLoop", cypherLoop, "cypherIterate", cypherIterate);
        this.validateQueries(fieldStatement);
        Stream<LoopingBatchAndTotalResult> allResults = Stream.empty();
        HashMap<String, Object> loopParams = new HashMap<String, Object>(1);
        Object value = null;
        while (true) {
            loopParams.put("previous", value);
            try (Result result = this.tx.execute(cypherLoop, loopParams);){
                value = result.next().get("loop");
                if (!Util.toBoolean(value)) {
                    Stream<LoopingBatchAndTotalResult> stream = allResults;
                    return stream;
                }
            }
            this.log.info("starting batched operation using iteration `%s` in separate thread", new Object[]{cypherIterate});
            result = this.tx.execute(cypherIterate);
            try {
                Stream<BatchAndTotalResult> oneResult = this.iterateAndExecuteBatchedInSeparateThread((int)batchSize, false, false, 0L, (Iterator<Map<String, Object>>)result, (tx, params) -> tx.execute(cypherAction, params), 50, -1);
                Object loopParam = value;
                allResults = Stream.concat(allResults, oneResult.map(r -> r.inLoop(loopParam)));
                continue;
            }
            finally {
                if (result == null) continue;
                result.close();
                continue;
            }
            break;
        }
    }

    private void validateQueries(Map<String, String> fieldStatement) {
        String error = fieldStatement.entrySet().stream().map(e -> {
            try {
                this.validateQuery((String)e.getValue());
                return null;
            }
            catch (Exception exception) {
                return String.format("Exception for field `%s`, message: %s", e.getKey(), exception.getMessage());
            }
        }).filter(e -> e != null).collect(Collectors.joining("\n"));
        if (!error.isEmpty()) {
            throw new RuntimeException(error);
        }
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.iterate('statement returning items', 'statement per item', {batchSize:1000,iterateList:true,parallel:false,params:{},concurrency:50,retries:0}) YIELD batches, total - run the second statement for each item returned by the first statement. Returns number of batches and total processed rows")
    public Stream<BatchAndTotalResult> iterate(@Name(value="cypherIterate") String cypherIterate, @Name(value="cypherAction") String cypherAction, @Name(value="config") Map<String, Object> config) {
        this.validateQuery(cypherIterate);
        long batchSize = Util.toLong(config.getOrDefault("batchSize", 10000));
        int concurrency = Util.toInteger(config.getOrDefault("concurrency", 50));
        boolean parallel = Util.toBoolean(config.getOrDefault("parallel", false));
        boolean iterateList = Util.toBoolean(config.getOrDefault("iterateList", true));
        long retries = Util.toLong(config.getOrDefault("retries", 0));
        Map params = config.getOrDefault("params", Collections.emptyMap());
        int failedParams = Util.toInteger(config.getOrDefault("failedParams", -1));
        try (Result result = this.tx.execute(Periodic.slottedRuntime(cypherIterate), params);){
            Pair<String, Boolean> prepared = this.prepareInnerStatement(cypherAction, iterateList, result.columns(), "_batch");
            String innerStatement = (String)prepared.first();
            iterateList = (Boolean)prepared.other();
            this.log.info("starting batching from `%s` operation using iteration `%s` in separate thread", new Object[]{cypherIterate, cypherAction});
            Stream<BatchAndTotalResult> stream = this.iterateAndExecuteBatchedInSeparateThread((int)batchSize, parallel, iterateList, retries, (Iterator<Map<String, Object>>)result, (tx, p) -> Iterators.count((Iterator)tx.execute(innerStatement, Util.merge(params, p))), concurrency, failedParams);
            return stream;
        }
    }

    static String slottedRuntime(String cypherIterate) {
        if (RUNTIME_PATTERN.matcher(cypherIterate).find()) {
            return cypherIterate;
        }
        Matcher matcher = CYPHER_PREFIX_PATTERN.matcher(cypherIterate.substring(0, Math.min(15, cypherIterate.length())));
        return matcher.find() ? CYPHER_PREFIX_PATTERN.matcher(cypherIterate).replaceFirst(CYPHER_RUNTIME_SLOTTED) : CYPHER_RUNTIME_SLOTTED + cypherIterate;
    }

    public Pair<String, Boolean> prepareInnerStatement(String cypherAction, boolean iterateList, List<String> columns, String iterator) {
        String names = columns.stream().map(Util::quote).collect(Collectors.joining("|"));
        boolean withCheck = this.regNoCaseMultiLine("[{$](" + names + ")\\}?\\s+AS\\s+").matcher(cypherAction).find();
        if (withCheck) {
            return Pair.of((Object)cypherAction, (Object)false);
        }
        if (iterateList) {
            if (this.regNoCaseMultiLine("UNWIND\\s+[{$]" + iterator + "\\}?\\s+AS\\s+").matcher(cypherAction).find()) {
                return Pair.of((Object)cypherAction, (Object)true);
            }
            String with = Util.withMapping(columns.stream(), c -> Util.quote(iterator) + "." + Util.quote(c) + " AS " + Util.quote(c));
            return Pair.of((Object)("UNWIND " + Util.param(iterator) + " AS " + Util.quote(iterator) + with + " " + cypherAction), (Object)true);
        }
        return Pair.of((Object)(Util.withMapping(columns.stream(), c -> Util.param(c) + " AS " + Util.quote(c)) + cypherAction), (Object)false);
    }

    public Pattern regNoCaseMultiLine(String pattern) {
        return Pattern.compile(pattern, 42);
    }

    @Deprecated
    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.rock_n_roll('some cypher for iteration', 'some cypher as action on each iteration', 10000) YIELD batches, total - run the action statement in batches over the iterator statement's results in a separate thread. Returns number of batches and total processed rows")
    public Stream<BatchAndTotalResult> rock_n_roll(@Name(value="cypherIterate") String cypherIterate, @Name(value="cypherAction") String cypherAction, @Name(value="batchSize") long batchSize) {
        Map<String, String> fieldStatement = Util.map("cypherIterate", cypherIterate, "cypherAction", cypherAction);
        this.validateQueries(fieldStatement);
        this.log.info("starting batched operation using iteration `%s` in separate thread", new Object[]{cypherIterate});
        try (Result result = this.tx.execute(cypherIterate);){
            Stream<BatchAndTotalResult> stream = this.iterateAndExecuteBatchedInSeparateThread((int)batchSize, false, false, 0L, (Iterator<Map<String, Object>>)result, (tx, p) -> tx.execute(cypherAction, p), 50, -1);
            return stream;
        }
    }

    private Stream<BatchAndTotalResult> iterateAndExecuteBatchedInSeparateThread(int batchsize, boolean parallel, boolean iterateList, long retries, Iterator<Map<String, Object>> iterator, BiConsumer<Transaction, Map<String, Object>> consumer, int concurrency, int failedParams) {
        boolean wasTerminated;
        ExecutorService pool = parallel ? this.pools.getDefaultExecutorService() : this.pools.getSingleExecutorService();
        ArrayList<Future<Long>> futures = new ArrayList<Future<Long>>(concurrency);
        BatchAndTotalCollector collector = new BatchAndTotalCollector(this.terminationGuard, failedParams);
        while (!Util.transactionIsTerminated(this.terminationGuard)) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("execute in batch no %d batch size ", new Object[]{batchsize});
            }
            List<Map<String, Object>> batch = Util.take(iterator, batchsize);
            long currentBatchSize = batch.size();
            Function<Transaction, Long> task = iterateList ? txInThread -> {
                if (Util.transactionIsTerminated(this.terminationGuard)) {
                    return 0L;
                }
                Map<String, Object> params = Util.map("_count", collector.getCount(), "_batch", batch);
                long successes = this.executeAndReportErrors((Transaction)txInThread, consumer, params, batch, batch.size(), null, collector);
                return successes;
            } : txInThread -> {
                if (Util.transactionIsTerminated(this.terminationGuard)) {
                    return 0L;
                }
                AtomicLong localCount = new AtomicLong(collector.getCount());
                return batch.stream().map(p -> {
                    if (localCount.get() % 1000L == 0L && Util.transactionIsTerminated(this.terminationGuard)) {
                        return 0;
                    }
                    Map<String, Object> params = Util.merge(p, Util.map("_count", localCount.get(), "_batch", batch));
                    return this.executeAndReportErrors((Transaction)txInThread, consumer, params, batch, 1, localCount, collector);
                }).mapToLong(n -> (Long)n).sum();
            };
            futures.add(Util.inTxFuture(this.log, pool, this.db, task, retries, aLong -> collector.incrementRetried(), _ignored -> collector.incrementBatches()));
            collector.incrementCount(currentBatchSize);
            if (iterator.hasNext()) continue;
        }
        ToLongFunction<Future> toLongFunction = (wasTerminated = Util.transactionIsTerminated(this.terminationGuard)) ? f -> Util.getFutureOrCancel(f, collector.getBatchErrors(), collector.getFailedBatches(), 0L) : f -> Util.getFuture(f, collector.getBatchErrors(), collector.getFailedBatches(), 0L);
        collector.incrementSuccesses(futures.stream().mapToLong(toLongFunction).sum());
        Util.logErrors("Error during iterate.commit:", collector.getBatchErrors(), this.log);
        Util.logErrors("Error during iterate.execute:", collector.getOperationErrors(), this.log);
        return Stream.of(collector.getResult());
    }

    private long executeAndReportErrors(Transaction tx, BiConsumer<Transaction, Map<String, Object>> consumer, Map<String, Object> params, List<Map<String, Object>> batch, int returnValue, AtomicLong localCount, BatchAndTotalCollector collector) {
        try {
            consumer.accept(tx, params);
            if (localCount != null) {
                localCount.getAndIncrement();
            }
            return returnValue;
        }
        catch (Exception e) {
            collector.incrementFailedOps(batch.size());
            collector.amendFailedParamsMap(batch);
            this.recordError(collector.getOperationErrors(), e);
            throw e;
        }
    }

    private class Countdown
    implements Runnable {
        private final String name;
        private final String statement;
        private final long rate;
        private final transient Log log;

        public Countdown(String name, String statement, long rate, Log log) {
            this.name = name;
            this.statement = statement;
            this.rate = rate;
            this.log = log;
        }

        @Override
        public void run() {
            if (Periodic.this.executeNumericResultStatement(this.statement, Collections.emptyMap()) > 0L) {
                Periodic.this.pools.getScheduledExecutorService().schedule(() -> Periodic.this.submit(this.name, this, this.log), this.rate, TimeUnit.SECONDS);
            }
        }
    }

    public static class JobInfo {
        public final String name;
        public long delay;
        public long rate;
        public boolean done;
        public boolean cancelled;

        public JobInfo(String name) {
            this.name = name;
        }

        public JobInfo(String name, long delay, long rate) {
            this.name = name;
            this.delay = delay;
            this.rate = rate;
        }

        public JobInfo update(Future future) {
            this.done = future.isDone();
            this.cancelled = future.isCancelled();
            return this;
        }

        public boolean equals(Object o) {
            return this == o || o instanceof JobInfo && this.name.equals(((JobInfo)o).name);
        }

        public int hashCode() {
            return this.name.hashCode();
        }
    }

    public static class LoopingBatchAndTotalResult {
        public Object loop;
        public long batches;
        public long total;

        public LoopingBatchAndTotalResult(Object loop, long batches, long total) {
            this.loop = loop;
            this.batches = batches;
            this.total = total;
        }
    }

    public static class BatchAndTotalResult {
        public final long batches;
        public final long total;
        public final long timeTaken;
        public final long committedOperations;
        public final long failedOperations;
        public final long failedBatches;
        public final long retries;
        public final Map<String, Long> errorMessages;
        public final Map<String, Object> batch;
        public final Map<String, Object> operations;
        public final boolean wasTerminated;
        public final Map<String, List<Map<String, Object>>> failedParams;

        public BatchAndTotalResult(long batches, long total, long timeTaken, long committedOperations, long failedOperations, long failedBatches, long retries, Map<String, Long> operationErrors, Map<String, Long> batchErrors, boolean wasTerminated, Map<String, List<Map<String, Object>>> failedParams) {
            this.batches = batches;
            this.total = total;
            this.timeTaken = timeTaken;
            this.committedOperations = committedOperations;
            this.failedOperations = failedOperations;
            this.failedBatches = failedBatches;
            this.retries = retries;
            this.errorMessages = operationErrors;
            this.wasTerminated = wasTerminated;
            this.failedParams = failedParams;
            this.batch = Util.map("total", batches, "failed", failedBatches, "committed", batches - failedBatches, "errors", batchErrors);
            this.operations = Util.map("total", total, "failed", failedOperations, "committed", committedOperations, "errors", operationErrors);
        }

        public LoopingBatchAndTotalResult inLoop(Object loop) {
            return new LoopingBatchAndTotalResult(loop, this.batches, this.total);
        }
    }

    public static class BatchAndTotalCollector {
        private final int failedParams;
        private long start = System.nanoTime();
        private AtomicLong batches = new AtomicLong();
        private long successes = 0L;
        private AtomicLong count = new AtomicLong();
        private AtomicLong failedOps = new AtomicLong();
        private AtomicLong retried = new AtomicLong();
        private Map<String, Long> operationErrors = new ConcurrentHashMap<String, Long>();
        private AtomicInteger failedBatches = new AtomicInteger();
        private Map<String, Long> batchErrors = new HashMap<String, Long>();
        private Map<String, List<Map<String, Object>>> failedParamsMap = new ConcurrentHashMap<String, List<Map<String, Object>>>();
        private final boolean wasTerminated;

        public BatchAndTotalCollector(TerminationGuard terminationGuard, int failedParams) {
            this.failedParams = failedParams;
            this.wasTerminated = Util.transactionIsTerminated(terminationGuard);
        }

        public BatchAndTotalResult getResult() {
            long timeTaken = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - this.start);
            return new BatchAndTotalResult(this.batches.get(), this.count.get(), timeTaken, this.successes, this.failedOps.get(), this.failedBatches.get(), this.retried.get(), this.operationErrors, this.batchErrors, this.wasTerminated, this.failedParamsMap);
        }

        public long getBatches() {
            return this.batches.get();
        }

        public long getCount() {
            return this.count.get();
        }

        public void incrementFailedOps(long size) {
            this.failedOps.addAndGet(size);
        }

        public void incrementBatches() {
            this.batches.incrementAndGet();
        }

        public void incrementSuccesses(long increment) {
            this.successes += increment;
        }

        public void incrementCount(long currentBatchSize) {
            this.count.addAndGet(currentBatchSize);
        }

        public Map<String, Long> getBatchErrors() {
            return this.batchErrors;
        }

        public Map<String, Long> getOperationErrors() {
            return this.operationErrors;
        }

        public Map<String, List<Map<String, Object>>> getFailedParamsMap() {
            return this.failedParamsMap;
        }

        public void amendFailedParamsMap(List<Map<String, Object>> batch) {
            if (this.failedParams >= 0) {
                this.failedParamsMap.put(Long.toString(this.batches.get()), new ArrayList<Map<String, Object>>(batch.subList(0, Math.min(this.failedParams + 1, batch.size()))));
            }
        }

        public AtomicInteger getFailedBatches() {
            return this.failedBatches;
        }

        public void incrementRetried() {
            this.retried.incrementAndGet();
        }
    }

    public static class RundownResult {
        public final long updates;
        public final long executions;
        public final long runtime;
        public final long batches;
        public final long failedBatches;
        public final Map<String, Long> batchErrors;
        public final long failedCommits;
        public final Map<String, Long> commitErrors;
        public final boolean wasTerminated;

        public RundownResult(long total, long executions, long timeTaken, long batches, long failedBatches, Map<String, Long> batchErrors, long failedCommits, Map<String, Long> commitErrors, boolean wasTerminated) {
            this.updates = total;
            this.executions = executions;
            this.runtime = timeTaken;
            this.batches = batches;
            this.failedBatches = failedBatches;
            this.batchErrors = batchErrors;
            this.failedCommits = failedCommits;
            this.commitErrors = commitErrors;
            this.wasTerminated = wasTerminated;
        }
    }
}

