/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.gis.spatial.index.curves;

import java.util.ArrayList;
import java.util.List;
import org.neo4j.gis.spatial.index.Envelope;

public abstract class SpaceFillingCurve {
    private final Envelope range;
    private final int nbrDim;
    private final int maxLevel;
    private final long width;
    private final long valueWidth;
    private final int quadFactor;
    private final long initialNormMask;
    private double[] scalingFactor;

    SpaceFillingCurve(Envelope range, int maxLevel) {
        this.range = range;
        this.nbrDim = range.getDimension();
        this.maxLevel = maxLevel;
        if (maxLevel < 1) {
            throw new IllegalArgumentException("Hilbert index needs at least one level");
        }
        if (range.getDimension() > 3) {
            throw new IllegalArgumentException("Hilbert index does not yet support more than 3 dimensions");
        }
        this.width = (long)Math.pow(2.0, maxLevel);
        this.scalingFactor = new double[this.nbrDim];
        for (int dim = 0; dim < this.nbrDim; ++dim) {
            this.scalingFactor[dim] = (double)this.width / range.getWidth(dim);
        }
        this.valueWidth = (long)Math.pow(2.0, maxLevel * this.nbrDim);
        this.initialNormMask = (long)(Math.pow(2.0, this.nbrDim) - 1.0) << (maxLevel - 1) * this.nbrDim;
        this.quadFactor = (int)Math.pow(2.0, this.nbrDim);
    }

    public int getMaxLevel() {
        return this.maxLevel;
    }

    public long getWidth() {
        return this.width;
    }

    public long getValueWidth() {
        return this.valueWidth;
    }

    public double getTileWidth(int dimension, int level) {
        return this.range.getWidth(dimension) / Math.pow(2.0, level);
    }

    public Envelope getRange() {
        return this.range;
    }

    protected abstract CurveRule rootCurve();

    public Long derivedValueFor(double[] coord) {
        return this.derivedValueFor(coord, this.maxLevel);
    }

    private Long derivedValueFor(double[] coord, int level) {
        this.assertValidLevel(level);
        long[] normalizedValues = this.getNormalizedCoord(coord);
        long derivedValue = 0L;
        long mask = 1L << this.maxLevel - 1;
        CurveRule currentCurve = this.rootCurve();
        for (int i = 1; i <= this.maxLevel; ++i) {
            int bitIndex = this.maxLevel - i;
            int npoint = 0;
            for (long val : normalizedValues) {
                npoint = npoint << 1 | (int)((val & mask) >> bitIndex);
            }
            int derivedIndex = currentCurve.indexForNPoint(npoint);
            derivedValue = derivedValue << this.nbrDim | (long)derivedIndex;
            mask >>= 1;
            currentCurve = currentCurve.childAt(derivedIndex);
        }
        if (level < this.maxLevel) {
            derivedValue <<= this.nbrDim * this.maxLevel - level;
        }
        return derivedValue;
    }

    public double[] centerPointFor(long derivedValue) {
        return this.centerPointFor(derivedValue, this.maxLevel);
    }

    private double[] centerPointFor(long derivedValue, int level) {
        long[] normalizedCoord = this.normalizedCoordinateFor(derivedValue, level);
        return this.getDoubleCoord(normalizedCoord, level);
    }

    long[] normalizedCoordinateFor(long derivedValue, int level) {
        this.assertValidLevel(level);
        long mask = this.initialNormMask;
        long[] coordinate = new long[this.nbrDim];
        CurveRule currentCurve = this.rootCurve();
        for (int i = 1; i <= level; ++i) {
            int bitIndex = this.maxLevel - i;
            int derivedIndex = (int)((derivedValue & mask) >> bitIndex * this.nbrDim);
            int npoint = currentCurve.npointForIndex(derivedIndex);
            int[] bitValues = this.bitValues(npoint);
            for (int dim = 0; dim < this.nbrDim; ++dim) {
                coordinate[dim] = coordinate[dim] << 1 | (long)bitValues[dim];
            }
            mask >>= this.nbrDim;
            currentCurve = currentCurve.childAt(derivedIndex);
        }
        if (level < this.maxLevel) {
            for (int dim = 0; dim < this.nbrDim; ++dim) {
                coordinate[dim] = coordinate[dim] << this.maxLevel - level;
            }
        }
        return coordinate;
    }

    public List<LongRange> getTilesIntersectingEnvelope(Envelope referenceEnvelope) {
        SearchEnvelope search = new SearchEnvelope(referenceEnvelope);
        ArrayList<LongRange> results = new ArrayList<LongRange>();
        this.addTilesIntersectingEnvelopeAt(search, new SearchEnvelope(0L, this.getWidth(), this.nbrDim), this.rootCurve(), 0L, this.getValueWidth(), results);
        return results;
    }

    private void addTilesIntersectingEnvelopeAt(SearchEnvelope search, SearchEnvelope currentExtent, CurveRule curve, long left, long right, ArrayList<LongRange> results) {
        if (right - left == 1L) {
            long[] coord = this.normalizedCoordinateFor(left, this.maxLevel);
            if (search.contains(coord)) {
                LongRange current;
                LongRange longRange = current = results.size() > 0 ? results.get(results.size() - 1) : null;
                if (current != null && current.max == left - 1L) {
                    current.expandToMax(left);
                } else {
                    current = new LongRange(left);
                    results.add(current);
                }
            }
        } else if (search.intersects(currentExtent)) {
            long width = (right - left) / (long)this.quadFactor;
            for (int i = 0; i < this.quadFactor; ++i) {
                int npoint = curve.npointForIndex(i);
                SearchEnvelope quadrant = currentExtent.quadrant(this.bitValues(npoint));
                this.addTilesIntersectingEnvelopeAt(search, quadrant, curve.childAt(i), left + (long)i * width, left + (long)(i + 1) * width, results);
            }
        }
    }

    private int[] bitValues(int npoint) {
        int[] bitValues = new int[this.nbrDim];
        for (int dim = 0; dim < this.nbrDim; ++dim) {
            int shift = this.nbrDim - dim - 1;
            bitValues[dim] = (npoint & 1 << shift) >> shift;
        }
        return bitValues;
    }

    private long[] getNormalizedCoord(double[] coord) {
        long[] normalizedCoord = new long[this.nbrDim];
        for (int dim = 0; dim < this.nbrDim; ++dim) {
            double value = this.clamp(coord[dim], this.range.getMin(dim), this.range.getMax(dim));
            normalizedCoord[dim] = value == this.range.getMax(dim) ? this.valueWidth - 1L : (long)((value - this.range.getMin(dim)) * this.scalingFactor[dim]);
        }
        return normalizedCoord;
    }

    private double[] getDoubleCoord(long[] normalizedCoord, int level) {
        double[] coord = new double[this.nbrDim];
        for (int dim = 0; dim < this.nbrDim; ++dim) {
            double coordinate = (double)normalizedCoord[dim] / this.scalingFactor[dim] + this.range.getMin(dim) + this.getTileWidth(dim, level) / 2.0;
            coord[dim] = this.clamp(coordinate, this.range.getMin(dim), this.range.getMax(dim));
        }
        return coord;
    }

    private double clamp(double val, double min, double max) {
        if (val <= min) {
            return min;
        }
        if (val >= max) {
            return max;
        }
        return val;
    }

    private void assertValidLevel(int level) {
        if (level > this.maxLevel) {
            throw new IllegalArgumentException("Level " + level + " greater than max-level " + this.maxLevel);
        }
    }

    private class SearchEnvelope {
        long[] min;
        long[] max;
        int nbrDim;

        private SearchEnvelope(Envelope referenceEnvelope) {
            this.min = SpaceFillingCurve.this.getNormalizedCoord(referenceEnvelope.getMin());
            this.max = SpaceFillingCurve.this.getNormalizedCoord(referenceEnvelope.getMax());
            this.nbrDim = referenceEnvelope.getDimension();
        }

        private SearchEnvelope(long[] min, long[] max) {
            this.min = min;
            this.max = max;
            this.nbrDim = min.length;
        }

        private SearchEnvelope(long min, long max, int nbrDim) {
            this.nbrDim = nbrDim;
            this.min = new long[nbrDim];
            this.max = new long[nbrDim];
            for (int dim = 0; dim < nbrDim; ++dim) {
                this.min[dim] = min;
                this.max[dim] = max;
            }
        }

        private SearchEnvelope quadrant(int[] quadNbrs) {
            long[] newMin = new long[this.nbrDim];
            long[] newMax = new long[this.nbrDim];
            for (int dim = 0; dim < this.nbrDim; ++dim) {
                long extent = (this.max[dim] - this.min[dim]) / 2L;
                newMin[dim] = this.min[dim] + (long)quadNbrs[dim] * extent;
                newMax[dim] = this.min[dim] + (long)(quadNbrs[dim] + 1) * extent;
            }
            return new SearchEnvelope(newMin, newMax);
        }

        private boolean contains(long[] coord) {
            for (int dim = 0; dim < this.nbrDim; ++dim) {
                if (coord[dim] >= this.min[dim] && coord[dim] <= this.max[dim]) continue;
                return false;
            }
            return true;
        }

        private boolean intersects(SearchEnvelope other) {
            for (int dim = 0; dim < this.nbrDim; ++dim) {
                if (this.max[dim] >= other.min[dim] && other.max[dim] >= this.min[dim]) continue;
                return false;
            }
            return true;
        }
    }

    public static class LongRange {
        public final long min;
        public long max;

        LongRange(long value) {
            this(value, value);
        }

        LongRange(long min, long max) {
            this.min = min;
            this.max = max;
        }

        void expandToMax(long other) {
            this.max = other;
        }

        public boolean equals(Object other) {
            return other instanceof LongRange && this.equals((LongRange)other);
        }

        public boolean equals(LongRange other) {
            return this.min == other.min && this.max == other.max;
        }

        public int hashCode() {
            return (int)(this.min << (int)(16L + this.max));
        }

        public String toString() {
            return "LongRange(" + this.min + "," + this.max + ")";
        }
    }

    static abstract class CurveRule {
        final int dimension;
        final int[] npointValues;

        CurveRule(int dimension, int[] npointValues) {
            this.dimension = dimension;
            this.npointValues = npointValues;
            assert (npointValues.length == this.length());
        }

        final int length() {
            return (int)Math.pow(2.0, this.dimension);
        }

        int npointForIndex(int derivedIndex) {
            return this.npointValues[derivedIndex];
        }

        int indexForNPoint(int npoint) {
            for (int index = 0; index < this.npointValues.length; ++index) {
                if (this.npointValues[index] != npoint) continue;
                return index;
            }
            return -1;
        }

        abstract CurveRule childAt(int var1);
    }
}

