/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.export;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.export.PushToCloudConsole;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.database.NormalizedDatabaseName;
import picocli.CommandLine;

@CommandLine.Command(name="upload", description={"Push a local database to a Neo4j Aura instance. The target location is a Neo4j Aura Bolt URI. If Neo4j Cloud username and password are not provided either as a command option or as an environment variable, they will be requested interactively "})
public class UploadCommand
extends AbstractAdminCommand {
    private static final String DEV_MODE_VAR_NAME = "P2C_DEV_MODE";
    private final Copier copier;
    private final PushToCloudConsole cons;
    @CommandLine.Parameters(paramLabel="<database>", description={"Name of the database that should be uploaded. The name is used to select a dump file which is expected to be named <database>.dump."}, converter={Converters.DatabaseNameConverter.class})
    private NormalizedDatabaseName database;
    @CommandLine.Option(names={"--from-path"}, paramLabel="<path>", description={"'/path/to/directory-containing-dump' Path to a directory containing a database dump to upload."}, required=true)
    private Path dumpDirectory;
    @CommandLine.Option(names={"--to-uri"}, paramLabel="<uri>", arity="1", required=true, description={"'neo4j://mydatabaseid.databases.neo4j.io' Bolt URI of target database"})
    private String boltURI;
    @CommandLine.Option(names={"--to-user"}, defaultValue="${NEO4J_USERNAME}", description={"Username of the target database to push this database to. Prompt will ask for username if not provided. Alternatively NEO4J_USERNAME environment variable can be used."})
    private String username;
    @CommandLine.Option(names={"--to-password"}, defaultValue="${NEO4J_PASSWORD}", description={"Password of the target database to push this database to. Prompt will ask for password if not provided. Alternatively NEO4J_PASSWORD environment variable can be used."})
    private String password;
    @CommandLine.Option(names={"--overwrite-destination"}, arity="0..1", paramLabel="true|false", fallbackValue="true", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, description={"Overwrite the data in the target database."})
    private boolean overwrite;
    @CommandLine.Option(names={"--to"}, paramLabel="<destination>", description={"The destination for the upload."}, defaultValue="aura", showDefaultValue=CommandLine.Help.Visibility.ALWAYS)
    private String to;

    public UploadCommand(ExecutionContext ctx, Copier copier, PushToCloudConsole cons) {
        super(ctx);
        this.copier = copier;
        this.cons = cons;
    }

    public static long readSizeFromDumpMetaData(ExecutionContext ctx, Path dump) {
        Loader.DumpMetaData metaData;
        try {
            FileSystemAbstraction fileSystem = ctx.fs();
            metaData = new Loader(fileSystem, System.out).getMetaData(() -> fileSystem.openAsInputStream(dump));
        }
        catch (IOException e) {
            throw new CommandFailedException("Unable to check size of database dump.", (Throwable)e);
        }
        return Long.parseLong(metaData.byteCount());
    }

    public static String sizeText(long size) {
        return String.format("%.1f GB", UploadCommand.bytesToGibibytes(size));
    }

    public static double bytesToGibibytes(long sizeInBytes) {
        return (double)sizeInBytes / 1.073741824E9;
    }

    public void execute() {
        try {
            char[] pass;
            if (!"aura".equals(this.to)) {
                throw new CommandFailedException(String.format("'%s' is not a supported destination. Supported destinations are: 'aura'", this.to));
            }
            if (StringUtils.isBlank((CharSequence)this.username) && StringUtils.isBlank((CharSequence)(this.username = this.cons.readLine("%s", "Neo4j aura username (default: neo4j):")))) {
                this.username = "neo4j";
            }
            if (StringUtils.isBlank((CharSequence)this.password)) {
                pass = this.cons.readPassword("Neo4j aura password for %s:", this.username);
                if (pass.length == 0) {
                    throw new CommandFailedException("Please supply a password, either by '--to-password' parameter, 'NEO4J_PASSWORD' environment variable, or prompt");
                }
            } else {
                pass = this.password.toCharArray();
            }
            boolean devMode = this.cons.readDevMode(DEV_MODE_VAR_NAME);
            String consoleURL = this.buildConsoleURI(this.boltURI, devMode);
            String bearerToken = this.copier.authenticate(this.verbose, consoleURL, this.username, pass, this.overwrite);
            DumpUploader uploader = this.makeDumpUploader(this.dumpDirectory, this.database.name());
            ((Uploader)uploader).process(consoleURL, bearerToken);
        }
        catch (Exception e) {
            throw new CommandFailedException(e.getMessage());
        }
    }

    private void verbose(String format, Object ... args) {
        if (this.verbose) {
            this.ctx.out().printf(format, args);
        }
    }

    String buildConsoleURI(String boltURI, boolean devMode) throws CommandFailedException {
        UrlMatcher[] urlMatcherArray;
        if (devMode) {
            UrlMatcher[] urlMatcherArray2 = new UrlMatcher[3];
            urlMatcherArray2[0] = new DevMatcher();
            urlMatcherArray2[1] = new ProdMatcher();
            urlMatcherArray = urlMatcherArray2;
            urlMatcherArray2[2] = new PrivMatcher();
        } else {
            UrlMatcher[] urlMatcherArray3 = new UrlMatcher[2];
            urlMatcherArray3[0] = new ProdMatcher();
            urlMatcherArray = urlMatcherArray3;
            urlMatcherArray3[1] = new PrivMatcher();
        }
        UrlMatcher[] matchers = urlMatcherArray;
        return Arrays.stream(matchers).filter(m -> m.match(boltURI)).findFirst().orElseThrow(() -> new CommandFailedException("Invalid Bolt URI '" + boltURI + "'")).consoleUrl();
    }

    public DumpUploader makeDumpUploader(Path dump, String database) {
        if (!this.ctx.fs().isDirectory(dump)) {
            throw new CommandFailedException(String.format("The provided source directory '%s' doesn't exist", dump));
        }
        Path dumpFile = dump.resolve(database + ".dump");
        if (!this.ctx.fs().fileExists(dumpFile)) {
            throw new CommandFailedException(String.format("Dump file '%s' does not exist", dumpFile.toAbsolutePath()));
        }
        return new DumpUploader(new Source(this.ctx.fs(), dumpFile, this.dumpSize(dumpFile)));
    }

    private long dumpSize(Path dump) {
        long sizeInBytes = UploadCommand.readSizeFromDumpMetaData(this.ctx, dump);
        this.verbose("Determined DumpSize=%d bytes from dump at %s\n", sizeInBytes, dump);
        return sizeInBytes;
    }

    public static interface Copier {
        public String authenticate(boolean var1, String var2, String var3, char[] var4, boolean var5) throws CommandFailedException;

        public void copy(boolean var1, String var2, String var3, Source var4, boolean var5, String var6) throws CommandFailedException;

        public void checkSize(boolean var1, String var2, long var3, String var5) throws CommandFailedException;
    }

    class DumpUploader
    extends Uploader {
        DumpUploader(Source source) {
            super(source);
        }

        @Override
        void process(String consoleURL, String bearerToken) {
            UploadCommand.this.verbose("Checking database size %s fits at %s\n", UploadCommand.sizeText(this.size()), consoleURL);
            UploadCommand.this.copier.checkSize(UploadCommand.this.verbose, consoleURL, this.size(), bearerToken);
            UploadCommand.this.verbose("Uploading data of %s to %s\n", UploadCommand.sizeText(this.size()), consoleURL);
            UploadCommand.this.copier.copy(UploadCommand.this.verbose, consoleURL, UploadCommand.this.boltURI, this.source, false, bearerToken);
        }
    }

    static abstract class Uploader {
        protected final Source source;

        Uploader(Source source) {
            this.source = source;
        }

        long size() {
            return this.source.size();
        }

        abstract void process(String var1, String var2);
    }

    static abstract class UrlMatcher {
        protected Matcher matcher;
        protected String url;

        UrlMatcher() {
        }

        protected abstract Pattern pattern();

        public abstract String consoleUrl();

        public boolean match(String url) {
            this.url = url;
            this.matcher = this.pattern().matcher(url);
            return this.matcher.matches();
        }
    }

    static class DevMatcher
    extends UrlMatcher {
        DevMatcher() {
        }

        @Override
        protected Pattern pattern() {
            return Pattern.compile("(?:bolt(?:\\+routing)?|neo4j(?:\\+s|\\+ssc)?)://([^-]+)(-(.+))?.databases.neo4j(-(.+))?.io$");
        }

        @Override
        public String consoleUrl() {
            String databaseId = this.matcher.group(1);
            String environment = this.matcher.group(2);
            String domain = "";
            if (environment == null) {
                throw new CommandFailedException("Expected to find an environment running in dev mode in bolt URI: " + this.url);
            }
            if (this.matcher.groupCount() == 5) {
                domain = this.matcher.group(4);
            }
            return String.format("https://console%s.neo4j%s.io/v1/databases/%s", environment, domain, databaseId);
        }
    }

    class ProdMatcher
    extends UrlMatcher {
        ProdMatcher() {
        }

        @Override
        protected Pattern pattern() {
            return Pattern.compile("(?:bolt(?:\\+routing)?|neo4j(?:\\+s|\\+ssc)?)://([^-]+)(-(.+))?.databases.neo4j.io$");
        }

        @Override
        public String consoleUrl() {
            String databaseId = this.matcher.group(1);
            String environment = this.matcher.group(2);
            return String.format("https://console%s.neo4j.io/v1/databases/%s", environment == null ? "" : environment, databaseId);
        }
    }

    static class PrivMatcher
    extends UrlMatcher {
        PrivMatcher() {
        }

        @Override
        protected Pattern pattern() {
            return Pattern.compile("(?:bolt(?:\\+routing)?|neo4j(?:\\+s|\\+ssc)?)://([a-zA-Z0-9]+)\\.(\\S+)-orch-(\\d+).neo4j(-\\S+)?.io$");
        }

        @Override
        public String consoleUrl() {
            String databaseId = this.matcher.group(1);
            Object environment = this.matcher.group(2);
            String domain = "";
            switch (environment) {
                case "production": {
                    environment = "";
                    break;
                }
                case "staging": 
                case "prestaging": {
                    environment = "-" + (String)environment;
                    break;
                }
                default: {
                    environment = "-" + (String)environment;
                    if (this.matcher.group(4) == null) {
                        throw new CommandFailedException("Invalid Bolt URI '" + this.url + "'");
                    }
                    domain = this.matcher.group(4);
                }
            }
            return String.format("https://console%s.neo4j%s.io/v1/databases/%s", environment, domain, databaseId);
        }
    }

    public record Source(FileSystemAbstraction fs, Path path, long size) {
        long crc32Sum() throws IOException {
            CRC32 crc = new CRC32();
            try (InputStream inputStream = this.fs.openAsInputStream(this.path);){
                int cnt;
                while ((cnt = inputStream.read()) != -1) {
                    crc.update(cnt);
                }
            }
            return crc.getValue();
        }
    }
}

