/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.export;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.time.Clock;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.export.AuraResponse;
import org.neo4j.export.CommandResponseHandler;
import org.neo4j.export.ProgressTrackingOutputStream;
import org.neo4j.export.SignedUploadURLFactory;
import org.neo4j.export.UploadCommand;
import org.neo4j.export.Util;
import org.neo4j.internal.helpers.progress.ProgressListener;
import org.neo4j.internal.helpers.progress.ProgressMonitorFactory;
import org.neo4j.time.Clocks;

public class AuraClient {
    static final int HTTP_UNPROCESSABLE_ENTITY = 422;
    static final String ERROR_REASON_EXCEEDS_MAX_SIZE = "ImportExceedsMaxSize";
    static final int HTTP_TOO_MANY_REQUESTS = 429;
    private static final long DEFAULT_MAXIMUM_RETRIES = 50L;
    private static final long DEFAULT_MAXIMUM_RETRY_BACKOFF_MILLIS = TimeUnit.SECONDS.toMillis(64L);
    private final String consoleURL;
    private final String username;
    private final char[] password;
    private final boolean consentConfirmed;
    private final String boltURI;
    private final Clock clock;
    private final ProgressListenerFactory progressListenerFactory;
    private final CommandResponseHandler commandResponseHandler;
    private final ExecutionContext ctx;
    private final Sleeper sleeper;
    private boolean verbose;

    public AuraClient(AuraClientBuilder auraClientBuilder) {
        this.ctx = auraClientBuilder.ctx;
        this.consoleURL = auraClientBuilder.consoleURL;
        this.username = auraClientBuilder.username;
        this.password = auraClientBuilder.password;
        this.consentConfirmed = auraClientBuilder.consentConfirmed;
        this.boltURI = auraClientBuilder.boltURI;
        this.sleeper = auraClientBuilder.sleeper;
        this.clock = auraClientBuilder.clock;
        this.progressListenerFactory = auraClientBuilder.progressListenerFactory;
        this.commandResponseHandler = auraClientBuilder.commandResponseHandler;
    }

    public String getConsoleURL() {
        return this.consoleURL;
    }

    public String authenticate(boolean verbose) throws CommandFailedException {
        try {
            return this.doAuthenticate(verbose);
        }
        catch (IOException e) {
            this.ctx.err().println("Failed to authenticate with the aura console");
            throw new CommandFailedException("Failed to authenticate", (Throwable)e);
        }
    }

    public AuraResponse.SignedURIBody initatePresignedUpload(long crc32Sum, long size, String bearerToken, String version) {
        URL importURL = Util.safeUrl(this.consoleURL + "/import");
        AuraResponse.SignedURIBody signedURIBody = this.retryOnUnavailable(() -> this.doInitatePresignedUpload(crc32Sum, size, bearerToken, version, importURL));
        return signedURIBody;
    }

    private AuraResponse.SignedURIBody doInitatePresignedUpload(long crc32Sum, long size, String bearerToken, String version, URL importURL) throws IOException {
        HttpURLConnection connection = (HttpURLConnection)importURL.openConnection();
        String bearerHeader = "Bearer " + bearerToken;
        try (Closeable c = connection::disconnect;){
            connection.setRequestMethod("POST");
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("Authorization", bearerHeader);
            connection.setRequestProperty("Accept", "application/json");
            connection.setRequestProperty("Neo4j-Version", version);
            connection.setDoOutput(true);
            try (OutputStream postData = connection.getOutputStream();){
                postData.write(String.format("{\"Crc32\":%d, \"FullSize\":%d}", crc32Sum, size).getBytes(StandardCharsets.UTF_8));
            }
            int responseCode = connection.getResponseCode();
            switch (responseCode) {
                case 301: 
                case 404: {
                    throw this.updatePluginErrorResponse(connection);
                }
                case 401: {
                    throw this.errorResponse(this.verbose, connection, "The given authorization token is invalid or has expired");
                }
                case 422: {
                    throw this.validationFailureErrorResponse(connection, size);
                }
                case 502: 
                case 503: 
                case 504: {
                    throw new RetryableHttpException(this.commandResponseHandler.unexpectedResponse(this.verbose, connection, "Initiating upload target"));
                }
                case 202: {
                    AuraResponse.SignedURIBody signedURIBody = this.extractSignedURIFromResponse(this.verbose, connection);
                    return signedURIBody;
                }
            }
            throw this.commandResponseHandler.unexpectedResponse(this.verbose, connection, "Initiating upload target");
        }
    }

    private AuraResponse.SignedURIBody extractSignedURIFromResponse(boolean verbose, HttpURLConnection connection) throws IOException {
        try (InputStream responseData = connection.getInputStream();){
            String json = new String(IOUtils.toByteArray((InputStream)responseData), StandardCharsets.UTF_8);
            this.commandResponseHandler.debug(verbose, "Got json '" + json + "' back expecting to contain the signed URL");
            AuraResponse.SignedURIBody signedURIBody = Util.parseJsonUsingJacksonParser(json, AuraResponse.SignedURIBody.class);
            return signedURIBody;
        }
    }

    private String doAuthenticate(boolean verbose) throws IOException {
        URL url = Util.safeUrl(this.consoleURL + "/import/auth");
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        Closeable c = connection::disconnect;
        connection.setRequestMethod("POST");
        connection.setRequestProperty("Authorization", "Basic " + Util.base64Encode(this.username, this.password));
        connection.setRequestProperty("Accept", "application/json");
        connection.setRequestProperty("Confirmed", String.valueOf(this.consentConfirmed));
        int responseCode = connection.getResponseCode();
        switch (responseCode) {
            case 404: {
                throw this.errorResponse(verbose, connection, "We encountered a problem while contacting your Neo4j Aura instance, please check your Bolt URI");
            }
            case 301: {
                throw this.updatePluginErrorResponse(connection);
            }
            case 401: {
                throw this.errorResponse(verbose, connection, "Invalid username/password credentials");
            }
            case 403: {
                throw this.errorResponse(verbose, connection, "The credentials provided do not give administrative access to the target database");
            }
            case 409: {
                throw this.errorResponse(verbose, connection, "No consent to overwrite database. Aborting");
            }
            case 502: 
            case 503: 
            case 504: {
                throw new SignedUploadURLFactory.RetryableHttpException(this.commandResponseHandler.unexpectedResponse(verbose, connection, "Authorization"));
            }
            case 200: {
                try (InputStream responseData = connection.getInputStream();){
                    String json = new String(IOUtils.toByteArray((InputStream)responseData), StandardCharsets.UTF_8);
                    this.commandResponseHandler.debug(true, "Got json response back from authorize request");
                    String string = Util.parseJsonUsingJacksonParser((String)json, TokenBody.class).Token;
                    return string;
                }
            }
        }
        throw this.commandResponseHandler.unexpectedResponse(verbose, connection, "Authorization");
        finally {
            if (c != null) {
                c.close();
            }
        }
    }

    <T> T retryOnUnavailable(IOExceptionSupplier<T> runnableCommand) {
        int attempt = 0;
        Throwable lastException = null;
        while (true) {
            try {
                return runnableCommand.get();
            }
            catch (RetryableHttpException e) {
                if ((long)attempt >= 50L) break;
                ThreadLocalRandom random = ThreadLocalRandom.current();
                long backoffFromRetryCount = TimeUnit.SECONDS.toMillis(1L << attempt++) + (long)random.nextInt(1000);
                try {
                    this.sleeper.sleep(Long.min(backoffFromRetryCount, DEFAULT_MAXIMUM_RETRY_BACKOFF_MILLIS));
                }
                catch (InterruptedException ex) {
                    throw new CommandFailedException(e.getMessage(), (Throwable)e);
                }
                lastException = e;
                continue;
            }
            catch (IOException e) {
                throw new CommandFailedException(e.getMessage(), (Throwable)e);
            }
            break;
        }
        throw (RuntimeException)lastException.getCause();
    }

    public void checkSize(boolean verbose, long size, String bearerToken) {
        this.retryOnUnavailable(() -> {
            this.doCheckSize(verbose, size, bearerToken);
            return null;
        });
    }

    private void doCheckSize(boolean verbose, long size, String bearerToken) throws IOException {
        URL url = Util.safeUrl(this.consoleURL + "/import/size");
        String bearerTokenHeader = "Bearer " + bearerToken;
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        try (Closeable c = connection::disconnect;){
            connection.setDoOutput(true);
            connection.setRequestMethod("POST");
            connection.setRequestProperty("Authorization", bearerTokenHeader);
            connection.setRequestProperty("Content-Type", "application/json");
            try (OutputStream postData = connection.getOutputStream();){
                postData.write(String.format("{\"FullSize\":%d}", size).getBytes(StandardCharsets.UTF_8));
            }
            int responseCode = connection.getResponseCode();
            switch (responseCode) {
                case 422: {
                    throw this.validationFailureErrorResponse(connection, size);
                }
                case 200: {
                    return;
                }
                case 502: 
                case 503: 
                case 504: {
                    throw new RetryableHttpException(this.commandResponseHandler.unexpectedResponse(verbose, connection, "Size check"));
                }
            }
            throw this.commandResponseHandler.unexpectedResponse(verbose, connection, "Size check");
        }
    }

    public void doStatusPolling(boolean verbose, String bearerToken, long fileSize) throws InterruptedException {
        this.ctx.out().println("We have received your export and it is currently being loaded into your Aura instance.");
        this.ctx.out().println("You can wait here, or abort this command and head over to the console to be notified of when your database is running.");
        String bearerTokenHeader = "Bearer " + bearerToken;
        ProgressTrackingOutputStream.Progress statusProgress = new ProgressTrackingOutputStream.Progress(this.progressListenerFactory.create("Import progress (estimated)", 100L), 0L);
        boolean importHasStarted = false;
        long importStarted = this.clock.millis();
        double importTimeEstimateMinutes = 5.0 + 3.0 * UploadCommand.bytesToGibibytes(fileSize);
        long importTimeEstimateMillis = TimeUnit.SECONDS.toMillis((long)(importTimeEstimateMinutes * 60.0));
        long importStartedTimeout = importStarted + 90000L;
        this.commandResponseHandler.debug(verbose, String.format("Rough guess for how long dump file import will take: %.0f minutes; file size is %.1f GB (%d bytes)", importTimeEstimateMinutes, UploadCommand.bytesToGibibytes(fileSize), fileSize));
        while (!statusProgress.isDone()) {
            StatusBody statusBody = this.getDatabaseStatus(verbose, Util.safeUrl(this.consoleURL + "/import/status"), bearerTokenHeader);
            switch (statusBody.Status) {
                case "running": {
                    if (importHasStarted) {
                        statusProgress.rewindTo(0L);
                        statusProgress.add(100);
                        statusProgress.done();
                        break;
                    }
                    this.throwIfImportDidNotStart(importStartedTimeout);
                    break;
                }
                case "loading failed": {
                    if (importHasStarted) {
                        throw this.formatCommandFailedExceptionError(statusBody.Error.getMessage(), statusBody.Error.getUrl());
                    }
                    this.throwIfImportDidNotStart(importStartedTimeout);
                    break;
                }
                default: {
                    importHasStarted = true;
                    long elapsed = this.clock.millis() - importStarted;
                    statusProgress.rewindTo(0L);
                    statusProgress.add(this.importStatusProgressEstimate(statusBody.Status, elapsed, importTimeEstimateMillis));
                }
            }
            this.sleeper.sleep(2000L);
        }
        this.ctx.out().println("Your data was successfully pushed to Aura and is now running.");
        long importDurationMillis = this.clock.millis() - importStarted;
        this.commandResponseHandler.debug(verbose, String.format("Import took about %d minutes to complete excluding upload (%d ms)", TimeUnit.MILLISECONDS.toMinutes(importDurationMillis), importDurationMillis));
    }

    private CommandFailedException formatCommandFailedExceptionError(String message, String url) {
        if (StringUtils.isEmpty((CharSequence)url)) {
            return new CommandFailedException(message);
        }
        String trimmedMessage = StringUtils.removeEnd((String)message, (String)".");
        return new CommandFailedException(String.format("Error: %s. See: %s", trimmedMessage, url));
    }

    int importStatusProgressEstimate(String databaseStatus, long elapsed, long importTimeEstimateMillis) {
        switch (databaseStatus) {
            case "running": {
                return 0;
            }
            case "updating": 
            case "loading": {
                int loadProgressEstimation = (int)Math.min(98L, elapsed * 98L / importTimeEstimateMillis);
                return 1 + loadProgressEstimation;
            }
        }
        throw new CommandFailedException(String.format("We're sorry, something has failed during the loading of your database. Please try again and if this problem persists, please open up a support case. Database status: %s", databaseStatus));
    }

    private StatusBody doGetDatabaseStatus(boolean verbose, URL statusURL, String bearerToken) throws IOException {
        HttpURLConnection connection = (HttpURLConnection)statusURL.openConnection();
        Closeable c = connection::disconnect;
        connection.setRequestMethod("GET");
        connection.setRequestProperty("Authorization", bearerToken);
        connection.setDoOutput(true);
        int responseCode = connection.getResponseCode();
        switch (responseCode) {
            case 301: 
            case 404: {
                throw this.updatePluginErrorResponse(connection);
            }
            case 200: {
                try (InputStream responseData = connection.getInputStream();){
                    String json = new String(IOUtils.toByteArray((InputStream)responseData), StandardCharsets.UTF_8);
                    StatusBody statusBody = Util.parseJsonUsingJacksonParser(json, StatusBody.class);
                    return statusBody;
                }
            }
            case 502: 
            case 503: 
            case 504: {
                throw new RetryableHttpException(this.commandResponseHandler.unexpectedResponse(verbose, connection, "Trigger import/restore after successful upload"));
            }
        }
        throw this.commandResponseHandler.unexpectedResponse(verbose, connection, "Trigger import/restore after successful upload");
        finally {
            if (c != null) {
                c.close();
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void triggerImportProtocol(boolean verbose, Path source, long crc32Sum, String bearerToken) throws IOException {
        URL completeImportURL = Util.safeUrl(this.consoleURL + "/import/upload-complete");
        HttpURLConnection connection = (HttpURLConnection)completeImportURL.openConnection();
        String bearerHeader = "Bearer " + bearerToken;
        try (Closeable c = connection::disconnect;){
            connection.setRequestMethod("POST");
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("Authorization", bearerHeader);
            connection.setDoOutput(true);
            try (OutputStream postData = connection.getOutputStream();){
                postData.write(String.format("{\"Crc32\":%d}", crc32Sum).getBytes(StandardCharsets.UTF_8));
            }
            int responseCode = connection.getResponseCode();
            switch (responseCode) {
                case 301: 
                case 404: {
                    throw this.updatePluginErrorResponse(connection);
                }
                case 429: {
                    throw this.resumePossibleErrorResponse(connection, source);
                }
                case 409: {
                    throw this.errorResponse(verbose, connection, "The target database contained data and consent to overwrite the data was not given. Aborting");
                }
                case 200: {
                    return;
                }
                default: {
                    throw this.resumePossibleErrorResponse(connection, source);
                }
            }
        }
    }

    private void throwIfImportDidNotStart(long importStartedTimeout) {
        boolean passedStartImportTimeout;
        boolean bl = passedStartImportTimeout = this.clock.millis() > importStartedTimeout;
        if (passedStartImportTimeout) {
            throw new CommandFailedException("Timed out waiting for database load to start as the database did not enter 'loading' state in time. Please retry the operation. You might find more information about the failure on the database status page in https://console.neo4j.io.");
        }
    }

    private StatusBody getDatabaseStatus(boolean verbose, URL statusURL, String bearerToken) {
        return this.retryOnUnavailable(() -> this.doGetDatabaseStatus(verbose, statusURL, bearerToken));
    }

    private CommandFailedException resumePossibleErrorResponse(HttpURLConnection connection, Path dump) throws IOException {
        this.commandResponseHandler.debugErrorResponse(true, connection);
        return new CommandFailedException("We encountered a problem while communicating to the Neo4j Aura system. \nYou can re-try using the existing dump by running this command: \n" + String.format("neo4j-admin push-to-cloud --%s=%s --%s=%s", "dump", dump.toAbsolutePath(), "bolt-uri", this.boltURI));
    }

    private CommandFailedException errorResponse(boolean verbose, HttpURLConnection connection, String errorDescription) throws IOException {
        this.commandResponseHandler.debugErrorResponse(verbose, connection);
        return new CommandFailedException(errorDescription);
    }

    private CommandFailedException updatePluginErrorResponse(HttpURLConnection connection) throws IOException {
        this.commandResponseHandler.debugErrorResponse(true, connection);
        return new CommandFailedException("We encountered a problem while communicating to the Neo4j Aura system. Please check that you are using the latest version of the push-to-cloud plugin and upgrade if necessary. If this problem persists after upgrading, please contact support and attach the logs shown below to your ticket in the support portal.");
    }

    private CommandFailedException validationFailureErrorResponse(HttpURLConnection connection, long size) throws IOException {
        try (InputStream responseData = connection.getErrorStream();){
            String responseString = new String(IOUtils.toByteArray((InputStream)responseData), StandardCharsets.UTF_8);
            this.commandResponseHandler.debugResponse(responseString, connection, true);
            ErrorBody errorBody = Util.parseJsonUsingJacksonParser(responseString, ErrorBody.class);
            String message = errorBody.getMessage();
            if (ERROR_REASON_EXCEEDS_MAX_SIZE.equals(errorBody.getReason())) {
                String trimmedMessage = StringUtils.removeEnd((String)message, (String)".");
                message = String.format("%s. Minimum storage space required: %s", trimmedMessage, UploadCommand.sizeText(size));
            }
            CommandFailedException commandFailedException = this.formatCommandFailedExceptionError(message, errorBody.getUrl());
            return commandFailedException;
        }
    }

    public static class AuraClientBuilder {
        ExecutionContext ctx;
        private String consoleURL;
        private String username;
        private char[] password;
        private boolean consentConfirmed;
        private String boltURI;
        private Clock clock;
        private Sleeper sleeper;
        private ProgressListenerFactory progressListenerFactory;
        private CommandResponseHandler commandResponseHandler;

        public AuraClientBuilder(ExecutionContext ctx) {
            this.ctx = ctx;
        }

        public AuraClientBuilder withConsoleURL(String consoleURL) {
            this.consoleURL = consoleURL;
            return this;
        }

        public AuraClientBuilder withUserName(String username) {
            this.username = username;
            return this;
        }

        public AuraClientBuilder withPassword(char[] password) {
            this.password = password;
            return this;
        }

        public AuraClientBuilder withConsent(boolean consentConfirmed) {
            this.consentConfirmed = consentConfirmed;
            return this;
        }

        public AuraClientBuilder withBoltURI(String boltURI) {
            this.boltURI = boltURI;
            return this;
        }

        public AuraClientBuilder withClock(Clock clock) {
            this.clock = clock;
            return this;
        }

        public AuraClientBuilder withSleeper(Sleeper sleeper) {
            this.sleeper = sleeper;
            return this;
        }

        public AuraClientBuilder withCommandResponseHandler(CommandResponseHandler commandResponseHandler) {
            this.commandResponseHandler = commandResponseHandler;
            return this;
        }

        public AuraClientBuilder withProgressListenerFactory(ProgressListenerFactory progressListenerFactory) {
            this.progressListenerFactory = progressListenerFactory;
            return this;
        }

        public AuraClientBuilder withDefaults() {
            if (this.sleeper == null) {
                this.sleeper = Thread::sleep;
            }
            if (this.clock == null) {
                this.clock = Clocks.nanoClock();
            }
            this.commandResponseHandler = new CommandResponseHandler(this.ctx);
            this.progressListenerFactory = (text, length) -> ProgressMonitorFactory.textual((OutputStream)this.ctx.out()).singlePart(text, length);
            return this;
        }

        public AuraClient build() {
            return new AuraClient(this);
        }
    }

    static interface Sleeper {
        public void sleep(long var1) throws InterruptedException;
    }

    public static interface ProgressListenerFactory {
        public ProgressListener create(String var1, long var2);
    }

    private static interface IOExceptionSupplier<T> {
        public T get() throws IOException;
    }

    static class RetryableHttpException
    extends RuntimeException {
        RetryableHttpException(CommandFailedException e) {
            super((Throwable)e);
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    private static class TokenBody {
        public String Token;

        private TokenBody() {
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    static class StatusBody {
        public String Status;
        public ErrorBody Error = new ErrorBody();

        StatusBody() {
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    static class ErrorBody {
        private static final String DEFAULT_MESSAGE = "an unexpected problem ocurred, please contact customer support for assistance";
        private static final String DEFAULT_REASON = "UnknownError";
        private final String message;
        private final String reason;
        private final String url;

        ErrorBody() {
            this(null, null, null);
        }

        @JsonCreator
        ErrorBody(@JsonProperty(value="Message") String message, @JsonProperty(value="Reason") String reason, @JsonProperty(value="Url") String url) {
            this.message = message;
            this.reason = reason;
            this.url = url;
        }

        public String getMessage() {
            return (String)StringUtils.defaultIfBlank((CharSequence)this.message, (CharSequence)DEFAULT_MESSAGE);
        }

        public String getReason() {
            return (String)StringUtils.defaultIfBlank((CharSequence)this.reason, (CharSequence)DEFAULT_REASON);
        }

        public String getUrl() {
            return this.url;
        }
    }
}

