/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.pushtocloud;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.function.LongConsumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.cli.AbstractCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.kernel.database.NormalizedDatabaseName;
import org.neo4j.pushtocloud.PushToCloudConsole;
import picocli.CommandLine;

@CommandLine.Command(name="push-to-cloud", description={"Push your local database to a Neo4j Aura instance. The database must be shutdown in order to take a dump to upload. The target location is your Neo4j Aura Bolt URI. You will be asked your Neo4j Cloud username and password during the push-to-cloud operation."})
public class PushToCloudCommand
extends AbstractCommand {
    private final Copier copier;
    private final DumpCreator dumpCreator;
    private final PushToCloudConsole cons;
    @CommandLine.Option(names={"--database"}, description={"Name of the database to push. Defaults to neo4j. This argument cannot be used together with --dump."}, converter={Converters.DatabaseNameConverter.class})
    private NormalizedDatabaseName database;
    @CommandLine.Option(names={"--dump"}, description={"'/path/to/my-neo4j-database-dump-file' Path to an existing database dump for upload. This argument cannot be used together with --database."})
    private Path dump;
    @CommandLine.Option(names={"--temp-file-location", "--dump-to"}, description={"'/path/to/temp-file' Target path for temporary database dump file to be uploaded. Used in combination with the --database argument."})
    private Path tmpDumpFile;
    @CommandLine.Option(names={"--bolt-uri"}, arity="1", required=true, description={"'neo4j://mydatabaseid.databases.neo4j.io' Bolt URI of target database"})
    private String boltURI;
    @CommandLine.Option(names={"--username"}, defaultValue="${NEO4J_USERNAME}", description={"Optional: Username of the target database to push this database to. Prompt will ask for username if not provided. Alternatively NEO4J_USERNAME environment variable can be used."})
    private String username;
    @CommandLine.Option(names={"--password"}, defaultValue="${NEO4J_PASSWORD}", description={"Optional: Password of the target database to push this database to. Prompt will ask for password if not provided. Alternatively NEO4J_PASSWORD environment variable can be used."})
    private String password;
    @CommandLine.Option(names={"--overwrite"}, description={"Optional: Overwrite the data in the target database."})
    private boolean overwrite;
    private static final double ACCEPTABLE_DUMP_CHANGE = 0.1;

    public PushToCloudCommand(ExecutionContext ctx, Copier copier, DumpCreator dumpCreator, PushToCloudConsole cons) {
        super(ctx);
        this.copier = copier;
        this.dumpCreator = dumpCreator;
        this.cons = cons;
    }

    public void execute() {
        try {
            char[] pass;
            if ((this.database == null || StringUtils.isBlank((CharSequence)this.database.name())) && (this.dump == null || StringUtils.isBlank((CharSequence)this.dump.toString()))) {
                this.database = new NormalizedDatabaseName("neo4j");
            }
            if (StringUtils.isBlank((CharSequence)this.username) && StringUtils.isBlank((CharSequence)(this.username = this.cons.readLine("%s", "Neo4j aura username (default: neo4j):")))) {
                this.username = "neo4j";
            }
            if (StringUtils.isBlank((CharSequence)this.password)) {
                pass = this.cons.readPassword("Neo4j aura password for %s:", this.username);
                if (pass.length == 0) {
                    throw new CommandFailedException("Please supply a password, either by '--password' parameter, 'NEO4J_PASSWORD' environment variable, or prompt");
                }
            } else {
                pass = this.password.toCharArray();
            }
            String consoleURL = this.buildConsoleURI(this.boltURI);
            String bearerToken = this.copier.authenticate(this.verbose, consoleURL, this.username, pass, this.overwrite);
            Uploader uploader = this.prepareUploader(this.dump, this.database, this.tmpDumpFile);
            uploader.process(consoleURL, bearerToken);
        }
        catch (Exception e) {
            if (this.verbose) {
                e.printStackTrace(this.ctx.out());
            }
            throw e;
        }
    }

    private void verbose(String format, Object ... args) {
        if (this.verbose) {
            this.ctx.out().printf(format, args);
        }
    }

    private String buildConsoleURI(String boltURI) throws CommandFailedException {
        Pattern pattern = Pattern.compile("(?:bolt(?:\\+routing)?|neo4j(?:\\+s|\\+ssc)?)://([^-]+)(-(.+))?.databases.neo4j.io$");
        Matcher matcher = pattern.matcher(boltURI);
        if (!matcher.matches()) {
            throw new CommandFailedException("Invalid Bolt URI '" + boltURI + "'");
        }
        String databaseId = matcher.group(1);
        String environment = matcher.group(2);
        return String.format("https://console%s.neo4j.io/v1/databases/%s", environment == null ? "" : environment, databaseId);
    }

    private Uploader prepareUploader(Path dump, NormalizedDatabaseName database, Path to) throws CommandFailedException {
        if (dump != null && database != null) {
            throw new CommandFailedException("Provide either a dump or database name, not both");
        }
        if (dump != null) {
            return this.makeDumpUploader(dump);
        }
        return this.makeFullUploader(to);
    }

    public DumpUploader makeDumpUploader(Path dump) {
        if (Files.notExists(dump, new LinkOption[0])) {
            throw new CommandFailedException(String.format("The provided dump '%s' file doesn't exist", dump));
        }
        return new DumpUploader(new Source(dump, this.dumpSize(dump)));
    }

    public FullUploader makeFullUploader(Path to) {
        Path dumpPath;
        Path path = dumpPath = to != null ? to : this.ctx.homeDir().resolve("dump-of-" + this.database.name() + "-" + System.currentTimeMillis());
        if (Files.exists(dumpPath, new LinkOption[0])) {
            throw new CommandFailedException(String.format("The provided dump-to target '%s' file already exists", dumpPath));
        }
        return new FullUploader(new Source(dumpPath, this.fullSize(this.ctx, this.database)));
    }

    private long fullSize(ExecutionContext ctx, NormalizedDatabaseName database) {
        Path configFile = ctx.confDir().resolve("neo4j.conf");
        DatabaseLayout layout = Neo4jLayout.of((Config)this.getConfig(configFile)).databaseLayout(database.name());
        long storeFilesSize = FileUtils.sizeOf((File)layout.databaseDirectory().toFile());
        long txLogSize = this.readTxLogsSize(layout.getTransactionLogsDirectory());
        long size = txLogSize + storeFilesSize;
        this.verbose("Determined FullSize=%d bytes from storeFileSize=%d + txLogSize=%d in database '%s'\n", size, storeFilesSize, txLogSize, database.name());
        return size;
    }

    private long dumpSize(Path dump) {
        long sizeInBytes = PushToCloudCommand.readSizeFromDumpMetaData(dump);
        this.verbose("Determined DumpSize=%d bytes from dump at %s\n", sizeInBytes, dump);
        return sizeInBytes;
    }

    private long readTxLogsSize(Path txLogs) {
        long txLogSize = 0L;
        if (Files.exists(txLogs, new LinkOption[0])) {
            if (Files.isDirectory(txLogs, new LinkOption[0])) {
                String[] logs = txLogs.toFile().list((dir, name) -> name.startsWith("neostore.transaction.db"));
                if (logs != null && logs.length > 0) {
                    TxSizeSetter setSize = new TxSizeSetter(logs.length);
                    Arrays.stream(logs).mapToLong(name -> new File(txLogs.toFile(), (String)name).length()).max().ifPresent(setSize);
                    txLogSize = setSize.txLogSize;
                }
            } else {
                throw new IllegalArgumentException("Cannot determine size of transaction logs: " + txLogs + " is not a directory");
            }
        }
        return txLogSize;
    }

    public static long readSizeFromDumpMetaData(Path dump) {
        Loader.DumpMetaData metaData;
        try {
            metaData = new Loader(System.out).getMetaData(() -> Files.newInputStream(dump, new OpenOption[0]));
        }
        catch (IOException e) {
            throw new CommandFailedException("Unable to check size of database dump.", (Throwable)e);
        }
        return Long.parseLong(metaData.byteCount);
    }

    private Config getConfig(Path configFile) {
        if (!this.ctx.fs().fileExists(configFile)) {
            throw new CommandFailedException("Unable to find config file, tried: " + configFile.toAbsolutePath());
        }
        try {
            return Config.newBuilder().fromFile(configFile).set(GraphDatabaseSettings.neo4j_home, (Object)this.ctx.homeDir().toAbsolutePath()).commandExpansion(this.allowCommandExpansion).build();
        }
        catch (Exception e) {
            throw new CommandFailedException("Failed to read config file: " + configFile.toAbsolutePath(), (Throwable)e);
        }
    }

    public static String sizeText(long size) {
        return String.format("%.1f GB", PushToCloudCommand.bytesToGibibytes(size));
    }

    public static double bytesToGibibytes(long sizeInBytes) {
        return (double)sizeInBytes / 1.073741824E9;
    }

    public static interface DumpCreator {
        public Path dumpDatabase(String var1, Path var2) throws CommandFailedException;
    }

    public static interface Copier {
        public String authenticate(boolean var1, String var2, String var3, char[] var4, boolean var5) throws CommandFailedException;

        public void copy(boolean var1, String var2, String var3, Source var4, boolean var5, String var6) throws CommandFailedException;

        public void checkSize(boolean var1, String var2, long var3, String var5) throws CommandFailedException;
    }

    public static class Source {
        private final Path path;
        private long size;

        public Source(Path path, long size) {
            this.path = path;
            this.size = size;
        }

        public Path path() {
            return this.path;
        }

        public long size() {
            return this.size;
        }

        protected void setSize(long newSize) {
            this.size = newSize;
        }

        long crc32Sum() throws IOException {
            CRC32 crc = new CRC32();
            try (BufferedInputStream inputStream = new BufferedInputStream(Files.newInputStream(this.path, new OpenOption[0]));){
                int cnt;
                while ((cnt = ((InputStream)inputStream).read()) != -1) {
                    crc.update(cnt);
                }
            }
            return crc.getValue();
        }

        public int hashCode() {
            return this.path.hashCode() + 31 * (int)this.size;
        }

        public boolean equals(Object obj) {
            if (obj instanceof Source) {
                Source other = (Source)obj;
                return this.path.equals(other.path) && this.size == other.size;
            }
            return false;
        }
    }

    private static class TxSizeSetter
    implements LongConsumer {
        long txLogSize;
        final long count;

        TxSizeSetter(int count) {
            this.count = count;
        }

        @Override
        public void accept(long size) {
            this.txLogSize = Math.min(10L, this.count) * size;
        }
    }

    class FullUploader
    extends Uploader {
        FullUploader(Source source) {
            super(source);
        }

        @Override
        void process(String consoleURL, String bearerToken) {
            PushToCloudCommand.this.verbose("Checking database size %s fits at %s\n", PushToCloudCommand.sizeText(this.size()), consoleURL);
            PushToCloudCommand.this.copier.checkSize(PushToCloudCommand.this.verbose, consoleURL, this.size(), bearerToken);
            Path dumpFile = PushToCloudCommand.this.dumpCreator.dumpDatabase(PushToCloudCommand.this.database.name(), this.path());
            long sizeFromDump = PushToCloudCommand.this.dumpSize(dumpFile);
            long sizeFromDatabase = this.size();
            PushToCloudCommand.this.verbose("Validating sizes: fromDump=%d, fromDatabase=%d", sizeFromDump, sizeFromDatabase);
            if ((double)Math.abs(sizeFromDump - sizeFromDatabase) > 0.1 * (double)sizeFromDatabase) {
                PushToCloudCommand.this.ctx.out().printf("Warning: unexpectedly large difference between size in dump, and original size: %d != %d", sizeFromDump, sizeFromDatabase);
            }
            this.source.setSize(sizeFromDump);
            PushToCloudCommand.this.verbose("Uploading data of %s to %s\n", PushToCloudCommand.sizeText(this.size()), consoleURL);
            PushToCloudCommand.this.copier.copy(PushToCloudCommand.this.verbose, consoleURL, PushToCloudCommand.this.boltURI, this.source, true, bearerToken);
        }
    }

    class DumpUploader
    extends Uploader {
        DumpUploader(Source source) {
            super(source);
        }

        @Override
        void process(String consoleURL, String bearerToken) {
            PushToCloudCommand.this.verbose("Checking database size %s fits at %s\n", PushToCloudCommand.sizeText(this.size()), consoleURL);
            PushToCloudCommand.this.copier.checkSize(PushToCloudCommand.this.verbose, consoleURL, this.size(), bearerToken);
            PushToCloudCommand.this.verbose("Uploading data of %s to %s\n", PushToCloudCommand.sizeText(this.size()), consoleURL);
            PushToCloudCommand.this.copier.copy(PushToCloudCommand.this.verbose, consoleURL, PushToCloudCommand.this.boltURI, this.source, false, bearerToken);
        }
    }

    static abstract class Uploader {
        protected final Source source;

        Uploader(Source source) {
            this.source = source;
        }

        long size() {
            return this.source.size();
        }

        Path path() {
            return this.source.path();
        }

        abstract void process(String var1, String var2);
    }
}

