/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.pushtocloud;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.neo4j.commandline.admin.AdminCommand;
import org.neo4j.commandline.admin.CommandFailed;
import org.neo4j.commandline.admin.IncorrectUsage;
import org.neo4j.commandline.admin.OutsideWorld;
import org.neo4j.commandline.arguments.Arguments;
import org.neo4j.commandline.arguments.MandatoryNamedArg;
import org.neo4j.commandline.arguments.NamedArgument;
import org.neo4j.commandline.arguments.OptionalNamedArg;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.Args;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.pushtocloud.DumpMetaData;

public class PushToCloudCommand
implements AdminCommand {
    static final String ARG_DATABASE = "database";
    static final String ARG_BOLT_URI = "bolt-uri";
    static final String ARG_DUMP = "dump";
    static final String ARG_DUMP_TO = "dump-to";
    static final String ARG_VERBOSE = "verbose";
    static final String ARG_OVERWRITE = "overwrite";
    static final String ARG_USERNAME = "username";
    static final String ARG_PASSWORD = "password";
    static final String ENV_USERNAME = "NEO4J_USERNAME";
    static final String ENV_PASSWORD = "NEO4J_PASSWORD";
    static final Arguments arguments = new Arguments().withDatabase().withArgument((NamedArgument)new OptionalNamedArg("dump", "/path/to/my-neo4j-database-dump-file", null, "Path to an existing database dump for upload. This arugment cannot be used together with --database.")).withArgument((NamedArgument)new OptionalNamedArg("dump-to", "/path/to/dump-file-to-be-created", null, "Target path for dump file. Used in combination with the --database argument.")).withArgument((NamedArgument)new MandatoryNamedArg("bolt-uri", "bolt+routing://mydatabaseid.databases.neo4j.io", "Bolt URI of target database")).withArgument((NamedArgument)new OptionalNamedArg("verbose", "true/false", null, "Enable verbose output.")).withArgument((NamedArgument)new OptionalNamedArg("username", "neo4j", null, "Optional: Username of the target database to push this database to. Prompt will ask for username if not provided. Alternatively NEO4J_USERNAME environment variable can be used.")).withArgument((NamedArgument)new OptionalNamedArg("password", "mYs3cr3tPa$$w0rd", null, "Optional: Password of the target database to push this database to. Prompt will ask for password if not provided. Alternatively NEO4J_PASSWORD environment variable can be used.")).withArgument((NamedArgument)new OptionalNamedArg("overwrite", "true/false", "false", "Optional: Overwrite the data in the target database."));
    private final Path homeDir;
    private final Path configDir;
    private final OutsideWorld outsideWorld;
    private final Copier copier;
    private final DumpCreator dumpCreator;
    private String database;
    private String dump;
    private String boltURI;
    private boolean verbose;
    private static double ACCEPTABLE_DUMP_CHANGE = 0.1;

    public PushToCloudCommand(Path homeDir, Path configDir, OutsideWorld outsideWorld, Copier copier, DumpCreator dumpCreator) {
        this.homeDir = homeDir;
        this.configDir = configDir;
        this.outsideWorld = outsideWorld;
        this.copier = copier;
        this.dumpCreator = dumpCreator;
    }

    public void execute(String[] args) throws IncorrectUsage, CommandFailed {
        Args arguments = Args.parse((String[])args);
        this.verbose = arguments.getBoolean(ARG_VERBOSE);
        try {
            String passwordFromArg = arguments.get(ARG_PASSWORD);
            String username = arguments.get(ARG_USERNAME);
            String usernameFromEnv = System.getenv(ENV_USERNAME);
            String passwordFromEnv = System.getenv(ENV_PASSWORD);
            if (username == null) {
                username = usernameFromEnv != null ? usernameFromEnv : this.outsideWorld.promptLine("Neo4j Aura database username (default: neo4j): ", new Object[0]);
            }
            if (username == null || "".equals(username)) {
                username = "neo4j";
            }
            char[] password = passwordFromArg != null ? passwordFromArg.toCharArray() : (passwordFromEnv != null ? passwordFromEnv.toCharArray() : this.outsideWorld.promptPassword(String.format("Neo4j Aura database password for %s: ", username), new Object[0]));
            this.boltURI = arguments.get(ARG_BOLT_URI);
            if (this.boltURI == null || "".equals(this.boltURI)) {
                this.boltURI = this.outsideWorld.promptLine("Neo4j Aura database Bolt URI: ", new Object[0]);
            }
            if (this.boltURI == null || "".equals(this.boltURI)) {
                throw new IncorrectUsage("Please provide a Neo4j Aura Bolt URI of the target location to push the database to, using the --bolt-uri argument.");
            }
            String confirmationViaArgument = arguments.get(ARG_OVERWRITE, "false", "true");
            String consoleURL = this.buildConsoleURI(this.boltURI);
            String bearerToken = this.copier.authenticate(this.verbose, consoleURL, username, password, "true".equals(confirmationViaArgument));
            String tmpDumpFile = arguments.get(ARG_DUMP_TO);
            this.dump = arguments.get(ARG_DUMP);
            this.database = arguments.get(ARG_DATABASE);
            if (this.dump == null && this.database == null) {
                this.database = this.getActiveDatabase();
            }
            Uploader uploader = this.prepareUploader(this.dump, this.database, tmpDumpFile);
            uploader.process(consoleURL, bearerToken);
        }
        catch (Exception e) {
            if (this.verbose) {
                this.outsideWorld.printStacktrace(e);
            }
            throw e;
        }
    }

    private void verbose(String format, Object ... args) {
        if (this.verbose) {
            this.outsideWorld.outStream().printf(format, args);
        }
    }

    private String buildConsoleURI(String boltURI) throws IncorrectUsage {
        Pattern pattern = Pattern.compile("(?:bolt(?:\\+routing)?|neo4j(?:\\+s|\\+ssc)?)://([^-]+)(-(.+))?.databases.neo4j.io$");
        Matcher matcher = pattern.matcher(boltURI);
        if (!matcher.matches()) {
            throw new IncorrectUsage("Invalid Bolt URI '" + boltURI + "'");
        }
        String databaseId = matcher.group(1);
        String environment = matcher.group(2);
        return String.format("https://console%s.neo4j.io/v1/databases/%s", environment == null ? "" : environment, databaseId);
    }

    private String getActiveDatabase() throws CommandFailed {
        File configFile = this.configDir.resolve("neo4j.conf").toFile();
        Config config = this.getConfig(configFile);
        String name = (String)config.get(GraphDatabaseSettings.active_database);
        if (!"graph.db".equals(name)) {
            this.outsideWorld.stdOutLine("Setting database name to non-default: " + name);
        }
        return name;
    }

    private Uploader prepareUploader(String dump, String database, String to) throws CommandFailed, IncorrectUsage {
        if (dump != null && database != null) {
            throw new IncorrectUsage("Provide either a dump or database name, not both");
        }
        if (dump != null) {
            return this.makeDumpUploader(new File(dump));
        }
        return this.makeFullUploader(database, to);
    }

    public DumpUploader makeDumpUploader(File dump) throws CommandFailed {
        Path path = dump.toPath();
        if (!Files.exists(path, new LinkOption[0])) {
            throw new CommandFailed(String.format("The provided dump '%s' file doesn't exist", path));
        }
        return new DumpUploader(new Source(dump.toPath(), this.dumpSize(dump)));
    }

    public FullUploader makeFullUploader(String database, String to) throws CommandFailed {
        Path dumpPath;
        Path path = dumpPath = to != null ? new File(to).toPath() : this.homeDir.resolve("dump-of-" + database + "-" + System.currentTimeMillis());
        if (Files.exists(dumpPath, new LinkOption[0])) {
            throw new CommandFailed(String.format("The provided dump-to target '%s' file already exists", dumpPath));
        }
        try {
            return new FullUploader(new Source(dumpPath, this.fullSize(database)));
        }
        catch (IOException e) {
            throw new CommandFailed(String.format("Failed to prepare a full database uploader: %s", e.getMessage()), (Throwable)e);
        }
    }

    private Config buildConfig(String databaseName) {
        return Config.fromFile((Path)this.configDir.resolve("neo4j.conf")).withHome(this.homeDir).withConnectorsDisabled().withNoThrowOnFileLoadFailure().withSetting(GraphDatabaseSettings.active_database, databaseName).build();
    }

    private long fullSize(String database) throws IOException, CommandFailed {
        File configFile = this.configDir.resolve("neo4j.conf").toFile();
        Config config = this.getConfig(configFile);
        File databasesRootDirectory = (File)config.get(GraphDatabaseSettings.databases_root_path);
        File dbPath = new File(databasesRootDirectory, database);
        long storeFilesSize = Files.size(dbPath.toPath());
        this.verbose("Determined FullSize=%d bytes in database '%s'\n", storeFilesSize, database);
        return storeFilesSize;
    }

    private long dumpSize(File dump) throws CommandFailed {
        long sizeInBytes = PushToCloudCommand.readSizeFromDumpMetaData(dump);
        this.verbose("Determined DumpSize=%d bytes from dump at %s\n", sizeInBytes, dump);
        return sizeInBytes;
    }

    public static long readSizeFromDumpMetaData(File dump) throws CommandFailed {
        Path path = dump.toPath();
        try {
            DumpMetaData metaData = DumpMetaData.getMetaData(path);
            return Long.parseLong(metaData.byteCount);
        }
        catch (IOException e) {
            throw new CommandFailed("Unable to check size of database dump.", (Throwable)e);
        }
    }

    private Config getConfig(File configFile) throws CommandFailed {
        if (!this.outsideWorld.fileSystem().fileExists(configFile)) {
            throw new CommandFailed("Unable to find config file, tried: " + configFile.getAbsolutePath());
        }
        try {
            return Config.builder().withFile(configFile).withSetting(GraphDatabaseSettings.neo4j_home, this.homeDir.toAbsolutePath().toString()).build();
        }
        catch (Exception e) {
            throw new CommandFailed("Failed to read config file: " + configFile.getAbsolutePath(), (Throwable)e);
        }
    }

    public static String sizeText(long size) {
        return String.format("%.1f GB", PushToCloudCommand.bytesToGibibytes(size));
    }

    public static double bytesToGibibytes(long sizeInBytes) {
        return (double)sizeInBytes / 1.073741824E9;
    }

    public static interface DumpCreator {
        public File dumpDatabase(String var1, Path var2) throws CommandFailed, IncorrectUsage;
    }

    public static interface Copier {
        public String authenticate(boolean var1, String var2, String var3, char[] var4, boolean var5) throws CommandFailed;

        public void copy(boolean var1, String var2, String var3, Source var4, boolean var5, String var6) throws CommandFailed;

        public void checkSize(boolean var1, String var2, long var3, String var5) throws CommandFailed;
    }

    public static class Source {
        private final Path path;
        private long size;

        public Source(Path path, long size) {
            this.path = path;
            this.size = size;
        }

        public Path path() {
            return this.path;
        }

        public long size() {
            return this.size;
        }

        protected void setSize(long newSize) {
            this.size = newSize;
        }

        long crc32Sum() throws IOException {
            CRC32 crc = new CRC32();
            try (BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(this.path.toFile()));){
                int cnt;
                while ((cnt = ((InputStream)inputStream).read()) != -1) {
                    crc.update(cnt);
                }
            }
            return crc.getValue();
        }

        public int hashCode() {
            return this.path.hashCode() + 31 * (int)this.size;
        }

        public boolean equals(Object obj) {
            if (obj instanceof Source) {
                Source other = (Source)obj;
                return this.path.equals(other.path) && this.size == other.size;
            }
            return false;
        }
    }

    class FullUploader
    extends Uploader {
        FullUploader(Source source) {
            super(source);
        }

        @Override
        void process(String consoleURL, String bearerToken) throws CommandFailed, IncorrectUsage {
            PushToCloudCommand.this.verbose("Checking database size %s fits at %s\n", new Object[]{PushToCloudCommand.sizeText(this.size()), consoleURL});
            PushToCloudCommand.this.copier.checkSize(PushToCloudCommand.this.verbose, consoleURL, this.size(), bearerToken);
            File dumpFile = PushToCloudCommand.this.dumpCreator.dumpDatabase(PushToCloudCommand.this.database, this.path());
            long sizeFromDump = PushToCloudCommand.this.dumpSize(dumpFile);
            long sizeFromDatabase = this.size();
            PushToCloudCommand.this.verbose("Validating sizes: fromDump=%d, fromDatabase=%d", new Object[]{sizeFromDump, sizeFromDatabase});
            if ((double)Math.abs(sizeFromDump - sizeFromDatabase) > ACCEPTABLE_DUMP_CHANGE * (double)sizeFromDatabase) {
                PushToCloudCommand.this.outsideWorld.outStream().printf("Warning: unexpectedly large difference between size in dump, and original size: %d != %d", sizeFromDump, sizeFromDatabase);
            }
            this.source.setSize(sizeFromDump);
            PushToCloudCommand.this.verbose("Uploading data of %s to %s\n", new Object[]{PushToCloudCommand.sizeText(this.size()), consoleURL});
            PushToCloudCommand.this.copier.copy(PushToCloudCommand.this.verbose, consoleURL, PushToCloudCommand.this.boltURI, this.source, true, bearerToken);
        }
    }

    class DumpUploader
    extends Uploader {
        DumpUploader(Source source) {
            super(source);
        }

        @Override
        void process(String consoleURL, String bearerToken) throws CommandFailed {
            PushToCloudCommand.this.verbose("Checking database size %s fits at %s\n", new Object[]{PushToCloudCommand.sizeText(this.size()), consoleURL});
            PushToCloudCommand.this.copier.checkSize(PushToCloudCommand.this.verbose, consoleURL, this.size(), bearerToken);
            PushToCloudCommand.this.verbose("Uploading data of %s to %s\n", new Object[]{PushToCloudCommand.sizeText(this.size()), consoleURL});
            PushToCloudCommand.this.copier.copy(PushToCloudCommand.this.verbose, consoleURL, PushToCloudCommand.this.boltURI, this.source, false, bearerToken);
        }
    }

    static abstract class Uploader {
        protected final Source source;

        Uploader(Source source) {
            this.source = source;
        }

        long size() {
            return this.source.size();
        }

        Path path() {
            return this.source.path();
        }

        abstract void process(String var1, String var2) throws CommandFailed, IncorrectUsage;
    }
}

