/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.procedure.builtin.graphschema;

import java.util.Map;
import java.util.stream.Stream;
import org.neo4j.graphdb.Transaction;
import org.neo4j.kernel.api.procedure.SystemProcedure;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Internal;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;

/**
 * UDF for creating a Graph database schema according to the format defined here:
 * <a href="https://github.com/neo4j/graph-schema-json-js-utils">graph-schema-json-js-utils</a>, see scheme
 * <a href="https://github.com/neo4j/graph-schema-json-js-utils/blob/main/json-schema.json">here</a> and an example
 * <a href="https://github.com/neo4j/graph-schema-json-js-utils/blob/main/test/validation/test-schemas/full.json">here</a>.
 * <p>
 * The instrospector creates JSON ids based on the labels and types by default. It can alternatively use
 * Time-Sorted Unique Identifiers (TSID) for the ids inside the generated schema by calling it via
 * {@code CALL experimental.introspect.asJson({useConstantIds: false})}
 */
public class Introspect {

    @Context
    public Transaction transaction;

    /**
     * Shared configuration of the functions and procedures in this class.
     *
     * @param useConstantIds Whether to use constant ids (derived from tokens) or generate ids
     * @param prettyPrint    Whether to pretty print the result or not
     * @param quoteTokens    Whether to always quote tokens or not
     * @param sampleOnly     Whether to sample relationships for determining properties on concrete relationships or not (defaults to {@literal true})
     */
    record Config(boolean useConstantIds, boolean prettyPrint, boolean quoteTokens, boolean sampleOnly) {

        Config(Map<String, Object> params) {
            this(
                    (boolean) params.getOrDefault("useConstantIds", true),
                    (boolean) params.getOrDefault("prettyPrint", false),
                    (boolean) params.getOrDefault("quoteTokens", true),
                    (boolean) params.getOrDefault("sampleOnly", true));
        }
    }

    @Deprecated
    @Internal
    @SystemProcedure
    @Procedure(name = "internal.introspect.asJson", mode = Mode.READ)
    @Description(""
            + "Call with {useConstantIds: false} to generate substitute ids for all tokens and use {prettyPrint: true} for enabling pretty printing;"
            + "{quoteTokens: false} will disable quotation of tokens.")
    public Stream<GraphSchemaJSONResultWrapper> introspectAsJson(@Name("params") Map<String, Object> params)
            throws Exception {

        var config = new Config(params);
        var graphSchema = GraphSchema.build(transaction, config);

        return Stream.of(GraphSchemaJSONResultWrapper.of(graphSchema, config));
    }

    @Deprecated
    @Internal
    @SystemProcedure
    @Procedure(name = "internal.introspect.asGraph", mode = Mode.READ)
    @Description("Visualizes the JSON generated by this introspector in a graphy way")
    public Stream<GraphSchemaGraphyResultWrapper> introspectAndVisualize(@Name("params") Map<String, Object> params)
            throws Exception {

        var graphSchema = GraphSchema.build(transaction, new Config(params));
        var flat = (boolean) params.getOrDefault("flat", false);
        return Stream.of(
                flat
                        ? GraphSchemaGraphyResultWrapper.flat(graphSchema)
                        : GraphSchemaGraphyResultWrapper.full(graphSchema));
    }
}
