/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.ogm.autoindex;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.neo4j.ogm.annotation.CompositeIndex;
import org.neo4j.ogm.autoindex.AutoIndex;
import org.neo4j.ogm.autoindex.IndexType;
import org.neo4j.ogm.autoindex.MissingIndexException;
import org.neo4j.ogm.config.AutoIndexMode;
import org.neo4j.ogm.config.Configuration;
import org.neo4j.ogm.metadata.ClassInfo;
import org.neo4j.ogm.metadata.FieldInfo;
import org.neo4j.ogm.metadata.MetaData;
import org.neo4j.ogm.model.Result;
import org.neo4j.ogm.request.DefaultRequest;
import org.neo4j.ogm.request.Statement;
import org.neo4j.ogm.response.Response;
import org.neo4j.ogm.session.Neo4jSession;
import org.neo4j.ogm.transaction.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AutoIndexManager {
    public static final Pattern COMPOSITE_KEY_MAP_COMPOSITE_PATTERN = Pattern.compile("(.+)\\.(.+)");
    private static final Logger LOGGER = LoggerFactory.getLogger(AutoIndexManager.class);
    private final AutoIndexMode mode;
    private final String dumpDir;
    private final String dumpFilename;
    private final Set<AutoIndex> indexes;
    private final Neo4jSession session;
    private DatabaseInformation databaseInfo;

    public AutoIndexManager(MetaData metaData, Configuration configuration, Neo4jSession session) {
        this.mode = configuration.getAutoIndex();
        this.dumpDir = configuration.getDumpDir();
        this.dumpFilename = configuration.getDumpFilename();
        this.session = session;
        this.indexes = AutoIndexManager.initialiseAutoIndex(metaData);
    }

    public void run() {
        this.databaseInfo = DatabaseInformation.parse((Map)((Iterable)this.session.query("call dbms.components()", Collections.emptyMap()).queryResults()).iterator().next());
        switch (this.mode) {
            case ASSERT: {
                this.assertIndexes();
                break;
            }
            case UPDATE: {
                this.updateIndexes();
                break;
            }
            case VALIDATE: {
                this.validateIndexes();
                break;
            }
            case DUMP: {
                this.dumpIndexes();
                break;
            }
        }
    }

    private void dumpIndexes() {
        ArrayList<String> dumpContent = new ArrayList<String>();
        for (AutoIndex index : this.indexes) {
            dumpContent.add(index.getCreateStatement().getStatement());
        }
        Path dumpPath = Paths.get(this.dumpDir, this.dumpFilename);
        LOGGER.debug("Dumping Indexes to: [{}]", (Object)dumpPath.toAbsolutePath());
        try {
            Files.write(dumpPath, dumpContent, StandardCharsets.UTF_8, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Could not write file to " + dumpPath.toAbsolutePath(), e);
        }
    }

    private void validateIndexes() {
        LOGGER.debug("Validating indexes and constraints");
        ArrayList<AutoIndex> copyOfIndexes = new ArrayList<AutoIndex>(this.indexes);
        List<AutoIndex> dbIndexes = this.loadIndexesFromDB();
        copyOfIndexes.removeAll(dbIndexes);
        if (!copyOfIndexes.isEmpty()) {
            StringBuilder missingIndexes = new StringBuilder("[");
            for (AutoIndex s : copyOfIndexes) {
                missingIndexes.append(s.getDescription()).append(", ");
            }
            missingIndexes.append("]");
            throw new MissingIndexException("Validation of Constraints and Indexes failed. Could not find the following : " + missingIndexes);
        }
    }

    private void assertIndexes() {
        LOGGER.debug("Asserting indexes and constraints");
        ArrayList<Statement> dropStatements = new ArrayList<Statement>();
        List<AutoIndex> dbIndexes = this.loadIndexesFromDB();
        for (AutoIndex dbIndex : dbIndexes) {
            LOGGER.debug("[{}] added to drop statements.", (Object)dbIndex.getDescription());
            dropStatements.add(dbIndex.getDropStatement());
        }
        org.neo4j.ogm.session.request.DefaultRequest dropIndexesRequest = new org.neo4j.ogm.session.request.DefaultRequest();
        dropIndexesRequest.setStatements(dropStatements);
        LOGGER.debug("Dropping all indexes and constraints");
        this.session.doInTransaction(() -> this.session.requestHandler().execute((DefaultRequest)dropIndexesRequest), Transaction.Type.READ_WRITE);
        this.create();
    }

    private List<AutoIndex> loadIndexesFromDB() {
        ArrayList<AutoIndex> dbIndexes = new ArrayList<AutoIndex>();
        this.session.doInTransaction(() -> {
            Result query = this.session.query("CALL db.constraints()", Collections.emptyMap());
            for (Map queryResult : (Iterable)query.queryResults()) {
                Optional<AutoIndex> dbIndex = AutoIndex.parseConstraint(queryResult, this.databaseInfo.version);
                dbIndex.ifPresent(dbIndexes::add);
            }
        }, Transaction.Type.READ_ONLY);
        this.session.doInTransaction(() -> {
            Result query = this.session.query("CALL db.indexes()", Collections.emptyMap());
            for (Map queryResult : (Iterable)query.queryResults()) {
                Optional<AutoIndex> dbIndex = AutoIndex.parseIndex(queryResult, this.databaseInfo.version);
                dbIndex.ifPresent(dbIndexes::add);
            }
        }, Transaction.Type.READ_ONLY);
        return dbIndexes;
    }

    private void updateIndexes() {
        LOGGER.info("Updating indexes and constraints");
        ArrayList<Statement> dropStatements = new ArrayList<Statement>();
        List<AutoIndex> dbIndexes = this.loadIndexesFromDB();
        for (AutoIndex dbIndex : dbIndexes) {
            if (!dbIndex.hasOpposite() || !this.indexes.contains(dbIndex.createOppositeIndex())) continue;
            dropStatements.add(dbIndex.getDropStatement());
        }
        this.executeStatements(dropStatements);
        ArrayList<Statement> createStatements = new ArrayList<Statement>();
        for (AutoIndex index : this.indexes) {
            if (dbIndexes.contains(index)) continue;
            createStatements.add(index.getCreateStatement());
        }
        this.executeStatements(createStatements);
    }

    private void executeStatements(List<Statement> statements) {
        org.neo4j.ogm.session.request.DefaultRequest request = new org.neo4j.ogm.session.request.DefaultRequest();
        request.setStatements(statements);
        this.session.doInTransaction(() -> {
            Response response = this.session.requestHandler().execute((DefaultRequest)request);
            Throwable throwable = null;
            if (response != null) {
                if (throwable != null) {
                    try {
                        response.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                } else {
                    response.close();
                }
            }
        }, Transaction.Type.READ_WRITE);
    }

    private void create() {
        ArrayList<Statement> statements = new ArrayList<Statement>();
        for (AutoIndex index : this.indexes) {
            Statement createStatement = index.getCreateStatement();
            statements.add(createStatement);
            LOGGER.debug("[{}] added to create statements.", (Object)createStatement);
        }
        LOGGER.debug("Creating indexes and constraints.");
        org.neo4j.ogm.session.request.DefaultRequest request = new org.neo4j.ogm.session.request.DefaultRequest();
        request.setStatements(statements);
        this.session.doInTransaction(() -> this.session.requestHandler().execute((DefaultRequest)request).close(), Transaction.Type.READ_WRITE);
    }

    private static Set<AutoIndex> initialiseAutoIndex(MetaData metaData) {
        LOGGER.debug("Building Index Metadata.");
        HashSet<AutoIndex> indexMetadata = new HashSet<AutoIndex>();
        for (ClassInfo classInfo : metaData.persistentEntities()) {
            String owningType = classInfo.neo4jName();
            if (AutoIndexManager.needsToBeIndexed(classInfo)) {
                IndexType type;
                HashSet decomposedFields = new HashSet();
                for (CompositeIndex index : classInfo.getCompositeIndexes()) {
                    type = index.unique() ? IndexType.NODE_KEY_CONSTRAINT : IndexType.COMPOSITE_INDEX;
                    ArrayList properties = new ArrayList();
                    Stream.of(index.value().length > 0 ? index.value() : index.properties()).forEach(p -> {
                        Matcher m = COMPOSITE_KEY_MAP_COMPOSITE_PATTERN.matcher((CharSequence)p);
                        if (m.matches()) {
                            decomposedFields.add(m.group(1));
                            properties.add(m.group(2));
                        } else {
                            properties.add(p);
                        }
                    });
                    AutoIndex autoIndex = new AutoIndex(type, owningType, properties.toArray(new String[properties.size()]));
                    LOGGER.debug("Adding composite index [description={}]", (Object)autoIndex);
                    indexMetadata.add(autoIndex);
                }
                for (FieldInfo fieldInfo : AutoIndexManager.getIndexFields(classInfo)) {
                    IndexType indexType = type = fieldInfo.isConstraint() ? IndexType.UNIQUE_CONSTRAINT : IndexType.SINGLE_INDEX;
                    if (fieldInfo.hasCompositeConverter()) {
                        if (decomposedFields.contains(fieldInfo.getName())) continue;
                        LOGGER.warn("\nThe field {} of {} should be indexed with an index of type {}.\nThis is not supported on a composite field (a field that is decomposed into a set of properties), no index will be created.\nUse a @CompositeIndex on the class instead and prefix the properties with `{}.`.", new Object[]{fieldInfo.getName(), classInfo.getUnderlyingClass(), type, fieldInfo.getName()});
                        continue;
                    }
                    AutoIndex autoIndex = new AutoIndex(type, owningType, new String[]{fieldInfo.property()});
                    LOGGER.debug("Adding Index [description={}]", (Object)autoIndex);
                    indexMetadata.add(autoIndex);
                }
            }
            if (!classInfo.hasRequiredFields()) continue;
            for (FieldInfo requiredField : classInfo.requiredFields()) {
                IndexType type = classInfo.isRelationshipEntity() ? IndexType.REL_PROP_EXISTENCE_CONSTRAINT : IndexType.NODE_PROP_EXISTENCE_CONSTRAINT;
                AutoIndex autoIndex = new AutoIndex(type, owningType, new String[]{requiredField.property()});
                LOGGER.debug("Adding required constraint [description={}]", (Object)autoIndex);
                indexMetadata.add(autoIndex);
            }
        }
        return indexMetadata;
    }

    private static boolean needsToBeIndexed(ClassInfo classInfo) {
        return (!classInfo.isAbstract() || classInfo.neo4jName() != null) && AutoIndexManager.containsIndexesInHierarchy(classInfo);
    }

    private static boolean containsIndexesInHierarchy(ClassInfo classInfo) {
        boolean containsIndexes = false;
        for (ClassInfo currentClassInfo = classInfo; !containsIndexes && currentClassInfo != null; currentClassInfo = currentClassInfo.directSuperclass()) {
            containsIndexes = currentClassInfo.containsIndexes();
        }
        return containsIndexes;
    }

    private static List<FieldInfo> getIndexFields(ClassInfo classInfo) {
        ArrayList<FieldInfo> indexFields = new ArrayList<FieldInfo>();
        for (ClassInfo currentClassInfo = classInfo.directSuperclass(); currentClassInfo != null; currentClassInfo = currentClassInfo.directSuperclass()) {
            if (AutoIndexManager.needsToBeIndexed(currentClassInfo)) continue;
            indexFields.addAll(currentClassInfo.getIndexFields());
        }
        indexFields.addAll(classInfo.getIndexFields());
        return indexFields;
    }

    private static class DatabaseInformation {
        final String version;
        final String edition;

        private DatabaseInformation(String version, String edition) {
            this.version = version;
            this.edition = edition;
        }

        static DatabaseInformation parse(Map<String, Object> databaseInformation) {
            return new DatabaseInformation(DatabaseInformation.extractVersion(databaseInformation), (String)databaseInformation.get("edition"));
        }

        private static String extractVersion(Map<String, Object> databaseInformation) {
            return ((String[])databaseInformation.get("versions"))[0];
        }
    }
}

