/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.ogm.autoindex;

import java.util.Arrays;
import java.util.Collections;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.neo4j.ogm.autoindex.IndexType;
import org.neo4j.ogm.request.Statement;
import org.neo4j.ogm.session.request.RowDataStatement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class AutoIndex {
    private static final Logger logger = LoggerFactory.getLogger(AutoIndex.class);
    private final String[] properties;
    private final String owningType;
    private final IndexType type;
    private final String description;

    AutoIndex(IndexType type, String owningType, String[] properties) {
        this.properties = properties;
        this.owningType = owningType;
        this.type = type;
        this.description = AutoIndex.createDescription(type, owningType, properties);
    }

    private static String createDescription(IndexType type, String owningType, String[] properties) {
        String name = owningType.toLowerCase();
        switch (type) {
            case SINGLE_INDEX: {
                AutoIndex.validatePropertiesLength(properties, IndexType.SINGLE_INDEX);
                return "INDEX ON :`" + owningType + "`(`" + properties[0] + "`)";
            }
            case UNIQUE_CONSTRAINT: {
                AutoIndex.validatePropertiesLength(properties, IndexType.UNIQUE_CONSTRAINT);
                return "CONSTRAINT ON (`" + name + "`:`" + owningType + "`) ASSERT `" + name + "`.`" + properties[0] + "` IS UNIQUE";
            }
            case COMPOSITE_INDEX: {
                return AutoIndex.buildCompositeIndex(name, owningType, properties);
            }
            case NODE_KEY_CONSTRAINT: {
                return AutoIndex.buildNodeKeyConstraint(name, owningType, properties);
            }
            case NODE_PROP_EXISTENCE_CONSTRAINT: {
                AutoIndex.validatePropertiesLength(properties, IndexType.NODE_PROP_EXISTENCE_CONSTRAINT);
                return "CONSTRAINT ON (`" + name + "`:`" + owningType + "`) ASSERT exists(`" + name + "`.`" + properties[0] + "`)";
            }
            case REL_PROP_EXISTENCE_CONSTRAINT: {
                AutoIndex.validatePropertiesLength(properties, IndexType.NODE_PROP_EXISTENCE_CONSTRAINT);
                return "CONSTRAINT ON ()-[`" + name + "`:`" + owningType + "`]-() ASSERT exists(`" + name + "`.`" + properties[0] + "`)";
            }
        }
        throw new UnsupportedOperationException("Index type " + (Object)((Object)type) + " not supported yet");
    }

    private static void validatePropertiesLength(String[] properties, IndexType violatedIndexType) {
        if (properties.length != 1) {
            throw new IllegalArgumentException((Object)((Object)violatedIndexType) + " must have exactly one property, got " + Arrays.toString(properties));
        }
    }

    private static String buildCompositeIndex(String name, String owningType, String[] properties) {
        StringBuilder sb = new StringBuilder();
        sb.append("INDEX ON :`").append(owningType).append("`(");
        AutoIndex.appendProperties(sb, properties);
        sb.append(")");
        return sb.toString();
    }

    private static String buildNodeKeyConstraint(String name, String owningType, String[] properties) {
        StringBuilder sb = new StringBuilder();
        sb.append("CONSTRAINT ON (`").append(name).append("`:`").append(owningType).append("`) ASSERT (");
        AutoIndex.appendPropertiesWithNode(sb, name, properties);
        sb.append(") IS NODE KEY");
        return sb.toString();
    }

    private static void appendProperties(StringBuilder sb, String[] properties) {
        for (int i = 0; i < properties.length; ++i) {
            sb.append('`');
            sb.append(properties[i]);
            sb.append('`');
            if (i >= properties.length - 1) continue;
            sb.append(',');
        }
    }

    private static void appendPropertiesWithNode(StringBuilder sb, String nodeName, String[] properties) {
        for (int i = 0; i < properties.length; ++i) {
            sb.append('`');
            sb.append(nodeName);
            sb.append("`.`");
            sb.append(properties[i]);
            sb.append('`');
            if (i >= properties.length - 1) continue;
            sb.append(',');
        }
    }

    public String[] getProperties() {
        return this.properties;
    }

    public String getOwningType() {
        return this.owningType;
    }

    public IndexType getType() {
        return this.type;
    }

    Statement getCreateStatement() {
        return new RowDataStatement("CREATE " + this.description, Collections.emptyMap());
    }

    public Statement getDropStatement() {
        return new RowDataStatement("DROP " + this.description, Collections.emptyMap());
    }

    String getDescription() {
        return this.description;
    }

    static Optional<AutoIndex> parse(String description) {
        Pattern pattern = Pattern.compile("INDEX ON :(?<label>.*)\\((?<property>.*)\\)");
        Matcher matcher = pattern.matcher(description);
        if (matcher.matches()) {
            String label = matcher.group("label");
            String[] properties = matcher.group("property").split(",");
            for (int i = 0; i < properties.length; ++i) {
                properties[i] = properties[i].trim();
            }
            if (properties.length > 1) {
                return Optional.of(new AutoIndex(IndexType.COMPOSITE_INDEX, label, properties));
            }
            return Optional.of(new AutoIndex(IndexType.SINGLE_INDEX, label, properties));
        }
        pattern = Pattern.compile("CONSTRAINT ON \\((?<name>.*):(?<label>.*)\\) ASSERT ?\\k<name>.(?<property>.*) IS UNIQUE");
        matcher = pattern.matcher(description);
        if (matcher.matches()) {
            String label = matcher.group("label").trim();
            String[] properties = matcher.group("property").split(",");
            return Optional.of(new AutoIndex(IndexType.UNIQUE_CONSTRAINT, label, properties));
        }
        pattern = Pattern.compile("CONSTRAINT ON \\((?<name>.*):(?<label>.*)\\) ASSERT \\((?<properties>.*)\\) IS NODE KEY");
        matcher = pattern.matcher(description);
        if (matcher.matches()) {
            String label = matcher.group("label").trim();
            String[] properties = matcher.group("properties").split(",");
            for (int i = 0; i < properties.length; ++i) {
                properties[i] = properties[i].trim().substring(label.length() + 1);
            }
            return Optional.of(new AutoIndex(IndexType.NODE_KEY_CONSTRAINT, label, properties));
        }
        pattern = Pattern.compile("CONSTRAINT ON \\(\\s?(?<name>.*):(?<label>.*)\\s?\\) ASSERT exists\\(?\\k<name>.(?<property>.*)\\)");
        matcher = pattern.matcher(description);
        if (matcher.matches()) {
            String label = matcher.group("label").trim();
            String[] properties = matcher.group("property").split(",");
            return Optional.of(new AutoIndex(IndexType.NODE_PROP_EXISTENCE_CONSTRAINT, label, properties));
        }
        pattern = Pattern.compile("CONSTRAINT ON \\(\\)-\\[\\s?(?<name>.*):(?<label>.*)\\s?\\]-\\(\\) ASSERT exists\\(?\\k<name>.(?<property>.*)\\)");
        matcher = pattern.matcher(description);
        if (matcher.matches()) {
            String label = matcher.group("label").trim();
            String[] properties = matcher.group("property").split(",");
            for (int i = 0; i < properties.length; ++i) {
                properties[i] = properties[i].trim();
            }
            return Optional.of(new AutoIndex(IndexType.REL_PROP_EXISTENCE_CONSTRAINT, label, properties));
        }
        logger.warn("Could not parse index description {}", (Object)description);
        return Optional.empty();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        AutoIndex autoIndex = (AutoIndex)o;
        if (!Arrays.equals(this.properties, autoIndex.properties)) {
            return false;
        }
        if (!this.owningType.equals(autoIndex.owningType)) {
            return false;
        }
        return this.type == autoIndex.type;
    }

    public int hashCode() {
        int result = Arrays.hashCode(this.properties);
        result = 31 * result + this.owningType.hashCode();
        result = 31 * result + this.type.hashCode();
        return result;
    }

    public String toString() {
        return "AutoIndex{description='" + this.description + '\'' + '}';
    }

    public boolean hasOpposite() {
        switch (this.type) {
            case SINGLE_INDEX: 
            case UNIQUE_CONSTRAINT: 
            case COMPOSITE_INDEX: 
            case NODE_KEY_CONSTRAINT: {
                return true;
            }
        }
        return false;
    }

    public AutoIndex createOppositeIndex() {
        switch (this.type) {
            case SINGLE_INDEX: {
                return new AutoIndex(IndexType.UNIQUE_CONSTRAINT, this.owningType, this.properties);
            }
            case UNIQUE_CONSTRAINT: {
                return new AutoIndex(IndexType.SINGLE_INDEX, this.owningType, this.properties);
            }
            case COMPOSITE_INDEX: {
                return new AutoIndex(IndexType.NODE_KEY_CONSTRAINT, this.owningType, this.properties);
            }
            case NODE_KEY_CONSTRAINT: {
                return new AutoIndex(IndexType.COMPOSITE_INDEX, this.owningType, this.properties);
            }
        }
        throw new IllegalStateException("Can not create opposite index for type=" + (Object)((Object)this.type));
    }
}

