/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.diagnostics.providers;

import java.io.IOException;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import org.neo4j.internal.diagnostics.DiagnosticsLogger;
import org.neo4j.internal.diagnostics.NamedDiagnosticsProvider;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.kernel.internal.NativeIndexFileFilter;
import org.neo4j.storageengine.api.StorageEngineFactory;

public class StoreFilesDiagnostics
extends NamedDiagnosticsProvider {
    private final StorageEngineFactory storageEngineFactory;
    private final FileSystemAbstraction fs;
    private final DatabaseLayout databaseLayout;

    public StoreFilesDiagnostics(StorageEngineFactory storageEngineFactory, FileSystemAbstraction fs, DatabaseLayout databaseLayout) {
        super("Store files");
        this.storageEngineFactory = storageEngineFactory;
        this.fs = fs;
        this.databaseLayout = databaseLayout;
    }

    public void dump(DiagnosticsLogger logger) {
        logger.log(StoreFilesDiagnostics.getDiskSpace(this.databaseLayout));
        logger.log("Storage files stored on file store: " + FileUtils.getFileStoreType((Path)this.databaseLayout.databaseDirectory()));
        logger.log("Storage files: (filename : modification date - size)");
        MappedFileCounter mappedCounter = new MappedFileCounter();
        long totalSize = this.logStoreFiles(logger, "  ", this.databaseLayout.databaseDirectory(), mappedCounter);
        logger.log("Storage summary: ");
        logger.log("  Total size of store: " + ByteUnit.bytesToString((long)totalSize));
        logger.log("  Total size of mapped files: " + ByteUnit.bytesToString((long)mappedCounter.getSize()));
    }

    private long logStoreFiles(DiagnosticsLogger logger, String prefix, Path dir, MappedFileCounter mappedCounter) {
        if (!Files.isDirectory(dir, new LinkOption[0])) {
            return 0L;
        }
        Path[] files = FileUtils.listPaths((Path)dir);
        if (files == null) {
            logger.log(prefix + "<INACCESSIBLE>");
            return 0L;
        }
        long total = 0L;
        List<Path> fileList = Arrays.asList(files);
        fileList.sort(Comparator.comparing(Path::getFileName));
        for (Path file : fileList) {
            long size = 0L;
            String filename = file.getFileName().toString();
            if (Files.isDirectory(file, new LinkOption[0])) {
                logger.log(prefix + filename + ":");
                size = this.logStoreFiles(logger, prefix + "  ", file, mappedCounter);
                filename = "- Total";
            } else {
                try {
                    size = Files.size(file);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                mappedCounter.addFile(file);
            }
            String fileModificationDate = StoreFilesDiagnostics.getFileModificationDate(file);
            String bytes = ByteUnit.bytesToString((long)size);
            String fileInformation = String.format("%s%s: %s - %s", prefix, filename, fileModificationDate, bytes);
            logger.log(fileInformation);
            total += size;
        }
        return total;
    }

    private static String getFileModificationDate(Path file) {
        try {
            ZonedDateTime modifiedDate = Files.getLastModifiedTime(file, new LinkOption[0]).toInstant().atZone(ZoneId.systemDefault()).withNano(0);
            return DateTimeFormatter.ISO_OFFSET_DATE_TIME.format(modifiedDate);
        }
        catch (IOException e) {
            return "<UNKNOWN>";
        }
    }

    private static String getDiskSpace(DatabaseLayout databaseLayout) {
        String header = "Disk space on partition (Total / Free / Free %%):";
        try {
            Path directory = databaseLayout.databaseDirectory();
            FileStore fileStore = Files.getFileStore(directory);
            long free = fileStore.getUnallocatedSpace();
            long total = fileStore.getTotalSpace();
            long percentage = total != 0L ? free * 100L / total : 0L;
            return String.format(header + " %s / %s / %s", total, free, percentage);
        }
        catch (IOException e) {
            return header + " Unable to determine disk space on the partition";
        }
    }

    private class MappedFileCounter {
        private final Set<Path> mappedCandidates = new HashSet<Path>();
        private long size;
        private final Predicate<Path> mappedIndexFilter;

        MappedFileCounter() {
            try {
                this.mappedCandidates.addAll(StoreFilesDiagnostics.this.storageEngineFactory.listStorageFiles(StoreFilesDiagnostics.this.fs, StoreFilesDiagnostics.this.databaseLayout));
            }
            catch (IOException iOException) {
                // empty catch block
            }
            this.mappedIndexFilter = new NativeIndexFileFilter(StoreFilesDiagnostics.this.databaseLayout.databaseDirectory());
        }

        void addFile(Path file) {
            if (this.canBeManagedByPageCache(file) || this.mappedIndexFilter.test(file)) {
                try {
                    this.size += Files.size(file);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }

        public long getSize() {
            return this.size;
        }

        boolean canBeManagedByPageCache(Path storeFile) {
            boolean isLabelScanStore = StoreFilesDiagnostics.this.databaseLayout.labelScanStore().equals(storeFile);
            return isLabelScanStore || this.mappedCandidates.contains(storeFile);
        }
    }
}

