/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.util.function.LongConsumer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.memory.HeapScopedBuffer;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.memory.MemoryTracker;

public class CorruptedLogsTruncator {
    public static final String CORRUPTED_TX_LOGS_BASE_NAME = "corrupted-neostore.transaction.db";
    private static final String LOG_FILE_ARCHIVE_PATTERN = "corrupted-neostore.transaction.db-%d-%d-%d.zip";
    private final File storeDir;
    private final LogFiles logFiles;
    private final FileSystemAbstraction fs;
    private final MemoryTracker memoryTracker;

    public CorruptedLogsTruncator(File storeDir, LogFiles logFiles, FileSystemAbstraction fs, MemoryTracker memoryTracker) {
        this.storeDir = storeDir;
        this.logFiles = logFiles;
        this.fs = fs;
        this.memoryTracker = memoryTracker;
    }

    public void truncate(LogPosition positionAfterLastRecoveredTransaction) throws IOException {
        long recoveredTransactionOffset;
        long recoveredTransactionLogVersion = positionAfterLastRecoveredTransaction.getLogVersion();
        if (this.isRecoveredLogCorrupted(recoveredTransactionLogVersion, recoveredTransactionOffset = positionAfterLastRecoveredTransaction.getByteOffset()) || this.haveMoreRecentLogFiles(recoveredTransactionLogVersion)) {
            this.backupCorruptedContent(recoveredTransactionLogVersion, recoveredTransactionOffset);
            this.truncateLogFiles(recoveredTransactionLogVersion, recoveredTransactionOffset);
        }
    }

    private void truncateLogFiles(long recoveredTransactionLogVersion, long recoveredTransactionOffset) throws IOException {
        File lastRecoveredTransactionLog = this.logFiles.getLogFileForVersion(recoveredTransactionLogVersion);
        this.fs.truncate(lastRecoveredTransactionLog, recoveredTransactionOffset);
        this.forEachSubsequentLogFile(recoveredTransactionLogVersion, fileIndex -> this.fs.deleteFile(this.logFiles.getLogFileForVersion(fileIndex)));
    }

    private void forEachSubsequentLogFile(long recoveredTransactionLogVersion, LongConsumer action) {
        long highestLogVersion = this.logFiles.getHighestLogVersion();
        for (long fileIndex = recoveredTransactionLogVersion + 1L; fileIndex <= highestLogVersion; ++fileIndex) {
            action.accept(fileIndex);
        }
    }

    private void backupCorruptedContent(long recoveredTransactionLogVersion, long recoveredTransactionOffset) throws IOException {
        File corruptedLogArchive = this.getArchiveFile(recoveredTransactionLogVersion, recoveredTransactionOffset);
        try (ZipOutputStream recoveryContent = new ZipOutputStream(new BufferedOutputStream(this.fs.openAsOutputStream(corruptedLogArchive, false)));
             HeapScopedBuffer bufferScope = new HeapScopedBuffer(1, ByteUnit.MebiByte, this.memoryTracker);){
            this.copyTransactionLogContent(recoveredTransactionLogVersion, recoveredTransactionOffset, recoveryContent, bufferScope.getBuffer());
            this.forEachSubsequentLogFile(recoveredTransactionLogVersion, fileIndex -> {
                try {
                    this.copyTransactionLogContent(fileIndex, 0L, recoveryContent, bufferScope.getBuffer());
                }
                catch (IOException io) {
                    throw new UncheckedIOException(io);
                }
            });
        }
    }

    private File getArchiveFile(long recoveredTransactionLogVersion, long recoveredTransactionOffset) throws IOException {
        File corruptedLogsFolder = new File(this.storeDir, CORRUPTED_TX_LOGS_BASE_NAME);
        this.fs.mkdirs(corruptedLogsFolder);
        return new File(corruptedLogsFolder, String.format(LOG_FILE_ARCHIVE_PATTERN, recoveredTransactionLogVersion, recoveredTransactionOffset, System.currentTimeMillis()));
    }

    private void copyTransactionLogContent(long logFileIndex, long logOffset, ZipOutputStream destination, ByteBuffer byteBuffer) throws IOException {
        File logFile = this.logFiles.getLogFileForVersion(logFileIndex);
        if (this.fs.getFileSize(logFile) == logOffset) {
            return;
        }
        ZipEntry zipEntry = new ZipEntry(logFile.getName());
        destination.putNextEntry(zipEntry);
        try (StoreChannel transactionLogChannel = this.fs.read(logFile);){
            transactionLogChannel.position(logOffset);
            while (transactionLogChannel.read(byteBuffer) >= 0) {
                byteBuffer.flip();
                destination.write(byteBuffer.array(), byteBuffer.position(), byteBuffer.remaining());
                byteBuffer.clear();
            }
        }
        destination.closeEntry();
    }

    private boolean haveMoreRecentLogFiles(long recoveredTransactionLogVersion) {
        return this.logFiles.getHighestLogVersion() > recoveredTransactionLogVersion;
    }

    private boolean isRecoveredLogCorrupted(long recoveredTransactionLogVersion, long recoveredTransactionOffset) {
        return this.logFiles.getLogFileForVersion(recoveredTransactionLogVersion).length() > recoveredTransactionOffset;
    }
}

