/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.index.schema;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.configuration.Config;
import org.neo4j.index.internal.gbptree.Layout;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.memory.ByteBufferFactory;
import org.neo4j.kernel.impl.index.schema.GenericKey;
import org.neo4j.kernel.impl.index.schema.GenericLayout;
import org.neo4j.kernel.impl.index.schema.IndexKeyStorage;
import org.neo4j.kernel.impl.index.schema.NativeIndexKey;
import org.neo4j.kernel.impl.index.schema.UnsafeDirectByteBufferAllocator;
import org.neo4j.kernel.impl.index.schema.config.IndexSpecificSpaceFillingCurveSettings;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.LocalMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.RandomRule;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.values.storable.Value;

@TestDirectoryExtension
@ExtendWith(value={RandomExtension.class})
class IndexKeyStorageTest {
    private static final int BLOCK_SIZE = 2000;
    private static final IndexSpecificSpaceFillingCurveSettings spatialSettings = IndexSpecificSpaceFillingCurveSettings.fromConfig((Config)Config.defaults());
    @Inject
    protected TestDirectory directory;
    @Inject
    protected RandomRule random;
    private GenericLayout layout;
    private int numberOfSlots;

    IndexKeyStorageTest() {
    }

    @BeforeEach
    void createLayout() {
        this.numberOfSlots = this.random.nextInt(1, 3);
        this.layout = new GenericLayout(this.numberOfSlots, spatialSettings);
    }

    @Test
    void shouldAddAndReadZeroKey() throws IOException {
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage();){
            keyStorage.doneAdding();
            try (IndexKeyStorage.KeyEntryCursor reader = (IndexKeyStorage.KeyEntryCursor)keyStorage.reader();){
                Assertions.assertFalse((boolean)reader.next(), (String)"Didn't expect reader to have any entries.");
            }
        }
    }

    @Test
    void shouldAddAndReadOneKey() throws IOException {
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage();){
            GenericKey expected = this.randomKey(1);
            keyStorage.add((Object)expected);
            keyStorage.doneAdding();
            try (IndexKeyStorage.KeyEntryCursor reader = (IndexKeyStorage.KeyEntryCursor)keyStorage.reader();){
                Assertions.assertTrue((boolean)reader.next(), (String)"Expected reader to have one entry");
                GenericKey actual = (GenericKey)reader.key();
                Assertions.assertEquals((int)0, (int)this.layout.compare((NativeIndexKey)expected, (NativeIndexKey)actual), (String)"Expected stored key to be equal to original.");
                Assertions.assertFalse((boolean)reader.next(), (String)("Expected reader to have only one entry, second entry was " + reader.key()));
            }
        }
    }

    @Test
    void shouldAddAndReadMultipleKeys() throws IOException {
        ArrayList<GenericKey> keys = new ArrayList<GenericKey>();
        int numberOfKeys = 1000;
        for (int i = 0; i < numberOfKeys; ++i) {
            keys.add(this.randomKey(i));
        }
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage();){
            for (GenericKey key : keys) {
                keyStorage.add((Object)key);
            }
            keyStorage.doneAdding();
            try (IndexKeyStorage.KeyEntryCursor reader = (IndexKeyStorage.KeyEntryCursor)keyStorage.reader();){
                for (GenericKey expected : keys) {
                    Assertions.assertTrue((boolean)reader.next());
                    GenericKey actual = (GenericKey)reader.key();
                    Assertions.assertEquals((int)0, (int)this.layout.compare((NativeIndexKey)expected, (NativeIndexKey)actual), (String)"Expected stored key to be equal to original.");
                }
                Assertions.assertFalse((boolean)reader.next(), (String)("Expected reader to have no more entries, but had at least one additional " + reader.key()));
            }
        }
    }

    @Test
    void shouldNotCreateFileIfNoData() throws IOException {
        FileSystemAbstraction fs = this.directory.getFileSystem();
        File makeSureImDeleted = this.directory.file("makeSureImDeleted", new String[0]);
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage(makeSureImDeleted);){
            Assertions.assertFalse((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to exist now so that we can assert deletion later.");
            keyStorage.doneAdding();
            Assertions.assertFalse((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to exist now so that we can assert deletion later.");
        }
        Assertions.assertFalse((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to be deleted on close.");
    }

    @Test
    void shouldDeleteFileOnCloseWithData() throws IOException {
        FileSystemAbstraction fs = this.directory.getFileSystem();
        File makeSureImDeleted = this.directory.file("makeSureImDeleted", new String[0]);
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage(makeSureImDeleted);){
            keyStorage.add((Object)this.randomKey(1));
            keyStorage.doneAdding();
            Assertions.assertTrue((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to exist now so that we can assert deletion later.");
        }
        Assertions.assertFalse((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to be deleted on close.");
    }

    @Test
    void shouldDeleteFileOnCloseWithDataBeforeDoneAdding() throws IOException {
        FileSystemAbstraction fs = this.directory.getFileSystem();
        File makeSureImDeleted = this.directory.file("makeSureImDeleted", new String[0]);
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage(makeSureImDeleted);){
            keyStorage.add((Object)this.randomKey(1));
            Assertions.assertTrue((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to exist now so that we can assert deletion later.");
        }
        Assertions.assertFalse((boolean)fs.fileExists(makeSureImDeleted), (String)"Expected this file to be deleted on close.");
    }

    @Test
    void mustAllocateResourcesLazilyAndCleanUpOnClose() throws IOException {
        FileSystemAbstraction fs = this.directory.getFileSystem();
        LocalMemoryTracker allocationTracker = new LocalMemoryTracker();
        File file = this.directory.file("file", new String[0]);
        try (UnsafeDirectByteBufferAllocator bufferFactory = new UnsafeDirectByteBufferAllocator();
             IndexKeyStorage<GenericKey> keyStorage = this.keyStorage(file, (ByteBufferFactory.Allocator)bufferFactory, (MemoryTracker)allocationTracker);){
            Assertions.assertEquals((long)0L, (long)allocationTracker.usedNativeMemory(), (String)"Expected to not have any buffers allocated yet");
            Assertions.assertFalse((boolean)fs.fileExists(file), (String)"Expected file to be created lazily");
            keyStorage.add((Object)this.randomKey(1));
            Assertions.assertEquals((long)2000L, (long)allocationTracker.usedNativeMemory(), (String)"Expected to have exactly one buffer allocated by now");
            Assertions.assertTrue((boolean)fs.fileExists(file), (String)"Expected file to be created by now");
        }
        Assertions.assertFalse((boolean)fs.fileExists(file), (String)"Expected file to be deleted on close");
    }

    @Test
    void shouldReportCorrectCount() throws IOException {
        try (IndexKeyStorage<GenericKey> keyStorage = this.keyStorage();){
            Assertions.assertEquals((long)0L, (long)keyStorage.count());
            keyStorage.add((Object)this.randomKey(1));
            Assertions.assertEquals((long)1L, (long)keyStorage.count());
            keyStorage.add((Object)this.randomKey(2));
            Assertions.assertEquals((long)2L, (long)keyStorage.count());
            keyStorage.doneAdding();
            Assertions.assertEquals((long)2L, (long)keyStorage.count());
        }
    }

    private GenericKey randomKey(int entityId) {
        GenericKey key = this.layout.newKey();
        key.initialize((long)entityId);
        for (int i = 0; i < this.numberOfSlots; ++i) {
            Value value = this.random.randomValues().nextValue();
            key.initFromValue(i, value, NativeIndexKey.Inclusion.NEUTRAL);
        }
        return key;
    }

    private IndexKeyStorage<GenericKey> keyStorage() {
        return this.keyStorage(this.directory.file("file", new String[0]));
    }

    private IndexKeyStorage<GenericKey> keyStorage(File file) {
        return this.keyStorage(file, ByteBufferFactory.heapBufferFactory((int)0).newLocalAllocator(), (MemoryTracker)EmptyMemoryTracker.INSTANCE);
    }

    private IndexKeyStorage<GenericKey> keyStorage(File file, ByteBufferFactory.Allocator bufferFactory, MemoryTracker memoryTracker) {
        return new IndexKeyStorage(this.directory.getFileSystem(), file, bufferFactory, 2000, (Layout)this.layout, memoryTracker);
    }
}

