/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.availability;

import java.time.Clock;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.stream.Collectors;
import org.neo4j.graphdb.DatabaseShutdownException;
import org.neo4j.internal.helpers.Format;
import org.neo4j.internal.helpers.Listeners;
import org.neo4j.kernel.availability.AvailabilityGuard;
import org.neo4j.kernel.availability.AvailabilityListener;
import org.neo4j.kernel.availability.AvailabilityRequirement;
import org.neo4j.kernel.availability.CompositeDatabaseAvailabilityGuard;
import org.neo4j.kernel.availability.UnavailableException;
import org.neo4j.kernel.database.NamedDatabaseId;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.logging.Log;

public class DatabaseAvailabilityGuard
extends LifecycleAdapter
implements AvailabilityGuard {
    private static final String DATABASE_AVAILABLE_MSG = "Fulfilling of requirement '%s' makes database %s available.";
    private static final String DATABASE_UNAVAILABLE_MSG = "Requirement `%s` makes database %s unavailable.";
    private final Set<AvailabilityRequirement> blockingRequirements = new CopyOnWriteArraySet<AvailabilityRequirement>();
    private volatile boolean shutdown = true;
    private volatile Throwable startupFailure;
    private final Listeners<AvailabilityListener> listeners = new Listeners();
    private final NamedDatabaseId namedDatabaseId;
    private final Clock clock;
    private final Log log;
    private final long databaseTimeMillis;
    private final CompositeDatabaseAvailabilityGuard globalGuard;

    public DatabaseAvailabilityGuard(NamedDatabaseId namedDatabaseId, Clock clock, Log log, long databaseTimeMillis, CompositeDatabaseAvailabilityGuard globalGuard) {
        this.namedDatabaseId = namedDatabaseId;
        this.clock = clock;
        this.log = log;
        this.databaseTimeMillis = databaseTimeMillis;
        this.globalGuard = globalGuard;
        this.listeners.add((Object)new LoggingAvailabilityListener(log, namedDatabaseId));
    }

    public void init() throws Exception {
        this.shutdown = false;
        this.startupFailure = null;
    }

    public void start() throws Exception {
        this.globalGuard.addDatabaseAvailabilityGuard(this);
    }

    public void stop() throws Exception {
        this.globalGuard.removeDatabaseAvailabilityGuard(this);
    }

    @Override
    public void require(AvailabilityRequirement requirement) {
        if (this.shutdown) {
            return;
        }
        if (!this.blockingRequirements.add(requirement)) {
            return;
        }
        if (this.blockingRequirements.size() == 1) {
            this.log.info(DATABASE_UNAVAILABLE_MSG, new Object[]{requirement.description(), this.namedDatabaseId.name()});
            this.listeners.notify(AvailabilityListener::unavailable);
        }
    }

    @Override
    public void fulfill(AvailabilityRequirement requirement) {
        if (this.shutdown) {
            return;
        }
        if (!this.blockingRequirements.remove(requirement)) {
            return;
        }
        if (this.blockingRequirements.isEmpty()) {
            this.log.info(DATABASE_AVAILABLE_MSG, new Object[]{requirement.description(), this.namedDatabaseId.name()});
            this.listeners.notify(AvailabilityListener::available);
        }
    }

    public void startupFailure(Throwable cause) {
        this.startupFailure = cause;
    }

    public void shutdown() {
        this.shutdown = true;
        this.blockingRequirements.clear();
    }

    @Override
    public boolean isAvailable() {
        return this.availability() == Availability.AVAILABLE;
    }

    @Override
    public boolean isShutdown() {
        return this.availability() == Availability.SHUTDOWN;
    }

    @Override
    public boolean isAvailable(long millis) {
        return this.availability(millis) == Availability.AVAILABLE;
    }

    public void assertDatabaseAvailable() throws UnavailableException {
        Availability availability = this.availability(this.databaseTimeMillis);
        switch (availability) {
            case AVAILABLE: {
                return;
            }
            case SHUTDOWN: {
                if (this.startupFailure != null) {
                    throw new DatabaseShutdownException(this.startupFailure);
                }
                throw new DatabaseShutdownException();
            }
            case UNAVAILABLE: {
                this.throwUnavailableException(this.databaseTimeMillis, availability);
            }
        }
        throw new IllegalStateException("Unsupported availability mode: " + availability);
    }

    @Override
    public void await(long millis) throws UnavailableException {
        Availability availability = this.availability(millis);
        if (availability == Availability.AVAILABLE) {
            return;
        }
        this.throwUnavailableException(millis, availability);
    }

    private void throwUnavailableException(long millis, Availability availability) throws UnavailableException {
        String description = availability == Availability.UNAVAILABLE ? "Timeout waiting for database to become available and allow new transactions. Waited " + Format.duration((long)millis) + ". " + this.describe() : "Database not available because it's shutting down";
        throw new UnavailableException(description);
    }

    private Availability availability() {
        if (this.shutdown) {
            return Availability.SHUTDOWN;
        }
        return this.blockingRequirements.isEmpty() ? Availability.AVAILABLE : Availability.UNAVAILABLE;
    }

    private Availability availability(long millis) {
        Availability availability = this.availability();
        if (availability == Availability.AVAILABLE) {
            return availability;
        }
        long timeout = this.clock.millis() + millis;
        do {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                break;
            }
        } while ((availability = this.availability()) != Availability.AVAILABLE && this.clock.millis() < timeout);
        return availability;
    }

    @Override
    public void addListener(AvailabilityListener listener) {
        this.listeners.add((Object)listener);
    }

    @Override
    public void removeListener(AvailabilityListener listener) {
        this.listeners.remove((Object)listener);
    }

    @Override
    public String describe() {
        Set<AvailabilityRequirement> requirementSet = this.blockingRequirements;
        int requirements = requirementSet.size();
        if (requirements > 0) {
            String causes = requirementSet.stream().map(AvailabilityRequirement::description).collect(Collectors.joining(", "));
            return requirements + " reasons for blocking: " + causes + ".";
        }
        return "No blocking components";
    }

    private static class LoggingAvailabilityListener
    implements AvailabilityListener {
        private final Log log;
        private final NamedDatabaseId namedDatabaseId;

        LoggingAvailabilityListener(Log log, NamedDatabaseId namedDatabaseId) {
            this.log = log;
            this.namedDatabaseId = namedDatabaseId;
        }

        @Override
        public void available() {
            this.log.info("Database %s is ready.", new Object[]{this.namedDatabaseId.name()});
        }

        @Override
        public void unavailable() {
            this.log.info("Database %s is unavailable.", new Object[]{this.namedDatabaseId.name()});
        }
    }

    private static enum Availability {
        AVAILABLE,
        UNAVAILABLE,
        SHUTDOWN;

    }
}

