/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.storemigration;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.store.format.RecordFormatSelector;
import org.neo4j.kernel.impl.storemigration.MigrationStatus;
import org.neo4j.kernel.impl.storemigration.StoreMigrationParticipant;
import org.neo4j.kernel.impl.storemigration.UpgradableDatabase;
import org.neo4j.kernel.impl.storemigration.UpgradeNotAllowedByConfigurationException;
import org.neo4j.kernel.impl.storemigration.monitoring.MigrationProgressMonitor;
import org.neo4j.kernel.impl.util.monitoring.ProgressReporter;
import org.neo4j.kernel.internal.Version;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;

public class StoreUpgrader {
    public static final String MIGRATION_DIRECTORY = "upgrade";
    public static final String MIGRATION_LEFT_OVERS_DIRECTORY = "upgrade_backup";
    private static final String MIGRATION_STATUS_FILE = "_status";
    private final UpgradableDatabase upgradableDatabase;
    private final MigrationProgressMonitor progressMonitor;
    private final List<StoreMigrationParticipant> participants = new ArrayList<StoreMigrationParticipant>();
    private final Config config;
    private final FileSystemAbstraction fileSystem;
    private final PageCache pageCache;
    private final Log log;
    private final LogProvider logProvider;

    public StoreUpgrader(UpgradableDatabase upgradableDatabase, MigrationProgressMonitor progressMonitor, Config config, FileSystemAbstraction fileSystem, PageCache pageCache, LogProvider logProvider) {
        this.upgradableDatabase = upgradableDatabase;
        this.progressMonitor = progressMonitor;
        this.fileSystem = fileSystem;
        this.config = config;
        this.pageCache = pageCache;
        this.logProvider = logProvider;
        this.log = logProvider.getLog(this.getClass());
    }

    public void addParticipant(StoreMigrationParticipant participant) {
        assert (participant != null);
        if (!StoreMigrationParticipant.NOT_PARTICIPATING.equals(participant)) {
            this.participants.add(participant);
        }
    }

    public void migrateIfNeeded(File storeDirectory) {
        File migrationDirectory = new File(storeDirectory, MIGRATION_DIRECTORY);
        this.cleanupLegacyLeftOverDirsIn(storeDirectory);
        File migrationStateFile = new File(migrationDirectory, MIGRATION_STATUS_FILE);
        if (this.upgradableDatabase.hasCurrentVersion(storeDirectory) && !this.fileSystem.fileExists(migrationStateFile)) {
            return;
        }
        if (this.isUpgradeAllowed()) {
            this.migrateStore(storeDirectory, migrationDirectory, migrationStateFile);
        } else if (!RecordFormatSelector.isStoreAndConfigFormatsCompatible(this.config, storeDirectory, this.fileSystem, this.pageCache, this.logProvider)) {
            throw new UpgradeNotAllowedByConfigurationException();
        }
    }

    private void migrateStore(File storeDirectory, File migrationDirectory, File migrationStateFile) {
        this.progressMonitor.started(this.participants.size());
        MigrationStatus migrationStatus = MigrationStatus.readMigrationStatus(this.fileSystem, migrationStateFile);
        String versionToMigrateFrom = null;
        if (MigrationStatus.migrating.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = this.upgradableDatabase.checkUpgradeable(storeDirectory).storeVersion();
            this.cleanMigrationDirectory(migrationDirectory);
            MigrationStatus.migrating.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.migrateToIsolatedDirectory(storeDirectory, migrationDirectory, versionToMigrateFrom);
            MigrationStatus.moving.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.moving.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.moving.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.moveMigratedFilesToStoreDirectory(this.participants, migrationDirectory, storeDirectory, versionToMigrateFrom, this.upgradableDatabase.currentVersion());
        }
        this.cleanup(this.participants, migrationDirectory);
        this.progressMonitor.completed();
    }

    List<StoreMigrationParticipant> getParticipants() {
        return this.participants;
    }

    private boolean isUpgradeAllowed() {
        return this.config.get(GraphDatabaseSettings.allow_upgrade);
    }

    private void cleanupLegacyLeftOverDirsIn(File storeDir) {
        Pattern leftOverDirsPattern = Pattern.compile("upgrade_backup(_\\d*)?");
        File[] leftOverDirs = storeDir.listFiles((file, name) -> file.isDirectory() && leftOverDirsPattern.matcher(name).matches());
        if (leftOverDirs != null) {
            for (File leftOverDir : leftOverDirs) {
                this.deleteSilently(leftOverDir);
            }
        }
    }

    private void cleanup(Iterable<StoreMigrationParticipant> participants, File migrationDirectory) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.cleanup(migrationDirectory);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure cleaning up after migration", e);
        }
    }

    private void moveMigratedFilesToStoreDirectory(Iterable<StoreMigrationParticipant> participants, File migrationDirectory, File storeDirectory, String versionToMigrateFrom, String versionToMigrateTo) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.moveMigratedFiles(migrationDirectory, storeDirectory, versionToMigrateFrom, versionToMigrateTo);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void migrateToIsolatedDirectory(File storeDir, File migrationDirectory, String versionToMigrateFrom) {
        try {
            for (StoreMigrationParticipant participant : this.participants) {
                ProgressReporter progressReporter = this.progressMonitor.startSection(participant.getName());
                participant.migrate(storeDir, migrationDirectory, progressReporter, versionToMigrateFrom, this.upgradableDatabase.currentVersion());
                progressReporter.completed();
            }
        }
        catch (IOException | UncheckedIOException e) {
            throw new UnableToUpgradeException("Failure doing migration", e);
        }
    }

    private void cleanMigrationDirectory(File migrationDirectory) {
        try {
            if (this.fileSystem.fileExists(migrationDirectory)) {
                this.fileSystem.deleteRecursively(migrationDirectory);
            }
        }
        catch (IOException | UncheckedIOException e) {
            throw new UnableToUpgradeException("Failure deleting upgrade directory " + migrationDirectory, e);
        }
        this.fileSystem.mkdir(migrationDirectory);
    }

    private void deleteSilently(File dir) {
        try {
            this.fileSystem.deleteRecursively(dir);
        }
        catch (IOException e) {
            this.log.error("Unable to delete directory: " + dir, (Throwable)e);
        }
    }

    static class DatabaseNotCleanlyShutDownException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "The database is not cleanly shutdown. The database needs recovery, in order to recover the database, please run the old version of the database on this store.";

        DatabaseNotCleanlyShutDownException() {
            super(MESSAGE);
        }
    }

    public static class UnexpectedUpgradingStoreFormatException
    extends UnableToUpgradeException {
        protected static final String MESSAGE = "This is an enterprise-only store. Please configure '%s' to open.";

        UnexpectedUpgradingStoreFormatException() {
            super(String.format(MESSAGE, GraphDatabaseSettings.record_format.name()));
        }
    }

    public static class AttemptedDowngradeException
    extends UnableToUpgradeException {
        static final String MESSAGE = "Downgrading stores are not supported.";

        AttemptedDowngradeException() {
            super(MESSAGE);
        }
    }

    public static class UnexpectedUpgradingStoreVersionException
    extends UnableToUpgradeException {
        static final String MESSAGE = "Not possible to upgrade a store with version '%s' to current store version `%s` (Neo4j %s).";

        UnexpectedUpgradingStoreVersionException(String fileVersion, String currentVersion) {
            super(String.format(MESSAGE, fileVersion, currentVersion, Version.getNeo4jVersion()));
        }
    }

    static class UpgradingStoreVersionNotFoundException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "'%s' does not contain a store version, please ensure that the original database was shut down in a clean state.";

        UpgradingStoreVersionNotFoundException(String filenameWithoutStoreVersion) {
            super(String.format(MESSAGE, filenameWithoutStoreVersion));
        }
    }

    static class UpgradeMissingStoreFilesException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "Missing required store file '%s'.";

        UpgradeMissingStoreFilesException(String filenameExpectedToExist) {
            super(String.format(MESSAGE, filenameExpectedToExist));
        }
    }

    public static class UnableToUpgradeException
    extends RuntimeException {
        public UnableToUpgradeException(String message, Throwable cause) {
            super(message, cause);
        }

        UnableToUpgradeException(String message) {
            super(message);
        }
    }
}

