/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.store;

import java.io.File;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.Exceptions;
import org.neo4j.helpers.collection.Visitor;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PageCursor;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.kernel.IdGeneratorFactory;
import org.neo4j.kernel.IdType;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.store.InvalidIdGeneratorException;
import org.neo4j.kernel.impl.store.InvalidRecordException;
import org.neo4j.kernel.impl.store.StoreNotFoundException;
import org.neo4j.kernel.impl.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.store.id.IdGenerator;
import org.neo4j.kernel.impl.store.id.IdSequence;
import org.neo4j.kernel.impl.store.record.Record;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.Logger;

public abstract class CommonAbstractStore
implements IdSequence,
AutoCloseable {
    public static final String ALL_STORES_VERSION = "v0.A.6";
    public static final String UNKNOWN_VERSION = "Unknown";
    protected final Config configuration;
    protected final PageCache pageCache;
    protected final File storageFileName;
    protected final IdType idType;
    protected final IdGeneratorFactory idGeneratorFactory;
    protected final Log log;
    protected PagedFile storeFile;
    private IdGenerator idGenerator;
    private boolean storeOk = true;
    private Throwable causeOfStoreNotOk;

    public CommonAbstractStore(File fileName, Config configuration, IdType idType, IdGeneratorFactory idGeneratorFactory, PageCache pageCache, LogProvider logProvider) {
        this.storageFileName = fileName;
        this.configuration = configuration;
        this.idGeneratorFactory = idGeneratorFactory;
        this.pageCache = pageCache;
        this.idType = idType;
        this.log = logProvider.getLog(this.getClass());
    }

    void initialise(boolean createIfNotExists) {
        try {
            this.checkStorage(createIfNotExists);
            this.loadStorage();
        }
        catch (Exception e) {
            if (this.storeFile != null) {
                try {
                    this.storeFile.close();
                }
                catch (IOException failureToClose) {
                    e.addSuppressed(failureToClose);
                }
            }
            throw Exceptions.launderedException(e);
        }
    }

    protected static long longFromIntAndMod(long base, long modifier) {
        return modifier == 0L && base == 0xFFFFFFFFL ? -1L : base | modifier;
    }

    protected abstract String getTypeDescriptor();

    protected abstract void initialiseNewStoreFile(PagedFile var1) throws IOException;

    protected void checkStorage(boolean createIfNotExists) {
        try {
            PagedFile ignore = this.pageCache.map(this.storageFileName, this.pageCache.pageSize(), new OpenOption[0]);
            Throwable throwable = null;
            if (ignore != null) {
                if (throwable != null) {
                    try {
                        ignore.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                } else {
                    ignore.close();
                }
            }
        }
        catch (NoSuchFileException e) {
            if (createIfNotExists) {
                try (PagedFile file = this.pageCache.map(this.storageFileName, this.pageCache.pageSize(), new OpenOption[]{StandardOpenOption.CREATE});){
                    this.initialiseNewStoreFile(file);
                    return;
                }
                catch (IOException e1) {
                    e.addSuppressed(e1);
                }
            }
            throw new StoreNotFoundException("Store file not found: " + this.storageFileName, e);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to open store file: " + this.storageFileName, e);
        }
    }

    protected void loadStorage() {
        try {
            this.readAndVerifyBlockSize();
            try {
                int filePageSize = this.pageCache.pageSize() - this.pageCache.pageSize() % this.getRecordSize();
                this.storeFile = this.pageCache.map(this.getStorageFileName(), filePageSize, new OpenOption[0]);
            }
            catch (IOException e) {
                throw new UnderlyingStorageException(e);
            }
            this.loadIdGenerator();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to load storage " + this.getStorageFileName(), e);
        }
    }

    protected long pageIdForRecord(long id) {
        return id * (long)this.getRecordSize() / (long)this.storeFile.pageSize();
    }

    protected int offsetForId(long id) {
        return (int)(id * (long)this.getRecordSize() % (long)this.storeFile.pageSize());
    }

    public int getRecordsPerPage() {
        return this.storeFile.pageSize() / this.getRecordSize();
    }

    protected abstract void readAndVerifyBlockSize() throws IOException;

    private void loadIdGenerator() {
        try {
            if (this.storeOk) {
                this.openIdGenerator();
            }
        }
        catch (InvalidIdGeneratorException e) {
            this.setStoreNotOk(e);
        }
        finally {
            if (!this.getStoreOk()) {
                this.log.debug(this.getStorageFileName() + " non clean shutdown detected");
            }
        }
    }

    protected int getHeaderRecord() throws IOException {
        int headerRecord = 0;
        try (PagedFile pagedFile = this.pageCache.map(this.getStorageFileName(), this.pageCache.pageSize(), new OpenOption[0]);
             PageCursor pageCursor = pagedFile.io(0L, 1);){
            if (pageCursor.next()) {
                do {
                    headerRecord = pageCursor.getInt();
                } while (pageCursor.shouldRetry());
            }
        }
        if (headerRecord <= 0) {
            throw new InvalidRecordException("Illegal block size: " + headerRecord + " in " + this.getStorageFileName());
        }
        return headerRecord;
    }

    protected abstract boolean isInUse(byte var1);

    final void rebuildIdGenerator() {
        boolean fastRebuild;
        long defraggedCount;
        block16: {
            int blockSize = this.getRecordSize();
            if (blockSize <= 0) {
                throw new InvalidRecordException("Illegal blockSize: " + blockSize);
            }
            this.log.info("Rebuilding id generator for[" + this.getStorageFileName() + "] ...");
            this.closeIdGenerator();
            this.createIdGenerator(this.getIdFileName());
            this.openIdGenerator();
            defraggedCount = 0L;
            fastRebuild = this.doFastIdGeneratorRebuild();
            try {
                long foundHighId = this.scanForHighId();
                this.setHighId(foundHighId);
                if (fastRebuild) break block16;
                try (PageCursor cursor = this.storeFile.io(0L, 10);){
                    defraggedCount = this.rebuildIdGeneratorSlow(cursor, this.getRecordsPerPage(), blockSize, foundHighId);
                }
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Unable to rebuild id generator " + this.getStorageFileName(), e);
            }
        }
        this.log.info("[" + this.getStorageFileName() + "] high id=" + this.getHighId() + " (defragged=" + defraggedCount + ")");
        this.log.info(this.getStorageFileName() + " rebuild id generator, highId=" + this.getHighId() + " defragged count=" + defraggedCount);
        if (!fastRebuild) {
            this.closeIdGenerator();
            this.openIdGenerator();
        }
    }

    private long rebuildIdGeneratorSlow(PageCursor cursor, int recordsPerPage, int blockSize, long foundHighId) throws IOException {
        long defragCount = 0L;
        long[] freedBatch = new long[recordsPerPage];
        int startingId = this.getNumberOfReservedLowIds();
        boolean done = false;
        while (!done && cursor.next()) {
            int i;
            int defragged;
            long idPageOffset = cursor.getCurrentPageId() * (long)recordsPerPage;
            block1: do {
                defragged = 0;
                done = false;
                for (i = startingId; i < recordsPerPage; ++i) {
                    int offset = i * blockSize;
                    cursor.setOffset(offset);
                    long recordId = idPageOffset + (long)i;
                    if (recordId >= foundHighId) {
                        done = true;
                        continue block1;
                    }
                    if (!this.isRecordInUse(cursor)) {
                        freedBatch[defragged++] = recordId;
                        continue;
                    }
                    if (!this.isRecordReserved(cursor)) continue;
                    cursor.setOffset(offset);
                    cursor.putByte(Record.NOT_IN_USE.byteValue());
                    cursor.putInt(0);
                    freedBatch[defragged++] = recordId;
                }
            } while (cursor.shouldRetry());
            for (i = 0; i < defragged; ++i) {
                this.freeId(freedBatch[i]);
            }
            defragCount += (long)defragged;
            startingId = 0;
        }
        return defragCount;
    }

    protected boolean doFastIdGeneratorRebuild() {
        return this.configuration.get(Configuration.rebuild_idgenerators_fast);
    }

    protected void setStoreNotOk(Throwable cause) {
        this.storeOk = false;
        this.causeOfStoreNotOk = cause;
        this.idGenerator = null;
    }

    protected boolean getStoreOk() {
        return this.storeOk;
    }

    protected final void checkStoreOk() {
        if (!this.storeOk) {
            throw new UnderlyingStorageException("Store is not OK", this.causeOfStoreNotOk);
        }
    }

    @Override
    public long nextId() {
        return this.idGenerator.nextId();
    }

    public void freeId(long id) {
        if (this.idGenerator != null) {
            this.idGenerator.freeId(id);
        }
    }

    public long getHighId() {
        return this.idGenerator != null ? this.idGenerator.getHighId() : this.scanForHighId();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setHighId(long highId) {
        if (this.idGenerator != null) {
            IdGenerator idGenerator = this.idGenerator;
            synchronized (idGenerator) {
                if (highId > this.idGenerator.getHighId()) {
                    this.idGenerator.setHighId(highId);
                }
            }
        }
    }

    public final void makeStoreOk() {
        if (!this.storeOk) {
            this.rebuildIdGenerator();
            this.storeOk = true;
            this.causeOfStoreNotOk = null;
        }
    }

    public File getStorageFileName() {
        return this.storageFileName;
    }

    private File getIdFileName() {
        return new File(this.getStorageFileName().getPath() + ".id");
    }

    protected void openIdGenerator() {
        this.idGenerator = this.openIdGenerator(this.getIdFileName(), this.idType.getGrabSize());
    }

    protected IdGenerator openIdGenerator(File fileName, int grabSize) {
        return this.idGeneratorFactory.open(fileName, grabSize, this.getIdType(), this.scanForHighId());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected long scanForHighId() {
        try (PageCursor cursor = this.storeFile.io(0L, 1);){
            long nextPageId = this.storeFile.getLastPageId();
            int recordsPerPage = this.getRecordsPerPage();
            int recordSize = this.getRecordSize();
            while (true) {
                if (nextPageId >= 0L && cursor.next(nextPageId)) {
                    --nextPageId;
                } else {
                    long l = this.getNumberOfReservedLowIds();
                    return l;
                }
                do {
                    int currentRecord = recordsPerPage;
                    while (currentRecord-- > 0) {
                        cursor.setOffset(currentRecord * recordSize);
                        long recordId = cursor.getCurrentPageId() * (long)recordsPerPage + (long)currentRecord;
                        if (!this.isRecordInUse(cursor)) continue;
                        long l = recordId + 1L;
                        return l;
                    }
                } while (cursor.shouldRetry());
            }
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to find high id by scanning backwards " + this.getStorageFileName(), e);
        }
    }

    public abstract int getRecordSize();

    protected boolean isRecordInUse(PageCursor cursor) {
        return this.isInUse(cursor.getByte());
    }

    protected boolean isRecordReserved(PageCursor cursor) {
        return false;
    }

    protected void createIdGenerator(File fileName) {
        this.idGeneratorFactory.create(fileName, 0L, false);
    }

    protected void closeIdGenerator() {
        if (this.idGenerator != null) {
            this.idGenerator.close();
        }
    }

    public void flush() {
        try {
            this.storeFile.flushAndForce();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Failed to flush", e);
        }
    }

    @Override
    public void close() {
        if (this.idGenerator == null || !this.storeOk) {
            try {
                this.storeFile.close();
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Failed to close store file: " + this.getStorageFileName(), e);
            }
            return;
        }
        try {
            this.storeFile.close();
            this.idGenerator.close();
        }
        catch (IOException | IllegalStateException e) {
            throw new UnderlyingStorageException("Failed to close store file: " + this.getStorageFileName(), e);
        }
    }

    public long getHighestPossibleIdInUse() {
        if (this.idGenerator != null) {
            return this.idGenerator.getHighestPossibleIdInUse();
        }
        return this.scanForHighId() - 1L;
    }

    public void setHighestPossibleIdInUse(long highId) {
        this.setHighId(highId + 1L);
    }

    public long getNumberOfIdsInUse() {
        return this.idGenerator.getNumberOfIdsInUse();
    }

    public int getNumberOfReservedLowIds() {
        return 0;
    }

    public IdType getIdType() {
        return this.idType;
    }

    public final void logVersions(Logger logger) {
        logger.log(this.getTypeDescriptor() + " " + ALL_STORES_VERSION);
    }

    public final void logIdUsage(Logger logger) {
        logger.log(String.format("  %s: used=%s high=%s", this.getTypeDescriptor() + " " + ALL_STORES_VERSION, this.getNumberOfIdsInUse(), this.getHighestPossibleIdInUse()));
    }

    public final void visitStore(Visitor<CommonAbstractStore, RuntimeException> visitor) {
        visitor.visit(this);
    }

    final void deleteIdGenerator() {
        if (this.idGenerator != null) {
            this.idGenerator.delete();
            this.idGenerator = null;
        }
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    public static abstract class Configuration {
        public static final Setting<Boolean> rebuild_idgenerators_fast = GraphDatabaseSettings.rebuild_idgenerators_fast;
    }
}

