/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.store.id;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicLong;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.kernel.impl.store.InvalidIdGeneratorException;
import org.neo4j.kernel.impl.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.store.id.IdGenerator;
import org.neo4j.kernel.impl.store.id.IdRange;

public class IdGeneratorImpl
implements IdGenerator {
    private static final int HEADER_SIZE = 9;
    private static final byte CLEAN_GENERATOR = 0;
    private static final byte STICKY_GENERATOR = 1;
    public static final long INTEGER_MINUS_ONE = 0xFFFFFFFFL;
    private int grabSize = -1;
    private final AtomicLong highId = new AtomicLong(-1L);
    private long readPosition;
    private long maxReadPosition = 9L;
    private long defraggedIdCount = -1L;
    private final File file;
    private final FileSystemAbstraction fs;
    private StoreChannel fileChannel = null;
    private final LinkedList<Long> idsReadFromFile = new LinkedList();
    private final LinkedList<Long> releasedIdList = new LinkedList();
    private final long max;
    private final boolean aggressiveReuse;

    public IdGeneratorImpl(FileSystemAbstraction fs, File file, int grabSize, long max, boolean aggressiveReuse, long highId) {
        this.fs = fs;
        this.aggressiveReuse = aggressiveReuse;
        if (grabSize < 1) {
            throw new IllegalArgumentException("Illegal grabSize: " + grabSize);
        }
        this.max = max;
        this.file = file;
        this.grabSize = grabSize;
        this.initGenerator();
        this.highId.set(Math.max(this.highId.get(), highId));
    }

    @Override
    public synchronized long nextId() {
        this.assertStillOpen();
        long nextDefragId = this.nextIdFromDefragList();
        if (nextDefragId != -1L) {
            return nextDefragId;
        }
        long id = this.highId.get();
        if (id == 0xFFFFFFFFL) {
            id = this.highId.incrementAndGet();
        }
        this.assertIdWithinCapacity(id);
        this.highId.incrementAndGet();
        return id;
    }

    private void assertIdWithinCapacity(long id) {
        if (id > this.max || id < 0L) {
            throw new UnderlyingStorageException("Id capacity exceeded: " + id + " is not within bounds [0; " + this.max + "] for " + this.file);
        }
    }

    private boolean canReadMoreIdBatches() {
        return this.readPosition < this.maxReadPosition;
    }

    private long nextIdFromDefragList() {
        Long id;
        if (this.aggressiveReuse && (id = this.releasedIdList.poll()) != null) {
            --this.defraggedIdCount;
            return id;
        }
        if (!this.idsReadFromFile.isEmpty() || this.canReadMoreIdBatches()) {
            if (this.idsReadFromFile.isEmpty()) {
                this.readIdBatch();
            }
            long id2 = this.idsReadFromFile.removeFirst();
            --this.defraggedIdCount;
            return id2;
        }
        return -1L;
    }

    private void assertStillOpen() {
        if (this.fileChannel == null) {
            throw new IllegalStateException("Closed id generator " + this.file);
        }
    }

    @Override
    public synchronized IdRange nextIdBatch(int size) {
        long id;
        this.assertStillOpen();
        int count = 0;
        long[] defragIds = new long[size];
        while (count < size && (id = this.nextIdFromDefragList()) != -1L) {
            defragIds[count++] = id;
        }
        long[] tmpArray = defragIds;
        defragIds = new long[count];
        System.arraycopy(tmpArray, 0, defragIds, 0, count);
        int sizeLeftForRange = size - count;
        long start = this.highId.get();
        this.setHighId(start + (long)sizeLeftForRange);
        return new IdRange(defragIds, start, sizeLeftForRange);
    }

    @Override
    public void setHighId(long id) {
        this.assertIdWithinCapacity(id);
        this.highId.set(id);
    }

    @Override
    public long getHighId() {
        return this.highId.get();
    }

    @Override
    public long getHighestPossibleIdInUse() {
        return this.getHighId() - 1L;
    }

    @Override
    public synchronized void freeId(long id) {
        if (id == 0xFFFFFFFFL) {
            return;
        }
        if (this.fileChannel == null) {
            throw new IllegalStateException("Generator closed " + this.file);
        }
        if (id < 0L || id >= this.highId.get()) {
            throw new IllegalArgumentException("Illegal id[" + id + "], highId is " + this.highId.get());
        }
        this.releasedIdList.add(id);
        ++this.defraggedIdCount;
        if (this.releasedIdList.size() >= this.grabSize) {
            this.writeIdBatch(ByteBuffer.allocate(this.grabSize * 8));
        }
    }

    @Override
    public synchronized void close() {
        if (this.isClosed()) {
            return;
        }
        ByteBuffer writeBuffer = ByteBuffer.allocate(this.grabSize * 8);
        if (!this.releasedIdList.isEmpty()) {
            this.writeIdBatch(writeBuffer);
        }
        if (!this.idsReadFromFile.isEmpty()) {
            while (!this.idsReadFromFile.isEmpty()) {
                this.releasedIdList.add(this.idsReadFromFile.removeFirst());
            }
            this.writeIdBatch(writeBuffer);
        }
        try {
            ByteBuffer buffer = ByteBuffer.allocate(9);
            this.writeHeader(buffer);
            this.defragReusableIdsInFile(writeBuffer);
            this.fileChannel.force(false);
            this.markAsCleanlyClosed(buffer);
            this.closeChannel();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to close id generator " + this.file, e);
        }
    }

    private boolean isClosed() {
        return this.highId.get() == -1L;
    }

    private void closeChannel() throws IOException {
        this.fileChannel.force(false);
        this.fileChannel.close();
        this.fileChannel = null;
        this.highId.set(-1L);
    }

    private void markAsCleanlyClosed(ByteBuffer buffer) throws IOException {
        buffer.clear();
        buffer.put((byte)0);
        buffer.limit(1);
        buffer.flip();
        this.fileChannel.position(0L);
        this.fileChannel.write(buffer);
    }

    private void defragReusableIdsInFile(ByteBuffer writeBuffer) throws IOException {
        if (this.readPosition > 9L) {
            int bytesRead;
            long writePosition = 9L;
            long position = Math.min(this.readPosition, this.maxReadPosition);
            do {
                writeBuffer.clear();
                this.fileChannel.position(position);
                bytesRead = this.fileChannel.read(writeBuffer);
                position += (long)bytesRead;
                writeBuffer.flip();
                this.fileChannel.position(writePosition);
                writePosition += (long)this.fileChannel.write(writeBuffer);
            } while (bytesRead > 0);
            this.fileChannel.truncate(writePosition);
        }
    }

    private void writeHeader(ByteBuffer buffer) throws IOException {
        this.fileChannel.position(0L);
        buffer.put((byte)1).putLong(this.highId.get());
        buffer.flip();
        this.fileChannel.write(buffer);
    }

    public static void createGenerator(FileSystemAbstraction fs, File fileName) {
        IdGeneratorImpl.createGenerator(fs, fileName, 0L);
    }

    public static void createGenerator(FileSystemAbstraction fs, File fileName, long highId) {
        if (fs == null) {
            throw new IllegalArgumentException("Null filesystem");
        }
        if (fileName == null) {
            throw new IllegalArgumentException("Null filename");
        }
        if (fs.fileExists(fileName)) {
            throw new IllegalStateException("Can't create IdGeneratorFile[" + fileName + "], file already exists");
        }
        try {
            StoreChannel channel = fs.create(fileName);
            ByteBuffer buffer = ByteBuffer.allocate(9);
            buffer.put((byte)0).putLong(highId).flip();
            channel.write(buffer);
            channel.force(false);
            channel.close();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to create id generator" + fileName, e);
        }
    }

    private synchronized void initGenerator() {
        try {
            this.fileChannel = this.fs.open(this.file, "rw");
            ByteBuffer buffer = this.readHeader();
            this.markAsSticky(buffer);
            this.fileChannel.position(9L);
            this.maxReadPosition = this.fileChannel.size();
            this.defraggedIdCount = (int)(this.maxReadPosition - 9L) / 8;
            this.readIdBatch();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to init id generator " + this.file, e);
        }
    }

    private void markAsSticky(ByteBuffer buffer) throws IOException {
        buffer.clear();
        buffer.put((byte)1).limit(1).flip();
        this.fileChannel.position(0L);
        this.fileChannel.write(buffer);
        this.fileChannel.force(false);
    }

    private ByteBuffer readHeader() throws IOException {
        try {
            ByteBuffer buffer = IdGeneratorImpl.readHighIdFromHeader(this.fileChannel, this.file);
            this.readPosition = 9L;
            this.highId.set(buffer.getLong());
            return buffer;
        }
        catch (InvalidIdGeneratorException e) {
            this.fileChannel.close();
            throw e;
        }
    }

    private static ByteBuffer readHighIdFromHeader(StoreChannel channel, File fileName) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(9);
        int read = channel.read(buffer);
        if (read != 9) {
            throw new InvalidIdGeneratorException("Unable to read header, bytes read: " + read);
        }
        buffer.flip();
        byte storageStatus = buffer.get();
        if (storageStatus != 0) {
            throw new InvalidIdGeneratorException("Sticky generator[ " + fileName + "] delete this id file and build a new one");
        }
        return buffer;
    }

    public static long readHighId(FileSystemAbstraction fileSystem, File file) throws IOException {
        try (StoreChannel channel = fileSystem.open(file, "r");){
            long l = IdGeneratorImpl.readHighIdFromHeader(channel, file).getLong();
            return l;
        }
    }

    private void readIdBatch() {
        if (!this.canReadMoreIdBatches()) {
            return;
        }
        try {
            int howMuchToRead = (int)Math.min((long)(this.grabSize * 8), this.maxReadPosition - this.readPosition);
            ByteBuffer readBuffer = ByteBuffer.allocate(howMuchToRead);
            this.fileChannel.position(this.readPosition);
            int bytesRead = this.fileChannel.read(readBuffer);
            assert (this.fileChannel.position() <= this.maxReadPosition);
            this.readPosition += (long)bytesRead;
            readBuffer.flip();
            assert (bytesRead % 8 == 0);
            int idsRead = bytesRead / 8;
            for (int i = 0; i < idsRead; ++i) {
                long id = readBuffer.getLong();
                if (id == 0xFFFFFFFFL) continue;
                this.idsReadFromFile.add(id);
            }
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Failed reading defragged id batch", e);
        }
    }

    private void writeIdBatch(ByteBuffer writeBuffer) {
        try {
            this.fileChannel.position(this.fileChannel.size());
            writeBuffer.clear();
            while (!this.releasedIdList.isEmpty()) {
                long id = this.releasedIdList.removeFirst();
                if (id == 0xFFFFFFFFL) continue;
                writeBuffer.putLong(id);
                if (writeBuffer.position() != writeBuffer.capacity()) continue;
                writeBuffer.flip();
                while (writeBuffer.hasRemaining()) {
                    this.fileChannel.write(writeBuffer);
                }
                writeBuffer.clear();
            }
            writeBuffer.flip();
            while (writeBuffer.hasRemaining()) {
                this.fileChannel.write(writeBuffer);
            }
            this.fileChannel.position(this.readPosition);
            if (this.aggressiveReuse) {
                this.maxReadPosition = this.fileChannel.size();
            }
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to write defragged id  batch", e);
        }
    }

    public synchronized void dumpFreeIds() {
        while (this.canReadMoreIdBatches()) {
            this.readIdBatch();
        }
        for (Long id : this.idsReadFromFile) {
            System.out.print(" " + id);
        }
        System.out.println("\nNext free id: " + this.highId);
        this.close();
    }

    @Override
    public synchronized long getNumberOfIdsInUse() {
        return this.highId.get() - this.defraggedIdCount;
    }

    @Override
    public long getDefragCount() {
        return this.defraggedIdCount;
    }

    public void clearFreeIds() {
        this.releasedIdList.clear();
        this.idsReadFromFile.clear();
        this.defraggedIdCount = -1L;
        try {
            FileUtils.truncateFile((SeekableByteChannel)this.fileChannel, (long)9L);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public synchronized void delete() {
        if (!this.isClosed()) {
            try {
                this.closeChannel();
            }
            catch (IOException e) {
                throw new UnderlyingStorageException("Unable to safe close id generator " + this.file, e);
            }
        }
        if (!this.fs.deleteFile(this.file)) {
            throw new UnderlyingStorageException("Unable to delete id generator " + this.file);
        }
    }

    public String toString() {
        return "IdGeneratorImpl " + this.hashCode() + " [highId=" + this.highId + ", defragged=" + this.defraggedIdCount + ", fileName=" + this.file + ", max=" + this.max + ", aggressive=" + this.aggressiveReuse + "]";
    }
}

