/*
 * Copyright (c) 2002-2015 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.impl.index;

import static java.lang.String.format;

import java.io.IOException;
import java.util.Map;

import org.neo4j.graphdb.index.Index;
import org.neo4j.kernel.impl.transaction.command.Command;
import org.neo4j.kernel.impl.transaction.command.CommandRecordVisitor;
import org.neo4j.kernel.impl.transaction.command.NeoCommandHandler;
import org.neo4j.kernel.impl.transaction.command.NeoCommandType;

/**
 * Created from {@link IndexDefineCommand} or read from a logical log.
 * Contains all the different types of commands that an {@link Index} need
 * to support.
 */
public abstract class IndexCommand extends Command
{
    public static final byte VALUE_TYPE_NULL = (byte) 0;
    public static final byte VALUE_TYPE_SHORT = (byte) 1;
    public static final byte VALUE_TYPE_INT = (byte) 2;
    public static final byte VALUE_TYPE_LONG = (byte) 3;
    public static final byte VALUE_TYPE_FLOAT = (byte) 4;
    public static final byte VALUE_TYPE_DOUBLE = (byte) 5;
    public static final byte VALUE_TYPE_STRING = (byte) 6;

    private byte commandType;
    protected byte indexNameId;
    protected byte entityType;
    protected long entityId;
    protected byte keyId;
    protected byte valueType;
    protected Object value;

    protected void init( byte commandType, byte indexNameId, byte entityType, long entityId, byte keyId, Object value )
    {
        this.commandType = commandType ;
        this.indexNameId = indexNameId;
        this.entityType = entityType;
        this.entityId = entityId;
        this.keyId = keyId;
        this.value = value;
        this.valueType = valueTypeOf( value );
    }

    public byte getIndexNameId()
    {
        return indexNameId;
    }

    public byte getEntityType()
    {
        return entityType;
    }

    public long getEntityId()
    {
        return entityId;
    }

    public byte getKeyId()
    {
        return keyId;
    }

    public Object getValue()
    {
        return value;
    }

    @Override
    public void accept( CommandRecordVisitor visitor )
    {
        // no op
    }

    public byte startNodeNeedsLong()
    {
        return 0;
    }

    public byte endNodeNeedsLong()
    {
        return 0;
    }

    private static byte valueTypeOf( Object value )
    {
        byte valueType = 0;
        if ( value == null )
        {
            valueType = VALUE_TYPE_NULL;
        }
        else if ( value instanceof Number )
        {
            if ( value instanceof Float )
            {
                valueType = VALUE_TYPE_FLOAT;
            }
            else if ( value instanceof Double )
            {
                valueType = VALUE_TYPE_DOUBLE;
            }
            else if ( value instanceof Long )
            {
                valueType = VALUE_TYPE_LONG;
            }
            else if ( value instanceof Short )
            {
                valueType = VALUE_TYPE_SHORT;
            }
            else
            {
                valueType = VALUE_TYPE_INT;
            }
        }
        else
        {
            valueType = VALUE_TYPE_STRING;
        }
        return valueType;
    }

    public static class AddNodeCommand extends IndexCommand
    {
        public void init( byte indexNameId, long entityId, byte keyId, Object value )
        {
            super.init( NeoCommandType.INDEX_ADD_COMMAND, indexNameId, IndexEntityType.Node.id(),
                    entityId, keyId, value );
        }

        @Override
        public boolean handle( NeoCommandHandler visitor ) throws IOException
        {
            return visitor.visitIndexAddNodeCommand( this );
        }

        @Override
        public String toString()
        {
            return "AddNode[index:" + indexNameId + ", id:" + entityId + ", key:" + keyId + ", value:" + value + "]";
        }
    }

    protected static byte needsLong( long value )
    {
        return value > Integer.MAX_VALUE ? (byte)1 : (byte)0;
    }

    public static class AddRelationshipCommand extends IndexCommand
    {
        private long startNode;
        private long endNode;

        public void init( byte indexNameId, long entityId, byte keyId,
                Object value, long startNode, long endNode )
        {
            super.init( NeoCommandType.INDEX_ADD_RELATIONSHIP_COMMAND, indexNameId, IndexEntityType.Relationship.id(),
                    entityId, keyId, value );
            this.startNode = startNode;
            this.endNode = endNode;
        }

        public long getStartNode()
        {
            return startNode;
        }

        public long getEndNode()
        {
            return endNode;
        }

        @Override
        public byte startNodeNeedsLong()
        {
            return needsLong( startNode );
        }

        @Override
        public byte endNodeNeedsLong()
        {
            return needsLong( endNode );
        }

        @Override
        public int hashCode()
        {
            int result = (int) (startNode ^ (startNode >>> 32));
            result = 31 * result + (int) (endNode ^ (endNode >>> 32));
            return result;
        }

        @Override
        public boolean equals( Object obj )
        {
            if ( !super.equals( obj ) )
            {
                return false;
            }
            AddRelationshipCommand other = (AddRelationshipCommand) obj;
            return startNode == other.startNode && endNode == other.endNode;
        }

        @Override
        public boolean handle( NeoCommandHandler visitor ) throws IOException
        {
            return visitor.visitIndexAddRelationshipCommand( this );
        }

        @Override
        public String toString()
        {
            return "AddRelationship[index:" + indexNameId + ", id:" + entityId + ", key:" + keyId +
                    ", value:" + value + "(" + (value != null ? value.getClass().getSimpleName() : "null") + ")" +
                    ", startNode:" + startNode +
                    ", endNode:" + endNode +
                    "]";
        }
    }

    public static class RemoveCommand extends IndexCommand
    {
        public void init( byte indexNameId, byte entityType, long entityId, byte keyId, Object value )
        {
            super.init( NeoCommandType.INDEX_REMOVE_COMMAND, indexNameId, entityType, entityId, keyId, value );
        }

        @Override
        public boolean handle( NeoCommandHandler visitor ) throws IOException
        {
            return visitor.visitIndexRemoveCommand( this );
        }

        @Override
        public String toString()
        {
            return format( "Remove%s[index:%d, id:%d, key:%d, value:%s]",
                    IndexEntityType.byId( entityType ).nameToLowerCase(), indexNameId, entityId, keyId, value );
        }
    }

    public static class DeleteCommand extends IndexCommand
    {
        public void init( byte indexNameId, byte entityType )
        {
            super.init( NeoCommandType.INDEX_DELETE_COMMAND, indexNameId, entityType, 0L, (byte)0, null );
        }

        @Override
        public boolean handle( NeoCommandHandler visitor ) throws IOException
        {
            return visitor.visitIndexDeleteCommand( this );
        }

        @Override
        public String toString()
        {
            return "Delete[index:" + indexNameId + ", type:" + IndexEntityType.byId( entityType ).nameToLowerCase() + "]";
        }
    }

    public static class CreateCommand extends IndexCommand
    {
        private Map<String, String> config;

        public void init( byte indexNameId, byte entityType, Map<String, String> config )
        {
            super.init( NeoCommandType.INDEX_CREATE_COMMAND, indexNameId, entityType, 0L, (byte)0, null );
            this.config = config;
        }

        public Map<String, String> getConfig()
        {
            return config;
        }

        @Override
        public int hashCode()
        {
            return config != null ? config.hashCode() : 0;
        }

        @Override
        public boolean equals( Object obj )
        {
            return super.equals( obj ) && config.equals( ((CreateCommand)obj).config );
        }

        @Override
        public boolean handle( NeoCommandHandler visitor ) throws IOException
        {
            return visitor.visitIndexCreateCommand( this );
        }

        @Override
        public String toString()
        {
            return format( "Create%sIndex[index:%d, config:%s]",
                    IndexEntityType.byId( entityType ).nameToLowerCase(), indexNameId, config );
        }
    }

    @Override
    public boolean equals( Object obj )
    {
        IndexCommand other = (IndexCommand) obj;
        boolean equals = getCommandType() == other.getCommandType() &&
                entityType == other.entityType &&
                indexNameId == other.indexNameId &&
                keyId == other.keyId &&
                getValueType() == other.getValueType();
        if ( !equals )
        {
            return false;
        }

        return value == null ? other.value == null : value.equals( other.value );
    }

    public byte getCommandType()
    {
        return commandType;
    }

    public byte getValueType()
    {
        return valueType;
    }
}
