/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;
import org.neo4j.graphdb.TransactionFailureException;
import org.neo4j.helpers.UTF8;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.kernel.Lifecycle;
import org.neo4j.kernel.impl.nioneo.xa.NeoStoreXaDataSource;
import org.neo4j.kernel.impl.transaction.TxLog;
import org.neo4j.kernel.impl.transaction.XidImpl;
import org.neo4j.kernel.impl.transaction.xaframework.XaDataSource;
import org.neo4j.kernel.impl.transaction.xaframework.XaResource;
import org.neo4j.kernel.impl.util.StringLogger;

public class XaDataSourceManager
implements Lifecycle {
    private static Logger log = Logger.getLogger(XaDataSourceManager.class.getName());
    private final Map<String, XaDataSource> dataSources = new HashMap<String, XaDataSource>();
    private final Map<String, XaDataSource> branchIdMapping = new HashMap<String, XaDataSource>();
    private final Map<String, byte[]> sourceIdMapping = new HashMap<String, byte[]>();
    private StringLogger msgLog;

    public XaDataSourceManager(StringLogger msgLog) {
        this.msgLog = msgLog;
    }

    @Override
    public void init() throws Throwable {
    }

    @Override
    public void start() throws Throwable {
    }

    @Override
    public void stop() throws Throwable {
    }

    @Override
    public void shutdown() throws Throwable {
        this.branchIdMapping.clear();
        this.sourceIdMapping.clear();
        for (XaDataSource dataSource : this.dataSources.values()) {
            dataSource.close();
        }
        this.dataSources.clear();
    }

    public XaDataSource getXaDataSource(String name) {
        return this.dataSources.get(name);
    }

    public NeoStoreXaDataSource getNeoStoreDataSource() {
        return (NeoStoreXaDataSource)this.getXaDataSource("nioneodb");
    }

    public synchronized void registerDataSource(XaDataSource dataSource) {
        this.dataSources.put(dataSource.getName(), dataSource);
        this.branchIdMapping.put(UTF8.decode(dataSource.getBranchId()), dataSource);
        this.sourceIdMapping.put(dataSource.getName(), dataSource.getBranchId());
    }

    public synchronized void unregisterDataSource(String name) {
        XaDataSource dataSource = this.dataSources.get(name);
        byte[] branchId = this.getBranchId(dataSource.getXaConnection().getXaResource());
        this.dataSources.remove(name);
        this.branchIdMapping.remove(UTF8.decode(branchId));
        this.sourceIdMapping.remove(name);
        dataSource.close();
    }

    synchronized byte[] getBranchId(XAResource xaResource) {
        byte[] branchId;
        if (xaResource instanceof XaResource && (branchId = ((XaResource)xaResource).getBranchId()) != null) {
            return branchId;
        }
        for (Map.Entry<String, XaDataSource> entry : this.dataSources.entrySet()) {
            XaDataSource dataSource = entry.getValue();
            XAResource resource = dataSource.getXaConnection().getXaResource();
            try {
                if (!resource.isSameRM(xaResource)) continue;
                String name = entry.getKey();
                return this.sourceIdMapping.get(name);
            }
            catch (XAException e) {
                throw new TransactionFailureException("Unable to check is same resource", e);
            }
        }
        throw new TransactionFailureException("Unable to find mapping for XAResource[" + xaResource + "]");
    }

    private XaDataSource getDataSource(byte[] branchId) {
        XaDataSource dataSource = this.branchIdMapping.get(UTF8.decode(branchId));
        if (dataSource == null) {
            throw new TransactionFailureException("No mapping found for branchId[0x" + UTF8.decode(branchId) + "]");
        }
        return dataSource;
    }

    public Collection<XaDataSource> getAllRegisteredDataSources() {
        return this.dataSources.values();
    }

    public void recover(Iterator<List<TxLog.Record>> danglingRecordList) {
        ArrayList<NonCompletedTransaction> commitList = new ArrayList<NonCompletedTransaction>();
        LinkedList<Xid> rollbackList = new LinkedList<Xid>();
        HashMap<Resource, XaDataSource> resourceMap = new HashMap<Resource, XaDataSource>();
        this.buildRecoveryInfo(commitList, rollbackList, resourceMap, danglingRecordList);
        LinkedList<Xid> recoveredXidsList = new LinkedList<Xid>();
        try {
            Xid[] xids;
            for (XaDataSource xaDataSource : this.dataSources.values()) {
                XAResource xaRes = xaDataSource.getXaConnection().getXaResource();
                xids = xaRes.recover(0);
                for (int i = 0; i < xids.length; ++i) {
                    if (XidImpl.isThisTm(xids[i].getGlobalTransactionId())) {
                        if (rollbackList.contains(xids[i])) {
                            log.fine("Found pre commit " + xids[i] + " rolling back ... ");
                            this.msgLog.logMessage("TM: Found pre commit " + xids[i] + " rolling back ... ", true);
                            rollbackList.remove(xids[i]);
                            xaRes.rollback(xids[i]);
                            continue;
                        }
                        Resource resource = new Resource(xids[i].getBranchQualifier());
                        if (!resourceMap.containsKey(resource)) {
                            resourceMap.put(resource, xaDataSource);
                        }
                        recoveredXidsList.add(xids[i]);
                        continue;
                    }
                    log.warning("Unknown xid: " + xids[i]);
                }
            }
            Collections.sort(commitList);
            for (NonCompletedTransaction nct : commitList) {
                int seq = nct.getSequenceNumber();
                xids = nct.getXids();
                log.fine("Marked as commit tx-seq[" + seq + "] branch length: " + xids.length);
                for (Xid xid : xids) {
                    if (!recoveredXidsList.contains(xid)) {
                        log.fine("Tx-seq[" + seq + "][" + xid + "] not found in recovered xid list, " + "assuming already committed");
                        continue;
                    }
                    recoveredXidsList.remove(xid);
                    Resource resource = new Resource(xid.getBranchQualifier());
                    if (!resourceMap.containsKey(resource)) {
                        TransactionFailureException ex = new TransactionFailureException("Couldn't find XAResource for " + xid);
                        throw this.logAndReturn("TM: recovery error", ex);
                    }
                    log.fine("Commiting tx seq[" + seq + "][" + xid + "] ... ");
                    this.msgLog.logMessage("TM: Committing tx " + xid, true);
                    ((XaDataSource)resourceMap.get(resource)).getXaConnection().getXaResource().commit(xid, false);
                }
            }
            for (Xid xid : recoveredXidsList) {
                Resource resource = new Resource(xid.getBranchQualifier());
                if (!resourceMap.containsKey(resource)) {
                    TransactionFailureException ex = new TransactionFailureException("Couldn't find XAResource for " + xid);
                    throw this.logAndReturn("TM: recovery error", ex);
                }
                log.fine("Rollback " + xid + " ... ");
                this.msgLog.logMessage("TM: no match found for " + xid + " removing", true);
                ((XaDataSource)resourceMap.get(resource)).getXaConnection().getXaResource().rollback(xid);
            }
            if (rollbackList.size() > 0) {
                log.fine("TxLog contained unresolved xids that needed rollback. They couldn't be matched to any of the XAResources recover list. Assuming " + rollbackList.size() + " transactions already rolled back.");
                this.msgLog.logMessage("TM: no match found for in total " + rollbackList.size() + " transaction that should have been rolled back", true);
            }
            for (XaDataSource participant : MapUtil.reverse(resourceMap).keySet()) {
                participant.rotateLogicalLog();
            }
        }
        catch (IOException e) {
            throw this.logAndReturn("TM: recovery failed", new TransactionFailureException("Recovery failed.", e));
        }
        catch (XAException e) {
            throw this.logAndReturn("TM: recovery failed", new TransactionFailureException("Recovery failed.", e));
        }
    }

    private void buildRecoveryInfo(List<NonCompletedTransaction> commitList, List<Xid> rollbackList, Map<Resource, XaDataSource> resourceMap, Iterator<List<TxLog.Record>> danglingRecordList) {
        while (danglingRecordList.hasNext()) {
            Iterator<TxLog.Record> dListItr = danglingRecordList.next().iterator();
            TxLog.Record startRecord = dListItr.next();
            if (startRecord.getType() != 1) {
                throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("First record not a start record, type=" + startRecord.getType()));
            }
            HashSet<Resource> branchSet = new HashSet<Resource>();
            int markedCommit = -1;
            while (dListItr.hasNext()) {
                TxLog.Record record = dListItr.next();
                if (record.getType() == 2) {
                    if (markedCommit != -1) {
                        throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("Already marked commit " + startRecord));
                    }
                    branchSet.add(new Resource(record.getBranchId()));
                    continue;
                }
                if (record.getType() == 3) {
                    if (markedCommit != -1) {
                        throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("Already marked commit " + startRecord));
                    }
                    markedCommit = record.getSequenceNumber();
                    continue;
                }
                throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("Illegal record type[" + record.getType() + "]"));
            }
            Iterator resourceItr = branchSet.iterator();
            LinkedList<Xid> xids = new LinkedList<Xid>();
            while (resourceItr.hasNext()) {
                Resource resource = (Resource)resourceItr.next();
                if (!resourceMap.containsKey(resource)) {
                    resourceMap.put(resource, this.getDataSource(resource.getResourceId()));
                }
                xids.add(new XidImpl(startRecord.getGlobalId(), resource.getResourceId()));
            }
            if (markedCommit != -1) {
                commitList.add(new NonCompletedTransaction(markedCommit, xids));
                continue;
            }
            rollbackList.addAll(xids);
        }
    }

    private <E extends Exception> E logAndReturn(String msg, E exception) {
        try {
            this.msgLog.logMessage(msg, exception, true);
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return exception;
    }

    public void rotateLogicalLogs() {
        for (XaDataSource dataSource : this.dataSources.values()) {
            try {
                dataSource.rotateLogicalLog();
            }
            catch (IOException e) {
                this.msgLog.logMessage("Couldn't rotate logical log for " + dataSource.getName(), e);
            }
        }
    }

    private static class Resource {
        private byte[] resourceId = null;
        private volatile int hashCode = 0;

        Resource(byte[] resourceId) {
            if (resourceId == null || resourceId.length == 0) {
                throw new IllegalArgumentException("Illegal resourceId");
            }
            this.resourceId = resourceId;
        }

        byte[] getResourceId() {
            return this.resourceId;
        }

        public boolean equals(Object o) {
            if (!(o instanceof Resource)) {
                return false;
            }
            byte[] otherResourceId = ((Resource)o).getResourceId();
            if (this.resourceId.length != otherResourceId.length) {
                return false;
            }
            for (int i = 0; i < this.resourceId.length; ++i) {
                if (this.resourceId[i] == otherResourceId[i]) continue;
                return false;
            }
            return true;
        }

        public int hashCode() {
            if (this.hashCode == 0) {
                int calcHash = 0;
                for (int i = 0; i < this.resourceId.length; ++i) {
                    calcHash += this.resourceId[i] << i * 8;
                }
                this.hashCode = 3217 * calcHash;
            }
            return this.hashCode;
        }
    }

    private static class NonCompletedTransaction
    implements Comparable<NonCompletedTransaction> {
        private int seqNr = -1;
        private List<Xid> xidList = null;

        NonCompletedTransaction(int seqNr, List<Xid> xidList) {
            this.seqNr = seqNr;
            this.xidList = xidList;
        }

        int getSequenceNumber() {
            return this.seqNr;
        }

        Xid[] getXids() {
            return this.xidList.toArray(new XidImpl[this.xidList.size()]);
        }

        public String toString() {
            return "NonCompletedTx[" + this.seqNr + "," + this.xidList + "]";
        }

        @Override
        public int compareTo(NonCompletedTransaction nct) {
            return this.getSequenceNumber() - nct.getSequenceNumber();
        }
    }
}

